/*
 * Copyright (c) 2007-2015 Freescale Semiconductor, Inc.
 * Copyright 2018-2020 NXP
 *
 * License: NXP LA_OPT_NXP_Software_License
 *
 * NXP Confidential. This software is owned or controlled by NXP and may
 * only be used strictly in accordance with the applicable license terms.
 * By expressly accepting such terms or by downloading, installing,
 * activating and/or otherwise using the software, you are agreeing that
 * you have read, and that you agree to comply with and are bound by,
 * such license terms.  If you do not agree to be bound by the applicable
 * license terms, then you may not retain, install, activate or otherwise
 * use the software.  This code may only be used in a microprocessor,
 * microcontroller, sensor or digital signal processor ("NXP Product")
 * supplied directly or indirectly from NXP.  See the full NXP Software
 * License Agreement in license/LA_OPT_NXP_Software_License.pdf
 *
 * FreeMASTER Communication Driver - msCAN low-level driver
 */

#include "freemaster.h"
#include "freemaster_private.h"

/* Numeric identifier to help pre-processor to identify whether our driver is used or not. */
#define FMSTR_FLEXCAN_MPC577xC_ID 1


#if (FMSTR_MK_IDSTR(FMSTR_CAN_DRV) == FMSTR_FLEXCAN_MPC577XC)
#if !(FMSTR_DISABLE)

#include "freemaster_can.h"
#include "freemaster_mpc5777c_can.h"

/******************************************************************************
* Local variables
******************************************************************************/
/* Serial base address */

#ifdef FMSTR_FLEXCAN_BASE
    static FMSTR_ADDR fmstr_canBaseAddr = (FMSTR_ADDR)FMSTR_FLEXCAN_BASE;
#else
    static FMSTR_ADDR fmstr_canBaseAddr = (FMSTR_ADDR)0;
#endif

flexcan_frame_t *fmstr_rxmsg =(flexcan_frame_t*) (FMSTR_FLEXCAN_BASE+0x80);								/*There is used FIFO for receiving the MBs, the MB0 address is used for pushing the data from FIFO*/

#if FMSTR_FLEXCAN_TXMB<8
flexcan_frame_t *fmstr_txmsg =(flexcan_frame_t*) (FMSTR_FLEXCAN_BASE+0x80+0x10*8);     /*MB[8] is the lowest MB which can be used for transmitting when the RX FIFO is   Buffer to prepare transmission */
#else
flexcan_frame_t *fmstr_txmsg =(flexcan_frame_t*) (FMSTR_FLEXCAN_BASE+0x80+0x10*FMSTR_FLEXCAN_TXMB);     /*MB[FMSTR_FLEXCAN_TXMB] -  Buffer to prepare transmission */
#endif
flexcan_received_mb rx_data,tx_data;
flexcan_received_mb *fmstr_rx_data = &rx_data;
flexcan_received_mb *fmstr_tx_data = &tx_data;

/******************************************************************************
* Local functions
******************************************************************************/

static FMSTR_BOOL _FMSTR_MPC577xCCan_Init(FMSTR_U32 idRx, FMSTR_U32 idTx);  /* Initialize CAN module on a given base address. */
static void _FMSTR_MPC577xCCan_EnableTxInterrupt(FMSTR_BOOL enable);     /* Enable CAN Transmit interrupt. */
static void _FMSTR_MPC577xCCan_EnableRxInterrupt(FMSTR_BOOL enable);     /* Enable CAN Receive interrupt. */
static void _FMSTR_MPC577xCCan_EnableRx(void);                           /* Enable/re-initialize Receiver buffer. */
static FMSTR_SIZE8 _FMSTR_MPC577xCCan_GetRxFrameLen(void);               /* Return size of received CAN frame, or 0 if no Rx frame is available. */
static FMSTR_BCHR _FMSTR_MPC577xCCan_GetRxFrameByte(FMSTR_SIZE8 index);  /* Get data byte at index (0..8). */
static void _FMSTR_MPC577xCCan_AckRxFrame(void);                         /* Discard received frame and enable receiving a next one. */
static FMSTR_BOOL _FMSTR_MPC577xCCan_PrepareTxFrame(void);               /* Initialize transmit buffer; return false when Tx buffer is not available. */
static void _FMSTR_MPC577xCCan_PutTxFrameByte(FMSTR_SIZE8 index, FMSTR_BCHR data);   /* Fill one byte of transmit data. */
static void _FMSTR_MPC577xCCan_SendTxFrame(FMSTR_SIZE8 len);             /* Send the Tx buffer. */

/******************************************************************************
* Driver interface
******************************************************************************/

const FMSTR_CAN_DRV_INTF FMSTR_FLEXCAN_MPC577XC =
{
    .Init = _FMSTR_MPC577xCCan_Init,
    .EnableTxInterrupt = _FMSTR_MPC577xCCan_EnableTxInterrupt,
    .EnableRxInterrupt = _FMSTR_MPC577xCCan_EnableRxInterrupt,
    .EnableRx =        _FMSTR_MPC577xCCan_EnableRx,
	.GetRxFrameLen =   _FMSTR_MPC577xCCan_GetRxFrameLen,
    .GetRxFrameByte =  _FMSTR_MPC577xCCan_GetRxFrameByte,
    .AckRxFrame =      _FMSTR_MPC577xCCan_AckRxFrame,
    .PrepareTxFrame =  _FMSTR_MPC577xCCan_PrepareTxFrame,
    .PutTxFrameByte =  _FMSTR_MPC577xCCan_PutTxFrameByte,
    .SendTxFrame =     _FMSTR_MPC577xCCan_SendTxFrame,
};

/****************************************************************************************
* General peripheral space access macros
*****************************************************************************************/

#define FMSTR_SETBIT(base, offset, bit)         (*(((volatile FMSTR_U32*)(base+offset))) |= (bit))
#define FMSTR_CLRBIT(base, offset, bit)         (*(((volatile FMSTR_U32*)(base+offset))) &= ~(bit))
#define FMSTR_TSTBIT(base, offset, bit)         (*(((volatile FMSTR_U32*)(base+offset))) & (bit))
#define FMSTR_SETREG(base, offset, value)       (*(((volatile FMSTR_U32*)(base+offset))) = (value))
#define FMSTR_GETREG(base, offset)              (*(((volatile FMSTR_U32*)(base+offset))))

/****************************************************************************************
* CAN module constants
*****************************************************************************************/

/* FLEXCAN module registers */
#define FMSTR_FLEXCAN_IMASK1_OFFSET 0x28
#define FMSTR_FLEXCAN_IMASK2_OFFSET 0x24

#define FMSTR_FLEXCAN_IFLAG2_OFFSET 0x2C
#define FMSTR_FLEXCAN_IFLAG1_OFFSET 0x30

#define FMSTR_FLEXCAN_RX_FIFO_ID0 0xE0
#define FMSTR_FLEXCAN_RX_FIFO_INT 0x00000020 /*MB[5] flag is connected with the FIFO entries*/


#if FMSTR_FLEXCAN_TXMB<32
#define FMSTR_FLEXCAN_TX_IFLAG_OFFSET FMSTR_FLEXCAN_IFLAG1_OFFSET
#define FMSTR_FLEXCAN_TX_IMASK_OFFSET FMSTR_FLEXCAN_IMASK1_OFFSET
#if FMSTR_FLEXCAN_TXMB<8
#define FMSTR_FLEXCAN_TX_INT (1<<8)
#else
#define FMSTR_FLEXCAN_TX_INT (1<<FMSTR_FLEXCAN_TXMB)
#endif
#else
#define FMSTR_FLEXCAN_TX_IFLAG_OFFSET FMSTR_FLEXCAN_IFLAG2_OFFSET
#define FMSTR_FLEXCAN_TX_IMASK_OFFSET FMSTR_FLEXCAN_IMASK2_OFFSET
#define FMSTR_FLEXCAN_TX_INT (1<<(FMSTR_FLEXCAN_TXMB-32))
#endif

/******************************************************************************
* Implementation
******************************************************************************/

static FMSTR_BOOL _FMSTR_MPC577xCCan_Init(FMSTR_U32 idRx, FMSTR_U32 idTx)
{
    /* initialize Rx FIFO - There is used Format A of the ID table*/

	fmstr_rxmsg->CODE =RX_MB_INACTIVE;

	if (idRx & FMSTR_CAN_EXTID) /*Extended format of the ID*/
	{
		*((volatile FMSTR_U32*)(FMSTR_FLEXCAN_BASE+FMSTR_FLEXCAN_RX_FIFO_ID0)) = idRx<<1; /*Store the FreeMASTER RX MB into the ID filter table*/
	}
	else /*Standard format of the ID*/
	{
		*((volatile FMSTR_U32*)(FMSTR_FLEXCAN_BASE+FMSTR_FLEXCAN_RX_FIFO_ID0)) = idRx<<19; /*Store the FreeMASTER RX MB into the ID filter table*/
	}

    /* initialize Tx MB */
	fmstr_txmsg->CODE =TX_MB_INACTIVE;

	if (idTx & FMSTR_CAN_EXTID) /*Extended format of the ID*/
	{
		fmstr_txmsg->ID=idTx&0x1FFFFFFF;
		fmstr_txmsg->IDE = 0x1;

	}
	else /*Standard format of the ID*/
	{
		fmstr_txmsg->ID=(idTx&0x7FF)<<18;
		fmstr_txmsg->IDE = 0x0;

	}

#if FMSTR_LONG_INTR || FMSTR_SHORT_INTR
	FMSTR_SETREG(FMSTR_FLEXCAN_BASE,FMSTR_FLEXCAN_IMASK2_OFFSET,0x0);
	FMSTR_SETREG(FMSTR_FLEXCAN_BASE,FMSTR_FLEXCAN_IMASK1_OFFSET,0x0);
#endif

    return FMSTR_TRUE;
}

static void _FMSTR_MPC577xCCan_EnableTxInterrupt(FMSTR_BOOL enable)
{
	if (enable)
	{
		FMSTR_SETBIT(FMSTR_FLEXCAN_BASE, FMSTR_FLEXCAN_TX_IMASK_OFFSET, FMSTR_FLEXCAN_TX_INT);
	}
	else
	{
		FMSTR_CLRBIT(FMSTR_FLEXCAN_BASE, FMSTR_FLEXCAN_TX_IMASK_OFFSET, FMSTR_FLEXCAN_TX_INT);
	}
}

static void _FMSTR_MPC577xCCan_EnableRxInterrupt(FMSTR_BOOL enable)
{
	if (enable)
	{
		FMSTR_SETBIT(FMSTR_FLEXCAN_BASE, FMSTR_FLEXCAN_IMASK1_OFFSET, FMSTR_FLEXCAN_RX_FIFO_INT);
	}
	else
	{
		FMSTR_CLRBIT(FMSTR_FLEXCAN_BASE, FMSTR_FLEXCAN_IMASK1_OFFSET, FMSTR_FLEXCAN_RX_FIFO_INT);
	}

}

static void _FMSTR_MPC577xCCan_EnableRx(void)
{
	/*enable receiving into the MB*/
	/*When RX FIFO is enabled receiving is enabled*/;
}

static FMSTR_SIZE8 _FMSTR_MPC577xCCan_GetRxFrameLen(void)
{

	uint8_t i=0;
#ifdef FMSTR_POLL_DRIVEN
    /* Is any data received? */
	if(!FMSTR_TSTBIT(FMSTR_FLEXCAN_BASE, FMSTR_FLEXCAN_IFLAG1_OFFSET,FMSTR_FLEXCAN_RX_FIFO_INT)) return 0;

#endif /* FMSTR_POLL_DRIVEN */

    /* Current cache still valid? */
    if(!fmstr_rx_data->length)
    {
    	/* Try to read, when successfull, the MB is acknowledged and set for next receive */
    	if(!FMSTR_TSTBIT(FMSTR_FLEXCAN_BASE, FMSTR_FLEXCAN_IFLAG1_OFFSET,FMSTR_FLEXCAN_RX_FIFO_INT)) return 0;

    	fmstr_rx_data->length = fmstr_rxmsg->DLC;

    	for (i=0;i<fmstr_rx_data->length;i++) fmstr_rx_data->data[i] = fmstr_rxmsg->data[i];

    	FMSTR_SETREG(FMSTR_FLEXCAN_BASE, FMSTR_FLEXCAN_IFLAG1_OFFSET,FMSTR_FLEXCAN_RX_FIFO_INT); /*Clearing the FIFO RX flag for pushing next RX MB if there any in the FIFO*/
    }

    /* we have got some frame, return its length */
    return fmstr_rx_data->length;
}

static FMSTR_BCHR _FMSTR_MPC577xCCan_GetRxFrameByte(FMSTR_SIZE8 index)
{
	FMSTR_BCHR data;

    /* need switch as data bytes are not necessarily ordered in the frame */
    switch(index)
    {
    case 0:  data = fmstr_rx_data->data[0]; break;
    case 1:  data = fmstr_rx_data->data[1]; break;
    case 2:  data = fmstr_rx_data->data[2]; break;
    case 3:  data = fmstr_rx_data->data[3]; break;
    case 4:  data = fmstr_rx_data->data[4]; break;
    case 5:  data = fmstr_rx_data->data[5]; break;
    case 6:  data = fmstr_rx_data->data[6]; break;
    default: data = fmstr_rx_data->data[7]; break;
    }

    return data;
}

static void _FMSTR_MPC577xCCan_AckRxFrame(void)
{
    /* The frame is already acknowledged in registers by calling FLEXCAN_ReadRxMb before. */
    /* We only clear the local cached buffer so it appears as if we have acknowledged it. */
    fmstr_rx_data->length = 0;
}

static FMSTR_BOOL _FMSTR_MPC577xCCan_PrepareTxFrame(void)
{
#ifdef FMSTR_POLL_DRIVEN
    /* Was all data sent? */

	if(fmstr_tx_data->length && (FMSTR_TSTBIT(FMSTR_FLEXCAN_BASE, FMSTR_FLEXCAN_TX_IFLAG_OFFSET, FMSTR_FLEXCAN_TX_INT)==0))

    return FMSTR_FALSE;

    /* Clear TX status flag */
    FMSTR_SETREG(FMSTR_FLEXCAN_BASE,FMSTR_FLEXCAN_TX_IFLAG_OFFSET, FMSTR_FLEXCAN_TX_INT);

    /* Acknowledge frame was transmitted */
    fmstr_tx_data->length = 0;
#else
    if(fmstr_tx_data->length)
        return FMSTR_FALSE;
#endif /* FMSTR_POLL_DRIVEN */

    /* set as transmit-emtpy MB */
    fmstr_txmsg->CODE = TX_MB_INACTIVE;

    fmstr_txmsg->DLC = 0;
    fmstr_txmsg->data[0] = 0;
    fmstr_txmsg->data[1] = 0;
    fmstr_txmsg->data[2] = 0;
    fmstr_txmsg->data[3] = 0;
    fmstr_txmsg->data[4] = 0;
    fmstr_txmsg->data[5] = 0;
    fmstr_txmsg->data[6] = 0;
    fmstr_txmsg->data[7] = 0;


    return FMSTR_TRUE;
}

static void _FMSTR_MPC577xCCan_PutTxFrameByte(FMSTR_SIZE8 index, FMSTR_BCHR data)
{
    /* need switch as data bytes are not necessarily ordered in the frame */
    switch(index)
    {
    case 0:  fmstr_tx_data->data[0] = data; break;
    case 1:  fmstr_tx_data->data[1] = data; break;
    case 2:  fmstr_tx_data->data[2] = data; break;
    case 3:  fmstr_tx_data->data[3] = data; break;
    case 4:  fmstr_tx_data->data[4] = data; break;
    case 5:  fmstr_tx_data->data[5] = data; break;
    case 6:  fmstr_tx_data->data[6] = data; break;
    default: fmstr_tx_data->data[7] = data; break;
    }
}

static void _FMSTR_MPC577xCCan_SendTxFrame(FMSTR_SIZE8 len)
{
	uint8_t i=0;
	//check
	//if (fmstr_txmsg->CODE != TX_MB_INACTIVE) return 0;
	fmstr_txmsg->DLC = len;

	for (i=0;i<len;i++) fmstr_txmsg->data[i] = fmstr_tx_data->data[i];

	fmstr_txmsg->CODE = TX_MB_DATA;
}

/**************************************************************************//*!
*
* @brief    Assigning FreeMASTER communication module base address
*
******************************************************************************/

void FMSTR_CanSetBaseAddress(FMSTR_ADDR base)
{
    fmstr_canBaseAddr = base;
}

/**************************************************************************//*!
*
* @brief    Process FreeMASTER CAN interrupt (call this function from msCAN ISR)
*
******************************************************************************/

void FMSTR_CanIsr(void)
{
#if FMSTR_LONG_INTR || FMSTR_SHORT_INTR

	if (FMSTR_TSTBIT(FMSTR_FLEXCAN_BASE, FMSTR_FLEXCAN_IFLAG1_OFFSET,FMSTR_FLEXCAN_RX_FIFO_INT))
	{
        /* Process received frame */

        FMSTR_ProcessCanRx();

    }

    if (FMSTR_TSTBIT(FMSTR_FLEXCAN_BASE, FMSTR_FLEXCAN_TX_IFLAG_OFFSET, FMSTR_FLEXCAN_TX_INT))
    {
        /* Send next frame, if needed */
        FMSTR_ProcessCanTx();
    }
#endif
}

#else /* (!(FMSTR_DISABLE)) */

/* Empty API functions when FMSTR_DISABLE is set */
void FMSTR_CanSetBaseAddress(FMSTR_ADDR base)
{
    FMSTR_UNUSED(base);
}

void FMSTR_CanIsr(void)
{
}

#endif /* !(FMSTR_DISABLE) */ 
#endif /* (FMSTR_MK_IDSTR(FMSTR_CAN_DRV) == FMSTR_CAN_MCUX_MSCAN_ID) */
