/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2010 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file     meas_s12zvm.c
*
* @author   b35258
*
* @version  1.0.2.0
*
* @date     Jun-11-2015
*
* @brief    Header file for measurement module
*
*******************************************************************************
*
*
******************************************************************************/

/******************************************************************************
| Includes
-----------------------------------------------------------------------------*/
#include "meas_S12zvm.h"
#include "gflib.h"
#include "gmclib.h"
#include "gdflib.h"

/******************************************************************************
| External declarations
-----------------------------------------------------------------------------*/

/******************************************************************************
| Defines and macros            (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Typedefs and structures       (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Global variable definitions   (scope: module-exported)
-----------------------------------------------------------------------------*/
extern volatile unsigned short ADC0ResultList[];
extern volatile unsigned short ADC1ResultList[];


/******************************************************************************
| Global variable definitions   (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function prototypes           (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-local)
-----------------------------------------------------------------------------*/

/******************************************************************************
| Function implementations      (scope: module-exported)
-----------------------------------------------------------------------------*/

/**************************************************************************//*!
@brief			Measurement module software initialisation 

@param[in,out]  *ptr

@return		# true - when Module initialisation ended successfully
            # false - when Module initialisation is ongoing, or error occurred

@details	Clear variables needed for both calibration as well as run time
			measurement. 

@note		It is not inteded to be executed when application is in run mode.

@warning
******************************************************************************/
tBool Meas_Clear(measModule_t *ptr)
{
    ptr->measured.f16PhA.filt   = 0x0;
    ptr->measured.f16PhA.raw    = 0x0;
    ptr->measured.f16PhB.filt   = 0x0;
    ptr->measured.f16PhB.raw    = 0x0;
    ptr->measured.f16PhC.filt   = 0x0;
    ptr->measured.f16PhC.raw    = 0x0;
    ptr->measured.f16Udcb.filt  = 0x0;
    ptr->measured.f16Udcb.raw   = 0x0;
    ptr->measured.f16Temp.filt  = 0x0;
    ptr->measured.f16Temp.raw   = 0x0;
    ptr->measured.f16Idcb.filt  = 0x0;
    ptr->measured.f16Idcb.raw   = 0x0;

    ptr->offset.f16PhA.f16Offset = 0;    
	ptr->offset.f16PhB.f16Offset = 0;
	ptr->offset.f16PhC.f16Offset = 0;
    
    ptr->flag.R             	= 0x0;

    ptr->param.u16CalibSamples  = 0;
    ptr->flag.B.calibInitDone   = 0;
    ptr->flag.B.calibDone       = 0;

    return 1;
}

/**************************************************************************//*!
@brief      3Phase current measurement software callibration routine.

@param[in,out]  *ptr    	Pointer to structure of measurement module variables and
                        	parameters
@param[in]      svmSector	Space Vector Modulation Secotr

@return     # true - when Calibration ended successfully
            # false - when Calibration is ongoing

@details    This function performs offset callibration for 3 phase current measurement
			during the calibration phase of the application. It is not intended to be 
			executed when application is in run mode.

@warning
******************************************************************************/
tBool Meas_CalibCurrentSense(measModule_t *ptr, tU16 svmSector)
{
    if (!(ptr->flag.B.calibInitDone))
    {
        ptr->calibCntr = 1<< (ptr->param.u16CalibSamples + 4); // +4 in order to accommodate settling time of the filter

        ptr->measured.f16PhA.filt   = 0x0;
        ptr->measured.f16PhB.filt   = 0x0;
        ptr->measured.f16PhC.filt   = 0x0;

        ptr->offset.f16PhA.filtParam.f32Acc		= 0x0;
        ptr->offset.f16PhB.filtParam.f32Acc		= 0x0;
        ptr->offset.f16PhC.filtParam.f32Acc		= 0x0;

        ptr->flag.B.calibDone       = 0;
        ptr->flag.B.calibInitDone   = 1;
    }

    if (!(ptr->flag.B.calibDone))
    {
        /* --------------------------------------------------------------
         * Phase A - DC offset data filtering using MA recursive filter
         * ------------------------------------------------------------ */
    	ptr->offset.f16PhA.f16Offset  = GDFLIB_FilterMA_F16(ptr->measured.f16PhA.raw, &ptr->offset.f16PhA.filtParam);

        /* --------------------------------------------------------------
         * Phase B - DC offset data filtering using MA recursive filter
         * ------------------------------------------------------------ */
    	ptr->offset.f16PhB.f16Offset  = GDFLIB_FilterMA_F16(ptr->measured.f16PhB.raw, &ptr->offset.f16PhB.filtParam);

        /* --------------------------------------------------------------
         * Phase C - DC offset data filtering using MA recursive filter
         * ------------------------------------------------------------ */
    	ptr->offset.f16PhC.f16Offset  = GDFLIB_FilterMA_F16(ptr->measured.f16PhC.raw, &ptr->offset.f16PhC.filtParam);

        if ((--ptr->calibCntr)<=0)
        {
        	ptr->flag.B.calibDone       = 1;    // end of DC offset calibration
        }

    }
    return (ptr->flag.B.calibDone);
}


/**************************************************************************//*!
@brief      3-phase current measurement reading.

@param[in,out]  *ptr    Pointer to structure of module variables and
                        parameters

@return     # true - when measurement ended successfully
            # false - when measurement is ongoing, or error occurred.

@details    This function performs measurement of three phase currents from
            two shunt resistors. Because a non-zero length PWM pulse width
            is required for successful current sample, this approach can not
            be utilized up to full PWM dutycycle.

@note

@warning
******************************************************************************/
tBool Meas_Get3PhCurrent(measModule_t *ptr, SWLIBS_3Syst_F16 *i,  tU16 svmSector)
{
	ptr->measured.f16PhA.raw    = ADC0ResultList[0]>>1;
	ptr->measured.f16PhB.raw    = ADC1ResultList[0]>>1;

    ptr->measured.f16PhA.raw = (FRAC16(0.5) - ptr->measured.f16PhA.raw)<<1;
    ptr->measured.f16PhB.raw = (FRAC16(0.5) - ptr->measured.f16PhB.raw)<<1;


	i->f16Arg1 = ptr->measured.f16PhA.raw - ptr->offset.f16PhA.f16Offset;
	i->f16Arg2 = ptr->measured.f16PhB.raw - ptr->offset.f16PhB.f16Offset;
	i->f16Arg3 = MLIB_AddSat(-1 * i->f16Arg1, -1 * i->f16Arg2);


    return(1);
}

/**************************************************************************//*!
@brief      DCB Voltage measurement routine.

@param[in,out]  *ptr    Pointer to structure of module variables and
                        parameters

@return     # true - when measurement ended successfully
            # false - when measurement is ongoing, or error occurred.

@details    This function performs measurement of DCBus Voltage.
            
@note

@warning
******************************************************************************/
tBool Meas_GetUdcVoltage(measModule_t *ptr, GDFLIB_FILTER_MA_T *uDcbFilter)
{
	ptr->measured.f16Udcb.raw   = ADC0ResultList[1]>>1;
	ptr->measured.f16Udcb.filt  = GDFLIB_FilterMA(ptr->measured.f16Udcb.raw, uDcbFilter);

    return(1);
}



/**************************************************************************//*!
@brief      Temperature measurement routine.

@param[in,out]  *ptr    Pointer to structure of module variables and
                        parameters

@return     # true - when measurement ended successfully
            # false - when measurement is ongoing, or error occurred.

@details    This function performs measurement of System temperature.
            
@note

@warning
******************************************************************************/
tBool Meas_GetTemperature(measModule_t *ptr)
{
	ptr->measured.f16Temp.raw = (tFrac16)(ADC1ResultList[1]>>1);
	ptr->measured.f16Temp.filt = MLIB_Mul_F16(ptr->measured.f16Temp.raw,FRAC16(0.73801));
	ptr->measured.f16Temp.filt = MLIB_Sub_F16(ptr->measured.f16Temp.filt, FRAC16(0.23801));
	ptr->measured.f16Temp.filt = ptr->measured.f16Temp.filt << 1;
    return(1);
}


/* End of file */
