#!/usr/bin/env python3

'''

Copyright 2019 NXP.

This software is owned or controlled by NXP and may only be used strictly in accordance with the
license terms that accompany it. By expressly accepting such terms or by downloading, installing,
activating and/or otherwise using the software, you are agreeing that you have read, and that you
agree to comply with and are bound by, such license terms. If you do not agree to be bound by the
applicable license terms, then you may not retain, install, activate or otherwise use the software.

'''

import base64
import subprocess
import sys
import os
import math

if len(sys.argv) < 5:
    print("Usage:")
    print("       generate_signing_artifacts.py ca_name country code country_name state organization")
    print("")
    print("       ca_name: Name of CA for image signature chain of trust")
    print("")
    print("       country code: GB/US")
    print("")
    print("       country_name: CA Country Name")
    print("")
    print("       state: CA Country State")
    print("")
    print("       organization: CA Company Organization")
    print("")
    sys.exit(1)
else:

    CA_NAME = sys.argv[1]
    COUNTRY_CODE = sys.argv[2]
    COUNTRY_NAME = sys.argv[3]
    STATE_NAME = sys.argv[4]
    ORG_NAME = sys.argv[5]

    # Get all the credentials for signing/verification entity
    CA_ROOT_FOLDER = os.path.dirname(os.path.realpath(__file__)) + '/ca'

    # Creating main directory
    print("Creating directories...")
    cmd1 = ['mkdir', 'ca']
    out = subprocess.run(cmd1, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode != 0:
        print("ERROR: Failed to prepare directories " + str(out.returncode))
        sys.exit(1)

    os.chdir('ca')

    # Prepare the directories for the CA/key/cert generation
    print("Creating directories...")
    cmd1 = ['mkdir', 'certs', 'crl', 'newcerts', 'private', 'csr']
    print(str(cmd1))
    out = subprocess.run(cmd1, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Successfully prepared the directories")
    else:
        print("ERROR: Failed to prepare directories " + str(out.returncode))
        sys.exit(1)

    print("chmod directories...")
    cmd2 = ['chmod', '700', 'private']
    print(str(cmd2))
    out = subprocess.run(cmd2, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Successfully prepared the directories")
    else:
        print("ERROR: Failed to prepare directories " + str(out.returncode))
        sys.exit(1)

    print("creating index file...")
    cmd3 = ['touch', 'index.txt', 'serial', 'crlnumber', 'index.txt.attr']
    print(str(cmd3))
    out = subprocess.run(cmd3, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Successfully prepared the directories")
    else:
        print("ERROR: Failed to prepare directories " + str(out.returncode))
        sys.exit(1)

    print("Creating Serial File...")
    with open('./serial', 'w+') as serial_file:
        serial_file.write('1000')
        serial_file.close()

    print("Modifying contents for local path...")    
    
    # Modifying openssl.cnf
    openssl_contents = []
    with open('../openssl.cnf.tmp', 'r') as openssl:
        openssl_contents = openssl.readlines()
        openssl.close()
    do_write = True
    with open('./openssl.cnf', 'w+') as openssl:
        for line in openssl_contents:
            if 'dir               = /root/ca' in line:
                do_write = False
                openssl.write('dir               = ' + CA_ROOT_FOLDER + '\n')
            if 'private_key       = $dir/private/ca.key.pem' in line:
                do_write = False
                openssl.write('private_key       = $dir/private/'+ CA_NAME + '.root.ca.key.pem\n')

            if 'certificate       = $dir/certs/ca.cert.pem' in line: 
                do_write = False
                openssl.write('certificate       = $dir/certs/' + CA_NAME + '.root.ca.crt.pem\n')
            if not do_write:
                do_write = True
            else:
                openssl.write(line)
        openssl.close()
    print("SUCCESS: openssl.cnf copied.")

    print("Creating Root Key...")    
    cmd6 = ['openssl', 'genrsa', '-aes256', '-out', 'private/' + CA_NAME + '.root.ca.key.pem', '4096']
    out = subprocess.run(cmd6, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Created Root Key")
    else:
        print("ERROR: Failed to Create Root Key")
        print(str(out.args))
        print(str(out.stdout.decode('utf-8')))
        sys.exit(1)

    print("Changing Root Key Permissions...")   
    cmd7 = ['chmod', '400', 'private/' + CA_NAME + '.root.ca.key.pem']
    out = subprocess.run(cmd7, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Changed Root Key Permissions")
    else:
        print("ERROR: Failed to change Root Key Permissions")
        print(str(out.args))
        print(str(out.stdout.decode('utf-8')))
        sys.exit(1)
    
    print("Creating Root Certificate...")   
    cmd8 = ['openssl', 'req', '-config', 'openssl.cnf', '-key', 'private/' + CA_NAME + '.root.ca.key.pem', '-new', '-x509', '-days', '7300', '-sha256' , '-extensions' , 'v3_ca' , '-out' , 'certs/' + CA_NAME + '.root.ca.crt.pem', '-subj', '/C=' + COUNTRY_CODE + '/ST=' + STATE_NAME + '/L=' + COUNTRY_NAME + '/O=' + ORG_NAME + '/CN=' + CA_NAME + ' CA Root' ]
    out = subprocess.run(cmd8, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Created Root Certificate")
    else:
        print("ERROR: Failed to Create Root Certificate")
        print(str(out.args))
        print(str(out.stdout.decode('utf-8')))
        sys.exit(1)

    print("Changing certificate permissions...")   
    cmd9 = ['chmod', '444', 'certs/' + CA_NAME + '.root.ca.crt.pem']
    out = subprocess.run(cmd9, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Changed certificate permissions")
    else:
        print("ERROR: Failed to change certificate permissions")
        print(str(out.args))
        print(str(out.stdout.decode('utf-8')))
        sys.exit(1)

    print("Creating Private Key...")   
    cmd11 = ['openssl', 'genrsa', '-aes256', '-out', 'private/' + CA_NAME + '.app.a.key.pem', '2048']
    out = subprocess.run(cmd11, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Created private key")
    else:
        print("ERROR: Failed to create private key")
        print(str(out.args))
        print(str(out.stdout.decode('utf-8')))
        sys.exit(1)


    print("Changing Key Permissions..")   
    cmd12 = ['chmod', '400', 'private/' + CA_NAME + '.app.a.key.pem']
    out = subprocess.run(cmd12, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Changing Key Permissions")
    else:
        print("ERROR: Failed to change key permissions")
        print(str(out.args))
        print(str(out.stdout))
        sys.exit(1)

    print("Creating Certificate..")   
    cmd13 = ['openssl', 'req', '-config', 'openssl.cnf', '-key', 'private/' + CA_NAME + '.app.a.key.pem', '-new', '-sha256', '-out', 'csr/' + CA_NAME + '.app.a.csr.pem', '-subj', '/C=' + COUNTRY_CODE + '/ST=' + STATE_NAME + '/L=' + COUNTRY_NAME + '/O=' + ORG_NAME + '/CN=' + CA_NAME + ' Application A' ]
    out = subprocess.run(cmd13, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Creating Certificate")
    else:
        print("ERROR: Failed to Create Certificate")
        print(str(out.args))
        print(str(out.stdout))
        sys.exit(1)

    print("Sign the CSR..")   
    cmd14 = ['openssl', 'ca', '-batch', '-config', 'openssl.cnf', '-extensions', 'server_cert', '-days', '375', '-notext', '-md', 'sha256', '-in', 'csr/' + CA_NAME + '.app.a.csr.pem', '-out', 'certs/' + CA_NAME + '.app.a.crt.pem']
    out = subprocess.run(cmd14, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Signed the CSR ")
    else:
        print("ERROR: Failed to Sign the CSR")
        print(str(out.args))
        print(str(out.stdout))
        sys.exit(1)
    
    print("Modifying certificate permissions...")   
    cmd15 = ['chmod', '444', 'certs/' + CA_NAME + '.app.a.crt.pem']
    out = subprocess.run(cmd15, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Modifed the certificate permissions")
    else:
        print("ERROR: Failed to modify certificate permissions")
        print(str(out.args))
        print(str(out.stdout))
        sys.exit(1)

    print("Creating Private Key...")   
    cmd11 = ['openssl', 'genrsa', '-aes256', '-out', 'private/' + CA_NAME + '.app.b.key.pem', '2048']
    out = subprocess.run(cmd11, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Created private key")
    else:
        print("ERROR: Failed to create private key")
        print(str(out.args))
        print(str(out.stdout.decode('utf-8')))
        sys.exit(1)


    print("Changing Key Permissions..")   
    cmd12 = ['chmod', '400', 'private/' + CA_NAME + '.app.b.key.pem']
    out = subprocess.run(cmd12, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Changing Key Permissions")
    else:
        print("ERROR: Failed to change key permissions")
        print(str(out.args))
        print(str(out.stdout))
        sys.exit(1)

    print("Creating Certificate..")   
    cmd13 = ['openssl', 'req', '-config', 'openssl.cnf', '-key', 'private/' + CA_NAME + '.app.b.key.pem', '-new', '-sha256', '-out', 'csr/' + CA_NAME + '.app.b.csr.pem', '-subj', '/C=' + COUNTRY_CODE + '/ST=' + STATE_NAME + '/L=' + COUNTRY_NAME + '/O=' + ORG_NAME + '/CN=' + CA_NAME + ' Application B' ]
    out = subprocess.run(cmd13, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Creating Certificate")
    else:
        print("ERROR: Failed to Create Certificate")
        print(str(out.args))
        print(str(out.stdout))
        sys.exit(1)

    print("Sign the CSR..")   
    cmd14 = ['openssl', 'ca', '-batch', '-config', 'openssl.cnf', '-extensions', 'server_cert', '-days', '375', '-notext', '-md', 'sha256', '-in', 'csr/' + CA_NAME + '.app.b.csr.pem', '-out', 'certs/' + CA_NAME + '.app.b.crt.pem']
    out = subprocess.run(cmd14, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Signed the CSR ")
    else:
        print("ERROR: Failed to Sign the CSR")
        print(str(out.args))
        print(str(out.stdout))
        sys.exit(1)
    
    print("Modifying certificate permissions...")   
    cmd15 = ['chmod', '444', 'certs/' + CA_NAME + '.app.b.crt.pem']
    out = subprocess.run(cmd15, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if out.returncode == 0:
        print("SUCCESS: Modifed the certificate permissions")
    else:
        print("ERROR: Failed to modify certificate permissions")
        print(str(out.args))
        print(str(out.stdout))
        sys.exit(1)

    sys.exit(0)
