#!/usr/bin/env python3

"""

Copyright 2019 NXP.

This software is owned or controlled by NXP and may only be used strictly in accordance with the
license terms that accompany it. By expressly accepting such terms or by downloading, installing,
activating and/or otherwise using the software, you are agreeing that you have read, and that you
agree to comply with and are bound by, such license terms. If you do not agree to be bound by the
applicable license terms, then you may not retain, install, activate or otherwise use the software.

File
++++
/Scripts/sln_viznas_iot_secure_boot/manf/prog_sec_app.py

Brief
+++++
** Programs secured app on NXP VIZN Module. **

.. versionadded:: 0.0

"""

import sys
import os
import time
import argparse
import logging
import subprocess
from shutil import copyfile

import Ivaldi.blhost as blhost
import Ivaldi.sdphost as sdphost
import Ivaldi.helpers as helpers
import Ivaldi.onboard.aws as ob

#logging.basicConfig(level=logging.DEBUG)

TOP_DIR = "../../.."
IMG_DIR = TOP_DIR + '/Image_Binaries/'
LD_PATH = TOP_DIR + '/Image_Binaries/ivt_flashloader_signed.bin'
SB_CRYPT_PATH = IMG_DIR + '/boot_crypt_image.sb'
SB_SIGN_PATH = IMG_DIR + '/boot_sign_image.sb'
FLDR_BIN_PATH = TOP_DIR + '/Image_Binaries/ivt_flashloader_signed.bin'
OUTPUT_DIR = TOP_DIR + '/Output'
PY3_DIR = TOP_DIR + '/Scripts'

ser_num = ''
ca_name = 'prod'

parser = argparse.ArgumentParser()
parser.add_argument("-s", "--signed-only", action="store_true", help="Run signed rather than encrypted app")
parser.add_argument("-c", "--ca-name", type=str, nargs='?', default='', help="Name of CA used to sign images.")
args = parser.parse_args()

app_sb_path = SB_CRYPT_PATH if not args.signed_only else SB_SIGN_PATH

if args.ca_name:
    ca_name = args.ca_name

if not os.access(app_sb_path, os.R_OK):
    logging.error("Unable to read {} boot app file: {}"
                  .format("signed" if args.signed_only else "encrypted", app_sb_path))
    sys.exit(1)

sdp = sdphost.SDPHost('0x1fc9', '0x135')

# Check communication with device
print('Establishing connection...')
if sdp.error_status()['ret']:
    print('ERROR: Could not establish communication with device. Power cycle device.')
    sys.exit(1)
else:
    print('SUCCESS: Communication established with device.')

# Load flashloader onto device
print('Loading flashloader...')
if sdp.write_file('0x20000000', FLDR_BIN_PATH)['ret']:
    print('ERROR: Could not write file!')
    sys.exit(1)
else:
    print('SUCCESS: Flashloader loaded successfully.')

# Jump to flash loader entry point
print('Jumping to flashloader entry point...')
if sdp.jump_to_address('0x20000400')['ret']:
    print('ERROR: Could not jump to address!')
    sys.exit(1)
else:
    print('SUCCESS: Device jumped to execute flashloader.')

bl = blhost.BLHost()

# Poll device to make sure it is ready
print('Waiting for device to be ready for blhost...')
waitCount = 0
while bl.get_property('0x01')['ret']:
    time.sleep(0.5)
    waitCount += 1
    if waitCount == 10:
        print('ERROR: Timeout waiting for device to be ready. Power cycle device and try again.')
        sys.exit(1)

print('SUCCESS: Device is ready for blhost!')

# Read out unique ID
print('Reading device unique ID...')
prop = bl.get_property('0x12')
if prop['ret']:
    print('ERROR: Could not read device unique ID!')
    sys.exit(1)
else:
    ser_num = helpers.encode_unique_id(prop['response'])
    print('SUCCESS: Device serial number is %s' %(ser_num))

# Write config option block to RAM
print('Writing memory config option block...')
if bl.fill_memory('0x2000', '0x4', '0xc0333006')['ret']:
    print('ERROR: Could not fill memory!')
    sys.exit(1)
else:
    print('SUCCESS: Config option block loaded into RAM.')

# Configure FlexSPI
print('Configuring FlexSPI...')
if bl.configure_memory('0x9', '0x2000')['ret']:
    print('ERROR: Could not configure memory!')
    sys.exit(1)
else:
    print('SUCCESS: FlexSPI configured.')

# Erase flash
print('Erasing flash...')
if bl.flash_erase_region('0x60000000', '0x2000000')['ret']:
    print('ERROR: Could not erase memory!')
    sys.exit(1)
else:
    print('SUCCESS: Flash erased.')

# Program root cert 
print('Programming flash with root cert...')
root_ca_path = PY3_DIR + '/ota_signing/ca/certs/' + ca_name + '.root.ca.crt.pem'
root_ca_bin = IMG_DIR + ca_name + '.root.ca.bin'
helpers.file_format(root_ca_path, root_ca_bin)
if bl.write_memory('0x61CC0000', root_ca_bin)['ret']:
    print('ERROR: Could not program flash with certificates for this "thing"!')
    sys.exit(1)
else:
    print('SUCCESS: Programmed flash with certificates for this "thing".')

# Program app cert 
print('Programming flash with app cert application A...')
leaf_ca_path = PY3_DIR + '/ota_signing/ca/certs/' + ca_name + '.app.a.crt.pem'
leaf_ca_bin = IMG_DIR + ca_name + '.app.a.bin'
helpers.file_format(leaf_ca_path, leaf_ca_bin)
if bl.write_memory('0x61D00000', leaf_ca_bin)['ret']:
    print('ERROR: Could not program flash with certificates for this "thing"!')
    sys.exit(1)
else:
    print('SUCCESS: Programmed flash with certificates for this "thing".')

# Program bootloader cert
print('Programming flash with app cert for bootloader...')
if bl.write_memory('0x61D80000', leaf_ca_bin)['ret']:
    print('ERROR: Could not program flash with certificates for this "thing"!')
    sys.exit(1)
else:
    print('SUCCESS: Programmed flash with certificates for this "thing".')

# Sign bootloader and application and generate fica table
try:
    copyfile(IMG_DIR + '/sln_viznas_iot_elock_oobe.bin', PY3_DIR + '/ota_signing/sign/sln_viznas_iot_elock_oobe.bin')
except:
    print('ERROR: Unable to copy ais demo!')
    sys.exit(1)

try:
    copyfile(IMG_DIR + '/sln_viznas_iot_bootloader.bin', PY3_DIR + '/ota_signing/sign/sln_viznas_iot_bootloader.bin')
except:
    print('ERROR: Unable to copy bootloader!')
    sys.exit(1)

# NOTE: Script is written to program the bank A image and signing certificate only
# python3 sign_package.py -p sln_viznas_iot_ -m elock_oobe -a ca_name.app.a
cmd = ['python3', 'sign_package.py', '-p', 'sln_viznas_iot_', '-m', 'elock_oobe', '-a', ca_name + '.app.a']
out = subprocess.run(cmd, cwd=PY3_DIR + '/ota_signing/sign', stdout=subprocess.PIPE, stderr=subprocess.PIPE)
if out.returncode == 0:
    print('SUCCESS: sign_package succeeded.')
else:
    print("ERROR: sign_package failed!")
    print(str(out.stdout.strip(), 'utf-8', 'strict'))
    print(str(out.stderr.strip(), 'utf-8', 'strict'))
    sys.exit(1)

try:
    copyfile(PY3_DIR + '/ota_signing/sign/fica_table.bin', IMG_DIR + '/fica_table.bin')
except:
    print('ERROR: Unable to copy fica table!')
    sys.exit(1)

# Program FICA table
print('Programming FICA table...')
fica_path = IMG_DIR + '/fica_table.bin'
if bl.write_memory('0x61FC0000', fica_path)['ret']:
    print('ERROR: Could not program flash with certificates for this "thing"!')
    sys.exit(1)
else:
    print('SUCCESS: Programmed flash with certificates for this "thing".')

# Load secure app file
print('Programming flash with secure app file...')
if bl.receive_sb_file(app_sb_path)['ret']:
    print('ERROR: Could not program secure app file!')
    sys.exit(1)
else:
    print('SUCCESS: Programmed flash with secure app file.')
    print('\nUnpower module, move the boot jumper in BOOT_MODE_1, and restore power\n')
