#!/usr/bin/env python3

"""

Copyright 2019 NXP.

This software is owned or controlled by NXP and may only be used strictly in accordance with the
license terms that accompany it. By expressly accepting such terms or by downloading, installing,
activating and/or otherwise using the software, you are agreeing that you have read, and that you
agree to comply with and are bound by, such license terms. If you do not agree to be bound by the
applicable license terms, then you may not retain, install, activate or otherwise use the software.

File
++++
/Scripts/sln_viznas_iot_secure_boot/oem/secure_app.py

Brief
+++++
** Secure application for execution with High Assurance Boot (HAB). **

.. versionadded:: 0.0

"""

import sys
import os
import logging
import argparse

import Ivaldi.elftosb as elftosb
import Ivaldi.helpers as hlpr

# logging.basicConfig(level=logging.DEBUG)

TOP_DIR = "../../.."
IMG_DIR = TOP_DIR + "/Image_Binaries/"
BD_SIGN_IMAGE = TOP_DIR + "/Tools/bd_file/imx-flexspinor-normal-signed-dcd.bd"
BD_GEN_SIGN_SB = TOP_DIR + "/Tools/bd_file/program_flexspinor_image_hyperflash.bd"
BD_GEN_CRYPT_SB = TOP_DIR + \
    "/Tools/bd_file/program_flexspinor_image_hyperflash_encrypt.bd"

DEF_HAB_PATH = IMG_DIR + "/sln_viznas_iot_bootstrap.srec"
DEF_BOOT_PATH = IMG_DIR + "/sln_viznas_iot_bootloader.bin"
DEF_APP_PATH = IMG_DIR + "/sln_viznas_iot_elock_oobe.bin"

parser = argparse.ArgumentParser()
parser.add_argument("-s", "--signed-only", action="store_true",
                    help="Only sign rather than encrypt app")
parser.add_argument("--bin", action="store_true",
                    help="App hab file in 'bin' format rather than 'srec'")
parser.add_argument("hab_path", nargs='?',
                    default=DEF_HAB_PATH, help="path to Srec file")
parser.add_argument("bootloader",  nargs='?',
                    default=DEF_BOOT_PATH, help="path to bin file")
parser.add_argument("main_app",  nargs='?',
                    default=DEF_APP_PATH, help="path to bin file")
args = parser.parse_args()

hab_path = args.hab_path
bootloader_path = args.bootloader
main_app_path = args.main_app

if args.signed_only:
    sec_boot_file = "boot_sign_image.sb"
    bd_gen_sb = BD_GEN_SIGN_SB
else:
    sec_boot_file = "boot_crypt_image.sb"
    bd_gen_sb = BD_GEN_CRYPT_SB

sec_boot_path = IMG_DIR + '/' + sec_boot_file

if not os.access(hab_path, os.R_OK):
    logging.error("Unable to read hab_path: %s", hab_path)
    sys.exit(1)

# Need trailing '/' to handle paths of form
# "../../../../<file_name>" as dirname removes
# trailing '/'
hab_dir = os.path.dirname(hab_path) + '/'
hab_file = os.path.basename(hab_path)
hab_name = os.path.splitext(hab_file)[0]

hab_image_path = IMG_DIR + '/' + hab_name + "_signed.bin"
hab_image_nopad_path = IMG_DIR + '/' + hab_name + "_signed_nopadding.bin"


if not os.access(IMG_DIR, os.W_OK):
    logging.error("Unable to write secure boot app directory: %s", IMG_DIR)
    sys.exit(1)

if os.path.exists(sec_boot_path) and not os.access(sec_boot_path, os.W_OK):
    logging.error("Unable to write secure boot app file: %s", sec_boot_path)
    sys.exit(1)


e2sb = elftosb.Elftosb()

# Create signed image
print('{} app image ...'.format(
    "Cryptographicly signing" if args.signed_only else "Encrypting"))
if e2sb.sign_srec(BD_SIGN_IMAGE, hab_image_path, hab_dir, hab_file
                  )['ret']:

    print('ERROR: Unable to create {} image.'.format(
        "signed" if args.signed_only else "encrypted"))
    sys.exit(1)

elif os.stat(hab_image_path).st_size == 0 or os.stat(hab_image_nopad_path).st_size == 0:
    # cst, a subprocess of elftosb, may fail without proper error propagation to elftosb
    print('ERROR: Zero size {} image.'.format(
        "signed" if args.signed_only else "encrypted"))
    sys.exit(1)
else:
    print('SUCCESS: Created {} image.'.format(
        "signed" if args.signed_only else "encrypted"))

# Create secure boot file
print('Creating {} app file ...'.format(
    "signed" if args.signed_only else "encrypted"))

if e2sb.create_sb(bd_gen_sb, sec_boot_path, IMG_DIR, [hab_image_nopad_path, bootloader_path, "NoToolbox", main_app_path]
                  )['ret']:

    print('ERROR: Unable to create {} boot app file.'.format(
        "signed" if args.signed_only else "encrypted"))
    sys.exit(1)
else:
    print('SUCCESS: Created {} app file.'.format(
        "signed" if args.signed_only else "encrypted"))
