/* main.c - FCCU Simulation for MPC5744P */
/* Description:  Measures eTimer pulse/period measurement */
/* Rev 1.0 Oct 24 2017 D Chung - production version */
/* Copyright NXP Semiconductor, Inc 2016 All rights reserved. */

/*******************************************************************************
* NXP Semiconductor Inc.
* (c) Copyright 2016 NXP Semiconductor, Inc.
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by NXP in this matter are performed AS IS and without
any warranty. CUSTOMER retains the final decision relative to the total design
and functionality of the end product. NXP neither guarantees nor will be
held liable by CUSTOMER for the success of this project.
NXP DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING,
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE ORE ADVISE SUPPLIED TO THE PROJECT
BY NXP, AND OR NAY PRODUCT RESULTING FROM NXP SERVICES. IN NO EVENT
SHALL NXP BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF
THIS AGREEMENT.

CUSTOMER agrees to hold NXP harmless against any and all claims demands or
actions by anyone on account of any damage, or injury, whether commercial,
contractual, or tortuous, rising directly or indirectly as a result of the advise
or assistance supplied CUSTOMER in connection with product, services or goods
supplied under this Agreement.
********************************************************************************
* File              main.c
* Owner             David Chung
* Version           1.0
* Date              Oct-24-2017
* Classification    General Business Information
* Brief            	This example simulates faults with the FCCU
*
********************************************************************************
* Detailed Description:
* This code example simulates faults with the FCCU. Press SW1 to simulate the fault.
* When FAULT status is detected, the MCU system moves into SAFE mode and flashes the
* red LED.  Press SW2 to clear the fault. Red LED will turn back off.
*
* ------------------------------------------------------------------------------
* Test HW:         Miriac(TM)-EK5744
* MCU:             MPC5744P
* Terminal:        19200, 8N1, None
* Fsys:            160 MHz PLL on 40 MHz XOSC
* Debugger:        USB Multilink
* Target:          FLASH
* EVB connection:  USB to PC
*
********************************************************************************
Revision History:
Version  Date         Author  			Description of Changes
1.0      Oct-24-2017  David Chung	  	Initial version

*******************************************************************************/

#include "derivative.h" /* include peripheral declarations */
#include "linflexd_uart.h"
#include <string.h>

#define COUNTERMAX 50000

extern void xcptn_xmpl(void);

void CLK_Init(void);

void SIUL_Init(void);

void FCCU_Init(void);

void Alarm_ISR(void);

void LinFlexD_1_Rx_ISR(void);

void MTC_ISR(void);

void MC_ME_Enable_Interrupts(void);

uint32_t mode_old = 0;

void MC_ME_Enable_Interrupts(){
	/* Clear the current flag */
	MC_ME.IS.R |= 0x00000001;

	/* Enable MC_ME interrupt in INTC */
	INTC_0.PSR[252].R = 0x800F;	//MTC Interrupt
}

void MTC_ISR(){
	/* Clear the flag */
	MC_ME.IS.R |= 0x00000001;

	/* See which mode it is in. If in RUN0 mode,
	 * Baud clock is 80 MHz. If in SAFE mode, baud
	 * clock is 8 MHz. Reinitialize LINFlexD_1
	 */
	if(MC_ME.GS.B.S_CURRENT_MODE == 4){
		uint8_t message[] = {"\n\rMiriac(TM)-EK5744 is now in RUN0 mode with system clock at 160 MHz. DOUT0 LED is not flashing. Press any key to simulate a fault again.\n\r"};
		initLINFlexD_1(80,19200);
		TransmitData((const char*)message,strlen((const char*)message));
	}else if(MC_ME.GS.B.S_CURRENT_MODE == 2){
		uint8_t message[] = {"\n\rMiriac(TM)-EK5744 is now in SAFE mode with system clock at 16 MHz. DOUT0 LED is flashing. Press any key to clear the fault and return to normal state.\n\r"};
		initLINFlexD_1(8,19200);
		TransmitData((const char*)message,strlen((const char*)message));
	}
}

void LinFlexD_1_Rx_ISR(){
	uint8_t data = 0; //Dummy fetch

	ReceiveData((char*)&data);

	/* If current mode is RUN0, transition to SAFE mode
	 * and vice-versa.
	 */
	if(MC_ME.GS.B.S_CURRENT_MODE == 4){
		uint8_t message[] = {"\n\rCharacter received. Program will fake a fault to simulate FCCU reaction.\n\r"};
		TransmitData((const char*)message,strlen((const char*)message));

		/* Implement fake fault */
		FCCU.NCFF.B.FNCFC = 7; //"Fake" trigger NCF[7]
	}else if(MC_ME.GS.B.S_CURRENT_MODE == 2){
		uint8_t message[] = {"\n\rCharacter received. Program will clear the fault and reset to normal state.\n\r"};
		TransmitData((const char*)message,strlen((const char*)message));

		uint32_t tempval = 0;

		/* Clear the fault NCF[7] */
		FCCU.NCFK.R =  0xAB3498FE;
		FCCU.NCF_S[0].R = 0xFFFFFFFF;
		while (FCCU.CTRL.B.OPS != 0x3);
		FCCU.CTRL.B.OPR = 0xA; //Read the NCF_S register to verify the fault has been cleared
		while (FCCU.CTRL.B.OPS != 0x3);


		FCCU.NCFK.R =  0xAB3498FE;
		FCCU.NCF_S[1].R = 0xFFFFFFFF;
		while (FCCU.CTRL.B.OPS != 0x3);
		FCCU.CTRL.B.OPR = 0xA; //Read the NCF_S register to verify the fault has been cleared
		while (FCCU.CTRL.B.OPS != 0x3);

		FCCU.NCFK.R =  0xAB3498FE;
		FCCU.NCF_S[2].R = 0xFFFFFFFF;
		while (FCCU.CTRL.B.OPS != 0x3);
		FCCU.CTRL.B.OPR = 0xA; //Read the NCF_S register to verify the fault has been cleared
		while (FCCU.CTRL.B.OPS != 0x3);

		/* Read the NCF_S register to verify the fault has been cleared */
		FCCU.CTRL.B.OPR = 10; //Set operation to OP10 (read FCCU_NCF_S)
		while(FCCU.CTRL.B.OPS != 3); //Wait until operation is successful
		tempval = (FCCU.NCF_S[0].R & 0x00000080); //Read value of NCF_S register and mask NCF[34] bit

		/* If NCF[7] remains 1, then recall function recursively to try
		 * to clear the flag again
		 */
		if(tempval){
			MTC_ISR();
		}

		/* Once flags cleared, read the FCCU state */
		FCCU.CTRL.B.OPR = 3;
		while(FCCU.CTRL.B.OPS != 3);
		tempval = FCCU.STAT.R;

		/* If NORMAL state indicated, switch back to
		 * old mode, IF current mode is safemode
		 */
		if(((tempval & 0x7) == 0) && (MC_ME.GS.B.S_CURRENT_MODE == 2)){
			/* RUN0 -> SAFE allowed; SAFE -> RUN0 not allowed.
			 * Must do SAFE -> DRUN -> RUN0
			 */
			MC_ME.MCTL.R = 0x30005AF0; //Transition to DRUN and key
			MC_ME.MCTL.R = 0x3000A50F; //DRUN and inverted key
			while(MC_ME.GS.B.S_MTRANS); //Wait while mode is in transition

			MC_ME.MCTL.R = (mode_old << 28) | 0x5AF0; //old mode and key
			MC_ME.MCTL.R = (mode_old << 28) | 0xA50F; //old mode and inverted key
			while(MC_ME.GS.B.S_MTRANS); //Wait while mode is in transition
		}

		/* Turn off DOUT0 LED */
		SIUL2.GPDO[PD10].R = 1;
	}

}

void CLK_Init(){
	MC_CGM.AC3_SC.R = 0x01000000; //Select XOSC as source of PLL

	/* PLL0_PHI = XOSC_freq * MFD/(prediv * rfdphi) */
	PLLDIG.PLL0DV.R = 0x00021008; //Multiply 40 MHz by 8 and divide by 2

	/* Turn on PLL0 and XOSC and switch system clock to PLL0_PHI */
	MC_ME.DRUN_MC.R = 0x00130072;

	/* Switch takes effect after mode transition.
	 * Mode can switch to itself
	 * Transition from DRUN to DRUN.
	 */
	MC_ME.MCTL.R = 0x30005AF0; //DRUN and key
	MC_ME.MCTL.R = 0x3000A50F; //DRUN and inverted key

	/* Wait until mode transition has completed */
	while(MC_ME.GS.R & 0x08000000);

	/* eTimer runs on PBRIDGEx_CLK, max 50MHz
	 * and MOTC_CLK, max 160MHz
	 * 160MHz/4 = 40MHz
	 */
	MC_CGM.SC_DC0.R = 0x80030000; //Enable system clock divide Divide value is 3 because formula is Actual Divider = divide value + 1

	/* FCCU does not require gating. Runs on PBRIDGE1_CLK and IRC.
	 * Configure SAFE mode settings for when FCCU enters FAULT state,
	 * when SW transitions the system to SAFE mode
	 */
	MC_ME.SAFE_MC.R = 0x00130010; //Disable safegating of pins in SAFE mode so normal pad operation

	/* Configure RUN0 to same as DRUN */
	MC_ME.RUN0_MC.R = 0x00130072;

	/* Enable LINFlexD_1 (to UART) */
	MC_ME.RUN_PC[1].R = 0x0000001C;	//Peripherals following this run profile are enabled in DRUN, RUN0, and SAFE mode
	MC_ME.PCTL91.B.RUN_CFG = 1; //Follow RUN_PC1, so LINflexD_1 is enabled in SAFE, DRUN, and RUN0 mode

	/* Clear flag just in case */
	MC_ME.IS.R |= 0x00000001;

	/* Enable mode transition interrupt */
	MC_ME.IM.R |= 0x00000001;

	/* Change mode from DRUN to RUN0
	 * to make change take effect.
	 */
	MC_ME.MCTL.R = 0x40005AF0; //RUN0 and key
	MC_ME.MCTL.R = 0x4000A50F; //RUN0 and inverted key

	/* Wait until mode transition has completed */
	while(MC_ME.GS.R & 0x08000000);

}

void SIUL_Init(){
	/* Enable DOUT0 LED */
	SIUL2.MSCR[PC10].B.OBE = 1;		//PWENA
	SIUL2.MSCR[PC13].B.OBE = 1;		//PWENB
	SIUL2.MSCR[PD10].B.OBE = 1;		//DOUT0

	SIUL2.GPDO[PC10].R = 0;
	SIUL2.GPDO[PC13].R = 0;
	SIUL2.GPDO[PD10].R = 0;			//Turn off LED
}



void FCCU_Init(){
	/* Clear all status bits so configuration does
	 * not cause an interrupt to instantly fire.
	 */
    FCCU.NCFK.R =  0xAB3498FE;
    FCCU.NCF_S[0].R = 0xFFFFFFFF;
    while (FCCU.CTRL.B.OPS != 0x3);

    FCCU.NCFK.R =  0xAB3498FE;
    FCCU.NCF_S[1].R = 0xFFFFFFFF;
    while (FCCU.CTRL.B.OPS != 0x3);

    FCCU.NCFK.R =  0xAB3498FE;
    FCCU.NCF_S[2].R = 0xFFFFFFFF;
    while (FCCU.CTRL.B.OPS != 0x3);

	/* Configure FCCU by entering CONFIG mode.
	 * NORMAL -> CONFIG. Do so by writing 0x913756AF
	 * to FCCU_CTRLK and then writing FCCU_CTRL[OPR] = 1
	 */
    FCCU.TRANS_LOCK.R = 0xBC;		//unlock configuration
	FCCU.CTRLK.R = 0x913756AF;
	FCCU.CTRL.R = 0x1;

	while(FCCU.CTRL.B.OPS != 0x3); //Wait until operation is successful

	/* By default, FCCU CONFIG mode timer gives user
	 * 4.096ms to configure FCCU. FCCU CONFIG timer is configurable
	 * in FCCU_CFG_TO[TO]. Options are integer multiples of 64us, up to 8.492ms
	 * Use (non-critical fault) NCF[7] which
	 * maps to STCU2 fault condition, "fake"-able.
	 */
	FCCU.NCF_CFG[0].R = 0x00000080; //Enable NCF[7] as a SW recoverable fault
	FCCU.NCFS_CFG[0].R = 0x00000000; //No reset reaction, short or long
	//FCCU.NCFS_CFG[2].R = 0x00000010; //Short functional reset for NCF[34]

	/* CONFIG state only available for a short time. Switch back to
	 * NORMAL state.  Then switch back to CONFIG for next step
	 * for CONFIG.
	 */
	FCCU.CTRLK.R = 0x825A132B; //Key for OP2
	FCCU.CTRL.R = 0x2;
	while(FCCU.CTRL.B.OPS != 0x3); //Wait until operation is successful

	/* Switch back to CONFIG */
    FCCU.TRANS_LOCK.R = 0xBC;		//unlock configuration
	FCCU.CTRLK.R = 0x913756AF;
	FCCU.CTRL.R = 0x1;

	while(FCCU.CTRL.B.OPS != 0x3); //Wait until operation is successful

	/* Enable NCF[7] */
	FCCU.NCF_E[0].R = 0x00000080;

	/* CONFIG state only available for a short time. Switch back to
	 * NORMAL state.  Then switch back to CONFIG for next step
	 * for CONFIG.
	 */
	FCCU.CTRLK.R = 0x825A132B; //Key for OP2
	FCCU.CTRL.R = 0x2;
	while(FCCU.CTRL.B.OPS != 0x3); //Wait until operation is successful

	/* Switch back to CONFIG */
    FCCU.TRANS_LOCK.R = 0xBC;		//unlock configuration
	FCCU.CTRLK.R = 0x913756AF;
	FCCU.CTRL.R = 0x1;

	while(FCCU.CTRL.B.OPS != 0x3); //Wait until operation is successful

	/* Enable NCF[7] timeout. When NCF[7] occurs
	 * FCCU will transition to ALARM mode and run
	 * the NCF timer.  When NCF timer runs out,
	 * FCCU will transition from ALARM to FAULT mode
	 * and system will go to SAFE mode
	 */
	FCCU.NCF_TOE[0].R = 0x00000080;
	//FCCU.NCF_TOE[0].R = 0;

	/* CONFIG state only available for a short time. Switch back to
	 * NORMAL state.  Then switch back to CONFIG for next step
	 * for CONFIG.
	 */
	FCCU.CTRLK.R = 0x825A132B; //Key for OP2
	FCCU.CTRL.R = 0x2;
	while(FCCU.CTRL.B.OPS != 0x3); //Wait until operation is successful

	/* Switch back to CONFIG */
    FCCU.TRANS_LOCK.R = 0xBC;		//unlock configuration
	FCCU.CTRLK.R = 0x913756AF;
	FCCU.CTRL.R = 0x1;

	while(FCCU.CTRL.B.OPS != 0x3); //Wait until operation is successful


	/* FCCU NCF timer runs on 16 MHz IRC
	 * Set 0x9896800 (1.6 x 10^8d) ticks, which
	 * makes for 10 seconds. When NCF[7] occurs,
	 * will remain in ALARM mode for maximum time allowed
	 * (4.096ms) before transition to FAULT mode
	 */
	FCCU.NCF_TO.R = 0x0000FFFF;

	/* CONFIG state only available for a short time. Switch back to
	 * NORMAL state.  Then switch back to CONFIG for next step
	 * for CONFIG.
	 */
	FCCU.CTRLK.R = 0x825A132B; //Key for OP2
	FCCU.CTRL.R = 0x2;
	while(FCCU.CTRL.B.OPS != 0x3); //Wait until operation is successful

	/* Switch back to CONFIG */
    FCCU.TRANS_LOCK.R = 0xBC;		//unlock configuration
	FCCU.CTRLK.R = 0x913756AF;
	FCCU.CTRL.R = 0x1;

	while(FCCU.CTRL.B.OPS != 0x3); //Wait until operation is successful

	/* Enable FCCU Alarm interrupt for NCF[7] */
	FCCU.IRQ_ALARM_EN[0].R = 0x00000080;

	/* Switch to NORMAL state */
	FCCU.CTRLK.R = 0x825A132B; //Key for OP2
	FCCU.CTRL.R = 0x2;
	while(FCCU.CTRL.B.OPS != 0x3); //Wait until operation is successful

	/* Configure ALARM interrupt in INTC */
	INTC_0.PSR[488].R = 0x800B;
}

void Alarm_ISR(){
	static uint32_t counter = 0; //Internal SW counter
	uint32_t tempval = 0;
	/* Read the STAT register to see
	 * current state of FCCU
	 */
	FCCU.CTRL.B.OPR = 3;
	while(FCCU.CTRL.B.OPS != 3);
	tempval = FCCU.STAT.R;

	/* If FAULT state indicated,
	 * switch system to SAFE mode by SW.
	 * MPC5744P is fault tolerant, meaning system
	 * will not automatically switch to SAFE mode
	 * when FCCU enters FAULT state.
	 * Only do this if not already in SAFE mode
	 */
	if(((tempval & 0x7) == 3) && (MC_ME.GS.B.S_CURRENT_MODE != 2)){
		/* Save the mode so we know where to reenter to
		 * when FCCU returns to NORMAL state
		 */
		mode_old = (MC_ME.GS.R & 0xF0000000) >> 28;
		MC_ME.MCTL.R = 0x20005AF0; //SAFE and key
		MC_ME.MCTL.R = 0x2000A50F; //SAFE and inverted key
		while(MC_ME.GS.B.S_MTRANS); //Wait while mode is in transition
	}

	/* When in SAFE mode, flash DOUT0 LED based on SW counter */
	if(MC_ME.GS.B.S_CURRENT_MODE == 2){
		counter++; //Increment counter
		if(counter > COUNTERMAX){
			SIUL2.GPDO[PD10].R ^= 1; //Toggle DOUT0 LED
			counter = 0; //Reset counter
		}
	}else{
		/* If not in SAFE mode but in this ISR,
		 * then FCCU was in ALARM state. Just turn on
		 * DOUT0 LED
		 */
		SIUL2.GPDO[PD10].R = 1; //Active high
	}
}

int main(void)
{
	uint8_t intro[] = {"\n\rWelcome to the Miriac(TM)-EK5744 FCCU code example!\n\r"};
	uint8_t message0[] = {"\n\rEK5744 is in RUN0 mode. Press any key to enter SAFE mode.\n\r"};
	xcptn_xmpl ();              /* Configure and Enable Interrupts */

	/* Initialize clock to 160MHz PLL */
	CLK_Init();

	/* Configure IO pins */
	SIUL_Init();

	/* Configure the FCCU */
	FCCU_Init();

	/* Initialize UART terminal */
	initLINFlexD_1(80,19200);
	TransmitData((const char*)intro,strlen((const char*)intro));
	TransmitData((const char*)message0,strlen((const char*)message0));

	/* Enable MC_ME Interrupts */
	MC_ME_Enable_Interrupts();

	/* Loop forever */
	for(;;){
		/* Make sure DOUT0 LED is off if
		 * the PC is here.
		 */
		SIUL2.GPDO[PD10].R = 0;
	}

	return 0;
}
