/**************************************************************************
* 
* Copyright 2005-2011 by Andrey Butok. Freescale Semiconductor, Inc.
*
**********************************************************************/ /*!
*
* @file fnet_timer.h
*
* @author Andrey Butok
*
* @date Feb-4-2011
*
* @version 0.1.10.0
*
* @brief FNET Timer API.
*
***************************************************************************/

#ifndef _FNET_TIMER_H

#define _FNET_TIMER_H

/*! @addtogroup fnet_timer
* An application can use the @ref fnet_timer_ticks() function to get a number of 
* ticks (periods, defined by the @ref FNET_TIMER_PERIOD_MS) since the
* hardware timer startup.
*/

/*! @{ */

/**************************************************************************/ /*!
 * @brief Timer period in milliseconds (period of one timer tick).
 ******************************************************************************/
#define FNET_TIMER_PERIOD_MS        (100) /* Do not change it.*/

/**************************************************************************/ /*!
 * @brief Number of timer ticks in one hour.
 ******************************************************************************/
#define FNET_TIMER_TICK_IN_HOUR     ((1000*60*60)/FNET_TIMER_PERIOD_MS)

/**************************************************************************/ /*!
 * @brief Number of timer ticks in one minute.
 ******************************************************************************/
#define FNET_TIMER_TICK_IN_MIN      ((1000*60)/FNET_TIMER_PERIOD_MS)

/**************************************************************************/ /*!
 * @brief Number of timer ticks in one second.
 ******************************************************************************/
#define FNET_TIMER_TICK_IN_SEC      (1000/FNET_TIMER_PERIOD_MS)

/***************************************************************************/ /*!
 *
 * @brief    Gets the timer counter value in ticks.
 *
 *
 * @return   This function returns a current value of the timer counter in ticks.
 *
 ******************************************************************************
 *
 * This function returns a current value of the timer counter that 
 * contains a number of periods from the moment of the hardware 
 * timer initialization (it's done in the FNET stack initialization).@n
 * The period of one timer tick is defined by the @ref FNET_TIMER_PERIOD_MS.
 *
 ******************************************************************************/
unsigned long fnet_timer_ticks( void );

/***************************************************************************/ /*!
 *
 * @brief    Converts milliseconds to timer ticks.
 *
 * @param    time_ms Time value in milliseconds.
 *
 * @return   This function returns the time value in timer ticks.
 *
 ******************************************************************************
 *
 * This function converts the time value @c time_ms in milliseconds to the time
 * value in timer ticks.@n
 * The period of one timer tick is defined by the @ref FNET_TIMER_PERIOD_MS.
 *
 ******************************************************************************/
unsigned long fnet_timer_ms2ticks( unsigned long time_ms );

/***************************************************************************/ /*!
 *
 * @brief    Calculates an interval between two moments in time.
 *
 * @param    start Start time in ticks.
 *
 * @param    end   End time in ticks.
 *
 * @return   This function returns an interval value between two time moments
 *           (in timer ticks).
 *
 ******************************************************************************
 *
 * This function calculates an interval between two moments in time, @c start 
 * and @c end. 
 * This function takes into account also a possible counter overrun @c (start>end).
 *
 ******************************************************************************/
unsigned long fnet_timer_get_interval( unsigned long start, unsigned long end );

/***************************************************************************/ /*!
 *
 * @brief    Performs a delay for the given number of timer ticks.
 *
 * @param    delay_ticks Time value used for delay, in ticks.
 *
 ******************************************************************************
 *
 * This function performs a delay for a given number of timer ticks. 
 * The function is blocked, till the @c delay_ticks expires. 
 *
 ******************************************************************************/
void fnet_timer_delay( unsigned long delay_ticks );

/*! @} */

#endif
