/*******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2006-2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
********************************************************************************
*
* $File Name:       PwSBC.c$
* @file             PwSBC.c
*
* $Date:            Feb-03-2015$
* @date             Feb-03-2015
*
* $Version:         1.3$
* @version          1.3
*
* Description      PowerSBC driver source file
* @brief The function            PowerSBC driver source file
*
* --------------------------------------------------------------------
* $Name:  $
*******************************************************************************/
/****************************************************************************//*!
*
*  @mainpage PowerSBC driver for MC339XX
*
*  @section Intro Introduction
*
*	This package contains PowerSBC driver allowing to control PowerBSC05, PowerSBC10 
*	and powerSBC20. PwSBC driver was created as a demo code to enable our customers
*	to start their development quickly and to demonstrate functionality of the PowerSBC
*	chip. Presented driver is not AUTOSAR nor ISO26262 compliant. This driver package 
*	needs so-called MPC5744 Basic software drivers, which present the basic layer
*	for this application.
* 
*  The key features of this package are the following:
*  - Offer driver allowing to write and read into and from PwSBC registers
*  - Switch PwSBC modes (low power, normal etc.)
*  - Treat all interrupt sources (events)
*  - Supervise voltages presented in the PwSBC interface
*
*  For more information about the functions and configuration items see these documents: 
*
*******************************************************************************
*
* @attention 
*            
*******************************************************************************/
/*==================================================================================================
*   Project              : SBC5744
*   Platform             : MPC5744
*   Dependencies         : MPC5744 - Basic SW drivers.
*   All Rights Reserved.
==================================================================================================*/

/*==================================================================================================
Revision History:
                             Modification     This function
Author (core ID)              Date D/M/Y       Name		  		Description of Changes
B35993		 				  01/03/2012 	   ALL		  		Driver created
B35993						  20/06/2013	 PwSBC_Init	  		Modified for Silicon Pass 3 (added 
														  		COMPILE_SILICON_PASS3 pre-compiler option)
B35993						  24/07/2013	  Mode_Rx/Tx  		Tags upgraded for Pass 3/4	
B35993						  24/07/2013  PwSBC_SetLPONmode   	This function deleted
B35993						  24/07/2013	  PwSBC_Init 		Upgraded for the Init bit
B35993						  05/12/2013		All				Structures updated
MicroSys                      03/02/2015       ALL              Modifications for MPC5744
MicroSys                      17/02/2015       ALL              undefined DEBUG_HANDLERS and COMPILE_SILICON_PASS3
---------------------------   ----------    ------------  ------------------------------------------
==================================================================================================*/

#include "hal_def.h"
#include "globals.h"
#include "PwSBC.h"
#include "siul2.h"
#include "adc.h"
#include "dspi.h"
#include "pit.h"

#undef  DEBUG_HANDLERS
#undef  COMPILE_SILICON_PASS3
#undef	GUI_SUPPORT

/***************************************************************************//*!
*   @brief The function PwSBC_HandleSPI_NO_COM signals a disconnected SPI.
*	@par Include: 
*					PwSBC.h
* 	@par Description: 
*					This function is called when a disconnected SPI has been 
*					detected. This function sends a warning message to the via UART 
*					in this case.
*	@remarks 	Normal mode has higher priority than the low power modes 
*				(LPON, LPOFF).
*	@par Code sample
*			PwSBC_HandleSPI_NO_COM();
*			- Program calls handler corresponding to disconnected SPI bus.
********************************************************************************/
void PwSBC_HandleSPI_NO_COM(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[10] = {'S','P','I','_','N','O','_','C','O','M'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 10);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_InitConfig1 loads a PwSBC Init configuration.
*	@par Include: 
*					PwSBC.h
* 	@par Description: 
*					This function loads PwSBC default Init configuration from 
*					program memory.
*	@remarks 	
*	@par Code sample
*			PwSBC_InitConfig1();
*			- Program loads Init configuration that will be used during Init 
*			phase of the PwSBC.
********************************************************************************/
//Initializes INITstructures that will be used during PwSBC init phase
void PwSBC_InitConfig1(){
	INITstruct.INIT_Vreg1.R = INIT_VREG1_DEF;		
	INITstruct.INIT_Vreg2.R = INIT_VREG2_DEF;		
	INITstruct.INIT_CAN_LIN.R = INIT_CAN_LIN_DEF;	
	INITstruct.INIT_IO_WU1.R = INIT_IO_WU1_DEF;	
	INITstruct.INIT_IO_WU2.R = INIT_IO_WU2_DEF;	
	INITstruct.INIT_INT.R = INIT_INT_DEF;		
	INITstruct.INIT_Supervisor1.R = INIT_SUPERVISOR1_DEF;
	INITstruct.INIT_Supervisor2.R = INIT_SUPERVISOR2_DEF;
	INITstruct.INIT_Supervisor3.R = INIT_SUPERVISOR3_DEF;
	INITstruct.INIT_FSSM1.R = INIT_FSSM1_DEF;		
	INITstruct.INIT_FSSM2.R = INIT_FSSM2_DEF;		
	INITstruct.WD_window.R = WD_WINDOW_DEF;		
	INITstruct.WD_LFSR.R = WD_LFSR_DEF;
#ifdef GUI_SUPPORT
	GUIstruct.PwSBCconfigLoaded = GUI_COM_INIT_LOADED;		//PwSBc init configuration has been successfuly loaded
#endif
}

/***************************************************************************//*!
*   @brief The function PwSBC_InitConfig2 loads a PwSBC Init configuration.
*	@par Include: 
*					PwSBC.h
* 	@par Description: 
*					This function loads PwSBC default Init configuration from GUI.
*	@remarks 	
*	@par Code sample
*			PwSBC_InitConfig2();
*			- Program asks GUI for the Init configuration.
********************************************************************************/
void PwSBC_InitConfig2(){
#ifdef GUI_SUPPORT
	GUI_GetInitConfig();
#endif
}


/*---------------------------------------------------------------------------\
* MODE functions
----------------------------------------------------------------------------*\

*S***************************************************************************//*!
*   @brief The function PwSBC_SetLPOFFmode switches the actual mode of the PwSBC 
*			to the low power OFF mode.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets bit LPOFF in the MODE register to switch PwSBC 
*					into the Low power Vreg OFF mode and checks whether the mode 
*					was entered or not.
* 	@return 	Return value can be returned only in the case that the MCU is 
*				supplied externally (not using Vcore of the PwSBC). In the 
*				other case Vcore, PowerSBC goes into the LPOFF mode, Vcore 
*				regulator is switched off and the MCU losts all data.
				- "0" - Device entered the LPOFF mode successfuly. <br>
*				- "1" - Operation failed.
*	@remarks 	Normal and LPON modes have higher priority than LPOFF. Applying 
*				this command will switch PwSBC into LPOFF mode, Vcore regulator 
*				will be switched OFF and MCU will lost all the data.
*	@par Code sample
*			while(PwSBC_SetLPmode());
*			- Program stays in an infinite loop till the until the LPOFF mode 
*			is entered.
********************************************************************************/
uint32_t PwSBC_SetLPOFFmode(){
	MODE_Tx_32B_tag cmd; 
	cmd.R = 0;
	
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = MODE_ADR;							//set address
	cmd.B.LPOFF = 1;								//LPOFF mode
	PwSBC_SendSecureCmdW(cmd.R);
	return 1;
}

/***************************************************************************//*!
*   @brief The function PwSBC_RequestINT requests an interrupt.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets bit INT_request in the MODE register, which 
*					will generate an interrupt.
*	@remarks 	INT_request will be cleared automatically when the interrupt is 
*				done.
*	@par Code sample
*			PwSBC_RequestINT();
*			- Commad requests an interrupt.
********************************************************************************/
void PwSBC_RequestINT(){
	MODE_Tx_32B_tag cmd;
	cmd.R = 0;
	
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = MODE_ADR;							//set address
	cmd.B.INT_request = 1;							//INTerruption request
	PwSBC_SendSecureCmdRW(cmd.R);
}

/*---------------------------------------------------------------------------\
* Vreg_mode functions
----------------------------------------------------------------------------*\

****************************************************************************//*!
*   @brief The function PwSBC_EnableVcore enables the Vcore regulator during 
*			the Normal mode.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets bit Vcore_EN in the Vreg Mode register, which 
*					enables the Vcore power supply.
* 	@return 	
*				- "0" - Vcore was successfuly enabled. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*			</para>
*			uint32_t status; </para>
*			status = PwSBC_EnableVcore(); </para>	
*			- Command enables Vcore power supply and stores returned status.	
********************************************************************************/
uint32_t PwSBC_EnableVcore(){
	Vreg_mode_Tx_32B_tag cmd;
	cmd.R = 0;
	
	cmd.R = INTstruct.Vreg_mode.R << 4;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = VREG_MODE_ADR;						//set address
	cmd.B.Vcore_EN = 1;								//enable Vcore
	PwSBC_SendSecureCmdRW(cmd.R);
	if(INTstruct.Vreg_mode.B.Vcore_EN == 1){
		return 0;									//success
	}
	else{
		return 1;									//error
	}	
}

/***************************************************************************//*!
*   @brief The function PwSBC_DisableVcore disables the Vcore regulator during the Normal mode.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function clears bit Vcore_EN in the Vreg Mode register, 
*					which disables the Vcore power supply.
* 	@return 	
*				- "0" - Vcore was successfuly disabled. <br>
*				- "1" - Operation failed.
*	@remarks 	Do not disable Vcore if it is configured as a safety critical.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>
*			status = PwSBC_DisableVcore();		</para>
*			- Command disables Vcore power supply and stores returned status.	
********************************************************************************/
uint32_t PwSBC_DisableVcore(){
	Vreg_mode_Tx_32B_tag cmd;
	cmd.R = 0;
	
	if(INTstruct.INIT_Supervisor1.B.Vcore_FS != 0){
		return 2;									//error -> Vcore is safety critical
	}
	
	cmd.R = INTstruct.Vreg_mode.R << 4;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = VREG_MODE_ADR;						//set address
	cmd.B.Vcore_EN = 0;								//disable Vcore
	PwSBC_SendSecureCmdRW(cmd.R);
	if(INTstruct.Vreg_mode.B.Vcore_EN == 0){
		return 0;									//success
	}
	else{
		return 1;									//error
	}	
}

/***************************************************************************//*!
*   @brief The function PwSBC_EnableVcca enables the Vcca regulator during the Normal mode.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets bit Vcca_EN in the Vreg Mode register, 
*					which enables the Vcca power supply.
* 	@return 	
*				- "0" - Vcca was successfuly enabled. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*											</para>
*			uint32_t status;				</para>
*			status = PwSBC_EnableVcca();	</para>
*			- Command enables Vcca power supply and stores returned status.	
********************************************************************************/
uint32_t PwSBC_EnableVcca(){
	Vreg_mode_Tx_32B_tag cmd;
	cmd.R = 0;
	
	cmd.R = INTstruct.Vreg_mode.R << 4;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = VREG_MODE_ADR;						//set address
	cmd.B.Vcca_EN = 1;								//enable Vcca
	PwSBC_SendSecureCmdRW(cmd.R);
	if(INTstruct.Vreg_mode.B.Vcca_EN == 1){
		return 0;									//success
	}
	else{
		return 1;									//error
	}	
}

/***************************************************************************//*!
*   @brief The function PwSBC_DisableVcca disables the Vcca regulator during the Normal mode.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function clears bit Vcca_EN in the Vreg Mode register, 
*					which disables the Vcca power supply.
* 	@return 	
*				- "0" - Vcca was successfuly disabled. <br>
*				- "1" - Operation failed.
*	@remarks 	Do not disable Vcca if it is configured as a safety critical.
*	@par Code sample
*											</para>
*			uint32_t status;				</para>
*			status = PwSBC_DisableVcca();	</para>
*			- Command disables Vcca power supply and stores returned status.	
********************************************************************************/
uint32_t PwSBC_DisableVcca(){
	Vreg_mode_Tx_32B_tag cmd;
	cmd.R = 0;
	
	if(INTstruct.INIT_Supervisor1.B.Vcca_FS != 0){
		return 2;									//error -> Vcca is safety critical
	}
	
	cmd.R = INTstruct.Vreg_mode.R << 4;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = VREG_MODE_ADR;						//set address
	cmd.B.Vcca_EN = 0;								//disable Vcca
	PwSBC_SendSecureCmdRW(cmd.R);
	if(INTstruct.Vreg_mode.B.Vcca_EN == 0){
		return 0;									//success
	}
	else{
		return 1;									//error
	}	
}

/***************************************************************************//*!
*   @brief The function PwSBC_EnableVaux enables the Vaux regulator during the Normal mode.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets bit Vaux_EN in the Vreg Mode register, which 
*					enables the Vaux power supply.
* 	@return 	
*				- "0" - Vaux was successfuly enabled. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*											</para>
*			uint32_t status;				</para>
*			status = PwSBC_EnableVaux();	</para>
*			- Command enables Vaux power supply and stores returned status.	
********************************************************************************/
uint32_t PwSBC_EnableVaux(){
	Vreg_mode_Tx_32B_tag cmd;
	cmd.R = 0;
	
	cmd.R = INTstruct.Vreg_mode.R << 4;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = VREG_MODE_ADR;						//set address
	cmd.B.Vaux_EN = 1;								//enable Vaux
	PwSBC_SendSecureCmdRW(cmd.R);
	if(INTstruct.Vreg_mode.B.Vaux_EN == 1){
		return 0;									//success
	}
	else{
		return 1;									//error
	}	
}

/***************************************************************************//*!
*   @brief The function PwSBC_DisableVaux disables the Vaux regulator during the Normal mode.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function clears bit Vaux_EN in the Vreg Mode register, 
*					which disables the Vaux power supply.
* 	@return 	
*				- "0" - Vaux was successfuly disabled. <br>
*				- "1" - Operation failed.
*	@remarks 	Do not disable Vaux if it is configured as a safety critical.
*	@par Code sample
*											</para>
*			uint32_t status;				</para>
*			status = PwSBC_DisableVaux();	</para>
*			- Command disables Vaux power supply and stores returned status.	
********************************************************************************/
uint32_t PwSBC_DisableVaux(){
	Vreg_mode_Tx_32B_tag cmd;
	cmd.R = 0;
	
	if(INTstruct.INIT_Supervisor2.B.Vaux_FS != 0){
		return 2;									//error -> Vaux is safety critical
	}
	
	cmd.R = INTstruct.Vreg_mode.R << 4;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = VREG_MODE_ADR;						//set address
	cmd.B.Vaux_EN = 0;								//disable Vaux
	PwSBC_SendSecureCmdRW(cmd.R);
	if(INTstruct.Vreg_mode.B.Vaux_EN == 0){
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_EnableVcan enables the Vcan regulator during the Normal mode.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets bit Vcan_EN in the Vreg Mode register, which 
*					enables the Vcan power supply.
* 	@return 	
*				- "0" - Vcan was successfuly enabled. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*											</para>
*			uint32_t status;				</para>
*			status = PwSBC_EnableVcan();	</para>
*			Command enables Vcan power supply and stores returned status.	
********************************************************************************/
uint32_t PwSBC_EnableVcan(){
	Vreg_mode_Tx_32B_tag cmd;
	cmd.R = 0;
	
	cmd.R = INTstruct.Vreg_mode.R << 4;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = VREG_MODE_ADR;						//set address
	cmd.B.Vcan_EN = 1;								//enable Vcan
	PwSBC_SendSecureCmdRW(cmd.R);
	if(INTstruct.Vreg_mode.B.Vcan_EN == 1){
		return 0;									//success
	}
	else{
		return 1;									//error
	}	
}

/***************************************************************************//*!
*   @brief The function PwSBC_DisableVcan disables the Vcan regulator during the Normal mode.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function clears bit Vcan_EN in the Vreg Mode register, 
*					which disables the Vcan power supply.
* 	@return 	
*				- "0" - Vcan was successfuly disabled. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*											</para>
*			uint32_t status;				</para>
*			status = PwSBC_DisableVcan();	</para>
*			- Command disables Vcan power supply and stores returned status.	
********************************************************************************/
uint32_t PwSBC_DisableVcan(){
	Vreg_mode_Tx_32B_tag cmd;
	cmd.R = 0;
	
	cmd.R = INTstruct.Vreg_mode.R << 4;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = VREG_MODE_ADR;						//set address
	cmd.B.Vcan_EN = 0;								//disable Vcan
	PwSBC_SendSecureCmdRW(cmd.R);
	if(INTstruct.Vreg_mode.B.Vcan_EN == 0){
		return 0;									//success
	}
	else{
		return 1;									//error
	}	
}

/*---------------------------------------------------------------------------\
* IO_out - AMUX functions
----------------------------------------------------------------------------*\

***************************************************************************//*!
*   @brief The function PwSBC_EnableOUT4 enables an output gate driver for the IO pins (IO4 and IO5).
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets bit IO_out_4_EN (register IO_out-AMUX), 
*					which enables output gate driver for IO_4.
* 	@return 	
*				- "0" - Output gate driver was successfuly enabled. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*											</para>
*			uint32_t status;				</para>
*			status = PwSBC_EnableOUT4();	</para>
*			- Command enables output gate driver for IO_4 and stores returned status.	
********************************************************************************/
uint32_t PwSBC_EnableOUT4(){
	IO_OUT_AMUX_32B_tag cmd;
	cmd.R = 0;
	
	cmd.R = INTstruct.IO_OUT_AMUX.R;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = IO_OUT_AMUX_ADR;					//set address
	cmd.B.IO_out_4_EN = 1;							//enable IO4 OUT
	 PwSBC_SendCmdRW(cmd.R);
	if(INTstruct.IO_OUT_AMUX.B.IO_out_4_EN == 1){
		return 0;									//success
	}
	else{
		return 1;									//error
	}	
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetOUT4 sets the specified IO output (logic High).
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets bit IO_out_4 (register IO_out-AMUX), which 
*					sets IO_4 to logical high state.
* 	@return 	
*				- "0" - Output was successfuly set to high state. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*											</para>
*			uint32_t status;				</para>
*			status = PwSBC_SetOUT4();		</para>
*			- Command sets IO_4 to high state and stores returned status.	
********************************************************************************/
uint32_t PwSBC_SetOUT4(){
	IO_OUT_AMUX_32B_tag cmd;
	cmd.R = 0;
	
	cmd.R = INTstruct.IO_OUT_AMUX.R;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = IO_OUT_AMUX_ADR;					//set address
	cmd.B.IO_out_4 = 1;								//set -> IO4 = 1
	 PwSBC_SendCmdRW(cmd.R);
	if(INTstruct.IO_OUT_AMUX.B.IO_out_4 == 1){
		return 0;									//success
	}
	else{
		return 1;									//error
	}	
}

/***************************************************************************//*!
*   @brief The function PwSBC_ClearOUT4 clears the specified IO output (logic Low).
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function clears bit IO_out_4 (register IO_out-AMUX), which 
*					sets IO_4 to logical low state.
* 	@return 	
*				- "0" - Output was successfuly set to low state. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*											</para>
*			uint32_t status;				</para>
*			status = PwSBC_ClearOUT4();		</para>
*			- Command sets IO_4 to low state and stores returned status.	
********************************************************************************/
uint32_t PwSBC_ClearOUT4(){
	IO_OUT_AMUX_32B_tag cmd;
	cmd.R = 0;
	
	cmd.R = INTstruct.IO_OUT_AMUX.R;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = IO_OUT_AMUX_ADR;					//set address
	cmd.B.IO_out_4 = 0;								//clear -> IO4 = 0
	 PwSBC_SendCmdRW(cmd.R);
	if(INTstruct.IO_OUT_AMUX.B.IO_out_4 == 0){
		return 0;									//success
	}
	else{
		return 1;									//error
	}	
}

/***************************************************************************//*!
*   @brief The function PwSBC_EnableOUT5 enables an output gate driver for the IO pins (IO4 and IO5).
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets bit IO_out_5_EN (register IO_out-AMUX), which 
*					enables output gate driver for IO_5.
* 	@return 	
*				- "0" - Output gate driver was successfuly enabled. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*												</para>
*			uint32_t status;					</para>
*			status = PwSBC_EnableOUT5();		</para>
*			- Command enables output gate driver for IO_5 and stores returned 
*			status.	
********************************************************************************/
uint32_t PwSBC_EnableOUT5(){
	IO_OUT_AMUX_32B_tag cmd;
	cmd.R = 0;
	
	cmd.R = INTstruct.IO_OUT_AMUX.R;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = IO_OUT_AMUX_ADR;					//set address
	cmd.B.IO_out_5_EN = 1;							//enable IO5 OUT
	 PwSBC_SendCmdRW(cmd.R);
	if(INTstruct.IO_OUT_AMUX.B.IO_out_5_EN == 1){
		return 0;									//success
	}
	else{
		return 1;									//error
	}	
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetOUT5 sets the specified IO output (logic High).
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets bit IO_out_5 (register IO_out-AMUX), which 
*					sets IO_5 to logical high state.
* 	@return 	
*				- "0" - Output was successfuly set to high state. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*										</para>
*			uint32_t status;			</para>
*			status = PwSBC_SetOUT5();	</para>
*			- Command sets IO_5 to high state and stores returned status.	
********************************************************************************/
uint32_t PwSBC_SetOUT5(){
	IO_OUT_AMUX_32B_tag cmd;
	cmd.R = 0;
	
	cmd.R = INTstruct.IO_OUT_AMUX.R;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = IO_OUT_AMUX_ADR;					//set address
	cmd.B.IO_out_5 = 1;								//set -> IO5 = 1
	 PwSBC_SendCmdRW(cmd.R);
	if(INTstruct.IO_OUT_AMUX.B.IO_out_5 == 1){
		return 0;									//success
	}
	else{
		return 1;									//error
	}	
}

/***************************************************************************//*!
*   @brief The function PwSBC_ClearOUT5 clears the specified IO output (logic Low).
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function clears bit IO_out_5 (register IO_out-AMUX), which 
*					sets IO_5 to logical low state.
* 	@return 	
*				- "0" - Output was successfuly set to low state. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*											</para>
*			uint32_t status;				</para>
*			status = PwSBC_ClearOUT5();		</para>
*			- Command sets IO_5 to low state and stores returned status.	
********************************************************************************/
uint32_t PwSBC_ClearOUT5(){
	IO_OUT_AMUX_32B_tag cmd;
	cmd.R = 0;
	
	cmd.R = INTstruct.IO_OUT_AMUX.R;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = IO_OUT_AMUX_ADR;					//set address
	cmd.B.IO_out_5 = 0;								//clear -> IO5 = 0
	 PwSBC_SendCmdRW(cmd.R);
	if(INTstruct.IO_OUT_AMUX.B.IO_out_5 == 0){
		return 0;									//success
	}
	else{
		return 1;									//error
	}	
}

/***************************************************************************//*!
*   @brief The function PwSBC_SwitchAMUXchannel switches a desired channel to the output of AMUX.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function switches AMUX channel to the desired one 
*					(bits Amux_2:0 in the IO_out-AMUX register.
* 	@param[in] nbChannel - Number of the desired AMUX channel (from 0 till 7).
* 	@return 	
*				- "0" - AMUX channel successfuly switched. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*													</para>
*			uint32_t status;						</para>
*			status = PwSBC_SwitchAMUXchannel(5);	</para>
*			- Command switches AMUX channel to the channel no. 5 and stores 
*			returned status.	
********************************************************************************/
uint32_t PwSBC_SwitchAMUXchannel(uint32_t nbChannel){
	IO_OUT_AMUX_32B_tag cmd;
	cmd.R = 0;
	
	if((nbChannel < 0) || (nbChannel > 7)){
		return 2;									//error -> bad number of AMUX channel
	}
	
	cmd.R = INTstruct.IO_OUT_AMUX.R;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = IO_OUT_AMUX_ADR;					//set address
	cmd.B.Amux = nbChannel;							//set AMUX channel
	 PwSBC_SendCmdRW(cmd.R);
	if(INTstruct.IO_OUT_AMUX.B.Amux == nbChannel){
		return 0;									//success
	}
	else{
		return 1;									//error
	}		 
}

/*---------------------------------------------------------------------------\
* CAN-LIN mode functions
----------------------------------------------------------------------------*\

***************************************************************************//*!
*   @brief The function PwSBC_SetCANmode sets the CAN mode.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function changes CAN mode in the CAN-LIN mode register.
* 	@param[in] mode - Desired CAN mode (from 0 till 3).
* 	@return 	
*				- "0" - CAN mode was successfuly changed. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t status;							</para>
*			status = PwSBC_SetCANmode(CAN_SLEEP);		</para>
*			- Command switches CAN transceiver into the Sleep mode without 
*			a wake-up capability.
********************************************************************************/
uint32_t PwSBC_SetCANmode(uint32_t mode){
	CAN_LIN_mode_Tx_32B_tag cmd;
	cmd.R = 0;
	
	if((mode < 0) || (mode>3)){
		return 2;									//error -> bad mode number
	}
	
	cmd.R = INTstruct.CAN_LIN_mode.R;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = CAN_LIN_MODE_ADR;					//set address
	cmd.B.CAN_mode = mode;							//set CAN mode
	 PwSBC_SendCmdRW(cmd.R);
	if(INTstruct.CAN_LIN_mode.B.CAN_mode == mode){
		return 0;									//success
	}
	else{
		return 1;									//error
	}		 
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetAutoCANTxDisable sets the Automatic CAN Tx disable.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets bit CAN_auto_dis in the CAN-LIN mode register. 
*					This will enable CAN transceiver to switch its mode from 
*					"Normal" to "Listen only" automatically when an event (CAN 
*					overtemperature/TXD dominant/RXD recessive) occurs.
* 	@return 	
*				- "0" - CAN_auto_dis bit was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*													</para>
*			uint32_t status;						</para>
*			status = PwSBC_SetAutoCANTxDisable();	</para>
*			- Command enables CAN transceiver to change its mode automatically.	
********************************************************************************/
uint32_t PwSBC_SetAutoCANTxDisable(){
	CAN_LIN_mode_Tx_32B_tag cmd;
	cmd.R = 0;
	
	cmd.R = INTstruct.CAN_LIN_mode.R;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = CAN_LIN_MODE_ADR;					//set address
	cmd.B.CAN_auto_dis = 1;							//set automatic CAN disable
	 PwSBC_SendCmdRW(cmd.R);
	if(INTstruct.CAN_LIN_mode.B.CAN_auto_dis == 1){
		return 0;									//success
	}
	else{
		return 1;									//error
	}		
}

/***************************************************************************//*!
*   @brief The function PwSBC_ClearAutoCANTxDisable disables the automatic 
*			transition of the CAN transceiver to the low-power mode.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function clears bit CAN_auto_dis in the CAN-LIN mode register. 
*					In this configuration CAN transceiver will not change its 
*					mode automatically if any event occurs.
* 	@return 	
*				- "0" - CAN_auto_dis bit was successfuly cleared. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t status;							</para>
*			status = PwSBC_ClearAutoCANTxDisable();		</para>
*			- Command  clears bit CAN_auto_dis and stores returned status.	
********************************************************************************/
uint32_t PwSBC_ClearAutoCANTxDisable(){
	CAN_LIN_mode_Tx_32B_tag cmd;
	cmd.R = 0;
	
	cmd.R = INTstruct.CAN_LIN_mode.R;				//restore old register setting
	cmd.B.RW = 1;									//write command
	cmd.B.ADR = CAN_LIN_MODE_ADR;					//set address
	cmd.B.CAN_auto_dis = 0;							//clear automatic CAN disable
	 PwSBC_SendCmdRW(cmd.R);
	if(INTstruct.CAN_LIN_mode.B.CAN_auto_dis == 0){
		return 0;									//success
	}
	else{
		return 1;									//error
	}		
}



/*---------------------------------------------------------------------------\
* WD_window functions
----------------------------------------------------------------------------*\

***************************************************************************//*!
*   @brief The function PwSBC_ChangeWDwindow changes the WD window duration.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function changes WD window duration.
* 	@param[in] duration - WD window duration coded in binary (from 0 till 15).
Mask of channels for injected conversion.
*				Predefined values:
*				- WD_DISABLE - WD window disabled
*				- WD_WIN_1 - 1 ms
*				- WD_WIN_2 - 2 ms
*				- WD_WIN_3 - 3 ms
*				- WD_WIN_4 - 4 ms
*				- WD_WIN_6 - 6 ms
*				- WD_WIN_8 - 8 ms
*				- WD_WIN_12	- 12 ms
*				- WD_WIN_16	- 16 ms
*				- WD_WIN_24	- 24 ms
*				- WD_WIN_32	- 32 ms
*				- WD_WIN_64	- 64 ms
*				- WD_WIN_128 - 128 ms
*				- WD_WIN_256 - 256 ms
*				- WD_WIN_512 - 512 ms
*				- WD_WIN_1024 - 1024 ms
* 	@return 	
*				- "0" - WD window duration was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	
*	@par Code sample
*														</para>	
*			uint32_t status;							</para>
*			status = PwSBC_ChangeWDwindow(WD_WIN_32);	</para>
*			- Command changes WD window duration to 32 ms.	
********************************************************************************/
uint32_t PwSBC_ChangeWDwindow(uint32_t duration){
	WD_window_Tx_32B_tag cmd;
	cmd.R = 0;
	
	if((duration < 0) || (duration > 15)){
		return 2;										//error -> given parameter is OUT of range
	}
	
	cmd.B.RW = 1;										//write command
	cmd.B.ADR = WD_WINDOW_ADR;							//set address
	cmd.B.WD_window = duration;							//set WD window duration
	PwSBC_SendSecureCmdRW(cmd.R);
	if(INTstruct.WD_window.B.WD_window == duration){
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}

/*---------------------------------------------------------------------------\
* WD_LFSR functions
----------------------------------------------------------------------------*\

***************************************************************************//*!
*   @brief The function PwSBC_SendSeed sends an LFSR seed from the MCU to the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sends new seed for the implemented LFSR.
* 	@param[in] seed - 	New seed for the LFSR. This can be any 8-bit value 
*						excluding 0.
* 	@return 	
*				- "0" - Seed was successfuly changed. <br>
*				- "1" - Operation failed. <br>
*				- "2" - Wrong seed parameter (cannot be 0).
*	@remarks 	Zero state is not allowed for this type of LFSR, because it 
*				should never evolve from this state.
*	@par Code sample
*											</para>
*			uint32_t status;				</para>	
*			status = PwSBC_SendSeed(20);	</para>	
*			- Command changes LFSR value to 20 (decimal).	
********************************************************************************/
uint32_t PwSBC_SendSeed(uint32_t seed){
	WD_LFSR_Tx_32B_tag cmd;
	cmd.R = 0;
	
	if(seed == 0){
		return 2;				//error ->	seed CANNOT be zero
	}
	
	cmd.B.RW = 1;											//write command
	cmd.B.ADR = WD_LFSR_ADR;								//set address
	cmd.B.WD_LFSR = seed;									//set LFSR seed
	 PwSBC_SendCmdRW(cmd.R);
	if(INTstruct.WD_LFSR.B.WD_LFSR == seed){
		return 0;									//success
	}
	else{
		return 1;									//error
	}
}
 
/*---------------------------------------------------------------------------\
* WD_answer functions
----------------------------------------------------------------------------*\

***************************************************************************//*!
*   @brief The function PwSBC_RefreshWD refreshes the WD.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function refreshes WD using the given WD answer. This is 
*					done by writing into the WD answer register.
* 	@param[in] answer - 8-bit WD answer to be sent.
* 	@return 	
*				- "0" - WD was successfuly refreshed. <br>
*				- "10" - SPI disconnected or no SPI answer (in decimal). <br>
*				- "11" - SPI_G error detected (in decimal).
*	@remarks 	8-bit answer must be constisted of the control computations 
*				that have been done on the actual LFSR content. 
*	@par Code sample
*												</para>
*			uint32_t status;					</para>
*			status = PwSBC_RefreshWD(50);		</para>
*			- Command sends WD answer with value 50.	
********************************************************************************/
uint32_t PwSBC_RefreshWD(uint32_t answer){
	WD_answer_Tx_32B_tag cmd;
	uint32_t resp = 0;
	//uint8_t error = 0;
	
	cmd.R = 0;
	
	cmd.B.RW = 1;										//write command
	cmd.B.ADR = WD_ANSWER_ADR;							//set address
	cmd.B.WD_answer = answer;							//set answer
	
	resp =  PwSBC_SendCmdRW(cmd.R);

	return resp;										//returns error status from the previous function
}

/***************************************************************************//*!
*   @brief The function PwSBC_GenerateLFSR generates, stores and returns a new state of the LFSR from the previous 
*			one, that is stored in memory.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function evolves LFSR implemented in the MCU into the 
*					following state and stores the new value in a global 
*					structure.
* 	@return 	New LFSR state.
*	@remarks 	If this function is used, then the synchronization between LFSR 
*				implemented in the MCU and the one in PwSBC must be guaranteed.
*	@par Code sample
*												</para>
*			uint32_t new_lfsr;					</para>	
*			new_lfsr = PwSBC_GenerateLFSR();	</para>	
*			- Command evolves actual LFSR content and returns the new one.	
********************************************************************************/
uint32_t PwSBC_GenerateLFSR(){
	register32_struct gate;
	gate.R = 0;
	
	gate.B.bit0 = ~(PITstruct.currentLFSR.B.bit7 ^ PITstruct.currentLFSR.B.bit5);
	gate.B.bit0 = ~(PITstruct.currentLFSR.B.bit4 ^ gate.B.bit0);
	gate.B.bit0 = ~(PITstruct.currentLFSR.B.bit3 ^ gate.B.bit0);
	
	PITstruct.currentLFSR.R <<= 1;
	PITstruct.currentLFSR.B.bit0 = gate.B.bit0;
	
	PITstruct.currentLFSR.R &= 0xFF;		//mask out only the lowest Byte
	
	return PITstruct.currentLFSR.R;	
}

/***************************************************************************//*!
*   @brief The function PwSBC_ComputeLFSR computes, stores and returns test based from actual LFSR.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function makes control computations with the given LFSR and 
*					returns result of the computation on the least 8 bits.
* 	@param[in] actualLFSR - 8-bit LFSR value on which will be applied control 
*						computations.
* 	@return 	Result of the control computations. Computations are made as 
*				follows:
*				~(((lfsr*4 + 6 - 4)/4
*	@remarks 	Control computations are made in assembler using instructions 
*				to prove ALU using basic operations (*, /, +, -).
*	@par Code sample
*												</para>
*			uint32_t result;					</para>
*			result = PwSBC_ComputeLFSR(50);		</para>
*			- Command makes control computations with the given LFSRvalue (50) 
*			and returns result.	
********************************************************************************/
uint32_t PwSBC_ComputeLFSR(uint32_t	actualLFSR){ 
#if 0
	asm{ 
		#if __option(vle)
			se_li 		r25,0x04		//load nb.4 -> r25
			mullw 		r3,r25,r3		//lfsr * 4 	-> r3
			e_add16i	r3,r3,6			//r3 + 6 		-> r3
			se_subi 	r3,4				//r3 - 4 		-> r3
			se_not 	r3						//NOT r3 		-> r3
			e_li 		r24,0xFFFF			//mask 			-> r24
			se_and 	r3,r24				//r24 & r3  -> r3
			divwu		r3,r3,r25			//r3 / 4		-> r3	-> as a return value
			e_li 		r24,0xFF			//mask 			-> r24
			se_and 	r3,r24				//store only lower 8 bits -> r3
		#else 
			li r25,0x04						//load nb.4 -> r25
			mullw r3,r25,r3				//lfsr * 4 	-> r3
			addi r3,r3,6					//r3 + 6		-> r3
			li r25,4							//load nb.4 -> r25
			subf r3,r25,r3				//r3 - r25	-> r3
			li r25,0xFFFF						//mask for negation	-> r25
			nand r3,r3,r25				//r25 NAND r3 -> r3
			li r25,4							//load nb.r3 -> r25
			divwu r3,r3,r25				//r3 / r25	->	r3 -> as a return value
			clrlwi r3,r3,24				//store only 8 lower bits
	
		#endif 	
	}
#else
	return (~(actualLFSR*4 + 6 - 4)/4);
#endif
}


/*---------------------------------------------------------------------------\
* Fail_Safe_OUT functions
----------------------------------------------------------------------------*\

***************************************************************************//*!
*   @brief The function PwSBC_ReleaseFSout releases the Fail Safe output FS0b.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function swaps actual LFSR content (stored in the LFSR) 
*					and sends command to release safe output FS0b.
*	@remarks 	Command should be applied to release fail safe outputs when RST 
*				error counter goes back to the state no. 0.
*	@par Code sample
*			PwSBC_ReleaseFSout();	
*			- Command releases fail safe outputs.	
********************************************************************************/
void PwSBC_ReleaseFSout(){
	FS_OUT_Tx_32B_tag cmd;
	cmd.R = 0;

	PwSBC_GetWDLFSR();									//get current LFSR content
	PITstruct.currentLFSR.R = INTstruct.WD_LFSR.R;		//store current LFSR in PIT structure
		
	cmd.B.FS_out_0 = PITstruct.currentLFSR.B.bit7;		//swap current LFSR register content
	cmd.B.FS_out_1 = PITstruct.currentLFSR.B.bit6;
	cmd.B.FS_out_2 = PITstruct.currentLFSR.B.bit5;
	cmd.B.FS_out_3 = PITstruct.currentLFSR.B.bit4;
	cmd.B.FS_out_4 = PITstruct.currentLFSR.B.bit3;
	cmd.B.FS_out_5 = PITstruct.currentLFSR.B.bit2;
	cmd.B.FS_out_6 = PITstruct.currentLFSR.B.bit1;
	cmd.B.FS_out_7 = PITstruct.currentLFSR.B.bit0;

	cmd.R = ~ cmd.R;									//negate already swapped bits

	cmd.B.RW = 1;										//write command
	cmd.B.ADR = FS_OUT_ADR;								//set address

	PwSBC_SendCmdW(cmd.R);
}

/*---------------------------------------------------------------------------\
* RSTb_request functions
----------------------------------------------------------------------------*\

***************************************************************************//*!
*   @brief The function PwSBC_RequestReset requests a low pulse on the RSTb - MCU reset.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function requests one Reset pulse.
*	@remarks 	Reset can cause a data loss thanks to the restart of the MCU.
*	@par Code sample
*			PwSBC_RequestReset();	
*			- Command requests one Reset pulse.	
********************************************************************************/
void PwSBC_RequestReset(){
	RSTb_request_Tx_32B_tag cmd;
	cmd.R = 0;
	
	cmd.B.RW = 1;										//write command
	cmd.B.ADR = RSTB_REQUEST_ADR;						//set address
	cmd.B.RSTb_request = 1;
	
	PwSBC_SendSecureCmdW(cmd.R);
}

/*---------------------------------------------------------------------------\
* Getters
----------------------------------------------------------------------------*\

***************************************************************************//*!
*   @brief The function PwSBC_GetInitVreg1 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the INIT_VREG1 register.
* 	@return 	Content of the INIT_VREG1 register.
*	@remarks 	
*	@par Code sample
*												</para>
*			uint32_t reg_setting;				</para>
*			reg_setting = PwSBC_GetInitVreg1();		</para>
*			- Command returns content of the INIT_VREG1 register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetInitVreg1(){
	PwSBC_SendCmdR(INIT_VREG1_ADR << 9);    //send a read command created from register address
	return INTstruct.INIT_Vreg1.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetInitVreg2 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the INIT_VREG2 register.
* 	@return 	Content of the INIT_VREG2 register.
*	@remarks 	
*	@par Code sample
*												</para>
*			uint32_t reg_setting;				</para>
*			reg_setting = PwSBC_GetInitVreg2();		</para>
*			- Command returns content of the INIT_VREG2 register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetInitVreg2(){
	PwSBC_SendCmdR(INIT_VREG2_ADR << 9);    //send a read command created from register address
	return INTstruct.INIT_Vreg2.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetInitCAN gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the INIT_CAN_LIN register.
* 	@return 	Content of the INIT_CAN_LIN register.
*	@remarks 	
*	@par Code sample
*													</para>
*			uint32_t reg_setting;					</para>
*			reg_setting = PwSBC_GetInitCAN();		</para>
*			- Command returns content of the INIT_CAN_LIN register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetInitCAN(){
	PwSBC_SendCmdR(INIT_CAN_LIN_ADR << 9);    //send a read command created from register address
	return INTstruct.INIT_CAN_LIN.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetInitIOWU1 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the INIT_IO_WU1 register.
* 	@return 	Content of the INIT_IO_WU1 register.
*	@remarks 	
*	@par Code sample
*												</para>
*			uint32_t reg_setting;				</para>	
*			reg_setting = PwSBC_GetInitIOWU1();		</para>
*			- Command returns content of the INIT_IO_WU1 register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetInitIOWU1(){
	PwSBC_SendCmdR(INIT_IO_WU1_ADR << 9);    //send a read command created from register address
	return INTstruct.INIT_IO_WU1.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetInitIOWU2 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the INIT_IO_WU2 register.
* 	@return 	Content of the INIT_IO_WU2 register.
*	@remarks 	
*	@par Code sample
*													</para>
*			uint32_t reg_setting;					</para>
*			reg_setting = PwSBC_GetInitIOWU2();			</para>		
*			- Command returns content of the INIT_IO_WU2 register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetInitIOWU2(){
	PwSBC_SendCmdR(INIT_IO_WU2_ADR << 9);    //send a read command created from register address
	return INTstruct.INIT_IO_WU2.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetInitINT gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the INIT_INT register.
* 	@return 	Content of the INIT_INT register.
*	@remarks 	
*	@par Code sample
*												</para>
*			uint32_t reg_setting;				</para>
*			reg_setting = PwSBC_GetInitINT();		</para>
*			- Command returns content of the INIT_INT register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetInitINT(){
	PwSBC_SendCmdR(INIT_INT_ADR << 9);    //send a read command created from register address
	return INTstruct.INIT_INT.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetHWconfig gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the HW_CONFIG register.
* 	@return 	Content of the HW_CONFIG register.
*	@remarks 	
*	@par Code sample
*												</para>
*			uint32_t reg_setting;				</para>
*			reg_setting = PwSBC_GetHWconfig();		</para>
*			- Command returns content of the HW_CONFIG register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetHWconfig(){
	PwSBC_SendCmdR(HW_CONFIG_ADR << 9);    //send a read command created from register address
	return INTstruct.HW_Config.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetWUsource gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the WU_SOURCE register.
* 	@return 	Content of the WU_SOURCE register.
*	@remarks 	
*	@par Code sample
*												</para>		
*			uint32_t reg_setting;				</para>
*			reg_setting = PwSBC_GetWUsource();		</para>
*			- Command returns content of the WU_SOURCE register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetWUsource(){
	PwSBC_SendCmdR(WU_SOURCE_ADR << 9);    //send a read command created from register address
	return INTstruct.WU_source.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetIOinputState gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the IO_INPUT_STATE register.
* 	@return 	Content of the IO_INPUT_STATE register.
*	@remarks 	
*	@par Code sample
*													</para>
*			uint32_t reg_setting;					</para>
*			reg_setting = PwSBC_GetIOinputState();		</para>
*			- Command returns content of the IO_INPUT_STATE register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetIOinputState(){
	PwSBC_SendCmdR(IO_INPUT_STATE_ADR << 9);    //send a read command created from register address
	return INTstruct.IO_input_state.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetStatusVreg1 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the STATUS_VREG1 register.
* 	@return 	Content of the STATUS_VREG1 register.
*	@remarks 	
*	@par Code sample								
*													</para>
*			uint32_t reg_setting;					</para>
*			reg_setting = PwSBC_GetStatusVreg1();		</para>			
*			- Command returns content of the STATUS_VREG1 register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetStatusVreg1(){
	PwSBC_SendCmdR(STATUS_VREG1_ADR << 9);    //send a read command created from register address
	return INTstruct.Status_Vreg1.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetStatusVreg2 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the STATUS_VREG2 register.
* 	@return 	Content of the STATUS_VREG2 register.
*	@remarks 	
*	@par Code sample
*													</para>
*			uint32_t reg_setting;					</para>
*			reg_setting = PwSBC_GetStatusVreg2();	</para>	
*			- Command returns content of the STATUS_VREG2 register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetStatusVreg2(){
	PwSBC_SendCmdR(STATUS_VREG2_ADR << 9);    //send a read command created from register address
	return INTstruct.Status_Vreg2.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetDiagVreg1 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the DIAG_VREG1 register.
* 	@return 	Content of the DIAG_VREG1 register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>
*			reg_setting = PwSBC_GetDiagVreg1();			</para>	
*			- Command returns content of the DIAG_VREG1 register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetDiagVreg1(){
	PwSBC_SendCmdR(DIAG_VREG1_ADR << 9);    //send a read command created from register address
	return INTstruct.Diag_Vreg1.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetDiagVreg2 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the DIAG_VREG2 register.
* 	@return 	Content of the DIAG_VREG2 register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>
*			reg_setting = PwSBC_GetDiagVreg2();			</para>	
*			- Command returns content of the DIAG_VREG2 register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetDiagVreg2(){
	PwSBC_SendCmdR(DIAG_VREG2_ADR << 9);    //send a read command created from register address
	return INTstruct.Diag_Vreg2.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetDiagVreg3 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the DIAG_VREG3 register.
* 	@return 	Content of the DIAG_VREG3 register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>
*			reg_setting = PwSBC_GetDiagVreg3();			</para>	
*			- Command returns content of the DIAG_VREG3 register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetDiagVreg3(){
	PwSBC_SendCmdR(DIAG_VREG3_ADR << 9);    //send a read command created from register address
	return INTstruct.Diag_Vreg3.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetDiagCAN gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the DIAG_CAN register.
* 	@return 	Content of the DIAG_CAN register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>	
*			reg_setting = PwSBC_GetDiagCAN();			</para>	
*			- Command returns content of the DIAG_CAN register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetDiagCAN(){
	PwSBC_SendCmdR(DIAG_CAN_ADR << 9);    //send a read command created from register address
	return INTstruct.Diag_CAN.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetCAN gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the DIAG_CAN_LIN register.
* 	@return 	Content of the DIAG_CAN_LIN register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>
*			reg_setting = PwSBC_GetCAN();		</para>	
*			- Command returns content of the DIAG_CAN_LIN register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetCAN(){
	PwSBC_SendCmdR(DIAG_CAN_LIN_ADR << 9);    //send a read command created from register address
	return INTstruct.Diag_CAN_LIN.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetDiagSPI gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the DIAG_SPI register.
* 	@return 	Content of the DIAG_SPI register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>	
*			reg_setting = PwSBC_GetDiagSPI();			</para>	
*			- Command returns content of the DIAG_SPI register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetDiagSPI(){
	PwSBC_SendCmdR(DIAG_SPI_ADR << 9);    //send a read command created from register address
	return INTstruct.Diag_SPI.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetMode gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the MODE register.
* 	@return 	Content of the MODE register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>
*			reg_setting = PwSBC_GetMode();				</para>	
*			- Command returns content of the MODE register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetMode(){
	PwSBC_SendCmdR(MODE_ADR << 9);    //send a read command created from register address
	return INTstruct.MODE.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetVregMode gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the VREG_MODE register.
* 	@return 	Content of the VREG_MODE register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>
*			reg_setting = PwSBC_GetVregMode();			</para>	
*			- Command returns content of the VREG_MODE register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetVregMode(){
	PwSBC_SendCmdR(VREG_MODE_ADR << 9);    //send a read command created from register address
	return INTstruct.Vreg_mode.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetIOAMUX gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the IO_OUT_AMUX register.
* 	@return 	Content of the IO_OUT_AMUX register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>
*			reg_setting = PwSBC_GetIOAMUX();			</para>	
*			- Command returns content of the IO_OUT_AMUX register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetIOAMUX(){
	PwSBC_SendCmdR(IO_OUT_AMUX_ADR << 9);    //send a read command created from register address
	return INTstruct.IO_OUT_AMUX.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetCANmode gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the CAN_LIN_MODE register.
* 	@return 	Content of the CAN_LIN_MODE register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>
*			reg_setting = PwSBC_GetCANmode();		</para>	
*			- Command returns content of the CAN_LIN_MODE register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetCANmode(){
	PwSBC_SendCmdR(CAN_LIN_MODE_ADR << 9);    //send a read command created from register address
	return INTstruct.CAN_LIN_mode.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetVregLP gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the VREG_LPON register.
* 	@return 	Content of the VREG_LPON register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>	
*			reg_setting = PwSBC_GetVregLP();			</para>	
*			- Command returns content of the VREG_LPON register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetVregLP(){
	PwSBC_SendCmdR(VREG_LPON_ADR << 9);    //send a read command created from register address
	return INTstruct.Vreg_LPON.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetInitSupervisor1 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the INIT_SUPERVISOR1 register.
* 	@return 	Content of the INIT_SUPERVISOR1 register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>
*			reg_setting = PwSBC_GetInitSupervisor1();	</para>	
*			- Command returns content of the INIT_SUPERVISOR1 register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetInitSupervisor1(){
	PwSBC_SendCmdR(INIT_SUPERVISOR1_ADR << 9);    //send a read command created from register address
	return INTstruct.INIT_Supervisor1.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetInitSupervisor2 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the INIT_SUPERVISOR2 register.
* 	@return 	Content of the INIT_SUPERVISOR2 register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>
*			reg_setting = PwSBC_GetInitSupervisor2();	</para>	
*			- Command returns content of the INIT_SUPERVISOR2 register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetInitSupervisor2(){
	PwSBC_SendCmdR(INIT_SUPERVISOR2_ADR << 9);    //send a read command created from register address
	return INTstruct.INIT_Supervisor2.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetInitSupervisor3 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the INIT_SUPERVISOR3 register.
* 	@return 	Content of the INIT_SUPERVISOR3 register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>
*			reg_setting = PwSBC_GetInitSupervisor3();	</para>	
*			- Command returns content of the INIT_SUPERVISOR3 register and 
*			stores it in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetInitSupervisor3(){
	PwSBC_SendCmdR(INIT_SUPERVISOR3_ADR << 9);    //send a read command created from register address
	return INTstruct.INIT_Supervisor3.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetInitFSSM1 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the INIT_FSSM1 register.
* 	@return 	Content of the INIT_FSSM1 register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>	
*			reg_setting = PwSBC_GetInitFSSM1();			</para>	
*			- Command returns content of the INIT_FSSM1 register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetInitFSSM1(){
	PwSBC_SendCmdR(INIT_FSSM1_ADR << 9);    //send a read command created from register address
	return INTstruct.INIT_FSSM1.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetInitFSSM2 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the INIT_FSSM2 register.
* 	@return 	Content of the INIT_FSSM2 register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>	
*			reg_setting = PwSBC_GetInitFSSM2();			</para>	
*			- Command returns content of the INIT_FSSM2 register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetInitFSSM2(){
	PwSBC_SendCmdR(INIT_FSSM2_ADR << 9);    //send a read command created from register address
	return INTstruct.INIT_FSSM2.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetWDwindow gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the WD_WINDOW register.
* 	@return 	Content of the WD_WINDOW register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>	
*			reg_setting = PwSBC_GetWDwindow();			</para>	
*			- Command returns content of the WD_WINDOW register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetWDwindow(){
	PwSBC_SendCmdR(WD_WINDOW_ADR << 9);    //send a read command created from register address
	return INTstruct.WD_window.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetWDLFSR gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the WD_LFSR register.
* 	@return 	Content of the WD_LFSR register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>	
*			reg_setting = PwSBC_GetWDLFSR();			</para>	
*			- Command returns content of the WD_LFSR register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetWDLFSR(){
	PwSBC_SendCmdR(WD_LFSR_ADR << 9);    //send a read command created from register address
	return INTstruct.WD_LFSR.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetWDanswer gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the WD_ANSWER register.
* 	@return 	Content of the WD_ANSWER register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>	
*			reg_setting = PwSBC_GetWDanswer();			</para>	
*			- Command returns content of the WD_ANSWER register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetWDanswer(){
	PwSBC_SendCmdR(WD_ANSWER_ADR << 9);    //send a read command created from register address
	return INTstruct.WD_answer.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetDiagFS1 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the DIAG_FS1 register.
* 	@return 	Content of the DIAG_FS1 register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>	
*			reg_setting = PwSBC_GetDiagFS1();			</para>	
*			- Command returns content of the DIAG_FS1 register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetDiagFS1(){
	PwSBC_SendCmdR(DIAG_FS1_ADR << 9);    //send a read command created from register address
	return INTstruct.Diag_FS1.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetWDcounter gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the WD_COUNTER register.
* 	@return 	Content of the WD_COUNTER register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>
*			reg_setting = PwSBC_GetWDcounter();			</para>	
*			- Command returns content of the WD_COUNTER register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetWDcounter(){
	PwSBC_SendCmdR(WD_COUNTER_ADR << 9);    //send a read command created from register address
	return INTstruct.WD_counter.R;          
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetDiagFS2 gets actual content of the register.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function returns content of the DIAG_FS2 register.
* 	@return 	Content of the DIAG_FS2 register.
*	@remarks 	
*	@par Code sample
*														</para>
*			uint32_t reg_setting;						</para>
*			reg_setting = PwSBC_GetDiagFS2();			</para>	
*			- Command returns content of the DIAG_FS2 register and stores it 
*			in the "reg_setting" variable.	
********************************************************************************/
uint32_t PwSBC_GetDiagFS2(){
	PwSBC_SendCmdR(DIAG_FS2_ADR << 9);    //send a read command created from register address
	return INTstruct.Diag_FS2.R;          
}


/*---------------------------------------------------------------------------\
* Setters for INIT functions
----------------------------------------------------------------------------*\

***************************************************************************//*!
*   @brief The function PwSBC_SetInitVreg1 sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the INIT_VREG1 register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>
*			status = PwSBC_SetInitVreg1();		</para>
*			- Command sets INIT_VREG1 register and stores the returned status. 
*			Security bits and parity bits can be neglected, because they will be 
*			set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetInitVreg1(uint32_t cmd){
	uint32_t errCode;
	errCode =  PwSBC_SendCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if((INTstruct.INIT_Vreg1.R & 0x00FF) == (cmd & 0x00FF)){		//compare lower Byte only
			return 0;							//success
		}
		else{
			return 1;							//error -> register content NOT verified
		}	
	}
	
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetInitVreg2 sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the INIT_VREG2 register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>
*			status = PwSBC_SetInitVreg2();		</para>
*			- Command sets INIT_VREG2 register and stores the returned status. 
*			Security bits and parity bits can be neglected, because they will be 
*			set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetInitVreg2(uint32_t cmd){
	uint32_t errCode;
	errCode =  PwSBC_SendCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if((INTstruct.INIT_Vreg2.R & 0x00FF) == (cmd & 0x00FF)){		//compare lower Byte only
			return 0;							//success
		}
		else{
			return 1;							//error -> register content NOT verified
		}	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetInitCAN sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the INIT_CAN_LIN register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>	
*			status = PwSBC_SetInitCAN();			</para>
*			- Command sets INIT_CAN_LIN register and stores the returned status. 
*			Security bits and parity bits can be neglected, because they will be 
*			set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetInitCAN(uint32_t cmd){
	uint32_t errCode;
	errCode =  PwSBC_SendCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if((INTstruct.INIT_CAN_LIN.R & 0x00FF) == (cmd & 0x00FF)){		//compare lower Byte only
			return 0;							//success
		}
		else{
			return 1;							//error -> register content NOT verified
		}	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetInitIOWU1 sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the INIT_IO_WU1 register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>	
*			status = PwSBC_SetInitIOWU1();			</para>
*			- Command sets INIT_IO_WU1 register and stores the returned status. 
*			Security bits and parity bits can be neglected, because they will be 
*			set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetInitIOWU1(uint32_t cmd){
	uint32_t errCode;
	errCode =  PwSBC_SendCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if((INTstruct.INIT_IO_WU1.R & 0x00FF) == (cmd & 0x00FF)){		//compare lower Byte only
			return 0;							//success
		}
		else{
			return 1;							//error -> register content NOT verified
		}	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetInitIOWU2 sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the INIT_IO_WU2 register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>	
*			status = PwSBC_SetInitIOWU2();		</para>
*			- Command sets INIT_IO_WU2 register and stores the returned status. 
*			Security bits and parity bits can be neglected, because they will be 
*			set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetInitIOWU2(uint32_t cmd){
	uint32_t errCode;
	errCode =  PwSBC_SendCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		
		if((INTstruct.INIT_IO_WU2.R & 0x00FF) == (cmd & 0x00FF)){		//compare lower Byte only
			return 0;							//success
		}
		else{
			return 1;							//error -> register content NOT verified
		}	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetInitINT sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the INIT_INT register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>	
*			status = PwSBC_SetInitINT();		</para>
*			- Command sets INIT_INT register and stores the returned status. 
*			Security bits and parity bits can be neglected, because they will be 
*			set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetInitINT(uint32_t cmd){
	uint32_t errCode;
	errCode =  PwSBC_SendCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if((INTstruct.INIT_INT.R & 0x00FF) == (cmd & 0x00FF)){		//compare lower Byte only
			return 0;									//success
		}
		else{
			return 1;									//error -> register content NOT verified
		}	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetInitSupervisor1 sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the INIT_SUPERVISOR1 register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>	
*			status = PwSBC_SetInitSupervisor1();		</para>	
*			- Command sets INIT_SUPERVISOR1 register and stores the returned 
*			status. Security bits and parity bits can be neglected, because they 
*			will be set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetInitSupervisor1(uint32_t cmd){
	uint32_t errCode;
	INIT_Supervisor1_Tx_32B_tag cmdTx;
	cmdTx.R = cmd;
	
	errCode = PwSBC_SendSecureCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if((INTstruct.INIT_Supervisor1.B.Vcca_FS == cmdTx.B.Vcca_FS) && (INTstruct.INIT_Supervisor1.B.Vcore_FS == cmdTx.B.Vcore_FS)){
			return 0;									//success
		}
		else{
			return 1;									//error -> register content NOT verified
		}	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetInitSupervisor2 sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the INIT_SUPERVISOR2 register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>
*			status = PwSBC_SetInitSupervisor2();			</para>
*			- Command sets INIT_SUPERVISOR2 register and stores the returned 
*			status. Security bits and parity bits can be neglected, because they 
*			will be set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetInitSupervisor2(uint32_t cmd){
	uint32_t errCode;
	INIT_Supervisor2_Tx_32B_tag cmdTx;
	cmdTx.R = cmd;
	
	errCode = PwSBC_SendSecureCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if((INTstruct.INIT_Supervisor2.B.Vaux_FS == cmdTx.B.Vaux_FS) && (INTstruct.INIT_Supervisor2.B.DIS_8s == cmdTx.B.DIS_8s)){
			return 0;									//success
		}
		else{
			return 1;									//error -> register content NOT verified
		}	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetInitSupervisor3 sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the INIT_SUPERVISOR3 register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>	
*			status = PwSBC_SetInitSupervisor3();			</para>
*			- Command sets INIT_SUPERVISOR3 register and stores the returned 
*			status. Security bits and parity bits can be neglected, because they 
*			will be set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetInitSupervisor3(uint32_t cmd){
	uint32_t errCode;
	INIT_Supervisor3_Tx_32B_tag cmdTx;
	cmdTx.R = cmd;
	
	errCode = PwSBC_SendSecureCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if((INTstruct.INIT_Supervisor3.B.Vcore_5D == cmdTx.B.Vcore_5D) && (INTstruct.INIT_Supervisor3.B.Vcca_5D == cmdTx.B.Vcca_5D) && (INTstruct.INIT_Supervisor3.B.Vaux_5D == cmdTx.B.Vaux_5D)){
			return 0;									//success
		}
		else{
			return 1;									//error -> register content NOT verified
		}	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetInitFSSM1 sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the INIT_FSSM1 register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>	
*			status = PwSBC_SetInitFSSM1();			</para>
*			- Command sets INIT_FSSM1 register and stores the returned status. 
*			Security bits and parity bits can be neglected, because they will be 
*			set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetInitFSSM1(uint32_t cmd){
	uint32_t errCode;
	
	errCode = PwSBC_SendSecureCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if(((cmd & 0x000000FF) >> 4) == (INTstruct.INIT_FSSM1.R & 0x0000000F)){
			return 0;									//success
		}
		else{
			return 1;									//error -> register content NOT verified
		}	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetInitFSSM2 sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the INIT_FSSM2 register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>	
*			status = PwSBC_SetInitFSSM2();			</para>
*			- Command sets INIT_FSSM2 register and stores the returned status. 
*			Security bits and parity bits can be neglected, because they will be 
*			set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetInitFSSM2(uint32_t cmd){
	uint32_t errCode;
	
	errCode = PwSBC_SendSecureCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if(((cmd & 0x000000FF) >> 4) == (INTstruct.INIT_FSSM2.R & 0x0000000F)){
			return 0;									//success
		}
		else{
			return 1;									//error -> register content NOT verified
		}	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetMode sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the MODE register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>	
*			status = PwSBC_SetMode();			</para>
*			- Command sets MODE register and stores the returned status. 
*			Security bits and parity bits can be neglected, because they will be 
*			set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetMode(uint32_t cmd){
	uint32_t errCode;
	
	errCode = PwSBC_SendSecureCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if((((cmd & 0x000000FF) >> 5) == (INTstruct.MODE.R & 0x00000007)) || ((cmd & 0x0010) == 0x0010)){
			return 0;									//success
		}
		else{
			return 1;									//error -> register content NOT verified
		}	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetVregMode sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the VREG_MODE register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>
*			status = PwSBC_SetVregMode();			</para>
*			- Command sets VREG_MODE register and stores the returned status. 
*			Security bits and parity bits can be neglected, because they will be 
*			set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetVregMode(uint32_t cmd){
	uint32_t errCode;
	
	errCode = PwSBC_SendSecureCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if(((cmd & 0x000000FF) >> 4) == (INTstruct.Vreg_mode.R & 0x0000000F)){
			return 0;									//success
		}
		else{
			return 1;									//error -> register content NOT verified
		}	
	}
}	

/***************************************************************************//*!
*   @brief The function PwSBC_SetIOAMUX sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the IO_OUT_AMUX register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>
*			status = PwSBC_SetIOAMUX();			</para>
*			- Command sets IO_OUT_AMUX register and stores the returned status. 
*			Security bits and parity bits can be neglected, because they will be 
*			set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetIOAMUX(uint32_t cmd){
	uint32_t errCode;
	
	errCode =  PwSBC_SendCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if(((cmd & 0x000000FF) & 0x000000F7) == (INTstruct.IO_OUT_AMUX.R & 0x000000F7)){
			return 0;									//success
		}
		else{
			return 1;									//error -> register content NOT verified
		}	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetCANregMode sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the CAN_LIN_MODE register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>	
*			status = PwSBC_SetCANregMode();			</para>
*			- Command sets CAN_LIN_MODE register and stores the returned status. 
*			Security bits and parity bits can be neglected, because they will be 
*			set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetCANregMode(uint32_t cmd){
	uint32_t errCode;
	
	errCode =  PwSBC_SendCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if(((cmd & 0x000000FF) & 0x000000FC) == (INTstruct.CAN_LIN_mode.R & 0x000000FC)){
			return 0;									//success
		}
		else{
			return 1;									//error -> register content NOT verified
		}	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetCANregMode2 sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the VREG_LPON register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>
*			status = PwSBC_SetCANregMode2();			</para>
*			- Command sets VREG_LPON register and stores the returned status. 
*			Security bits and parity bits can be neglected, because they will be 
*			set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetCANregMode2(uint32_t cmd){
	uint32_t errCode;
	
	errCode = PwSBC_SendSecureCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if(((cmd & 0x000000FF) >> 5) == (INTstruct.Vreg_LPON.R & 0x00000007)){
			return 0;									//success
		}
		else{
			return 1;									//error -> register content NOT verified
		}	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetWDwindow sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the WD_WINDOW register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>	
*			status = PwSBC_SetWDwindow();			</para>
*			- Command sets WD_WINDOW register and stores the returned status. 
*			Security bits and parity bits can be neglected, because they will be 
*			set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetWDwindow(uint32_t cmd){
	uint32_t errCode;
	
	errCode = PwSBC_SendSecureCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if(((cmd & 0x000000FF) >> 4) == (INTstruct.WD_window.R & 0x0000000F)){
			return 0;									//success
		}
		else{
			return 1;									//error -> register content NOT verified
		}	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SetWDLFSR sets register with the given value.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sets content of the WD_LFSR register.
* 	@param[in] cmd - Desired register setting placed on the lower 16 bits.
* 	@return 	
*				- "0" - Register was successfuly set. <br>
*				- "1" - Operation failed.
*	@remarks 	Commmand cmd should be a full 16-bit register setting. Parity 
*				setting and security bits in this command can be neglected, 
*				because they will be recomputed by the function automatically.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>	
*			status = PwSBC_SetWDLFSR();			</para>
*			- Command sets WD_LFSR register and stores the returned status. 
*			Security bits and parity bits can be neglected, because they will be 
*			set automatically by the function.
********************************************************************************/
uint32_t PwSBC_SetWDLFSR(uint32_t cmd){
	uint32_t errCode;
	
	errCode =  PwSBC_SendCmdRW(cmd);
	
	if(errCode != 0){						
		return errCode;							//error in the communication
	}
	else
	{
		if((cmd & 0x000000FF) == (INTstruct.WD_LFSR.R & 0x000000FF)){
			return 0;									//success
		}
		else{
			return 1;									//error -> register content NOT verified
		}	
	}
}

/*---------------------------------------------------------------------------\
* INIT PwSBC function
----------------------------------------------------------------------------*\

***************************************************************************//*!
*   @brief The function PwSBC_Init initializes the PwSBC registers.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function initializes PwSBC registers that are marked as 
*					INIT. This initialization is made using default setting 
*					that can be changed in the global defines. Initialization 
*					of the INIT registers is finished by writing to INIT_INT 
*					register. Then the global structures are refreshed by reading 
*					content of all registers and first WD answer is computed and 
*					sent. This WD refresh closes the whole init phase and 
*					function ends successfuly.
*	@remarks 	This function tries to write into each INIT register and verify this 
*				operation. If the write operation does not succeed, then program 
*				tries to repeat the same operation until it succeeds. If the 
*				program is not able to write into any of INIT registers 
*				successfuly, program can stay blocked in an infinite loop till 
*				the first reset (RSTb activation) arrives. This case can occur 
*				in case of any fatal error on the SPI bus is presented (then 
*				reset activation is the only solution).
*	@par Code sample
*			PwSBC_Init();	
*			- This function initializes PwSBC, updates global structures and sends 
*			first WD refresh.	
********************************************************************************/
void PwSBC_Init(){
	uint32_t firstAnswer = 0;
	uint32_t status = 0;
	/* Port for S32-Studio Issue */
	/* to prevent compiler warning */
	status = status;

	//uint32_t resp = 0;
	uint32_t newLFSR = 0;
	
	/* Distinguish INIT phase regarding state of the RST counter */
	
	if(INTstruct.MODE.B.Init == 1){	//Initialize both - FS and MAIN part
		status = PwSBC_SetInitSupervisor1(INITstruct.INIT_Supervisor1.R);		//FS command
		status = PwSBC_SetInitVreg1(INITstruct.INIT_Vreg1.R);		
		status = PwSBC_SetInitSupervisor2(INITstruct.INIT_Supervisor2.R);		//FS command
		status = PwSBC_SetInitVreg2(INITstruct.INIT_Vreg2.R);				
		status = PwSBC_SetInitSupervisor3(INITstruct.INIT_Supervisor3.R);		//FS command
		status = PwSBC_SetInitCAN(INITstruct.INIT_CAN_LIN.R);	
		status = PwSBC_SetInitFSSM1(INITstruct.INIT_FSSM1.R);					//FS command
		status = PwSBC_SetInitIOWU1(INITstruct.INIT_IO_WU1.R);	
		status = PwSBC_SetInitFSSM2(INITstruct.INIT_FSSM2.R);					//FS command
		status = PwSBC_SetInitIOWU2(INITstruct.INIT_IO_WU2.R);	
		status = PwSBC_SetWDwindow(INITstruct.WD_window.R);						//FS command 25Ons between commands
		status = PwSBC_SetWDLFSR(INITstruct.WD_LFSR.R);							//FS command
		status = PwSBC_SetInitINT(INITstruct.INIT_INT.R);
	}
	else{		//Initialize only FS part after RESET
		status = PwSBC_SetInitSupervisor1(INITstruct.INIT_Supervisor1.R);		//FS command 25Ons between commands
		status = PwSBC_SetInitSupervisor2(INITstruct.INIT_Supervisor2.R);		//FS command
		status = PwSBC_SetInitSupervisor3(INITstruct.INIT_Supervisor3.R);		//FS command
		status = PwSBC_SetInitFSSM1(INITstruct.INIT_FSSM1.R);					//FS command
		status = PwSBC_SetInitFSSM2(INITstruct.INIT_FSSM2.R);					//FS command
		status = PwSBC_SetWDwindow(INITstruct.WD_window.R);						//FS command
		status = PwSBC_SetWDLFSR(INITstruct.WD_LFSR.R);							//FS command */
	}

	/* clear undervoltage flags that are 1 by default */ 
	PwSBC_GetDiagVreg1();																							
	PwSBC_GetDiagVreg2();	
	PwSBC_GetDiagVreg3();
	/* clear over-current flags that are 1 on the beginning */
	PwSBC_GetStatusVreg1();
	PwSBC_GetStatusVreg2();
	/* initialize structure by reading all necessary registers, that have not been initialized yet */
	PwSBC_UpdateRegisters();
	
	/* first WD answer */
	newLFSR = PwSBC_GetWDLFSR();
	firstAnswer = PwSBC_ComputeLFSR(newLFSR);				//first WD refresh computations
	PwSBC_RefreshWD(firstAnswer);
	PITstruct.currentLFSR.R = INTstruct.WD_LFSR.R;							//store current LFSR in PIT structure
	PITstruct.WD_answer = firstAnswer;								//make the test computations and store result in PIT structure

	/* Initialize structure with previous values */
	INTstructPrevious.INIT_Vreg1.R = INTstruct.INIT_Vreg1.R;
	INTstructPrevious.INIT_Vreg2.R = INTstruct.INIT_Vreg2.R;
	INTstructPrevious.INIT_CAN_LIN.R = INTstruct.INIT_CAN_LIN.R;
	INTstructPrevious.INIT_IO_WU1.R = INTstruct.INIT_IO_WU1.R;
	INTstructPrevious.INIT_IO_WU2.R = INTstruct.INIT_IO_WU2.R;
	INTstructPrevious.INIT_INT.R = INTstruct.INIT_INT.R;
	INTstructPrevious.HW_Config.R = INTstruct.HW_Config.R;
	INTstructPrevious.WU_source.R = INTstruct.WU_source.R;
	INTstructPrevious.IO_input_state.R = INTstruct.IO_input_state.R;
	INTstructPrevious.Status_Vreg1.R = INTstruct.Status_Vreg1.R;
	INTstructPrevious.Status_Vreg2.R = INTstruct.Status_Vreg2.R;
	INTstructPrevious.Diag_Vreg1.R = INTstruct.Diag_Vreg1.R;
	INTstructPrevious.Diag_Vreg2.R = INTstruct.Diag_Vreg2.R;
	INTstructPrevious.Diag_Vreg3.R = INTstruct.Diag_Vreg3.R;
	INTstructPrevious.Diag_CAN.R = INTstruct.Diag_CAN.R;
	INTstructPrevious.Diag_CAN_LIN.R = INTstruct.Diag_CAN_LIN.R;
	INTstructPrevious.Diag_SPI.R = INTstruct.Diag_SPI.R;
	INTstructPrevious.MODE.R = INTstruct.MODE.R;
	INTstructPrevious.Vreg_mode.R = INTstruct.Vreg_mode.R;
	INTstructPrevious.IO_OUT_AMUX.R = INTstruct.IO_OUT_AMUX.R;
	INTstructPrevious.CAN_LIN_mode.R = INTstruct.CAN_LIN_mode.R;
	INTstructPrevious.Vreg_LPON.R = INTstruct.Vreg_LPON.R;
	INTstructPrevious.INIT_Supervisor1.R = INTstruct.INIT_Supervisor1.R;
	INTstructPrevious.INIT_Supervisor2.R = INTstruct.INIT_Supervisor2.R;
	INTstructPrevious.INIT_Supervisor3.R = INTstruct.INIT_Supervisor3.R;
	INTstructPrevious.INIT_FSSM1.R = INTstruct.INIT_FSSM1.R;
	INTstructPrevious.INIT_FSSM2.R = INTstruct.INIT_FSSM2.R;
	INTstructPrevious.WD_window.R = INTstruct.WD_window.R;
	INTstructPrevious.WD_LFSR.R = INTstruct.WD_LFSR.R;
	INTstructPrevious.WD_answer.R = INTstruct.WD_answer.R;
	INTstructPrevious.Diag_FS1.R = INTstruct.Diag_FS1.R;
	INTstructPrevious.WD_counter.R = INTstruct.WD_counter.R;
	INTstructPrevious.Diag_FS2.R = INTstruct.Diag_FS2.R;

}

/***************************************************************************//*!
*   @brief The function PwSBC_ComputeParity computes a parity for a specified 
*			command and returns the whole command.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function computes odd parity for the given command and 
*					returns modified command.
* 	@param[in] cmd - 16-bit command.
* 	@return 	Modified 16-bit command including odd parity.
*	@remarks 	If a secured command is being used, then the 
*				PwSBC_ComputeSecurityBits function must be applied before 
*				computing parity.
*	@par Code sample
*																		</para>
*			uint32_t cmd_with_parity;									</para>
*			cmd_with_parity = PwSBC_ComputeParity(0b1010111000000001);	</para>
*			- This function computes parity for the command (IO_out - AMUX) and 
*			returns its modified version in the variable cmd_with_ parity.	
********************************************************************************/
uint32_t PwSBC_ComputeParity(uint32_t cmd){
		uint8_t sum = 0;
		uint8_t i = 0;		
		
		for(i = 0;i < 16;i++){
			if(i != 8){
				sum = sum + (uint8_t)((cmd >> i) & 0x00000001);
			}
		}
		if(sum%2 == 0){			//even number -> P = 1
			cmd = cmd | 0x00000100;
		}
		else{
			cmd = cmd & 0xFFFFFEFF;		
		}
		return cmd;
}

/***************************************************************************//*!
*   @brief The function PwSBC_ComputeSecurityBits computes the security bits for a specified command and returns 
*			the whole command.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function computes 4 security bits for the given command and 
*					returns modified command.
* 	@param[in] cmd - 16-bit command.
* 	@return 	Modified 16-bit command including 4 security bits.
*	@remarks 	This function should be used only for secure commands and has 
*				to be used before applying PwSBC_ComputeParity function.
*	@par Code sample
*																	</para>
*			uint32_t cmd_secured;									</para>
*			cmd_secured = PwSBC_ComputeParity(0b1010101001000000);	</para>
*			- This function computes security bits for the command (MODE) and returns 
*			its modified version in the variable cmd_secured.	
********************************************************************************/
uint32_t PwSBC_ComputeSecurityBits(uint32_t cmd){
		register32_struct command;
		
		command.R = cmd;
		command.B.bit0 = command.B.bit6;
		command.B.bit1 = command.B.bit7;
		command.B.bit2 = ~command.B.bit4;
		command.B.bit3 = ~command.B.bit5;
		
		return command.R;
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetVoltageTight reads a value from the ADC and recomputes it into a voltage considering 
*			a tight voltage range ratio.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function reads value from ADC and recomputes it into voltage 
*					considering tight voltage range ratio.
* 	@return 	Voltage [V] that was measured on the active AMUX channel.
*				The returned value is computed as follows:	
*				(Vcca/ADC_RESOLUTION)*x*VAMUX_TG_X, where:  
*				- Vcca - if the parameter ADC_SOURCE_CALIB is defined in 
*				the global defines, then its value is used as a Vcca. If this 
*				parameter is not defined, then either 3.3 or 5.0 V is used 
*				instead depending on the Vcca_HW bit. This bit is checked and 
*				the appropriate voltage is chosen automatically by this function.
*				- ADC_RESOLUTION - this parameter is defined in the global 
*				defines. By default for the MPC5643L the 12-bit ADC is used, 
*				so the number (2^12)-1 = 4095 corresponds to this parameter.
*				- x - number obtained from ADC in the 12-bit range (value from 
*				0 till 4095)
*				- VAMUX_TG_X - divider ration tight input voltage range. 	
*				(VAMUX_TG_X = 2 for Vcca = 5 V; VAMUX_TG_X = 3 for Vcca = 3.3 V)
*	@remarks 	This function works only when the tight range is being used. 
*				This range allows to measure voltages from 2 till 10 V. If 
*				a wider range is necessary, then PwSBC_GetVoltageWide should be 
*				used instead.
*				- Example of computations (Vcca = 3.3 V, ADC_SOURCE_CALIB is not 
*				defined, ADC_RESOLUTION = 4095 by default, VAMUX_TG_X = 3,
*				x = 1089):	(3.3/4095)*1089*3 = 2.63V
*	@par Code sample
*													</para>
*			float voltage;							</para>
*			voltage = PwSBC_GetVoltageTight();		</para>
*			- This function reads value from ADC (depending on actual AMUX channel) 
*			and returns value as a voltage [V].	
********************************************************************************/
float PwSBC_GetVoltageTight(){
	double voltage = 0;
	uint32_t x;
	float treshold = 4.15;
	/* Port for S32-Studio Issue */
	/* to prevent compiler warning */
	treshold = treshold;
	
	x = ADC_GetChannelValue(ADC_NB, ADC_CH);
	#ifdef ADC_SOURCE_CALIB																	//reference voltage for ADC is stored in ADC_SOURCE_CALIB
		voltage = (ADC_SOURCE_CALIB / ADC_RESOLUTION) * x;
		if(ADC_SOURCE_CALIB < treshold){													//choose between 3.3V and 5V ratio for computations
			voltage *= VAMUX_TG_33;			
		}
		else{
			voltage *= VAMUX_TG_5;
		}
	#else																										//reference voltage is set to Vcca
		 if(INTstruct.HW_Config.B.Vcca_HW == 0){							//choose between 3.3V and 5V used as Vcca
		 		voltage = (3.3 / ADC_RESOLUTION) * x;
		 		voltage *= VAMUX_TG_33;		
		 }
		 else{
		 		voltage = (5.0 / ADC_RESOLUTION) * x;
		 		voltage *= VAMUX_TG_5;		
		 }
	#endif			
	
	return (float)voltage;	
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetVoltageWide reads a value from ADC and recomputes it into a voltage considering 
*			a wide voltage range ratio.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function reads value from ADC and recomputes it into voltage 
*					considering wide voltage range ratio.
* 	@return 	Voltage [V] that was measured on the active AMUX channel. 
*				The returned value is computed as follows:
*				(Vcca/ADC_RESOLUTION)*x*VAMUX_TG_X, where:
*				- Vcca - if the parameter ADC_SOURCE_CALIB is defined in 
*				the global defines, then its value is used as a Vcca. If this 
*				parameter is not defined, then either 3.3 or 5.0 V is used 
*				instead depending on the Vcca_HW bit. This bit is checked 
*				and the appropriate voltage is chosen automatically by this 
*				function.
*				- ADC_RESOLUTION - this parameter is defined in the global 
*				defines. By default for the MPC5643L the 12-bit ADC is used, 
*				so the number (2^12)-1 = 4095 corresponds to this parameter.
*				- x - number obtained from ADC in the 12-bit range (value from 0 
*				till 4095)
*				- VAMUX_WD_X - divider ration tight input voltage range 
*				(VAMUX_WD_X = 5 for Vcca = 5 V; VAMUX_WD_X = 7 for Vcca = 3.3 V)
*	@remarks 	This function works for wide range of voltages 
*				(from 2 till 24 V). If the voltage is less than 10 V than usage 
*				of tight range (function PwSBC_GetVoltageTight) will be more 
*				precise.
*				- Example of computations (Vcca = 3.3 V, ADC_SOURCE_CALIB is 
*				not defined, ADC_RESOLUTION = 4095 by default, VAMUX_WD_X = 7, 
*				x = 3054): (3.3/4095)*3054*7 = 17.23V
*	@par Code sample
*													</para>
*			float voltage;							</para>
*			voltage = PwSBC_GetVoltageWide();		</para>
*			- This function reads value from ADC (depending on actual AMUX channel) 
*			and returns value as a voltage [V].	
********************************************************************************/
float PwSBC_GetVoltageWide(){
	double voltage = 0;
	uint32_t x;
	float treshold = 4.15;
	/* Port for S32-Studio Issue */
	/* to prevent compiler warning */
	treshold = treshold;
	
	x = ADC_GetChannelValue(ADC_NB, ADC_CH);
	#ifdef ADC_SOURCE_CALIB																	//reference voltage for ADC is stored in ADC_SOURCE_CALIB
		voltage = (ADC_SOURCE_CALIB / ADC_RESOLUTION) * x;
		if(ADC_SOURCE_CALIB < treshold){													//choose between 3.3V and 5V ratio for computations
			voltage *= VAMUX_WD_33;			
		}
		else{
			voltage *= VAMUX_WD_5;
		}
	#else																										//reference voltage is set to Vcca
		 if(INTstruct.HW_Config.B.Vcca_HW == 0){							//choose between 3.3V and 5V used as Vcca
		 		voltage = (3.3 / ADC_RESOLUTION) * x;
		 		voltage *= VAMUX_WD_33;		
		 }
		 else{
		 		voltage = (5.0 / ADC_RESOLUTION) * x;
		 		voltage *= VAMUX_WD_5;		
		 }
	#endif			
	
	return (float)voltage;	
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetVoltage reads a value from the ADC and recomputes it into a voltage considering 
*			no voltage ratio.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function reads value from ADC and recomputes it into voltage 
*					considering no voltage ratio.
* 	@return 	Voltage [V] that was measured on the active AMUX channel. 
*				The returned value is computed as follows:
*				(Vcca/ADC_RESOLUTION)*x, where:
*				- Vcca - if the parameter ADC_SOURCE_CALIB is defined in 
*				the global defines, then its value is used as a Vcca. 
*				If this parameter is not defined, then either 3.3 or 5.0 V 
*				is used instead depending on the Vcca_HW bit. This bit is 
*				checked and the appropriate voltage is chosen automatically 
*				by this function.
*				- ADC_RESOLUTION - this parameter is defined in the global 
*				defines. By default for the MPC5643L the 12-bit ADC is used, 
*				so the number (2^12)-1 = 4095 corresponds to this parameter.
*				- x - number obtained from ADC in the 12-bit range (value from 
*				0 till 4095)
*	@remarks 	This function should be used for voltages between 0 and 
*				Vcca - voltages that does not use any ratio (Vref, Die 
*				temperature sensor). For other volatges please use either 
*				PwSBC_GetVoltageTight or PwSBC_GetVoltageWide.
*				- Example of computations (Vcca = 5.0 V, ADC_SOURCE_CALIB is not 
*				defined, ADC_RESOLUTION = 4095 by default, x = 2056):
*				(5.0/4095)*2056 = 2.51V
*	@par Code sample
*												</para>
*			float voltage;						</para>
*			voltage = PwSBC_GetVoltage();		</para>	
*			- This function reads value from ADC and returns value as a voltage [V].	
********************************************************************************/
float PwSBC_GetVoltage(){
	double voltage = 0;
	uint32_t x;
	
	x = ADC_GetChannelValue(ADC_NB, ADC_CH);
	#ifdef ADC_SOURCE_CALIB																	//reference voltage for ADC is stored in ADC_SOURCE_CALIB
		voltage = (ADC_SOURCE_CALIB / ADC_RESOLUTION) * x;
	#else																										//reference voltage is set to Vcca
		 if(INTstruct.HW_Config.B.Vcca_HW == 0){							//choose between 3.3V and 5V used as Vcca
		 		voltage = (3.3 / ADC_RESOLUTION) * x;
		 }
		 else{
		 		voltage = (5.0 / ADC_RESOLUTION) * x;
		 }
	#endif			
	
	return (float)voltage;	
}

/***************************************************************************//*!
*   @brief The function PwSBC_GetTemperature reads a value from the ADC and recomputes it into a temperature 
*			(no voltage ratio).
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function reads value from ADC and recomputes it into 
*					temperature.
* 	@return 	Temperature in degrees of Celsius. Temerature is computed using 
*				the following formula:
*				t = ((V*1000)-VAMUX_TP_0)/VAMUX_TP_C0, where:
*				- V - voltage that is computed as shown in the function 
*				PwSBC_GetVoltage.
*				- VAMUX_TP_0 - typical voltage for zero temperature (0 degrees 
*				of Celsius)
*				- VAMUX_TP_CO - temperature coefficient for the implemented 
*				sensor
*	@remarks 	Temperature coefficients are set by default in the global 
*				defines.
*				- Example of computations (Vcca = 5.0 V, ADC_SOURCE_CALIB is not 
*				defined, ADC_RESOLUTION = 4095 by default, x = 2056, V = 2.51 V, 
*				VAMUX_TP_0 = 850 mV, VAMUX_TP_CO = 10 mV/degrees of Celsius):
*				t = ((2.51*1000)-850)10 = 166 deg.
*	@par Code sample
*											</para>
*			float temp;						</para>
*			temp = PwSBC_GetTemperature();	</para>
*			- This function reads value from ADC and returns it as a die temperature.	
********************************************************************************/
float PwSBC_GetTemperature(){
	double voltage = 0;
	double temp = 0;
	uint32_t x;
	
	x = ADC_GetChannelValue(ADC_NB, ADC_CH);
	#ifdef ADC_SOURCE_CALIB																	//reference voltage for ADC is stored in ADC_SOURCE_CALIB
		voltage = (ADC_SOURCE_CALIB / ADC_RESOLUTION) * x;
	#else																										//reference voltage is set to Vcca
		 if(INTstruct.HW_Config.B.Vcca_HW == 0){							//choose between 3.3V and 5V used as Vcca
		 		voltage = (3.3 / ADC_RESOLUTION) * x;
		 }
		 else{
		 		voltage = (5.0 / ADC_RESOLUTION) * x;
		 }
	#endif			
	
	// Temperature computations from input voltage:
	// temp				temperature in degrees of Celsius
	// VAMUX_TP_0		nominal voltage at 0 degrees of Celsius
	// VAMUX_TP_CO		coefficient in mV/degrees of Celsius
	
	temp = ((voltage * 1000) - VAMUX_TP_0)/VAMUX_TP_CO;
	
	return (float)temp;	
}

/***************************************************************************//*!
*   @brief The function PwSBC_ProcessSPI treats the data received on the SPI MISO line.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function reads word received on the SPI MISO line, 
*					decodes the frame address and stores received word in 
*					the global structures.
*	@remarks 	This function is called everytime by the PwSBC_SendCmdXX and 
*				PwSBC_SendSecureCmdXX. This calling causes that the global 
*				structures are updated with every command sent to the PwSBC.
*	@par Code sample
*			PwSBC_ProcessSPI();	
*			- This function treats received command and stores its content in 
*			the global structures.	
********************************************************************************/
void PwSBC_ProcessSPI(){

	uint32_t address = 0;

	SPIstruct.response = DSPI_Read(DSPI_NB);
	SPIstruct.statusPwSBC.R = SPIstruct.response >> 8;
																		
	address = (SPIstruct.readCmd & 0x00007E00) >> 9;									//mask register address from the read command
	switch(address){
		case	INIT_VREG1_ADR				:	INTstruct.INIT_Vreg1.R = SPIstruct.response; break;
		case	INIT_VREG2_ADR				:	INTstruct.INIT_Vreg2.R = SPIstruct.response; break;
		case	INIT_CAN_LIN_ADR			:	INTstruct.INIT_CAN_LIN.R = SPIstruct.response; break;
		case	INIT_IO_WU1_ADR				:	INTstruct.INIT_IO_WU1.R = SPIstruct.response; break;
		case	INIT_IO_WU2_ADR				:	INTstruct.INIT_IO_WU2.R = SPIstruct.response; break;
		case	INIT_INT_ADR				:	INTstruct.INIT_INT.R = SPIstruct.response; break;
		case	HW_CONFIG_ADR				:	INTstruct.HW_Config.R = SPIstruct.response; break;
		case	WU_SOURCE_ADR				:	INTstruct.WU_source.R = SPIstruct.response; break;
		case	IO_INPUT_STATE_ADR			:	INTstruct.IO_input_state.R = SPIstruct.response; break;
		case	STATUS_VREG1_ADR			:	INTstruct.Status_Vreg1.R = SPIstruct.response; break;
		case	STATUS_VREG2_ADR			:	INTstruct.Status_Vreg2.R = SPIstruct.response; break;
		case	DIAG_VREG1_ADR				:	INTstruct.Diag_Vreg1.R = SPIstruct.response; break;
		case	DIAG_VREG2_ADR				:	INTstruct.Diag_Vreg2.R = SPIstruct.response; break;
		case	DIAG_VREG3_ADR				:	INTstruct.Diag_Vreg3.R = SPIstruct.response; break;
		case	DIAG_CAN_ADR				:	INTstruct.Diag_CAN.R = SPIstruct.response; break;
		case	DIAG_CAN_LIN_ADR			:	INTstruct.Diag_CAN_LIN.R = SPIstruct.response; break;
		case	DIAG_SPI_ADR				:	INTstruct.Diag_SPI.R = SPIstruct.response; break;
		case	MODE_ADR					:	INTstruct.MODE.R = SPIstruct.response; break;
		case	VREG_MODE_ADR				:	INTstruct.Vreg_mode.R = SPIstruct.response; break;
		case	IO_OUT_AMUX_ADR				:	INTstruct.IO_OUT_AMUX.R = SPIstruct.response; break;
		case	CAN_LIN_MODE_ADR			:	INTstruct.CAN_LIN_mode.R = SPIstruct.response; break;
		case	VREG_LPON_ADR				:	INTstruct.Vreg_LPON.R = SPIstruct.response; break;
		case	INIT_SUPERVISOR1_ADR		:	INTstruct.INIT_Supervisor1.R = SPIstruct.response; break;
		case	INIT_SUPERVISOR2_ADR		:	INTstruct.INIT_Supervisor2.R = SPIstruct.response; break;
		case	INIT_SUPERVISOR3_ADR		:	INTstruct.INIT_Supervisor3.R = SPIstruct.response; break;
		case	INIT_FSSM1_ADR				:	INTstruct.INIT_FSSM1.R = SPIstruct.response; break;
		case	INIT_FSSM2_ADR				:	INTstruct.INIT_FSSM2.R = SPIstruct.response; break;
		case	WD_WINDOW_ADR				:	INTstruct.WD_window.R = SPIstruct.response; break;
		case	WD_LFSR_ADR					:	INTstruct.WD_LFSR.R = SPIstruct.response; break;
		case	WD_ANSWER_ADR				:	INTstruct.WD_answer.R = SPIstruct.response; break;
		case	DIAG_FS1_ADR				:	INTstruct.Diag_FS1.R = SPIstruct.response; break;
		case	WD_COUNTER_ADR				:	INTstruct.WD_counter.R = SPIstruct.response; break;
		case	DIAG_FS2_ADR				:	INTstruct.Diag_FS2.R = SPIstruct.response; break;	
	}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SendCmdR sends a read command and waits until the end of transmission.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sends a read command to the PwSBC and waits 
*					the end of transmission. Data received on the MISO pin are 
*					decoded and the global structure is refreshed using this 
*					value.
* 	@param[in] cmd - 	16-bit read command (parity bit should be 0 for all read 
*						commands).
* 	@return 	
*				- "0" - Command was sent without any error. <br>
*				- "10" - SPI disconnected or no SPI answer (in decimal).<br>
*				- "11" - SPI_G error detected (in decimal).
*	@remarks 	This presents the lowest level of the PwSBC driver to send 
*				commands to PwSBC and is being used by the other functions. 
*				This function is designed just for read commands.
*	@par Code sample
*													</para>
*				uint32_t status;					</para>
*				status = PwSBC_SendCmdR(0x2000);	</para>
*				- This function sends read command Diag Vreg3 and stores returned 
*				status.	
********************************************************************************/
uint32_t PwSBC_SendCmdR(uint32_t cmd){
		uint32_t stockPriority = 0;
		  
//		stockPriority = INTC.CPR.B.PRI;			//save current priority
//		INTC.CPR.B.PRI = INT_CEIL_PRIORITY;		//block DSPI resource
		stockPriority = INTC_0.CPR[0].B.PRI;			//save current priority
		INTC_0.CPR[0].B.PRI = INT_CEIL_PRIORITY;		//block DSPI resource
		SPIstruct.writeCmd = 0;					//NO write cmd
		SPIstruct.readCmd = cmd;				//set read cmd
				
		DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.readCmd);	//send the read command, function waits for result
		PwSBC_ProcessSPI();									//read received cmd and save it in the global structure
		
		if(SPIstruct.statusPwSBC.B.SPI_G == 1){
			if(SPIstruct.response == 0xFFFF){	
				SPIstruct.writeCmd = 0;								//NO write cmd
				SPIstruct.readCmd = (DIAG_SPI_ADR <<9);				//set read cmd to Diag SPI command
				DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.readCmd);	//send the read command, function waits for result
				PwSBC_ProcessSPI();
				if((INTstruct.Diag_SPI.B.bit0 & INTstruct.Diag_SPI.B.bit2) == 0){
					INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
					return 11;										//error -> SPI_G error					
				}
				else{
					INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
					PwSBC_HandleSPI_NO_COM(); 						//handle disconnected SPI
					return 10;										//error -> no answer on SPI
				}
			}
			else{
				INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
				return 11;										//error -> SPI_G error
			}
		}
		else{
			INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
			return 0;
		}

}

/***************************************************************************//*!
*   @brief The function PwSBC_SendCmdW sends a write command and waits until the end of transmission.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sends a write command to the PwSBC and waits 
*					the end of transmission. Data received on the MISO pin are 
*					decoded and the global structure is refreshed using this 
*					value.
* 	@param[in] cmd - 	16-bit write command. Parity bit will be computed and 
*						set automatically by this function.
* 	@return 	
*				- "0" - Command was sent without any error. <br>
*				- "10" - SPI disconnected or no SPI answer (in decimal).<br>
*				- "11" - SPI_G error detected (in decimal).
*	@remarks 	This presents the lowest level of the PwSBC driver to send 
*				commands to PwSBC and is being used by the other functions. 
*				This function simply sends a write command, but does not apply 
*				any verification after writing.
*	@par Code sample
*												</para>	
*			uint32_t status;					</para>
*			status = PwSBC_SendCmdW(0x2000);	</para>
*			- This function sends read command Diag Vreg3 and stores returned status.	
********************************************************************************/
uint32_t PwSBC_SendCmdW(uint32_t cmd){
		uint32_t stockPriority = 0;
		  
		stockPriority = INTC_0.CPR[0].B.PRI;			//save current priority
		INTC_0.CPR[0].B.PRI = INT_CEIL_PRIORITY;		//block DSPI resource
	
		SPIstruct.writeCmd = cmd;
		SPIstruct.writeCmd = PwSBC_ComputeParity(SPIstruct.writeCmd);
		
		DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.writeCmd);		//send the write command, function waits for result
  		PwSBC_ProcessSPI();											//read received cmd and save it in the global structure
		
		if(SPIstruct.statusPwSBC.B.SPI_G == 1){
			if(SPIstruct.response == 0xFFFF){
				SPIstruct.writeCmd = 0;								//NO write cmd
				SPIstruct.readCmd = (DIAG_SPI_ADR <<9);				//set read cmd to Diag SPI command
				DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.readCmd);	//send the read command, function waits for result
				PwSBC_ProcessSPI();
				if((INTstruct.Diag_SPI.B.bit0 & INTstruct.Diag_SPI.B.bit2) == 0){
					INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
					return 11;										//error -> SPI_G error					
				}
				else{
					INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
					PwSBC_HandleSPI_NO_COM(); 						//handle disconnected SPI
					return 10;										//error -> no answer on SPI
				}
			}
			else{
				INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
				return 11;										//error -> SPI_G error
			}
		}
		else{
			INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
			return 0;
		}
}	


/***************************************************************************//*!
*   @brief The function PwSBC_SendCmdRW	sends a write and then a read command and waits until the end 
*			of transmission.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sends a write command followed by a read command 
*					to PwSBC. Data sent on the MISO during the read command is 
*					stocked in global structure, so it can be used by functions 
*					on higher level to verify register content (if the write 
*					command was applied correctly or not).
* 	@param[in] cmd - 	16-bit write or read command (function recomputes 
*						the R/W bit automatically so either read or write 
*						command can be given). Parity bit will be computed 
*						and set automatically by this function.
* 	@return 	
*				- "0" - Command was sent without any error. <br>
*				- "10" - SPI disconnected or no SPI answer (in decimal).<br>
*				- "11" - SPI_G error detected (in decimal).
*	@remarks 	This presents the lowest level of the PwSBC driver to send 
*				commands to PwSBC and is being used by the other functions. 
*				This function should be used for Read/Write commands.
*	@par Code sample
*												</para>
*			uint32_t status;					</para>
*			status = PwSBC_SendCmdRW(0x8544);	</para>
*			- This function sets INIT VREG2 register and reads its content to allow 
*			verification.	
********************************************************************************/
	uint32_t  PwSBC_SendCmdRW(uint32_t cmd){
			uint32_t stockPriority = 0;
			  
			stockPriority = INTC_0.CPR[0].B.PRI;			//save current priority
			INTC_0.CPR[0].B.PRI = INT_CEIL_PRIORITY;		//block DSPI resource
			
			SPIstruct.writeCmd = cmd | 0x8000;							//create write cmd
			SPIstruct.readCmd = cmd & 0x7E00;							//create read cmd
			SPIstruct.writeCmd = PwSBC_ComputeParity(SPIstruct.writeCmd);
			
			DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.writeCmd);			//send the write command, function waits for result
			SPIstruct.response = DSPI_Read(DSPI_NB);					//read result and release inp. buffer
			SPIstruct.statusPwSBC.R = SPIstruct.response >> 8;			
			
			DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.readCmd);				//send the read command, function waits for result
			PwSBC_ProcessSPI();												//read received cmd and save it in the global structure
			
			if(SPIstruct.statusPwSBC.B.SPI_G == 1){
				if(SPIstruct.response == 0xFFFF){
					SPIstruct.writeCmd = 0;								//NO write cmd
					SPIstruct.readCmd = (DIAG_SPI_ADR <<9);				//set read cmd to Diag SPI command
					DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.readCmd);	//send the read command, function waits for result
					PwSBC_ProcessSPI();
					if((INTstruct.Diag_SPI.B.bit0 & INTstruct.Diag_SPI.B.bit2) == 0){
						INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
						return 11;										//error -> SPI_G error					
					}
					else{
						INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
						PwSBC_HandleSPI_NO_COM(); 						//handle disconnected SPI 
						return 10;										//error -> no answer on SPI
					}
				}
				else{
					INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
					return 11;										//error -> SPI_G error
				}
			}
			else{
				INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
				return 0;
			}
	}


/***************************************************************************//*!
*   @brief The function PwSBC_SendSecureCmdRW sends a write command with secure bits and then a read command 
			and waits until the end of transmission.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sends a secured write command (commands that use 
*					four security bits) followed by a read command to PwSBC. 
*					Data sent on the MISO during the read command is stocked in 
*					global structure, so it can be used by functions on higher 
*					level to verify register content (if the write command was 
*					applied correctly or not).
* 	@param[in] cmd - 	16-bit write or read command (function recomputes 
*						the R/W bit automatically so either read or write command 
*						can be given). Parity bit and security bits will be 
*						computed and set automatically by this function.
* 	@return 	
*				- "0" - Command was sent without any error. <br>
*				- "10" - SPI disconnected or no SPI answer (in decimal).<br>
*				- "11" - SPI_G error detected (in decimal).
*	@remarks 	This presents the lowest level of the PwSBC driver to send 
*				commands to PwSBC and is being used by the other functions. 
*				This function should be used for secured Read/Write commands.
*	@par Code sample
*														</para>
*			uint32_t status;							</para>
*			status = PwSBC_SendSecureCmdRW(0xAB8E);		</para>
*			- This function sets Normal mode for the PwSBC and verifies 
*			the operation.	
********************************************************************************/
uint32_t PwSBC_SendSecureCmdRW(uint32_t cmd){
		uint32_t stockPriority = 0;
		  
		stockPriority = INTC_0.CPR[0].B.PRI;			//save current priority
		INTC_0.CPR[0].B.PRI = INT_CEIL_PRIORITY;	//block DSPI resource
		
		SPIstruct.writeCmd = cmd | 0x8000;									//create write cmd
		SPIstruct.readCmd = cmd & 0x7E00;									//create read cmd
		

		SPIstruct.writeCmd = PwSBC_ComputeSecurityBits(SPIstruct.writeCmd);
		SPIstruct.writeCmd = PwSBC_ComputeParity(SPIstruct.writeCmd);
		
		DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.writeCmd);					//send the write command, function waits for result
		SPIstruct.response = DSPI_Read(DSPI_NB);							//read result and release inp. buffer
		SPIstruct.statusPwSBC.R = SPIstruct.response >> 8;			
		
		DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.readCmd);						//send the read command, function waits for result
		PwSBC_ProcessSPI();														//read received cmd and save it in the global structure
		
		if(SPIstruct.statusPwSBC.B.SPI_G == 1){
			if(SPIstruct.response == 0xFFFF){
				SPIstruct.writeCmd = 0;								//NO write cmd
				SPIstruct.readCmd = (DIAG_SPI_ADR <<9);				//set read cmd to Diag SPI command
				DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.readCmd);	//send the read command, function waits for result
				PwSBC_ProcessSPI();
				if((INTstruct.Diag_SPI.B.bit0 & INTstruct.Diag_SPI.B.bit2) == 0){
					INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
					return 11;										//error -> SPI_G error					
				}
				else{
					INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
					PwSBC_HandleSPI_NO_COM(); 						//handle disconnected SPI
					return 10;										//error -> no answer on SPI
				}
			}
			else{
				INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
				return 11;										//error -> SPI_G error
			}
		}
		else{
			INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
			return 0;
		}
}

/***************************************************************************//*!
*   @brief The function PwSBC_SendSecureCmdW sends a write command with secure bits and waits until the end 
*			of transmission.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function sends a secured write command (commands that use 
*					four security bits) to the PwSBC and waits the end of 
*					transmission. Data received on the MISO pin are decoded 
*					and the global structure is refreshed using this value.
* 	@param[in] cmd - 	16-bit write command. Parity bit and four security bits 
*						will be computed and set automatically by this function.
* 	@return 	
*				- "0" - Command was sent without any error. <br>
*				- "10" - SPI disconnected or no SPI answer (in decimal).<br>
*				- "11" - SPI_G error detected (in decimal).
*	@remarks 	This presents the lowest level of the PwSBC driver to send 
*				commands to PwSBC and is being used by the other functions. 
*				This function simply sends a write command, but does not apply 
*				any verification after writing.
*	@par Code sample
*														</para>
*			uint32_t status;							</para>
*			status = PwSBC_SendSecureCmdRW(0xAB8E);		</para>
*			- This function sets Normal mode for the PwSBC without any verification.	
********************************************************************************/
uint32_t PwSBC_SendSecureCmdW(uint32_t cmd){
		uint32_t stockPriority = 0;
				
			stockPriority = INTC_0.CPR[0].B.PRI;			//save current priority
			INTC_0.CPR[0].B.PRI = INT_CEIL_PRIORITY;	//block DSPI resource
		
			SPIstruct.writeCmd = cmd;
			SPIstruct.writeCmd = PwSBC_ComputeSecurityBits(SPIstruct.writeCmd);
			SPIstruct.writeCmd = PwSBC_ComputeParity(SPIstruct.writeCmd);
			
			DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.writeCmd);				//send the write command, function waits for result
			PwSBC_ProcessSPI();													//read received cmd and save it in the global structure
			
			if(SPIstruct.statusPwSBC.B.SPI_G == 1){
				if(SPIstruct.response == 0xFFFF){
					SPIstruct.writeCmd = 0;								//NO write cmd
					SPIstruct.readCmd = (DIAG_SPI_ADR <<9);				//set read cmd to Diag SPI command
					DSPI_Send(DSPI_NB, DSPI_CS, SPIstruct.readCmd);	//send the read command, function waits for result
					PwSBC_ProcessSPI();
					if((INTstruct.Diag_SPI.B.bit0 & INTstruct.Diag_SPI.B.bit2) == 0){
						INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
						return 11;										//error -> SPI_G error					
					}
					else{
						INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
						PwSBC_HandleSPI_NO_COM(); 						//handle disconnected SPI 
						return 10;										//error -> no answer on SPI
					}
				}
				else{
					INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
					return 11;										//error -> SPI_G error
				}
			}
			else{
				INTC_0.CPR[0].B.PRI = stockPriority;					//release DSPI resource
				return 0;
			}
}


/*---------------------------------------------------------------------------\
* EVENT handlers
\****************************************************************************/

/**
* event handlers for the Diag_SPI register
*/

/***************************************************************************//*!
*   @brief The function PwSBC_HandleSPI_err handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleSPI_err();	
*			- This function handles SPI_ERR event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleSPI_err(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'S','P','I','_','e','r','r'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleSPI_CLK handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleSPI_CLK();	
*			- This function handles SPI_CLK event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleSPI_CLK(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'S','P','I','_','C','L','K'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleSPI_Req handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleSPI_Req();	
*			- This function handles SPI_REQ event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleSPI_Req(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'S','P','I','_','R','e','q'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleSPI_Parity handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleSPI_Parity();	
*			- This function handles SPI_PARITY event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleSPI_Parity(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[10] = {'S','P','I','_','P','a','r','i','t','y'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 10);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}


/**
* event handlers for the WU_source register
*/

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIO_5_WU handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIO_5_WU();	
*			- This function handles IO_5_WU event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIO_5_WU(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'I','0','_','5','_','W','U'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIO_4_WU handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIO_4_WU();	
*			- This function handles IO_4_WU event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIO_4_WU(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'I','0','_','4','_','W','U'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIO_3_WU handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIO_3_WU();	
*			- This function handles IO_3_WU event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIO_3_WU(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'I','0','_','3','_','W','U'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIO_2_WU handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIO_2_WU();	
*			- This function handles IO_2_WU event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIO_2_WU(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'I','0','_','2','_','W','U'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIO_1_WU handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIO_1_WU();	
*			- This function handles IO_1_WU event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIO_1_WU(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'I','0','_','1','_','W','U'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIO_0_WU handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIO_0_WU();	
*			- This function handles IO_0_WU event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIO_0_WU(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'I','0','_','0','_','W','U'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/**
* event handlers for the Status_Vreg1 register
*/

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIpFF handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIpFF();	
*			- This function handles IPFF event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIpFF(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[4] = {'I','p','f','f'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 4);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIlim_pre handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIlim_pre();	
*			- This function handles ILIM_PRE event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIlim_pre(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[8] = {'I','l','i','m','_','p','r','e'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 8);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleTwarn_pre handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleTwarn_pre();	
*			- This function handles TWARN_PRE event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleTwarn_pre(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[9] = {'T','w','a','r','n','_','p','r','e'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 9);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleBoB handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleBoB();	
*			- This function handles BOB event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleBoB(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[3] = {'B','o','B'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 3);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleVpre_state handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			TEST();	
*			- This function handles TEST event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleVpre_state(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[10] = {'V','p','r','e','_','s','t','a','t','e'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 10);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}



/*
* event handlers for the CAN_LIN_mode register
*/

/***************************************************************************//*!
*   @brief The function PwSBC_HandleCAN_wu handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleCAN_wu();	
*			- This function handles CAN_WU event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleCAN_wu(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[6] = {'C','A','N','_','W','U'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 6);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}



/**
* event handlers for the Diag_CAN register
*/

/***************************************************************************//*!
*   @brief The function PwSBC_HandleCANH_batt handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleCANH_batt();	
*			- This function handles CANH_BATT event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleCANH_batt(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[9] = {'C','A','N','H','_','b','a','t','t'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 9);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleCANH_gnd handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleCANH_gnd();	
*			- This function handles CANH_GND event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleCANH_gnd(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[8] = {'C','A','N','H','_','g','n','d'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 8);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleCANL_batt handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleCANL_batt();	
*			- This function handles CANL_BATT event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleCANL_batt(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[9] = {'C','A','N','L','_','b','a','t','t'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 9);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleCANL_gnd handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleCANL_gnd();	
*			- This function handles CANL_GND event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleCANL_gnd(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[8] = {'C','A','N','L','_','g','n','d'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 8);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}
/***************************************************************************//*!
*   @brief The function PwSBC_HandleCAN_dominant handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleCAN_dominant();	
*			- This function handles CANH_DOMINANT event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleCAN_dominant(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[12] = {'C','A','N','_','d','o','m','i','n','a','n','t'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 12);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleRXD_recessive handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleRXD_recessive();	
*			- This function handles RXD_RECESSIVE event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleRXD_recessive(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[13] = {'R','X','D','_','r','e','c','e','s','s','i','v','e'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 13);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleTXD_dominant handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleTXD_dominant();	
*			- This function handles TXD_DOMINANT event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleTXD_dominant(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[12] = {'T','X','D','_','d','o','m','i','n','a','n','t'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 12);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/**
* event handlers for the Diag_CAN_LIN register
*/

/***************************************************************************//*!
*   @brief The function PwSBC_HandleTXDL_dominant handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleTXDL_dominant();	
*			- This function handles TXDL_DOMINANT event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleTXDL_dominant(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[13] = {'T','X','D','L','_','d','o','m','i','n','a','n','t'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 13);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}


/***************************************************************************//*!
*   @brief The function PwSBC_HandleCAN_OT handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleCAN_OT();	
*			- This function handles CAN_OT event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleCAN_OT(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[6] = {'C','A','N','_','O','T'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 6);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleCAN_OC handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleCAN_OC();	
*			- This function handles CAN_OC event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleCAN_OC(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[6] = {'C','A','N','_','O','C'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 6);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/**
* event handlers for the IO_input_state register
*/

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIO_5 handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIO_5();	
*			- This function handles IO_5 event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIO_5(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[4] = {'I','O','_','5'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 4);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIO_4 handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIO_4();	
*			- This function handles IO_4 event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIO_4(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[4] = {'I','O','_','4'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 4);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIO_3 handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIO_3();	
*			- This function handles IO_3 event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIO_3(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[4] = {'I','O','_','3'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 4);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIO_2 handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIO_2();	
*			- This function handles IO_2 event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIO_2(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'I','O','_','2'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 4);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIO_1 handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIO_1();	
*			- This function handles IO_1 event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIO_1(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[4] = {'I','O','_','1'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 4);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIO_0 handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIO_0();	
*			- This function handles IO_0 event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIO_0(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[4] = {'I','O','_','0'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 4);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

//
// event handlers for the Diag_Vreg1 register
//

/***************************************************************************//*!
*   @brief The function PwSBC_HandleVsns_uv handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleVsns_uv();	
*			- This function handles VSNS_UV event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleVsns_uv(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'V','s','n','s','_','u','v'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}
	
/***************************************************************************//*!
*   @brief The function PwSBC_HandleVsup_uv_H handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleVsup_uv_H();	
*			- This function handles VSUP_UV_H event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleVsup_uv_H(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[9] = {'V','s','u','p','_','u','v','_','H'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 9);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleVpre_OV handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleVpre_OV();	
*			- This function handles VPRE_OV event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleVpre_OV(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'V','p','r','e','_','O','V'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleVpre_UV handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleVpre_UV();	
*			- This function handles VPRE_UV event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleVpre_UV(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'V','p','r','e','_','U','V'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleVcore_OV handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleVcore_OV();	
*			- This function handles VCORE_OV event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleVcore_OV(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[8] = {'V','c','o','r','e','_','O','V'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 8);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleVcore_UV handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleVcore_UV();	
*			- This function handles VCORE_UV event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleVcore_UV(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[8] = {'V','c','o','r','e','_','U','V'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 8);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

//
// event handlers for the Status_Vreg2 register
//

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIlim_core handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIlim_core();	
*			- This function handles ILIM_CORE event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIlim_core(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[9] = {'I','l','i','m','_','c','o','r','e'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 9);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleTwarn_core handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleTwarn_core();	
*			- This function handles TWARN_CORE event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleTwarn_core(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[10] = {'T','w','a','r','n','_','c','o','r','e'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 10);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleVcore_state handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleVcore_state();	
*			- This function handles VCORE_STATE event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleVcore_state(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[11] = {'V','c','o','r','e','_','s','t','a','t','e'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 11);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleTwarn_cca handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleTwarn_cca();	
*			- This function handles TWARN_CCA event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleTwarn_cca(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[9] = {'T','w','a','r','n','_','c','c','a'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 9);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIlim_cca handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIlim_cca();	
*			- This function handles ILIM_CCA event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIlim_cca(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[8] = {'I','l','i','m','_','c','c','a'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 8);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIlim_aux handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIlim_aux();	
*			- This function handles ILIM_AUX event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIlim_aux(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[8] = {'I','l','i','m','_','a','u','x'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 8);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIlim_can handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIlim_can();	
*			- This function handles ILIM_CAN event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIlim_can(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[8] = {'I','l','i','m','_','c','a','n'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 8);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/*
* event handlers for the Diag_Vreg3 register
*/

/***************************************************************************//*!
*   @brief The function PwSBC_HandleTsd_cca handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleTsd_cca();	
*			- This function handles TSD_CCA event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleTsd_cca(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'T','s','d','_','c','c','a'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIlim_cca_off handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIlim_cca_off();	
*			- This function handles ILIM_CCA_OFF event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIlim_cca_off(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[12] = {'I','l','i','m','_','c','c','a','_','o','f','f'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 12);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleVcca_OV handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleVcca_OV();	
*			- This function handles VCCA_OV event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleVcca_OV(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'V','c','c','a','_','O','V'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleVcca_UV handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleVcca_UV();	
*			- This function handles VCCA_UV event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleVcca_UV(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'V','c','c','a','_','U','V'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

//
// event PwSBC_Handlers for the Diag_Vreg2 register
//

/***************************************************************************//*!
*   @brief The function PwSBC_HandleTsd_can handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleTsd_can();	
*			- This function handles TSD_CAN event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleTsd_can(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'T','s','d','_','c','a','n'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}
/***************************************************************************//*!
*   @brief The function PwSBC_HandleVcan_ov handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleVcan_ov();	
*			- This function handles VCAN_OV event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleVcan_ov(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'V','c','a','n','_','o','v'};		//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleVcan_uv handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleVcan_uv();	
*			- This function handles VCAN_UV event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleVcan_uv(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'V','c','a','n','_','u','v'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleTsd_aux handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleTsd_aux();	
*			- This function handles TSD_AUX event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleTsd_aux(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'T','s','d','_','a','u','x'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleIlim_aux_off handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleIlim_aux_off();	
*			- This function handles SPI_ERR event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleIlim_aux_off(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'S','P','I','_','e','r','r'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleVaux_OV handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleVaux_OV();	
*			- This function handles VAUX_OV event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleVaux_OV(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'V','a','u','x','_','O','V'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
	INTstruct.Diag_Vreg2.B.Vaux_OV = 0;        //clear flag
}

/***************************************************************************//*!
*   @brief The function PwSBC_HandleVaux_UV handles event in the PwSBC.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					Event handlers are defined for all interrupt sources in 
*					the PwSBC. These handlers have predefined headers, but 
*					shall be defined by customers regarding their special needs 
*					(different safety strategies etc.). This code should be put 
*					in so-called "TO DO" sections. In the end function clears 
*					the appropriate event flag in global structures.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_HandleVaux_UV();	
*			- This function handles VAUX_UV event that has been captured during 
*			the SIUL interrupt routine.	
********************************************************************************/
void PwSBC_HandleVaux_UV(){
	#ifdef DEBUG_HANDLERS
	unsigned char data[7] = {'V','a','u','x','_','U','V'};	//send error via UART, for DEBUG purposes only
	LINFLEX_SendWordUART(LINFLEX_UART_NB, data, 7);
	#endif
	//
	// .... 
	// TO DO part
	// ....
	//
}


/*---------------------------------------------------------------------------\
* PIT interruption service routine for WD refresh called by a TIF flag
\****************************************************************************/
 
 /***************************************************************************//*!
*   @brief The function PwSBC_IsrPIT_WD is a PIT interrupt service routine for WD refresh.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function is an interrupt service routine for the periodical 
*					WD refresh that is launched by Periodical Interrupt Timer 
*					(PIT). On the beginning function gets actual LFSR value from 
*					the PwSBC, makes control computations and sends WD answer. 
*					If WD is refreshed without any error, function clears PIT 
*					interrupt flag and ends. If any error occurs, error strategy 
*					will be used reach a successfulWD refresh (see ALGORITHMS 
*					for details).
*	@remarks 	PIT channel used for periodical WD refresh is defined in global 
*				defines as a PIT_WD_CH. This function shall be registered as 
*				an interrupt service routine for specified interrupt vector with 
*				specified priority (parameter INT_WD_PRIORITY in the global 
*				defines). This priority is set by default as the highest one.
*	@par Code sample
*			INTC_InstallINTCInterruptHandler(PwSBC_IsrPIT_WD,227,INT_WD_PRIORITY);	
*			- This function registers PwSBC_IsrPIT_WD interrupt routine with interrupt 
*			vector no. 227 and priority defined by parameter INT_WD_PRIORITY 
*			(placed in global defines).	
********************************************************************************/
void PwSBC_IsrPIT_WD(){
	uint32_t errorCounter = 0;
	uint32_t resp = 10;															//some non-zero value
	uint32_t newLFSR = 0;
#ifdef COMPILE_SILICON_PASS3
	static uint32_t nbWDrefresh = 0;
	static uint32_t FSOUTreleased = 0;
	///Refresh WD counter in the SW
	nbWDrefresh++;
#endif	
	newLFSR = PwSBC_GetWDLFSR();
	PITstruct.currentLFSR.R = INTstruct.WD_LFSR.R;		//store current LFSR in PIT structure

	if(GUIstruct.WDwinChangeRequest == 1){
		GUIstruct.WDwinChangeRequest = 0;			
		PIT_DisableChannel(PIT_WD_CH);		//disable PIT
		PwSBC_ChangeWDwindow(GUIstruct.WDwinNb);	//Set new WD window duration in the PwSBC
		PIT_Setup(PIT_WD_CH, MCU_SYS_CLK, GUIstruct.WDwinPeriod);
		PIT_EnableChannel(PIT_WD_CH);
	}


	PITstruct.WD_answer = PwSBC_ComputeLFSR(newLFSR);						//make the test computations and store result in PIT structure
		
	do
	{
		resp = PwSBC_RefreshWD(PITstruct.WD_answer);
		if(resp != 0){
			PIT_DisableChannel(PIT_WD_CH);							//disable timer if an error has occured
			PwSBC_GetDiagSPI();
			errorCounter++;
			if(errorCounter > 20)
			{
			//	PwSBC_SignalError(ERR_SPI_NO_REFRESH);
				errorCounter = 1;									//error signaled -> decrement error counter
			}	
		}
		else{
			errorCounter = 0;										//no error -> reset errorCounter
		}
	}while(errorCounter > 0);
	  	  
#ifdef COMPILE_SILICON_PASS3
	if((FSOUTreleased == 0) && (nbWDrefresh >= 7)){
		FSOUTreleased = 1;
		PwSBC_ReleaseFSout(); 	
	}
#endif  	  
	if(!PIT_IsChannelEnabled(PIT_WD_CH)){
		PIT_EnableChannel(PIT_WD_CH);		
	}	
	if(ADCstruct.scanVoltage.R > 0){
		ADC_StartNormalConversion(ADC_NB, ADC_MASK);				//start new ADC conversion if required by scanVoltage mask	
	} 	
	PIT_ClearFlag(PIT_WD_CH);		//clear interrupt TIF flag
}

/*---------------------------------------------------------------------------\
* PIT interruption service routine to send data via UART called by a TIF flag
\****************************************************************************/
 
void PwSBC_IsrPIT_UART(){
	
	
	PwSBC_UpdateRegisters();												
	
//	PwSBC_SendCommonRegisters();

	PIT_ClearFlag(PIT_UART_CH);														//clear interrupt TIF flag	
	

}

/*---------------------------------------------------------------------------\
* EXTernal pin interruption service routine called by RFDF flag
\****************************************************************************/

 /***************************************************************************//*!
*   @brief The function PwSBC_IsrSIUL is an SIUL external interrupt service routine.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function is an interrupt service routine for external event 
*					handling. This function reads general status register and looks 
*					for the interrupt source. If the an active interrupt flag 
*					in the PwSBC has been found, function calls appropriate 
*					event handler. Event handlers are predefined functions with 
*					"TO DO" sections. These sections must be finished 
*					depending on special needs for each application (different 
*					safety strategies etc., see ALGORITHMS for details).
*	@remarks 	Input that is being used for external interrupt handling is 
*				defined by parameter SIUL_INT_EIRQ. This function shall be registered 
*				as an interrupt service routine for specified interrupt vector 
*				with specified priority (parameter INT_SIUL_PRIORITY in the 
*				global defines).
*	@par Code sample
*			INTC_InstallINTCInterruptHandler(PwSBC_IsrSIUL,41, INT_SIUL_PRIORITY);	
*			- This function registers PwSBC_IsrSIUL interrupt routine with interrupt 
*			vector no. 41 (corresponds to EIRQ no. 7) and priority defined by 
*			parameter INT_SIUL_PRIORITY (placed in global defines).	
********************************************************************************/
void PwSBC_IsrSIUL(){
	uint32_t WUsource = 0;
	/* Port for S32-Studio Issue */
		/* to prevent compiler warning */
	WUsource = WUsource;
	
	PwSBC_GetMode();		//any get function to obtain actual status

/* SPI_G event */
	
	if(SPIstruct.statusPwSBC.B.SPI_G == 1){
		PwSBC_GetDiagSPI();
		//PwSBC_SignalEvent(EV_SPI);
	}
				
/* WU event */
	if(SPIstruct.statusPwSBC.B.WU == 1){
			if((PwSBC_GetWUsource() & 0x000000FF) > 0){			//mask only the register without status
				if(INTstruct.WU_source.B.Phy_WU == 1){		//event in the physical layer of the CAN or LIN bus
					PwSBC_GetCANmode();				
				}
				if(INTstruct.WU_source.B.Vreg_WU == 1){		//event in the regulator
					PwSBC_GetStatusVreg1();
				}
			}
		}
		
/* CAN_G event */
	if(SPIstruct.statusPwSBC.B.CAN_G == 1){
		if((PwSBC_GetDiagCAN() & 0x000000FF) == 0){													//mask only the register without status
			PwSBC_GetCAN();		
		}
	}
	
/* LIN_G event */
	if(SPIstruct.statusPwSBC.B.LIN_G == 1){
		PwSBC_GetCAN();		
	}

	
/* IO_G event */
	if(SPIstruct.statusPwSBC.B.IO_G == 1){
		PwSBC_GetIOinputState();		
	}


/* Vpre_G event */
	if(SPIstruct.statusPwSBC.B.Vpre_G == 1){
		PwSBC_GetDiagVreg1();
		if((INTstruct.Diag_Vreg1.R & 0x000000D8) == 0){								//mask only the bits associated with Vpre -> Vsns_uv, Vsup_uv_H, Tsd_pre,	Vpre_OV,	Vpre_UV
			PwSBC_GetStatusVreg1();
		}
	}


/* Vcore_G event */
	if(SPIstruct.statusPwSBC.B.Vcore_G == 1){
		PwSBC_GetDiagVreg1();
		PwSBC_GetStatusVreg2();
	}

/* Vothers_G */
	if(SPIstruct.statusPwSBC.B.Vothers_G == 1){
		PwSBC_GetDiagVreg2();
		PwSBC_GetDiagVreg3();
		PwSBC_GetStatusVreg2();				

	}
	
	SIUL_ClearExtIntFlag(SIUL_INT_EIRQ);											//clear interrupt EIF flag 		

}


/*---------------------------------------------------------------------------\
* ADC interruption service routine called by EOC flag
\****************************************************************************/

 /***************************************************************************//*!
*   @brief The function PwSBC_IsrADC si an ADC interrupt service routine.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function is an interrupt service routine to process results 
*					from AD conversions (called with every end of conversion). 
*					Depending on the active AMUX channel, function treats ADC 
*					results and stores them in the global structures. Then AMUX 
*					channel is switched and function ends (for details see 
*					ALGORITHMS).
*	@remarks 	ADC module that is being used for ADC conversions is defined by 
*				ADC_NB parameter in global defines. Priority of the ADC interrupt 
*				is defined by INT_ADC_PRIORITY parameter.
*	@par Code sample
*			INTC_InstallINTCInterruptHandler(PwSBC_IsrADC,(62 + 20*ADC_NB),
*				INT_ADC_PRIORITY);	
*			- This function registers PwSBC_IsrADC function as an interrupt service 
*			routine for interrupt vector no. 62 (corresponds to the end of 
*			conversion event for ADC module no. 0) with priority 
*			INT_ADC_PRIORITY.	
********************************************************************************/
void PwSBC_IsrADC(){
	uint8_t actualCH = 0;
	uint8_t nbAMUX = 0;
	uint8_t mask = 0;
	uint8_t i = 0;
	
	actualCH = (uint8_t)INTstruct.IO_OUT_AMUX.B.Amux;				//actual channel used by ADC
	
/* store actual channel to the structure */	
	switch(actualCH){
		case	AMUX_VREF				:	ADCstruct.actualVoltage.Vref = PwSBC_GetVoltage(); break;
		case	AMUX_VSNS_WIDE	:	ADCstruct.actualVoltage.VsnsW = PwSBC_GetVoltageWide(); break;
		case	AMUX_IO0_WIDE		:	ADCstruct.actualVoltage.IO0W = PwSBC_GetVoltageWide(); break;
		case	AMUX_IO1_WIDE		:	ADCstruct.actualVoltage.IO1W = PwSBC_GetVoltageWide(); break;
		case	AMUX_VSNS_TIGHT	:	ADCstruct.actualVoltage.VsnsT = PwSBC_GetVoltageTight(); break;
		case	AMUX_IO0_TIGHT	: ADCstruct.actualVoltage.IO0T = PwSBC_GetVoltageTight(); break;
		case	AMUX_IO1_TIGHT	: ADCstruct.actualVoltage.IO1T = PwSBC_GetVoltageTight(); break;
		case	AMUX_TEMP				: ADCstruct.actualVoltage.Temp = PwSBC_GetTemperature(); break;
	}

/* switch AMUX to the following masked channel and start next conversion */
	nbAMUX = actualCH;
	
	for(i = 0;i < 8;i++){ 
		nbAMUX++;
		if(nbAMUX >= 8){				//if higher number than number of channels -> return to the start index (0)
			nbAMUX = 0;					
		}
		
		mask = 1 << nbAMUX;				//mask just one bit every time - masks are powers of 2
		
		if((ADCstruct.scanVoltage.R & mask) > 0){		//if the specified channel is masked
			PwSBC_SwitchAMUXchannel(nbAMUX);									//change AMUX channel
			break;																		
		}
	}
	
	ADC_ClearAllEOCflags(ADC_NB);										//clear EOC flags
	
}

/*---------------------------------------------------------------------------\
* EVENT handler - check of all PwSBC interruption event
\****************************************************************************/

 /***************************************************************************//*!
*   @brief The function PwSBC_EventHandler checks events in the system. 
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function checks all event flags in the global structures 
*					that were previously stored during processing interrupt 
*					routine. If any flag is active, then the appropriate event 
*					handler (function PwSBC_HandleXXXX) is called. As an active 
*					flag is considered flag that changed its state (on any 
*					edge - bits that can be active always e.g. IO_5) or flag 
*					that has been set from 0 to 1 regardless its previous state 
*					(flags that can be active only for a short period e.g. 
*					Vcca_OV). The "any edge" flags are stored in the structures, 
*					but the "temporary" flags are cleared after their processing.
*	@remarks 	This event handler does not send any command to PwSBC, it works 
*				only with the flags stored in the global structures.
*	@par Code sample
*			PwSBC_EventHandler();	
*			- This function handles all events that have been captured during SIUL 
*			interrupt routine.	
********************************************************************************/
void PwSBC_EventHandler(){
	if(INTstruct.Diag_SPI.B.SPI_err == 1){	//check SPI_err event
		PwSBC_HandleSPI_err();	
		INTstruct.Diag_SPI.B.SPI_err = 0;	//Clear flag
		INTstructPrevious.Diag_SPI.B.SPI_err = INTstruct.Diag_SPI.B.SPI_err;	//Update previous structure with new results
	}
	if(INTstruct.Diag_SPI.B.SPI_CLK == 1){	//check SPI_CLK event
		PwSBC_HandleSPI_CLK();	
		INTstruct.Diag_SPI.B.SPI_CLK = 0;	//Clear flag
		INTstructPrevious.Diag_SPI.B.SPI_CLK = INTstruct.Diag_SPI.B.SPI_CLK;	//Update previous structure with new results
	}
	if(INTstruct.Diag_SPI.B.SPI_Parity == 1){	//check SPI_Parity event
		PwSBC_HandleSPI_Parity();	
		INTstruct.Diag_SPI.B.SPI_Parity = 0;	//Clear flag
		INTstructPrevious.Diag_SPI.B.SPI_Parity = INTstruct.Diag_SPI.B.SPI_Parity;	//Update previous structure with new results
	}
	if(INTstruct.Diag_SPI.B.SPI_Req == 1){	//check SPI_Req event
		PwSBC_HandleSPI_Req();	
		INTstruct.Diag_SPI.B.SPI_Req = 0;	//Clear flag
		INTstructPrevious.Diag_SPI.B.SPI_Req = INTstruct.Diag_SPI.B.SPI_Req;	//Update previous structure with new results
	}
	if(INTstruct.WU_source.B.IO_5_WU == 1){	//check IO_5_WU event
		PwSBC_HandleIO_5_WU();	
		INTstruct.WU_source.B.IO_5_WU = 0;	//Clear flag
		INTstructPrevious.WU_source.B.IO_5_WU = INTstruct.WU_source.B.IO_5_WU;	//Update previous structure with new results
	}

	if(INTstruct.WU_source.B.IO_4_WU == 1){	//check IO_4_WU event
		PwSBC_HandleIO_4_WU();	
		INTstruct.WU_source.B.IO_4_WU = 0;	//Clear flag
		INTstructPrevious.WU_source.B.IO_4_WU = INTstruct.WU_source.B.IO_4_WU;	//Update previous structure with new results
	}

	if(INTstruct.WU_source.B.IO_3_WU == 1){	//check IO_3_WU event
		PwSBC_HandleIO_3_WU();	
		INTstruct.WU_source.B.IO_3_WU = 0;	//Clear flag
		INTstructPrevious.WU_source.B.IO_3_WU = INTstruct.WU_source.B.IO_3_WU;	//Update previous structure with new results
	}

	if(INTstruct.WU_source.B.IO_2_WU == 1){	//check IO_2_WU event
		PwSBC_HandleIO_2_WU();	
		INTstruct.WU_source.B.IO_2_WU = 0;	//Clear flag
		INTstructPrevious.WU_source.B.IO_2_WU = INTstruct.WU_source.B.IO_2_WU;	//Update previous structure with new results
	}

	if(INTstruct.WU_source.B.IO_1_WU == 1){	//check IO_1_WU event
		PwSBC_HandleIO_1_WU();	
		INTstruct.WU_source.B.IO_1_WU = 0;	//Clear flag
		INTstructPrevious.WU_source.B.IO_1_WU = INTstruct.WU_source.B.IO_1_WU;	//Update previous structure with new results
	}

	if(INTstruct.WU_source.B.IO_0_WU == 1){	//check IO_0_WU event
		PwSBC_HandleIO_0_WU();	
		INTstruct.WU_source.B.IO_0_WU = 0;	//Clear flag
		INTstructPrevious.WU_source.B.IO_0_WU = INTstruct.WU_source.B.IO_0_WU;	//Update previous structure with new results
	}


	if(INTstruct.CAN_LIN_mode.B.CAN_wu == 1){	//check CAN_wu event
		PwSBC_HandleCAN_wu();	
		INTstruct.CAN_LIN_mode.B.CAN_wu = 0;	//Clear flag
		INTstructPrevious.CAN_LIN_mode.B.CAN_wu = INTstruct.CAN_LIN_mode.B.CAN_wu;	//Update previous structure with new results
	}

	if(INTstruct.Diag_CAN.B.CANL_batt == 1){	//check CANL_batt event
		PwSBC_HandleCANL_batt();	
		INTstruct.Diag_CAN.B.CANL_batt = 0;	//Clear flag
		INTstructPrevious.Diag_CAN.B.CANL_batt = INTstruct.Diag_CAN.B.CANL_batt;	//Update previous structure with new results
	}

	if(INTstruct.Diag_CAN.B.CANL_gnd == 1){	//check CANL_gnd event
		PwSBC_HandleCANL_gnd();	
		INTstruct.Diag_CAN.B.CANL_gnd = 0;	//Clear flag
		INTstructPrevious.Diag_CAN.B.CANL_gnd = INTstruct.Diag_CAN.B.CANL_gnd;	//Update previous structure with new results
	}

	if(INTstruct.Diag_CAN.B.CANH_batt == 1){	//check CANH_batt event
		PwSBC_HandleCANH_batt();	
		INTstruct.Diag_CAN.B.CANH_batt = 0;	//Clear flag
		INTstructPrevious.Diag_CAN.B.CANH_batt = INTstruct.Diag_CAN.B.CANH_batt;	//Update previous structure with new results
	}

	if(INTstruct.Diag_CAN.B.CANH_gnd == 1){	//check CANH_gnd event
		PwSBC_HandleCANH_gnd();	
		INTstruct.Diag_CAN.B.CANH_gnd = 0;	//Clear flag
		INTstructPrevious.Diag_CAN.B.CANH_gnd = INTstruct.Diag_CAN.B.CANH_gnd;	//Update previous structure with new results
	}

	if(INTstruct.Diag_CAN_LIN.B.CAN_OT == 1){	//check CAN_OT event
		PwSBC_HandleCAN_OT();	
		INTstruct.Diag_CAN_LIN.B.CAN_OT = 0;	//Clear flag
		INTstructPrevious.Diag_CAN_LIN.B.CAN_OT = INTstruct.Diag_CAN_LIN.B.CAN_OT;	//Update previous structure with new results
	}

	if(INTstruct.Diag_CAN.B.RXD_recessive == 1){	//check RXD_recessive event
		PwSBC_HandleRXD_recessive();	
		INTstruct.Diag_CAN.B.RXD_recessive = 0;	//Clear flag
		INTstructPrevious.Diag_CAN.B.RXD_recessive = INTstruct.Diag_CAN.B.RXD_recessive;	//Update previous structure with new results
	}

	if(INTstruct.Diag_CAN.B.TXD_dominant == 1){	//check TXD_dominant event
		PwSBC_HandleTXD_dominant();	
		INTstruct.Diag_CAN.B.TXD_dominant = 0;	//Clear flag
		INTstructPrevious.Diag_CAN.B.TXD_dominant = INTstruct.Diag_CAN.B.TXD_dominant;	//Update previous structure with new results
	}

	if(INTstruct.Diag_CAN.B.CAN_dominant == 1){	//check CAN_dominant event
		PwSBC_HandleCAN_dominant();	
		INTstruct.Diag_CAN.B.CAN_dominant = 0;	//Clear flag
		INTstructPrevious.Diag_CAN.B.CAN_dominant = INTstruct.Diag_CAN.B.CAN_dominant;	//Update previous structure with new results
	}

	if(INTstruct.Diag_CAN_LIN.B.CAN_OC == 1){	//check CAN_OC event
		PwSBC_HandleCAN_OC();	
		INTstruct.Diag_CAN_LIN.B.CAN_OC = 0;	//Clear flag
		INTstructPrevious.Diag_CAN_LIN.B.CAN_OC = INTstruct.Diag_CAN_LIN.B.CAN_OC;	//Update previous structure with new results
	}


	if(INTstruct.IO_input_state.B.IO_5 != INTstructPrevious.IO_input_state.B.IO_5){	//check IO_5 event
		PwSBC_HandleIO_5();	
		INTstructPrevious.IO_input_state.B.IO_5 = INTstruct.IO_input_state.B.IO_5;	//Update previous structure with new results
	}

	if(INTstruct.IO_input_state.B.IO_4 != INTstructPrevious.IO_input_state.B.IO_4){	//check IO_4 event
		PwSBC_HandleIO_4();	
		INTstructPrevious.IO_input_state.B.IO_4 = INTstruct.IO_input_state.B.IO_4;	//Update previous structure with new results
	}

	if(INTstruct.IO_input_state.B.IO_3 != INTstructPrevious.IO_input_state.B.IO_3){	//check IO_3 event
		PwSBC_HandleIO_3();	
		INTstructPrevious.IO_input_state.B.IO_3 = INTstruct.IO_input_state.B.IO_3;	//Update previous structure with new results
	}

	if(INTstruct.IO_input_state.B.IO_2 != INTstructPrevious.IO_input_state.B.IO_2){	//check IO_2 event
		PwSBC_HandleIO_2();	
		INTstructPrevious.IO_input_state.B.IO_2 = INTstruct.IO_input_state.B.IO_2;	//Update previous structure with new results
	}

	if(INTstruct.IO_input_state.B.IO_1 != INTstructPrevious.IO_input_state.B.IO_1){	//check IO_1 event
		PwSBC_HandleIO_1();	
		INTstructPrevious.IO_input_state.B.IO_1 = INTstruct.IO_input_state.B.IO_1;	//Update previous structure with new results
	}

	if(INTstruct.IO_input_state.B.IO_0 != INTstructPrevious.IO_input_state.B.IO_0){	//check IO_0 event
		PwSBC_HandleIO_0();	
		INTstructPrevious.IO_input_state.B.IO_0 = INTstruct.IO_input_state.B.IO_0;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg1.B.Vsns_uv != INTstructPrevious.Diag_Vreg1.B.Vsns_uv){	//check Vsns_uv event
		PwSBC_HandleVsns_uv();	
		INTstructPrevious.Diag_Vreg1.B.Vsns_uv = INTstruct.Diag_Vreg1.B.Vsns_uv;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg1.B.Vsup_uv_7!= INTstructPrevious.Diag_Vreg1.B.Vsup_uv_7){	//check Vsup_uv_H event
		PwSBC_HandleVsup_uv_H();	
		INTstructPrevious.Diag_Vreg1.B.Vsup_uv_7= INTstruct.Diag_Vreg1.B.Vsup_uv_7; //Update previous structure with new results
	}

	if(INTstruct.Status_Vreg1.B.IpFF != INTstructPrevious.Status_Vreg1.B.IpFF){	//check IpFF event
		PwSBC_HandleIpFF();	
		INTstructPrevious.Status_Vreg1.B.IpFF = INTstruct.Status_Vreg1.B.IpFF;	//Update previous structure with new results
	}

	if(INTstruct.Status_Vreg1.B.Ilim_pre != INTstructPrevious.Status_Vreg1.B.Ilim_pre){	//check Ilim_pre event
		PwSBC_HandleIlim_pre();	
		INTstructPrevious.Status_Vreg1.B.Ilim_pre = INTstruct.Status_Vreg1.B.Ilim_pre;	//Update previous structure with new results
	}

	if(INTstruct.Status_Vreg1.B.Twarn_pre != INTstructPrevious.Status_Vreg1.B.Twarn_pre){	//check Twarn_pre event
		PwSBC_HandleTwarn_pre();	
		INTstructPrevious.Status_Vreg1.B.Twarn_pre = INTstruct.Status_Vreg1.B.Twarn_pre;	//Update previous structure with new results
	}

	if(INTstruct.Status_Vreg1.B.BoB != INTstructPrevious.Status_Vreg1.B.BoB){	//check BoB event
		PwSBC_HandleBoB();	
		INTstructPrevious.Status_Vreg1.B.BoB = INTstruct.Status_Vreg1.B.BoB;	//Update previous structure with new results
	}

	if(INTstruct.Status_Vreg1.B.Vpre_state != INTstructPrevious.Status_Vreg1.B.Vpre_state){	//check Vpre_state event
		PwSBC_HandleVpre_state();	
		INTstructPrevious.Status_Vreg1.B.Vpre_state = INTstruct.Status_Vreg1.B.Vpre_state;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg1.B.Vpre_OV == 1){	//check Vpre_OV event
		PwSBC_HandleVpre_OV();	
		INTstruct.Diag_Vreg1.B.Vpre_OV = 0;	//Clear flag
		INTstructPrevious.Diag_Vreg1.B.Vpre_OV = INTstruct.Diag_Vreg1.B.Vpre_OV;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg1.B.Vpre_UV != INTstructPrevious.Diag_Vreg1.B.Vpre_UV){	//check Vpre_UV event
		PwSBC_HandleVpre_UV();	
		INTstructPrevious.Diag_Vreg1.B.Vpre_UV = INTstruct.Diag_Vreg1.B.Vpre_UV;	//Update previous structure with new results
	}

	if(INTstruct.Status_Vreg2.B.Ilim_core == 1){	//check Ilim_core event
		PwSBC_HandleIlim_core();	
		INTstruct.Status_Vreg2.B.Ilim_core = 0;	//Clear flag
		INTstructPrevious.Status_Vreg2.B.Ilim_core = INTstruct.Status_Vreg2.B.Ilim_core;	//Update previous structure with new results
	}

	if(INTstruct.Status_Vreg2.B.Twarn_core == 1){	//check Twarn_core event
		PwSBC_HandleTwarn_core();	
		INTstruct.Status_Vreg2.B.Twarn_core = 0;	//Clear flag
		INTstructPrevious.Status_Vreg2.B.Twarn_core = INTstruct.Status_Vreg2.B.Twarn_core;	//Update previous structure with new results
	}

	if(INTstruct.Status_Vreg2.B.Vcore_state != INTstructPrevious.Status_Vreg2.B.Vcore_state){	//check Vcore_state event
		PwSBC_HandleVcore_state();	
		INTstructPrevious.Status_Vreg2.B.Vcore_state = INTstruct.Status_Vreg2.B.Vcore_state;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg1.B.Vcore_UV != INTstructPrevious.Diag_Vreg1.B.Vcore_UV){	//check Vcore_UV event
		PwSBC_HandleVcore_UV();	
		INTstructPrevious.Diag_Vreg1.B.Vcore_UV = INTstruct.Diag_Vreg1.B.Vcore_UV;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg1.B.Vcore_OV == 1){	//check Vcore_OV event
		PwSBC_HandleVcore_OV();	
		INTstruct.Diag_Vreg1.B.Vcore_OV = 0;	//Clear flag
		INTstructPrevious.Diag_Vreg1.B.Vcore_OV = INTstruct.Diag_Vreg1.B.Vcore_OV;	//Update previous structure with new results
	}

	if(INTstruct.Status_Vreg2.B.Ilim_cca == 1){	//check Ilim_cca event
		PwSBC_HandleIlim_cca();	
		INTstruct.Status_Vreg2.B.Ilim_cca = 0;	//Clear flag
		INTstructPrevious.Status_Vreg2.B.Ilim_cca = INTstruct.Status_Vreg2.B.Ilim_cca;	//Update previous structure with new results
	}

	if(INTstruct.Status_Vreg2.B.Twarn_cca == 1){	//check Twarn_cca event
		PwSBC_HandleTwarn_cca();	
		INTstruct.Status_Vreg2.B.Twarn_cca = 0;	//Clear flag
		INTstructPrevious.Status_Vreg2.B.Twarn_cca = INTstruct.Status_Vreg2.B.Twarn_cca;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg3.B.Tsd_cca == 1){	//check Tsd_cca event
		PwSBC_HandleTsd_cca();	
		INTstruct.Diag_Vreg3.B.Tsd_cca = 0;	//Clear flag
		INTstructPrevious.Diag_Vreg3.B.Tsd_cca = INTstruct.Diag_Vreg3.B.Tsd_cca;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg3.B.Ilim_cca_off != INTstructPrevious.Diag_Vreg3.B.Ilim_cca_off){	//check Ilim_cca_off event
		PwSBC_HandleIlim_cca_off();	
		INTstructPrevious.Diag_Vreg3.B.Ilim_cca_off = INTstruct.Diag_Vreg3.B.Ilim_cca_off;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg3.B.Vcca_UV != INTstructPrevious.Diag_Vreg3.B.Vcca_UV){	//check Vcca_UV event
		PwSBC_HandleVcca_UV();	
		INTstructPrevious.Diag_Vreg3.B.Vcca_UV = INTstruct.Diag_Vreg3.B.Vcca_UV;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg3.B.Vcca_OV == 1){	//check Vcca_OV event
		PwSBC_HandleVcca_OV();	
		INTstruct.Diag_Vreg3.B.Vcca_OV = 0;	//Clear flag
		INTstructPrevious.Diag_Vreg3.B.Vcca_OV = INTstruct.Diag_Vreg3.B.Vcca_OV;	//Update previous structure with new results
	}

	if(INTstruct.Status_Vreg2.B.Ilim_aux == 1){	//check Ilim_aux event
		PwSBC_HandleIlim_aux();	
		INTstruct.Status_Vreg2.B.Ilim_aux = 0;	//Clear flag
		INTstructPrevious.Status_Vreg2.B.Ilim_aux = INTstruct.Status_Vreg2.B.Ilim_aux;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg2.B.Tsd_aux == 1){	//check Tsd_aux event
		PwSBC_HandleTsd_aux();	
		INTstruct.Diag_Vreg2.B.Tsd_aux = 0;	//Clear flag
		INTstructPrevious.Diag_Vreg2.B.Tsd_aux = INTstruct.Diag_Vreg2.B.Tsd_aux;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg2.B.Ilim_aux_off != INTstructPrevious.Diag_Vreg2.B.Ilim_aux_off){	//check Ilim_aux_off event
		PwSBC_HandleIlim_aux_off();	
		INTstructPrevious.Diag_Vreg2.B.Ilim_aux_off = INTstruct.Diag_Vreg2.B.Ilim_aux_off;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg2.B.Vaux_UV != INTstructPrevious.Diag_Vreg2.B.Vaux_UV){	//check Vaux_UV event
		PwSBC_HandleVaux_UV();	
		INTstructPrevious.Diag_Vreg2.B.Vaux_UV = INTstruct.Diag_Vreg2.B.Vaux_UV;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg2.B.Vaux_OV == 1){	//check Vaux_OV event
		PwSBC_HandleVaux_OV();	
		INTstruct.Diag_Vreg2.B.Vaux_OV = 0;	//Clear flag
		INTstructPrevious.Diag_Vreg2.B.Vaux_OV = INTstruct.Diag_Vreg2.B.Vaux_OV;	//Update previous structure with new results
	}

	if(INTstruct.Status_Vreg2.B.Ilim_can == 1){	//check Ilim_can event
		PwSBC_HandleIlim_can();	
		INTstruct.Status_Vreg2.B.Ilim_can = 0;	//Clear flag
		INTstructPrevious.Status_Vreg2.B.Ilim_can = INTstruct.Status_Vreg2.B.Ilim_can;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg2.B.Vcan_uv != INTstructPrevious.Diag_Vreg2.B.Vcan_uv){	//check Vcan_uv event
		PwSBC_HandleVcan_uv();	
		INTstructPrevious.Diag_Vreg2.B.Vcan_uv = INTstruct.Diag_Vreg2.B.Vcan_uv;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg2.B.Vcan_ov == 1){	//check Vcan_ov event
		PwSBC_HandleVcan_ov();	
		INTstruct.Diag_Vreg2.B.Vcan_ov = 0;	//Clear flag
		INTstructPrevious.Diag_Vreg2.B.Vcan_ov = INTstruct.Diag_Vreg2.B.Vcan_ov;	//Update previous structure with new results
	}

	if(INTstruct.Diag_Vreg2.B.Tsd_can == 1){	//check Tsd_can event
		PwSBC_HandleTsd_can();	
		INTstruct.Diag_Vreg2.B.Tsd_can = 0;	//Clear flag
		INTstructPrevious.Diag_Vreg2.B.Tsd_can = INTstruct.Diag_Vreg2.B.Tsd_can;	//Update previous structure with new results
	}
	
	/* Update diagnostics */
	PwSBC_GetDiagVreg1();																							
	PwSBC_GetDiagVreg2();	
	PwSBC_GetDiagVreg3();
	PwSBC_GetStatusVreg1();
	PwSBC_GetStatusVreg2();
	PwSBC_GetDiagSPI();
	PwSBC_GetWUsource();
	PwSBC_GetCANmode();
	PwSBC_GetDiagCAN();
	PwSBC_GetCAN();
	PwSBC_GetIOinputState();

}

/*---------------------------------------------------------------------------\
* COMMUNICATION functions
----------------------------------------------------------------------------*\

 ***************************************************************************//*!
*   @brief The function PwSBC_UpdateRegisters gets the actual content of all necessary registers and stores it into 
*			a structure.
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function updates global data structures that contain states 
*					of PwSBC registers. This is done for all PwSBC registers 
*					excluding INIT registers that cannot be changed during normal 
*					operation.
*	@remarks 	
*	@par Code sample
*			PwSBC_UpdateRegisters();	
*			- This function updates global data structures with actual content of 
*			the PwSBC registers.	
********************************************************************************/
void	PwSBC_UpdateRegisters(){			//INIT registers are not read
	PwSBC_GetInitVreg1();
	PwSBC_GetInitVreg2();
	PwSBC_GetInitCAN();
	PwSBC_GetInitIOWU1();
	PwSBC_GetInitIOWU2();
	PwSBC_GetInitINT();
	
	PwSBC_GetInitSupervisor1();
	PwSBC_GetInitSupervisor2();
	PwSBC_GetInitSupervisor3();
	PwSBC_GetInitFSSM1();
	PwSBC_GetInitFSSM2();
	PwSBC_GetWDwindow();
	PwSBC_GetWDLFSR();
	PwSBC_GetWDanswer();
		
	PwSBC_GetHWconfig();
	PwSBC_GetWUsource();
	PwSBC_GetDiagFS1();					//FS command
	PwSBC_GetIOinputState();
	PwSBC_GetStatusVreg1();
	PwSBC_GetWDcounter();				//FS command
	PwSBC_GetStatusVreg2();
	PwSBC_GetDiagVreg1();
	PwSBC_GetDiagFS2();					//FS command
	PwSBC_GetDiagVreg2();
	PwSBC_GetDiagVreg3();
	PwSBC_GetDiagCAN();
	PwSBC_GetCAN();
	PwSBC_GetDiagSPI();
	PwSBC_GetMode();
	PwSBC_GetVregMode();
	PwSBC_GetIOAMUX();
	PwSBC_GetCANmode();
	PwSBC_GetVregLP();
}
#ifdef GUI_SUPPORT
/*---------------------------------------------------------------------------\
* UART interruption service routine to receive data via UART called by a  flag
\****************************************************************************/
 /***************************************************************************//*!
*   @brief The function PwSBC_IsrUART_Rx is a UART interrupt servcie routine
*	@par Include 
*					PwSBC.h
* 	@par Description 
*					This function treats all received frames from GUI via UART. 
*					Frames are decoded and checked. Decoded content is stored 
*					in the global structures.
*	@remarks 	
*	@par Code sample
*			INTC_InstallINTCInterruptHandler(PwSBC_IsrUART_Rx,99,
*			INT_UART_RX_PRIORITY);	
*			- This function registers PwSBC_IsrUART_Rx function as an interrupt service 
*			routine for interrupt vector no. 99 (corresponds to a received frame 
*			via UART) with priority INT_UART_RX_PRIORITY.		
********************************************************************************/
void PwSBC_IsrUART_Rx(){
	static unsigned char receivedCounter = 0;
	static unsigned char receivedArray[6];
	GUImsg receivedMsg;
	uint32_t cmd = 0;
	uint32_t address = 0;
	GUImsg msg;

	
	LINFLEX_ReadUART (LINFLEX_UART_NB, LINFLEX_UART_RX_BUFFER_SIZE, (unsigned char*)&(receivedArray[receivedCounter]));
		
	receivedCounter+=LINFLEX_UART_RX_BUFFER_SIZE;
	
	
	
	//construct frame
	if(receivedCounter == MSG_FRAME_SIZE){			//full frame received
		receivedCounter = 0;
		if(GUI_CheckFrame(receivedArray) == 1){		//frame OK
			//process received frame
			receivedMsg.ID = receivedArray[0];		//copy received frame to the message structure
			receivedMsg.Byte0 = receivedArray[1];
			receivedMsg.Byte1 = receivedArray[2];
			receivedMsg.Byte2 = receivedArray[3];
			receivedMsg.Byte3 = receivedArray[4]; 
			receivedMsg.CHSUM = receivedArray[5];
			
			cmd = (receivedMsg.Byte2) << 8;	//Reconstruct command
			cmd = cmd | receivedMsg.Byte3;			
			
			//Error message
			if((receivedMsg.ID >= GUI_COM_ERR_LOW) & (receivedMsg.ID <= GUI_COM_ERR_HIGH)){	
				//
				// .... 
				// TO DO part
				// ....
				//
			}
			//Event message
			if((receivedMsg.ID >= GUI_COM_EV_LOW) & (receivedMsg.ID <= GUI_COM_EV_HIGH)){	
				//
				// .... 
				// TO DO part
				// ....
				//
			}
			
			//Standard message
			if((receivedMsg.ID >= GUI_COM_STD_LOW) & (receivedMsg.ID <= GUI_COM_STD_HIGH)){	
				switch(receivedMsg.ID){
					case GUI_COM_STD41 : PwSBC_SendCmdR(cmd); break;			//Resend command received from GUI to the PwSBC
			
				}
	
			}
			
			//Setting message			
			if((receivedMsg.ID >= GUI_COM_SET_LOW) & (receivedMsg.ID <= GUI_COM_SET_HIGH)){	
				switch(receivedMsg.ID){
					case GUI_COM_SET26 : 	
										 	//Set new WD window period during the next WD refresh
										 	GUIstruct.WDwinNb = (receivedMsg.Byte3 & 0xF0) >> 4;		//Mask WD window duration
										 	if(GUIstruct.WDwinNb != WD_DISABLE){
										 		//Set new WD window and enable WD refresh if disabled
										 		switch(GUIstruct.WDwinNb){
											 		case WD_WIN_1		: GUIstruct.WDwinPeriod = PIT_WD_PERIOD1; break;
											 		case WD_WIN_2		: GUIstruct.WDwinPeriod = PIT_WD_PERIOD2; break;
											 		case WD_WIN_3		: GUIstruct.WDwinPeriod = PIT_WD_PERIOD3; break;
												 	case WD_WIN_4		: GUIstruct.WDwinPeriod = PIT_WD_PERIOD4; break;
												    case WD_WIN_6		: GUIstruct.WDwinPeriod = PIT_WD_PERIOD6; break;
												    case WD_WIN_8		: GUIstruct.WDwinPeriod = PIT_WD_PERIOD8; break;
												    case WD_WIN_12		: GUIstruct.WDwinPeriod = PIT_WD_PERIOD12; break;
												    case WD_WIN_16		: GUIstruct.WDwinPeriod = PIT_WD_PERIOD16; break;	
												    case WD_WIN_24		: GUIstruct.WDwinPeriod = PIT_WD_PERIOD24; break;
												    case WD_WIN_32		: GUIstruct.WDwinPeriod = PIT_WD_PERIOD32; break;
												    case WD_WIN_64		: GUIstruct.WDwinPeriod = PIT_WD_PERIOD64; break;
												    case WD_WIN_128		: GUIstruct.WDwinPeriod = PIT_WD_PERIOD128; break;
												    case WD_WIN_256		: GUIstruct.WDwinPeriod = PIT_WD_PERIOD256; break;
												    case WD_WIN_512		: GUIstruct.WDwinPeriod = PIT_WD_PERIOD512; break;
												    case WD_WIN_1024	: GUIstruct.WDwinPeriod = PIT_WD_PERIOD1024; break;
																												 		
											 	}
											 	if(GUIstruct.WDwinDisabled == 1){		//If WD window is disabled -> enable it and setup with new period
										 			GUIstruct.WDwinDisabled = 0;
										 			PwSBC_ChangeWDwindow(GUIstruct.WDwinNb);				//Set new WD window duration in the PwSBC				
										 			PIT_Setup(PIT_WD_CH, MCU_SYS_CLK, GUIstruct.WDwinPeriod);	//WD window duration will be changed immediatelly
													PIT_EnableChannel(PIT_WD_CH);	
										 			
										 		}
										 		else{
										 			GUIstruct.WDwinDisabled = 0;
										 			GUIstruct.WDwinChangeRequest = 1;		//WD window duration will be changed during next WD refresh
										 			
										 		}
										 	}
										 	else{
										 		//Disable WD window
										 		GUIstruct.WDwinDisabled = 1;
										 		PwSBC_ChangeWDwindow(GUIstruct.WDwinNb);				//Set new WD window duration in the PwSBC
										 		PIT_DisableChannel(PIT_WD_CH);	//disable PIT
										 													 		
										 	} break;
					case GUI_COM_SET27:
										//Request actual LFSR content update and resend it to GUI
										cmd = (receivedMsg.Byte2) << 8;		//Reconstruct command
										cmd = cmd | receivedMsg.Byte3;			
									 	PwSBC_SendCmdR(cmd);		//Resend command received from GUI to the PwSBC
										msg = GUI_CreateStdMsgFrame(129, (unsigned char*)&(INTstruct.WD_LFSR.R));
										LINFLEX_SendWordUART(LINFLEX_UART_NB,&(msg.ID), 6);
										//Refresh counters
										PwSBC_GetWDcounter();
										PwSBC_GetDiagFS2();	
										//Send them to GUI
										msg = GUI_CreateStdMsgFrame(GUI_COM_STD32, (uint8_t*)&(INTstruct.WD_counter.R));
										LINFLEX_SendWordUART(LINFLEX_UART_NB,&(msg.ID), 6);
										msg = GUI_CreateStdMsgFrame(GUI_COM_STD33, (uint8_t*)&(INTstruct.Diag_FS2.R));
										LINFLEX_SendWordUART(LINFLEX_UART_NB,&(msg.ID), 6);
										break;
					case GUI_COM_SET28:
										//GUI detected -> verify the whole frame
										if((receivedMsg.Byte0 == GUI_COM_DETECT0) && (receivedMsg.Byte1 == GUI_COM_DETECT1) && (receivedMsg.Byte2 == GUI_COM_DETECT2) && (receivedMsg.Byte3 == GUI_COM_DETECT3)){
											GUIstruct.GUIdetected = 1;		//GUI successfuly detected
										}
										else{
											GUIstruct.GUIdetected = 0;		//GUI NOT successfuly detected
										} break;
					//Load init config from the GUI into the INITstruct
					case GUI_COM_SET29: INITstruct.INIT_Vreg1.R = cmd;
										GUIstruct.PwSBCconfigLoaded = 1; break;
					case GUI_COM_SET30: INITstruct.INIT_Vreg2.R = cmd; 
										GUIstruct.PwSBCconfigLoaded++; break;
					case GUI_COM_SET31: INITstruct.INIT_CAN_LIN.R = cmd; 
										GUIstruct.PwSBCconfigLoaded++; break;
					case GUI_COM_SET32: INITstruct.INIT_IO_WU1.R = cmd;
										GUIstruct.PwSBCconfigLoaded++; break;
					case GUI_COM_SET33: INITstruct.INIT_IO_WU2.R = cmd;
										GUIstruct.PwSBCconfigLoaded++; break;
					case GUI_COM_SET34: INITstruct.INIT_INT.R = cmd; 
										GUIstruct.PwSBCconfigLoaded++; break;
					case GUI_COM_SET35: INITstruct.INIT_Supervisor1.R = cmd;
										GUIstruct.PwSBCconfigLoaded++; break; 
					case GUI_COM_SET36: INITstruct.INIT_Supervisor2.R = cmd;
										GUIstruct.PwSBCconfigLoaded++; break; 
					case GUI_COM_SET37: INITstruct.INIT_Supervisor3.R = cmd;
										GUIstruct.PwSBCconfigLoaded++; break; 
					case GUI_COM_SET38: INITstruct.INIT_FSSM1.R = cmd; 
										GUIstruct.PwSBCconfigLoaded++; break;
					case GUI_COM_SET39: INITstruct.INIT_FSSM2.R = cmd; 
										GUIstruct.PwSBCconfigLoaded++; break;
					case GUI_COM_SET40: INITstruct.WD_window.R = cmd; 
										GUIstruct.PwSBCconfigLoaded++; break;
					case GUI_COM_SET41: INITstruct.WD_LFSR.R = cmd;
										GUIstruct.PwSBCconfigLoaded++; break;
					//Enable or disable sending frames to the GUI
					case GUI_COM_SET42: GUIstruct.SendFrames = receivedMsg.Byte3; break;
					case GUI_COM_SET43: GUIstruct.SendFramesOnce = receivedMsg.Byte3; break;	
										
				}
			

			} 

			
		}
		else{										//frame NOT OK
			receivedArray[0] = receivedArray[1];	//shift all array to the left by one Byte
			receivedArray[1] = receivedArray[2];
			receivedArray[2] = receivedArray[3];
			receivedArray[3] = receivedArray[4];
			receivedArray[4] = receivedArray[5];
			receivedCounter = MSG_FRAME_SIZE - 1;	//set counter to the last position
		}

	}
		

	LINFLEX_ClearReceiveFlagUART(LINFLEX_UART_NB);		//clear Rx flags
}
#endif /*GUI_SUPPORTED */

/***************************************************************************//*!
*   @brief Sends a word via UART.
*	@par Include: 
*					LINFLEX.h
* 	@par Description: 
*					Function sends a word consisted of several Bytes via UART.
*	@param[in] nbLINFlex -  Number of LINFlex module (0 or 1).
*	@param[in] p_string -  Pointer to the array of Bytes to be sent.
*	@param[in] length -  Number of Bytes to be sent.
*	@remarks 	Function blocks UART as a resource, this could block execution 
*				of the main routines if longer frames are sent.
********************************************************************************/
void LINFLEX_SendWordUART(vuint8_t nbLINFlex, uint8_t* p_string, uint8_t length)
{
}


