/** @file wlmgr_debug.c
  *
  * @brief This file contains WLAN application specific defines etc.
  *
  * Copyright 2019-2020 NXP
  *
  * NXP CONFIDENTIAL
  * The source code contained or described herein and all documents related to
  * the source code ("Materials") are owned by NXP, its
  * suppliers and/or its licensors. Title to the Materials remains with NXP,
  * its suppliers and/or its licensors. The Materials contain
  * trade secrets and proprietary and confidential information of NXP, its
  * suppliers and/or its licensors. The Materials are protected by worldwide copyright
  * and trade secret laws and treaty provisions. No part of the Materials may be
  * used, copied, reproduced, modified, published, uploaded, posted,
  * transmitted, distributed, or disclosed in any way without NXP's prior
  * express written permission.
  *
  * No license under any patent, copyright, trade secret or other intellectual
  * property right is granted to or conferred upon you by disclosure or delivery
  * of the Materials, either expressly, by implication, inducement, estoppel or
  * otherwise. Any license under such intellectual property rights must be
  * express and approved by NXP in writing.
  *
  */

#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include <sys/time.h>
#include <unistd.h>
#include <pthread.h>
#include "wlmgr_debug.h"

#define WLMGR_HOUR_TO_SECOND    3600

#ifdef DEBUG_ENABLED
u8 wlmgr_debug_flag = 1;
#else
u8 wlmgr_debug_flag = 0;
#endif /* DEBUG_ENABLED */

u8 wlmgr_debug_level = WLMGR_INFO;
u8 wlmgr_debug_timestamp = 1;

static FILE *out_file = NULL;

static u8 wlmgr_debug_file_name[WLMGR_DEBUG_FILE_NAME_LEN];
static u8 wlmgr_dbg_file[WLMGR_DEBUG_FILE_NAME_LEN];
static s32 wlmgr_debug_file_count = 0;
static struct timespec wlmgr_keep_file_ts;

static pthread_mutex_t wlmgr_debug_mutex;

static s32 wlmgr_debug_keep_file(void);

void wlmgr_debug_set_flag(u8 flag)
{
    wlmgr_debug_flag = flag;
}

u8 wlmgr_debug_get_flag(void)
{
    return wlmgr_debug_flag;
}

void wlmgr_debug_set_level(u8 level)
{
    if ((level >= WLMGR_EXCESSIVE) && (level <= WLMGR_ERROR))
        wlmgr_debug_level = level;
}

u8 wlmgr_debug_get_level(void)
{
    return wlmgr_debug_level;
}

void wlmgr_debug_print_timestamp(void)
{
    struct timespec ts;
    s32 res;

    if (!wlmgr_debug_timestamp)
        return;

    clock_gettime(CLOCK_MONOTONIC, &ts);
    if (out_file) {
        if ((ts.tv_sec - wlmgr_keep_file_ts.tv_sec) > WLMGR_HOUR_TO_SECOND) {
            wlmgr_debug_keep_file();
        }
        fprintf(out_file, "%ld.%06u: ", (long) ts.tv_sec, (unsigned int) ts.tv_nsec/1000);
    } else {
        printf("%ld.%06u: ", (long) ts.tv_sec, (unsigned int) ts.tv_nsec/1000);
    }
}


/**
 * _wlmgr_printf - conditional printf
 * @level: priority level (WLMGR_*) of the message
 * @lf: New line '\n' is added to the end of the text when printing to stdout
 * @fmt: printf format string, followed by optional arguments
 *
 */
void _wlmgr_printf(u8 level, u8 lf, const u8 *fmt, ...)
{
    va_list ap;

    if (!wlmgr_debug_flag)
        return;

    pthread_mutex_lock(&wlmgr_debug_mutex);

    va_start(ap, fmt);
    if (level >= wlmgr_debug_level) {
        if (lf)
            wlmgr_debug_print_timestamp();
        if (out_file) {
            vfprintf(out_file, fmt, ap);
            if (lf)
                fprintf(out_file, "\n");
        } else {
            vprintf(fmt, ap);
            if (lf)
                printf("\n");
        }
    }

    va_end(ap);

    pthread_mutex_unlock(&wlmgr_debug_mutex);
}

s32 wlmgr_debug_open_file(const u8 *path)
{
    s32 res;

    if (!path)
        return 0;

	res = pthread_mutex_init(&wlmgr_debug_mutex,NULL);

    if (out_file) {
        wlmgr_debug_close_file();
    }
    memset(wlmgr_debug_file_name, 0, sizeof(wlmgr_debug_file_name));
    memcpy(wlmgr_debug_file_name, path, strlen(path));

    memset(wlmgr_dbg_file, 0, sizeof(wlmgr_dbg_file));
    sprintf(wlmgr_dbg_file, "%s_%d", wlmgr_debug_file_name, wlmgr_debug_file_count);
    wlmgr_debug_file_count ++;

    out_file = fopen(wlmgr_dbg_file, "w");
    if (out_file == NULL) {
        wlmgr_printf(WLMGR_ERROR, "wpa_debug_open_file: Failed to open "
                                    "output file, using standard output");
        return -1;
    }
    clock_gettime(CLOCK_MONOTONIC, &wlmgr_keep_file_ts);
    return 0;
}

static s32 wlmgr_debug_keep_file(void)
{
    s32 res;

    if (!out_file) {
        return 0;
    }

    clock_gettime(CLOCK_MONOTONIC, &wlmgr_keep_file_ts);
    if (wlmgr_debug_file_count >= 10) {
        /* delete old debug file */
        memset(wlmgr_dbg_file, 0, sizeof(wlmgr_dbg_file));
        sprintf(wlmgr_dbg_file, "%s_%d", wlmgr_debug_file_name, (wlmgr_debug_file_count-10));
        unlink(wlmgr_dbg_file);
    }
    /* create new debug file */
    wlmgr_debug_close_file();
    memset(wlmgr_dbg_file, 0, sizeof(wlmgr_dbg_file));
    sprintf(wlmgr_dbg_file, "%s_%d", wlmgr_debug_file_name, (wlmgr_debug_file_count));
    wlmgr_debug_file_count ++;

    out_file = fopen(wlmgr_dbg_file, "w");
    if (out_file == NULL) {
        wlmgr_printf(WLMGR_ERROR, "wpa_debug_open_file: Failed to open "
                                    "output file, using standard output");
        return -1;
    }
    return 0;
}

void wlmgr_debug_close_file(void)
{
	if (!out_file)
		return;
	fclose(out_file);
	out_file = NULL;
}


