/** @file ctrl_iface.c
  *
  * @brief This file contains WLAN application specific defines etc.
  *
  * Copyright 2019-2020 NXP
  *
  * NXP CONFIDENTIAL
  * The source code contained or described herein and all documents related to
  * the source code ("Materials") are owned by NXP, its
  * suppliers and/or its licensors. Title to the Materials remains with NXP,
  * its suppliers and/or its licensors. The Materials contain
  * trade secrets and proprietary and confidential information of NXP, its
  * suppliers and/or its licensors. The Materials are protected by worldwide copyright
  * and trade secret laws and treaty provisions. No part of the Materials may be
  * used, copied, reproduced, modified, published, uploaded, posted,
  * transmitted, distributed, or disclosed in any way without NXP's prior
  * express written permission.
  *
  * No license under any patent, copyright, trade secret or other intellectual
  * property right is granted to or conferred upon you by disclosure or delivery
  * of the Materials, either expressly, by implication, inducement, estoppel or
  * otherwise. Any license under such intellectual property rights must be
  * express and approved by NXP in writing.
  *
  */

#include <malloc.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <errno.h>
#include <pthread.h>
#include "wlmgr.h"
#include "ctrl_iface.h"
#include "wlmgr_debug.h"

static int ctrl_iface_cmd_get(wlmgr_bss_t *wlmgr, char *cmd,
                                    char *reply, size_t *reply_len)
{
    int ret = 0;

    wlmgr_printf(WLMGR_DEBUG, "CTRL_IFACE GET \"%s\"", cmd);

    ret = wlmgr_cmd_get((void *)wlmgr, cmd, reply, reply_len);

    return ret;
}

static int ctrl_iface_cmd_set(wlmgr_bss_t *wlmgr, char *cmd,
                              char *reply, size_t *reply_len)
{
    int ret = 0;
    *reply_len = 0;

    wlmgr_printf(WLMGR_DEBUG, "CTRL_IFACE SET \"%s\"", cmd);

    ret = wlmgr_cmd_set((void *)wlmgr, cmd);

    return ret;
}

static void
ctrl_iface_receive_parse(wlmgr_bss_t *wlmgr, const u8 *src_addr,
                               char *buf, size_t len,
                               char *reply, size_t *reply_len)
{
    int ret = 0;

    buf[len] = '\0';

    if (strncmp(buf, "GET ", 4) == 0)
        ret = ctrl_iface_cmd_get(wlmgr, buf + 4, reply, reply_len);
    else if (strncmp(buf, "SET ", 4) == 0)
        ret = ctrl_iface_cmd_set(wlmgr, buf + 4, reply, reply_len);
    else {
        ret = -2;
        printf("unknown command\n");
    }

    if (ret == 1)
        return;

    if (ret == 0) {
        if (*reply_len == 0) {
            *reply_len = strlen("OK\n");
            strncpy(reply, "OK\n", *reply_len);
        }
    } else {
        *reply_len = strlen("FAIL\n");
        strncpy(reply, "FAIL\n", *reply_len);
    }
}

static void ctrl_iface_receive(void *ctx)
{
    wlmgr_bss_t *wlmgr = (wlmgr_bss_t *)ctx;
    char recvBuf[2048];
    size_t recvLen;
    struct sockaddr_ll from;
    socklen_t fromlen;
    char replyBuf[2048];
    size_t replyLen;

    if (!wlmgr)
        return;

    while (1) {
        memset(&from, 0, sizeof(struct sockaddr_ll));
        memset(recvBuf, 0, sizeof(recvBuf));

        fromlen = sizeof(struct sockaddr_ll);
        recvLen = recvfrom(wlmgr->ctrl_iface, recvBuf, sizeof(recvBuf), 0,
                           (struct sockaddr *)&from, &fromlen);
        if (recvLen < 0) {
            wlmgr_printf(WLMGR_ERROR, "%s: recvfrom failed: %s", __func__, strerror(errno));
            sleep(1);
	    continue;
        }

        memset(replyBuf, 0, sizeof(replyBuf));
        replyLen = 0;
        ctrl_iface_receive_parse(wlmgr, from.sll_addr, recvBuf, recvLen,
                                       replyBuf, &replyLen);

        sendto(wlmgr->ctrl_iface, replyBuf, replyLen, 0,
               (struct sockaddr *)&from, fromlen);
    }
}

void ctrl_iface_deinit(s32 ctrl_sock)
{

    if (ctrl_sock >= 0) {
        close(ctrl_sock);
        ctrl_sock = -1;
    }
}

s32 ctrl_iface_init(void *data, const char *dir, const char *file)
{
    struct sockaddr_un source;
    s32 ctrl_sock = -1;
    u8 *ctrl_path;
    size_t len;
    pthread_t t_ctrl_iface;

    if ((data == NULL) || (dir == NULL) || (file == NULL) ||
        (strlen(dir) == 0) || (strlen(file) == 0))
        return -1;

    if (mkdir(dir, S_IRWXU | S_IRWXG) < 0) {
        if (errno == EEXIST) {
            wlmgr_printf(WLMGR_DEBUG, "Using existing control interface directory");
        } else {
            wlmgr_printf(WLMGR_ERROR, "mkdir[ctrl_iface_dir %s] failed", dir);
            goto exit;
        }
    }

    ctrl_sock = socket(PF_UNIX, SOCK_DGRAM, 0);
    if (ctrl_sock < 0) {
        wlmgr_printf(WLMGR_ERROR, "%s: socket(PF_UNIX) failed: %s", __func__, strerror(errno));
        goto exit;
    }

    len = strlen(dir) + strlen(file) + 2;
    if (len >= sizeof(source.sun_path))
        goto exit;

    ctrl_path = (char *)wlmgr_malloc(len);
    if (ctrl_path == NULL)
        goto exit;

    snprintf(ctrl_path, len, "%s/%s", dir, file);
    ctrl_path[len - 1] = '\0';

    memset(&source, 0, sizeof(source));
    source.sun_family = AF_UNIX;
    strncpy(source.sun_path, ctrl_path, sizeof(source.sun_path));

    if (bind(ctrl_sock, (struct sockaddr *)&source, sizeof(source)) < 0) {
        //wlmgr_printf(WLMGR_ERROR, "bind(PF_UNIX) failed: %s", strerror(errno));
        if (connect(ctrl_sock, (struct sockaddr *)&source, sizeof(source)) < 0) {
             //wlmgr_printf(WLMGR_ERROR, "ctrl_iface exists, but does not allow connections");
            if (unlink(ctrl_path) < 0) {
                //wlmgr_printf(WLMGR_ERROR, "unlink[ctrl_iface %s] failed", ctrl_path);
                goto exit;
            }
            if (bind(ctrl_sock, (struct sockaddr *) &source, sizeof(source)) < 0) {
                //wlmgr_printf(WLMGR_ERROR, "bind(PF_UNIX) failed: %s", strerror(errno));
                goto exit;
            }
            //wlmgr_printf(WLMGR_INFO, "Replaced leftover ctrl_iface socket '%s' Successfully",
            //       ctrl_path);
        } else {
            wlmgr_printf(WLMGR_INFO, "ctrl_iface exists and seems to be in use - cannot override it");
            wlmgr_printf(WLMGR_INFO, "Delete '%s' manually if it is not used anymore", ctrl_path);
            goto exit;
        }
    }

    if (chmod(ctrl_path, S_IRWXU | S_IRWXG) < 0) {
        wlmgr_printf(WLMGR_ERROR, "chmod[ctrl_interface/ifname] failed");
        goto exit;
    }

    if (pthread_create(&t_ctrl_iface, NULL,
                       (void *)ctrl_iface_receive, data) != 0) {
        wlmgr_printf(WLMGR_ERROR, "%s: pthread_create failed", __func__);
        goto exit;
    }

    wlmgr_free(ctrl_path);

    return ctrl_sock;

exit:
    if (ctrl_path) {
        unlink(ctrl_path);
        wlmgr_free(ctrl_path);
    }

    ctrl_iface_deinit(ctrl_sock);

    return -1;
}
