/** @file driver.c
  *
  * @brief This file contains WLAN application specific defines etc.
  *
  * Copyright 2019-2020 NXP
  *
  * NXP CONFIDENTIAL
  * The source code contained or described herein and all documents related to
  * the source code ("Materials") are owned by NXP, its
  * suppliers and/or its licensors. Title to the Materials remains with NXP,
  * its suppliers and/or its licensors. The Materials contain
  * trade secrets and proprietary and confidential information of NXP, its
  * suppliers and/or its licensors. The Materials are protected by worldwide copyright
  * and trade secret laws and treaty provisions. No part of the Materials may be
  * used, copied, reproduced, modified, published, uploaded, posted,
  * transmitted, distributed, or disclosed in any way without NXP's prior
  * express written permission.
  *
  * No license under any patent, copyright, trade secret or other intellectual
  * property right is granted to or conferred upon you by disclosure or delivery
  * of the Materials, either expressly, by implication, inducement, estoppel or
  * otherwise. Any license under such intellectual property rights must be
  * express and approved by NXP in writing.
  *
  */

#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <signal.h>
#include <netinet/in.h>
#include "driver.h"
#include "netlink.h"
#include "driver_LnxIoctl.h"
#include "wlmgr_debug.h"
#include "mu_mode_func.h"

#define KILO    1e3
#define MEGA    1e6
#define GIGA    1e9

#define WE_POINT_VERSION 18

static int we_version_compiled = 0;

static const char *ether_sprintf(const u8 *addr)
{
    static char buf[sizeof(MACSTR)];

    if (addr != NULL)
        snprintf(buf, sizeof(buf), MACSTR, MAC2STR(addr));
    else
        snprintf(buf, sizeof(buf), MACSTR, 0,0,0,0,0,0);

    return buf;
}

static int driver_commit(struct driver_data *drv) {
	struct iwreq iwr;

	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, drv->iface, IFNAMSIZ);

	if (ioctl(drv->ioctl_sock, SIOCSIWCOMMIT, &iwr) < 0) {
		wlmgr_printf(WLMGR_ERROR, "ioctl[SIOCSIWCOMMIT]");
		return -1;
	}

	return 0;
}

static int set80211param(struct driver_data *drv,
                         int param, void *arg, boolean commit)
{
    struct iwreq iwr;

    memset(&iwr, 0, sizeof(iwr));
    strncpy(iwr.ifr_name, drv->iface, IFNAMSIZ);
    iwr.u.mode = param;
    memcpy(iwr.u.name + sizeof(u32), arg, sizeof(arg));

    if (ioctl(drv->ioctl_sock, WL_IOCTL_WL_PARAM, &iwr) < 0) {
        wlmgr_printf(WLMGR_ERROR, "ioctl[WL_IOCTL_WL_PARAM] failed");
        return -1;
    }

    if (commit) {
        memset(&iwr, 0, sizeof(iwr));
        strncpy(iwr.ifr_name, drv->iface, IFNAMSIZ);
        if (ioctl(drv->ioctl_sock, SIOCSIWCOMMIT, &iwr) < 0) {
            wlmgr_printf(WLMGR_ERROR, "ioctl[SIOCSIWCOMMIT] failed");
            return -1;
        }
    }

    return 0;
}

static int get80211param(struct driver_data *drv, int param, void *value)
{
    struct iwreq iwr;

    memset(&iwr, 0, sizeof(iwr));
    strncpy(iwr.ifr_name, drv->iface, IFNAMSIZ);
    iwr.u.mode = param;

    if (ioctl(drv->ioctl_sock, WL_IOCTL_WL_GET_PARAM, &iwr) < 0) {
        wlmgr_printf(WLMGR_ERROR, "ioctl[WL_IOCTL_WL_GET_PARAM] failed");
        return -1;
    }

    if (sizeof(int) < IFNAMSIZ)
        memcpy(value, iwr.u.name, sizeof(iwr.u));

    return 0;
}

static int set80211priv(struct driver_data *drv, int cmd, void *data, int len)
{
    struct iwreq iwr;

    memset(&iwr, 0, sizeof(iwr));
    strncpy(iwr.ifr_name, drv->iface, IFNAMSIZ);

    iwr.u.data.pointer = data;
    iwr.u.data.length = len;

    if (ioctl(drv->ioctl_sock, cmd, &iwr) < 0) {
        wlmgr_printf(WLMGR_ERROR, "set80211priv ioctl failed!");
        return -1;
    }

    return 0;
}

static int get80211priv(struct driver_data *drv, int cmd, void *data, int len)
{
    struct iwreq iwr;

    memset(&iwr, 0, sizeof(iwr));

    strncpy(iwr.ifr_name, drv->iface, IFNAMSIZ);
    iwr.u.data.pointer = data;
    iwr.u.data.length = len;

    if (ioctl(drv->ioctl_sock, cmd, &iwr) < 0) {
        wlmgr_printf(WLMGR_ERROR, "get80211priv ioctl failed!");
        return -1;
    }

    return iwr.u.data.length;
}

static void driver_wireless_event_custom(struct driver_data *drv, char *custom, size_t len)
{
    //wlmgr_printf(WLMGR_DEBUG, "Custom driver event:%s", custom);

    if (strncmp(custom, "1905_BTM", 8) == 0) {
        wlmgr_printf(WLMGR_DEBUG, "Custom driver event:BTM Response");
    } else if (strncmp(custom, "wlmgr: ", 7) == 0) {
        if (strncmp(custom + 7, "mumode ", 7) == 0) {
            mums_custom_event((void *)drv, custom + 14);
        }
    }
}

static void driver_wireless_event(struct driver_data *drv,char *data, int len)
{
    struct iw_event iw_event, *iw_event_p = &iw_event;
    char *pos, *end, *event, *buf;
    size_t buf_len;

    pos = data;
    end = data + len;
    while (pos + IW_EV_LCP_LEN <= end) {
        if (pos == NULL)
            return;

        memcpy(&iw_event, pos, IW_EV_LCP_LEN);

        if (iw_event_p->len <= IW_EV_LCP_LEN)
            return;

        wlmgr_printf(WLMGR_EXCESSIVE, "Wireless event received, cmd=0x%x", iw_event_p->cmd);

        event = pos + IW_EV_POINT_LEN;
        if (drv->we_version > WE_POINT_VERSION &&
            (iw_event_p->cmd == IWEVMICHAELMICFAILURE ||
             iw_event_p->cmd == IWEVASSOCREQIE ||
             iw_event_p->cmd == IWEVCUSTOM)) {
            char *dpos = (char *) &iw_event.u.data.length;
            int dlen = dpos - (char *) &iw_event;
            memcpy(dpos, pos + IW_EV_LCP_LEN,
                    sizeof(struct iw_event) - dlen);
        } else {
            memcpy(&iw_event, pos, sizeof(struct iw_event));
            event += IW_EV_POINT_OFF;
        }

        switch (iw_event_p->cmd) {
        case IWEVEXPIRED:
            wlmgr_printf(WLMGR_DEBUG, "Driver event: IWEVEXPIRED");
            mums_sta_disassoc_event((void *)drv, (u8 *)iw_event_p->u.addr.sa_data);
            break;
        case IWEVREGISTERED:
            wlmgr_printf(WLMGR_DEBUG, "Driver event: IWEVREGISTERED");
            mums_sta_assoc_event((void *)drv, (u8 *)iw_event_p->u.addr.sa_data);
            break;
        case IWEVCUSTOM:
            if (event + iw_event_p->u.data.length > end)
                return;
            buf = wlmgr_malloc(iw_event_p->u.data.length + 1);
            if (buf == NULL)
                return;
            memset(buf, 0, sizeof(buf));
            memcpy(buf, event, iw_event_p->u.data.length);
            buf[iw_event_p->u.data.length] = '\0';
            buf_len = iw_event_p->u.data.length;
            driver_wireless_event_custom(drv, buf, buf_len);
            wlmgr_free(buf);
            break;
        }
        pos += iw_event_p->len;
    }
}

static void driver_wireless_event_newlink(void *ctx, struct ifinfomsg *ifi,
                                        u8 *buf, size_t len)
{
    struct driver_data *drv = (struct driver_data *)ctx;
    int attrlen, rta_len;
    struct rtattr *attr;

    if (ifi->ifi_index != drv->ifindex)
        return;

    /* header || ifinfomsg || rtattr(optional) */
    attrlen = len;
    attr = (struct rtattr *) buf;

    rta_len = RTA_ALIGN(sizeof(struct rtattr));
    while (RTA_OK(attr, attrlen)) {
        if (attr->rta_type == IFLA_WIRELESS) {
            driver_wireless_event(drv, ((char *) attr) + rta_len,
				             attr->rta_len - rta_len);
	}
        attr = RTA_NEXT(attr, attrlen);
    }
}

static int wl_get_range(struct driver_data *drv)
{
	struct iw_range *range;
	struct iwreq iwr;
	size_t buf_len;

	buf_len = sizeof(struct iw_range) + 500;
	range = wlmgr_malloc(buf_len);
	if (range == NULL) {
	    wlmgr_printf(WLMGR_ERROR, "wlmgr: allocate memory for iw_range failed");
	    return -ENOMEM;
	}
	memset(range, 0, buf_len);
	memset(&iwr, 0, sizeof(struct iwreq));
	iwr.u.data.pointer = (caddr_t) range;
	iwr.u.data.length = buf_len;
	strncpy(iwr.ifr_name, drv->iface, sizeof(iwr.ifr_name));

	if ((ioctl(drv->ioctl_sock, SIOCGIWRANGE, &iwr)) < 0) {
	    wlmgr_printf(WLMGR_ERROR, "ioctl[SIOCGIWRANGE] failed");
	    wlmgr_free(range);
	    return -1;
	}
	we_version_compiled = range->we_version_compiled;
	drv->we_version = we_version_compiled;
	//wlmgr_printf(WLMGR_INFO, "Driver build with Wireless Extension %d",
	//         drv->we_version);
	wlmgr_free(range);
	return 0;
}

static int wl_get_ifindex(struct driver_data *drv)
{
    struct ifreq ifr;

    memset(&ifr, 0, sizeof(ifr));
    strncpy(ifr.ifr_name, drv->iface, sizeof(ifr.ifr_name));

    if (ioctl(drv->ioctl_sock, SIOCGIFINDEX, &ifr) != 0) {
	 wlmgr_printf(WLMGR_ERROR, "ioctl[SIOCGIFINDEX] failed");
	 return -1;
    }
    drv->ifindex = ifr.ifr_ifindex;

    return 0;
}

static int wl_get_ifhwaddr(struct driver_data *drv)
{
    struct ifreq ifr;

    memset(&ifr, 0, sizeof(ifr));
    strncpy(ifr.ifr_name, drv->iface, sizeof(ifr.ifr_name));

    if (ioctl(drv->ioctl_sock, SIOCGIFHWADDR, &ifr) != 0) {
	 wlmgr_printf(WLMGR_ERROR, "ioctl[SIOCGIFHWADDR] failed");
	 return -1;
    }
    memcpy(drv->wdev_hwaddr, ifr.ifr_hwaddr.sa_data, ETH_ADDR_LEN);
    //wlmgr_printf(WLMGR_DEBUG, "Wireless interface: \"%s\" " MACSTR " #%d",
    //        drv->iface, MAC2STR(drv->wdev_hwaddr), drv->ifindex);

    return 0;
}

static int wl_get_sw_version(struct driver_data *drv)
{
    int len = -1;
    u8 *smac = NULL, *pfw = NULL, *drv_v = NULL, *ptr = NULL, *tmp = NULL;
    u8 version[64];
    u8 majorVer, minorVer, relVer, patchVer, pfwBuildVer, drvBuildVer;
    u16 pfwVer, drvVer;

    memset(version, 0,sizeof(version));
    len = get80211priv(drv, WL_IOCTL_GET_VERSION, version, 0);

    if (len >= sizeof(version))
        return -1;

    version[len - 1] = '\0';

    ptr = version + strlen("\n\n");
    tmp = strtok(ptr, "-");
    if (tmp != NULL) {
        smac = tmp;
        tmp = strtok(NULL, "-");
        if (tmp != NULL) {
            pfw = tmp;
            tmp = strtok(NULL, "-");
            if (tmp != NULL) {
                drv_v = tmp;
            }
        }
    }

    if (smac != NULL) {
        tmp = strtok(smac, ".");
        if (tmp != NULL) {
            majorVer = (u8)atoi(tmp);
            tmp = strtok(NULL, ".");
            if (tmp != NULL) {
                minorVer = (u8)atoi(tmp);
                tmp = strtok(NULL, ".");
                if (tmp != NULL) {
                    relVer = (u8)atoi(tmp);
                }
            }
        }
    }

    if ((pfw != NULL) && (*pfw == 'P')) {
        if ((pfw + 1) != NULL)
            pfwVer = (u16)atoi(pfw + 1);

        tmp = strchr(pfw, '.');
        if ((tmp != NULL) && (pfw + 5) != NULL)
            pfwBuildVer = (u8)atoi(pfw + 5);
    }

    if ((drv_v != NULL) && (*drv_v == 'D')) {
        if ((drv_v + 1) != NULL)
            drvVer = (u16)atoi(drv_v + 1);

        tmp = strchr(drv_v, '.');
        if ((tmp != NULL) && (drv_v + 5) != NULL)
            drvBuildVer = (u8)atoi(drv_v + 5);
    }

    drv->ver.major = majorVer;
    drv->ver.minor = minorVer;
    drv->ver.rel = relVer;
    drv->ver.pfw = pfwVer;
    drv->ver.pfwBuild = pfwBuildVer;
    drv->ver.drv = drvVer;
    drv->ver.drvBuild = drvBuildVer;

    return 0;
}

static void driver_drv_deinit(void* priv)
{
    struct driver_data *drv = (struct driver_data *)priv;

    if (drv == NULL)
        return;

    if (drv->netlink != NULL)
        netlink_deinit(drv->netlink);

    if (drv->ioctl_sock >= 0) {
        close(drv->ioctl_sock);
        drv->ioctl_sock = -1;
    }

    if (drv != NULL) {
        wlmgr_free(drv);
        drv = NULL;
    }
}

static void *driver_drv_init(void *data, const char *iface)
{
    struct driver_data *drv;

    drv = (struct driver_data *)wlmgr_malloc(sizeof(struct driver_data));
    if (drv == NULL) {
	wlmgr_printf(WLMGR_ERROR, "Could not allocate memory for nxp driver data");
        goto exit;
    }

    /* create sock to communicate with driver layer via ioctl */
    memset(drv, 0, sizeof(struct driver_data));
    drv->data = data;
    strncpy(drv->iface, iface, IFNAMSIZ);
    drv->ioctl_sock = socket(PF_INET, SOCK_DGRAM, 0);
    if (drv->ioctl_sock < 0) {
        wlmgr_printf(WLMGR_ERROR, "driver: socket(PF_INET) failed: %s", strerror(errno));
        goto exit;
    }

    if (wl_get_range(drv) < 0) {
        wlmgr_printf(WLMGR_ERROR, "driver: Cannot get range.");
        goto exit;
    }

    if (wl_get_ifindex(drv) < 0) {
        wlmgr_printf(WLMGR_ERROR, "driver: Cannot get ifindex.");
        goto exit;
    }

    if (wl_get_ifhwaddr(drv) < 0) {
        wlmgr_printf(WLMGR_ERROR, "driver: Cannot get ifhwaddr.");
        goto exit;
    }

    if (wl_get_sw_version(drv) < 0) {
        wlmgr_printf(WLMGR_ERROR, "driver: Cannot get SW version.");
        goto exit;
    }

    /* create sock to receive data from driver layer via netlink */
    drv->netlink = netlink_init(driver_wireless_event_newlink, NULL, (void *)drv);
    if (drv->netlink == NULL) {
        wlmgr_printf(WLMGR_ERROR, "driver: Cannot open netlink sock.");
        goto exit;
    }

    return (void *)drv;

exit:
    driver_drv_deinit(drv);

    return NULL;
}

double freq2float(struct iw_freq *in)
{
    int i;
    double res = (double) in->m;

    for(i = 0; i < in->e; i++)
        res *= 10;

    return(res);
}

int driver_get_bssid(void *priv, u8 *buf)
{
    struct driver_data *drv = (struct driver_data *)priv;
    int len = -1;
    u8 bssid[100];

    len = get80211priv(drv, WL_IOCTL_GET_BSSID, bssid, 0);

    if (len > 0) {
        /*the format of getten bssid is \"MAC xx:xx:xx:xx:xx:xx\"*/
        len -= strlen("MAC ");
        if (len > 0)
            strncpy((char *)buf, (char *)(bssid + strlen("MAC ")), len);
    }

    return len;
}

int driver_get_currchan(void *priv, u8 *buf)
{
    struct driver_data *drv = (struct driver_data *)priv;
    struct iwreq iwr;
    struct iw_range *range;
    double freq;
    double ref_freq;
    int i, channel = -1;

    memset(&iwr, 0, sizeof(iwr));
    strncpy(iwr.ifr_name, drv->iface, IFNAMSIZ);
    if (ioctl(drv->ioctl_sock, SIOCGIWFREQ, &iwr) < 0) {
        wlmgr_printf(WLMGR_ERROR, "ioctl[SIOCGIWFREQ] failed");
        return -1;
    }

    freq = freq2float(&(iwr.u.freq));
    if (freq < KILO)
        channel = (int)freq;
    else {
        memset(range, 0, sizeof(struct iw_range));
	memset(&iwr, 0, sizeof(struct iwreq));
	iwr.u.data.pointer = (caddr_t) range;
	iwr.u.data.length = sizeof(struct iw_range);
	strncpy(iwr.ifr_name, drv->iface, sizeof(iwr.ifr_name));

	if ((ioctl(drv->ioctl_sock, SIOCGIWRANGE, &iwr)) < 0) {
	    wlmgr_printf(WLMGR_ERROR, "ioctl[SIOCGIWRANGE] failed");
	    return -1;
	}

        for (i = 0; i < range->num_frequency; i++)
        {
            ref_freq = freq2float(&(range->freq[i]));
            if(freq == ref_freq)
	        channel = range->freq[i].i;
        }

        if(channel < 0)
            return -1;
    }
   *buf = (u8)channel;

    return 0;
}

unsigned int driver_get_chanutil(void *priv)
{
    struct driver_data *drv = (struct driver_data *)priv;
    u8 buf[256];
    u8 *pos = NULL;
    int chanutil = 0, len = -1;

    memset(buf, 0, sizeof(buf));
    memcpy(buf, "getchutil", strlen("getchutil"));

    len = get80211priv(drv, WL_IOCTL_GETCMD,
                       buf, strlen(buf) + 1);

    if (len < 0)
        return 0;
    else {
        buf[len] = '\0';

        pos = buf;
        if ((pos = strstr(pos, "chutil:"))) {
            chanutil = atoi(pos+strlen("chutil:"));
        }

      return chanutil;
    }
    return 1;
}

unsigned int driver_get_chanutil_by_cnt(void *priv)
{
    struct driver_data *drv = (struct driver_data *)priv;
    u8 buf[256];
    u8 *pos = NULL;
    int chanutil = 0, len = -1;

    memset(buf, 0, sizeof(buf));
    sprintf(buf, "getchutil -f bandsteer 1000 0 5");

    len = get80211priv(drv, WL_IOCTL_GETCMD,
                       buf, strlen(buf) + 1);

    return 1;
}

unsigned int driver_get_agingtime(void *priv)
{
    struct driver_data *drv = (struct driver_data *)priv;
    struct wlreq_key wk;
    unsigned int agingtime = 0;

    memset(&wk, 0, sizeof(wk));
    if (get80211param(drv, WL_PARAM_AGINGTIME, &agingtime) < 0)
        return 0;

    return agingtime;
}

unsigned int driver_get_stalist(void *priv, u8 *buf)
{
    struct driver_data *drv = (struct driver_data *)priv;
    int len = -1;

    memcpy(buf, "getstalist_for_msan", strlen("getstalist_for_msan"));

    len = get80211priv(drv, WL_IOCTL_GETCMD,
                       buf, strlen("getstalist_for_msan") + 1);
    buf[len] = '\0';

    return len;
}

unsigned int driver_get_stacnt(void *priv)
{
    struct driver_data *drv = (struct driver_data *)priv;
    u8 buf[256];
    unsigned int stacnt = 0;
    int len = -1;

    memset(buf, 0, sizeof(buf));
    memcpy(buf, "getstacnt", strlen("getstacnt"));

    len = get80211priv(drv, WL_IOCTL_GETCMD,
                       buf, strlen(buf) + 1);

    if (len < 0)
        return 0;
    else {
        buf[len] = '\0';

        stacnt = atoi(buf);

      return stacnt;
    }

    return 0;
}

unsigned short driver_get_maxsta(void *priv)
{
    FILE *fp_read = NULL;
    unsigned int sta_num = 0;

    fp_read = fopen("/sys/module/ap8x/parameters/sta_num", "r");
    if (fp_read != NULL) {
	fscanf(fp_read, "%d", &sta_num);
        fclose(fp_read);
    }

    return (unsigned short)sta_num;
}

DRV_DEV_ID driver_get_deviceid(void *priv)
{
    struct driver_data *drv = (struct driver_data *)priv;
    FILE *fp_read = NULL;
    char file_path[128];
    char *device_info = NULL;
    char *device_str = NULL;
    DRV_DEV_ID device_id = DRV_DEV_ID_FAIL;

    device_info = (char *) wlmgr_malloc(4096);
    if (device_info == NULL)
        return device_id;

    memset(device_info, 0, 4096);

    memset(file_path, 0, sizeof(file_path));
    sprintf(file_path, "/sys/class/net/%s/vap_info/info", drv->iface);
    fp_read = fopen(file_path, "r");
    if (fp_read != NULL) {
        char *pos = NULL;
        size_t size = 0;

        size = fread(device_info, 1, 4096, fp_read);
        fclose(fp_read);
        if (size > 0) {
            pos = strstr(device_info, "deviceid:        ");
            if (pos) {
                device_str = strtok(pos + strlen("deviceid:        "), " ");
            }
        }
    }

    if (strncmp(device_str, "SC5", 3) == 0)
        device_id = DRV_DEV_ID_SC5;
    else if (strncmp(device_str, "SCBT", 4) == 0)
        device_id = DRV_DEV_ID_SCBT;

    return device_id;
}

DRV_BW_TYPE driver_get_bandwidth(void *priv)
{
    struct driver_data *drv = (struct driver_data *)priv;
    struct wlreq_key wk;
    unsigned int bandwidth = 0;

    memset(&wk, 0, sizeof(wk));
    if (get80211param(drv, WL_PARAM_HTBANDWIDTH, &wk) < 0)
        return DRV_BW_TYPE_FAIL;

    memcpy(&bandwidth, &wk, 1);
    return (DRV_BW_TYPE)bandwidth;
}

static int
driver_config_btm_target(void *priv, const u8 *target, const u8 channel)
{
    struct driver_data *drv = (struct driver_data *)priv;
    char buf[100];
    char target_mac_str[18] = {0};
    char str_tmp[4] = {0};
    int i;

    for (i=0; i<ETH_ADDR_LEN; i++)
    {
        sprintf(str_tmp, "%02x", target[i]);
        strcat(target_mac_str, str_tmp);
    }

    memset(buf, 0, sizeof(buf));
    sprintf(buf, "nlistcfg %s 3 1 1 0 0 1 1 0 %d 9",
            target_mac_str, channel);

    wlmgr_printf(WLMGR_DEBUG, "Config btm target %s", ether_sprintf(target));

    return set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));
}

static int driver_send_wtsbtmreq(void *priv, const u8 *addr, const u8 *target)
{
    struct driver_data *drv = (struct driver_data *)priv;
    char buf[100];

    memset(buf, 0, sizeof(buf));
    sprintf(buf, "wts_btmreq %s ", ether_sprintf(addr));
    sprintf(buf + strlen(buf), "%s 1 1 0 1 50", ether_sprintf(target));

    wlmgr_printf(WLMGR_DEBUG, "Send BTM request to %s", ether_sprintf(addr));

    return set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));
}

static int driver_send_btmreq(void *priv, const u8 *addr)
{
    struct driver_data *drv = (struct driver_data *)priv;
    char buf[100];

    memset(buf, 0, sizeof(buf));
    sprintf(buf, "btmreq %s 1 1 0 1 50 10", ether_sprintf(addr));
    wlmgr_printf(WLMGR_DEBUG, "Send BTM request to %s\n", ether_sprintf(addr));

    return set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));
}

static int driver_set_btm(void *priv, const u8 enable)
{
    struct driver_data *drv = (struct driver_data *)priv;
    char buf[100];

    memset(buf, 0, sizeof(buf));
    sprintf(buf, "sta_btm_enable %d", enable);

    return set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));
}

static int driver_set_apsteermode(void *priv, u8 enable)
{
    struct driver_data *drv = (struct driver_data *)priv;
    char buf[100];

    wlmgr_printf(WLMGR_DEBUG, "%s AP Steering....", enable ? "Enabling" : "Disabling");

    memset(buf, 0, sizeof(buf));
    strcpy(buf, enable ? "ap_steer_enable" : "ap_steer_disable");

    return set80211priv(drv, WL_IOCTL_SETCMD, buf, sizeof(buf));
}

static int driver_get_aprssi(void *priv, u8 *buf)
{
    struct driver_data *drv = (struct driver_data *)priv;
    int len = -1;

    memcpy(buf, "getaprssi", strlen("getaprssi"));

    len = get80211priv(drv, WL_IOCTL_GETCMD,
                       buf, strlen("getaprssi") + 1);
    buf[len] = '\0';

    return len;
}

static int driver_send_disassoc(void *priv, const u8 *addr, int reason)
{
    struct driver_data *drv = (struct driver_data *)priv;
    struct wlreq_mlme mlme;

    memset(&mlme, 0, sizeof(struct wlreq_mlme));
    mlme.im_op = WL_MLME_DISASSOC;
    mlme.im_reason = reason;
    memcpy(mlme.im_macaddr, addr, ETH_ADDR_LEN);

    return set80211param(drv, WL_PARAM_MLME_REQ, (void *)&mlme, FALSE);
}

static int driver_set_acnt_enable(void *priv, u16 action)
{
    struct driver_data *drv = (struct driver_data *)priv;
    char buf[100];
    s32  ret = 0;
    u16  i;

    memset(buf, 0, sizeof(buf));
    sprintf(buf, "gettxscheinfo %s", action == 1 ? "enable" : "disable");
    wlmgr_printf(WLMGR_DEBUG, "%s %s", __func__, buf);
    ret = set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));
    if (ret < 0) {
        return ret;
    }
    memset(buf, 0, sizeof(buf));
    sprintf(buf, "HostSetAcntWithMu %d", action);
    wlmgr_printf(WLMGR_DEBUG, "%s %s", __func__, buf);
    return set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));
}

static int driver_set_muset(void *priv, u16 sta_cnt, u8 option, u8 gid, u16 *StnId)
{
    struct driver_data *drv = (struct driver_data *)priv;
    char buf[100];
    char value[5];
    u16  i;

    memset(buf, 0, sizeof(buf));
    sprintf(buf, "HostSetMUSet %d %d %d", option, gid, sta_cnt);
    for (i = 0; i < sta_cnt; i ++) {
        memset(value, 0, sizeof(value));
        sprintf(value, " %d", StnId[i]);
        strcat(buf, value);
    }
    wlmgr_printf(WLMGR_DEBUG, "%s %s", __func__, buf);
    return set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));
}

static int driver_set_ofdma_config(void *priv, u8 enable, u8 sta_count)
{
    struct driver_data *drv = (struct driver_data *)priv;
    char buf[100];
    u16  i;

    if (sta_count > 16) {
        wlmgr_printf(WLMGR_ERROR, "SetOfdma sta count(%d) > 16, Failed!", sta_count);
        return -1;
    }
    memset(buf, 0, sizeof(buf));
    sprintf(buf, "SetOfdma %d 0 20000 %d", enable, sta_count);
    wlmgr_printf(WLMGR_DEBUG, "%s %s", __func__, buf);
    return set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));
}

static int driver_set_ofdma(void *priv, u8 enable, u8 sta_cnt, u16 *StnId)
{
    struct driver_data *drv = (struct driver_data *)priv;
    char buf[100];
    char value[5];
    u16  i;

    memset(buf, 0, sizeof(buf));
    sprintf(buf, "HostSetOfdma %d %d", enable, sta_cnt);
    for (i = 0; i < sta_cnt; i ++) {
        memset(value, 0, sizeof(value));
        sprintf(value, " %d", StnId[i]);
        strcat(buf, value);
    }
    wlmgr_printf(WLMGR_DEBUG, "%s %s", __func__, buf);
    return set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));
}

static int driver_set_ul_muset(void *priv, u8 action, u32 rateinfo, u32 flag, u8 gid, u8 mode, u8 bw, u16 sta_cnt, u32 param_cnt, u32 *param, u32 mu_version)
{
    struct driver_data  *drv = (struct driver_data *)priv;
    char                *buf, *pos;
    char                pading[100];
    u32                 param1, param2;
    u16                 i;
    s32                 ret = 0;

    if (drv == NULL || drv->data == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s driver not ready", __func__);
        return -1;
    }

    buf = (char *)wlmgr_malloc(1024);
    if (buf == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s allocate memory failed", __func__);
        return -1;
    }

    memset(buf, 0, sizeof(char) * 1024);
    sprintf(buf, "HostSetULMUSet %d %d 0x%08x %d %d %d %d", action, gid, rateinfo, flag, mode, bw, sta_cnt);
    if (param != NULL) {
        for (i = 0; i < param_cnt; i+=4) {
            memset(pading, 0, sizeof(pading));
            if (mu_version >= DRV_MU_VERSION_1) {
                sprintf(pading, " 0x%x 0x%x %d %d", param[i], param[i+1], param[i+2], param[i+3]);
            } else {
                sprintf(pading, " 0x%x 0x%x %d", param[i], param[i+1], param[i+2]);
            }
            strcat(buf, pading);
        }
    }
    wlmgr_printf(WLMGR_DEBUG, "%s %s", __func__, buf);
    ret = set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));
    wlmgr_free(buf);

    return ret;
}

static u8 driver_get_dl_gid(void *priv)
{
    struct driver_data *drv = (struct driver_data *)priv;
    u8    *buf, gid;
    s32   len = -1;

    buf = (char *)wlmgr_malloc(256);
    if (buf == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s allocate memory failed", __func__);
        return 0;
    }

    memset(buf, 0, sizeof(char) * 256);
    memcpy(buf, "HostGetDlGid", strlen("HostGetDlGid"));

    len = get80211priv(drv, WL_IOCTL_GETCMD,
                       buf, strlen(buf) + 1);
    buf[len] = '\0';
    gid = atoi(buf);
    wlmgr_free(buf);

    return gid;
}

static u8 driver_get_ul_gid(void *priv)
{
    struct driver_data *drv = (struct driver_data *)priv;
    u8    *buf, gid;
    s32   len = -1;

    buf = (char *)wlmgr_malloc(256);
    if (buf == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s allocate memory failed", __func__);
        return 0;
    }

    memset(buf, 0, sizeof(char) * 256);
    memcpy(buf, "HostGetUlGid", strlen("HostGetUlGid"));

    len = get80211priv(drv, WL_IOCTL_GETCMD,
                       buf, strlen(buf) + 1);
    buf[len] = '\0';
    gid = atoi(buf);
    wlmgr_free(buf);

    return gid;
}

static s32 driver_set_dl_gid(void *priv, u8 gid)
{
    struct driver_data *drv = (struct driver_data *)priv;
    u8      *buf;
    char    value[5];
    u16     i;
    s32     ret = 0;

    buf = (char *)wlmgr_malloc(256);
    if (buf == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s allocate memory failed", __func__);
        return 0;
    }
    memset(buf, 0, 256);
    sprintf(buf, "HostDelDlGid %d", gid);
    wlmgr_printf(WLMGR_DEBUG, "%s %s", __func__, buf);
    ret = set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));
    wlmgr_free(buf);

    return ret;
}

static s32 driver_set_ul_gid(void *priv, u8 gid)
{
    struct driver_data *drv = (struct driver_data *)priv;
    u8      *buf;
    char    value[5];
    u16     i;
    s32     ret = 0;

    buf = (char *)wlmgr_malloc(256);
    if (buf == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s allocate memory failed", __func__);
        return 0;
    }
    memset(buf, 0, 256);
    sprintf(buf, "HostDelUlGid %d", gid);
    wlmgr_printf(WLMGR_DEBUG, "%s %s", __func__, buf);
    ret = set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));
    wlmgr_free(buf);

    return ret;
}

static u8 driver_get_dl_ofdma(void *priv)
{
    struct driver_data *drv = (struct driver_data *)priv;
    u8    *buf, gid;
    s32   len = -1;

    buf = (char *)wlmgr_malloc(256);
    if (buf == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s allocate memory failed", __func__);
        return 0;
    }

    memset(buf, 0, sizeof(char) * 256);
    memcpy(buf, "Host_dl_ofdma", strlen("Host_dl_ofdma"));

    len = get80211priv(drv, WL_IOCTL_GETCMD,
                       buf, strlen(buf) + 1);
    buf[len] = '\0';
    gid = atoi(buf);
    wlmgr_free(buf);

    return gid;
}

static u8 driver_get_dl_mimo(void *priv)
{
    struct driver_data *drv = (struct driver_data *)priv;
    u8    *buf, gid;
    s32   len = -1;

    buf = (char *)wlmgr_malloc(256);
    if (buf == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s allocate memory failed", __func__);
        return 0;
    }

    memset(buf, 0, sizeof(char) * 256);
    memcpy(buf, "Host_dl_mimo", strlen("Host_dl_mimo"));

    len = get80211priv(drv, WL_IOCTL_GETCMD,
                       buf, strlen(buf) + 1);
    buf[len] = '\0';
    gid = atoi(buf);
    wlmgr_free(buf);

    return gid;
}

static u8 driver_get_ul_ofdma(void *priv)
{
    struct driver_data *drv = (struct driver_data *)priv;
    u8    *buf, gid;
    s32   len = -1;

    buf = (char *)wlmgr_malloc(256);
    if (buf == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s allocate memory failed", __func__);
        return 0;
    }

    memset(buf, 0, sizeof(char) * 256);
    memcpy(buf, "Host_ul_ofdma", strlen("Host_ul_ofdma"));

    len = get80211priv(drv, WL_IOCTL_GETCMD,
                       buf, strlen(buf) + 1);
    buf[len] = '\0';
    gid = atoi(buf);
    wlmgr_free(buf);

    return gid;
}

static u8 driver_get_ul_mimo(void *priv)
{
    struct driver_data *drv = (struct driver_data *)priv;
    u8    *buf, gid;
    s32   len = -1;

    buf = (char *)wlmgr_malloc(256);
    if (buf == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s allocate memory failed", __func__);
        return 0;
    }

    memset(buf, 0, sizeof(char) * 256);
    memcpy(buf, "Host_ul_mimo", strlen("Host_ul_mimo"));

    len = get80211priv(drv, WL_IOCTL_GETCMD,
                       buf, strlen(buf) + 1);
    buf[len] = '\0';
    gid = atoi(buf);
    wlmgr_free(buf);

    return gid;
}

static s32 driver_set_muedca(void *priv, u8 on_off)
{
    struct driver_data *drv = (struct driver_data *)priv;
    int val = (int)on_off;
    s32     ret = 0;

    wlmgr_printf(WLMGR_DEBUG, "%s %d", __func__, val);
    ret = set80211param(drv, WL_PARAM_MU_EDCA_EN, &val, FALSE);
    return ret;
}

static s32 driver_set_fw_mib(void *priv, char *mib_name, u8 valcnt, u32 *val)
{
    struct driver_data *drv = (struct driver_data *)priv;
    u8      *buf;
    char    value[20];
    u16     i;
    s32     ret = 0;

    buf = (char *)wlmgr_malloc(256);
    if (buf == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s allocate memory failed", __func__);
        return 0;
    }
    memset(buf, 0, 256);
    sprintf(buf, "fw_mib %s ", mib_name);
    for (i=0 ; i<valcnt ; i++) {
        sprintf(value, " %d", val[i]);
        strcat(buf, value);
    }

    wlmgr_printf(WLMGR_DEBUG, "%s %s\n", __func__, buf);
    ret = set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));

    wlmgr_free(buf);
    return ret;
}

u8 driver_compare_drv_ver(void *priv, u16 version)
{
    struct driver_data *drv = (struct driver_data *)priv;

    if (version >= drv->ver.drv)
        return 1;
    else
        return 0;
}

static s32 driver_set_sched_mode(void *priv, u8 *mode, u8 *on_off)
{
    struct driver_data *drv = (struct driver_data *)priv;
    u8      *buf;
    u16     buf_len;
    s32     ret = 0;

    if (mode == NULL || on_off == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s parameter NULL", __func__);
        return -1;
    }
    buf_len = strlen(mode) + strlen(on_off) + 16;
    buf = (char *)wlmgr_malloc(buf_len);
    if (buf == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s allocate memory failed", __func__);
        return -1;
    }
    memset(buf, 0, buf_len);
    sprintf(buf, "sched_mode %s %s", mode, on_off);
    wlmgr_printf(WLMGR_DEBUG, "%s %s\n", __func__, buf);
    ret = set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));

    wlmgr_free(buf);
    return ret;
}

static s32 driver_set_dl_expiretime(void *priv, u32 expiretime)
{
    struct driver_data *drv = (struct driver_data *)priv;
    u8      *buf;
    u16     buf_len;
    s32     ret = 0;

    buf = (char *)wlmgr_malloc(256);
    if (buf == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s allocate memory failed", __func__);
        return -1;
    }
    memset(buf, 0, 256);
    sprintf(buf, "setreg addr 0x20000610 %d", expiretime);
    wlmgr_printf(WLMGR_DEBUG, "%s %s\n", __func__, buf);
    ret = set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));

    wlmgr_free(buf);
    return ret;
}

static s32 driver_set_sta_aggregation(void *priv, u8 enable, u16 StnId)
{
    struct driver_data *drv = (struct driver_data *)priv;
    u8      *buf;
    u16     buf_len;
    s32     ret = 0;

    if (enable != 0 && enable != 2) {
        wlmgr_printf(WLMGR_ERROR, "%s parameter NULL", __func__);
        return -1;
    }
    buf = (char *)wlmgr_malloc(256);
    if (buf == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s allocate memory failed", __func__);
        return 0;
    }
    memset(buf, 0, 256);
    if (enable == 0) {
        /* Disable AMSDU and set AMPDU length to 1 */
        sprintf(buf, "changeba %d 0xFF 0 1", StnId);
    } else if (enable == 1) {
        /* Enable AMSDU and set AMPDU length to 16 */
        sprintf(buf, "changeba %d 0xFF 1 16", StnId);
    } else if (enable == 2) {
        /* Restore AMSDU/AMPDU setting */
        sprintf(buf, "changeba %d 0xFE 0 32", StnId);
    }
    wlmgr_printf(WLMGR_DEBUG, "%s %s", __func__, buf);
    ret = set80211priv(drv, WL_IOCTL_SETCMD, buf, strlen(buf));
    wlmgr_free(buf);

    return ret;
}


const struct driver_ops driver_ops = {
    .name               = "nxp",
    .init               = driver_drv_init,
    .deinit             = driver_drv_deinit,
    .get_bssid          = driver_get_bssid,
    .get_curchan        = driver_get_currchan,
    .get_chanutil       = driver_get_chanutil,
    .get_stacnt         = driver_get_stacnt,
    .get_maxsta         = driver_get_maxsta,
    .get_agingtime      = driver_get_agingtime,
    .get_stalist        = driver_get_stalist,
    .get_deviceid       = driver_get_deviceid,
    .get_bandwidth      = driver_get_bandwidth,
    .config_btm_target  = driver_config_btm_target,
    .send_wtsbtmreq     = driver_send_wtsbtmreq,
    .send_btmreq        = driver_send_btmreq,
    .set_btm            = driver_set_btm,
    .set_apsteermode    = driver_set_apsteermode,
    .get_aprssi         = driver_get_aprssi,
    .send_disassoc      = driver_send_disassoc,
    .set_acnt_enable    = driver_set_acnt_enable,
    .set_muset          = driver_set_muset,
    .set_ofdma_config   = driver_set_ofdma_config,
    .set_ofdma          = driver_set_ofdma,
    .set_ul_muset       = driver_set_ul_muset,
    .get_dl_gid         = driver_get_dl_gid,
    .get_ul_gid         = driver_get_ul_gid,
    .set_dl_gid         = driver_set_dl_gid,
    .set_ul_gid         = driver_set_ul_gid,
    .get_dl_ofdma       = driver_get_dl_ofdma,
    .get_dl_mimo        = driver_get_dl_mimo,
    .get_ul_ofdma       = driver_get_ul_ofdma,
    .get_ul_mimo        = driver_get_ul_mimo,
    .set_muedca         = driver_set_muedca,
    .set_fw_mib         = driver_set_fw_mib,
    .compare_drv_ver    = driver_compare_drv_ver,
    .set_sched_mode     = driver_set_sched_mode,
    .set_dl_expiretime  = driver_set_dl_expiretime,
    .set_sta_aggregation       = driver_set_sta_aggregation,
};

const struct driver_ops *const drivers[] =
{
    &driver_ops,
    NULL
};

