/** @file mu_mode.h
  *
  * @brief This file contains WLAN application specific defines etc.
  *
  * Copyright 2019-2020 NXP
  *
  * NXP CONFIDENTIAL
  * The source code contained or described herein and all documents related to
  * the source code ("Materials") are owned by NXP, its
  * suppliers and/or its licensors. Title to the Materials remains with NXP,
  * its suppliers and/or its licensors. The Materials contain
  * trade secrets and proprietary and confidential information of NXP, its
  * suppliers and/or its licensors. The Materials are protected by worldwide copyright
  * and trade secret laws and treaty provisions. No part of the Materials may be
  * used, copied, reproduced, modified, published, uploaded, posted,
  * transmitted, distributed, or disclosed in any way without NXP's prior
  * express written permission.
  *
  * No license under any patent, copyright, trade secret or other intellectual
  * property right is granted to or conferred upon you by disclosure or delivery
  * of the Materials, either expressly, by implication, inducement, estoppel or
  * otherwise. Any license under such intellectual property rights must be
  * express and approved by NXP in writing.
  *
  */


#ifndef mu_mode_H_
#define mu_mode_H_

#include <mqueue.h>
#include "utils.h"
#include "ieeetypes.h"
#include "stadb.h"

//#define MU_MODE_FS_DIR_PATH             "/root/mu_mode/"
#define MU_MODE_FS_DIR_PATH             "/sys/class/net/"

#define MU_GROUP_ID_MAX     64
#define MU_TOTAL_USERS      320
#define MU_MAX_USERS        16      /* MAX user of one Group */
#define OFDMA_MAX_USERS     16      /* MAX user of OFDMA */
#define MU_MAX_GROUPS       8       /* MAX Group */

#define MU_MODE_QUEUE_NAME  "/mums_mq"
#define MU_MODE_MSG_LEN     512 /* sizeof(struct MU_Notify_t) */

#define HostCmd_ACT_GEN_SET 0x0001
#define HostCmd_ACT_GEN_DEL 0x0002

/* Mapping to SFW define */
#define TXINFO_TF_TYPE_MUMIMO       1
#define TXINFO_TF_TYPE_OFDMA        2

/* Feature enable/disable/force */
#define MUMS_DISABLE        0
#define MUMS_ENABLE         1
#define MUMS_FORCE          2

#define TF_OFDMA_TXOP               300    /* 300 us txop, p#2 of "SCBT -Basic UL test cases.pptx"*/

#define MUMS_SCBT_MAX_NSS       4
#define MUMS_SC5_MAX_NSS        8
#define MUMS_MAX_NSS            MUMS_SCBT_MAX_NSS

#define MUMS_PRE_MODE_COUNTS            3
#define MUMS_OFDMA_AIRTIME_UP_LIMIT     300

typedef enum _mode_type
{
    MODE_IS_SU = 0,
    MODE_IS_MU
} mode_type;

typedef enum {
    RSSI_NEAR,
    RSSI_MIDDLE,
    RSSI_FAR,
    RSSI_END
} MU_MODE_RANGE_e;

typedef enum {
    RANGE_IDLE,             /* Can be send to do advanced grouping */
    RANGE_IN_PROCESS,       /* range list still in advanced grouping */
    RANGE_CALC_DONE,        /* advanced grouping done */
    RANGE_STATUS_END
} mums_range_status;

typedef enum {
    CALC_NEXT,              /* After calculate, one more group list */
    CALC_LAST_GROUP,        /* This is last group list */
    CALC_SKIP,              /* SM data failed, Skip advanced grouping */
    CALC_END
} mums_calc_status;

/**
 * enum SFW_BW_TYPE - BandWidth types
 */
typedef enum
{
    SFW_BW_TYPE_20MHZ,
    SFW_BW_TYPE_40MHZ,
    SFW_BW_TYPE_80MHZ,
    SFW_BW_TYPE_160MHZ,
    SFW_BW_TYPE_MAX
} SFW_BW_TYPE;


typedef struct MUCapStaNode_t
{
    struct MUCapStaNode_t   *nxt;
    struct MUCapStaNode_t   *prv;
    struct stadb_t          *StaInfo_p;
    u8                      find;    /* been find */
} MUCapStaNode_t;


typedef struct Group_List
{
    u16     StnId[MU_MAX_USERS];
    u8      gid;                    /* group id */
    u16     cnt;                    /* total MU capable station in list */
    u8      mode;                   /* MS_DL_MIMO_11AC or MS_DL_MIMO_11AX */
    u8      range;                  /* RSSI of range RSSI_NEAR/RSSI_MIDDLE/RSSI_FAR */
    u8      changed;                /* list members changed */
    u8      done;                   /* already do advance grouping */
} Group_List;

typedef struct Range_List
{
    u16                 StnId[MU_TOTAL_USERS];
    u16                 cnt;                    /* total MU capable station in list */
    u8                  mode;                   /* MS_DL_MIMO_11AC or MS_DL_MIMO_11AX */
    boolean             changed;                /* list members changed */
    mums_range_status   status;                 /* range list status */
} Range_List;

/**
 * MU_List - a STA list that have been selected mode.
 * node: link list for STA node.
 * changed: After auto mode selection, if mu_list has changed, set to 1.
 * gid: group id of this mu_list.
 * cnt: STA counts of this mu_list.
 */
typedef struct MU_List
{
    struct MUCapStaNode_t   *node;
    u8                      changed;    /* list members changed */
    u8                      gid;        /* group id */
    u32                     cnt;        /* total MU capable station in list */
    u32                     nss_cnt;    /* total station nss in list */
} MU_List;



/**
 * Mode_List - Main struct for OFDMA/MIMO list.
 *             Each OFDMA/MIMO type has a Mode_List
 * mu_list: contains STAs that have been selected mode.
 * range_list: contains STAs that have been selected by rssi from mu_list.
 * g_list: contains STAs that have been calculated by advanced grouping.
 */
typedef struct Mode_List
{
    MU_List                 mu_list;
    struct Range_List       range_list[RSSI_END];
    struct Group_List       g_list[MU_MAX_GROUPS];
} Mode_List;

typedef struct OMI_data
{
    u16                     stnid;
    u8                      sub_type;
} OMI_data;


enum {
    N_TYPE_IE = 1,
    N_TYPE_GROUP_LIST,
    N_TYPE_END
};

enum {
    N_OMI_SUBTYPE_DISABLE_UL_OFDMA = 0,
    N_OMI_SUBTYPE_END
};

typedef struct MU_Notify_t
{
    u8                      type;
    u8                      eid;    /* WL_MLME_ASSOC */
    union
    {
        IEEEtypes_MacAddr_t     sta_mac;
        Group_List              cal_g_list[MU_MAX_GROUPS];
        OMI_data                OMI;
    } Body;
} MU_Notify_t;

/* For Driver */
typedef struct ul_stnid_ru_s
{
    u16             StnID;
    u16             RU_alloc;       /* [0]: ru_allocation region
                                     *      1: 2 STA(996 + 996)
                                     *      0: others
                                     * [1:7]: ru_allocation 0 ~67*/
    u8              SU_Rate_NSS;    /* 0:1SS ...7:8SS, Others: reserved*/
    u8              SU_Rate_MCS;    /* 0~11, Others: reserved */
    s16             SU_rssi;        /* Signed Value */
    u32             CSI;            /* TBD */
    u32             DataLen;        /* 0 */
} ul_stnid_ru_t;

typedef struct mu_mode_config_t
{
    u8              enabled;
    u8              interval;
    u64             Th_11ac_mimo_tx_bytes;
    u64             Th_11ax_mimo_tx_bytes;
    u64             Th_11ax_ofdma_tx_bytes;
    u64             Th_11ac_mimo_rx_bytes;
    u64             Th_11ax_mimo_rx_bytes;
    u64             Th_11ax_ofdma_rx_bytes;
    u32             Th_ofdma_su;        /* default:5000ms */
    u32             Th_ofdma_airtime;   /* default:0 */
    s32             TH_rssi_near;       /* default:-45 */
    s32             TH_rssi_far;        /* default:-70 */
    u8              max_nss;
    u8              mu_ignore_client_capabilities;
    /* DL config */
    u8              DL_enabled;
    u8              DL_MIMO_enabled;
    u8              DL_OFDMA_enabled;
    u8              DL_MIMO_max_groups;
    u8              DL_OFDMA_No_Aggregation;
    /* UL config */
    u8              UL_enabled;
    u8              UL_MIMO_enabled;
    u8              UL_OFDMA_enabled;
    u8              UL_sched_bsrp;
    u32             UL_MIMO_Data_Len;
    u32             UL_OFDMA_Data_Len;
    u8              UL_MIMO_max_groups;
    u8              UL_OFDMA_max_groups;
    /* signed char, wait someone to fix s8 */
    signed char     UL_fix_mcs[MU_MAX_USERS];         /* UL fix mcs rate by order in group */
} mu_mode_config_t;

typedef struct mums_bss_data_t
{
    pthread_cond_t  mu_mode_cond;
    pthread_mutex_t mu_mode_mutex;
    mqd_t           message_queue;      /* message queue */
    u16             sta_count;          /* sta count for per vap */
    stadb_t         *stadb;
    u8              mu_time_count;
    Group_List      cal_g_list[MU_MAX_GROUPS];  /* mimo calculate result list */
    u8              mq_name[128];       /* message queue name */
    /* DL mode selection */
//    u8              DL_GroupSeq;
//    u64             DL_GroupSet;
    u8              In_Grouping;        /* calculation in advance grouping */
    u16             In_cal_mode;        /* [0:7] is mu_mode_e, [8:16] is MU_MODE_RANGE_e */
    Mode_List       DL_OFDMA_list[MU_MAX_GROUPS];
    Mode_List       DL_11AC_MIMO_list[MU_MAX_GROUPS];
    Mode_List       DL_11AX_MIMO_list[MU_MAX_GROUPS];
    Mode_List       DL_VHT_MIMO_list;
    Mode_List       DL_HE_MIMO_list;
    Mode_List       DL_VHT_MIMO2_list;         //WAR
    Mode_List       DL_HE_MIMO2_list;         //WAR
    /* UL mode selection */
//    u8              UL_GroupSeq;
    u32             ul_tf_rate;
//    u64             UL_GroupSet;
    Mode_List       UL_MIMO_list[MU_MAX_GROUPS];
    Mode_List       UL_OFDMA_list[MU_MAX_GROUPS];
    ul_stnid_ru_t   UL_StaList[MU_MAX_USERS];
} mums_bss_data_t;

typedef struct mums_radio_data_t
{
    u8              DL_GroupSeq;
    u64             DL_GroupSet;
    u8              UL_GroupSeq;
    u64             UL_GroupSet;
    u32             mu_version;
    u8              device_id;
    u8              max_nss;
} mums_radio_data_t;

#endif /* mu_mode_H_ */
