/** @file stadb.c
  *
  * @brief This file contains WLAN application specific defines etc.
  *
  * Copyright 2019-2020 NXP
  *
  * NXP CONFIDENTIAL
  * The source code contained or described herein and all documents related to
  * the source code ("Materials") are owned by NXP, its
  * suppliers and/or its licensors. Title to the Materials remains with NXP,
  * its suppliers and/or its licensors. The Materials contain
  * trade secrets and proprietary and confidential information of NXP, its
  * suppliers and/or its licensors. The Materials are protected by worldwide copyright
  * and trade secret laws and treaty provisions. No part of the Materials may be
  * used, copied, reproduced, modified, published, uploaded, posted,
  * transmitted, distributed, or disclosed in any way without NXP's prior
  * express written permission.
  *
  * No license under any patent, copyright, trade secret or other intellectual
  * property right is granted to or conferred upon you by disclosure or delivery
  * of the Materials, either expressly, by implication, inducement, estoppel or
  * otherwise. Any license under such intellectual property rights must be
  * express and approved by NXP in writing.
  *
  */

#include <malloc.h>
#include <stdio.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <fcntl.h>
#include <errno.h>
#include <pthread.h>
#include "wlmgr_debug.h"
#include "stadb_func.h"

static struct stadb_t *stadb_list = NULL;

stadb_t *stadb_new(mums_bss_data_t *data_p, char *sta_addr)
{
    stadb_t *node, *cur;
    u8      i;

    if (data_p == NULL) {
        wlmgr_printf(WLMGR_ERROR,"%s data_p == NULL", __func__);
        return NULL;
    }
    if (sta_addr == NULL) {
        wlmgr_printf(WLMGR_ERROR,"%s sta_addr == NULL", __func__);
        return NULL;
    }

    if ((node = stadb_find(data_p, sta_addr)) != NULL) {
        return node;
    }
    node = (stadb_t *)wlmgr_malloc(sizeof(stadb_t));
    if (node == NULL) {
        wlmgr_printf(WLMGR_ERROR,"%s allocate node failed!!", __func__);
        return NULL;
    }
    memset(node, 0, sizeof(stadb_t));
    memcpy(node->addr, sta_addr, 6);
    if (data_p->stadb == NULL) {
        data_p->stadb = node;
    } else {
        cur = data_p->stadb;
        while (cur->next) {
            cur = cur->next;
        }
        cur->next = node;
    }
    node->status = STAT_INIT;
    data_p->sta_count ++;
    return node;
}

stadb_t *stadb_find(mums_bss_data_t *data_p, char *sta_addr)
{
    stadb_t *node;

    if (data_p == NULL) {
        return NULL;
    }
    if (sta_addr == NULL) {
        return NULL;
    }
    node = data_p->stadb;
    while(node) {
        if(!memcmp(node->addr, sta_addr, 6)) {
            return node;
        }
        node = node->next;
    }
    return NULL;
}

stadb_t *stadb_find_by_id(mums_bss_data_t *data_p, u16 StnId)
{
    stadb_t *node;

    if (data_p == NULL) {
        return NULL;
    }
    node = data_p->stadb;
    while(node) {
        if(node->StnId == StnId) {
            return node;
        }
        node = node->next;
    }
    return NULL;
}

int stadb_del(mums_bss_data_t *data_p, char *sta_addr)
{
    stadb_t *node, *prev;

    if (data_p == NULL) {
        return STADB_ENTRY_FAILED;
    }
    if (sta_addr == NULL) {
        return STADB_ADDR_FAILED;
    }
    prev = node = data_p->stadb;
    if(!memcmp(node->addr, sta_addr, 6)) {
        data_p->stadb = prev->next;
        wlmgr_free(prev);
        data_p->sta_count -- ;
        return STADB_SUCCESS;
    } else {
        while(node) {
            if(!memcmp(node->addr, sta_addr, 6)) {
                prev->next = node->next;
                wlmgr_free(node);
                data_p->sta_count -- ;
                return STADB_SUCCESS;
            }
            prev = node;
            node = prev->next;
        }
    }

    return STADB_NOT_FOUND;
}

int stadb_all_del(mums_bss_data_t *data_p)
{
    stadb_t *node, *next;

    if (data_p == NULL) {
        return STADB_ENTRY_FAILED;
    }
    next = node = data_p->stadb;
    while(node) {
        next = node->next;
        wlmgr_free(node);
        node = next;
    }
    data_p->stadb = NULL;
    data_p->sta_count = 0;
    return STADB_SUCCESS;
}

MUCapStaNode_t *mu_node_new(mu_mode_e mode, MU_List *mu_list, stadb_t *sta_node)
{
    MUCapStaNode_t  *node, *cur;
    u8              switch_flag = 0;

    if (mu_list == NULL || sta_node == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s mu_list or sta_node is NULL", __func__);
        return NULL;
    }
    node = (MUCapStaNode_t *)wlmgr_malloc(sizeof(MUCapStaNode_t));
    if (node == NULL) {
        wlmgr_printf(WLMGR_ERROR, "%s node == NULL", __func__);
        return NULL;
    }
    memset(node, 0, sizeof(MUCapStaNode_t));

    if (mu_list->node == NULL) {
        mu_list->node = node;
    } else {
        cur = mu_list->node;
        while (cur != NULL) {
            switch_flag = 0;
            if (mode < MS_DL_MODE_END) {
                if (sta_node->nss > cur->StaInfo_p->nss) {
                    /* High NSS STA can in front of list */
                    switch_flag = 1;
                } else if (sta_node->nss == cur->StaInfo_p->nss) {
                    if ((sta_node->tx_air_time > 0) && (cur->StaInfo_p->tx_air_time < sta_node->tx_air_time)) {
                        /* The same NSS STA can compare tx_air_time */
                        switch_flag = 1;
                    }
                }
            }

            if (switch_flag) {
                if (cur->prv != NULL) {
                    node->nxt = cur;
                    node->prv = cur->prv;
                    node->prv->nxt = node;
                    cur->prv = node;
                } else {
                    mu_list->node = node;
                    node->nxt = cur;
                    cur->prv = node;
                }
                break;
            }
            if (cur->nxt == NULL) {
                cur->nxt = node;
                node->prv = cur;
                break;
            }
            cur = cur->nxt;
        }
    }
    mu_list->cnt ++;
    node->StaInfo_p = sta_node;
    return node;
}

MUCapStaNode_t *mu_node_find(MU_List *mu_list, stadb_t *sta_node)
{
    MUCapStaNode_t  *node;

    if (mu_list == NULL) {
        return NULL;
    }
    if (sta_node == NULL) {
        return NULL;
    }

    node = mu_list->node;
    while(node) {
        if(node->StaInfo_p == sta_node) {
            return node;
        }
        node = node->nxt;
    }
    return NULL;
}

s32 mu_node_del(MU_List *mu_list, MUCapStaNode_t *node)
{
    if (mu_list == NULL || node == NULL) {
        return WL_STATUS_FAILURE;
    }
    if (node->prv == NULL) {
        mu_list->node = node->nxt;
    } else {
        node->prv->nxt = node->nxt;
    }
    if (node->nxt != NULL) {
        node->nxt->prv = node->prv;
    }

    wlmgr_free(node);
    if (mu_list->cnt > 0)
        mu_list->cnt --;
    return WL_STATUS_SUCCESS;
}

void mu_node_all_del(MU_List *mu_list)
{
    MUCapStaNode_t  *node, *next;

    if (mu_list == NULL) {
        return;
    }
    next = node = mu_list->node;
    while(node) {
        next = node->nxt;
        wlmgr_free(node);
        node = next;
    }
    mu_list->node = NULL;
    mu_list->cnt = 0;
    return;
}

void mu_node_move_last_to_other_list(mu_mode_e mode, Mode_List *mode_list_from, Mode_List *mode_list_to)
{
    MUCapStaNode_t  *node, *next;
    u8              switch_flag = 0;

    node = mode_list_from->mu_list.node;
    if (node == NULL) {
        return;
    }
    while(node->nxt){
        node = node->nxt;
    }
    if (node->prv == NULL) {
        mode_list_from->mu_list.node = NULL;
    } else {
        node->prv->nxt = NULL;
    }
    if (mode_list_from->mu_list.cnt > 0) {
        mode_list_from->mu_list.cnt --;
    }
    node->prv = NULL;
    node->nxt = NULL;

    mu_node_new(mode, &(mode_list_to->mu_list), node->StaInfo_p);
    wlmgr_free(node);
}

u16 mu_node_total_nss(MU_List *mu_list, u8 *all_one_nss)
{
    MUCapStaNode_t  *node;
    u16             nss_cnt = 0;

    if (mu_list == NULL) {
        return 0;
    }
    if (all_one_nss == NULL) {
        return 0;
    }

    node = mu_list->node;
    *all_one_nss = 1;
    while(node) {
        if(node->StaInfo_p) {
            nss_cnt += node->StaInfo_p->nss;
            if (node->StaInfo_p->nss > 1) {
                *all_one_nss = 0;
            }
        }
        node = node->nxt;
    }
    return nss_cnt;
}
