/** @file wlmgr.c
  *
  * @brief This file contains WLAN application specific defines etc.
  *
  * Copyright 2019-2020 NXP
  *
  * NXP CONFIDENTIAL
  * The source code contained or described herein and all documents related to
  * the source code ("Materials") are owned by NXP, its
  * suppliers and/or its licensors. Title to the Materials remains with NXP,
  * its suppliers and/or its licensors. The Materials contain
  * trade secrets and proprietary and confidential information of NXP, its
  * suppliers and/or its licensors. The Materials are protected by worldwide copyright
  * and trade secret laws and treaty provisions. No part of the Materials may be
  * used, copied, reproduced, modified, published, uploaded, posted,
  * transmitted, distributed, or disclosed in any way without NXP's prior
  * express written permission.
  *
  * No license under any patent, copyright, trade secret or other intellectual
  * property right is granted to or conferred upon you by disclosure or delivery
  * of the Materials, either expressly, by implication, inducement, estoppel or
  * otherwise. Any license under such intellectual property rights must be
  * express and approved by NXP in writing.
  *
  */

#include <pthread.h>
#include <sys/time.h>
#include <signal.h>
#include "wlmgr.h"
#include "drv_ops.h"
#include "ctrl_iface.h"
#include "wlmgr_debug.h"
#include "mu_mode_func.h"

#define VERSION          "v2.12"
#define IFACE_DEF        "wdev0ap0"
#define CIFACE_DIR_DEF   "/var/run/wlmgr"

#define WLMGR_MAX_CARDS_SUPPORT 2 //sync. with driver MAX_CARDS_SUPPORT
#define WLMGR_MAX_BSS_NUM 32 //sync. with driver NUMOFAPS
#define WLMGR_MAX_STANUM  320 //sync. with driver MAX_STNS

#define WLMGR_MAIN_DEBUG

typedef enum
{
    WLMGR_ACTION_MUMIMO,
    WLMGR_ACTION_MUMODE,
    WLMGR_ACTION_DOALL,
    WLMGR_ACTION_NONE
} WLMGR_ACTION_e;

#define WLMGR_ACTION_e u8

typedef struct masnd_action_t {
    u8 *actStr;
    u8 actNum;
} PACK_END masnd_action_t;

static struct masnd_action_t actions[] = {
    {"mumimo", WLMGR_ACTION_MUMIMO},
    {"mumode", WLMGR_ACTION_MUMODE},
    {"all", WLMGR_ACTION_DOALL},
};

static s8 *usage[] = {
    "Usage: ",
    "   wlmgr [-v][-h][-d[d[d]]][-c <wlmgr.conf>]/[-i <wlanif> -a <action>][-f filename]",
    "   where",
    "   wlanif: wireless network interface",
    "   wlmgr.conf: wlmgr configuration filename"
    "   action: ",
    "     : mumimo",
    "     : mumode",
    "     : all",
};

extern const struct driver_ops *const drivers[];

wlmgr_interfaces_t wlmgr_interfaces;

static int wlmgr_run = 1;

static void displayUsage(void)
{
    u32 i;
    for (i = 0; i < NELEMENTS(usage); i++)
        printf("%s\n", usage[i]);
}

static int findAction(s8 *actStr)
{
    s32 i;

    for (i = 0; i < 2; i++) {
        if (!strcasecmp(actions[i].actStr, actStr)) {
            return actions[i].actNum;
        }
    }
    return actions[i].actNum;
}

static char *config_get_line(FILE * fp, s8 * str, s32 size, int *lineno)
{
    char *start, *end;
    int out, next_line;

    if (!fp || !str)
        return NULL;

    do {
read_line:
        if (!fgets(str, size, fp))
            break;
            start = str;
            start[size - 1] = '\0';
            end = start + strlen(str);
            (*lineno)++;

            out = 1;
            while (out && (start < end)) {
            next_line = 0;
            /* Remove empty lines and lines starting with # */
            switch (start[0]) {
            case ' ':          /* White space */
            case '\t':         /* Tab */
                start++;
                break;
            case '#':
            case '\n':
            case '\0':
                next_line = 1;
                break;
            case '\r':
                if (start[1] == '\n')
                    next_line = 1;
                else
                    start++;
                break;
            default:
                out = 0;
                break;
            }
            if (next_line)
                goto read_line;
        }

        /* Remove # comments unless they are within a double quoted string.
           Remove trailing white space. */
        if ((end = strstr(start, "\""))) {
            if (!(end = strstr(end + 1, "\"")))
                end = start;
        } else
            end = start;

        if ((end = strstr(end + 1, "#")))
            *end-- = '\0';
        else
            end = start + strlen(start) - 1;

        out = 1;
        while (out && (start < end)) {
            switch (*end) {
            case ' ':          /* White space */
            case '\t':         /* Tab */
            case '\n':
            case '\r':
                *end = '\0';
                end--;
                break;
            default:
                out = 0;
                break;
            }
            }
        if (start == '\0')
            continue;

        return start;
    } while (1);
    return NULL;
}

static int parse_line(char *line, char *args[])
{
    int arg_num = 0;
    int is_start = 0;
    int is_quote = 0;
    int length = 0;
    int i = 0;

    arg_num = 0;
    length = strlen(line);
    /* Process line */

    /* Find number of arguments */
    is_start = 0;
    is_quote = 0;
    for (i = 0; i < length; i++) {
        /* Ignore leading spaces */
        if (is_start == 0) {
            if (line[i] == ' ') {
                continue;
            } else if (line[i] == '\t') {
                continue;
            } else if (line[i] == '\n') {
                break;
            } else {
                is_start = 1;
                args[arg_num] = &line[i];
                arg_num++;
            }
        }
        if (is_start == 1) {
            /* Ignore comments */
            if (line[i] == '#') {
                if (is_quote == 0) {
                    line[i] = '\0';
                    arg_num--;
                }
                break;
            }
            /* Separate by '=' */
            if (line[i] == '=') {
                line[i] = '\0';
                is_start = 0;
                continue;
            }
            /* Separate by ',' */
            if (line[i] == ',') {
                line[i] = '\0';
                is_start = 0;
                continue;
            }
            /* Change ',' to ' ', but not inside quotes */
            if ((line[i] == ',') && (is_quote == 0)) {
                line[i] = ' ';
                continue;
            }
        }
        /* Remove newlines */
        if (line[i] == '\n') {
            line[i] = '\0';
        }
        /* Check for quotes */
        if (line[i] == '"') {
            is_quote = (is_quote == 1) ? 0 : 1;
            continue;
        }
        if (((line[i] == ' ') || (line[i] == '\t')) && (is_quote == 0)) {
            line[i] = '\0';
            is_start = 0;
            continue;
        }
    }
    return arg_num;
}

static void wlmgr_config_defaults(wlmgr_bss_config_t *conf)
{
    strncpy(conf->ifname, IFACE_DEF, strlen(IFACE_DEF));
    mumimo_config_defaults(&conf->mumimo);
    mu_mode_config_defaults(&conf->mu_mode);
}

static s32 wlmgr_config_read(wlmgr_bss_config_t *conf, const char *fname)
{
    FILE *f = NULL;
    s8 *line = NULL;
    s32 li = 0, arg_num = 0;
    s8 *args[100], *pos = NULL;
    s32 ret = WL_STATUS_FAILURE;

    f = fopen(fname, "r");
    if (f == NULL) {
        wlmgr_printf(WLMGR_ERROR, "Could not open configuration file '%s' "
                "for reading.", fname);
        goto exit;
    }

    line = (char *)wlmgr_malloc(MAX_CONFIG_LINE);
    if (line == NULL) {
        wlmgr_printf(WLMGR_ERROR, "Cannot allocate memory for line...");
        goto exit;
    }
    memset(line, 0, MAX_CONFIG_LINE);

    while ((pos = config_get_line(f, line, MAX_CONFIG_LINE, &li))) {
        arg_num = parse_line(line, args);
        /* fill in configurations */
        if (strcmp(args[0], "interface") == 0) {
            if (strlen(args[1]) < sizeof(conf->ifname)) {
                memset(conf->ifname, 0, sizeof(conf->ifname));
                strncpy(conf->ifname, args[1], sizeof(conf->ifname));
            }
        }
        /* MU-MOMO configuration */
        else if (strcmp(args[0], "mumimo_enabled") == 0) {
            conf->mumimo.enabled = atoi(args[1]);
	} else if (strcmp(args[0], "mumimo_11axac_enabled") == 0) {
	    u8 axac_enabled = atoi(args[1]);
            if (axac_enabled < MUMIMO_ENABLE_NUM)
                conf->mumimo.axac_enabled = axac_enabled;
            else
                wlmgr_printf(WLMGR_ERROR, "Incorrect MU-MIMO 11AXAC enable mode - %d", axac_enabled);
	} else if (strcmp(args[0], "mumimo_interval") == 0) {
            conf->mumimo.interval = atoi(args[1]);
	} else if (strcmp(args[0], "mumimo_tonefactor") == 0) {
            conf->mumimo.tone_factor = atoi(args[1]);
	}
        /* Mode Selection configuration */
        else if (strcmp(args[0], "mumode_enabled") == 0) {
            conf->mu_mode.enabled = atoi(args[1]);
        } else if (strcmp(args[0], "mumode_DL_enabled") == 0) {
            conf->mu_mode.DL_enabled = atoi(args[1]);
        } else if (strcmp(args[0], "mumode_DL_MIMO_enabled") == 0) {
            conf->mu_mode.DL_MIMO_enabled = atoi(args[1]);
        } else if (strcmp(args[0], "mumode_DL_OFDMA_enabled") == 0) {
            conf->mu_mode.DL_OFDMA_enabled = atoi(args[1]);
        } else if (strcmp(args[0], "mumode_UL_enabled") == 0) {
            conf->mu_mode.UL_enabled = atoi(args[1]);
        } else if (strcmp(args[0], "mumode_UL_MIMO_enabled") == 0) {
            conf->mu_mode.UL_MIMO_enabled = atoi(args[1]);
        } else if (strcmp(args[0], "mumode_UL_OFDMA_enabled") == 0) {
            conf->mu_mode.UL_OFDMA_enabled = atoi(args[1]);
        } else if (strcmp(args[0], "mumode_interval") == 0) {
            conf->mu_mode.interval = atoi(args[1]);
        } else if (strcmp(args[0], "Th_11ac_mimo_tx_bytes") == 0) {
            conf->mu_mode.Th_11ac_mimo_tx_bytes = atoi(args[1]);
        } else if (strcmp(args[0], "Th_11ax_mimo_tx_bytes") == 0) {
            conf->mu_mode.Th_11ax_mimo_tx_bytes = atoi(args[1]);
        } else if (strcmp(args[0], "Th_11ax_ofdma_tx_bytes") == 0) {
            conf->mu_mode.Th_11ax_ofdma_tx_bytes = atoi(args[1]);
        } else if (strcmp(args[0], "Th_11ac_mimo_rx_bytes") == 0) {
            conf->mu_mode.Th_11ac_mimo_rx_bytes = atoi(args[1]);
        } else if (strcmp(args[0], "Th_11ax_mimo_rx_bytes") == 0) {
            conf->mu_mode.Th_11ax_mimo_rx_bytes = atoi(args[1]);
        } else if (strcmp(args[0], "Th_11ax_ofdma_rx_bytes") == 0) {
            conf->mu_mode.Th_11ax_ofdma_rx_bytes = atoi(args[1]);
        } else if (strcmp(args[0], "Th_ofdma_su") == 0) {
            conf->mu_mode.Th_ofdma_su = atoi(args[1]);
        } else if (strcmp(args[0], "TH_rssi_near") == 0) {
            conf->mu_mode.TH_rssi_near = atoi(args[1]);
        } else if (strcmp(args[0], "TH_rssi_far") == 0) {
            conf->mu_mode.TH_rssi_far = atoi(args[1]);
        }
    }

    ret = WL_STATUS_SUCCESS;

exit:
    if (f != NULL) {
        fclose(f);
        f = NULL;
    }

    if (line != NULL) {
        wlmgr_free(line);
        line = NULL;
    }

    return ret;
}

static wlmgr_bss_config_t *wlmgr_config_init(const char *config_file, char *ifname, u8 action)
{
    wlmgr_bss_config_t *conf;

    conf = wlmgr_malloc(sizeof(wlmgr_bss_config_t));
    if (conf == NULL) {
        wlmgr_printf(WLMGR_ERROR, "Failed to allocate memory for configuration data.");
        return NULL;
    }

    memset(conf, 0, sizeof(wlmgr_bss_config_t));
    wlmgr_config_defaults(conf);

    /* wlmgr -c <wlmgr.conf> */
    if ((config_file != NULL) && (strlen(config_file) > 0)) {
        if (wlmgr_config_read(conf, config_file) != WL_STATUS_SUCCESS)
            goto exit;
    }

    /* wlmgr -a <action> */
    if (action < WLMGR_ACTION_NONE) {
        switch (action)
        {
        case WLMGR_ACTION_MUMIMO:
            conf->mumimo.enabled = MUMIMO_MODE_ENABLEBYCMD;
            conf->mu_mode.enabled = MUMS_DISABLE;
            break;
        case WLMGR_ACTION_MUMODE:
            conf->mumimo.enabled = MUMIMO_MODE_ENABLEBYMODESEL;
            conf->mu_mode.enabled               = MUMS_ENABLE;
            conf->mu_mode.DL_enabled            = MUMS_ENABLE;
            conf->mu_mode.DL_MIMO_enabled       = MUMS_ENABLE;
            conf->mu_mode.DL_OFDMA_enabled      = MUMS_ENABLE;
            conf->mu_mode.UL_enabled            = MUMS_ENABLE;
            conf->mu_mode.UL_MIMO_enabled       = MUMS_ENABLE;
            conf->mu_mode.UL_OFDMA_enabled      = MUMS_ENABLE;
            break;
        default:
            /* Add default configuration in wlmgr_config_defaults               *
             * xxx_config_defaults API should be provided by each functionality */
            break;
        }
    }

    /* wlmgr -i <wlanif> */
    if ((ifname != NULL) && (strlen(ifname) > 0)) {
        memset(conf->ifname, 0, sizeof(conf->ifname));
        strncpy(conf->ifname, ifname, strlen(ifname));
    }
    /* execute wlmgr without any arguments */
    else
        wlmgr_printf(WLMGR_INFO, "No config file and No arguments assigned, use default configuration!");

    return conf;

exit:
    if (conf != NULL)
        wlmgr_free(conf);

    return NULL;
}

static void wlmgr_global_deinit(void)
{
    wlmgr_interfaces_t *interfaces = &wlmgr_interfaces;
    u8 i, j;

    for (i = 0; i < interfaces->iface_num; i++) {
        wlmgr_iface_t *iface = interfaces->iface[i];
        if (iface == NULL)
            continue;

        for (j = 0; j < iface->bss_num; j++) {
            wlmgr_bss_t *bss = iface->bss[j];
            if (bss == NULL)
                continue;

            mumimo_bss_deinit((void *)bss);
            if (bss->config) {
                if (bss->config->mu_mode.enabled)
                    mums_bss_deinit((void *)bss);

                wlmgr_free(bss->config);
                bss->config = NULL;
            }

            drv_ops_deinit(bss, bss->drv_priv);

            if (bss->ctrl_iface >= 0) {
                ctrl_iface_deinit(bss->ctrl_iface);
                bss->ctrl_iface = -1;
            }
            wlmgr_free(bss);
            iface->bss[j] = NULL;
        }

        mumimo_radio_deinit((void *)iface);
        mums_radio_deinit((void *)iface);

        wlmgr_free(iface->bss);
        wlmgr_free(iface);
        interfaces->iface[i] = NULL;
    }
    wlmgr_free(interfaces->iface);
    interfaces->iface = NULL;
}

static void wlmgr_handle_term(int signum)
{
    wlmgr_run = 0;
}

static s32 wlmgr_get_ifname_byBSS(char *bss_name, char *ifname)
{
    FILE *fp_read = NULL;
    char *vap_info = NULL;
    char *ifname_str = NULL;
    char file_path[128];

    vap_info = (char *) wlmgr_malloc(4096);
    if (vap_info == NULL)
        return WL_STATUS_FAILURE;

    memset(file_path, 0, sizeof(file_path));
    sprintf(file_path, "/sys/class/net/%s/vap_info/info", bss_name);

    fp_read = fopen(file_path, "r");
    if (fp_read != NULL) {
        char *pos = NULL;
        size_t size = 0;

        memset(vap_info, 0, 4096);
        size = fread(vap_info, 1, 4096, fp_read);
        fclose(fp_read);
        if (size > 0) {
            pos = strstr(vap_info, "rootdev:         ");
            if (pos == NULL)
                return WL_STATUS_FAILURE;
            ifname_str = strtok(pos + strlen("rootdev:         "), "\n");
            strncpy(ifname, ifname_str, strlen(ifname_str));
        }
    }

    wlmgr_free(vap_info);

    return WL_STATUS_SUCCESS;
}

static s32 wlmgr_get_ifindex(char *ifname, u8 *ifindex)
{
    FILE *fp_read = NULL;
    char *hw_info = NULL;
    char file_path[128];

    hw_info = (char *) wlmgr_malloc(4096);
    if (hw_info == NULL)
        return WL_STATUS_FAILURE;

    memset(file_path, 0, sizeof(file_path));
    sprintf(file_path, "/sys/class/net/%s/hw/hw_info", ifname);

    fp_read = fopen(file_path, "r");
    if (fp_read != NULL) {
        char *pos = NULL;
        size_t size = 0;

        memset(hw_info, 0, 4096);
        size = fread(hw_info, 1, 4096, fp_read);
        fclose(fp_read);
        if (size > 0) {
            pos = strstr(hw_info, "cardindex:");
            if (pos == NULL)
                return WL_STATUS_FAILURE;
            pos += strlen("cardindex:");
            if (pos == NULL)
                return WL_STATUS_FAILURE;
            *ifindex = (u8)atoi(pos);
        }
    }

    wlmgr_free(hw_info);

    return WL_STATUS_SUCCESS;
}

static s32 wlmgr_bss_init(wlmgr_interfaces_t *interfaces,
                          char *config_file,
                          char **bss_names,
                          u8 bss_names_num,
                          u8 action)
{
    u8 idx;
    s32 ret = WL_STATUS_FAILURE;

    for (idx = 0; idx < bss_names_num; idx++) {
        wlmgr_iface_t *iface;
        wlmgr_bss_t *bss;
        char *bss_name;
        char ifname[IFNAMSIZ + 1];
        u8 ifindex;

        bss = wlmgr_malloc(sizeof(wlmgr_bss_t));
        if (bss == NULL)
            return WL_STATUS_FAILURE;

        memset(bss, 0, sizeof(wlmgr_bss_t));

        bss_name = bss_names[idx];

        /* fill-in configurations */
        bss->config = wlmgr_config_init(config_file, bss_name, action);
        if (bss->config == NULL) {
            wlmgr_printf(WLMGR_ERROR, "config initialized Failed!");
            wlmgr_free(bss);
            return ret;
        }

        /* initialize driver */
        bss->driver = drivers[0];

        bss->drv_priv =
            drv_ops_init(bss, bss->config->ifname);
        if ((bss->driver == NULL) || (bss->drv_priv == NULL)) {
            wlmgr_printf(WLMGR_ERROR, "Driver interface \"%s\" initialized Failed!",
                bss->config->ifname);
            wlmgr_free(bss);
            return ret;
        } else
            wlmgr_printf(WLMGR_INFO, "Driver interface \"%s\" initialized Successfully!",
                bss->config->ifname);

        /* initialize control interface for wlmgr_cli */
        bss->ctrl_iface = -1;
        bss->ctrl_iface =
        ctrl_iface_init((void *)bss, CIFACE_DIR_DEF, bss->config->ifname);

        //if (bss->ctrl_iface >= 0)
        //    wlmgr_printf(WLMGR_INFO, "Control interface \"%s/%s\" initialized Successfully!",
        //        CIFACE_DIR_DEF, bss->config->ifname);

        memset(ifname, 0, sizeof(ifname));
        ret = wlmgr_get_ifname_byBSS(bss_name, ifname);
        if (ret != WL_STATUS_SUCCESS)
            return ret;

        ret = wlmgr_get_ifindex(ifname, &ifindex);
        if (ret != WL_STATUS_SUCCESS)
            return ret;

        iface = interfaces->iface[ifindex];
        bss->iface = iface;
        strncpy(iface->ifname, ifname, strlen(ifname));
        iface->bss[iface->bss_num++] = bss;
    }

    return WL_STATUS_SUCCESS;
}

static wlmgr_iface_t *wlmgr_iface_init(wlmgr_interfaces_t *interfaces, u8 idx)
{
    wlmgr_iface_t *iface = NULL;
    char ifname[IFNAMSIZ + 1];

    iface = wlmgr_malloc(sizeof(wlmgr_iface_t));
    if (!iface)
        return NULL;

    memset(iface, 0, sizeof(wlmgr_iface_t));
    iface->ifindex = idx;
    /* TBD: set the default interface name by config */
    //sprintf(ifname, "%s%d", IFACE_PREFIX, idx);
    //strncpy(iface->ifname, ifname, strlen(ifname));
    iface->interfaces = interfaces;
    iface->bss = wlmgr_malloc(interfaces->max_bss_num * sizeof(wlmgr_bss_t *));

    return iface;
}

static s32 wlmgr_interfaces_init(wlmgr_interfaces_t *interfaces)
{
    FILE *fp_read = NULL;
    u32 max_bss_num = 0, idx = 0;

    interfaces->iface_num = WLMGR_MAX_CARDS_SUPPORT; //TBD: read from /sys
    //if (interfaces->iface_num == 0) {
    //	wlmgr_printf(WLMGR_ERROR, "The number of radio interfaces is incorrect!");
    //    return WL_STATUS_FAILURE;
    //}

    fp_read = fopen("/sys/module/ap8x/parameters/bss_num", "r");
    if (fp_read != NULL) {
        fscanf(fp_read, "%d", &max_bss_num);
        fclose(fp_read);
    }

    interfaces->max_bss_num = (max_bss_num > 0) ? (u8)max_bss_num : WLMGR_MAX_BSS_NUM;
    if (interfaces->max_bss_num > WLMGR_MAX_BSS_NUM)
        interfaces->max_bss_num = WLMGR_MAX_BSS_NUM;

    interfaces->iface =
        wlmgr_malloc((interfaces->iface_num + interfaces->max_bss_num) *
                      sizeof(wlmgr_iface_t *));
    if (interfaces->iface == NULL)
        return WL_STATUS_FAILURE;

    for (idx = 0; idx < interfaces->iface_num; idx++) {
        interfaces->iface[idx] = wlmgr_iface_init(interfaces, idx);
        if (!interfaces->iface[idx]) {
            wlmgr_printf(WLMGR_ERROR, "Failed to initialize interface.");
            wlmgr_free(interfaces->iface);
            interfaces->iface = NULL;
            return WL_STATUS_FAILURE;
        }
    }

    return WL_STATUS_SUCCESS;
}

static s32 wlmgr_global_init(char *config_file,
                             char **bss_names,
                             u8 bss_names_num,
                             u8 action)
{
    wlmgr_interfaces_t *interfaces = &wlmgr_interfaces;
    struct sigaction act;
    s32 ret;
    u8 i, j;

    /* register the callback for signal_terminate */
    memset(&act, 0, sizeof(struct sigaction));
    act.sa_handler = wlmgr_handle_term;
    sigaction(SIGTERM, &act, NULL);

    /* initialize radio interfaces */
    ret = wlmgr_interfaces_init(interfaces);
    if (ret != WL_STATUS_SUCCESS)
        return ret;

    /* initialize bss data */
    ret = wlmgr_bss_init(interfaces, config_file, bss_names, bss_names_num, action);
    if (ret != WL_STATUS_SUCCESS)
        return ret;

    /* Initialize global data per radio and BSS interface for each functionality */
    for (i = 0; i < interfaces->iface_num; i++) {
        wlmgr_iface_t *iface = interfaces->iface[i];
        if (iface == NULL)
            continue;

        iface->mumimo_data = mumimo_radio_init((void *)iface);
        iface->mumode_iface_data = mums_radio_init((void *)iface);

        for (j = 0; j < iface->bss_num; j++) {
            wlmgr_bss_t *bss = iface->bss[j];
            if (bss == NULL)
                continue;

            bss->mumimo_data = mumimo_bss_init((void *)bss);
            if (bss->config) {
                if (bss->config->mu_mode.enabled)
                    bss->mumode_bss_data = mums_bss_init((void *)bss);
            }
        }
    }

    return WL_STATUS_SUCCESS;
}

static s32 wlmgr_global_run(void)
{
    wlmgr_interfaces_t *interfaces = &wlmgr_interfaces;
    u8 i, j;

    /* trigger thread to do routine jobs per BSS */
    for (i = 0; i < interfaces->iface_num; i++) {
        wlmgr_iface_t *iface = interfaces->iface[i];
	for (j = 0; j < iface->bss_num; j++) {
            wlmgr_bss_t *bss = iface->bss[j];
            pthread_t t_mumimo;
            pthread_t t_mu_mode;
            /* MU-MIMO grouping */
            if (bss->config->mumimo.enabled == MUMIMO_MODE_ENABLEBYCMD) {
                if (pthread_create(&t_mumimo, NULL, (void *)mumimo_monitor,
                                   (void *)bss) != 0) {
                    wlmgr_printf(WLMGR_ERROR,
                                 "pthread_create for MU MIMO Monitor failed!");
                    return WL_STATUS_FAILURE;
                }
            } else {
                /* MU-MIMO enabled mode is DISABLE or  ENABLEBYMODESEL     *
                 * Create handler to wait message sent from Mode Selection */
                if (pthread_create(&t_mumimo, NULL, (void *)mumimo_handler,
                                   (void *)bss) != 0) {
                    wlmgr_printf(WLMGR_ERROR,
                                 "pthread_create for MU MIMO Handler failed!");
                    return WL_STATUS_FAILURE;
                }
            }

            /* MU-MODE selection */
            if (bss->config->mu_mode.enabled) {
                if (pthread_create(&t_mu_mode, NULL, (void *)mu_mode_task,
                                   (void *)bss) != 0) {
                    wlmgr_printf(WLMGR_ERROR, "pthread_create for MU MODE failed!");
                    return WL_STATUS_FAILURE;
                }
            }
        }
    }

    return WL_STATUS_SUCCESS;
}

static s32 wlmgr_get_ifnames_byCmd(char *argv[], int argc, int optind,
				     char ***bss_names, u8 *bss_names_num)
{
    char *bss_name, **nnames;
    s32 i, j;

    if (!argv)
        return WL_STATUS_FAILURE;

    for (i = optind, j = 0; i < argc; i++) {
        if (strchr(argv[i], '-')) {
            j = argc - i;
            break;
        }
    }

    argc -= j;
    if (optind >= argc)
        return WL_STATUS_FAILURE;

    *bss_names_num = (argc - optind);

    nnames = wlmgr_malloc((*bss_names_num) * sizeof(char *));

    if (!nnames)
            goto exit;

    *bss_names = nnames;

    for (i = 0, j = optind; j < argc; i++, j++) {
        if (!argv[j])
            continue;

        (*bss_names)[i] = strdup(argv[j]);
    }

    return WL_STATUS_SUCCESS;

exit:
    for (i = 0; i < *bss_names_num; i++)
        wlmgr_free((*bss_names)[i]);
    wlmgr_free(*bss_names);
    *bss_names = NULL;
    *bss_names_num = 0;
    return WL_STATUS_FAILURE;
}

int main(int argc, char *argv[])
{
    //char *ifname = NULL;
    char *config_fname = NULL;
    char *action_str = NULL;
    char *debug_file = NULL;
    u8 action = WLMGR_ACTION_DOALL;
    char **bss_names = NULL;
    u8 bss_names_num = 0;
    u8 debug_level = wlmgr_debug_get_level();
    s32 c = -1;

    /* check arguments */
    for (;;) {
        c = getopt(argc, argv, "di:a:c:f:vh");
        if (c < 0)
            break;

        switch (c) {
        case 'd':
            wlmgr_debug_set_flag(1);
            /* Defaul wlmgr_debug_level is INFO  *
             * -d,   DEBUG                       *
             * -dd,  DUMP                        *
             * -ddd,  ALL                        */
            if (debug_level > WLMGR_EXCESSIVE)
               wlmgr_debug_set_level(--debug_level);
            break;
        case 'i':
#if 0
            if (ifname != NULL)
                wlmgr_free(ifname);
            ifname = strdup(optarg);
            if (strlen(ifname) >= (IFNAMSIZ + 1)) {
                wlmgr_printf(WLMGR_ERROR, "Too long length of interface name %s", ifname);
                goto exit;
            }
#endif
            if (wlmgr_get_ifnames_byCmd(argv, argc, (optind - 1),
                                          &bss_names, &bss_names_num)) {
                wlmgr_printf(WLMGR_ERROR, "Interface name is incorrect!");
                goto exit;
            }
            optind += (bss_names_num - 1);
            break;
        case 'a':
            if (action_str != NULL)
                wlmgr_free(action_str);
            action_str = strdup(optarg);
            if (strlen(action_str) > 0) {
                action = findAction(action_str);
            }
            break;
       case 'c':
            if (config_fname != NULL)
                wlmgr_free(config_fname);
            config_fname = strdup(optarg);
            if (strlen(config_fname) > MAX_CONFIG_LINE) {
                wlmgr_printf(WLMGR_ERROR, "Too long length of Configuration file %s", config_fname);
                goto exit;
            }
            wlmgr_printf(WLMGR_INFO, "Configuration file: %s", config_fname);
            break;
        case 'f':
            if (debug_file != NULL)
                wlmgr_free(debug_file);
            debug_file = strdup(optarg);
            if (strlen(debug_file) > WLMGR_DEBUG_FILE_NAME_LEN) {
            wlmgr_printf(WLMGR_ERROR, "Too long length of debug file name %s", debug_file);
            wlmgr_free(debug_file);
            goto exit;
            }
            break;
        case 'v':
            printf("wlmgr version %s\n", VERSION);
            goto exit;
        case 'h':
        default:
            displayUsage();
            goto exit;
        }
    }

    if (debug_file) {
        wlmgr_debug_open_file(debug_file);
    }
    wlmgr_free(debug_file);

    if (wlmgr_global_init(config_fname, bss_names, bss_names_num, action) != WL_STATUS_SUCCESS)
        goto exit;

    if (wlmgr_global_run() != WL_STATUS_SUCCESS)
        goto exit;

    while (wlmgr_run) {
        sleep(1);
    }

exit:
    wlmgr_global_deinit();
    wlmgr_debug_close_file();
    return WL_STATUS_SUCCESS;
}

/***** The handler implementation for wlmgr_cli command ******/
s32 wlmgr_cmd_get_config(void *ctx, char *cmd, char *reply, size_t *reply_len)
{
    wlmgr_bss_t *wlmgr = (wlmgr_bss_t *)ctx;
    wlmgr_bss_config_t *config = NULL;

    *reply_len = 0;

    if ((wlmgr == NULL) || (wlmgr->config == NULL))
        return -2;

    config = wlmgr->config;
    if (config == NULL)
        return -2;

    printf("\nwlmgr - interface: %s\n", config->ifname);

    mumimo_cmd_get((void *)wlmgr, "config", reply, reply_len);
    printf("\n");

    return 1;
}

s32 wlmgr_cmd_get_debug(void *ctx, char *cmd, char *reply, size_t *reply_len)
{
    u8 debug_flag = wlmgr_debug_get_flag();
    u8 debug_level = wlmgr_debug_get_level();
    *reply_len = 0;

    printf("wlmgr - debug %s",
           (debug_flag) ? "enabled " : "disabled.\n");

    if (debug_flag) {
        switch (debug_level) {
        case WLMGR_EXCESSIVE:
            printf("(level-ALL)\n");
            break;
        case WLMGR_DUMP:
            printf("(level-DUMP)\n");
            break;
        case WLMGR_DEBUG:
            printf("(level-DEBUG)\n");
            break;
        case WLMGR_WARNING:
            printf("(level-WARNING)\n");
            break;
        case WLMGR_ERROR:
            printf("(level-ERROR)\n");
            break;
        case WLMGR_INFO:
        default:
            printf("(level-INFO)\n");
            break;
        }
    }

    return 1;
}

s32 wlmgr_cmd_get(void *ctx, char *cmd, char *reply, size_t *reply_len)
{
    wlmgr_bss_t *wlmgr = (wlmgr_bss_t *)ctx;
    int ret = 1;

    *reply_len = 0;

    if (wlmgr == NULL)
        return -2;

    if (strncmp(cmd, "version", 6) == 0) {
        printf("wlmgr version: %s\n", VERSION);
    } else if (strncmp(cmd, "config", 6) == 0)
        ret = wlmgr_cmd_get_config((void *)wlmgr, (cmd + 6), reply, reply_len);
    else if (strncmp(cmd, "debug", 5) == 0)
        ret = wlmgr_cmd_get_debug((void *)wlmgr, (cmd + 5), reply, reply_len);
    else if (strncmp(cmd, "mumimo-", 7) == 0)
        ret = mumimo_cmd_get((void *)wlmgr, (cmd + 7), reply, reply_len);
    else if (strncmp(cmd, "mumode-", 7) == 0)
        ret = mu_mode_cmd_get((void *)wlmgr, (cmd + 7), reply, reply_len);
    else {
        ret = -2;
        printf("unknown command\n");
    }

    printf("\n");
    return ret;
}

s32 wlmgr_cmd_set_config(void *ctx, char *cmd)
{
    wlmgr_bss_t *wlmgr = (wlmgr_bss_t *)ctx;
    wlmgr_bss_config_t *config = NULL;

    if ((wlmgr == NULL) || (wlmgr->config == NULL))
        return -2;

    config = wlmgr->config;
    if (config == NULL)
        return -2;

    //{
    //    printf("unknown command\n");
    //    return -2;
    //}

    return 0;
}

s32 wlmgr_cmd_set_debug(void *ctx, char *cmd)
{
    wlmgr_debug_set_flag(1);

    if (strncmp(cmd, "disable", 7) == 0)
        wlmgr_debug_set_flag(0);
    else if (strncmp(cmd, "ALL", 3) == 0)
        wlmgr_debug_set_level(WLMGR_EXCESSIVE);
    else if (strncmp(cmd, "DUMP", 3) == 0)
        wlmgr_debug_set_level(WLMGR_DUMP);
    else if (strncmp(cmd, "DEBUG", 5) == 0)
        wlmgr_debug_set_level(WLMGR_DEBUG);
    else if (strncmp(cmd, "INFO", 4) == 0)
        wlmgr_debug_set_level(WLMGR_INFO);
    else if (strncmp(cmd, "WARNING", 7) == 0)
        wlmgr_debug_set_level(WLMGR_WARNING);
    else if (strncmp(cmd, "ERROR", 5) == 0)
        wlmgr_debug_set_level(WLMGR_ERROR);
    else
        wlmgr_debug_set_level(WLMGR_INFO);

    return 0;
}

s32 wlmgr_cmd_set(void *ctx, char *cmd)
{
    wlmgr_bss_t *wlmgr = (wlmgr_bss_t *)ctx;
    int ret = 0;

    if (wlmgr == NULL)
        return -2;

    if (strncmp(cmd, "config-", 7) == 0)
        ret = wlmgr_cmd_set_config((void *)wlmgr, (cmd + 7));
    else if (strncmp(cmd, "debug ", 6) == 0)
        ret = wlmgr_cmd_set_debug((void *)wlmgr, (cmd + 6));
    else if (strncmp(cmd, "mumimo-", 7) == 0)
        ret = mumimo_cmd_set((void *)wlmgr, (cmd + 7));
    else if (strncmp(cmd, "mumode-", 7) == 0)
        ret = mu_mode_cmd_set((void *)wlmgr, (cmd + 7));
    else {
        ret = -2;
        printf("unknown command\n");
    }

    return ret;
}
