//MCU and driver header files
#include <hidef.h>
#include "derivative.h"

//auxiliary functions
void PLL_init(void);
void CAN_init(void);
void CAN_send(unsigned int id, unsigned char * ptr, unsigned char cnt);

//main loop sends and example message
void main(void) {
	int id;
	unsigned char data[8];
	
	PLL_init();
	CAN_init();
	
	//send a test message
	id = 0x5A;
	data[0] = 0xBB;
	data[1] = 0xDD;
	CAN_send(id, &data[0], 2);
	
	for(;;) {
		__RESET_WATCHDOG();
	}
}

//initialises the PLL
void PLL_init(void){
	CPMUCLKS_PLLSEL = 1;				//FBUS = FPLL/2.   FBUS = 32MHz, 
	CPMUREFDIV_REFFRQ = 1;				//Reference clock between 2MHZ and 6MHZ.	
	CPMUREFDIV_REFDIV = 0x1;		    //FREF=8/(1+1) = 4MHZ		
	CPMUSYNR_VCOFRQ = 0x1;             	//FVCO is between 48MHZ and 80MHZ	
	CPMUSYNR_SYNDIV = 0x7;				//FVCO = 2xFREFx(SYNDIV+1)   =   FVCO = 2x4x(7+1) = 64MHZ
	CPMUPOSTDIV_POSTDIV = 0x0;			//FPLL = FVCO/(POSTDIV+1).  FPLL = 64MHZ/(0+1)    FPLL = 64MHz	
	CPMUOSC_OSCE = 1;					//External oscillator enable. 8MHZ.        FREF=FOSC/(REFDIV+1)		
	while(!CPMUIFLG_LOCK){}				// Wait for LOCK.      	
	CPMUIFLG = 0xFF;					// clear CMPMU int flags - not needed but good practice    
}

//initialises CAN communications
void CAN_init(void){

	//initialise MSCAN
	CAN0CTL0 = 0x01;
	while (!(CAN0CTL1_INITAK)){}
	CAN0CTL1_CANE = 1;      /* Enable MSCAN module */
	CAN0CTL1_CLKSRC = 0;    /* Clock source is OSCCLK = 4 MHz */    
	CAN0CTL1_LOOPB = 0;     /* Set to 1 for LoopBack Mode, 0 otherwise */
	CAN0CTL1_LISTEN = 0;    /* Not listen only mode */  
	CAN0CTL0_WUPE = 1;		/* Enable WAKEUP */
	
	//Baud rate = CANCLK/(Pre-scaler * time quanta)                                                                           
	CAN0BTR1_TSEG_10 = 0x0A;
	CAN0BTR1_TSEG_20 = 0x03;
	CAN0BTR0_BRP = 0x01;
	CAN0BTR0_SJW = 0x03;
	CAN0BTR1_SAMP = 0;
	
	//accept all messages
	CAN0IDAC_IDAM = 0; 		/* Two 32 bit filters */
	CAN0IDMR0 = 0xFF;		/* Accept all incoming ID's */
	CAN0IDMR1 = 0xFF;
	CAN0IDMR2 = 0xFF;
	CAN0IDMR3 = 0xFF;
	CAN0IDMR4 = 0xFF;
	CAN0IDMR5 = 0xFF;
	CAN0IDMR6 = 0xFF;
	CAN0IDMR7 = 0xFF;
	
	//exit initialisation mode
	CAN0CTL0_INITRQ = 0;
	while (CAN0CTL1_INITAK){};
	while(!(CAN0CTL0_SYNCH)){};
	
	//clear flags
	CAN0RFLG_RXF = 1;                                   
	CAN0RIER_RXFIE = 1;
	
	//enable physical layer
	CP0CR_CPE = 1;
	CP0CR_SPE = 1;
	
}

//sends a CAN message
void CAN_send(unsigned int id, unsigned char * ptr, unsigned char cnt){
	
	unsigned char index, bufnum;
	
	//prepare frame
	CAN0TIER = 0x00;		/* disable TX interrupts */
	CAN0TBSEL = CAN0TFLG;	/* select lowest empty buffer */
	bufnum = CAN0TBSEL;
	//load ID
	CAN0TXIDR0 = (id&0b11111111000)>>3;
	CAN0TXIDR1 = (id&0b00000000111)<<5;
	
	//load data
	for(index=0; index<cnt; index++){
		*(&CAN0TXDSR0 + index) = ptr[index];
	}
	
	//send frame
	CAN0TXDLR = cnt;	/* set data length */
	CAN0TXTBPR = 0;		/* set priority */
	CAN0TFLG = bufnum;	/* start transmission */
}
