#include "derivative.h" /* include peripheral declarations */
#include "CAN.h"



/*Transmission Structures*/
CAN_Tx_Struct Tx_Struct[STRUCTURE_LENGTH];
CAN_Tx_Struct Tx_TempStruct;

/* Pointers */
CAN_Tx_Struct *pstructBuffer;
CAN_Tx_Struct *pstructTempBuffer;

/*Variables*/
UINT8 u8CanSendIndex;
UINT8 u8BufferStatus;

/*Reception Structure*/
CAN_Rx_Struct Rx_Struct;
UINT8         u8RxCompleteFlg;

/*BUFFER VARIABLES*/
/* Pointers */
volatile CAN_Tx_Struct *StartAddress;
volatile CAN_Tx_Struct *EndAddress;
volatile CAN_Tx_Struct *App_CanPointer;
volatile CAN_Tx_Struct *Tx_CanPointer;

/* Variables */
volatile UINT8 gu8BufferMaxSize;
volatile UINT8 gu8BufferOverFlow;


/* Function prototypes */
void CANRx_Isr(void);
void CANTx_Isr(void);
void Buffer_Init(CAN_Tx_Struct* pstructBufferPointer ,UINT8 structMaxSize);
UINT8 Buffer_Request(CAN_Tx_Struct* pstructDataPointer ,UINT16 u8RequestSize);

void Buffer_Init(CAN_Tx_Struct* pstructBufferPointer ,UINT8 structMaxSize)
{
    /* Buffer Initialization */
    EndAddress=pstructBufferPointer+structMaxSize-1;
    StartAddress=pstructBufferPointer;
    gu8BufferMaxSize=structMaxSize;
    App_CanPointer=pstructBufferPointer;
    Tx_CanPointer=pstructBufferPointer;
    gu8BufferOverFlow=0;
}

UINT8 Buffer_Request(CAN_Tx_Struct* pstructDataPointer ,UINT16 u8RequestSize)
{
    UINT8 u8FreeSpace;
    
    /* Check for OverFlow */
    if(gu8BufferOverFlow)
        return(NOT_ENOUGH_SPACE);
    
    
    /* Calculate Free Space */
    if(App_CanPointer < Tx_CanPointer)
        u8FreeSpace=(UINT8)(Tx_CanPointer-App_CanPointer);
    
    else
        u8FreeSpace=gu8BufferMaxSize-(App_CanPointer-Tx_CanPointer);
    
    /* Validate requested size */
    if(u8FreeSpace<u8RequestSize)
        return(NOT_ENOUGH_SPACE);
    
    
    if(u8FreeSpace==u8RequestSize)
        gu8BufferOverFlow=1;
    
    /* Buffer Copy */
    while(u8RequestSize--)
    {
        *App_CanPointer=*pstructDataPointer;
        App_CanPointer++;
        pstructDataPointer++;
        if(App_CanPointer > EndAddress)
        {
            App_CanPointer=StartAddress;
        }
    }
    return(OK);
}


void MSCAN_Disable(void){
    CAN0CTL0_INITRQ = 1;    /* MSCAN in initialization mode */
        while (!(CAN0CTL1_INITAK))
            ;	        /* Wait for initialization mode acknowledge */	
		
	CAN0CTL1_CANE = 0;    /* Disable MSCAN */
	
}


void CANPHY_Init(void)
{
 CP0CR_CPE = 1;		//CANPHY enable
 CP0CR_SPE = 1;		//CANPHY SPLIT enable  
}


void MSCAN_Init(void)
{
    CAN0CTL0 = 0x01;    /* MSCAN in initialization mode */
        while (!(CAN0CTL1_INITAK))
            ;	        /* Wait for initialization mode acknowledge */	
        CAN0CTL1_CANE = 1;      /* Enable MSCAN module */
        CAN0CTL1_CLKSRC = 0;    /* Clock source is OSCCLK, CANCLK = 4MHz */    
        CAN0CTL1_LOOPB = 0;     /* Set to 1 for LoopBack Mode, 0 otherwise */
        CAN0CTL1_LISTEN = 0;    /* Not listen only mode */  

        CAN0CTL0_WUPE = 1;		/* Enable WAKEUP */
        
        
        /* Baud rate = CANCLK/(Prescaler * time quantas) */                                                                           
        CAN0BTR1_TSEG_10 = 0x0A;    /* Time Segment 1 = 11 */
        CAN0BTR1_TSEG_20 = 0x03;    /* Time Segment 2 = 4 */
                                    /* TSEG1 + TSEG2 + SYNCH_SEG = 16 time quantas */

        /* Prescaler = CANCLK/(Baud rate * time quantas) = 8MHz/(125kHz * 16) = 4 */    
        CAN0BTR0_BRP = 0x03;    /* Baud rate prescaler = 4 */  
        CAN0BTR0_SJW = 0x03;    /* Sinchronization jump width = 3 clock cycles */
                                      
        CAN0BTR1_SAMP = 0;      /* One sample per bit */	      

        /* Four 16-bit acceptance filters */ 
        CAN0IDAC_IDAM = 0x01;                                                                        
        
        CAN0IDAR0 = 0x00;                   /* 16 bit Filter 0 */ 
        CAN0IDMR0 = MASK_CODE_ALL_ID_HIGH;  
        CAN0IDAR1 = 0x00;      
        CAN0IDMR1 = MASK_CODE_ALL_ID_LOW;
        
        CAN0IDAR2 = 0x00;                   /* 16 bit Filter 1 */
        CAN0IDMR2 = MASK_CODE_ST_ID_HIGH;   
        CAN0IDAR3 = 0x00;                   
        CAN0IDMR3 = MASK_CODE_ST_ID_LOW;    
        
        CAN0IDAR4 = 0x00;                   /* 16 bit Filter 2 */
        CAN0IDMR4 = MASK_CODE_ST_ID_HIGH;   
        CAN0IDAR5 = 0x00;                   
        CAN0IDMR5 = MASK_CODE_ST_ID_LOW;    
        
        CAN0IDAR6 = 0x00;                   /* 16 bit Filter 3 */
        CAN0IDMR6 = MASK_CODE_ST_ID_HIGH;   
        CAN0IDAR7 = 0x00;                   
        CAN0IDMR7 = MASK_CODE_ST_ID_LOW; 
                             
        CAN0CTL0_INITRQ = 0; ;            /* Exit initialization mode request */
            while (CAN0CTL1_INITAK)
                ;               	/* Wait for normal mode */
                       
            while(!(CAN0CTL0_SYNCH))
                ;                   /* Wait for CAN synchronization */
            /*Tx Buffers Init*/
            pstructBuffer=&Tx_Struct[0];
            pstructTempBuffer=&Tx_TempStruct;
            Buffer_Init(pstructBuffer,STRUCTURE_LENGTH);    /*Tx Buffer Init*/
            
            
            CAN0RFLG_RXF = 1;       /* Clear receiver flags */                                   
            CAN0RIER_RXFIE = 1;     /* Enable Full Receive Buffer interrupt */               
}


void CAN_EnterSleep(void){
    CAN0CTL0_SLPRQ = 1;    /* MSCAN enter sleep mode */
        while (!(CAN0CTL1_SLPAK))
            ;	        /* Wait for sleep mode acknowledge */		
}


void CAN_ListenOnly(void)
{
    CAN0CTL0_INITRQ = 1;    /* MSCAN in initialization mode */
        while (!(CAN0CTL1_INITAK))
            ;	        /* Wait for initialization mode acknowledge */	
		
	CAN0CTL1_LISTEN = 1;    /* Set MSCAN module in listen mode */
	
	CAN0CTL0_INITRQ = 0;           /* Exit initialization mode request */
        while (CAN0CTL1_INITAK)
            ;
}




UINT8 CAN_SendFrame(UINT32 u32ID)
{   
	UINT8 u8Prio = 0x00;
	UINT8 u8Length = sizeof(Tx_TempStruct.u8Tx_Buffer);    
	
	CAN0TIER = 0x00;                                           /* Disable Tx Buffer empty interrupt*/
    
    Tx_TempStruct.u32Id=u32ID;                                 /*Load TxID in TempStruct*/
    Tx_TempStruct.u8Prio=u8Prio;                               /*Load Priority in TempStruct*/
    Tx_TempStruct.u8Length=u8Length;                           /*Load Length in TempStruct*/
    
    u8BufferStatus=Buffer_Request(pstructTempBuffer,0x01);     /*Load TempStruct in TxBuffer*/
    u8CanSendIndex++;                                          /*Increment Index*/
    CAN0TIER = 0x07;                                           /* Enable Tx Buffer empty interrupt*/               
        
        return NO_ERR;   
 }

void CAN_ReceivedFrame(void)
{
    
    /* Index for extracting/storing received data */
    UINT8 u8Index;
    
	    Rx_Struct.u8Length = (CAN0RXDLR & 0x0F);                   /* Extract received frame data length */
	    
	    /* Read and store each of the received data */
	    for (u8Index=0; u8Index<Rx_Struct.u8Length; u8Index++)
  	        Rx_Struct.u8Rx_Buffer[u8Index] = *(&CAN0RXDSR0 + u8Index);
  	        
  	  
  	  Rx_Struct.u32Id= *((UINT32 *) ((UINT32)(&CAN0RXIDR0)));    /* Extract received frame Identifier*/
      Rx_Struct.u8IdHit=(CAN0IDAC & 0x07);                       /* Extract received frame IDHit */
      u8RxCompleteFlg = 1;                                         /* Reception Complete Flag(Needs to be cleaned in app_Scheduler)*/

      Rx_Struct.u32Id = Rx_Struct.u32Id & 0xFFF00000;
      
      CAN0RFLG_RXF = 1;   /* Clear reception flag */   
      
      //CAN_Callback(Rx_Struct.u32Id);
  
}


/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ INTERRUPTS ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */

interrupt VectorNumber_Vcan0rx void CANRx_Isr(void)
{    	
	(void)CAN_ReceivedFrame(); /* Read received frame */
}

interrupt VectorNumber_Vcan0tx void CANTx_Isr(void)
{    
    /* Transmission buffer */
    UINT8 u8Txbuffer = {0};
    
    /* Index to data within the transmission buffer */ 
    UINT8 u8Index;
    
    if((u8CanSendIndex--)>0)       /*Enable Tx Interrupts if there are messages in TxBuffer*/
             CAN0TIER=0x07;
    if(u8CanSendIndex==0)          /*Disable Tx Interrupts if there are no messages in TxBuffer*/
             CAN0TIER=0x00;

    CAN0TBSEL = CAN0TFLG;       /* Select lowest empty buffer */
    u8Txbuffer = CAN0TBSEL;		  /* Backup selected buffer */
    
    
    *((UINT32 *) ((UINT32)(&CAN0TXIDR0)))= Tx_CanPointer->u32Id;   /* Load Id to IDR Registers */
        
        /* Load data to Data Segment Registers */
        for (u8Index=0;u8Index<(Tx_CanPointer->u8Length);u8Index++) {
            *(&CAN0TXDSR0 + u8Index) = Tx_CanPointer->u8Tx_Buffer[u8Index];  
        }

    CAN0TXDLR = Tx_CanPointer->u8Length;   /* Set Data Length Code */
    CAN0TXTBPR = Tx_CanPointer->u8Prio;    /* Set Priority */

    CAN0TFLG = u8Txbuffer;                 /* Start transmission */
    
    gu8BufferOverFlow=0;                   /*Clean OverFlow flag*/
    Tx_CanPointer++;                       /*Increment TxPointer*/
    
    if(Tx_CanPointer > EndAddress)         /*If TxPointer is greater than EndAddres go to StartAddres*/
    {
       Tx_CanPointer=StartAddress;
    }         
}
