/******************************************************************************								
*     Pulse Width Timer (PWT)					                            	*								
*       																		*
*       This example demonstrates how to initialize and use the PWT module 		*
*       on the Kinetis EA series MCUs.           								*
*       																		*
*       This examples use the RTC timer to generate a square signal of 5 Hz in 	*
*       the PTC5 port. Frequency and period of the signal is measure by the 	*
*       PWT input 0  (PWT_IN0) located in PTD5. The result of the PWT 			*
*       measurement is printed to a serial terminal.							*
*       																		*
*       To use this example connect PTC5 pin to PTD5 pin on the TRK-KEA board. 	*                                                         												
*		Open a serial terminal with a baud rate of 9600 to see the PWT			*
*		 measurement result.													*	
******************************************************************************/		
/**********************************************************************************************
* External objects
**********************************************************************************************/

#include "derivative.h" /* include peripheral declarations */
#include "CLK.h"
#include "GPIO.h"
#include "PWT.h"
#include "RTC.h"
#include "UART.h"
#include "printf.h"

/**********************************************************************************************
* Global variables
**********************************************************************************************/

/**********************************************************************************************
* Constants and macros
**********************************************************************************************/


#define LED0_TOGGLE		OUTPUT_TOGGLE(C,PTC0)
#define LED1_TOGGLE		OUTPUT_TOGGLE(C,PTC1)
#define LED2_TOGGLE		OUTPUT_TOGGLE(C,PTC2)
#define LED3_TOGGLE		OUTPUT_TOGGLE(C,PTC3)

#define LED0_OFF		OUTPUT_CLEAR(C,PTC0);
#define LED1_OFF		OUTPUT_CLEAR(C,PTC1);
#define LED2_OFF		OUTPUT_CLEAR(C,PTC2);
#define LED3_OFF		OUTPUT_CLEAR(C,PTC3);

#define LED0_ON			OUTPUT_SET(C,PTC0);
#define LED1_ON			OUTPUT_SET(C,PTC1);
#define LED2_ON			OUTPUT_SET(C,PTC2);
#define LED3_ON			OUTPUT_SET(C,PTC3);

/**********************************************************************************************
* Local types
**********************************************************************************************/


/**********************************************************************************************
* Local function prototypes
*********************************************************************************************/
void Enable_Interrupt(UINT8 vector_number);
void RTC_ISR();
void Pulse_Measurement(void);
/**********************************************************************************************
* Local variables
**********************************************************************************************/
UINT32 timer_counter=0,positive_measure=0,negative_measure=0;
float frequency=0.0,period=0.0;
float decimals=0;
const float pwt_clock=156250.0; // pwt_clock=clock source/prescaler, in this case 20MHz/128

/**********************************************************************************************
* Local functions
**********************************************************************************************/


/**********************************************************************************************
* Global functions
**********************************************************************************************/
void GPIO_Init()
{
	CONFIG_PIN_AS_GPIO(C,PTC0,OUTPUT); /* Configure LED 0 (PTC0) as an output */
	CONFIG_PIN_AS_GPIO(C,PTC1,OUTPUT); /* Configure LED 1 (PTC1) as an output */
	CONFIG_PIN_AS_GPIO(C,PTC2,OUTPUT); /* Configure LED 2 (PTC2) as an output */
	CONFIG_PIN_AS_GPIO(C,PTC3,OUTPUT); /* Configure LED 3 (PTC3) as an output */
	
	CONFIG_PIN_AS_GPIO(D,PTD0,INPUT); /* Configure BTN0 (PTD0) as an input */	
	CONFIG_PIN_AS_GPIO(D,PTD1,INPUT); /* Configure BTN1 (PTD1) as an input */
	
	ENABLE_INPUT(D,PTD0);			 /* Enable input SW1*/	
	ENABLE_INPUT(D,PTD1);			/*  Enable input SW2*/

	LED0_OFF;							/* Turn off LED0 */
	LED1_OFF;							/* Turn off LED1 */
	LED2_OFF;							/* Turn off LED2 */
	LED3_OFF;							/* Turn off LED3 */

}



/***********************************************************************************************
*
* @brief    main() - Program entry function
* @param    none
* @return   none
*
************************************************************************************************/
int main(void)
{	
	Clk_Init();		/* Configure clocks to run at 20 MHz */
	UART_Init();	/* Initialize UART*/
	GPIO_Init();	/* Configure button pins as inputs and LED pins as outputs */
	PWT_Init();		/* Initialize PWT*/
	RTC_Init();		/* Initialize RTC*/
	RTC_SetCallback(RTC_ISR); /* Set the callback function to be called */
	Enable_Interrupt(INT_RTC);	/* Enable RTC interrupt*/
	Enable_Interrupt(INT_PWT);	/* Enable PWT interrupt*/
	
	for(;;) {	   
	  
	   	if(timer_counter%50==0)  /* Print measure every 5 second*/
	   	{
	   		if(counter_overflow==0)
	   		{
	   			(void)printf("\f********** Pulse Width Timer (PWT)**********\n\r");
	   			(void)printf("Received: %d.%d Hz , Period: %d ms \n\r",(int)frequency,(int)decimals,(int)period);
	   			(void)printf("Positive Pulse: %d  , Negative Pulse: %d  \n\r",(int)positive_measure,(int)negative_measure);
	   		}
	   		else
	   		{
				(void)printf("\f********** Pulse Width Timer (PWT)**********\n\r");
		   		(void)printf("PWT counter overflow\n\r");
		   	
	   		}
	   	}
	   	
		Pulse_Measurement();
	}
	
	return 0;
}


void Pulse_Measurement(void)
{
	
	positive_measure=Positive_pulse_measure(); /* Read positive_measurement*/
   	negative_measure=Negative_pulse_measure();/* Read negative_measurement*/
   	frequency=(pwt_clock)/(positive_measure + negative_measure);				/* Calculate signal frequency*/
   	period = 1.0/frequency;					/* Calculate signal period*/
   	period=period*1000;						/* Conversion of seconds to ms*/
   	
   	decimals=frequency- (int)frequency;
   	decimals=decimals*10;


}
/***********************************************************************************************
*
* @brief    RTC_ISR. RTC interrupts every 100 ms.Toggles LED0. Counter increase by one.		
* @param    none
* @return   none
*
************************************************************************************************/  
void RTC_ISR()
{
	timer_counter++;	
	LED0_TOGGLE;	
}

/***********************************************************************************************
*
* @brief    Enable_Interrupt(UINT8 vector_number). Enable interrupts from desired module.
* @param    Module interrupt number from the interrupts vector table
* @return   none
*
************************************************************************************************/  
void Enable_Interrupt(UINT8 vector_number)
{
	
	vector_number= vector_number-16;

	
	/* Set the ICPR and ISER registers accordingly */
	NVIC_ICPR |= 1 << (vector_number%32);
	NVIC_ISER |= 1 << (vector_number%32);

}
