/******************************************************************************								
*    LCD.c                                                                    *								
*                                                                             *								
*       Created on: 05/06/2012                                                *								
*       Author: B34981                                                        *								
*                                                                             *								
*       The LCD Controller is capable of handling up to 40 frontaples and 4   *								
*    backplanes for a total of 160 segments. This driver file contains the    *								
*    routines for the LCD initialization and animation.                       *								
******************************************************************************/								

#include "LCD.h"

/**********************************************************************************************
* External objects
**********************************************************************************************/


/**********************************************************************************************
* Global variables
**********************************************************************************************/


/**********************************************************************************************
* Constants and macros
**********************************************************************************************/


/**********************************************************************************************
* Local types
**********************************************************************************************/


/**********************************************************************************************
* Local function prototypes
*********************************************************************************************/
void SetC1(UINT8 in);
void SetC2(UINT8 in);
void SetC3(UINT8 in);
void SetC4(UINT8 in);
void SetC5(UINT8 in);
void SetC6(UINT8 in);
void SetO1(UINT8 in);
void SetO2(UINT8 in);
void SetO3(UINT8 in);
void SetO4(UINT8 in);
void SetO5(UINT8 in);
void SetO6(UINT8 in);

/**********************************************************************************************
* Local variables
**********************************************************************************************/
UINT8 bar = 0, accumulate = 0, dir = 0;
UINT16 seg = 0, data = 9999;
UINT8 odo_mode = 0, data_mode = 0, clock_mode = 0, logo = 0; 

/**********************************************************************************************
* Local functions
**********************************************************************************************/


/**********************************************************************************************
* Global functions
**********************************************************************************************/

/***********************************************************************************************
*
* @brief    LCD_Init() - Initialize the LCD module to drive a 40 x 4 Segment LCD
* @param    none
* @return   none
*
************************************************************************************************/  
void LCD_Init()
  {
                      //Configure LCD clock (running from RTCCLK 32.768 KHz)
  LCDCR0_LCLK = 1;	  //Divide clk by 4096 = 61Hz  
  LCDCR1_LCDSWAI = 1; //LCD in Wait
  LCDCR0_DUTY = 0;	  //Duty = 1/4
  LCDCR0_BIAS = 1;	  //Bias = 1/3
    
    /* Enable Frontplanes */  
  LCDFPENR0 = 0xFF;	//Segments 0..7
  LCDFPENR1 = 0xFF;	//Segments 8..15
  LCDFPENR2 = 0xFF;	//Segments 16..23
  LCDFPENR3 = 0xFF;	//Segments 24..31
  LCDFPENR4 = 0xFF;	//Segments 32..39
  
  LCDCR0_LCDEN = 1;    // Enable LCD       
  }

/***********************************************************************************************
*
* @brief    LCD_Animate - Animate the LCD
* @param    none
* @return   none
*
************************************************************************************************/  
void LCD_Animate()
{
	setLines();
			
	SetG((seg/3)%14);

	/*DATA_COUNT*/
	SetData(data);
	data--;
	if(data==0){data = 9999;}
	/*DATA_COUNT*/

	/*GAS_BAR*/
	setBar(bar, accumulate);
	if(dir == 0){bar++;} else{bar--;}
	if(bar == 8){dir = 1;}
	if(bar == 0){dir = 0;  if(accumulate){accumulate = 0;}else{accumulate = 1;} }
	/*GAS_BAR*/

	/*ODOMODE*/
	if(seg%3==0)
	  {
	  odo_mode++;
	  if(odo_mode == 0){setOdoMode();}
	  if(odo_mode == 1){setTripAMode();}
	  if(odo_mode == 2){setTripBMode();}
	  if(odo_mode == 3){setTempMode();}
	  if(odo_mode == 4){setRangeMode();}
	  if(odo_mode == 5){setOdoMode(); odo_mode = 0;}
	  }
	/*ODOMODE*/

	/*DATAMODE*/
	if(seg%3==0)
	  {
	  data_mode++;
	  if(data_mode == 0){setSpeedMode(); setGasIndicators(1);}
	  if(data_mode == 1){setFuelEfficiencyMode(); setGasIndicators(0);}
	  if(data_mode == 2){setSpeedMode(); setGasIndicators(1); data_mode = 0;}
	  }
	/*DATAMODE*/

	/*CLOCKMODE*/
	if(seg%4==0)
	  {
	  clock_mode++;
	  if(clock_mode == 0){setAMMode();}
	  if(clock_mode == 1){setPMMode();}
	  if(clock_mode == 2){set24HMode();}
	  if(clock_mode == 3){setAMMode(); clock_mode = 0;}
	  }
	/*CLOCKMODE*/

	/*LOGO*/
	if(seg%4==0)
	  {
	  logo++;
	  if(logo == 0){setEcoMode(1); setCheckService(0); setFreescale(0);}
	  if(logo == 1){setEcoMode(0); setCheckService(1); setFreescale(0);}
	  if(logo == 2){setEcoMode(0); setCheckService(0); setFreescale(1);}
	  if(logo == 3){setEcoMode(1); setCheckService(0); setFreescale(0);}
	  if(logo == 4){setEcoMode(1); setCheckService(1); setFreescale(0);}
	  if(logo == 5){setEcoMode(1); setCheckService(1); setFreescale(1);}
	  if(logo == 6){setEcoMode(0); setCheckService(1); setFreescale(1);}  
	  if(logo == 7){setEcoMode(0); setCheckService(0); setFreescale(1);}
	  if(logo == 8){setEcoMode(1); setCheckService(0); setFreescale(0); logo = 0;}
	  }
	/*LOGO*/	
	
	seg++;
}


/***********************************************************************************************
*
* @brief    SetD1 - Display a number to the first digit of the central numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/  
void SetD1(UINT8 in)
  {
  switch(in)
    {
      case 0: D1_0; break;
      case 1: D1_1; break;
      case 2: D1_2; break;
      case 3: D1_3; break;
      case 4: D1_4; break;
      case 5: D1_5; break;
      case 6: D1_6; break;
      case 7: D1_7; break;
      case 8: D1_8; break;
      case 9: D1_9; break;
    }  
  }
  
/***********************************************************************************************
*
* @brief    SetD2 - Display a number to the second digit of the central numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/
void SetD2(UINT8 in)
  {
  switch(in)
    {
      case 0: D2_0; break;
      case 1: D2_1; break;
      case 2: D2_2; break;
      case 3: D2_3; break;
      case 4: D2_4; break;
      case 5: D2_5; break;
      case 6: D2_6; break;
      case 7: D2_7; break;
      case 8: D2_8; break;
      case 9: D2_9; break;
    }  
  }  
  
/***********************************************************************************************
*
* @brief    SetD3 - Display a number to the third digit of the central numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/
void SetD3(UINT8 in)
  {
  switch(in)
    {
      case 0: D3_0; break;
      case 1: D3_1; break;
      case 2: D3_2; break;
      case 3: D3_3; break;
      case 4: D3_4; break;
      case 5: D3_5; break;
      case 6: D3_6; break;
      case 7: D3_7; break;
      case 8: D3_8; break;
      case 9: D3_9; break;
    }          
  }  
  
/***********************************************************************************************
*
* @brief    SetD4 - Display a number to the fourth digit of the central numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/
void SetD4(UINT8 in)
  {
  switch(in)
    {
      case 0: D4_0; break;
      case 1: D4_1; break;
      case 2: D4_2; break;
      case 3: D4_3; break;
      case 4: D4_4; break;
      case 5: D4_5; break;
      case 6: D4_6; break;
      case 7: D4_7; break;
      case 8: D4_8; break;
      case 9: D4_9; break;
    }  
  }  
  
/***********************************************************************************************
*
* @brief    SetC1 - Display a number to the first digit of the clock numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/
void SetC1(UINT8 in)
  {
  switch(in)
    {
      case 0: C1_0; break;
      case 1: C1_1; break;
      case 2: C1_2; break;
      case 3: C1_3; break;
      case 4: C1_4; break;
      case 5: C1_5; break;
      case 6: C1_6; break;
      case 7: C1_7; break;
      case 8: C1_8; break;
      case 9: C1_9; break;
    }  
  }
    
/***********************************************************************************************
*
* @brief    SetC2 - Display a number to the second digit of the clock numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/
void SetC2(UINT8 in)
  {
  switch(in)
    {
      case 0: C2_0; break;
      case 1: C2_1; break;
      case 2: C2_2; break;
      case 3: C2_3; break;
      case 4: C2_4; break;
      case 5: C2_5; break;
      case 6: C2_6; break;
      case 7: C2_7; break;
      case 8: C2_8; break;
      case 9: C2_9; break;
    }  
  }  
  
/***********************************************************************************************
*
* @brief    SetC3 - Display a number to the third digit of the clock numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/
void SetC3(UINT8 in)
  {
  switch(in)
    {
      case 0: C3_0; break;
      case 1: C3_1; break;
      case 2: C3_2; break;
      case 3: C3_3; break;
      case 4: C3_4; break;
      case 5: C3_5; break;
      case 6: C3_6; break;
      case 7: C3_7; break;
      case 8: C3_8; break;
      case 9: C3_9; break;
    }  
  }  
  
/***********************************************************************************************
*
* @brief    SetC4 - Display a number to the fourth digit of the clock numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/
void SetC4(UINT8 in)
  {
  switch(in)
    {
      case 0: C4_0; break;
      case 1: C4_1; break;
      case 2: C4_2; break;
      case 3: C4_3; break;
      case 4: C4_4; break;
      case 5: C4_5; break;
      case 6: C4_6; break;
      case 7: C4_7; break;
      case 8: C4_8; break;
      case 9: C4_9; break;
    }  
  }  
  
/***********************************************************************************************
*
* @brief    SetC5 - Display a number to the fifth digit of the clock numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/
void SetC5(UINT8 in)
  {
  switch(in)
    {
      case 0: C5_0; break;
      case 1: C5_1; break;
      case 2: C5_2; break;
      case 3: C5_3; break;
      case 4: C5_4; break;
      case 5: C5_5; break;
      case 6: C5_6; break;
      case 7: C5_7; break;
      case 8: C5_8; break;
      case 9: C5_9; break;
    }  
  }  
  
/***********************************************************************************************
*
* @brief    SetC6 - Display a number to the sixth digit of the clock numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/
void SetC6(UINT8 in)
  {
  switch(in)
    {
      case 0: C6_0; break;
      case 1: C6_1; break;
      case 2: C6_2; break;
      case 3: C6_3; break;
      case 4: C6_4; break;
      case 5: C6_5; break;
      case 6: C6_6; break;
      case 7: C6_7; break;
      case 8: C6_8; break;
      case 9: C6_9; break;
    }  
  } 
  
/***********************************************************************************************
*
* @brief    SetClk - Set the hour, minutes and seconds on the clock segments
* @param    UINT8 h - hours, UINT8 m - minutes, UINT8 - s - seconds
* @return   none
*
************************************************************************************************/
void SetClk(UINT8 h, UINT8 m, UINT8 s)
  {
     SetC6(s % 10);
     s /= 10;
     SetC5(s % 10);     

     SetC4(m % 10);
     m /= 10;
     SetC3(m % 10);
     
     SetC2(h % 10);
     h /= 10;
     SetC1(h % 10);     
  }  
   
/***********************************************************************************************
*
* @brief    SetO1 - Display a number to the first digit of the odometer numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/
void SetO1(UINT8 in)
  {
  switch(in)
    {
      case 0: O1_0; break;
      case 1: O1_1; break;
      case 2: O1_2; break;
      case 3: O1_3; break;
      case 4: O1_4; break;
      case 5: O1_5; break;
      case 6: O1_6; break;
      case 7: O1_7; break;
      case 8: O1_8; break;
      case 9: O1_9; break;
    }  
  }  
  
/***********************************************************************************************
*
* @brief    SetO2 - Display a number to the second digit of the odometer numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/
void SetO2(UINT8 in)
  {
  switch(in)
    {
      case 0: O2_0; break;
      case 1: O2_1; break;
      case 2: O2_2; break;
      case 3: O2_3; break;
      case 4: O2_4; break;
      case 5: O2_5; break;
      case 6: O2_6; break;
      case 7: O2_7; break;
      case 8: O2_8; break;
      case 9: O2_9; break;
    }  
  }    
  
/***********************************************************************************************
*
* @brief    SetO3 - Display a number to the third digit of the odometer numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/
void SetO3(UINT8 in)
  {
  switch(in)
    {
      case 0: O3_0; break;
      case 1: O3_1; break;
      case 2: O3_2; break;
      case 3: O3_3; break;
      case 4: O3_4; break;
      case 5: O3_5; break;
      case 6: O3_6; break;
      case 7: O3_7; break;
      case 8: O3_8; break;
      case 9: O3_9; break;
    }  
  }    
  
/***********************************************************************************************
*
* @brief    SetO4 - Display a number to the fourth digit of the odometer numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/
void SetO4(UINT8 in)
  {
  switch(in)
    {
      case 0: O4_0; break;
      case 1: O4_1; break;
      case 2: O4_2; break;
      case 3: O4_3; break;
      case 4: O4_4; break;
      case 5: O4_5; break;
      case 6: O4_6; break;
      case 7: O4_7; break;
      case 8: O4_8; break;
      case 9: O4_9; break;
    }  
  }    
  
/***********************************************************************************************
*
* @brief    SetO5 - Display a number to the fifth digit of the odometer numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/
void SetO5(UINT8 in)
  {
  switch(in)
    {
      case 0: O5_0; break;
      case 1: O5_1; break;
      case 2: O5_2; break;
      case 3: O5_3; break;
      case 4: O5_4; break;
      case 5: O5_5; break;
      case 6: O5_6; break;
      case 7: O5_7; break;
      case 8: O5_8; break;
      case 9: O5_9; break;
    }  
  }    
  
/***********************************************************************************************
*
* @brief    SetO6 - Display a number to the sixth digit of the odometer numeric display
* @param    Decimal number from 0 - 9
* @return   none
*
************************************************************************************************/
void SetO6(UINT8 in)
  {
  switch(in)
    {
      case 0: O6_0; break;
      case 1: O6_1; break;
      case 2: O6_2; break;
      case 3: O6_3; break;
      case 4: O6_4; break;
      case 5: O6_5; break;
      case 6: O6_6; break;
      case 7: O6_7; break;
      case 8: O6_8; break;
      case 9: O6_9; break;
    }  
  } 

/***********************************************************************************************
*
* @brief    SetOdo - Set a UINT16 value to the odometer segments
* @param    UINT16 in - value to display on the odometer
* @return   none
*
************************************************************************************************/  
void SetOdo(UINT16 in)
  {
     SetO6(in % 10);
     in /= 10;
     SetO5(in % 10);
     in /= 10;
     SetO4(in % 10);
     in /= 10;
     SetO3(in % 10);
     in /= 10;
     SetO2(in % 10);
     in /= 10;
     SetO1(in % 10);                          
  }

/***********************************************************************************************
*
* @brief    SetData - Set a UINT16 value to the central data numeric segments
* @param    UINT16 in - value to display on the central data segments
* @return   none
*
************************************************************************************************/  
void SetData(UINT16 in)
  {
     SetD4(in % 10);
     in /= 10;
     SetD3(in % 10);
     in /= 10;
     SetD2(in % 10);
     in /= 10;
     SetD1(in % 10);     
  }
  
/***********************************************************************************************
*
* @brief    SetG - Set a value on the Gear display segments
* @param    UINT16 in - value to display on the gear segments 0-9, 10-P, 11-R, 12-N and 13-D
* @return   none
*
************************************************************************************************/  
void SetG(UINT8 in)
  {
  switch(in)
    {
      case 0:  G_0; break;
      case 1:  G_1; break;
      case 2:  G_2; break;
      case 3:  G_3; break;
      case 4:  G_4; break;
      case 5:  G_5; break;
      case 6:  G_6; break;
      case 7:  G_7; break;
      case 8:  G_8; break;
      case 9:  G_9; break;
      case 10: G_P; break;
      case 11: G_R; break;
      case 12: G_N; break;
      case 13: G_D; break;
    }  
  }      
  
/***********************************************************************************************
*
* @brief    SetBar - Set the value to display on the gas bar segments
* @param    UINT16 in - value to display on the gas bar segments
* 			UINT8 accumulate - 0 only displays the selected value, 1 - displays the selected value and all the lower ones.
* @return   none
*
************************************************************************************************/
void setBar(UINT8 in, UINT8 accumulate)
  {
  if(accumulate)
    {
    switch(in)
      {
        case 0: SEG_B0 = 0; SEG_B1 = 0;  SEG_B2 = 0;  SEG_B3 = 0;  SEG_B4 = 0;  SEG_B5 = 0;  SEG_B6 = 0;  SEG_B7 = 0; SEG_E = 1; SEG_F = 0; break;
        case 1: SEG_B0 = 1; SEG_B1 = 0;  SEG_B2 = 0;  SEG_B3 = 0;  SEG_B4 = 0;  SEG_B5 = 0;  SEG_B6 = 0;  SEG_B7 = 0; SEG_E = 0; SEG_F = 0; break;
        case 2: SEG_B0 = 1; SEG_B1 = 1;  SEG_B2 = 0;  SEG_B3 = 0;  SEG_B4 = 0;  SEG_B5 = 0;  SEG_B6 = 0;  SEG_B7 = 0; SEG_E = 0; SEG_F = 0; break;
        case 3: SEG_B0 = 1; SEG_B1 = 1;  SEG_B2 = 1;  SEG_B3 = 0;  SEG_B4 = 0;  SEG_B5 = 0;  SEG_B6 = 0;  SEG_B7 = 0; SEG_E = 0; SEG_F = 0; break;
        case 4: SEG_B0 = 1; SEG_B1 = 1;  SEG_B2 = 1;  SEG_B3 = 1;  SEG_B4 = 0;  SEG_B5 = 0;  SEG_B6 = 0;  SEG_B7 = 0; SEG_E = 0; SEG_F = 0; break;
        case 5: SEG_B0 = 1; SEG_B1 = 1;  SEG_B2 = 1;  SEG_B3 = 1;  SEG_B4 = 1;  SEG_B5 = 0;  SEG_B6 = 0;  SEG_B7 = 0; SEG_E = 0; SEG_F = 0; break;
        case 6: SEG_B0 = 1; SEG_B1 = 1;  SEG_B2 = 1;  SEG_B3 = 1;  SEG_B4 = 1;  SEG_B5 = 1;  SEG_B6 = 0;  SEG_B7 = 0; SEG_E = 0; SEG_F = 0; break;
        case 7: SEG_B0 = 1; SEG_B1 = 1;  SEG_B2 = 1;  SEG_B3 = 1;  SEG_B4 = 1;  SEG_B5 = 1;  SEG_B6 = 1;  SEG_B7 = 0; SEG_E = 0; SEG_F = 0; break;
        case 8: SEG_B0 = 1; SEG_B1 = 1;  SEG_B2 = 1;  SEG_B3 = 1;  SEG_B4 = 1;  SEG_B5 = 1;  SEG_B6 = 1;  SEG_B7 = 1; SEG_E = 0; SEG_F = 1; break;
      }    
    }
  else
    {
    switch(in)
      {
        case 0: SEG_B0 = 0; SEG_B1 = 0;  SEG_B2 = 0;  SEG_B3 = 0;  SEG_B4 = 0;  SEG_B5 = 0;  SEG_B6 = 0;  SEG_B7 = 0; SEG_E = 1; SEG_F = 0; break;
        case 1: SEG_B0 = 1; SEG_B1 = 0;  SEG_B2 = 0;  SEG_B3 = 0;  SEG_B4 = 0;  SEG_B5 = 0;  SEG_B6 = 0;  SEG_B7 = 0; SEG_E = 0; SEG_F = 0; break;
        case 2: SEG_B0 = 0; SEG_B1 = 1;  SEG_B2 = 0;  SEG_B3 = 0;  SEG_B4 = 0;  SEG_B5 = 0;  SEG_B6 = 0;  SEG_B7 = 0; SEG_E = 0; SEG_F = 0; break;
        case 3: SEG_B0 = 0; SEG_B1 = 0;  SEG_B2 = 1;  SEG_B3 = 0;  SEG_B4 = 0;  SEG_B5 = 0;  SEG_B6 = 0;  SEG_B7 = 0; SEG_E = 0; SEG_F = 0; break;
        case 4: SEG_B0 = 0; SEG_B1 = 0;  SEG_B2 = 0;  SEG_B3 = 1;  SEG_B4 = 0;  SEG_B5 = 0;  SEG_B6 = 0;  SEG_B7 = 0; SEG_E = 0; SEG_F = 0; break;
        case 5: SEG_B0 = 0; SEG_B1 = 0;  SEG_B2 = 0;  SEG_B3 = 0;  SEG_B4 = 1;  SEG_B5 = 0;  SEG_B6 = 0;  SEG_B7 = 0; SEG_E = 0; SEG_F = 0; break;
        case 6: SEG_B0 = 0; SEG_B1 = 0;  SEG_B2 = 0;  SEG_B3 = 0;  SEG_B4 = 0;  SEG_B5 = 1;  SEG_B6 = 0;  SEG_B7 = 0; SEG_E = 0; SEG_F = 0; break;
        case 7: SEG_B0 = 0; SEG_B1 = 0;  SEG_B2 = 0;  SEG_B3 = 0;  SEG_B4 = 0;  SEG_B5 = 0;  SEG_B6 = 1;  SEG_B7 = 0; SEG_E = 0; SEG_F = 0; break;
        case 8: SEG_B0 = 0; SEG_B1 = 0;  SEG_B2 = 0;  SEG_B3 = 0;  SEG_B4 = 0;  SEG_B5 = 0;  SEG_B6 = 0;  SEG_B7 = 1; SEG_E = 0; SEG_F = 1; break;
      }        
    }  
  }

/***********************************************************************************************
*
* @brief    setEcoMode - Set the EcoMode icon
* @param    UINT8 in - 1 turns the icon ON, 0 turns the icon OFF
* @return   none
*
************************************************************************************************/ 
void setEcoMode(UINT8 in){SEG_ECOMODE = in;}

/***********************************************************************************************
*
* @brief    setCheckService - Set the CheckService icon
* @param    UINT8 in - 1 turns the icon ON, 0 turns the icon OFF
* @return   none
*
************************************************************************************************/
void setCheckService(UINT8 in){SEG_CHKENG = in;}

/***********************************************************************************************
*
* @brief    setGasIndicators - Set the GasPump icon and the sidebars 
* @param    UINT8 in - 1 turns the icons ON, 0 turns the icons OFF
* @return   none
*
************************************************************************************************/
void setGasIndicators(UINT8 in){SEG_GP=in; SEG_SBARS=in;}

/***********************************************************************************************
*
* @brief    setFreescale - Set the Freescale icon 
* @param    UINT8 in - 1 turns the icon ON, 0 turns the icon OFF
* @return   none
*
************************************************************************************************/
void setFreescale(UINT8 in){SEG_FSL=in;}

/***********************************************************************************************
*
* @brief    setOdoMode - Sets the display on ODO mode  
* @param    none
* @return   none
*
************************************************************************************************/
void setOdoMode()
  {
  SEG_ODO=1;
  SEG_KM = 1;
  SEG_OP = 0;  
  SEG_TRIP = 0;
  SEG_A = 0;
  SEG_B = 0;
  SEG_RANGE = 0;
  SEG_TEMP = 0;
  SEG_C = 0;
  SEG_TOZ = 0;
  }
  
/***********************************************************************************************
*
* @brief    setTripAMode - Sets the display on TripA mode  
* @param    none
* @return   none
*
************************************************************************************************/
void setTripAMode()
  {
  SEG_ODO=0;
  SEG_KM = 1;
  SEG_OP = 0;  
  SEG_TRIP = 1;  
  SEG_A = 1;
  SEG_B = 0;
  SEG_RANGE = 0;
  SEG_TEMP = 0;
  SEG_C = 0;
  SEG_TOZ = 0;  
  }

/***********************************************************************************************
*
* @brief    setTripBMode - Sets the display on TripB mode  
* @param    none
* @return   none
*
************************************************************************************************/
void setTripBMode()
  {
  SEG_ODO=0;
  SEG_KM = 1;
  SEG_OP = 0;  
  SEG_TRIP = 1;  
  SEG_A = 0;
  SEG_B = 1;
  SEG_RANGE = 0;
  SEG_TEMP = 0;
  SEG_C = 0;
  SEG_TOZ = 0;  
  }
  
/***********************************************************************************************
*
* @brief    setTempMode - Sets the display on Temperature mode  
* @param    none
* @return   none
*
************************************************************************************************/
void setTempMode()
  {
  SEG_ODO=0;
  SEG_KM = 0;
  SEG_OP = 0;  
  SEG_TRIP = 0;  
  SEG_A = 0;
  SEG_B = 0;
  SEG_RANGE = 0;
  SEG_TEMP = 1;
  SEG_C = 1;
  SEG_TOZ = 0;  
  }  
  
/***********************************************************************************************
*
* @brief    setRangeMode - Sets the display on Range mode  
* @param    none
* @return   none
*
************************************************************************************************/
void setRangeMode()
  {
  SEG_ODO=0;
  SEG_KM = 1;
  SEG_OP = 0;  
  SEG_TRIP = 0;  
  SEG_A = 0;
  SEG_B = 0;
  SEG_RANGE = 1;
  SEG_TEMP = 0;
  SEG_C = 0;
  SEG_TOZ = 1;  
  }  
  
/***********************************************************************************************
*
* @brief    setSpeedMode - Sets the display on Speed mode  
* @param    none
* @return   none
*
************************************************************************************************/
void setSpeedMode()
  {
  SEG_DCOL = 0;
  SEG_DP = 0;
  SEG_KMH = 1;
  SEG_KML = 0;
  SEG_ET = 0;
  }
 
/***********************************************************************************************
*
* @brief    setFuelEfficiencyMode - Sets the display on Fuel Efficiency mode  
* @param    none
* @return   none
*
************************************************************************************************/
void setFuelEfficiencyMode()
  {
  SEG_DCOL = 0;
  SEG_DP = 1;
  SEG_KMH = 0;
  SEG_KML = 1;
  SEG_ET = 1;
  }  
  
/***********************************************************************************************
*
* @brief    setAMMode - Sets the display on AM clock display mode  
* @param    none
* @return   none
*
************************************************************************************************/
void setAMMode(){SEG_AM=1; SEG_PM=0; SEG_24H=0; SEG_CCOL = 1;}

/***********************************************************************************************
*
* @brief    setPMMode - Sets the display on PM clock display mode  
* @param    none
* @return   none
*
************************************************************************************************/
void setPMMode(){SEG_AM=0; SEG_PM=1; SEG_24H=0; SEG_CCOL = 1;}

/***********************************************************************************************
*
* @brief    set24HMode - Sets the display on 24H clock display mode  
* @param    none
* @return   none
*
************************************************************************************************/
void set24HMode(){SEG_AM=0; SEG_PM=0; SEG_24H=1; SEG_CCOL = 1;}

/***********************************************************************************************
*
* @brief    setLines - Sets the display lines to divide sections  
* @param    none
* @return   none
*
************************************************************************************************/
void setLines()
{
	  SEG_HLINE = 1;   /*Set Lines*/
	  SEG_VLINE = 1;   /*Set Lines*/	
}