/*
 * Copyright 2016 - 2018, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/* lrp_dll.cpp : Defines the exported functions for the DLL application. */


#ifdef _WIN32
/*#include <stdio.h>
#include <tchar.h>
#include <string.h>
#include <stdint.h>
#include "phCryptoSym_Lrp_Aes.h" */
#include "phCryptoSym_Lrp.h"
#include <ph_Status.h>
#include <phCryptoSym.h>
#else
#include <phhalSFR.h>
#include <ph_TypeDefs.h>
#include <phhalLrp.h>
#include <phhalMem.h>
#include <phhalSymCipher.h>
#endif

#define LRP_LEGACY_IMPLEMENTATION 0

#define LRP_DEBUG 0
#define LRP_EMULATE_ATHENIUM 1
#define LRP_MDC2_LENFIELDLEN 4
#define LRP_SPT_BUF_SIZE ((4 * LRP_DIM ) * LRP_BLOCK_SIZE)

/*
* Simulate SFRs via static global variables
*/
#ifdef _WIN32
static uint8_t pbKeyCopro[LRP_BLOCK_SIZE];
static uint8_t pbDataCopro[LRP_BLOCK_SIZE];
static uint8_t pbIOCopro[LRP_BLOCK_SIZE];
static uint8_t pbTMPCopro[LRP_BLOCK_SIZE];
static uint8_t LRP_copro_running = 0;
static uint8_t LRP_copro_enable = 0;
#else
#ifdef LRP_EMULATE_ATHENIUM
uint8_t pbIOCopro[LRP_BLOCK_SIZE];
#endif
#endif

phStatus_t phCryptoSym_Lrp_Encrypt(
                                  phCryptoSym_Sw_DataParams_t * pDataParams,
                                  uint16_t wOption,
                                  const uint8_t * pPlainBuffer,
                                  uint16_t  wBufferLength,
                                  uint8_t * pEncryptedBuffer
                                  )
{
#if LRP_LEGACY_IMPLEMENTATION
	phRetCode_t status;
	SecretPlaintexts_t stSecretPlaintexts;
	PlaintextVector_t stPT;
    InputVector_t stIV;
	uint32_t EnCtr  = pDataParams->dwEncCtr;
	uint8_t *pEncCtr =  (uint8_t *) &EnCtr;

	uint8_t bIV[LRP_SPT_BUF_SIZE] = {'\0'};
	/* uint8_t pbK1[LRP_BLOCK_SIZE] = {0x0};*/
	uint8_t SPTBuf[LRP_SPT_BUF_SIZE]  = {'\0'};
	uint8_t bWorkspace[LRP_BLOCK_SIZE] = {'\0'};
	uint8_t bOutput[LRP_BLOCK_SIZE * LRP_BLOCK_SIZE] = {'\0'};
	uint8_t pKeyBuff[LRP_BLOCK_SIZE] = {'\0'};
	uint8_t index=0;
	uint8_t i=1;

	wOption = 0x00; /* Avoid warning */

	/* copy the Keysession master, Directly assigning pDataParams->pKey to stSecretPlaintexts.pbUpdatedKey will modify the pDataParams->pKey. Hence copy it to local buffer */
	memcpy(pKeyBuff, pDataParams->pKey, LRP_BLOCK_SIZE);

	/* Frame Secret Plain Text */
	stSecretPlaintexts.pbUpdatedKey = pKeyBuff;
	stSecretPlaintexts.pbSecretPlaintexts =  SPTBuf;

	/* Prepare Plain text vector */
	stPT.wNumBytesInput = wBufferLength;
    stPT.pbInput = (uint8_t*)pPlainBuffer;

	/* Prepare IV Data */
	bIV[3] = (uint8_t) *pEncCtr;
	pEncCtr++;

	bIV[2] =  (uint8_t)*pEncCtr;
	pEncCtr++;

	bIV[1] =  (uint8_t)*pEncCtr;
	pEncCtr++;

	bIV[0] =  (uint8_t)*pEncCtr;

	stIV.pbInput = bIV;
	stIV.bNumNibblesOffset =  0x00;
	stIV.bNumNibblesInput = 0x08;

	/* Get the SPT */
	status = LRP_GeneratePlaintexts(&stSecretPlaintexts);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}
	for(index = 0; index <= 0x01 ; index++)
	{
		/* First loop generates MAC Key, Second loop generates ENC Key */
		status = LRP_GenerateUpdatedKey(stSecretPlaintexts.pbUpdatedKey, i);
		if(status != LRP_OK)
		{
			return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
		}
		/* Decrement the i value by 1*/
		i -= 1;
	}

	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}
	LRP_PrintBlock("Enciphered Text : ", stPT.pbInput);

	status = LRP_CTR(&stSecretPlaintexts, &stPT,  &stIV, bWorkspace, bOutput);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	LRP_PrintBlock("Plain Text : ", bOutput);

	/* The Output contains the encrypted data */
	memcpy(pEncryptedBuffer,bOutput, wBufferLength);

	memset(bWorkspace, 0x00, LRP_BLOCK_SIZE);
#else
	LRP_Result_t status =  LRP_ERROR;
	uint8_t bSecretPlaintext[LRP_BLOCK_SIZE * LRP_BLOCK_SIZE] = {'\0'};
	uint8_t bUpdatedKey[LRP_BLOCK_SIZE * 2] = {'\0'};
	uint8_t pKeyBuff[LRP_BLOCK_SIZE] = {'\0'};
	uint8_t bCounter[LRP_DIM] = {'\0'};
	uint8_t bOutput[LRP_BLOCK_SIZE * LRP_BLOCK_SIZE * 2] = {'\0'};
	uint8_t bPadding = 0x00;

	uint32_t EnCtr  = pDataParams->dwEncCtr;
	uint8_t *pEncCtr =  (uint8_t *) &EnCtr;
	uint8_t bNumberOfRounds = 0x00;
	wOption = 0x00; /* Avoid warning */

	/* copy the Keysession master, Directly assigning pDataParams->pKey to the LRP_KEY_GEN function results in key modification. Hence copy it to local buffer */
	memcpy(pKeyBuff, pDataParams->pKey, LRP_BLOCK_SIZE);

	/* Frame the counter Buffer  */
	bCounter[3] = (uint8_t) *pEncCtr;
	pEncCtr++;

	bCounter[2] =  (uint8_t)*pEncCtr;
	pEncCtr++;

	bCounter[1] =  (uint8_t)*pEncCtr;
	pEncCtr++;

	bCounter[0] =  (uint8_t)*pEncCtr;

	/* Generate the Secret Plain text
	 * Here pDataParams->pKey is our Session Auth Master Key Through this Generate Session MAC Key and use it for MACing data.
	 * As an output, bUpdatedKey[0 - 15] is the Session MAC key
	 * 	bUpdatedKey[16 - 31] is the  Session Enc key
	 */
    /* Get the number of rounds from the master data params. This is required because to Encrypt any data with 1 round LRP key(SAM_EncipherOfflineData),
	 * this  should be used. For rest other decryption, 2 round LRP key should be used
	*/
	bNumberOfRounds = pDataParams->bNumofKeystoUpdate;

	status = LRP_key_gen(bSecretPlaintext, bUpdatedKey, pKeyBuff, bNumberOfRounds);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/* Check whether padding is required or not */
	if(wBufferLength % LRP_BLOCK_SIZE)
	{
		bPadding =  0x01;
	}
	/* Now Encrypt the data using session Enc key */
	if(bNumberOfRounds == 0x01)
	{
		status  = LRP_LRICB_encrypt(bOutput, pPlainBuffer, bCounter, LRP_DIM, wBufferLength, bSecretPlaintext, &bUpdatedKey[0], bPadding);
	}
	else
	{
	     status  = LRP_LRICB_encrypt(bOutput, pPlainBuffer, bCounter, LRP_DIM, wBufferLength, bSecretPlaintext, &bUpdatedKey[16], bPadding);
	}

	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/*Copy the encrypted data */
	memcpy(pEncryptedBuffer, bOutput, wBufferLength);

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_CRYPTOSYM);

#endif
}


phStatus_t phCryptoSym_Lrp_Decrypt(
                                  phCryptoSym_Sw_DataParams_t * pDataParams,
                                  uint16_t wOption,
                                  const uint8_t * pEncryptedBuffer,
                                  uint16_t  wBufferLength,
                                  uint8_t * pPlainBuffer
                                  )
{
	LRP_Result_t status =  LRP_ERROR;
	uint8_t bSecretPlaintext[LRP_BLOCK_SIZE * LRP_BLOCK_SIZE] = {'\0'};
	uint8_t bUpdatedKey[LRP_BLOCK_SIZE * 2] = {'\0'};
	uint8_t pKeyBuff[LRP_BLOCK_SIZE] = {'\0'};
	uint8_t bCounter[LRP_DIM] = {'\0'};
	uint8_t bOutput[LRP_BLOCK_SIZE * LRP_BLOCK_SIZE] = {'\0'};
	uint8_t bPadding = 0x00;
	uint16_t wSizeofPlainText = 0x00;

	uint32_t EnCtr  = pDataParams->dwEncCtr;
	uint8_t *pEncCtr =  (uint8_t *) &EnCtr;
	uint8_t bNumberOfRounds = 0x00;
	wOption = 0x00; /* Avoid warning */

	/* copy the Keysession master, Directly assigning pDataParams->pKey to the LRP_KEY_GEN function results in key modification. Hence copy it to local buffer */
	memcpy(pKeyBuff, pDataParams->pKey, LRP_BLOCK_SIZE);

	/* Frame the counter Buffer  */
	bCounter[3] = (uint8_t) *pEncCtr;
	pEncCtr++;

	bCounter[2] =  (uint8_t)*pEncCtr;
	pEncCtr++;

	bCounter[1] =  (uint8_t)*pEncCtr;
	pEncCtr++;

	bCounter[0] =  (uint8_t)*pEncCtr;

	/* Generate the Secret Plain text
	 * Here pDataParams->pKey is our Session Auth Master Key Through this Generate Session MAC Key and use it for MACing data.
	 * As an output, bUpdatedKey[0 - 15] is the Session MAC key
	 * 	bUpdatedKey[16 - 31] is the  Session Enc key
	 */
	/* Get the number of rounds from the master data params. This is required because to Decrypt PICCData, 1 round LRP key should
	 * be used. For rest other decryption, 2 round LRP key should be used
	*/
	bNumberOfRounds = pDataParams->bNumofKeystoUpdate;

	status = LRP_key_gen(bSecretPlaintext, bUpdatedKey, pKeyBuff, bNumberOfRounds);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/* Check whether padding is required or not */
	if(wBufferLength % LRP_BLOCK_SIZE)
	{
		bPadding =  0x01;
	}

	/* Now Encrypt the data using session Enc key */
	if(bNumberOfRounds == 0x01)
	{
		status  = LRP_LRICB_decrypt(bOutput, pEncryptedBuffer, bCounter, LRP_DIM, &wSizeofPlainText, wBufferLength, bSecretPlaintext, &bUpdatedKey[0], bPadding);
	}
	else
	{
	 	status  = LRP_LRICB_decrypt(bOutput, pEncryptedBuffer, bCounter, LRP_DIM, &wSizeofPlainText, wBufferLength, bSecretPlaintext, &bUpdatedKey[16], bPadding);
	}
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/*Copy the decrypted data */
	memcpy(pPlainBuffer, bOutput, wBufferLength);

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_CRYPTOSYM);
}

phStatus_t phCryptoSym_Lrp_Decrypt_VariableIV(
                                  phCryptoSym_Sw_DataParams_t * pDataParams,
                                  uint16_t wOption,
								  uint8_t *pIv,
								  uint8_t wIvSize,
                                  const uint8_t * pEncryptedBuffer,
                                  uint16_t  wBufferLength,
                                  uint8_t * pPlainBuffer
                                  )
{
	LRP_Result_t status =  LRP_ERROR;
	uint8_t bSecretPlaintext[LRP_BLOCK_SIZE * LRP_BLOCK_SIZE] = {'\0'};
	uint8_t bUpdatedKey[LRP_BLOCK_SIZE * 2] = {'\0'};
	uint8_t pKeyBuff[LRP_BLOCK_SIZE] = {'\0'};
	uint8_t bCounter[LRP_BLOCK_SIZE] = {'\0'};
	uint8_t bOutput[LRP_BLOCK_SIZE * LRP_BLOCK_SIZE] = {'\0'};
	uint8_t bPadding = 0x00;
	uint16_t wSizeofPlainText = 0x00;
	uint16_t i=0;
	/*uint32_t EnCtr  = pDataParams->dwEncCtr;
	uint8_t *pEncCtr =  (uint8_t *) &EnCtr;*/
	uint8_t keyIndex = 0x00;
	wOption = 0x00; /* Avoid warning */

	memset(bCounter, 0x00, LRP_BLOCK_SIZE);
	/* copy the Keysession master, Directly assigning pDataParams->pKey to the LRP_KEY_GEN function results in key modification. Hence copy it to local buffer */
	memcpy(pKeyBuff, pDataParams->pKey, LRP_BLOCK_SIZE);

	/* Frame the counter Buffer  */

	for(i=0; wIvSize>i; i++){
		bCounter[i]=pIv[i];
	}

	/* Generate the Secret Plain text
	 * Here pDataParams->pKey is our Session Auth Master Key Through this Generate Session MAC Key and use it for MACing data.
	 * As an output, bUpdatedKey[0 - 15] is the Session MAC key
	 * 	bUpdatedKey[16 - 31] is the  Session Enc key
	 */
	status = LRP_key_gen(bSecretPlaintext, bUpdatedKey, pKeyBuff, 0x02);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/* Check whether padding is required or not */
	if(wBufferLength % LRP_BLOCK_SIZE)
	{
		bPadding =  0x01;
	}
	if(wIvSize == 0x06)  /*TODO:Hack needs to be done elegantly */
	{
		keyIndex = 16;
	}
	/* Now Encrypt the data using session Enc key */
	status  = LRP_LRICB_decrypt(bOutput, pEncryptedBuffer, bCounter, wIvSize, &wSizeofPlainText, wBufferLength, bSecretPlaintext, &bUpdatedKey[keyIndex], bPadding);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/*Copy the decrypted data */
	memcpy(pPlainBuffer, bOutput, wBufferLength);

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_CRYPTOSYM);
}

phStatus_t phCryptoSym_Lrp_LoadKeyDirect(phCryptoSym_Sw_DataParams_t *pDataParams,
								const uint8_t * pKey,
								uint16_t wKeyType
								)
{
	 /* Clear existing key */
    memset(pDataParams->pKey, 0x00, sizeof(pDataParams->pKey));

	/* We only need to copy the key provided... */
	if((wKeyType == PH_CRYPTOSYM_KEY_TYPE_AES128) || (wKeyType == PH_CRYPTOSYM_KEY_TYPE_2K3DES))
	{
		memcpy(pDataParams->pKey, pKey, PH_CRYPTOSYM_AES128_KEY_SIZE); /* PRQA S 3200 */
	}
	else if((wKeyType == PH_CRYPTOSYM_KEY_TYPE_AES192) || (wKeyType == PH_CRYPTOSYM_KEY_TYPE_3K3DES))
	{
		memcpy(pDataParams->pKey, pKey, PH_CRYPTOSYM_AES192_KEY_SIZE);
	}
	else if(wKeyType == PH_CRYPTOSYM_KEY_TYPE_DES)
	{
		memcpy(pDataParams->pKey, pKey, PH_CRYPTOSYM_DES_KEY_SIZE);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_CRYPTOSYM);
}


phStatus_t phCryptoSym_Lrp_CalculateMac(
                                       phCryptoSym_Sw_DataParams_t * pDataParams,
                                       uint16_t wOption,
                                       const uint8_t * pData,
                                       uint16_t  wDataLength,
                                       uint8_t * pMac,
                                       uint8_t * pMacLength
                                       )
{
	/* Frame the data as per the LRP implementation */
	phRetCode_t status;
	if( wOption == PH_CRYPTOSYM_MAC_MODE_LRP_SKM)
	{
		/* Set the IV as 0 IV */
		memset(pDataParams->pIV, 0x00, LRP_BLOCK_SIZE);

		/* Generate Session key master */
		PH_CHECK_SUCCESS_FCT(status, phCryptoSym_Lrp_Int_GenerateSessionAuthMasterKey(pDataParams, pData, wDataLength, pMac, pMacLength));
	}
	else if(wOption == PH_CRYPTOSYM_MAC_MODE_LRP_GENPCDRESP)
	{
		PH_CHECK_SUCCESS_FCT(status, phCryptoSym_Lrp_Int_GeneratePcdResponse(pDataParams, wOption, pData, wDataLength, pMac, pMacLength));
	}
	else
	{
		/* Regular Macing operation */
		 if ((wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT) ||
        (pDataParams->wKeepIV == PH_CRYPTOSYM_VALUE_KEEP_IV_ON))
		{
			/* better leave the IV */
		}else
		{
			memset(pDataParams->pIV, 0x00, LRP_BLOCK_SIZE);
		}

		/*Let's find out whether we should complete the MAC or if this is just an intermediate MAC calculation */
		if (wOption & PH_EXCHANGE_BUFFERED_BIT)
		{
			/* This is just an intermediate MAC */

			/* In this case we do not allow incomplete blocks. */
			if (wDataLength % LRP_BLOCK_SIZE)
			{
				return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_CRYPTOSYM);
			}
		}
		PH_CHECK_SUCCESS_FCT(status, phCryptoSym_Lrp_Int_GenerateMac(pDataParams, wOption, pData, wDataLength, pMac, pMacLength));

		if ((pDataParams->wKeepIV == PH_CRYPTOSYM_VALUE_KEEP_IV_ON) || (wOption & PH_EXCHANGE_BUFFERED_BIT))
		{
			memcpy(pDataParams->pIV, pMac, LRP_BLOCK_SIZE );
		}
		else
		{
			/* Clear the IV for security reasons */
			memset(pDataParams->pIV, 0, LRP_BLOCK_SIZE );
		}
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_CRYPTOSYM);
}

phStatus_t phCryptoSym_Lrp_Int_GenerateSessionAuthMasterKey(
                                       phCryptoSym_Sw_DataParams_t * pDataParams,
                                       const uint8_t * pData,
                                       uint16_t  wDataLength,
                                       uint8_t * pMac,
                                       uint8_t * pMacLength
                                       )
{
#if LRP_LEGACY_IMPLEMENTATION
		/* Frame the data as per the LRP implementation */
	phRetCode_t status;
	SecretPlaintexts_t stSecretPlaintexts;
	PlaintextVector_t stPT;
    InputVector_t stIV;
	/* uint8_t pbK1[LRP_BLOCK_SIZE] = {0x0}; */
	uint8_t *pbK1;
	uint8_t SPTBuf[LRP_SPT_BUF_SIZE]  = {'\0'};
	uint8_t pbWorkspace[LRP_BLOCK_SIZE] = {'\0'};
    uint8_t finalize = LRP_EVAL_FINALIZE;

	/* Prepare the Input data */
	stPT.wNumBytesInput = wDataLength;
    stPT.pbInput = (uint8_t*)pData;

	/* Frame IV */
    stIV.bNumNibblesInput = LRP_BLOCK_SIZE*2;
    stIV.bNumNibblesOffset = 0;
	stIV.pbInput = pDataParams->pIV;

	/* Frame Secre Plain Text */
	stSecretPlaintexts.pbUpdatedKey = pDataParams->pKey;
	stSecretPlaintexts.pbSecretPlaintexts =  SPTBuf;

	/* Set pbK1 */
	pbK1 = 0;

	/* Get the SPT */
	status = LRP_GeneratePlaintexts(&stSecretPlaintexts);
	/* For new Implementation
		/* LRP_key_gen(array[256], array[16], Kx, 0x01);
		array[0 - 15] -- Session Auth Master key
		array[16 - 31] -- Session Enc k
		Remove all other checks.
		*/
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/* Generate Updated Key based on the given input key Kx */
	status = LRP_GenerateUpdatedKey(stSecretPlaintexts.pbUpdatedKey, 0x01);

	/* For new Implementation
	LRP_Result_t LRP_key_gen(&stSecretPlaintexts, stSecretPlaintexts.pbUpdatedKey, pDataParams->pKey, 1)

	**************/
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	LRP_PrintBlock("Input for MAC: ", stPT.pbInput);

	/* Start MAC Operation */
	status = LRP_CMAC(&stSecretPlaintexts, &stPT, &stIV, pbK1, pbWorkspace, finalize);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/* Print the output mac */
	LRP_PrintBlock("Maced Data : ", stIV.pbInput);

	/* Copy the Requested Mac to the output buffer */
	memcpy(pMac, stIV.pbInput, wDataLength);

	*pMacLength =  LRP_BLOCK_SIZE;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_CRYPTOSYM);
#else
	LRP_Result_t status =  LRP_ERROR;
	uint8_t bSecretPlaintext[LRP_BLOCK_SIZE * LRP_BLOCK_SIZE] = {'\0'};
	uint8_t bUpdatedKey[LRP_BLOCK_SIZE] = {'\0'};
	uint8_t bSessionAuthMaster[LRP_BLOCK_SIZE] = {'\0'};

	/* Generate SPT  and Updated Key */
	status =  LRP_key_gen(bSecretPlaintext, bUpdatedKey, pDataParams->pKey, 0x01);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}
	/* Generate Session Auth master */
	status = LRP_CMAC(bSessionAuthMaster, pData, wDataLength, bSecretPlaintext, bUpdatedKey);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/* Copy the Requested Mac to the output buffer */
	memcpy(pMac, bSessionAuthMaster, LRP_BLOCK_SIZE);
	*pMacLength =  (uint8_t)LRP_BLOCK_SIZE;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_CRYPTOSYM);

#endif
}

phStatus_t phCryptoSym_Lrp_Int_GenerateMac(
                                       phCryptoSym_Sw_DataParams_t * pDataParams,
									   uint16_t wOption,
                                       const uint8_t * pData,
                                       uint16_t  wDataLength,
                                       uint8_t * pMac,
                                       uint8_t * pMacLength
                                       )
{
#if LRP_LEGACY_IMPLEMENTATION
	/* Frame the data as per the LRP implementation */
	phRetCode_t status;
	uint8_t i = 1;
	SecretPlaintexts_t stSecretPlaintexts;
	PlaintextVector_t stPT;
	uint8_t pKeyBuff[LRP_BLOCK_SIZE] = {'\0'};
    InputVector_t stIV;
	uint8_t *pbK1;
	uint8_t SPTBuf[LRP_SPT_BUF_SIZE]  = {'\0'};
	uint8_t pbWorkspace[LRP_BLOCK_SIZE] = {'\0'};
    uint8_t finalize = LRP_EVAL_FINALIZE;
    uint8_t lrpIvSize = LRP_BLOCK_SIZE;

	/* Prepare the Input data */
	stPT.wNumBytesInput = wDataLength;
    stPT.pbInput = (uint8_t*)pData;

	/* Frame IV */
    stIV.bNumNibblesInput = LRP_BLOCK_SIZE*2;
    stIV.bNumNibblesOffset = 0;

	/* Manage IV based on the wOption */
	stIV.pbInput = pDataParams->pIV;

	/* copy the Keysession master, Directly assigning pDataParams->pKey to stSecretPlaintexts.pbUpdatedKey will modify the pDataParams->pKey. Hence copy it to local buffer */
	memcpy(pKeyBuff, pDataParams->pKey, LRP_BLOCK_SIZE);

	/* Frame Secre Plain Text */
	stSecretPlaintexts.pbUpdatedKey = pKeyBuff;
	stSecretPlaintexts.pbSecretPlaintexts =  SPTBuf;

	/* Set pbK1 */
	pbK1 = 0;

	/* Set the Finalize */
	if(wOption & PH_EXCHANGE_BUFFERED_BIT)
	{
		finalize = LRP_EVAL_DONT_FINALIZE;
	}

	/* Get the SPT */
	status = LRP_GeneratePlaintexts(&stSecretPlaintexts);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/* set the stSecretPlaintexts.pbUpdatedKey to KSessionKeyMaster */
	status = LRP_GenerateUpdatedKey(stSecretPlaintexts.pbUpdatedKey, i);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	LRP_PrintBlock("Input for MAC: ", stPT.pbInput);

	/* Start MAC Operation */
	status = LRP_CMAC(&stSecretPlaintexts, &stPT, &stIV, pbK1, pbWorkspace, finalize);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/* Print the output mac */

	LRP_PrintBlock("Maced Data : ", stIV.pbInput);

	/* Copy the Requested Mac to the output buffer */
	memcpy(pMac, stIV.pbInput, LRP_BLOCK_SIZE);

	*pMacLength =  (uint8_t)lrpIvSize;

	/* Check whether IV to be updated */
	if ((pDataParams->wKeepIV == PH_CRYPTOSYM_VALUE_KEEP_IV_ON) || (wOption & PH_EXCHANGE_BUFFERED_BIT))
    {
        memcpy(pDataParams->pIV, pMac, LRP_BLOCK_SIZE ); /* PRQA S 3200 */
    }else
    {
        /* Clear the IV for security reasons */
        memset(pDataParams->pIV, 0, LRP_BLOCK_SIZE ); /* PRQA S 3200 */
    }
#else
	LRP_Result_t status =  LRP_ERROR;
	uint8_t bSecretPlaintext[LRP_BLOCK_SIZE * LRP_BLOCK_SIZE] = {'\0'};
	uint8_t bUpdatedKey[LRP_BLOCK_SIZE * 2] = {'\0'};
	uint8_t bMacOutPut[LRP_BLOCK_SIZE] = {'\0'};
	uint8_t pKeyBuff[LRP_BLOCK_SIZE] = {'\0'};

	/* copy the Keysession master, Directly assigning pDataParams->pKey to LRP_KEY_GEN may modify the key. Hence copy it to local buffer */
	memcpy(pKeyBuff, pDataParams->pKey, LRP_BLOCK_SIZE);

	/* Generate the Secret Plain text
	 * Here pDataParams->pKey is our Session Auth Master Key Through this Generate Session MAC Key and use it for MACing data.
	 * As an output, bUpdatedKey[0 - 15] is the Session MAC key
	 * 	bUpdatedKey[16 - 31] is the  Session Enc key
	 */
	status = LRP_key_gen(bSecretPlaintext, bUpdatedKey, pKeyBuff, 0x01);

	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	if ((pDataParams->wKeepIV == PH_CRYPTOSYM_VALUE_KEEP_IV_ON) || (wOption & PH_EXCHANGE_BUFFERED_BIT))
	{
		status =  LRP_CMAC_incremental(bMacOutPut, pData, wDataLength, pDataParams->pIV, bSecretPlaintext, &bUpdatedKey[0], 0x00);
	}
	else
	{
		status =  LRP_CMAC_incremental(bMacOutPut, pData, wDataLength, pDataParams->pIV, bSecretPlaintext, &bUpdatedKey[0], 0x01);
	}

	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/* Check whether IV to be updated */
	if ((pDataParams->wKeepIV == PH_CRYPTOSYM_VALUE_KEEP_IV_ON) || (wOption & PH_EXCHANGE_BUFFERED_BIT))
    {
        memcpy(pDataParams->pIV, pMac, LRP_BLOCK_SIZE );
    }else
    {
        /* Clear the IV for security reasons */
        memset(pDataParams->pIV, 0, LRP_BLOCK_SIZE );
    }

	/* Copy the Requested Mac to the output buffer */
	memcpy(pMac, bMacOutPut, LRP_BLOCK_SIZE);

	*pMacLength =  (uint8_t)LRP_BLOCK_SIZE;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_CRYPTOSYM);
#endif
}

phStatus_t phCryptoSym_Lrp_Int_GeneratePcdResponse(
                                       phCryptoSym_Sw_DataParams_t * pDataParams,
									   uint16_t wOption,
                                       const uint8_t * pData,
                                       uint16_t  wDataLength,
                                       uint8_t * pMac,
                                       uint8_t * pMacLength
                                       )
{

#if LRP_LEGACY_IMPLEMENTATION
	/* Frame the data as per the LRP implementation */
	phRetCode_t status;
	uint8_t index = 0;
	uint8_t i = 1;
	SecretPlaintexts_t stSecretPlaintexts;
	PlaintextVector_t stPT;
	uint8_t pKeyBuff[LRP_BLOCK_SIZE] = {'\0'};
    InputVector_t stIV;
	uint8_t *pbK1;
	uint8_t SPTBuf[LRP_SPT_BUF_SIZE]  = {'\0'};
	uint8_t StIVBuf[LRP_BLOCK_SIZE * 2]  = {'\0'};
	uint8_t pbWorkspace[LRP_BLOCK_SIZE] = {'\0'};
    uint8_t finalize = LRP_EVAL_FINALIZE;
    uint8_t lrpIvSize = LRP_BLOCK_SIZE;

	/* Prepare the Input data */
	stPT.wNumBytesInput = wDataLength;
    stPT.pbInput = (uint8_t*)pData;

	/* Frame IV */
    stIV.bNumNibblesInput = LRP_BLOCK_SIZE*2;
    stIV.bNumNibblesOffset = 0;

	/* Manage IV based on the wOption */
	stIV.pbInput = StIVBuf;

	/* copy the Keysession master, Directly assigning pDataParams->pKey to stSecretPlaintexts.pbUpdatedKey will modify the pDataParams->pKey. Hence copy it to local buffer */
	memcpy(pKeyBuff, pDataParams->pKey, LRP_BLOCK_SIZE);

	/* Frame Secre Plain Text */
	stSecretPlaintexts.pbUpdatedKey = pKeyBuff;
	stSecretPlaintexts.pbSecretPlaintexts =  SPTBuf;

	/* Set pbK1 */
	pbK1 = 0;

	wOption = 0; /* To avoid warning */

	/* Get the SPT */
	status = LRP_GeneratePlaintexts(&stSecretPlaintexts);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/* set the stSecretPlaintexts.pbUpdatedKey to KSessionKeyMaster */
	for(index = 0; index <= pDataParams->bNumofKeystoUpdate ; index++)
	{
		/* LRP_key_gen(array[256], array[32], Session Auth Master Key, 0x02);
		array[0 - 15] -- Session MAC k
		array[16 - 31] -- Session Enc k
		Remove all other checks.
		*/

		status = LRP_GenerateUpdatedKey(stSecretPlaintexts.pbUpdatedKey, i);
		if(status != LRP_OK)
		{
			return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
		}
		if(index == 1)
		{
			/* First loop generates Session ENC Key: stSecretPlaintexts.pbUpdatedKey has to be stored in internal structure
				* Copy the 16 bytes of Session ENC Key to (384 - 32)nd position
				*/
			memcpy(&pDataParams->pKey[(PH_CRYPTOSYM_SW_KEY_BUFFER_SIZE * 2) - (LRP_BLOCK_SIZE * 2)], stSecretPlaintexts.pbUpdatedKey, LRP_BLOCK_SIZE);
		}
		if(index==0)
		{

			/* Second loop generates Session MAC  Key: stSecretPlaintexts.pbUpdatedKey has to be stored in internal structure
			 * Copy the 16 bytes of Session MAC Key to (384 - 16)th position
			 */
			memcpy(&pDataParams->pKey[(PH_CRYPTOSYM_SW_KEY_BUFFER_SIZE * 2) - LRP_BLOCK_SIZE], stSecretPlaintexts.pbUpdatedKey, LRP_BLOCK_SIZE);


		}
		/* Decrement the i value by 1*/
		i -= 1;
	}

	LRP_PrintBlock("Input for MAC: ", stPT.pbInput);

	/* Copy the Session MAC key which has to be restored to stSecretPlaintexts.pbUpdatedKey. Otherwise second loop has already
	 * overwrite the stSecretPlaintexts.pbUpdatedKey Key with Session ENC Key */
	memcpy(stSecretPlaintexts.pbUpdatedKey, &pDataParams->pKey[(PH_CRYPTOSYM_SW_KEY_BUFFER_SIZE * 2) - LRP_BLOCK_SIZE], LRP_BLOCK_SIZE);

	/* Start MAC Operation to generate PCD Response */
	status = LRP_CMAC(&stSecretPlaintexts, &stPT, &stIV, pbK1, pbWorkspace, finalize);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/* Print the output mac */

	LRP_PrintBlock("Maced Data : ", stIV.pbInput);

	/* Copy the Requested Mac to the output buffer */
	memcpy(pMac, stIV.pbInput, LRP_BLOCK_SIZE);

	*pMacLength =  (uint8_t)lrpIvSize;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_CRYPTOSYM);

#else
	LRP_Result_t status =  LRP_ERROR;
	uint8_t bSecretPlaintext[LRP_BLOCK_SIZE * LRP_BLOCK_SIZE] = {'\0'};
	uint8_t bUpdatedKey[LRP_BLOCK_SIZE * 2] = {'\0'};
	uint8_t bPcdResponse[LRP_BLOCK_SIZE] = {'\0'};

	wOption = 0; /* To avoid warning */

	/* Generate the Secret Plain text
	 * Here pDataParams->pKey is our Key Session auth master.
	 * As an output, bUpdatedKey[0 - 15] is the Session MAC key
	 * 	bUpdatedKey[16 - 31] is the  Session Enc key
	 */
	status = LRP_key_gen(bSecretPlaintext, bUpdatedKey, pDataParams->pKey, 0x02);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/* Now copy Session MAC Key to pDataParams Key Buffer */
	memcpy(&pDataParams->pKey[(PH_CRYPTOSYM_SW_KEY_BUFFER_SIZE * 2) - (LRP_BLOCK_SIZE * 2)], &bUpdatedKey[0], LRP_BLOCK_SIZE);

	/* Now copy Session ENC Key to pDataParams Key Buffer */
	memcpy(&pDataParams->pKey[(PH_CRYPTOSYM_SW_KEY_BUFFER_SIZE * 2) - (LRP_BLOCK_SIZE)], &bUpdatedKey[16], LRP_BLOCK_SIZE);


	/* Now Generate the PCD response using session key */
	status = LRP_CMAC(bPcdResponse, pData, wDataLength, bSecretPlaintext, &bUpdatedKey[0]);
	if(status != LRP_OK)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_CRYPTOSYM);
	}

	/* Copy the Requested Mac to the output buffer */
	memcpy(pMac, bPcdResponse, LRP_BLOCK_SIZE);

	*pMacLength =  (uint8_t)LRP_BLOCK_SIZE;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_CRYPTOSYM);
#endif
}