/*
 * Copyright 2021 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal functions of Software implementation of MIFARE DUOX application layer.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7464 $
 * $Date: 2025-08-29 16:56:35 +0530 (Fri, 29 Aug 2025) $
 *
 * History:
 *  Rajendran Kumar: Generated 14. Sep 2021
 *
 */

#include <ph_Status.h>

#ifdef NXPBUILD__PHAL_MFDUOX_SW

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
#include <phKeyStore.h>
#include <phCryptoASym.h>
#include <phCryptoSym.h>
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

#include <phpalMifare.h>
#include <phpalI14443p4.h>

#ifdef NXPBUILD__PH_TMIUTILS
#include <phTMIUtils.h>
#endif /* NXPBUILD__PH_TMIUTILS */

#ifdef NXPBUILD__PHAL_VCA
#include <phalVca.h>
#endif /* NXPBUILD__PHAL_VCA */

#include "phalMfDuoX_Sw_Int.h"
#include "../phalMfDuoX_Int.h"





#ifdef NXPBUILD__PHAL_MFDUOX_NDA
uint8_t PH_MEMLOC_CONST_ROM phalMfDuoX_Sw_ZeroIv[PH_CRYPTOSYM_AES_BLOCK_SIZE];
#endif /*NXPBUILD__PHAL_MFDUOX_NDA */

static const uint16_t PH_MEMLOC_CONST_ROM aFrameSize[] = { 16, 24, 32, 40, 48, 64, 96, 128, 256, 512, 1024, 2048, 4096 };





phStatus_t phalMfDuoX_Sw_Int_ValidateResponse(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint16_t wStatus,
    uint16_t wPiccRetCode)
{
    uint16_t PH_MEMLOC_REM wPICCStatusCode = 0;

    /* Evaluate the response. */
    if(wStatus == PH_ERR_SUCCESS)
    {
        /* Frame PICC Status Code. */
        wPICCStatusCode = (uint16_t) ((bOption == PHAL_MFDUOX_ISO7816_APDU_CMD) ? wPiccRetCode : (wPiccRetCode & 0x00FFU));

        /* Validate the PICC Status. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_ComputeErrorResponse(pDataParams, wPICCStatusCode));
    }
    else
    {
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_AL_MFDUOX);
        }

        PH_CHECK_SUCCESS(wStatus);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_Int_CardExchange(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wBufferOption, uint8_t bChainingState,
    uint8_t bCmdOption, uint16_t wTotDataLen, uint8_t bExchangeLE, uint8_t * pData, uint16_t wDataLen, uint8_t ** ppResponse,
    uint16_t * pRespLen, uint8_t * pPiccErrCode)
{
    phStatus_t      PH_MEMLOC_REM wStatus = 0;
    phStatus_t      PH_MEMLOC_REM wPICCStatus = 0;
    uint16_t        PH_MEMLOC_REM wOption = 0;
    uint16_t        PH_MEMLOC_REM wFrameLen = 0;
    uint16_t        PH_MEMLOC_REM wWrappedLen = 0;
    uint16_t        PH_MEMLOC_REM wBytesPending = 0;
    uint16_t        PH_MEMLOC_REM wLc = 0;
    uint16_t        PH_MEMLOC_REM wRspLen = 0;
    uint8_t         PH_MEMLOC_REM bPICCStatLen = 0;
    uint8_t         PH_MEMLOC_REM bLcLen = 0;
    uint8_t         PH_MEMLOC_REM bOffset = 0;
    uint8_t         PH_MEMLOC_REM bCheckStatus = 0;
    uint8_t         PH_MEMLOC_REM bCmdFormat = 0;
    uint8_t         PH_MEMLOC_REM bChaining = PH_OFF;

    static uint16_t PH_MEMLOC_REM wBytesExchanged;
    static uint8_t  PH_MEMLOC_REM bLeLen;

    uint8_t*        PH_MEMLOC_REM pResponse = NULL;
    uint8_t         PH_MEMLOC_REM aLe[3U] = { 0x00, 0x00, 0x00 };

    uint8_t         PH_MEMLOC_REM aISO7816Header[8] = { PHAL_MFDUOX_WRAPPEDAPDU_CLA, 0x00, PHAL_MFDUOX_WRAPPEDAPDU_P1,
        PHAL_MFDUOX_WRAPPEDAPDU_P2, 0x00, 0x00, 0x00 };
    uint8_t         PH_MEMLOC_REM bISO7816HeaderLen = 4U;

    /* Get PICC Frame size. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_GetFrameLen(pDataParams, PH_OFF, &wFrameLen));

    /* Get Remaining bytes to be exchanged. */
    phpalMifare_GetConfig(pDataParams->pPalMifareDataParams, PHPAL_I14443P4_CONFIG_REMAINING_BYTES_TO_EXCHANGE, &wBytesPending);

    if(pDataParams->bWrappedMode)
    {
        if(!(wBufferOption & PH_EXCHANGE_LEAVE_BUFFER_BIT) || (wBufferOption == PH_EXCHANGE_DEFAULT))
        {
            wWrappedLen = (uint16_t) (4U /* CLA, P1, P2, LC. */ + bExchangeLE);
            wWrappedLen = (uint16_t) (pDataParams->bShortLenApdu ? wWrappedLen :
                (wWrappedLen + (2U /* Extended LC */ + bExchangeLE /* Extended LE */)));
        }
        else
        {
            wWrappedLen = bExchangeLE;
            wWrappedLen = (uint16_t) (pDataParams->bShortLenApdu ? wWrappedLen : (wWrappedLen + 1U /* Extended LE */));
        }
    }
    else
    {
        /* Nothing to do. */
    }

    /* Set the buffering options to be given to PAL layer. */
    if((wBufferOption & PH_EXCHANGE_BUFFERED_BIT) || (wBufferOption == PH_EXCHANGE_DEFAULT))
    {
        /* Update bytes exchanged */
        wBytesExchanged += (uint16_t) (((wBufferOption & PH_EXCHANGE_LEAVE_BUFFER_BIT) == 0U) ? 0U : wDataLen);

        /* Set the Chaining Options */
        bChaining = (uint8_t) (((wTotDataLen - wBytesExchanged) + wWrappedLen) > wFrameLen);
        bChaining = (uint8_t) ((wBytesPending > wFrameLen) ? PH_ON : bChaining);
        bChaining = (uint8_t) ((wBufferOption == PH_EXCHANGE_DEFAULT) ? PH_OFF : bChaining);

        /* Update bytes exchanged */
        wBytesExchanged += (uint16_t) (((wBufferOption & PH_EXCHANGE_LEAVE_BUFFER_BIT) == 0U) ? wDataLen : 0U);
        wBytesExchanged = (uint16_t) ((wBufferOption == PH_EXCHANGE_DEFAULT) ? 0U : wBytesExchanged);
    }
    else
    {
        bChaining = PH_OFF;
        wBytesExchanged = 0U;
    }

    /* Set the Options */
    bChaining = (uint8_t) ((bChainingState != PHAL_MFDUOX_CHAINING_BIT_INVALID) ? bChainingState : bChaining);
    wOption = (uint16_t) (bChaining | wBufferOption);

        /* Exchange the command in Iso7816 wrapped format ----------------------------------------------------------------------------- */
    if(pDataParams->bWrappedMode)
    {
        if((wBufferOption == PH_EXCHANGE_BUFFER_FIRST) || (wBufferOption == PH_EXCHANGE_DEFAULT))
        {
            bLeLen = 1U;

            /* Set the LC information. */
            wLc = (uint16_t) (wTotDataLen - 1 /* Excluding the command code. */);

            /* Update the command code to Iso7816 header */
            aISO7816Header[1U] = pData[0];

            /* Add LC if available */
            if(wLc)
            {
                /* Update LC bytes according to Extended APDU option. */
                if(pDataParams->bShortLenApdu == PH_OFF)
                {
                    aISO7816Header[bISO7816HeaderLen + bLcLen++] = 0x00U;
                    aISO7816Header[bISO7816HeaderLen + bLcLen++] = (uint8_t) ((wLc & 0x0000FF00U) >> 8U);

                    /* Le length is updated to two if LC is present and the APDU is extended. */
                    bLeLen = 2U;
                }

                aISO7816Header[bISO7816HeaderLen + bLcLen++] = (uint8_t) (wLc & 0x000000FFU);

                /* Update IHeader Length */
                bISO7816HeaderLen += bLcLen;
            }
            else
            {
                /* Update Le count */
                if(pDataParams->bShortLenApdu == PH_OFF)
                {
                    bLeLen = 3U;
                }
            }

            /* Add the ISO 7816 header to layer 4 buffer. */
            PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
                pDataParams->pPalMifareDataParams,
                (uint16_t) (bChaining | PH_EXCHANGE_BUFFER_FIRST),
                &aISO7816Header[0],
                bISO7816HeaderLen,
                NULL,
                NULL));

            /* Add the data to layer 4 buffer. */
            if((wDataLen - 1U) != 0U)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
                    pDataParams->pPalMifareDataParams,
                    PH_EXCHANGE_BUFFER_CONT,
                    &pData[1U],  /* Exclude the command code because it is added to INS. */
                    (uint16_t) (wDataLen - 1U),
                    NULL,
                    NULL));
            }
        }

        if(wBufferOption == PH_EXCHANGE_BUFFER_CONT)
        {
            /* Add the data to layer 4 buffer. */
            PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
                pDataParams->pPalMifareDataParams,
                wOption,
                pData,
                wDataLen,
                NULL,
                NULL));
        }

        if((wBufferOption == PH_EXCHANGE_BUFFER_LAST) || (wBufferOption == PH_EXCHANGE_DEFAULT))
        {
            if(wBufferOption == PH_EXCHANGE_BUFFER_LAST)
            {
                /* Add the data to layer 4 buffer. */
                PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
                    pDataParams->pPalMifareDataParams,
                    (uint16_t) (bChaining | PH_EXCHANGE_BUFFER_CONT),
                    pData,
                    wDataLen,
                    NULL,
                    NULL));
            }

            /* Add Le to L4 buffer and exchange the command. */
            wStatus = phpalMifare_ExchangeL4(
                pDataParams->pPalMifareDataParams,
                PH_EXCHANGE_BUFFER_LAST,
                &aLe[0],
                (uint8_t) (bExchangeLE ? bLeLen : 0),
                &pResponse,
                &wRspLen);

            /* Validate the Status. */
            if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
            {
                /* Update PICC Error code to INVALID. */
                if(pPiccErrCode != NULL)
                    *pPiccErrCode = PHAL_MFDUOX_PICC_STATUS_INVALID;

                return wStatus;
            }

            /* Should the status needs to be verified. */
            bCheckStatus = (uint8_t) ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS);

            /* Combine Sw1 and Sw2 status codes. */
            if(bCheckStatus)
                wPICCStatus = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);

            /* Evaluate the Status. */
            wStatus = phalMfDuoX_Sw_Int_ValidateResponse(pDataParams, PHAL_MFDUOX_PRODUCT_CMD, wStatus, wPICCStatus);

            /* Create memory for updating the response of ISO 14443 format. */
            *ppResponse = pResponse;

            /* Update the response buffer length excluding SW1SW2. */
            *pRespLen = (uint16_t) (wRspLen - (bCheckStatus ? 2U : 0));

            /* Copy the second byte of response (SW2) to RxBuffer */
            if(bCmdOption & PHAL_MFDUOX_RETURN_PICC_STATUS)
                if(pPiccErrCode != NULL)
                    *pPiccErrCode = pResponse[wRspLen - 1U];
        }

        if(wBufferOption == PH_EXCHANGE_RXCHAINING)
        {
            /* Exchange the command */
            wStatus = phpalMifare_ExchangeL4(
                pDataParams->pPalMifareDataParams,
                wBufferOption,
                pData,
                wDataLen,
                &pResponse,
                &wRspLen);

            /* Validate the Status. */
            if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
            {
                /* Update PICC Error code to INVALID. */
                if(pPiccErrCode != NULL)
                    *pPiccErrCode = PHAL_MFDUOX_PICC_STATUS_INVALID;

                return wStatus;
            }

            /* Should the status needs to be verified. */
            bCheckStatus = (uint8_t) ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS);

            if(wRspLen != 0)
            {
                /* Combine Sw1 and Sw2 status codes. */
                if(bCheckStatus)
                    wPICCStatus = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);

                /* Evaluate the Status. */
                wStatus = phalMfDuoX_Sw_Int_ValidateResponse(pDataParams, PHAL_MFDUOX_PRODUCT_CMD, wStatus, wPICCStatus);

                /* Create memory for updating the response of ISO 14443 format. */
                *ppResponse = pResponse;

                /* Update the response buffer length excluding SW1SW2. */
                *pRespLen = (uint16_t) (wRspLen - (bCheckStatus ? 2U : 0));

                /* Copy the second byte of response (SW2) to RxBuffer */
                if(bCmdOption & PHAL_MFDUOX_RETURN_PICC_STATUS)
                    if(pPiccErrCode != NULL)
                        *pPiccErrCode = pResponse[wRspLen - 1U];
            }
        }
    }

    /* Exchange the command in Native format or ISO7816 Standard Commands ---------------------------------------------------------- */
    else
    {
        /* Exchange the data to the card in Native format. */
        wStatus = phpalMifare_ExchangeL4(
            pDataParams->pPalMifareDataParams,
            wOption,
            pData,
            wDataLen,
            &pResponse,
            &wRspLen);

        /* Validate the Status. */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            /* Update PICC Error code to INVALID. */
            if(pPiccErrCode != NULL)
                *pPiccErrCode = PHAL_MFDUOX_PICC_STATUS_INVALID;

            return wStatus;
        }

        /* Verify the received data and update the response buffer with received data. */
        if((bCmdOption & PHAL_MFDUOX_OPTION_PENDING) ||
            (bCmdOption & PHAL_MFDUOX_OPTION_COMPLETE))
        {
            if(!(bCmdOption & PHAL_MFDUOX_EXCLUDE_PICC_STATUS))
            {
                if(bCmdOption & PHAL_MFDUOX_PICC_STATUS_WRAPPED)
                {
                    /* Combine Sw1 and Sw2 status codes. */
                    wPICCStatus = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);
                    bPICCStatLen = 2U;

                    bCmdFormat = PHAL_MFDUOX_ISO7816_APDU_CMD;
                }
                else
                {
                    wPICCStatus = pResponse[0];
                    bPICCStatLen = 1U;

                    bCmdFormat = PHAL_MFDUOX_PRODUCT_CMD;
                }
            }

            /* Evaluate the Status. */
            wStatus = phalMfDuoX_Sw_Int_ValidateResponse(pDataParams, bCmdFormat, wStatus, wPICCStatus);

            /* Update the response buffer length excluding PICC Status Code. */
            *pRespLen = wRspLen - bPICCStatLen;

            /* Set the Offset from where the data needs to be copied. */
            bOffset = (uint8_t) ((bCmdOption & PHAL_MFDUOX_PICC_STATUS_WRAPPED) ? 0 : 1U);
            bOffset = (uint8_t) ((bCmdOption & PHAL_MFDUOX_EXCLUDE_PICC_STATUS) ? 0 : bOffset);

            /* Add the Response data excluding PICC Status Code. */
            *ppResponse = &pResponse[bOffset];

            /* Update the PICC Status parameter. */
            if(bCmdOption & PHAL_MFDUOX_RETURN_PICC_STATUS)
                if(pPiccErrCode != NULL)
                    *pPiccErrCode = (uint8_t) wPICCStatus;
        }
    }

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_Send7816Apdu(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint16_t wBufOption,
    uint8_t bExtendedLenApdu, uint8_t bClass, uint8_t bIns, uint8_t bP1, uint8_t bP2, uint8_t * pData,
    uint16_t wDataLen, uint32_t dwExpBytes, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM aLe[3U] = { 0x00, 0x00, 0x00 };
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;

    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint8_t     PH_MEMLOC_REM bLeLen = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Set the command code to DataParams. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_SetCmdCode(pDataParams, bIns));

    /* Frame Standard ISO7816 - 4 Header. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bClass;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bIns;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bP1;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bP2;

    /* Check whether LC needs to be exchanged. */
    if(bOption & PHAL_MFDUOX_EXCHANGE_LC_ONLY)
    {
        /* Check whether Length LC is represented in short APDU or extended APDU */
        if(bExtendedLenApdu == PH_ON)
        {
            /* First byte will be 0x00 if Ext APDU present. Next 2 byte contains actual data. */
            PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = 0x00;

            /* As of now this field will be set to 0x00 since maximum data that can be sent is 16 bytes.
             * In case if data to be sent exceeds 255 bytes, this byte shall be used.
             */
            PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) ((wDataLen & 0xFF00U) >> 8U);
        }

        /* Short Length APDU. */
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) (wDataLen & 0x00FFU);
    }

    /* Buffer ISO7816-4 Command Header. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        &pResponse,
        &wRspLen));

    /* Check whether Le needs to be exchanged. */
    if(bOption & PHAL_MFDUOX_EXCHANGE_LE_ONLY)
    {
        /* As per ISO/IEC:7816-4(2005), Section 5, An extended LE field consists of either three bytes
         * (one byte set to '00' followed by two bytes with any value) if the LC field is absent, or
         * two bytes (with any value) if an extended LC field is present.
         */

        /* Check whether Length is represented in extended APDU format and LC is present.
         * If true, then Le should represented in 2 bytes else LE should be represented in 3 bytes
         */
        if(bExtendedLenApdu == PH_ON)
        {
            if(!(bOption & PHAL_MFDUOX_EXCHANGE_LC_ONLY))
            {
                aLe[bLeLen++] = (uint8_t) ((dwExpBytes & 0x00FF0000U) >> 16U);
            }

            aLe[bLeLen++] = (uint8_t) ((dwExpBytes & 0x0000FF00U) >> 8U);
        }

        /* Short APDU */
        aLe[bLeLen++] = (uint8_t) (dwExpBytes & 0x000000FFU);
    }

    /* Exchange the command based on the INS. */
    switch(bIns)
    {
        case PHAL_MFDUOX_CMD_ISO7816_SELECT_FILE:
            wStatus = phalMfDuoX_Sw_Int_ISOSelectFile(pDataParams, pData, wDataLen, aLe, bLeLen, ppResponse, pRspLen);
            break;

        case PHAL_MFDUOX_CMD_ISO7816_READ_BINARY:
            wStatus = phalMfDuoX_Sw_Int_ISOReadBinary(pDataParams, wBufOption, aLe, bLeLen, ppResponse, pRspLen);
            break;

        case PHAL_MFDUOX_CMD_ISO7816_UPDATE_BINARY:
            wStatus = phalMfDuoX_Sw_Int_ISOUpdateBinary(pDataParams, pData, wDataLen);
            break;

        case PHAL_MFDUOX_CMD_ISO7816_READ_RECORD:
            wStatus = phalMfDuoX_Sw_Int_ISOReadRecord(pDataParams, wBufOption, aLe, bLeLen, ppResponse, pRspLen);
            break;

        case PHAL_MFDUOX_CMD_ISO7816_APPEND_RECORD:
            wStatus = phalMfDuoX_Sw_Int_ISOAppendRecord(pDataParams, pData, wDataLen);
            break;

        case PHAL_MFDUOX_CMD_ISO7816_GET_CHALLENGE:
            wStatus = phalMfDuoX_Sw_Int_ISOGetChallenge(pDataParams, aLe, bLeLen, ppResponse, pRspLen);
            break;

        case PHAL_MFDUOX_CMD_VDE_READ_DATA:
            wStatus = phalMfDuoX_Sw_Int_VdeReadData(pDataParams, wBufOption, aLe, bLeLen, ppResponse, pRspLen);
            break;

        case PHAL_MFDUOX_CMD_VDE_WRITE_DATA:
            wStatus = phalMfDuoX_Sw_Int_VdeWriteData(pDataParams, pData, wDataLen, aLe, bLeLen);
            break;

        case PHAL_MFDUOX_CMD_VDE_ECDSA_SIGN:
            wStatus = phalMfDuoX_Sw_Int_VdeECDSASign(pDataParams, pData, wDataLen, aLe, bLeLen, ppResponse, pRspLen);
            break;

        /*
         * This case cannot be achieved as the INS that are implemented are the supported ones only.
         * This case is kept for completeness of the switch statement and to avoid error while
         * implementing new command support in future.
         */
        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_MFDUOX);
            break;
    }

    return wStatus;
}





#ifdef NXPBUILD__PHAL_MFDUOX_NDA
phStatus_t phalMfDuoX_Sw_Int_ISOGeneralAuthenticate_Part1(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bCARootKeyNo,
    uint8_t * pOptsA, uint16_t bOptsALen, uint8_t * pExpRspLen, uint8_t bExpRspLen, uint8_t ** ppResponse,
    uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wE_PubAKeyLen = 0;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint8_t     PH_MEMLOC_REM bCurveID = 0;

    uint8_t     PH_MEMLOC_REM *pResponse = NULL;

    /* Frame Cmd.ISOGeneralAuthenticate Part1 command ---------------------------------------------------------------------------------- */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_ISO7816_GENERIC_CLA;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_AUTHENTICATE_ISO_GENERAL;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_WRAPPEDAPDU_P1;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bCARootKeyNo;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_WRAPPEDAPDU_LC;

    /* Frame Extended LC. */
    if(bExpRspLen > 1)
    {
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_WRAPPEDAPDU_LC;
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_WRAPPEDAPDU_LC;
    }

    /* Add OptsA to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pOptsA, bOptsALen);
    PHAL_MFDUOX_CMD_BUF_LEN += bOptsALen;

    /* Add Ephemeral Public Key (E.Pub.A). */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_ExportKey(
        pDataParams->pCryptoDataParamsASym,
        PH_CRYPTOASYM_PUBLIC_KEY,
        PHAL_MFDUOX_PUBLIC_KEY_LEN,
        &bCurveID,
        &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN + 4],
        &wE_PubAKeyLen));

    /* Add AuthDOHdr (Authentication Data Object Header) to command buffer (TL). */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_AUTH_ISO_GENERAL_AUTH_DO_HDR_TAG;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) (2U /* Tag (T) + Length (L) */ + wE_PubAKeyLen /* Length of Ephemeral Public Key */);

    /* Add E.Pub.A Authentication Data Object to command buffer (TLV). */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_AUTH_ISO_GENERAL_EPH_PUB_KEY_TAG;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) wE_PubAKeyLen;
    PHAL_MFDUOX_CMD_BUF_LEN += wE_PubAKeyLen;

    /* Add LE. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pExpRspLen, bExpRspLen);
    PHAL_MFDUOX_CMD_BUF_LEN += bExpRspLen;

    /* Update LC data. */
    phalMfDuoX_Int_UpdateLC(PHAL_MFDUOX_CMD_BUF, PHAL_MFDUOX_CMD_BUF_LEN, PH_ON, bExpRspLen);

    /* Exchange command to PICC -------------------------------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        PHAL_MFDUOX_CHAINING_BIT_INVALID,
        (uint8_t) (PHAL_MFDUOX_OPTION_COMPLETE | PHAL_MFDUOX_PICC_STATUS_WRAPPED),
        0,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        &pResponse,
        &wRspLen,
        NULL));

    /* Validate AuthDOHdr from response. */
    if(pResponse[0] != PHAL_MFDUOX_AUTH_ISO_GENERAL_AUTH_DO_HDR_TAG)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);

    /* Validate Tag information from Ephemeral Public B (E.Pub.B) TLV */
    if(pResponse[2] != PHAL_MFDUOX_AUTH_ISO_GENERAL_EPH_PUB_KEY_TAG)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);

    /* Remove AuthDoHdr and Ephemeral Public B TL. */
    *ppResponse = &pResponse[4U];
    *pRspLen = (uint16_t) (wRspLen - 4U);

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_ISOGeneralAuthenticate_Part2(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wKeyNo_PrivA,
    uint16_t wKeyPos_PrivA, uint8_t bCurveID, uint8_t * pOptsA, uint16_t wOptsALen, uint8_t * pE_PubB,
    uint16_t wE_PubBLen, uint8_t * pCertA, uint16_t wCertALen, uint8_t * pExpRspLen, uint8_t bExpRspLen,
    uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wBER_TLV_Len = 0;
    uint16_t    PH_MEMLOC_REM wMsgEncLen = 0;
    uint16_t    PH_MEMLOC_REM wL1 = 0;
    uint8_t     PH_MEMLOC_REM bBER_Len = 0;
    uint8_t     PH_MEMLOC_REM bIsMutualAuth = PH_OFF;
    uint8_t     PH_MEMLOC_REM bInclude_CertA = PH_OFF;

    uint8_t     PH_MEMLOC_REM aSharedSecret[64];
    uint16_t    PH_MEMLOC_REM wSharedSecretLen = 0;

    uint8_t     PH_MEMLOC_REM *pE_PubA = NULL;
    uint16_t    PH_MEMLOC_REM wE_PubALen = 0;

    uint8_t     PH_MEMLOC_REM *pSignature = NULL;
    uint16_t    PH_MEMLOC_REM wSignLen = 0;

    uint8_t     PH_MEMLOC_REM *pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Clear Buffers. */
    (void) memset(aSharedSecret, 0x00, sizeof(aSharedSecret));
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Get the Status to Auth Method. */
    bIsMutualAuth = (uint8_t) ((pOptsA[2U] & PHAL_MFDUOX_ISO_GENERAL_AUTH_MUTUAL) ? PH_ON : PH_OFF);

    /* Get the Status to include Certificate (Cert.A) or not. */
    bInclude_CertA = (uint8_t) !(pOptsA[2U] & PHAL_MFDUOX_ISO_GENERAL_AUTH_EXCLUDE_CERTIFICATE);

    /* Compute Shared Secret (ShS = ECDH(E.Priv.A, E.Pub.B)) --------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_SharedSecret(
        pDataParams->pCryptoDataParamsASym,
        bCurveID,
        pE_PubB,
        wE_PubBLen,
        aSharedSecret,
        &wSharedSecretLen));

    /* Export Ephemeral Public Key (E.Pub.A) ------------------------------------------------------------------------------------------- */
    pE_PubA = PHAL_MFDUOX_CMD_BUF;
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_ExportKey(
        pDataParams->pCryptoDataParamsASym,
        PH_CRYPTOASYM_PUBLIC_KEY,
        PHAL_MFDUOX_PUBLIC_KEY_LEN,
        &bCurveID,
        pE_PubA,
        &wE_PubALen));

    /* Compute Session Keys (KSesAuthENC, KSesAuthMAC = KDF(ShS)) ---------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_GenerateSessionKeys_ASym(
        pDataParams,
        pE_PubA,
        wE_PubALen,
        pE_PubB,
        wE_PubBLen,
        aSharedSecret,
        wSharedSecretLen));

    /* Generate Signature (ECDSASign(Priv.A, Msg.A.ext)) ------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_SignMessage(
        pDataParams,
        wKeyNo_PrivA,
        wKeyPos_PrivA,
        pOptsA,
        wOptsALen,
        pE_PubA,
        wE_PubALen,
        pE_PubB,
        wE_PubBLen,
        &pSignature,
        &wSignLen));

    /* Exchange Cmd.ISOGeneralAuthenticate Part2 command --------------------------------------------------------------------------------- */

    /* Check if Host Certificate (Cert.A) should be exchanged. */
    wCertALen = (uint16_t) (bInclude_CertA ? wCertALen : 0U);

    /* Manipulate the Length information (L1). */
    wL1 = (uint16_t) (1U /* Msg.A.enc Tag */ + wCertALen + wSignLen);
    if(wL1 <= 127U)
    {
        /* Update the BER-TLV Len to 1 bytes format. */
        bBER_Len = PHAL_MFDUOX_ISO7816_BER_TLV_L_NO_CONST; /* 1 bytes of Actual Length */
    }
    else if( wL1 > 255U)
    {
        /* Update the BER-TLV Len to 2 bytes format. */
        bBER_Len = PHAL_MFDUOX_ISO7816_BER_TLV_L_82; /* Constant Length Byte + 2 bytes of Actual Length */
    }
    else
    {
        /* Update the BER-TLV Len to 1 bytes format. */
        bBER_Len = PHAL_MFDUOX_ISO7816_BER_TLV_L_81; /* Constant Length Byte + 1 bytes of Actual Length */
    }

    /* Force Extend LE information to be exchanged if greater than 255 bytes for Mutual Authentication. */
    if((wL1 > 255U) || bIsMutualAuth)
    {
        bExpRspLen = 2U;
    }

    /* Clear Buffer */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_ISO7816_GENERIC_CLA;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_AUTHENTICATE_ISO_GENERAL;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_WRAPPEDAPDU_P1;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_WRAPPEDAPDU_P2;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_WRAPPEDAPDU_LC;

    /* Frame Extended LC. */
    if(bExpRspLen > 1)
    {
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_WRAPPEDAPDU_LC;
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_WRAPPEDAPDU_LC;
    }

    /* Compute Encrypted Message A Length. */
    wMsgEncLen = (uint16_t) (2 /* constant: 0xE0E0 */ + wSignLen + wCertALen);
    wMsgEncLen = (uint16_t) (PHAL_MFDUOX_IS_MULTIPLE_AES_BLOCK_SIZE(wMsgEncLen) ? (wMsgEncLen + 16U) : wMsgEncLen);
    PHAL_MFDUOX_NEAREST_MULTIPLE(wMsgEncLen, wMsgEncLen);

    /* Compute complete BER-TLV Data Object Length. */
    wBER_TLV_Len = (uint16_t)
    (
        1U          /* Msg.A.enc Tag (T) */     +
        bBER_Len    /* Length (L) */            +
        wMsgEncLen  /* Value (V) */
    );

    /* Add AuthDOHdr (Authentication Data Object Header) to command buffer. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_AUTH_ISO_GENERAL_AUTH_DO_HDR_TAG;
    phalMfDuoX_Int_EncodeBER_TLV_Len(wBER_TLV_Len, PHAL_MFDUOX_CMD_BUF, &PHAL_MFDUOX_CMD_BUF_LEN);

    /* Add Authentication Data Object for Encrypted Message. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_AUTH_ISO_GENERAL_ENC_MSG_A_TAG;
    phalMfDuoX_Int_EncodeBER_TLV_Len(wMsgEncLen, PHAL_MFDUOX_CMD_BUF, &PHAL_MFDUOX_CMD_BUF_LEN);

    /* Update LC */
    phalMfDuoX_Int_UpdateLC(PHAL_MFDUOX_CMD_BUF, (uint16_t) (PHAL_MFDUOX_CMD_BUF_LEN + wMsgEncLen), PH_OFF, bExpRspLen);

    /* Buffer the information to be exchanged. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHAL_MFDUOX_CHAINING_BIT_INVALID,
        PHAL_MFDUOX_OPTION_NONE,
        (uint16_t) (PHAL_MFDUOX_CMD_BUF_LEN + wMsgEncLen),
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        NULL,
        NULL));

    /* Add Encrypted Message to be exchanged. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_EncryptMessage_Exchange(
        pDataParams,
        pSignature,
        wSignLen,
        pCertA,
        wCertALen,
        wMsgEncLen));

    /* Add LE and Exchange command to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        PHAL_MFDUOX_CHAINING_BIT_DISABLE,
        (uint8_t) (PHAL_MFDUOX_OPTION_COMPLETE | PHAL_MFDUOX_PICC_STATUS_WRAPPED),
        0,
        PH_OFF,
        pExpRspLen,
        bExpRspLen,
        &pResponse,
        &wRspLen,
        NULL));

    /* Validate AuthDOHdr from response. */
    if(pResponse[0] != PHAL_MFDUOX_AUTH_ISO_GENERAL_AUTH_DO_HDR_TAG)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);

    pResponse++;
    wRspLen--;

    /* Encode BER-TLV Length of AuthDOHdr */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_DecodeBER_TLV_Len(&pResponse, &wBER_TLV_Len, &wRspLen));

    /* Check the response length. */
    if(wRspLen != wBER_TLV_Len)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);

    /* Validate Tag information from Encrypted Message B TLV. */
    if(pResponse[0] != PHAL_MFDUOX_AUTH_ISO_GENERAL_ENC_MSG_B_TAG)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);

    pResponse++;
    wRspLen--;

    /* Encode BER-TLV Length of Encrypted Message B TLV */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_DecodeBER_TLV_Len(&pResponse, &wBER_TLV_Len, &wRspLen));

    /* Check the response length. */
    if(wRspLen != wBER_TLV_Len)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);

    /* Decrypt Cmd.ISOGeneralAuthenticate Part2 Response ------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_DecryptMessage(
        pDataParams,
        bIsMutualAuth,
        pResponse,
        wRspLen,
        &pResponse,
        &wRspLen));

    /* Copy Decrypted message to parameter. */
    *ppResponse = pResponse;
    *pRspLen = wRspLen;

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_ISOGeneralAuthenticate_Final(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wKeyNo_PrivA,
    uint16_t wKeyPos_PrivA, uint8_t bCARootKeyNo, uint8_t bCurveID, uint8_t * pOptsA, uint16_t wOptsALen,
    uint8_t * pE_PubB, uint16_t wE_PubBLen, uint8_t * pCertA, uint16_t wCertALen, uint8_t * pExpRspLen,
    uint8_t bExpRspLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wBER_TLV_Len = 0;
    uint16_t    PH_MEMLOC_REM wMsgEncLen = 0;
    uint16_t    PH_MEMLOC_REM wL1 = 0;
    uint8_t     PH_MEMLOC_REM bBER_Len = 0;
    uint8_t     PH_MEMLOC_REM bIsMutualAuth = PH_OFF;
    uint8_t     PH_MEMLOC_REM bInclude_CertA = PH_OFF;

    uint8_t     PH_MEMLOC_REM aSharedSecret[64];
    uint16_t    PH_MEMLOC_REM wSharedSecretLen = 0;

    uint8_t     PH_MEMLOC_REM *pE_PubA = NULL;
    uint16_t    PH_MEMLOC_REM wE_PubALen = 0;

    uint8_t     PH_MEMLOC_REM *pSignature = NULL;
    uint16_t    PH_MEMLOC_REM wSignLen = 0;

    uint8_t     PH_MEMLOC_REM *pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Clear Buffers. */
    (void) memset(aSharedSecret, 0x00, sizeof(aSharedSecret));
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    /* Get the Status to Auth Method. */
    bIsMutualAuth = (uint8_t) ((pOptsA[2] & PHAL_MFDUOX_ISO_GENERAL_AUTH_MUTUAL) ? PH_ON : PH_OFF);

    /* Get the Status to include Certificate (Cert.A) or not. */
    bInclude_CertA = (uint8_t) !(pOptsA[2] & PHAL_MFDUOX_ISO_GENERAL_AUTH_EXCLUDE_CERTIFICATE);

    /* Compute Shared Secret (ShS = ECDH(E.Priv.A, E.Pub.B)) --------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_SharedSecret(
        pDataParams->pCryptoDataParamsASym,
        bCurveID,
        pE_PubB,
        wE_PubBLen,
        aSharedSecret,
        &wSharedSecretLen));

    /* Export Ephemeral Public Key (E.Pub.A) ------------------------------------------------------------------------------------------- */
    pE_PubA = PHAL_MFDUOX_CMD_BUF;
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_ExportKey(
        pDataParams->pCryptoDataParamsASym,
        PH_CRYPTOASYM_PUBLIC_KEY,
        PHAL_MFDUOX_PUBLIC_KEY_LEN,
        &bCurveID,
        pE_PubA,
        &wE_PubALen));

    /* Compute Session Keys (KSesAuthENC, KSesAuthMAC = KDF(ShS)) ---------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_GenerateSessionKeys_ASym(
        pDataParams,
        pE_PubA,
        wE_PubALen,
        pE_PubB,
        wE_PubBLen,
        aSharedSecret,
        wSharedSecretLen));

    /* Generate Signature (ECDSASign(Priv.A, Msg.A.ext)) ------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_SignMessage(
        pDataParams,
        wKeyNo_PrivA,
        wKeyPos_PrivA,
        pOptsA,
        wOptsALen,
        pE_PubA,
        wE_PubALen,
        pE_PubB,
        wE_PubBLen,
        &pSignature,
        &wSignLen));

    /* Exchange Cmd.ISOGeneralAuthenticate Final command ------------------------------------------------------------------------------- */

    /* Check if Host Certificate (Cert.A) should be exchanged. */
    wCertALen = (uint16_t) (bInclude_CertA ? wCertALen : 0U);

    /* Manipulate the Length information (L1). */
    wL1 = (uint16_t) (1U /* Msg.A.enc Tag */ + wCertALen + wSignLen);
    if(wL1 <= 127U)
    {
        /* Update the BER-TLV Len to 1 bytes format. */
        bBER_Len = PHAL_MFDUOX_ISO7816_BER_TLV_L_NO_CONST; /* 1 bytes of Actual Length */
    }
    else if( wL1 > 255U)
    {
        /* Update the BER-TLV Len to 2 bytes format. */
        bBER_Len = PHAL_MFDUOX_ISO7816_BER_TLV_L_82; /* Constant Length Byte + 2 bytes of Actual Length */
    }
    else
    {
        /* Update the BER-TLV Len to 1 bytes format. */
        bBER_Len = PHAL_MFDUOX_ISO7816_BER_TLV_L_81; /* Constant Length Byte + 1 bytes of Actual Length */
    }

    /* Force Extend LE information to be exchanged if greater than 255 bytes for Mutual Authentication. */
    if((wL1 > 255U) || bIsMutualAuth)
    {
        bExpRspLen = 2U;
    }

    /* Clear Buffer */
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
    PHAL_MFDUOX_CMD_BUF_LEN = 0;

    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_ISO7816_GENERIC_CLA;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_CMD_AUTHENTICATE_ISO_GENERAL;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_WRAPPEDAPDU_P1;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bCARootKeyNo;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_WRAPPEDAPDU_LC;

    /* Frame Extended LC. */
    if(bExpRspLen > 1)
    {
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_WRAPPEDAPDU_LC;
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_WRAPPEDAPDU_LC;
    }

    /* Compute Encrypted Message A Length. */
    wMsgEncLen = (uint16_t) (2U /* constant: 0xE0E0 */ + wSignLen + wCertALen);
    wMsgEncLen = (uint16_t) (PHAL_MFDUOX_IS_MULTIPLE_AES_BLOCK_SIZE(wMsgEncLen) ? (wMsgEncLen + 16U) : wMsgEncLen);
    PHAL_MFDUOX_NEAREST_MULTIPLE(wMsgEncLen, wMsgEncLen);

    /* Compute complete BER-TLV Data Object Length. */
    wBER_TLV_Len = (uint16_t)
    (
        2U              /* E.Pub.A Tag (T) and Length (L) */    +
        wE_PubALen      /* Value (V) */                         +

        1U              /* Msg.A.enc Tag (T) */                 +
        bBER_Len        /* Length (L) */                        +
        wMsgEncLen      /* Value (V) */
    );

    /* Add OptsA to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pOptsA, wOptsALen);
    PHAL_MFDUOX_CMD_BUF_LEN += wOptsALen;

    /* Add AuthDOHdr (Authentication Data Object Header) to command buffer. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_AUTH_ISO_GENERAL_AUTH_DO_HDR_TAG;
    phalMfDuoX_Int_EncodeBER_TLV_Len(wBER_TLV_Len, PHAL_MFDUOX_CMD_BUF, &PHAL_MFDUOX_CMD_BUF_LEN);

    /* Add Authentication Data Object for Ephemeral Public Key. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_AUTH_ISO_GENERAL_EPH_PUB_KEY_TAG;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) wE_PubALen;

    /* Add Ephemeral Public Key (E.Pub.A) */
    wE_PubALen = 0;
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_ExportKey(
        pDataParams->pCryptoDataParamsASym,
        PH_CRYPTOASYM_PUBLIC_KEY,
        PHAL_MFDUOX_PUBLIC_KEY_LEN,
        &bCurveID,
        &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN],
        &wE_PubALen));
    PHAL_MFDUOX_CMD_BUF_LEN += wE_PubALen;

    /* Add Authentication Data Object for Encrypted Message. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = PHAL_MFDUOX_AUTH_ISO_GENERAL_ENC_MSG_A_TAG;
    phalMfDuoX_Int_EncodeBER_TLV_Len(wMsgEncLen, PHAL_MFDUOX_CMD_BUF, &PHAL_MFDUOX_CMD_BUF_LEN);

    /* Update LC */
    phalMfDuoX_Int_UpdateLC(PHAL_MFDUOX_CMD_BUF, (uint16_t) (PHAL_MFDUOX_CMD_BUF_LEN + wMsgEncLen), PH_OFF, bExpRspLen);

    /* Buffer the information to be exchanged. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHAL_MFDUOX_CHAINING_BIT_INVALID,
        PHAL_MFDUOX_OPTION_NONE,
        (uint16_t) (PHAL_MFDUOX_CMD_BUF_LEN + wMsgEncLen),
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        NULL,
        NULL));

    /* Add Encrypted Message to be exchanged. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_EncryptMessage_Exchange(
        pDataParams,
        pSignature,
        wSignLen,
        pCertA,
        wCertALen,
        wMsgEncLen));

    /* Add LE and Exchange command to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        PHAL_MFDUOX_CHAINING_BIT_DISABLE,
        (uint8_t) (PHAL_MFDUOX_OPTION_COMPLETE | PHAL_MFDUOX_PICC_STATUS_WRAPPED),
        0,
        PH_OFF,
        pExpRspLen,
        bExpRspLen,
        &pResponse,
        &wRspLen,
        NULL));

    /* Validate AuthDOHdr from response. */
    if(pResponse[0] != PHAL_MFDUOX_AUTH_ISO_GENERAL_AUTH_DO_HDR_TAG)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);

    pResponse++;
    wRspLen--;

    /* Encode BER-TLV Length of AuthDOHdr */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_DecodeBER_TLV_Len(&pResponse, &wBER_TLV_Len, &wRspLen));

    /* Check the response length. */
    if(wRspLen != wBER_TLV_Len)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);

    /* Validate Tag information from Encrypted Message B TLV. */
    if(pResponse[0] != PHAL_MFDUOX_AUTH_ISO_GENERAL_ENC_MSG_B_TAG)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);

    pResponse++;
    wRspLen--;

    /* Encode BER-TLV Length of Encrypted Message B TLV */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Int_DecodeBER_TLV_Len(&pResponse, &wBER_TLV_Len, &wRspLen));

    /* Check the response length. */
    if(wRspLen != wBER_TLV_Len)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);

    /* Decrypt Cmd.ISOGeneralAuthenticate Part2 Response ------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_DecryptMessage(
        pDataParams,
        bIsMutualAuth,
        pResponse,
        wRspLen,
        &pResponse,
        &wRspLen));

    /* Copy Decrypted message to parameter. */
    *ppResponse = pResponse;
    *pRspLen = wRspLen;

    return wStatus;
}





phStatus_t phalMfDuoX_Sw_Int_SignMessage(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wKeyNo_PrivA, uint16_t wKeyPos_PrivA,
    uint8_t * pOptsA, uint16_t wOptsALen, uint8_t * pE_PubA, uint16_t wE_PubALen, uint8_t * pE_PubB, uint16_t wE_PubBLen,
    uint8_t ** ppSigA, uint16_t * pSigALen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aMsgACont[2U] = { 0xE0U, 0xE0U };

    /* Sign (Msg.A.ext = 0xE0E0 || OptsA || E.Pub.A || E.Pub.B) ------------------------------------------------------------------------ */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Sign(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_FIRST,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        aMsgACont,
        2U,
        NULL,
        NULL));

    /* Add OptsA for Hashing. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Sign(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_CONT,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pOptsA,
        wOptsALen,
        NULL,
        NULL));

    /* Add Ephemeral Public Key A for Hashing. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Sign(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_CONT,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pE_PubA,
        wE_PubALen,
        NULL,
        NULL));

    /* Add Ephemeral Public Key B for Hashing. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Sign(
        pDataParams->pCryptoDataParamsASym,
        (uint16_t) (PH_EXCHANGE_BUFFER_CONT),
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pE_PubB,
        wE_PubBLen,
        NULL,
        NULL));

    /* Clear Buffer and Update the Length. */
    (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Load the Known Private Key (Priv.A) to ASymmetric Crypto component. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_LoadKey(
        pDataParams->pCryptoDataParamsASym,
        PH_CRYPTOASYM_PRIVATE_KEY,
        wKeyNo_PrivA,
        wKeyPos_PrivA));

    /* Add Ephemeral Public Key B for Hashing. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Sign(
        pDataParams->pCryptoDataParamsASym,
        (uint16_t) (PH_EXCHANGE_BUFFER_LAST),
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        NULL,
        0,
        PHAL_MFDUOX_PRS_BUF,
        &PHAL_MFDUOX_PRS_BUF_LEN));

    /* Assign the Signature to the Parameter. */
    *ppSigA = PHAL_MFDUOX_PRS_BUF;
    *pSigALen = PHAL_MFDUOX_PRS_BUF_LEN;

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_EncryptMessage_Exchange(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t * pSigA, uint16_t wSigALen,
    uint8_t * pCertA, uint16_t wCertALen, uint16_t wMsgAEncLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wRemLen = 0;

    uint16_t    PH_MEMLOC_REM wCmdBufSize = 0;
    uint16_t    PH_MEMLOC_REM wCert_Offset = 0;
    uint16_t    PH_MEMLOC_REM wCopyLen = 0;

    /* Clear Command Buffer. */
    PHAL_MFDUOX_CMD_BUF_LEN = 0;
    (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);

    /* Load IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Encrypt Constant and Certificate */
    PHAL_MFDUOX_NEAREST_MULTIPLE(PHAL_MFDUOX_CMD_BUF_SIZE, wCmdBufSize);

    /* Add Constant to command buffer. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = 0xE0U;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = 0xE0U;

    /* Add Certificate to command buffer. */
    if(wCertALen != 0U)
    {
        /* Save the Certificate length. */
        wRemLen = wCertALen;

        /* Add Certificate to command buffer ------------------------------------------------------------------------------------------- */
        do
        {
            /* Compute the Length of bytes to copy from Certificate to bring up AES block size. */
            wCopyLen = (uint16_t) ((wRemLen > wCmdBufSize) ? wCmdBufSize : wRemLen);
            if(wRemLen > PH_CRYPTOSYM_AES_BLOCK_SIZE)
                PHAL_MFDUOX_PREVIOUS_MULTIPLE(wCopyLen, wCopyLen);
            wCopyLen -= PHAL_MFDUOX_CMD_BUF_LEN;

            /* Add Certificate to command buffer. */
            (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], &pCertA[wCert_Offset], wCopyLen);
            PHAL_MFDUOX_CMD_BUF_LEN += wCopyLen;

            /* Update Remaining length. */
            wRemLen -= wCopyLen;

            /* Break if the information to be copied is of AES block size. */
            if(PHAL_MFDUOX_IS_MULTIPLE_AES_BLOCK_SIZE(PHAL_MFDUOX_CMD_BUF_LEN))
            {
                /* Encrypt the Message. */
                PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                    pDataParams->pCryptoDataParamsEnc,
                    (uint16_t) (PH_EXCHANGE_BUFFER_CONT | PH_CRYPTOSYM_CIPHER_MODE_CBC),
                    PHAL_MFDUOX_CMD_BUF,
                    PHAL_MFDUOX_CMD_BUF_LEN,
                    PHAL_MFDUOX_CMD_BUF));

                /* Append the encrypted information to exchange buffer. */
                PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_CardExchange(
                    pDataParams,
                    PH_EXCHANGE_BUFFER_CONT,
                    PHAL_MFDUOX_CHAINING_BIT_DISABLE,
                    PHAL_MFDUOX_OPTION_NONE,
                    0,
                    PH_OFF,
                    PHAL_MFDUOX_CMD_BUF,
                    PHAL_MFDUOX_CMD_BUF_LEN,
                    NULL,
                    NULL,
                    NULL));

                /* Clear Command Buffer. */
                PHAL_MFDUOX_CMD_BUF_LEN = 0;
                (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);

                /* Update Certificate Offset. */
                wCert_Offset += wCopyLen;
            }
            else
            {
                wRemLen = 0U;
            }

        } while(wRemLen > 0U);
    }

    /* Add Signature to command buffer ------------------------------------------------------------------------------------------------- */

    /* Add Signature to command buffer. */
    (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pSigA, wSigALen);
    PHAL_MFDUOX_CMD_BUF_LEN += wSigALen;

    /* Apply Padding. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_ApplyPadding(
        PH_CRYPTOSYM_PADDING_MODE_2,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        wMsgAEncLen,
        PHAL_MFDUOX_CMD_BUF,
        &PHAL_MFDUOX_CMD_BUF_LEN));

    /* Encrypt Signature --------------------------------------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_EXCHANGE_BUFFER_LAST | PH_CRYPTOSYM_CIPHER_MODE_CBC),
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        PHAL_MFDUOX_CMD_BUF));

    /* Buffer Encrypted Signature for Exchange. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        PHAL_MFDUOX_CHAINING_BIT_DISABLE,
        PHAL_MFDUOX_OPTION_NONE,
        0,
        PH_OFF,
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        NULL,
        NULL,
        NULL));

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_DecryptMessage(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bIsMutualAuth, uint8_t * pResponse,
    uint16_t wRspLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Check if there is a response from PICC. */
    if(wRspLen == 0)
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDUOX);

    /* Load IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Decrypt Response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_CBC),
        pResponse,
        wRspLen,
        pResponse));

    /* Remove Padding. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_RemovePadding(
        PH_CRYPTOSYM_PADDING_MODE_2,
        pResponse,
        wRspLen,
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        wRspLen,
        pResponse,
        &wRspLen));

    /* Update Authentication State ----------------------------------------------------------------------------------------------------- */
    if(bIsMutualAuth == PH_OFF)
    {
        /* Update AuthState to Authenticated. */
        pDataParams->bAuthState = PHAL_MFDUOX_ECC_AUTHENTICATED;
        pDataParams->wKeyType = PH_CRYPTOSYM_KEY_TYPE_AES128;

        /* Set TI and Command Counter to zero. */
        (void) memset(pDataParams->aTi, 0x00, sizeof(pDataParams->aTi));
        pDataParams->wCmdCtr = 0;

        /* Clear the local buffers. */
        (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUFFER_SIZE_MINIMUM);
        PHAL_MFDUOX_CMD_BUF_LEN = 0;
        PHAL_MFDUOX_CMD_BUF_OFFSET = 0;

        (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUFFER_SIZE_MINIMUM);
        PHAL_MFDUOX_PRS_BUF_LEN = 0;
        PHAL_MFDUOX_PRS_BUF_OFFSET = 0;

        PHAL_MFDUOX_IS_PICC_DATA_COMPLETE = PH_OFF;
        PHAL_MFDUOX_HAS_MAC_PROCESSED = PH_OFF;
    }
    else
    {
        /*
         * Response Validation for Mutual Authentication is taken care by
         * phalMfDuoX_ISOGeneralAuthenticateVerify interface.
         */
    }

    /* Update the response parameter. */
    *ppResponse = pResponse;
    *pRspLen = wRspLen;

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_GenerateSessionKeys_ASym(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t * pE_PubA,
    uint16_t wE_PubALen, uint8_t * pE_PubB, uint16_t wE_PubBLen, uint8_t * pSharedSecret, uint16_t wSharedSecretLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wPubA_X_Offset = 0;
    uint16_t    PH_MEMLOC_REM wPubB_X_Offset = 0;
    uint8_t     PH_MEMLOC_REM bKDK_Len = 0;
    uint8_t     PH_MEMLOC_REM bMac_Len = 0;

    uint8_t     PH_MEMLOC_REM aKey[PH_CRYPTOSYM_AES_BLOCK_SIZE];
    uint8_t     PH_MEMLOC_REM aKDK[PH_CRYPTOSYM_AES_BLOCK_SIZE];

    /* Clear Buffers. */
    (void) memset(aKey, 0x00, sizeof(aKey));
    (void) memset(aKDK, 0x00, sizeof(aKDK));
    (void) memset(pDataParams->aSesAuthENCKey, 0x00, sizeof(pDataParams->aSesAuthENCKey));
    (void) memset(pDataParams->aSesAuthMACKey, 0x00, sizeof(pDataParams->aSesAuthMACKey));

    /* Get the Offsets for utilizing the data in Key framing. */
    wPubA_X_Offset = (uint16_t) (((wE_PubALen - 1U) / 2U) - 8U);
    wPubB_X_Offset = (uint16_t) (((wE_PubBLen - 1U) / 2U) - 8U);

    /* Frame the Key using Public Keys's X points. */
    (void) memcpy(&aKey[0], &pE_PubA[wPubA_X_Offset + 1U], 8U);
    (void) memcpy(&aKey[8], &pE_PubB[wPubB_X_Offset + 1U], 8U);

    /* Compute KDK (Key-Derivation Key) ------------------------------------------------------------------------------------------------ */
    /* Load Key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsMac,
        aKey,
        PH_CRYPTOSYM_KEY_TYPE_AES128));

    /* Load IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Compute KDK using computed Shared Secret. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        pSharedSecret,
        wSharedSecretLen,
        aKDK,
        &bKDK_Len));

    /* Load KDK Key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsMac,
        aKDK,
        PH_CRYPTOSYM_KEY_TYPE_AES128));

    /* Generate Session Auth Encryption Key -------------------------------------------------------------------------------------------- */
    pDataParams->aSesAuthENCKey[0U] = 0xB4U; pDataParams->aSesAuthENCKey[1U] = 0x4BU;
    pDataParams->aSesAuthENCKey[3U] = 0x01U; pDataParams->aSesAuthENCKey[5U] = 0x80U;

    /* Load IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Generate Session Auth Encryption Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        pDataParams->aSesAuthENCKey,
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        pDataParams->aSesAuthENCKey,
        &bMac_Len));

    /* Load Session Auth Encryption Key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsEnc,
        pDataParams->aSesAuthENCKey,
        PH_CRYPTOSYM_KEY_TYPE_AES128));

    /* Load IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Set the keep Iv ON */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_SetConfig(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CONFIG_KEEP_IV,
        PH_CRYPTOSYM_VALUE_KEEP_IV_ON));

    /* Generate Session Auth MAC Key ------------------------------------------------------------------------------------------------ */
    pDataParams->aSesAuthMACKey[0U] = 0x4BU; pDataParams->aSesAuthMACKey[1U] = 0xB4U;
    pDataParams->aSesAuthMACKey[3U] = 0x01U; pDataParams->aSesAuthMACKey[5U] = 0x80U;

    /* Load IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Generate Session Auth MAC Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        pDataParams->aSesAuthMACKey,
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        pDataParams->aSesAuthMACKey,
        &bMac_Len));

    /* Load Session Auth MAC Key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsMac,
        pDataParams->aSesAuthMACKey,
        PH_CRYPTOSYM_KEY_TYPE_AES128));

    /* Load IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_Int_GenerateSessionKeys_Sym(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wKeyType, uint8_t * pRndA,
    uint8_t * pRndB)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aSV[PH_CRYPTOSYM_AES_BLOCK_SIZE * 2U];
    uint8_t     PH_MEMLOC_REM bSvLen = 0;
    uint8_t     PH_MEMLOC_REM bMacLen = 0;
    uint8_t     PH_MEMLOC_REM bIsAES256 = PH_OFF;

    /* Check if the Key Type is AES256 */
    bIsAES256 = (uint8_t) (wKeyType == PH_KEYSTORE_KEY_TYPE_AES256);

    /* Frame Session Vector 1 information. */
    aSV[bSvLen++] = 0xA5U; aSV[bSvLen++] = 0x5AU; aSV[bSvLen++] = 0x00U; aSV[bSvLen++] = 0x01U;
    aSV[bSvLen++] = (uint8_t) (bIsAES256 ? 0x01U : 0x00U);
    aSV[bSvLen++] = (uint8_t) (bIsAES256 ? 0x00U : 0x80U);

    aSV[bSvLen++] = pRndA[0]; aSV[bSvLen++] = pRndA[1];

    aSV[bSvLen++] = pRndA[2U] ^ pRndB[0U];
    aSV[bSvLen++] = pRndA[3U] ^ pRndB[1U];
    aSV[bSvLen++] = pRndA[4U] ^ pRndB[2U];
    aSV[bSvLen++] = pRndA[5U] ^ pRndB[3U];
    aSV[bSvLen++] = pRndA[6U] ^ pRndB[4U];
    aSV[bSvLen++] = pRndA[7U] ^ pRndB[5U];

    (void) memcpy(&aSV[14U], &pRndB[6U], 10U);
    (void) memcpy(&aSV[24U], &pRndA[8U], 8U);

    bSvLen = (uint8_t) sizeof(aSV);

    /* Generate Session ENC Key --------------------------------------------------------------------------------------------------------
     * AES128
     *      SV1 = 0xA5 || 0x5A || 0x00 || 0x01 || 0x00 || 0x80 || RndA[15:14] || (RndA[13::8] XOR RndB[15::10]) || RndB[9::0] || RndA[7::0]
     *      KSesAuthENC = PRF(Kx, SV1)
     *
     * AES256
     *      SV1a = 0xA5 || 0x5A || 0x00 || 0x01 || 0x01 || 0x00 || RndA[15:14] || (RndA[13::8] XOR RndB[15::10]) || RndB[9::0] || RndA[7::0]
     *      SV1b = 0xA5 || 0x5A || 0x00 || 0x02 || 0x01 || 0x00 || RndA[15:14] || (RndA[13::8] XOR RndB[15::10]) || RndB[9::0] || RndA[7::0]
     *      KSesAuthENC = PRF(Kx, SV1a)||PRF(Kx, SV1b)
     */
    /* Load Zero IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Start CMAC calculation */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_MAC_MODE_CMAC,
        aSV,
        bSvLen,
        pDataParams->aSesAuthENCKey,
        &bMacLen));

    /* Extend the computation for AES256 KeyType */
    if(wKeyType == PH_KEYSTORE_KEY_TYPE_AES256)
    {
        /* Update the Session Vector */
        aSV[0x03U] = 0x02U;

        /* Load Zero IV */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pCryptoDataParamsEnc,
            phalMfDuoX_Sw_ZeroIv,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));

        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pCryptoDataParamsEnc,
            PH_CRYPTOSYM_MAC_MODE_CMAC,
            aSV,
            bSvLen,
            &pDataParams->aSesAuthENCKey[PH_CRYPTOSYM_AES_BLOCK_SIZE],
            &bMacLen));

        /* Revert back. */
        aSV[0x03U] = 0x01U;
    }

    /* Generate Session MAC Key --------------------------------------------------------------------------------------------------------
     * AES128
     *      SV2 = 0x5A || 0xA5 || 0x00 || 0x01 || 0x00 || 0x80 || RndA[15:14] || (RndA[13::8] XOR RndB[15::10]) || RndB[9::0] || RndA[7::0]
     *      KSesAuthMAC = PRF(Kx, SV2)
     *
     * AES256
     *      SV2a = 0x5A || 0xA5 || 0x00 || 0x01 || 0x01 || 0x00 || RndA[15:14] || (RndA[13::8] XOR RndB[15::10]) || RndB[9::0] || RndA[7::0]
     *      SV2b = 0x5A || 0xA5 || 0x00 || 0x02 || 0x01 || 0x00 || RndA[15:14] || (RndA[13::8] XOR RndB[15::10]) || RndB[9::0] || RndA[7::0]
     *      KSesAuthMAC = PRF(Kx, SV1a)||PRF(Kx, SV1b)
     */
    aSV[0U] = 0x5AU;
    aSV[1U] = 0xA5U;

    /* Load IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Start CMAC calculation */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_MAC_MODE_CMAC,
        aSV,
        bSvLen,
        pDataParams->aSesAuthMACKey,
        &bMacLen));

    /* Extend the computation for AES256 KeyType */
    if(wKeyType)
    {
        aSV[0x03U] = 0x02U;

        /* Load Zero IV */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pCryptoDataParamsEnc,
            phalMfDuoX_Sw_ZeroIv,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));

        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pCryptoDataParamsEnc,
            PH_CRYPTOSYM_MAC_MODE_CMAC,
            aSV,
            bSvLen,
            &pDataParams->aSesAuthMACKey[PH_CRYPTOSYM_AES_BLOCK_SIZE],
            &bMacLen));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */





phStatus_t phalMfDuoX_Sw_Int_ApplySM(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bIsFirstFrame, uint8_t bIsLastFrame,
    uint8_t bCommMode, uint8_t * pCmdHeader, uint16_t wCmdHeaderLen, uint8_t * pCmdData, uint16_t wCmdDataLen,
    uint8_t ** ppSMBuf, uint16_t * pSMBufLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint16_t    PH_MEMLOC_REM wOption = 0;
    uint16_t    PH_MEMLOC_REM wBlockSize = 0;
    uint16_t    PH_MEMLOC_REM wOffset = 0;
    uint16_t    PH_MEMLOC_REM wPaddedDataLen = 0;
    uint16_t    PH_MEMLOC_REM wBuffOptions = 0;
    uint16_t    PH_MEMLOC_REM wUnProcessedDataLen_ENC = 0U;
    uint8_t     PH_MEMLOC_REM bMacLen = 0;
    uint8_t     PH_MEMLOC_REM bMacProcessed = PH_OFF;
    uint8_t     PH_MEMLOC_REM bUnProcessedDataLen_MAC = 0U;

    uint8_t     PH_MEMLOC_REM aMac[PH_CRYPTOSYM_AES_BLOCK_SIZE];
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

    /* Set the Status to SUCCESS. */
    wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
    /* Perform Secure Messaging on data. */
    if(bCommMode != PHAL_MFDUOX_COMMUNICATION_PLAIN)
    {
        /* Compute Encryption IV and Frame initial MAC --------------------------------------------------------------------------------- */
        if(bIsFirstFrame)
        {
            /* Compute IV for encryption. */
            if(bCommMode == PHAL_MFDUOX_COMMUNICATION_FULL)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_GenerateIv(
                    pDataParams,
                    PH_OFF,
                    pDataParams->aTi,
                    pDataParams->wCmdCtr));
            }

            /* Load zero IV for MAC Computation. */
            PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_LoadIv(
                pDataParams->pCryptoDataParamsMac,
                phalMfDuoX_Sw_ZeroIv,
                PH_CRYPTOSYM_AES_BLOCK_SIZE));

            /* Copy Command code to processing buffer. */
            PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_LEN++] = pCmdHeader[0U];

            /* Add command counter. */
            PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_LEN++] = (uint8_t) (pDataParams->wCmdCtr);
            PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_LEN++] = (uint8_t) (pDataParams->wCmdCtr >> 8U);

            /* Copy TI information to processing buffer. */
            (void) memcpy(&PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_LEN], pDataParams->aTi, PHAL_MFDUOX_SIZE_TI);
            PHAL_MFDUOX_PRS_BUF_LEN += (uint16_t) PHAL_MFDUOX_SIZE_TI;

            /* Copy Command Header to processing buffer. */
            (void) memcpy(&PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_LEN], &pCmdHeader[1U], (wCmdHeaderLen - 1U));
            PHAL_MFDUOX_PRS_BUF_LEN += (uint16_t) (wCmdHeaderLen - 1U);

            /* Update Processing buffer offset. */
            PHAL_MFDUOX_PRS_BUF_OFFSET = PHAL_MFDUOX_PRS_BUF_LEN;

            /* Clear Command buffer length and offset */
            PHAL_MFDUOX_CMD_BUF_LEN = 0U;
            PHAL_MFDUOX_CMD_BUF_OFFSET = 0U;
        }
        else
        {
            /* Shift processing buffer for next operation. */
            if((PHAL_MFDUOX_IS_ENC_PENDING == PH_OFF) && (PHAL_MFDUOX_PRS_BUF_OFFSET > 0U))
            {
                (void) memcpy(PHAL_MFDUOX_PRS_BUF, &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_OFFSET], PHAL_MFDUOX_PRS_BUF_LEN);
                PHAL_MFDUOX_PRS_BUF_OFFSET = PHAL_MFDUOX_PRS_BUF_LEN;
            }

            /* Copy plain data to processing */
            if(PHAL_MFDUOX_CMD_BUF_LEN > 0U)
            {
                (void) memcpy(&PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_OFFSET], PHAL_MFDUOX_CMD_BUF,
                    PHAL_MFDUOX_CMD_BUF_LEN);

                /* Update Processing buffer length. */
                PHAL_MFDUOX_PRS_BUF_LEN += PHAL_MFDUOX_CMD_BUF_LEN;
                PHAL_MFDUOX_PRS_BUF_OFFSET += PHAL_MFDUOX_CMD_BUF_LEN;

                /* Set unprocessed plain data. */
                wUnProcessedDataLen_ENC = PHAL_MFDUOX_CMD_BUF_LEN;

                /*Clear command buffer length and offset.*/
                PHAL_MFDUOX_CMD_BUF_LEN = 0U;
                PHAL_MFDUOX_CMD_BUF_OFFSET = 0U;
            }

            /* Update Processing buffer offset */
            PHAL_MFDUOX_PRS_BUF_OFFSET = PHAL_MFDUOX_PRS_BUF_LEN;

            /* Clear Processing buffer Length */
            PHAL_MFDUOX_PRS_BUF_LEN = wUnProcessedDataLen_ENC;
        }

        /* Save the starting offset of data to be returned. */
        wOffset = (uint16_t) (PHAL_MFDUOX_PRS_BUF_OFFSET - wUnProcessedDataLen_ENC);

        /* Copy command data to processing buffer. */
        (void) memcpy(&PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_OFFSET], pCmdData, wCmdDataLen);

        /* Set length of bytes to be encrypted. */
        wUnProcessedDataLen_ENC += wCmdDataLen;

        /* Update Processing buffer length and offset. */
        PHAL_MFDUOX_PRS_BUF_LEN += (uint16_t) (wCmdDataLen + (bIsFirstFrame ? 0U : wOffset /* UnProcessed MAC */));
        PHAL_MFDUOX_PRS_BUF_OFFSET += wCmdDataLen;

        /* Encrypt the Data */
        if(bCommMode == PHAL_MFDUOX_COMMUNICATION_FULL)
        {
            /* Apply padding. */
            if(bIsLastFrame)
            {
                PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_ApplyPadding(
                    PH_CRYPTOSYM_PADDING_MODE_2,
                    &PHAL_MFDUOX_PRS_BUF[wOffset],
                    wUnProcessedDataLen_ENC,
                    PH_CRYPTOSYM_AES_BLOCK_SIZE,
                    PHAL_MFDUOX_PRS_BUF_SIZE,
                    &PHAL_MFDUOX_PRS_BUF[wOffset],
                    &wPaddedDataLen));

                /* Update Processing buffer length to including padding bytes. */
                PHAL_MFDUOX_PRS_BUF_LEN += (uint16_t) (wPaddedDataLen - wUnProcessedDataLen_ENC);
                PHAL_MFDUOX_PRS_BUF_OFFSET = PHAL_MFDUOX_PRS_BUF_LEN;

                /* Update un processed encrypted data length. */
                wUnProcessedDataLen_ENC = wPaddedDataLen;
            }

            /* Set the buffering option and Cipher mode .*/
            wBuffOptions = (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC |
                (bIsLastFrame ? PH_EXCHANGE_BUFFER_LAST : PH_EXCHANGE_BUFFER_CONT));

            /* Get the Block size to encrypt. */
            PHAL_MFDUOX_PREVIOUS_MULTIPLE(wUnProcessedDataLen_ENC, wBlockSize);
            wBlockSize = (uint16_t) (bIsLastFrame ? wUnProcessedDataLen_ENC : wBlockSize);

            /* Encrypt the CmdData information. */
            if(wBlockSize != 0U)
            {
                PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_Encrypt(
                    pDataParams->pCryptoDataParamsEnc,
                    wBuffOptions,
                    &PHAL_MFDUOX_PRS_BUF[wOffset],
                    wBlockSize,
                    &PHAL_MFDUOX_PRS_BUF[wOffset]));

                PHAL_MFDUOX_IS_ENC_PENDING = PH_OFF;

                /* Copy un processed data to command buffer. */
                if((wUnProcessedDataLen_ENC - wBlockSize) > 0U)
                {
                    (void) memcpy(PHAL_MFDUOX_CMD_BUF, &PHAL_MFDUOX_PRS_BUF[wOffset + wBlockSize],
                        (wUnProcessedDataLen_ENC - wBlockSize));
                    PHAL_MFDUOX_CMD_BUF_LEN = (uint16_t) (wUnProcessedDataLen_ENC - wBlockSize);

                    /* Revert back processing buffer offset and length. */
                    PHAL_MFDUOX_PRS_BUF_LEN -= (uint16_t) (wUnProcessedDataLen_ENC - wBlockSize);
                    PHAL_MFDUOX_PRS_BUF_OFFSET -= (uint16_t) (wUnProcessedDataLen_ENC - wBlockSize);
                }
            }
            else
            {
                /* Copy plain data to command buffer. */
                (void) memcpy(PHAL_MFDUOX_CMD_BUF, &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_LEN - wUnProcessedDataLen_ENC],
                    wUnProcessedDataLen_ENC);
                PHAL_MFDUOX_CMD_BUF_LEN = wUnProcessedDataLen_ENC;

                /* Revert back processing buffer offset and length. */
                PHAL_MFDUOX_PRS_BUF_LEN -= wUnProcessedDataLen_ENC;
                PHAL_MFDUOX_PRS_BUF_OFFSET -= wUnProcessedDataLen_ENC;

                PHAL_MFDUOX_IS_ENC_PENDING = PH_ON;
            }
        }

        /* Compute size of information to be MACed. */
        PHAL_MFDUOX_PREVIOUS_MULTIPLE(PHAL_MFDUOX_PRS_BUF_LEN, wBlockSize);
        wBlockSize = (uint16_t) (bIsLastFrame ? PHAL_MFDUOX_PRS_BUF_LEN : wBlockSize);

        /* Compute MAC. */
        if(((wBlockSize != 0U) || (bIsLastFrame == PH_ON)) && (PHAL_MFDUOX_IS_ENC_PENDING == PH_OFF))
        {
            /* Update Buffer Options */
            wOption = (uint16_t) (bIsLastFrame ? PH_EXCHANGE_BUFFER_LAST : PH_EXCHANGE_BUFFER_CONT);
            wOption = (uint16_t) (wOption | PH_CRYPTOSYM_MAC_MODE_CMAC);

            PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_CalculateMac(
                pDataParams->pCryptoDataParamsMac,
                wOption,
                PHAL_MFDUOX_PRS_BUF,
                wBlockSize,
                aMac,
                &bMacLen));

            /* Compute unprocessed MAC */
            bUnProcessedDataLen_MAC = (uint8_t) (PHAL_MFDUOX_PRS_BUF_LEN - wBlockSize);

            /* Set MAC processed flag. */
            bMacProcessed = PH_ON;
        }
        else
        {
            /* Set unprocessed MAC */
            bUnProcessedDataLen_MAC = (uint8_t) PHAL_MFDUOX_PRS_BUF_LEN;

            /* Set MAC processed flag. */
            bMacProcessed = PH_OFF;
        }

        /* Add final data for MAC computation and get 16 bytes of MAC */
        if(bIsLastFrame)
        {
            /* Truncate MAC. */
            phalMfDuoX_Sw_Int_TruncateMac(aMac);

            /* Copy Truncated MAC to Processing buffer. */
            (void) memcpy(&PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_OFFSET], aMac, PHAL_MFDUOX_TRUNCATED_MAC_LEN);
            PHAL_MFDUOX_PRS_BUF_LEN += PHAL_MFDUOX_TRUNCATED_MAC_LEN;

            /* Update Status */
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
        }

        /* Copy the SM information to be exchanged. */
        *ppSMBuf = &PHAL_MFDUOX_PRS_BUF[wOffset];
        *pSMBufLen = (uint16_t) (PHAL_MFDUOX_PRS_BUF_LEN - wOffset);

        /* Update Offset */
        wOffset = (uint16_t) (bMacProcessed ? wBlockSize : 0U);

        if(PHAL_MFDUOX_IS_ENC_PENDING == PH_OFF)
        {
            /* Update processing buffer offset for next operation. */
            PHAL_MFDUOX_PRS_BUF_OFFSET = (uint16_t) (wOffset - (wBlockSize - wOffset));

            /* Update processing buffer length with unprocessed MAC bytes. */
            PHAL_MFDUOX_PRS_BUF_LEN = bUnProcessedDataLen_MAC;
        }
    }
    else
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */
    {
        /* Mark the parameters taht are not used. */
        PH_UNUSED_VARIABLE(pDataParams);
        PH_UNUSED_VARIABLE(bIsFirstFrame);
        PH_UNUSED_VARIABLE(bIsLastFrame);
        PH_UNUSED_VARIABLE(bCommMode);
        PH_UNUSED_VARIABLE(pCmdHeader);
        PH_UNUSED_VARIABLE(wCmdHeaderLen);

        *ppSMBuf = pCmdData;
        *pSMBufLen = wCmdDataLen;
    }

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_RemoveSM(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bIsISOChained, uint8_t bIsFirstFrame,
    uint8_t bIsLastFrame, uint8_t bCommMode, uint8_t * pResponse, uint16_t wRespLen, uint8_t bPiccStat, uint8_t ** ppOutBuffer,
    uint16_t * pOutBufLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wRemData = 0;
    uint16_t    PH_MEMLOC_REM wBuffOption = 0;
    uint16_t    PH_MEMLOC_REM wAESBlockSize = 0;
    uint16_t    PH_MEMLOC_REM wDataLen = 0;
    uint8_t     PH_MEMLOC_REM bHasMoreData = PH_OFF;
    uint8_t     PH_MEMLOC_REM bDecryptData = PH_ON;

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint16_t    PH_MEMLOC_REM wOffset = 0;
    uint16_t    PH_MEMLOC_REM wBuffEncLen = 0;
    uint16_t    PH_MEMLOC_REM wBuffMacLen = 0;
    uint8_t     PH_MEMLOC_REM bFinished = PH_ON;
    uint8_t     PH_MEMLOC_REM bMacLen = 0;

    uint8_t     PH_MEMLOC_REM aMac[PH_CRYPTOSYM_AES_BLOCK_SIZE];
    uint8_t     PH_MEMLOC_REM * pBufferEnc = NULL;
    uint8_t     PH_MEMLOC_REM * pBufferMac = NULL;
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

    /* Compute DataLen excluding MAC information. */
    wDataLen = (uint16_t) (wRespLen ? (wRespLen - (bIsLastFrame ? 8U /* MAC excluded */ : 0U)) : 0U);
    wDataLen = (uint16_t) ((bCommMode != PHAL_MFDUOX_COMMUNICATION_PLAIN) ? wDataLen : wRespLen);

    if(bIsFirstFrame)
    {
        /* Clear Buffers. */
        (void) memset(PHAL_MFDUOX_CMD_BUF, 0x00, PHAL_MFDUOX_CMD_BUF_SIZE);
        PHAL_MFDUOX_CMD_BUF_LEN = 0;

        (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
        PHAL_MFDUOX_PRS_BUF_LEN = 0;

        /* Increment the command counter. */
        if(pDataParams->bAuthState != PHAL_MFDUOX_NOT_AUTHENTICATED)
            pDataParams->wCmdCtr++;

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
        if(bCommMode != PHAL_MFDUOX_COMMUNICATION_PLAIN)
        {
            /* Add PICC Response Code and Command Counter information. */
            PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bPiccStat;
            PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) (pDataParams->wCmdCtr);
            PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) (pDataParams->wCmdCtr >> 8U);

            /* Add TI (Transaction Identifier) information. */
            (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pDataParams->aTi, PHAL_MFDUOX_SIZE_TI);
            PHAL_MFDUOX_CMD_BUF_LEN += (uint16_t) PHAL_MFDUOX_SIZE_TI;

            /* Compute the Remaining bytes to be filled for obtaining AES block size. */
            wAESBlockSize = PHAL_MFCC_AES_BLOCK_SIZE_DIFF(PHAL_MFDUOX_CMD_BUF_LEN);

            /* Check if data higher than AES block size is available. */
            bHasMoreData = (wDataLen > wAESBlockSize);
            bHasMoreData = bIsISOChained;
            bHasMoreData = (wDataLen > (PHAL_MFDUOX_CMD_BUF_SIZE - PHAL_MFDUOX_CMD_BUF_LEN));

            /* Update Index information. */
            wRemData = (uint16_t) (bHasMoreData ? wAESBlockSize : wDataLen);

            /* Copy the initial information from Response Data to fill Command buffer upto Single AES block Size. */
            (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pResponse, wRemData);
            PHAL_MFDUOX_CMD_BUF_LEN += (uint16_t) wRemData;

            /* Set the offset to pick the data. */
            wOffset = (uint16_t) (bHasMoreData ? wRemData : 0);

            /* Clear the flag that in case of ISOChainned exchanged. */
            bHasMoreData = (uint8_t) (bIsISOChained ? PH_OFF : bHasMoreData);

            /* Load Zero IV for Macing. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
                pDataParams->pCryptoDataParamsMac,
                phalMfDuoX_Sw_ZeroIv,
                PH_CRYPTOSYM_AES_BLOCK_SIZE));

            /* Compute and Load IV for Decryption. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_GenerateIv(
                pDataParams,
                PH_ON,
                pDataParams->aTi,
                pDataParams->wCmdCtr));
        }
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */
    }

    /* Performs the below code operation for,
     *  - DataManagement commands with Native Chaining where PICC data do not fit in one single frame.
     *  - All the commands where No Chaining is involved. The data fits in one complete PICC frame.
     *  - For commands that involve ISO/IEC 14443-4 Chaining, whether the frame fits in one PICC frame
     *    or chains into multiple PICC frames,
     *      - Processing of data is performed in if condition where processing for last frame
     *        of data is performed (Refer the if Statement if(bIsLastFrame...)).
     */
    if(bIsISOChained == PH_OFF)
    {
        /* Copy data from command buffer to processing buffer.
         * The below procedure is required in case if there is more PICC data that
         * could not fit into Processing buffer. In the next call the previously
         * remained PICC data needs to be copied to processing buffer.
         */
        if(PHAL_MFDUOX_CMD_BUF_OFFSET)
        {
            /* Compute the remaining data to be copied form offset. */
            wRemData = (uint16_t) (PHAL_MFDUOX_CMD_BUF_LEN - PHAL_MFDUOX_CMD_BUF_OFFSET);

            /* Clear Buffers. */
            (void) memset(PHAL_MFDUOX_PRS_BUF, 0x00, PHAL_MFDUOX_PRS_BUF_SIZE);
            PHAL_MFDUOX_PRS_BUF_LEN = 0;

            /* Compute the data. */
            (void) memcpy(PHAL_MFDUOX_PRS_BUF, &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_OFFSET], wRemData);
            PHAL_MFDUOX_PRS_BUF_LEN = wRemData;

            /* Reset Offset and Remaining Data. */
            wRemData = 0;
            PHAL_MFDUOX_CMD_BUF_OFFSET = 0;

            /* Set Remaining data decryption if its last frame. */
            if(bIsLastFrame)
                bDecryptData = PH_ON;
        }

        /* Buffer response information to command buffer for macing. */
        if((PHAL_MFDUOX_CMD_BUF_LEN + wDataLen) < PHAL_MFDUOX_CMD_BUF_SIZE)
        {
            (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], &pResponse[wRemData], (wDataLen - wRemData));
            PHAL_MFDUOX_CMD_BUF_LEN += (uint16_t) (wDataLen - wRemData);

            bHasMoreData = PH_ON;
        }

        /* Perform MACing of data based on the command buffer length and size.
         * Here if the command buffer is unable to copy the newly received PICC data, MACing of data based on AES
         * block size is first performed.
         * Then the remaining data that is not processed for macing is shifted to beginning of the command buffer
         * and then newly received PICC data is copied.
         * Additional, if there is room to allocate PICC data to processing buffer, required amount of PICC data
         * is copied to processing buffer. The remaining PICC data that is not copied will be taken from command
         * buffer for the next call.
         */
        else
        {
            /* Frame Buffering Option. */
            wBuffOption = (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_BUFFER_CONT);

            /* Perform MAC for Multiple of AES block size. */
            PHAL_MFDUOX_PREVIOUS_MULTIPLE(PHAL_MFDUOX_CMD_BUF_LEN, wAESBlockSize);
#ifdef NXPBUILD__PHAL_MFDUOX_NDA
            if(bCommMode != PHAL_MFDUOX_COMMUNICATION_PLAIN)
            {
                /* Compute MAC if command buffer is multiple of AES block size. */
                PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
                    pDataParams->pCryptoDataParamsMac,
                    wBuffOption,
                    PHAL_MFDUOX_CMD_BUF,
                    wAESBlockSize,
                    aMac,
                    &bMacLen));
            }
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

            /* Shift the remaining information. */
            if(PHAL_MFDUOX_IS_NOT_MULTIPLE_AES_BLOCK_SIZE(PHAL_MFDUOX_CMD_BUF_LEN))
            {
                /* Compute Remaining data to be shifted. */
                wRemData = (uint16_t) (PHAL_MFDUOX_CMD_BUF_LEN - wAESBlockSize);

                /* Shift the remaining data. */
                (void) memcpy(PHAL_MFDUOX_CMD_BUF, &PHAL_MFDUOX_CMD_BUF[wAESBlockSize], wRemData);
                PHAL_MFDUOX_CMD_BUF_LEN = wRemData;

                /* Reset command buffer. */
                (void) memset(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], 0x00, (PHAL_MFDUOX_CMD_BUF_SIZE - PHAL_MFDUOX_CMD_BUF_LEN));

                /* Backup the offset from command buffer. */
                PHAL_MFDUOX_CMD_BUF_OFFSET = wRemData;

                /* Copy the data. */
                (void) memcpy(&PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN], pResponse, wDataLen);
                PHAL_MFDUOX_CMD_BUF_LEN += wDataLen;

                /* Compute Remaining data to be copied. */
                wRemData = (uint16_t) (PHAL_MFDUOX_PRS_BUF_SIZE - PHAL_MFDUOX_PRS_BUF_LEN);
                wRemData = (uint16_t) ((wRemData < wDataLen) ? wRemData : wDataLen);
                wRemData = (uint16_t) ((wRemData > PHAL_MFDUOX_CMD_BUF_LEN) ? (PHAL_MFDUOX_CMD_BUF_LEN - PHAL_MFDUOX_CMD_BUF_OFFSET) : wRemData);

                /* Copy the data. */
                (void) memcpy(&PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_LEN], &PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_OFFSET], wRemData);
                PHAL_MFDUOX_PRS_BUF_LEN += wRemData;

                /* Update Offset. */
                PHAL_MFDUOX_CMD_BUF_OFFSET = (uint16_t) (PHAL_MFDUOX_CMD_BUF_OFFSET + wRemData);

                /* Reset command buffer offset. */
                if(PHAL_MFDUOX_CMD_BUF_OFFSET == PHAL_MFDUOX_CMD_BUF_LEN)
                {
                    /* Check if more data needs to be copied to processing buffer. */
                    bHasMoreData = PH_OFF;
                    PHAL_MFDUOX_CMD_BUF_OFFSET = 0;
                    PHAL_MFDUOX_PRS_BUF_OFFSET = PHAL_MFDUOX_CMD_BUF_LEN;
                }
                else
                    bHasMoreData = PH_ON;

                wRemData = 0;
            }
            else
            {
                /* Reset command buffer length. */
                PHAL_MFDUOX_CMD_BUF_LEN = 0;
            }
        }

        /* Add PICC data to processing buffer only if required. */
        if(bHasMoreData)
        {
            /* Move the data to processing buffer. */
            if((PHAL_MFDUOX_PRS_BUF_LEN + wDataLen) < PHAL_MFDUOX_PRS_BUF_SIZE)
            {
                (void) memcpy(&PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_LEN], pResponse, wDataLen);
                PHAL_MFDUOX_PRS_BUF_LEN += (uint16_t) wDataLen;

                /* Reset Offset. */
                PHAL_MFDUOX_CMD_BUF_OFFSET = 0;
            }
            else
            {
                /* Compute Remaining data to be copied. */
                wRemData = (uint16_t) (PHAL_MFDUOX_PRS_BUF_SIZE - PHAL_MFDUOX_PRS_BUF_LEN);

                /* Copy the remaining data. */
                (void) memcpy(&PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_LEN], pResponse, wRemData);
                PHAL_MFDUOX_PRS_BUF_LEN += (uint16_t) wRemData;

                /* Backup the offset from command buffer. */
                if(PHAL_MFDUOX_PRS_BUF_OFFSET)
                {
                    PHAL_MFDUOX_CMD_BUF_OFFSET = (uint16_t) (wRemData ? (PHAL_MFDUOX_CMD_BUF_LEN - (wDataLen - wRemData))
                        : PHAL_MFDUOX_CMD_BUF_OFFSET);
                    PHAL_MFDUOX_PRS_BUF_OFFSET = 0;
                }
                else
                    PHAL_MFDUOX_CMD_BUF_OFFSET += (uint16_t) (wRemData + PHAL_MFDUOX_PRS_BUF_OFFSET);

                /* Update Has More Flag.*/
                bHasMoreData = (uint8_t) ((wDataLen - wRemData) ? PH_ON : PH_OFF);
#ifdef NXPBUILD__PHAL_MFDUOX_NDA
                /* Decrypt the data. */
                if(bCommMode == PHAL_MFDUOX_COMMUNICATION_FULL)
                {
                    /* Frame Buffering Option. */
                    wBuffOption = (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_BUFFER_CONT);

                    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
                        pDataParams->pCryptoDataParamsEnc,
                        wBuffOption,
                        PHAL_MFDUOX_PRS_BUF,
                        PHAL_MFDUOX_PRS_BUF_LEN,
                        PHAL_MFDUOX_PRS_BUF));

                    /* Remove Padding. */
                    if(!bHasMoreData && bIsLastFrame)
                        PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_RemovePadding(
                            PH_CRYPTOSYM_PADDING_MODE_2,
                            PHAL_MFDUOX_PRS_BUF,
                            PHAL_MFDUOX_PRS_BUF_LEN,
                            PH_CRYPTOSYM_AES_BLOCK_SIZE,
                            PHAL_MFDUOX_PRS_BUF_LEN,
                            PHAL_MFDUOX_PRS_BUF,
                            &PHAL_MFDUOX_PRS_BUF_LEN));

                    /* Set if Decryption for remaining data needs to be performed in this call. */
                    if(bIsLastFrame)
                        bDecryptData = PH_OFF;
                }
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

                /* Update status for Chaining. */
                wStatus = PH_ADD_COMPCODE((uint16_t) (bHasMoreData ? PH_ERR_SUCCESS_CHAINING : PH_ERR_SUCCESS), PH_COMP_AL_MFDUOX);
                wStatus = (uint16_t) (((bPiccStat == PHAL_MFDUOX_RESP_OPERATION_OK) && bIsLastFrame && !bHasMoreData) ?
                    PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX) : wStatus);
            }
        }
    }

    /* Compute MAC and verify. */
    if(bIsLastFrame || bIsISOChained)
    {
#ifdef NXPBUILD__PHAL_MFDUOX_NDA
        if(bCommMode != PHAL_MFDUOX_COMMUNICATION_PLAIN)
        {
            /* Get the Buffer to use. */
            pBufferMac = PHAL_MFDUOX_CMD_BUF;
            wBuffMacLen = PHAL_MFDUOX_CMD_BUF_LEN;

            /* Set the Remaining Data  */
            wRemData = (uint16_t) ((wDataLen == wRemData) ? 0 : wDataLen);
            wRemData = (uint16_t) ((wRemData > wOffset) ? (wRemData - wOffset) : wRemData);

            do
            {
                /* Compute and Verify MAC if still pending. */
                if(PHAL_MFDUOX_HAS_MAC_PROCESSED == PH_OFF)
                {
                    /* Frame Buffering Option. */
                    wBuffOption = (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_BUFFER_LAST);

                    /* Update Buffering state. */
                    if(bIsISOChained)
                    {
                        wBuffOption = (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | (((wRemData <= (PHAL_MFDUOX_CMD_BUF_SIZE - wBuffMacLen)) &&
                            bIsLastFrame) ? PH_EXCHANGE_BUFFER_LAST : PH_EXCHANGE_BUFFER_CONT));

                        /* Copy Remaining data to command buffer to fill up AES block Size. */
                        if(!PHAL_MFDUOX_IS_MULTIPLE_AES_BLOCK_SIZE(wBuffMacLen))
                        {
                            /* Compute the Remaining bytes to be filled for obtaining one AES block size. */
                            wAESBlockSize = PHAL_MFCC_AES_BLOCK_SIZE_DIFF(wBuffMacLen);
                            wAESBlockSize = (uint16_t) ((wRemData < (PHAL_MFDUOX_CMD_BUF_SIZE - wBuffMacLen)) ? wRemData : wAESBlockSize);
                            if(wBuffOption != PH_EXCHANGE_BUFFER_LAST)
                                wAESBlockSize = PHAL_MFCC_AES_BLOCK_SIZE_DIFF(wBuffMacLen);

                            /* Copy Remaining data to command buffer. */
                            (void) memcpy(&pBufferMac[wBuffMacLen], &pResponse[wOffset], wAESBlockSize);
                            wBuffMacLen += wAESBlockSize;

                            /* Update Remaining data and offset. */
                            wRemData -= wAESBlockSize;
                            wOffset += wAESBlockSize;
                        }

                        /* Check if More data is available for Macing. */
                        bFinished = (uint8_t) (wRemData < PH_CRYPTOSYM_AES_BLOCK_SIZE);
                        bHasMoreData = bFinished;

                        /* Update Finished Settings. */
                        bFinished = (uint8_t) ((wRemData == 0) && bIsLastFrame);

                        /* Update Buffering option. */
                        wBuffOption = (uint16_t) (wRemData ? PH_EXCHANGE_BUFFER_CONT : wBuffOption);
                    }

                    /* Compute the Final MAC */
                    PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_CalculateMac(
                        pDataParams->pCryptoDataParamsMac,
                        wBuffOption,
                        pBufferMac,
                        wBuffMacLen,
                        aMac,
                        &bMacLen));

                    /* Reset command buffer length and Offset. */
                    if(!bHasMoreData)
                    {
                        PHAL_MFDUOX_CMD_BUF_LEN = 0;
                        PHAL_MFDUOX_CMD_BUF_OFFSET = 0;
                    }

                    /* Move the next set of data for MAC calculation. */
                    if(bIsISOChained)
                    {
                        if(!bFinished)
                        {
                            /* Get AES block size length from remaining data length. */
                            PHAL_MFDUOX_PREVIOUS_MULTIPLE(wRemData, wAESBlockSize);
                            wAESBlockSize = (uint16_t) (bIsLastFrame ? wRemData : wAESBlockSize);
                            wRemData -= wAESBlockSize;

                            /* Copy the data that needs to be maced. */
                            pBufferMac = &pResponse[wOffset];
                            wBuffMacLen = wAESBlockSize;

                            /* Update Offset */
                            wOffset += wAESBlockSize;
                        }

                        /* Copy Remaining data to Command buffer length. */
                        else
                        {
                            (void) memcpy(PHAL_MFDUOX_CMD_BUF, &pResponse[wOffset], wRemData);
                            PHAL_MFDUOX_CMD_BUF_LEN = wRemData;
                        }
                    }

                    /* Verify MAC if its the last frame. */
                    if((bIsLastFrame == PH_ON) && (bFinished == PH_ON))
                    {
                        /* Truncate MAC. */
                        phalMfDuoX_Sw_Int_TruncateMac(aMac);

                        /* Verify the MAC. */
                        if(memcmp(&pResponse[wRespLen - PHAL_MFDUOX_TRUNCATED_MAC_LEN], aMac, PHAL_MFDUOX_TRUNCATED_MAC_LEN) != 0)
                        {
                            phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
                            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_AL_MFDUOX);
                        }
                        else
                            PHAL_MFDUOX_HAS_MAC_PROCESSED = PH_ON;
                    }
                }
            } while(!bFinished);

            /* Decrypt the data. */
            if(bCommMode == PHAL_MFDUOX_COMMUNICATION_FULL)
            {
                /* Perform Decryption of data only if the processing buffer is not holding data based on AES block size. */
                if(bDecryptData == PH_ON)
                {
                    /* Get the Actual Buffer that needs to be decrypted. */
                    pBufferEnc = bIsISOChained ? pResponse : PHAL_MFDUOX_PRS_BUF;
                    wBuffEncLen = bIsISOChained ? wDataLen : PHAL_MFDUOX_PRS_BUF_LEN;

                    /* Frame Buffering Option. */
                    wBuffOption = (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | (bIsLastFrame ? PH_EXCHANGE_BUFFER_LAST
                        : PH_EXCHANGE_BUFFER_CONT));

                    /* Decrypt the information. */
                    PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_Decrypt(
                        pDataParams->pCryptoDataParamsEnc,
                        wBuffOption,
                        pBufferEnc,
                        wBuffEncLen,
                        pBufferEnc));

                    /* Remove Padding. */
                    if(bIsLastFrame)
                        PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_RemovePadding(
                            PH_CRYPTOSYM_PADDING_MODE_2,
                            pBufferEnc,
                            wBuffEncLen,
                            PH_CRYPTOSYM_AES_BLOCK_SIZE,
                            wBuffEncLen,
                            pBufferEnc,
                            &wBuffEncLen));

                    /* Copy back the decrypted data to respective buffers. */
                    if(bIsISOChained == PH_ON)
                    {
                        (void) memcpy(pResponse, pBufferEnc, wBuffEncLen);
                        wDataLen = wBuffEncLen;
                    }
                    else
                    {
                        (void) memcpy(PHAL_MFDUOX_PRS_BUF, pBufferEnc, wBuffEncLen);
                        PHAL_MFDUOX_PRS_BUF_LEN = wBuffEncLen;
                    }
                }
            }
        }
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */
    }

    if(bIsLastFrame || ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING))
    {
        /* Copy the data to response parameter. */
        if(pOutBufLen != NULL)
        {
            *ppOutBuffer = bIsISOChained ? pResponse : PHAL_MFDUOX_PRS_BUF;
            *pOutBufLen = (uint16_t) (bIsISOChained ? wDataLen : PHAL_MFDUOX_PRS_BUF_LEN);
        }
    }
    else
    {
        if(bIsISOChained == PH_ON)
        {
            *ppOutBuffer = pResponse;
            *pOutBufLen = wDataLen;
        }
    }

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_ReadData(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bIsISOChained,
    uint8_t bCmd_ComMode, uint8_t bResp_ComMode, uint8_t * pCmdHeader, uint16_t wCmdHeaderLen, uint32_t dwDataToRead,
    uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus_PICC = 0;
    phStatus_t  PH_MEMLOC_REM wStatus_SM = 0;
    uint32_t    PH_MEMLOC_REM dwBlockSize = 0;
    uint32_t    PH_MEMLOC_REM dwTotalReadLen = 0;
    uint16_t    PH_MEMLOC_REM wBuffOption = PH_EXCHANGE_BUFFER_LAST;
    uint16_t    PH_MEMLOC_REM wTotalFrameLen = 0;
    uint16_t    PH_MEMLOC_REM wSMBufLen = 0;
    uint16_t    PH_MEMLOC_REM wRspLen_PICC = 0;
    uint16_t    PH_MEMLOC_REM wRspLen_SM = 0;
    uint8_t     PH_MEMLOC_REM bHasError = PH_OFF;
    uint8_t     PH_MEMLOC_REM bCmdOptions = 0;
    uint8_t     PH_MEMLOC_REM bHasChainedRsp = PH_OFF;
    uint8_t     PH_MEMLOC_REM bFirstFrame = PH_ON;
    uint8_t     PH_MEMLOC_REM bLastFrame = PH_OFF;
    uint8_t     PH_MEMLOC_REM bFinished = PH_OFF;
    uint8_t     PH_MEMLOC_REM bPiccErrCode = 0;
    uint8_t     PH_MEMLOC_REM bShortLenApdu = PH_OFF;
    uint8_t     PH_MEMLOC_REM bChainnedCmd = PHAL_MFDUOX_ADDITIONAL_FRAME;

    uint8_t *   PH_MEMLOC_REM pCmdHeader_Tmp = NULL;
    uint8_t *   PH_MEMLOC_REM pSMBuf = NULL;
    uint8_t *   PH_MEMLOC_REM pResponse_PICC = NULL;
    uint8_t *   PH_MEMLOC_REM pResponse_SM = NULL;

    if((bResp_ComMode != PHAL_MFDUOX_COMMUNICATION_PLAIN) && (pDataParams->bAuthState == PHAL_MFDUOX_NOT_AUTHENTICATED))
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_AL_MFDUOX);
    }

    /* Move the Command Header to local variable. */
    pCmdHeader_Tmp = pCmdHeader;

    /* Reset Processing Buffer Length. */
    PHAL_MFDUOX_PRS_BUF_LEN = 0;

    /* Frame options for PICC Exchange. */
    bCmdOptions = (uint8_t) (PHAL_MFDUOX_OPTION_COMPLETE | PHAL_MFDUOX_RETURN_PICC_STATUS);

    /* Compute Total frame information length.
     * This includes Command Header + Secure Messaging (Based on Communication modes)
     */
    if((wOption & 0xFF0F) == PH_EXCHANGE_DEFAULT)
    {
#ifdef NXPBUILD__PHAL_MFDUOX_NDA
        wTotalFrameLen = (uint16_t) (wCmdHeaderLen + ((bCmd_ComMode == PHAL_MFDUOX_COMMUNICATION_MAC) ? 8U : 0U));
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */
#ifndef NXPBUILD__PHAL_MFDUOX_NDA
        wTotalFrameLen = (uint16_t) wCmdHeaderLen;
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

        /* Set Extended Length APDU format in case of ISO Chaining and data size is greater than frame size. */
        if(pDataParams->bWrappedMode)
        {
            /* Back up current state of ShortLenAPDU information. */
            bShortLenApdu = pDataParams->bShortLenApdu;

            /* Manipulate for Read commands. */
            switch(pDataParams->bCmdCode)
            {
                case PHAL_MFDUOX_CMD_READ_DATA_NATIVE:
                case PHAL_MFDUOX_CMD_READ_DATA_ISO:
                case PHAL_MFDUOX_CMD_READ_RECORD_NATIVE:
                case PHAL_MFDUOX_CMD_READ_RECORD_ISO:
                    /* Compute Total Read Length */
                    PHAL_MFDUOX_NEAREST_MULTIPLE(dwDataToRead, dwBlockSize);
                    dwBlockSize = (uint16_t) ((dwDataToRead == dwBlockSize) ? (dwBlockSize + PH_CRYPTOSYM_AES_BLOCK_SIZE /* Padding */) : dwBlockSize);
#ifdef NXPBUILD__PHAL_MFDUOX_NDA
                    dwTotalReadLen = (uint16_t) ((bResp_ComMode == PHAL_MFDUOX_COMMUNICATION_PLAIN) ? dwDataToRead :
                        (((bResp_ComMode == PHAL_MFDUOX_COMMUNICATION_MAC) ? dwDataToRead : dwBlockSize) + 8U /* MAC */));
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */
#ifndef NXPBUILD__PHAL_MFDUOX_NDA
                    dwTotalReadLen = (uint16_t) dwDataToRead;
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

                    /* Set the format of data to be sent as short APDU when,
                     * 1. Bit[1] of bIsISOChained is set. This means user is force sending the data in short APDU format in case of BIGISO read.
                     * 2. In case data to be read is not BIGISO (Less than 256 bytes).
                     */
                    if((bIsISOChained == PHAL_MFDUOX_CHAINING_ISO_SHORT_LEN) ||
                        ((dwTotalReadLen <= 0xFFU) && (dwDataToRead != 0)))
                    {
                        /* Enable Short Length APDU. */
                        pDataParams->bShortLenApdu = PH_ON;

                        /* Reset Bit[1] of 'bIsISOChained' for subsequent operations */
                        bIsISOChained &= 0xFDU;
                    }

                    /* Enable Extended length APDU format in case if Response exceeds the frame size. */
                    else
                        pDataParams->bShortLenApdu = PH_OFF;
                    break;

                default:
                        pDataParams->bShortLenApdu = PH_OFF;
                    break;
            }
        }
    }
    else
    {
        wTotalFrameLen = (uint16_t) (bIsISOChained ? 0 : 1U);

        pCmdHeader_Tmp = &bChainnedCmd;
        wCmdHeaderLen = wTotalFrameLen;

        wBuffOption = (uint16_t) (bIsISOChained ? wOption : PH_EXCHANGE_BUFFER_LAST);
        bFirstFrame = PH_OFF;

        /* Frame options for PICC Exchange. */
        bCmdOptions = (uint8_t) (PHAL_MFDUOX_OPTION_COMPLETE | (bIsISOChained ? PHAL_MFDUOX_EXCLUDE_PICC_STATUS : bCmdOptions));
    }

    /* Buffer Command Header for Exchange to PICC -------------------------------------------------------------------------------------- */
    if((PHAL_MFDUOX_IS_PICC_DATA_COMPLETE == PH_OFF) && (wTotalFrameLen != 0))
    {
        PH_CHECK_SUCCESS_FCT(wStatus_PICC, phalMfDuoX_Sw_Int_CardExchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHAL_MFDUOX_CHAINING_BIT_INVALID,
            PHAL_MFDUOX_OPTION_NONE,
            wTotalFrameLen,
            PH_ON,
            pCmdHeader_Tmp,
            wCmdHeaderLen,
            NULL,
            NULL,
            NULL));
    }

    /* Apply MAC on Command ------------------------------------------------------------------------------------------------------------ */
    if((wOption & 0xFF0F) == PH_EXCHANGE_DEFAULT)
    {
        PH_CHECK_SUCCESS_FCT(wStatus_SM, phalMfDuoX_Sw_Int_ApplySM(
            pDataParams,
            PH_ON,
            PH_ON,
            bCmd_ComMode,
            pCmdHeader,
            wCmdHeaderLen,
            NULL,
            0,
            &pSMBuf,
            &wSMBufLen));
    }

    /* Exchange SM Information to PICC ------------------------------------------------------------------------------------------------- */
    do
    {
        if(PHAL_MFDUOX_IS_PICC_DATA_COMPLETE == PH_OFF)
        {
            wTotalFrameLen = (uint16_t) (bHasChainedRsp ? 1U : 0);
            wStatus_PICC = phalMfDuoX_Sw_Int_CardExchange(
                pDataParams,
                wBuffOption,
                PHAL_MFDUOX_CHAINING_BIT_INVALID,
                bCmdOptions,
                wTotalFrameLen,
                PH_ON,
                bHasChainedRsp ? &bChainnedCmd : pSMBuf,
                (uint16_t) (bHasChainedRsp ? 1 : wSMBufLen),
                &pResponse_PICC,
                &wRspLen_PICC,
                &bPiccErrCode);
        }

        /* Update the PICC status to generic status variable. */
        wStatus = wStatus_PICC;

        /* Set the finished flag to end the loop. */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            bFinished = PH_ON;
            bHasError = PH_ON;
        }

        /* Check if response has Success status. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            bFinished = PH_ON;
            bLastFrame = PH_ON;
            PHAL_MFDUOX_IS_PICC_DATA_COMPLETE = PH_ON;
        }

        /* Verify and Remove Secure Messaging ---------------------------------------------------------------------------------------------- */
        if(!bHasError)
        {
            wStatus_SM = phalMfDuoX_Sw_Int_RemoveSM(
                pDataParams,
                bIsISOChained,
                bFirstFrame,
                bLastFrame,
                bResp_ComMode,
                pResponse_PICC,
                wRspLen_PICC,
                (uint8_t) (bFirstFrame ? PHAL_MFDUOX_RESP_OPERATION_OK : bPiccErrCode),
                &pResponse_SM,
                &wRspLen_SM);
        }

        /* Verify Status of SM */
        if(((wStatus_SM & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus_SM & PH_ERR_MASK ) != PH_ERR_SUCCESS_CHAINING))
            bFinished = PH_ON;

        /* Update the SM status to generic status variable. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            wStatus = wStatus_SM;

        /* Check if response has chaining status. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
        {
            wBuffOption = PH_EXCHANGE_DEFAULT;
            bHasChainedRsp = PH_ON;
            bFirstFrame = PH_OFF;

            /* Complete the loop in case if ISO Chained command. */
            if(bIsISOChained == PH_ON)
                bFinished = PH_ON;
        }

        /* Process response from SM. */
        if((wStatus_SM & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
        {
            /* End the loop. */
            bFinished = PH_ON;

            /* Update the Generic Status. */
            wStatus = wStatus_SM;
        }
        else
        {
            PHAL_MFDUOX_IS_PICC_DATA_COMPLETE = PH_OFF;
#ifdef NXPBUILD__PHAL_MFDUOX_NDA
            PHAL_MFDUOX_HAS_MAC_PROCESSED = PH_OFF;
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */
            PHAL_MFDUOX_CMD_BUF_OFFSET = 0;
        }

        /* Copy the response to parameter. */
        if(pRespLen != NULL)
        {
            *ppResponse = pResponse_SM;
            *pRespLen = wRspLen_SM;
        }
    } while(!bFinished);

    /* Perform Reset Authentication. */
    if(bHasError)
    {
        /* Reset Authentication State. */
        phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
    }

    /* Revert back state of ShortLenAPDU information. */
    if(pDataParams->bWrappedMode)
        pDataParams->bShortLenApdu = bShortLenApdu;

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_WriteData(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bIsNativeChained, uint8_t bCmd_ComMode,
    uint8_t bResp_ComMode, uint8_t bResetAuth, uint8_t * pCmdHeader, uint16_t wCmdHeaderLen, uint8_t * pCmdData,
    uint32_t dwCmdDataLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus_PICC = 0;
    phStatus_t  PH_MEMLOC_REM wStatus_SM = 0;
    uint32_t    PH_MEMLOC_REM dwRemData = 0;
    uint32_t    PH_MEMLOC_REM dwBlockSize = 0;
    uint32_t    PH_MEMLOC_REM dwOffset = 0;
    uint16_t    PH_MEMLOC_REM wTotalFrameLen = 0;
    uint16_t    PH_MEMLOC_REM wExchangeLen = 0;
    uint16_t    PH_MEMLOC_REM wWrappedLen = 0;
    uint16_t    PH_MEMLOC_REM wFrameLen = 0;
    uint16_t    PH_MEMLOC_REM wSMBufLen = 0;
    uint16_t    PH_MEMLOC_REM wDataLen = 0;
    uint16_t    PH_MEMLOC_REM wRspLen_PICC = 0;
    uint16_t    PH_MEMLOC_REM wBuffOption_PICC = PH_EXCHANGE_BUFFER_LAST;
    uint16_t    PH_MEMLOC_REM wRemData_Exchange = 0U;
    uint16_t    PH_MEMLOC_REM wBytesExchanged = 0;
    uint16_t    PH_MEMLOC_REM wTotalBytesExchanged = 0;
    uint8_t     PH_MEMLOC_REM bFinished = PH_OFF;
    uint8_t     PH_MEMLOC_REM bPiccErrCode = PH_OFF;
    uint8_t     PH_MEMLOC_REM bCmdOptions = 0;
    uint8_t     PH_MEMLOC_REM bHasError = PH_OFF;
    uint8_t     PH_MEMLOC_REM bShortLenApdu = PH_OFF;
    uint8_t     PH_MEMLOC_REM bChainnedCmd = PHAL_MFDUOX_ADDITIONAL_FRAME;
    uint8_t     PH_MEMLOC_REM bIsChainnedFrame = PH_OFF;
    uint8_t     PH_MEMLOC_REM bIsFirstFrame = PH_OFF;
    uint8_t     PH_MEMLOC_REM bIsLastFrame = PH_OFF;

    /* In case if there is no more data to exchange with PICC and PICC is still expecting data. */
    uint8_t     PH_MEMLOC_REM bRetry = 1;

    uint8_t *   PH_MEMLOC_REM pSMBuf = NULL;
    uint8_t *   PH_MEMLOC_REM pResponse_PICC = NULL;

    if((bCmd_ComMode != PHAL_MFDUOX_COMMUNICATION_PLAIN) && (pDataParams->bAuthState == PHAL_MFDUOX_NOT_AUTHENTICATED))
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_AL_MFDUOX);
    }

    /* Clear Processing buffer. */
    if(PHAL_MFDUOX_IS_PICC_DATA_COMPLETE == PH_OFF)
    {
        PHAL_MFDUOX_PRS_BUF_LEN = 0;
    }

    /* Get PICC Frame size. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfDuoX_Sw_Int_GetFrameLen(pDataParams, bIsNativeChained, &wFrameLen));

    /* Get Encrypted Data Block Size. */
    PHAL_MFDUOX_NEAREST_MULTIPLE(dwCmdDataLen, dwBlockSize);
    dwBlockSize = (uint16_t) ((dwCmdDataLen == dwBlockSize) ? (dwBlockSize + PH_CRYPTOSYM_AES_BLOCK_SIZE /* Padding */) : dwBlockSize);

    /* Compute Total frame information length.
     * This includes Command Header + Command Data (Based on Communication modes)
     */
#ifdef NXPBUILD__PHAL_MFDUOX_NDA
    wTotalFrameLen = (uint16_t) (wCmdHeaderLen + ((bCmd_ComMode == PHAL_MFDUOX_COMMUNICATION_PLAIN) ? dwCmdDataLen :
        (((bCmd_ComMode == PHAL_MFDUOX_COMMUNICATION_MAC) ? dwCmdDataLen : dwBlockSize) + 8U /* MAC */)));
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */
#ifndef NXPBUILD__PHAL_MFDUOX_NDA
    wTotalFrameLen = (uint16_t) (wCmdHeaderLen + dwCmdDataLen);
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

    /* Compute the maximum bytes that be transferred in one frame. */
    wExchangeLen = wTotalFrameLen;
    wExchangeLen = (uint16_t) (bIsNativeChained ? ((wExchangeLen < wFrameLen) ? wExchangeLen : wFrameLen) : wExchangeLen);

    /* Set Extended Length APDU format in case of ISO Chaining and data size is greater than frame size. */
    if(pDataParams->bWrappedMode)
    {
        if(bIsNativeChained == PH_OFF)
        {
            /* Back up current state of ShortLenAPDUinformation. */
            bShortLenApdu = pDataParams->bShortLenApdu;

            /* Enable Extended length APDU format in case if CommandLen + DataLen + ISO7816 Header exceeds frame size. */
            if((wExchangeLen - 5U) > wFrameLen)
                pDataParams->bShortLenApdu = PH_OFF;
        }

        /* Update wrapped length. */
        else
        {
            wWrappedLen = 6U; /* CLA, INS, P1, P2, LC, LE. */
            wWrappedLen = (uint16_t) (pDataParams->bShortLenApdu ? wWrappedLen : (wWrappedLen + 3U /* Extended LC and LE */));

            /* Update Frame Length in case */
            wFrameLen -= wWrappedLen;

            /* Update Total Frame length. */
            wExchangeLen = (uint16_t) ((wExchangeLen > wFrameLen) ? (wFrameLen + 1U) : wExchangeLen);
        }
    }

    /* Buffer Command Header to PICC exchange buffer ----------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus_PICC, phalMfDuoX_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHAL_MFDUOX_CHAINING_BIT_INVALID,
        PHAL_MFDUOX_OPTION_NONE,
        wExchangeLen,
        PH_ON,
        pCmdHeader,
        wCmdHeaderLen,
        NULL,
        NULL,
        NULL));

    /* Update Remaining data for SM and. */
    dwRemData = dwCmdDataLen;

    /* Set the Frame options. */
    bIsFirstFrame = PH_ON;
    bIsLastFrame = PH_OFF;

    /* Set Command header as exchanged information  */
    wTotalBytesExchanged = wCmdHeaderLen;
    wBytesExchanged = (uint16_t) ((pDataParams->bWrappedMode) ? (wCmdHeaderLen - 1U) : wCmdHeaderLen);

    do
    {
        /* Exclude if all data is processed but PICC exchange is pending. */
        if(bIsLastFrame == PH_OFF)
        {
            /* Update the data to be used for SM Application. */
            wDataLen = (uint16_t) ((dwRemData > wFrameLen) ? wFrameLen : dwRemData);
            wDataLen = (uint16_t) (((wCmdHeaderLen + wDataLen) > wFrameLen) ? PHAL_MFDUOX_ABS(wFrameLen, wCmdHeaderLen) : wDataLen);
            wDataLen = (uint16_t) (bIsChainnedFrame && (wDataLen >= wFrameLen) ? (wDataLen - 1U) : wDataLen);

            /* Update Remaining data to be used. */
            dwRemData -= wDataLen;

            /* Set Framing Options. */
            bIsLastFrame = (uint8_t) (dwRemData <= 0U);

            /* Apply Secure Messaging on Command ------------------------------------------------------------------------------------------- */
            wStatus_SM = phalMfDuoX_Sw_Int_ApplySM(
                pDataParams,
                bIsFirstFrame,
                bIsLastFrame,
                bCmd_ComMode,
                pCmdHeader,
                wCmdHeaderLen,
                &pCmdData[dwOffset],
                wDataLen,
                &pSMBuf,
                &wSMBufLen);

            /* Clear Framing Option */
            bIsFirstFrame = PH_OFF;

            /* Update the PICC status to generic status variable. */
            wStatus = wStatus_SM;

            /* Update the Offset */
            dwOffset += wDataLen;
        }

        /* Verify Status of SM */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            bFinished = PH_ON;
            bHasError = PH_ON;
        }

        /* Set the finished flag to end the loop. */
        bFinished = bIsLastFrame;

        /* Exchange data to PICC. */
        if(bHasError == PH_OFF)
        {
            /* Backup data to be exchanged. */
            wRemData_Exchange = wSMBufLen;
            wDataLen = wSMBufLen;

            do
            {
                if(bIsNativeChained == PH_ON)
                {
                    wBuffOption_PICC = (uint16_t) (((wBytesExchanged + wDataLen) >= wFrameLen) ?
                        PH_EXCHANGE_BUFFER_LAST : PH_EXCHANGE_BUFFER_CONT);

                    wDataLen = (uint16_t) (((wBytesExchanged + wDataLen) > wFrameLen) ?
                        PHAL_MFDUOX_ABS(wFrameLen, wBytesExchanged) : wRemData_Exchange);
                }
                else
                {
                    wBuffOption_PICC = PH_EXCHANGE_BUFFER_CONT;
                }

                /* Update remaining data to exchange */
                wRemData_Exchange -= wDataLen;

                /* Update Buffer Options */
                wBuffOption_PICC = (uint16_t) (bIsLastFrame ? PH_EXCHANGE_BUFFER_LAST : wBuffOption_PICC);

                /* Update options for exchange interface. */
                bCmdOptions = (uint8_t) ((wBuffOption_PICC == PH_EXCHANGE_BUFFER_CONT) ? PHAL_MFDUOX_OPTION_NONE : PHAL_MFDUOX_OPTION_COMPLETE);
                bCmdOptions |= PHAL_MFDUOX_RETURN_PICC_STATUS;

                /* Exchange data. */
                if((wDataLen > 0U) || (bIsLastFrame == PH_ON))
                {
                    /* Clear Total exchange length */
                    wExchangeLen = (uint16_t) ((bIsNativeChained == PH_ON) ? 0U : wExchangeLen);

                    wStatus_PICC = phalMfDuoX_Sw_Int_CardExchange(
                        pDataParams,
                        wBuffOption_PICC,
                        PHAL_MFDUOX_CHAINING_BIT_INVALID,
                        bCmdOptions,
                        wExchangeLen,
                        PH_ON,
                        pSMBuf,
                        wDataLen,
                        &pResponse_PICC,
                        &wRspLen_PICC,
                        &bPiccErrCode);

                    /* Set bytes exchanged fro current frame. */
                    wBytesExchanged += wDataLen;

                    /* Set bytes exchanged */
                    wTotalBytesExchanged += wDataLen;
                }

                /* Update the SM status to generic status variable. */
                wStatus = wStatus_PICC;

                /* Set the finished flag to end the loop. */
                if((bPiccErrCode != PHAL_MFDUOX_RESP_OPERATION_OK) && (bPiccErrCode != PHAL_MFDUOX_RESP_OPERATION_OK_LIM) &&
                    (bPiccErrCode != PHAL_MFDUOX_ADDITIONAL_FRAME))
                {
                    bHasError = PH_ON;
                    bFinished = PH_ON;
                    wRemData_Exchange = 0U;
                }

                /* Exchange chaining command */
                bIsChainnedFrame = (uint8_t) ((wBuffOption_PICC == PH_EXCHANGE_BUFFER_LAST) && (bIsNativeChained == PH_ON));
                bIsChainnedFrame = (uint8_t) ((bPiccErrCode != PHAL_MFDUOX_ADDITIONAL_FRAME) ? PH_OFF : bIsChainnedFrame);

                if(bIsChainnedFrame == PH_ON)
                {
                    while(bRetry != 10U)
                    {
                        /* Update command header length */
                        wCmdHeaderLen = 1U;

                        /* Clear PICC Response Code */
                        bPiccErrCode = 0U;

                        wBytesExchanged = (uint16_t) ((pDataParams->bWrappedMode) ? 0U : wCmdHeaderLen);

                        wExchangeLen = (uint16_t) (((wTotalFrameLen - wTotalBytesExchanged) >= wFrameLen) ? wFrameLen :
                            (wCmdHeaderLen + (wTotalFrameLen - wTotalBytesExchanged)));
                        wExchangeLen = (uint16_t) (wExchangeLen + ((wTotalFrameLen - wTotalBytesExchanged) >= wFrameLen));

                        /* Update Buffer Options */
                        wBuffOption_PICC = (uint16_t) (((dwRemData != 0U) || (wRemData_Exchange != 0U)) ? PH_EXCHANGE_BUFFER_FIRST :
                            PH_EXCHANGE_DEFAULT);

                        /* Update options for exchange interface. */
                        bCmdOptions = PHAL_MFDUOX_RETURN_PICC_STATUS;
                        bCmdOptions |= (uint8_t) ((wBuffOption_PICC == PH_EXCHANGE_BUFFER_FIRST) ? PHAL_MFDUOX_OPTION_NONE :
                            PHAL_MFDUOX_OPTION_COMPLETE);

                        wStatus_PICC = phalMfDuoX_Sw_Int_CardExchange(
                            pDataParams,
                            wBuffOption_PICC,
                            PHAL_MFDUOX_CHAINING_BIT_DISABLE,
                            bCmdOptions,
                            wExchangeLen,
                            PH_ON,
                            &bChainnedCmd,
                            wCmdHeaderLen,
                            &pResponse_PICC,
                            &wRspLen_PICC,
                            &bPiccErrCode);

                        /* Shift SMBuffer to next starting position */
                        pSMBuf += ((wRemData_Exchange > 0U) ? wDataLen : 0U);
                        wDataLen = wRemData_Exchange;

                        /* Update Retry count. */
                        bRetry = (uint8_t) ((wBuffOption_PICC == PH_EXCHANGE_BUFFER_FIRST) ? 10U : (bRetry + 1U));
                        bRetry = (uint8_t) ((bPiccErrCode != PHAL_MFDUOX_ADDITIONAL_FRAME) ? 10U : bRetry);
                    }

                    /* Reset Retry count. */
                    bRetry = 1U;
                }
                else
                {
                    wCmdHeaderLen = (uint16_t) ((bIsNativeChained == PH_OFF) ? 0U : wCmdHeaderLen);
                }

                /* Clear PICC Response Code */
                bPiccErrCode = 0U;

            } while(wRemData_Exchange != 0U);
        }
        else
        {
            /* Set the finished flag to end the loop. */
            bFinished = PH_ON;
        }
    } while(!bFinished);

    /* Clear Offsets and length. */
    PHAL_MFDUOX_CMD_BUF_LEN = 0U;
    PHAL_MFDUOX_CMD_BUF_OFFSET = 0U;

    PHAL_MFDUOX_PRS_BUF_LEN = 0U;
    PHAL_MFDUOX_PRS_BUF_OFFSET = 0U;

    /* Verify and Remove Secure Messaging ---------------------------------------------------------------------------------------------- */
    if(!bHasError)
    {
        if(bResp_ComMode != PHAL_MFDUOX_COMMUNICATION_PLAIN)
        {
            PH_CHECK_SUCCESS_FCT(wStatus_SM, phalMfDuoX_Sw_Int_RemoveSM(
                pDataParams,
                PH_OFF,
                PH_ON,
                PH_ON,
                bResp_ComMode,
                pResponse_PICC,
                wRspLen_PICC,
                bPiccErrCode,
                ppResponse,
                pRespLen));
        }
        else
        {
            /* In case of SUCCESS and Communication mode as PLAIN, increment the command counter. */
            if(wStatus == PH_ERR_SUCCESS)
                pDataParams->wCmdCtr++;

            /* Copy response data to parameter */
            if (pRespLen != NULL)
            {
                *ppResponse = pResponse_PICC;
                *pRespLen = wRspLen_PICC;
            }
        }
    }

    /* Perform Reset Authentication. */
    if(bResetAuth || bHasError)
    {
        /* Clear Reset Authentication flag. */
        bResetAuth = PH_OFF;

        /* Additional Operation for Delete Application command execution. */
        if((pDataParams->bCmdCode == PHAL_MFDUOX_CMD_DELETE_APPLICATION) && (bHasError == PH_OFF))
        {
            /* Reset the Application Identifier. */
            (void) memset(pDataParams->aAid, 0x00, 3);
        }

        /* Reset Authentication State. */
        phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
    }

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
    PHAL_MFDUOX_HAS_MAC_PROCESSED = PH_OFF;
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */
    PHAL_MFDUOX_IS_PICC_DATA_COMPLETE = PH_OFF;

    /* Revert back state of ShortLenAPDU information. */
    if(pDataParams->bWrappedMode && (bIsNativeChained == PH_OFF))
        pDataParams->bShortLenApdu = bShortLenApdu;

    return wStatus;
}





phStatus_t phalMfDuoX_Sw_Int_ISOSelectFile(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t * pData, uint16_t wDataLen,
    uint8_t * pLe, uint8_t bLeLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatusTmp = PH_ERR_SUCCESS;
    phStatus_t  PH_MEMLOC_REM wStatusTmp1 = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint16_t    PH_MEMLOC_REM wVal = 0;

    /* Buffer Command Data. */
    PH_CHECK_SUCCESS_FCT(wStatusTmp, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pData,
        wDataLen,
        &pResponse,
        &wRspLen));

    /* Buffer LE and Exchange the command to PICC.  */
    PH_CHECK_SUCCESS_FCT(wStatusTmp, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pLe,
        bLeLen,
        &pResponse,
        &wRspLen));

    /* Combine Sw1 and Sw2 status codes. */
    wStatusTmp = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);

    /* Evaluate the Status. */
    wStatusTmp = phalMfDuoX_Int_ComputeErrorResponse(pDataParams, wStatusTmp);

    /*  */
    if((wStatusTmp & PH_ERR_MASK) == PHAL_MFDUOX_ERR_DF_7816_GEN_ERROR)
    {
        PH_CHECK_SUCCESS_FCT(wStatusTmp1, phalMfDuoX_GetConfig(pDataParams, PHAL_MFDUOX_ADDITIONAL_INFO, &wVal));
    }

    /*  Check for Success and for LIMITED FUNCTIONALITY error. In both cases, FCI would be returned */
    if((wStatusTmp == PH_ERR_SUCCESS) || (wVal == PHAL_MFDUOX_ISO7816_ERR_6283))
    {
        if(pRspLen != NULL)
        {
            *pRspLen = wRspLen - 2U;
        }

        if(ppResponse != NULL)
        {
            *ppResponse = pResponse;
        }
    }
    else
    {
        /* Nothing to do here */
    }

    return wStatusTmp;
}

phStatus_t phalMfDuoX_Sw_Int_ISOReadBinary(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pLe,
    uint8_t bLeLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    phStatus_t  PH_MEMLOC_REM wStatus_Rsp = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint16_t    PH_MEMLOC_REM wOptions_Tmp = 0;

    if((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_RXCHAINING)
    {
        bLeLen = 0;
        wOptions_Tmp = PH_EXCHANGE_RXCHAINING;
    }
    else
    {
        wOptions_Tmp = PH_EXCHANGE_BUFFER_LAST;
    }

    /* Buffer LE and Exchange the command to PICC.  */
    wStatus = phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        wOptions_Tmp,
        pLe,
        bLeLen,
        &pResponse,
        &wRspLen);

    /* Reset Authentication state. */
    if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
    {
        phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
    }
    else
    {
        /* Combine Sw1 and Sw2 status codes. */
        wStatus_Rsp = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);

        /* Evaluate the Status. */
        wStatus = phalMfDuoX_Sw_Int_ValidateResponse(pDataParams, PHAL_MFDUOX_ISO7816_APDU_CMD, wStatus, wStatus_Rsp);

        /* Copy the response to parameter. */
        *ppResponse = pResponse;
        *pRspLen = wRspLen;

        /* Decrement Status code. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            *pRspLen = *pRspLen - 2U;
    }

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_ISOUpdateBinary(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t * pData, uint16_t wDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    phStatus_t  PH_MEMLOC_REM wStatus_Rsp = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Buffer LE and Exchange the command to PICC.  */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pData,
        wDataLen,
        &pResponse,
        &wRspLen));

    /* Combine Sw1 and Sw2 status codes. */
    wStatus_Rsp = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);

    /* Evaluate the Status. */
    wStatus = phalMfDuoX_Int_ComputeErrorResponse(pDataParams, wStatus_Rsp);

    /* Reset Authentication state. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
    }
    else
    {
        /* Do Nothing */
    }

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_ISOReadRecord(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pLe,
    uint8_t bLeLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    phStatus_t  PH_MEMLOC_REM wStatus_Rsp = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint16_t    PH_MEMLOC_REM wOptions_Tmp = 0;

    if((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_RXCHAINING)
    {
        bLeLen = 0;
        wOptions_Tmp = PH_EXCHANGE_RXCHAINING;
    }
    else
    {
        wOptions_Tmp = PH_EXCHANGE_BUFFER_LAST;
    }

    /* Buffer LE and Exchange the command to PICC.  */
    wStatus = phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        wOptions_Tmp,
        pLe,
        bLeLen,
        &pResponse,
        &wRspLen);

    /* Reset Authentication state. */
    if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
    {
        phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
    }
    else
    {
        /* Combine Sw1 and Sw2 status codes. */
        wStatus_Rsp = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);

        /* Evaluate the Status. */
        wStatus = phalMfDuoX_Sw_Int_ValidateResponse(pDataParams, PHAL_MFDUOX_ISO7816_APDU_CMD, wStatus, wStatus_Rsp);

        /* Copy the response to parameter. */
        *ppResponse = pResponse;
        *pRspLen = wRspLen;

        /* Decrement Status code. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            *pRspLen = *pRspLen - 2U;
    }

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_ISOAppendRecord(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t * pData, uint16_t wDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    phStatus_t  PH_MEMLOC_REM wStatus_Rsp = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Buffer LE and Exchange the command to PICC.  */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pData,
        wDataLen,
        &pResponse,
        &wRspLen));

    /* Combine Sw1 and Sw2 status codes. */
    wStatus_Rsp = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);

    /* Evaluate the Status. */
    wStatus = phalMfDuoX_Int_ComputeErrorResponse(pDataParams, wStatus_Rsp);

    /* Reset Authentication state. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
    }
    else
    {
        /* Do Nothing */
    }

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_ISOGetChallenge(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t * pLe, uint8_t bLeLen,
    uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    phStatus_t  PH_MEMLOC_REM wStatus_Rsp = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Buffer LE and Exchange the command to PICC.  */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pLe,
        bLeLen,
        &pResponse,
        &wRspLen));

    /* Combine Sw1 and Sw2 status codes. */
    wStatus_Rsp = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);

    /* Evaluate the Status. */
    wStatus = phalMfDuoX_Int_ComputeErrorResponse(pDataParams, wStatus_Rsp);

    /* Reset Authentication state. */
    phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);

    /* Copy the response. */
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* Copy the response to parameter. */
        *ppResponse = pResponse;
        *pRspLen = wRspLen - 2U;
    }

    return wStatus;
}





phStatus_t phalMfDuoX_Sw_Int_VdeReadData(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pLe,
    uint8_t bLeLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    phStatus_t  PH_MEMLOC_REM wStatus_Rsp = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint16_t    PH_MEMLOC_REM wOptions_Tmp = 0;

    if((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_RXCHAINING)
    {
        bLeLen = 0;
        wOptions_Tmp = PH_EXCHANGE_RXCHAINING;
    }
    else
    {
        wOptions_Tmp = PH_EXCHANGE_BUFFER_LAST;
    }

    /* Buffer LE and Exchange the command to PICC.  */
    wStatus = phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        wOptions_Tmp,
        pLe,
        bLeLen,
        &pResponse,
        &wRspLen);

    /* Process the PICC response */
    if(pResponse != NULL)
    {
        /* Combine Sw1 and Sw2 status codes. */
        wStatus_Rsp = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);

        /* Evaluate the Status. */
        wStatus = phalMfDuoX_Sw_Int_ValidateResponse(pDataParams, PHAL_MFDUOX_ISO7816_APDU_CMD, wStatus, wStatus_Rsp);

        /* Reset Authentication state. */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
        }
        else
        {
            /* Copy the response to parameter. */
            *ppResponse = pResponse;
            *pRspLen = wRspLen;

            /* Decrement Status code. */
            if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
                *pRspLen = *pRspLen - 2U;
        }
    }
    else
    {
        phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
    }

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_VdeWriteData(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t * pData, uint16_t wDataLen,
    uint8_t * pLe, uint8_t bLeLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    phStatus_t  PH_MEMLOC_REM wStatus_Rsp = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Buffer Command Data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pData,
        wDataLen,
        &pResponse,
        &wRspLen));

    /* Buffer LE and Exchange the command to PICC.  */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pLe,
        bLeLen,
        &pResponse,
        &wRspLen));

    /* Combine Sw1 and Sw2 status codes. */
    wStatus_Rsp = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);

    /* Evaluate the Status. */
    wStatus = phalMfDuoX_Int_ComputeErrorResponse(pDataParams, wStatus_Rsp);

    /* Reset Authentication state. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
    }
    else
    {
        /* Do Nothing */
    }

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_VdeECDSASign(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t * pData, uint16_t wDataLen,
    uint8_t * pLe, uint8_t bLeLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    phStatus_t  PH_MEMLOC_REM wStatus_Rsp = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Buffer Command Data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pData,
        wDataLen,
        &pResponse,
        &wRspLen));

    /* Buffer LE and Exchange the command to PICC.  */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pLe,
        bLeLen,
        &pResponse,
        &wRspLen));

    /* Combine Sw1 and Sw2 status codes. */
    wStatus_Rsp = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);

    /* Evaluate the Status. */
    wStatus = phalMfDuoX_Int_ComputeErrorResponse(pDataParams, wStatus_Rsp);

    /* Reset Authentication state. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        phalMfDuoX_Sw_Int_ResetAuthStatus(pDataParams);
    }
    else
    {
        /* Copy the response to parameter. */
        *ppResponse = pResponse;
        *pRspLen = wRspLen;
    }

    return wStatus;
}





#ifdef NXPBUILD__PHAL_MFDUOX_NDA
phStatus_t phalMfDuoX_Sw_Int_Generate_ICUpgradeKey(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer,
    uint8_t bUpgradeInfo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;
    uint8_t     PH_MEMLOC_REM bKeyLen = 0;

    uint8_t     PH_MEMLOC_REM *pKey = NULL;

    /* Set the pointer. */
    pKey = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 3U)];

    /* Get Key out of the key store object */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNo,
        wKeyVer,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pKey,
        &wKeyType));

    /* Validate KeyType. */
    PHAL_MFDUOX_VALIDATE_KEYTYPE(wKeyType);

    /* Load the IC Upgrade key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsMac,
        pKey,
        wKeyType));

    /* Load Zero IV. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Frame Information to be maced. */
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = 0x96U;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = 0x69U;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = 0x00U;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = 0x01U;
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) ((wKeyType == PH_KEYSTORE_KEY_TYPE_AES128) ? 0x00U : 0x01U);
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = (uint8_t) ((wKeyType == PH_KEYSTORE_KEY_TYPE_AES128) ? 0x80U : 0x00U);
    PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bUpgradeInfo;

    /* Compute MAC to perform PRF of the data and arrive at Upgrade key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        (uint16_t)(PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        PHAL_MFDUOX_CMD_BUF,
        PHAL_MFDUOX_CMD_BUF_LEN,
        pKey,
        &bKeyLen));

    /* Compute Upgrade Key for AES256 */
    if(wKeyType == PH_KEYSTORE_KEY_TYPE_AES256)
    {
        /* Clear Command buffer Length */
        PHAL_MFDUOX_CMD_BUF_LEN = 0U;

        /* Frame Information to be maced. */
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = 0x96U;
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = 0x69U;
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = 0x00U;
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = 0x02U;
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = 0x01U;
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = 0x00U;
        PHAL_MFDUOX_CMD_BUF[PHAL_MFDUOX_CMD_BUF_LEN++] = bUpgradeInfo;

        /* Compute MAC to perform PRF of the data and arrive at Upgrade key. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pCryptoDataParamsMac,
            (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
            PHAL_MFDUOX_CMD_BUF,
            PHAL_MFDUOX_CMD_BUF_LEN,
            &pKey[bKeyLen],
            &bKeyLen));
    }

    /* Load UpgradeKey. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsEnc,
        pKey,
        wKeyType));

    /* Load Zero IV. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */





phStatus_t phalMfDuoX_Sw_Int_ResetAuthStatus(phalMfDuoX_Sw_DataParams_t * pDataParams)
{
    phStatus_t PH_MEMLOC_REM wStatus = PH_ERR_USE_CONDITION;

    pDataParams->wCmdBufLen = 0;
    pDataParams->wCmdBufOffset = 0;
    pDataParams->wPrsBufLen = 0;
    pDataParams->wPrsBufOffset = 0;
    pDataParams->wCmdCtr = 0;
    pDataParams->bCmdCode = PHAL_MFDUOX_CMD_INVALID;
    pDataParams->bAuthState = PHAL_MFDUOX_NOT_AUTHENTICATED;
    pDataParams->bKeyNo = 0xFF;
    pDataParams->bPICCDataComplete = PH_OFF;
#ifdef NXPBUILD__PHAL_MFDUOX_NDA
    pDataParams->bHasMACProcessed = PH_OFF;
    pDataParams->bIsENCPending = PH_OFF;
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

    (void) memset(pDataParams->pCmdBuf, 0x00, PHAL_MFDUOX_CMD_BUFFER_SIZE_MINIMUM);
    (void) memset(pDataParams->pPrsBuf, 0x00, PHAL_MFDUOX_PRS_BUFFER_SIZE_MINIMUM);

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
    (void) memset(pDataParams->aSesAuthENCKey, 0x00, sizeof(pDataParams->aSesAuthENCKey));
    (void) memset(pDataParams->aSesAuthMACKey, 0x00, sizeof(pDataParams->aSesAuthMACKey));
    (void) memset(pDataParams->aTi, 0x00, PHAL_MFDUOX_SIZE_TI);

    (void) phCryptoASym_InvalidateKey(pDataParams->pCryptoDataParamsASym);

    wStatus = phTMIUtils_ActivateTMICollection((phTMIUtils_t *) pDataParams->pTMIDataParams,
        PH_TMIUTILS_RESET_TMI);
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
#ifdef NXPBUILD__PHAL_VCA
    /* Update the authentication state if VCA PC feature is required by the application. */
    if(pDataParams->pVCADataParams != NULL)
    {
        wStatus = phalVca_SetSessionKeyUtility(pDataParams->pVCADataParams, pDataParams->aSesAuthENCKey,
            pDataParams->bAuthState);
    }
#endif /* NXPBUILD__PHAL_VCA */
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */

    return wStatus;
}

phStatus_t phalMfDuoX_Sw_Int_GetFrameLen(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bIsNativeChained, uint16_t * pFrameLen)
{
    uint16_t    PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wFSI = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_GetConfig(pDataParams->pPalMifareDataParams,
        PHPAL_I14443P4_CONFIG_FSI, &wFSI));

    /* Extract FSCI (PICC Frame Size) and Update the parameter. */
    *pFrameLen = aFrameSize[(uint8_t) (wFSI & 0x00FFU)];

    /* Fix Frame length to 64 Bytes if Native Chaining is Enabled */
    if((bIsNativeChained == PH_ON) && (*pFrameLen > PHAL_MFDUOX_MAX_NATIVE_DATA_LEN))
        *pFrameLen = PHAL_MFDUOX_MAX_NATIVE_DATA_LEN;

    /* Remove the ISO header. */
    *pFrameLen -= 4;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

#ifdef NXPBUILD__PHAL_MFDUOX_NDA
phStatus_t phalMfDuoX_Sw_Int_GenerateIv(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bIsResponse, uint8_t * pTi, uint16_t wCmdCtr)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bIndex = 0;
    uint8_t     PH_MEMLOC_REM bCmdCtrMsb = (uint8_t) (wCmdCtr >> 8U);
    uint8_t     PH_MEMLOC_REM bCmdCtrLsb = (uint8_t) (wCmdCtr & 0x00FFU);
    uint8_t     PH_MEMLOC_REM aIv[PH_CRYPTOSYM_AES_BLOCK_SIZE];

    memset(aIv, 0, PH_CRYPTOSYM_AES_BLOCK_SIZE); /* PRQA S 3200 */

    if(bIsResponse)
    {
        /* Form the IV for RespData as 0x5A||0xA5||TI||CmdCtr||0x0000000000000000 */
        aIv[bIndex++] = 0x5AU;
        aIv[bIndex++] = 0xA5U;
    }
    else
    {
        /* Form the IV for CmdData as 0xA5||0x5A||TI||CmdCtr||0x0000000000000000  */
        aIv[bIndex++] = 0xA5U;
        aIv[bIndex++] = 0x5AU;
    }

    aIv[bIndex++] = pTi[0U];
    aIv[bIndex++] = pTi[1U];
    aIv[bIndex++] = pTi[2U];
    aIv[bIndex++] = pTi[3U];
    aIv[bIndex++] = bCmdCtrLsb;
    aIv[bIndex++] = bCmdCtrMsb;

    /* Load Zero IV. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Encrypt IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_ECB,
        aIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        aIv));

    /* Load Encrypted Iv. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        aIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_Int_BackUpIV(void * pCryptoSymDataParams, uint8_t ** ppIV, uint8_t bIV_Len)
{
    switch(PH_GET_COMPID(pCryptoSymDataParams))
    {
#ifdef NXPBUILD__PH_CRYPTOSYM_SW
        case PH_CRYPTOSYM_SW_ID:
            (void) memcpy(ppIV[0], ((phCryptoSym_Sw_DataParams_t *) pCryptoSymDataParams)->pIV, bIV_Len);
            break;
#endif /* NXPBUILD__PH_CRYPTOSYM_SW */

#ifdef NXPBUILD__PH_CRYPTOSYM_MBEDTLS
        case PH_CRYPTOSYM_MBEDTLS_ID:
            (void) memcpy(ppIV[0], ((phCryptoSym_mBedTLS_DataParams_t *) pCryptoSymDataParams)->aIV, bIV_Len);
            break;
#endif /* NXPBUILD__PH_CRYPTOSYM_MBEDTLS */

        default:
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_Int_GenerateSessionKey_TMAC(phalMfDuoX_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bType,
    uint16_t wKeyNoTMACKey, uint16_t wKeyVerTMACKey, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pTMC, uint8_t * pUid,
    uint8_t bUidLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwTMC = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;
    uint8_t     PH_MEMLOC_REM bKeyLen = 0;
    uint8_t     PH_MEMLOC_REM bSV1_ALen = 0;
    uint8_t     PH_MEMLOC_REM bSV1_BLen = 0;

    uint8_t     PH_MEMLOC_REM *pKey = NULL;

    uint8_t     PH_MEMLOC_REM *pSV_A = NULL;
    uint8_t     PH_MEMLOC_REM *pSV_B = NULL;

    /* Formation of TMC as double word value- TMC shall be communicated LSB first. */
    dwTMC = pTMC[0] | (pTMC[1] << 8U) | (pTMC[2] << 16U) | (pTMC[3] << 24U);

    /* If TMC is 0xFFFFFFFF, then return error */
    if(dwTMC == 0xFFFFFFFFU)
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_AL_MFDUOX);

    /* Set the pointer for the Random Numbers. */
    pKey = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 2U)];

    pSV_A = &PHAL_MFDUOX_CMD_BUF[PH_CRYPTOSYM_AES256_KEY_SIZE * 0U];
    pSV_B = &PHAL_MFDUOX_CMD_BUF[PH_CRYPTOSYM_AES256_KEY_SIZE * 1U];

    /* Get Key out of the key store object */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNoTMACKey,
        wKeyVerTMACKey,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pKey,
        &wKeyType));

    /* Validate KeyType. */
    PHAL_MFDUOX_VALIDATE_KEYTYPE(wKeyType);

    /* Diversify Transaction MAC key. */
    if(wOption == PHAL_MFDUOX_TM_KEY_DIVERSIFICATION_ON)
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyDirectKey(
        pDataParams->pCryptoDataParamsEnc,
        wOption,
        pKey,
        wKeyType,
        pDivInput,
        bDivInputLen,
        pKey));

    /* Increment dwTMC */
    dwTMC++;

    /* Frame Session Vector
     * For AES128
     *      SV1  = 0x5A || 0x00 || 0x01 || 0x00 || 0x80 || (TMC + 1) || UID [||0x00::0x00]
     *      SV2  = 0xA5 || 0x00 || 0x01 || 0x00 || 0x80 || (TMC + 1) || UID [||0x00::0x00]
     *
     *      SesTMMAC    => PRF(TMacKey, SV1)
     *      SesTMENC    => PRF(TMacKey, SV2)
     *
     * For AES256
     *      SV1a = 0x5A || 0x00 || 0x01 || 0x01 || 0x00 || (TMC + 1) || UID [||0x00::0x00]
     *      SV1b = 0x5A || 0x00 || 0x02 || 0x01 || 0x00 || (TMC + 1) || UID [||0x00::0x00]
     *      SV2a = 0xA5 || 0x00 || 0x01 || 0x01 || 0x00 || (TMC + 1) || UID [||0x00::0x00]
     *      SV2b = 0xA5 || 0x00 || 0x02 || 0x01 || 0x00 || (TMC + 1) || UID [||0x00::0x00]
     *
     *      SesTMMAC    => PRF(TMacKey, SV1a) || PRF(TMacKey, SV1b)
     *      SesTMENC    => PRF(TMacKey, SV2a) || PRF(TMacKey, SV2b)
     *
     */
    pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = (uint8_t) ((bType == PHAL_MFDUOX_SESSION_MAC) ? 0x5AU : 0xA5U);
    pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = 0x00U;

    pSV_A[bSV1_ALen++] = 0x01U;
    pSV_B[bSV1_BLen++] = 0x02U;

    pSV_A[bSV1_ALen++] = (uint8_t) ((wKeyType == PH_KEYSTORE_KEY_TYPE_AES128) ? 0x00U : 0x01U);
    pSV_B[bSV1_BLen++] = 0x01U;

    pSV_A[bSV1_ALen++] = (uint8_t) ((wKeyType == PH_KEYSTORE_KEY_TYPE_AES128) ? 0x80U : 0x00U);
    pSV_B[bSV1_BLen++] = 0x00U;

    pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = (uint8_t) (dwTMC & 0xFFU);
    pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = (uint8_t) ((dwTMC >> 8U) & 0xFFU);
    pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = (uint8_t) ((dwTMC >> 16U) & 0xFFU);
    pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = (uint8_t) ((dwTMC >> 24U) & 0xFFU);

    /* Add UID. */
    (void) memcpy(&pSV_A[bSV1_ALen], pUid, bUidLen);
    bSV1_ALen += bUidLen;
    (void) memcpy(&pSV_B[bSV1_BLen], pUid, bUidLen);
    bSV1_BLen += bUidLen;

    /* Get AES block size for Session vector input. */
    PHAL_MFDUOX_NEAREST_MULTIPLE(bSV1_ALen, bSV1_ALen);
    PHAL_MFDUOX_NEAREST_MULTIPLE(bSV1_BLen, bSV1_BLen);

    /* load key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsMac,
        pKey,
        wKeyType));

    /* Load Iv */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Generate Transaction MAC Session Key (SV1 or SV1a). */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        pSV_A,
        bSV1_ALen,
        pKey,
        &bKeyLen));

    /* Generate Transaction MAC Session Key (SV1b). */
    if(wKeyType == PH_KEYSTORE_KEY_TYPE_AES256)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pCryptoDataParamsMac,
            (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
            pSV_B,
            bSV1_BLen,
            &pKey[bKeyLen],
            &bKeyLen));

        /* Increment Key to make it as 32 bytes. */
        bKeyLen += PH_CRYPTOSYM_AES_BLOCK_SIZE;
    }

    /* Load Transaction MAC Session Key (SesTMMAC) */
    if(bType == PHAL_MFDUOX_SESSION_MAC)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsMac,
            pKey,
            wKeyType));
    }

    /* Load Transaction MAC Session Key (SesTMENC) */
    else
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsEnc,
            pKey,
            wKeyType));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalMfDuoX_Sw_Int_ComputeSessionKey_SDM(phalMfDuoX_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bSdmOption,
    uint16_t wKeyNo_SDM, uint16_t wKeyVer_SDM, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwSDMReadCtr = 0;
    uint16_t    PH_MEMLOC_REM wKeyType;
    uint8_t     PH_MEMLOC_REM bKeyLen = 0;
    uint8_t     PH_MEMLOC_REM bSV1_ALen = 0;
    uint8_t     PH_MEMLOC_REM bSV1_BLen = 0;

    uint8_t     PH_MEMLOC_REM *pKey = NULL;

    uint8_t     PH_MEMLOC_REM *pSV_A = NULL;
    uint8_t     PH_MEMLOC_REM *pSV_B = NULL;

    /* Formation of SDM Read Counter as double word value - SDM Read Counter shall be communicated LSB first. */
    if(pSDMReadCtr != NULL)
    {
        dwSDMReadCtr = pSDMReadCtr[0U] | (pSDMReadCtr[1U] << 8U) | (pSDMReadCtr[2U] << 16U) | (pSDMReadCtr[3U] << 24U);

        /* If SDMRead Counter is 0x00FFFFFF, then return error */
        if(dwSDMReadCtr >= 0x00FFFFFFU)
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_AL_MFDUOX);
    }

    /* Set the pointer for the Random Numbers. */
    pKey = &PHAL_MFDUOX_PRS_BUF[PHAL_MFDUOX_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 2U)];

    pSV_A = &PHAL_MFDUOX_CMD_BUF[PH_CRYPTOSYM_AES256_KEY_SIZE * 0U];
    pSV_B = &PHAL_MFDUOX_CMD_BUF[PH_CRYPTOSYM_AES256_KEY_SIZE * 1U];

    /* Get Key out of the key store object */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNo_SDM,
        wKeyVer_SDM,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pKey,
        &wKeyType));

    /* Validate KeyType. */
    PHAL_MFDUOX_VALIDATE_KEYTYPE(wKeyType);

    /* Frame Session Vector
     * For AES128
     *      SV1  = 0xC3 || 0x3C || 0x00 || 0x01 || 0x00 || 0x80 || [VCUID] || [SDMReadCtr] || [Zero Padding]
     *      SV2  = 0x3C || 0xC3 || 0x00 || 0x01 || 0x00 || 0x80 || [VCUID] || [SDMReadCtr] || [Zero Padding]
     *
     *      SesSDMFileReadENC   => MAC(SDMFileReadKey, SV1)
     *      SesSDMFileReadMAC   => MAC(SDMFileReadKey, SV2)
     *
     * For AES256
     *      SV1a = 0xC3 || 0x3C || 0x00 || 0x01 || 0x01 || 0x00 || [VCUID] || [SDMReadCtr] || [Zero Padding]
     *      SV1b = 0xC3 || 0x3C || 0x00 || 0x02 || 0x01 || 0x00 || [VCUID] || [SDMReadCtr] || [Zero Padding]
     *      SV2a = 0x3C || 0xC3 || 0x00 || 0x01 || 0x01 || 0x00 || [VCUID] || [SDMReadCtr] || [Zero Padding]
     *      SV2b = 0x3C || 0xC3 || 0x00 || 0x02 || 0x01 || 0x00 || [VCUID] || [SDMReadCtr] || [Zero Padding]
     *
     *      SesSDMFileReadENC   => PRF(SDMFileReadKey, SV1a) || PRF(SDMFileReadKey, SV1b)
     *      SesSDMFileReadMAC   => PRF(SDMFileReadKey, SV2a) || PRF(SDMFileReadKey, SV2b)
     *
     */
    pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = (uint8_t) ((bOption == PHAL_MFDUOX_SESSION_MAC) ? 0x3CU : 0xC3U);
    pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = (uint8_t) ((bOption == PHAL_MFDUOX_SESSION_MAC) ? 0xC3U : 0x3CU);
    pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = 0x00U;

    pSV_A[bSV1_ALen++] = 0x01U;
    pSV_B[bSV1_BLen++] = 0x02U;

    pSV_A[bSV1_ALen++] = (uint8_t) ((wKeyType == PH_KEYSTORE_KEY_TYPE_AES128) ? 0x00U : 0x01U);
    pSV_B[bSV1_BLen++] = 0x01U;

    pSV_A[bSV1_ALen++] = (uint8_t) ((wKeyType == PH_KEYSTORE_KEY_TYPE_AES128) ? 0x80U : 0x00U);
    pSV_B[bSV1_BLen++] = 0x00U;

    /* Add UID. */
    if(bSdmOption & PHAL_MFDUOX_VCUID_PRESENT)
    {
        if(pUid == NULL)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFDUOX);
        }
        else
        {
            (void) memcpy(&pSV_A[bSV1_ALen], pUid, bUidLen);
            bSV1_ALen += bUidLen;
            (void) memcpy(&pSV_B[bSV1_BLen], pUid, bUidLen);
            bSV1_BLen += bUidLen;
        }
    }

    /* Add SDM Read Counter. */
    if(bSdmOption & PHAL_MFDUOX_RDCTR_PRESENT)
    {
        if(pSDMReadCtr != NULL)
        {
            pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = (uint8_t) (dwSDMReadCtr & 0xFFU);
            pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = (uint8_t) ((dwSDMReadCtr >> 8U) & 0xFFU);
            pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = (uint8_t) ((dwSDMReadCtr >> 16U) & 0xFFU);
        }
    }

    /* Get AES block size for Session vector input. */
    PHAL_MFDUOX_NEAREST_MULTIPLE(bSV1_ALen, bSV1_ALen);
    PHAL_MFDUOX_NEAREST_MULTIPLE(bSV1_BLen, bSV1_BLen);

    /* load key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsMac,
        pKey,
        wKeyType));

    /* Load Iv */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalMfDuoX_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Generate SDMFileRead MAC Session Key (SV1 or SV1a). */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        pSV_A,
        bSV1_ALen,
        pKey,
        &bKeyLen));

    /* Generate SDMFileRead MAC Session Key (SV1b). */
    if(wKeyType == PH_KEYSTORE_KEY_TYPE_AES256)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pCryptoDataParamsMac,
            (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
            pSV_B,
            bSV1_BLen,
            &pKey[bKeyLen],
            &bKeyLen));

        /* Increment Key to make it as 32 bytes. */
        bKeyLen += PH_CRYPTOSYM_AES_BLOCK_SIZE;
    }

    /* Load SDM MAC Session Key (SesSDMMAC) */
    if(bOption == PHAL_MFDUOX_SESSION_MAC)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsMac,
            pKey,
            wKeyType));
    }

    /* Load SDM MAC Session Key (SesSDMENC) */
    else
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsEnc,
            pKey,
            wKeyType));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

void phalMfDuoX_Sw_Int_TruncateMac(uint8_t * pMac)
{
    uint8_t PH_MEMLOC_REM bIndex = 0;
    uint8_t PH_MEMLOC_REM bIndex2 = 0;

    for(bIndex = 1U, bIndex2 = 0; bIndex < 16U; bIndex += 2U, bIndex2++)
    {
        pMac[bIndex2] = pMac[bIndex];
    }
}
#endif /* NXPBUILD__PHAL_MFDUOX_NDA */
#endif /* NXPBUILD__PHAL_MFDUOX_SW */
