/*
 * Copyright 2009 - 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
* Sam (X-Mode) MIFARE(R) Classic Component of Reader Library Framework.
* $Author: Rajendran Kumar (nxp99556) $
* $Revision: 7467 $
* $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
*
* History:
*  CHu: Generated 31. July 2009
*
*/

#ifndef PHALMFC_SAM_X_H
#define PHALMFC_SAM_X_H

#include <ph_Status.h>
#include <phalMfc.h>

/*
 * Authenticates with the card using MIFARE keys provided.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for Authentication.
 *		bKeyType				: The type of key to be used.
 *									0x0A: PHAL_MFC_KEYA.
 *									0x0B: PHAL_MFC_KEYB.
 *		bKeyNo					: Reference key number to be used from hardware keystore.
 *		bKeyVer					: Reference key version to be used form hardware keystore.
 *		pUid					: The complete UID of the Card.
 *		bUidLen					: The length of the UID.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sam_X_Authenticate(void * pDataParams, uint8_t bBlockNo, uint8_t bKeyType, uint8_t bKeyNo, uint8_t bKeyVer,
    uint8_t * pUid, uint8_t bUidLen);

/*
 * Reads the data from MIFARE PICC.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for reading the data.
 *
 * Output Parameters:
 *		pData					: The data read from MIFARE PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sam_X_Read(void * pDataParams, uint8_t bBlockNo, uint8_t * pData);

/*
 * Reads the data from from MIFARE PICC. This read data will be in value format.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for reading the value.
 *
 * Output Parameters:
 *		pValue					: The value read from MIFARE PICC.
 *		pAddr					: The address read from MIFARE PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sam_X_ReadValue(void * pDataParams, uint8_t bBlockNo, uint8_t * pValue, uint8_t * pAddr);

/*
 * Write the data to MIFARE PICC.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for writing the data.
 *		pData					: The data to be written to MIFARE PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sam_X_Write(void * pDataParams, uint8_t bBlockNo, uint8_t * pData);

/*
 * Write the value to MIFARE PICC.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for writing the value.
 *		pValue					: The value to be written to MIFARE PICC.
 *		bAddrData				: The address to written to MIFARE PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sam_X_WriteValue(void * pDataParams, uint8_t bBlockNo, uint8_t * pValue, uint8_t bAddr);

/*
 * Increments the value in a MIFARE PICC. Performs the communication with PICC using SAM's Cmd.ISO14443-3_TransparentExchange
 * command. As per MIFARE Classic product, Increment just increments the valus but not saves. As per SAM Cmd.MF_Increment
 * increments the value and saves to the memory.
 * So for avoiding any issues while communicating to PICC with SAM attached and not to break the execution flow, this
 * command is directly communicated to PICC uisng SAM's L3 Transparent Exchange command. The actual communication of Sam will be
 * performed using IncrementTransfer command.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for incrementing the value.
 *		pValue					: The value to be incremented in a MIFARE PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sam_X_Increment(void * pDataParams, uint8_t bBlockNo, uint8_t * pValue);

/*
 * Decrements the value in a MIFARE PICC. Performs the communication with PICC using SAM's Cmd.ISO14443-3_TransparentExchange
 * command. As per MIFARE Classic product, Decrement just decrements the valus but not saves. As per SAM Cmd.MF_Increment
 * decrements the value and saves to the memory.
 * So for avoiding any issues while communicating to PICC with SAM attached and not to break the execution flow, this
 * command is directly communicated to PICC uisng SAM's L3 Transparent Exchange command. The actual communication of Sam will be
 * performed using DecrementTransfer command.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for decrementing the value.
 *		pValue					: The value to be Decremented in a MIFARE PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sam_X_Decrement(void * pDataParams, uint8_t bBlockNo, uint8_t * pValue);

/*
 * Transfers the value in a MIFARE PICC. Performs the communication with PICC using SAM's Cmd.ISO14443-3_TransparentExchange
 * command. As per MIFARE Classic product, transfer saves the increments or decremented value to the memory but as per SAM there
 * is no seperate command to Transfer the value.
 * So for avoiding any issues while communicating to PICC with SAM attached and not to break the execution flow, this
 * command is directly communicated to PICC uisng SAM's L3 Transparent Exchange command.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for transferring the value.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sam_X_Transfer(void * pDataParams, uint8_t bBlockNo);

/*
 * Restores the value in a MIFARE PICC. Performs the communication with PICC using SAM's Cmd.ISO14443-3_TransparentExchange
 * command. As per MIFARE Classic product, restores already saved value to the memory but as per SAM there
 * is no seperate command to Transfer the value.
 * So for avoiding any issues while communicating to PICC with SAM attached and not to break the execution flow, this
 * command is directly communicated to PICC uisng SAM's L3 Transparent Exchange command.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bBlockNo				: The block number to be used for restoring the value.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sam_X_Restore(void * pDataParams, uint8_t bBlockNo);

/*
 * Increments and Transfers the value in a MIFARE PICC. This operation is performed using the SAM hardware.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bSrcBlockNo				: The source block number used for incrementing the value.
 *		bDstBlockNo				: The destination block number used for transferring the value.
 *		pValue					: The value to be incremented and transferred.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sam_X_IncrementTransfer(void * pDataParams, uint8_t bSrcBlockNo, uint8_t bDstBlockNo, uint8_t * pValue);

/*
 * Decrements and Transfers the value in a MIFARE PICC. This operation is performed using the SAM hardware.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bSrcBlockNo				: The source block number used for Decrementing the value.
 *		bDstBlockNo				: The destination block number used for transferring the value.
 *		pValue					: The value to be decremented and transferred.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sam_X_DecrementTransfer(void * pDataParams, uint8_t bSrcBlockNo, uint8_t bDstBlockNo, uint8_t * pValue);

/*
 * Restores the transferred value in a MIFARE PICC. This operation is performed using the SAM hardware.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bSrcBlockNo				: The source block number used for restoring the value.
 *		bDstBlockNo				: The destination block number used for restoring the value.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sam_X_RestoreTransfer(void * pDataParams, uint8_t bSrcBlockNo, uint8_t bDstBlockNo);

/*
 * Personalizes the UID based om the UID type provided. Performing communication to PICC instead of SAM because SAM do not support
 * UID Personalization. So for avoiding any issues while communicating to PICC with SAM attached and not to break the execution flow,
 * this command is directly communicated to PICC.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		bUidType				: One of the below types for personalizing the UID.
 *									0x00: PHAL_MFC_UID_TYPE_UIDF0
 *									0x40: PHAL_MFC_UID_TYPE_UIDF1
 *									0x20: PHAL_MFC_UID_TYPE_UIDF2
 *									0x60: PHAL_MFC_UID_TYPE_UIDF3
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sam_X_PersonalizeUid(void * pDataParams, uint8_t bUidType);

/*
 * Perform MIFARE(R) GET VERSION usage command sequence with MIFARE Picc. GET_VERSION command is used to retrieve information
 * about the NTAG family, the product version, storage size and other product data required to identify the specific NTAG-I2C.
 * Performing communication to PICC instead of SAM because SAM do not support retreiving the version. So for avoiding any issues
 * while communicating to PICC with SAM attached and not to break the execution flow, this command is directly communicated to PICC.
 *
 * Output Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		pVersion				: The version returned by PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sam_X_GetVersion(void * pDataParams, uint8_t * pVersion);

/*
 * Convert HAL status codes to palMifare status codes.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to this layer's parameter structure.
 *		wStatus					: The status code returned by Sam.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Sam_X_ConvertNak(void * pDataParams, phStatus_t wStatus);
#endif /* PHALMFC_SAM_X_H */
