/*
 * Copyright 2009 - 2018, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal functions for MIFARE (R) application layer.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 *
 * History:
 *  CHu: Generated 19. May 2009
 *
 */

#ifndef PHALMFC_INT_H
#define PHALMFC_INT_H

#include <ph_Status.h>

/** \addtogroup phalMfc
 * @{
 */
/** \name Status Codes of MIFARE(R) Classic PICC. */
/** @{ */
#define PHAL_MFC_RESP_ACK       0x0A    /**< MIFARE Classic ACK response code */
#define PHAL_MFC_RESP_NAK0      0x00    /**< MIFARE Classic NAK0 response code */
#define PHAL_MFC_RESP_NAK1      0x01    /**< MIFARE Classic NAK1 response code */
#define PHAL_MFC_RESP_NAK4      0x04    /**< MIFARE Classic NAK4 response code */
#define PHAL_MFC_RESP_NAK5      0x05    /**< MIFARE Classic NAK5 response code */
/** @} */

/** \name Command Codes of MIFARE(R) Classic PICC. */
/** @{ */
#define PHAL_MFC_CMD_RESTORE    0xC2    /**< MIFARE Classic Restore command byte */
#define PHAL_MFC_CMD_INCREMENT  0xC1    /**< MIFARE Classic Increment command byte */
#define PHAL_MFC_CMD_DECREMENT  0xC0    /**< MIFARE Classic Decrement command byte */
#define PHAL_MFC_CMD_TRANSFER   0xB0    /**< MIFARE Classic Transfer command byte */
#define PHAL_MFC_CMD_READ       0x30    /**< MIFARE Classic Read command byte */
#define PHAL_MFC_CMD_WRITE      0xA0    /**< MIFARE Classic Write command byte */
#define PHAL_MFC_CMD_AUTHA      0x60    /**< MIFARE Classic Authenticate A command byte */
#define PHAL_MFC_CMD_AUTHB      0x61    /**< MIFARE Classic Authenticate B command byte */
#define PHAL_MFC_CMD_PERSOUID   0x40    /**< MIFARE Classic Personalize UID command */
#define PHAL_MFC_CMD_GETVERSION 0x62    /**< MIFARE Classic GET VERSION command */
/** @} */

#define PHAL_MFC_VERSION_LENGTH 0x08	/**< Length of a Version MIFARE(R) Classic data block */

/**
 * end of defgroup phalMfc
 * @}
 */

/*
 * Perform MIFARE(R) Transfer command with MIFARE Picc. This interface will be called by Software and
 * Sam component.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to PAL Mifare parameter structure.
 *		bBlockNo				: The block number to be used for transferring the value.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Int_Transfer(void * pPalMifareDataParams, uint8_t bBlockNo);

/*
 * Perform a MIFARE(R) Value operation with MIFARE Picc. This is used by INCREMENT, DECREMENT and RESTORE commands.
 * This interface will be called by Software and Sam component.
 *
 * Input Parameters:
 *		pDataParams				: Pointer to PAL Mifare layer's parameter structure.
 *		bBlockNo				: The block number to be used for decrementing the value.
 *		pValue					: The value to be Decremented in a MIFARE PICC.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Int_Value(void * pPalMifareDataParams, uint8_t bCmdCode, uint8_t bBlockNo,
	uint8_t * pValue);

/*
 * Create a Value block for a given value/addr pair.
 *
 * Input Parameters:
 *		pValue					: The value to be used for valueblock format creation.
 *		bAddrData				: The address to be used for valueblock format creation.
 *		pBlock					: The block to be used for valueblock format creation.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Int_CreateValueBlock(uint8_t * pValue, uint8_t bAddrData, uint8_t * pBlock);

/*
 * Check value block format of a given block.
 *
 * Input Parameters:
 *		pBlock					: The block to be used for valueblock format verification.
 *
 * Return:
 *			PH_ERR_SUCCESS for successfull operation.
 *			Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfc_Int_CheckValueBlockFormat(uint8_t * pBlock);

#endif /* PHALMFC_INT_H */
