/*
 * Copyright 2016 - 2018, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Mifare Plus application's Sam NonX layer's internal component of Reader Library framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <ph_RefDefs.h>
#include <string.h>
#include <ph_TypeDefs.h>
#include <phpalMifare.h>
#include <phalMfp.h>

#ifdef NXPBUILD__PHAL_MFP_SAM_NONX

#include "../phalMfp_Int.h"
#include "phalMfp_Sam_NonX_Int.h"

phStatus_t phalMfp_Sam_NonX_Int_CardExchange(void * pDataParams, uint16_t wOption, uint8_t bIsoLayer, uint8_t * pPayload,
	uint16_t wPayloadLen, uint8_t ** ppResponse, uint16_t * pRespLen, uint8_t * pPiccErrCode)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
	uint8_t *	PH_MEMLOC_REM pResponse = NULL;

	/* Exchange the command to card based on activation level. */
	if(bIsoLayer)
	{
		/* Exchange the command in ISO14443 L4 activated state. */
		wStatus = phpalMifare_ExchangeL4(
			PHAL_MFP_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
			wOption,
			pPayload,
			wPayloadLen,
			&pResponse,
			pRespLen);

		if ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
		{
			PH_CHECK_SUCCESS(wStatus);
		}
	}
	else
	{
		/* Exchange the command in ISO14443 L3 activated state. */
		wStatus = phpalMifare_ExchangeL3(
			PHAL_MFP_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
			wOption,
			pPayload,
			wPayloadLen,
			&pResponse,
			pRespLen);

		*pPiccErrCode = (uint8_t) (wStatus & PH_ERR_MASK);
	}

	if((wOption == PH_EXCHANGE_DEFAULT) || (wOption == PH_EXCHANGE_BUFFER_LAST) || (wOption == PH_EXCHANGE_RXCHAINING))
	{
		if(wOption != PH_EXCHANGE_RXCHAINING)
		{
			/* Add the status code. */
			*pPiccErrCode = pResponse[0];

			/* Decrement the response length to discard status code. */
			(*pRespLen)--;
		}

		/* Add the Response data excluding the status code. */
		*ppResponse = &pResponse[(wOption == PH_EXCHANGE_RXCHAINING) ? 0 : 1];

		/* Validate the status. */
		if(wOption != PH_EXCHANGE_RXCHAINING)
		{
			PH_CHECK_SUCCESS_FCT(wStatus1, phalMfp_Int_ComputeErrorResponse(1, pResponse[0], bIsoLayer));
		}
	}

	return wStatus;
}

phStatus_t phalMfp_Sam_NonX_Int_AuthenticateMFP(void * pDataParams, uint8_t bLayer4Comm, uint8_t bFirstAuth, uint8_t bKdf, uint16_t wBlockNr,
	uint16_t wKeyNum, uint16_t wKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pPcdCap2In, uint8_t bPcdCap2InLen,
	uint8_t * pPcdCap2Out, uint8_t * pPdCap2)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;
	uint8_t		PH_MEMLOC_REM bPiccRetCode = 0;
	uint8_t		PH_MEMLOC_REM bOption = 0;
    uint8_t		PH_MEMLOC_REM aCmdBuff[PHAL_MFP_AUTH_CMD_SIZE];
    uint8_t		PH_MEMLOC_REM bCmdLen = 0;
	uint8_t *	PH_MEMLOC_REM pResp_Card = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen_Card = 0;
	uint8_t *	PH_MEMLOC_REM pResp_SAM = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen_SAM = 0;
	uint8_t 	PH_MEMLOC_REM aResp_SAM[PHAL_MFP_RESP_PCD_CHAL_SIZE];
    uint8_t		PH_MEMLOC_REM bRespLen_SAM = 0;

	/* Check if bPcdCap2InLen data is not greater than 6. */
    if (bPcdCap2InLen > PHAL_MFP_CAPABILITY_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFP);
    }

/* Frame the First part of Authenticate command to be sent to card. */
	/* Reset the command buffer and length. */
	bCmdLen = 0;
	memset(aCmdBuff, 0x00, sizeof(aCmdBuff));	/* PRQA S 3200 */

	/* Add the Authentication command code to command buffer. */
    if (bFirstAuth)
    {
		aCmdBuff[bCmdLen++] = PHAL_MFP_CMD_AUTH_FIRST;
    }
    else
    {
        aCmdBuff[bCmdLen++] = PHAL_MFP_CMD_AUTH_NON_FIRST;
    }

	/* Append the block number. */
	aCmdBuff[bCmdLen++] = (uint8_t)(wBlockNr & 0xFF); /* LSB */
    aCmdBuff[bCmdLen++] = (uint8_t)(wBlockNr >> 8);   /* MSB */

	/* Append PCD Capibilites to command buffer in case of First authentication and Layer 3 / 4 Activated state. */
	if(bFirstAuth)
	{
		/* Update the PCD capability length as zero to command buffer by default. */
		aCmdBuff[bCmdLen++] = 0;

		if(bLayer4Comm)
		{
			/* Update the PCD capability length to command buffer. */
			aCmdBuff[bCmdLen - 1] = bPcdCap2InLen;

			memcpy(&aCmdBuff[bCmdLen], pPcdCap2In, bPcdCap2InLen);	/* PRQA S 3200 */
			bCmdLen += bPcdCap2InLen;
		}
	}

/* Exchange First part of authentication command to card ============================================================================= */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_NonX_Int_CardExchange (
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		bLayer4Comm,
		aCmdBuff,
		bCmdLen,
		&pResp_Card,
		&wRespLen_Card,
		&bPiccErrCode));

	/* Check if response consists of correct data size. */
    if (wRespLen_Card != PHAL_MFP_RESP_PD_CHAL_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFP);
    }

/* Exchange the First part of Authenticate command to SAM hardware ------------------------------------------------------------------- */

	/* Exchagne the command to Sam layer. */
	if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_ID)
	{
		bOption |= bFirstAuth;

		if(bKdf == PHAL_MFP_AUTHENTICATE_SL2_KDF)
		{
			bOption |= PHHAL_HW_SAMAV2_CMD_SAM_AUTHENTICATE_MFP_SL2_KDF;
		}

		if(bKdf == PHAL_MFP_AUTHENTICATE_SL3_KDF)
		{
			bOption |= PHHAL_HW_SAMAV2_CMD_SAM_AUTHENTICATE_MFP_SL3_KDF;
		}

		wStatus = phhalHw_SamAV2_Cmd_SAM_AuthenticateMfp_Part1(
			((phalMfp_SamAV2_DataParams_t *)pDataParams)->pHalSamDataParams,
			bOption,
			(uint8_t) wKeyNum,
			(uint8_t) wKeyVer,
			pResp_Card,
			pDivInput,
			bDivInputLen,
			aResp_SAM,
			&bRespLen_SAM);

		/* Copy the information to the pointer. */
		pResp_SAM = aResp_SAM;
		wRespLen_SAM = bRespLen_SAM;
	}
	else
	{
		#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
		/* Update the option variable with proper authentication type and input PCD Caps. */
		if (!bFirstAuth)
		{
			bOption |= PHHAL_HW_SAMAV3_MFP_AUTHENTICATE_NON_FIRST;
			bPcdCap2InLen = 0;
		}

		/* Update the option variable with key derivation information. */
		bOption |= bKdf;

		/* Update the option variable with proper diversification information. */
		if (bDivInputLen)
		{
			bOption |= PHHAL_HW_SAMAV3_MFP_AUTHENTICATE_DIVERSIFICATION_ON;
		}

		wStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticateMFP_Part1(
			((phalMfp_SamAV3_NonX_DataParams_t *)pDataParams)->pHalSamDataParams,
			bOption,
			(uint8_t) wKeyNum,
			(uint8_t) wKeyVer,
			pResp_Card,
			(uint8_t) wRespLen_Card,
			pDivInput,
			bDivInputLen,
			&pResp_SAM,
			&wRespLen_SAM);
		#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
	}

	/* Check if chaining status is returned from HAL. */
	if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
	{
		/* Return the status. */
		return wStatus;
	}

	/* Check if the response received from SAM is not of reqired size. */
	if(wRespLen_SAM != PHAL_MFP_RESP_PCD_CHAL_SIZE)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFP);
	}

/* Exchange Second part of Authentication command (PCDChalResp) to Card ============================================================== */
	/* Reset the command buffer and length. */
	bCmdLen = 0;
	memset(aCmdBuff, 0x00, sizeof(aCmdBuff));	/* PRQA S 3200 */

	/* Frame the command. */
	aCmdBuff[bCmdLen++] = PHAL_MFP_CMD_AUTH_CONTINUE;

	/* Append PCD Challange response received from SAM. */
	memcpy(&aCmdBuff[1], pResp_SAM, wRespLen_SAM);	/* PRQA S 3200 */
	bCmdLen += (uint8_t) wRespLen_SAM;

	/* Exchange the data. */
	bPiccErrCode = 0;
	wStatus = phalMfp_Sam_NonX_Int_CardExchange (
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		bLayer4Comm,
		aCmdBuff,
		bCmdLen,
		&pResp_Card,
		&wRespLen_Card,
		&bPiccErrCode);

	/* Exchagne the command to Sam layer. */
	if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_ID)
	{
		/* Reset the command buffer and length. */
		bCmdLen = 0;
		memset(aCmdBuff, 0x00, sizeof(aCmdBuff));	/* PRQA S 3200 */

		/* Frame the second part of command. */
		if((bPiccErrCode != 0x90) && (bPiccErrCode != 0x0A))
		{
			aCmdBuff[bCmdLen++] = bPiccErrCode;
		}
		else
		{
			memcpy(aCmdBuff, pResp_Card, wRespLen_Card);	/* PRQA S 3200 */
			bCmdLen = (uint8_t) wRespLen_Card;
		}

		wStatus = phhalHw_SamAV2_Cmd_SAM_AuthenticateMfp_Part2(
			((phalMfp_SamAV2_DataParams_t *)pDataParams)->pHalSamDataParams,
			aCmdBuff,
			bCmdLen,
			pPdCap2,
			pPcdCap2Out);

		bPiccRetCode = bPiccErrCode;
	}
	else
	{
		#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
		wStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticateMFP_Part2(
			((phalMfp_SamAV3_NonX_DataParams_t *)pDataParams)->pHalSamDataParams,
			bPiccErrCode,
			pResp_Card,
			(uint8_t) wRespLen_Card,
			&pPdCap2,
			&pPcdCap2Out,
			&bPiccRetCode);
		#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
	}

	/* Return the error code. */
	if ((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV2_ERR_MIFARE_PLUS_GEN)
	{
		/* Compute the response code. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_ComputeErrorResponse(1, bPiccRetCode, bLayer4Comm));
	}
	else
	{
		/* Return the error code other than success.*/
		if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
		{
			/* Return the status. */
			return wStatus;
		}
		else
		{
			/* In case of Originality Key - reset secure messaging */
			if ((wBlockNr >= PHAL_MFP_SAM_NONX_ORIGINALITY_KEY_FIRST) && (wBlockNr <= PHAL_MFP_SAM_NONX_ORIGINALITY_KEY_LAST))
			{
				/* Kill only card Auth */
				PH_CHECK_SUCCESS_FCT(wStatus, PHHAL_HW_SAM_KILL_AUTHENTICATION(pDataParams, 0x01));
			}
		}
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_NonX_Int_CombinedWriteMFP(void * pDataParams, uint8_t bCmdCode, uint16_t wSrcBlockNr, uint16_t wDstBlockNr,
	uint8_t * pData, uint8_t bDataLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;
	uint8_t		PH_MEMLOC_REM bPiccRetCode = 0;
    uint8_t		PH_MEMLOC_REM aCmdBuff[PHAL_MFP_COMBINED_WRITE_CMD_SIZE];
    uint8_t		PH_MEMLOC_REM bCmdLen = 0;
	uint8_t *	PH_MEMLOC_REM pResp_Card = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen_Card = 0;
	uint8_t *	PH_MEMLOC_REM pResp_SAM = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen_SAM = 0;

	/* Reset the command buffer and its length variable. */
	bCmdLen = 0;
	memset(aCmdBuff, 0x00, sizeof(aCmdBuff));	/* PRQA S 3200 */

/* Frame the command and send it to card and SAM. ---------------------------------------------------------------------------- */
	aCmdBuff[bCmdLen++] = bCmdCode;
    aCmdBuff[bCmdLen++] = (uint8_t)(wSrcBlockNr & 0xff); /* LSB */
    aCmdBuff[bCmdLen++] = (uint8_t)(wSrcBlockNr >> 8);   /* MSB */

    if ((bCmdCode == PHAL_MFP_CMD_INCRTR) || (bCmdCode == PHAL_MFP_CMD_INCRTR_M) ||
        (bCmdCode == PHAL_MFP_CMD_DECRTR) || (bCmdCode == PHAL_MFP_CMD_DECRTR_M))
    {
        aCmdBuff[bCmdLen++] = (uint8_t)(wDstBlockNr & 0xff); /* LSB */
        aCmdBuff[bCmdLen++] = (uint8_t)(wDstBlockNr >> 8);   /* MSB */
    }

/* Send the command buffer to SAM for encrypting the data and calculating the MAC if required. ------------------------------ */
	if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_ID)
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_SAM_CombinedWriteMfp(
			((phalMfp_SamAV2_DataParams_t *)pDataParams)->pHalSamDataParams,
			PH_EXCHANGE_BUFFER_FIRST | PHHAL_HW_SAMAV2_CMD_SAM_COMBINED_WRITE_MFP_COMMAND,
			aCmdBuff,
			bCmdLen,
			&pResp_SAM,
			&wRespLen_SAM));

		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_SAM_CombinedWriteMfp(
			((phalMfp_SamAV2_DataParams_t *)pDataParams)->pHalSamDataParams,
			PH_EXCHANGE_BUFFER_LAST,
			pData,
			bDataLen,
			&pResp_SAM,
			&wRespLen_SAM));
	}
	else
	{
		#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_CombinedWriteMFP(
			((phalMfp_SamAV3_NonX_DataParams_t *)pDataParams)->pHalSamDataParams,
			PH_EXCHANGE_BUFFER_FIRST | PHHAL_HW_SAMAV3_OPTION_COMBINED_WRITE_MFP_COMMAND,
			aCmdBuff,
			bCmdLen,
			NULL,
			NULL,
			NULL));

		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_CombinedWriteMFP(
			((phalMfp_SamAV3_NonX_DataParams_t *)pDataParams)->pHalSamDataParams,
			PH_EXCHANGE_BUFFER_LAST,
			pData,
			bDataLen,
			&pResp_SAM,
			&wRespLen_SAM,
			NULL));
		#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
	}

/* Exchange the data received from SAM to the card. ------------------------------------------------------------------------- */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_NonX_Int_CardExchange(
		pDataParams,
		PH_EXCHANGE_BUFFER_FIRST,
		PH_ON,
		aCmdBuff,
		bCmdLen,
		NULL,
		NULL,
		NULL));

	wStatus = phalMfp_Sam_NonX_Int_CardExchange(
		pDataParams,
		PH_EXCHANGE_BUFFER_LAST,
		PH_ON,
		pResp_SAM,
		wRespLen_SAM,
		&pResp_Card,
		&wRespLen_Card,
		&bPiccErrCode);

/* Send the received data to SAM for processing the MAC is available. -------------------------------------------------------- */
	/* Clear the command buffer. */
	bCmdLen = 0;
	memset(aCmdBuff, 0x00, sizeof(aCmdBuff));	/* PRQA S 3200 */

	/* Append the Response code. */
	aCmdBuff[bCmdLen++] = bPiccErrCode;

	/* Exchange the command to SAM for MAC verification if MAC is returned by the Card. */
	if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_ID)
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_SAM_CombinedWriteMfp(
			((phalMfp_SamAV2_DataParams_t *)pDataParams)->pHalSamDataParams,
			PH_EXCHANGE_BUFFER_FIRST | PHHAL_HW_SAMAV2_CMD_SAM_COMBINED_WRITE_MFP_RESPONSE,
			aCmdBuff,
			bCmdLen,
			&pResp_SAM,
			&wRespLen_SAM));

		wStatus = phhalHw_SamAV2_Cmd_SAM_CombinedWriteMfp(
			((phalMfp_SamAV2_DataParams_t *)pDataParams)->pHalSamDataParams,
			PH_EXCHANGE_BUFFER_LAST,
			pResp_Card,
			(uint8_t) wRespLen_Card,
			&pResp_SAM,
			&wRespLen_SAM);

		bPiccRetCode = bPiccErrCode;
	}
	else
	{
		#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_CombinedWriteMFP(
			((phalMfp_SamAV3_NonX_DataParams_t *)pDataParams)->pHalSamDataParams,
			PH_EXCHANGE_BUFFER_FIRST | PHHAL_HW_SAMAV3_OPTION_COMBINED_WRITE_MFP_RESPONSE,
			aCmdBuff,
			bCmdLen,
			&pResp_SAM,
			&wRespLen_SAM,
			&bPiccErrCode));

		wStatus = phhalHw_SamAV3_Cmd_SAM_CombinedWriteMFP(
			((phalMfp_SamAV3_NonX_DataParams_t *)pDataParams)->pHalSamDataParams,
			PH_EXCHANGE_BUFFER_LAST,
			pResp_Card,
			(uint8_t) wRespLen_Card,
			&pResp_SAM,
			&wRespLen_SAM,
			&bPiccRetCode);
		#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
	}

	/* Return the error code. */
	if ((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV2_ERR_MIFARE_PLUS_GEN)
	{
		/* Compute the response code. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_ComputeErrorResponse(1, bPiccRetCode, PH_ON));
	}
	else
	{
		/* Return the error code other than success.*/
		if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
		{
			return wStatus;
		}
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_NonX_Int_CombinedReadMFP(void * pDataParams, uint8_t bCmdCode, uint16_t wBlockNr,
	uint8_t bNumBlocks, uint8_t * pBlocks)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM aCmdBuff[PHAL_MFP_COMBINED_READ_CMD_SIZE];
    uint8_t		PH_MEMLOC_REM bCmdLen = 0;
    uint8_t *	PH_MEMLOC_REM pResp_Card = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen_Card = 0;
	uint8_t *	PH_MEMLOC_REM pResp_Sam = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen_Sam = 0;
    uint8_t		PH_MEMLOC_REM bPayloadType = 0;
	uint8_t		PH_MEMLOC_REM bLFI = 0x00;
    uint16_t	PH_MEMLOC_REM wOption = PH_EXCHANGE_DEFAULT;
	uint16_t	PH_MEMLOC_REM wRemBytes = 0;
	uint16_t	PH_MEMLOC_REM wDataLen = 0;
    uint16_t	PH_MEMLOC_REM wOffset = 0;
    uint16_t	PH_MEMLOC_REM wOffset1 = 0;
	uint8_t		PH_MEMLOC_REM bMacLen = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;
	uint8_t		PH_MEMLOC_REM bPiccRetCode = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;

	/* Compute the Payload Type information. */
	if (bCmdCode < PHAL_MFP_CMD_READ_ENU /* Command code from which the MacOnCmd is not available. */ )
	{
		/* This payload type will be used if Cmd.SAM_CombinedRead called for Mac On Command calculation.
		 * Which means the PICC command information will first be exchanged with Sam, then with PICC and
		 * at last again to the Sam.
		 *
		 * For this type the following will be applicable.
		 *		First exchagne will be performed with Sam to receive MAC. Then the MAC will be appended with
		 *		command and will be exachnge with PICC. The PICC response will then be excahnge with SAM again.
		 *		If the data is encrypted, Sam will decrypt the data and respond the plain data.
		 *		If the data is not encrypted, Sam will not respond back the data.
		 */
		if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_ID)
		{
			bPayloadType = PHHAL_HW_SAMAV2_CMD_SAM_COMBINED_READ_MFP_RESPONSE;
		}
		else
		{
			#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
			bPayloadType = PHHAL_HW_SAMAV3_OPTION_COMBINED_READ_MFP_RESPONSE;
			#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
		}
	}
	else
	{
		/* This payload type will be used if Cmd.SAM_CombinedRead is not called for Mac On Command calculation.
		 * Which means the PICC command information will first be exchanged with PICC and at last it will be
		 * exchanged with Sam.
		 *
		 * For this type the following will be applicable.
		 *		First exchange will be performed with PICC and the data will be received. The received data along
		 *		with command information and PICC status code will be exchanged with SAM.
		 *		If the data is encrypted, Sam will decrypt the data and respond the plain data.
		 *		If the data is not encrypted, Sam will not respond back the data.
		 */
		if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_ID)
		{
			bPayloadType = PHHAL_HW_SAMAV2_CMD_SAM_COMBINED_READ_MFP_COMMAND_RESPONSE;
		}
		else
		{
			#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
			bPayloadType = PHHAL_HW_SAMAV3_OPTION_COMBINED_READ_MFP_BOTH;
			#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
		}

	}

	/* Update the variable if MacOnResponse is available. This variable is used to remove the Mac if
	 * PICC response data is PLAIN as SAM will not respond back the data.
	 */
	if(bCmdCode & 0x01)
	{
		bMacLen = 8;
	}

    /* Reset the command buffer and its length variable. */
	bCmdLen = 0;
	memset(aCmdBuff, 0x00, sizeof(aCmdBuff));	/* PRQA S 3200 */

	/* Frame the command information. */
	aCmdBuff[bCmdLen++] = bCmdCode;
    aCmdBuff[bCmdLen++] = (uint8_t)(wBlockNr & 0xff); /* LSB */
    aCmdBuff[bCmdLen++] = (uint8_t)(wBlockNr >> 8);   /* MSB */
    aCmdBuff[bCmdLen++] = bNumBlocks;

/* Exchange the command information to Sam for MAC reception ------------------------------------------------------------------------- */
	if((bPayloadType == PHHAL_HW_SAMAV2_CMD_SAM_COMBINED_READ_MFP_RESPONSE)
		#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
		|| (bPayloadType == PHHAL_HW_SAMAV3_OPTION_COMBINED_READ_MFP_RESPONSE)
		#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
	  )
    {
		/* Get the Mac for command information from Sam hardware. */
		if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_ID)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_SAM_CombinedReadMfp(
				((phalMfp_SamAV2_DataParams_t *)pDataParams)->pHalSamDataParams,
				PH_EXCHANGE_DEFAULT | PHHAL_HW_SAMAV2_CMD_SAM_COMBINED_READ_MFP_COMMAND,
				aCmdBuff,
				bCmdLen,
				&pResp_Sam,
				&wRespLen_Sam));
		}
		else
		{
			#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
			PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_CombinedReadMFP(
				((phalMfp_SamAV3_NonX_DataParams_t *)pDataParams)->pHalSamDataParams,
				PHHAL_HW_SAMAV3_ISO7816_LAST_FRAME,
				PH_EXCHANGE_DEFAULT | PHHAL_HW_SAMAV3_OPTION_COMBINED_READ_MFP_COMMAND,
				aCmdBuff,
				bCmdLen,
				&pResp_Sam,
				&wRespLen_Sam,
				&bPiccErrCode));
			#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
		}
	}

/* Exchange the command information to PICC ========================================================================================== */
	wOption = PH_EXCHANGE_BUFFER_LAST;
	do
	{
		/* Buffer the command information to exchange buffer. */
		if(wOption != PH_EXCHANGE_RXCHAINING)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Sam_NonX_Int_CardExchange(
				pDataParams,
				PH_EXCHANGE_BUFFER_FIRST,
				PH_ON,
				aCmdBuff,
				bCmdLen,
				NULL,
				NULL,
				NULL));
		}

		/* Buffer the SAM's response information to exchange buffer and exchange the bufferred information card. */
		wStatus = phalMfp_Sam_NonX_Int_CardExchange(
			pDataParams,
			wOption,
			PH_ON,
			pResp_Sam,
			(uint16_t) ((wOption == PH_EXCHANGE_RXCHAINING) ? 0 : wRespLen_Sam),
			&pResp_Card,
			&wRespLen_Card,
			&bPiccErrCode);

		memcpy(&pBlocks[wOffset], pResp_Card, wRespLen_Card);  /* PRQA S 3200 */

		/* Update the Bufferring Option to Chaining. */
		if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
		{
			wOption = PH_EXCHANGE_RXCHAINING;
		}
		else
		{
			bFinished = PH_ON;
		}

		/* Set the offset. */
		wOffset += wRespLen_Card;

	}while(!bFinished);

	/* Check the response if Mac On Cmd is not executed. */
	if((bPayloadType == PHHAL_HW_SAMAV2_CMD_SAM_COMBINED_READ_MFP_COMMAND_RESPONSE)
		#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
		|| (bPayloadType == PHHAL_HW_SAMAV3_OPTION_COMBINED_READ_MFP_BOTH)
		#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
	  )
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_ComputeErrorResponse(1, bPiccErrCode, PH_ON));
	}

	/* Update the remaining bytes. */
	wRemBytes = wOffset;
	wDataLen = wOffset;

	/* Update bufferring option to First. */
	wOption = PH_EXCHANGE_BUFFER_FIRST;

	/* Update LFI in case if there are more than one frame. */
	if(wRemBytes > PHAL_MFP_SAM_COMBINED_READ_MAX_FRAME_SIZE)
	{
		/* Update the remaining bytes. */
		wDataLen = PHAL_MFP_SAM_COMBINED_READ_MAX_FRAME_SIZE;

		if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_ID)
		{
			bLFI = PH_EXCHANGE_TXCHAINING;
		}
		else
		{
			#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
			bLFI = PHHAL_HW_SAMAV3_ISO7816_CHAINED_FRAME;
			#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
		}
	}

/* Exchange the information to SAM --------------------------------------------------------------------------------------------------- */
	/* Add the command information to Sam exchange buffer if the payload is BOTH. */
	if((bPayloadType == PHHAL_HW_SAMAV2_CMD_SAM_COMBINED_READ_MFP_COMMAND_RESPONSE)
		#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
		|| (bPayloadType == PHHAL_HW_SAMAV3_OPTION_COMBINED_READ_MFP_BOTH)
		#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
	  )
	{
		/* Buffer command information to excahnge buffer. */
		if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_ID)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_SAM_CombinedReadMfp(
				((phalMfp_SamAV2_DataParams_t *)pDataParams)->pHalSamDataParams,
				wOption | bPayloadType | bLFI,
				aCmdBuff,
				bCmdLen,
				&pResp_Sam,
				&wRespLen_Sam));

			/* Update bufferring option to CONT. */
			wOption = PH_EXCHANGE_BUFFER_CONT;
		}
		else
		{
			#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
			PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_CombinedReadMFP(
				((phalMfp_SamAV3_NonX_DataParams_t *)pDataParams)->pHalSamDataParams,
				bLFI,
				wOption | bPayloadType,
				aCmdBuff,
				bCmdLen,
				NULL,
				NULL,
				NULL));

			/* Update bufferring option to CONT. */
			wOption = PH_EXCHANGE_BUFFER_CONT;
			#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
		}
	}

	/* Buffer Picc error code to excahnge buffer. */
	if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_ID)
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_SAM_CombinedReadMfp(
			((phalMfp_SamAV2_DataParams_t *)pDataParams)->pHalSamDataParams,
			wOption | bPayloadType | bLFI,
			&bPiccErrCode,
			1,
			&pResp_Sam,
			&wRespLen_Sam));
	}
	else
	{
		#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_CombinedReadMFP(
			((phalMfp_SamAV3_NonX_DataParams_t *)pDataParams)->pHalSamDataParams,
			bLFI,
			wOption | bPayloadType,
			&bPiccErrCode,
			1,
			NULL,
			NULL,
			NULL));
		#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
	}

	/* Update bufferring option to Last. */
	wOption = PH_EXCHANGE_BUFFER_LAST;

    /* Start the chaining loop */
	wOffset = 0;

    /* Start the chaining loop */
    do
    {
		/* Buffer Picc response data to excahnge buffer. */
		if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_ID)
		{
			wStatus = phhalHw_SamAV2_Cmd_SAM_CombinedReadMfp(
				((phalMfp_SamAV2_DataParams_t *)pDataParams)->pHalSamDataParams,
				wOption | bPayloadType | bLFI,
				&pBlocks[wOffset],
				(uint8_t) wDataLen,
				&pResp_Sam,
				&wRespLen_Sam);

			bPiccRetCode = bPiccErrCode;
		}
		else
		{
			#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
			wStatus = phhalHw_SamAV3_Cmd_SAM_CombinedReadMFP(
				((phalMfp_SamAV3_NonX_DataParams_t *)pDataParams)->pHalSamDataParams,
				bLFI,
				wOption | bPayloadType,
				&pBlocks[wOffset],
				(uint8_t) wDataLen,
				&pResp_Sam,
				&wRespLen_Sam,
				&bPiccRetCode);
			#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
		}

		/* Check for chaining response. */
		if((bLFI == PH_EXCHANGE_TXCHAINING)
			#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
			|| (bLFI == PHHAL_HW_SAMAV3_ISO7816_CHAINED_FRAME)
			#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
		  )
		{
			/* Check if chaining status is returned from HAL. */
			if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
			{
				/* Clear the memory in case of error. */
				memset(pBlocks, 0, bNumBlocks * 16);  /* PRQA S 3200 */

				return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFP);
			}
		}
		/* Check for PICC status information. */
		else if ((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV2_ERR_MIFARE_PLUS_GEN)
		{
			/* Clear the memory in case of error. */
			memset(pBlocks, 0, bNumBlocks * 16);  /* PRQA S 3200 */

			/* Compute the PICC error response. */
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_ComputeErrorResponse(1, bPiccRetCode, PH_ON));
		}
		/* Check for other status information. */
		else
		{
			if(wStatus != PH_ERR_SUCCESS)
			{
				/* Clear the memory in case of error. */
				memset(pBlocks, 0, bNumBlocks * 16);  /* PRQA S 3200 */
			}

			PH_CHECK_SUCCESS(wStatus);
		}

		/* Copy the data to the parameter. */
		memcpy(&pBlocks[wOffset1], pResp_Sam, wRespLen_Sam);  /* PRQA S 3200 */
		wOffset1 += (uint16_t) wRespLen_Sam;

		/* Update the variables. */
		wOffset += PHAL_MFP_SAM_COMBINED_READ_MAX_FRAME_SIZE;
		wRemBytes = (uint16_t) (wRemBytes - PHAL_MFP_SAM_COMBINED_READ_MAX_FRAME_SIZE);

		/* Update bufferring option to DEFAULT. */
		wOption = PH_EXCHANGE_DEFAULT;

		/* Reset the Payload Type. */
		bPayloadType = 0;

		/* Update LFI in case if there are more than one frame. */
		if(wRemBytes >= PHAL_MFP_SAM_COMBINED_READ_MAX_FRAME_SIZE)
		{
			if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_ID)
			{
				bLFI = PH_EXCHANGE_TXCHAINING;
			}
			else
			{
				#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
				bLFI = PHHAL_HW_SAMAV3_ISO7816_CHAINED_FRAME;
				#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
			}
		}
		/* Update the LFI flag if last frame. */
		else
		{
			wDataLen = wRemBytes;

			if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_ID)
			{
				bLFI = 0x00;
			}
			else
			{
				#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
				bLFI = PHHAL_HW_SAMAV3_ISO7816_LAST_FRAME;
				#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
			}
		}

    } while (wStatus != (PH_ERR_MASK & PH_ERR_SUCCESS));

	/* Clear the MAC information. */
	memset(&pBlocks[bNumBlocks * 16], 0, 8);  /* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_NonX_Int_ChangeKeyMFP(void * pDataParams, uint8_t bCommand, uint16_t wBlockNr, uint16_t wKeyNum, uint16_t wKeyVer,
	uint8_t bDivInputLen, uint8_t * pDivInput)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;
	uint8_t		PH_MEMLOC_REM bPiccRetCode = 0;
    uint8_t		PH_MEMLOC_REM aCmdBuff[PHAL_MFP_CHANGE_KEY_CMD_SIZE];
    uint8_t		PH_MEMLOC_REM bCmdLen = 0;
	uint8_t *	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint8_t 	PH_MEMLOC_REM aPayload[36];
	uint8_t 	PH_MEMLOC_REM bPayloadLen = 0;

#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
	uint8_t		PH_MEMLOC_REM bOption = 0;
#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */

	if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_ID)
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_SAM_ChangeKeyMfp_Part1(
			((phalMfp_SamAV2_DataParams_t *)pDataParams)->pHalSamDataParams,
			bCommand,
			wBlockNr,
			(uint8_t) wKeyNum,
			(uint8_t) wKeyVer,
			pDivInput,
			bDivInputLen,
			&pResponse,
			&wRespLen));
	}
	else
	{
		#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
		/* Reset the payload buffer and its length variable. */
		bPayloadLen = 0;
		memset(aPayload, 0x00, sizeof(aPayload));	/* PRQA S 3200 */

		/* Frame the payload to be sent to Sam hardware. */
		aPayload[bPayloadLen++] = bCommand;
		aPayload[bPayloadLen++] = (uint8_t) (wBlockNr & 0x00FF);
		aPayload[bPayloadLen++] = (uint8_t) ((wBlockNr & 0xFF00) >> 8);
		aPayload[bPayloadLen++] = (uint8_t) wKeyNum;
		aPayload[bPayloadLen++] = (uint8_t) wKeyVer;

		/* Update option variable with diversification off flag. */
		bOption = PHHAL_HW_SAMAV3_MFP_CHANGE_KEY_DIVERSIFICATION_OFF;

		/* Add Diversification input to command buffer. */
 		if(bDivInputLen)
		{
			bOption = PHHAL_HW_SAMAV3_MFP_CHANGE_KEY_DIVERSIFICATION_ON;

			/* Add the DivInput information to command buffer. */
			memcpy(&aPayload[bPayloadLen], pDivInput, bDivInputLen);	/* PRQA S 3200 */
			bPayloadLen += bDivInputLen;
		}

		/* Exchange the details to SAM hardware and get the protected data --------------------------------------------------------------- */
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ChangeKeyMFP(
			((phalMfp_SamAV3_NonX_DataParams_t *)pDataParams)->pHalSamDataParams,
			(uint8_t) (PHHAL_HW_SAMAV3_OPTION_MFP_CHANGE_KEY_COMMAND | bOption),
			aPayload,
			bPayloadLen,
			&pResponse,
			&wRespLen,
			NULL));
		#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
	}

	/* Exchange the command information to the card ================================================================================== */

	/* Reset the command buffer and its length variable. */
	bCmdLen = 0;
	memset(aCmdBuff, 0x00, sizeof(aCmdBuff));	/* PRQA S 3200 */

	/* Frame the command information. */
	aCmdBuff[bCmdLen++] = bCommand;
    aCmdBuff[bCmdLen++] = (uint8_t)(wBlockNr & 0xff); /* LSB */
    aCmdBuff[bCmdLen++] = (uint8_t)(wBlockNr >> 8);   /* MSB */

	/* Append the protected data to command buffer. */
	memcpy(&aCmdBuff[bCmdLen], pResponse, wRespLen);	/* PRQA S 3200 */
	bCmdLen += (uint8_t) wRespLen;

	wStatus = phalMfp_Sam_NonX_Int_CardExchange(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PH_ON,
		aCmdBuff,
		bCmdLen,
		&pResponse,
        &wRespLen,
		&bPiccErrCode);

	/*Exchange the response received from card to SAM hardware ----------------------------------------------------------------------- */

	/* Reset the payload buffer and its length variable. */
	bPayloadLen = 0;
	memset(aPayload, 0x00, sizeof(aPayload));	/* PRQA S 3200 */

	/* Add Picc return code. */
	aPayload[bPayloadLen++] = bPiccErrCode;

	/* Add the response received from PICC. */
	memcpy(&aPayload[bPayloadLen], pResponse, wRespLen);	/* PRQA S 3200 */
	bPayloadLen += (uint8_t) wRespLen;

	/* Reset the buffers and variables. */
	pResponse = NULL;
	wRespLen = 0;

	/* Exchange the details to SAM hardware and get the protected data. */
	if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_ID)
	{
		wStatus = phhalHw_SamAV2_Cmd_SAM_ChangeKeyMfp_Part2(
			((phalMfp_SamAV2_DataParams_t *)pDataParams)->pHalSamDataParams,
			aPayload,
			bPayloadLen);

		bPiccRetCode = bPiccErrCode;
	}
	else
	{
		#ifdef NXPBUILD__PHAL_MFP_SAMAV3_NONX
		bPiccRetCode = 0;

		wStatus = phhalHw_SamAV3_Cmd_SAM_ChangeKeyMFP(
			((phalMfp_SamAV3_NonX_DataParams_t *)pDataParams)->pHalSamDataParams,
			PHHAL_HW_SAMAV3_OPTION_MFP_CHANGE_KEY_RESPONSE,
			aPayload,
			bPayloadLen,
			&pResponse,
			&wRespLen,
			&bPiccRetCode);
		#endif /* NXPBUILD__PHAL_MFP_SAMAV3_NONX */
	}

	/* Return the error code. */
	if ((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV2_ERR_MIFARE_PLUS_GEN)
	{
		/* Compute the response code. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_ComputeErrorResponse(1, bPiccRetCode, PH_ON));
	}
	else
	{
		/* Return the error code other than success.*/
		if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
		{
			return wStatus;
		}
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_NonX_Int_ProximityCheck(void * pDataParams, uint8_t * pRndC, uint8_t bPps1, uint8_t bNumSteps)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuff[9];
	uint8_t		PH_MEMLOC_REM bCmdLen = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;
	uint8_t		PH_MEMLOC_REM aRndRC[14];
	uint8_t		PH_MEMLOC_REM aMac[8];
	uint8_t *	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Perform Prepare PC command. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_PrepareProximityCheck(PHAL_MFP_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams)));

	/* Perform ProximityCheck command. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_ProximityCheck(
		PHAL_MFP_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
		bNumSteps,
		pRndC,
		aRndRC));

	/* Exchange the Random RC information to Sam Hardware. */
	wStatus = phhalHw_SamAV2_Cmd_SAM_ProximityCheckMfp_Part1(
		((phalMfp_SamAV2_DataParams_t *)pDataParams)->pHalSamDataParams,
		0xFF,
		0xFF,
		bPps1,
		aRndRC,
		NULL,
		0,
		aMac);

	/* Check if chaining status is returned from HAL. */
	if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
	{
		/* Return the status. */
		return wStatus;
	}

	/* Exchange the Mac to PICC. */
	aCmdBuff[bCmdLen++] = 0xFD;

	memcpy ( &aCmdBuff[bCmdLen], aMac, 8);	/* PRQA S 3200 */
	bCmdLen += 8;

	/* Exchange VerifyProximityCheck command to PICC. */
	 wStatus = phalMfp_Sam_NonX_Int_CardExchange(
		pDataParams,
		PH_EXCHANGE_DEFAULT,
		PH_ON,
		aCmdBuff,
		bCmdLen,
		&pResponse,
		&wRespLen,
		&bPiccErrCode);

	/*Exchange the MAC received from PICC to SAM. */
	 PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_SAM_ProximityCheckMfp_Part2(
		((phalMfp_SamAV2_DataParams_t *)pDataParams)->pHalSamDataParams,
		pResponse));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}
#endif /* NXPBUILD__PHAL_MFP_SAM_NONX */
