/*
 * Copyright 2020, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM X MIFARE Plus EVx (Ev1, and future versions) Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phTMIUtils.h>
#include <phalMfpEVx.h>
#include <phpalMifare.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHAL_MFPEVX_SAM_X

#include "../phalMfpEVx_Int.h"
#include "phalMfpEVx_Sam_X.h"
#include "phalMfpEVx_Sam_X_Int.h"
#include <phhalHw_SamAV3_Cmd.h>

phStatus_t phalMfpEVx_SamAV3_X_Init(phalMfpEVx_SamAV3_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams,
    phhalHw_SamAV3_DataParams_t * pHalSamDataParams, phpalMifare_SamAV3_X_DataParams_t * pPalMifareDataParams, phTMIUtils_t * pTMIDataParams)
{
    /* Check DataParams size. */
    if(sizeof(phalMfpEVx_SamAV3_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPEVX);
    }

    /* Verify the parameters. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPEVX);
    PH_ASSERT_NULL_PARAM(pHalSamDataParams, PH_COMP_AL_MFPEVX);
    PH_ASSERT_NULL_PARAM(pPalMifareDataParams, PH_COMP_AL_MFPEVX);

    /* Store the parameter info to internal structure members. */
    pDataParams->wId = PH_COMP_AL_MFPEVX | PHAL_MFPEVX_SAMAV3_X_ID;
    pDataParams->pHalSamDataParams = pHalSamDataParams;
    pDataParams->pPalMifareDataParams = pPalMifareDataParams;
    pDataParams->pTMIDataParams = pTMIDataParams;
    pDataParams->bAuthMode = 0x00U;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}


/***************************************************************************************************************************************/
/* MIFARE Plus EVx Generic command for personalization.																				   */
/***************************************************************************************************************************************/
phStatus_t phalMfpEVx_Sam_X_WritePerso(void * pDataParams, uint8_t bLayer4Comm, uint16_t wBlockNr, uint8_t bNumBlocks, uint8_t * pValue)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bRespCode = 0;
    uint16_t	PH_MEMLOC_REM wOption = 0;
    uint8_t		PH_MEMLOC_REM bBlockNo = 0;
    uint8_t		PH_MEMLOC_REM aBlocks[18];

    /* Validate if Number of blocks is greater than 13. */
    if(bNumBlocks > 13)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPEVX);
    }

    /* Call Hal WritePerso command to frame the initial header. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_MFP_WritePerso(
        PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        PH_EXCHANGE_BUFFER_FIRST,
        NULL,
        0U,
        NULL));


    /* Buffer the blocks information to Hal layer. */
    for(bBlockNo = 0; bBlockNo < bNumBlocks; bBlockNo++)
    {
        /* Reset the blocks buffer. */
        (void) memset(aBlocks, 0x00U, sizeof(aBlocks));

        /* Update the buffering Option. */
        wOption = (uint16_t) ((bBlockNo == (bNumBlocks - 1)) ? PH_EXCHANGE_BUFFER_LAST : PH_EXCHANGE_BUFFER_CONT);

        /* Frame the blocks information. */
        aBlocks[0U] = (uint8_t) ((wBlockNr + bBlockNo) & 0x00FFU);
        aBlocks[1U] = (uint8_t) (((wBlockNr + bBlockNo) & 0xFF00U) >> 8U);
        (void) memcpy(&aBlocks[2U], &pValue[16U * bBlockNo], 16U);

        /* Call Hal WritePerso command to frame the blocks information and exchange the command. */
        wStatus = phhalHw_SamAV3_Cmd_MFP_WritePerso(
            PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
            wOption,
            aBlocks,
            18U,
            &bRespCode);

        /* Return the error code. */
        if(((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_PLUS_GEN) ||
            ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS))
        {
            if(wOption == PH_EXCHANGE_BUFFER_LAST)
            {
                /* Compute the response code. */
                PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_ComputeErrorResponse(1, bRespCode, bLayer4Comm));
            }
        }
        else
        {
            return wStatus;
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_CommitPerso(void * pDataParams, uint8_t bOption, uint8_t bLayer4Comm)
{
    return phalMfpEVx_Int_CommitPerso(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams, bOption,
        bLayer4Comm, PH_OFF, PH_OFF);
}


/***************************************************************************************************************************************/
/* MIFARE Plus EVx Generic command for authentication.																				   */
/***************************************************************************************************************************************/

phStatus_t phalMfpEVx_Sam_X_AuthenticateMfc(void * pDataParams, uint8_t bOption, uint8_t bBlockNo, uint8_t bKeyType,
    uint16_t wKeyNo, uint16_t wKeyVer, uint8_t * pUid, uint8_t bUidLen)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;

    /* Exchange the information to Hal Sam layer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_MF_Authenticate(
        PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        bOption,
        &pUid[bUidLen - 4U],
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        bKeyType,
        bBlockNo,
        bBlockNo));

    /* Update the Auth Mode to MIFARE Authenticated. */
    PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode = (uint8_t) PHAL_MFPEVX_SL1_MIFARE_AUTHENTICATED;

    /* Enable MIFARE Crypto. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams), PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1,
        PH_OFF));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_AuthenticateSL0(void * pDataParams, uint8_t bLayer4Comm, uint8_t bFirstAuth, uint16_t wBlockNr,
    uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bDivInputLen, uint8_t * pDivInput, uint8_t bPcdCap2InLen, uint8_t * pPcdCap2In,
    uint8_t * pPcdCap2Out, uint8_t * pPdCap2)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_MFP_Authenticate(
        pDataParams,
        bLayer4Comm,
        bFirstAuth,
        PHHAL_HW_SAMAV3_MFP_AUTHENTICATE_SL1_NO_KDF,
        wBlockNr,
        wKeyNo,
        wKeyVer,
        pDivInput,
        bDivInputLen,
        pPcdCap2In,
        bPcdCap2InLen,
        pPcdCap2Out,
        pPdCap2));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_AuthenticateSL1(void * pDataParams, uint8_t bLayer4Comm, uint8_t bFirstAuth, uint16_t wBlockNr,
    uint16_t wKeyNo, uint16_t wKeyVer, uint8_t bDivInputLen, uint8_t * pDivInput, uint8_t bPcdCap2InLen, uint8_t * pPcdCap2In,
    uint8_t * pPcdCap2Out, uint8_t * pPdCap2)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bKdf = PHHAL_HW_SAMAV3_MFP_AUTHENTICATE_SL3_KDF;

    /* Update the KDF. */
    if(((wBlockNr == PHAL_MFPEVX_SL1CARDAUTHKEY) && (bLayer4Comm == PHAL_MFPEVX_ISO14443_L3)) ||
        ((wBlockNr >= PHAL_MFPEVX_ORIGINALITY_KEY_FIRST) && (wBlockNr <= PHAL_MFPEVX_ORIGINALITY_KEY_LAST)))
        bKdf = PHHAL_HW_SAMAV3_MFP_AUTHENTICATE_SL1_NO_KDF;

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_MFP_Authenticate(
        pDataParams,
        bLayer4Comm,
        bFirstAuth,
        bKdf,
        wBlockNr,
        wKeyNo,
        wKeyVer,
        pDivInput,
        bDivInputLen,
        pPcdCap2In,
        bPcdCap2InLen,
        pPcdCap2Out,
        pPdCap2));

    /* Not updating the state in case authenticated using special keys. */
    if(((wBlockNr != PHAL_MFPEVX_SL1CARDAUTHKEY) || (bLayer4Comm != 0x00U)) &&
        (wBlockNr != PHAL_MFPEVX_L3SECTORSWITCHKEY) && (wBlockNr != PHAL_MFPEVX_L3SWITCHKEY) &&
        ((wBlockNr <= PHAL_MFPEVX_ORIGINALITY_KEY_FIRST) || (wBlockNr >= PHAL_MFPEVX_ORIGINALITY_KEY_LAST)))
    {
        PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode = (uint8_t) PHAL_MFPEVX_SL1_MFP_AUTHENTICATED;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_AuthenticateSL3(void * pDataParams, uint8_t bFirstAuth, uint16_t wBlockNr, uint16_t wKeyNo,
    uint16_t wKeyVer, uint8_t bDivInputLen, uint8_t * pDivInput, uint8_t bPcdCap2InLen, uint8_t * pPcdCap2In,
    uint8_t * pPcdCap2Out, uint8_t * pPdCap2)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bKdf = PHHAL_HW_SAMAV3_MFP_AUTHENTICATE_SL3_KDF;

    /* Update the KDF. */
    if((wBlockNr >= PHAL_MFPEVX_ORIGINALITY_KEY_FIRST) && (wBlockNr <= PHAL_MFPEVX_ORIGINALITY_KEY_LAST))
        bKdf = PHHAL_HW_SAMAV3_MFP_AUTHENTICATE_SL1_NO_KDF;

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_MFP_Authenticate(
        pDataParams,
        PH_ON,
        bFirstAuth,
        bKdf,
        wBlockNr,
        wKeyNo,
        wKeyVer,
        pDivInput,
        bDivInputLen,
        pPcdCap2In,
        bPcdCap2InLen,
        pPcdCap2Out,
        pPdCap2));

    if(wStatus == PH_ERR_SUCCESS)
    {
        PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode = (uint8_t) PHAL_MFPEVX_SL3_MFP_AUTHENTICATED;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_SSAuthenticate(void * pDataParams, uint8_t bOption, uint16_t wSSKeyBNr, uint16_t wSSKeyNr, uint16_t wSSKeyVer,
    uint8_t bLenDivInputSSKey, uint8_t * pDivInputSSKey, uint8_t  bSecCount, uint16_t * pSectorNos, uint16_t * pKeyNos,
    uint16_t * pKeyVers, uint8_t bLenDivInputSectorKeyBs, uint8_t * pDivInputSectorKeyBs)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM aPayload[32];
    uint8_t		PH_MEMLOC_REM bPayloadLen = 0;
    uint8_t		PH_MEMLOC_REM bIteration = 0;
    uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;

    /* Buffer the initial set if information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_MFP_AuthSectorSwitch(
        PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        (uint16_t) (PH_EXCHANGE_BUFFER_FIRST | bOption),
        wSSKeyBNr,
        (uint8_t) wSSKeyNr,
        (uint8_t) wSSKeyVer,
        (uint8_t) ((bOption & 0x04U) ? pKeyNos[0U] : 0x00U),
        (uint8_t) ((bOption & 0x04U) ? pKeyVers[0U] : 0x00U),
        bSecCount,
        NULL,
        0U,
        NULL,
        0U,
        NULL));

    /* Buffer the KeyBlocks information Sam exchange buffer. */
    for(bIteration = 0U; bIteration < bSecCount; bIteration++)
    {
        /* Reset payload length variable. */
        bPayloadLen = 0U;

        aPayload[bPayloadLen++] = (uint8_t) (pSectorNos[bIteration] & 0x00FFU);
        aPayload[bPayloadLen++] = (uint8_t) ((pSectorNos[bIteration] & 0xFF00U) >> 8U);

        if((bOption & 0x04) != 0x04 /* Master Key disabled. */)
        {
            aPayload[bPayloadLen++] = (uint8_t) pKeyNos[bIteration];
            aPayload[bPayloadLen++] = (uint8_t) pKeyVers[bIteration];
        }

        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_MFP_AuthSectorSwitch(
            PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
            PH_EXCHANGE_BUFFER_CONT | PHHAL_HW_SAMAV3_MFP_SSAUTH_BUFFER_KEY_BLOCKS,
            0U,
            0U,
            0U,
            0x00U,
            0x00U,
            0U,
            aPayload,
            bPayloadLen,
            NULL,
            0U,
            NULL));
    }

    /* Add Sector Switch Key diversification input to Sam exchange buffer. */
    if(bOption & 0x01U)
    {
        /* Reset payload length variable. */
        bPayloadLen = 0U;

        /* Add the length information. */
        aPayload[bPayloadLen++] = bLenDivInputSSKey;

        /* Add the diversification input. */
        (void) memcpy(&aPayload[bPayloadLen], pDivInputSSKey, bLenDivInputSSKey);
        bPayloadLen += bLenDivInputSSKey;

        /* Buffer the Sector Switch Key DivInput information Sam exchange buffer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_MFP_AuthSectorSwitch(
            PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
            PH_EXCHANGE_BUFFER_CONT | PHHAL_HW_SAMAV3_MFP_SSAUTH_BUFFER_DIV_INPUT,
            0U,
            0U,
            0U,
            0U,
            0U,
            0U,
            NULL,
            0U,
            aPayload,
            bPayloadLen,
            NULL));
    }

    /* Add Sector Key diversification input to Sam exchange buffer. */
    if(bOption & 0x02U)
    {
        /* Reset payload length variable. */
        bPayloadLen = 0U;

        /* Adding the length information because diversification input for Sector keys are different. */
        if(bLenDivInputSectorKeyBs)
        {
            aPayload[bPayloadLen++] = bLenDivInputSectorKeyBs;
        }

        /* Adding the length information as zero because diversification input for Sector keys are same. */
        else
        {
            aPayload[bPayloadLen++] = 0x00U;
        }

        (void) memcpy(&aPayload[bPayloadLen], pDivInputSectorKeyBs, bLenDivInputSectorKeyBs);
        bPayloadLen += bLenDivInputSectorKeyBs;

        /* Buffer the KeyB DivInput information Sam exchange buffer. */
        wStatus = phhalHw_SamAV3_Cmd_MFP_AuthSectorSwitch(
            PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
            PH_EXCHANGE_BUFFER_CONT | PHHAL_HW_SAMAV3_MFP_SSAUTH_BUFFER_DIV_INPUT,
            0U,
            0U,
            0U,
            0U,
            0U,
            0U,
            NULL,
            0U,
            aPayload,
            bPayloadLen,
            &bPiccErrCode);
    }

    /* Exchange the buffered information to Sam hardware. */
    wStatus = phhalHw_SamAV3_Cmd_MFP_AuthSectorSwitch(
        PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        PH_EXCHANGE_BUFFER_LAST,
        0U,
        0U,
        0U,
        0U,
        0U,
        0U,
        NULL,
        0U,
        NULL,
        0U,
        &bPiccErrCode);

    /* Validate the error code. */
    if(((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_PLUS_GEN) ||
        ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS))
    {
        /* Check for PICC response code in case of error only. For Success the PiccErrCode = 0. */
        if(bPiccErrCode)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_ComputeErrorResponse(1, bPiccErrCode, PH_ON));
        }
    }
    else
    {
        return wStatus;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_AuthenticatePDC(void * pDataParams, uint16_t wBlockNr, uint16_t wKeyNo, uint16_t wKeyVer,
    uint8_t bDivInputLen, uint8_t * pDivInput, uint8_t bUpgradeInfo)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;

    wStatus = phhalHw_SamAV3_Cmd_PDC_Authenticate(
        PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        PHHAL_HW_SAMAV3_PDC_AUTH_DERIVE_UPGRADE_KEY,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        wBlockNr,
        &bUpgradeInfo,
        1U,
        pDivInput,
        bDivInputLen,
        &bPiccErrCode);

    /* Return the error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_GEN)
    {
        /* Compute the response code. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_ComputeErrorResponse(1U, bPiccErrCode, PH_ON));
    }
    else
    {
        return wStatus;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}



/***************************************************************************************************************************************/
/* MIFARE Plus EVx Generic command for data operations.																				   */
/***************************************************************************************************************************************/
phStatus_t phalMfpEVx_Sam_X_Write(void * pDataParams, uint8_t bEncrypted, uint8_t bWriteMaced, uint16_t wBlockNr, uint8_t bNumBlocks,
    uint8_t * pBlocks, uint8_t * pTMC, uint8_t * pTMV)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bCmdCode = 0;
    uint8_t		PH_MEMLOC_REM bOption = 0;
    uint8_t		PH_MEMLOC_REM aData[18];
    uint8_t		PH_MEMLOC_REM bDataLen = 0;
    uint8_t *	PH_MEMLOC_REM pTMData = NULL;
    uint16_t	PH_MEMLOC_REM wHostMode = 0;
    uint16_t	PH_MEMLOC_REM wTMDataLen = 0;

    /* Perform Write according to the auth mode.*/
    switch(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode)
    {
        /* Perform write command in MIFARE mode. (ISO14443 Layer 3 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L3:
        case PHAL_MFPEVX_SL1_MIFARE_AUTHENTICATED:
            /* Frame the data buffer. */
            aData[bDataLen++] = PHAL_MFPEVX_CMD_MFC_WRITE;
            aData[bDataLen++] = (uint8_t) wBlockNr;
            (void) memcpy(&aData[bDataLen], pBlocks, PHAL_MFPEVX_DATA_BLOCK_SIZE);
            bDataLen += PHAL_MFPEVX_DATA_BLOCK_SIZE;

            /* Get the Host Mode. */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams), PHHAL_HW_SAMAV3_CONFIG_HOSTMODE,
                &wHostMode));

            /* Perform TMICollection. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_CollectTMI(pDataParams, aData, bDataLen, NULL, 0U));

            /* Frame the Option. */
            bOption = PHHAL_HW_SAMAV3_CMD_MF_WRITE_CLASSIC;
            bOption |= (uint8_t) ((wHostMode == PHHAL_HW_SAMAV2_HC_AV2_MODE) ? PHHAL_HW_SAMAV3_CMD_MF_WRITE_TMDATA_NOT_RETURNED :
                PHHAL_HW_SAMAV3_CMD_MF_WRITE_TMDATA_RETURNED);

            wStatus = phhalHw_SamAV3_Cmd_MF_Write(
                PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption,
                &aData[1],
                (uint16_t) (bDataLen - 1),
                &pTMData,
                &wTMDataLen);

            /* Convert to Pal codes. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_ConvertNak(wStatus));

            if(wTMDataLen)
            {
                /* Copy the TMData to parameters. */
                (void) memcpy(pTMC, &pTMData[0U], 4U);
                (void) memcpy(pTMV, &pTMData[4U], 8U);
            }
            break;

        /* Perform write command in MFP mode. (ISO14443 Layer 4 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L4:
        case PHAL_MFPEVX_SL1_MFP_AUTHENTICATED:
        case PHAL_MFPEVX_SL3_MFP_AUTHENTICATED:
            /* Evaluate the command code. */
            bCmdCode = (uint8_t) (PHAL_MFPEVX_CMD_WRITE_EN | ((!bEncrypted & 0x01U) << 1U) | bWriteMaced);

            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_MFP_CombinedWrite(
                pDataParams,
                bCmdCode,
                wBlockNr,
                0x00U,
                pBlocks,
                (bNumBlocks * PHAL_MFPEVX_DATA_BLOCK_SIZE),
                pTMC,
                pTMV));
            break;

        /* Return error in case of not authenticated in any one of the auth mode.*/
        default:
            return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPEVX);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_Read(void * pDataParams, uint8_t bEncrypted, uint8_t bReadMaced, uint8_t bMacOnCmd, uint16_t wBlockNr,
    uint8_t bNumBlocks, uint8_t * pBlocks)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bCmdCode = 0;
    uint8_t		PH_MEMLOC_REM aData[2];
    uint8_t		PH_MEMLOC_REM bDataLen = 0;
    uint8_t *	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Perform Read according to the auth mode.*/
    switch(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode)
    {
        /* Perform read command in MIFARE mode. (ISO14443 Layer 3 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L3:
        case PHAL_MFPEVX_SL1_MIFARE_AUTHENTICATED:
            /* Frame the command information. */
            aData[bDataLen++] = PHAL_MFPEVX_CMD_MFC_READ;
            aData[bDataLen++] = (uint8_t) (wBlockNr & 0x00FFU);

            wStatus = phhalHw_SamAV3_Cmd_MF_Read(
                PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                &aData[1],
                (uint16_t) (bDataLen - 1U),
                &pResponse,
                &wRespLen);

            /* Convert to Pal codes. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_ConvertNak(wStatus));

            /* Copy the response to the parameter. */
            (void) memcpy(pBlocks, pResponse, wRespLen);

            /* Perform TMICollection. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_CollectTMI(pDataParams, aData, bDataLen, pBlocks, wRespLen));
            break;

        /* Perform read command in MFP mode. (ISO14443 Layer 4 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L4:
        case PHAL_MFPEVX_SL1_MFP_AUTHENTICATED:
        case PHAL_MFPEVX_SL3_MFP_AUTHENTICATED:
            /* Evaluate the command code. */
            bCmdCode = (uint8_t) (PHAL_MFPEVX_CMD_READ_ENM | (((!bMacOnCmd & 0x01U) << 2U) | ((!bEncrypted & 0x01U) << 1U) | bReadMaced));

            /* Perform Read command. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_CombinedReadMFP(
                pDataParams,
                bCmdCode,
                wBlockNr,
                bNumBlocks,
                pBlocks));
            break;

        /* Return error in case if not authenticated in any one of the auth mode.*/
        default:
            return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPEVX);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}



/***************************************************************************************************************************************/
/* MIFARE Plus EVx Generic command for value operations.																			   */
/***************************************************************************************************************************************/
phStatus_t phalMfpEVx_Sam_X_WriteValue(void * pDataParams, uint8_t bEncrypted, uint8_t bWriteMaced, uint16_t wBlockNr, uint8_t * pValue,
	uint8_t bAddrData, uint8_t * pTMC, uint8_t * pTMV)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bCmdCode = 0;
	uint8_t		PH_MEMLOC_REM bOption = 0;
	uint8_t		PH_MEMLOC_REM aWriteDesc[9];
    uint8_t		PH_MEMLOC_REM aBlock[PHAL_MFPEVX_DATA_BLOCK_SIZE];
	uint8_t *	PH_MEMLOC_REM pTMData = NULL;
	uint16_t	PH_MEMLOC_REM wHostMode = 0;
	uint16_t	PH_MEMLOC_REM wTMDataLen = 0;

	/* Perform Write according to the auth mode. (ISO14443 Layer 3 activated)  */
	switch(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode)
	{
		case PHAL_MFPEVX_NOT_AUTHENTICATED_L3:
		case PHAL_MFPEVX_SL1_MIFARE_AUTHENTICATED:
			/*Frame the WriteDesc buffer information. */
			aWriteDesc[0] = (uint8_t) wBlockNr;
            (void) memcpy(&aWriteDesc[1], pValue, PHAL_MFPEVX_VALUE_BLOCK_SIZE);
			aWriteDesc[5] = bAddrData;
			aWriteDesc[6] = (uint8_t) ~bAddrData;
			aWriteDesc[7] = bAddrData;
			aWriteDesc[8] = (uint8_t) ~bAddrData;

			/* Get the Host Mode. */
			PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams), PHHAL_HW_SAMAV3_CONFIG_HOSTMODE,
				&wHostMode));

			/* Frame the Option. */
			bOption = (uint8_t) ((wHostMode == PHHAL_HW_SAMAV2_HC_AV2_MODE) ? PHHAL_HW_SAMAV3_CMD_MF_VALUE_WRITE_TMDATA_NOT_RETURNED :
				PHHAL_HW_SAMAV3_CMD_MF_VALUE_WRITE_TMDATA_RETURNED);

			wStatus = phhalHw_SamAV3_Cmd_MF_ValueWrite(
				PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
				bOption,
				aWriteDesc,
				9,
				&pTMData,
				&wTMDataLen);

			/* Convert to Pal codes. */
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_ConvertNak(wStatus));

			if(wTMDataLen)
			{
				/* Copy the TMData to parameters. */
                (void) memcpy(pTMC, &pTMData[0U], 4U);
                (void) memcpy(pTMV, &pTMData[4U], 8U);
			}

			/* Frame the buffer for TMI collection. */
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_CreateValueBlock(pValue, bAddrData, aBlock));

			aWriteDesc[0U] = PHAL_MFPEVX_CMD_MFC_WRITE;
			aWriteDesc[1U] = (uint8_t) wBlockNr;

			/* Perform TMICollection. */
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_CollectTMI(pDataParams, aWriteDesc, 2U, aBlock, PHAL_MFPEVX_DATA_BLOCK_SIZE));
			break;

		/* Perform write command in MFP mode. (ISO14443 Layer 4 activated) */
		case PHAL_MFPEVX_NOT_AUTHENTICATED_L4:
		case PHAL_MFPEVX_SL1_MFP_AUTHENTICATED:
		case PHAL_MFPEVX_SL3_MFP_AUTHENTICATED:
			/* Form the value to be written in block format.
			 *				 | 00 01 02 03 | 04 05 06 07 | 08 09 0A 0B |  0C  |    0D  |  0E  |   0F   |
			 * Value Block = |    Value    |    ~Value   |    Value    | Addr | ~ Addr | Addr | ~ Addr |
			 */
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_CreateValueBlock(pValue, bAddrData, aBlock));

			/* Evaluate the command code. */
			bCmdCode = (uint8_t) (PHAL_MFPEVX_CMD_WRITE_EN | ((!bEncrypted & 0x01U) << 1U) | bWriteMaced);

			PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_MFP_CombinedWrite(
				pDataParams,
				bCmdCode,
				wBlockNr,
				0x00U,
				aBlock,
				PHAL_MFPEVX_DATA_BLOCK_SIZE,
				pTMC,
				pTMV));
			break;

		/* Return error in case of not authenticated in any one of the auth mode.*/
		default:
			return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPEVX);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_ReadValue(void * pDataParams, uint8_t bEncrypted, uint8_t bReadMaced, uint8_t bMacOnCmd, uint16_t wBlockNr,
    uint8_t * pValue, uint8_t * pAddrData)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bCmdCode = 0;
    uint8_t		PH_MEMLOC_REM aData[PHAL_MFPEVX_DATA_BLOCK_SIZE];
    uint8_t		PH_MEMLOC_REM bDataLen = 0;
    uint16_t	PH_MEMLOC_REM wBlockLen = 0;
    uint8_t	*	PH_MEMLOC_REM pData = NULL;

    /* Perform Read according to the auth mode.*/
    switch(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode)
    {
        /* Perform read command in MIFARE mode. (ISO14443 Layer 3 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L3:
        case PHAL_MFPEVX_SL1_MIFARE_AUTHENTICATED:
            /* Frame the command information. */
            aData[bDataLen++] = PHAL_MFPEVX_CMD_MFC_READ;
            aData[bDataLen++] = (uint8_t) (wBlockNr & 0x00FFU);

            wStatus = phhalHw_SamAV3_Cmd_MF_Read(
                PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                &aData[1U],
                (uint16_t) (bDataLen - 1U),
                &pData,
                &wBlockLen);

            /* Convert to Pal codes. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_ConvertNak(wStatus));

            /* Perform TMICollection. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_CollectTMI(pDataParams, aData, bDataLen, pData, wBlockLen));

            /* Update the buffer. */
            (void) memcpy(aData, pData, PHAL_MFPEVX_DATA_BLOCK_SIZE);
            break;

        /* Perform read command in MFP mode. (ISO14443 Layer 4 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L4:
        case PHAL_MFPEVX_SL1_MFP_AUTHENTICATED:
        case PHAL_MFPEVX_SL3_MFP_AUTHENTICATED:
            /* Evaluate the command code. */
            bCmdCode = (uint8_t) (PHAL_MFPEVX_CMD_READ_ENM | (((!bMacOnCmd & 0x01U) << 2U) | ((!bEncrypted & 0x01U) << 1U) | bReadMaced));

            /* Perform Read command. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_CombinedReadMFP(
                pDataParams,
                bCmdCode,
                wBlockNr,
                0x01U,
                aData));
            break;

        /* Return error in case if not authenticated in any one of the auth mode.*/
        default:
            return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPEVX);
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_CheckValueBlockFormat(aData));

    *pAddrData = aData[12];
    (void) memcpy(pValue, aData, 4);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_Increment(void * pDataParams, uint8_t bIncrementMaced, uint16_t wBlockNr, uint8_t * pValue)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    /* Perform increment according to the auth mode.*/
    switch(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode)
    {
        /* Perform increment command in MIFARE mode. (ISO14443 Layer 3 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L3:
        case PHAL_MFPEVX_SL1_MIFARE_AUTHENTICATED:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_WriteExtMfc(
                pDataParams,
                PHAL_MFPEVX_CMD_MFC_INCREMENT,
                (uint8_t) wBlockNr,
                pValue,
                PHAL_MFPEVX_VALUE_BLOCK_SIZE,
                NULL,
                NULL));
            break;

        /* Perform increment command in MFP mode. (ISO14443 Layer 4 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L4:
        case PHAL_MFPEVX_SL1_MFP_AUTHENTICATED:
        case PHAL_MFPEVX_SL3_MFP_AUTHENTICATED:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_MFP_CombinedWrite(
                pDataParams,
                (uint8_t) (PHAL_MFPEVX_CMD_INCR | bIncrementMaced),
                wBlockNr,
                0x00U,
                pValue,
                PHAL_MFPEVX_VALUE_BLOCK_SIZE,
                NULL,
                NULL));
            break;

        /* Return error in case of not authenticated in any one of the auth mode.*/
        default:
            return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPEVX);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_Decrement(void * pDataParams, uint8_t bDecrementMaced, uint16_t wBlockNr, uint8_t * pValue)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    /* Perform decrement according to the auth mode.*/
    switch(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode)
    {
        /* Perform decrement command in MIFARE mode. (ISO14443 Layer 3 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L3:
        case PHAL_MFPEVX_SL1_MIFARE_AUTHENTICATED:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_WriteExtMfc(
                pDataParams,
                PHAL_MFPEVX_CMD_MFC_DECREMENT,
                (uint8_t) wBlockNr,
                pValue,
                PHAL_MFPEVX_VALUE_BLOCK_SIZE,
                NULL,
                NULL));
            break;

        /* Perform decrement command in MFP mode. (ISO14443 Layer 4 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L4:
        case PHAL_MFPEVX_SL1_MFP_AUTHENTICATED:
        case PHAL_MFPEVX_SL3_MFP_AUTHENTICATED:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_MFP_CombinedWrite(
                pDataParams,
                (uint8_t) (PHAL_MFPEVX_CMD_DECR | bDecrementMaced),
                wBlockNr,
                0x00U,
                pValue,
                PHAL_MFPEVX_VALUE_BLOCK_SIZE,
                NULL,
                NULL));
            break;

        /* Return error in case of not authenticated in any one of the auth mode.*/
        default:
            return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPEVX);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_IncrementTransfer(void * pDataParams, uint8_t bIncrementTransferMaced, uint16_t wSrcBlockNr,
    uint16_t wDstBlockNr, uint8_t * pValue, uint8_t * pTMC, uint8_t * pTMV)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bOption = 0;
    uint8_t		PH_MEMLOC_REM aIncrDesc[6];
    uint8_t *	PH_MEMLOC_REM pTMData = NULL;
    uint16_t	PH_MEMLOC_REM wHostMode = 0;
    uint16_t	PH_MEMLOC_REM wTMDataLen = 0;

    /* Perform increment transfer according to the auth mode.*/
    switch(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode)
    {
        /* Perform increment transfer command in MIFARE mode. (ISO14443 Layer 3 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L3:
        case PHAL_MFPEVX_SL1_MIFARE_AUTHENTICATED:
            /* Frame the IncrDesc information. */
            aIncrDesc[0] = (uint8_t) (wSrcBlockNr & 0x00FF);
            aIncrDesc[1] = (uint8_t) (wDstBlockNr & 0x00FF);
            (void) memcpy(&aIncrDesc[2], pValue, PHAL_MFPEVX_VALUE_BLOCK_SIZE);

            /* Get the Host Mode. */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams), PHHAL_HW_SAMAV3_CONFIG_HOSTMODE,
                &wHostMode));

            /* Frame the Option. */
            bOption |= (uint8_t) ((wHostMode == PHHAL_HW_SAMAV2_HC_AV2_MODE) ? PHHAL_HW_SAMAV3_CMD_MF_INCREMENT_TMDATA_NOT_RETURNED :
                PHHAL_HW_SAMAV3_CMD_MF_INCREMENT_TMDATA_RETURNED);

            /* Exchange the command to Sam hardware. */
            wStatus = phhalHw_SamAV3_Cmd_MF_Increment(
                PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption,
                aIncrDesc,
                6U,
                &pTMData,
                &wTMDataLen);

            /* Convert to Pal codes. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_ConvertNak(wStatus));

            /* Perform TMICollection. */
            aIncrDesc[0U] = PHAL_MFPEVX_CMD_MFC_INCREMENT;
            aIncrDesc[1U] = (uint8_t) wSrcBlockNr;
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_CollectTMI(pDataParams, aIncrDesc, 2U, pValue,
                PHAL_MFPEVX_VALUE_BLOCK_SIZE));

            aIncrDesc[0U] = PHAL_MFPEVX_CMD_MFC_TRANSFER;
            aIncrDesc[1U] = (uint8_t) wDstBlockNr;
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_CollectTMI(pDataParams, aIncrDesc, 2U, NULL, 0U));

            if(wTMDataLen)
            {
                /* Copy the TMData to parameters. */
                (void) memcpy(pTMC, &pTMData[0], 4);
                (void) memcpy(pTMV, &pTMData[4], 8);
            }
            break;

        /* Perform increment transfer command in MFP mode. (ISO14443 Layer 4 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L4:
        case PHAL_MFPEVX_SL1_MFP_AUTHENTICATED:
        case PHAL_MFPEVX_SL3_MFP_AUTHENTICATED:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_MFP_CombinedWrite(
                pDataParams,
                (uint8_t) (PHAL_MFPEVX_CMD_INCRTR | bIncrementTransferMaced),
                wSrcBlockNr,
                wDstBlockNr,
                pValue,
                PHAL_MFPEVX_VALUE_BLOCK_SIZE,
                pTMC,
                pTMV));
            break;

        /* Return error in case of not authenticated in any one of the auth mode.*/
        default:
            return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPEVX);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_DecrementTransfer(void * pDataParams, uint8_t bDecrementTransferMaced, uint16_t wSrcBlockNr,
    uint16_t wDstBlockNr, uint8_t * pValue, uint8_t * pTMC, uint8_t * pTMV)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bOption = 0;
    uint8_t		PH_MEMLOC_REM aDecrDesc[6];
    uint8_t *	PH_MEMLOC_REM pTMData = NULL;
    uint16_t	PH_MEMLOC_REM wHostMode = 0;
    uint16_t	PH_MEMLOC_REM wTMDataLen = 0;

    /* Perform decrement transfer according to the auth mode.*/
    switch(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode)
    {
        /* Perform decrement transfer command in MIFARE mode. (ISO14443 Layer 3 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L3:
        case PHAL_MFPEVX_SL1_MIFARE_AUTHENTICATED:
            /* Frame the DecrDesc information. */
            aDecrDesc[0] = (uint8_t) (wSrcBlockNr & 0x00FF);
            aDecrDesc[1] = (uint8_t) (wDstBlockNr & 0x00FF);
            (void) memcpy(&aDecrDesc[2], pValue, PHAL_MFPEVX_VALUE_BLOCK_SIZE);

            /* Get the Host Mode. */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams), PHHAL_HW_SAMAV3_CONFIG_HOSTMODE,
                &wHostMode));

            /* Frame the Option. */
            bOption |= (uint8_t) ((wHostMode == PHHAL_HW_SAMAV2_HC_AV2_MODE) ? PHHAL_HW_SAMAV3_CMD_MF_DECREMENT_TMDATA_NOT_RETURNED :
                PHHAL_HW_SAMAV3_CMD_MF_DECREMENT_TMDATA_RETURNED);

            /* Exchange the command to Sam hardware. */
            wStatus = phhalHw_SamAV3_Cmd_MF_Decrement(
                PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption,
                aDecrDesc,
                6U,
                &pTMData,
                &wTMDataLen);

            /* Convert to Pal codes. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_ConvertNak(wStatus));

            /* Perform TMICollection. */
            aDecrDesc[0] = PHAL_MFPEVX_CMD_MFC_DECREMENT;
            aDecrDesc[1] = (uint8_t) wSrcBlockNr;
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_CollectTMI(pDataParams, aDecrDesc, 2, pValue, PHAL_MFPEVX_VALUE_BLOCK_SIZE));

            aDecrDesc[0] = PHAL_MFPEVX_CMD_MFC_TRANSFER;
            aDecrDesc[1] = (uint8_t) wDstBlockNr;
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_CollectTMI(pDataParams, aDecrDesc, 2, NULL, 0));

            if(wTMDataLen)
            {
                /* Copy the TMData to parameters. */
                (void) memcpy(pTMC, &pTMData[0], 4);
                (void) memcpy(pTMV, &pTMData[4], 8);
            }
            break;

        /* Perform decrement transfer command in MFP mode. (ISO14443 Layer 4 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L4:
        case PHAL_MFPEVX_SL1_MFP_AUTHENTICATED:
        case PHAL_MFPEVX_SL3_MFP_AUTHENTICATED:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_MFP_CombinedWrite(
                pDataParams,
                (uint8_t) (PHAL_MFPEVX_CMD_DECRTR | bDecrementTransferMaced),
                wSrcBlockNr,
                wDstBlockNr,
                pValue,
                PHAL_MFPEVX_VALUE_BLOCK_SIZE,
                pTMC,
                pTMV));
            break;

        /* Return error in case of not authenticated in any one of the auth mode.*/
        default:
            return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPEVX);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_Transfer(void * pDataParams, uint8_t bTransferMaced, uint16_t wBlockNr, uint8_t * pTMC, uint8_t * pTMV)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    /* Perform transfer according to the auth mode.*/
    switch(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode)
    {
        /* Perform transfer command in MIFARE mode. (ISO14443 Layer 3 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L3:
        case PHAL_MFPEVX_SL1_MIFARE_AUTHENTICATED:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_WriteExtMfc(
                pDataParams,
                PHAL_MFPEVX_CMD_MFC_TRANSFER,
                (uint8_t) wBlockNr,
                NULL,
                0x00U,
                pTMC,
                pTMV));
            break;

        /* Perform transfer command in MFP mode. (ISO14443 Layer 4 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L4:
        case PHAL_MFPEVX_SL1_MFP_AUTHENTICATED:
        case PHAL_MFPEVX_SL3_MFP_AUTHENTICATED:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_MFP_CombinedWrite(
                pDataParams,
                (uint8_t) (PHAL_MFPEVX_CMD_TRANS | bTransferMaced),
                wBlockNr,
                0x00U,
                NULL,
                0U,
                pTMC,
                pTMV));
            break;

        /* Return error in case of not authenticated in any one of the auth mode.*/
        default:
            return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPEVX);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_Restore(void * pDataParams, uint8_t bRestoreMaced, uint16_t wBlockNr)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    /* Perform restore according to the auth mode.*/
    switch(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode)
    {
        /* Perform restore command in MIFARE mode. (ISO14443 Layer 3 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L3:
        case PHAL_MFPEVX_SL1_MIFARE_AUTHENTICATED:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_WriteExtMfc(
                pDataParams,
                PHAL_MFPEVX_CMD_MFC_RESTORE,
                (uint8_t) wBlockNr,
                NULL,
                0x00U,
                NULL,
                NULL));
            break;

        /* Perform restore command in MFP mode. (ISO14443 Layer 4 activated) */
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L4:
        case PHAL_MFPEVX_SL1_MFP_AUTHENTICATED:
        case PHAL_MFPEVX_SL3_MFP_AUTHENTICATED:
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_MFP_CombinedWrite(
                pDataParams,
                (uint8_t) (PHAL_MFPEVX_CMD_REST | bRestoreMaced),
                wBlockNr,
                0x00U,
                NULL,
                0U,
                NULL,
                NULL));
            break;

        /* Return error in case of not authenticated in any one of the auth mode.*/
        default:
            return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPEVX);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}



/***************************************************************************************************************************************/
/* MIFARE Plus EVx Generic command for special commands.																			   */
/***************************************************************************************************************************************/

phStatus_t phalMfpEVx_Sam_X_GetVersion(void * pDataParams, uint8_t * pVerInfo, uint8_t * pVerLen)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM aCmdBuf[1];
    uint8_t		PH_MEMLOC_REM bCmdLen = 0;
    uint8_t		PH_MEMLOC_REM bRespCode = 0;
    uint8_t*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t		PH_MEMLOC_REM aVersion[PHAL_MFPEVX_VERSION_INFO_LENGTH];
    uint8_t		PH_MEMLOC_REM bVerLen = 0;

    /* Frame the GetVersion command. */
    aCmdBuf[bCmdLen++] = PHAL_MFPEVX_CMD_GET_VERSION;

    /* Check if authenticated in MFP mode to append the MAC to the command. */
    if((PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPEVX_SL1_MFP_AUTHENTICATED) ||
        (PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPEVX_SL3_MFP_AUTHENTICATED))
    {
        /* Read the version information of the PICC using MFP_CombinedRead command. */
        wStatus = phhalHw_SamAV3_Cmd_MFP_CombinedRead(
            PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
            PH_EXCHANGE_DEFAULT,
            aCmdBuf,
            bCmdLen,
            &pResponse,
            &wRespLen,
            &bRespCode);

        /* Return the error code. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_PLUS_GEN)
        {
            /* Compute the response code. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_ComputeErrorResponse(1, bRespCode, PH_ON));
        }
        else
        {
            /* Return the HAL error code if not success. */
            if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
            {
                return wStatus;
            }
        }

        /* Copy the local version buffer to the reference parameter. */
        (void) memcpy(pVerInfo, pResponse, wRespLen);
        *pVerLen = (uint8_t) wRespLen;
    }

    /* Read the version from the PICC. */
    else
    {
        do
        {
            wStatus = phalMfpEVx_Sam_X_Int_CardExchange(
                pDataParams,
                PH_EXCHANGE_DEFAULT,
                PH_ON,
                aCmdBuf,
                bCmdLen,		/* Command code is included as part of length. */
                &pResponse,
                &wRespLen,
                &bRespCode);

            /* Check the status. */
            if((bRespCode != PHAL_MFPEVX_RESP_ACK_ISO4) && (bRespCode != PHAL_MFPEVX_RESP_ADDITIONAL_FRAME))
                break;

            /* Copy the Version A into version buffer and update the version buffer length .*/
            (void) memcpy(&aVersion[bVerLen], pResponse, wRespLen);
            bVerLen += (uint8_t) wRespLen;

            /* Update the command information. */
            aCmdBuf[0U] = PHAL_MFPEVX_RESP_ADDITIONAL_FRAME;
            bCmdLen = 1U;

        } while(bRespCode != PHAL_MFPEVX_RESP_ACK_ISO4);

        /* Copy the local version buffer to the reference parameter. */
        (void) memcpy(pVerInfo, aVersion, bVerLen);
        *pVerLen = bVerLen;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_ReadSign(void * pDataParams, uint8_t bLayer4Comm, uint8_t bAddr, uint8_t ** pSignature)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM aCmdBuf[2];
    uint8_t		PH_MEMLOC_REM bCmdLen = 0;
    uint8_t		PH_MEMLOC_REM bRespCode = 0;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Frame the ReadSignature command. */
    aCmdBuf[bCmdLen++] = PHAL_MFPEVX_CMD_READ_SIG;
    aCmdBuf[bCmdLen++] = bAddr;

    switch(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode)
    {
        case PHAL_MFPEVX_SL1_MFP_AUTHENTICATED:
        case PHAL_MFPEVX_SL3_MFP_AUTHENTICATED:
            /* Read the signature information of the PICC using MFP_CombinedRead command. */
            wStatus = phhalHw_SamAV3_Cmd_MFP_CombinedRead(
                PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
                PH_EXCHANGE_DEFAULT,
                aCmdBuf,
                bCmdLen,
                pSignature,
                &wRespLen,
                &bRespCode);

            /* Return the error code. */
            if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_PLUS_GEN)
            {
                /* Compute the response code. */
                PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_ComputeErrorResponse(1U, bRespCode, bLayer4Comm));
            }
            else
            {
                /* Return the HAL error code if not success. */
                if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
                {
                    return wStatus;
                }
            }
            break;

        case PHAL_MFPEVX_NOTAUTHENTICATED:
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L3:
        case PHAL_MFPEVX_NOT_AUTHENTICATED_L4:
        case PHAL_MFPEVX_SL1_MIFARE_AUTHENTICATED:
            /* Perform direct communication with PICC. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_CardExchange(
                pDataParams,
                PH_EXCHANGE_DEFAULT,
                bLayer4Comm,
                aCmdBuf,
                bCmdLen,		/* Command code is included as part of length. */
                pSignature,
                &wRespLen,
                &bRespCode));
            break;

        default:
            break;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_ResetAuth(void * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Perform ResetAuth. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_ResetAuth(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
        PH_OFF, PH_OFF));

    /* Reset the crypto layer */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_ResetSecMsgState(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_PersonalizeUid(void * pDataParams, uint8_t bUidType)
{
    return phalMfpEVx_Int_PersonalizeUid(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams, bUidType);
}

phStatus_t phalMfpEVx_Sam_X_SetConfigSL1(void * pDataParams, uint8_t bOption)
{
    return phalMfpEVx_Int_SetConfigSL1(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams, bOption);
}

phStatus_t phalMfpEVx_Sam_X_ReadSL1TMBlock(void * pDataParams, uint16_t wBlockNr, uint8_t * pBlocks)
{
    return phalMfpEVx_Int_ReadSL1TMBlock(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams, wBlockNr, pBlocks);
}

phStatus_t phalMfpEVx_Sam_X_VCSupportLastISOL3(void * pDataParams, uint8_t * pIid, uint8_t * pPcdCapL3, uint8_t * pInfo)
{
    return phalMfpEVx_Int_VCSupportLastISOL3(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams, pIid, pPcdCapL3,
        pInfo);
}

phStatus_t phalMfpEVx_Sam_X_ChangeKey(void * pDataParams, uint8_t bChangeKeyMaced, uint16_t wBlockNr, uint16_t wKeyNo, uint16_t wKeyVer,
    uint8_t bDivInputLen, uint8_t * pDivInput)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;

    /* Exchange the information to Sam hardware. */
    wStatus = phhalHw_SamAV3_Cmd_MFP_ChangeKey(
        PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        (uint8_t) (bDivInputLen ? PHHAL_HW_SAMAV3_MFP_CHANGE_KEY_DIVERSIFICATION_ON : PHHAL_HW_SAMAV3_MFP_CHANGE_KEY_DIVERSIFICATION_OFF),
        (uint8_t) (bChangeKeyMaced ? PHAL_MFPEVX_CMD_WRITE_EM : PHAL_MFPEVX_CMD_WRITE_EN),
        wBlockNr,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        pDivInput,
        bDivInputLen,
        &bPiccErrCode);

    /* Return the error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_PLUS_GEN)
    {
        /* Compute the response code. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_ComputeErrorResponse(1U, bPiccErrCode, PH_ON));
    }
    else
    {
        /* Return the HAL error code if not success. */
        if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
        {
            return wStatus;
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_CommitReaderID(void * pDataParams, uint16_t wBlockNr, uint8_t * pEncTMRI)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM aCmdBuff[19];
    uint8_t		PH_MEMLOC_REM bCmdLen = 0;
    uint8_t *	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;
    uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    /* Exchange the command to SamAV3 hardware .*/
    wStatus = phhalHw_SamAV3_Cmd_TMRI_CommitReaderID(
        PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        PHHAL_HW_CMD_SAMAV3_TMRI_ISO_MODE_NATIVE,
        PHHAL_HW_CMD_SAMAV3_COMMIT_READER_ID_PICC_STATE_MFP,
        wBlockNr,
        &pResponse,
        &wRespLen,
        &bPiccErrCode);

    /* Return the error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_GEN)
    {
        /* Compute the response code. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_ComputeErrorResponse(1U, bPiccErrCode, PH_ON));
    }
    else
    {
        /* Return the HAL error code if not success. */
        if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
        {
            return wStatus;
        }
    }

    /* Update the pEncTMRI parameter. */
    memcpy(pEncTMRI, pResponse, wRespLen);

    /* Reset the command buffer and its length variable. */
    bCmdLen = 0U;
    memset(aCmdBuff, 0x00, sizeof(aCmdBuff));

    /* Frame the command information. */
    aCmdBuff[bCmdLen++] = PHAL_MFPEVX_CMD_COMMIT_READER_ID;
    aCmdBuff[bCmdLen++] = (uint8_t) (wBlockNr & 0xffU); /* LSB */
    aCmdBuff[bCmdLen++] = (uint8_t) (wBlockNr >> 8U);   /* MSB */

    /* Frame the TMRI information. */
    (void) memcpy(&aCmdBuff[bCmdLen], (*((PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams))->pHalSamDataParams)).bUid, 7);
    bCmdLen += 7U;

    aCmdBuff[bCmdLen] = 0x80U;

    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            PH_TMIUTILS_NO_PADDING, aCmdBuff, 19U, pEncTMRI, PHAL_MFPEVX_SIZE_ENCTMRI, PHAL_MFPEVX_DATA_BLOCK_SIZE));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}



/***************************************************************************************************************************************/
/* MIFARE Plus EVx Generic command for utility operations.																			   */
/***************************************************************************************************************************************/
phStatus_t phalMfpEVx_Sam_X_ResetSecMsgState(void * pDataParams)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;

    /* Kill only card Auth */
    phhalHw_SamAV3_Cmd_SAM_KillAuthentication(PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams), 0x01);

    PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bSMMode = (uint8_t) PHAL_MFPEVX_SECURE_MESSAGE_EV0;

    /* State machine should be handled in a way where L3 activation or L4 activation should not be lost */
    if((PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPEVX_SL3_MFP_AUTHENTICATED) ||
        (PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPEVX_SL1_MFP_AUTHENTICATED) ||
        (PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPEVX_NOT_AUTHENTICATED_L4))
    {
        PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode = PHAL_MFPEVX_NOT_AUTHENTICATED_L4;
    }
    else if((PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPEVX_NOT_AUTHENTICATED_L3) ||
        (PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPEVX_SL1_MIFARE_AUTHENTICATED))
    {
        PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode = PHAL_MFPEVX_NOT_AUTHENTICATED_L3;
    }
    else
    {
        PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode = PHAL_MFPEVX_NOTAUTHENTICATED;
    }

    /* Reset the TMI Collection. */
    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_ActivateTMICollection(
        PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams, PH_TMIUTILS_RESET_TMI));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_SetConfig(void * pDataParams, uint16_t wOption, uint16_t wValue)
{
    switch(wOption)
    {
        case PHAL_MFPEVX_AUTH_MODE:
            PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode = (uint8_t) wValue;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFPEVX);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_GetConfig(void *pDataParams, uint16_t wOption, uint16_t * pValue)
{
    switch(wOption)
    {
        case PHAL_MFPEVX_AUTH_MODE:
            *pValue = (uint16_t) PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFPEVX);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_CalculateTMV(void * pDataParams, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, uint16_t wDstKeyNo,
    uint16_t wDstKeyVer, uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen, uint8_t * pTMI, uint16_t wTMILen, uint8_t * pTMV)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t		PH_MEMLOC_REM bFinished = PH_OFF;
    uint8_t *	PH_MEMLOC_REM pMac = NULL;
    uint16_t	PH_MEMLOC_REM wMacLen = 0;

    uint16_t	PH_MEMLOC_REM wBuffOption = PH_EXCHANGE_DEFAULT;
    uint8_t		PH_MEMLOC_REM bExchangeLen = 0;
    uint16_t	PH_MEMLOC_REM wRemLen = 0;
    uint16_t	PH_MEMLOC_REM wTMIOffset = 0;

    /* Validate the key information. */
    if((wSrcKeyNo > 0x7fU) || (wSrcKeyVer > 0xffU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPEVX);
    }

    if(((wDstKeyNo < 0xE0U) || (wDstKeyNo > 0xE3U)) || (wDstKeyVer > 0xffU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPEVX);
    }

    /* Derive Transaction MAC (KSesTMMAC) session key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_ComputeTMACSessionVectors(
        pDataParams,
        PHAL_MFPEVX_SAM_X_SESSION_TMAC_MAC,
        wSrcKeyNo,
        wSrcKeyVer,
        wDstKeyNo,
        pTMC,
        pUid,
        bUidLen));

    /* Perform Offline activation using Ram Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
        PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
        (uint8_t) wDstKeyNo,
        (uint8_t) wDstKeyVer,
        NULL,
        0U));

    /* Perform MAC verification. */
    wRemLen = (uint16_t) wTMILen;
    wBuffOption = PH_EXCHANGE_TXCHAINING;

    do
    {
        /* Update the finished flag and buffering option. */
        if(wRemLen <= PHAL_MFPEVX_SAM_DATA_FRAME_LENGTH)
        {
            bFinished = PH_ON;
            wBuffOption = PH_EXCHANGE_DEFAULT;
            bExchangeLen = (uint8_t) wRemLen;
        }
        else
        {
            bExchangeLen = PHAL_MFPEVX_SAM_DATA_FRAME_LENGTH;
            wRemLen = (uint16_t) (wRemLen - PHAL_MFPEVX_SAM_DATA_FRAME_LENGTH);
        }

        /* Exchange the TMI information to SAM. */
        wStatus = phhalHw_SamAV3_Cmd_SAM_GenerateMAC(
            PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
            wBuffOption,
            PHHAL_HW_SAMAV3_TRUNCATION_MODE_MFP,
            &pTMI[wTMIOffset],
            bExchangeLen,
            &pMac,
            &wMacLen);

        /* Validate the response. */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            bFinished = PH_ON;
        }

        /* Update the TMI offset information. */
        wTMIOffset += PHAL_MFPEVX_SAM_DATA_FRAME_LENGTH;

    } while(!bFinished);

    /* Copy the Mac to the parameter. */
    (void) memcpy(pTMV, pMac, wMacLen);

    return wStatus;
}

phStatus_t phalMfpEVx_Sam_X_DecryptReaderID(void * pDataParams, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, uint16_t wDstKeyNo,
    uint16_t wDstKeyVer, uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen, uint8_t * pEncTMRI, uint8_t * pTMRIPrev)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;
    uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
    uint16_t	PH_MEMLOC_REM wRespLen = 0;

    /* Validate the key information. */
    if((wSrcKeyNo > 0x7fU) || (wSrcKeyVer > 0xffU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPEVX);
    }

    if(((wDstKeyNo < 0xE0U) || (wDstKeyNo > 0xE3U)) || (wDstKeyVer > 0xffU))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPEVX);
    }

    /* Derive Transaction MAC (KSesTMMAC) session key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_ComputeTMACSessionVectors(
        pDataParams,
        PHAL_MFPEVX_SAM_X_SESSION_TMAC_ENC,
        wSrcKeyNo,
        wSrcKeyVer,
        wDstKeyNo,
        pTMC,
        pUid,
        bUidLen));

    /* Perform Offline activation using Ram Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
        PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
        (uint8_t) wDstKeyNo,
        (uint8_t) wDstKeyVer,
        NULL,
        0U));

    /* Exchange the TMI information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DecipherOfflineData(
        PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        PH_EXCHANGE_DEFAULT,
        pEncTMRI,
        16U,
        &pResponse,
        &wRespLen));

    /* Copy the decrypted information to the parameter. */
    memcpy(pTMRIPrev, pResponse, wRespLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}
#endif /* NXPBUILD__PHAL_MFPEVX_SAM_X */
