/*
 * Copyright 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/**
 * \file
 * MIFARE PRIME NTAG application SamAV3 NonX component of Reader Library framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <ph_TypeDefs.h>
#include <phhalHw.h>
#include <phpalMifare.h>
#include <string.h>

#include <phCryptoSym.h>
#include <phCryptoRng.h>
#include <phKeyStore.h>
#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX

#include "../phalMfPrimeNtag_Int.h"
#include "phalMfPrimeNtag_Sam_NonX.h"
#include "phalMfPrimeNtag_Sam_NonX_Int.h"
#include <phhalHw_SamAV3_Cmd.h>

phStatus_t phalMfPrimeNtag_SamAV3_NonX_Init(phalMfPrimeNtag_SamAV3_NonX_DataParams_t * pDataParams, uint16_t wSizeOfDataParams,
	phhalHw_SamAV3_DataParams_t * pHalSamDataParams, void * pPalMifareDataParams )
{
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pHalSamDataParams, PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM(pPalMifareDataParams, PH_COMP_AL_MFPRIMENTAG);

    /* Data Params size Check. */
    if (sizeof(phalMfPrimeNtag_SamAV3_NonX_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
    }

    /* Initialize dataparams structure members. */
    pDataParams->wId                    = PH_COMP_AL_MFPRIMENTAG | PHAL_MFPRIMENTAG_SAMAV3_NONX_ID;
    pDataParams->pHalSamDataParams      = pHalSamDataParams;
    pDataParams->pPalMifareDataParams   = pPalMifareDataParams;
    pDataParams->bKeyNo                 = 0xFF;
    pDataParams->bAuthMode              = PHAL_MFPRIMENTAG_NOT_AUTHENTICATED;
    pDataParams->wAdditionalInfo        = 0x0000;

	memset(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->aAid, 0x00, 3);	/* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}




/* MIFARE PRIME NTAG secure messaging related commands. ------------------------------------------------------------------------------ */
phStatus_t phalMfPrimeNtag_Sam_NonX_AuthenticateEv2(void *pDataParams, uint8_t bAuthType, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer,
	uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pPcdCapsIn, uint8_t bPcdCapsInLen, uint8_t * pPcdCapsOut,
	uint8_t * pPdCapsOut)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM bFirstAuth = 0;

	if((bAuthType != PHAL_MFPRIMENTAG_AUTHFIRST_NON_LRP) && (bAuthType != PHAL_MFPRIMENTAG_AUTHNONFIRST_NON_LRP))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}

	/* For EV2 mode the length should not exceed 6 bytes. */
	if(bPcdCapsInLen > 6)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Set First or following Auth to wOption parameter. */
	bFirstAuth = (uint8_t) ((bAuthType & 0x01) ? PHAL_MFPRIMENTAG_CMD_AUTHENTICATE_EV2_FIRST : PHAL_MFPRIMENTAG_CMD_AUTHENTICATE_EV2_NON_FIRST);

	/* Exchange the commands between Card and SAM hardware to complete Authentication. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_AuthenticatePICC(
		pDataParams,
		bFirstAuth,
		PHHAL_HW_CMD_SAMAV3_AUTH_MODE_EV2,
		wOption,
		wKeyNo,
		wKeyVer,
		bKeyNoCard,
		pDivInput,
		bDivInputLen,
		pPcdCapsIn,
		bPcdCapsInLen,
		pPcdCapsOut,
		pPdCapsOut));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}




/* MIFARE PRIME NTAG Memory and Configuration mamangement commands. ------------------------------------------------------------------ */
phStatus_t phalMfPrimeNtag_Sam_NonX_SetConfiguration(void * pDataParams, uint8_t bOption, uint8_t * pData, uint8_t bDataLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM aCmdBuff[2];

    /* Frame the command information. */
    aCmdBuff[0] = PHAL_MFPRIMENTAG_CMD_SET_CONFIG;
    aCmdBuff[1] = bOption;

	/* Exchange Cmd.SetConfiguration information to Sam and PICC. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_WriteData(
		pDataParams,
		PH_OFF,
		PHAL_MFPRIMENTAG_COMMUNICATION_ENC,
		PHAL_MFPRIMENTAG_COMMUNICATION_MACD,
		aCmdBuff,
		2,
		pData,
		bDataLen,
		NULL,
		NULL));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_GetVersion(void * pDataParams, uint8_t * pVerInfo)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuff[1];
	uint8_t		PH_MEMLOC_REM bComMode = 0;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Frame the command information. */
	aCmdBuff[0] = PHAL_MFPRIMENTAG_CMD_GET_VERSION;

	/* Frame the communication mode to be applied. */
	bComMode = (uint8_t) ((PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_AUTHENTICATEEV2) ?
		PHAL_MFPRIMENTAG_COMMUNICATION_MACD : PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN);

	/* Exchange Cmd.GetVersion information to Sam and PICC. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_ReadData(
		pDataParams,
		bComMode,
		bComMode,
		aCmdBuff,
		1,
		&pVerInfo,
		&wRespLen));

	PH_CHECK_SUCCESS_FCT(wStatus, phalMfprimeNtag_SetConfig(pDataParams, PHAL_MFPRIMENTAG_ADDITIONAL_INFO, wRespLen));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}




/* MIFARE PRIME NTAG Key mamangement commands. -------------------------------------------------------------------------------------- */
phStatus_t phalMfPrimeNtag_Sam_NonX_ChangeKey(void * pDataParams, uint16_t wOption, uint16_t wCurrKeyNo, uint16_t wCurrKeyVer, uint16_t wNewKeyNo, uint16_t wNewKeyVer,
	uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivInputLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;

	/* Exchange the commands between Card and SAM hardware to Change Key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_ChangeKeyPICC(
		pDataParams,
		wOption,
		bKeyNoCard,
		wCurrKeyNo,
		wCurrKeyVer,
		wNewKeyNo,
		wNewKeyVer,
		pDivInput,
		bDivInputLen));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_GetKeyVersion(void * pDataParams, uint8_t bKeyNo, uint8_t * pKeyVersion, uint8_t * pKeyVerLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuff[3];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint8_t		PH_MEMLOC_REM bComMode = 0;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	uint8_t		PH_MEMLOC_REM aAppId[3] = {0x00, 0x00, 0x00};

	/* Only if seleted Aid is 0x000000. */
	if (memcmp(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00)
    {
		/* At PICC level, 1,2,3,4 are valid ones. 0 is excluded */
		if (bKeyNo >= 0x05)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
    }
	else
	{
		/* At App level, 0,1,2,3,4 are valid ones. */
		if (bKeyNo >= 0x03)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
	}
#endif

	/* Frame the command information. */
	aCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_GET_KEY_VERSION;
	aCmdBuff[wCmdLen++] = bKeyNo;

	/* Frame the communication mode to be applied. */
	bComMode = (uint8_t) ((PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_AUTHENTICATEEV2) ?
		PHAL_MFPRIMENTAG_COMMUNICATION_MACD : PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN);

	/* Exchange Cmd.GetKeyVersion information to Sam and PICC. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_ReadData(
		pDataParams,
		bComMode,
		bComMode,
		aCmdBuff,
		wCmdLen,
		&pKeyVersion,
		&wRespLen));

	*pKeyVerLen = (uint8_t) wRespLen;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}




/* MIFARE PRIME NTAG File mamangement commands. -------------------------------------------------------------------------------------- */
phStatus_t phalMfPrimeNtag_Sam_NonX_GetFileSettings(void * pDataParams, uint8_t bFileNo, uint8_t * pFSBuffer, uint8_t * pBufferLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuff[2];
	uint8_t		PH_MEMLOC_REM bComMode = 0;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Frame the command information. */
	aCmdBuff[0] = PHAL_MFPRIMENTAG_CMD_GET_FILE_SETTINGS;
	aCmdBuff[1] = bFileNo;

	/* Frame the communication mode to be applied. */
	bComMode = (uint8_t) ((PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_AUTHENTICATEEV2) ?
		PHAL_MFPRIMENTAG_COMMUNICATION_MACD : PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN);

	/* Exchange Cmd.GetFileSettings information to Sam and PICC. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_ReadData(
		pDataParams,
		bComMode,
		PHAL_MFPRIMENTAG_COMMUNICATION_MACD,
		aCmdBuff,
		2,
		&pFSBuffer,
		&wRespLen));

	/* Update the length. */
	*pBufferLen = (uint8_t) wRespLen;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_GetFileCountersSDM(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pFileCounters, uint8_t * pRespLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuff[2];
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Validate the parameters */
	if ((bOption != PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) &&
        (bOption != PHAL_MFPRIMENTAG_COMMUNICATION_ENC))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	/* Frame the command information. */
	aCmdBuff[0] = PHAL_MFPRIMENTAG_CMD_GET_SDM_FILE_COUNTER;
	aCmdBuff[1] = bFileNo;

	/* Exchange Cmd.GetFileCounters information to Sam and PICC. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_ReadData(
		pDataParams,
		bOption,
		bOption,
		aCmdBuff,
		2,
		&pFileCounters,
		&wRespLen));

	/* Update the length. */
	*pRespLen = (uint8_t) wRespLen;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_ChangeFileSettings(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bCommSett, uint8_t *pAccessRights)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM aCmdBuff[10];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint8_t		PH_MEMLOC_REM bCmd_ComMode = 0;
	uint8_t		PH_MEMLOC_REM bResp_ComMode = 0;

	/* Frame the Crypto information.
	 * Masking the MSB bit becuase its not support by SAM.
	 * Its actually supported in Software only.
	 */
	bCmd_ComMode = (uint8_t) (bOption & 0xFE);

	/* Validate the parameters */
	if ((bCmd_ComMode != PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) &&
        (bCmd_ComMode != PHAL_MFPRIMENTAG_COMMUNICATION_ENC) &&
        (bCmd_ComMode != PHAL_MFPRIMENTAG_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	/* Frame the command information. */
	aCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_CHANGE_FILE_SETTINGS;
	aCmdBuff[wCmdLen++] = bFileNo;
	aCmdBuff[wCmdLen++] = (uint8_t) (bCommSett >> 4);

    /* Append access rights. */
	memcpy(&aCmdBuff[wCmdLen], pAccessRights, 2); /* PRQA S 3200 */
	wCmdLen += 2;

	/* Frame the SM to be applied for command. */
	bResp_ComMode = (uint8_t) ((bCmd_ComMode == PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) ? PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN :
		PHAL_MFPRIMENTAG_COMMUNICATION_MACD);

	/* Exchange Cmd.ChangeFileSettings information to Sam and PICC. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_WriteData(
		pDataParams,
		PH_OFF,
		bCmd_ComMode,
		bResp_ComMode,
		aCmdBuff,
		(uint16_t) ((bCmd_ComMode == PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) ? wCmdLen : 2),
		&aCmdBuff[2],
		(uint16_t) ((bCmd_ComMode == PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) ? 0 : (wCmdLen - 2)),
		NULL,
		NULL));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_ChangeFileSettingsSDM(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bFileOption, uint8_t *pAccessRights,
	uint8_t bSdmOptions, uint8_t *pSdmAccessRights, uint8_t *pVCUIDOffset, uint8_t *pSDMReadCtrOffset, uint8_t *pSDMMACInputOffset, uint8_t *pSDMMACOffset)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM aCmdBuff[40];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint8_t		PH_MEMLOC_REM bCmd_ComMode = 0;
	uint8_t		PH_MEMLOC_REM bResp_ComMode = 0;

	/* Validate the parameters */
#if RDR_LIB_PARAM_CHECK
	if(bFileNo != PHAL_MFPRIMENTAG_SDM_FILE_ID)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}
#endif

	if ((bOption != PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) &&
        (bOption != PHAL_MFPRIMENTAG_COMMUNICATION_ENC) &&
        (bOption != PHAL_MFPRIMENTAG_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	/* Frame the command information. */
	aCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_CHANGE_FILE_SETTINGS;
	aCmdBuff[wCmdLen++] = bFileNo;
	aCmdBuff[wCmdLen++] = bFileOption;

    /* Append access rights. */
	memcpy(&aCmdBuff[wCmdLen], pAccessRights, 2); /* PRQA S 3200 */
	wCmdLen += 2;

	/* Append SDMOption information. */
	aCmdBuff[wCmdLen++] = bSdmOptions;

	/* Append SDM access rights. */
	memcpy(&aCmdBuff[wCmdLen], pSdmAccessRights, 2); /* PRQA S 3200 */
	wCmdLen += 2;

	/* Append VCUID Offset. */
	if(pVCUIDOffset != NULL)
	{
		memcpy(&aCmdBuff[wCmdLen], pVCUIDOffset, 3); /* PRQA S 3200 */
		wCmdLen += 3;
	}

	/* Append SDM Read Ctr Offset. */
	if(pSDMReadCtrOffset != NULL)
	{
		memcpy(&aCmdBuff[wCmdLen], pSDMReadCtrOffset, 3); /* PRQA S 3200 */
		wCmdLen += 3;
	}

	/* Append SDM MAC Input Offset. */
	memcpy(&aCmdBuff[wCmdLen], pSDMMACInputOffset, 3); /* PRQA S 3200 */
	wCmdLen += 3;

	/* Append SDM MAC Offset. */
	memcpy(&aCmdBuff[wCmdLen], pSDMMACOffset, 3); /* PRQA S 3200 */
	wCmdLen += 3;

	/* Frame the command and response communication modes. */
	bCmd_ComMode = bOption;
	bResp_ComMode = (uint8_t) ((bCmd_ComMode == PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) ? PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN :
		PHAL_MFPRIMENTAG_COMMUNICATION_MACD);

	/* Exchange Cmd.ChangeFileSettings information to Sam and PICC. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_WriteData(
		pDataParams,
		PH_OFF,
		bCmd_ComMode,
		bResp_ComMode,
		aCmdBuff,
		(uint16_t) ((bCmd_ComMode == PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) ? wCmdLen : 2),
		&aCmdBuff[2],
		(uint16_t) ((bCmd_ComMode == PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) ? 0 : (wCmdLen - 2)),
		NULL,
		NULL));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}





/* MIFARE PRIME NTAG Data mamangement commands. -------------------------------------------------------------------------------------- */
phStatus_t phalMfPrimeNtag_Sam_NonX_ReadData(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pLength, uint8_t ** ppResponse,
	uint16_t * pRespLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuff[8];
	uint16_t	PH_MEMLOC_REM wCmdLen = 0;
	uint8_t		PH_MEMLOC_REM bCmd_ComMode = 0;
	uint8_t		PH_MEMLOC_REM bResp_ComMode = 0;

	/* Validate the parameters. */
    if (((bOption & 0xF0) != PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) &&
        ((bOption & 0xF0) != PHAL_MFPRIMENTAG_COMMUNICATION_ENC) &&
        ((bOption & 0xF0) != PHAL_MFPRIMENTAG_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	if ((bOption & 0x0FU) != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	/* Frame the command information. */
    aCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_READ_DATA_ISO;
    aCmdBuff[wCmdLen++] = bFileNo;

    memcpy(&aCmdBuff[wCmdLen], pOffset, 3); /* PRQA S 3200 */
    wCmdLen += 3;

    memcpy(&aCmdBuff[wCmdLen], pLength, 3); /* PRQA S 3200 */
    wCmdLen += 3;

	/* Frame the SM to be applied for command. */
	bCmd_ComMode = (uint8_t) (((bOption & 0xF0) == PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) ? PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN :
		PHAL_MFPRIMENTAG_COMMUNICATION_MACD);

	/* Frame the SM to be applied for response. */
	bResp_ComMode = (uint8_t) (bOption & 0xF0);

	/* Exchange Cmd.ReadData information to Sam and PICC. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_ReadData(
		pDataParams,
		bCmd_ComMode,
		bResp_ComMode,
		aCmdBuff,
		wCmdLen,
		ppResponse,
		pRespLen));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_WriteData(void * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pOffset, uint8_t * pData, uint8_t * pDataLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t     PH_MEMLOC_REM aCmdBuff[8];
	uint16_t	PH_MEMLOC_REM wCmdLen = 0;
	uint32_t	PH_MEMLOC_REM dwDataLen = 0;
	uint8_t		PH_MEMLOC_REM bCmd_ComMode = 0;
	uint8_t		PH_MEMLOC_REM bResp_ComMode = 0;

	/* Validate the parameters. */
#ifdef RDR_LIB_PARAM_CHECK
	if ((bOption != PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) &&
		(bOption != PHAL_MFPRIMENTAG_COMMUNICATION_ENC) &&
		(bOption != PHAL_MFPRIMENTAG_COMMUNICATION_MACD))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}
#endif

    /* Frame the command information. */
    aCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_WRITE_DATA_ISO;
    aCmdBuff[wCmdLen++] = bFileNo;

	memcpy(&aCmdBuff[wCmdLen], pOffset, 3);
	wCmdLen += 3;

	memcpy(&aCmdBuff[wCmdLen], pDataLen, 3);
	wCmdLen += 3;

	/* Set the lengths. */
	dwDataLen = (uint32_t) (pDataLen[0] | (pDataLen[1] << 8) | (pDataLen[2] << 16));

	/* Frame the SM to be applied for command. */
	bCmd_ComMode = (uint8_t) (bOption & 0xF0);

	/* Frame the SM to be applied for command. */
	bResp_ComMode = (uint8_t) ((bOption == PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) ? PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN :
		PHAL_MFPRIMENTAG_COMMUNICATION_MACD);

	/* Exchange Cmd.WriteData information to Sam and PICC. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_WriteData(
		pDataParams,
		PH_ON,
		bCmd_ComMode,
		bResp_ComMode,
		aCmdBuff,
		wCmdLen,
		pData,
		dwDataLen,
		NULL,
		NULL));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}




/* MIFARE PRIME NTAG ISO7816-4 commands. --------------------------------------------------------------------------------------------- */
phStatus_t phalMfPrimeNtag_Sam_NonX_IsoSelectFile(void * pDataParams, uint8_t bOption, uint8_t bSelector, uint8_t * pFid, uint8_t * pDFname, uint8_t bDFnameLen,
	uint8_t	bExtendedLenApdu, uint8_t ** ppFCI, uint16_t * pFCILen)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuff[26];
	uint16_t	PH_MEMLOC_REM wCmdLen = 0;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint8_t		PH_MEMLOC_REM aFileId[3] = {'\0'};
	uint8_t		PH_MEMLOC_REM aPiccDfName[7] = {0xD2, 0x76, 0x00, 0x00, 0x85, 0x01, 0x00};
	uint8_t		PH_MEMLOC_REM aPiccStat[2] = {0x00, 0x00};

	/* Validate the parameters. */
#ifdef RDR_LIB_PARAM_CHECK
	if (bDFnameLen > 16)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}
	/* If User intends to select through DFName, check for validity of DFName passed */
	if((bSelector ==  PHAL_MFPRIMENTAG_SELECTOR_4) && (bDFnameLen == 0x00))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}
	if ((bOption != PHAL_MFPRIMENTAG_FCI_RETURNED) && (bOption != PHAL_MFPRIMENTAG_FCI_NOT_RETURNED))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}
	if((bSelector != PHAL_MFPRIMENTAG_SELECTOR_4) && (bSelector != PHAL_MFPRIMENTAG_SELECTOR_3) && (pFid == NULL))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}
#endif

	/* Frame the command. */
	aCmdBuff[wCmdLen++] = 0x00;
	aCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_ISO7816_SELECT_FILE;
	aCmdBuff[wCmdLen++] = bSelector;
	aCmdBuff[wCmdLen++] = bOption;

	/* Append LC. */
	if(bSelector != PHAL_MFPRIMENTAG_SELECTOR_3)
	{
		if(bExtendedLenApdu)
		{
			aCmdBuff[wCmdLen++] = 0;
			aCmdBuff[wCmdLen++] = 0;
		}
	}

	/* Append the payload and LC. */
	if(bSelector == PHAL_MFPRIMENTAG_SELECTOR_4)
	{
		/* Append LC. */
		aCmdBuff[wCmdLen++] = bDFnameLen;

		memcpy(&aCmdBuff[wCmdLen], pDFname, bDFnameLen); /* PRQA S 3200 */
		wCmdLen += bDFnameLen;
	}
	else if((bSelector ==  PHAL_MFPRIMENTAG_SELECTOR_0) || (bSelector == PHAL_MFPRIMENTAG_SELECTOR_1) ||
		(bSelector == PHAL_MFPRIMENTAG_SELECTOR_2))
	{
		/* Append LC. */
		aCmdBuff[wCmdLen++] = 2;

		/* Select MF, DF or EF, by file identifier
		 * Select child DF
		 * Select EF under the current DF, by file identifier
		 * Select parent DF of the current DF
		 */
		aFileId[1] = aCmdBuff[wCmdLen++] = pFid[1];
		aFileId[0] = aCmdBuff[wCmdLen++] = pFid[0];
		aFileId[2] = 0;
	}
	else
	{
		/* Exchange the contents to PICC. */
		aCmdBuff[wCmdLen++] = 0;
	}

	/* Append LE. */
	if(bSelector != PHAL_MFPRIMENTAG_SELECTOR_3)
	{
		if(bExtendedLenApdu)
		{
			aCmdBuff[wCmdLen++] = 0;
		}
		aCmdBuff[wCmdLen++] = 0;
	}

	/* Exchange Cmd.ISOSelectFile information to PICC. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_CardExchange(
		pDataParams,
		PH_OFF,
		PH_EXCHANGE_DEFAULT,
		0,
		PH_OFF,
		aCmdBuff,
		wCmdLen,
		&pResponse,
		&wRespLen,
		aPiccStat));

	/* Reset Authentication should not be targeted for elementary file selection using file ID */
	if(bSelector !=  PHAL_MFPRIMENTAG_SELECTOR_2)
	{
		/* Reset Authentication Status here */
		PH_CHECK_SUCCESS_FCT(wStatus1, phalMfPrimeNtag_Sam_NonX_ResetAuthentication(pDataParams));
	}

	/* once the selection Success, update the File Id to master data structure if the selection is done through AID */
	if((bSelector ==  PHAL_MFPRIMENTAG_SELECTOR_0) || (bSelector == PHAL_MFPRIMENTAG_SELECTOR_1) ||
		(bSelector == PHAL_MFPRIMENTAG_SELECTOR_2))
	{
		memcpy(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->aAid, aFileId, sizeof(aFileId));	/* PRQA S 3200 */
	}
	else if((bSelector ==  PHAL_MFPRIMENTAG_SELECTOR_4))
	{
		/* Update the file ID to all zeros if DF Name is of PICC. */
		if(memcmp( pDFname, aPiccDfName, 7) == 0)
		{
			aFileId[0] = 0x00;
			aFileId[1] = 0x00;
			aFileId[2] = 0x00;
		}
		else
		{
			aFileId[0] = 0xff;
			aFileId[1] = 0xff;
			aFileId[2] = 0xff;
		}

		memcpy(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->aAid, aFileId, sizeof(aFileId));	/* PRQA S 3200 */
	}
	else
	{
			/* Nothing for Secector 0x03. */
	}

	/* Copy the response to the buffer */
	*ppFCI = pResponse;
	*pFCILen = wRespLen;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_IsoReadBinary ( void * pDataParams, uint16_t wOption, uint8_t bOffset, uint8_t bSfid, uint32_t dwBytesToRead, uint8_t bExtendedLenApdu,
	uint8_t ** ppResponse, uint32_t * pBytesRead )
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuff[8];
	uint16_t	PH_MEMLOC_REM wCmdLen = 0;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint8_t		PH_MEMLOC_REM aPiccStat[2] = { 0x00, 0x00 };

#ifdef RDR_LIB_PARAM_CHECK
	/* Validate the parameter. */
	if (bSfid & 0x80)
	{
		/* Short file id is supplied */
		if (( bSfid & 0x7FU ) > 0x1F)
		{
			/* Error condition */
			return PH_ADD_COMPCODE ( PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG );
		}
	}
#endif
	if(wOption != PH_EXCHANGE_DEFAULT)
	{
		return PH_ADD_COMPCODE ( PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG );
	}

	/* Frame the command information based on the option. */
	aCmdBuff[wCmdLen++] = 0x00;
	aCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_ISO7816_READ_BINARY;
	aCmdBuff[wCmdLen++] = bSfid;
	aCmdBuff[wCmdLen++] = bOffset;

	if (bExtendedLenApdu)
	{
		aCmdBuff[wCmdLen++] = ( uint8_t ) ( ( dwBytesToRead & 0x00FF0000 ) >> 16 );
		aCmdBuff[wCmdLen++] = ( uint8_t ) ( ( dwBytesToRead & 0x0000FF00 ) >> 8 );
	}
	aCmdBuff[wCmdLen++] = ( uint8_t ) ( dwBytesToRead & 0x000000FF );

	/* Exchange Cmd.ISOReadBinary information to PICC. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_CardExchange (
		pDataParams,
		PH_OFF,
		wOption,
		0,
		PH_OFF,
		aCmdBuff,
		wCmdLen,
		ppResponse,
		&wRespLen,
		aPiccStat));

	*pBytesRead = wRespLen;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_IsoUpdateBinary ( void * pDataParams, uint8_t bOffset, uint8_t bSfid, uint8_t bExtendedLenApdu, uint8_t * pData, uint32_t dwDataLen )
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuff[7];
	uint16_t	PH_MEMLOC_REM wCmdLen = 0;
	uint8_t *	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint8_t		PH_MEMLOC_REM aPiccStat[2] = { 0x00, 0x00 };

#ifdef RDR_LIB_PARAM_CHECK
	/* Validate the parameter. */
	if (bSfid & 0x80)
	{
		/* Short file id is supplied */
		if (( bSfid & 0x7FU ) > 0x1F)
		{
			/* Error condition */
			return PH_ADD_COMPCODE ( PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG );
		}
	}
#endif

	/* Frame the command. */
	aCmdBuff[wCmdLen++] = 0x00;										/* CLA */
	aCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_ISO7816_UPDATE_BINARY;	/* INS */
	aCmdBuff[wCmdLen++] = bSfid;									/* P1 */
	aCmdBuff[wCmdLen++] = bOffset;

	if (bExtendedLenApdu)
	{
		aCmdBuff[wCmdLen++] = ( uint8_t ) ( ( dwDataLen & 0x00FF0000 ) >> 16 );
		aCmdBuff[wCmdLen++] = ( uint8_t ) ( ( dwDataLen & 0x0000FF00 ) >> 8 );
	}
	aCmdBuff[wCmdLen++] = ( uint8_t ) ( dwDataLen & 0x000000FF );

	/* Exchange Cmd.ISOSUpdateBinary information to PICC. */
	PH_CHECK_SUCCESS_FCT ( wStatus, phalMfPrimeNtag_Sam_NonX_Int_CardExchange (
		pDataParams,
		PH_OFF,
		PH_EXCHANGE_BUFFER_CONT,
		0,
		PH_OFF,
		aCmdBuff,
		wCmdLen,
		&pResponse,
		&wRespLen,
		aPiccStat ) );

	wStatus = phalMfPrimeNtag_Sam_NonX_Int_CardExchange (
		pDataParams,
		PH_OFF,
		PH_EXCHANGE_BUFFER_LAST,
		0,
		PH_OFF,
		pData,
		( uint16_t ) dwDataLen,
		&pResponse,
		&wRespLen,
		aPiccStat );

	if (wStatus != PH_ERR_SUCCESS)
	{
		/* Reset authentication status */
		PH_CHECK_SUCCESS_FCT ( wStatus1, phalMfPrimeNtag_Sam_NonX_ResetAuthentication ( pDataParams ) );
	}

	return wStatus;
}




/* MIFARE PRIME NTAG Originality Check functions. ------------------------------------------------------------------------------------ */
phStatus_t phalMfPrimeNtag_Sam_NonX_ReadSign(void * pDataParams, uint8_t bAddr, uint8_t ** ppSignature)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuff[2];
	uint8_t		PH_MEMLOC_REM bComMode = 0;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Frame the command information. */
	aCmdBuff[0] = PHAL_MFPRIMENTAG_CMD_READ_SIG;
	aCmdBuff[1] = bAddr;

	/* Frame the communication mode to be applied. */
	bComMode = (uint8_t) ((PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_AUTHENTICATEEV2) ?
		PHAL_MFPRIMENTAG_COMMUNICATION_ENC : PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN);

	/* Exchange Cmd.ReadSign information to Sam and PICC. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_ReadData(
		pDataParams,
		bComMode,
		PHAL_MFPRIMENTAG_COMMUNICATION_ENC,
		aCmdBuff,
		2,
		ppSignature,
		&wRespLen));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}




/* MIFARE PRIME NTAG Miscellaneous functions. ---------------------------------------------------------------------------------------- */
phStatus_t phalMfPrimeNtag_Sam_NonX_GetConfig(void * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    switch (wConfig)
    {
		case PHAL_MFPRIMENTAG_ADDITIONAL_INFO:
			*pValue = PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->wAdditionalInfo;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_SetConfig(void * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    switch (wConfig)
    {
		case PHAL_MFPRIMENTAG_ADDITIONAL_INFO:
			PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->wAdditionalInfo = wValue;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_ResetAuthentication(void * pDataParams)
{
	phalMfPrimeNtag_Sam_NonX_Int_ResetAuthStatus(pDataParams);

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_CalculateMACSDM(void * pDataParams, uint8_t bSdmOption, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer,
	uint16_t wDstKeyNo, uint16_t wDstKeyVer, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr, uint8_t * pInData, uint16_t wInDataLen,
	uint8_t * pMac)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t *	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;

	/* Validate the key information. */
	if ((wSrcKeyNo > 0x7f) || (wSrcKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	if (((wDstKeyNo < 0xE0) || (wDstKeyNo > 0xE3)) || (wDstKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	/* Derive SDM MAC (KSesSDMFileReadMAC) session key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_ComputeSDMSessionVectors(
		pDataParams,
		bSdmOption,
		wSrcKeyNo,
		wSrcKeyVer,
		wDstKeyNo,
		pUid,
		bUidLen,
		pSDMReadCtr));

	/* Perform Offline activation using Ram Key. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
		PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
		(uint8_t) wDstKeyNo,
		(uint8_t) wDstKeyVer,
		NULL,
		0));

	/* Exchange the Input information to SAM. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_GenerateMAC(
		PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PH_EXCHANGE_DEFAULT,
		PHHAL_HW_SAMAV3_TRUNCATION_MODE_MFP,
		pInData,
		(uint8_t) wInDataLen,
		&pResponse,
		&wRespLen));

	/* Copy the Mac to the parameter. */
	memcpy(pMac, pResponse, wRespLen);	/* PRQA S 3200 */

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */
