/*
 * Copyright 2016 - 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <ph_TypeDefs.h>
#include <string.h>
#include <phTools.h>
#include <phKeyStore.h>
#ifdef NXPBUILD__PH_CRYPTOSYM
#include <phCryptoSym.h>
#endif /* NXPBUILD__PH_CRYPTOSYM */
#ifdef NXPBUILD__PH_CRYPTORNG
#include <phCryptoRng.h>
#endif /* NXPBUILD__PH_CRYPTORNG */
#include <phTMIUtils.h>
#include <phalVca.h>

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SW

#include "../phalMfprimeNtag_Int.h"
#include "phalMfprimeNtag_Sw.h"
#include "phalMfprimeNtag_Sw_Int.h"

phStatus_t phalMfprimeNtag_Sw_Init(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, void * pPalMifareDataParams,
	void * pKeyStoreDataParams, void * pCryptoDataParamsEnc, void * pCryptoDataParamsMac, void * pCryptoRngDataParams, void * pTMIDataParams,
	void * pHalDataParams)
{
    /* data param check */
    if (sizeof(phalMfprimeNtag_Sw_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFPRIMENTAG);
    }
    PH_ASSERT_NULL_DATA_PARAM(pDataParams,PH_COMP_AL_MFPRIMENTAG);
    PH_ASSERT_NULL_PARAM (pPalMifareDataParams,PH_COMP_AL_MFPRIMENTAG);
#ifdef NXPBUILD__PH_CRYPTOSYM
    PH_ASSERT_NULL_PARAM (pKeyStoreDataParams,PH_COMP_AL_MFPRIMENTAG);
    PH_ASSERT_NULL_PARAM (pCryptoDataParamsEnc,PH_COMP_AL_MFPRIMENTAG);
	PH_ASSERT_NULL_PARAM (pCryptoDataParamsMac,PH_COMP_AL_MFPRIMENTAG);
    PH_ASSERT_NULL_PARAM (pCryptoRngDataParams,PH_COMP_AL_MFPRIMENTAG);
#endif /* NXPBUILD__PH_CRYPTOSYM */
	PH_ASSERT_NULL_PARAM (pHalDataParams,PH_COMP_AL_MFPRIMENTAG);
    /* init private data */
    pDataParams->wId                    = PH_COMP_AL_MFPRIMENTAG | PHAL_MFPRIMENTAG_SW_ID;
    pDataParams->pPalMifareDataParams   = pPalMifareDataParams;
    pDataParams->pKeyStoreDataParams    = pKeyStoreDataParams;
    pDataParams->pCryptoDataParamsEnc   = pCryptoDataParamsEnc;
    pDataParams->pCryptoDataParamsMac   = pCryptoDataParamsMac;
    pDataParams->pCryptoRngDataParams   = pCryptoRngDataParams;
    pDataParams->pTMIDataParams         = pTMIDataParams;
    pDataParams->pHalDataParams         = pHalDataParams;
    /* 2 Byte CRC initial value in Authenticate mode. */
    pDataParams->wCrc = PH_TOOLS_CRC16_PRESET_ISO14443A;

    /* 4 Byte CRC initial value in 0x1A, 0xAA mode. */
    pDataParams->dwCrc = PH_TOOLS_CRC32_PRESET_DF8;

    memset(pDataParams->bSesAuthENCKey, 0x00, 24);  /* PRQA S 3200 */
    pDataParams->bKeyNo = 0xFF; /* Set to invalid */
    memset(pDataParams->bIv, 0x00, 16); /* PRQA S 3200 */
    memset(pDataParams->pAid, 0x00, 3);  /* PRQA S 3200 */
    pDataParams->bAuthMode = PHAL_MFPRIMENTAG_NOT_AUTHENTICATED; /* Set to invalid */
    pDataParams->bWrappedMode = 0x01; /* Set to true */
    pDataParams->bCryptoMethod = 0xFF; /* No crypto just after init */
    pDataParams->wAdditionalInfo = 0x0000;
	pDataParams->bShortLenApdu = 0x01; /* By default, extended length APDU format is used for BIG ISO Read */
    pDataParams->dwPayLoadLen = 0;
    pDataParams->wCmdCtr = 0;
    memset(pDataParams->bTi, 0x00, PHAL_MFPRIMENTAG_SIZE_TI);  /* PRQA S 3200 */
    memset(pDataParams->bSesAuthMACKey, 0x00, 16);  /* PRQA S 3200 */
    memset(pDataParams->pUnprocByteBuff, 0x00, PHAL_MFPRIMENTAG_SIZE_MAC);  /* PRQA S 3200 */
    pDataParams->bNoUnprocBytes = 0;
    memset(pDataParams->bLastBlockBuffer, 0x00, 16);  /* PRQA S 3200 */
    pDataParams->bLastBlockIndex = 0;

	pDataParams->bSdmOptionVal = 0;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}


#ifdef NXPBUILD__PH_CRYPTOSYM
/*
*   For following authentication, when you send a command and get encrypted bRndB
you should decrypt it with a computed IV. The IV is the decryption IV with option = PH_OFF (Pls verify)
Similarly, when you are encrypting RndA||RndB' you should then encrypt with computed IV (option PHON)
When you receive encrypted RndA', again decrypt this using computed IV with PH_OFF
Lastly, It is still not very clear, what is the IV to be used for encrypting SV1 and SV2.
You can track this as an open question.

AuthenticateEV2 is also used to authenticate PICC originality check keys.
In this case, the AuthEv2 nonfirst is used without AuthEV2First being done before.
Here having secure messaging i.e., Session keys is of no use. Just like in MIFARE Plus auth,
we should reset the secure messaging if AuthEV2nonfirst is done successfully without a
AuthEV2First done previously.

*/
phStatus_t phalMfprimeNtag_Sw_AuthenticateEv2(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint8_t bAuthOption, uint16_t wOption, uint16_t wKeyNo,
	uint16_t wKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivLen, uint8_t bLenPcdCapsIn, uint8_t *pPcdCapsIn, uint8_t *pPcdCapsOut,
	uint8_t *pPdCapsOut)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;

	/* Decide whether the Auth option is Non LRP or Regular EV2 */
	if((bAuthOption == PHAL_MFPRIMENTAG_AUTHFIRST_NON_LRP) || (bAuthOption == PHAL_MFPRIMENTAG_AUTHNONFIRST_NON_LRP))
	{
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfprimeNtag_Sw_Int_AuthenticateEv2(
            pDataParams,
            bAuthOption,
            wOption,
            wKeyNo,
            wKeyVer,
            bKeyNoCard,
            pDivInput,
            bDivLen,
            bLenPcdCapsIn,
            pPcdCapsIn,
            pPcdCapsOut,
            pPdCapsOut
            ));
	}
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}




phStatus_t phalMfprimeNtag_Sw_SetConfiguration(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pData, uint8_t bDataLen)
{
	uint8_t  PH_MEMLOC_REM bCmdBuff[8];
	uint16_t PH_MEMLOC_REM wCmdLen = 0;
	uint8_t  PH_MEMLOC_REM bPaddingMethod = PH_CRYPTOSYM_PADDING_MODE_1;
	phStatus_t PH_MEMLOC_REM statusTmp;

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_SET_CONFIG;
	bCmdBuff[wCmdLen++] = bOption;
	switch(bOption)
	{
		/* Data = 1B configuration data */
		case PHAL_MFPRIMENTAG_SET_CONFIG_OPTION0:
			/* Data =  KEY || 1BYTE KEY VERSION    Key data is 25 bytes */
		case PHAL_MFPRIMENTAG_SET_CONFIG_OPTION1:
			/* User defined SAK */
		case PHAL_MFPRIMENTAG_SET_CONFIG_OPTION3:
			/* Secure Messaging Configuration */
		case PHAL_MFPRIMENTAG_SET_CONFIG_OPTION4:
			/* Capability data, consisting of VCTID Override, PDCap1 and PDCap2 */
		case PHAL_MFPRIMENTAG_SET_CONFIG_OPTION5:
			/* Virtual Card Installation Identifier(VCIID) or application ISODFName */
		case PHAL_MFPRIMENTAG_SET_CONFIG_OPTION6:
			break;
			/* User defined ATS */
		case PHAL_MFPRIMENTAG_SET_CONFIG_OPTION2:
			bPaddingMethod = PH_CRYPTOSYM_PADDING_MODE_2;
			break;

		default:
			/* Do not check for Invalid parameter here. */
			break;
	}

	/* In case of Option-5, where LRP to be enabled(i.e pData[4] = 01), Data is Sent in FULL and the response is expected in PLAIN.
	* Hence the followiung interface should know that it should not process the Response data in FULL mode. Hence
	* using the below flag variable */

	statusTmp = phalMfprimeNtag_Sw_Int_Write_Enc(
		pDataParams,
		PHAL_MFPRIMENTAG_DEFAULT_MODE,
		bCmdBuff,
		wCmdLen,
		bPaddingMethod,
		0x00,
		pData,
		(uint16_t) bDataLen
	);

	/* Store back the original Crypto method the pDataParams->bCryptoMethod once the data is complete */

	return statusTmp;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalMfprimeNtag_Sw_GetVersion(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint8_t * pVerInfo)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[8];
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint8_t     PH_MEMLOC_REM *pRecv;

	/* form the command */
	bCmdBuff[0] = PHAL_MFPRIMENTAG_CMD_GET_VERSION;

	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfprimeNtag_Sw_Int_ReadData_Plain(
		pDataParams,
		(pDataParams->bAuthMode == PHAL_MFPRIMENTAG_AUTHENTICATEEV2) ? PHAL_MFPRIMENTAG_COMMUNICATION_MACD : PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN,
		bCmdBuff,
		1,
		&pRecv,
		&wRxlen
	));

	/* If received Data length is not equal to 28B(In case of 7BUID) or 30B(In case of 10B UID), 27B(In case of 4B UID)
	* then its a Protocol Error
	*/
	if((wRxlen != PHAL_MFPRIMENTAG_DEF_VERSION_LENGTH) &&
		(wRxlen != PHAL_MFPRIMENTAG_10B_VERSION_LENGTH) &&
		(wRxlen != PHAL_MFPRIMENTAG_4B_VERSION_LENGTH))
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFPRIMENTAG);
	}

	memcpy(pVerInfo, pRecv, wRxlen); /* PRQA S 3200 */

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}




#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfprimeNtag_Sw_ChangeKey(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint16_t wOption, uint16_t wOldKeyNo, uint16_t wOldKeyVer, uint16_t wNewKeyNo,
	uint16_t wNewKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivLen)
{
	uint16_t    PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[42];
	uint8_t     PH_MEMLOC_REM bWorkBuffer[42];
	uint8_t     PH_MEMLOC_REM bOldKey[32];
	uint8_t     PH_MEMLOC_REM bNewKey[32];
	uint8_t     PH_MEMLOC_REM bNewKeyLen = 0;
	uint8_t     PH_MEMLOC_REM bIndex;
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint16_t    PH_MEMLOC_REM wOldKeyType;
	uint16_t    PH_MEMLOC_REM wNewKeyType;
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint8_t     PH_MEMLOC_REM bCMAC[PH_CRYPTOSYM_AES_BLOCK_SIZE];
	uint8_t     PH_MEMLOC_REM bCMacCard[8];
	uint8_t     PH_MEMLOC_REM bMacLen;
	uint32_t    PH_MEMLOC_REM dwCrc;
	uint8_t     PH_MEMLOC_REM bIvLen = 0;
	uint16_t    PH_MEMLOC_REM wTmp;
	uint16_t    PH_MEMLOC_REM wTmpOption = 0x0000;
	uint8_t     PH_MEMLOC_REM bAppId[3] = {0x00, 0x00, 0x00};
	uint8_t *   PH_MEMLOC_REM pRecv;
	uint16_t    PH_MEMLOC_REM wWorkBufferLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	/* Change key should also take care of changing other keys at PICC level like
	* the Proximity check key,
	* VCA keys
	* Transaction MAC key
	*/
	if(memcmp(pDataParams->pAid, bAppId, 3) == 0x00)
	{
		/* Only if seleted Aid is 0x000000, PICC level key change is targeted. */
		if((bKeyNoCard & 0x80) != 0x80)
		{
			/* Invalid card key number supplied */
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
		else
		{
			if((bKeyNoCard & 0x1F) >= 0x01)
			{
				/* Invalid card key number supplied */
				return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
			}
			else
			{
				/*Do Nothing. This is for PRQA compliance */
			}
		}
	}
	else
	{
		if((bKeyNoCard & 0xC0) != 0x00)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
		else
		{
			if(bKeyNoCard >= 0x03)
			{
				/* Invalid application key specified */
				return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
			}
			else
			{
			}
		}
	}
	if((wOption == 0x0000) || (bDivLen > 31))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}
	if(pDataParams->bAuthMode != PHAL_MFPRIMENTAG_AUTHENTICATEEV2)
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPRIMENTAG);
	}
#endif
	memset(bWorkBuffer, 0x00, 42); /* PRQA S 3200 */
	memset(bCmdBuff, 0x00, 42); /* PRQA S 3200 */
	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_CHANGE_KEY;
	bCmdBuff[wCmdLen++] = bKeyNoCard;

	bIvLen = PH_CRYPTOSYM_AES_BLOCK_SIZE;


	/* the IV is constructed by encrypting with KeyID.SesAuthENCKey according to the ECB mode
	 * As ECB encription doesnot use IV during the encription so we need not backup/ update with zero IV */
	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfprimeNtag_Sw_Int_ComputeIv(
		PH_OFF,
		pDataParams->bTi,
		pDataParams->wCmdCtr,
		pDataParams->bIv
	));

	/* Encrypt IV */
	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_Encrypt(
		pDataParams->pCryptoDataParamsEnc,
		PH_CRYPTOSYM_CIPHER_MODE_ECB,
		pDataParams->bIv,
		bIvLen,
		pDataParams->bIv
	));
	/* Load Iv */
	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
		pDataParams->pCryptoDataParamsEnc,
		pDataParams->bIv,
		bIvLen
	));

	PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
		pDataParams->pKeyStoreDataParams,
		wOldKeyNo,
		wOldKeyVer,
		sizeof(bOldKey),
		bOldKey,
		&wOldKeyType
	));

	PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
		pDataParams->pKeyStoreDataParams,
		wNewKeyNo,
		wNewKeyVer,
		sizeof(bNewKey),
		bNewKey,
		&wNewKeyType
	));

	/*
	It is allowed to change a key type for PICC master key.
	Old key may not be diversified but new key can be.
	Old key may be diversified with one round but new key can
	be diversified with two rounds.

	Key diversification method (DESFire or MFPlus) cannot be changed
	between old and new key.

	It is assumed that the diversification input specified is the same
	for both old key and new key
	*/

	if((wOption != PHAL_MFPRIMENTAG_NO_DIVERSIFICATION) && (bDivLen != 0x00))
	{
		if(wOption & PHAL_MFPRIMENTAG_CHGKEY_DIV_NEW_KEY)
		{
			if(wOption & PHAL_MFPRIMENTAG_CHGKEY_DIV_METHOD_CMAC)
			{
				wTmpOption = PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS;
			}
			else
			{
				wTmpOption = PH_CRYPTOSYM_DIV_MODE_DESFIRE;
				if(wOption & PHAL_MFPRIMENTAG_CHGKEY_DIV_NEW_KEY_ONERND)
				{
					wTmpOption |= PH_CRYPTOSYM_DIV_OPTION_2K3DES_HALF;
				}
			}
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_DiversifyDirectKey(
				pDataParams->pCryptoDataParamsEnc,
				wTmpOption,
				bNewKey,
				wNewKeyType,
				pDivInput,
				bDivLen,
				bNewKey
			));
		}
		if(wOption & PHAL_MFPRIMENTAG_CHGKEY_DIV_OLD_KEY)
		{
			if(wOption & PHAL_MFPRIMENTAG_CHGKEY_DIV_METHOD_CMAC)
			{
				wTmpOption |= PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS;
			}
			else
			{
				wTmpOption |= PH_CRYPTOSYM_DIV_MODE_DESFIRE;
				if(wOption & PHAL_MFPRIMENTAG_CHGKEY_DIV_OLD_KEY_ONERND)
				{
					wTmpOption |= PH_CRYPTOSYM_DIV_OPTION_2K3DES_HALF;
				}
			}
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_DiversifyDirectKey(
				pDataParams->pCryptoDataParamsEnc,
				wTmpOption,
				bOldKey,
				wOldKeyType,
				pDivInput,
				bDivLen,
				bOldKey
			));
		}

		/* Reload the IV and key since the diversify function has invalidated the key */
		/* Load the Session key which is valid for this authentication */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsEnc,
			pDataParams->bSesAuthENCKey,
			pDataParams->bCryptoMethod
		));
		/* The IV will be different if AuthMode is AV2. Here the
		* ENC IV has to be computed and used for encryption.
		* The MAC IV is required to generate the MAC and append this to
		* the command before sending to the card.
		*/

		/* the IV is constructed by encrypting with KeyID.SesAuthENCKey according to the ECB mode
		 * As ECB encription doesnot use IV during the encription so we need not backup/ update with zero IV*/
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfprimeNtag_Sw_Int_ComputeIv(
			PH_OFF,
			pDataParams->bTi,
			pDataParams->wCmdCtr,
			pDataParams->bIv
		));

		/* Encrypt IV */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_Encrypt(
			pDataParams->pCryptoDataParamsEnc,
			PH_CRYPTOSYM_CIPHER_MODE_ECB,
			pDataParams->bIv,
			bIvLen,
			pDataParams->bIv
		));

		/* Load Iv */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
			pDataParams->pCryptoDataParamsEnc,
			pDataParams->bIv,
			bIvLen
		));

		/* Need to check whether this is required for 0x0A mode also*/
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_SetConfig(
			pDataParams->pCryptoDataParamsEnc,
			PH_CRYPTOSYM_CONFIG_KEEP_IV,
			PH_CRYPTOSYM_VALUE_KEEP_IV_ON
		));
	}

	if(wNewKeyType == PH_CRYPTOSYM_KEY_TYPE_AES128)
	{
		bNewKeyLen = PH_CRYPTOSYM_AES128_KEY_SIZE;
	}
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFPRIMENTAG);
	}


	/* Need new else if statement for AuthEV2
	* In AuthEV2, it is sent in FULL mode, meaning the Data is encrypted
	* and a MAC is calculated and attached at the end.
	* ENC Session key is used for encryption and MAC session key is
	* used for MAC calculation
	*/

	if((pDataParams->bKeyNo & 0x3FU) != (bKeyNoCard & 0x3FU))
	{
		/* Copy bNewKey to the bWorkBuffer */
		for(bIndex = 0; bIndex < bNewKeyLen; bIndex++)
		{
			bWorkBuffer[bIndex] = bNewKey[bIndex];
		}

		/* Append Key Version */
		bWorkBuffer[bIndex++] = (uint8_t) wNewKeyVer;

		/* Calculate CRC32 over the new key  ie CRC32NK shall be the 4-byte CRC value taken over NewKey */
		PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc32(
			PH_TOOLS_CRC_OPTION_DEFAULT,
			PH_TOOLS_CRC32_PRESET_DF8,
			PH_TOOLS_CRC32_POLY_DF8,
			bWorkBuffer,
			(uint16_t) bIndex - 1,
			&dwCrc
		));

		/* Key number authenticated with is not the key to be
		* changed
		* xored_Data = pNewKey ^ wKey
		* bWorkBuffer contains pNewKey
		*/
		for(bIndex = 0; bIndex < bNewKeyLen; bIndex++)
		{
			bWorkBuffer[bIndex] = bOldKey[bIndex] ^ bWorkBuffer[bIndex];
		}
		/* xored_Data+ [AES key version] + CRC32 (all prev. data) + CRC32(new key)+padding */
		/* Adding key version should always be true because we are only dealing with
		* AES128 keys here
		*/
		bIndex++;   /* Just increment bIndex because it already contains wNewKeyVer */
		memcpy(&bCmdBuff[2], bWorkBuffer, bIndex); /* PRQA S 3200 */
		wCmdLen = wCmdLen + bIndex;

		memcpy(&bCmdBuff[wCmdLen], &dwCrc, 4); /* PRQA S 3200 */
		wCmdLen += 4;

		wTmp = wCmdLen - 2;
		{
			if(wTmp % bIvLen)
			{
				/* Apply padding */
				PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_ApplyPadding(
					PH_CRYPTOSYM_PADDING_MODE_2,
					&bCmdBuff[2],
					wTmp,
					bIvLen,
					sizeof(bCmdBuff) - 2,
					&bCmdBuff[2],
					&wTmp
				));
			}

			/* Encrypt */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_Encrypt(
				pDataParams->pCryptoDataParamsEnc,
				PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_BUFFER_CONT,
				&bCmdBuff[2],
				wTmp,
				&bCmdBuff[2]
			));
			wCmdLen = wTmp + 2;
		}
		/* Encrypt */
	}
	else
	{
		memcpy(&bCmdBuff[wCmdLen], bNewKey, bNewKeyLen); /* PRQA S 3200 */
		wCmdLen = wCmdLen + bNewKeyLen;

		/* Also check if it is PICC master key and is an AES key is to be written.
		If so then key version also needs to be appended to the new key before
		calculating CRC */
		if(memcmp(pDataParams->pAid, bAppId, 3) == 0x00)
		{
			if((bKeyNoCard & 0xC0) == 0x80)
			{
				/* PICC master key is being changed to AES key. Version is relevant */
				bCmdBuff[wCmdLen++] = (uint8_t) wNewKeyVer;
			}
		}
		else
		{
			bCmdBuff[wCmdLen++] = (uint8_t) wNewKeyVer;
		}

		wTmp = wCmdLen - 2;
		{
			/* Apply padding */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_ApplyPadding(
				PH_CRYPTOSYM_PADDING_MODE_2,
				&bCmdBuff[2],
				wTmp,
				bIvLen,
				sizeof(bCmdBuff) - 2,
				&bCmdBuff[2],
				&wTmp
			));

			/* Encrypt */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_Encrypt(
				pDataParams->pCryptoDataParamsEnc,
				PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_BUFFER_CONT,
				&bCmdBuff[2],
				wTmp,
				&bCmdBuff[2]
			));
			/* Update Cmd len */
			wCmdLen = wTmp + 2;
		}
		/* Encrypt */
	}

	memset(pDataParams->bIv, 0x00, bIvLen);  /* PRQA S 3200 */
	wWorkBufferLen = 0;
	bWorkBuffer[wWorkBufferLen++] = bCmdBuff[0];
	/* Add CmdCtr and TI for MAC calculation */
	bWorkBuffer[wWorkBufferLen++] = (uint8_t) (pDataParams->wCmdCtr);
	bWorkBuffer[wWorkBufferLen++] = (uint8_t) (pDataParams->wCmdCtr >> 8);
	memcpy(&bWorkBuffer[wWorkBufferLen], pDataParams->bTi, PHAL_MFPRIMENTAG_SIZE_TI); /* PRQA S 3200 */
	wWorkBufferLen += PHAL_MFPRIMENTAG_SIZE_TI;

	memcpy(&bWorkBuffer[wWorkBufferLen], &bCmdBuff[1], (wCmdLen - 1)); /* PRQA S 3200 */
	wWorkBufferLen += (wCmdLen - 1);

	/* Load Iv */
	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
		pDataParams->pCryptoDataParamsMac,
		pDataParams->bIv,
		bIvLen
	));

	/* Append MAC */
	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(
		pDataParams->pCryptoDataParamsMac,
		(PH_CRYPTOSYM_MAC_MODE_CMAC),
		bWorkBuffer,
		wWorkBufferLen,
		bCMAC,
		&bMacLen
	));

	/* Truncate the MAC generated */
	phalMfprimeNtag_Sw_Int_TruncateMac(bCMAC);

	memcpy(&bCmdBuff[wCmdLen], bCMAC, 8); /* PRQA S 3200 */
	wCmdLen += 8;

	/* Send the command */
	statusTmp = phalMfprimeNtag_ExchangeCmd(
		pDataParams,
		pDataParams->pPalMifareDataParams,
		pDataParams->bWrappedMode,
		bCmdBuff,
		wCmdLen,
		&pRecv,
		&wRxlen
	);
	if(statusTmp != PH_ERR_SUCCESS)
	{
		phalMfprimeNtag_Sw_Int_ResetAuthStatus(pDataParams);
		return statusTmp;
	}

	/* Max 8 byte CMAC is expected nothing more. */
	if(wRxlen > 8)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFDF);
	}

	memcpy(bWorkBuffer, pRecv, wRxlen);  /* PRQA S 3200 */

	/* Verification of MAC also required for AuthEV2
	*/
	/* reset authentication status if the key authenticated with is changed in
	* this case the card does not return a MAC because authentication is lost
	*/
	if((pDataParams->bKeyNo & 0x3FU) == (bKeyNoCard & 0x3FU))
	{
		phalMfprimeNtag_Sw_Int_ResetAuthStatus(pDataParams);
	}
	else
	{
		if(wRxlen < 8) /* If no CMAC received */
		{
			return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFPRIMENTAG);
		}

		/* Increment the command counter. */
		pDataParams->wCmdCtr++;

		/* Load IV */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
			pDataParams->pCryptoDataParamsMac,
			pDataParams->bIv,
			bIvLen
		));

		/* copy CMAC received from card*/
		memcpy(bCMacCard, &bWorkBuffer[wRxlen - 8], 8);  /* PRQA S 3200 */
		wRxlen -= 8;

		/*
		* Calculate MAC on RC || wCmdCtr || TI || RespData
		* bWorkBuffer is used as receive buffer so pDataParams->pUnprocByteBuff is used
		*/
		pDataParams->bNoUnprocBytes = 0x00;
		pDataParams->pUnprocByteBuff[pDataParams->bNoUnprocBytes++] = 0x00;
		pDataParams->pUnprocByteBuff[pDataParams->bNoUnprocBytes++] = (uint8_t) (pDataParams->wCmdCtr);
		pDataParams->pUnprocByteBuff[pDataParams->bNoUnprocBytes++] = (uint8_t) (pDataParams->wCmdCtr >> 8);
		memcpy(&pDataParams->pUnprocByteBuff[pDataParams->bNoUnprocBytes], pDataParams->bTi, PHAL_MFPRIMENTAG_SIZE_TI); /* PRQA S 3200 */
		pDataParams->bNoUnprocBytes += PHAL_MFPRIMENTAG_SIZE_TI;

		/* verify the MAC */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(
			pDataParams->pCryptoDataParamsMac,
			(PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
			pDataParams->pUnprocByteBuff,
			pDataParams->bNoUnprocBytes,
			bCMAC,
			&bMacLen
		));

		/* Truncate the MAC generated */
		phalMfprimeNtag_Sw_Int_TruncateMac(bCMAC);

		/* Compare the CMAC from card and CMAC calculated */
		if(memcmp(bCMacCard, bCMAC, 8) != 0)
		{
			phalMfprimeNtag_Sw_Int_ResetAuthStatus(pDataParams);
			return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_AL_MFPRIMENTAG);
		}
	}
	/*Do Nothing. This is for PRQA compliance */
	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalMfprimeNtag_Sw_GetKeyVersion(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint8_t bKeyNo, uint8_t bKeySetNo, uint8_t * pKeyVersion, uint8_t * bRxLen)
{
    /**
    * This command can be issued without valid authentication
    */
    uint16_t    PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bCmdBuff[20];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint16_t    PH_MEMLOC_REM wRxlen;
    uint8_t     PH_MEMLOC_REM *pRecv;

#ifdef RDR_LIB_PARAM_CHECK
    uint8_t     PH_MEMLOC_REM bAppId[3] = {0x00, 0x00, 0x00};
	if (memcmp(pDataParams->pAid, bAppId, 3) == 0x00)
    {
		if (bKeyNo  >= 0x05)
		{
        	return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
    }
    else
	{
		if (bKeyNo  >= 0x03)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
	}
#endif
	bKeySetNo = 0x00;
    /* form the command */
    bCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_GET_KEY_VERSION;
    bCmdBuff[wCmdLen++] = bKeyNo;

	/* Check for bit[6] of bKeyNo.
	 * if set then pass bKeySetNo in the command buffer as per ref arch.v15 */

    PH_CHECK_SUCCESS_FCT(statusTmp, phalMfprimeNtag_Sw_Int_ReadData_Plain(
        pDataParams,
        (pDataParams->bAuthMode == PHAL_MFPRIMENTAG_AUTHENTICATEEV2)? PHAL_MFPRIMENTAG_COMMUNICATION_MACD : PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN,
        bCmdBuff,
        wCmdLen,
        &pRecv,
        &wRxlen
        ));

    /*
    * If Key set version retrieval (KeySetNo[b7] is set), then expected wRxlen can be upto 16 bytes(KeySet version ranges between 2 to 16)
    * Else the wRxlen should be equal to 1(1 byte of KeyVer).
    */
	/* If bit[7] of bKeySetNo is set, then wRxLen must not be equal to 0x01 */
    if (wRxlen != 0x01)
    {
		 return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFPRIMENTAG);
    }

    if(memcpy(pKeyVersion, pRecv, wRxlen) == NULL) /* PRQA S 3200 */
	{
		return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_AL_MFPRIMENTAG);
	}

    *bRxLen = (uint8_t)wRxlen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}




phStatus_t phalMfprimeNtag_Sw_GetFileSettings(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint8_t bFileNo, uint8_t * pFSBuffer, uint8_t * pBufferLen)
{
    uint16_t    PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bCmdBuff[8];
    uint16_t    PH_MEMLOC_REM wRxlen;
    uint8_t     PH_MEMLOC_REM *pRecv;

    /* form the command */
    bCmdBuff[0] = PHAL_MFPRIMENTAG_CMD_GET_FILE_SETTINGS;
    bCmdBuff[1] = bFileNo;

    PH_CHECK_SUCCESS_FCT(statusTmp, phalMfprimeNtag_Sw_Int_ReadData_Plain(
        pDataParams,
        (pDataParams->bAuthMode == PHAL_MFPRIMENTAG_AUTHENTICATEEV2)? PHAL_MFPRIMENTAG_COMMUNICATION_MACD : PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN,
        bCmdBuff,
        2,
        &pRecv,
        &wRxlen
        ));

    /*
    * File type can be obtained by reading the zeroth index of the receive buffer
    * For Std data file, pRecv[0] = 0x00
    * For Backup data file, pRecv[0] = 0x01
    * For Value file, pRecv[0] = 0x02
    * For Linear Record file, pRecv[0] = 0x03
    * For Cyclic Record file, pRecv[0] = 0x04
    * For Transaction MAC file, pRecv[0] = 0x05
    * For TransactionMAC file wRxLen = 6
    * For Standard file or Backup file mandatory 7 bytes and optional No. of AAR with AAR are received
    * For Value file mandatory 17 bytes and optional No. of AAR with AAR are received
    * For Linear/Cyclic Record file mandatory 13 bytes and optional No. of AAR with AAR are received
    */

	if(wRxlen > 0)
	{
		memcpy(pFSBuffer, pRecv, wRxlen); /* PRQA S 3200 */
	}
    /* Update pBufferLen and return  */
    *pBufferLen = (uint8_t)wRxlen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfprimeNtag_Sw_GetFileCountersSDM(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t * pFileCounters,
	uint8_t * pRxLen)
{
	uint16_t    PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[20];
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint8_t     PH_MEMLOC_REM *pRecv;

	bCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_GET_SDM_FILE_COUNTER;
	bCmdBuff[wCmdLen++] = bFileNo;

	if(bOption == PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN)
	{
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfprimeNtag_Sw_Int_ReadData_Plain(
			pDataParams,
			PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN,
			bCmdBuff,
			wCmdLen,
			&pRecv,
			&wRxlen
		));
	}
	else
	{
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfprimeNtag_Sw_Int_ReadData_Enc(
			pDataParams,
			PHAL_MFPRIMENTAG_COMMUNICATION_ENC,
			bCmdBuff,
			wCmdLen,
			&pRecv,
			&wRxlen
		));
	}

	/* Response will be received as
	* 1. 3 Byte SDMReadCounter
	* 2. 2 Bytes Reserved
	*/
	if(wRxlen != PHAL_MFPRIMENTAG_SDM_FILE_CTR_LENGTH)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFPRIMENTAG);
	}

	*pRxLen = (uint8_t) wRxlen;
	memcpy(pFileCounters, pRecv, *pRxLen);

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalMfprimeNtag_Sw_ChangeFileSettings(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bCommSett, uint8_t *pAccessRights,
	uint8_t bNumAddARs, uint8_t * pAddARs)
{
    uint8_t     PH_MEMLOC_REM bCmdBuff[24];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
#ifdef RDR_LIB_PARAM_CHECK
    if ((bFileNo != PHAL_MFPRIMENTAG_STANDARD_FILE_ID) && (bFileNo != PHAL_MFPRIMENTAG_SDM_FILE_ID ))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }
#endif
    /* form the command */
    bCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_CHANGE_FILE_SETTINGS;
    bCmdBuff[wCmdLen++] = bFileNo;

	/* Copy communication settings. Communication settings in the first nibble so right shifting */
    bCmdBuff[wCmdLen++] = (uint8_t)(bCommSett >> 4);

    bCmdBuff[wCmdLen++] = pAccessRights[0];
    bCmdBuff[wCmdLen++] = pAccessRights[1];

	/* To avoid warning */
	bNumAddARs =  0;
	pAddARs = NULL;

	if(((bOption & 0xF0U) == PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN))
	{
		/* COMMUNICATION IS PLAIN */
		return phalMfprimeNtag_Sw_Int_Write_Plain(pDataParams,
		PHAL_MFPRIMENTAG_DEFAULT_MODE,
        bCmdBuff,
        wCmdLen,
        PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN,
        NULL,
        0x0000
        );
	}
#ifdef NXPBUILD__PH_CRYPTOSYM
	else if((bOption & 0xF0U) == PHAL_MFPRIMENTAG_COMMUNICATION_MACD)
	{
		/* COMMUNICATION IS Mac */
		return phalMfprimeNtag_Sw_Int_Write_Plain(pDataParams,
		PHAL_MFPRIMENTAG_DEFAULT_MODE,
        bCmdBuff,
        wCmdLen,
        PHAL_MFPRIMENTAG_COMMUNICATION_MACD,
        NULL,
        0x0000
        );
	}
	else if((bOption & 0xF0U) == PHAL_MFPRIMENTAG_COMMUNICATION_ENC)
	{
		return phalMfprimeNtag_Sw_Int_Write_Enc(pDataParams,
			PHAL_MFPRIMENTAG_DEFAULT_MODE,
			bCmdBuff,
			0x0002,
			PH_CRYPTOSYM_PADDING_MODE_1,
			0x00,
			&bCmdBuff[2],
			wCmdLen - 2
			);
    }
#endif /* NXPBUILD__PH_CRYPTOSYM */
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }
}

phStatus_t phalMfprimeNtag_Sw_ChangeFileSettingsSDM(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo, uint8_t bFileOption,
	uint8_t *pAccessRights, uint8_t bSdmOptions, uint8_t *pSdmAccessRights, uint8_t *pVCUIDOffset, uint8_t *pSDMReadCtrOffset, uint8_t *pSDMMACInputOffset,
	uint8_t *pSDMMACOffset)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[24];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	phStatus_t  PH_MEMLOC_REM statusTmp;

#if RDR_LIB_PARAM_CHECK
	if(bFileNo != PHAL_MFPRIMENTAG_SDM_FILE_ID)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}
#endif

	bCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_CHANGE_FILE_SETTINGS;
	bCmdBuff[wCmdLen++] = bFileNo;
	bCmdBuff[wCmdLen++] = bFileOption;
	if(pAccessRights != NULL)
	{
		bCmdBuff[wCmdLen++] = pAccessRights[0];
		bCmdBuff[wCmdLen++] = pAccessRights[1];
	}
	bCmdBuff[wCmdLen++] = bSdmOptions;
	if(pSdmAccessRights != NULL)
	{
		bCmdBuff[wCmdLen++] = pSdmAccessRights[0];
		bCmdBuff[wCmdLen++] = pSdmAccessRights[1];
	}

	if(pVCUIDOffset != NULL)
	{
		bCmdBuff[wCmdLen++] = pVCUIDOffset[0];
		bCmdBuff[wCmdLen++] = pVCUIDOffset[1];
		bCmdBuff[wCmdLen++] = pVCUIDOffset[2];
	}
	if(pSDMReadCtrOffset != NULL)
	{
		bCmdBuff[wCmdLen++] = pSDMReadCtrOffset[0];
		bCmdBuff[wCmdLen++] = pSDMReadCtrOffset[1];
		bCmdBuff[wCmdLen++] = pSDMReadCtrOffset[2];
	}
	if(pSDMMACInputOffset != NULL)
	{
		bCmdBuff[wCmdLen++] = pSDMMACInputOffset[0];
		bCmdBuff[wCmdLen++] = pSDMMACInputOffset[1];
		bCmdBuff[wCmdLen++] = pSDMMACInputOffset[2];
	}
	if(pSDMMACOffset != NULL)
	{
		bCmdBuff[wCmdLen++] = pSDMMACOffset[0];
		bCmdBuff[wCmdLen++] = pSDMMACOffset[1];
		bCmdBuff[wCmdLen++] = pSDMMACOffset[2];
	}

	if(((bOption & 0xF0U) == PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN))
	{
		/* COMMUNICATION IS PLAIN */
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfprimeNtag_Sw_Int_Write_Plain(pDataParams,
			PHAL_MFPRIMENTAG_DEFAULT_MODE,
			bCmdBuff,
			wCmdLen,
			PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN,
			NULL,
			0x0000
		));
	}
#ifdef NXPBUILD__PH_CRYPTOSYM
	else if((bOption & 0xF0U) == PHAL_MFPRIMENTAG_COMMUNICATION_MACD)
	{
		/* COMMUNICATION IS Mac */
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfprimeNtag_Sw_Int_Write_Plain(pDataParams,
			PHAL_MFPRIMENTAG_DEFAULT_MODE,
			bCmdBuff,
			wCmdLen,
			PHAL_MFPRIMENTAG_COMMUNICATION_MACD,
			NULL,
			0x0000
		));
		/* Set SDM Option */
	}
	else if((bOption & 0xF0U) == PHAL_MFPRIMENTAG_COMMUNICATION_ENC)
	{
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfprimeNtag_Sw_Int_Write_Enc(pDataParams,
			PHAL_MFPRIMENTAG_DEFAULT_MODE,
			bCmdBuff,
			0x0002,
			PH_CRYPTOSYM_PADDING_MODE_1,
			0x00,
			&bCmdBuff[2],
			wCmdLen - 2
		));
	}
#endif /* NXPBUILD__PH_CRYPTOSYM */
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}
	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}




phStatus_t phalMfprimeNtag_Sw_ReadData(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo, uint8_t * pOffset,
	uint8_t * pLength, uint8_t ** ppRxdata, uint16_t * pRxdataLen)
{
    /* The signature of this is changed. We include
    * the bIns as third parameter that will differentiate
    * between application chaining and ISO chaining modes
    */
    phStatus_t  PH_MEMLOC_REM status = 0;
    uint8_t     PH_MEMLOC_REM bCmdBuff[16];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint32_t    PH_MEMLOC_REM dwDataLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	/* bit[1] of bIns will also be used. Hence bIns should be checked for above 0x03*/
    if (((bFileNo & 0x1f) >= 0x03) || (bIns > 0x03))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }
#endif
    if ((bOption & 0x0FU) == PH_EXCHANGE_RXCHAINING)
    {
        bCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_RESP_ADDITIONAL_FRAME;
    }
    else if ((bOption & 0x0FU) == PH_EXCHANGE_DEFAULT)
    {
		/* copy data length */
		dwDataLen = pLength[2];
		dwDataLen = dwDataLen << 8 | pLength[1];
		dwDataLen = dwDataLen << 8 | pLength[0];

        if (bIns == 0x00)
        {
            bCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_READ_DATA;
        }
        else
        {
            bCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_READ_DATA_ISO;
        }

        bCmdBuff[wCmdLen++] = bFileNo;
        memcpy(&bCmdBuff[wCmdLen], pOffset, 3); /* PRQA S 3200 */
        wCmdLen += 3;
        memcpy(&bCmdBuff[wCmdLen], pLength, 3); /* PRQA S 3200 */
        wCmdLen += 3;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

    if ((bOption & 0xF0U) == PHAL_MFPRIMENTAG_COMMUNICATION_ENC)
    {
#ifdef NXPBUILD__PH_CRYPTOSYM
        /* Upload Payload size for proper CRC calculation */
        if ((bOption & 0x0FU) !=  PH_EXCHANGE_RXCHAINING)
        {
            pDataParams->dwPayLoadLen = dwDataLen;
        }

        status = phalMfprimeNtag_Sw_Int_ReadData_Enc(
            pDataParams,
            bOption | ((bIns == 0x00) ? PHAL_MFPRIMENTAG_DEFAULT_MODE : PHAL_MFPRIMENTAG_ISO_CHAINING_MODE),
            bCmdBuff,
            wCmdLen,
            ppRxdata,
            pRxdataLen
            );
#endif /* NXPBUILD__PH_CRYPTOSYM */
    }
    else if (((bOption & 0xF0U) == PHAL_MFPRIMENTAG_COMMUNICATION_MACD) ||
        ((bOption & 0xF0U) == PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN))
    {
        status = phalMfprimeNtag_Sw_Int_ReadData_Plain(
            pDataParams,
            bOption | ((bIns == 0x00) ? PHAL_MFPRIMENTAG_DEFAULT_MODE : PHAL_MFPRIMENTAG_ISO_CHAINING_MODE),
            bCmdBuff,
            wCmdLen,
            ppRxdata,
            pRxdataLen
            );
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	if ((status == PH_ERR_SUCCESS) && (bCmdBuff[0] != PHAL_MFPRIMENTAG_RESP_ADDITIONAL_FRAME))
    {
        if ((dwDataLen != *pRxdataLen) && (dwDataLen != 0))
        {
            /* Reset authentication status */
                phalMfprimeNtag_Sw_Int_ResetAuthStatus(pDataParams);
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFPRIMENTAG);
        }
    }
    return status;
}

phStatus_t phalMfprimeNtag_Sw_WriteData(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint8_t bCommOption, uint8_t bIns, uint8_t bFileNo, uint8_t * pOffset,
	uint8_t * pData, uint8_t * pDataLen)
{
    /* The signature of this is changed. We include
    * the bIns as third parameter that will differentiate
    * between application chaining and ISO chaining modes
    */
    phStatus_t  PH_MEMLOC_REM statusTmp = 0;
	uint8_t     PH_MEMLOC_REM bCommOptionTemp = bCommOption;
    uint8_t     PH_MEMLOC_REM bCmdBuff[16];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint16_t    PH_MEMLOC_REM wDataLenTemp;
    uint32_t    PH_MEMLOC_REM dwDataLen;
    uint32_t    PH_MEMLOC_REM dwDataWritten = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if (((bFileNo & 0x7f) > 0x1f) || (bIns > 0x01))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }
    if ((bCommOption != PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN) &&
        (bCommOption != PHAL_MFPRIMENTAG_COMMUNICATION_ENC) &&
        (bCommOption != PHAL_MFPRIMENTAG_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }
#endif
    /* form the command depending on bIns */
    if (bIns == 0x00)
    {
        bCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_WRITE_DATA;
    }
    else
    {
        bCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_WRITE_DATA_ISO;
    }
    bCmdBuff[wCmdLen++] = bFileNo;
    memcpy(&bCmdBuff[wCmdLen], pOffset, 3); /* PRQA S 3200 */
    wCmdLen += 3;
    memcpy(&bCmdBuff[wCmdLen], pDataLen, 3); /* PRQA S 3200 */
    wCmdLen += 3;

	/* copy data length */
	dwDataLen = pDataLen[2];
	dwDataLen = dwDataLen << 8 | pDataLen[1];
	dwDataLen = dwDataLen << 8 | pDataLen[0];

	/* to handle 2 MB of data update maximum of data bytes that can be sent in a single loop */
	wDataLenTemp = (uint16_t)dwDataLen;


	/* update the total number of data to be written
	 * when INS and wrapped mode is enable, we need to send the total number of bytes with Le */
	pDataParams->dwPayLoadLen = dwDataLen;

    if (pDataParams->bAuthMode == PHAL_MFPRIMENTAG_NOT_AUTHENTICATED)
    {
        bCommOption = PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN;
    }
#ifdef NXPBUILD__PH_CRYPTOSYM
	else
	{
		if((bCommOption & 0xF0U) == PHAL_MFPRIMENTAG_COMMUNICATION_MACD)
		{
			pDataParams->dwPayLoadLen = pDataParams->dwPayLoadLen + 0x08;
		}
		if((bCommOption & 0xF0U) == PHAL_MFPRIMENTAG_COMMUNICATION_ENC)
		{
			if(pDataParams->bAuthMode == PHAL_MFPRIMENTAG_AUTHENTICATEEV2)
			{
				pDataParams->dwPayLoadLen = 8 + pDataParams->dwPayLoadLen + ((pDataParams->dwPayLoadLen % PH_CRYPTOSYM_AES_BLOCK_SIZE)?
																		(PH_CRYPTOSYM_AES_BLOCK_SIZE - (pDataParams->dwPayLoadLen % PH_CRYPTOSYM_AES_BLOCK_SIZE)) : PH_CRYPTOSYM_AES_BLOCK_SIZE) ;
			}

		}
	}
#endif /* NXPBUILD__PH_CRYPTOSYM */

    /* Check TMI Collection Status */

		if ((bCommOption & 0xF0U) == PHAL_MFPRIMENTAG_COMMUNICATION_ENC)
		{
#ifdef NXPBUILD__PH_CRYPTOSYM
			statusTmp =  phalMfprimeNtag_Sw_Int_Write_Enc(
				pDataParams,
				((bIns == 0x00) ? PHAL_MFPRIMENTAG_DEFAULT_MODE : PHAL_MFPRIMENTAG_ISO_CHAINING_MODE),
				bCmdBuff,
				wCmdLen,
				PH_CRYPTOSYM_PADDING_MODE_1,
				bCommOptionTemp,
				&pData[dwDataWritten],
				wDataLenTemp
				);
#endif /* NXPBUILD__PH_CRYPTOSYM */
		}
		else
		{
			statusTmp =  phalMfprimeNtag_Sw_Int_Write_Plain(
				pDataParams,
				((bIns == 0x00) ? PHAL_MFPRIMENTAG_DEFAULT_MODE : PHAL_MFPRIMENTAG_ISO_CHAINING_MODE),
				bCmdBuff,
				wCmdLen,
				bCommOptionTemp,
				&pData[dwDataWritten],
				wDataLenTemp
				);
		}
	return statusTmp;

}




phStatus_t phalMfprimeNtag_Sw_IsoSelectFile(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bSelector, uint8_t * pFid, uint8_t * pDFname,
	uint8_t bDFnameLen, uint8_t	bExtendedLenApdu, uint8_t ** ppFCI, uint16_t * pwFCILen)
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bData[24];
    uint32_t     PH_MEMLOC_REM bLc = 0;
	uint32_t     PH_MEMLOC_REM bLe = 0;
	uint8_t		PH_MEMLOC_REM bFileId[3] = {'\0'};
	uint8_t     PH_MEMLOC_REM bPICCDefaultDFName[8] = {0xD2,0x76,0x00,0x00,0x85,0x01,0x00}; /*D2760000850100 PICC DFNAME*/

#ifdef RDR_LIB_PARAM_CHECK
    if (bDFnameLen > 16)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }
    if ((bOption != 0x00) && (bOption != 0x0C))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }
#endif
    if((bSelector ==  0x00) || (bSelector == 0x01) || (bSelector == 0x02))
	{
        /* Selection by EF Id*/
        /* Send MSB first to card */
		if(pFid != NULL)
		{
			 bFileId[1] = bData[0] = pFid[1];
			 bFileId[0] = bData[1] = pFid[0];
			 bFileId[2] = 0x00;
			 bLc = 2;
		}
		else
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
	}
	else if((bSelector ==  0x04))
	{
        /* Selection by DF Name */
		if(pDFname != NULL)
		{
			memcpy(bData, pDFname, bDFnameLen); /* PRQA S 3200 */
			bLc = bDFnameLen;
		}
		else
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
	}
	else
	{
		/* Do Nothing, Pass it to PICC. Let Card throw error */
	}

    PH_CHECK_SUCCESS_FCT(statusTmp, phalMfprimeNtag_Int_Send7816Apdu(
        pDataParams,
        pDataParams->pPalMifareDataParams,
        0x03,
        PHAL_MFPRIMENTAG_CMD_ISO7816_SELECT_FILE,
        bSelector,
        bOption,
        bLc,
		bExtendedLenApdu,
        bData,
        bLe,
        ppFCI,
        pwFCILen));

	/* Reset Authentication should not be targeted for elementary file selection using file ID */
	if(bSelector !=  0x02)
	{
		/* Reset Authentication Status here */
		phalMfprimeNtag_Sw_Int_ResetAuthStatus(pDataParams);
	}

    /* ISO wrapped mode is on */
    pDataParams->bWrappedMode = 1;

	/* once the selection Success, update the File Id to master data structure if the selection is done through AID*/
	if((bSelector ==  0x00) || (bSelector == 0x01) || (bSelector == 0x02))
	{
		memcpy(pDataParams->pAid, bFileId, sizeof(bFileId));
	}
	else /*if((bSelector ==  0x04))*/
    {
		/* Check for PICC Level default DF Name */
		if (memcmp(bPICCDefaultDFName, pDFname, bDFnameLen) == 0)
		{
			bFileId[0] = 0x00;
			bFileId[1] = 0x00;
			bFileId[2] = 0x00;
		}
		else /*if (memcmp(bAppDefaultDFName, pDFname, bDFnameLen) == 0)*/
		{
			/* This is done to avoid session key generation at the time of authenticate API */
			bFileId[0] = 0xff;
			bFileId[1] = 0xff;
			bFileId[2] = 0xff;
		}
		memcpy(pDataParams->pAid, bFileId, sizeof(bFileId));
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfprimeNtag_Sw_IsoReadBinary(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bOffset, uint8_t bSfid, uint32_t dwBytesToRead,
	uint8_t bExtendedLenApdu, uint8_t ** ppRxBuffer, uint32_t * pBytesRead)
{
    uint8_t     PH_MEMLOC_REM bP1 = 0;
    uint8_t     PH_MEMLOC_REM bP2 = 0;
    uint8_t     PH_MEMLOC_REM bCmdBuff[8];
    uint16_t    PH_MEMLOC_REM wOffset;
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint8_t		PH_MEMLOC_REM *pLePtr=NULL;

    if ((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_DEFAULT)
    {
        if (bSfid & 0x80)
        {
#ifdef RDR_LIB_PARAM_CHECK
            /* Short file id is supplied */
            if ((bSfid & 0x7FU) > 0x1F)
            {
                /* Error condition */
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
            }
#endif
            bP1 = bSfid;
            bP2 = bOffset;
        }
        else
        {
            /* P1 and P2 code the offset */
            wOffset = bP1 = bSfid;
            wOffset <<= 8; /* Left shift */
            wOffset |= bOffset;
            bP2 = bOffset;
        }
        pLePtr = (uint8_t *) &dwBytesToRead;
        bCmdBuff[wCmdLen++] = 0x00; /* Class */
        bCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_ISO7816_READ_BINARY; /* Ins */
        bCmdBuff[wCmdLen++] = bP1;
        bCmdBuff[wCmdLen++] = bP2;

		/* Check whether Length Le should be represented in Short APDU or extended length APDU */
		if(bExtendedLenApdu == 0x01)
		{
			/*
			 * An extended Le field consists of either three bytes (one * byte set to '00' followed by two bytes with any
			 * value) if the Lc field is absent, or two bytes (with any * value) if an extended Lc field is present.
			 * From '0001' to 'FFFF', the two bytes encode Ne from one
			 * to 65 535.
			 * If the two bytes are set to '0000', then Ne is 65 536.
			 */
			 bCmdBuff[wCmdLen++] = 0x00;
			 bCmdBuff[wCmdLen++] = *(pLePtr + 1);
			 bCmdBuff[wCmdLen++] = *(pLePtr);
			 /* Need to handle the case where the expected data to be read is more than 0xFFFF */
		}
		else
		{
			/* Short APDU */
			bCmdBuff[wCmdLen++] = *(pLePtr);
		}
    }
    else if ((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_RXCHAINING)
    {
        wCmdLen = 0;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

    return phalMfprimeNtag_Sw_Int_IsoRead(
        pDataParams,
        wOption,
        bCmdBuff,
        wCmdLen,
        ppRxBuffer,
        pBytesRead
        );
}

phStatus_t phalMfprimeNtag_Sw_IsoUpdateBinary(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint8_t bOffset, uint8_t bSfid, uint8_t bExtendedLenApdu, uint8_t * pData,
	uint32_t dwDataLen)

{
    uint8_t     PH_MEMLOC_REM bP1 = 0;
    uint8_t     PH_MEMLOC_REM bP2 = 0;
    uint32_t    PH_MEMLOC_REM bLc = 0;
    uint16_t    PH_MEMLOC_REM wOffset;
    phStatus_t  PH_MEMLOC_REM status;

    if (bSfid & 0x80)
    {
#ifdef RDR_LIB_PARAM_CHECK
        /* Short file id is supplied */
        if ((bSfid & 0x7FU) > 0x1F)
        {
            /* Error condition */
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
        }
#endif
		bP1 = bSfid;
        bP2 = bOffset;
    }
    else
    {
        /* P1 and P2 code the offset */
        wOffset = bP1 = bSfid;
        wOffset <<= 8; /* Left shift */
        wOffset |= bOffset;
        bP2 = bOffset;
    }

    bLc = dwDataLen;
    status = phalMfprimeNtag_Int_Send7816Apdu(
        pDataParams,
        pDataParams->pPalMifareDataParams,
        0x01,
        PHAL_MFPRIMENTAG_CMD_ISO7816_UPDATE_BINARY,
        bP1,
        bP2,
        bLc,
		bExtendedLenApdu,
        pData,
        0x00,
        NULL,
        NULL
		);

    if (status != PH_ERR_SUCCESS)
    {
        /* Reset authentication status */
        phalMfprimeNtag_Sw_Int_ResetAuthStatus(pDataParams);
    }

    return status;
}




phStatus_t phalMfprimeNtag_Sw_ReadSign(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint8_t bAddr, uint8_t ** pSignature)
{

	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[2];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint16_t    PH_MEMLOC_REM wRxLength = 0;

	/* build command frame */
	bCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_READ_SIG;
	bCmdBuff[wCmdLen++] = bAddr;

	/* Req spec(ver 0.14 says),
	* 1. Cmd.Read_Sig shall return the NXPOriginalitySignature as written during wafer test in plain if not authenticated
	* 2. Cmd.Read_Sig shall require MACed command if authenticated.
	*/
	if(pDataParams->bAuthMode == PHAL_MFPRIMENTAG_NOT_AUTHENTICATED)
	{
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfprimeNtag_Sw_Int_ReadData_Plain(
			pDataParams,
			PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN,
			bCmdBuff,
			wCmdLen,
			pSignature,
			&wRxLength
		));
	}
#ifdef NXPBUILD__PH_CRYPTOSYM
	else
	{
		/* Set the expected data length as 56 bytes */
		pDataParams->dwPayLoadLen = PHAL_MFPRIMENTAG_SIG_LENGTH;

		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfprimeNtag_Sw_Int_ReadData_Enc(
			pDataParams,
			PHAL_MFPRIMENTAG_COMMUNICATION_ENC,
			bCmdBuff,
			wCmdLen,
			pSignature,
			&wRxLength
		));
	}
#endif /* NXPBUILD__PH_CRYPTOSYM */
	/* check received length :- 56 byte signature */
	if(wRxLength != PHAL_MFPRIMENTAG_SIG_LENGTH)
	{
		return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_AL_MFPRIMENTAG);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);

}




phStatus_t phalMfprimeNtag_Sw_GetConfig(phalMfprimeNtag_Sw_DataParams_t *pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    switch (wConfig)
    {
    case PHAL_MFPRIMENTAG_ADDITIONAL_INFO:
        *pValue = pDataParams->wAdditionalInfo;
        break;


    case PHAL_MFPRIMENTAG_SHORT_LENGTH_APDU:
        *pValue = (uint16_t)pDataParams->bShortLenApdu;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfprimeNtag_Sw_SetConfig(phalMfprimeNtag_Sw_DataParams_t * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    switch (wConfig)
    {
    case PHAL_MFPRIMENTAG_ADDITIONAL_INFO:
        pDataParams->wAdditionalInfo = wValue;
        break;

    case PHAL_MFPRIMENTAG_SHORT_LENGTH_APDU:
        pDataParams->bShortLenApdu = (uint8_t)wValue;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfprimeNtag_Sw_ResetAuthentication(phalMfprimeNtag_Sw_DataParams_t * pDataParams)
{
    phalMfprimeNtag_Sw_Int_ResetAuthStatus(pDataParams);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfprimeNtag_Sw_CalculateMACSDM(phalMfprimeNtag_Sw_DataParams_t *pDataParams, uint16_t wOption, uint8_t bSdmOption, uint16_t wSDMFileReadKeyNo,
	uint16_t wSDMFileReadKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr, uint8_t * pIndata,
	uint16_t wInDataLen, uint8_t * pRespMac)
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bKey[PH_CRYPTOSYM_AES128_KEY_SIZE];
    uint16_t    PH_MEMLOC_REM wKeyType;
    uint8_t     PH_MEMLOC_REM bTmpIV[PH_CRYPTOSYM_AES_BLOCK_SIZE];
    uint8_t     PH_MEMLOC_REM bSV[PH_CRYPTOSYM_AES128_KEY_SIZE];
    uint8_t     PH_MEMLOC_REM bCMAC[PH_CRYPTOSYM_AES_BLOCK_SIZE];
    uint32_t    PH_MEMLOC_REM dwSDMctr = 0;
	uint32_t    PH_MEMLOC_REM dwSDMctrtemp = 0;
    uint8_t     PH_MEMLOC_REM bSVMacLen = 0;

	pDivInput =  NULL; /* Avoid Warnings */
	bDivInputLen = 0x00; /* Avoid Warnings */
	wOption =  PHAL_MFPRIMENTAG_NO_DIVERSIFICATION; /* Avoid Warnings */

	if(pSDMReadCtr != NULL)
	{
		/* Formation of SDM Ctr as double word value- SDM Ctr shall be communicated LSB first. */
		dwSDMctr = pSDMReadCtr[0];
		dwSDMctrtemp = pSDMReadCtr[1];
		dwSDMctr |= (dwSDMctrtemp << 8);
		dwSDMctrtemp = pSDMReadCtr[2];
		dwSDMctr |= (dwSDMctrtemp << 16);
		dwSDMctrtemp = pSDMReadCtr[3];
		dwSDMctr |= (dwSDMctrtemp << 24);
	}

	/* If SDM Ctr is 0xFFFFFF, then return error */
	if (dwSDMctr == 0xFFFFFF)
	{
		return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Get Key out of the key store object */
	PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
		pDataParams->pKeyStoreDataParams,
		wSDMFileReadKeyNo,
		wSDMFileReadKeyVer,
		PH_CRYPTOSYM_AES128_KEY_SIZE,
		bKey,
		&wKeyType
		));

	/* Invalid key type at wKeyNoTMACKey and wKeyVerTMACKey */
	if (wKeyType != PH_CRYPTOSYM_KEY_TYPE_AES128)
	{
		return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFPRIMENTAG);
	}

	/* SV2 = 0x3C || 0xC3 || 0x00 || 0x01 || 0x00 || 0x80 [|| VCUID ][|| SDMReadCtr][||ZeroPadding] */
	/* Padding not required as we have 16 Bytes block */
	bSV[bSVMacLen++] = 0x3C;
	bSV[bSVMacLen++] = 0xC3;
	bSV[bSVMacLen++] = 0x00;
	bSV[bSVMacLen++] = 0x01;
	bSV[bSVMacLen++] = 0x00;
	bSV[bSVMacLen++] = 0x80;

	/* Copy UID into SV - UID should not more then 7 byte in
		* this case else bSV array size needs to be changed
		*/
	if(bSdmOption & PHAL_MFPRIMENTAG_VCUID_PRESENT)
	{
		/* If VCUID should be considered for MAC calcluation, and pUID is passed as NULL, throw error */
		if(pUid != NULL)
		{
			memcpy(&bSV[bSVMacLen], pUid, bUidLen); /* PRQA S 3200 */
			bSVMacLen += bUidLen;
		}
		else
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
	}

	if(bSdmOption & PHAL_MFPRIMENTAG_RDCTR_PRESENT)
	{
		/* If SDMReadCounter should be considered for MAC calcluation, and pUID is passed as NULL, throw error */
		if(dwSDMctr != NULL)
		{
			bSV[bSVMacLen++] = (uint8_t)(dwSDMctr & 0xFF);
			bSV[bSVMacLen++] = (uint8_t)((dwSDMctr >> 8) & 0xFF);
			bSV[bSVMacLen++] = (uint8_t)((dwSDMctr >> 16) & 0xFF);
		}
		else
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
	}

	/* SV padded with the zero bytes up to a length of 16 bytes (if needed)*/
	if(bSVMacLen < PH_CRYPTOSYM_AES128_KEY_SIZE)
	{
		memset(&bSV[bSVMacLen], 0x00, (PH_CRYPTOSYM_AES128_KEY_SIZE - bSVMacLen)); /* PRQA S 3200 */
	}

	/* load key */
	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
		pDataParams->pCryptoDataParamsMac,
		bKey,
		wKeyType));

	/* Create a Back up of the current IV */
	memcpy(bTmpIV, pDataParams->bIv, PH_CRYPTOSYM_AES_BLOCK_SIZE); /* PRQA S 3200 */

	/* Load zero to IV */
	memset(pDataParams->bIv, 0x00, PH_CRYPTOSYM_AES_BLOCK_SIZE);  /* PRQA S 3200 */

	/* Load Iv */
	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
		pDataParams->pCryptoDataParamsMac,
		pDataParams->bIv,
		PH_CRYPTOSYM_AES_BLOCK_SIZE
		));

	/* MAC SV to obtain KSesSDMFileReadMAC */
	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(
		pDataParams->pCryptoDataParamsMac,
		(PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
		bSV,
		PH_CRYPTOSYM_AES_BLOCK_SIZE,
		bKey,
		&bSVMacLen
		));

	/* load KSesSDMFileReadMAC */
	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
		pDataParams->pCryptoDataParamsMac,
		bKey,
		PH_CRYPTOSYM_KEY_TYPE_AES128));

	/* Load zero to IV */
	memset(pDataParams->bIv, 0x00, PH_CRYPTOSYM_AES_BLOCK_SIZE);  /* PRQA S 3200 */

	/* Load Iv */
	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
		pDataParams->pCryptoDataParamsMac,
		pDataParams->bIv,
		PH_CRYPTOSYM_AES_BLOCK_SIZE
		));

	/* Mac The Input Data using K(sessionSDMMacKey) to obtain SDMMac.  */
	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(
		pDataParams->pCryptoDataParamsMac,
		(PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
		pIndata,
		wInDataLen,
		bCMAC,
		&bSVMacLen
		));

	/* Truncate the MAC generated */
	phalMfprimeNtag_Sw_Int_TruncateMac(bCMAC);

	/* Copy the Truncated CMAC into the return buffer */
	memcpy(pRespMac, bCMAC, PHAL_MFPRIMENTAG_TRUNCATED_MAC_SIZE); /* PRQA S 3200 */

	/* Restore back the IV */
	memcpy(pDataParams->bIv, bTmpIV, PH_CRYPTOSYM_AES_BLOCK_SIZE); /* PRQA S 3200 */

	if(pDataParams->bAuthMode != PHAL_MFPRIMENTAG_NOT_AUTHENTICATED)
	{
		/* Load the session key */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsMac,
			pDataParams->bSesAuthMACKey,
			pDataParams->bCryptoMethod
			));
	}
	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SW */
