/*
 * Copyright 2013 - 2017, 2019 - 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal functions of Software implementation of MIFARE (R) Ultralight application layer.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phpalMifare.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHAL_MFUL

#include <phCryptoSym.h>
#include <phalMful.h>

#include "phalMful_Int.h"

phStatus_t phalMful_Int_Read(void * pDataParams, uint8_t bAddress, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Process CMAC and exchange the command. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_Exchange(
        pDataParams,
        PHAL_MFUL_CMD_READ,
        &bAddress,
        1U,
        &pResponse,
        &wRespLen));

    /* Check received length. */
    if(wRespLen != PHAL_MFUL_READ_BLOCK_LENGTH)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFUL);
    }

    /* Copy received data block. */
    (void) memcpy(pData, pResponse, wRespLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_Write(void * pDataParams, uint8_t bAddress, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aDataBuff[5];
    uint8_t     PH_MEMLOC_REM bDataBufLen = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame data information. */
    aDataBuff[bDataBufLen++] = bAddress;
    (void) memcpy(&aDataBuff[bDataBufLen], pData, PHAL_MFUL_WRITE_BLOCK_LENGTH);
    bDataBufLen += PHAL_MFUL_WRITE_BLOCK_LENGTH;

    /* Process CMAC and exchange the command. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_Exchange(
        pDataParams,
        PHAL_MFUL_CMD_WRITE,
        aDataBuff,
        bDataBufLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_FastWrite(void * pDataParams, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t     PH_MEMLOC_REM bCommand[3];
    uint8_t *   PH_MEMLOC_REM pResponse;
    uint16_t    PH_MEMLOC_REM wRspLen;
    void *      PH_MEMLOC_REM pPalMifareDataParams = NULL;

    /* Get the PAL DataParams. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_GetPalMifareDataParams(pDataParams, &pPalMifareDataParams));

    /* Build command frame */
    bCommand[0U] = PHAL_MFUL_CMD_FAST_WRITE;
    bCommand[1U] = 0xF0U; /* Start Address */
    bCommand[2U] = 0xFFU; /* End Address */

    /* Buffer the command frame */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
        pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        bCommand,
        3U,
        &pResponse,
        &wRspLen));

    /* Transmit the data */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
        pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pData,
        (uint16_t) (16U * PHAL_MFUL_WRITE_BLOCK_LENGTH),
        &pResponse,
        &wRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_CompatibilityWrite(void * pDataParams, uint8_t bAddress, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t     PH_MEMLOC_REM bCommand[2];
    uint8_t *   PH_MEMLOC_REM pResponse;
    uint16_t    PH_MEMLOC_REM wRspLen;
    void *      PH_MEMLOC_REM pPalMifareDataParams = NULL;

    /* Get the PAL DataParams. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_GetPalMifareDataParams(pDataParams, &pPalMifareDataParams));

    /* Build command frame */
    bCommand[0U] = PHAL_MFUL_CMD_COMPWRITE;
    bCommand[1U] = bAddress;

    /* Send the first part */
    status = phpalMifare_ExchangeL3(
        pPalMifareDataParams,
        PH_EXCHANGE_DEFAULT,
        bCommand,
        2U,
        &pResponse,
        &wRspLen);

    /* Either ACK (newer UL cards) or TO (older UL cards) is expected */
    if((status & PH_ERR_MASK) != PH_ERR_IO_TIMEOUT)
    {
        PH_CHECK_SUCCESS(status);
    }

    /* Buffer the data */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
        pPalMifareDataParams,
        PH_EXCHANGE_DEFAULT,
        pData,
        PHAL_MFUL_COMPWRITE_BLOCK_LENGTH,
        &pResponse,
        &wRspLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_IncrCnt(void * pDataParams, uint8_t bCntNum, uint8_t * pCnt)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aDataBuff[5];
    uint8_t     PH_MEMLOC_REM bDataBufLen = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame data information. */
    aDataBuff[bDataBufLen++] = bCntNum;
    (void) memcpy(&aDataBuff[bDataBufLen], pCnt, PHAL_MFUL_COUNTER_WR_VALUE_LENGTH);
    bDataBufLen += PHAL_MFUL_COUNTER_WR_VALUE_LENGTH;

    /* Process CMAC and exchange the command. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_Exchange(
        pDataParams,
        PHAL_MFUL_CMD_INCR_CNT,
        aDataBuff,
        bDataBufLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_ReadCnt(void * pDataParams, uint8_t bCntNum, uint8_t * pCntValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Process CMAC and exchange the command. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_Exchange(
        pDataParams,
        PHAL_MFUL_CMD_READ_CNT,
        &bCntNum,
        1U,
        &pResponse,
        &wRespLen));

    /* Check received length. */
    if(wRespLen != PHAL_MFUL_COUNTER_RD_VALUE_LENGTH)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFUL);
    }

    /* Copy received data block. */
    (void) memcpy(pCntValue, pResponse, wRespLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_PwdAuth(void * pDataParams, uint8_t * pPwd, uint8_t * pPack)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t     PH_MEMLOC_REM bCommand;
    uint8_t *   PH_MEMLOC_REM pResponse;
    uint16_t    PH_MEMLOC_REM wRspLen;
    void *      PH_MEMLOC_REM pPalMifareDataParams = NULL;

    /* Get the PAL DataParams. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_GetPalMifareDataParams(pDataParams, &pPalMifareDataParams));

    /* Build command frame */
    bCommand = PHAL_MFUL_CMD_PWD_AUTH;

    /* Transmit the command frame */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
        pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        &bCommand,
        1U,
        &pResponse,
        &wRspLen));

    /* Transmit the data */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
        pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pPwd,
        PHAL_MFUL_WRITE_BLOCK_LENGTH,
        &pResponse,
        &wRspLen));

    /* Check received length */
    if(wRspLen != PHAL_MFUL_PACK_LENGTH)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFUL);
    }

    /* Copy received data block */
    (void) memcpy(pPack, pResponse, wRspLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_GetVersion(void * pDataParams, uint8_t * pVersion)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Process CMAC and exchange the command. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_Exchange(
        pDataParams,
        PHAL_MFUL_CMD_GET_VER,
        NULL,
        0U,
        &pResponse,
        &wRespLen));

    /* Check received length. */
    if(wRespLen != PHAL_MFUL_VERSION_LENGTH)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFUL);
    }

    /* Copy received data block. */
    (void) memcpy(pVersion, pResponse, wRespLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_FastRead(void * pDataParams, uint8_t  bStartAddr, uint8_t bEndAddr,
    uint8_t ** ppData, uint16_t * pNumBytes)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aDataBuff[5];
    uint8_t     PH_MEMLOC_REM bDataBufLen = 0;

    /* Frame data information. */
    aDataBuff[bDataBufLen++] = bStartAddr;
    aDataBuff[bDataBufLen++] = bEndAddr;

    /* Process CMAC and exchange the command. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_Exchange(
        pDataParams,
        PHAL_MFUL_CMD_FAST_READ,
        aDataBuff,
        bDataBufLen,
        ppData,
        pNumBytes));

    /* Check received length */
    if(*pNumBytes != ((uint16_t) ((bEndAddr - bStartAddr + 1U) * 4U)))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFUL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_SectorSelect(void * pDataParams, uint8_t bSecNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t     PH_MEMLOC_REM bCommand1[2];
    uint8_t     PH_MEMLOC_REM bCommand2[4];
    uint8_t *   PH_MEMLOC_REM pResponse;
    uint16_t    PH_MEMLOC_REM wRspLen;
    void *      PH_MEMLOC_REM pPalMifareDataParams = NULL;

    /* Get the PAL DataParams. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_GetPalMifareDataParams(pDataParams, &pPalMifareDataParams));

    /* build command frame packet 1 */
    bCommand1[0U] = PHAL_MFUL_CMD_SECTOR_SELECT;
    bCommand1[1U] = 0xFFU;

    /* transmit the command frame packet 1 */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
        pPalMifareDataParams,
        PH_EXCHANGE_DEFAULT,
        bCommand1,
        2U,
        &pResponse,
        &wRspLen));

    /* Build command frame packet 2 */
    bCommand2[0U] = bSecNo;
    bCommand2[1U] = 0x00U;
    bCommand2[2U] = 0x00U;
    bCommand2[3U] = 0x00U;

    /* transmit the command frame packet 2 */
    wStatus = phpalMifare_ExchangeL3(
        pPalMifareDataParams,
        PH_EXCHANGE_DEFAULT,
        bCommand2,
        4U,
        &pResponse,
        &wRspLen);

    /* No response expected for packet 2 */
    if((wStatus & PH_ERR_MASK) != PH_ERR_IO_TIMEOUT)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFUL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_ReadSign(void * pDataParams, uint8_t bAddr, uint8_t ** pSignature, uint16_t* pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Process CMAC and exchange the command. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_Exchange(
        pDataParams,
        PHAL_MFUL_CMD_READ_SIG,
        &bAddr,
        1U,
        pSignature,
        pDataLen));

    /* Check received length. */
    if((*pDataLen != PHAL_MFUL_SIG_LENGTH_32)
#ifdef NXPBUILD__PH_CRYPTOASYM
        && (*pDataLen != PHAL_MFUL_SIG_LENGTH_48)
#endif /* NXPBUILD__PH_CRYPTOASYM */
        )
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFUL);
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_SetConfig(pDataParams, PHAL_MFUL_ADDITIONAL_INFO, *pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_ChkTearingEvent(void * pDataParams, uint8_t bCntNum, uint8_t * pValidFlag)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t     PH_MEMLOC_REM bCommand[2];
    uint8_t *   PH_MEMLOC_REM pResponse;
    uint16_t    PH_MEMLOC_REM wRspLen;
    void *      PH_MEMLOC_REM pPalMifareDataParams = NULL;

    /* Get the PAL DataParams. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_GetPalMifareDataParams(pDataParams, &pPalMifareDataParams));

    /* build command frame */
    bCommand[0U] = PHAL_MFUL_CMD_CHK_TRG_EVT;
    bCommand[1U] = bCntNum;

    /* transmit the command frame */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
        pPalMifareDataParams,
        PH_EXCHANGE_DEFAULT,
        bCommand,
        2U,
        &pResponse,
        &wRspLen));


    /* check received length */
    if(wRspLen != 1U)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFUL);
    }

    /* copy received data block */
    (void) memcpy(pValidFlag, pResponse, wRspLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_WriteSign(void * pDataParams, uint8_t bAddress, uint8_t * pSignature)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aDataBuff[5];
    uint8_t     PH_MEMLOC_REM bDataBufLen = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame data information. */
    aDataBuff[bDataBufLen++] = bAddress;
    (void) memcpy(&aDataBuff[bDataBufLen], pSignature, PHAL_MFUL_WRITE_BLOCK_LENGTH);
    bDataBufLen += PHAL_MFUL_WRITE_BLOCK_LENGTH;

    /* Process CMAC and exchange the command. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_Exchange(
        pDataParams,
        PHAL_MFUL_CMD_WRITE_SIGN,
        aDataBuff,
        bDataBufLen,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_LockSign(void * pDataParams, uint8_t bLockMode)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Process CMAC and exchange the command. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_Exchange(
        pDataParams,
        PHAL_MFUL_CMD_LOCK_SIGN,
        &bLockMode,
        1U,
        &pResponse,
        &wRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_VirtualCardSelect(void * pDataParams, uint8_t * pVCIID, uint8_t bVCIIDLen,
    uint8_t * pVCTID)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t     PH_MEMLOC_REM aCommand[1];
    uint8_t *   PH_MEMLOC_REM pResponse;
    uint16_t    PH_MEMLOC_REM wRspLen;
    void *      PH_MEMLOC_REM pPalMifareDataParams = NULL;

    /* Get the PAL DataParams. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_GetPalMifareDataParams(pDataParams, &pPalMifareDataParams));

    /*Frame the command buffer. */
    aCommand[0U] = PHAL_MFUL_CMD_VCSL;

    /* Transmit the command */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
        pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        aCommand,
        1U,
        &pResponse,
        &wRspLen));

    /* Transmit the IID data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
        pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pVCIID,
        bVCIIDLen,
        &pResponse,
        &wRspLen));

    /* Check received length */
    if(wRspLen != 1U)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFUL);
    }

    /* Copy received data block */
    (void) memcpy(pVCTID, pResponse, wRspLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_ReadTTStatus(void * pDataParams, uint8_t bAddr, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t     PH_MEMLOC_REM aCommand[2];
    uint8_t *   PH_MEMLOC_REM pResponse;
    uint16_t    PH_MEMLOC_REM wRspLen;
    void *      PH_MEMLOC_REM pPalMifareDataParams = NULL;

    /* Get the PAL DataParams. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_GetPalMifareDataParams(pDataParams, &pPalMifareDataParams));

    /*Frame the command buffer. */
    aCommand[0U] = PHAL_MFUL_CMD_READ_TT_STATUS;
    aCommand[1U] = bAddr;

    /* Transmit the command */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
        pPalMifareDataParams,
        PH_EXCHANGE_DEFAULT,
        aCommand,
        0x02U,
        &pResponse,
        &wRspLen));

    /* check received length */
    if(wRspLen != 5U)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFUL);
    }

    /* copy received data block */
    (void) memcpy(pData, pResponse, wRspLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_Exchange(void * pDataParams, uint8_t bCmd, uint8_t * pData, uint8_t bDataLen,
    uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aMac[PH_CRYPTOSYM_AES_BLOCK_SIZE];
    uint8_t     PH_MEMLOC_REM bMacLen = 0;
    void *      PH_MEMLOC_REM pPalMifareDataParams = NULL;

#ifdef NXPBUILD__PH_CRYPTOSYM
    uint8_t     PH_MEMLOC_REM bCMACVal = NULL;
    uint8_t     PH_MEMLOC_REM bAuthMode = NULL;
    uint8_t     PH_MEMLOC_REM aInBuff[256];
    uint8_t     PH_MEMLOC_REM bInBufLen = 0;
    uint8_t     PH_MEMLOC_REM aIV[PH_CRYPTOSYM_AES_BLOCK_SIZE];

#ifdef NXPBUILD__PHAL_MFUL_SAMAV3_NONX
    uint8_t *   PH_MEMLOC_REM pMac = NULL;
    uint16_t    PH_MEMLOC_REM wMLen = 0;
#endif /* NXPBUILD__PHAL_MFUL_SAMAV3_NONX */

    /* Set the buffers Buffer to zero. */
    (void) memset(aIV, 0x00U, sizeof(aIV));
#endif /* NXPBUILD__PH_CRYPTOSYM */

    /* Get the PAL DataParams. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_GetPalMifareDataParams(pDataParams, &pPalMifareDataParams));

#ifdef NXPBUILD__PH_CRYPTOSYM
    /* Get the CMAC Value Enabled or Disabled. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_GetCMAC(pDataParams, &bCMACVal));

    /* Get the Auth Mode */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_GetAuthMode(pDataParams, &bAuthMode));

    /* Check if CMAC is enabled. */
    if(bCMACVal)
    {
        /* Check if its Authenticated. */
        if(bAuthMode == PHAL_MFUL_CMD_AUTH)
        {
            /* Compute CMAC for Software mode */
            if(PH_GET_COMPID(pDataParams) == PHAL_MFUL_SW_ID)
            {
                /* Frame the MAC input buffer.
                    * wCmdCtr || Cmd || CmdData
                    */
                aInBuff[bInBufLen++] = (uint8_t) (((phalMful_Sw_DataParams_t *) pDataParams)->wCmdCtr);
                aInBuff[bInBufLen++] = (uint8_t) (((phalMful_Sw_DataParams_t *) pDataParams)->wCmdCtr >> 8U);
                aInBuff[bInBufLen++] = bCmd;
                memcpy(&aInBuff[bInBufLen], pData, bDataLen); /* PRQA S 3200 */
                bInBufLen += bDataLen;

                /* Load the IV. */
                PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
                    ((phalMful_Sw_DataParams_t *) pDataParams)->pCryptoDataParams,
                    aIV,
                    PH_CRYPTOSYM_AES_BLOCK_SIZE));

                /* Compute the MAC. */
                PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
                    ((phalMful_Sw_DataParams_t *) pDataParams)->pCryptoDataParams,
                    (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
                    aInBuff,
                    bInBufLen,
                    aMac,
                    &bMacLen));

                /* Truncate the MAC generated */
                PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_TruncateMac(aMac));
                bMacLen = 8U; /* Truncated length. */

                /* Reset Input buffer length. */
                bInBufLen = 0U;
            }
#ifdef NXPBUILD__PHAL_MFUL_SAMAV3_NONX
            else if(PH_GET_COMPID(pDataParams) == PHAL_MFUL_SAMAV3_NONX_ID)
            {
                /* Perform Offline activation using Ram Key. */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
                    (((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->pHalSamDataParams),
                    PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
                    (((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->bRamKeyNo),
                    (((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->bRamKeyVer),
                    NULL,
                    0U));

                /* Frame the MAC input buffer.
                 * wCmdCtr || Cmd || CmdData
                 */
                aInBuff[bInBufLen++] = (uint8_t) (((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->wCmdCtr);
                aInBuff[bInBufLen++] = (uint8_t) (((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->wCmdCtr >> 8U);
                aInBuff[bInBufLen++] = bCmd;
                (void) memcpy(&aInBuff[bInBufLen], pData, bDataLen);
                bInBufLen += bDataLen;

                /* Load zero IV. */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_LoadInitVector(
                    (((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->pHalSamDataParams),
                    PHHAL_HW_SAMAV3_CMD_SAM_LOAD_IV_MODE_SET_IV,
                    aIV,
                    PH_CRYPTOSYM_AES_BLOCK_SIZE));

                /* Generate Mac. */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_GenerateMAC(
                    (((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->pHalSamDataParams),
                    PH_EXCHANGE_DEFAULT,
                    PHHAL_HW_SAMAV3_TRUNCATION_MODE_MFP,
                    aInBuff,
                    bInBufLen,
                    &pMac,
                    &wMLen));

                /* Copy the Mac to the parameter. */
                (void) memcpy(aMac, pMac, wMLen);
                bMacLen = (uint8_t) wMLen;

                /* Reset Input buffer length. */
                bInBufLen = 0U;
            }
#endif /* NXPBUILD__PHAL_MFUL_SAMAV3_NONX */
            else
            {
                /* Nothing to do. */
            }
        }
    }
    else
    {
        /* Nothing to do. */
    }
#endif /* NXPBUILD__PH_CRYPTOSYM */

    /* Buffer the command information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
        pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        &bCmd,
        1U,
        NULL,
        NULL));

    /* Buffer the data information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
        pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pData,
        bDataLen,
        NULL,
        NULL));

    /* Buffer the mac information and exchange the buffered information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
        pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        aMac,
        bMacLen,
        ppResponse,
        pRespLen));

#ifdef NXPBUILD__PH_CRYPTOSYM
    /* Check if CMAC is enabled. */
    if(bCMACVal)
    {
        /* Check if its Authenticated. */
        if(bAuthMode == PHAL_MFUL_CMD_AUTH)
        {
            /* Verify CMAC for Software mode only.*/
            if(PH_GET_COMPID(pDataParams) == PHAL_MFUL_SW_ID)
            {
                /* Increment the command counter. */
                ((phalMful_Sw_DataParams_t *) pDataParams)->wCmdCtr++;

                /* Frame the MAC input buffer.
                 * wCmdCtr || CmdData
                 */
                aInBuff[bInBufLen++] = (uint8_t) (((phalMful_Sw_DataParams_t *) pDataParams)->wCmdCtr);
                aInBuff[bInBufLen++] = (uint8_t) (((phalMful_Sw_DataParams_t *) pDataParams)->wCmdCtr >> 8U);

                (void) memcpy(&aInBuff[bInBufLen], *ppResponse, (*pRespLen - 8 /* MAC */));
                bInBufLen += (uint8_t) (*pRespLen - 8U /* MAC */);

                /* Load the IV. */
                PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
                    ((phalMful_Sw_DataParams_t *) pDataParams)->pCryptoDataParams,
                    aIV,
                    PH_CRYPTOSYM_AES_BLOCK_SIZE));

                /* Compute the MAC. */
                PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
                    ((phalMful_Sw_DataParams_t *) pDataParams)->pCryptoDataParams,
                    (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
                    aInBuff,
                    bInBufLen,
                    aMac,
                    &bMacLen));

                /* Truncate the MAC generated */
                PH_CHECK_SUCCESS_FCT(wStatus, phalMful_Int_TruncateMac(aMac));
                bMacLen = 8U; /* Truncated length */

                /* Compare the MAC. */
                if(memcmp(&ppResponse[0U][*pRespLen - 8U], aMac, 8U) != 0U)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_AL_MFUL);
                }

                /* Increment the command counter. */
                ((phalMful_Sw_DataParams_t *) pDataParams)->wCmdCtr++;

                /* Update the length to remove the mac from response. */
                *pRespLen -= 8U;
            }
#ifdef NXPBUILD__PHAL_MFUL_SAMAV3_NONX
            else if(PH_GET_COMPID(pDataParams) == PHAL_MFUL_SAMAV3_NONX_ID)
            {
                /* Increment the command counter. */
                ((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->wCmdCtr++;

                /* Perform Offline activation using Ram Key. */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(
                    (((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->pHalSamDataParams),
                    PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
                    (((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->bRamKeyNo),
                    (((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->bRamKeyVer),
                    NULL,
                    0U));

                /* Frame the MAC input buffer.
                 * wCmdCtr || CmdData
                 */
                aInBuff[bInBufLen++] = (uint8_t) (((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->wCmdCtr);
                aInBuff[bInBufLen++] = (uint8_t) (((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->wCmdCtr >> 8U);

                (void) memcpy(&aInBuff[bInBufLen], *ppResponse, (*pRespLen - 8 /* MAC */));
                bInBufLen += (uint8_t) (*pRespLen - 8U /* MAC */);

                /* Load zero IV. */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_LoadInitVector(
                    (((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->pHalSamDataParams),
                    PHHAL_HW_SAMAV3_CMD_SAM_LOAD_IV_MODE_SET_IV,
                    aIV,
                    PH_CRYPTOSYM_AES_BLOCK_SIZE));

                /* Generate Mac. */
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_GenerateMAC(
                    (((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->pHalSamDataParams),
                    PH_EXCHANGE_DEFAULT,
                    PHHAL_HW_SAMAV3_TRUNCATION_MODE_MFP,
                    aInBuff,
                    bInBufLen,
                    &pMac,
                    &wMLen));

                /* Copy the Mac to the parameter. */
                (void) memcpy(aMac, pMac, wMLen);
                bMacLen = (uint8_t) wMLen;

                /* Compare the MAC. */
                if(memcmp(&ppResponse[0U][*pRespLen - 8U], aMac, 8U) != 0U)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_AL_MFUL);
                }

                /* Increment the command counter. */
                ((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->wCmdCtr++;

                /* Update the length to remove the mac from response. */
                *pRespLen -= 8U;
            }
#endif /* NXPBUILD__PHAL_MFUL_SAMAV3_NONX */
            else
            {
                /* Nothing to do. */
            }
        }
    }
    else
    {
        /* Nothing to do. */
    }
#endif /* NXPBUILD__PH_CRYPTOSYM */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_GetPalMifareDataParams(void * pDataParams, void** pPalMifareDataParams)
{
    switch(PH_GET_COMPID(pDataParams))
    {
        case PHAL_MFUL_SW_ID:
            *pPalMifareDataParams = ((phalMful_Sw_DataParams_t *) pDataParams)->pPalMifareDataParams;
            break;

#ifdef NXPBUILD__PHAL_MFUL_SAM_NONX
#ifdef NXPBUILD__PHAL_MFUL_SAMAV2_NONX
        case PHAL_MFUL_SAMAV2_ID:
            *pPalMifareDataParams = ((phalMful_SamAV2_DataParams_t *) pDataParams)->pPalMifareDataParams;
            break;
#endif /* NXPBUILD__PHAL_MFUL_SAMAV2_NONX */

#ifdef NXPBUILD__PHAL_MFUL_SAMAV3_NONX
        case PHAL_MFUL_SAMAV3_NONX_ID:
            *pPalMifareDataParams = ((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->pPalMifareDataParams;
            break;
#endif /* NXPBUILD__PHAL_MFUL_SAMAV3_NONX */
#endif /* NXPBUILD__PHAL_MFUL_SAM_NONX */

#ifdef NXPBUILD__PHAL_MFUL_SAM_X
#ifdef NXPBUILD__PHAL_MFUL_SAMAV2_X
        case PHAL_MFUL_SAMAV2_X_ID:
            *pPalMifareDataParams = ((phalMful_SamAV2_X_DataParams_t *) pDataParams)->pPalMifareDataParams;
            break;
#endif /* NXPBUILD__PHAL_MFUL_SAMAV2_X */

#ifdef NXPBUILD__PHAL_MFUL_SAMAV3_X
        case PHAL_MFUL_SAMAV3_X_ID:
            *pPalMifareDataParams = ((phalMful_SamAV3_X_DataParams_t *) pDataParams)->pPalMifareDataParams;
            break;
#endif /* NXPBUILD__PHAL_MFUL_SAMAV3_X */
#endif /* NXPBUILD__PHAL_MFUL_SAM_X */

        default:
            *pPalMifareDataParams = NULL;
            break;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_GetCMAC(void * pDataParams, uint8_t* pCMACVal)
{
    switch(PH_GET_COMPID(pDataParams))
    {
        case PHAL_MFUL_SW_ID:
            *pCMACVal = ((phalMful_Sw_DataParams_t *) pDataParams)->bCMACReq;
            break;

#ifdef NXPBUILD__PHAL_MFUL_SAM_NONX
#ifdef NXPBUILD__PHAL_MFUL_SAMAV3_NONX
        case PHAL_MFUL_SAMAV3_NONX_ID:
            *pCMACVal = ((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->bCMACReq;
            break;
#endif /* NXPBUILD__PHAL_MFUL_SAMAV3_NONX */
#endif /* NXPBUILD__PHAL_MFUL_SAM_NONX */

        default:
            *pCMACVal = 0;
            break;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_GetAuthMode(void * pDataParams, uint8_t* pAuthMode)
{
    switch(PH_GET_COMPID(pDataParams))
    {
        case PHAL_MFUL_SW_ID:
            *pAuthMode = ((phalMful_Sw_DataParams_t *) pDataParams)->bAuthMode;
            break;

#ifdef NXPBUILD__PHAL_MFUL_SAM_NONX
#ifdef NXPBUILD__PHAL_MFUL_SAMAV3_NONX
        case PHAL_MFUL_SAMAV3_NONX_ID:
            *pAuthMode = ((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->bAuthMode;
            break;
#endif /* NXPBUILD__PHAL_MFUL_SAMAV3_NONX */
#endif /* NXPBUILD__PHAL_MFUL_SAM_NONX */

        default:
            *pAuthMode = 0;
            break;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_GetAdditionalInfo(void * pDataParams, uint8_t* pAddInfo)
{
    switch(PH_GET_COMPID(pDataParams))
    {
        case PHAL_MFUL_SW_ID:
            *pAddInfo = ((phalMful_Sw_DataParams_t *) pDataParams)->bAuthMode;
            break;

#ifdef NXPBUILD__PHAL_MFUL_SAM_NONX
#ifdef NXPBUILD__PHAL_MFUL_SAMAV2_NONX
        case PHAL_MFUL_SAMAV2_ID:
            *pAddInfo = ((phalMful_SamAV2_DataParams_t *) pDataParams)->bAdditionalInfo;
            break;
#endif /* NXPBUILD__PHAL_MFUL_SAMAV2_NONX */

#ifdef NXPBUILD__PHAL_MFUL_SAMAV3_NONX
        case PHAL_MFUL_SAMAV3_NONX_ID:
            *pAddInfo = ((phalMful_SamAV3_NonX_DataParams_t *) pDataParams)->bAdditionalInfo;
            break;
#endif /* NXPBUILD__PHAL_MFUL_SAMAV3_NONX */
#endif /* NXPBUILD__PHAL_MFUL_SAM_NONX */

        default:
            *pAddInfo = 0;
            break;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}

phStatus_t phalMful_Int_TruncateMac(uint8_t * pMac)
{
    uint8_t PH_MEMLOC_REM bIndex;
    uint8_t PH_MEMLOC_REM bIndex2;

    for (bIndex = 1U, bIndex2 = 0U; bIndex < 16U; bIndex += 2U, bIndex2++)
    {
        pMac[bIndex2] = pMac[bIndex];
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFUL);
}
#endif /* NXPBUILD__PHAL_MFUL */
