/*
 * Copyright 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic NATG X DNA Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7462 $
 * $Date: 2025-08-29 14:09:06 +0530 (Fri, 29 Aug 2025) $
 *
 * History:
 *  Rajendran Kumar: Generated 23 Aug 2024
 */

#include <ph_Status.h>

#ifdef NXPBUILD__PHAL_NTAGXDNA

#include <phalNtagXDna.h>
#include "phalNtagXDna_Int.h"
phStatus_t phalNtagXDna_Int_ComputeErrorResponse(void * pDataParams, uint16_t wStatusIn)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    phStatus_t  PH_MEMLOC_REM wStatusTmp = PH_ERR_SUCCESS;

    switch(wStatusIn)
    {
        case PHAL_NTAGXDNA_RESP_OPERATION_OK:
        case PHAL_NTAGXDNA_ISO7816_9000:
            wStatus = PH_ERR_SUCCESS;
            break;

        case PHAL_NTAGXDNA_RESP_ADDITIONAL_FRAME:
            wStatus = PH_ERR_SUCCESS_CHAINING;
            break;

        case PHAL_NTAGXDNA_RESP_LENGTH_ERROR:
            wStatus = PH_ERR_LENGTH_ERROR;
            break;

        case PHAL_NTAGXDNA_RESP_AUTHENTICATION_ERROR:
            wStatus = PH_ERR_AUTH_ERROR;
            break;

        case PHAL_NTAGXDNA_RESP_OUT_OF_EEPROM_ERROR:
            wStatus = PHAL_NTAGXDNA_ERR_OUT_OF_EEPROM;
            break;

        case PHAL_NTAGXDNA_RESP_NO_SUCH_KEY:
            wStatus = PHAL_NTAGXDNA_ERR_NO_SUCH_KEY;
            break;

        case PHAL_NTAGXDNA_RESP_PERMISSION_DENIED:
            wStatus = PHAL_NTAGXDNA_ERR_PERMISSION_DENIED;
            break;

        case PHAL_NTAGXDNA_RESP_BOUNDARY_ERROR:
            wStatus = PHAL_NTAGXDNA_ERR_BOUNDARY_ERROR;
            break;

        case PHAL_NTAGXDNA_RESP_COMMAND_ABORTED:
            wStatus = PHAL_NTAGXDNA_ERR_COMMAND_ABORTED;
            break;

        case PHAL_NTAGXDNA_RESP_DUPLICATE:
            wStatus = PHAL_NTAGXDNA_ERR_DUPLICATE;
            break;

        case PHAL_NTAGXDNA_RESP_FILE_NOT_FOUND:
            wStatus = PHAL_NTAGXDNA_ERR_FILE_NOT_FOUND;
            break;

        case PHAL_NTAGXDNA_RESP_INTEGRITY_ERROR:
            wStatus = PHAL_NTAGXDNA_ERR_PICC_CRYPTO;
            break;

        case PHAL_NTAGXDNA_RESP_PARAMETER_ERROR:
            wStatus = PHAL_NTAGXDNA_ERR_PARAMETER_ERROR;
            break;

        case PHAL_NTAGXDNA_RESP_MEMORY_ERROR:
        case PHAL_NTAGXDNA_RESP_ILLEGAL_COMMAND_CODE:
            wStatus = PHAL_NTAGXDNA_ERR_DF_GEN_ERROR;
            PH_CHECK_SUCCESS_FCT(wStatusTmp, phalNtagXDna_SetConfig(pDataParams, PHAL_NTAGXDNA_ADDITIONAL_INFO, wStatusIn));
            break;

        case PHAL_NTAGXDNA_ISO7816_ERR_6300:
        case PHAL_NTAGXDNA_ISO7816_ERR_6700:
        case PHAL_NTAGXDNA_ISO7816_ERR_6982:
        case PHAL_NTAGXDNA_ISO7816_ERR_6984:
        case PHAL_NTAGXDNA_ISO7816_ERR_6985:
        case PHAL_NTAGXDNA_ISO7816_ERR_6987:
        case PHAL_NTAGXDNA_ISO7816_ERR_6988:
        case PHAL_NTAGXDNA_ISO7816_ERR_6A80:
        case PHAL_NTAGXDNA_ISO7816_ERR_6A82:
        case PHAL_NTAGXDNA_ISO7816_ERR_6A86:
        case PHAL_NTAGXDNA_ISO7816_ERR_6A87:
        case PHAL_NTAGXDNA_ISO7816_ERR_6A88:
        case PHAL_NTAGXDNA_ISO7816_ERR_6C00:
        case PHAL_NTAGXDNA_ISO7816_ERR_6E00:
            wStatus = PHAL_NTAGXDNA_ERR_DF_7816_GEN_ERROR;
            PH_CHECK_SUCCESS_FCT(wStatusTmp, phalNtagXDna_SetConfig(pDataParams, PHAL_NTAGXDNA_ADDITIONAL_INFO, wStatusIn));
            break;

        case PHAL_NTAGXDNA_RESP_CERT_ERROR:
            wStatus = PHAL_NTAGXDNA_ERR_CERTIFICATE_ERROR;
            break;

        case PHAL_NTAGXDNA_RESP_WEAK_FIELD:
            wStatus = PHAL_NTAGXDNA_ERR_WEAK_FIELD_ERROR;
            break;

        case PHAL_NTAGXDNA_RESP_PAD_VOLTAGE_UNRELIABLE:
            wStatus = PHAL_NTAGXDNA_ERR_PAD_VOLTAGE_UNRELIABLE_ERROR;
            break;

        default:
            wStatus = PH_ERR_PROTOCOL_ERROR;
    }
    return PH_ADD_COMPCODE(wStatus, PH_COMP_AL_NTAGXDNA);
}

void phalNtagXDna_Int_RotateLeft(uint8_t * pData, uint8_t bDataLen, uint8_t bTimes)
{
    uint8_t bIndex_Times = 0, bIndex_Len = 0;
    uint8_t bTmp = 0;

    for(bIndex_Times = 0; bIndex_Times < bTimes; bIndex_Times++)
    {
        bTmp = pData[0];
        for(bIndex_Len = 0; bIndex_Len < bDataLen - 1U; bIndex_Len++)
            pData[bIndex_Len] = pData[bIndex_Len + 1U];

        pData[bDataLen - 1U] = bTmp;
    }
}

void phalNtagXDna_Int_RotateRight(uint8_t * pData, uint8_t bDataLen, uint8_t bTimes)
{
    uint8_t bIndex_Times = 0, bIndex_Len = 0;
    uint8_t bTmp = 0;

    for(bIndex_Times = 0; bIndex_Times < bTimes; bIndex_Times++)
    {
        bTmp = pData[bDataLen - 1U];
        for(bIndex_Len = (bDataLen - 1U); bIndex_Len > 0; bIndex_Len--)
            pData[bIndex_Len] = pData[bIndex_Len - 1U];

        pData[0] = bTmp;
    }
}

void phalNtagXDna_Int_GetCommMode(uint8_t bAuthState, uint8_t bOption, uint8_t * pCommMode)
{
    if(bOption != PHAL_NTAGXDNA_COMMUNICATION_INVALID)
    {
        switch(bOption)
        {
#ifdef NXPBUILD__PH_CRYPTOSYM
            case PHAL_NTAGXDNA_COMMUNICATION_FULL:
            case PHAL_NTAGXDNA_COMMUNICATION_MAC:
                *pCommMode = PHAL_NTAGXDNA_COMMUNICATION_MAC;
                break;
#endif /* NXPBUILD__PH_CRYPTOSYM */

            default:
                *pCommMode = PHAL_NTAGXDNA_COMMUNICATION_PLAIN;
                break;
        }
    }
    else
    {
        switch(bAuthState)
        {
#ifdef NXPBUILD__PH_CRYPTOSYM
            case PHAL_NTAGXDNA_EV2_AUTHENTICATED:
            case PHAL_NTAGXDNA_ECC_AUTHENTICATED:
                *pCommMode = PHAL_NTAGXDNA_COMMUNICATION_MAC;
                break;
#endif /* NXPBUILD__PH_CRYPTOSYM */

            default:
                *pCommMode = PHAL_NTAGXDNA_COMMUNICATION_PLAIN;
                break;
        }
    }
}

phStatus_t phalNtagXDna_Int_Validate_ComOption(uint8_t bComOption)
{
    switch(bComOption)
    {
        case PHAL_NTAGXDNA_COMMUNICATION_PLAIN:
        case PHAL_NTAGXDNA_COMMUNICATION_PLAIN_20:
#ifdef NXPBUILD__PH_CRYPTOSYM
        case PHAL_NTAGXDNA_COMMUNICATION_MAC:
        case PHAL_NTAGXDNA_COMMUNICATION_FULL:
#endif /* NXPBUILD__PH_CRYPTOSYM */
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

void phalNtagXDna_Int_EncodeBER_TLV_Len(uint8_t * pBuffer, uint16_t * pBuffLen, uint16_t wAddLen)
{
    if(wAddLen > 255U)
    {
        pBuffer[(*pBuffLen)++] = PHAL_NTAGXDNA_ISO7816_BER_TLV_C_82;
        pBuffer[(*pBuffLen)++] = (uint8_t) (wAddLen >> 8U);
    }
    else
    {
        pBuffer[(*pBuffLen)++] = PHAL_NTAGXDNA_ISO7816_BER_TLV_C_81;
    }

    pBuffer[(*pBuffLen)++] = (uint8_t) wAddLen;
}

phStatus_t phalNtagXDna_Int_DecodeBER_TLV_Len(uint8_t ** ppBuffer, uint16_t * pBER_TLV_Len, uint16_t * pRspLen)
{
        switch((*ppBuffer)[0])
        {
            case PHAL_NTAGXDNA_ISO7816_BER_TLV_C_81:
                *pBER_TLV_Len = (uint16_t) (*ppBuffer)[1U];

                (*ppBuffer) += 2U;
                (*pRspLen) -= 2U;
                break;

            case PHAL_NTAGXDNA_ISO7816_BER_TLV_C_82:
                *pBER_TLV_Len = (uint16_t) (((*ppBuffer)[1U] << 8U) | (*ppBuffer)[2U]);

                (*ppBuffer) += 3U;
                (*pRspLen) -= 3U;
                break;

            default:
                *pBER_TLV_Len = (uint16_t) (*ppBuffer)[0];

                (*ppBuffer) += 1U;
                (*pRspLen) -= 1U;
                break;
        }

    return PH_ERR_SUCCESS;
}

void phalNtagXDna_Int_UpdateLC(uint8_t * pData, uint16_t wDataLen, uint8_t bLE_Len)
{
    uint16_t    PH_MEMLOC_REM wISO7816_Hdr_Len = 0;
    uint16_t    PH_MEMLOC_REM wLC = 0;
    uint8_t     PH_MEMLOC_REM bLC_Len = 0;

    /* Compute ISO7816 Header Length. */
    wISO7816_Hdr_Len = (uint16_t) ((bLE_Len == 1) ? PHAL_NTAGXDNA_WRAPPED_HDR_LEN_NORMAL : PHAL_NTAGXDNA_WRAPPED_HDR_LEN_EXTENDED);

    /* Compute Actual LC. */
    wLC = (uint16_t) (wDataLen - (wISO7816_Hdr_Len + bLE_Len));

    /* Update LC. */
    if(bLE_Len > 1U)
    {
        pData[PHAL_NTAGXDNA_LC_POS + bLC_Len++] = 0x00;
        pData[PHAL_NTAGXDNA_LC_POS + bLC_Len++] = (uint8_t) ((wLC & 0xFF00U) >> 8U);
    }
    pData[PHAL_NTAGXDNA_LC_POS + bLC_Len++] = (uint8_t) (wLC & 0x00FFU);
}

#endif /* NXPBUILD__PHAL_NTAGXDNA */
