/*
 * Copyright 2013 - 2014, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * PipeLinux specific BAL-Component of Reader Library Framework.
 * $RCSfile $
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHBAL_REG_PIPELIN

#include "phbalReg_PipeLin.h"
#include "phbalReg_PipeLin_Int.h"

phStatus_t phbalReg_PipeLin_Init(
                                 phbalReg_PipeLin_DataParams_t * pDataParams,
                                 uint16_t wSizeOfDataParams,
                                 uint8_t bMode,
                                 uint16_t wTxBufSize,
                                 uint8_t * pTxBuffer
                                 )
{
    if (sizeof(phbalReg_PipeLin_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_BAL);
    }
    /* Verify exchange buffer */
    if (wTxBufSize == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pTxBuffer);

    /* Check operating mode */
    switch (bMode)
    {
    case PHBAL_REG_PIPELIN_MODE_RAW:
    case PHBAL_REG_PIPELIN_MODE_RS232:
    case PHBAL_REG_PIPELIN_MODE_I2C:
    case PHBAL_REG_PIPELIN_MODE_SPI:
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* initialize the data parameters */
    pDataParams->wId            = PH_COMP_BAL | PHBAL_REG_PIPELIN_ID;
    pDataParams->bMode          = bMode;
    pDataParams->pPipeNameRx    = (uint8_t*)PHBAL_REG_PIPELIN_RX_PIPENAME_DEFAULT;
    pDataParams->pPipeNameTx    = (uint8_t*)PHBAL_REG_PIPELIN_TX_PIPENAME_DEFAULT;
    pDataParams->pPipeRx        = NULL;
    pDataParams->pPipeTx        = NULL;
    pDataParams->pTxBuffer      = pTxBuffer;
    pDataParams->wTxBufSize     = wTxBufSize;
    pDataParams->bI2cSlaveAddr  = 0x00;
    pDataParams->bConfigValue   = PHBAL_REG_PIPELIN_VALUE_BITRATE_RS232_115200;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_PipeLin_GetPortList(
                                        phbalReg_PipeLin_DataParams_t * pDataParams,
                                        uint16_t wPortBufSize,
                                        uint8_t * pPortNames,
                                        uint16_t * pNumOfPorts
                                        )
{
#ifdef LINUX
    uint16_t wLen;

    if (wPortBufSize < (uint16_t)strlen(PHBAL_REG_PIPELIN_RX_PIPENAME_DEFAULT) + strlen(PHBAL_REG_PIPELIN_TX_PIPENAME_DEFAULT) + 2)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }
    wLen = (uint16_t)strlen(PHBAL_REG_PIPELIN_RX_PIPENAME_DEFAULT) + 1;
    memcpy(pPortNames, PHBAL_REG_PIPELIN_RX_PIPENAME_DEFAULT, wLen); /* PRQA S 3200 */
    pPortNames += wLen;

    wLen = (uint16_t)strlen(PHBAL_REG_PIPELIN_TX_PIPENAME_DEFAULT) + 1;
    memcpy(pPortNames, PHBAL_REG_PIPELIN_TX_PIPENAME_DEFAULT, wLen); /* PRQA S 3200 */

    *pNumOfPorts = 2;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    pDataParams = NULL;
    wPortBufSize = 0;
    pPortNames = NULL;
    *pNumOfPorts = 0;
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_PipeLin_SetPort(
                                    phbalReg_PipeLin_DataParams_t * pDataParams,
                                    uint8_t * pPortName
                                    )
{
#ifdef LINUX
    if (strlen((char*)pPortName) > 0)
    {
        pDataParams->pPipeNameRx = pPortName;
    }
    else
    {
        pDataParams->pPipeNameRx = (uint8_t*)PHBAL_REG_PIPELIN_RX_PIPENAME_DEFAULT;
    }
    pPortName += strlen((char*)pPortName) + 1;
    if (strlen((char*)pPortName) > 0)
    {
        pDataParams->pPipeNameTx = pPortName;
    }
    else
    {
        pDataParams->pPipeNameTx = (uint8_t*)PHBAL_REG_PIPELIN_TX_PIPENAME_DEFAULT;
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    pDataParams = NULL;
    pPortName = NULL;
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_PipeLin_OpenPort(
                                     phbalReg_PipeLin_DataParams_t * pDataParams
                                     )
{
#ifdef LINUX

    phStatus_t  PH_MEMLOC_REM status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
    int32_t     PH_MEMLOC_REM dFlags;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName,      "phbalReg_PipeLin_OpenPipe");
    PH_LOG_HELPER_ALLOCATE_TEXT(bInfoRxPipeExists,  "NOTE - RX PIPE already exists.");
    PH_LOG_HELPER_ALLOCATE_TEXT(bErrorRxPipeCreate, "ERROR - Cannot create RX PIPE.");
    PH_LOG_HELPER_ALLOCATE_TEXT(bErrorRxPipeOpen,   "ERROR - Cannot open RX PIPE.");
    PH_LOG_HELPER_ALLOCATE_TEXT(bInfoTxPipeExists,  "NOTE - TX PIPE already exists.");
    PH_LOG_HELPER_ALLOCATE_TEXT(bErrorTxPipeCreate, "ERROR - Cannot create TX PIPE.");
    PH_LOG_HELPER_ALLOCATE_TEXT(bErrorTxPipeOpen,   "ERROR - Cannot open TX PIPE.");
    PH_LOG_HELPER_ALLOCATE_TEXT(bErrorFcntl,        "ERROR - fcntl returned -1.");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

    /* Entry Debug Message */
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* interface is already open */
    if (pDataParams->pPipeTx != NULL || pDataParams->pPipeRx != NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    /* Create Rx-Pipe */
    if (mkfifo((char*)pDataParams->pPipeNameRx, 0666) != 0)
    {
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);

        if (errno == EEXIST)  /* PRQA S 5119 */
        {
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bInfoRxPipeExists);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
        }
        else
        {
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorRxPipeCreate);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
    }

    /* Create Tx-Pipe */
    if (mkfifo((char*)pDataParams->pPipeNameTx, 0666) != 0)
    {
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);

        if (errno == EEXIST)  /* PRQA S 5119 */
        {
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bInfoTxPipeExists);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
        }
        else
        {
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorTxPipeCreate);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
    }

    /* Open Rx-Pipe for reading */
    pDataParams->pPipeRx = (void*)fopen((char*)pDataParams->pPipeNameRx, "r");
    if (pDataParams->pPipeRx == FNULL)
    {
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorRxPipeOpen);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* Disable Rx buffering */
    setbuf((FILE*)pDataParams->pPipeRx, NULL);

    /*  get current flag settings of file */
    dFlags = fcntl(fileno((FILE*)pDataParams->pPipeRx), F_GETFL);
    if (dFlags == -1)
    {
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorFcntl);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* clear O_NONBLOCK and reset file flags  */
    dFlags = fcntl(fileno((FILE*)pDataParams->pPipeRx), F_SETFL, dFlags & O_NONBLOCK);
    if (dFlags == -1)
    {
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorFcntl);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* Open Tx-Pipe for writing */
    pDataParams->pPipeTx = (void*)fopen((char*)pDataParams->pPipeNameTx, "w");
    if (pDataParams->pPipeTx == FNULL)
    {
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorTxPipeOpen);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* Disable Tx buffering */
    setbuf((FILE*)pDataParams->pPipeTx, NULL);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
    return status;
#else
    /* satisfy compiler */
    pDataParams = NULL;
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_PipeLin_ClosePort(
                                      phbalReg_PipeLin_DataParams_t * pDataParams
                                      )
{
#ifdef LINUX

    /* interface is already closed */
    if (pDataParams->pPipeRx == NULL || pDataParams->pPipeTx == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    fclose(pDataParams->pPipeTx);  /* PRQA S 3200 */
    fclose(pDataParams->pPipeRx);  /* PRQA S 3200 */
    pDataParams->pPipeRx = NULL;
    pDataParams->pPipeTx = NULL;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    pDataParams = NULL;
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_PipeLin_Exchange(
                                     phbalReg_PipeLin_DataParams_t * pDataParams,
                                     uint16_t wOption,
                                     uint8_t * pTxBuffer,
                                     uint16_t wTxLength,
                                     uint16_t wRxBufSize,
                                     uint8_t * pRxBuffer,
                                     uint16_t * pRxLength
                                     )
{
    size_t      PH_MEMLOC_REM iResult;
    uint8_t *   PH_MEMLOC_REM pReadResult;
	phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bProtBuffer[5];
    uint8_t *   PH_MEMLOC_REM pTmpBuffer;
    uint16_t    PH_MEMLOC_REM wTmpBufferLen;
    uint16_t    PH_MEMLOC_REM wRxLengthTmp;

    /* Check options */
    if (wOption != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* Parameter check */
    if (wTxLength == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }
    /* interface should have been opened before */
    if (pDataParams->pPipeTx == NULL || pDataParams->pPipeRx == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    /* Use dummy storage for RxLength if not given */
    if (pRxLength == NULL)
    {
        pRxLength = &wRxLengthTmp;
    }

    /* Reset received length */
    *pRxLength = 0;

    /* Set TxBuffer */
    pTmpBuffer = pDataParams->pTxBuffer;
    wTmpBufferLen = pDataParams->wTxBufSize;

    /* Tx-Coding for different protocols */
    switch (pDataParams->bMode)
    {
        /* I2C mode */
    case PHBAL_REG_SOCKETWIN_MODE_I2C:
        /* Read operation */
        if (wTxLength == 1)
        {
            /* Perform write operation with zero data upfront */
            bProtBuffer[0] = 0x42;
            /* shift address and clear RD/NWR bit to indicate write operation */
            bProtBuffer[1] = (uint8_t)(pDataParams->bI2cSlaveAddr << 1);
            /* append register address */
            bProtBuffer[2] = pTxBuffer[0];

            /* Convert to string */
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_PipeLin_HexToString(
                bProtBuffer,
                3,
                wTmpBufferLen,
                pTmpBuffer));

            /* Send Write */
            wTxLength = (uint16_t)strlen((char*)pTmpBuffer);
            iResult = fwrite((char*)pTmpBuffer, 1, wTxLength, (FILE*)pDataParams->pPipeTx);

            /* Flush the pipe */
            fflush(pDataParams->pPipeTx); /* PRQA S 3200 */

            if ((uint16_t)iResult != wTxLength)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* Receive response */
            pReadResult = (uint8_t*)fgets((char*)pTmpBuffer, (int)wTmpBufferLen, (FILE*)pDataParams->pPipeRx);
            if (pReadResult == NULL)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* Retrieve length */
            *pRxLength = (uint16_t)strlen((char*)pTmpBuffer);

            /* Length check */
            if (*pRxLength < 2)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* remove end of line symbols at the end of the recv string and terminate the string with 0 */
            pTmpBuffer[*pRxLength - 2] = 0;

            /* Convert to HEX format */
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_PipeLin_StringToHex(pTmpBuffer, pRxLength));

            /* Length check */
            if (*pRxLength != 2)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* ACK check */
            if (pTmpBuffer[1] != 0x02)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* reset TxLength */
            wTxLength = 0;

            bProtBuffer[0] = (uint8_t)(0x80 | (uint8_t)(wRxBufSize & 0x7F));
            /* shift address and set RD/NWR bit to indicate read operation */
            bProtBuffer[1] = (uint8_t)(pDataParams->bI2cSlaveAddr << 1) | 0x01;
        }
        /* Write operation */
        else
        {
            bProtBuffer[0] = (uint8_t)(0x40 | (uint8_t)((wTxLength + 1) & 0x3F));
            /* shift address and clear RD/NWR bit to indicate write operation */
            bProtBuffer[1] = (uint8_t)(pDataParams->bI2cSlaveAddr << 1);
        }

        /* Convert to string */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_PipeLin_HexToString(
            bProtBuffer,
            2,
            wTmpBufferLen,
            pTmpBuffer));

        /* Adjust TxBuffer pointer */
        pTmpBuffer[4] = '\0';
        wTmpBufferLen = wTmpBufferLen - (uint16_t)strlen((char*)pTmpBuffer);
        pTmpBuffer += strlen((char*)pTmpBuffer);
        break;

        /* SPI mode */
    case PHBAL_REG_SOCKETWIN_MODE_SPI:
        bProtBuffer[0] = (uint8_t)(wTxLength << 3 >> 8);
        bProtBuffer[1] = (uint8_t)(wTxLength << 3);

        /* Convert to string */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_PipeLin_HexToString(
            bProtBuffer,
            2,
            wTmpBufferLen,
            pTmpBuffer));

        /* Adjust TxBuffer pointer */
        pTmpBuffer[4] = '\0';
        wTmpBufferLen = wTmpBufferLen - (uint16_t)strlen((char*)pTmpBuffer);
        pTmpBuffer += strlen((char*)pTmpBuffer);
        break;

        /* nothing to do for other protocols */
    default:
        break;
    }

	/* Convert from HEX format if neccessary */
    if (pDataParams->bMode != PHBAL_REG_SOCKETWIN_MODE_RAW)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_PipeLin_HexToString(
            pTxBuffer,
            wTxLength,
            wTmpBufferLen,
            pTmpBuffer));

        pTxBuffer = pDataParams->pTxBuffer;
        wTxLength = (uint16_t)strlen((char*)pDataParams->pTxBuffer);
    }
    else
    {
        /* String check */
        if (wTxLength != (uint16_t)strlen((char*)pTxBuffer))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
        }
    }

    /* Send TxBuffer */
    iResult = fwrite((char*)pTxBuffer, 1, wTxLength, (FILE*)pDataParams->pPipeTx);

    /* Flush the pipe */
    fflush(pDataParams->pPipeTx); /* PRQA S 3200 */

    /* Check result */
    if ((uint16_t)iResult != wTxLength)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

	/* Read data from socket */
    if (pDataParams->bMode == PHBAL_REG_SOCKETWIN_MODE_RAW)
    {
        /* Receive response */
        pReadResult = (uint8_t*)fgets((char*)pRxBuffer, (int)wRxBufSize, (FILE*)pDataParams->pPipeRx);
        if (pReadResult == NULL)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }

        /* Retrieve length */
        *pRxLength = (uint16_t)strlen((char*)pRxBuffer);

        /* Length check */
        if (*pRxLength == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }

        /* remove end of line symbols at the end of the recv string and terminate the string with 0 */
        pRxBuffer[*pRxLength - 1] = 0;
    }
    else
    {
        /* Reception loop */
        wTmpBufferLen = 0;
		do
		{
            /* Receive response */
            pReadResult = (uint8_t*)fgets(
                (char*)&pDataParams->pTxBuffer[wTmpBufferLen],
                (int)(pDataParams->wTxBufSize - wTmpBufferLen),
                (FILE*)pDataParams->pPipeRx);

            if (pReadResult == NULL)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* Retrieve length */
            wTmpBufferLen = wTmpBufferLen + (uint16_t)strlen((char*)pReadResult);
		}
        while ((strlen((char*)pReadResult) >= 2) && (pDataParams->pTxBuffer[wTmpBufferLen - 2] != 0x0D) && (pDataParams->pTxBuffer[wTmpBufferLen - 1] != 0x0A));

        /* remove end of line symbols at the end of the recv string and terminate the string with 0 */
        pDataParams->pTxBuffer[wTmpBufferLen - 2] = 0;
        --wTmpBufferLen;

        /* Convert to HEX format if neccessary */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_PipeLin_StringToHex(pDataParams->pTxBuffer, &wTmpBufferLen));

        /* Set initial RxBuffer pointer */
        pTmpBuffer = pDataParams->pTxBuffer;

        /* Rx-Coding for different protocols */
        switch (pDataParams->bMode)
        {
            /* I2C mode */
        case PHBAL_REG_SOCKETWIN_MODE_I2C:
            /* Read operation */
            if (bProtBuffer[0] & 0x80) /* PRQA S 3353 */
            {
                /* ACK check */
                if (pTmpBuffer[0] != 0x01)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
                }

                /* Advance to data */
                ++pTmpBuffer;
                --wTmpBufferLen;
            }
            /* Write operation */
            else
            {
                /* Length check */
                if (wTmpBufferLen != 2)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
                }

                /* Ignore length byte */
                ++pTmpBuffer;
                --wTmpBufferLen;

                /* ACK check */
                if (pTmpBuffer[0] != (bProtBuffer[0] & 0x3F))
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
                }

                /* Advance to data */
                ++pTmpBuffer;
                --wTmpBufferLen;
            }
            break;

            /* SPI mode */
        case PHBAL_REG_SOCKETWIN_MODE_SPI:

            /* Length check */
            if (wTmpBufferLen < 2)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* Length word check */
            if ((pTmpBuffer[0] != bProtBuffer[0]) ||
                (pTmpBuffer[1] != bProtBuffer[1]))
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }

            /* Advance to data */
            pTmpBuffer += 2;
            wTmpBufferLen = wTmpBufferLen - 2;
            break;

            /* nothing to do for other protocols */
        default:
            break;
        }

		/* Copy resulting HEX data to receive buffer */
        if (wTmpBufferLen > 0)
        {
            if (wTmpBufferLen <= wRxBufSize)
            {
                memcpy(pRxBuffer, pTmpBuffer, wTmpBufferLen);  /* PRQA S 3200 */
                *pRxLength = wTmpBufferLen;
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
            }
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_PipeLin_SetConfig(
                                      phbalReg_PipeLin_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint16_t wValue
                                      )
{
    phStatus_t  PH_MEMLOC_REM status;
    uint8_t     PH_MEMLOC_REM bMode;
    uint8_t     PH_MEMLOC_REM bCommand[13];
    uint16_t    PH_MEMLOC_REM wRespLen;

    switch (wConfig)
    {
    case PHBAL_REG_PIPELIN_CONFIG_MODE:

        /* Check operating mode */
        switch (wValue)
        {
        case PHBAL_REG_PIPELIN_MODE_RAW:
        case PHBAL_REG_PIPELIN_MODE_RS232:
        case PHBAL_REG_PIPELIN_MODE_I2C:
        case PHBAL_REG_PIPELIN_MODE_SPI:
            pDataParams->bMode = (uint8_t)wValue;
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
        }
        break;

    case PHBAL_REG_PIPELIN_CONFIG_BITRATE:

        /* This is not valid for RAW mode */
        if (pDataParams->bMode == PHBAL_REG_PIPELIN_MODE_RAW)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
        }
        /* Check if the value is valid for RS232 mode */
        else if (pDataParams->bMode == PHBAL_REG_PIPELIN_MODE_RS232)
        {
            if (wValue > PHBAL_REG_PIPELIN_VALUE_BITRATE_RS232_1228800)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
            }
        }
        /* Check if the value is valid for I2C mode */
        else if (pDataParams->bMode == PHBAL_REG_PIPELIN_MODE_I2C)
        {
            if ((wValue < PHBAL_REG_PIPELIN_VALUE_BITRATE_I2C_10K) ||
                (wValue > PHBAL_REG_PIPELIN_VALUE_BITRATE_I2C_400K) &&
                (
                (wValue < PHBAL_REG_PIPELIN_VALUE_BITRATE_I2CL_1M) ||
                (wValue > PHBAL_REG_PIPELIN_VALUE_BITRATE_I2CL_5M)
                ))
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
            }
        }
        /* Check if the value is valid for SPI mode */
        else if (pDataParams->bMode == PHBAL_REG_PIPELIN_MODE_SPI)
        {
            if ((wValue < PHBAL_REG_PIPELIN_VALUE_BITRATE_SPI_10K) ||
                (wValue > PHBAL_REG_PIPELIN_VALUE_BITRATE_SPI_8M))
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
            }
        }
        /* This should never happen */
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
        }

        /* save configuration */
        pDataParams->bConfigValue = (uint8_t)wValue;

        /* backup current mode */
        bMode = pDataParams->bMode;

        /* set mode to RAW */
        pDataParams->bMode = PHBAL_REG_PIPELIN_MODE_RAW;

        /* build config command */
        sprintf((char*)bCommand, "config%02X\n", pDataParams->bConfigValue);  /* PRQA S 3200 */

        /* send config command */
        status = phbalReg_PipeLin_Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            bCommand,
            (uint16_t)strlen((char*)bCommand),
            sizeof(bCommand),
            bCommand,
            &wRespLen);

        /* restore previous mode */
        pDataParams->bMode = bMode;

        /* Check success */
        PH_CHECK_SUCCESS(status);
        break;

    case PHBAL_REG_PIPELIN_CONFIG_I2C_SLAVE_ADDR:

        pDataParams->bI2cSlaveAddr = (uint8_t)wValue;
        break;

    case PHBAL_REG_PIPELIN_CONFIG_SETCARD:

        /* backup current mode */
        bMode = pDataParams->bMode;

        /* set mode to RAW */
        pDataParams->bMode = PHBAL_REG_PIPELIN_MODE_RAW;

        /* build setcard command */
        sprintf((char*)bCommand, "setcard%02X%02X\n", (uint8_t)(wValue >> 8), (uint8_t)(wValue));  /* PRQA S 3200 */

        /* send config command */
        status = phbalReg_PipeLin_Exchange(
            pDataParams,
            PH_EXCHANGE_DEFAULT,
            bCommand,
            (uint16_t)strlen((char*)bCommand),
            sizeof(bCommand),
            bCommand,
            &wRespLen);

        /* restore previous mode */
        pDataParams->bMode = bMode;

        /* Check success */
        PH_CHECK_SUCCESS(status);
        break;

    case PHBAL_REG_PIPELIN_CONFIG_RESET:

        if (wValue)
        {
            /* backup current mode */
            bMode = pDataParams->bMode;

            /* set mode to RAW */
            pDataParams->bMode = PHBAL_REG_PIPELIN_MODE_RAW;

            /* build reset command */
            sprintf((char*)bCommand, "reset\n");  /* PRQA S 3200 */

            /* send reset command */
            status = phbalReg_PipeLin_Exchange(
                pDataParams,
                PH_EXCHANGE_DEFAULT,
                bCommand,
                (uint16_t)strlen((char*)bCommand),
                sizeof(bCommand),
                bCommand,
                &wRespLen);

            /* Check success */
            PH_CHECK_SUCCESS(status);

            if (memcmp(bCommand, "ACK", 3))
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }
        }
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_PipeLin_GetConfig(
                                      phbalReg_PipeLin_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint16_t * pValue
                                      )
{
    switch (wConfig)
    {
    case PHBAL_REG_PIPELIN_CONFIG_MODE:

        *pValue = (uint16_t)pDataParams->bMode;
        break;

    case PHBAL_REG_PIPELIN_CONFIG_BITRATE:

        *pValue = (uint16_t)pDataParams->bConfigValue;
        break;

    case PHBAL_REG_PIPELIN_CONFIG_I2C_SLAVE_ADDR:

        *pValue = (uint16_t)pDataParams->bI2cSlaveAddr;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

#endif /* NXPBUILD__PHBAL_REG_PIPELIN */
