/*
 * Copyright 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal FPGA BOX BAL Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHBAL_REG_SERIALWIN

#include "phbalReg_SerialWin_Int.h"

phStatus_t phbalReg_SerialWin_Int_SetMode(
                                          void * phPort,
                                          uint32_t dwBitRate,
                                          uint8_t bStopBits,
                                          uint8_t bParity,
                                          uint8_t bDtr,
                                          uint8_t bRts
                                          )
{
#ifdef _WIN32
    DCB PH_MEMLOC_BUF dcb;

    /* Prepare DCB structure */
    SecureZeroMemory(&dcb, sizeof(DCB));  /* PRQA S 3200 */
    GetCommState( phPort, &dcb );

    dcb.DCBlength = sizeof(DCB);
    dcb.BaudRate = dwBitRate;
    dcb.fBinary = TRUE;
    dcb.fParity = FALSE;
    dcb.fOutxCtsFlow = FALSE;
    dcb.fOutxDsrFlow = FALSE;
    dcb.fDsrSensitivity = FALSE;
    dcb.fTXContinueOnXoff = TRUE;
    dcb.fOutX = FALSE;
    dcb.fInX = FALSE;
    dcb.fErrorChar = FALSE;
    dcb.fNull = FALSE;
    dcb.fAbortOnError = FALSE;
    dcb.XonLim = 0;
    dcb.XoffLim = 0;
    dcb.ByteSize = 8;

    switch ( bDtr )
    {
        case PHBAL_REG_SERIALWIN_VALUE_DTR_HIGH:
            dcb.fDtrControl = DTR_CONTROL_ENABLE;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_DTR_LOW:
            dcb.fDtrControl = DTR_CONTROL_DISABLE;
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    switch ( bRts )
    {
        case PHBAL_REG_SERIALWIN_VALUE_RTS_HIGH:
            dcb.fRtsControl = RTS_CONTROL_ENABLE;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_RTS_LOW:
            dcb.fRtsControl = RTS_CONTROL_DISABLE;
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    switch( dwBitRate )
    {
        case PHBAL_REG_SERIALWIN_VALUE_BITRATE_9600:
            dcb.BaudRate = CBR_9600;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_BITRATE_19200:
            dcb.BaudRate = CBR_19200;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_BITRATE_38400:
            dcb.BaudRate = CBR_38400;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_BITRATE_57600:
            dcb.BaudRate = CBR_57600;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_BITRATE_115200:
            dcb.BaudRate = CBR_115200;
            break;
        default:
            dcb.BaudRate = dwBitRate;
            break;
    }

    switch( bParity )
    {
        case PHBAL_REG_SERIALWIN_VALUE_PARITY_NONE:
            dcb.Parity = NOPARITY;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_PARITY_ODD:
            dcb.Parity = ODDPARITY;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_PARITY_EVEN:
            dcb.Parity = EVENPARITY;
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    switch( bStopBits )
    {
        case PHBAL_REG_SERIALWIN_VALUE_STOPBITS_ONE:
            dcb.StopBits = ONESTOPBIT;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_STOPBITS_ONE5:
            dcb.StopBits = ONE5STOPBITS;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_STOPBITS_TWO:
            dcb.StopBits = TWOSTOPBITS;
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* Apply the settings */
    if (SetCommState(phPort, &dcb) == FALSE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* Set DTR, some boards needs a DTR = 1 level */
    if (EscapeCommFunction(phPort, SETDTR) == FALSE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (phPort || dwBitRate);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SerialWin_Int_SetBitRate(
                                             void * phPort,
                                             uint32_t dwBitRate
                                             )
{
#ifdef _WIN32
    DCB PH_MEMLOC_BUF dcb;

    /* Prepare CDB structure */
    SecureZeroMemory( &dcb, sizeof(DCB) );  /* PRQA S 3200 */
    GetCommState( phPort, &dcb );

    switch( dwBitRate )
    {
        case PHBAL_REG_SERIALWIN_VALUE_BITRATE_9600:
            dcb.BaudRate = CBR_9600;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_BITRATE_19200:
            dcb.BaudRate = CBR_19200;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_BITRATE_38400:
            dcb.BaudRate = CBR_38400;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_BITRATE_57600:
            dcb.BaudRate = CBR_57600;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_BITRATE_115200:
            dcb.BaudRate = CBR_115200;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_BITRATE_230400:
            dcb.BaudRate = 230400;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_BITRATE_460800:
            dcb.BaudRate = 460800;
            break;
        default:
            dcb.BaudRate = dwBitRate;
            break;
    }

    /* Apply the settings */
    if (SetCommState(phPort, &dcb) == FALSE)
    {
        DWORD err = GetLastError();
        if ( err != ERROR_INVALID_HANDLE )
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
    }

     return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);

#else
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SerialWin_Int_SetTimeout(
                                             phbalReg_SerialWin_DataParams_t * pDataParams
                                             )
{
#ifdef _WIN32
    COMMTIMEOUTS PH_MEMLOC_BUF cto;
    uint32_t     PH_MEMLOC_REM dwAddTimeoutMs = 0;
    uint32_t     PH_MEMLOC_REM dwBaudRate = 0;
    uint16_t     PH_MEMLOC_REM wStopBits = 0;
    float32_t    PH_MEMLOC_REM StopBits = 0;
    uint16_t     PH_MEMLOC_REM wParity = 0;
    phStatus_t   PH_MEMLOC_REM statusTmp;

    if(pDataParams->wAdditionalRxDelayBytes != 0)
    {
        switch(pDataParams->dwBitRate)
        {
            case PHBAL_REG_SERIALWIN_VALUE_BITRATE_9600:
                dwBaudRate = CBR_9600;
                break;
            case PHBAL_REG_SERIALWIN_VALUE_BITRATE_19200:
                dwBaudRate = CBR_19200;
                break;
            case PHBAL_REG_SERIALWIN_VALUE_BITRATE_38400:
                dwBaudRate = CBR_38400;
                break;
            case PHBAL_REG_SERIALWIN_VALUE_BITRATE_57600:
                dwBaudRate = CBR_57600;
                break;
            case PHBAL_REG_SERIALWIN_VALUE_BITRATE_115200:
                dwBaudRate = CBR_115200;
                break;
            case PHBAL_REG_SERIALWIN_VALUE_BITRATE_230400:
                dwBaudRate = 230400;
                break;
            case PHBAL_REG_SERIALWIN_VALUE_BITRATE_460800:
                dwBaudRate = 460800;
                break;
            default:
                dwBaudRate = pDataParams->dwBitRate;
                break;
        }

        /* Get number of stop bits */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams, PHBAL_REG_SERIALWIN_CONFIG_STOPBITS, &wStopBits));

        switch (wStopBits)
        {
        case PHBAL_REG_SERIALWIN_VALUE_STOPBITS_ONE:
            StopBits = 1;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_STOPBITS_ONE5:
            StopBits = 1.5;
            break;
        case PHBAL_REG_SERIALWIN_VALUE_STOPBITS_TWO:
            StopBits = 2;
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
        }

        /* Get Parity */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams, PHBAL_REG_SERIALWIN_CONFIG_PARITY, &wParity));

        switch(wParity)
        {
            case PHBAL_REG_SERIALWIN_VALUE_PARITY_NONE:
                wParity = 0;
                break;
            case PHBAL_REG_SERIALWIN_VALUE_PARITY_ODD:
            case PHBAL_REG_SERIALWIN_VALUE_PARITY_EVEN:
                wParity = 1;
                break;
            default:
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
        }

        /* Calculate additional timeout */
        dwAddTimeoutMs = (uint32_t)(1.0 / dwBaudRate * (9 + StopBits + wParity) * pDataParams->wAdditionalRxDelayBytes * 1000.0 + 0.5);
    }

    /* Prepare CTO structure */
    SecureZeroMemory( &cto, sizeof(COMMTIMEOUTS) );
    cto.ReadTotalTimeoutConstant = (uint32_t)pDataParams->wTimeout + dwAddTimeoutMs;
    cto.ReadTotalTimeoutMultiplier = (uint32_t)pDataParams->wTimeoutMultiplier;
    cto.ReadIntervalTimeout = (uint32_t)pDataParams->wTimeoutMultiplier;
    cto.WriteTotalTimeoutConstant = 0;
    cto.WriteTotalTimeoutMultiplier = 0;

    /* Apply timeouts */
    if (SetCommTimeouts(pDataParams->pComHandle, &cto) == FALSE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

phStatus_t phbalReg_SerialWin_Int_SetBitRateFromFiDi(
                                                     phbalReg_SerialWin_DataParams_t * pDataParams
                                                     )
{
#ifdef _WIN32
    phStatus_t PH_MEMLOC_REM status;
    const uint16_t wFtable[16] = { 372,    372, 558, 744, 1116, 1488, 1860, 0, 0 ,512, 768, 1024, 1536, 2048 };
    const uint16_t wDtable[16] = { 1, 1, 2, 4, 8, 16, 32, 64, 12, 20 };
    uint16_t wF, wD;
    uint32_t br;

    if ( ((0 <= pDataParams->bFI) && (pDataParams->bFI <= 6)) || ((9 <= pDataParams->bFI) && (pDataParams->bFI <= 13)) )
    {
        wF = wFtable[pDataParams->bFI];
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    if ( (1 <= pDataParams->bDI) && (pDataParams->bDI <= 9) )
    {
        wD = wDtable[pDataParams->bDI];
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    br = (uint32_t) ((1000000*3.5712*wD)/wF);
    switch (br)
    {
        case 9600:
            pDataParams->dwBitRate = PHBAL_REG_SERIALWIN_VALUE_BITRATE_9600;
            break;
        case 19200:
            pDataParams->dwBitRate = PHBAL_REG_SERIALWIN_VALUE_BITRATE_19200;
            break;
        case 38400:
            pDataParams->dwBitRate = PHBAL_REG_SERIALWIN_VALUE_BITRATE_38400;
            break;
        case 57600:
            pDataParams->dwBitRate = PHBAL_REG_SERIALWIN_VALUE_BITRATE_57600;
            break;
        case 115200:
            pDataParams->dwBitRate = PHBAL_REG_SERIALWIN_VALUE_BITRATE_115200;
            break;
        case 230400:
            pDataParams->dwBitRate = PHBAL_REG_SERIALWIN_VALUE_BITRATE_230400;
            break;
        case 460800:
            pDataParams->dwBitRate = PHBAL_REG_SERIALWIN_VALUE_BITRATE_460800;
            break;
        default:
            pDataParams->dwBitRate = br;
            break;
            //return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* set bit rate */
    status = phbalReg_SerialWin_Int_SetBitRate(pDataParams->pComHandle, pDataParams->dwBitRate);

    return PH_ADD_COMPCODE(status, PH_COMP_BAL);
#else
    /* satisfy compiler */
    if (pDataParams);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
#endif
}

#endif /* NXPBUILD__PHBAL_REG_SERIALWIN */
