/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * DUT specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7463 $
 * $Date: 2025-08-29 15:56:52 +0530 (Fri, 29 Aug 2025) $
 *
 * History:
 *      Created On October 12, 2022
 *
 */

#include <ph_Status.h>

#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_HAL
#include <ph_RefDefs.h>
#include <phhalHw_DUT_Cmd.h>

#include "../phhalHw_DUT.h"

phStatus_t phhalHw_DUT_Cmd_HAL_Exchange(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pTxBuff, uint16_t wTxLen,
    uint8_t ** ppRxBuff, uint16_t * pRxLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wPosition = 0;
    uint16_t    PH_MEMLOC_REM wRxLen = 0;
    uint8_t *   PH_MEMLOC_REM pRxBuff = NULL;

    /* Check options */
    if((wOption & (uint16_t) ~(uint16_t) (PH_EXCHANGE_BUFFERED_BIT | PH_EXCHANGE_LEAVE_BUFFER_BIT)) != 0x0000U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* Check if caller has provided valid RxBuffer */
    if(ppRxBuff == NULL)
    {
        ppRxBuff = &pRxBuff;
    }
    if(pRxLen == NULL)
    {
        pRxLen = &wRxLen;
    }

    /* clear internal buffer if requested */
    if((0U == ((wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))))
    {
        pDataParams->wTxBufLen = 0;
    }

    /* Calculate start position */
    wPosition = PHHAL_HW_DUT_RESERVED_BUFFER_LEN + pDataParams->wTxBufStartPos + pDataParams->wTxBufLen;

    /* Buffer overflow check */
    if((wPosition + wTxLen) > pDataParams->wTxBufSize)
    {
        pDataParams->wTxBufLen = 0;
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /* Copy transmit buffer*/
    (void) memcpy(&pDataParams->pTxBuffer[wPosition], pTxBuff, wTxLen);
    pDataParams->wTxBufLen = pDataParams->wTxBufLen + wTxLen;

    /* Set receive buffer properties */
    pDataParams->wRxBufLen = pDataParams->wRxBufStartPos;

    /* Perform the actual exchange */
    /* Execute via Hal Command */
    wStatus = PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_HAL_EXCHANGE,
        wOption,
        pTxBuff,
        wTxLen,
        ppRxBuff,
        pRxLen);

    /* Return correct data length */
    *ppRxBuff -= pDataParams->wRxBufStartPos;
    *pRxLen = *pRxLen + pDataParams->wRxBufStartPos;

    /* Store RxBufLen in DataParams */
    pDataParams->wRxBufLen = *pRxLen;

    /* Reset buffered bytes */
    pDataParams->wTxBufLen = 0;

    /* Below Check is added to mask INCOMPLETE_BYTE satus from PN7462AU as 0x72 since the fw retuns INCOMPLETE_BYTE as 0x73.
    This can be removed once RdLib is merged*/
    if((wStatus & PH_ERR_MASK) == 0x73)
    {
        wStatus = (wStatus & PH_COMP_MASK) | PH_ERR_SUCCESS_INCOMPLETE_BYTE;
    }

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_HAL_SetConfig(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmd[4];

    switch(wConfig)
    {
        case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:

            /* Boundary check */
            if((PHHAL_HW_DUT_RESERVED_BUFFER_LEN + wValue) >= pDataParams->wRxBufSize)
            {
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
            }

            /* Set start position */
            pDataParams->wRxBufStartPos = wValue;
            pDataParams->wRxBufLen = wValue;

            /* Preserve RxBuffer contents if needed */
            if(pDataParams->pTxBuffer == pDataParams->pRxBuffer)
            {
                pDataParams->wTxBufStartPos = pDataParams->wRxBufStartPos;
            }
            else
            {
                pDataParams->wTxBufStartPos = 0;
            }
            break;

        case PHHAL_HW_CONFIG_SETMINFDT:
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_DUT_SetMinFDT(pDataParams, wValue));
            break;

        case PHHAL_HW_CONFIG_SERIAL_BITRATE:
            /* FIXME:PN7462AU To be reviewed. 2014.04.16 : Porting BFL Test Wrapper
            * No purpose of serail bitrate for DUT Hal */
            break;

        case PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL:
            break;

        case PHHAL_HW_CONFIG_TIMING_MODE:
        case PHHAL_HW_CONFIG_TIMING_US:
        case PHHAL_HW_CONFIG_TIMING_MS:
        case PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT:
            aCmd[0] = (uint8_t) (wConfig & 0xFFU);
            aCmd[1] = 0x80;
            aCmd[2] = (uint8_t) (wValue & 0xFFU);
            aCmd[3] = (uint8_t) (wValue >> 8U);

            /* Execute via Hal Command */
            return PHHALHW_DUT_INT(
                pDataParams,
                PHHAL_HW_DUT_CMD_HAL_SET_CONFIG,
                PH_EXCHANGE_DEFAULT,
                aCmd,
                4,
                NULL,
                NULL);
            break;

        case PHHAL_HW_CONFIG_RXDEAFBITS: /* To be reverted when RdLib is merged. */
            wValue = (wValue * 9U);

        default:
            aCmd[0] = (uint8_t) (wConfig & 0xFFU);
            aCmd[1] = (uint8_t) (wConfig >> 8U);
            aCmd[2] = (uint8_t) (wValue & 0xFFU);
            aCmd[3] = (uint8_t) (wValue >> 8U);

            /* Execute via Hal Command */
            return PHHALHW_DUT_INT(
                pDataParams,
                PHHAL_HW_DUT_CMD_HAL_SET_CONFIG,
                PH_EXCHANGE_DEFAULT,
                aCmd,
                4,
                NULL,
                NULL);
    }

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_DUT_Cmd_HAL_GetConfig(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t *   PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t    PH_MEMLOC_REM wRxLength = 0;
    uint8_t     PH_MEMLOC_REM aCmd[2];

    switch(wConfig)
    {
        case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:
            *pValue = pDataParams->wRxBufStartPos;
            break;

        case PHHAL_HW_CONFIG_RXBUFFER_BUFSIZE:
            *pValue = pDataParams->wRxBufSize;
            break;

        case PHHAL_HW_CONFIG_TXBUFFER_BUFSIZE:
            *pValue = pDataParams->wTxBufSize - pDataParams->wTxBufStartPos;
            break;

        case PHHAL_HW_CONFIG_TIMING_MODE:
        case PHHAL_HW_CONFIG_TIMING_US:
        case PHHAL_HW_CONFIG_TIMING_MS:
        case PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT:
            aCmd[0] = (uint8_t) (wConfig & 0xFFU);
            aCmd[1] = 0x80;

            /* Execute via Hal Command */
            PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
                pDataParams,
                PHHAL_HW_DUT_CMD_HAL_GET_CONFIG,
                PH_EXCHANGE_DEFAULT,
                aCmd,
                2,
                &pRxBuffer,
                &wRxLength));

            /* check received length */
            if(wRxLength != 2U)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
            }

            *pValue = (uint16_t) pRxBuffer[0] | ((uint16_t) pRxBuffer[1] << 8U);
            break;

        case PHHAL_HW_CONFIG_CARD_TYPE:
            /* Return parameter */
            *pValue = (uint16_t) pDataParams->bCardType;
            break;

        default:
            aCmd[0] = (uint8_t) (wConfig & 0xFFU);
            aCmd[1] = (uint8_t) (wConfig >> 8U);

            /* Execute via Hal Command */
            PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
                pDataParams,
                PHHAL_HW_DUT_CMD_HAL_GET_CONFIG,
                PH_EXCHANGE_DEFAULT,
                aCmd,
                2,
                &pRxBuffer,
                &wRxLength));

            /* check received length */
            if(wRxLength != 2U)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
            }

            *pValue = (uint16_t) pRxBuffer[0] | ((uint16_t) pRxBuffer[1] << 8U);
            break;
    }
    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_DUT_Cmd_HAL_ApplyProtocolSettings(phhalHw_DUT_DataParams_t * pDataParams, uint8_t bCardType)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t PH_MEMLOC_REM aCmd[1];

    /* Prepare command */
    aCmd[0] = bCardType;

    /* Execute via Hal Command */
    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_HAL_APPLY_PROT_SETTINGS,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        1,
        NULL,
        NULL));

    /* Store new card type */
    if(bCardType != PHHAL_HW_CARDTYPE_CURRENT)
    {
        pDataParams->bCardType = bCardType;
    }

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_DUT_Cmd_HAL_Wait(phhalHw_DUT_DataParams_t * pDataParams, uint8_t bUnit, uint16_t wTimeout)
{
    uint8_t PH_MEMLOC_REM aCmd[3];

    aCmd[0] = bUnit;
    aCmd[1] = (uint8_t) (wTimeout & 0xFFU);
    aCmd[2] = (uint8_t) (wTimeout >> 8U);

    /* Execute via Hal Command */
    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_HAL_WAIT,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        3,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_HAL_MfcAuthenticate(phhalHw_DUT_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t bKeyType, uint8_t * pKey, uint8_t * pUid)
{
    uint8_t PH_MEMLOC_REM aCmd[12];

    aCmd[0] = bBlockNo;
    aCmd[1] = bKeyType;
    (void) memcpy(&aCmd[2], pKey, 6);
    (void) memcpy(&aCmd[8], pUid, 4);

    /* Exchange with lower layer */
    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_HAL_MFC_AUTH,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        12,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_HAL_Execute(phhalHw_DUT_DataParams_t * pDataParams, uint8_t bCmd, uint16_t wOption, uint8_t * pTxBuff, uint16_t wTxLen,
    uint8_t ** ppRxBuff, uint16_t * pRxLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmd[3];

    aCmd[0] = bCmd;
    aCmd[1] = (uint8_t) (wOption & 0xFFU);
    aCmd[2] = (uint8_t) (wOption >> 8U);

    /* Buffer the command-frame */
    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_HAL_EXECUTE,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmd,
        3,
        NULL,
        NULL));

    /* Append given data and Exchange with lower layer */
    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_HAL_EXECUTE,
        PH_EXCHANGE_BUFFER_LAST,
        pTxBuff,
        wTxLen,
        ppRxBuff,
        pRxLen);
}

phStatus_t phhalHw_DUT_Cmd_HAL_MfcAuthenticateKeyNo(phhalHw_DUT_DataParams_t * pDataParams, uint8_t bBlockNo, uint8_t bKeyType, uint16_t wKeyNo,
    uint16_t wKeyVer, uint8_t * pUid)
{
    uint8_t PH_MEMLOC_REM aCmd[10];

    aCmd[0] = bBlockNo;
    aCmd[1] = bKeyType;
    (void) memcpy(&aCmd[2], pUid, 4);
    aCmd[6] = (uint8_t) (wKeyNo & 0xFFU);
    aCmd[7] = (uint8_t) (wKeyNo >> 8U);
    aCmd[8] = (uint8_t) (wKeyVer & 0xFFU);
    aCmd[9] = (uint8_t) (wKeyVer >> 8U);

    /* Exchange with lower layer */
    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_HAL_MFC_AUTH_KEY_NO,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        10,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_HAL_Transmit(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pTxBuff, uint16_t wTxLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint16_t    PH_MEMLOC_REM wPosition;

    /* Calculate start position */
    wPosition = PHHAL_HW_DUT_RESERVED_BUFFER_LEN + pDataParams->wTxBufStartPos + pDataParams->wTxBufLen;

    /* Buffer overflow check */
    if((wPosition + wTxLen) > pDataParams->wTxBufSize)
    {
        pDataParams->wTxBufLen = 0;
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /* Copy transmit buffer*/
    (void) memcpy(&pDataParams->pTxBuffer[wPosition], pTxBuff, wTxLen);
    pDataParams->wTxBufLen = pDataParams->wTxBufLen + wTxLen;

    /* Perform the actual exchange */
    /* Execute via Hal Command */
    wStatus = PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_HAL_TRANSMIT,
        wOption,
        pTxBuff,
        wTxLen,
        NULL,
        NULL);

    /* Reset buffered bytes */
    pDataParams->wTxBufLen = 0;

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_HAL_Receive(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wOption, uint8_t ** ppRxBuff, uint16_t * pRxLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wRxLen = 0;
    uint8_t *   PH_MEMLOC_REM pRxBuff = NULL;

    /* Check if caller has provided valid RxBuffer */
    if(ppRxBuff == NULL)
    {
        ppRxBuff = &pRxBuff;
    }
    if(pRxLen == NULL)
    {
        pRxLen = &wRxLen;
    }

    /* Set receive buffer properties */
    pDataParams->wRxBufLen = pDataParams->wRxBufStartPos;

    /* Perform the actual exchange */
    /* Execute via Hal Command */
    wStatus = PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_HAL_RECEIVE,
        wOption,
        NULL,
        0,
        ppRxBuff,
        pRxLen);

    /* Return correct data length */
    *ppRxBuff -= pDataParams->wRxBufStartPos;
    *pRxLen = *pRxLen + pDataParams->wRxBufStartPos;

    /* Store RxBufLen in DataParams */
    pDataParams->wRxBufLen = *pRxLen;

    return wStatus;
}

phStatus_t phhalHw_DUT_Cmd_HAL_I18000p3m3Inventory(phhalHw_DUT_DataParams_t * pDataParams, uint8_t * pSelCmd, uint8_t bSelCmdLen, uint8_t bNumValidBitsinLastByte,
    uint8_t * pBeginRndCmd, uint8_t bTSprocessing, uint8_t ** ppRxBuff, uint16_t * pRxLen)
{
    uint8_t PH_MEMLOC_REM aCmd[50];
    uint16_t PH_MEMLOC_REM wStatus = 0;

    aCmd[0] = bSelCmdLen;
    (void) memcpy(&aCmd[1], pSelCmd, bSelCmdLen);

    aCmd[bSelCmdLen + 1U] = bNumValidBitsinLastByte;
    (void) memcpy(&aCmd[bSelCmdLen + 2U], pBeginRndCmd, 3);

    aCmd[bSelCmdLen + 5U] = bTSprocessing;

    /* Execute via Hal Command */
    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_HAL_I18000P3M3_INVENTORY,
        PH_EXCHANGE_DEFAULT,
        aCmd,
        (uint8_t) (bSelCmdLen + 6U),
        ppRxBuff,
        pRxLen));

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_DUT_Cmd_HAL_I18000p3m3ResumeInventory(phhalHw_DUT_DataParams_t * pDataParams, uint8_t ** ppRxBuff, uint16_t * pRxLen)
{
    uint16_t PH_MEMLOC_REM wStatus = 0;

    /* Execute via Hal Command */
    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_HAL_I18000P3M3_RESUME_INVENTORY,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        ppRxBuff,
        pRxLen));

    return PH_ERR_SUCCESS;
}

#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_HAL */
