/*
 * Copyright 2013, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * RC523 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <ph_RefDefs.h>
#include <phTools.h>
#include <phCryptoSym.h>

#ifdef NXPBUILD__PHHAL_HW_MFCCRYPTO

#include "phhalHw_MfcCrypto.h"
#include "phhalHw_MfcCrypto_Int.h"
#include "phhalHw_MfcCrypto_Cmd.h"

#pragma message("********** [Including Crypo1] ********** ") /* PRQA S 3116 */

phStatus_t phhalHw_MfcCrypto_Init(
                                  phhalHw_MfcCrypto_DataParams_t * pDataParams,
                                  uint16_t wSizeOfDataParams,
                                  void * pHalDataParams,
                                  uint8_t * pTxBuffer,
                                  uint16_t wTxBufSize
                                  )
{
    if (sizeof(phhalHw_MfcCrypto_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pHalDataParams);

    /* Init. private data */
    pDataParams->wId                = PH_COMP_HAL | PHHAL_HW_MFCCRYPTO_ID;
    pDataParams->pHalDataParams     = pHalDataParams;
    pDataParams->bCryptoState       = PHHAL_HW_MFCCRYPTO_STATE_ENCRYPT_OFF;
    pDataParams->bActivated         = 0;
    pDataParams->uIdForSpFeed1      = 0;
    pDataParams->bEnabled           = 1;
    pDataParams->pTxBuffer          = pTxBuffer;
    pDataParams->wTxBufSize         = wTxBufSize;
    pDataParams->wTxBufLen          = 0;
    pDataParams->bRxBits            = 0;

    memset(pDataParams->rA_buf, 0x00, sizeof(pDataParams->rA_buf));         /* PRQA S 3200 */
    memset(pDataParams->rBp_buf, 0x00, sizeof(pDataParams->rBp_buf));       /* PRQA S 3200 */
    memset(pDataParams->cryptoReg, 0x00, sizeof(pDataParams->cryptoReg));   /* PRQA S 3200 */
    memset(pDataParams->randReg, 0x00, sizeof(pDataParams->randReg));       /* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_MfcCrypto_Exchange(
                                      phhalHw_MfcCrypto_DataParams_t * pDataParams,
                                      uint16_t wOption,
                                      uint8_t * pTxBuffer,
                                      uint16_t wTxLength,
                                      uint8_t ** ppRxBuffer,
                                      uint16_t * pRxLength
                                      )
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint16_t    PH_MEMLOC_REM wParity;
    uint16_t    PH_MEMLOC_REM wRxCrc;
    uint16_t    PH_MEMLOC_REM wTxCrc;
    uint16_t    PH_MEMLOC_REM wCrc;
    uint16_t    PH_MEMLOC_REM wTxBytes;
    uint8_t     PH_MEMLOC_REM bTxBits;
    uint32_t    PH_MEMLOC_REM dwDataBits;
    uint8_t     PH_MEMLOC_BUF aTmpBuffer[9];

    if ((pDataParams->bEnabled == 0) || (pDataParams->bCryptoState == PHHAL_HW_MFCCRYPTO_STATE_ENCRYPT_OFF))
    {
        return phhalHw_Exchange(pDataParams->pHalDataParams, wOption, pTxBuffer, wTxLength, ppRxBuffer, pRxLength );
    }
    else
    {
        /* TxBuffer overflow check */
        if ((pDataParams->wTxBufLen + wTxLength) > pDataParams->wTxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* copy data */
        memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pTxBuffer, wTxLength); /* PRQA S 3200 */
        pDataParams->wTxBufLen = pDataParams->wTxBufLen + wTxLength;

        /* do not perform real exchange, just fill the global TxBuffer */
        if (wOption & PH_EXCHANGE_BUFFERED_BIT)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
        }

        /* Retrieve Parity-setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_PARITY, &wParity));
        /* Retrieve CRC TX-setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXCRC, &wTxCrc));
        /* Retrieve CRC RX-setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXCRC, &wRxCrc));

        if (wTxCrc)
        {
            /* TxBuffer overflow check */
            if ((pDataParams->wTxBufSize - 2) < pDataParams->wTxBufLen)
            {
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
            }

            /* Append the CRC */
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                PH_TOOLS_CRC_OPTION_DEFAULT,
                PH_TOOLS_CRC16_PRESET_ISO14443A,
                PH_TOOLS_CRC16_POLY_ISO14443,
                pDataParams->pTxBuffer,
                pDataParams->wTxBufLen,
                &wCrc));

            pDataParams->pTxBuffer[pDataParams->wTxBufLen++] = (uint8_t)wCrc;
            pDataParams->pTxBuffer[pDataParams->wTxBufLen++] = (uint8_t)(wCrc>> 8);
        }

        /* Disable parity */
        if (wParity != PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_PARITY, PH_OFF));
        }

        /* Disable TX CRC */
        if (wTxCrc != PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXCRC, PH_OFF));
        }

        /* Disable RX CRC */
        if (wRxCrc != PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXCRC, PH_OFF));
        }

        /* Prepare overall length */
        pTxBuffer = pDataParams->pTxBuffer;
        wTxLength = pDataParams->wTxBufLen;

        /* Reset number of buffered bytes */
        pDataParams->wTxBufLen = 0;

        /* Perform encoding and buffering */
        wOption = PH_EXCHANGE_BUFFER_FIRST;
        do
        {
            /* Calc number of bytes to encode */
            wTxBytes = (wTxLength < 8) ? wTxLength : 8;

            /* Decrement overall TxLenght */
            wTxLength = wTxLength - wTxBytes;

            /* Calculate Parity */
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_EncodeParity(
                PH_TOOLS_PARITY_OPTION_ODD,
                pTxBuffer,
                wTxBytes,
                0,
                sizeof(aTmpBuffer),
                aTmpBuffer,
                &wTxBytes,
                &bTxBits));

            /* Calc. Bitlength for full bytes */
            if ((wTxLength > 0) || (bTxBits == 0))
            {
                dwDataBits = (uint32_t)((uint32_t)wTxBytes << 0x03U);
            }
            /* Calc. Bitlength for last (incomplete) byte */
            else
            {
                dwDataBits = (uint32_t)((uint32_t)(wTxBytes - 1) << 0x03U) + bTxBits;
                /* Set TxLastBits */
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXLASTBITS, bTxBits));
            }

            /* Perform MIFARE Enciphering */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_MfcCrypto_Cmd_CodeData(
                pDataParams,
                aTmpBuffer,
                dwDataBits,
                aTmpBuffer));

            /* If this is the last chunk perform sending */
            if (wTxLength == 0)
            {
                wOption &= (uint16_t)~(uint16_t)PH_EXCHANGE_BUFFERED_BIT;
            }

            /* Increment Transmit Buffer */
            pTxBuffer = &pTxBuffer[8];

            /* Perform the exchange */
            status = phhalHw_Exchange(
                pDataParams->pHalDataParams,
                wOption,
                aTmpBuffer,
                wTxBytes,
                ppRxBuffer,
                pRxLength);

            /* Retain buffered data */
            wOption |= PH_EXCHANGE_LEAVE_BUFFER_BIT;
        }
        while (wTxLength > 0);

        /* Restore parity */
        if (wParity != PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_PARITY, wParity));
        }

        /* Restore TX CRC */
        if (wTxCrc != PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXCRC, wTxCrc));
        }

        /* Restore RX CRC */
        if (wRxCrc != PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXCRC, wRxCrc));
        }

        /* Check error */
        if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS_INCOMPLETE_BYTE)
        {
            PH_CHECK_SUCCESS(status);
        }

        /* Get RX LAST BITS */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXLASTBITS, &wTxBytes));
        pDataParams->bRxBits = (uint8_t)wTxBytes;

        /* Calc. Bitlength for full bytes */
        if (pDataParams->bRxBits == 0)
        {
            dwDataBits = (uint32_t)((uint32_t)*pRxLength << 0x03U);
        }
        /* Calc. Bitlength for last (incomplete) byte */
        else
        {
            dwDataBits = (uint32_t)((uint32_t)(*pRxLength - 1) << 0x03U) + pDataParams->bRxBits;
        }

        /* Remove MIFARE Cipher */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_MfcCrypto_Cmd_CodeData(
            pDataParams,
            *ppRxBuffer,
            dwDataBits,
            *ppRxBuffer));

        /* Parity / CRC check s*/
        if (*pRxLength > 1)
        {
            /* Verify Parity */
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_DecodeParity(
                PH_TOOLS_PARITY_OPTION_ODD,
                *ppRxBuffer,
                *pRxLength,
                pDataParams->bRxBits,
                *pRxLength,
                *ppRxBuffer,
                pRxLength,
                &pDataParams->bRxBits));

            /* Check the CRC if necessary */
            if (wRxCrc)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
                    PH_TOOLS_CRC_OPTION_DEFAULT,
                    PH_TOOLS_CRC16_PRESET_ISO14443A,
                    PH_TOOLS_CRC16_POLY_ISO14443,
                    *ppRxBuffer,
                    *pRxLength,
                    &wCrc));

                if (wCrc)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
                }
                *pRxLength = *pRxLength - 2;
            }

            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
        }
        else
        {
            return status;
        }
    }
}

phStatus_t phhalHw_MfcCrypto_WriteRegister(
    phhalHw_MfcCrypto_DataParams_t * pDataParams,
    uint8_t bAddress,
    uint8_t bValue
    )
{
    /* Redirect to lower layer */
    return phhalHw_WriteRegister(pDataParams->pHalDataParams, bAddress, bValue);
}

phStatus_t phhalHw_MfcCrypto_ReadRegister(
    phhalHw_MfcCrypto_DataParams_t * pDataParams,
    uint8_t bAddress,
    uint8_t * pValue
    )
{
    /* Redirect to lower layer */
    return phhalHw_ReadRegister(pDataParams->pHalDataParams, bAddress, pValue);
}

phStatus_t phhalHw_MfcCrypto_ApplyProtocolSettings(
    phhalHw_MfcCrypto_DataParams_t * pDataParams,
    uint8_t bCardType
    )
{
    /* Redirect to lower layer */
    return phhalHw_ApplyProtocolSettings(pDataParams->pHalDataParams, bCardType);
}

phStatus_t phhalHw_MfcCrypto_SetConfig(
                                       phhalHw_MfcCrypto_DataParams_t * pDataParams,
                                       uint16_t wConfig,
                                       uint16_t wValue
                                       )
{
    phStatus_t PH_MEMLOC_REM statusTmp;

    /* We only need to handle MF Crypto reset here! */
    switch(wConfig)
    {
    case PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1:
        if (wValue == PH_ON)
        {
            pDataParams->bCryptoState = PHHAL_HW_MFCCRYPTO_STATE_ENCRYPT_OFF;
            phhalHw_MfcCrypto_ResetCrypto(pDataParams);
        }
        break;

    case PHHAL_HW_MFCCRYPTO_CONFIG_ENABLE:
        pDataParams->bEnabled = (uint8_t)wValue;

        /* if the MIFARE Crypto will be enabled, we have to switch off the CRC's,
        parity and an eventaully activated MF crypto on the hal layer below */
        if (wValue == PH_ON)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1, PH_ON));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXCRC, PH_OFF));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXCRC, PH_OFF));
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_PARITY, PH_OFF));
        }
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

    default:
        break;
    }

    return phhalHw_SetConfig(pDataParams->pHalDataParams, wConfig, wValue);
}

phStatus_t phhalHw_MfcCrypto_GetConfig(
                                       phhalHw_MfcCrypto_DataParams_t * pDataParams,
                                       uint16_t wConfig,
                                       uint16_t * pValue
                                       )
{
    switch (wConfig)
    {
    case PHHAL_HW_CONFIG_RXLASTBITS:
        if ((pDataParams->bEnabled) && (pDataParams->bCryptoState != PHHAL_HW_MFCCRYPTO_STATE_ENCRYPT_OFF))
        {
            *pValue = pDataParams->bRxBits;
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
        }
        break;

    default:
        break;
    }

    return phhalHw_GetConfig(pDataParams->pHalDataParams, wConfig, pValue);
}

phStatus_t phhalHw_MfcCrypto_FieldOn(
                                     phhalHw_MfcCrypto_DataParams_t * pDataParams
                                     )
{
    /* Redirect to lower layer */
    return phhalHw_FieldOn(pDataParams->pHalDataParams);
}

phStatus_t phhalHw_MfcCrypto_FieldOff(
                                      phhalHw_MfcCrypto_DataParams_t * pDataParams
                                      )
{
    /* Redirect to lower layer */
    return phhalHw_FieldOff(pDataParams->pHalDataParams);
}

phStatus_t phhalHw_MfcCrypto_FieldReset(
                                        phhalHw_MfcCrypto_DataParams_t * pDataParams
                                        )
{
    /* Redirect to lower layer */
    return phhalHw_FieldReset(pDataParams->pHalDataParams);
}

phStatus_t phhalHw_MfcCrypto_Wait(
                                  phhalHw_MfcCrypto_DataParams_t * pDataParams,
                                  uint8_t bUnit,
                                  uint16_t wTimeout
                                  )
{
    /* Redirect to lower layer */
    return phhalHw_Wait(pDataParams->pHalDataParams, bUnit, wTimeout);
}

phStatus_t phhalHw_MfcCrypto_MfcAuthenticateKeyNo(
    phhalHw_MfcCrypto_DataParams_t * pDataParams,
    uint8_t bBlockNo,
    uint8_t bKeyType,
    uint16_t wKeyNo,
    uint16_t wKeyVersion,
    uint8_t * pUid)
{
    if (pDataParams->bEnabled)
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
    }
    else
    {
        return phhalHw_MfcAuthenticateKeyNo(pDataParams->pHalDataParams, bBlockNo, bKeyType, wKeyNo, wKeyVersion, pUid);
    }
}

phStatus_t phhalHw_MfcCrypto_MfcAuthenticate(
    phhalHw_MfcCrypto_DataParams_t * pDataParams,
    uint8_t bBlockNo,
    uint8_t bKeyType,
    uint8_t * pKey,
    uint8_t * pUid)
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM aCmd[9];
    uint8_t     PH_MEMLOC_REM aRa[4];
    uint8_t     PH_MEMLOC_REM aRb[4];
    uint8_t     PH_MEMLOC_REM aRb2[4];
    uint16_t    PH_MEMLOC_REM wParity;
    uint16_t    PH_MEMLOC_REM wTxCrc;
    uint16_t    PH_MEMLOC_REM wRxCrc;
    uint16_t    PH_MEMLOC_REM wRxBits;
    uint8_t *   PH_MEMLOC_REM pResponse;
    uint16_t    PH_MEMLOC_REM wResponseLength;

    if (pDataParams->bEnabled)
    {
        /* Send Auth1 Command */
        if (bKeyType == PHHAL_HW_MFC_KEYA)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_MfcCrypto_Cmd_EncodeAuth1(pDataParams, 0x60, bBlockNo, aCmd));
        }
		else if (bKeyType == PHHAL_HW_MFC_KEYB)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_MfcCrypto_Cmd_EncodeAuth1(pDataParams, 0x61, bBlockNo, aCmd));
		}
		else
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
		}

        /* Retrieve Parity-setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_PARITY, &wParity));
        /* Retrieve CRC TX-setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXCRC, &wTxCrc));
        /* Retrieve CRC RX-setting */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXCRC, &wRxCrc));

        /* Disable parity */
        if (wParity != PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_PARITY, PH_OFF));
        }

        /* Disable TX CRC */
        if (wTxCrc != PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXCRC, PH_OFF));
        }

        /* Disable RX CRC */
        if (wRxCrc != PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXCRC, PH_OFF));
        }

        /* Set TxLastBits */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXLASTBITS, 4));

        /* Perform the exchange */
        status = phhalHw_Exchange(
            pDataParams->pHalDataParams,
            PH_EXCHANGE_DEFAULT,
            aCmd,
            5,
            &pResponse,
            &wResponseLength);

        /* Only allow incomplete byte status */
        if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS_INCOMPLETE_BYTE)
        {
            /* Restore parity */
            if (wParity != PH_OFF)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_PARITY, wParity));
            }

            /* Restore TX CRC */
            if (wTxCrc != PH_OFF)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXCRC, wTxCrc));
            }

            /* Restore RX CRC */
            if (wRxCrc != PH_OFF)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXCRC, wRxCrc));
            }

            /* return error */
            if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_HAL);
            }
            else
            {
                return status;
            }
        }

        /* Get number of valid bits */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXLASTBITS, &wRxBits));

        /* Decode Auth1 Response */
        status = phhalHw_MfcCrypto_Cmd_DecodeAuth1(pDataParams, pResponse, wResponseLength, (uint8_t)wRxBits, pUid, pKey, aRa,aRb);

        if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Send Auth2 Command */
            status = phhalHw_MfcCrypto_Cmd_EncodeAuth2(pDataParams, aCmd);

            if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
            {
                /* Set TxLastBits */
                PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXLASTBITS, 0));

                /* Perform the exchange */
                status = phhalHw_Exchange(
                    pDataParams->pHalDataParams,
                    PH_EXCHANGE_DEFAULT,
                    aCmd,
                    9,
                    &pResponse,
                    &wResponseLength);
            }
        }

        /* Restore parity */
        if (wParity != PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_PARITY, wParity));
        }

        /* Restore TX CRC */
        if (wTxCrc != PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_TXCRC, wTxCrc));
        }

        /* Restore RX CRC */
        if (wRxCrc != PH_OFF)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXCRC, wRxCrc));
        }

        /* Only allow incomplete byte status */
        if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS_INCOMPLETE_BYTE)
        {
            /* return error */
            if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_HAL);
            }
            else
            {
                return status;
            }
        }

        /* Get number of valid bits */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(pDataParams->pHalDataParams, PHHAL_HW_CONFIG_RXLASTBITS, &wRxBits));

        /* Decode Auth1 Response */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_MfcCrypto_Cmd_DecodeAuth2(pDataParams, pResponse, wResponseLength, (uint8_t)wRxBits,aRb2));

        /* Crypto is now active */
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }
    else
    {
        return phhalHw_MfcAuthenticate(pDataParams->pHalDataParams, bBlockNo, bKeyType, pKey, pUid);
    }
}

#endif /* NXPBUILD__PHHAL_HW_MFCCRYPTO */
