/*
 * Copyright 2013, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * RD710 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phhalHw.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHHAL_HW_MFCCRYPTO

#include "phhalHw_MfcCrypto.h"
#include "phhalHw_MfcCrypto_Int.h"

void phhalHw_MfcCrypto_SwapByteArray(uint8_t * pByteArray, uint16_t wArrayLen)
{
    uint8_t tmp;
    uint16_t i;

    for (i = 0; i < wArrayLen >> 1; i++)
    {
        tmp = pByteArray[i];
        pByteArray[i] = pByteArray[wArrayLen - i - 1];
        pByteArray[wArrayLen - i - 1] = tmp;
    }
}

/*!
*    Computes the boolean function based on the current crypto register content
*
*    \param[in] reg  	The crypto register
**	 \return 	the ouput bit
*
*/
uint8_t phhalHw_MfcCrypto_ComputeF(uint8_t * reg)
{
    /* tables for f-functions: MS input bit for any f-function will be MSB in index variable */
    uint8_t PH_MEMLOC_CONST_ROM F4[16] = {1, 1, 0, 0, 1, 0, 1, 1, 1, 0, 1, 1, 0, 0, 0, 0};
    uint8_t PH_MEMLOC_CONST_ROM F4s[16] = {1, 1, 1, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0, 1, 0, 0};
    uint8_t PH_MEMLOC_CONST_ROM F5[32] = {1, 1, 0, 0, 1, 1, 0, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0};

    uint8_t f4In=0, f5In=0;

    f4In = reg[38] << 3 | reg[36] << 2 | reg[34] << 1 | reg[32];

    f5In = F4s[f4In] << 4;

    f4In = reg[30] << 3 | reg[28] << 2 | reg[26] << 1 | reg[24];

    f5In |= F4[f4In] << 3;

    f4In = reg[22] << 3 | reg[20] << 2 | reg[18] << 1 | reg[16];

    f5In |= F4[f4In] << 2;

    f4In = reg[14] << 3 | reg[12] << 2 | reg[10] << 1 | reg[8];

    f5In |= F4s[f4In] << 1;

    f4In = reg[6] << 3 | reg[4] << 2 | reg[2] << 1 | reg[0];

    f5In |= F4[f4In];

    return F5[f5In];
}

/*!
*    Loads register with input bit (direct load mode)
*
*    \param[in,out] reg 	The Rand register
*    \param[in] 	in 		An input bit
*
*/
void phhalHw_MfcCrypto_SeedRandReg(uint8_t * reg, uint8_t in)
{
    uint8_t i;
    for (i=0; i<15; ++i)
    {
        reg[i] = reg[i+1];
    }
    reg[15] = in;
}

/*!
*   Loads register with input bits (direct load mode).
* 	\note LSB is loaded first.
*
*   \param[in, out] reg  	The rand register
*	\param[in] 		in 		An input buffer
* 	\param[in] 		size 	Size of the input buffer (nr of bytes)
*
*/
void phhalHw_MfcCrypto_SeedRandRegWithBuffer(uint8_t * reg, uint8_t* in, uint16_t size)
{
    uint16_t i;
    uint8_t j;
    uint8_t curr;
    for (i=size; i>0; i--)
    {
        curr = in[i-1];
        for (j=0; j<8; ++j)
        {
            phhalHw_MfcCrypto_SeedRandReg(reg, curr & 1);
            curr >>= 1;
        }
    }
}

/*!
*    Clocks the register one tick (normal mode).
*
*   \param[in, out] reg  The rand register
*	\return 		The output bit
*/
uint8_t phhalHw_MfcCrypto_ClockRandReg(uint8_t * reg)
{
    uint8_t out = reg[0];

    uint8_t in = reg[0] ^ reg[2] ^ reg[3] ^ reg[5];

    phhalHw_MfcCrypto_SeedRandReg(reg, in);
    return out;
}

/*!
*   Clocks the register several ticks (normal mode). The number of ticks is equal to the output
* 	buffer size.
*
*   \param[in, out] reg 	The rand register
*	\param[out] 	out 	An output buffer
*	\param[in] 		size 	Size of the output buffer (nr of bytes = nr clocks wanted / 8)
*/
void phhalHw_MfcCrypto_ClockRandRegWithBuffer(uint8_t * reg, uint8_t* out, uint16_t size)
{
    uint16_t i;
    uint8_t j;

    for (i=size; i>0; i--)
    {
        out[i-1]=0;
        for (j=0; j<8; ++j)
        {
            out[i-1] = (uint8_t)out[i-1] | (uint8_t)((uint8_t)phhalHw_MfcCrypto_ClockRandReg(reg) << j);
        }
    }
}

/*!
*    Loads register with input bit (direct load mode)
*
*    \param[in, out] reg  	The crypto register
*    \param[in]		 in 	An input bit
*
*/
void phhalHw_MfcCrypto_LoadCryptoReg(uint8_t *  reg, uint8_t in)
{
    uint8_t i;
    for (i=47; i>0; --i)
    {
        reg[i] = reg[i-1];
    }
    reg[0] = in;
}

/*!
*   Loads register with input bits (direct load mode).
* 	\note LSB is loaded first.
*
*    \param[in, out] 	reg  	The crypto register
*	 \param[in] 		in	 	An input buffer
* 	 \param[in] 		size 	Size of the input buffer (nr of bytes)
*/
void phhalHw_MfcCrypto_LoadCryptoRegWithBuffer(uint8_t * reg, uint8_t* in, uint16_t size)
{
    uint16_t i;
    uint8_t j;
    uint8_t curr;
    for (i=size; i>0; i--)
    {
        curr = in[i-1];
        for (j=0; j<8; ++j)
        {
            phhalHw_MfcCrypto_LoadCryptoReg(reg, curr & 1);
            curr >>= 1;
        }
    }
}

/*!
*    Feeds register with input bit (simple feedback mode)
*
*    \param[in, out] 	reg  	The crypto register
*	 \return 			An output bit (nb: this is a keystream bit)
*/
uint8_t phhalHw_MfcCrypto_SimFeedCryptoReg(uint8_t * reg)
{
    uint8_t out = phhalHw_MfcCrypto_ComputeF(reg);

    uint8_t in = reg[47] ^ reg[42] ^ reg[38] ^ reg[37] ^ reg[35] ^ reg[33] ^ reg[32] ^ reg[30] ^ reg[28]
    ^ reg[23] ^ reg[22] ^ reg[20] ^ reg[18] ^ reg[12] ^ reg[8]  ^ reg[6]  ^ reg[5]  ^ reg[4];

    phhalHw_MfcCrypto_LoadCryptoReg(reg, in);
    return out;
}

/*!
*    Feeds register with input bit (special feedback 1 mode),
*
*    \param[in, out] 	reg  	The crypto register
*    \param[in] 		in 		An input bit
*	 \return 			An output bit (nb: this is a keystream bit)
*/
uint8_t phhalHw_MfcCrypto_Feed1CryptoReg(uint8_t * reg, uint8_t in)
{
    uint8_t out = phhalHw_MfcCrypto_ComputeF(reg);

    in = in ^ reg[47] ^ reg[42] ^ reg[38] ^ reg[37] ^ reg[35] ^ reg[33] ^ reg[32] ^ reg[30] ^ reg[28]
    ^ reg[23] ^ reg[22] ^ reg[20] ^ reg[18] ^ reg[12] ^ reg[8]  ^ reg[6]  ^ reg[5]  ^ reg[4];

    phhalHw_MfcCrypto_LoadCryptoReg(reg, in);
    return out;
}

/*!
*    Feeds register with input bit (special feedback 2 mode).
*
*    \param[in, out] 	reg  	The crypto register
*    \param[in] 		in 		An input bit
*	 \return 			An output bit (nb: this is a keystream bit)
*/
uint8_t phhalHw_MfcCrypto_Feed2CryptoReg(uint8_t * reg, uint8_t in)
{
    uint8_t out = phhalHw_MfcCrypto_ComputeF(reg);

    in = in ^ reg[47] ^ reg[42] ^ reg[38] ^ reg[37] ^ reg[35] ^ reg[33] ^ reg[32] ^ reg[30] ^ reg[28]
    ^ reg[23] ^ reg[22] ^ reg[20] ^ reg[18] ^ reg[12] ^ reg[8]  ^ reg[6]  ^ reg[5]  ^ reg[4];

    phhalHw_MfcCrypto_LoadCryptoReg(reg, in ^ out);

    return out;
}

/* Bitwise insertion into BitArray
*  Note: pBitArray and pBits are normal ByteArrays, which means each byte represents 8 bits.
*/
void phhalHw_MfcCrypto_InsertBits(
                                  uint8_t * pBitArray,
                                  uint32_t * pdwBitArrayLen,
                                  uint32_t dwBitPosition,
                                  uint8_t * pBits,
                                  uint32_t dwBitCount,
                                  uint8_t bReversed,
                                  uint8_t bReplace)
{
    uint8_t bBit;
    uint8_t  bValue;
    uint16_t wByteIndex = 0;
    uint32_t dwBitIndex;
    uint32_t dwBitPositionSource;

    /* Check input parameters */
    if ((pBitArray == NULL) ||
        (pdwBitArrayLen == NULL) ||
        (pBits == NULL) ||
        (dwBitCount == 0))
    {
        return;
    }

    /* Move present bits out of the way */
    if ((bReplace == 0) && (*pdwBitArrayLen > 0))
    {
        for (dwBitIndex = *pdwBitArrayLen; dwBitIndex > dwBitPosition; dwBitIndex--)
        {
            /* Copy the very right bit of source to very right bit of destination */
            if (bReversed == 1)
            {
                bBit = (uint8_t)((pBitArray[(uint32_t)(dwBitIndex-1) >> 3] & (uint8_t)((uint8_t)0x01U << (uint8_t)((uint32_t)(dwBitIndex-1) % 8))) != 0);
                bValue = (uint8_t)(0x01 << (((dwBitIndex-1)+dwBitCount) % 8));
            }
            else
            {
                bBit = (uint8_t)((pBitArray[(uint32_t)(dwBitIndex-1) >> 3] & (uint8_t)((uint8_t)0x80U >> (uint8_t)((uint32_t)(dwBitIndex-1) % 8))) != 0);
                bValue = (uint8_t)0x80U >> (uint8_t)(((uint32_t)(dwBitIndex-1)+dwBitCount) % 8);
            }

            /* Move it out of the way */
            if (bBit == 1)
            {
                pBitArray[(uint32_t)((dwBitIndex-1)+dwBitCount) >> 3] |= bValue;
            }
            else
            {
                pBitArray[(uint32_t)((dwBitIndex-1)+dwBitCount) >> 3] &= (uint8_t)~(uint8_t)bValue;
            }
        }
    }

    dwBitPositionSource = 0;
    do
    {
        /* Increment the ByteIndex if neccessary */
        if ((dwBitPositionSource > 0) &&
            (dwBitPositionSource % 8) == 0)
        {
            wByteIndex++;
        }

        /* Get next bit value */
        if (bReversed == 1)
        {
            bBit = (uint8_t)((pBits[wByteIndex] & (uint8_t)((uint8_t)0x01U << (uint32_t)dwBitPositionSource % 8)) != 0);
            bValue = (uint8_t)0x01U << (uint8_t)((uint32_t)dwBitPosition % 8);
        }
        else
        {
            bBit = (uint8_t)((pBits[wByteIndex] & (uint8_t)(0x80U >> (dwBitPositionSource % 8))) != 0);
            bValue = (uint8_t)0x80U >> (uint8_t)((uint32_t)dwBitPosition % 8);
        }

        /* Add new bit to array */
        if (bBit == 1)
        {
            pBitArray[dwBitPosition >> 3] |= bValue;
        }
        else
        {
            pBitArray[dwBitPosition >> 3] &= (uint8_t)~(uint8_t)bValue;
        }

        /* Increment Bit Positions */
        dwBitPosition++;
        dwBitPositionSource++;

        /* Increment Array length */
        if (bReplace == 0)
        {
            (*pdwBitArrayLen)++;
        }
    }
    while (--dwBitCount > 0);
}

void phhalHw_MfcCrypto_GetBits(
                               uint8_t * pBitArray,
                               uint32_t dwBitArrayLen,
                               uint32_t dwBitPosition,
                               uint8_t * pBits,
                               uint32_t dwBitCount,
                               uint8_t bReversed)
{
    uint8_t bBit;
    uint16_t wByteIndex = 0;
    uint16_t wBitIndex = 0;

    /* Check input parameters */
    if ((pBitArray == NULL) ||
        (dwBitArrayLen == 0) ||
        (pBits == NULL) ||
        (dwBitCount == 0) ||
        (dwBitCount > (dwBitArrayLen - dwBitPosition)))
    {
        return;
    }

    do
    {
        /* Increment the byteIndex if neccessary */
        if ((wBitIndex > 0) &&
            (wBitIndex % 8) == 0)
        {
            wByteIndex++;
        }

        /* Get next bit value */
        if (bReversed == 1)
        {
            bBit = (uint8_t)((pBitArray[dwBitPosition >> 3] & (uint8_t)((uint8_t)0x01U << (uint8_t)((uint32_t)dwBitPosition % 8))) != 0);
        }
        else
        {
            bBit = (uint8_t)((pBitArray[dwBitPosition >> 3] & (uint8_t)((uint8_t)0x80U >> (uint8_t)((uint32_t)dwBitPosition % 8))) != 0);
        }

        /* Add bitvalue to output array */
        if (bBit == 1)
        {
            if (bReversed == 1)
            {
                pBits[wByteIndex] = (uint8_t)pBits[wByteIndex] | (uint8_t)((uint8_t)0x01U << (uint8_t)((uint16_t)wBitIndex % 8));
            }
            else
            {
                pBits[wByteIndex] = (uint8_t)pBits[wByteIndex] | (uint8_t)((uint8_t)0x80U >> (uint8_t)((uint16_t)wBitIndex % 8));
            }
        }
        else
        {
            if (bReversed == 1)
            {
                pBits[wByteIndex] = (uint8_t)pBits[wByteIndex] & (uint8_t)~(0x01U << (uint8_t)((uint16_t)wBitIndex % 8));
            }
            else
            {
                pBits[wByteIndex] = (uint8_t)pBits[wByteIndex] & (uint8_t)~(0x80U >> (uint8_t)((uint16_t)wBitIndex % 8));
            }
        }

        /* Increment Bit Position */
        dwBitPosition++;

        /* Increment Bit Index */
        wBitIndex++;
    }
    while (--dwBitCount > 0);
}

/*
*   Reset the crypto ..
*/
void phhalHw_MfcCrypto_ResetCrypto(phhalHw_MfcCrypto_DataParams_t * pDataParams)
{
    /* Initialize pDataParams->cryptoReg, pDataParams->randReg, key */
    memset(pDataParams->cryptoReg, 0x00, MF_KEYSIZE_BITS); /* PRQA S 3200 */
    memset(pDataParams->randReg, 0x00, MF_RANDREGSIZE_BITS); /* PRQA S 3200 */

    pDataParams->bCryptoState = PHHAL_HW_MFCCRYPTO_STATE_ENCRYPT_OFF;
}

#endif /* NXPBUILD__PHHAL_HW_MFCCRYPTO */
