/*
 * Copyright 2017 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * PCSC ISO14443-3A Component of Reader Library.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_RefDefs.h>
#include <phpalI14443p3a.h>

#ifdef NXPBUILD__PHPAL_I14443P3A_PCSC

#include <phhalHw.h>
#include <phhalHw_Pcsc_Cmd.h>
#include "phpalI14443p3a_Pcsc.h"

phStatus_t phpalI14443p3a_Pcsc_Init(
                                    phpalI14443p3a_Pcsc_DataParams_t * pDataParams,
                                    uint16_t wSizeOfDataParams,
                                    void * pHalDataParams
                                    )
{
    if (sizeof(phpalI14443p3a_Pcsc_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_ISO14443P3A);
    }

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pHalDataParams);

    /* Check data parameters */
    if (PH_GET_COMPCODE(pHalDataParams) != PH_COMP_HAL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P3A);
    }

    /* Extract BAL */
    switch (PH_GET_COMPID(pHalDataParams))
    {
#ifdef NXPBUILD__PHHAL_HW_CALLBACK
        case PHHAL_HW_CALLBACK_ID:
            /* Check if required PCSC HAL is privided with pCbContext and BAL is provided with PCSC HAL */
            if (((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext == NULL)
            {
                pDataParams->pBalDataParams = NULL;
                pDataParams->pHalPcscDataParams = NULL;
                break;
            }

            if (PH_GET_COMPCODE(((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext) != PH_COMP_HAL)
            {
                pDataParams->pBalDataParams = NULL;
                pDataParams->pHalPcscDataParams = NULL;
                break;
            }

            if(PH_GET_COMPID(((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext) != PHHAL_HW_PCSC_ID)
            {
                pDataParams->pBalDataParams = NULL;
                pDataParams->pHalPcscDataParams = NULL;
                break;
            }

            /* Set internal pcsc hal dataparams */
            pDataParams->pHalPcscDataParams = (phhalHw_Pcsc_DataParams_t *)(((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext);

            /* Check if required BAL is provided with HAL */
            if (((phhalHw_Pcsc_DataParams_t *)(((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext))->pBalDataParams == NULL)
            {
                pDataParams->pBalDataParams = NULL;
                break;
            }
            pDataParams->pBalDataParams = (phbalReg_PcscWin_DataParams_t *)
                ((phhalHw_Pcsc_DataParams_t *)(((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext))->pBalDataParams;
            break;
#endif /* NXPBUILD__PHHAL_HW_CALLBACK */

#ifdef NXPBUILD__PHHAL_HW_PCSC
        case PHHAL_HW_PCSC_ID:
            /* Set internal pcsc hal dataparams */
            pDataParams->pHalPcscDataParams = (phhalHw_Pcsc_DataParams_t *)pHalDataParams;

            /* Check if required BAL is provided with HAL */
            if (((phhalHw_Pcsc_DataParams_t *)pHalDataParams)->pBalDataParams == NULL)
            {
                pDataParams->pBalDataParams = NULL;
                break;
            }
            pDataParams->pBalDataParams = (phbalReg_PcscWin_DataParams_t *)
                ((phhalHw_Pcsc_DataParams_t *)pHalDataParams)->pBalDataParams;
            break;
#endif /* NXPBUILD__PHHAL_HW_PCSC */

        default:
            pDataParams->pBalDataParams = NULL;
            pDataParams->pHalPcscDataParams = NULL;
            break;
    }

    /* Initialize private data */
    pDataParams->wId            = PH_COMP_PAL_ISO14443P3A | PHPAL_I14443P3A_PCSC_ID;
    pDataParams->pHalDataParams = pHalDataParams;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P3A);
}

phStatus_t phpalI14443p3a_Pcsc_RequestA(
                                        phpalI14443p3a_Pcsc_DataParams_t * pDataParams,
                                        uint8_t * pAtqa
                                        )
{
    /* Variables not used */
    PH_UNUSED_VARIABLE(pDataParams);
    PH_UNUSED_VARIABLE(pAtqa);

    /* Not available for PCSC Reader */
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P3A);
}

phStatus_t phpalI14443p3a_Pcsc_WakeUpA(
                                       phpalI14443p3a_Pcsc_DataParams_t * pDataParams,
                                       uint8_t * pAtqa
                                       )
{
    /* Variables not used */
    PH_UNUSED_VARIABLE(pDataParams);
    PH_UNUSED_VARIABLE(pAtqa);

    /* Not available for PCSC Reader */
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P3A);
}

phStatus_t phpalI14443p3a_Pcsc_HaltA(
                                     phpalI14443p3a_Pcsc_DataParams_t * pDataParams
                                     )
{
    /* Variables not used */
    PH_UNUSED_VARIABLE(pDataParams);

    /* Not available for PCSC Reader */
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P3A);
}

phStatus_t phpalI14443p3a_Pcsc_Anticollision(
                                             phpalI14443p3a_Pcsc_DataParams_t * pDataParams,
                                             uint8_t bCascadeLevel,
                                             uint8_t * pUidIn,
                                             uint8_t bNvbUidIn,
                                             uint8_t * pUidOut,
                                             uint8_t * pNvbUidOut
                                             )
{
    /* Variables not used */
    PH_UNUSED_VARIABLE(pDataParams);
    PH_UNUSED_VARIABLE(bCascadeLevel);
    PH_UNUSED_VARIABLE(pUidIn);
    PH_UNUSED_VARIABLE(bNvbUidIn);
    PH_UNUSED_VARIABLE(pUidOut);
    PH_UNUSED_VARIABLE(pNvbUidOut);

    /* Not available for PCSC Reader */
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P3A);
}

phStatus_t phpalI14443p3a_Pcsc_Select(
                                      phpalI14443p3a_Pcsc_DataParams_t * pDataParams,
                                      uint8_t bCascadeLevel,
                                      uint8_t * pUidIn,
                                      uint8_t * pSak
                                      )
{
    /* Variables not used */
    PH_UNUSED_VARIABLE(pDataParams);
    PH_UNUSED_VARIABLE(bCascadeLevel);
    PH_UNUSED_VARIABLE(pUidIn);
    PH_UNUSED_VARIABLE(pSak);

    /* Not available for PCSC Reader */
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P3A);
}

phStatus_t phpalI14443p3a_Pcsc_ActivateCard(
                                            phpalI14443p3a_Pcsc_DataParams_t * pDataParams,
                                            uint8_t * pUidIn,
                                            uint8_t bLenUidIn,
                                            uint8_t * pUidOut,
                                            uint8_t * pLenUidOut,
                                            uint8_t * pSak,
                                            uint8_t * pMoreCardsAvailable
                                            )
{
    phStatus_t  PH_MEMLOC_REM status;
    uint16_t    PH_MEMLOC_REM wSession = 0U;
    uint16_t    PH_MEMLOC_REM wCardState;

    /* Variables not used */
    PH_UNUSED_VARIABLE(pUidIn);
    PH_UNUSED_VARIABLE(bLenUidIn);
    PH_UNUSED_VARIABLE(pMoreCardsAvailable);

    /* Check if transparent session is supported */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_GetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_PCSC_CONFIG_EXCHANGE,
        &wSession));

    /* Activate till layer-3 if transparent session is supported */
    if (wSession == PHHAL_HW_PCSC_TRANSPARENT)
    {
        /* Get UID of card */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Pcsc_Cmd_GetData(
            pDataParams->pHalPcscDataParams,
            PHHAL_HW_PCSC_UID,
            pUidOut,
            (uint16_t *)pLenUidOut));

        /* Activate layer 3 command */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Pcsc_Cmd_SwitchProtocol(
            pDataParams->pHalPcscDataParams,
            PHHAL_HW_PCSC_ISO14443A,
            PHHAL_HW_PCSC_LAYER_3));

        /* Get SAK from HAL */
        *pSak = pDataParams->pHalPcscDataParams->bSak;
    }
    else
    {
        /* If bal is available use alternative methode */
        if (pDataParams->pBalDataParams != NULL)
        {
            /* If card is already open check if the card is still available.
            If so the card is still activated to the second activation will fail with HAL I/O timeout */
            if (pDataParams->pBalDataParams->hContext != NULL ||
                pDataParams->pBalDataParams->hCard != NULL)
            {
                /* Card Connection is open check status */
                PH_CHECK_SUCCESS_FCT(status, phbalReg_GetConfig(pDataParams->pBalDataParams, PHBAL_REG_PCSCWIN_CONFIG_CARD_STATE, &wCardState));

                /* card still active -> no reactivation posible */
                if (wCardState == 6 /*SCARD_SPECIFIC*/)
                {
                    return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
                }

                /* Card not active longer -> Close/Reset also reader (and wait field recovery time) */
                status = phhalHw_FieldReset(pDataParams->pHalDataParams);
            }

            /* Activate tries to open the PCSC reader.
            if this fails it is an HAL I/O Timeout
            except there is a BAL IO Timeout*/
            /* Open Port to start Connection */
            status = phbalReg_OpenPort(pDataParams->pBalDataParams);
            if (status == PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_BAL))
            {
                return status;
            }
            if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS)
            {
                return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
            }
        }
        else
        {
            /* Card already activated to highest supported layer. No control over activation. */
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P3A);
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P3A);
}

phStatus_t phpalI14443p3a_Pcsc_Exchange(
                                        phpalI14443p3a_Pcsc_DataParams_t * pDataParams,
                                        uint16_t wOption,
                                        uint8_t * pTxBuffer,
                                        uint16_t wTxLength,
                                        uint8_t ** ppRxBuffer,
                                        uint16_t * pRxLength
                                        )
{
    /* Variables not used */
    PH_UNUSED_VARIABLE(pDataParams);
    PH_UNUSED_VARIABLE(wOption);
    PH_UNUSED_VARIABLE(pTxBuffer);
    PH_UNUSED_VARIABLE(wTxLength);
    PH_UNUSED_VARIABLE(ppRxBuffer);
    PH_UNUSED_VARIABLE(pRxLength);

    /* Not available for PCSC Reader */
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P3A);
}

phStatus_t phpalI14443p3a_Pcsc_GetSerialNo(
                                           phpalI14443p3a_Pcsc_DataParams_t * pDataParams,
                                           uint8_t * pRxBuffer,
                                           uint8_t * pRxLength
                                           )
{
    phStatus_t  PH_MEMLOC_REM status;
    uint16_t    PH_MEMLOC_REM wSession = 0U;

    /* Check if transparent session is supported */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_GetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_PCSC_CONFIG_EXCHANGE,
        &wSession));

    /* GetSerialNo if transparent session is supported */
    if (wSession == PHHAL_HW_PCSC_TRANSPARENT)
    {
        /* Get UID of card */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_Pcsc_Cmd_GetData(
            pDataParams->pHalPcscDataParams,
            PHHAL_HW_PCSC_UID,
            pRxBuffer,
            (uint16_t *)pRxLength));

        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P3A);
    }
    else
    {
        /* Not available for PCSC Reader */
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P3A);
    }
}

#endif /* NXPBUILD__PHPAL_I14443P3A_PCSC */
