/*
 * Copyright 2017 - 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * PCSC ISO14443-4 layer of Reader Library.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phpalI14443p4.h>
#include <ph_RefDefs.h>

/* MS Windows PC/SC Subsystem */
#ifdef _WIN32
#pragma warning(push)                   /* PRQA S 3116 */
#pragma warning(disable:4201)           /* PRQA S 3116 */
#include <winscard.h>
#pragma warning(pop)                    /* PRQA S 3116 */
#pragma comment(lib, "Winscard.lib")    /* PRQA S 3116 */
#endif

#include <phhalHw_Pcsc_Cmd.h>

#ifdef NXPBUILD__PHPAL_I14443P4_PCSC

#include "phpalI14443p4_Pcsc.h"
#include "phpalI14443p4_Pcsc_Int.h"

phStatus_t phpalI14443p4_Pcsc_Init(
                                   phpalI14443p4_Pcsc_DataParams_t * pDataParams,
                                   uint16_t wSizeOfDataParams,
                                   void * pHalDataParams
                                   )
{
    if (sizeof(phpalI14443p4_Pcsc_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_ISO14443P4);
    }

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pHalDataParams);

    /* Check data parameters */
    if (PH_GET_COMPCODE(pHalDataParams) != PH_COMP_HAL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
    }

    /* Extract BAL */
    switch (PH_GET_COMPID(pHalDataParams))
    {
#ifdef NXPBUILD__PHHAL_HW_CALLBACK
        case PHHAL_HW_CALLBACK_ID:
            /* Check if required PCSC HAL is privided with pCbContext and BAL is provided with PCSC HAL */
            if (((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext == NULL)
            {
                pDataParams->pBalDataParams = NULL;
                pDataParams->pHalPcscDataParams = NULL;
                break;
            }

            if (PH_GET_COMPCODE(((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext) != PH_COMP_HAL)
            {
                pDataParams->pBalDataParams = NULL;
                pDataParams->pHalPcscDataParams = NULL;
                break;
            }

            if(PH_GET_COMPID(((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext) != PHHAL_HW_PCSC_ID)
            {
                pDataParams->pBalDataParams = NULL;
                pDataParams->pHalPcscDataParams = NULL;
                break;
            }

            /* Set internal pcsc hal dataparams */
            pDataParams->pHalPcscDataParams = (phhalHw_Pcsc_DataParams_t *)(((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext);

            /* Check if required BAL is provided with HAL */
            if (((phhalHw_Pcsc_DataParams_t *)(((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext))->pBalDataParams == NULL)
            {
                pDataParams->pBalDataParams = NULL;
                break;
            }
            pDataParams->pBalDataParams = (phbalReg_PcscWin_DataParams_t *)
                ((phhalHw_Pcsc_DataParams_t *)(((phhalHw_Callback_DataParams_t *)pHalDataParams)->pCbContext))->pBalDataParams;
            break;
#endif /* NXPBUILD__PHHAL_HW_CALLBACK */

#ifdef NXPBUILD__PHHAL_HW_PCSC
        case PHHAL_HW_PCSC_ID:
            /* Set internal pcsc hal dataparams */
            pDataParams->pHalPcscDataParams = (phhalHw_Pcsc_DataParams_t *)pHalDataParams;

            /* Check if required BAL is provided with HAL */
            if (((phhalHw_Pcsc_DataParams_t *)pHalDataParams)->pBalDataParams == NULL)
            {
                pDataParams->pBalDataParams = NULL;
                break;
            }
            pDataParams->pBalDataParams = (phbalReg_PcscWin_DataParams_t *)
                ((phhalHw_Pcsc_DataParams_t *)pHalDataParams)->pBalDataParams;
            break;
#endif /* NXPBUILD__PHHAL_HW_PCSC */

        default:
            pDataParams->pBalDataParams = NULL;
            pDataParams->pHalPcscDataParams = NULL;
            break;
    }

    /* Initialize private data */
    pDataParams->wId            = PH_COMP_PAL_ISO14443P4 | PHPAL_I14443P4_PCSC_ID;
    pDataParams->pHalDataParams = pHalDataParams;

    /* Reset protocol to defaults */
    return phpalI14443p4_Pcsc_ResetProtocol(pDataParams);
}

phStatus_t phpalI14443p4_Pcsc_SetProtocol(
                                          phpalI14443p4_Pcsc_DataParams_t * pDataParams,
                                          uint8_t   bCidEnable,
                                          uint8_t   bCid,
                                          uint8_t   bNadEnable,
                                          uint8_t   bNad,
                                          uint8_t   bFwi,
                                          uint8_t   bFsdi,
                                          uint8_t   bFsci
                                          )
{
    phStatus_t PH_MEMLOC_REM status;
    uint16_t   PH_MEMLOC_REM wSession = 0U;

    /* Check if transparent session is supported */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_GetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_PCSC_CONFIG_EXCHANGE,
        &wSession));

    if (wSession == PHHAL_HW_PCSC_TRANSPARENT)
    {
        pDataParams->bCidEnabled        = bCidEnable;
        pDataParams->bCid               = bCid;
        pDataParams->bNadEnabled        = bNadEnable;
        pDataParams->bNad               = bNad;
        pDataParams->bFwi               = bFwi;
        pDataParams->bFsdi              = bFsdi;
        pDataParams->bFsci              = bFsci;

        if (bCidEnable)
        {
            /* Send CID to reader */
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_PCSC_CONFIG_CID,
                bCid));
        }

        pDataParams->bCidEnabled = bCidEnable;

        if (bNadEnable)
        {
            /* Send NAD to reader */
            PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_PCSC_CONFIG_NAD,
                bNad));
        }

        pDataParams->bNadEnabled = bNadEnable;

        /* Send FWI to reader */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_PCSC_CONFIG_FWTI,
            bFwi));

        /* Send FSDI to reader */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_PCSC_CONFIG_FSDI,
            bFsdi));

        /* Send FSCI to reader */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_PCSC_CONFIG_FSCI,
            bFsci));
    }
    else
    {
        /* If bal is available use alternative methode */
        if (pDataParams->pBalDataParams != NULL)
        {
            /* Not available for PCSC Reader */
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
        }
        else
        {
            /* Card already activated to highest supported layer. No control over activation. */
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P4);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Pcsc_ResetProtocol(
                                            phpalI14443p4_Pcsc_DataParams_t * pDataParams
                                            )
{
    /* Set default state */
    pDataParams->bStateNow = PHPAL_I14443P4_PCSC_STATE_I_BLOCK_TX;

    /* Rule A, ISO/IEC 14443-4:2008(E), init. Blocknumber to zero */
    pDataParams->bPcbBlockNum = 0;

    /* Apply default parameters */
    pDataParams->bCidEnabled        = 0x00;
    pDataParams->bCid               = 0x00;
    pDataParams->bNadEnabled        = 0x00;
    pDataParams->bNad               = 0x00;
    pDataParams->bFwi               = PHPAL_I14443P4_PCSC_FWI_DEFAULT;
    pDataParams->bFsdi              = PHPAL_I14443P4_PCSC_FSDI_DEFAULT;
    pDataParams->bFsci              = PHPAL_I14443P4_PCSC_FSCI_DEFAULT;
    pDataParams->bMaxRetryCount     = PHPAL_I14443P4_PCSC_MAX_RETRIES_DEFAULT;
    pDataParams->bExchangeType      = PHPAL_I14443P4_PCSC_NORMAL;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Pcsc_Deselect(
                                       phpalI14443p4_Pcsc_DataParams_t * pDataParams
                                       )
{
    phStatus_t PH_MEMLOC_REM status;
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t    PH_MEMLOC_REM bIsoFrame[3];
    uint16_t   PH_MEMLOC_REM wIsoFrameLen = 0;
    uint8_t *  PH_MEMLOC_REM pResp = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;
    uint8_t    PH_MEMLOC_REM bInvalidBlock;
    uint8_t    PH_MEMLOC_REM bResponseReceived;
    uint16_t   PH_MEMLOC_REM wRetries;
    uint16_t   PH_MEMLOC_REM wSession = 0U;

    /* Check if transparent session is supported */
    PH_CHECK_SUCCESS_FCT(status, phhalHw_GetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_PCSC_CONFIG_EXCHANGE,
        &wSession));

    if (wSession == PHHAL_HW_PCSC_TRANSPARENT)
    {
        /* Build S(DESELECT) frame */
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI14443p4_Pcsc_Int_BuildSBlock(
            pDataParams->bCidEnabled,
            pDataParams->bCid,
            0,
            0,
            bIsoFrame,
            &wIsoFrameLen));

        /* Set DESELECT timeout */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,
            PHPAL_I14443P4_PCSC_FWT_DESELECT_US + PHPAL_I14443P4_PCSC_EXT_TIME_US));

        /* Retrieve max. retry count */
        wRetries = (uint16_t)pDataParams->bMaxRetryCount + 1U;

        /* Reset response received flag */
        bResponseReceived = 0;

        /* Do as long as invalid responses are received
         * and the retry counter has not reached zero. */
        do
        {
            /* Send the S(DESELECT) request */
            status = phhalHw_Pcsc_Cmd_Transceive(
                pDataParams->pHalPcscDataParams,
                PH_EXCHANGE_DEFAULT,
                bIsoFrame,
                wIsoFrameLen,
                &pResp,
                &wRespLen);

            /* Status --> InvalidBlock mapping */
            bInvalidBlock = (uint8_t)PHPAL_I14443P4_PCSC_IS_INVALID_BLOCK_STATUS(status);

            if (0U == (bInvalidBlock))
            {
                /* Check for other errors */
                PH_CHECK_SUCCESS(status);

                /* Signal that we've received something */
                bResponseReceived = 1;

                /* Rule 8, ISO/IEC 14443-4:2008(E), "...the S(DESELECT) request may be re-transmitted..." */
                if (wRespLen == wIsoFrameLen)
                {
                    /* Mask out Power-Level-Indication */
                    if (0U != (pDataParams->bCidEnabled))
                    {
                        pResp[1] &= 0x3FU;
                    }

                    /* Verify S(DESELECT) Response */
                    if (memcmp(bIsoFrame, pResp, wRespLen) != 0)
                    {
                        status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                    }
                }
                else
                {
                    status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
                }
            }
        }
        while ((0U != bInvalidBlock) && (--wRetries));

        /* Operation not successful */
        if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS)
        {
            /* Return ERR_RECOVERY_FAILED if some response has been received
             * before (bMaxRetryCount = 0 suppresses the retry behaviour). */
            if ((pDataParams->bMaxRetryCount > 0U) && (bResponseReceived))
            {
                status = PH_ADD_COMPCODE(PHPAL_I14443P4_ERR_RECOVERY_FAILED, PH_COMP_PAL_ISO14443P4);
            }
        }

        /* Apply default parameters */
        (void)phpalI14443p4_Pcsc_ResetProtocol(pDataParams);
    }
    else
    {
        /* If bal is available use alternative methode */
        if (pDataParams->pBalDataParams != NULL)
        {
            /* PCSC Field Off and Field On (Close & Open Port) */
            status = phhalHw_FieldOff(pDataParams->pHalDataParams);
            if(status != (PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4))
                return status;

            return phhalHw_FieldOn(pDataParams);
        }
        else
        {
            /* No control over activation. */
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P4);
        }
    }

    return status;
}

phStatus_t phpalI14443p4_Pcsc_PresCheck(
                                        phpalI14443p4_Pcsc_DataParams_t * pDataParams
                                        )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint8_t    PH_MEMLOC_REM bIsoFrame[3];
    uint8_t *  PH_MEMLOC_REM pIsoFrameResp = NULL;
    uint16_t   PH_MEMLOC_REM wIsoFrameLen = 0;
    uint16_t   PH_MEMLOC_REM wSession = 0U;
	uint16_t wCardState;

    /* Check if transparent session is supported */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_PCSC_CONFIG_EXCHANGE,
        &wSession));

    if (wSession == PHHAL_HW_PCSC_TRANSPARENT)
    {
        /* Build R(NAK) frame */
        PH_CHECK_SUCCESS_FCT(statusTmp, phpalI14443p4_Pcsc_Int_BuildRBlock(
            pDataParams->bCidEnabled,
            pDataParams->bCid,
            pDataParams->bPcbBlockNum,
            0,
            bIsoFrame,
            &wIsoFrameLen));

        /* Send frame */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Cmd_Transceive(
            pDataParams->pHalPcscDataParams,
            PH_EXCHANGE_DEFAULT,
            bIsoFrame,
            wIsoFrameLen,
            &pIsoFrameResp,
            &wIsoFrameLen));

        /* Check if we got a valid R(ACK) response */
        if (PHPAL_I14443P4_PCSC_IS_R_BLOCK(pIsoFrameResp[PHPAL_I14443P4_PCSC_PCB_POS]) &&
            PHPAL_I14443P4_PCSC_IS_ACK(pIsoFrameResp[PHPAL_I14443P4_PCSC_PCB_POS]))
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phpalI14443p4_Pcsc_Int_IsValidRBlock(
                pDataParams->bCidEnabled,
                pDataParams->bCid,
                pIsoFrameResp,
                wIsoFrameLen));
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4);
        }
    }
    else
    {
        /* If bal is available use alternative methode */
        if (pDataParams->pBalDataParams != NULL)
        {
            /* Card Connection is open check status */
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalDataParams, PHBAL_REG_PCSCWIN_CONFIG_CARD_STATE, &wCardState));

            switch(wCardState)
            {
            case SCARD_UNKNOWN:
            case SCARD_ABSENT:
            case SCARD_PRESENT:
            case SCARD_SWALLOWED:
            case SCARD_POWERED:
            default:
                return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
            case SCARD_NEGOTIABLE:
            case SCARD_SPECIFIC:
                return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
            }
        }
        else
        {
            /* No control over activation / detection. */
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P4);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Pcsc_Exchange(
                                       phpalI14443p4_Pcsc_DataParams_t * pDataParams,
                                       uint16_t wOption,
                                       uint8_t * pTxBuffer,
                                       uint16_t wTxLength,
                                       uint8_t ** ppRxBuffer,
                                       uint16_t * pRxLength
                                       )
{
    phStatus_t PH_MEMLOC_REM status;
	phStatus_t PH_MEMLOC_REM statustmp;
    uint16_t   PH_MEMLOC_REM wRxLength;
    uint8_t *  PH_MEMLOC_REM pRxBuffer;

    /* Check if caller has provided valid pointers */
    if (ppRxBuffer == NULL)
    {
        ppRxBuffer = &pRxBuffer;
    }
    if (pRxLength == NULL)
    {
        pRxLength = &wRxLength;
    }

    if (pDataParams->bExchangeType == PHPAL_I14443P4_PCSC_NORMAL)
    {
        /* Configure HAL exchange to use normal exchange (not transparent) */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_PCSC_CONFIG_EXCHANGE,
            PHHAL_HW_PCSC_NORMAL));

        /* Perform Exchange (chaining / error handling done by PCSC reader) */
        status = phhalHw_Exchange(
            pDataParams->pHalDataParams,
            wOption,
            pTxBuffer,
            wTxLength,
            ppRxBuffer,
            pRxLength);

		/* Configure Rx Buffer Start position to zero. */
		PH_CHECK_SUCCESS_FCT(statustmp, phhalHw_SetConfig(
			pDataParams->pHalDataParams,
			PHHAL_HW_CONFIG_RXBUFFER_STARTPOS,
			0U));
    }
    else
    {
        /* Configure HAL exchange to use transparent exchange. */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_PCSC_CONFIG_EXCHANGE,
            PHHAL_HW_PCSC_TRANSPARENT));

        /* Do transparent exchange (chaining / error handling done here in PAL) */
        status = phpalI14443p4_Pcsc_Int_TransparentExchange(
            pDataParams,
            wOption,
            pTxBuffer,
            wTxLength,
            ppRxBuffer,
            pRxLength);
    }

    return status;
}

phStatus_t phpalI14443p4_Pcsc_SetConfig(
                                        phpalI14443p4_Pcsc_DataParams_t * pDataParams,
                                        uint16_t wConfig,
                                        uint16_t wValue
                                        )
{
    phStatus_t PH_MEMLOC_REM status;

    switch (wConfig)
    {
    case PHPAL_I14443P4_CONFIG_BLOCKNO:
        if (wValue == 0)
        {
            pDataParams->bPcbBlockNum = 0;
        }
        else
        {
            pDataParams->bPcbBlockNum = 1;
        }
        break;

    case PHPAL_I14443P4_CONFIG_CID:
        if (wValue & 0xFF00)
        {
            if ((wValue & 0x00FF) > PHPAL_I14443P4_CID_MAX)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
            }
            pDataParams->bCidEnabled = 1;
            pDataParams->bCid = (uint8_t)(wValue);
        }
        else
        {
            pDataParams->bCidEnabled = 0;
            pDataParams->bCid = 0;
        }

        /* Send CID to reader */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_PCSC_CONFIG_CID,
            pDataParams->bCid));
        break;

    case PHPAL_I14443P4_CONFIG_NAD:
        if (wValue & 0xFF00)
        {
            pDataParams->bNadEnabled = 1;
            pDataParams->bNad = (uint8_t)(wValue);
        }
        else
        {
            pDataParams->bNadEnabled = 0;
            pDataParams->bNad = 0;
        }

        /* Send NAD to reader */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_PCSC_CONFIG_NAD,
            pDataParams->bNad));
        break;

    case PHPAL_I14443P4_CONFIG_FWI:
        if (wValue > PHPAL_I14443P4_FWI_MAX)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }

        pDataParams->bFwi = (uint8_t)(wValue);

        /* Send FWI to reader */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_PCSC_CONFIG_FWTI,
            pDataParams->bFwi));
        break;

    case PHPAL_I14443P4_CONFIG_FSI:
        if (((wValue >> 8) > PHPAL_I14443P4_FRAMESIZE_MAX) ||
            ((wValue & 0xFF) > PHPAL_I14443P4_FRAMESIZE_MAX))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }

        pDataParams->bFsdi = (uint8_t)(wValue >> 8);
        pDataParams->bFsci = (uint8_t)(wValue);

        /* Send FSDI to reader */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_PCSC_CONFIG_FSDI,
            pDataParams->bFsdi));

        /* Send FSCI to reader */
        PH_CHECK_SUCCESS_FCT(status, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_PCSC_CONFIG_FSCI,
            pDataParams->bFsci));
        break;

    case PHPAL_I14443P4_CONFIG_MAXRETRYCOUNT:
        pDataParams->bMaxRetryCount = (uint8_t)(wValue);
        break;

    case PHPAL_I14443P4_PCSC_CONFIG_EXCHANGE:
        /* Parameter Check */
        if ((wValue != PHPAL_I14443P4_PCSC_NORMAL) && (wValue != PHPAL_I14443P4_PCSC_TRANSPARENT))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
        }

        pDataParams->bExchangeType = (uint8_t)wValue;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_ISO14443P4);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Pcsc_GetConfig(
                                        phpalI14443p4_Pcsc_DataParams_t * pDataParams,
                                        uint16_t wConfig,
                                        uint16_t * pValue
                                        )
{
   switch (wConfig)
    {
    case PHPAL_I14443P4_CONFIG_BLOCKNO:
        if (pDataParams->bPcbBlockNum == 0)
        {
            *pValue = 0;
        }
        else
        {
            *pValue = 1;
        }
        break;

    case PHPAL_I14443P4_CONFIG_CID:
        if (pDataParams->bCidEnabled)
        {
            *pValue = (uint16_t)(0x0100 | pDataParams->bCid);
        }
        else
        {
            *pValue = (uint16_t)(pDataParams->bCid);
        }
        break;

    case PHPAL_I14443P4_CONFIG_NAD:
        if (pDataParams->bNadEnabled)
        {
            *pValue = (uint16_t)(0x0100 | pDataParams->bNad);
        }
        else
        {
            *pValue = (uint16_t)(pDataParams->bNad);
        }
        break;

    case PHPAL_I14443P4_CONFIG_FWI:
        *pValue = (uint16_t)pDataParams->bFwi;
        break;

    case PHPAL_I14443P4_CONFIG_FSI:
        *pValue  = (uint16_t)((uint16_t)(pDataParams->bFsdi) << 8);
        *pValue |= (uint16_t)(pDataParams->bFsci);
        break;

    case PHPAL_I14443P4_CONFIG_MAXRETRYCOUNT:
        *pValue = (uint16_t)pDataParams->bMaxRetryCount;
        break;

    case PHPAL_I14443P4_PCSC_CONFIG_EXCHANGE:
        *pValue = (uint16_t)pDataParams->bExchangeType;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_ISO14443P4);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

#endif /* NXPBUILD__PHPAL_I14443P4_PCSC */
