/*
 * Copyright 2013, 2015, 2017, 2020, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software ISO15693 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>

#ifdef NXPBUILD__PHPAL_SLI15693_SW

#include <phhalHw.h>
#include <phpalSli15693.h>
#include <ph_RefDefs.h>

#include "phpalSli15693_Sw.h"
#include "phpalSli15693_Sw_Int.h"

phStatus_t phpalSli15693_Sw_Init(phpalSli15693_Sw_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, void * pHalDataParams)
{
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_SLI15693);
    PH_ASSERT_NULL_PARAM(pHalDataParams, PH_COMP_PAL_SLI15693);

    /* Parameter structure length check */
    if(sizeof(phpalSli15693_Sw_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_SLI15693);
    }

    /* Initialize the structure members. */
    pDataParams->wId                    = PH_COMP_PAL_SLI15693 | PHPAL_SLI15693_SW_ID;
    pDataParams->pHalDataParams         = pHalDataParams;
    pDataParams->wAdditionalInfo        = 0;
    pDataParams->bFlags                 = PHPAL_SLI15693_SW_FLAGS_DEFAULT;
    pDataParams->bUidBitLen             = 0;
    pDataParams->bExplicitlyAddressed   = 0;
    pDataParams->bOpeMode               = RD_LIB_MODE_ISO; /* Default Mode is ISO */
    pDataParams->bBuffering             = 0;
    pDataParams->bMaxRetryCount         = PHPAL_SLI15693_SW_RETRYCOUNT_DEFAULT;
    pDataParams->bIcMfgCode             = PHPAL_SLI15693_SW_NXP_MFG_CODE;
    pDataParams->bResFlags              = 0;


    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_SLI15693);
}





/* ISO15693 Mandatory commands ------------------------------------------------------------------------------------- */
phStatus_t phpalSli15693_Sw_Inventory(phpalSli15693_Sw_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t * pDsfid, uint8_t * pUid)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bUidLen = 0;
    uint16_t    PH_MEMLOC_REM wDataLen = 0;

    /* Perform Inventory command. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Sw_Int_Inventory(
        pDataParams,
        PH_OFF,
        PHPAL_SLI15693_SW_CMD_INVENTORY,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        PH_OFF,
        NULL,
        0,
        0,
        NULL,
        pUid,
        &bUidLen,
        pDsfid,
        &wDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_SLI15693);
}

phStatus_t phpalSli15693_Sw_StayQuiet(phpalSli15693_Sw_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Clear selected flag */
    pDataParams->bFlags &= (uint8_t) ~(uint8_t) PHPAL_SLI15693_FLAG_SELECTED;

    /* Set addressed flag */
    pDataParams->bFlags |= PHPAL_SLI15693_FLAG_ADDRESSED;

    /* Reset to ready command */
    aCmdBuff[0] = PHPAL_SLI15693_SW_CMD_STAY_QUIET;

    /* Set short timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Exchange command. */
    wStatus = phpalSli15693_Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1,
        &pResponse,
        &wRespLen);

    /* No response to the command allowed */
    if((wStatus & PH_ERR_MASK) != PH_ERR_IO_TIMEOUT)
    {
        PH_CHECK_SUCCESS(wStatus);
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_SLI15693);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_SLI15693);
}





/* ISO15693 Optional commands -------------------------------------------------------------------------------------- */
phStatus_t phpalSli15693_Sw_Select(phpalSli15693_Sw_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* clear selected flag */
    pDataParams->bFlags &= (uint8_t) ~(uint8_t) PHPAL_SLI15693_FLAG_SELECTED;

    /* set addressed flag */
    pDataParams->bFlags |= PHPAL_SLI15693_FLAG_ADDRESSED;

    /* reset to ready command */
    aCmdBuff[0] = PHPAL_SLI15693_SW_CMD_SELECT;

    /* Set short timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Exchange command */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1,
        &pResponse,
        &wRespLen));

    /* We shall not receive any data */
    if(wRespLen != 0U)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_SLI15693);
    }

    /* clear addressed flag */
    pDataParams->bFlags &= (uint8_t) ~(uint8_t) PHPAL_SLI15693_FLAG_ADDRESSED;

    /* set selected flag */
    pDataParams->bFlags |= PHPAL_SLI15693_FLAG_SELECTED;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_SLI15693);
}

phStatus_t phpalSli15693_Sw_ResetToReady(phpalSli15693_Sw_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* reset to ready command */
    aCmdBuff[0] = PHPAL_SLI15693_SW_CMD_RESET_TO_READY;

    /* Set short timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Exchange command */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1,
        &pResponse,
        &wRespLen));

    /* We shall not receive any data */
    if(wRespLen != 0U)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_SLI15693);
    }

    /* clear selected flag */
    pDataParams->bFlags &= (uint8_t) ~(uint8_t) PHPAL_SLI15693_FLAG_SELECTED;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_SLI15693);
}





/* ISO15693 Custom commands ---------------------------------------------------------------------------------------- */
phStatus_t phpalSli15693_Sw_InventoryRead(phpalSli15693_Sw_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bBlockNo, uint16_t wNoOfBlocks, uint8_t * pUid, uint8_t * bUidLen, uint8_t * pData, uint16_t * wLength)
{
    return phpalSli15693_Sw_Int_Inventory(
        pDataParams,
        PH_OFF,
        PHPAL_SLI15693_SW_CMD_INVENTORY_READ,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        PH_OFF,
        NULL,
        bBlockNo,
        wNoOfBlocks,
        NULL,
        pUid,
        bUidLen,
        pData,
        wLength);
}

phStatus_t phpalSli15693_Sw_InventoryReadExtended(phpalSli15693_Sw_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bExtendedOptions, uint8_t * pCID, uint8_t bBlockNo, uint16_t wNumOfBlocks, uint8_t * pCDIDOut, uint8_t * pUid,
    uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    return phpalSli15693_Sw_Int_Inventory(
        pDataParams,
        PH_ON,
        PHPAL_SLI15693_SW_CMD_INVENTORY_READ,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        bExtendedOptions,
        pCID,
        bBlockNo,
        wNumOfBlocks,
        pCDIDOut,
        pUid,
        pUidLen,
        pData,
        pDataLen);
}

phStatus_t phpalSli15693_Sw_FastInventoryRead(phpalSli15693_Sw_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bBlockNo, uint16_t wNoOfBlocks, uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    return phpalSli15693_Sw_Int_Inventory(
        pDataParams,
        PH_OFF,
        PHPAL_SLI15693_SW_CMD_FAST_INVENTORY_READ,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        PH_OFF,
        NULL,
        bBlockNo,
        wNoOfBlocks,
        NULL,
        pUid,
        pUidLen,
        pData,
        pDataLen);
}

phStatus_t phpalSli15693_Sw_FastInventoryReadExtended(phpalSli15693_Sw_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bExtendedOptions, uint8_t * pCID, uint8_t bBlockNo, uint16_t wNumOfBlocks, uint8_t * pCDIDOut, uint8_t * pUid,
    uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    return phpalSli15693_Sw_Int_Inventory(
        pDataParams,
        PH_ON,
        PHPAL_SLI15693_SW_CMD_FAST_INVENTORY_READ,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        bExtendedOptions,
        pCID,
        bBlockNo,
        wNumOfBlocks,
        pCDIDOut,
        pUid,
        pUidLen,
        pData,
        pDataLen);
}

phStatus_t phpalSli15693_Sw_InventoryPageRead(phpalSli15693_Sw_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bPageNo, uint16_t wNoOfPages, uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    return phpalSli15693_Sw_Int_Inventory(
        pDataParams,
        PH_OFF,
        PHPAL_SLI15693_SW_CMD_INVENTORY_PAGE_READ,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        PH_OFF,
        NULL,
        bPageNo,
        wNoOfPages,
        NULL,
        pUid,
        pUidLen,
        pData,
        pDataLen);
}

phStatus_t phpalSli15693_Sw_FastInventoryPageRead(phpalSli15693_Sw_DataParams_t * pDataParams, uint8_t bFlags, uint8_t bAfi, uint8_t * pMask,
    uint8_t bMaskBitLen, uint8_t bPageNo, uint16_t wNoOfPages, uint8_t * pUid, uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    return phpalSli15693_Sw_Int_Inventory(
        pDataParams,
        PH_OFF,
        PHPAL_SLI15693_SW_CMD_FAST_INVENTORY_PAGE_READ,
        bFlags,
        bAfi,
        pMask,
        bMaskBitLen,
        PH_OFF,
        NULL,
        bPageNo,
        wNoOfPages,
        NULL,
        pUid,
        pUidLen,
        pData,
        pDataLen);
}

phStatus_t phpalSli15693_Sw_StayQuietPersistent(phpalSli15693_Sw_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Clear selected flag */
    pDataParams->bFlags &= (uint8_t) ~(uint8_t) PHPAL_SLI15693_FLAG_SELECTED;

    /* Set addressed flag */
    pDataParams->bFlags |= PHPAL_SLI15693_FLAG_ADDRESSED;

    /* Stay Quiet Persistent command */
    aCmdBuff[0] = PHPAL_SLI15693_SW_CMD_STAY_QUIET_PERS;

    /* Set short timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Exchange command */
    wStatus = phpalSli15693_Exchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1,
        &pResponse,
        &wRespLen);

    /* No response to the command allowed */
    if((wStatus & PH_ERR_MASK) != PH_ERR_IO_TIMEOUT)
    {
        PH_CHECK_SUCCESS(wStatus);
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_SLI15693);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_SLI15693);
}





/* ISO15693 Common commands ---------------------------------------------------------------------------------------- */
phStatus_t phpalSli15693_Sw_ActivateCard(phpalSli15693_Sw_DataParams_t * pDataParams, uint8_t  bOption, uint8_t bFlags, uint8_t bAfi,
    uint8_t * pMask, uint8_t bMaskBitLen, uint8_t * pDsfid, uint8_t * pUid, uint8_t * pMoreCardsAvailable)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bSlot = 0;
    uint8_t     PH_MEMLOC_REM bTimeOutNotAllowed = 0;
    uint8_t     PH_MEMLOC_REM aMaskBuffer[PHPAL_SLI15693_UID_LENGTH];
    uint8_t     PH_MEMLOC_REM bMaskByteLen = 0;
    uint8_t     PH_MEMLOC_REM bUidLenDummy = 0;
    uint8_t     PH_MEMLOC_REM bDataDummy[1];
    uint16_t    PH_MEMLOC_REM wDataLenDummy = 0;

    /* Check option byte */
    if((bOption != PHPAL_SLI15693_ACTIVATE_ADDRESSED) && (bOption != PHPAL_SLI15693_ACTIVATE_SELECTED))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_SLI15693);
    }

    /* Init return value */
    *pMoreCardsAvailable = 0;

    /* Perform inventory command */
    wStatus = phpalSli15693_Sw_Inventory(pDataParams, bFlags, bAfi, pMask, bMaskBitLen, pDsfid, pUid);

    /* Check for invalid parameter */
    if((wStatus & PH_ERR_MASK) == PH_ERR_INVALID_PARAMETER)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_SLI15693);
    }

    /* Retrieve mask byte length */
    bMaskByteLen = ((bMaskBitLen % 8U) != 0U) ? ((bMaskBitLen >> 3U) + 1U) : (bMaskBitLen >> 3U);

    /* Init mask buffer */
    (void) memset(aMaskBuffer, 0, sizeof(aMaskBuffer));
    (void) memcpy(aMaskBuffer, pMask, bMaskByteLen);

    /* If we have more than one slot, we need to send slot markers (b6 = 0) */
    if((0U == ((bFlags & PHPAL_SLI15693_FLAG_NBSLOTS))))
    {
        /* Slot marker loop */
        bSlot = 0;
        while((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
        {
            switch(wStatus & PH_ERR_MASK)
            {
                /*  If a CRC error or collision error occurred --> resolve collision */
                case PH_ERR_COLLISION_ERROR:
                case PH_ERR_INTEGRITY_ERROR:

                    *pMoreCardsAvailable = 1;

                    /* Get bit-length of last valid byte */
                    bUidLenDummy = (uint8_t) (bMaskBitLen % 8U);

                    /* Whole byte is valid -> append slot number to next byte */
                    if(bUidLenDummy == 0U)
                    {
                        aMaskBuffer[bMaskBitLen >> 3U] = bSlot;
                    }

                    /* Partial byte is valid */
                    else
                    {
                        /* Fill the invalid bits of the incomplete byte with the 4 bits slot number */
                        aMaskBuffer[bMaskBitLen >> 3U] &= (uint8_t) (0xFFU >> (8U - bUidLenDummy));
                        aMaskBuffer[bMaskBitLen >> 3U] |= (uint8_t) (bSlot << bUidLenDummy);

                        /* If not all 4 bits of the Slot number fit in the incomplete byte, put the rest in the next byte */
                        if(bUidLenDummy > 4)
                        {
                            aMaskBuffer[(bMaskBitLen >> 3U) + 1U] = (uint8_t) (bSlot >> bUidLenDummy);
                        }
                    }

                    /* Increment the mask bit length by the 4 bits slot number */
                    bMaskBitLen = bMaskBitLen + 4U;

                    /* Start again from Slot number 0 */
                    bSlot = 0;
                    wStatus = phpalSli15693_Sw_Inventory(pDataParams, bFlags, bAfi, aMaskBuffer, bMaskBitLen, pDsfid, pUid);
                    break;

                /* Timeout -> advance to next slot */
                case PH_ERR_IO_TIMEOUT:

                    /* Send EOF and increment slot number */
                    ++bSlot;
                    wStatus = phpalSli15693_Sw_SendEof(
                        pDataParams,
                        PHPAL_SLI15693_EOF_NEXT_SLOT,
                        pDsfid,
                        pUid,
                        &bUidLenDummy,
                        bDataDummy,
                        &wDataLenDummy);
                    break;

                /* Other error -> bail out */
                default:
                    return wStatus;
            }

            /* If we ran through all slots, we are done */
            if(bSlot == 15U)
            {
                break;
            }
        }
    }

    /* Use only one slot */
    else
    {
        /* Do not allow a time out after the first Inventory command since then, no
        card with the specified mask is present in the field. */
        bTimeOutNotAllowed = 1;

        /* Bit mask loop */
        while(bMaskBitLen < PHPAL_SLI15693_SW_UID_COMPLETE)
        {
            switch(wStatus & PH_ERR_MASK)
            {
                /* If there was a card, the loop is done */
                case PH_ERR_SUCCESS:
                    bMaskBitLen = PHPAL_SLI15693_SW_UID_COMPLETE;
                    break;

                /* In case of a timeout, no card with the actual mask is in the field, so toggle the last bit of the mask */
                case PH_ERR_IO_TIMEOUT:
                    /* Abort, if this bit was already tested */
                    if(0U != bTimeOutNotAllowed)
                    {
                        return wStatus;
                    }

                    /* The bit mask is > 0, because in the case of BitMask = 0 a timeout can not happen */
                    aMaskBuffer[(uint8_t) (bMaskBitLen - 1) >> 3] = aMaskBuffer[(uint8_t) (bMaskBitLen - 1) >> 3] ^ (uint8_t) (0x01 << (uint8_t) ((bMaskBitLen - 1) % 8));
                    bTimeOutNotAllowed = 1;
                    break;

                /*  If a CRC error or collision error occurred add one bit to the mask. */
                case PH_ERR_COLLISION_ERROR:
                case PH_ERR_INTEGRITY_ERROR:
                    /* Note: The value of this bit does not really matter since it will be toggled in case of a timeout */
                    *pMoreCardsAvailable = 1;

                    ++bMaskBitLen;
                    bTimeOutNotAllowed = 0;
                    break;

                /* If it is an other error than Collision, CRC or Timeout Error the function should return the error */
                default:
                    return wStatus;
            }

            /* Perform next inventory command to perform Anti-Collision */
            if(bMaskBitLen < PHPAL_SLI15693_SW_UID_COMPLETE)
            {
                wStatus = phpalSli15693_Sw_Inventory(pDataParams, bFlags, bAfi, aMaskBuffer, bMaskBitLen, pDsfid, pUid);
            }
        }
    }

    /* No need to proceed if an error occurred. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        /* Clear option flag */
        pDataParams->bFlags &= (uint8_t) ~(uint8_t) PHPAL_SLI15693_FLAG_OPTION;
        return wStatus;
    }

    /* Select the card if necessary */
    if(0U != (bOption & PHPAL_SLI15693_ACTIVATE_SELECTED))
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Select(pDataParams));
    }

    /* Set long timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pDataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_SLI15693);
}

phStatus_t phpalSli15693_Sw_SendEof(phpalSli15693_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pDsfid, uint8_t * pUid,
    uint8_t * pUidLen, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM bStoredUidLen = 0;
    uint16_t    PH_MEMLOC_REM wAsk = 0;
    uint16_t    PH_MEMLOC_REM wTimeout = 0;

    /* Check bOption. */
    switch(bOption)
    {
        case PHPAL_SLI15693_EOF_NEXT_SLOT:
        case PHPAL_SLI15693_EOF_NEXT_SLOT_INV_READ:
        case PHPAL_SLI15693_EOF_WRITE_ALIKE:
        case PHPAL_SLI15693_EOF_WRITE_ALIKE_WITH_WAIT:
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_SLI15693);
    }

    /* Reset UID and data lengths */
    *pDsfid = 0;
    *pUidLen = 0;
    *pDataLen = 0;

    /* If requested, wait ~20ms upon sending EOF. */
    if(bOption == PHPAL_SLI15693_EOF_WRITE_ALIKE_WITH_WAIT)
    {
        /* in case of write alike commands wait 20 ms. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Wait(
            pDataParams->pHalDataParams,
            PHHAL_HW_TIME_MICROSECONDS,
            PHPAL_SLI15693_TIMEOUT_LONG_US));
    }

    /* Get the ASK 100 Condition */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_CONFIG_ASK100,
        &wAsk));

    if(bOption == PHPAL_SLI15693_EOF_NEXT_SLOT)
    {
        if(0U != (wAsk))
        {
            if(0U != (pDataParams->bFlags & PHPAL_SLI15693_FLAG_DATA_RATE))
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_HIGH_SOF_US;
            }
            else
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_LOW_SOF_US;
            }
        }
        else
        {
            if(0U != (pDataParams->bFlags & PHPAL_SLI15693_FLAG_DATA_RATE))
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_HIGH_NRT_US;
            }
            else
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_LOW_NRT_US;
            }
        }

        /* Set  timeout. */
        PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
            pDataParams,
            PHPAL_SLI15693_CONFIG_TIMEOUT_US,
            wTimeout));
    }
    else if(bOption == PHPAL_SLI15693_EOF_NEXT_SLOT_INV_READ)
    {
        if(0U != wAsk)
        {
            if(0U != (pDataParams->bFlags & PHPAL_SLI15693_FLAG_DATA_RATE))
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_FASTHIGH_SOF_US;
            }
            else
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_FASTLOW_SOF_US;
            }
        }
        else
        {
            if(0U != (pDataParams->bFlags & PHPAL_SLI15693_FLAG_DATA_RATE))
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_FASTHIGH_NRT_US;
            }
            else
            {
                wTimeout = PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_SW_FASTLOW_NRT_US;
            }
        }

        /* Set  timeout. */
        PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
            pDataParams,
            PHPAL_SLI15693_CONFIG_TIMEOUT_US,
            wTimeout));
    }
    else
    {
        wTimeout = (uint16_t) ((PHPAL_SLI15693_TIMEOUT_MAX2_US + PHPAL_SLI15693_TIMEOUT_DELTA_US) / 1000);
        wTimeout += (uint16_t) ((((PHPAL_SLI15693_TIMEOUT_MAX2_US + PHPAL_SLI15693_TIMEOUT_DELTA_US) % 1000) > 500) ? 1 : 0);

        /* Set FDT time max2(19.95ms) + Tolerance(Delta) Timeout(50ms). */
        PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
            pDataParams,
            PHPAL_SLI15693_CONFIG_TIMEOUT_MS,
            wTimeout));
    }

    /* Disable SOF, so only EOF is sent */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_CONFIG_SYMBOL_START,
        PH_OFF));

    /* Exchange command. */
    wStatus = phhalHw_Exchange(
        pDataParams->pHalDataParams,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        &pResponse,
        &wRespLen);

    /* Reset HAL to send SOF and EOF. */
    PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_CONFIG_SYMBOL_START,
        PHHAL_HW_SYMBOL_I15693_SOF));

    /* Check Success */
    PH_CHECK_SUCCESS(wStatus);

    /* Check RespLength */
    if(wRespLen == 0U)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_SLI15693);
    }

    /* Check error flag */
    if(pResponse[0] & PHPAL_SLI15693_FLAG_RESP_ERROR)
    {
        /* Check the length in case of error */
        if(wRespLen != 2U)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_SLI15693);
        }

        pDataParams->wAdditionalInfo = pResponse[1];
        return PH_ADD_COMPCODE(PHPAL_SLI15693_ERR_ISO15693, PH_COMP_PAL_SLI15693);
    }

    if(bOption == PHPAL_SLI15693_EOF_NEXT_SLOT)
    {
        /* Ignore flag byte */
        ++pResponse;
        --wRespLen;

        /* The UID is only returned on a next slot command. */
        if(wRespLen != (1U + PHPAL_SLI15693_UID_LENGTH))
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_SLI15693);
        }

        /* Wait T2 */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Wait(
            pDataParams->pHalDataParams,
            PHHAL_HW_TIME_MICROSECONDS,
            PHPAL_SLI15693_SW_T2_WAITING_TIME));

        /* Copy the DSFID to the data buffer */
        *pDsfid = pResponse[0];

        /* Retrieve the UID */
        (void) memcpy(pDataParams->pUid, &pResponse[1], PHPAL_SLI15693_UID_LENGTH);
        pDataParams->bUidBitLen = PHPAL_SLI15693_SW_UID_COMPLETE;

        /* Return the UID */
        (void) memcpy(pUid, &pResponse[1], PHPAL_SLI15693_UID_LENGTH);
        *pUidLen = PHPAL_SLI15693_UID_LENGTH;

        /* Set addressed flag */
        pDataParams->bFlags |= PHPAL_SLI15693_FLAG_ADDRESSED;
    }
    else if(bOption == PHPAL_SLI15693_EOF_NEXT_SLOT_INV_READ)
    {
        /* Ignore flag byte */
        ++pResponse;
        --wRespLen;

        /* Option flag is set -> we also received (partial) UID */
        if(pDataParams->bFlags & PHPAL_SLI15693_FLAG_OPTION)
        {
            /* The response length should not be less than the remaining UID */
            bStoredUidLen = pDataParams->bUidBitLen >> 3U;
            *pUidLen = PHPAL_SLI15693_UID_LENGTH - bStoredUidLen;
            if(wRespLen < *pUidLen)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_SLI15693);
            }

            if(bStoredUidLen < PHPAL_SLI15693_UID_LENGTH)
            {
                /* We need to merge the contents of the mask buffer and the received data */
                if(pDataParams->bUidBitLen % 8U)
                {
                    if(bStoredUidLen < 7U)
                    {
                        /* Copy the UID bytes we received from the card */
                        (void) memcpy(&(pDataParams->pUid[bStoredUidLen + 1]), &pResponse[1], (*pUidLen) - 1);
                    }

                    /* Merge mask-bits with received bits */
                    pDataParams->pUid[bStoredUidLen] |= pResponse[0];
                }
                else
                {
                    /* Copy the UID bytes we received from the card */
                    (void) memcpy(&(pDataParams->pUid[bStoredUidLen]), pResponse, *pUidLen);
                }

                /* Return the received (partial) UID */
                (void) memcpy(pUid, pResponse, *pUidLen);
            }

            /* UID is now complete */
            pDataParams->bUidBitLen = PHPAL_SLI15693_SW_UID_COMPLETE;

            /* Shift pointer and length */
            pResponse += *pUidLen;
            wRespLen = wRespLen - *pUidLen;
        }

        /* Copy the received data to the provided buffer */
        (void) memcpy(pData, pResponse, wRespLen);

        /* The remaining bytes of the response are the data bytes */
        *pDataLen = wRespLen;

        /* set addressed flag */
        pDataParams->bFlags |= PHPAL_SLI15693_FLAG_ADDRESSED;

        /* Wait T2 */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Wait(
            pDataParams->pHalDataParams,
            PHHAL_HW_TIME_MICROSECONDS,
            PHPAL_SLI15693_SW_T2_WAITING_TIME));
    }
    else
    {
        /* 1 byte is only returned on a EOF for a write alike command. */
        if(wRespLen != 1U)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_SLI15693);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_SLI15693);
}

phStatus_t phpalSli15693_Sw_Exchange(phpalSli15693_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pTxBuffer, uint16_t wTxLen,
    uint8_t ** ppRxBuffer, uint16_t * pRxLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bFirstBlock[3];
    uint16_t    PH_MEMLOC_REM wTransmitLen = 0;
    uint16_t    PH_MEMLOC_REM RxLength = 0;
    uint8_t     PH_MEMLOC_REM bIsInProcessReply = PH_OFF;
    uint8_t *   PH_MEMLOC_REM pRxBuffer = NULL;

    /* Check if InProcess Reply flag is set */
    bIsInProcessReply = (uint8_t) ((wOption & PH_EXCHANGE_CUSTOM_BITS_MASK) == PHPAL_SLI15693_SW_IN_PROCESS_REPLY);

    /* Remove Custom flag configuration from wOptions */
    wOption = (uint16_t) (wOption & ~PH_EXCHANGE_CUSTOM_BITS_MASK);

    /* The frame to exchange looks like the following:
    {optional} [one byte]
    [flags][CMD]{[MFC]}{[UID0] .. [UID3]}[CMDParam(0)] .. [CMDParam(N)] */

    /* Check for valid UID in addressed mode */
    if ((0U == ((pDataParams->bFlags & PHPAL_SLI15693_FLAG_INVENTORY))) && (0U != ((pDataParams->bFlags & PHPAL_SLI15693_FLAG_ADDRESSED))))
    {
        /* Return an error if UID is not valid */
        if(pDataParams->bUidBitLen != PHPAL_SLI15693_SW_UID_COMPLETE)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_PAL_SLI15693);
        }
    }

    /* Check if caller has provided valid RxBuffer */
    if(ppRxBuffer == NULL)
    {
        ppRxBuffer = &pRxBuffer;
    }
    if(pRxLen == NULL)
    {
        pRxLen = &RxLength;
    }

    if((0U == ((wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))) && (pTxBuffer != NULL))
    {
        /* This is the first chunk of a command frame. Add the Flags byte */
        bFirstBlock[wTransmitLen++] = pDataParams->bFlags;

        /* Add Command byte */
        bFirstBlock[wTransmitLen++] = pTxBuffer[0];

        /* Add MFC byte for custom commands */
        if((pTxBuffer[0] > PHPAL_SLI15693_SW_CUSTOM_CMD_CODE_BEGIN) && (pTxBuffer[0] < PHPAL_SLI15693_SW_CUSTOM_CMD_CODE_END))
        {
            /* if no serial number is valid we assume that we use only NXP tags. */
            bFirstBlock[wTransmitLen++] = pDataParams->bIcMfgCode;
        }
        else
        {
            if((pTxBuffer[0] == PHPAL_SLI15693_SW_CMD_EXTENDED_GET_SYSTEM_INFORMATION) && (wTxLen > 1U))
            {
                /* Handle the Frame format preparation for Extended Get System Information specifically since this takes 1
                 *  UID from 3rd Byte whereas rest other commands take UID from 2nd Byte
                 *  SOF + FLAGS + COMMAND CODE +  INFO FLAGS  + UID  + CRC + EOF.
                 */

                /* Copy Info Flags */
                bFirstBlock[wTransmitLen++] = pTxBuffer[1];

                /* First byte was the command byte */
                ++pTxBuffer;
                --wTxLen;
            }
        }

        /* Buffer the initial information to be exchanged */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Exchange(
            pDataParams->pHalDataParams,
            PH_EXCHANGE_BUFFERED_BIT,
            bFirstBlock,
            wTransmitLen,
            ppRxBuffer,
            pRxLen));

        /* First byte was the command byte */
        ++pTxBuffer;
        --wTxLen;

        /* If we are not waiting any more we can transmit the UID bytes, if necessary. */
        if((0U == ((pDataParams->bFlags & PHPAL_SLI15693_FLAG_INVENTORY))) && (0U != ((pDataParams->bFlags & PHPAL_SLI15693_FLAG_ADDRESSED))))
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Exchange(
                pDataParams->pHalDataParams,
                PH_EXCHANGE_BUFFER_CONT,    /* do not clear, do not send */
                pDataParams->pUid,
                PHPAL_SLI15693_UID_LENGTH,
                ppRxBuffer,
                pRxLen));
        }

        /* Now we need to keep our buffered contents */
        wOption |= PH_EXCHANGE_LEAVE_BUFFER_BIT;
    }

    /* Exchange command */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Exchange(
        pDataParams->pHalDataParams,
        wOption,
        pTxBuffer,
        wTxLen,
        ppRxBuffer,
        pRxLen));

    /* Skip response flag validation if In-Process reply is enabled.
     * Validation will be taken care in phpalSli15693_Sw_InProcessReply
     * interface.
     */
    if (bIsInProcessReply == PH_OFF)
    {
        /* If this was the last (or only) chunk, we should have received a response,
        so we extract the Flags byte and check the wStatus */
        if ((0U == ((wOption & PH_EXCHANGE_BUFFERED_BIT))))
        {
            /* First Check RespLength*/
            if (*pRxLen == 0U)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_SLI15693);
            }

            /* Check error flag */
            pDataParams->bResFlags = 0;
            if (0U != ((*ppRxBuffer[0]) & PHPAL_SLI15693_FLAG_RESP_ERROR))
            {
                pDataParams->bResFlags = PHPAL_SLI15693_FLAG_RESP_ERROR;

                /* Check the length in case of error */
                if (*pRxLen != 2U)
                {
                    return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_SLI15693);
                }

                /* Copy error code to additional info */
                pDataParams->wAdditionalInfo = (*ppRxBuffer)[1];

                /* Return empty receive buffer */
                *ppRxBuffer = NULL;
                *pRxLen = 0;

                return PH_ADD_COMPCODE(PHPAL_SLI15693_ERR_ISO15693, PH_COMP_PAL_SLI15693);
            }

            /* Copy response flag to additional info */
            pDataParams->wAdditionalInfo = (*ppRxBuffer)[0];

            /* Adjust pointer and length */
            ++(*ppRxBuffer);
            --(*pRxLen);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_SLI15693);
}






/* ISO15693 Utility commands --------------------------------------------------------------------------------------- */
phStatus_t phpalSli15693_Sw_GetSerialNo(phpalSli15693_Sw_DataParams_t * pDataParams, uint8_t * pUid, uint8_t * pUidLen)
{
    /* Return an error if UID is not valid */
    if (pDataParams->bUidBitLen != PHPAL_SLI15693_SW_UID_COMPLETE)
    {
        *pUidLen = 0;
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_PAL_SLI15693);
    }

    /* Copy the UID */
    (void) memcpy(pUid, pDataParams->pUid, PHPAL_SLI15693_UID_LENGTH);
    *pUidLen = PHPAL_SLI15693_UID_LENGTH;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_SLI15693);
}

phStatus_t phpalSli15693_Sw_SetSerialNo(phpalSli15693_Sw_DataParams_t * pDataParams, uint8_t * pUid, uint8_t bUidLen)
{
    /* Check for valid UID length */
    if(bUidLen != PHPAL_SLI15693_UID_LENGTH)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_SLI15693);
    }

    /* Copy UID and UID length */
    (void) memcpy(pDataParams->pUid, pUid, PHPAL_SLI15693_UID_LENGTH);
    pDataParams->bUidBitLen = PHPAL_SLI15693_SW_UID_COMPLETE;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_SLI15693);
}

phStatus_t phpalSli15693_Sw_SetConfig(phpalSli15693_Sw_DataParams_t * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wValueOld = 0;

    switch(wConfig)
    {
        case PHPAL_SLI15693_CONFIG_FLAGS:
            /* Retrieve current Sub-Carrier setting */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONFIG_SUBCARRIER,
                &wValueOld));

            /* Set dual Sub-Carrier if HAL is not already configured for it */
            if(0U != (wValue & PHPAL_SLI15693_FLAG_TWO_SUB_CARRIERS))
            {
                /* NFC mode supports only single sub-carrier mode */
                if(pDataParams->bOpeMode == RD_LIB_MODE_NFC)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_SLI15693);
                }

                if(wValueOld != PHHAL_HW_SUBCARRIER_DUAL)
                {
                    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
                        pDataParams->pHalDataParams,
                        PHHAL_HW_CONFIG_SUBCARRIER,
                        PHHAL_HW_SUBCARRIER_DUAL));
                }
            }

            /* Set single Sub-Carrier if HAL is not already configured for it */
            else
            {
                if(wValueOld != PHHAL_HW_SUBCARRIER_SINGLE)
                {
                    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
                        pDataParams->pHalDataParams,
                        PHHAL_HW_CONFIG_SUBCARRIER,
                        PHHAL_HW_SUBCARRIER_SINGLE));
                }
            }

            /* Retrieve current DataRate setting */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONFIG_RXDATARATE,
                &wValueOld));

            /* Set low data rate if HAL is not already configured for it */
            if(0U == (wValue & PHPAL_SLI15693_FLAG_DATA_RATE))
            {
                /* NFC mode supports only HIGH Data Rate */
                if(pDataParams->bOpeMode == RD_LIB_MODE_NFC)
                {
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_SLI15693);
                }

                if(wValueOld != PHHAL_HW_RF_RX_DATARATE_LOW)
                {
                    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
                        pDataParams->pHalDataParams,
                        PHHAL_HW_CONFIG_RXDATARATE,
                        PHHAL_HW_RF_RX_DATARATE_LOW));
                }
            }

            /* Set high data rate if HAL is not already configured for it */
            else
            {
                if(wValueOld != PHHAL_HW_RF_RX_DATARATE_HIGH)
                {
                    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
                        pDataParams->pHalDataParams,
                        PHHAL_HW_CONFIG_RXDATARATE,
                        PHHAL_HW_RF_RX_DATARATE_HIGH));
                }
            }

            /* Retrieve flags */
            pDataParams->bFlags = (uint8_t) wValue;
            break;

        case PHPAL_SLI15693_CONFIG_TIMEOUT_US:
            if(((uint32_t) wValue + PHPAL_SLI15693_SW_EXT_TIME_US) > 0xFFFFU)
            {
                /* Return error */
                /* Maximum allowed value is 0xFFC3 (0xFFFF - PHPAL_SLI15693_SW_EXT_TIME_US) */
                return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_PAL_SLI15693);
            }

            /* Extend timeout a little */
            wValue += PHPAL_SLI15693_SW_EXT_TIME_US;

            /* Retrieve current timeout */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,
                &wValueOld));

            /* Set the new timeout */
            if((((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS) && (wValue != wValueOld)) ||
                ((wStatus & PH_ERR_MASK) == PH_ERR_PARAMETER_OVERFLOW))
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
                    pDataParams->pHalDataParams,
                    PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,
                    wValue));
            }
            else
            {
                /* Return error if the error is other than PH_ERR_PARAMETER_OVERFLOW */
                PH_CHECK_SUCCESS(wStatus);
            }
            break;

        case PHPAL_SLI15693_CONFIG_TIMEOUT_MS:
            /* Retrieve current timeout */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,
                &wValueOld));

            /* Set the new timeout */
            if(wValue != wValueOld)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
                    pDataParams->pHalDataParams,
                    PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,
                    wValue));
            }
            break;

        case PHPAL_SLI15693_CONFIG_T1_PARAMETER:

            if((wValue != PHPAL_SLI15693_TIMEOUT_SHORT_US) && (pDataParams->bOpeMode == RD_LIB_MODE_NFC))
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_SLI15693);
            }

            /* Retrieve current T1 Value */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,
                &wValueOld));

            /* Set the new T1 Value */
            if(wValue != wValueOld)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
                    pDataParams->pHalDataParams,
                    PHHAL_HW_CONFIG_RXDEAFBITS,
                    (wValue - 10U)));

                PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
                    pDataParams->pHalDataParams,
                    PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,
                    wValue));
            }
            break;

        case PHPAL_SLI15693_CONFIG_TRANSPARENT:
            /* Set Flag byte for Transparent Exchange */
            pDataParams->bFlags = (uint8_t) wValue;
            break;

        case PHPAL_SLI15693_CONFIG_ENABLE_BUFFERING:
            pDataParams->bBuffering = (uint8_t) wValue;
            break;

        case PHPAL_SLI15693_CONFIG_TXDATARATE:
            if((pDataParams->bOpeMode == RD_LIB_MODE_NFC) && (wValue != PHPAL_SLI15693_26KBPS_DATARATE))
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_SLI15693);
            }

            if(wValue > PHPAL_SLI15693_212KBPS_DATARATE)
            {
                return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_SLI15693);
            }

            switch(wValue)
            {
                case PHPAL_SLI15693_26KBPS_DATARATE: wValue = PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4; break;
                case PHPAL_SLI15693_53KBPS_DATARATE: wValue = PHHAL_HW_RF_I15693_53KBPS_DATARATE; break;
                case PHPAL_SLI15693_106KBPS_DATARATE: wValue = PHHAL_HW_RF_DATARATE_106; break;
                case PHPAL_SLI15693_212KBPS_DATARATE: wValue = PHHAL_HW_RF_DATARATE_212; break;
                default: break;
            }

            /* Set the new TX data rate */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONFIG_TXDATARATE,
                wValue));
            break;

        case PHPAL_SLI15693_CONFIG_RXDATARATE:
            if((pDataParams->bOpeMode == RD_LIB_MODE_NFC) && (wValue != PHPAL_SLI15693_26KBPS_DATARATE))
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_SLI15693);
            }

            switch(wValue)
            {
                case PHPAL_SLI15693_26KBPS_DATARATE: wValue = PHHAL_HW_RF_RX_DATARATE_HIGH; break;
                case PHPAL_SLI15693_53KBPS_DATARATE: wValue = PHHAL_HW_RF_RX_DATARATE_FAST_HIGH; break;
                case PHPAL_SLI15693_106KBPS_DATARATE: wValue = PHHAL_HW_RF_DATARATE_106; break;
                case PHPAL_SLI15693_212KBPS_DATARATE: wValue = PHHAL_HW_RF_DATARATE_212; break;
                default: break;
            }

            /* Set the new RX data rate */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONFIG_RXDATARATE,
                wValue));
            break;

        case PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT:
            if(pDataParams->bMaxRetryCount > PHPAL_SLI15693_SW_RETRYCOUNT_MAX)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_SLI15693);
            }
            pDataParams->bMaxRetryCount = (uint8_t) wValue;
            break;

        case PHPAL_SLI15693_CONFIG_ICMFGCODE:
            pDataParams->bIcMfgCode = (uint8_t) wValue;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_SLI15693);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_SLI15693);
}

phStatus_t phpalSli15693_Sw_GetConfig(phpalSli15693_Sw_DataParams_t * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    switch(wConfig)
    {
        case PHPAL_SLI15693_CONFIG_FLAGS:
            *pValue = pDataParams->bFlags;
            break;

        case PHPAL_SLI15693_CONFIG_ADD_INFO:
            *pValue = pDataParams->wAdditionalInfo;
            break;

        case PHPAL_SLI15693_CONFIG_TIMEOUT_US:
            /* Get HAL timeout value */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,
                pValue));

                /* Remove timeout extension */
            *pValue -= PHPAL_SLI15693_SW_EXT_TIME_US;
            break;

        case PHPAL_SLI15693_CONFIG_TIMEOUT_MS:
            /* Get HAL timeout value */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,
                pValue));
            break;

        case PHPAL_SLI15693_CONFIG_TXDATARATE:
            /* Get HAL timeout value */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONFIG_TXDATARATE,
                pValue));

            /* Map the HAL TX BaudRates to generic 15693 BaudRates */
            switch(*pValue)
            {
                case PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4: *pValue = PHPAL_SLI15693_26KBPS_DATARATE; break;
                case PHHAL_HW_RF_I15693_53KBPS_DATARATE: *pValue = PHPAL_SLI15693_53KBPS_DATARATE; break;
                case PHHAL_HW_RF_DATARATE_106: *pValue = PHPAL_SLI15693_106KBPS_DATARATE; break;
                case PHHAL_HW_RF_DATARATE_212: *pValue = PHPAL_SLI15693_212KBPS_DATARATE; break;
                default: break;
            }
            break;

        case PHPAL_SLI15693_CONFIG_RXDATARATE:
            /* Get HAL timeout value */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONFIG_RXDATARATE,
                pValue));

            /* Map the HAL RX BaudRates to generic 15693 BaudRates */
            switch(*pValue)
            {
                case PHHAL_HW_RF_RX_DATARATE_HIGH: *pValue = PHPAL_SLI15693_26KBPS_DATARATE; break;
                case PHHAL_HW_RF_RX_DATARATE_FAST_HIGH: *pValue = PHPAL_SLI15693_53KBPS_DATARATE; break;
                case PHHAL_HW_RF_DATARATE_106: *pValue = PHPAL_SLI15693_106KBPS_DATARATE; break;
                case PHHAL_HW_RF_DATARATE_212: *pValue = PHPAL_SLI15693_212KBPS_DATARATE; break;
                default: break;
            }
            break;

        case PHPAL_SLI15693_CONFIG_T1_PARAMETER:
            /* Retrieve current T1 Value */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,
                pValue));
            break;

        case PHPAL_SLI15693_CONFIG_TRANSPARENT:
            /* Set Flag byte for Transparent Exchange */
            *pValue = pDataParams->bFlags;
            break;

        case PHPAL_SLI15693_CONFIG_ENABLE_BUFFERING:
            *pValue = pDataParams->bBuffering;
            break;

        case PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT:
            *pValue = pDataParams->bMaxRetryCount;
            break;

        case PHPAL_SLI15693_CONFIG_ICMFGCODE:
            *pValue = pDataParams->bIcMfgCode;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_PAL_SLI15693);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_SLI15693);
}

#endif /* NXPBUILD__PHPAL_SLI15693_SW */
