/*
 * Copyright 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#include "Example-SAMAV3.h"

int __cdecl main()
{
    /* Local variables */
    uint8_t aAtr_Contactless[256];
    uint8_t aAtr_Contact[256];

    uint8_t aTx_BalBuffer[512];
    uint8_t aRx_BalBuffer[512];

    uint8_t aTx_HalBuffer[512];
    uint8_t aRx_HalBuffer[512];

    uint8_t aTx_SamBuffer[512];
    uint8_t aRx_SamBuffer[512];

    uint8_t aSeed[8];
    uint8_t aPLUploadBuffer[4096];

    uint8_t aComPort[20];
    uint8_t aEnumBuff[512];

    phKeyStore_Sw_KeyEntry_t aKeyEntry[0xFF];
    phKeyStore_Sw_KeyVersionPair_t aKeyVersion[0xFF * 0x01];
    phKeyStore_Sw_KUCEntry_t aKeyUsage[0xFF];

    /* Variables for Common data parameter. */
    phCryptoSym_Sw_DataParams_t stCryptoEnc;
    phCryptoSym_Sw_DataParams_t stCryptoMAC;
    phCryptoSym_Sw_DataParams_t stPLUpload_CryptoEnc;
    phCryptoSym_Sw_DataParams_t stPLUpload_CryptoMAC;
    phCryptoSym_Sw_DataParams_t stCryptoSymRnd;
    phCryptoRng_Sw_DataParams_t stCryptoRng;

    /* Variables for PAL data parameter. */
    phpalI14443p3a_Sw_DataParams_t stI14443p3a_Sw;
    phpalI14443p4a_Sw_DataParams_t stI14443p4a_Sw;
    phpalI14443p4_Sw_DataParams_t stI14443p4_Sw;
    phpalMifare_Sw_DataParams_t stPalMifare_Sw;
    phpalI14443p3a_SamAV3_X_DataParams_t stI14443p3a_X;
    phpalI14443p4a_SamAV3_X_DataParams_t stI14443p4a_X;
    phpalI14443p4_SamAV3_X_DataParams_t stI14443p4_X;
    phpalMifare_SamAV3_X_DataParams_t stPalMifare_X;
    phpalSli15693_Sw_DataParams_t stPalI15693;

    int dwChoice = 0;

    printf("----------------------------------------------------------------------------------------------------------- \n\n");
    printf("NxpRdLib example program to demonstrate SAM AV3 features. \n");
    printf("Note: \n");
    printf("     The below connection for S mode are supported. \n");
    printf("     Rd730 (Pegoda-3) configured for both contactless and contact mode connection ");
    printf("(Communication via VCom interface). \n");

    printf("     Rd710 (Pegoda-2) configured for both contactless and contact mode connection");
    printf(" (Communication via PCSCWin interface). \n");

    printf("     Rd710 (Pegoda-2) configured for contactless and BroadCom for contact mode connection");
    printf(" (Communication via PCSCWin interface). \n");

    printf("\n");
    printf("----------------------------------------------------------------------------------------------------------- \n\n");

    /* Initialize Software KeyStore component. */
    CHECK_SUCCESS_PRINT(phKeyStore_Sw_Init(&stKeystoreSw, sizeof(phKeyStore_Sw_DataParams_t), aKeyEntry, 0xFF, aKeyVersion, 0x01, aKeyUsage, 0xFF));

    /** Initialize Crypto components. */
    CHECK_SUCCESS_PRINT(phCryptoSym_Sw_Init(&stCryptoEnc, sizeof(phCryptoSym_Sw_DataParams_t), &stKeystoreSw));
    CHECK_SUCCESS_PRINT(phCryptoSym_Sw_Init(&stCryptoMAC, sizeof(phCryptoSym_Sw_DataParams_t), &stKeystoreSw));
    CHECK_SUCCESS_PRINT(phCryptoSym_Sw_Init(&stCryptoEnc1, sizeof(phCryptoSym_Sw_DataParams_t), &stKeystoreSw));
    CHECK_SUCCESS_PRINT(phCryptoSym_Sw_Init(&stCryptoMAC1, sizeof(phCryptoSym_Sw_DataParams_t), &stKeystoreSw));
    CHECK_SUCCESS_PRINT(phCryptoSym_Sw_Init(&stPLUpload_CryptoEnc, sizeof(phCryptoSym_Sw_DataParams_t), &stKeystoreSw));
    CHECK_SUCCESS_PRINT(phCryptoSym_Sw_Init(&stPLUpload_CryptoMAC, sizeof(phCryptoSym_Sw_DataParams_t), &stKeystoreSw));
    CHECK_SUCCESS_PRINT(phCryptoSym_Sw_Init(&stCryptoSymRnd, sizeof(phCryptoSym_Sw_DataParams_t), &stKeystoreSw));
    CHECK_SUCCESS_PRINT(phCryptoRng_Sw_Init(&stCryptoRng, sizeof(phCryptoRng_Sw_DataParams_t), &stCryptoSymRnd));

    /* Seed the random number generator with the given seed. */
    memset(aSeed, 0x00, 0x08);
    CHECK_SUCCESS_PRINT(phCryptoRng_Seed(&stCryptoRng, aSeed, sizeof(aSeed)));

    /* Get the reader connection to be performed. */
    printf("Press 1 to connect the reader in S Mode.\n");
    printf("Press 2 to connect the reader in X Mode with RC523 reader IC (Support by Pegoda-2 only).\n");
    printf("Press x to exit the application.\n");
    dwChoice = _getch();

    /* Exit if requested. */
    EXIT_IF_REQUESTED(dwChoice);

    /* Update Reader Mode flag. */
    eReaderMode = (enum ReaderMode) (dwChoice - 0x30);

    /* Perform connection based on the reader communication. */
    switch(eReaderMode)
    {
        case S_MODE:
            /* Get the type of S Mode connection to perform. */
            printf("\n");
            printf("Press 1 to connect with RD730 (Pegoda-3) for both contactless and contact communication via VCom interface.\n");
            printf("Press 2 to connect with RD710 (Pegoda-2) for both contactless and contact communication via PCSC(SCard) interface.\n");
            printf("Press 3 to connect with RD710 (Pegoda-2) for contactless and BroadComm for contact communication via PCSC(SCard) interface.\n");
            printf("Press x to exit the application.\n");
            dwChoice = _getch();

            /* Exit if requested. */
            EXIT_IF_REQUESTED(dwChoice);

            dwChoice = (uint16_t) (dwChoice - 0x30);
            switch(dwChoice)
            {
                case 1:
                    /* Add escape sequence. */
                    aComPort[0] = '\\'; aComPort[1] = '\\'; aComPort[2] = '.'; aComPort[3] = '\\';

                    /* Get the Comport to use. */
                    printf("\n");
                    printf("Please provide the Comport number to use - ");
                    scanf("%s", &aComPort[4U]);

                    /* Initialize and connect to Contactless Bal component. */
                    CHECK_SUCCESS_PRINT(phbalReg_SerialWin_Init(&stBal_Serial, sizeof(phbalReg_SerialWin_DataParams_t), sizeof(aEnumBuff), aEnumBuff));
                    CHECK_SUCCESS_PRINT(phbalReg_SetPort(&stBal_Serial, aComPort));
                    CHECK_SUCCESS_PRINT(phbalReg_OpenPort(&stBal_Serial));

                    /* Set the BitRate for BAL. */
                    CHECK_SUCCESS_PRINT(phbalReg_SetConfig(&stBal_Serial, PHBAL_REG_SERIALWIN_CONFIG_BITRATE, PHBAL_REG_SERIALWIN_VALUE_BITRATE_115200));

                    /* Update the generic Bal variable. */
                    pBal_Generic = &stBal_Serial;

                    /* Initialize the reader HAL component. */
                    CHECK_SUCCESS_PRINT(phhalHw_DUT_Init(&stHal_DUT, sizeof(stHal_DUT), pBal_Generic, 0, aTx_HalBuffer, sizeof(aTx_HalBuffer), aRx_HalBuffer,
                        sizeof(aRx_HalBuffer)));
                    CHECK_SUCCESS_PRINT(phhalHw_DUT_InitReader(&stHal_DUT, PHHAL_HW_DUT_INIT_CONTACTLESS));

                    /* Update the generic Hal variable. */
                    pHal_Generic = &stHal_DUT;

                    /* Initialize BAL SAM. */
                    CHECK_SUCCESS_PRINT(phbalReg_Sam_Init(&stBal_Sam, sizeof(phbalReg_Sam_DataParams_t), pBal_Generic, aAtr_Contact, 256,
                        aTx_BalBuffer, sizeof(aTx_BalBuffer), aRx_BalBuffer, sizeof(aRx_BalBuffer)));
                    CHECK_SUCCESS_PRINT(phbalReg_OpenPort(&stBal_Sam));

                    /* Update the generic Bal variable. */
                    pBal_Generic = &stBal_Sam;
                    break;

                case 2:
                case 3:
                    /* Initialize and connect to Contactless Bal component. */
                    CHECK_SUCCESS_PRINT(phbalReg_PcscWin_Init(&stBal_Pcsc_ContactLess, sizeof(phbalReg_PcscWin_DataParams_t), aAtr_Contactless, 256));
                    CHECK_SUCCESS_PRINT(phbalReg_SetConfig(&stBal_Pcsc_ContactLess, PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL, PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_UNDEFINED));
                    CHECK_SUCCESS_PRINT(phbalReg_SetConfig(&stBal_Pcsc_ContactLess, PHBAL_REG_PCSCWIN_CONFIG_SHARE, PHBAL_REG_PCSCWIN_VALUE_SHARE_DIRECT));
                    CHECK_SUCCESS_PRINT(phbalReg_SetPort(&stBal_Pcsc_ContactLess, (uint8_t *) ((dwChoice == 2) ? PCSC_READER_P2_S_CONTACTLESS_NAME :
                        PCSC_READER_P2_S_CONTACTLESS_NAME_EXT)));
                    CHECK_SUCCESS_PRINT(phbalReg_OpenPort(&stBal_Pcsc_ContactLess));

                    /* Connect to Contact bal. */
                    if(dwChoice == 2)
                    {
                        CHECK_SUCCESS_PRINT(phbalReg_Sam_Init(&stBal_Sam, sizeof(phbalReg_Sam_DataParams_t), &stBal_Pcsc_ContactLess, aAtr_Contact, 256,
                            aTx_BalBuffer, sizeof(aTx_BalBuffer), aRx_BalBuffer, sizeof(aRx_BalBuffer)));
                        CHECK_SUCCESS_PRINT(phbalReg_OpenPort(&stBal_Sam));

                        /* Update the generic Bal variable. */
                        pBal_Generic = &stBal_Sam;
                    }
                    else
                    {
                        /* Initialize and connect to Contact Bal component. */
                        CHECK_SUCCESS_PRINT(phbalReg_PcscWin_Init(&stBal_Pcsc_Contact, sizeof(phbalReg_PcscWin_DataParams_t), aAtr_Contact, 256));
                        CHECK_SUCCESS_PRINT(phbalReg_SetPort(&stBal_Pcsc_Contact, (uint8_t *) PCSC_READER_P2_S_CONTACT_NAME_EXT));
                        CHECK_SUCCESS_PRINT(phbalReg_OpenPort(&stBal_Pcsc_Contact));

                        /* Update the generic Bal variable. */
                        pBal_Generic = &stBal_Pcsc_Contact;
                    }

                    /* Initialize Hal component for contactless reader. */
                    CHECK_SUCCESS_PRINT(phhalHw_Rd710_Init(&stHal_Rd710, sizeof(phhalHw_Rd710_DataParams_t), &stBal_Pcsc_ContactLess, 0, aTx_HalBuffer, sizeof(aTx_HalBuffer),
                        aRx_HalBuffer, sizeof(aRx_HalBuffer)));
                    CHECK_SUCCESS_PRINT(phhalHw_Rd710_Cmd_InitReader(&stHal_Rd710));

                    /* Update the generic Hal variable. */
                    pHal_Generic = &stHal_Rd710;
                    break;

                default:
                    printf("\n\n");
                    printf("Invalid option selected.\n");
                    printf("Press any key to exit with the demonstration.\n");
                    _getch();
                    exit(0);
                    break;
            }
            break;

        case X_MODE_RC523:
            /* Connect to contact reader. */
            CHECK_SUCCESS_PRINT(phbalReg_PcscWin_Init(&stBal_Pcsc_Contact, sizeof(phbalReg_PcscWin_DataParams_t), aAtr_Contact, 256));
            CHECK_SUCCESS_PRINT(phbalReg_SetPort(&stBal_Pcsc_Contact, (uint8_t *) PCSC_READER_P2_X_NAME));
            CHECK_SUCCESS_PRINT(phbalReg_OpenPort(&stBal_Pcsc_Contact));

            /* Update the generic Bal and Hal variable. */
            pBal_Generic = &stBal_Pcsc_Contact;
            pHal_Generic = NULL;
            break;

        default:
            printf("\n\n");
            printf("Invalid option selected.\n");
            printf("Press any key to exit with the demonstration.\n");
            _getch();
            exit(0);
            break;
    }

    /* Initialize Hal component for contact reader. */
    CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Init(&stHal_SamAv3, sizeof(stHal_SamAv3), pBal_Generic, pHal_Generic, &stKeystoreSw, &stCryptoEnc, &stCryptoMAC, &stCryptoRng,
        &stPLUpload_CryptoEnc, &stPLUpload_CryptoMAC, ((eReaderMode == S_MODE) ? PHHAL_HW_SAMAV3_OPMODE_NON_X : (eReaderMode == X_MODE_RC523) ? PHHAL_HW_SAMAV3_OPMODE_X_RC523 :
        PHHAL_HW_SAMAV3_OPMODE_X_RC663), 0x00, aTx_SamBuffer, sizeof(aTx_SamBuffer), aRx_SamBuffer, sizeof(aRx_SamBuffer), aPLUploadBuffer));

    /* Update generic Hal to Sam aV3 DataParams. */
    pHal_Generic = &stHal_SamAv3;

    /* Initialize SAM KeyStore component. */
    CHECK_SUCCESS_PRINT(phKeyStore_SamAV3_Init(&stKeystoreSAM, sizeof(phKeyStore_SamAV3_DataParams_t), &stHal_SamAv3));

    /* Detect SAM settings. This will update the Host version internally. */
    CHECK_SUCCESS_PRINT(phhalHw_SamAV3_DetectMode(&stHal_SamAv3));

    /* Load a the key. */
    CHECK_SUCCESS_PRINT(phKeyStore_FormatKeyEntry(&stKeystoreSw, SAM_MASTER_KEY_ADDRESS, PH_KEYSTORE_KEY_TYPE_AES128));
    CHECK_SUCCESS_PRINT(phKeyStore_SetKeyAtPos(&stKeystoreSw, SAM_MASTER_KEY_ADDRESS, 0, PH_KEYSTORE_KEY_TYPE_AES128, aHostAuthKey, SAM_MASTER_KEY_VERSION));

    /* Perform HOST AUTHENTICATION in PLAIN mode. */
    CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_AuthenticateHost(&stHal_SamAv3, bCommMode, SAM_MASTER_KEY_ADDRESS, SAM_MASTER_KEY_VERSION, SAM_MASTER_KEY, 0));

    /* Perform Reader IC initialization for X mode and PAL initialization. */
    if(eReaderMode == S_MODE)
    {
        /* Initialize the PAL components. */
        CHECK_SUCCESS_PRINT(phpalI14443p3a_Sw_Init(&stI14443p3a_Sw, sizeof(phpalI14443p3a_Sw_DataParams_t), &stHal_SamAv3));
        CHECK_SUCCESS_PRINT(phpalI14443p4a_Sw_Init(&stI14443p4a_Sw, sizeof(phpalI14443p4a_Sw_DataParams_t), &stHal_SamAv3));
        CHECK_SUCCESS_PRINT(phpalI14443p4_Sw_Init(&stI14443p4_Sw, sizeof(phpalI14443p4_Sw_DataParams_t), &stHal_SamAv3));
        CHECK_SUCCESS_PRINT(phpalMifare_Sw_Init(&stPalMifare_Sw, sizeof(phpalMifare_Sw_DataParams_t), &stHal_SamAv3, &stI14443p4_Sw));
        CHECK_SUCCESS_PRINT(phpalSli15693_Sw_Init(&stPalI15693, sizeof(phpalSli15693_Sw_DataParams_t), &stHal_SamAv3));

        /* Update the specific data params to generic ones. */
        pPalIso14443_3A_Generic = &stI14443p3a_Sw;
        pPalIso14443_4A_Generic = &stI14443p4a_Sw;
        pPalIso14443_4_Generic = &stI14443p4_Sw;
        pPalMifare_Generic = &stPalMifare_Sw;
        pPalIso15693 = &stPalI15693;
    }
    else
    {
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_RC_Init(&stHal_SamAv3, 0x00));

        /* Initialize the PAL components. */
        CHECK_SUCCESS_PRINT(phpalI14443p3a_SamAV3_X_Init(&stI14443p3a_X, sizeof(phpalI14443p3a_SamAV3_X_DataParams_t), &stHal_SamAv3));
        CHECK_SUCCESS_PRINT(phpalI14443p4a_SamAV3_X_Init(&stI14443p4a_X, sizeof(phpalI14443p4a_SamAV3_X_DataParams_t), &stHal_SamAv3));
        CHECK_SUCCESS_PRINT(phpalI14443p4_SamAV3_X_Init(&stI14443p4_X, sizeof(phpalI14443p4_SamAV3_X_DataParams_t), &stHal_SamAv3));
        CHECK_SUCCESS_PRINT(phpalMifare_SamAV3_X_Init(&stPalMifare_X, sizeof(phpalMifare_SamAV3_X_DataParams_t), &stHal_SamAv3, &stI14443p4_X));

        /* Update the specific data params to generic ones. */
        pPalIso14443_3A_Generic = &stI14443p3a_X;
        pPalIso14443_4A_Generic = &stI14443p4a_X;
        pPalIso14443_4_Generic = &stI14443p4_X;
        pPalMifare_Generic = &stPalMifare_X;
    }

    /* Perform the sample scenarios. */
    do
    {
        dwChoice = 0;

        /* Clear the console. */
        system("cls");

        printf("Host Authentication features --------------------------------------------------------------------------\n");
        printf("Press %d for Demo on Host Authentication command. \n", HOST_AUTHENTICATION);
        printf("Press %d for Demo on Lock / Unlock command. \n", LOCK_UNLOCK);
        printf("\n");

        printf("Security and Communication features -------------------------------------------------------------------\n");
        printf("Press %d for Demo on GetVersion command. \n", GET_VERSION);
        printf("Press %d for Demo on Activate Offline Key command. \n", ACTIVATE_OFFLINE);
        printf("Press %d for Demo on Load Init Vector command. \n", LOAD_INIT_VECTOR);
        printf("Press %d for Demo on Kill Authentication command. \n", KILL_AUTHENTICATION);
        printf("Press %d for Demo on Select Application command. \n", SELECT_APPLICATION);
        printf("Press %d for Demo on Get Random command. \n", GET_RANDOM);
        printf("Press %d for Demo on Sleep command. \n", SLEEP);
        printf("\n");

        printf("Key Management features -------------------------------------------------------------------------------\n");
        printf("Press %d for Demo on Get Key Entry command. \n", GET_KEY_ENTRY);
        printf("Press %d for Demo on Change Key Entry command. \n", CHANGE_KEY_ENTRY);
        printf("Press %d for Demo on Get KUC Entry command. \n", GET_KUC_ENTRY);
        printf("Press %d for Demo on Change KUC Entry command. \n", CHANGE_KUC_ENTRY);
        printf("Press %d for Demo on Dump Session Key command. \n", DUMP_SESSION_KEY);
        printf("Press %d for Demo on Dump Secret Key command. \n", DUMP_SECRET_KEY);
        printf("Press %d for Demo on Generate & Verify MAC command. \n", GENERATE_VERIFY_MAC);
        printf("Press %d for Demo on Encipher & Decipher Data command. \n", ENCIPHER_DECIPHER_DATA);
        printf("Press %d for Demo on Encipher & Decipher Offline Data command. \n", ENCIPHER_DECIPHER_DATA_OFFLINE);
        printf("\n");

        printf("PICC features -----------------------------------------------------------------------------------------\n");
        printf("Press %d for Demo on MIFARE DESFire EV2 / EV3 PICC feature. \n", DESFIRE);
        printf("Press %d for Demo on MIFARE PLUS EV1 / EV2 PICC feature. \n", PLUS);
        printf("Press %d for Demo on MIFARE Classic PICC feature. \n", CLASSIC);
        printf("Press %d for Demo on MIFARE Ultralight PICC feature. \n", ULTRALIGHT);
        printf("Press %d for Demo on ISO / IEC 29167-10 feature. \n", ISO_IEC_29167);
        printf("Press %d for Demo on Transaction MAC feature. \n", TMAC);
        printf("Press %d for Demo on Secure Dynamic Messaging feature. \n", SDM);
        printf("\n");

        printf("Programmable Logic feature ----------------------------------------------------------------------------\n");
        printf("Press %d for Demo on Programmable Logic feature. \n", PROGRAMMABLE_LOGIC);

        printf("\n");
        printf("Press x to exit the application.\n");
        printf("Enter the option and press Enter to perform the demo - ");
        scanf_s("%d", &dwChoice);

        printf("\n\n");

        /* Exit if requested by user. */
        EXIT_IF_REQUESTED(dwChoice);

        switch(dwChoice)
        {
            case HOST_AUTHENTICATION:
                printf("Demo on Host Authentication command -------------------------------------------------- \n\n");
                bCommMode = 0;
                Demo_HostAuthentication();
                break;

            case LOCK_UNLOCK:
                printf("Demo on Lock Unlock command ---------------------------------------------------------- \n\n");
                Demo_LockUnlock();
                break;

            case GET_VERSION:
                printf("Demo on GetVersion command ----------------------------------------------------------- \n\n");
                Demo_GetVersion();
                break;

            case ACTIVATE_OFFLINE:
                printf("Demo on Activate Offline Key command ------------------------------------------------- \n\n");
                Demo_ActivateOfflineKey();
                break;

            case LOAD_INIT_VECTOR:
                printf("Demo on Load Init Vector command ---------------------------------------------------- \n\n");
                Demo_LoadInitVector();
                break;

            case KILL_AUTHENTICATION:
                printf("Demo on Kill Authentication command -------------------------------------------------- \n\n");
                Demo_KillAuthentication();
                break;

            case SELECT_APPLICATION:
                printf("Demo on Select Application command --------------------------------------------------- \n\n");
                Demo_SelectApplication();
                break;

            case GET_RANDOM:
                printf("Demo on Get Random command ---------------------------------------------------------- \n\n");
                Demo_GetRandom();
                break;

            case SLEEP:
                printf("Demo on Sleep command ---------------------------------------------------------------- \n\n");
                Demo_Sleep();
                break;

            case GET_KEY_ENTRY:
                printf("Demo on Get Key Entry command -------------------------------------------------------- \n\n");
                Demo_GetKeyEntry(PICC_AES128_KEY_ADDRESS, 0x01);
                break;

            case CHANGE_KEY_ENTRY:
                printf("Demo on Change Key Entry command ----------------------------------------------------- \n\n");
                Demo_ChangeKeyEntry(PICC_AES128_KEY_ADDRESS, KEY_TYPE_AES128, KEY_CLASS_PICC, aAESKey, sizeof(aAESKey), 0x01);
                break;

            case GET_KUC_ENTRY:
                printf("Demo on Get KUC Entry command -------------------------------------------------------- \n\n");
                Demo_GetKUCEntry(0x01);
                break;

            case CHANGE_KUC_ENTRY:
                printf("Demo on Change KUC Entry command ----------------------------------------------------- \n\n");
                Demo_ChangeKUCEntry();
                break;

            case DUMP_SESSION_KEY:
                printf("Demo on DumpSessionKey command ------------------------------------------------------ \n\n");
                printf("\tMake sure that DESFire EV2 / EV3 card is placed in field.\n");
                printf("\tMake sure that DESFire EV2 / EV3 card's master key (0x00) is of type DES.\n");
                printf("\tPress any key to proceed with the demonstration.\n");
                _getch();
                printf("\n");

                Demo_DumpSessionKey();
                break;

            case DUMP_SECRET_KEY:
                printf("Demo on DumpSecretKey command ------------------------------------------------------- \n\n");
                Demo_DumpSecretKey();
                break;

            case GENERATE_VERIFY_MAC:
                printf("Demo on GenerateMAC / VerifyMAC command --------------------------------------------- \n\n");
                Demo_GenerateVerifyMAC();
                break;

            case ENCIPHER_DECIPHER_DATA:
                printf("Demo on EncipherData / DecipherData command ----------------------------------------- \n\n");
                Demo_EncipherDecipherData();
                break;

            case ENCIPHER_DECIPHER_DATA_OFFLINE:
                printf("Demo on EncipherOfflineData / DecipherOfflineData command --------------------------- \n\n");
                Demo_EncipherDecipherDataOffline();
                break;

            case DESFIRE:
                printf("Demo on MIFARE DESFire PICC --------------------------------------------------------- \n\n");
                printf("\tMake sure that DESFire EV2 / EV3 card is placed in field.\n");
                printf("\tMake sure that DESFire EV2 / EV3 card's master key (0x00) is of type DES.\n");
                printf("\tPress any key to proceed with the demonstration.\n");
                _getch();
                printf("\n");

                Demo_MIFARE_DESFireEV2();
                break;

            case PLUS:
                printf("Demo on MIFARE PLUS PICC ------------------------------------------------------------ \n\n");
                printf("\tMake sure that PLUS EV1 / EV2 card is placed in field.\n");
                printf("\tMake sure that PLUS EV1 / EV2 card is personalized to Security Level 3.\n");
                printf("\tMake sure that PLUS EV1 / EV2 card's sector key are personalized to 0 values.\n");
                printf("\tPress any key to proceed with the demonstration.\n");
                _getch();
                printf("\n");

                Demo_MIFARE_PlusEV1();
                break;

            case CLASSIC:
                printf("Demo on MIFARE Classic PICC ------------------------------------------------------------ \n\n");
                printf("\tMake sure that Classic card is placed in field.\n");
                printf("\tPress any key to proceed with the demonstration.\n");
                _getch();
                printf("\n");

                Demo_MIFARE_Classic();
                break;

            case ULTRALIGHT:
                printf("Demo on MIFARE Ultralight PICC ----------------------------------------------------------\n\n");

                Demo_MIFARE_Ultralight();
                break;

            case ISO_IEC_29167:
                /* Include for SAM AV3 and S Mode reader connection only. */
                if((IncludeDemo_Reader(S_MODE) == PH_ERR_SUCCESS) && (IncludeDemo_Version(0x03) == PH_ERR_SUCCESS))
                {
                    printf("Demo on ISO / IEC 29167 - 10 -------------------------------------------------------- \n\n");
                    printf("\tMake sure that ICODE DNA tag is placed in field.\n");
                    printf("\tMake sure that ICODE DNA tag is personalized with proper configuration values.\n");
                    printf("\tMake sure that ICODE DNA tags's key 0 are personalized to 0 values.\n");
                    printf("\tPress any key to proceed with the demonstration.\n");
                    _getch();
                    printf("\n");

                    Demo_ISO_IEC29167();
                }
                break;

            case TMAC:
                /* Include for Sam AV3 only. */
                if(IncludeDemo_Version(0x03) == PH_ERR_SUCCESS)
                {
                    printf("Demo on Transaction MAC feature ----------------------------------------------------- \n\n");
                    printf("\tMake sure that MIFARE DESFire Light card is placed in field.\n");
                    printf("\tMake sure that MIFARE DESFire Light card is of factory default one.\n");
                    printf("\tPress any key to proceed with the demonstration.\n");
                    _getch();
                    printf("\n");

                    Demo_TransactionMAC(dwChoice);
                }
                break;

            case SDM:
                /* Include for Sam AV3 only. */
                if(IncludeDemo_Version(0x03) == PH_ERR_SUCCESS)
                {
                    printf("Demo on SecureDynamic Messaging feature -------------------------------------------- \n\n");
                    printf("\tMake sure that NTAG 42x DNA card is placed in field.\n");
                    printf("\tMake sure that NTAG 42x DNA card is of factory default one.\n");
                    printf("\tPress any key to proceed with the demonstration.\n");
                    _getch();
                    printf("\n");

                    Demo_SecureDynamicMessaging(dwChoice);
                }
                break;

            case PROGRAMMABLE_LOGIC:
                /* Include for Sam AV3 only. */
                if(IncludeDemo_Version(0x03) == PH_ERR_SUCCESS)
                {
                    printf("Demo on Programmable Logic feature -------------------------------------------------- \n\n");

                    Demo_ProgrammableLogic();
                }
                break;

            default:
                printf("Invalid option selected.\n");
                break;
        }

        printf("\n\n");
        printf("Press any key to continue with the demonstration.\n");
        printf("Press x to exit the demonstration.\n");
        dwChoice = _getch();
    } while((dwChoice != 'x') && (dwChoice != 'X'));

    /* Close the connection. */
    if(stBal_Pcsc_Contact.wId != 0x0000)
        CHECK_SUCCESS_PRINT(phbalReg_ClosePort(&stBal_Pcsc_Contact));

    if(stBal_Sam.wId != 0x0000)
        CHECK_SUCCESS_PRINT(phbalReg_ClosePort(&stBal_Sam));

    if(stBal_Pcsc_ContactLess.wId != 0x0000)
        CHECK_SUCCESS_PRINT(phbalReg_ClosePort(&stBal_Pcsc_ContactLess));

    return (0);
}

void PrintData(uint8_t* pBuffer, uint32_t dwLength, char* pFormat, char* pSpecialChar)
{
    uint32_t dwIndex;

    for(dwIndex = 0; dwIndex < dwLength; ++dwIndex)
        printf(pFormat, pBuffer[dwIndex]);

    printf(pSpecialChar);
}

phStatus_t IncludeDemo_Version(uint8_t bSamVer)
{
    if(bSamVer != stHal_SamAv3.bHostMode)
    {
        printf("\n\n");
        printf("This demo feature is not supported with Sam AV2 version.");
        printf("\n\n");

        return PH_ERR_UNSUPPORTED_COMMAND;
    }

    return PH_ERR_SUCCESS;
}

phStatus_t IncludeDemo_Reader(enum ReaderMode eMode)
{
    if(eReaderMode != eMode)
    {
        printf("\n\n");
        printf("This demo feature is supported in %s reader configuration only.", TO_STRING(eMode));
        printf("\n\n");

        return PH_ERR_UNSUPPORTED_COMMAND;
    }

    return PH_ERR_SUCCESS;
}

void Reverse(uint8_t * pData, uint16_t wLength)
{
    uint8_t bTemp;

    uint8_t bLSB = 0;
    uint8_t bMSB = (uint8_t) (wLength - 1);

    while(bLSB <= bMSB)
    {
        bTemp = pData[bLSB];
        pData[bLSB] = pData[bMSB];
        pData[bMSB] = bTemp;

        bLSB++;
        bMSB--;
    }
}

void ConvertAscii_Hex(uint8_t * pInput, uint32_t dwInLen, uint8_t * pOutString)
{
    uint32_t dwIteration = 0;

    for(dwIteration = 0; dwIteration < (dwInLen / 2); dwIteration++)
    {
        pOutString[dwIteration] = (pInput[2 * dwIteration] <= '9') ? ((pInput[2 * dwIteration] - '0') * 16) : (((pInput[2 * dwIteration] - 'A') + 10) << 4);
        pOutString[dwIteration] |= (pInput[(2 * dwIteration) + 1] <= '9') ? (pInput[(2 * dwIteration) + 1] - '0') : ((pInput[(2 * dwIteration) + 1] - 'A') + 10);
    }
}



phStatus_t ActivateCard(uint8_t bIsoLayer)
{
    phStatus_t wStatus;

    uint8_t bSak = 0;
    uint8_t bCardPlaced = 0;
    uint8_t bMoreCardsAvailable;
    uint8_t bCidEnabled;
    uint8_t bCid = 1;
    uint8_t bNadSupported;
    uint8_t bFwi;
    uint8_t bFsdi = 8;
    uint8_t bFsci;
    uint8_t bAfi = 0;
    uint8_t bDsfid = 0;
    uint8_t bMaskLength = 0;

    uint8_t aAts[256];
    uint8_t aMask[PHPAL_SLI15693_UID_LENGTH];

    /* Reset the field */
    PH_CHECK_SUCCESS(phhalHw_FieldReset(pHal_Generic));

    PH_CHECK_SUCCESS(phhalHw_Wait(pHal_Generic, PHHAL_HW_TIME_MILLISECONDS, 5));

    /* Perform ISO14443 Layer 3 / 4 activation. */
    if((bIsoLayer == ISO14443_L3) || (bIsoLayer == ISO14443_L4))
    {
        /* Configure Hardware for ISO14443 tag. */
        CHECK_SUCCESS_PRINT(phhalHw_ApplyProtocolSettings(pHal_Generic, PHHAL_HW_CARDTYPE_ISO14443A));

        /* Activate Layer 3 card. In loop till a card is detected. */
        do
        {
            wStatus = phpalI14443p3a_ActivateCard(pPalIso14443_3A_Generic, NULL, 0x00, aUid, &bUid_Len, &bSak, &bMoreCardsAvailable);

            if((wStatus != PH_ERR_SUCCESS) && (!bCardPlaced))
            {
                printf("Place MIFARE PICC card into field.\n");
                bCardPlaced = 1;
            }
        } while(wStatus != PH_ERR_SUCCESS);
        CHECK_SUCCESS_PRINT(wStatus);

        /* Activate L4 */
        if(bIsoLayer)
        {
            PH_CHECK_SUCCESS(phpalI14443p4a_ActivateCard(pPalIso14443_4A_Generic, bFsdi, bCid, PHPAL_I14443P4A_DATARATE_106, PHPAL_I14443P4A_DATARATE_106, aAts));

            /* Retrieve the ISO14443-4A protocol parameters. */
            PH_CHECK_SUCCESS(phpalI14443p4a_GetProtocolParams(pPalIso14443_4A_Generic, &bCidEnabled, &bCid, &bNadSupported, &bFwi,
                &bFsdi, &bFsci));

            /* Set the ISO14443-4 protocol parameters. */
            if(eReaderMode == S_MODE)
            {
                PH_CHECK_SUCCESS(phpalI14443p4_SetProtocol(pPalIso14443_4_Generic, bCidEnabled, bCid, bNadSupported, 0, bFwi, bFsdi, bFsci));
            }
        }
    }

    /* Perform ISO1569 activation. */
    else
    {
        /* Configure Hardware for ISO14443 tag. */
        CHECK_SUCCESS_PRINT(phhalHw_ApplyProtocolSettings(pHal_Generic, PHHAL_HW_CARDTYPE_ISO15693));

        PH_CHECK_SUCCESS(phpalSli15693_ActivateCard(pPalIso15693, PHPAL_SLI15693_ACTIVATE_ADDRESSED, (PHPAL_SLI15693_FLAG_DATA_RATE |
            PHPAL_SLI15693_FLAG_INVENTORY | PHPAL_SLI15693_FLAG_NBSLOTS), bAfi, aMask, bMaskLength, &bDsfid, aUid, &bMoreCardsAvailable));
    }

    return PH_ERR_SUCCESS;
}


phStatus_t MFD_Format()
{
    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Frame Format command information.  */
    aCmdBuf[0] = 0xFC;

    CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, 1, &pPicc_Response, &wPicc_RespLen));
    CHECK_SUCCESS_PICC(pPicc_Response[0], 0x00);

    return PH_ERR_SUCCESS;
}

phStatus_t MFD_Authenticate(uint8_t bAuthType, uint8_t bKeyNo, uint8_t bKeyVer)
{
    uint8_t aPcdCapIn[6] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
    uint8_t aPcdCapOut[6];
    uint8_t aPdCapOut[6];

    uint8_t bOption = bAuthType;

    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Update Option for EV2 type. */
    if((bOption == MFD_AUTH_TYPE_EV2_FIRST) || (bOption == MFD_AUTH_TYPE_EV2_NON_FIRST))
        bOption |= PHHAL_HW_CMD_SAMAV3_KDF_AV2;

    /* Authenticate in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Frame DESFire EV2 PICC part 1 Auth command. */
        aCmdBuf[wCmdLen++] = (uint8_t) ((bAuthType == MFD_AUTH_TYPE_D40) ? 0x0A : (bAuthType == MFD_AUTH_TYPE_EV2_FIRST) ? 0x71 : 0x77);
        aCmdBuf[wCmdLen++] = 0x00; /* KeyNo of the Card. */
        if((bAuthType == MFD_AUTH_TYPE_EV2_FIRST))
        {
            aCmdBuf[wCmdLen++] = 0x06; /* PCD Cap length. */
            memcpy(&aCmdBuf[wCmdLen], aPcdCapIn, sizeof(aPcdCapIn));
            wCmdLen += (uint8_t) sizeof(aPcdCapIn);
        }

        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, (uint8_t) wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC(pPicc_Response[0], 0xAF);

        /* Exchange the PICC response to Sam. */
        CHECK_SUCCESS_CHAINING(phhalHw_SamAV3_Cmd_SAM_AuthenticatePICC_Part1(&stHal_SamAv3, bOption, bKeyNo, bKeyVer, MFD_AUTH_TYPE_EV2, NULL, 0,
            &pPicc_Response[1], (uint8_t) (wPicc_RespLen - 1), &pSam_Response, &wSam_RespLen));

        /* Frame DESFire EV2 PICC part 2 Auth command. */
        wCmdLen = 0;
        aCmdBuf[wCmdLen++] = 0xAF;
        memcpy(&aCmdBuf[wCmdLen], pSam_Response, wSam_RespLen);
        wCmdLen += (uint8_t) wSam_RespLen;
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, (uint8_t) wCmdLen, &pPicc_Response, &wPicc_RespLen));

        /* Exchange the PICC response to Sam. */
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_AuthenticatePICC_Part2(&stHal_SamAv3, pPicc_Response[0], &pPicc_Response[1], (uint8_t) (wPicc_RespLen - 1), aPdCapOut, aPcdCapOut,
            &bPiccErrCode));
        CHECK_SUCCESS_PICC(pPicc_Response[0], 0x00);
    }

    /* Authenticate in X mode. */
    else
    {
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_DESFire_AuthenticatePICC(&stHal_SamAv3, bOption, PHHAL_HW_CMD_SAMAV3_ISO_MODE_NATIVE, 0x00 /* DF KeyNo */, bKeyNo,
            bKeyVer, (uint8_t) sizeof(aPcdCapIn), aPcdCapIn, NULL, 0, aPdCapOut, aPcdCapOut, &bPiccErrCode));
    }
    return PH_ERR_SUCCESS;
}

phStatus_t MFD_ChangeKey(uint8_t bCurKeyNo, uint8_t bCurKeyVer, uint8_t bNewKeyNo, uint8_t bNewKeyVer)
{
    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Authenticate in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Exchange the PICC response to Sam. */
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_ChangeKeyPICC(&stHal_SamAv3, PHHAL_HW_CMD_SAMAV3_CRYPTO_MODE_SAME_KEY, PHHAL_HW_CMD_SAMAV3_MASTER_KEY_UPDATE_INCLUDE_KEYTYPE,
            0x00, 0x00, bCurKeyNo, bCurKeyVer, bNewKeyNo, bNewKeyVer, NULL, 0, &pSam_Response, &wSam_RespLen));

        /* Frame DESFire ChangeKey command and exchange it to PICC. */
        aCmdBuf[wCmdLen++] = 0xC4;  aCmdBuf[wCmdLen++] = 0x00;
        memcpy(&aCmdBuf[wCmdLen], pSam_Response, wSam_RespLen); wCmdLen += (uint8_t) wSam_RespLen;
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, (uint8_t) wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC(pPicc_Response[0], 0x00);
    }

    /* Authenticate in X mode. */
    else
    {
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_DESFire_ChangeKeyPICC(&stHal_SamAv3, PHHAL_HW_CMD_SAMAV3_CRYPTO_MODE_SAME_KEY, PHHAL_HW_CMD_SAMAV3_MASTER_KEY_UPDATE_INCLUDE_KEYTYPE,
            0x00, 0x00, bCurKeyNo, bCurKeyVer, bNewKeyNo, bNewKeyVer, NULL, 0, aPICCError));
    }
    return PH_ERR_SUCCESS;
}

phStatus_t MFD_GetVersion()
{
    phStatus_t wStatus;
    uint8_t aVersion[40];
    uint8_t bVerLen = 0;

    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Frame GetVersion command information.  */
    aCmdBuf[0] = 0x60;

    /* Authenticate in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Frame DESFire GetVersion command and exchange it to PICC. */
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, 1, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC(pPicc_Response[0], 0xAF);
        memcpy(&aVersion[bVerLen], &pPicc_Response[1], (wPicc_RespLen - 1));
        bVerLen += (uint8_t) (wPicc_RespLen - 1);

        aCmdBuf[0] = 0xAF;
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, 1, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC(pPicc_Response[0], 0xAF);
        memcpy(&aVersion[bVerLen], &pPicc_Response[1], (wPicc_RespLen - 1));
        bVerLen += (uint8_t) (wPicc_RespLen - 1);

        aCmdBuf[0] = 0xAF;
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, 1, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC(pPicc_Response[0], 0x00);
        memcpy(&aVersion[bVerLen], &pPicc_Response[1], (wPicc_RespLen - 1));
        bVerLen += (uint8_t) (wPicc_RespLen - 1);
    }

    /* Authenticate in X mode. */
    else
    {
        do
        {
            wStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, 0x00, aCmdBuf, 1, &pSam_Response, &wSam_RespLen, aPICCError, &bPiccErrLen);
            if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
            {
                return wStatus;
            }

            memcpy(&aVersion[bVerLen], pSam_Response, wSam_RespLen);
            bVerLen += (uint8_t) wSam_RespLen;

            aCmdBuf[0] = 0xAF;
        } while(wStatus != PH_ERR_SUCCESS);
    }

    PrintData(aVersion, bVerLen, "%02X", "");

    return PH_ERR_SUCCESS;
}

phStatus_t MFD_CreateSelectApplication(uint8_t bKeyType, uint8_t * pAID)
{
    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Frame Create Application command information.  */
    aCmdBuf[wCmdLen++] = 0xCA;
    aCmdBuf[wCmdLen++] = pAID[0];
    aCmdBuf[wCmdLen++] = pAID[0];
    aCmdBuf[wCmdLen++] = pAID[0];
    aCmdBuf[wCmdLen++] = 0xEF;
    aCmdBuf[wCmdLen++] = (uint8_t) (bKeyType | 0x05);

    CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, (uint8_t) wCmdLen, &pPicc_Response, &wPicc_RespLen));
    CHECK_SUCCESS_PICC(pPicc_Response[0], 0x00);

    /* Frame Select Application command information.  */
    wCmdLen = 0;
    aCmdBuf[wCmdLen++] = 0x5A;
    aCmdBuf[wCmdLen++] = pAID[0];
    aCmdBuf[wCmdLen++] = pAID[0];
    aCmdBuf[wCmdLen++] = pAID[0];

    CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, (uint8_t) wCmdLen, &pPicc_Response, &wPicc_RespLen));
    CHECK_SUCCESS_PICC(pPicc_Response[0], 0x00);

    return PH_ERR_SUCCESS;
}

phStatus_t MFD_CreateFile()
{
    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Frame Create Standard File command information.  */
    aCmdBuf[wCmdLen++] = 0xCD;
    aCmdBuf[wCmdLen++] = 0x01;
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0xE0;
    aCmdBuf[wCmdLen++] = 0x0F;
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x04;
    aCmdBuf[wCmdLen++] = 0x00;

    CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, (uint8_t) wCmdLen, &pPicc_Response, &wPicc_RespLen));
    CHECK_SUCCESS_PICC(pPicc_Response[0], 0x00);

    return PH_ERR_SUCCESS;
}

phStatus_t MFD_Write()
{
    uint8_t aData[] = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10 };

    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Frame Write command information.  */
    aCmdBuf[wCmdLen++] = 0x3D;
    aCmdBuf[wCmdLen++] = 0x01;
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = (uint8_t) sizeof(aData);
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x00;
    memcpy(&aCmdBuf[wCmdLen], aData, sizeof(aData));
    wCmdLen += (uint8_t) sizeof(aData);

    PrintData(aData, sizeof(aData), "%02X ", "");

    /* Authenticate in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Frame DESFire Write command and exchange it to PICC. */
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC(pPicc_Response[0], 0x00);
    }

    /* Authenticate in X mode. */
    else
    {
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_DESFire_WriteX(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, 0x00, aCmdBuf, (uint8_t) wCmdLen,
            aPICCError, &bPiccErrLen));
    }

    return PH_ERR_SUCCESS;
}

phStatus_t MFD_Read()
{
    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Frame Read command information.  */
    aCmdBuf[wCmdLen++] = 0xBD;
    aCmdBuf[wCmdLen++] = 0x01;
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x10;
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x00;

    /* Authenticate in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Frame DESFire Read command and exchange it to PICC. */
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, (uint8_t) wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC(pPicc_Response[0], 0x00);

        pPicc_Response++;
        wPicc_RespLen--;
    }

    /* Authenticate in X mode. */
    else
    {
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_DESFire_ReadX(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, 0x00, aCmdBuf, (uint8_t) wCmdLen,
            &pPicc_Response, &wPicc_RespLen, aPICCError, &bPiccErrLen));
    }

    PrintData(pPicc_Response, wPicc_RespLen, "%02X ", "");

    return PH_ERR_SUCCESS;
}


phStatus_t PRIME_TMAC(uint8_t bAuthType)
{
    /* Authenticate the Application. */
    printf("\tAuthenticate the Application (APP_03 Key)       : ");
    CHECK_SUCCESS(PRIME_Authenticate((bAuthType ? MFD_AUTH_TYPE_LRP : MFD_AUTH_TYPE_EV2), 0x03, PICC_AES128_KEY_ADDRESS, PICC_AES128_KEY_VERSION));
    printf("SUCCESS\n");

    /* Perform Write Command. */
    printf("\tWrite data                                      : ");
    CHECK_SUCCESS(PRIME_Write());
    printf("\t\tSUCCESS\n");

    /* Perform Read Command. */
    printf("\tRead data                                       : ");
    CHECK_SUCCESS(PRIME_Read(STD_DATA_FILE_256_DFL));
    printf("\t\tSUCCESS\n");

    /* Authenticate the Application. */
    printf("\tAuthenticate the Application (APP_01 Key)       : ");
    CHECK_SUCCESS(PRIME_Authenticate((bAuthType ? MFD_AUTH_TYPE_LRP : MFD_AUTH_TYPE_EV2), 0x01, PICC_AES128_KEY_ADDRESS, PICC_AES128_KEY_VERSION));
    printf("SUCCESS\n\n");

    /* Perform CommitReaderID Command. */
    printf("\tCommitReaderID\n");
    CHECK_SUCCESS(PRIME_CommitReaderID());
    printf("\t\tStatus                                  : SUCCESS\n\n");

    /* Decrypt the Reader ID. */
    printf("\tDecryptReaderID\n");
    CHECK_SUCCESS(PRIME_DecryptReaderID(bAuthType));
    printf("\n");

    /* Authenticate the Application. */
    printf("\tAuthenticate the Application (APP_01 Key)       : ");
    CHECK_SUCCESS(PRIME_Authenticate((bAuthType ? MFD_AUTH_TYPE_LRP : MFD_AUTH_TYPE_EV2), 0x01, PICC_AES128_KEY_ADDRESS, PICC_AES128_KEY_VERSION));
    printf("SUCCESS\n");

    /* Perform CommitTransaction Command. */
    printf("\tCommitTransaction                               : ");
    CHECK_SUCCESS(PRIME_CommitTransaction());
    printf("\t\t\t\tSUCCESS\n\n");

    /* Compute the TMV. */
    printf("\tCalculateTMV\n");
    CHECK_SUCCESS(PRIME_CalculateTMV(bAuthType));

    return PH_ERR_SUCCESS;
}

phStatus_t PRIME_SDM(uint8_t bAuthType)
{
    /* Authenticate the Application. */
    printf("\tAuthenticate the Application (APP MASTER Key)   : ");
    CHECK_SUCCESS(PRIME_Authenticate((bAuthType ? MFD_AUTH_TYPE_LRP : MFD_AUTH_TYPE_EV2), 0x00, PICC_AES128_KEY_ADDRESS, PICC_AES128_KEY_VERSION));
    printf("SUCCESS\n");

    /* Perform ChangeFileSettings Command to enable SDM functionality. */
    printf("\tChangeFileSettings                              : ");
    CHECK_SUCCESS(PRIME_ChangeFileSettings(NTAG_42X_DNA_CFS_SDM));
    printf("SUCCESS\n\n");

    printf("\tPICC Activation                                 : ");
    CHECK_SUCCESS_PRINT(ActivateCard(ISO14443_L4));
    printf("SUCCESS\n");

    /* Select the PICC. */
    printf("\tPICC Selection                                  : ");
    CHECK_SUCCESS(PRIME_ISOSelect(NULL, 0, PICC_DF_NAME_NTAG42X, 7, 4));
    printf("SUCCESS\n");

    /* Select the Application. */
    printf("\tApplication Selection                           : ");
    CHECK_SUCCESS(PRIME_ISOSelect(APP_ISO_FILE_ID_NTAG42X, 0, APP_DF_NAME_NTAG42X, 7, 4));
    printf("SUCCESS\n");

    /* Perform Read Command. */
    printf("\tRead data                                       : ");
    CHECK_SUCCESS(PRIME_Read(STD_DATA_FILE_256_NTAG42X));

    printf("\tDecryption of SDM PICC Data: \n");
    CHECK_SUCCESS(PRIME_DecryptSDMPICCData(bAuthType));
    printf("\n");

    printf("\tDecryption of SDMENC File Data: \n");
    CHECK_SUCCESS(PRIME_DecryptSDMENCFileData(bAuthType));
    printf("\n");

    printf("\tSDMMAC computation and verification: \n");
    CHECK_SUCCESS(PRIME_CalculateSDMMAC(bAuthType));
    printf("\n");

    return PH_ERR_SUCCESS;
}

phStatus_t PRIME_Iso7816Wrap(uint8_t bLePresent, uint8_t * pPlainData, uint16_t * pDataLen)
{
    uint8_t     PH_MEMLOC_REM bCmdCode = 0;
    uint8_t     PH_MEMLOC_REM bLC = 0;

    /* Extract the command code. */
    bCmdCode = pPlainData[0];

    /* Set the LC value */
    bLC = (uint8_t) (*pDataLen - 1 /* Removing the command code. */);

    /* Copy the data to the Pointer. */
    memmove(&pPlainData[5], &pPlainData[1], bLC);   /* PRQA S 3200 */

    /* Reset the length buffer. */
    *pDataLen = 0;

    /* Frame the initial ISO7816 header. */
    pPlainData[(*pDataLen)++] = 0x90;
    pPlainData[(*pDataLen)++] = bCmdCode;
    pPlainData[(*pDataLen)++] = 0x00;
    pPlainData[(*pDataLen)++] = 0x00;

    /* Add LC if there is data. */
    if(bLC)
    {
        pPlainData[(*pDataLen)++] = bLC;

        /* Update Frame length based on LC. */
        *pDataLen = *pDataLen + bLC;
    }

    if(bLePresent)
    {
        pPlainData[(*pDataLen)++] = 0x00;
    }

    return PH_ERR_SUCCESS;
}

phStatus_t PRIME_ISOSelect(uint8_t* pISOFileID, uint8_t bISOFileID_Len, uint8_t* pDFName, uint8_t bDFName_Len, uint8_t bSelector)
{
    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Frame ISOSelect command information.  */
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0xA4;
    aCmdBuf[wCmdLen++] = bSelector;
    aCmdBuf[wCmdLen++] = 0x0C;
    aCmdBuf[wCmdLen++] = (uint8_t) (bDFName_Len | bISOFileID_Len);

    memcpy(&aCmdBuf[wCmdLen], pDFName, bDFName_Len);
    wCmdLen += bDFName_Len;

    memcpy(&aCmdBuf[wCmdLen], pISOFileID, bISOFileID_Len);
    wCmdLen += bISOFileID_Len;

    aCmdBuf[wCmdLen++] = 0x00;

    CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
    CHECK_SUCCESS_PICC_EXTENDED(((pPicc_Response[0] << 8) | pPicc_Response[1]), 0x9000);

    return PH_ERR_SUCCESS;
}

phStatus_t PRIME_Authenticate(uint8_t bAuthType, uint8_t bKeyNoCard, uint8_t bKeyNo, uint8_t bKeyVer)
{
    uint8_t aPcdCapIn[6] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
    uint8_t aPcdCapOut[6];
    uint8_t aPdCapOut[6];
    uint16_t wPicc_RespOffset = 0;

    /* Reset the command buffer. */
    wCmdLen = 0;

    aPcdCapIn[0] = (uint8_t) (bAuthType ? 0x02 : 0x00);

    /* Authenticate in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Frame DESFire EV2 PICC part 1 Auth command. */
        aCmdBuf[wCmdLen++] = 0x71; /* EV2 First Auth. */
        aCmdBuf[wCmdLen++] = bKeyNoCard; /* KeyNo of the Card. */

        /* Add PCD Capability info. */
        aCmdBuf[wCmdLen++] = 0x06; /* PCD Cap len. */
        memcpy(&aCmdBuf[wCmdLen], aPcdCapIn, sizeof(aPcdCapIn));
        wCmdLen += (uint8_t) sizeof(aPcdCapIn);

        /* Perform ISO 7816 Wrapping. */
        CHECK_SUCCESS(PRIME_Iso7816Wrap(PH_ON, aCmdBuf, &wCmdLen));

        /* Exchange the information to PICC. */
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC_EXTENDED(((pPicc_Response[wPicc_RespLen - 2] << 8) | pPicc_Response[wPicc_RespLen - 1]), 0x91AF);

        /* Set the Offset to use. */
        wPicc_RespOffset = (uint16_t) ((bAuthType == PHHAL_HW_CMD_SAMAV3_AUTH_MODE_LRP) ? 1 : 0);

        /* Exchange the PICC response to Sam. */
        CHECK_SUCCESS_CHAINING(phhalHw_SamAV3_Cmd_SAM_AuthenticatePICC_Part1(&stHal_SamAv3, PHHAL_HW_CMD_SAMAV3_AUTH_MODE_EV2_FIRST_AUTH, bKeyNo, bKeyVer,
            bAuthType, NULL, 0, &pPicc_Response[wPicc_RespOffset], (uint8_t) (wPicc_RespLen - (2 /* PICC Status code */ + wPicc_RespOffset)),
            &pSam_Response, &wSam_RespLen));

        /* Frame DESFire EV2 PICC part 2 Auth command. */
        wCmdLen = 0;
        aCmdBuf[wCmdLen++] = 0xAF;
        memcpy(&aCmdBuf[wCmdLen], pSam_Response, wSam_RespLen);
        wCmdLen += (uint8_t) wSam_RespLen;

        /* Perform ISO 7816 Wrapping. */
        CHECK_SUCCESS(PRIME_Iso7816Wrap(PH_ON, aCmdBuf, &wCmdLen));

        /* Exchange the information to PICC. */
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC_EXTENDED(((pPicc_Response[wPicc_RespLen - 2] << 8) | pPicc_Response[wPicc_RespLen - 1]), 0x9100);

        /* Exchange the PICC response to Sam. */
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_AuthenticatePICC_Part2(&stHal_SamAv3, pPicc_Response[wPicc_RespLen - 1], &pPicc_Response[0],
            (uint8_t) (wPicc_RespLen - 2), aPdCapOut, aPcdCapOut, &bPiccErrCode));
        CHECK_SUCCESS_PICC(bPiccErrCode, 0x00);
    }

    /* Authenticate in X mode. */
    else
    {
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_DESFire_AuthenticatePICC(&stHal_SamAv3, PHHAL_HW_CMD_SAMAV3_AUTH_MODE_EV2_FIRST_AUTH,
            PHHAL_HW_CMD_SAMAV3_ISO_MODE_ISO7816, bKeyNoCard, bKeyNo, bKeyVer, (uint8_t) sizeof(aPcdCapIn), aPcdCapIn, NULL, 0,
            aPdCapOut, aPcdCapOut, &bPiccErrCode));
    }
    return PH_ERR_SUCCESS;
}

phStatus_t PRIME_Write()
{
    uint8_t aData[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F };

    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Frame Write command information.  */
    aCmdBuf[wCmdLen++] = 0x8D;
    aCmdBuf[wCmdLen++] = STD_DATA_FILE_256_DFL;
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = (uint8_t) sizeof(aData);
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x00;

    /* Frame the TMI. */
    memcpy(&aTMI[bTMI_Len], aCmdBuf, wCmdLen);
    bTMI_Len += (uint8_t) wCmdLen;
    memset(&aTMI[bTMI_Len], 0x00, 8);
    bTMI_Len += 8;

    memcpy(&aCmdBuf[wCmdLen], aData, sizeof(aData));
    wCmdLen += (uint16_t) sizeof(aData);

    /* Frame the TMI. */
    memcpy(&aTMI[bTMI_Len], aData, sizeof(aData));
    bTMI_Len += (uint8_t) sizeof(aData);

    PrintData(aData, sizeof(aData), "%02X ", "");

    /* Authenticate in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Apply EV2 Secure Messaging. */
        wCmdLen = (uint16_t) (wCmdLen - sizeof(aData));
        CHECK_SUCCESS(phhalHw_SamAV3_Cmd_SAM_ApplySM(&stHal_SamAv3, (uint16_t) (PH_EXCHANGE_BUFFER_FIRST | PHHAL_HW_SAMAV3_CMD_APPLY_SM_INCLUDE_OFFSET),
            PHHAL_HW_SAMAV3_CMD_APPLY_REMOVE_SM_COMM_MODE_FULL, (uint8_t) wCmdLen, 0x00, aCmdBuf, (uint8_t) wCmdLen, &pSam_Response, &wSam_RespLen));
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_ApplySM(&stHal_SamAv3, PH_EXCHANGE_BUFFER_LAST, 0x00, 0x00, 0x00, aData, (uint8_t) sizeof(aData),
            &pSam_Response, &wSam_RespLen));

        memcpy(&aCmdBuf[wCmdLen], pSam_Response, wSam_RespLen);
        wCmdLen += (uint8_t) wSam_RespLen;

        /* Perform ISO 7816 Wrapping. */
        CHECK_SUCCESS(PRIME_Iso7816Wrap(PH_ON, aCmdBuf, &wCmdLen));

        /* Frame DESFire Write command and exchange it to PICC. */
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC_EXTENDED(((pPicc_Response[wPicc_RespLen - 2] << 8) | pPicc_Response[wPicc_RespLen - 1]), 0x9100);

        /* Remove EV2 Secure Messaging. */
        pSam_Response = NULL;
        wSam_RespLen = 0;
        CHECK_SUCCESS(phhalHw_SamAV3_Cmd_SAM_RemoveSM(&stHal_SamAv3, PH_EXCHANGE_BUFFER_FIRST, PHHAL_HW_SAMAV3_CMD_APPLY_REMOVE_SM_COMM_MODE_MAC,
            &pPicc_Response[wPicc_RespLen - 1], 1, NULL, NULL));
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_RemoveSM(&stHal_SamAv3, PH_EXCHANGE_BUFFER_LAST, 0x00, pPicc_Response, (uint8_t) (wPicc_RespLen - 2),
            &pSam_Response, &wSam_RespLen));
    }

    /* Authenticate in X mode. */
    else
    {
        /* Perform ISO 7816 Wrapping. */
        CHECK_SUCCESS(PRIME_Iso7816Wrap(PH_ON, aCmdBuf, &wCmdLen));

        /* Exchange the information to SAM */
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_DESFire_WriteX(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, 0x00, aCmdBuf, (uint8_t) wCmdLen, aPICCError, &bPiccErrLen));
    }


    return PH_ERR_SUCCESS;
}

phStatus_t PRIME_Read(uint8_t bFileNo)
{
    uint16_t wIteration = 0;

    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Frame Read command information.  */
    aCmdBuf[wCmdLen++] = 0xAD;
    aCmdBuf[wCmdLen++] = bFileNo;
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = (uint8_t) ((bFileNo == STD_DATA_FILE_256_DFL) ? 0x10 : 0x00);
    aCmdBuf[wCmdLen++] = (uint8_t) ((bFileNo == STD_DATA_FILE_256_NTAG42X) ? 0x01 : 0x00);
    aCmdBuf[wCmdLen++] = 0x00;

    if(bFileNo == STD_DATA_FILE_256_DFL)
    {
        /* Frame the TMI. */
        memcpy(&aTMI[bTMI_Len], aCmdBuf, wCmdLen);
        bTMI_Len += (uint8_t) wCmdLen;
        memset(&aTMI[bTMI_Len], 0x00, 8);
        bTMI_Len += 8;
    }

    /* Authenticate in S mode. */
    if(eReaderMode == S_MODE)
    {
        if(bFileNo != STD_DATA_FILE_256_NTAG42X)
        {
            /* Apply EV2 Secure Messaging. */
            CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_ApplySM(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, PHHAL_HW_SAMAV3_CMD_APPLY_REMOVE_SM_COMM_MODE_MAC,
                0x00, 0x00, aCmdBuf, (uint8_t) wCmdLen, &pSam_Response, &wSam_RespLen));

            memcpy(&aCmdBuf[wCmdLen], pSam_Response, wSam_RespLen);
            wCmdLen += (uint8_t) wSam_RespLen;
        }

        /* Perform ISO 7816 Wrapping. */
        CHECK_SUCCESS(PRIME_Iso7816Wrap(PH_ON, aCmdBuf, &wCmdLen));

        /* Frame DESFire Read command and exchange it to PICC. */
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC_EXTENDED(((pPicc_Response[wPicc_RespLen - 2] << 8) | pPicc_Response[wPicc_RespLen - 1]), 0x9100);

        /* Remove EV2 Secure Messaging. */
        if(bFileNo != STD_DATA_FILE_256_NTAG42X)
        {
            pSam_Response = NULL;
            wSam_RespLen = 0;
            CHECK_SUCCESS(phhalHw_SamAV3_Cmd_SAM_RemoveSM(&stHal_SamAv3, PH_EXCHANGE_BUFFER_FIRST, PHHAL_HW_SAMAV3_CMD_APPLY_REMOVE_SM_COMM_MODE_FULL,
                &pPicc_Response[wPicc_RespLen - 1], 1, NULL, NULL));
            CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_RemoveSM(&stHal_SamAv3, PH_EXCHANGE_BUFFER_LAST, 0x00, pPicc_Response, (uint8_t) (wPicc_RespLen - 2),
                &pSam_Response, &wSam_RespLen));
        }
    }

    /* Authenticate in X mode. */
    else
    {
        /* Perform ISO 7816 Wrapping. */
        CHECK_SUCCESS(PRIME_Iso7816Wrap(PH_ON, aCmdBuf, &wCmdLen));

        /* Exchange the information to SAM */
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_DESFire_ReadX(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, 0x00, aCmdBuf, (uint8_t) wCmdLen, &pPicc_Response,
            &wPicc_RespLen, aPICCError, &bPiccErrLen));
    }

    if(bFileNo == STD_DATA_FILE_256_DFL)
    {
        PrintData(pSam_Response, wSam_RespLen, "%02X ", "");

        /* Frame the TMI. */
        memcpy(&aTMI[bTMI_Len], pSam_Response, wSam_RespLen);
        bTMI_Len += (uint8_t) wSam_RespLen;
    }
    else if(bFileNo == STD_DATA_FILE_256_NTAG42X)
    {
        printf("SUCCESS\n");

        printf("                                                        : ");
        for(wIteration = 0; wIteration < (uint16_t) ((wPicc_RespLen - 2) / 32); wIteration++)
        {
            PrintData(&pPicc_Response[wIteration * 32], 32, "%02X ", "\n");
            if(wIteration < (uint16_t) (((wPicc_RespLen - 2) / 32) - 1))
                printf("                                                        : ");
        }
    }
    else
    {
        memcpy(aTMC, pSam_Response, 4);
        memcpy(aTMV, &pSam_Response[4], 8);
    }

    return PH_ERR_SUCCESS;
}

phStatus_t PRIME_CommitReaderID()
{
    phStatus_t wStatus = 0;

    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Frame CommitReaderID command information.  */
    aCmdBuf[wCmdLen++] = 0xC8;

    /* Authenticate in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Excahgne the information to SAM. */
        CHECK_SUCCESS_CHAINING(phhalHw_SamAV3_Cmd_SAM_CommitReaderID_Part1(&stHal_SamAv3, PHHAL_HW_CMD_SAMAV3_COMMIT_READER_ID_PICC_STATE_DESFIRE,
            0x00, &pSam_Response, &wSam_RespLen));

        /* Print the TMRI. */
        printf("\t\tTMRI                                    : ");
        PrintData(pSam_Response, 16, "%02X ", "");
        printf("\n");

        /* Frame the TMI. */
        memcpy(&aTMI[bTMI_Len], aCmdBuf, wCmdLen);
        bTMI_Len += (uint8_t) wCmdLen;
        memcpy(&aTMI[bTMI_Len], pSam_Response, 16);
        bTMI_Len += 16;

        memcpy(&aCmdBuf[wCmdLen], pSam_Response, wSam_RespLen);
        wCmdLen += (uint8_t) wSam_RespLen;

        /* Perform ISO 7816 Wrapping. */
        CHECK_SUCCESS(PRIME_Iso7816Wrap(PH_ON, aCmdBuf, &wCmdLen));

        /* Frame DESFire CommitReaderID command and exchange it to PICC. */
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC_EXTENDED(((pPicc_Response[wPicc_RespLen - 2] << 8) | pPicc_Response[wPicc_RespLen - 1]), 0x9100);

        /* Print the TMRI. */
        printf("\t\tEncTMRI                                 : ");
        PrintData(pPicc_Response, 16, "%02X ", "");
        printf("\n");

        /* Excahgne the information to SAM. */
        wStatus = phhalHw_SamAV3_Cmd_SAM_CommitReaderID_Part2(&stHal_SamAv3, pPicc_Response[wPicc_RespLen - 1], pPicc_Response,
            (uint8_t) (wPicc_RespLen - 2), &bPiccErrCode);
        CHECK_SUCCESS_PICC(bPiccErrCode, 0x00);

        /* Copy EncTMRI to global buffer. */
        memcpy(aEncTMRI, pPicc_Response, 16);

        /* Frame the TMI. */
        memcpy(&aTMI[bTMI_Len], aEncTMRI, 16);
        bTMI_Len += 16;
        memset(&aTMI[bTMI_Len], 0x00, 15);
        bTMI_Len += 15 /* Padding */;
    }

    /* Authenticate in X mode. */
    else
    {
        /* Perform ISO 7816 Wrapping. */
        CHECK_SUCCESS(PRIME_Iso7816Wrap(PH_ON, aCmdBuf, &wCmdLen));

        /* Exchange the information to SAM */
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_DESFire_ReadX(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, 0x00, aCmdBuf, (uint8_t) wCmdLen, &pPicc_Response,
            &wPicc_RespLen, aPICCError, &bPiccErrLen));
    }

    return PH_ERR_SUCCESS;
}

phStatus_t PRIME_DecryptReaderID(uint8_t bAuthType)
{
    uint8_t aSV[16];
    uint8_t aTmc[4];
    uint8_t bSvLen = 0;
    uint8_t bSvBak = 0;
    uint16_t wActTMC = 0;
    uint16_t wSesTMC = 0;
    uint32_t dwTMC = 0;

    /* Print the UID. */
    printf("\t\tUID                                     : ");
    PrintData(aUid, bUid_Len, "%02X ", "");
    printf("\n");

    /* Compute the SessionVector. ---------------------------------------------------------------------------------- */

    /* Read the Current TMC and TMV information. */
    CHECK_SUCCESS(PRIME_Read(TMAC_FILE_DFL));

    /* Increment the TMC by 1. */
    if(bAuthType == MFD_AUTH_TYPE_EV2)
    {
        dwTMC = (uint32_t) (aTMC[0] | (aTMC[1] << 8) | (aTMC[2] << 16) | (aTMC[3] << 24));
        dwTMC++;
    }
    else
    {
        wActTMC = (uint16_t) (aTMC[0] | (aTMC[1] << 8));
        wSesTMC = (uint16_t) (aTMC[2] | (aTMC[3] << 8));

        /*
         * The session TMC is not incremented as the whole transaction is
         * still performed using the current session.
         */
        wActTMC++;
        dwTMC = (uint32_t) ((wSesTMC << 16) | wActTMC);
    }

    aTmc[0] = (uint8_t) (dwTMC & 0xFF);
    aTmc[1] = (uint8_t) ((dwTMC & 0xFF00) >> 8);
    aTmc[2] = (uint8_t) ((dwTMC & 0xFF0000) >> 16);
    aTmc[3] = (uint8_t) ((dwTMC & 0xFF000000) >> 24);

    /* Print the Transaction MAC counter. */
    printf("\t\tTMC                                     : ");
    PrintData(aTmc, 4, "%02X ", "");
    printf("\n");

    /* Print the EncTMRI. */
    printf("\t\tEncTMRI                                 : ");
    PrintData(aEncTMRI, 16, "%02X ", "");
    printf("\n");

    /* Compute the session vector. */
    aSV[bSvLen++] = 0xA5;
    aSV[bSvLen++] = 0x00;
    aSV[bSvLen++] = 0x01;
    aSV[bSvLen++] = 0x00;
    aSV[bSvLen++] = 0x80;

    /* Append the TMC information. */
    memcpy(&aSV[bSvLen], aTmc, 4);
    bSvLen += 4;

    memcpy(&aSV[bSvLen], aUid, bUid_Len);
    bSvLen = 16;

    /* Rotate the SV information by 1 for LRP. */
    if(bAuthType == MFD_AUTH_TYPE_LRP)
    {
        bSvBak = aSV[0];
        memcpy(&aSV[0], &aSV[1], 15);   /* PRQA S 3200 */
        aSV[15] = bSvBak;
    }

    /* Exchange the session vector information to SAM. */
    CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_DeriveKey(&stHal_SamAv3, OFFLINE_CRYPTO_KEY_ADDRESS, OFFLINE_CRYPTO_KEY_VERSION,
        RAM_KEY_ADDRESS, aSV, bSvLen));

    /* Perform Offline activation using Ram Key. */
    CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(&stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
        RAM_KEY_ADDRESS, RAM_KEY_VERSION, NULL, 0));

    /* Exchange the TMI information to SAM. */
    CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_DecipherOfflineData(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, aEncTMRI, 16, &pSam_Response,
        &wSam_RespLen));

    /* Print the TMRIPrev. */
    printf("\t\tTMRIPrev                                : ");
    PrintData(pSam_Response, 16, "%02X ", "");
    printf("\n");

    return PH_ERR_SUCCESS;
}

phStatus_t PRIME_CommitTransaction()
{
    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Frame CommitTransaction command information.  */
    aCmdBuf[wCmdLen++] = 0xC7;
    aCmdBuf[wCmdLen++] = 0x01;

    /* Authenticate in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Apply EV2 Secure Messaging. */
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_ApplySM(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, PHHAL_HW_SAMAV3_CMD_APPLY_REMOVE_SM_COMM_MODE_MAC,
            0x00, 0x00, aCmdBuf, (uint8_t) wCmdLen, &pSam_Response, &wSam_RespLen));

        memcpy(&aCmdBuf[wCmdLen], pSam_Response, wSam_RespLen);
        wCmdLen += (uint8_t) wSam_RespLen;

        /* Perform ISO 7816 Wrapping. */
        CHECK_SUCCESS(PRIME_Iso7816Wrap(PH_ON, aCmdBuf, &wCmdLen));

        /* Frame DESFire CommitTransaction command and exchange it to PICC. */
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC_EXTENDED(((pPicc_Response[wPicc_RespLen - 2] << 8) | pPicc_Response[wPicc_RespLen - 1]), 0x9100);

        /* Remove EV2 Secure Messaging. */
        pSam_Response = NULL;
        wSam_RespLen = 0;
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_RemoveSM(&stHal_SamAv3, PH_EXCHANGE_BUFFER_FIRST, PHHAL_HW_SAMAV3_CMD_APPLY_REMOVE_SM_COMM_MODE_MAC,
            &pPicc_Response[wPicc_RespLen - 1], 1, NULL, NULL));
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_RemoveSM(&stHal_SamAv3, PH_EXCHANGE_BUFFER_LAST, 0x00, pPicc_Response, (uint8_t) (wPicc_RespLen - 2),
            &pSam_Response, &wSam_RespLen));
    }

    /* Authenticate in X mode. */
    else
    {
        /* Perform ISO 7816 Wrapping. */
        CHECK_SUCCESS(PRIME_Iso7816Wrap(PH_ON, aCmdBuf, &wCmdLen));

        /* Exchange the information to SAM */
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_DESFire_ReadX(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, 0x00, aCmdBuf, (uint8_t) wCmdLen, &pPicc_Response,
            &wPicc_RespLen, aPICCError, &bPiccErrLen));
    }

    PrintData(pPicc_Response, 12, "%02X ", "");

    return PH_ERR_SUCCESS;
}

phStatus_t PRIME_CalculateTMV(uint8_t bAuthType)
{
    uint8_t aSV[16];
    uint8_t aTmc[4];
    uint8_t bSvLen = 0;
    uint8_t bSvBak = 0;
    uint16_t wIteration = 0;
    uint16_t wActTMC = 0;
    uint16_t wSesTMC = 0;
    uint32_t dwTMC = 0;

    uint8_t bTotalIteration = 0;
    uint8_t bRemData = 0;
    uint8_t bDataLen = 0;

    /* Print the UID. */
    printf("\t\tUID                                     : ");
    PrintData(aUid, bUid_Len, "%02X ", "");
    printf("\n");

    /* Compute the SessionVector. ---------------------------------------------------------------------------------- */

    /* Read the Current TMC and TMV information. */
    CHECK_SUCCESS(PRIME_Read(TMAC_FILE_DFL));

    /* Increment the TMC by 1. */
    if(bAuthType == MFD_AUTH_TYPE_EV2)
        dwTMC = (uint32_t) (aTMC[0] | (aTMC[1] << 8) | (aTMC[2] << 16) | (aTMC[3] << 24));
    else
    {
        wActTMC = (uint16_t) (aTMC[0] | (aTMC[1] << 8));
        wSesTMC = (uint16_t) (aTMC[2] | (aTMC[3] << 8));
        dwTMC = (uint32_t) ((wSesTMC << 16) | wActTMC);
    }

    aTmc[0] = (uint8_t) (dwTMC & 0xFF);
    aTmc[1] = (uint8_t) ((dwTMC & 0xFF00) >> 8);
    aTmc[2] = (uint8_t) ((dwTMC & 0xFF0000) >> 16);
    aTmc[3] = (uint8_t) ((dwTMC & 0xFF000000) >> 24);

    /* Print the Transaction MAC counter. */
    printf("\t\tTMC                                     : ");
    PrintData(aTmc, 4, "%02X ", "");
    printf("\n");

    /* Print the TMI. */
    printf("\t\tTMI                                     : ");
    bTotalIteration = (uint8_t) (bTMI_Len / 32);
    bTotalIteration += (uint8_t) ((bTMI_Len / 32) ? 1 : 0);
    bRemData = bTMI_Len;
    bDataLen = 32;
    for(wIteration = 0; wIteration < bTotalIteration; wIteration++)
    {
        PrintData(&aTMI[wIteration * 32], bDataLen, "%02X ", "\n");

        bRemData -= 32;
        bDataLen = (uint8_t) ((bRemData < 32) ? bRemData : 32);

        if(wIteration < (uint16_t) (bTotalIteration - 1))
            printf("\t\t                                        : ");
    }

    /* Compute the session vector. */
    aSV[bSvLen++] = 0x5A;
    aSV[bSvLen++] = 0x00;
    aSV[bSvLen++] = 0x01;
    aSV[bSvLen++] = 0x00;
    aSV[bSvLen++] = 0x80;

    /* Append the TMC information. */
    memcpy(&aSV[bSvLen], aTmc, 4);
    bSvLen += 4;

    memcpy(&aSV[bSvLen], aUid, bUid_Len);
    bSvLen = 16;

    /* Rotate the SV information by 1 for LRP. */
    if(bAuthType == MFD_AUTH_TYPE_LRP)
    {
        bSvBak = aSV[0];
        memcpy(&aSV[0], &aSV[1], 15);   /* PRQA S 3200 */
        aSV[15] = bSvBak;
    }

    /* Exchange the session vector information to SAM. */
    CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_DeriveKey(&stHal_SamAv3, OFFLINE_CRYPTO_KEY_ADDRESS, OFFLINE_CRYPTO_KEY_VERSION,
        RAM_KEY_ADDRESS, aSV, bSvLen));

    /* Perform Offline activation using Ram Key. */
    CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(&stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
        RAM_KEY_ADDRESS, RAM_KEY_VERSION, NULL, 0));

    /* Exchange the TMI information to SAM. */
    CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_GenerateMAC(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, PHHAL_HW_SAMAV3_TRUNCATION_MODE_MFP,
        aTMI, bTMI_Len, &pSam_Response, &wSam_RespLen));

    /* Print the actual TMV. */
    printf("\t\tTMV Exp                                 : ");
    PrintData(aTMV, 8, "%02X ", "");
    printf("\n");

    /* Print the TMRIPrev. */
    printf("\t\tTMV Act                                 : ");
    PrintData(pSam_Response, wSam_RespLen, "%02X ", "");
    printf("\n");

    return PH_ERR_SUCCESS;
}

phStatus_t PRIME_ChangeFileSettings(uint8_t bType)
{
    uint16_t wCmdLen_Tmp = 0;

    /* Reset the command buffer. */
    wCmdLen = 0;

    if(eReaderMode == X_MODE_RC523)
    {
        aCmdBuf[wCmdLen++] = 0x05;
        aCmdBuf[wCmdLen++] = 0x00;
        aCmdBuf[wCmdLen++] = 0x00;
    }

    /* Frame ChangeFileSettings command information.  */
    aCmdBuf[wCmdLen++] = 0x5F;
    aCmdBuf[wCmdLen++] = STD_DATA_FILE_256_NTAG42X;
    aCmdBuf[wCmdLen++] = (uint8_t) (bType /* SDM */ | 0x03 /* FULL mode. */);
    aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x00;

    /* Frame SDM information. */
    if(bType == NTAG_42X_DNA_CFS_SDM)
    {
         /*
          * VCUID Mirroring Enabled, SDMRdCtr Mirroring Enabled,
          * SDMReadCtrLimit Enabled, SDMENCFileData Enabled, ASCII Enabled
          */
        aCmdBuf[wCmdLen++] = 0xF1;

        /*
         * Disabled, SDMReadCtr Free Access
         * SDMMetaDataRead - App Master Key, SDMFileRead - App Master Key
         */
        aCmdBuf[wCmdLen++] = 0xFE;
        aCmdBuf[wCmdLen++] = 0x00;

        memcpy(&aCmdBuf[wCmdLen], aPICCDataOffset, 3);
        wCmdLen += 3;

        memcpy(&aCmdBuf[wCmdLen], aSDMMACInputOffset, 3);
        wCmdLen += 3;

        memcpy(&aCmdBuf[wCmdLen], aSDMENCOffset, 3);
        wCmdLen += 3;

        memcpy(&aCmdBuf[wCmdLen], aSDMENCLen, 3);
        wCmdLen += 3;

        memcpy(&aCmdBuf[wCmdLen], aSDMMACOffset, 3);
        wCmdLen += 3;

        memcpy(&aCmdBuf[wCmdLen], aSDMReadCtrLimit, 3);
        wCmdLen += 3;
    }

    /* Authenticate in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Apply EV2 Secure Messaging. */
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_ApplySM(&stHal_SamAv3, (uint16_t) (PHHAL_HW_SAMAV3_CMD_APPLY_SM_INCLUDE_OFFSET | PH_EXCHANGE_DEFAULT),
            PHHAL_HW_SAMAV3_CMD_APPLY_REMOVE_SM_COMM_MODE_FULL, 0x02, 0x00, aCmdBuf, (uint8_t) wCmdLen, &pSam_Response, &wSam_RespLen));

        wCmdLen = 2;
        memcpy(&aCmdBuf[wCmdLen], pSam_Response, wSam_RespLen);
        wCmdLen += (uint8_t) wSam_RespLen;

        /* Perform ISO 7816 Wrapping. */
        CHECK_SUCCESS(PRIME_Iso7816Wrap(PH_ON, aCmdBuf, &wCmdLen));

        /* Frame ChangeFileSettings command and exchange it to PICC. */
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC_EXTENDED(((pPicc_Response[wPicc_RespLen - 2] << 8) | pPicc_Response[wPicc_RespLen - 1]), 0x9100);

        /* Remove EV2 Secure Messaging. */
        pSam_Response = NULL;
        wSam_RespLen = 0;
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_RemoveSM(&stHal_SamAv3, PH_EXCHANGE_BUFFER_FIRST, PHHAL_HW_SAMAV3_CMD_APPLY_REMOVE_SM_COMM_MODE_MAC,
            &pPicc_Response[wPicc_RespLen - 1], 1, NULL, NULL));
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_RemoveSM(&stHal_SamAv3, PH_EXCHANGE_BUFFER_LAST, 0x00, pPicc_Response, (uint8_t) (wPicc_RespLen - 2),
            &pSam_Response, &wSam_RespLen));
    }

    /* Authenticate in X mode. */
    else
    {
        wCmdLen_Tmp = (uint16_t) (wCmdLen - 3);

        /* Perform ISO 7816 Wrapping. */
        CHECK_SUCCESS(PRIME_Iso7816Wrap(PH_ON, &aCmdBuf[3], &wCmdLen_Tmp));

        /* Exchange the information to SAM */
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_DESFire_WriteX(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, (uint8_t) (PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_EXTENDED_OFFSET |
            PHHAL_HW_CMD_SAMAV3_CRYPTO_CONFIG_COMM_MODE_FULL), aCmdBuf, (uint8_t) wCmdLen_Tmp, aPICCError, &bPiccErrLen));
    }

    return PH_ERR_SUCCESS;
}

phStatus_t PRIME_DecryptSDMPICCData(uint8_t bAuthType)
{
    phStatus_t wStatus = 0;
    uint8_t bEncDataOffset = 0;
    uint8_t * pResponse = NULL;
    uint16_t wRespLen = 0;
    uint8_t aPICCDataRnd_Hex[20];
    uint8_t aPICCData_Hex[20];

    /* Perform Offline activation using Ram Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(&stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
        OFFLINE_CRYPTO_KEY_ADDRESS, OFFLINE_CRYPTO_KEY_VERSION, NULL, 0));

    /* Load the IV and update the Offset in case of LRP type. */
    if(bAuthType == MFD_AUTH_TYPE_LRP)
    {
        /* Convert ASCII to Hex. */
        ConvertAscii_Hex(pPicc_Response, 16, aPICCDataRnd_Hex);

        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_LoadInitVector(&stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_LOAD_IV_MODE_SET_LRP_ENC_CTR,
            aPICCDataRnd_Hex, 8));

        /* Update the Offset. */
        bEncDataOffset = 16;
    }

    /* Convert ASCII to Hex. */
    ConvertAscii_Hex(&pPicc_Response[bEncDataOffset], 32, aPICCData_Hex);

    /* Perform DecipherDataOffline. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DecipherOfflineData(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, aPICCData_Hex,
        16, &pResponse, &wRespLen));

    /* Extract SDMReadCtr. */
    memcpy(aSDMReadCtr, &pResponse[1 /* Constant C7 */ + 7 /* UID length */], 3);

    printf("\t\tPICC Data (ASCII)                       : ");
    PrintData(&pPicc_Response[bEncDataOffset], 32, "%02X ", "\n");

    printf("\t\tPICC Data (HEX)                         : ");
    PrintData(aPICCData_Hex, 16, "%02X ", "\n");

    printf("\t\tDecrypted PICC Data (HEX)               : ");
    PrintData(pResponse, wRespLen, "%02X ", "\n");

    return PH_ERR_SUCCESS;
}

phStatus_t PRIME_DecryptSDMENCFileData(uint8_t bAuthType)
{
    phStatus_t wStatus = 0;
    uint8_t aSV[16];
    uint8_t bSvLen = 0;
    uint32_t dwSDMReadCtr = 0;
    uint8_t bOption = 0;
    uint8_t aIV[16];
    uint8_t * pIV = NULL;
    uint16_t wIvLen = 0;
    uint8_t * pResponse = NULL;
    uint16_t wRespLen = 0;
    uint8_t aSDMENCData_Hex[20];
    uint32_t dwSDMENCOffset = 0;
    uint32_t dwSDMENCLen = 0;

    /* Frame SDMReadCtr. */
    dwSDMReadCtr = (uint32_t) (aSDMReadCtr[0] | (aSDMReadCtr[1] << 8) | (aSDMReadCtr[2] << 16));

    /* Clear SV and IV Buffer. */
    memset(aSV, 0x00, sizeof(aSV));   /* PRQA S 3200 */
    memset(aIV, 0x00, sizeof(aIV));   /* PRQA S 3200 */

    /* Frame the Session Vector. */
    if(bAuthType == MFD_AUTH_TYPE_LRP)
    {
        aSV[bSvLen++] = 0x00; aSV[bSvLen++] = 0x01; aSV[bSvLen++] = 0x00; aSV[bSvLen++] = 0x80;
        aSV[14] = 0x1E; aSV[15] = 0xE1;
    }
    else
    {
        aSV[bSvLen++] = 0xC3; aSV[bSvLen++] = 0x3C; aSV[bSvLen++] = 0x00; aSV[bSvLen++] = 0x01;
        aSV[bSvLen++] = 0x00; aSV[bSvLen++] = 0x80;
    }

    memcpy(&aSV[bSvLen], aUid, bUid_Len);   /* PRQA S 3200 */
    bSvLen += bUid_Len;

    aSV[bSvLen++] = (uint8_t) (dwSDMReadCtr & 0xFF);
    aSV[bSvLen++] = (uint8_t) ((dwSDMReadCtr & 0xFF00) >> 8);
    aSV[bSvLen++] = (uint8_t) ((dwSDMReadCtr & 0xFF0000) >> 16);

    /* Update the SV length to 16. */
    bSvLen = 16;

    /* Exchange the session vector information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DeriveKey(&stHal_SamAv3, OFFLINE_CRYPTO_KEY_ADDRESS, OFFLINE_CRYPTO_KEY_VERSION,
        RAM_KEY_ADDRESS, aSV, bSvLen));

    /* Perform Offline activation using Ram Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(&stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
        RAM_KEY_ADDRESS, RAM_KEY_VERSION, NULL, 0));

    /* Frame the IV. */
    if(bAuthType == MFD_AUTH_TYPE_LRP)
    {
        /* IV computation is SDMReadCtr || 0x000000 */

        memcpy(&aIV[wIvLen], aSDMReadCtr, 3);   /* PRQA S 3200 */
        wIvLen += 3;

        /* Update IVLen to allocate three zeros. */
        wIvLen += 3;

        /* Set the Option. */
        bOption = PHHAL_HW_SAMAV3_CMD_SAM_LOAD_IV_MODE_SET_LRP_ENC_CTR;
    }
    else
    {
        /* IV computation is E(KSesSDMFileReadENC; SDMReadCtr || 0x00000000000000000000000000) */
        memcpy(&aIV[wIvLen], aSDMReadCtr, 3);   /* PRQA S 3200 */

        /* Set the IV length to 16. */
        wIvLen = 16;

        /* Encrypt the IV. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_EncipherOfflineData(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, aIV, (uint8_t) wIvLen,
            &pIV, &wIvLen));

        /* Copy the enciphered data to local buffer. */
        memcpy(aIV, pIV, wIvLen);        /* PRQA S 3200 */
        pIV = NULL;

        /* Set the Option. */
        bOption = PHHAL_HW_SAMAV3_CMD_SAM_LOAD_IV_MODE_SET_IV;
    }

    /* Load the IV. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_LoadInitVector(&stHal_SamAv3, bOption, aIV, (uint8_t) wIvLen));

    /* Frame SDMENCData Offset and Length. */
    dwSDMENCLen = (uint32_t) (aSDMENCLen[0] | (aSDMENCLen[1] << 8) | (aSDMENCLen[2] << 16));
    dwSDMENCOffset = (uint32_t) (aSDMENCOffset[0] | (aSDMENCOffset[1] << 8) | (aSDMENCOffset[2] << 16));

    /* Convert ASCII to Hex. */
    ConvertAscii_Hex(&pPicc_Response[dwSDMENCOffset], dwSDMENCLen, aSDMENCData_Hex);

    /* Exchange the Encrypted information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DecipherOfflineData(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, aSDMENCData_Hex, 16,
        &pResponse, &wRespLen));

    printf("\t\tSDM Encrypted Data Offset               : ");
    PrintData(aSDMENCOffset, 3, "%02X ", "\n");

    printf("\t\tSDM Encrypted Data Length               : ");
    PrintData(aSDMENCLen, 3, "%02X ", "\n");

    printf("\t\tSDM Read Ctr                            : ");
    PrintData(aSDMReadCtr, 3, "%02X ", "\n");

    printf("\t\tSDM Encrypted Data (ASCII)              : ");
    PrintData(&pPicc_Response[dwSDMENCOffset], dwSDMENCLen, "%02X ", "\n");

    printf("\t\tSDM Encrypted Data (HEX)                : ");
    PrintData(aSDMENCData_Hex, 16, "%02X ", "\n");

    printf("\t\tDecrypted SDM Data                      : ");
    PrintData(pResponse, wRespLen, "%02X ", "\n");

    return PH_ERR_SUCCESS;
}

phStatus_t PRIME_CalculateSDMMAC(uint8_t bAuthType)
{
    phStatus_t wStatus = 0;
    uint8_t aSV[16];
    uint8_t bSvLen = 0;
    uint32_t dwSDMReadCtr = 0;
    uint8_t * pResponse = NULL;
    uint16_t wRespLen = 0;
    uint8_t aSDMMACData_Hex[256];
    uint32_t dwSDMMACInputOffset = 0;
    uint32_t dwSDMMACOffset = 0;
    uint8_t bFinished = PH_OFF;
    uint16_t wBuffOption = PH_EXCHANGE_DEFAULT;
    uint8_t bExchangeLen = 0;
    uint16_t wRemLen = 0;
    uint16_t wInputOffset = 0;

    /* Frame SDMReadCtr. */
    dwSDMReadCtr = (uint32_t) (aSDMReadCtr[0] | (aSDMReadCtr[1] << 8) | (aSDMReadCtr[2] << 16));

    /* Clear SV and IV Buffer. */
    memset(aSV, 0x00, sizeof(aSV));   /* PRQA S 3200 */

    /* Frame the Session Vector. */
    if(bAuthType == MFD_AUTH_TYPE_LRP)
    {
        aSV[bSvLen++] = 0x00; aSV[bSvLen++] = 0x01; aSV[bSvLen++] = 0x00; aSV[bSvLen++] = 0x80;
        aSV[14] = 0x1E; aSV[15] = 0xE1;
    }
    else
    {
        aSV[bSvLen++] = 0x3C; aSV[bSvLen++] = 0xC3; aSV[bSvLen++] = 0x00; aSV[bSvLen++] = 0x01;
        aSV[bSvLen++] = 0x00; aSV[bSvLen++] = 0x80;
    }

    memcpy(&aSV[bSvLen], aUid, bUid_Len);   /* PRQA S 3200 */
    bSvLen += bUid_Len;

    aSV[bSvLen++] = (uint8_t) (dwSDMReadCtr & 0xFF);
    aSV[bSvLen++] = (uint8_t) ((dwSDMReadCtr & 0xFF00) >> 8);
    aSV[bSvLen++] = (uint8_t) ((dwSDMReadCtr & 0xFF0000) >> 16);

    /* Update the SV length to 16. */
    bSvLen = 16;

    /* Exchange the session vector information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DeriveKey(&stHal_SamAv3, OFFLINE_CRYPTO_KEY_ADDRESS, OFFLINE_CRYPTO_KEY_VERSION,
        RAM_KEY_ADDRESS, aSV, bSvLen));

    /* Perform Offline activation using Ram Key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey(&stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
        RAM_KEY_ADDRESS, RAM_KEY_VERSION, NULL, 0));

    /* Frame SDMMAC Offset and Input Offset. */
    dwSDMMACInputOffset = (uint32_t) (aSDMMACInputOffset[0] | (aSDMMACInputOffset[1] << 8) | (aSDMMACInputOffset[2] << 16));
    dwSDMMACOffset = (uint32_t) (aSDMMACOffset[0] | (aSDMMACOffset[1] << 8) | (aSDMMACOffset[2] << 16));

    /* Perform MAC generation. */
    wRemLen = (uint16_t) ((wPicc_RespLen - 2) - (dwSDMMACInputOffset + 16));
    wBuffOption = PH_EXCHANGE_TXCHAINING;

    do
    {
        /* Update the finished flag and buffering option. */
        if(wRemLen <= 224)
        {
            bFinished = PH_ON;
            wBuffOption = PH_EXCHANGE_DEFAULT;
            bExchangeLen = (uint8_t) wRemLen;
        }
        else
        {
            bExchangeLen = 224;
            wRemLen = (uint16_t) (wRemLen - 224);
        }

        /* Exchange the Input information to SAM. */
        wStatus = phhalHw_SamAV3_Cmd_SAM_GenerateMAC(&stHal_SamAv3, wBuffOption, PHHAL_HW_SAMAV3_TRUNCATION_MODE_MFP,
            &pPicc_Response[dwSDMMACInputOffset + wInputOffset], bExchangeLen, &pResponse, &wRespLen);

        /* Validate the response. */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            bFinished = PH_ON;
        }

        /* Update the Input offset information. */
        wInputOffset += 224;

    } while(!bFinished);

    printf("\t\tSDM MAC Offset                          : ");
    PrintData(aSDMMACOffset, 3, "%02X ", "\n");

    printf("\t\tSDM MAC Input Offset                    : ");
    PrintData(aSDMMACInputOffset, 3, "%02X ", "\n");

    printf("\t\tSDM Read Ctr                            : ");
    PrintData(aSDMReadCtr, 3, "%02X ", "\n");

    printf("\t\tSDM MAC Data (ASCII)                    : ");
    PrintData(&pPicc_Response[dwSDMMACOffset], 16, "%02X ", "\n");

    printf("\t\tSDM MAC Data (HEX)                      : ");
    ConvertAscii_Hex(&pPicc_Response[dwSDMMACOffset], ((wPicc_RespLen - 2) - (dwSDMMACInputOffset + 16)), aSDMMACData_Hex);
    PrintData(aSDMMACData_Hex, 8, "%02X ", "\n");

    printf("\t\tCalculated SDM MAC Data                 : ");
    PrintData(pResponse, wRespLen, "%02X ", "\n");

    return PH_ERR_SUCCESS;
}


phStatus_t MFP_Authenticate(uint8_t bSMMode, uint8_t bKeyNo, uint8_t bKeyVer)
{
    phStatus_t wStatus = 0;

    uint8_t aPcdCapIn[6] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
    uint8_t aPcdCapOut[6];
    uint8_t aPdCapOut[6];
    uint8_t * pPcdCapOut = NULL;
    uint8_t * pPdCapOut = NULL;

    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Update SM Mode */
    aPcdCapIn[0] = bSMMode;

    /* Authenticate in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Frame PLUS EV2 PICC part 1 Auth command. */
        aCmdBuf[wCmdLen++] = 0x70;  aCmdBuf[wCmdLen++] = 0x00; aCmdBuf[wCmdLen++] = 0x40; aCmdBuf[wCmdLen++] = 0x06;
        memcpy(&aCmdBuf[wCmdLen], aPcdCapIn, 6);
        wCmdLen += 6;
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC(pPicc_Response[0], 0x90);

        /* Exchange the PICC response to Sam. */
        CHECK_SUCCESS_CHAINING(phhalHw_SamAV3_Cmd_SAM_AuthenticateMFP_Part1(&stHal_SamAv3, (PHHAL_HW_SAMAV3_MFP_AUTHENTICATE_FIRST | PHHAL_HW_SAMAV3_MFP_AUTHENTICATE_SL3_KDF),
            bKeyNo, bKeyVer, &pPicc_Response[1], (uint8_t) (wPicc_RespLen - 1), NULL, 0, &pSam_Response, &wSam_RespLen));

        /* Frame PLUS EV1 PICC part 2 Auth command. */
        wCmdLen = 0;
        aCmdBuf[wCmdLen++] = 0x72;
        memcpy(&aCmdBuf[wCmdLen], pSam_Response, wSam_RespLen);
        wCmdLen += (uint8_t) wSam_RespLen;
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC(pPicc_Response[0], 0x90);

        /* Exchange the PICC response to Sam. */
        wStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticateMFP_Part2(&stHal_SamAv3, pPicc_Response[0], &pPicc_Response[1], (uint8_t) (wPicc_RespLen - 1), &pPdCapOut,
            &pPcdCapOut, &bPiccErrCode);
        CHECK_SUCCESS_PICC(bPiccErrCode, 0x00);
    }

    /* Authenticate in X mode. */
    else
    {
        wStatus = phhalHw_SamAV3_Cmd_MFP_Authenticate(&stHal_SamAv3, (PHHAL_HW_SAMAV3_MFP_AUTHENTICATE_FIRST | PHHAL_HW_SAMAV3_MFP_AUTHENTICATE_SL3_KDF),
            bKeyNo, bKeyVer, 0x4000, aPcdCapIn, 6, NULL, 0, aPcdCapOut, aPdCapOut, &bPiccErrCode);
        CHECK_SUCCESS_PICC(bPiccErrCode, 0x00);
    }

    return wStatus;
}

phStatus_t MFP_ChangeKey(uint8_t bKeyNo, uint8_t bKeyVer)
{
    phStatus_t wStatus = 0;

    /* Reset the command buffer. */
    wCmdLen = 0;

    /* ChangeKey in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Frame PLUS EV1 PICC ChangeKey command. */
        aCmdBuf[wCmdLen++] = 0xA0;  aCmdBuf[wCmdLen++] = 0x00; aCmdBuf[wCmdLen++] = 0x40; aCmdBuf[wCmdLen++] = bKeyNo;
        aCmdBuf[wCmdLen++] = bKeyVer;

        /* Exchange the command code to Sam. */
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_ChangeKeyMFP(&stHal_SamAv3, PHHAL_HW_SAMAV3_OPTION_MFP_CHANGE_KEY_COMMAND, aCmdBuf, (uint8_t) wCmdLen,
            &pSam_Response, &wSam_RespLen, &bPiccErrCode));

        /* Exchange the protected data to PICC. */
        memcpy(&aCmdBuf[3], pSam_Response, wSam_RespLen);
        wCmdLen = (uint8_t) (3 + wSam_RespLen);
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC(pPicc_Response[0], 0x90);

        /* Exchange the command code to Sam. */
        wCmdLen = 0;
        aCmdBuf[wCmdLen++] = pPicc_Response[0];
        memcpy(&aCmdBuf[wCmdLen], &pPicc_Response[1], (wPicc_RespLen - 1));
        wCmdLen += (uint8_t) (wPicc_RespLen - 1);
        wStatus = phhalHw_SamAV3_Cmd_SAM_ChangeKeyMFP(&stHal_SamAv3, PHHAL_HW_SAMAV3_OPTION_MFP_CHANGE_KEY_RESPONSE, aCmdBuf, (uint8_t) wCmdLen, &pSam_Response,
            &wSam_RespLen, &bPiccErrCode);
        CHECK_SUCCESS_PICC(bPiccErrCode, 0x00);
    }

    /* ChangeKey in X mode. */
    else
    {
        wStatus = phhalHw_SamAV3_Cmd_MFP_ChangeKey(&stHal_SamAv3, PHHAL_HW_SAMAV3_MFP_CHANGE_KEY_DIVERSIFICATION_OFF, 0xA0, 0x4000, bKeyNo,
            bKeyVer, NULL, 0, &bPiccErrCode);
        CHECK_SUCCESS_PICC(bPiccErrCode, 0x00);
    }

    return wStatus;
}

phStatus_t MFP_Write()
{
    phStatus_t wStatus = 0;

    uint8_t aData[] = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10 };

    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Frame the command. */
    aCmdBuf[wCmdLen++] = 0xA0;                                  /* Command Code. MAC ON COMMAND, Encrypted, NO MAC ON RESPONSE */
    aCmdBuf[wCmdLen++] = 0x01;                                  /* Block Number */
    aCmdBuf[wCmdLen++] = 0x00;                                  /* Block Number */
    memcpy(&aCmdBuf[wCmdLen], aData, sizeof(aData));      /* Data */
    wCmdLen += (uint8_t) sizeof(aData);

    PrintData(aData, sizeof(aData), "%02X ", "");

    /* Write in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Exchange the command code to Sam. */
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_CombinedWriteMFP(&stHal_SamAv3, (PH_EXCHANGE_DEFAULT | PHHAL_HW_SAMAV3_OPTION_COMBINED_WRITE_MFP_COMMAND),
            aCmdBuf, (uint8_t) wCmdLen, &pSam_Response, &wSam_RespLen, &bPiccErrCode));

        /* Exchange the protected data to PICC. */
        memcpy(&aCmdBuf[3], pSam_Response, wSam_RespLen);
        wCmdLen = (uint8_t) (3 + wSam_RespLen);
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PICC(pPicc_Response[0], 0x90);

        /* Exchange the command code to Sam. */
        wCmdLen = 0;
        aCmdBuf[wCmdLen++] = pPicc_Response[0];
        memcpy(&aCmdBuf[wCmdLen], &pPicc_Response[1], (wPicc_RespLen - 1));
        wCmdLen += (uint8_t) (wPicc_RespLen - 1);
        wStatus = phhalHw_SamAV3_Cmd_SAM_CombinedWriteMFP(&stHal_SamAv3, (PH_EXCHANGE_DEFAULT | PHHAL_HW_SAMAV3_OPTION_COMBINED_WRITE_MFP_RESPONSE),
            aCmdBuf, (uint8_t) wCmdLen, &pSam_Response, &wSam_RespLen, &bPiccErrCode);
        CHECK_SUCCESS_PICC(bPiccErrCode, 0x00);
    }

    /* Write in X mode. */
    else
    {
        wStatus = phhalHw_SamAV3_Cmd_MFP_CombinedWrite(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, aCmdBuf, (uint8_t) wCmdLen, aTMC, aTMV, &bPiccErrCode);
        CHECK_SUCCESS_PICC(bPiccErrCode, 0x90);
    }

    return wStatus;
}

phStatus_t MFP_Read(uint8_t bCmdType)
{
    phStatus_t wStatus = 0;

    uint8_t aVersion[40];
    uint8_t bVerLen = 0;

    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Frame PICC Read command information. */
    if(bCmdType == MFP_COMBINED_READ_READ)
    {
        aCmdBuf[wCmdLen++] = 0x30;  /* Read Cmd Code (MAC ON CMD, MAC ON RESP, ENCRYPTED) */
        aCmdBuf[wCmdLen++] = 0x01;  /* Block Number. */
        aCmdBuf[wCmdLen++] = 0x00; /* Block Number. */
        aCmdBuf[wCmdLen++] = 0x01; /* Number of Blocks. */
    }
    /* Frame PICC GetVersion command information. */
    if(bCmdType == MFP_COMBINED_READ_GET_VERSION)
    {
        aCmdBuf[wCmdLen++] = 0x60;
    }

    /* Read in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Exchange the command code to Sam. */
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_CombinedReadMFP(&stHal_SamAv3, PHHAL_HW_SAMAV3_ISO7816_LAST_FRAME, (PH_EXCHANGE_DEFAULT |
            PHHAL_HW_SAMAV3_OPTION_COMBINED_READ_MFP_COMMAND), aCmdBuf, (uint8_t) wCmdLen, &pSam_Response, &wSam_RespLen, &bPiccErrCode));

        /* Exchange the protected data to PICC. */
        memcpy(&aCmdBuf[wCmdLen], pSam_Response, wSam_RespLen);
        wCmdLen += (uint8_t) wSam_RespLen;

        do
        {
            CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL4(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));

            if(bCmdType == MFP_COMBINED_READ_READ)
            {
                CHECK_SUCCESS_PICC(pPicc_Response[0], 0x90);
            }

            if((pPicc_Response[0] != 0x90) && (pPicc_Response[0] != 0xAF))
                break;

            if(bCmdType == MFP_COMBINED_READ_GET_VERSION)
            {
                memcpy(&aVersion[bVerLen], &pPicc_Response[1], (wPicc_RespLen - 1));
                bVerLen += (uint8_t) (wPicc_RespLen - 1);

                /* Update Command buffer and length. */
                aCmdBuf[0] = 0xAF;
                wCmdLen = 1;
            }
        } while(pPicc_Response[0] != 0x90);

        /* Exchange the command code to Sam. */
        wCmdLen = 0;

        /* Copy data based on command type. */
        if(bCmdType == MFP_COMBINED_READ_GET_VERSION)
        {
            aCmdBuf[wCmdLen++] = 0x90;
            memcpy(&aCmdBuf[wCmdLen], aVersion, bVerLen);
            wCmdLen += bVerLen;
        }
        else
        {
            aCmdBuf[wCmdLen++] = pPicc_Response[0];
            memcpy(&aCmdBuf[wCmdLen], &pPicc_Response[1], (wPicc_RespLen - 1));
            wCmdLen += (uint8_t) (wPicc_RespLen - 1);
        }
        wStatus = phhalHw_SamAV3_Cmd_SAM_CombinedReadMFP(&stHal_SamAv3, PHHAL_HW_SAMAV3_ISO7816_LAST_FRAME, (PH_EXCHANGE_DEFAULT |
            PHHAL_HW_SAMAV3_OPTION_COMBINED_READ_MFP_RESPONSE), aCmdBuf, (uint8_t) wCmdLen, &pSam_Response, &wSam_RespLen, &bPiccErrCode);
        CHECK_SUCCESS_PICC(bPiccErrCode, 0x00);
    }

    /* Read in X mode. */
    else
    {
        wStatus = phhalHw_SamAV3_Cmd_MFP_CombinedRead(&stHal_SamAv3, PH_EXCHANGE_DEFAULT, aCmdBuf, (uint8_t) wCmdLen, &pSam_Response,
            &wSam_RespLen, &bPiccErrCode);
        CHECK_SUCCESS_PICC(bPiccErrCode, 0x90);
    }

    if(wStatus == PH_ERR_SUCCESS)
    {
        /* Print data based on command type. */
        if((bCmdType == MFP_COMBINED_READ_GET_VERSION) && (eReaderMode == S_MODE))
        {
            PrintData(aVersion, (bVerLen - 8 /* MAC */), "%02X ", "");
        }
        else
        {
            PrintData(pSam_Response, wSam_RespLen, "%02X ", "");
        }
    }

    return wStatus;
}


phStatus_t MFC_Authenticate()
{
    phStatus_t wStatus = 0;

    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Authenticate in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Exchange the information to PICC. */
        aCmdBuf[wCmdLen++] = 0x60; aCmdBuf[wCmdLen++] = 0x00;
        CHECK_SUCCESS_PRINT(phhalHw_SetConfig(&stHal_Rd710, PHHAL_HW_CONFIG_RXCRC, PH_OFF));
        CHECK_SUCCESS_PRINT(phhalHw_Exchange(&stHal_Rd710, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));

        /* Exchange PICC response to SAM. */
        wCmdLen = 0;
        memcpy(aCmdBuf, pPicc_Response, wPicc_RespLen); wCmdLen = (uint8_t) wPicc_RespLen;
        aCmdBuf[wCmdLen++] = 0x00;
        CHECK_SUCCESS_CHAINING(phhalHw_SamAV3_Cmd_SAM_AuthenticateMIFARE_Part1(&stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_AUTHENTICATE_MIFARE_DIV_OFF, &aUid[bUid_Len - 4],
            PICC_MIFARE_KEY_ADDRESS, PICC_MIFARE_KEY_VERSION, PHHAL_HW_MFC_KEYA, 0, 0, aCmdBuf, (uint8_t) wCmdLen, &pSam_Response, &wSam_RespLen));

        /* Exchange the information to PICC. */
        aCmdBuf[wCmdLen++] = 0x60; aCmdBuf[wCmdLen++] = 0x00;
        CHECK_SUCCESS_PRINT(phhalHw_SetConfig(&stHal_Rd710, PHHAL_HW_CONFIG_TXCRC, PH_OFF));
        CHECK_SUCCESS_PRINT(phhalHw_SetConfig(&stHal_Rd710, PHHAL_HW_CONFIG_RXCRC, PH_OFF));
        CHECK_SUCCESS_PRINT(phhalHw_SetConfig(&stHal_Rd710, PHHAL_HW_CONFIG_PARITY, PH_OFF));
        CHECK_SUCCESS_PRINT(phhalHw_SetConfig(&stHal_Rd710, PHHAL_HW_CONFIG_TXLASTBITS, (uint16_t) ((wSam_RespLen * 8) % 8)));
        wStatus = phhalHw_Exchange(&stHal_Rd710, PH_EXCHANGE_DEFAULT, pSam_Response, (uint8_t) wSam_RespLen, &pPicc_Response, &wPicc_RespLen);
        if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_INCOMPLETE_BYTE)
            CHECK_SUCCESS_EXPECTED(wStatus, PH_ADD_COMPCODE(PH_ERR_SUCCESS_INCOMPLETE_BYTE, PH_COMP_HAL));

        CHECK_SUCCESS_PRINT(phhalHw_SetConfig(&stHal_Rd710, PHHAL_HW_CONFIG_TXCRC, PH_ON));
        CHECK_SUCCESS_PRINT(phhalHw_SetConfig(&stHal_Rd710, PHHAL_HW_CONFIG_RXCRC, PH_ON));
        CHECK_SUCCESS_PRINT(phhalHw_SetConfig(&stHal_Rd710, PHHAL_HW_CONFIG_PARITY, PH_ON));

        /* Exchange PICC response to SAM. */
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_AuthenticateMIFARE_Part2(&stHal_SamAv3, pPicc_Response, (uint8_t) wPicc_RespLen));
    }

    /* Authenticate in X mode. */
    else
    {
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_MF_Authenticate(&stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_AUTHENTICATE_MIFARE_DIV_OFF, &aUid[bUid_Len - 4], PICC_MIFARE_KEY_ADDRESS,
            PICC_MIFARE_KEY_VERSION, PHHAL_HW_MFC_KEYA, 0, 0));
    }

    return PH_ERR_SUCCESS;
}

phStatus_t MFC_Write()
{
    uint8_t aData[] = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10 };

    PrintData(aData, sizeof(aData), "%02X ", "");

    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Write in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Disable Crypto */
        CHECK_SUCCESS_PRINT(phhalHw_SetConfig(&stHal_SamAv3, PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1, PH_OFF));

        /* Exchange the information to PICC. */
        aCmdBuf[wCmdLen++] = 0xA0; aCmdBuf[wCmdLen++] = 0x01;
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL3(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL3(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aData, (uint8_t) sizeof(aData), &pPicc_Response, &wPicc_RespLen));

        /* Re-Enable Crypto */
        CHECK_SUCCESS_PRINT(phhalHw_SetConfig(&stHal_SamAv3, PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1, PH_ON));
    }

    /* Write in X mode. */
    else
    {
        /* Frame the command. */
        aCmdBuf[wCmdLen++] = 0x01;
        memcpy(&aCmdBuf[wCmdLen], aData, sizeof(aData));
        wCmdLen += (uint8_t) sizeof(aData);
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_MF_Write(&stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_MF_WRITE_CLASSIC, aCmdBuf, (uint8_t) wCmdLen,
            &pSam_Response, &wSam_RespLen));
    }

    return PH_ERR_SUCCESS;
}

phStatus_t MFC_Read()
{
    phStatus_t wStatus = 0;

    /* Reset the command buffer. */
    wCmdLen = 0;

    /* Read in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Disable Crypto */
        CHECK_SUCCESS_PRINT(phhalHw_SetConfig(&stHal_SamAv3, PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1, PH_OFF));

        /* Exchange the information to PICC. */
        aCmdBuf[wCmdLen++] = 0x30; aCmdBuf[wCmdLen++] = 0x01;
        CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL3(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));

        /* Re-Enable Crypto */
        CHECK_SUCCESS_PRINT(phhalHw_SetConfig(&stHal_SamAv3, PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1, PH_ON));
    }

    /* Read in X mode. */
    else
    {
        /* Frame the command information. */
        aCmdBuf[wCmdLen++] = 0x01;
        CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_MF_Read(&stHal_SamAv3, aCmdBuf, (uint8_t) wCmdLen, &pPicc_Response, &wPicc_RespLen));
    }

    if(wStatus == PH_ERR_SUCCESS)
    {
        /* Print the received data. */
        PrintData(pPicc_Response, wPicc_RespLen, "%02X ", "");
    }

    return PH_ERR_SUCCESS;
}


phStatus_t MFUL_Authenticate(uint8_t bULCProduct)
{
    phStatus_t wStatus = 0;

    uint8_t aPcdCapOut[6];
    uint8_t aPdCapOut[6];

    uint16_t wPack = 0;
    uint8_t bPICC_Status = 0;

    /* Authenticate in S mode. */
    if(eReaderMode == S_MODE)
    {
        /* Perform Authentication for MIFARE Ultralight C PICC. */
        if(bULCProduct == MFUL_AUTH_CLASSIC)
        {
            /* Exchange the information to PICC. */
            aCmdBuf[wCmdLen++] = 0x1A; aCmdBuf[wCmdLen++] = 0x00;
            CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL3(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
            CHECK_SUCCESS_PICC(pPicc_Response[0], 0xAF);

            /* Exchange PICC response to SAM. */
            CHECK_SUCCESS_CHAINING(phhalHw_SamAV3_Cmd_SAM_AuthenticatePICC_Part1(&stHal_SamAv3, 0x00, PICC_MFUL_2K3DES_KEY_ADDRESS, PICC_MFUL_2K3DES_KEY_VERSION,
                0x00, NULL, 0, &pPicc_Response[1], (uint8_t) (wPicc_RespLen - 1), &pSam_Response, &wSam_RespLen));

            /* Exchange the information to PICC. */
            wCmdLen = 0;
            aCmdBuf[wCmdLen++] = 0xAF;
            memcpy(&aCmdBuf[wCmdLen], pSam_Response, wSam_RespLen);
            wCmdLen += wSam_RespLen;
            CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL3(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));
            CHECK_SUCCESS_PICC(pPicc_Response[0], 0x00);

            /* Exchange PICC response to SAM. */
            CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_AuthenticatePICC_Part2(&stHal_SamAv3, pPicc_Response[0], &pPicc_Response[1], (uint8_t) (wPicc_RespLen - 1),
                aPdCapOut, aPcdCapOut, &bPICC_Status));
        }

        /* Perform Authentication for MIFARE Ultralight EV1 PICC. */
        if(bULCProduct == MFUL_AUTH_EV1)
        {
            /* Update the password and pack for MIFARE Ultralight EV1 PICC. */
            MFUL_UpdateDiversifiedPassword();

            /* Activate the PICC. */
            CHECK_SUCCESS_PRINT(ActivateCard(ISO14443_L3));

            /* Perform SAM exchange and to get the password. */
            CHECK_SUCCESS_CHAINING(phhalHw_SamAV3_Cmd_SAM_PwdAuthUL_Part1(&stHal_SamAv3, PICC_MFUL_AES_KEY_ADDRESS, PICC_MFUL_AES_KEY_VERSION, aDivInput, (uint8_t) sizeof(aDivInput),
                &pSam_Response, &wSam_RespLen));

            /* Exchange the password information to PICC. */
            aCmdBuf[wCmdLen++] = 0x1B; aCmdBuf[wCmdLen++] = pSam_Response[0]; aCmdBuf[wCmdLen++] = pSam_Response[1]; aCmdBuf[wCmdLen++] = pSam_Response[2];
            aCmdBuf[wCmdLen++] = pSam_Response[3];
            CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL3(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));

            /* Perform SAM exchange to verify Pack information. */
            wPack = (uint16_t) ((pPicc_Response[0] << 8) | pPicc_Response[1]);
            CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_PwdAuthUL_Part2(&stHal_SamAv3, wPack));

            /* Activate the PICC. */
            CHECK_SUCCESS_PRINT(ActivateCard(ISO14443_L3));

            /* Revert Password and Pack to default one. */
            MFUL_RevertPassword();
        }
    }

    /* Authenticate in X mode. */
    else
    {
        /* Perform Authentication for MIFARE Ultralight C PICC. */
        if(bULCProduct == MFUL_AUTH_CLASSIC)
        {
            wStatus = phhalHw_SamAV3_Cmd_ULC_AuthenticatePICC(&stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_ULC_AUTHENTICATE_DIV_OFF, PICC_MFUL_2K3DES_KEY_ADDRESS, PICC_MFUL_2K3DES_KEY_VERSION,
                NULL, 0, &bPICC_Status);

            /* Validate the status. */
            CHECK_SUCCESS_PICC(bPICC_Status, 0x00);
            CHECK_SUCCESS_PRINT(wStatus);
        }

        /* Perform Authentication for MIFARE Ultralight EV1 PICC. */
        if(bULCProduct == MFUL_AUTH_EV1)
        {
            /* Update the password and pack for MIFARE Ultralight EV1 PICC. */
            MFUL_UpdateDiversifiedPassword();

            /* Activate the PICC. */
            CHECK_SUCCESS_PRINT(ActivateCard(ISO14443_L3));

            /* Perform Authentication. */
            wStatus = phhalHw_SamAV3_Cmd_UL_PwdAuthUL(&stHal_SamAv3, PICC_MFUL_AES_KEY_ADDRESS, PICC_MFUL_AES_KEY_VERSION, aDivInput, (uint8_t) sizeof(aDivInput), &bPICC_Status);
            CHECK_SUCCESS_PICC(bPICC_Status, 0x00);
            CHECK_SUCCESS_PRINT(wStatus);

            /* Activate the PICC. */
            CHECK_SUCCESS_PRINT(ActivateCard(ISO14443_L3));

            /* Revert Password and Pack to default one. */
            MFUL_RevertPassword();
        }
    }

    return PH_ERR_SUCCESS;
}

phStatus_t MFUL_UpdateDiversifiedPassword()
{
    uint8_t aData[40];
    uint8_t bDataLen = 0;

    uint8_t aMac[16];
    uint8_t bMacLen = 0;

    uint8_t aIV[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

    CHECK_SUCCESS_PRINT(phCryptoSym_LoadKeyDirect(&stCryptoMAC1, aMFUL_AES_Key, PH_CRYPTOSYM_KEY_TYPE_AES128));
    CHECK_SUCCESS_PRINT(phCryptoSym_LoadIv(&stCryptoMAC1, aIV, (uint8_t) sizeof(aIV)));

    aData[bDataLen++] = 0x02;
    memcpy(&aData[bDataLen], aDivInput, sizeof(aDivInput));
    bDataLen += (uint8_t) sizeof(aDivInput);
    CHECK_SUCCESS_PRINT(phCryptoSym_CalculateMac(&stCryptoMAC1, PH_CRYPTOSYM_MAC_MODE_CMAC, aData, bDataLen, aMac, &bMacLen));

    memcpy(aMFUL_Password, aMac, 4);
    memcpy(aMFUL_Pack, &aMac[bMacLen - 2], 2);

    /* Perform Authentication. */
    wCmdLen = 0;
    aCmdBuf[wCmdLen++] = 0x1B; aCmdBuf[wCmdLen++] = 0xFF; aCmdBuf[wCmdLen++] = 0xFF; aCmdBuf[wCmdLen++] = 0xFF; aCmdBuf[wCmdLen++] = 0xFF;
    CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL3(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));

    /* Update the password. */
    wCmdLen = 0;
    aCmdBuf[wCmdLen++] = 0xA2; aCmdBuf[wCmdLen++] = 0x27; aCmdBuf[wCmdLen++] = aMac[0]; aCmdBuf[wCmdLen++] = aMac[1]; aCmdBuf[wCmdLen++] = aMac[2];
    aCmdBuf[wCmdLen++] = aMac[3];
    CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL3(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));

    /* Update the Pack. */
    wCmdLen = 0;
    aCmdBuf[wCmdLen++] = 0xA2; aCmdBuf[wCmdLen++] = 0x28; aCmdBuf[wCmdLen++] = aMac[14]; aCmdBuf[wCmdLen++] = aMac[15]; aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x00;
    CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL3(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));

    return PH_ERR_SUCCESS;
}

phStatus_t MFUL_RevertPassword()
{
    /* Perform Authentication. */
    wCmdLen = 0;
    aCmdBuf[wCmdLen++] = 0x1B; aCmdBuf[wCmdLen++] = aMFUL_Password[0]; aCmdBuf[wCmdLen++] = aMFUL_Password[1]; aCmdBuf[wCmdLen++] = aMFUL_Password[2];
    aCmdBuf[wCmdLen++] = aMFUL_Password[3];
    CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL3(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));

    /* Update the password. */
    wCmdLen = 0;
    aCmdBuf[wCmdLen++] = 0xA2; aCmdBuf[wCmdLen++] = 0x27; aCmdBuf[wCmdLen++] = 0xFF; aCmdBuf[wCmdLen++] = 0xFF; aCmdBuf[wCmdLen++] = 0xFF;
    aCmdBuf[wCmdLen++] = 0xFF;
    CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL3(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));

    /* Update the Pack. */
    wCmdLen = 0;
    aCmdBuf[wCmdLen++] = 0xA2; aCmdBuf[wCmdLen++] = 0x28; aCmdBuf[wCmdLen++] = 0x00; aCmdBuf[wCmdLen++] = 0x00; aCmdBuf[wCmdLen++] = 0x00;
    aCmdBuf[wCmdLen++] = 0x00;
    CHECK_SUCCESS_PRINT(phpalMifare_ExchangeL3(pPalMifare_Generic, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pPicc_Response, &wPicc_RespLen));

    return PH_ERR_SUCCESS;
}


phStatus_t ICODE_AuthenticateTAM()
{
    phStatus_t wStatus = 0;

    uint16_t wError = 0;

    uint8_t * pResponse = NULL;
    uint16_t wRespLen = 0;

    uint8_t * pIChallange = NULL;
    uint16_t wIChallangeLen = 0;

    /* Perform TamAuthentication to Get the IChallange from SAM. */
    CHECK_SUCCESS_CHAINING(phhalHw_SamAV3_Cmd_SAM_AuthenticateTAM1(&stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_TAM_GET_RND, PICC_AES128_KEY_ADDRESS,
        PICC_AES128_KEY_VERSION, NULL, 0, &pIChallange, &wIChallangeLen));

    /* Reverse the buffer. */
    Reverse(pIChallange, wIChallangeLen);

    /* Set DataRate and addressed mode. */
    CHECK_SUCCESS_PRINT(phpalSli15693_SetConfig(pPalIso15693, PHPAL_SLI15693_CONFIG_FLAGS, (uint16_t) (PHPAL_SLI15693_FLAG_DATA_RATE |
        PHPAL_SLI15693_FLAG_ADDRESSED)));
    CHECK_SUCCESS_PRINT(phpalSli15693_SetConfig(pPalIso15693, PHPAL_SLI15693_CONFIG_TIMEOUT_US, 20000));

    /* Exchange the IChallange to tag. */
    wCmdLen = 0;
    aCmdBuf[wCmdLen++] = 0x35; aCmdBuf[wCmdLen++] = 0x00; aCmdBuf[wCmdLen++] = 0x00; aCmdBuf[wCmdLen++] = 0x00;
    memcpy(&aCmdBuf[wCmdLen], pIChallange, wIChallangeLen);
    wCmdLen += (uint8_t) wIChallangeLen;
    wStatus = phpalSli15693_Exchange(pPalIso15693, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pResponse, &wRespLen);

    /*  Validate the Error code if available. */
    if(wStatus != PH_ERR_SUCCESS)
    {
        CHECK_SUCCESS_PRINT(phpalSli15693_GetConfig(pPalIso15693, PHPAL_SLI15693_CONFIG_ADD_INFO, &wError));
        CHECK_SUCCESS_TAG(wStatus, wError);
    }

    /* Reverse the buffer. */
    Reverse(&pResponse[1], (uint16_t) (wRespLen - 1));

    /* Exchange the response received from Tag to Sam. */
    CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_AuthenticateTAM1(&stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_TAM_PROCESS_TRESPONE, 0, 0, &pResponse[1],
        (uint8_t) (wRespLen - 1), &pIChallange, &wIChallangeLen));

    return PH_ERR_SUCCESS;
}

phStatus_t ICODE_AuthenticateMAM()
{
    phStatus_t wStatus = 0;

    uint16_t wError = 0;

    uint8_t * pIChallange = NULL;
    uint16_t wIChallangeLen = 0;

    uint8_t * pIResponse = NULL;
    uint16_t wIResponseLen = 0;

    uint8_t * pResponse = NULL;
    uint16_t wRespLen = 0;

    /* Perform MAM Authentication part 1 to Get the IChallange from SAM. */
    CHECK_SUCCESS_CHAINING(phhalHw_SamAV3_Cmd_SAM_AuthenticateMAM1(&stHal_SamAv3, PICC_AES128_KEY_ADDRESS, PICC_AES128_KEY_VERSION,
        NULL, 0, 0, &pIChallange, &wIChallangeLen));

    /* Reverse the buffer. */
    Reverse(pIChallange, wIChallangeLen);

    /* Set DataRate and addressed mode. */
    CHECK_SUCCESS_PRINT(phpalSli15693_SetConfig(pPalIso15693, PHPAL_SLI15693_CONFIG_FLAGS, (uint16_t) (PHPAL_SLI15693_FLAG_DATA_RATE |
        PHPAL_SLI15693_FLAG_ADDRESSED)));
    CHECK_SUCCESS_PRINT(phpalSli15693_SetConfig(pPalIso15693, PHPAL_SLI15693_CONFIG_TIMEOUT_US, 20000));

    /* Exchange the IChallange to tag. */
    wCmdLen = 0;
    aCmdBuf[wCmdLen++] = 0x35; aCmdBuf[wCmdLen++] = 0x00; aCmdBuf[wCmdLen++] = 0x02; aCmdBuf[wCmdLen++] = 0x00;
    memcpy(&aCmdBuf[wCmdLen], pIChallange, wIChallangeLen);
    wCmdLen += (uint8_t) wIChallangeLen;
    wStatus = phpalSli15693_Exchange(pPalIso15693, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pResponse, &wRespLen);

    /*  Validate the Error code if available. */
    if(wStatus != PH_ERR_SUCCESS)
    {
        CHECK_SUCCESS_PRINT(phpalSli15693_GetConfig(pPalIso15693, PHPAL_SLI15693_CONFIG_ADD_INFO, &wError));
        CHECK_SUCCESS_TAG(wStatus, wError);
    }

    /* Reverse the buffer. */
    Reverse(&pResponse[1], (uint16_t) (wRespLen - 1));

    /* Exchange the response received from Tag to Sam. */
    CHECK_SUCCESS_PRINT(phhalHw_SamAV3_Cmd_SAM_AuthenticateMAM2(&stHal_SamAv3, &pResponse[1], (uint8_t) (wRespLen - 1), &pIResponse, &wIResponseLen));

    /* Reverse the buffer. */
    Reverse(pIResponse, wIResponseLen);

    /* Exchange the IChallange to tag. */
    wCmdLen = 0;
    aCmdBuf[wCmdLen++] = 0x35; aCmdBuf[wCmdLen++] = 0x00; aCmdBuf[wCmdLen++] = 0x06;
    memcpy(&aCmdBuf[wCmdLen], pIResponse, wIResponseLen);
    wCmdLen += (uint8_t) wIResponseLen;
    wStatus = phpalSli15693_Exchange(pPalIso15693, PH_EXCHANGE_DEFAULT, aCmdBuf, wCmdLen, &pResponse, &wRespLen);

    /*  Validate the Error code if available. */
    if(wStatus != PH_ERR_SUCCESS)
    {
        CHECK_SUCCESS_PRINT(phpalSli15693_GetConfig(pPalIso15693, PHPAL_SLI15693_CONFIG_ADD_INFO, &wError));
        CHECK_SUCCESS_TAG(wStatus, wError);
    }

    return PH_ERR_SUCCESS;
}


phStatus_t PL_DetectFileSize ( FILE ** ppFile, uint16_t * pFileSize )
{
    uint32_t dwStartPos = 0, dwData = 0;

    /* Open the file. */
    *ppFile = fopen ( PL_CODE_FILE, "r" );

    /* Detect the size. */
    dwStartPos = ftell ( *ppFile );
    while ( fscanf ( *ppFile, "%c", ( char * ) &dwData ) == 1 )
    {
        ++(*pFileSize);
    }

    fseek ( *ppFile, dwStartPos, SEEK_SET );
    return PH_ERR_SUCCESS;
}

phStatus_t PL_ExtractPLCode ( FILE ** ppFile, uint8_t * pPLCode, uint16_t * pPLCodeLen )
{
    uint32_t dwPos = 0;
    uint32_t dwStartPos = 0;
    char * pFileData = NULL;

    char * pString_Duplicate = NULL;
    char *pToken = NULL, *pStrtok_Ctx = NULL;
    char ** pTokens = NULL;

    size_t dwTokens_Alloc = 1;
    size_t dwTokens_Used = 0;

    /* Read the file contents -------------------------------------------------------------------------------------------------------------------- */
    pFileData = ( char * ) malloc ( *pPLCodeLen );
    fread ( pFileData, 1, *pPLCodeLen, *ppFile );

    /* Split the strings ------------------------------------------------------------------------------------------------------------------------- */
    pString_Duplicate = _strdup ( pFileData );
    free ( pFileData );
    pTokens = ( char ** ) calloc ( dwTokens_Alloc, sizeof ( char * ) );
    for ( pToken = strtok_s ( pString_Duplicate, ":", &pStrtok_Ctx ); pToken != NULL; pToken = strtok_s ( NULL, ":", &pStrtok_Ctx ) )
    {
        if ( dwTokens_Used == dwTokens_Alloc )
        {
            dwTokens_Alloc *= 2;
            pTokens = ( char ** ) realloc ( pTokens, dwTokens_Alloc * sizeof ( char * ) );
        }
        pTokens[dwTokens_Used++] = _strdup (pToken );
    }

    if ( dwTokens_Used == 0 )
    {
        free ( pTokens );
        pTokens = NULL;
    }
    else
    {
        pTokens = ( char ** ) realloc ( pTokens, dwTokens_Used * sizeof ( char * ) );
    }
    free ( pString_Duplicate );

    /* Convert the characters to Hex bytes ------------------------------------------------------------------------------------------------------- */
    *pPLCodeLen = 0;
    free ( pTokens[0] ); free ( pTokens[dwTokens_Used - 1] );
    for ( dwStartPos = 1; dwStartPos <  ( dwTokens_Used - 1 ); dwStartPos++ )
    {
        for ( dwPos = 8; dwPos < 40; dwPos += 2 )
        {

            pPLCode[*pPLCodeLen] = ( uint8_t ) ( ( ( ( pTokens[dwStartPos][dwPos] | ( 'A' ^ 'a' ) ) - '0' ) % 39 ) << 4 ) |
                ( uint8_t ) ( ( ( pTokens[dwStartPos][dwPos + 1] | ( 'A' ^ 'a' ) ) - '0' ) % 39 );
            *pPLCodeLen = *pPLCodeLen + 1;
        }
        free ( pTokens[dwStartPos] );
    }

    free ( pTokens );
    fclose ( *ppFile );

    return PH_ERR_SUCCESS;
}


phStatus_t Demo_HostAuthentication ()
{
    uint32_t dwChoice = 0;

    if ( !bCommMode )
    {
        printf ( "\tPress 1 to perform Host Authentication to enable Plain mode. \n" );
        printf ( "\tPress 2 to perform Host Authentication to enable Mac mode. \n" );
        printf ( "\tPress 3 to perform Host Authentication to enable Full mode. \n" );
        dwChoice = _getch ();
        printf("\n\n");

        /* Check if proper option is selected. */
        if ( ( dwChoice != '1' ) && ( dwChoice != '2' ) && ( dwChoice != '3' ) )
        {
            printf ( "\tInvalid communication option selected. \n" );
            return 0xFF;
        }

        /* Perform HOST AUTHENTICATION. */
        bCommMode = ( uint8_t ) ( dwChoice - 0x30 );
    }

    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_AuthenticateHost ( &stHal_SamAv3, ( uint8_t ) ( bCommMode - 1 ), SAM_MASTER_KEY_ADDRESS, SAM_MASTER_KEY_VERSION,
        SAM_MASTER_KEY, 0 ) );

    //if ( !bCommMode )
    {
        printf ( "\tAuthentication to Sam is Success. \n" );
        printf ( "\tNote: \n" );
        printf ( "\t\tThe current communication mode enabled is " );
        printf ( ( dwChoice == '1' ) ? "PLAIN.\n" : ( dwChoice == '2' ) ? "MAC.\n" : "FULL.\n" );
        printf ( "\t\tAny demo performed post this authentication will be based on the communication mode selected.\n" );
    }

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_LockUnlock ()
{
    uint8_t aVersion[31];
    uint8_t bVerLen = 0;

    /* Exclude this scenario for MAC and FULL mode. */
    if ( ( bCommMode == 0x02 ) || ( bCommMode == 0x03 ) )
    {
        printf ( "\tThe keys are invalidated because after Host Authentication Session keys, the Crypto DataParams are updated with new\n" );
        printf ( "\tkeys inside LockUnlock interface. In the application, when LockUnlock command is called and CryptoSym params are not\n" );
        printf ( "\tcleared, the preceeding commands will fail in MAC and ENC mode. To avoid the failure of commands its necessary to\n" );
        printf ( "\tclear the existing keys in the Crypto DataParams and force application or user to perform Host Authentication again\n" );
        printf ( "\tto generate new key and store them.\n" );

        return PH_ERR_SUCCESS;
    }

    /* Perform ChangeKey Entry for PICC class. */
    printf ( "\tChanging KeyNo %d as AES HOST key                        : ", LOCK_UNLOCK_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  LOCK_UNLOCK_KEY_ADDRESS, KEY_TYPE_AES128, KEY_CLASS_HOST, aAESKey, sizeof ( aAESKey ), 0x00 ) );
    printf ( "SUCCESS\n" );

    /* Perform LockUnock to lock the SAM. */
    printf ( "\tPerform LockUnlock command to Lock SAM with UnlockKey   : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_LockUnlock ( &stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_LOCK_UNLOCK_TYPE_LOCK_KEY, 0, 0, LOCK_UNLOCK_KEY_ADDRESS, LOCK_UNLOCK_KEY_VERSION,
        LOCK_UNLOCK_KEY_ADDRESS, LOCK_UNLOCK_KEY_VERSION, 0xFFFFFFFF ) );
    printf ( "SUCCESS\n" );
    printf ( "\n" );

    /* Perform some commands to verify the locked state. */
    printf ( "\tPerforming some commands in locked state ------------------------------------- \n" );

    printf ( "\tPerform GetVersion                                      : " );
    CHECK_SUCCESS ( phhalHw_SamAV3_Cmd_SAM_GetVersion ( &stHal_SamAv3, aVersion, & bVerLen ) );
    PrintData ( aVersion, bVerLen, "%02X ", "\n" );

    printf ( "\tPerform GetKeyEntry                                     : " );
    CHECK_SUCCESS ( Demo_GetKeyEntry ( LOCK_UNLOCK_KEY_ADDRESS, 0x00 ) );

    /* Perform Kill Authentication. */
    printf ( "\tPerforming KillAuthentication to kill any Authentication: " );
    CHECK_SUCCESS_EXPECTED ( phhalHw_SamAV3_Cmd_SAM_KillAuthentication ( &stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_KILL_AUTHENTICATION_FULL ), 0x028A );
    printf ( "\n" );

    /* Perform LockUnock to Unlock the SAM. */
    printf ( "\tPerform LockUnlock command to Unlock SAM                : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_LockUnlock ( &stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_LOCK_UNLOCK_TYPE_UNLOCK, 0, 0, LOCK_UNLOCK_KEY_ADDRESS, LOCK_UNLOCK_KEY_VERSION,
        LOCK_UNLOCK_KEY_ADDRESS, LOCK_UNLOCK_KEY_VERSION, 0xFFFFFFFF ) );
    printf ( "SUCCESS\n" );
    printf ( "\n" );

    /* Perform some commands to verify the unlocked state. */
    printf ( "\tPerforming some commands in unlocked state ----------------------------------- \n" );
    if ( !bCommMode ) bCommMode = 1;
    printf ( "\tPerform Host Authentication                             : " );
    CHECK_SUCCESS ( Demo_HostAuthentication () );
    printf ( "SUCCESS\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_GetVersion ()
{
    uint8_t aVersion[31];
    uint8_t bVerLen = 0;

    /* Get the Version information. */
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_GetVersion ( &stHal_SamAv3, aVersion, & bVerLen ) );

    printf ( "\tVersion information : " );
    PrintData ( aVersion, bVerLen, "%02X ", "\n\n" );
    bVerLen = 0;

    printf ( "\tHardware Information ----------------------------\n" );
    printf ( "\t\tVendor ID         : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tType              : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tSub Type          : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tMajor Version     : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tMinor Version     : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tStorage Size      : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tProtocol          : %02X\n", aVersion[bVerLen++] );
    printf ( "\n" );

    printf ( "\tSoftware Information ----------------------------\n" );
    printf ( "\t\tVendor ID         : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tType              : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tSub Type          : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tMajor Version     : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tMinor Version     : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tStorage Size      : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tProtocol          : %02X\n", aVersion[bVerLen++] );
    printf ( "\n" );

    printf ( "\tManufacturer Information ----------------------------\n" );
    printf ( "\t\tUID               : " );
    PrintData ( &aVersion[bVerLen], 7, "%02X ", "\n" );
    bVerLen += 7;

    printf ( "\t\tBatch Number      : " );
    PrintData ( &aVersion[bVerLen], 5, "%02X ", "\n" );
    bVerLen += 5;

    printf ( "\t\tProduction Day    : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tProduction Month  : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tProduction Year   : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tCrypto            : %02X\n", aVersion[bVerLen++] );
    printf ( "\t\tState             : %02X\n", aVersion[bVerLen++] );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_ActivateOfflineKey ()
{
    /* Perform ChangeKey Entry for Offlinekey class. */
    printf ( "\tChanging KeyNo %d as AES128 OfflineCrypto key   : ", OFFLINE_CRYPTO_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  OFFLINE_CRYPTO_KEY_ADDRESS, KEY_TYPE_AES128, KEY_CLASS_OFFLINE_CRYPTO, aAESKey, sizeof ( aAESKey ), 0x00 ) );
    printf ( "SUCCESS\n" );

    /* Perform ActivateOffline command. */
    printf ( "\tActivating KeyNo %d for OfflineCrypto operations: ", OFFLINE_CRYPTO_KEY_ADDRESS );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey ( &stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
        OFFLINE_CRYPTO_KEY_ADDRESS, OFFLINE_CRYPTO_KEY_VERSION, NULL, 0 ) );
    printf ( "SUCCESS\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_LoadInitVector ()
{
    uint8_t * pMac = NULL;
    uint16_t wMacLen = 0;

    uint8_t aInputBuffer[16] = { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF };
    uint8_t aInitVector[16] = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10 };

    /* Activate Offline Key. */
    CHECK_SUCCESS ( Demo_ActivateOfflineKey () );

    printf("\n");
    printf ( "\tPlain Data                                     : " );
    PrintData ( aInputBuffer, sizeof ( aInputBuffer ), "%02X ", "\n" );

    /* Perform GenerateMAC. */
    printf ( "\tPerform GenerateMAC                            : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_GenerateMAC ( &stHal_SamAv3, PH_EXCHANGE_DEFAULT, 16, aInputBuffer, sizeof ( aInputBuffer ), &pMac, &wMacLen ) );
    printf ( "SUCCESS\n" );

    printf ( "\tMAC data before LoadInitVector command         : " );
    PrintData ( pMac, wMacLen, "%02X ", "\n\n" );

    printf ( "\tInitVector Data                                : " );
    PrintData ( aInitVector, sizeof ( aInitVector ), "%02X ", "\n" );

    /* Perform LoadInitVector. */
    printf ( "\tPerform Load Init Vector                       : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_LoadInitVector ( &stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_LOAD_IV_MODE_SET_IV, aInitVector, sizeof ( aInitVector ) ) );
    printf ( "SUCCESS\n" );

    /* Perform GenerateMAC. */
    printf ( "\tPerform GenerateMAC                            : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_GenerateMAC ( &stHal_SamAv3, PH_EXCHANGE_DEFAULT, 16, aInputBuffer, sizeof ( aInputBuffer ), &pMac, &wMacLen ) );
    printf ( "SUCCESS\n" );

    printf ( "\tMAC data after LoadInitVector command          : " );
    PrintData ( pMac, wMacLen, "%02X ", "\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_KillAuthentication ()
{
    uint8_t aRnd[16];

    /* Perform Get Random Number. */
    printf ( "\tGet Random Number                                       : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_GetRandom ( &stHal_SamAv3, ( uint8_t ) sizeof ( aRnd ), aRnd ) );
    printf ( "SUCCESS\n" );

    /* Perform Kill Authentication. */
    printf ( "\tPerforming KillAuthentication to kill any Authentication: " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_KillAuthentication ( &stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_KILL_AUTHENTICATION_FULL ) );
    printf ( "SUCCESS\n" );

    /* Perform Get Random Number. */
    printf ( "\tGet Random Number                                       : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_GetRandom ( &stHal_SamAv3, ( uint8_t ) sizeof ( aRnd ), aRnd ) );
    printf ( "SUCCESS\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_SelectApplication ()
{
    uint8_t aAid[3];

    /* Perform Select Application. */
    printf ( "\tPerform Select Application                                 : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_SelectApplication ( &stHal_SamAv3, aAid ) );
    printf ( "SUCCESS\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_GetRandom ()
{
    uint8_t aRnd[16];

    /* Clear the buffer. */
    memset ( aRnd, 0x00, sizeof ( aRnd ) );

    printf ( "\tRandom buffer before GetRandom command                     : " );
    PrintData ( aRnd, sizeof ( aRnd ), "%02X ", "\n" );

    /* Perform GetRandom. */
    printf ( "\tPerform GetRandom                                          : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_GetRandom ( &stHal_SamAv3, ( uint8_t ) sizeof ( aRnd ), aRnd ) );
    printf ( "SUCCESS\n" );

    printf ( "\tRandom buffer after GetRandom command                      : " );
    PrintData ( aRnd, sizeof ( aRnd ), "%02X ", "\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_Sleep ()
{
    /* Perform Sleep. */
    printf ( "\tPerform Sleep                                           : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_Sleep ( &stHal_SamAv3 ) );
    printf ( "SUCCESS\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_GetKeyEntry ( uint8_t bKeyNo, uint8_t bDisplayContents )
{
    uint8_t aKeyEntry[20];
    uint8_t bKeyEntryLen = 0;

    /* Get the KeyEntry information. */
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_GetKeyEntry ( &stHal_SamAv3, bKeyNo, ( uint8_t ) ( ( stHal_SamAv3.bHostMode == 0x02 ) ?
        PHHAL_HW_SAMAV3_CMD_SAM_GET_KEY_ENTRY_KEY_ENTRY_SAM_AV2 : PHHAL_HW_SAMAV3_CMD_SAM_GET_KEY_ENTRY_KEY_ENTRY_NEW ), aKeyEntry, &bKeyEntryLen ) );

    if ( bDisplayContents ) printf ( "\tKeyEntry information: " );
    PrintData ( aKeyEntry, bKeyEntryLen, "%02X ", "\n" );

    if ( bDisplayContents )
    {
        printf ( "\n" );
        printf ( "\tVersion A           : %02X\n", aKeyEntry[0] );
        printf ( "\tVersion B           : %02X\n", aKeyEntry[1] );

        if ( ( bKeyEntryLen == 13 /* for Sam AV2 */) || ( bKeyEntryLen == 16 ) /* for Sam AV3 */)
        {
            printf ( "\tVersion C           : %02X\n", aKeyEntry[3] );
            bKeyEntryLen = 3;
        }
        else
            bKeyEntryLen = 2;

        printf ( "\tDF_AID              : " );
        PrintData ( &aKeyEntry[bKeyEntryLen], 3, "%02X ", "\n" );
        bKeyEntryLen += 3;

        printf ( "\tDF_KeyNo            : %02X\n", aKeyEntry[bKeyEntryLen++] );
        printf ( "\tKeyNoCEK            : %02X\n", aKeyEntry[bKeyEntryLen++] );
        printf ( "\tKeyVCEK             : %02X\n", aKeyEntry[bKeyEntryLen++] );
        printf ( "\tRefNoKUC            : %02X\n", aKeyEntry[bKeyEntryLen++] );

        printf ( "\tSET                 : " );
        PrintData ( &aKeyEntry[bKeyEntryLen], 2, "%02X ", "\n" );
        bKeyEntryLen += 2;

        printf ( "\tExtSET              : " );
        if ( ( stHal_SamAv3.bHostMode == 0x02 ) )
        {
            printf  ( "%02X\n", aKeyEntry[bKeyEntryLen] );
        }
        else
        {
            PrintData ( &aKeyEntry[bKeyEntryLen], 2, "%02X ", "\n" );
            bKeyEntryLen += 2;

            printf ( "\tKeyNoAEK            : %02X\n", aKeyEntry[bKeyEntryLen++] );
            printf ( "\tKeyVAEK             : %02X\n", aKeyEntry[bKeyEntryLen++] );
        }
    }

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_ChangeKeyEntry ( uint8_t bKeyNo, uint8_t bKeyType, uint8_t bKeyClass, uint8_t *pKey, uint32_t dwKeyLen, uint8_t bDisplayContents )
{
    uint8_t aKeyEntry[64];
    uint8_t bKeyEntryLen = 61;

    /* Print the existing key entry information. */
    if ( bDisplayContents )
    {
        printf ( "\tKeyEntry information before ChangeKeyEntry command      : " );
        CHECK_SUCCESS ( Demo_GetKeyEntry ( bKeyNo, 0x00 ) );
    }

    /* Clear the buffer. */
    memset ( aKeyEntry, 0x00, sizeof ( aKeyEntry ) );

    /* Copy the Key. */
    memcpy ( aKeyEntry, pKey, dwKeyLen );

    /* Fill the key entry buffer with required values. */
    aKeyEntry[52] = 0xFE;       /* KeyNoCEK */
    aKeyEntry[54] = 0xFF;       /* RefNoKUC */
    aKeyEntry[55] = bKeyType;   /* SET */
    aKeyEntry[56] = 0x00;       /* SET */
    aKeyEntry[60] = bKeyClass;  /* ExtSET */
    aKeyEntry[61] = 0x00;       /* ExtSET */
    aKeyEntry[62] = 0xFE;       /* KeyNoAEK */

    /* Update Set and ExtSET with LockKey enabled. */
    if ( bKeyClass == KEY_CLASS_HOST )
    {
        aKeyEntry[56] |= 0x01 /* Enable Auth Key */;
        aKeyEntry[56] |= 0x04 /* Enable Lock Key */;

        if ( bKeyNo == SAM_MASTER_KEY )
            aKeyEntry[55] |= 0x80 /* Enable PL Key */;
    }

    /* Update Set and ExtSET with DumpSessionKey and DumpSecretKey enabled. */
    if ( bKeyClass == KEY_CLASS_PICC )
    {
        if ( bKeyType != KEY_TYPE_ISO_TDEA16 )
        {
            aKeyEntry[55] |= 0x01 /* Allow Dump SessionKey */;
            aKeyEntry[60] |= 0x08 /* Allow Dump SecretKey */;
        }

        if ( bKeyType == KEY_TYPE_MIFARE )
        {
            aKeyEntry[6]  = 0x00;
            aKeyEntry[7]  = 0x00;
            aKeyEntry[14] = 0x00;
            aKeyEntry[15] = 0x00;
        }

        if ( bKeyType == KEY_TYPE_ISO_TDEA16 )
        {
            aKeyEntry[55] |= 0x04;
        }
    }

    /* Update KeyEntry buffer if Host is AV3. */
    if ( stHal_SamAv3.bHostMode == 0x03)
        bKeyEntryLen = 64;

    /* Update the key entry. */
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_ChangeKeyEntry ( &stHal_SamAv3, bKeyNo, ( ( bKeyNo == RAM_KEY_ADDRESS ) ? 0x8F : 0xFF ),
        aKeyEntry, bKeyEntryLen ) );

    /* Print the update key entry information. */
    if ( bDisplayContents )
    {
        printf ( "\tKeyEntry information after ChangeKeyEntry command       : " );
        CHECK_SUCCESS ( Demo_GetKeyEntry ( bKeyNo, 0x00 ) );
    }

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_GetKUCEntry ( uint8_t bDisplayContents )
{
    uint8_t aKUCEntry[20];
    uint8_t bKUCEntryLen = 0;

    /* Get the KeyEntry information. */
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_GetKUCEntry ( &stHal_SamAv3, PICC_AES128_KEY_ADDRESS, aKUCEntry ) );

    if ( bDisplayContents ) printf ( "\tKUCEntry information: " );
    PrintData ( aKUCEntry, 10, "%02X ", "\n" );

    if ( bDisplayContents )
    {
        printf ( "\tLimit               : " );
        PrintData ( &aKUCEntry[bKUCEntryLen], 4, "%02X ", "\n" );
        bKUCEntryLen += 4;

        printf ( "\tKeyNoCKUC           : %02X\n", aKUCEntry[bKUCEntryLen++] );
        printf ( "\tKeyVCKUC            : %02X\n", aKUCEntry[bKUCEntryLen++] );

        printf ( "\tCurVal              : " );
        PrintData ( &aKUCEntry[bKUCEntryLen], 4, "%02X ", "\n" );
    }

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_ChangeKUCEntry ()
{
    uint8_t aKUCEntry[6];
    uint8_t bKUCEntryLen = 6;

    /* Clear the buffer. */
    memset ( aKUCEntry, 0x00, sizeof ( aKUCEntry ) );

    /* Print the existing key entry information. */
    printf ( "\tKeyEntry information before ChangeKUCEntry command      : " );
    CHECK_SUCCESS ( Demo_GetKUCEntry ( 0x00 ) );

    /* Fill the kuc entry buffer with required values. */
    aKUCEntry[0] = 0x00;    /* Limit */
    aKUCEntry[1] = 0x00;    /* Limit */
    aKUCEntry[2] = 0x00;    /* Limit */
    aKUCEntry[3] = 0x01;    /* Limit */
    aKUCEntry[4] = 0xFE;    /* KeyNoCKUC */
    aKUCEntry[5] = 0x00;    /* KeyVCKUC */

    /* Update the key entry. */
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_ChangeKUCEntry ( &stHal_SamAv3, PICC_AES128_KEY_ADDRESS, 0xE0, aKUCEntry, bKUCEntryLen ) );

    /* Print the update key entry information. */
    printf ( "\tKeyEntry information after ChangeKUCEntry command       : " );
    CHECK_SUCCESS ( Demo_GetKUCEntry ( 0x00 ) );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_DumpSessionKey ()
{
    uint8_t aSessionKey[16];
    uint8_t bSessionKeyLen = 0;

    /* Clear the buffer. */
    memset ( aSessionKey, 0x00, sizeof ( aSessionKey ) );

    /* Perform ChangeKey Entry for PICC class. */
    printf ( "\tChanging KeyNo %d as DESFire4 PICC key            : ", PICC_DESFIRE4_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  PICC_DESFIRE4_KEY_ADDRESS, KEY_TYPE_DESFire4, KEY_CLASS_PICC, aDESFire4Key, sizeof ( aDESFire4Key ), 0x00 ) );
    printf ( "SUCCESS\n" );

    /* Activate card. */
    printf ( "\tActivate DESFire PICC                            : " );
    CHECK_SUCCESS_PRINT ( ActivateCard ( ISO14443_L4 ) );
    printf ( "SUCCESS\n" );

    /* Authenticate with PICC. */
    printf ( "\tAuthenticate with DESFire PICC                   : " );
    CHECK_SUCCESS ( MFD_Authenticate ( MFD_AUTH_TYPE_D40, PICC_DESFIRE4_KEY_ADDRESS, PICC_DESFIRE4_KEY_VERSION ) );
    printf ( "SUCCESS\n" );

    printf ( "\tSessionKey buffer before DumpSessionKey command  : " );
    PrintData ( aSessionKey, sizeof ( aSessionKey ), "%02X ", "\n" );

    /* Perform DumpSessionKey. */
    printf ( "\tPerform DumpSessionKey                           : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_DumpSessionKey ( &stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_DUMP_MODE_PLAIN, aSessionKey, &bSessionKeyLen ) );
    printf ( "SUCCESS\n" );

    printf ( "\tSessionKey buffer after DumpSessionKey command   : " );
    PrintData ( aSessionKey, bSessionKeyLen, "%02X ", "\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_DumpSecretKey ()
{
    uint8_t aSecretKey[16];
    uint8_t bSecretKeyLen = 0;

    /* Clear the buffer. */
    memset ( aSecretKey, 0x00, sizeof ( aSecretKey ) );

    /* Perform ChangeKey Entry for PICC class. */
    printf ( "\tChanging KeyNo %d as AES PICC key                 : ", PICC_AES128_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  PICC_AES128_KEY_ADDRESS, KEY_TYPE_AES128, KEY_CLASS_PICC, aAESKey, sizeof ( aAESKey ), 0x00 ) );
    printf ( "SUCCESS\n" );

    printf ( "\tSecretKey buffer before DumpSecretKey command    : " );
    PrintData ( aSecretKey, sizeof ( aSecretKey ), "%02X ", "\n" );

    /* Perform DumpSecretKey. */
    printf ( "\tPerform DumpSecretKey                            : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_DumpSecretKey ( &stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_DUMP_MODE_PLAIN, PICC_AES128_KEY_ADDRESS, PICC_AES128_KEY_VERSION, NULL, 0,
        aSecretKey, &bSecretKeyLen ) );
    printf ( "SUCCESS\n" );

    printf ( "\tSecretKey buffer after DumpSecretKey command     : " );
    PrintData ( aSecretKey, bSecretKeyLen, "%02X ", "\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_GenerateVerifyMAC ()
{
    uint8_t * pMac = NULL;
    uint16_t wMacLen = 0;

    uint8_t aInputBuffer[24] = { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF };

    /* Activate Offline Key. */
    CHECK_SUCCESS ( Demo_ActivateOfflineKey () );

    printf ( "\tPlain Data                                     : " );
    PrintData ( aInputBuffer, 16, "%02X ", "\n" );

    /* Perform GenerateMAC. */
    printf ( "\tPerform GenerateMAC                            : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_GenerateMAC ( &stHal_SamAv3, PH_EXCHANGE_DEFAULT, 0x08, aInputBuffer, 16, &pMac, &wMacLen ) );
    printf ( "SUCCESS\n" );

    printf ( "\tMAC data                                       : " );
    PrintData ( pMac, wMacLen, "%02X ", "\n" );

    /* Copy the MAC to input buffer. */
    memcpy ( &aInputBuffer[16], pMac, wMacLen );

    printf ( "\tPlain Data with MAC                            : " );
    PrintData ( aInputBuffer, 24, "%02X ", "\n" );

    /* Perform VerifyMAC. */
    printf ( "\tPerform VerifyMAC                              : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_VerifyMAC ( &stHal_SamAv3, PH_EXCHANGE_DEFAULT, 0x08, aInputBuffer, 24 ) );
    printf ( "SUCCESS\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_EncipherDecipherData ()
{
    uint8_t * pEncData = NULL;
    uint16_t wEncDataLen = 0;
    uint8_t * pDecData = NULL;
    uint16_t wDecDataLen = 0;

    uint8_t aLength[3];
    uint8_t aPlainData[16] = { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF };

    /* Activate Offline Key. */
    CHECK_SUCCESS ( Demo_ActivateOfflineKey () );

    printf ( "\tPlain Data                                     : " );
    PrintData ( aPlainData, sizeof ( aPlainData ), "%02X ", "\n" );

    /* Perform EncipherData. */
    printf ( "\tPerform EncipherData                           : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_EncipherData ( &stHal_SamAv3, PH_EXCHANGE_DEFAULT, aPlainData, ( uint8_t ) sizeof ( aPlainData ), 0, &pEncData, &wEncDataLen ) );
    printf ( "SUCCESS\n" );

    printf ( "\tEnciphered Data                                : " );
    PrintData ( pEncData, wEncDataLen, "%02X ", "\n" );

    /* Update Length Buffer. */
    aLength[0] = ( uint8_t ) ( wEncDataLen & 0x00FF );
    aLength[1] = ( uint8_t ) ( ( wEncDataLen & 0xFF ) >> 8 );
    aLength[2] = 0x00;

    /* Perform DecipherData. */
    printf ( "\tPerform DecipherData                           : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_DecipherData ( &stHal_SamAv3, PH_EXCHANGE_DEFAULT, pEncData, ( uint8_t ) wEncDataLen, aLength, &pDecData, &wDecDataLen ) );
    printf ( "SUCCESS\n" );

    printf ( "\tDeciphered Data                                : " );
    PrintData ( pDecData, wDecDataLen, "%02X ", "\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_EncipherDecipherDataOffline ()
{
    uint8_t * pEncData = NULL;
    uint16_t wEncDataLen = 0;
    uint8_t * pDecData = NULL;
    uint16_t wDecDataLen = 0;

    uint8_t aLength[3];
    uint8_t aPlainData[16] = { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF };

    /* Activate Offline Key. */
    CHECK_SUCCESS ( Demo_ActivateOfflineKey () );

    printf ( "\tPlain Data                                     : " );
    PrintData ( aPlainData, sizeof ( aPlainData ), "%02X ", "\n" );

    /* Perform EncipherOfflineData. */
    printf ( "\tPerform EncipherOfflineData                    : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_EncipherOfflineData ( &stHal_SamAv3, PH_EXCHANGE_DEFAULT, aPlainData, ( uint8_t ) sizeof ( aPlainData ), &pEncData, &wEncDataLen ) );
    printf ( "SUCCESS\n" );

    printf ( "\tEnciphered Data                                : " );
    PrintData ( pEncData, wEncDataLen, "%02X ", "\n" );

    /* Update Length Buffer. */
    aLength[0] = ( uint8_t ) ( wEncDataLen & 0x00FF );
    aLength[1] = ( uint8_t ) ( ( wEncDataLen & 0xFF ) >> 8 );
    aLength[2] = 0x00;

    /* Perform DecipherOfflineData. */
    printf ( "\tPerform DecipherOfflineData                    : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_DecipherOfflineData ( &stHal_SamAv3, PH_EXCHANGE_DEFAULT, pEncData, ( uint8_t ) wEncDataLen, &pDecData, &wDecDataLen ) );
    printf ( "SUCCESS\n" );

    printf ( "\tDeciphered Data                                : " );
    PrintData ( pDecData, wDecDataLen, "%02X ", "\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_MIFARE_DESFireEV2 ()
{
    uint8_t aAID_DES[3] = { 0x01, 0x00, 0x00 };
    uint8_t aAID_AES[3] = { 0x02, 0x00, 0x00 };

    /* Perform ChangeKey Entry for PICC class. */
    printf ( "\tChanging KeyNo %d as DES PICC key               : ", PICC_DESFIRE4_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  PICC_DESFIRE4_KEY_ADDRESS, KEY_TYPE_DESFire4, KEY_CLASS_PICC, aDESFire4Key, sizeof ( aDESFire4Key ), PH_OFF ) );
    printf ( "SUCCESS\n" );

    /* Perform ChangeKey Entry for PICC class. */
    printf ( "\tChanging KeyNo %d as DES PICC key               : ", PICC_DESFIRE4_KEY_ADDRESS_NEW );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  PICC_DESFIRE4_KEY_ADDRESS_NEW, KEY_TYPE_DESFire4, KEY_CLASS_PICC, aDESFire4Key1, sizeof ( aDESFire4Key1 ), PH_OFF ) );
    printf ( "SUCCESS\n" );

    /* Perform ChangeKey Entry for PICC class. */
    printf ( "\tChanging KeyNo %d as AES PICC key               : ", PICC_AES128_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  PICC_AES128_KEY_ADDRESS, KEY_TYPE_AES128, KEY_CLASS_PICC, aAESKey, sizeof ( aAESKey ), PH_OFF ) );
    printf ( "SUCCESS\n\n" );

    printf ( "\tDemo on Key change feature -----------------------------------------------------------------------------\n" );

    /* Activate the PICC. */
    CHECK_SUCCESS_PRINT ( ActivateCard ( ISO14443_L4 ) );

    /* Authenticate PICC. */
    printf ( "\tD40 Authentication using PICC Master Key       : " );
    CHECK_SUCCESS ( MFD_Authenticate ( MFD_AUTH_TYPE_D40, PICC_DESFIRE4_KEY_ADDRESS, PICC_DESFIRE4_KEY_VERSION ) );
    printf ( "SUCCESS\n" );

    /* Format PICC. */
    CHECK_SUCCESS ( MFD_Format () );

    /* Perform PICC ChangeKey. */
    printf ( "\tChanging PICC Master Key data                  : " );
    CHECK_SUCCESS ( MFD_ChangeKey ( PICC_DESFIRE4_KEY_ADDRESS, PICC_DESFIRE4_KEY_VERSION, PICC_DESFIRE4_KEY_ADDRESS_NEW, PICC_DESFIRE4_KEY_VERSION_NEW ) );
    printf ( "SUCCESS\n" );

    /* Authenticate PICC. */
    printf ( "\tD40 Authentication using PICC Master Key       : " );
    CHECK_SUCCESS ( MFD_Authenticate ( MFD_AUTH_TYPE_D40, PICC_DESFIRE4_KEY_ADDRESS_NEW, PICC_DESFIRE4_KEY_VERSION_NEW ) );
    printf ( "SUCCESS\n" );

    /* Perform PICC ChangeKey. */
    printf ( "\tReverting PICC Master Key data                 : " );
    CHECK_SUCCESS ( MFD_ChangeKey ( PICC_DESFIRE4_KEY_ADDRESS_NEW, PICC_DESFIRE4_KEY_VERSION_NEW, PICC_DESFIRE4_KEY_ADDRESS, PICC_DESFIRE4_KEY_VERSION ) );
    printf ( "SUCCESS\n\n" );

    printf ( "\tDemo on D40 feature ------------------------------------------------------------------------------------\n" );

    /* Authenticate PICC. */
    printf ( "\tD40 Authentication using PICC Master Key       : " );
    CHECK_SUCCESS ( MFD_Authenticate ( MFD_AUTH_TYPE_D40, PICC_DESFIRE4_KEY_ADDRESS, PICC_DESFIRE4_KEY_VERSION ) );
    printf ( "SUCCESS\n" );

    /* Perform GetVersion Command. */
    printf ( "\tGetVersion                                     : " );
    CHECK_SUCCESS ( MFD_GetVersion () );
    printf ( "\tSUCCESS\n" );

    /* Perform Create Application Command. */
    printf ( "\tCreate and Select Application                  : " );
    CHECK_SUCCESS ( MFD_CreateSelectApplication ( MFD_KEYTYPE_DES, aAID_DES ) );
    printf ( "SUCCESS\n" );

    /* Perform Create File Command. */
    printf ( "\tCreate File                                    : " );
    CHECK_SUCCESS ( MFD_CreateFile () );
    printf ( "SUCCESS\n" );

    /* Authenticate PICC. */
    printf ( "\tD40 Authentication using PICC Master Key       : " );
    CHECK_SUCCESS ( MFD_Authenticate ( MFD_AUTH_TYPE_D40, PICC_DESFIRE4_KEY_ADDRESS, PICC_DESFIRE4_KEY_VERSION ) );
    printf ( "SUCCESS\n" );

    /* Perform Write Command. */
    printf ( "\tWrite data                                     : " );
    CHECK_SUCCESS ( MFD_Write () );
    printf ( "\t\tSUCCESS\n" );

    /* Perform Read Command. */
    printf ( "\tRead data                                      : " );
    CHECK_SUCCESS ( MFD_Read () );
    printf ( "\t\tSUCCESS\n\n" );

    printf ( "\tDemo on EV2 feature ------------------------------------------------------------------------------------\n" );

    if (IncludeDemo_Version ( 0x03 ) == PH_ERR_SUCCESS )
     {
        /* Activate the PICC. */
        CHECK_SUCCESS_PRINT ( ActivateCard ( ISO14443_L4 ) );

        /* Authenticate PICC. */
        printf ( "\tD40 Authentication using PICC Master Key       : " );
        CHECK_SUCCESS ( MFD_Authenticate ( MFD_AUTH_TYPE_D40, PICC_DESFIRE4_KEY_ADDRESS, PICC_DESFIRE4_KEY_VERSION ) );
        printf ( "SUCCESS\n" );

        /* Perform GetVersion Command. */
        printf ( "\tGetVersion                                     : " );
        CHECK_SUCCESS ( MFD_GetVersion () );
        printf ( "\tSUCCESS\n" );

        /* Perform Create Application Command. */
        printf ( "\tCreate and Select Application                  : " );
        CHECK_SUCCESS ( MFD_CreateSelectApplication ( MFD_KEYTYPE_AES, aAID_AES ) );
        printf ( "SUCCESS\n" );

        /* Perform Create File Command. */
        printf ( "\tCreate File                                    : " );
        CHECK_SUCCESS ( MFD_CreateFile () );
        printf ( "SUCCESS\n" );

        /* Authenticate PICC. */
        printf ( "\tEV2_First Authentication using PICC Master Key : " );
        CHECK_SUCCESS ( MFD_Authenticate ( MFD_AUTH_TYPE_EV2_FIRST, PICC_AES128_KEY_ADDRESS, PICC_AES128_KEY_VERSION ) );
        printf ( "SUCCESS\n" );

        /* Perform Write Command. */
        printf ( "\tWrite data                                     : " );
        CHECK_SUCCESS ( MFD_Write () );
        printf ( "\t\tSUCCESS\n" );

        /* Perform Read Command. */
        printf ( "\tRead data                                      : " );
        CHECK_SUCCESS ( MFD_Read () );
        printf ( "\t\tSUCCESS\n" );
     }

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_MIFARE_PlusEV1 ()
{
    /* Perform ChangeKey Entry for PICC class. */
    printf ( "\tChanging KeyNo %d as AES128 PICC key            : ", PICC_AES128_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  PICC_AES128_KEY_ADDRESS, KEY_TYPE_AES128, KEY_CLASS_PICC, aAESKey, sizeof ( aAESKey ), PH_OFF ) );
    printf ( "SUCCESS\n" );

    /* Perform ChangeKey Entry for PICC class. */
    printf ( "\tChanging KeyNo %d as AES128 PICC key            : ", PICC_AES128_KEY_ADDRESS_NEW );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  PICC_AES128_KEY_ADDRESS_NEW, KEY_TYPE_AES128, KEY_CLASS_PICC, aAESKey1, sizeof ( aAESKey1 ), PH_OFF ) );
    printf ( "SUCCESS\n\n" );

    printf ( "\tDemo on Key change feature -----------------------------------------------------------------------------\n" );

    /* Activate the PICC. */
    CHECK_SUCCESS_PRINT ( ActivateCard ( ISO14443_L4 ) );

    /* Authenticate PICC. */
    printf ( "\tAuthentication to block number 0x4000          : " );
    CHECK_SUCCESS ( MFP_Authenticate ( MFP_EV0_SECURE_MESSAGING, PICC_AES128_KEY_ADDRESS, PICC_AES128_KEY_VERSION ) );
    printf ( "SUCCESS\n" );

    /* Perform PICC ChangeKey. */
    printf ( "\tChanging key of block number 0x4000            : " );
    CHECK_SUCCESS ( MFP_ChangeKey ( PICC_AES128_KEY_ADDRESS_NEW, PICC_AES128_KEY_VERSION_NEW ) );
    printf ( "SUCCESS\n" );

    /* Authenticate PICC. */
    printf ( "\tAuthentication to block number 0x4000          : " );
    CHECK_SUCCESS ( MFP_Authenticate ( MFP_EV0_SECURE_MESSAGING, PICC_AES128_KEY_ADDRESS_NEW, PICC_AES128_KEY_VERSION_NEW ) );
    printf ( "SUCCESS\n" );

    /* Perform PICC ChangeKey. */
    printf ( "\tReverting key of block number 0x4000           : " );
    CHECK_SUCCESS ( MFP_ChangeKey ( PICC_AES128_KEY_ADDRESS, PICC_AES128_KEY_VERSION ) );
    printf ( "SUCCESS\n\n" );

    printf ( "\tDemo on EV0 SM feature ---------------------------------------------------------------------------------\n" );

    /* Authenticate PICC. */
    printf ( "\tAuthentication to block number 0x4000          : " );
    CHECK_SUCCESS ( MFP_Authenticate ( MFP_EV0_SECURE_MESSAGING, PICC_AES128_KEY_ADDRESS, PICC_AES128_KEY_VERSION ) );
    printf ( "SUCCESS\n" );

    /* Perform GetVersion Command. */
    printf ( "\tGetVersion                                     : " );
    if ( ( stHal_SamAv3.bHostMode == 0x03 ) )
    {
        CHECK_SUCCESS ( MFP_Read ( MFP_COMBINED_READ_GET_VERSION ) );
        printf ( "\tSUCCESS\n" );
    }
    else
    {
        printf ( "Not Supported with Sam AV2. \n" );
    }

    /* Perform Write Command. */
    printf ( "\tWrite to block number 1                        : " );
    CHECK_SUCCESS ( MFP_Write () );
    printf ( "\tSUCCESS\n" );

    /* Perform Read Command. */
    printf ( "\tRead from block number 1                       : " );
    CHECK_SUCCESS ( MFP_Read ( MFP_COMBINED_READ_READ ) );
    printf ( "\tSUCCESS\n\n" );

    if (IncludeDemo_Version ( 0x03 ) == PH_ERR_SUCCESS )
    {
        printf ( "\tDemo on EV1 SM feature ---------------------------------------------------------------------------------\n" );

        /* Authenticate PICC. */
        printf ( "\tAuthentication to block number 0x4000          : " );
        CHECK_SUCCESS ( MFP_Authenticate ( MFP_EV1_SECURE_MESSAGING, PICC_AES128_KEY_ADDRESS, PICC_AES128_KEY_VERSION ) );
        printf ( "SUCCESS\n" );

        /* Perform GetVersion Command. */
        printf ( "\tGetVersion                                     : " );
        CHECK_SUCCESS ( MFP_Read ( MFP_COMBINED_READ_GET_VERSION ) );
        printf ( "\tSUCCESS\n" );

        /* Perform Write Command. */
        printf ( "\tWrite to block number 1                        : " );
        CHECK_SUCCESS ( MFP_Write () );
        printf ( "\tSUCCESS\n" );

        /* Perform Read Command. */
        printf ( "\tRead from block number 1                       : " );
        CHECK_SUCCESS ( MFP_Read ( MFP_COMBINED_READ_READ ) );
        printf ( "\tSUCCESS\n" );
    }

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_MIFARE_Classic ()
{
    /* Perform ChangeKey Entry for PICC class. */
    printf ( "\tChanging KeyNo %d as MIFARE PICC key            : ", PICC_MIFARE_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  PICC_MIFARE_KEY_ADDRESS, KEY_TYPE_MIFARE, KEY_CLASS_PICC, aClassicKey, sizeof ( aClassicKey ), PH_OFF ) );
    printf ( "SUCCESS\n" );

    /* Activate the PICC. */
    CHECK_SUCCESS_PRINT ( ActivateCard ( ISO14443_L3 ) );

    /* Authenticate PICC. */
    printf ( "\tAuthentication to block number 0               : " );
    CHECK_SUCCESS ( MFC_Authenticate () );
    printf ( "SUCCESS\n" );

    /* Perform Write Command. */
    printf ( "\tWrite to block number 1                        : " );
    CHECK_SUCCESS ( MFC_Write () );
    printf ( "\tSUCCESS\n" );

    /* Perform Read Command. */
    printf ( "\tRead from block number 1                       : " );
    CHECK_SUCCESS ( MFC_Read () );
    printf ( "\tSUCCESS\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_MIFARE_Ultralight ()
{
    int dwPICC_Choice = 0;

    printf ( "\n" );
    printf ( "\tPress 1 to peform demo with MIFARE Ultralight C card.\n" );
    printf ( "\tPress 2 to peform demo with MIFARE Ultralight EV1 card.\n" );
    printf ( "\tEnter the option and press Enter to perform the demo - " );
    scanf_s ( "%d", &dwPICC_Choice );
    dwPICC_Choice = dwPICC_Choice - 1;

    printf("\n\n");
    if( ( dwPICC_Choice != MFUL_AUTH_CLASSIC ) && ( dwPICC_Choice != MFUL_AUTH_EV1 ) )
    {
        printf ( "\tInvalid option selected. Please restart the application and select proper option.\n" );
        _getch ();
        exit ( 0 );
    }

    printf ( "\tMake sure that MIFARE Ultralight %s card is placed in field.\n", ( dwPICC_Choice ) ? "EV1" : "C" );
    printf ( "\tPress any key to proceed with the demonstration.\n" );
    _getch ();
    printf ( "\n" );

    if ( dwPICC_Choice == MFUL_AUTH_CLASSIC )
    {
        /* Perform ChangeKey Entry for PICC class. */
        printf ( "\tChanging KeyNo %d as DES PICC key            : ", PICC_MFUL_2K3DES_KEY_ADDRESS );
        CHECK_SUCCESS ( Demo_ChangeKeyEntry (  PICC_MFUL_2K3DES_KEY_ADDRESS, KEY_TYPE_ISO_TDEA16, KEY_CLASS_PICC, aMFUL_2K3DES_Key, sizeof ( aMFUL_2K3DES_Key ), PH_OFF ) );
        printf ( "SUCCESS\n" );
    }

    if ( dwPICC_Choice == MFUL_AUTH_EV1 )
    {
        /* Perform ChangeKey Entry for PICC class. */
        printf ( "\tChanging KeyNo %d as AES PICC key           : ", PICC_MFUL_AES_KEY_ADDRESS );
        CHECK_SUCCESS ( Demo_ChangeKeyEntry (  PICC_MFUL_AES_KEY_ADDRESS, KEY_TYPE_AES128, KEY_CLASS_PICC, aMFUL_AES_Key, sizeof ( aMFUL_AES_Key ), PH_OFF ) );
        printf ( "SUCCESS\n" );
    }

    /* Activate the PICC. */
    CHECK_SUCCESS_PRINT ( ActivateCard ( ISO14443_L3 ) );

    /* Authenticate PICC. */
    printf ( "\tAuthentication                              : " );
    CHECK_SUCCESS ( MFUL_Authenticate ( ( uint8_t ) dwPICC_Choice ) );
    printf ( "SUCCESS\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_ISO_IEC29167 ()
{
    /* Perform ChangeKey Entry for PICC class. */
    printf ( "\tChanging KeyNo %d as AES128 PICC key            : ", PICC_AES128_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  PICC_AES128_KEY_ADDRESS, KEY_TYPE_AES128, KEY_CLASS_PICC, aAESKey, sizeof ( aAESKey ), PH_OFF ) );
    printf ( "SUCCESS\n" );

    /* Activate the tag. */
    CHECK_SUCCESS_PRINT ( ActivateCard ( ISO15693 ) );

    /* Perform TAM Authentication. */
    printf ( "\tTAM Authentication                             : " );
    CHECK_SUCCESS ( ICODE_AuthenticateTAM () );
    printf ( "SUCCESS\n" );

    /* Perform MAM Authentication. */
    printf ( "\tMAM Authentication                             : " );
    CHECK_SUCCESS ( ICODE_AuthenticateMAM () );
    printf ( "SUCCESS\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_TransactionMAC ( uint32_t dwFeature )
{
    uint32_t dwChoice = 0;
    uint32_t dwAuthType = 0;
    uint8_t bKeyType = 0;

    printf ( "\tPress 1 to perform TMAC with Authuentication mode as EV2. \n" );
    printf ( "\tPress 2 to perform TMAC with Authuentication mode as LRP. \n" );
    dwAuthType = _getch ();
    printf("\n");

    /* Check if proper option is selected. */
    if ( ( dwAuthType != '1' ) && ( dwAuthType != '2' ) )
    {
        printf ( "\tInvalid Authuentication mode selected. \n" );
        return 0xFF;
    }
    else
    {
        if ( ( enum Demo ) dwFeature == TMAC ) printf ( "\tMake sure MIFARE DESFire Light PICC has Transaction MAC file.\n" );

        printf ( "\tMake sure MIFARE DESFire Light PICC is configured to %s mode. \n", ( ( dwAuthType == 0x31 ) ? "EV2" : "LRP" ) );
        printf ( "\tPress any key to continue with the demo. \n" );
        dwChoice = _getch ();
        printf("\n\n");
    }

    /* Compute the KeyType to be changed. */
    bKeyType = ( uint8_t ) ( ( dwAuthType == 0x31 ) ? KEY_TYPE_AES128 : KEY_TYPE_AES128_LRP );
    bKeyType = ( uint8_t ) ( ( dwFeature == TMAC ) ? KEY_TYPE_AES128 : bKeyType );

    /* Perform some commands to verify the unlocked state. */
    bCommMode = 1;
    printf ( "\tPerform Host Authentication                     : " );
    CHECK_SUCCESS ( Demo_HostAuthentication () );
    printf ( "SUCCESS\n" );

    /* Perform ChangeKey Entry for PICC class. */
    printf ( "\tChanging KeyNo %02X as AES PICC key               : ", PICC_AES128_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  PICC_AES128_KEY_ADDRESS, ( ( dwAuthType == 0x31 ) ? KEY_TYPE_AES128 : KEY_TYPE_AES128_LRP ),
        KEY_CLASS_PICC, aAESKey, sizeof ( aAESKey ), PH_OFF ) );
    printf ( "SUCCESS\n" );

    /* Perform ChangeKey Entry for OfflineCrypto class. */
    printf ( "\tChanging KeyNo %02X as AES OfflineCrypto key      : ", OFFLINE_CRYPTO_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  OFFLINE_CRYPTO_KEY_ADDRESS, ( ( dwAuthType == 0x31 ) ? KEY_TYPE_AES128 : KEY_TYPE_AES128_LRP ),
        KEY_CLASS_OFFLINE_CRYPTO, aAESKey, sizeof ( aAESKey ), PH_OFF ) );
    printf ( "SUCCESS\n" );

    /* Perform ChangeKey Entry for OfflineCrypto class. */
    printf ( "\tChanging KeyNo %02X as AES RAM OfflineCrypto key  : ", RAM_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  RAM_KEY_ADDRESS, bKeyType, KEY_CLASS_OFFLINE_CRYPTO, aAESKey, sizeof ( aAESKey ), PH_OFF ) );
    printf ( "SUCCESS\n\n" );

    /* Activate PICC. */
    printf ( "\tPICC Activation                                 : " );
    CHECK_SUCCESS_PRINT ( ActivateCard ( ISO14443_L4 ) );
    printf ( "SUCCESS\n" );

    /* Select the PICC. */
    printf ( "\tPICC Selection                                  : " );
    CHECK_SUCCESS ( PRIME_ISOSelect ( NULL, 0, PICC_DF_NAME_DFL, 7, 4 ) );
    printf ( "SUCCESS\n" );

    /* Select the Application. */
    printf ( "\tApplication Selection                           : " );
    CHECK_SUCCESS ( PRIME_ISOSelect ( APP_ISO_FILE_ID_DFL, 0, APP_DF_NAME_DFL, 16, 4 ) );
    printf ( "SUCCESS\n" );

    /* Perform TAMC feature. */
    CHECK_SUCCESS ( PRIME_TMAC ( ( uint8_t ) ( ( dwAuthType - 0x30 ) - 1 ) ) );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_SecureDynamicMessaging ( uint32_t dwFeature )
{
    uint32_t dwChoice = 0;
    uint32_t dwAuthType = 0;
    uint8_t bKeyType = 0;

    printf ( "\tPress 1 to perform TMAC with Authuentication mode as EV2. \n" );
    printf ( "\tPress 2 to perform TMAC with Authuentication mode as LRP. \n" );
    dwAuthType = _getch ();
    printf("\n");

    /* Check if proper option is selected. */
    if ( ( dwAuthType != '1' ) && ( dwAuthType != '2' ) )
    {
        printf ( "\tInvalid Authuentication mode selected. \n" );
        return 0xFF;
    }
    else
    {
        printf ( "\tMake sure NTAG 42x DNA tag is configured to %s mode. \n", ( ( dwAuthType == 0x31 ) ? "EV2" : "LRP" ) );
        printf ( "\tPress any key to continue with the demo. \n" );
        dwChoice = _getch ();
        printf("\n\n");
    }

    /* Compute the KeyType to be changed. */
    bKeyType = ( uint8_t ) ( ( dwAuthType == 0x31 ) ? KEY_TYPE_AES128 : KEY_TYPE_AES128_LRP );
    bKeyType = ( uint8_t ) ( ( dwFeature == TMAC ) ? KEY_TYPE_AES128 : bKeyType );

    /* Perform some commands to verify the unlocked state. */
    bCommMode = 1;
    printf ( "\tPerform Host Authentication                     : " );
    CHECK_SUCCESS ( Demo_HostAuthentication () );
    printf ( "SUCCESS\n" );

    /* Perform ChangeKey Entry for PICC class. */
    printf ( "\tChanging KeyNo %02X as AES PICC key               : ", PICC_AES128_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  PICC_AES128_KEY_ADDRESS, ( ( dwAuthType == 0x31 ) ? KEY_TYPE_AES128 : KEY_TYPE_AES128_LRP ),
        KEY_CLASS_PICC, aAESKey, sizeof ( aAESKey ), PH_OFF ) );
    printf ( "SUCCESS\n" );

    /* Perform ChangeKey Entry for OfflineCrypto class. */
    printf ( "\tChanging KeyNo %02X as AES OfflineCrypto key      : ", OFFLINE_CRYPTO_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  OFFLINE_CRYPTO_KEY_ADDRESS, ( ( dwAuthType == 0x31 ) ? KEY_TYPE_AES128 : KEY_TYPE_AES128_LRP ),
        KEY_CLASS_OFFLINE_CRYPTO, aAESKey, sizeof ( aAESKey ), PH_OFF ) );
    printf ( "SUCCESS\n" );

    /* Perform ChangeKey Entry for OfflineCrypto class. */
    printf ( "\tChanging KeyNo %02X as AES RAM OfflineCrypto key  : ", RAM_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  RAM_KEY_ADDRESS, bKeyType, KEY_CLASS_OFFLINE_CRYPTO, aAESKey, sizeof ( aAESKey ), PH_OFF ) );
    printf ( "SUCCESS\n\n" );

    printf ( "\tPICC Activation                                 : " );
    CHECK_SUCCESS_PRINT ( ActivateCard ( ISO14443_L4 ) );
    printf ( "SUCCESS\n" );

    /* Select the PICC. */
    printf ( "\tPICC Selection                                  : " );
    CHECK_SUCCESS ( PRIME_ISOSelect ( NULL, 0, PICC_DF_NAME_NTAG42X, 7, 4 ) );
    printf ( "SUCCESS\n" );

    /* Select the Application. */
    printf ( "\tApplication Selection                           : " );
    CHECK_SUCCESS ( PRIME_ISOSelect ( APP_ISO_FILE_ID_NTAG42X, 0, APP_DF_NAME_NTAG42X, 7, 4 ) );
    printf ( "SUCCESS\n" );

    /* Perform SDM feature. */
    CHECK_SUCCESS ( PRIME_SDM ( ( uint8_t ) ( ( dwAuthType - 0x30 ) - 1 ) ) );
    printf ( "\n" );

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_ProgrammableLogic ()
{
    phStatus_t wStatus = 0;
    uint32_t dwUploadCtr = 0;
    uint8_t * pPLCode = NULL;
    uint16_t wPLCodeLen = 0;
    uint16_t wIteration = 0;

    uint8_t aPLData[20];
    uint8_t bPLDataLen = 0;

    uint8_t aMac[8];
    uint8_t * pPLResp = NULL;
    uint16_t wPLRespLen = 0;

    uint8_t * pMAC = NULL;
    uint16_t wMacLen = 0;

    FILE * pFile;

    /* Detect the size of the file. */
    CHECK_SUCCESS_PRINT ( PL_DetectFileSize ( &pFile, &wPLCodeLen ) );

    /* Extract the PLCode from the HEX file. */
    pPLCode = ( uint8_t * ) malloc ( wPLCodeLen );
    wStatus = PL_ExtractPLCode ( &pFile, pPLCode, &wPLCodeLen );
    if ( wStatus != PH_ERR_SUCCESS )
    {
        free ( pPLCode );
    }

    /* Get the UploadCtr vlaue form the user. */
    printf ( "\tPlease provide the Upload Counter value to be used for PLUpload command execution.\n" );
    printf ( "\tEnter the UploadCtr value in decimal format and press Enter : " );
    scanf_s ( "%d", &dwUploadCtr );
    printf("\n\n");

    /* Load the keys to software keystore. */
    CHECK_SUCCESS_PRINT ( phKeyStore_FormatKeyEntry ( &stKeystoreSw, OFFLINE_UPLOAD_KEY_ADDRESS, PH_KEYSTORE_KEY_TYPE_AES128 ) );
    CHECK_SUCCESS_PRINT ( phKeyStore_FormatKeyEntry ( &stKeystoreSw, UNLOCK_PL_KEY_ADDRESS, PH_KEYSTORE_KEY_TYPE_AES128 ) );
    CHECK_SUCCESS_PRINT ( phKeyStore_SetKeyAtPos ( &stKeystoreSw, OFFLINE_UPLOAD_KEY_ADDRESS, 0, PH_KEYSTORE_KEY_TYPE_AES128, aOfflineUploadKey, OFFLINE_UPLOAD_KEY_VERSION) );
    CHECK_SUCCESS_PRINT ( phKeyStore_SetKeyAtPos ( &stKeystoreSw, UNLOCK_PL_KEY_ADDRESS, 0, PH_KEYSTORE_KEY_TYPE_AES128, aPLUploadKey, UNLOCK_PL_KEY_VERSION) );

    /* Perform ChangeKey Entry for Offline class. */
    printf ( "\tChanging KeyNo %d as AES128 Offline Crypto key       : ", ISO_TDEA_32_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  ISO_TDEA_32_KEY_ADDRESS, KEY_TYPE_ISO_TDEA32, KEY_CLASS_OFFLINE_CRYPTO, aAESKey, sizeof ( aAESKey ), PH_OFF ) );
    printf ( "SUCCESS\n" );

    /* Perform ChangeKey Entry for Offline class. */
    printf ( "\tChanging KeyNo %d as AES128 Offline Upload key        : ", OFFLINE_UPLOAD_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  OFFLINE_UPLOAD_KEY_ADDRESS, KEY_TYPE_AES128, KEY_CLASS_OFFLINE_UPLOAD, aOfflineUploadKey, sizeof ( aOfflineUploadKey ), PH_OFF ) );
    printf ( "SUCCESS\n" );

    /* Perform ChangeKey Entry for Host class. */
    printf ( "\tChanging KeyNo %d as AES128 Host key with PL Enabled  : ", SAM_MASTER_KEY_ADDRESS );
    CHECK_SUCCESS ( Demo_ChangeKeyEntry (  SAM_MASTER_KEY_ADDRESS, KEY_TYPE_AES128, KEY_CLASS_HOST, aHostAuthKey, sizeof ( aHostAuthKey ), PH_OFF ) );
    printf ( "SUCCESS\n" );

    /* Perform LockUnlock to enable Programmable Logic feature. */
    printf ( "\tPerform LockUnlock for UnlockPL option               : " );
    CHECK_SUCCESS ( phhalHw_SamAV3_Cmd_SAM_LockUnlock ( &stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_LOCK_UNLOCK_TYPE_UNLOCK_PL, UNLOCK_PL_KEY_ADDRESS, UNLOCK_PL_KEY_VERSION,
        SAM_UNLOCK_PL_KEY, 0, SAM_UNLOCK_PL_KEY, 0, 0 ) );
    printf ( "SUCCESS\n" );

    /* Perform Host Authentication with the PL key enabled. */
    if ( !bCommMode ) bCommMode = 1;
    printf ( "\tPerform Host Authentication with PL Key enabled      : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_AuthenticateHost ( &stHal_SamAv3, ( uint8_t ) ( bCommMode - 1 ), SAM_MASTER_KEY_ADDRESS, SAM_MASTER_KEY_VERSION,
        SAM_MASTER_KEY_ADDRESS, 0 ) );
    printf ( "SUCCESS\n" );

    /* Perform Offline Activation. */
    printf ( "\tPerform Activation using Offline Upload Key          : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey ( &stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
        OFFLINE_UPLOAD_KEY_ADDRESS, OFFLINE_UPLOAD_KEY_VERSION, NULL, 0 ) );
    printf ( "SUCCESS\n" );

    /* Upload the PLCode. */
    printf ( "\nUploading Hex file contents --------------------------------------------------------------------------------------------------------------------------------------- \n" );
    printf ( "\tUpload the Hex file information                      : " );
    for ( wIteration = 0; wIteration < ( uint16_t ) ( wPLCodeLen / 32 ); wIteration++ )
    {
        PrintData ( &pPLCode[wIteration * 32], 32, "%02X ", "\n" );
        if ( wIteration < ( uint16_t ) ( ( wPLCodeLen / 32 ) - 1 ) ) printf ( "\t                                                     : " );
    }
    wStatus = phhalHw_SamAV3_Cmd_SAM_PLUpload ( &stHal_SamAv3, PH_ON, PH_ON, ( uint16_t ) dwUploadCtr, OFFLINE_UPLOAD_KEY_ADDRESS, OFFLINE_UPLOAD_KEY_VERSION, pPLCode,
        wPLCodeLen, NULL, 0 );
    free ( pPLCode );
    CHECK_SUCCESS_PRINT ( wStatus );
    printf ( "\t                                                     : SUCCESS\n" );

    printf ( "\nExecuting commands with respect to the uploaded Hex file ---------------------------------------------------------------------------------------------------------- \n" );

    printf ( "\tReset Quota. Quota ID = 00 01, Amount = 00 00 00 64\n" );
    bPLDataLen = 0;
    aPLData[bPLDataLen++] = 0x01 /* Reset Quota command */; memcpy ( &aPLData[bPLDataLen], stHal_SamAv3.bUid, 7 ) /* UID */; bPLDataLen += 7;
    aPLData[bPLDataLen++] = 0x00 /* Quota ID */; aPLData[bPLDataLen++] = 0x01 /* Quota ID */; aPLData[bPLDataLen++] = 0x00; /* Amount */
    aPLData[bPLDataLen++] = 0x00; /* Amount */ aPLData[bPLDataLen++] = 0x00; /* Amount */ aPLData[bPLDataLen++] = 0x64; /* Amount */
    printf ( "\t\tCommand  : " ); PrintData ( aPLData, bPLDataLen, "%02X ", "" ); printf ( "\n" );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_PLExec ( &stHal_SamAv3, PHHAL_HW_CMD_SAMAV3_PROG_LOGIC_LFI_LAST, aPLData, bPLDataLen, &pPLResp, &wPLRespLen ) );
    printf ( "\t\tResponse : " ); PrintData ( pPLResp, wPLRespLen, "%02X ", "" );  printf ( "\n\n" );

    printf ( "\tReset Quota. Quota ID = 00 02, Amount = 00 00 00 5A\n" );
    bPLDataLen = 0;
    aPLData[bPLDataLen++] = 0x01 /* Reset Quota command */; memcpy ( &aPLData[bPLDataLen], stHal_SamAv3.bUid, 7 ) /* UID */; bPLDataLen += 7;
    aPLData[bPLDataLen++] = 0x00 /* Quota ID */; aPLData[bPLDataLen++] = 0x02 /* Quota ID */; aPLData[bPLDataLen++] = 0x00; /* Amount */
    aPLData[bPLDataLen++] = 0x00; /* Amount */ aPLData[bPLDataLen++] = 0x00; /* Amount */ aPLData[bPLDataLen++] = 0x5A; /* Amount */
    printf ( "\t\tCommand  : " ); PrintData ( aPLData, bPLDataLen, "%02X ", "" ); printf ( "\n" );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_PLExec ( &stHal_SamAv3, PHHAL_HW_CMD_SAMAV3_PROG_LOGIC_LFI_LAST, aPLData, bPLDataLen, &pPLResp, &wPLRespLen ) );
    printf ( "\t\tResponse : " ); PrintData ( pPLResp, wPLRespLen, "%02X ", "" );  printf ( "\n\n" );

    printf ( "\tExport Quota. Quota ID = 00 01, Transfer Amount = 00 00 00 32\n" );
    bPLDataLen = 0;
    aPLData[bPLDataLen++] = 0x02 /* Export Quota command */; memcpy ( &aPLData[bPLDataLen], stHal_SamAv3.bUid, 7 ) /* UID */; bPLDataLen += 7;
    aPLData[bPLDataLen++] = 0x00 /* Quota ID */; aPLData[bPLDataLen++] = 0x01 /* Quota ID */; aPLData[bPLDataLen++] = 0x00; /* Amount */
    aPLData[bPLDataLen++] = 0x00; /* Amount */ aPLData[bPLDataLen++] = 0x00; /* Amount */ aPLData[bPLDataLen++] = 0x32; /* Amount */
    printf ( "\t\tCommand  : " ); PrintData ( aPLData, bPLDataLen, "%02X ", "" ); printf ( "\n" );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_PLExec ( &stHal_SamAv3, PHHAL_HW_CMD_SAMAV3_PROG_LOGIC_LFI_LAST, aPLData, bPLDataLen, &pPLResp, &wPLRespLen ) );
    printf ( "\t\tResponse : " ); PrintData ( pPLResp, wPLRespLen, "%02X ", "" );  printf ( "\n\n" );

    printf ( "\tImport Quota. Quota ID = 00 02, Transfer Amount = 00 00 00 32\n" );
    bPLDataLen = 0;
    aPLData[bPLDataLen++] = 0x03 /* Import Quota command */; memcpy ( &aPLData[bPLDataLen], stHal_SamAv3.bUid, 7 ) /* UID */; bPLDataLen += 7;
    aPLData[bPLDataLen++] = 0x00 /* Quota ID */; aPLData[bPLDataLen++] = 0x02 /* Quota ID */; aPLData[bPLDataLen++] = 0x00; /* Amount */
    aPLData[bPLDataLen++] = 0x00; /* Amount */ aPLData[bPLDataLen++] = 0x00; /* Amount */ aPLData[bPLDataLen++] = 0x32; /* Amount */
    printf ( "\t\tCommand  : " ); PrintData ( aPLData, bPLDataLen, "%02X ", "" ); printf ( "\n" );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_PLExec ( &stHal_SamAv3, PHHAL_HW_CMD_SAMAV3_PROG_LOGIC_LFI_LAST, aPLData, bPLDataLen, &pPLResp, &wPLRespLen ) );
    printf ( "\t\tResponse : " ); PrintData ( pPLResp, wPLRespLen, "%02X ", "" );  printf ( "\n\n" );

    /* Perform Offline Activation. */
    printf ( "\tPerform Activation using Offline Crypto Key         : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey ( &stHal_SamAv3, PHHAL_HW_SAMAV3_CMD_SAM_AO_LRP_UPDATE_KEY_RFU,
        ISO_TDEA_32_KEY_ADDRESS, ISO_TDEA_32_KEY_VERSION, NULL, 0 ) );
    printf ( "SUCCESS\n" );

    /* Compute the MAC. */
    bPLDataLen = 0;
    aPLData[bPLDataLen++] = 0x00; /* Amount */ aPLData[bPLDataLen++] = 0x00; /* Amount */ aPLData[bPLDataLen++] = 0x00; /* Amount */ aPLData[bPLDataLen++] = 0x32; /* Amount */
    printf ( "\tPerform GenerateMAC                                 : " );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_GenerateMAC ( &stHal_SamAv3, PH_EXCHANGE_DEFAULT, PHHAL_HW_SAMAV3_TRUNCATION_MODE_STANDARD, aPLData, bPLDataLen, &pMAC, &wMacLen ) );
    printf ( "Mac      : " ); PrintData ( pMAC, wMacLen, "%02X ", "" );  printf ( "\n\n" );

    printf ( "\tDelete Quota. Quota ID = 00 01, KeyNo = %02X, KeyVer = %02X\n", ISO_TDEA_32_KEY_ADDRESS, ISO_TDEA_32_KEY_VERSION );
    bPLDataLen = 0;
    aPLData[bPLDataLen++] = 0x05 /* Delete Quota command */; memcpy ( &aPLData[bPLDataLen], stHal_SamAv3.bUid, 7 ) /* UID */; bPLDataLen += 7;
    aPLData[bPLDataLen++] = 0x00 /* Quota ID */; aPLData[bPLDataLen++] = 0x01 /* Quota ID */; aPLData[bPLDataLen++] = ISO_TDEA_32_KEY_ADDRESS; /* KeyNo */
    aPLData[bPLDataLen++] = ISO_TDEA_32_KEY_VERSION; /* KeyVer */
    memcpy ( &aPLData[bPLDataLen], pMAC, wMacLen ) /* Mac */; bPLDataLen += ( uint8_t ) wMacLen;
    memcpy ( aMac, pMAC, wMacLen );
    printf ( "\t\tCommand  : " ); PrintData ( aPLData, bPLDataLen, "%02X ", "" ); printf ( "\n" );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_PLExec ( &stHal_SamAv3, PHHAL_HW_CMD_SAMAV3_PROG_LOGIC_LFI_LAST, aPLData, bPLDataLen, &pPLResp, &wPLRespLen ) );
    printf ( "\t\tResponse : " ); PrintData ( pPLResp, wPLRespLen, "%02X ", "" );  printf ( "\n\n" );

    printf ( "\tAdd Quota. Quota ID = 00 02, KeyNo = %02X, KeyVer = %02X\n", ISO_TDEA_32_KEY_ADDRESS, ISO_TDEA_32_KEY_VERSION );
    bPLDataLen = 0;
    aPLData[bPLDataLen++] = 0x04 /* Add Quota command */; memcpy ( &aPLData[bPLDataLen], stHal_SamAv3.bUid, 7 ) /* UID */; bPLDataLen += 7;
    aPLData[bPLDataLen++] = 0x00 /* Quota ID */; aPLData[bPLDataLen++] = 0x02 /* Quota ID */; aPLData[bPLDataLen++] = ISO_TDEA_32_KEY_ADDRESS; /* KeyNo */
    aPLData[bPLDataLen++] = ISO_TDEA_32_KEY_VERSION; /* KeyVer */
    memcpy ( &aPLData[bPLDataLen], aMac, wMacLen ) /* Mac */; bPLDataLen += ( uint8_t ) wMacLen;
    printf ( "\t\tCommand  : " ); PrintData ( aPLData, bPLDataLen, "%02X ", "" ); printf ( "\n" );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_PLExec ( &stHal_SamAv3, PHHAL_HW_CMD_SAMAV3_PROG_LOGIC_LFI_LAST, aPLData, bPLDataLen, &pPLResp, &wPLRespLen ) );
    printf ( "\t\tResponse : " ); PrintData ( pPLResp, wPLRespLen, "%02X ", "" );  printf ( "\n\n" );

    printf ( "\tRead Quota. Quota ID = 00 01\n" );
    bPLDataLen = 0;
    aPLData[bPLDataLen++] = 0x06 /* Read Quota command */; memcpy ( &aPLData[bPLDataLen], stHal_SamAv3.bUid, 7 ) /* UID */; bPLDataLen += 7;
    aPLData[bPLDataLen++] = 0x00 /* Quota ID */; aPLData[bPLDataLen++] = 0x01 /* Quota ID */;
    printf ( "\t\tCommand  : " ); PrintData ( aPLData, bPLDataLen, "%02X ", "" ); printf ( "\n" );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_PLExec ( &stHal_SamAv3, PHHAL_HW_CMD_SAMAV3_PROG_LOGIC_LFI_LAST, aPLData, bPLDataLen, &pPLResp, &wPLRespLen ) );
    printf ( "\t\tResponse : " ); PrintData ( pPLResp, wPLRespLen, "%02X ", "" );  printf ( "\n\n" );

    printf ( "\tRead Quota. Quota ID = 00 02\n" );
    bPLDataLen = 0;
    aPLData[bPLDataLen++] = 0x06 /* Read Quota command */; memcpy ( &aPLData[bPLDataLen], stHal_SamAv3.bUid, 7 ) /* UID */; bPLDataLen += 7;
    aPLData[bPLDataLen++] = 0x00 /* Quota ID */; aPLData[bPLDataLen++] = 0x02 /* Quota ID */;
    printf ( "\t\tCommand  : " ); PrintData ( aPLData, bPLDataLen, "%02X ", "" ); printf ( "\n" );
    CHECK_SUCCESS_PRINT ( phhalHw_SamAV3_Cmd_SAM_PLExec ( &stHal_SamAv3, PHHAL_HW_CMD_SAMAV3_PROG_LOGIC_LFI_LAST, aPLData, bPLDataLen, &pPLResp, &wPLRespLen ) );
    printf ( "\t\tResponse : " ); PrintData ( pPLResp, wPLRespLen, "%02X ", "" );  printf ( "\n\n" );

    return PH_ERR_SUCCESS;
}
