/*
 * Copyright 2013, 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for MF DesFire EV1 Application.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 *
 * Demonstrates:
 * Two applications on same card:
 *   AFC Application with AES Crypto
 *   NextPACs application with 3KEY DES crypto
 * Card personalization and Check out.
 */

/* Header for this file */


/* Reader Library Headers */
#include <phbalReg.h>
#include <ph_Status.h>
#include <phhalHw.h>
#include <phpalI14443p3a.h>
#include <phpalI14443p4a.h>
#include <phpalI14443p4.h>
#include <phpalMifare.h>
#include <phKeyStore.h>
#include <phCryptoRng.h>
#include <phCryptoSym.h>
#include "../intfs/phalMfdf.h"
#include "Example-Rd70x_Mfdf.h"



#pragma warning(disable:4996)

#define KEYCOUNT 20
#define KEYVERSIONS 1

#ifdef NXPBUILD__PH_LOG
phLog_RegisterEntry_t logRegisterEntries[16];
phLog_LogEntry_t logEntries[48];
#endif /* NXPBUILD__PH_LOG */

static uint8_t pLogBuffer[1024];
static uint8_t gaHalRxReader[0xFF];
static uint8_t gaHalTxReader[0xFF];
static uint8_t gbGenBuffer[1024];

/* Utility function to check return status */
#define CHECK_SUCCESS(x)                                                \
    if ((x) != PH_ERR_SUCCESS)                                              \
{                                                                       \
    printf("An error occured (0x%04X), press any key to exit...", (x)); \
    _getch();                                              \
    return (x);                                                           \
}

/* Utility function to print a buffer */
void PRINT_BUFFER(uint8_t * pBuffer, uint16_t wLength)
{
    uint16_t wIndex;

    for (wIndex = 0; wIndex < wLength; ++wIndex)
    {
        printf("%02X ", pBuffer[wIndex]);
    }
    printf("\n");
}

/* Logging function. Turned OFF currently */
#ifdef NXPBUILD__PH_LOG
void phalMfdf_log(void * pDataParams, uint8_t bOption, phLog_LogEntry_t * pEntries, uint16_t wEntryCount)
{
    uint16_t i;
    PHAL_MFDF_UNUSED_VARIABLE(bOption);
    PHAL_MFDF_UNUSED_VARIABLE(pDataParams);

    for (i = 0; i < wEntryCount; i++ )
    {
        if ( i != 0 )
        {
            printf("%s : ", pEntries[i].pString);
        }
        else
        {
            printf("-----------------------------------------\n");
            printf("%s \n", pEntries[0].pString);
            printf("-----------------------------------------\n");
        }
        PRINT_BUFFER((uint8_t *)(pEntries[i].pData), pEntries[i].wDataLen);
    }
    printf("\n\n");
}
#endif /* NXPBUILD__PH_LOG */

#define AMKEY           0       /* Application Master Key number. */
#define ACKEY           1       /* Application Common Key number. */
#define ARKEY           2       /* Application Read Key number. */
#define AWKEY           3       /* Application Write Key number. */
#define AAKEY           4       /* Application Accreditation Key number. */
#define PICCKEY         5       /* PICC Key entry number in key store. */
#define PICCAESKEY      6       /* New PICC (AES128) Key entry number in key store. */
#define AMKEY_NEW       7       /* New Application Master Key entry number in Key store. */
#define SYSTEM_KEY      8       /* System key entry number in key store. */
#define NPKEY0          9       /* Next PACS App Key 0. */
#define NPKEY1          10      /* Next PACS App Key 1. */
#define SYSTEM_KEY_NP   11      /* Next PACS System key. */
#define DEFAULT_3K3DES  12      /* Default 3K3DES KEY. All zeros. */
#define NPKEY_FOR_UID   13      /* Used to retrieve UID */
#define AFCKEY_FOR_UID  14      /* Used to retrieve UID */

#define STDDATAFILE1    1       /* File number of Standard data file 1. */
#define STDDATAFILE2    2       /* File number of Standard data file 2. */
#define VALUEFILE       3       /* File number of Value file. */
#define RECORDFILE      4       /* File number of Record file. */
#define NPCARDOBJ       1       /* File number of NP Card object. */
#define NPPACSOBJ       2       /* File number of NP PACS object. */

/* Default PICC key */
static uint8_t bDefaultPICCKey[16] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

/* AES PICC key */
static uint8_t bAESPICCKey[16] = {
    0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
    0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A
};
/* Application Master key of AFC App */
static uint8_t bAMKey[16] =     {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* Application Common Key of AFC App */
static uint8_t bACKey[16] =     {
    0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
    0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11
};
/* Application Read Key of AFC App */
static uint8_t bARKey[16] =     {
    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22
};
/* Application Write Key of AFC App */
static uint8_t bAWKey[16] =     {
    0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33,
    0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33
};
/* Application Accreditation Key of AFC App */
static uint8_t bAAKey[16] =     {
    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44
};
/* New Application Master key of AFC App */
static uint8_t bAMKey_New[16] = {
    0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
    0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA
};

/* AFC App Key to retrieve UID */
static uint8_t bAfcKeyN[24]  =   {
    0x1A, 0x2B, 0x3C, 0x4D, 0x5E, 0x6F, 0x70, 0x81,
    0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
};

/* System key of AFC App */
static uint8_t bSysKey[16] =    {
    0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
    0x09, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16
};
/* Next PACs System key */
static uint8_t bSysKeyNP[24] =  {
    0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF, 0xCA, 0xCB,
    0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
    0x09, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16
};
/* Data for AFC App data file 1 */
static uint8_t bData1[32] =     {
    0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7, 0xA8,
    0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 0xB8,
    0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8,
    0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8
};
/* Data for AFC App data file 2 */
static uint8_t bData2[28] =     {
    0x4D, 0x55, 0x53, 0x54, 0x45, 0x52, 0x3C, 0x53,
    0x41, 0x4D, 0x50, 0x4C, 0x45, 0x3C, 0x3C, 0x01,
    0x07, 0x80, 0x01, 0x07, 0x09, 30, 06, 10, 0, 0,
    0, 0
};
/* NextPac key 0 */
static uint8_t bNPKey0[24]  =   {
    0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF, 0x00, 0x11,
    0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99,
    0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80
};
/* NextPac key 1 */
static uint8_t bNPKey1[24]  =   {
    0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF, 0xE0, 0xE1,
    0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9,
    0xC0, 0xD0, 0xE0, 0xF0, 0x20, 0x30, 0x40, 0x50
};

/* NP Key to retrieve UID */
static uint8_t bNPKey2[24]  =   {
    0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xC0, 0xA1,
    0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7, 0xA8, 0xA9,
    0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80
};

/* NextPac Card Data */
static uint8_t bNPCardData[32] = {
    'M', 'A', 'N', 'U', 'F', 'A', 'C', 'T', 'U', 'R', 'E', 'R', 'A', 'B', 'C', 'D',
    0x45, 0x03, /* Mutual Authentication Mode */
    0x02,       /* Communication mode = fully encrypted */
    0x00, 0x00, 0x12, 0x34, /* Customer ID */
    0x00,       /* Application verification key version */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 /* 8B digital signature. To be updated */
};
/* NextPacs Data */
static uint8_t bNPPacsData[48] = {
    0x01,       /* Major version */
    0x01,       /* Minor Version */
    0x00, 0x00, 0x55, 0x44, 0x33, /* Customer/Site code */
    0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, /* Credential ID */
    0x01,       /* Re issue code */
    0x00, 0x00, 0x12, 0x34,     /* Pin code */
    0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC,
    0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, /* 20B customer data*/
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00  /* 8B Digital sign to be updated */
};

/* AFC APPLICATION id */
static uint8_t bAfcApp[3] = {0x2A, 0x8C, 0xF1};

/* Next PACS APPLICATION id */
static uint8_t bNpacsApp[3] = {0x2A, 0x8C, 0xF2};

static uint8_t bCardUid[7];

/* Creates and pre personalizes the AFC Application */
phStatus_t PrePersonalization_AFCApp(phalMfdf_Sw_DataParams_t * pAlMfdf,
                                     uint8_t * pCardUid)
{
    phStatus_t status;
    uint8_t bLength;

    uint8_t bOption;
    uint8_t bKeySettings1;
    uint8_t bKeySettings2;
    uint8_t bISOFileId[2];
    uint8_t pISODFName[16];
    uint8_t bDFLen;

    uint8_t bLLimit[4];
    uint8_t bULimit[4];
    uint8_t bValue[4];
    uint8_t bAccessRights[2];
    uint8_t pISOFileId[2];
    uint8_t bFileSize[3];
    uint8_t bLimitedCredit = 0;
    uint8_t bMaxNoRec[3];
    uint8_t bRecordSize[3];
    uint8_t bConfigByte;

    printf("Performing Pre Personalization ...... \n");
    /* ####################Pre personalization############################### */

    /* Authenticate with the PICC Master key */
    status = phalMfdf_AuthenticateISO(pAlMfdf,  /* [In] Pointer to parameters data structure */
        PHAL_MFDF_NO_DIVERSIFICATION,           /* [In] Diversification option */
        PICCKEY,                                /* [In]KEY number in key store */
        0x00,                                   /* [In] Key version in key store */
        0x00,                                   /* [In] Key number on Card */
        NULL,                                   /* [In] Diversification input */
        0);                                     /* [In] Diversification input length */
    CHECK_SUCCESS(status);

    /* Get a fresh card */
    status = phalMfdf_FormatPICC(pAlMfdf);
    CHECK_SUCCESS(status);

    /* Set Configuration - default key and version */
    bOption = PHAL_MFDF_SET_CONFIG_OPTION2; /* option to set default key and version */
    memset(gbGenBuffer, 0x00, 24);          /* Default key */
    gbGenBuffer[24] = 0x00;                 /* default key version */
    bLength = 25;                           /* length */
    status = phalMfdf_SetConfiguration(pAlMfdf,
        bOption,
        gbGenBuffer,
        bLength);
    CHECK_SUCCESS(status);

    /* Set Configuration - Configure for Random UID */
    bOption = PHAL_MFDF_SET_CONFIG_OPTION1; /* option to set configuration byte */
    bConfigByte = 0x02;                     /* 00000010 : Random ID Enabled */
    status = phalMfdf_SetConfiguration(pAlMfdf,
        bOption,
        &bConfigByte,
        0x01);
    CHECK_SUCCESS(status);

    /* Get the card UID */
    status = phalMfdf_GetCardUID(pAlMfdf,   /* [In] Pointer to parameters data structure */
        bCardUid);                          /* [Out] 7 Byte Card UID */
    CHECK_SUCCESS(status);

    /* Create AFC Application */

    /* bOption = 0x03 indicates that application has both ISO Fid and DF Name */
    bOption = 0x03;

    /* Iso File Id = 0x00AA. Sent LSB first */
    bISOFileId[0] = 0xAA;
    bISOFileId[1] = 0x00;

    /* DF Name = AFCAPPLICATION */
    bDFLen = 14;
    memcpy(pISODFName, "AFCAPPLICATION", bDFLen);

    /* Application masterkey settings 00001011
    * =  conf. changeable +
    *    create/delete file requires authentication +
    *   directory access is free +
    *   App MK is changeable
    *   Change access rights = 0x00 (AMK)
    */
    bKeySettings1 = 0x0B;

    /* Keysettings 2 = 10100110
    * AES crypto + supports 2 byte ISO fids +
    * max 6 keys in this app
    */
    bKeySettings2 = 0xA6;

    status = phalMfdf_CreateApplication(pAlMfdf,    /* [In] Pointer to parameters data structure */
        bOption,        /* [In] Option to indicate whether this application has ISO Fid and DF */
        bAfcApp,        /* [In] 3 byte AID, LSB First */
        bKeySettings1,  /* [In] Key Settings 1 - 1 Byte */
        bKeySettings2,  /* [In] Key Settings 2 - 1 Byte */
        bISOFileId,     /* [In] Two byte ISO File Id, LSB First */
        pISODFName,     /* [In] ISO DF Name. Maximum 16 bytes */
        bDFLen);        /* [In] Length of DF Name provided above */
    CHECK_SUCCESS(status);

    /* Select the AFC Application */
    status = phalMfdf_SelectApplication(pAlMfdf, bAfcApp);
    CHECK_SUCCESS(status);

    /* Authenticate */
    status = phalMfdf_AuthenticateAES(pAlMfdf,  /* [In] Pointer to parameters data structure */
        PHAL_MFDF_NO_DIVERSIFICATION,   /* [In] Diversification option */
        AMKEY,                          /* [In] KEY number in key store */
        0,                              /* [In] Key version in key store */
        0,                              /* [In] Key number on Card */
        NULL,                           /* [In] Diversification input */
        0);                             /* [In] Diversification input length */
    CHECK_SUCCESS(status);

    pISOFileId[0] = 01;
    pISOFileId[1] = 01;

    /* File Size = 32 bytes */
    bFileSize[0] = 0x20;
    bFileSize[1] = 0x00;
    bFileSize[2] = 0x00;

    bAccessRights[0] = 0x00; /* Change access rights = AMK, RW access = never(for now AMK) */
    bAccessRights[1] = 0x10; /* Read access = ACKey, Write access = never (for now AMK) */

    /* Create standard data file # 01 */
    status = phalMfdf_CreateStdDataFile(pAlMfdf,    /* [In] Pointer to parameters data structure */
        0x01,                                       /* [In] Option to indicate ISO File Id provided */
        STDDATAFILE1,                               /* [In] File number. 1 byte */
        pISOFileId,                                 /* [In] Two byte ISO File ID. LSB First */
        PHAL_MFDF_COMMUNICATION_MACD,               /* [In] Communication settings of the file */
        bAccessRights,                              /* [In] 2 Byte access rights. LSB First */
        bFileSize                                   /* [In] 3 Byte file size. LSB First */
        );
    CHECK_SUCCESS(status);

    /* Create Std data file #02 */
    pISOFileId[0] = 02;
    pISOFileId[1] = 01;

    /* File Size = 32 bytes */
    bFileSize[0] = 0x20;
    bFileSize[1] = 0x00;
    bFileSize[2] = 0x00;

    bAccessRights[0] = 0x00; /* RW access = AMK, Change access rights = AMK  */
    bAccessRights[1] = 0x20; /* Read access = ARKey, Write access = AMK */

    status = phalMfdf_CreateStdDataFile(pAlMfdf,    /* [In] Pointer to parameters data structure */
        0x01,                                       /* [In] Option to indicate ISO File Id provided */
        STDDATAFILE2,                               /* [In] File number. 1 byte */
        pISOFileId,                                 /* [In] Two byte ISO File ID. LSB First */
        PHAL_MFDF_COMMUNICATION_ENC,                /* [In] Communication settings of the file */
        bAccessRights,                              /* [In] 2 Byte access rights. LSB First */
        bFileSize                                   /* [In] 3 Byte file size. LSB First */
        );
    CHECK_SUCCESS(status);

    /* Create Value File */
    bAccessRights[0] = 0x40; /* RW access = AAK, Change access rights = AMK  */
    bAccessRights[1] = 0x33; /* Read access = AWKey, Write access = AWKey */

    memset(bLLimit, 0x00, 4);
    memset(bULimit, 0x00, 4);
    memset(bValue, 0x00, 4);

    bLLimit[0] = 0x14; /* 20 */
    bValue[0] = 0x14;  /* 20 */
    bULimit[0] = 0xF4;
    bULimit[1] = 0x01; /* 500 is UL */

    /* Limited Credit enabled */
    bLimitedCredit = PHAL_MFDF_ENABLE_LIMITEDCREDIT;

    status = phalMfdf_CreateValueFile(pAlMfdf,  /* [In] Pointer to parameters data structure */
        VALUEFILE,                              /* [In] File number */
        PHAL_MFDF_COMMUNICATION_ENC,            /* [In] Communication settings of the file */
        bAccessRights,                          /* [In] 2 Byte access rights. LSB First */
        bLLimit,                                /* [In] Lower Limit. 4 Byte value LSB First */
        bULimit,                                /* [In] Upper Limit. 4 Byte value LSB First */
        bValue,                                 /* [In] 4 Byte value LSB First */
        bLimitedCredit                          /* [In] Limited Credit and free getvalue setting. */
        );
    CHECK_SUCCESS(status);

    /* Create cyclic record file #04 */
    bRecordSize[0] = 12;
    bRecordSize[1] = 0;
    bRecordSize[2] = 0;

    bMaxNoRec[0] = 0x64; /* 100 records */
    bMaxNoRec[1] = 0;
    bMaxNoRec[2] = 0;

    bAccessRights[0] = 0x40; /* RW access = AAK, Change access rights = AMK  */
    bAccessRights[1] = 0x23; /* Read access = ARKey, Write access = AWKey */

    pISOFileId[0] = 4;
    pISOFileId[1] = 0;

    status = phalMfdf_CreateCyclicRecordFile(pAlMfdf,   /* [In] Pointer to this layers param structure */
        0x01,                                           /* [In] Indicates ISO file ID is present or not. */
        RECORDFILE,                                     /* [In] Linear record File No. */
        pISOFileId,                                     /* [In] 2 Byte IsoFileId. Sent LSB first to PICC. */
        PHAL_MFDF_COMMUNICATION_ENC,                    /* [In] communication settings */
        bAccessRights,                                  /* [In] 2 byte access rights. Sent LSB first to PICC */
        bRecordSize,                                    /* [In] 2 byte Record Size. Sent LSB first to PICC */
        bMaxNoRec                                       /* [In] 3 byte Max Number of Records. Sent LSB first to PICC */
        );
    CHECK_SUCCESS(status);

    /* Change default application keys to actual values */
    status = phalMfdf_ChangeKey(pAlMfdf,    /* [In] Pointer to this layer's parameter structure. */
        PHAL_MFDF_NO_DIVERSIFICATION,       /* [In] Diversification option. */
        AMKEY,                              /* [In] Old key number in keystore. */
        0,                                  /* [In] Old key version in keystore. */
        ACKEY,                              /* [In] New key number in keystore. */
        0,                                  /* [In] New key version in keystore. */
        ACKEY,                              /* [In] One byte Card number to be changed. */
        NULL,                               /* [In] Diversification input. Can be NULL. */
        0);                                 /* [In] Length of diversification input max 31B. */
    CHECK_SUCCESS(status);

    /* Application Read key. Diversified using card uid */
    status = phalMfdf_ChangeKey(pAlMfdf,
        (PHAL_MFDF_CHGKEY_DIV_NEW_KEY |
            PHAL_MFDF_CHGKEY_DIV_METHOD_CMAC),   /* ARKEY is diversified using Sam AV2 method */
        AMKEY,
        0,
        ARKEY,
        0,
        ARKEY,
        pCardUid,
        7);
    CHECK_SUCCESS(status);

    /* Application Write key. Diversified using card uid */
    status = phalMfdf_ChangeKey(pAlMfdf,
        (PHAL_MFDF_CHGKEY_DIV_NEW_KEY |
            PHAL_MFDF_CHGKEY_DIV_METHOD_CMAC),   /* AWKEY is diversified using Sam AV2 method */
        AMKEY,
        0,
        AWKEY,
        0,
        AWKEY,
        pCardUid,
        7);
    CHECK_SUCCESS(status);

    /* Application Accreditation key diversified using card UID */
    status = phalMfdf_ChangeKey(pAlMfdf,
        (PHAL_MFDF_CHGKEY_DIV_NEW_KEY |
            PHAL_MFDF_CHGKEY_DIV_METHOD_CMAC),   /* AAKEY is diversified using Sam AV2 method */
        AMKEY,
        0,
        AAKEY,
        0,
        AAKEY,
        pCardUid,
        7);
    CHECK_SUCCESS(status);

    /* UID Retriever key not diversified */
    status = phalMfdf_ChangeKey(pAlMfdf,
        PHAL_MFDF_NO_DIVERSIFICATION,   /* NO Diversification */
        AMKEY,
        0,
        AFCKEY_FOR_UID,
        0,
        5,  /* Key number 5 on card */
        NULL,
        0);
    CHECK_SUCCESS(status);

    /* Finally change the application master key */
    status = phalMfdf_ChangeKey(pAlMfdf,
        PHAL_MFDF_NO_DIVERSIFICATION,   /* NO Diversification */
        AMKEY,
        0,
        AMKEY_NEW,
        0,
        AMKEY,
        NULL,
        0);
    CHECK_SUCCESS(status);
    /* ####################End Pre personalization########################### */
    printf("\tPre Personalization DONE! \n");

    return 0;
}

/* Personalizes the AFC Application. Writes Data to data files 1 and 2 */
phStatus_t Personalization_AFCApp(phKeyStore_Sw_DataParams_t * pKeyStore,
                                  phalMfdf_Sw_DataParams_t * pAlMfdf,
                                  uint8_t * pCardUid)
{
    phStatus_t status;
    /* Bfl data parameter storage */
    phKeyStore_Sw_DataParams_t keyStore;
    phalMfdf_Sw_DataParams_t alMfdf;
    uint8_t bAccessRights[2];
    uint8_t bOffSet[3] = { 0, 0, 0 };

    uint8_t bInitVector[16];
    uint8_t bGenLen;
    uint8_t bDataLen[3] = { 0, 0, 0 };

    printf("Performing Personalization ...... \n");
    /* #################### Personalization############################### */

    keyStore = *pKeyStore;
    alMfdf = *pAlMfdf;

    /* Prepare the data - for file STDDATAFILE2 */
    /* IV  = 0x88 + 7ByteUID + 8B SN */
    bInitVector[0] = 0x88;
    memcpy(&bInitVector[1], pCardUid, 7);
    memcpy(&bInitVector[8], &bData1[8], 8); /* Copy the serial number */

    /* Load System Key to generate the digital signature */
    status = phCryptoSym_LoadKey(
        alMfdf.pCryptoDataParamsEnc,    /* [In] Pointer to this layer's parameter structure. */
        SYSTEM_KEY,                     /* [In] Number of the Key */
        0,                              /* [In] Version of the key */
        PH_KEYSTORE_KEY_TYPE_AES128);   /* [In] Type of Key */
    CHECK_SUCCESS(status);

    /* Load IV */
    status = phCryptoSym_LoadIv(
        alMfdf.pCryptoDataParamsEnc,    /* [In] Pointer to this layer's parameter structure. */
        bInitVector,                    /* [In] IV[bIVLength]. */
        16);                            /* [In] Length of IV. */
    CHECK_SUCCESS(status);

    /* Calculate CMAC */
    status = phCryptoSym_CalculateMac(
        alMfdf.pCryptoDataParamsEnc,    /* [In] Pointer to this layer's parameter structure. */
        PH_CRYPTOSYM_MAC_MODE_CMAC,     /* [In] Option byte specifying the MAC mode and the update behavior of the IV and the completion flag.*/
        bData2,                         /* [In] input block; uint8_t[wDataLength] */
        24,                             /* [In] number of input data bytes */
        gbGenBuffer,                    /* [Out] output MAC block; uint8_t[16] */
        &bGenLen);                      /* [Out] Length of MAC */
    CHECK_SUCCESS(status);

    /* Copy the four bytes of the generated CMAC */
    memcpy(&bData2[24], gbGenBuffer, 4);

    /* Select the AFC application */
    status = phalMfdf_SelectApplication(&alMfdf,    /* [In] Pointer to this layer's parameter structure. */
        bAfcApp);                                   /* [In] 3 byte UID. LSB First */
    CHECK_SUCCESS(status);

    status = phalMfdf_AuthenticateAES(&alMfdf,
        PHAL_MFDF_NO_DIVERSIFICATION,
        AMKEY_NEW,
        0x00,
        0x00,       /* Key number on Card is 0x00, i.e., AMK */
        NULL,
        0);
    CHECK_SUCCESS(status);

    /* Data for data file 1 */
    bDataLen[0] = 32;
    bDataLen[1] = 0;
    bDataLen[2] = 0;

    status = phalMfdf_WriteData(
        &alMfdf,                        /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_MACD,   /* [In] Communication Mode. Plain, Mac'd or encrypted. */
        STDDATAFILE1,                   /* [In] 1 byte file number. */
        bOffSet,                        /* [In] 3 bytes offset. LSB First. */
        bData1,                         /* [In] Data to be written. */
        bDataLen);                      /* [in] 3 bytes LSB First. length of data to be written. */
    CHECK_SUCCESS(status);

    bAccessRights[0] = 0xF0;    /* Change access rights = AMK, RW access = never */
    bAccessRights[1] = 0x1F;    /* Read access = ACKey, Write access = never */

    status = phalMfdf_ChangeFileSettings(
        &alMfdf,                        /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_ENC,    /* [In] Indicates whether the settings to be sent enciphered or plain. */
        STDDATAFILE1,                   /* [In] file number. */
        PHAL_MFDF_COMMUNICATION_MACD,   /* [In] new communication settings for the file . */
        bAccessRights);                 /* [In] 2 byte access rights. */
    CHECK_SUCCESS(status);

    /* Write to data file 2 */
    bDataLen[0] = 28;
    bDataLen[1] = 0;
    bDataLen[2] = 0;

    status = phalMfdf_WriteData(
        &alMfdf,                        /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_ENC,    /* [In] Communication Mode. Plain, Mac'd or encrypted. */
        STDDATAFILE2,                   /* [In] 1 byte file number. */
        bOffSet,                        /* [In] 3 bytes offset. LSB First. */
        bData2,                         /* [In] Data to be written. */
        bDataLen);                      /* [in] 3 bytes LSB First. length of data to be written. */
    CHECK_SUCCESS(status);

    /* ########################## End Personalization ############################# */
    printf("\tPersonalization of AFC App DONE!\n");
    return 0;
}

/* Credits a given value to the AFC Application */
phStatus_t Accreditation_WithValue(phalMfdf_Sw_DataParams_t * pAlMfdf,
                                   uint8_t * pValue)
{
    phStatus_t status;
    uint8_t bRecordSize[3];
    uint8_t bOffSet[3] = { 0, 0, 0 };
    uint8_t bRecord[12];
    uint8_t bValue[4];
    uint32_t dwVal1;
    uint32_t dwVal2;
    time_t rawtime;
    struct tm * timeinfo;

    /* ########################## Accreditation ################################### */
    printf("Performing Accreditation in AFC App.... \n");

    status = phalMfdf_SelectApplication(pAlMfdf, bAfcApp);
    CHECK_SUCCESS(status);

    /* First authenticate with UID retriever key */
    status = phalMfdf_AuthenticateAES(
        pAlMfdf,                        /* [In] Pointer to this layer's parameter structure. */
        PHAL_MFDF_NO_DIVERSIFICATION,   /* [In] Diversification option. */
        AFCKEY_FOR_UID,                 /* [In] key number in keystore to authenticate with. */
        0,                              /* [In] Key version in the key store. */
        5,                              /* [In] Key number on card. */
        NULL,                           /* [In] Diversification input. Can be NULL. */
        0);                             /* [In] Length of diversification input max 31B. */
    CHECK_SUCCESS(status);

    /* Get Card UID */
    status = phalMfdf_GetCardUID(pAlMfdf,   /* [In] Pointer to this layer's parameter structure. */
        bCardUid);                          /* [Out] 7 Byte Card UID. */
    CHECK_SUCCESS(status);

    /* Authenticate with key# 4 AAKEY */
    status = phalMfdf_AuthenticateAES(pAlMfdf,  /* [In] Pointer to this layer's parameter structure. */
        PHAL_MFDF_DIV_METHOD_CMAC,              /* [In] Diversification option. */
        AAKEY,                                  /* [In] key number in keystore to authenticate with. */
        0,                                      /* [In] Key version in the key store. */
        AAKEY,                                  /* [In] Key number on card. */
        bCardUid,                               /* [In] Diversification input. Can be NULL. */
        7);                                     /* [In] Length of diversification input max 31B. */
    CHECK_SUCCESS(status);

    memset(bValue, 0, 4);
    /* Get Current Value */
    status = phalMfdf_GetValue(
        pAlMfdf,                        /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_ENC,    /* [In] Communication option. */
        VALUEFILE,                      /* [In] 1 byte file number. */
        bValue);                        /* [Out] 4 Byte array to store the value read out. LSB First. */
    CHECK_SUCCESS(status);

    dwVal1 = bValue[3];
    dwVal1 = dwVal1 << 8 | bValue[2];
    dwVal1 = dwVal1 << 8 | bValue[1];
    dwVal1 = dwVal1 << 8 | bValue[0];

    /* Credit the value provided */
    memcpy(bValue, pValue, 4);

    /* Add money (100) to the account */
    status = phalMfdf_Credit(
        pAlMfdf,                        /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_ENC,    /* [In] Communication option. */
        VALUEFILE,                      /* [In] 1 byte file number. */
        bValue);                        /* [In] 4 byte value array. LSB first. */
    CHECK_SUCCESS(status);

    dwVal2 = bValue[3];
    dwVal2 = dwVal2 << 8 | bValue[2];
    dwVal2 = dwVal2 << 8 | bValue[1];
    dwVal2 = dwVal2 << 8 | bValue[0];

    dwVal1 += dwVal2;
    bValue[0] = (uint8_t)dwVal1;
    bValue[1] = (uint8_t)(dwVal1 >> 8);
    bValue[2] = (uint8_t)(dwVal1 >> 16);
    bValue[3] = (uint8_t)(dwVal1 >> 24);

    /* Write record */
    /* Get the system date and time */
    time(&rawtime);
    timeinfo = localtime(&rawtime);
    printf("Current local time and date: %s", asctime(timeinfo));

    bRecord[0] = (uint8_t)timeinfo->tm_mday;
    bRecord[1] = (uint8_t)(timeinfo->tm_mon + 1);
    bRecord[2] = (uint8_t)(timeinfo->tm_year % 100);

    bRecord[3] = (uint8_t)timeinfo->tm_hour;
    bRecord[4] = (uint8_t)timeinfo->tm_min;
    bRecord[5] = (uint8_t)timeinfo->tm_sec;

    /* Terminal ID */
    bRecord[6] = 0xDE;
    bRecord[7] = 2;

    bRecord[8] = 1; /* 0 - debit, 1 - credit, 2 limited credit */
    bRecord[11] = 'X'; /* Other info */

    bRecord[9] = bValue[0];
    bRecord[10] = bValue[1];

    memset(bRecordSize, 0, 3);
    bRecordSize[0] = 12;

    status = phalMfdf_WriteRecord(
        pAlMfdf,                        /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_ENC,    /* [In] Communication option. Plain, Mac'd or enc. */
        RECORDFILE,                     /* [In] 1 byte file number. */
        bOffSet,                        /* [In] 3 bytes offset. LSB First. */
        bRecord,                        /* [In] data to be written. */
        bRecordSize);                   /* [In] 3 bytes. length of data to be written. */
    CHECK_SUCCESS(status);

    /* Commit after write record */
    status = phalMfdf_CommitTransaction(pAlMfdf);
    CHECK_SUCCESS(status);

    printf(" The amount in your account is %x %x %x %x \n", bValue[3], bValue[2], bValue[1], bValue[0]);
    /* ########################## End Accreditation ############################### */
    printf("\tAccreditation DONE! \n");
    return 0;
}

/* Credits 100 units to the AFC Application */
phStatus_t Accreditation(phalMfdf_Sw_DataParams_t * pAlMfdf)
{
    uint8_t bValue[4] = {100, 0, 0, 0};
    return Accreditation_WithValue(pAlMfdf,
        bValue);
}

/* Checkout of AFC App. Debits 10 units from the AFC Application */
phStatus_t CheckOutAfcApp(phalMfdf_Sw_DataParams_t * pAlMfdf)
{
    phalMfdf_Sw_DataParams_t alMfdf;
    phStatus_t status;
    uint8_t bOffSet[3] = {0, 0, 0};
    uint8_t * pRecv;
    uint16_t wRxlen;
    uint8_t bIndex;
    uint8_t bDataLen[3] = {0, 0, 0};
    uint8_t bNumRec[3] = {0, 0, 0};
    uint8_t bValue[4] = {0, 0, 0, 0};
    time_t rawtime;
    struct tm * timeinfo;
    uint8_t bRecord[12];
    uint8_t bName[16];
    uint32_t dwVal1;

    alMfdf = *pAlMfdf;

    /* Select the AFC Application */
    status = phalMfdf_SelectApplication(&alMfdf, bAfcApp);
    CHECK_SUCCESS(status);

    /* Authenticate with ACKey */
    status = phalMfdf_AuthenticateAES(&alMfdf,
        PHAL_MFDF_NO_DIVERSIFICATION,
        ACKEY,
        0x00,
        ACKEY,
        NULL,
        0);
    CHECK_SUCCESS(status);

    bDataLen[0] = 32;

    /* Read low secure data */
    status = phalMfdf_ReadData(
        &alMfdf,                        /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_MACD,   /* [In] Is either plain or encrypted or MAC'd. */
        STDDATAFILE1,                   /* [In] 1 byte file number. */
        bOffSet,                        /* [In] 3 bytes offset. LSB First. */
        bDataLen,                       /* [In] 3 bytes. length of data to be read. If 00, entire file will be read. */
        &pRecv,                         /* [Out] Pointer to HAL Rx buffer returned back to user. */
        &wRxlen);                       /* [Out] Pointer to Length of RxData. */
    CHECK_SUCCESS(status);

    if (wRxlen != 32)
    {
        printf("Error occured in reading \n");
        return 0xFF;
    }

    /* Print Ticket SN */
    printf("Detected ticket # ");
    for(bIndex = 0; bIndex < 8; bIndex++)
    {
        printf("%x", pRecv[8+bIndex]);
    }
    printf("\n");

    status = phalMfdf_GetCardUID(
        &alMfdf,    /* [In] Pointer to this layers param structure. */
        bCardUid);  /* [Out] UID of the card. Buffer size should be 10bytes. */
    CHECK_SUCCESS(status);

    /* Authenticate with the diversified ARKey */
    status = phalMfdf_AuthenticateAES(
        &alMfdf,                        /* [In] Pointer to this layer's parameter structure. */
        PHAL_MFDF_DIV_METHOD_CMAC,    /* [In] Diversification option. */
        ARKEY,                          /* [In] key number in keystore to authenticate with. */
        0x00,                           /* [In] Key version in the key store. */
        ARKEY,                          /* [In] Key number on card. */
        bCardUid,                       /* [In] Diversification input. Can be NULL. */
        7);                             /* [In] Length of diversification input max 31B. */
    CHECK_SUCCESS(status);

    bDataLen[0] = 28;

    /* Read secure data */
    status = phalMfdf_ReadData(
        &alMfdf,                        /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_ENC,    /* [In] Is either plain or encrypted or MAC'd. */
        STDDATAFILE2,                   /* [In] 1 byte file number. */
        bOffSet,                        /* [In] 3 bytes offset. LSB First. */
        bDataLen,                       /* [In] 3 bytes. length of data to be read. If 00, entire file will be read. */
        &pRecv,                         /* [Out] Pointer to HAL Rx buffer returned back to user. */
        &wRxlen);                       /* [Out] Pointer to Length of RxData. */
    CHECK_SUCCESS(status);

    memcpy(bName, pRecv, 15);
    bName[15] = '\0';
    printf("Card belongs to %s \n", bName);

    bDataLen[0] = 12; /* Record size */
    bNumRec[0] = 1;
    status = phalMfdf_ReadRecords(
        &alMfdf,                        /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_ENC,    /* [In] communcation option. */
        RECORDFILE,                     /* [In] 1 byte file number */
        bOffSet,                        /* [In] 3 bytes offset to the record. LSB First. */
        bNumRec,                        /* [In] 3 bytes LSB first. Number of records to be read. If 0x00 00 00, then all the records are read. */
        bDataLen,                       /* [In] Record size. 3Bytes LSB first. */
        &pRecv,                         /* [Out] pointer to the HAL buffer that stores the read data. */
        &wRxlen);                       /* [Out] number of bytes read (= number of records read * size of record). */
    CHECK_SUCCESS(status);

    memcpy(bRecord, pRecv, 12);

    /* Authenticate with Application Write Key */
    status = phalMfdf_AuthenticateAES(&alMfdf,
        PHAL_MFDF_DIV_METHOD_CMAC,
        AWKEY,
        0x00,
        AWKEY,
        bCardUid,
        7);
    CHECK_SUCCESS(status);

    status = phalMfdf_GetValue(&alMfdf, /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_ENC,    /* [In] Communication option. */
        VALUEFILE,                      /* [In] 1 byte file number. */
        bValue);                        /* [Out] 4 Byte array to store the value read out. LSB First. */
    CHECK_SUCCESS(status);

    if (memcmp(bValue, &bRecord[9], 2) != 0)
    {
        printf("Validation Error... Data in records does not match the available value \n");
        return 0xFF;
    }

    dwVal1 = bValue[3];
    dwVal1 = dwVal1 << 8 | bValue[2];
    dwVal1 = dwVal1 << 8 | bValue[1];
    dwVal1 = dwVal1 << 8 | bValue[0];

    /* Prepare value to be debited */
    memset(bValue, 0, 4);
    bValue[0] = 10;

    /* Debit 10 units */
    status = phalMfdf_Debit(&alMfdf,    /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_ENC,    /* [In] Communication option. */
        VALUEFILE,                      /* [In] 1 byte file number. */
        bValue);                        /* [In] 4 byte value array. LSB first. */
    CHECK_SUCCESS(status);

    /* Do the arithmetic */
    dwVal1 = dwVal1 - 10;

    /* Convert back to 4 byte array */
    bValue[0] = (uint8_t)dwVal1;
    bValue[1] = (uint8_t)(dwVal1 >> 8);
    bValue[2] = (uint8_t)(dwVal1 >> 16);
    bValue[3] = (uint8_t)(dwVal1 >> 24);

    /* Get the system date and time */
    time(&rawtime);
    timeinfo = localtime(&rawtime);
    printf("Current local time and date: %s", asctime(timeinfo));

    /* Store current date and time in the record */
    bRecord[0] = (uint8_t)(timeinfo->tm_mday);
    bRecord[1] = (uint8_t)(timeinfo->tm_mon + 1);
    bRecord[2] = (uint8_t)(timeinfo->tm_year % 100);

    bRecord[3] = (uint8_t)timeinfo->tm_hour;
    bRecord[4] = (uint8_t)timeinfo->tm_min;
    bRecord[5] = (uint8_t)timeinfo->tm_sec;

    /* Terminal ID */
    bRecord[6] = 0xDE;
    bRecord[7] = 2;
    bRecord[8] = 0;     /* 0 - debit, 1 - credit, 2 limited credit */
    bRecord[11] = 'X';  /* Other info */
    bRecord[9] = bValue[0];
    bRecord[10] = bValue[1];

    bDataLen[0] = 12;
    status = phalMfdf_WriteRecord(
        &alMfdf,                        /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_ENC,    /* [In] Communication option. Plain, Mac'd or enc. */
        RECORDFILE,                     /* [In] 1 byte file number. */
        bOffSet,                        /* [In] 3 bytes offset. LSB First. */
        bRecord,                        /* [In] data to be written. */
        bDataLen);                      /* [In] 3 bytes. length of data to be written. LSB First. */
    CHECK_SUCCESS(status);

    /* Commit after writing record */
    status = phalMfdf_CommitTransaction(&alMfdf);
    CHECK_SUCCESS(status);

    return 0;
}

/* Personalization of NextPACS Application */
phStatus_t Perso_NextPACSApp(
                             phalMfdf_Sw_DataParams_t * pAlMfdf
                             )
{
    phStatus_t status;
    /* Bfl data parameter storage */

    phalMfdf_Sw_DataParams_t alMfdf;
    uint8_t bAccessRights[2];
    uint8_t bOffSet[3] = { 0, 0, 0 };
    uint8_t bOption;
    uint8_t bKeySettings1;
    uint8_t bKeySettings2;
    uint8_t bISOFileId[2];
    uint8_t pISODFName[16];
    uint8_t bDFLen;
    uint8_t bInitVector[16];
    uint8_t bGenLen;
    uint8_t bDataLen[3] = { 0, 0, 0 };
    uint8_t bNewKey[24];
    uint8_t bMac[16];

    alMfdf = *pAlMfdf;
    /* ######################### Pre personalization ###########################*/
    /* Authenticate with the PICC Master key */
    status = phalMfdf_AuthenticateISO(
        &alMfdf,                        /* [In] Pointer to this layer's parameter structure. */
        PHAL_MFDF_NO_DIVERSIFICATION,   /* [In] Diversification option. */
        PICCKEY,                        /* [In] key number in keystore to authenticate with. */
        0,                              /* [In] Key version in the key store. */
        0,                              /* [In] Key number on card. */
        NULL,                           /* [In] Diversification input. Can be NULL. */
        0);                             /* [In] Length of diversification input max 31B. */
    CHECK_SUCCESS(status);

    /* Get card UID. This UID is supplied to function below */
    status = phalMfdf_GetCardUID(
        &alMfdf,    /* [In] Pointer to this layer's parameter structure. */
        bCardUid);  /* [Out] UID of the card. Buffer size should be 7 bytes. */
    CHECK_SUCCESS(status);

    /* Already authenticated with PICC MK */
    /* Create NEXT PACs application */
    /* bOption = 0x03 indicates that application has both ISO Fid and DF Name */
    bOption = 0x03;

    /* Iso File Id = 0x01AB. Sent LSB first */
    bISOFileId[0] = 0xAB;
    bISOFileId[1] = 0x01;

    /* DF Name = NEXTPACSCAPP */
    bDFLen = 12;
    memcpy(pISODFName, "NEXTPACSCAPP", bDFLen);

    /* Application masterkey settings 00001011
    * =  conf. changeable +
    *    create/delete file requires authentication +
    *   directory access is free +
    *   App MK is changeable
    *   Change access rights = 0x00 (AMK)
    */
    bKeySettings1 = 0x0B;

    /* Keysettings 2 = 01100011
    * 3K3DES crypto + supports 2 byte ISO fids +
    * max 3 keys in this app
    */
    bKeySettings2 = 0x63;

    status = phalMfdf_CreateApplication(&alMfdf,
        bOption,        /* [In] Option to indicate whether this application has ISO Fid and DF */
        bNpacsApp,      /* [In] 3 byte AID, LSB First */
        bKeySettings1,  /* [In] Key Settings 1 - 1 Byte */
        bKeySettings2,  /* [In] Key Settings 2 - 1 Byte */
        bISOFileId,     /* [In] Two byte ISO File Id, LSB First */
        pISODFName,     /* [In] ISO DF Name. Maximum 16 bytes */
        bDFLen);        /* [In] Length of DF Name provided above */
    CHECK_SUCCESS(status);

    /* Select the application */
    status = phalMfdf_SelectApplication(&alMfdf, bNpacsApp);
    CHECK_SUCCESS(status);

    /* Authenticate */
    status = phalMfdf_AuthenticateISO(&alMfdf,
        PHAL_MFDF_NO_DIVERSIFICATION,
        DEFAULT_3K3DES,     /* Authenticate with default 3k3des key. All zeros */
        0,
        0,
        NULL,
        0);
    CHECK_SUCCESS(status);

    /* Create the Card Identifier Object */
    bISOFileId[0] = 1;
    bISOFileId[1] = 0;

    bAccessRights[0] = 0xF0; /* LSB of access rights. RW = never, Change config key = bNPKey0 */
    bAccessRights[1] = 0xE0; /* MSB. R = free, W = bNPKey0 (Never later) */

    bDataLen[0] = 32; /* File Size = 32 bytes */
    bDataLen[1] = 0;
    bDataLen[2] = 0;
    status = phalMfdf_CreateStdDataFile(&alMfdf,
        0x01,
        NPCARDOBJ,
        bISOFileId,
        PHAL_MFDF_COMMUNICATION_PLAIN,
        bAccessRights,
        bDataLen);
    CHECK_SUCCESS(status);

    /* Create the PACS Data Object */
    bISOFileId[0] = 2;
    bISOFileId[1] = 0;

    bAccessRights[0] = 0xF0; /* LSB of access rights. RW = never, Change config key = bNPKey0 */
    bAccessRights[1] = 0x10; /* MSB. R = bNPKey1, W = W = bNPKey0 (Never later) */

    bDataLen[0] = 48; /* File Size = 48 bytes */
    bDataLen[1] = 0;
    bDataLen[2] = 0;
    status = phalMfdf_CreateStdDataFile(&alMfdf,
        0x01,
        NPPACSOBJ,
        bISOFileId,
        PHAL_MFDF_COMMUNICATION_PLAIN,
        bAccessRights,
        bDataLen);
    CHECK_SUCCESS(status);
    /* ##################### End Pre personalization ########################### */

    /* ############################# personalization ########################### */
    /* Update bNPCardData with digital signature */
    status = phCryptoSym_LoadKey(alMfdf.pCryptoDataParamsEnc,
        SYSTEM_KEY_NP,
        0,
        PH_CRYPTOSYM_KEY_TYPE_3K3DES);
    CHECK_SUCCESS(status);

    /* Prepare the init vector */
    memset(bInitVector, 0x00, 16);
    memcpy(bInitVector, bCardUid, 7);

    /* Diversify the system key */
    status = phCryptoSym_LoadIv(alMfdf.pCryptoDataParamsEnc,
        bInitVector,
        8);
    CHECK_SUCCESS(status);

    /* Diversify the system key using card uid as div input */
    status = phCryptoSym_DiversifyKey(alMfdf.pCryptoDataParamsEnc,
        PHAL_MFDF_DIV_METHOD_CMAC,
        SYSTEM_KEY_NP,
        0,
        bCardUid,
        7,
        bNewKey);
    CHECK_SUCCESS(status);

    /* Load the diversified key and the init vector */
    status = phCryptoSym_LoadKeyDirect(alMfdf.pCryptoDataParamsEnc,
        bNewKey,
        PH_CRYPTOSYM_KEY_TYPE_3K3DES
        );
    CHECK_SUCCESS(status);
    status = phCryptoSym_LoadIv(alMfdf.pCryptoDataParamsEnc,
        bInitVector,
        8);
    CHECK_SUCCESS(status);

    /* Calculate CMAC which is the digital signature */
    status = phCryptoSym_CalculateMac(alMfdf.pCryptoDataParamsEnc,
        PH_CRYPTOSYM_MAC_MODE_CMAC,
        bNPCardData,
        24,
        bMac,
        &bGenLen);
    CHECK_SUCCESS(status);

    /* Copy the digital signature at the end of the card data */
    memcpy(&bNPCardData[24], bMac, 8);

    /* Update bNPPacsData with digital signature */
    /* Load the diversified key and the init vector */
    status = phCryptoSym_LoadKeyDirect(alMfdf.pCryptoDataParamsEnc,
        bNewKey,
        PH_CRYPTOSYM_KEY_TYPE_3K3DES
        );
    CHECK_SUCCESS(status);
    status = phCryptoSym_LoadIv(alMfdf.pCryptoDataParamsEnc,
        bInitVector,
        8);
    CHECK_SUCCESS(status);

    /* Calculate CMAC which is the digital signature */
    status = phCryptoSym_CalculateMac(
        alMfdf.pCryptoDataParamsEnc,    /* [In] Pointer to this layer's parameter structure. */
        PH_CRYPTOSYM_MAC_MODE_CMAC,     /* [In] Option byte specifying the MAC mode and the update behavior of the IV and the completion flag.*/
        bNPPacsData,                    /* [In] input block */
        40,                             /* [In] number of input data bytes */
        bMac,                           /* [Out] output MAC block */
        &bGenLen);                      /* [Out] Length of MAC */
    CHECK_SUCCESS(status);

    /* Store the signature in the last 8 bytes */
    memcpy(&bNPPacsData[40], bMac, 8);

    /* Finally change the key 0 and key 1 */
    /* Select the application */
    status = phalMfdf_SelectApplication(&alMfdf, bNpacsApp);
    CHECK_SUCCESS(status);

    /* Authenticate */
    status = phalMfdf_AuthenticateISO(&alMfdf,
        PHAL_MFDF_NO_DIVERSIFICATION,
        DEFAULT_3K3DES,
        0,
        0,
        NULL,
        0);
    CHECK_SUCCESS(status);

    /* Change key2 on card without Diversification */
    status = phalMfdf_ChangeKey(&alMfdf,
        PHAL_MFDF_NO_DIVERSIFICATION ,
        DEFAULT_3K3DES,
        0,
        NPKEY_FOR_UID,
        0,
        2, /* Key number on card */
        NULL,
        0);
    CHECK_SUCCESS(status);

    /* Change key1 on card with Diversification */
    status = phalMfdf_ChangeKey(&alMfdf,
        PHAL_MFDF_CHGKEY_DIV_NEW_KEY | PHAL_MFDF_CHGKEY_DIV_METHOD_CMAC ,
        DEFAULT_3K3DES,
        0,
        NPKEY1,
        0,
        1, /* Key number on card */
        bCardUid,
        7);
    CHECK_SUCCESS(status);

    /* Change key 0 on card with Diversification */
    status = phalMfdf_ChangeKey(&alMfdf,
        PHAL_MFDF_CHGKEY_DIV_NEW_KEY | PHAL_MFDF_CHGKEY_DIV_METHOD_CMAC ,
        DEFAULT_3K3DES,
        0,
        NPKEY0,
        0,
        0,
        bCardUid,
        7);
    CHECK_SUCCESS(status);

    /* Re authenticate */
    status = phalMfdf_SelectApplication(&alMfdf,    /* [In] Pointer to this layer's parameter structure. */
        bNpacsApp);                                 /* [In] 3 Byte UID. LSB First. */
    CHECK_SUCCESS(status);

    /* Authenticate */
    status = phalMfdf_AuthenticateISO(
        &alMfdf,                        /* [In] Pointer to this layer's parameter structure. */
        PHAL_MFDF_DIV_METHOD_CMAC,    /* [In] Diversification option. */
        NPKEY0,                         /* [In] key number in keystore to authenticate with. */
        0,                              /* [In] Key version in the key store. */
        0,                              /* [In] Key number on card. */
        bCardUid,                       /* [In] Diversification input. Can be NULL. */
        7);                             /* [In] Length of diversification input max 31B. */
    CHECK_SUCCESS(status);

    bDataLen[0] = 32;
    /* Write Data to Card Object */
    status = phalMfdf_WriteData(
        &alMfdf,                        /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_PLAIN,  /* [In] Communication Mode. Plain, Mac'd or encrypted. */
        NPCARDOBJ,                      /* [In] 1 byte file number. */
        bOffSet,                        /* [In] 3 bytes offset. LSB First. */
        bNPCardData,                    /* [In] Data to be written. */
        bDataLen);                      /* [in] 3 bytes LSB First. length of data to be written. */
    CHECK_SUCCESS(status);

    bDataLen[0] = 48;
    /* Write Data to PACS Object */
    status = phalMfdf_WriteData(
        &alMfdf,                        /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_PLAIN,  /* [In] Communication Mode. Plain, Mac'd or encrypted. */
        NPPACSOBJ,                      /* [In] 1 byte file number. */
        bOffSet,                        /* [In] 3 bytes offset. LSB First. */
        bNPPacsData,                    /* [In] Data to be written. */
        bDataLen);                      /* [in] 3 bytes LSB First. length of data to be written. */
    CHECK_SUCCESS(status);

    /* Change file settings of card object */
    bAccessRights[0] = 0xF0; /* LSB of access rights. RW = never, Change config key = bNPKey0 */
    bAccessRights[1] = 0xEF; /* MSB. R = free, W = Never ) */

    status = phalMfdf_ChangeFileSettings(&alMfdf,
        PHAL_MFDF_COMMUNICATION_ENC,
        NPCARDOBJ,
        PHAL_MFDF_COMMUNICATION_PLAIN,
        bAccessRights);
    CHECK_SUCCESS(status);

    /* Change file settings of PACS object */
    bAccessRights[0] = 0xF0; /* LSB of access rights. RW = never, Change config key = bNPKey0 */
    bAccessRights[1] = 0x1F; /* MSB. R = bNPKey1, W = W = Never) */
    status = phalMfdf_ChangeFileSettings(&alMfdf,
        PHAL_MFDF_COMMUNICATION_ENC,
        NPPACSOBJ,
        PHAL_MFDF_COMMUNICATION_PLAIN,
        bAccessRights);
    CHECK_SUCCESS(status);

    printf(" \nPersonalization of NextPacs application done \n");
    /* ########################End personalization ############################# */
    return 0;
}

/* Print the contents of data files of the NextPacs Application */
phStatus_t PrintNextPACSApp(phalMfdf_Sw_DataParams_t * pAlMfdf)
{
    phStatus_t status;
    uint8_t bOffSet[3] = {0, 0, 0};
    uint8_t bLength[3] = {0, 0, 0};
    uint8_t bName[17];
    uint8_t * pRecv;
    uint16_t wRxlen;

    status = phalMfdf_SelectApplication(pAlMfdf, bNpacsApp);
    CHECK_SUCCESS(status);

    /* Read and print the Card Data Object */
    status = phalMfdf_ReadData(
        pAlMfdf,                        /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_PLAIN,  /* [In] Is either plain or encrypted or MAC'd. */
        NPCARDOBJ,                      /* [In] 1 byte file number. */
        bOffSet,                        /* [In] 3 bytes offset. LSB First. */
        bLength,                        /* [In] 3 bytes. length of data to be read. If 00, entire file will be read. */
        &pRecv,                         /* [Out] Pointer to HAL Rx buffer returned back to user. */
        &wRxlen);                       /* [Out] Pointer to Length of RxData. */
    CHECK_SUCCESS(status);

    if (wRxlen != 32)
    {
        printf(" An error occured. Cannot read all the data from Card object. Quitting ... \n");
        return 0xFF;
    }

    printf("\nContents of Card Data Object \n");
    memcpy(bName, pRecv, 16);
    bName[16] = '\0';
    printf("Manufacturer name           : %s \n", bName);
    printf("Mutual Authentication code  : ");
    PRINT_BUFFER(&pRecv[16], 2);
    printf("Communication Encryption    : %02x \n", pRecv[18]);
    printf("Customer Id                 : ");
    PRINT_BUFFER(&pRecv[19], 4);
    printf("Key Version                 : %02x \n", pRecv[23]);

    /* Authenticate with uid retriever key */
    status = phalMfdf_AuthenticateISO(pAlMfdf,
        PHAL_MFDF_NO_DIVERSIFICATION,
        NPKEY_FOR_UID,
        0,
        2, /* Key number 2 on the card */
        NULL,
        0);
    CHECK_SUCCESS(status);

    /* Get Card UID */
    status = phalMfdf_GetCardUID(
        pAlMfdf,    /* [In] Pointer to this layer's parameter structure. */
        bCardUid);  /* [Out] UID of the card. Buffer size should be 7 bytes. */
    CHECK_SUCCESS(status);

    /* Authenticate with NPKEY 0 */
    status = phalMfdf_AuthenticateISO(pAlMfdf,
        PHAL_MFDF_DIV_METHOD_CMAC,
        NPKEY1,
        0,
        1,
        bCardUid,
        7);
    CHECK_SUCCESS(status);

    bLength[0] = 48;
    /* Next print the contents of the NP Pacs data object */
    status = phalMfdf_ReadData(
        pAlMfdf,                        /* [In] Pointer to this layers param structure. */
        PHAL_MFDF_COMMUNICATION_PLAIN,  /* [In] Is either plain or encrypted or MAC'd. */
        NPPACSOBJ,                      /* [In] 1 byte file number. */
        bOffSet,                        /* [In] 3 bytes offset. LSB First. */
        bLength,                        /* [In] 3 bytes. length of data to be read. If 00, entire file will be read. */
        &pRecv,                         /* [Out] Pointer to HAL Rx buffer returned back to user. */
        &wRxlen);                       /* [Out] Pointer to Length of RxData. */
    CHECK_SUCCESS(status);

    if (wRxlen != 48)
    {
        printf(" An error occured. Cannot read all the data from PACS object. Quitting ... \n");
        return 0xFF;
    }
    printf("\nContents of PACS Data Object \n");
    printf("Version Major               : %02x\n", pRecv[0]);
    printf("Version Minor               : %02x\n", pRecv[1]);
    printf("Customer / Site code        : ");
    PRINT_BUFFER(&pRecv[2], 5);
    printf("Credential Id               : ");
    PRINT_BUFFER(&pRecv[7], 8);
    printf("Reissue Code                : %02x\n", pRecv[15]);
    printf("PIN Code                    : ");
    PRINT_BUFFER(&pRecv[16], 4);
    printf("Customer Data               : ");
    PRINT_BUFFER(&pRecv[20], 20);

    return 0;
}
/* Stores all the keys in the key store */
phStatus_t StoreKeysInKeyStore(phKeyStore_Sw_DataParams_t * pKeyStore)
{
    phStatus_t status;
    phKeyStore_Sw_DataParams_t keyStore;
    uint8_t bDefKey[24];

    keyStore = *pKeyStore;

    /* Set the default PICC Master key in the key store */
    status = phKeyStore_FormatKeyEntry(&keyStore, PICCKEY, PH_CRYPTOSYM_KEY_TYPE_2K3DES);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(&keyStore,
        PICCKEY,
        0x00,
        PH_CRYPTOSYM_KEY_TYPE_2K3DES,
        bDefaultPICCKey,
        0x00);
    CHECK_SUCCESS(status);

    /* Set the Application Master key in the key store */
    status = phKeyStore_FormatKeyEntry(&keyStore, AMKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(&keyStore,
        AMKEY,
        0x00,
        PH_CRYPTOSYM_KEY_TYPE_AES128,
        bAMKey,
        0x00);
    CHECK_SUCCESS(status);

    /* Set the Application Common key in the key store */
    status = phKeyStore_FormatKeyEntry(&keyStore, ACKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(&keyStore,
        ACKEY,
        0x00,
        PH_CRYPTOSYM_KEY_TYPE_AES128,
        bACKey,
        0x00);
    CHECK_SUCCESS(status);

    /* Set the Application Read key in the key store */
    status = phKeyStore_FormatKeyEntry(&keyStore, ARKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(&keyStore,
        ARKEY,
        0x00,
        PH_CRYPTOSYM_KEY_TYPE_AES128,
        bARKey,
        0x00);
    CHECK_SUCCESS(status);

    /* Set the Application Write key in the key store */
    status = phKeyStore_FormatKeyEntry(&keyStore, AWKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(&keyStore,
        AWKEY,
        0x00,
        PH_CRYPTOSYM_KEY_TYPE_AES128,
        bAWKey,
        0x00);
    CHECK_SUCCESS(status);

    /* Set the Application Accreditation key in the key store */
    status = phKeyStore_FormatKeyEntry(&keyStore, AAKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(&keyStore,
        AAKEY,
        0x00,
        PH_CRYPTOSYM_KEY_TYPE_AES128,
        bAAKey,
        0x00);
    CHECK_SUCCESS(status);

    /* Set the new PICC AES128 key in the key store */
    status = phKeyStore_FormatKeyEntry(&keyStore, PICCAESKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(&keyStore,
        PICCAESKEY,
        0x00,
        PH_CRYPTOSYM_KEY_TYPE_AES128,
        bAESPICCKey,
        0x00);
    CHECK_SUCCESS(status);

    /* Set the new Application Master key in the key store */
    status = phKeyStore_FormatKeyEntry(&keyStore, AMKEY_NEW, PH_CRYPTOSYM_KEY_TYPE_AES128);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(&keyStore,
        AMKEY_NEW,
        0x00,
        PH_CRYPTOSYM_KEY_TYPE_AES128,
        bAESPICCKey,
        0x00);
    CHECK_SUCCESS(status);

    /* Set the SYSTEM key in the key store */
    status = phKeyStore_FormatKeyEntry(&keyStore, SYSTEM_KEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(&keyStore,
        SYSTEM_KEY,
        0x00,
        PH_CRYPTOSYM_KEY_TYPE_AES128,
        bSysKey,
        0x00);
    CHECK_SUCCESS(status);

    /* Set the UID retriever key in the key store */
    status = phKeyStore_FormatKeyEntry(&keyStore, AFCKEY_FOR_UID, PH_CRYPTOSYM_KEY_TYPE_AES128);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(&keyStore,
        AFCKEY_FOR_UID,
        0x00,
        PH_CRYPTOSYM_KEY_TYPE_AES128,
        bAfcKeyN,
        0x00);
    CHECK_SUCCESS(status);

    /* Set the NPKEY0 in the key store */
    status = phKeyStore_FormatKeyEntry(&keyStore, NPKEY0, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(&keyStore,
        NPKEY0,
        0x00,
        PH_CRYPTOSYM_KEY_TYPE_3K3DES,
        bNPKey0,
        0x00);
    CHECK_SUCCESS(status);

    /* Set the NPKEY1 in the key store */
    status = phKeyStore_FormatKeyEntry(&keyStore, NPKEY1, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(&keyStore,
        NPKEY0,
        0x00,
        PH_CRYPTOSYM_KEY_TYPE_3K3DES,
        bNPKey1,
        0x00);
    CHECK_SUCCESS(status);

    /* Set the NP system KEY in the key store */
    status = phKeyStore_FormatKeyEntry(&keyStore, SYSTEM_KEY_NP, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(&keyStore,
        SYSTEM_KEY_NP,
        0x00,
        PH_CRYPTOSYM_KEY_TYPE_3K3DES,
        bSysKeyNP,
        0x00);
    CHECK_SUCCESS(status);

    memset(bDefKey, 0, 24);
    /* Set the default 3K3DES KEY in the key store */
    status = phKeyStore_FormatKeyEntry(&keyStore, DEFAULT_3K3DES, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(&keyStore,
        DEFAULT_3K3DES,
        0x00,
        PH_CRYPTOSYM_KEY_TYPE_3K3DES,
        bDefKey,
        0x00);
    CHECK_SUCCESS(status);

    /* Set the UID retriever KEY for NP App in the key store */
    status = phKeyStore_FormatKeyEntry(&keyStore, NPKEY_FOR_UID, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(&keyStore,
        NPKEY_FOR_UID,
        0x00,
        PH_CRYPTOSYM_KEY_TYPE_3K3DES,
        bNPKey2,
        0x00);
    CHECK_SUCCESS(status);

    return 0;
}

int __cdecl main(int argc, char *argv[])
{
    phStatus_t status;
    uint8_t bReaderListPegoda[512];
    uint8_t bCidEnable, bCid, bNadSupported, bFwi, bFsdi, bFsci;
    uint8_t bSak[1];
    uint8_t bUid[10];
    uint8_t pSeed[8];
    uint8_t bMoreCardsAvaliable;
    uint8_t bLength;
    uint16_t wCount;
    int iChoice;
    phKeyStore_Sw_KeyEntry_t keyEntry[KEYCOUNT];
    phKeyStore_Sw_KeyVersionPair_t keyVersion[KEYCOUNT * KEYVERSIONS];
    phKeyStore_Sw_KUCEntry_t keyUsage[KEYCOUNT];

    /* Bfl data parameter storage */
    phbalReg_Rd70xUsbWin_DataParams_t balReader;
    phhalHw_Rd70x_DataParams_t halReader;
    phpalI14443p3a_Sw_DataParams_t I14443p3a;
    phpalI14443p4a_Sw_DataParams_t I14443p4a;
    phpalI14443p4_Sw_DataParams_t I14443p4;
    phpalMifare_Sw_DataParams_t palMifare;
    phCryptoSym_Sw_DataParams_t cryptoEnc;
    phCryptoSym_Sw_DataParams_t cryptoSymRnd;
    phCryptoRng_Sw_DataParams_t cryptoRng;
    phKeyStore_Sw_DataParams_t keyStore;
    phalMfdf_Sw_DataParams_t alMfdf;

    void * pHal;
	void * pKeyStore;                                 /**< Pointer to a KeyStore parameter structure */

    uint8_t pAts[255];
    uint8_t bValue[4] = {0, 0, 0, 0};

    PHAL_MFDF_UNUSED_VARIABLE(argc);
    PHAL_MFDF_UNUSED_ARRAY(argv);

    printf("\nNxpRdLib ANSI-C Example MIFARE DesFire EV1 Program\n\n");
    printf("Please ensure that a Pegoda reader is connected and in working condition.\n\n");
    printf("Firmware Version 2.11 or higher must be used!\n\n");
    printf("Performing startup...\n\n");

    /* init. keystore */
    status = phKeyStore_Sw_Init(
        &keyStore,
        sizeof(phKeyStore_Sw_DataParams_t),
        keyEntry,
        KEYCOUNT,
        keyVersion,
        KEYVERSIONS,
        keyUsage,
        KEYCOUNT);
    CHECK_SUCCESS(status);

    pKeyStore = &keyStore;

    /* init. crypto */
    status = phCryptoSym_Sw_Init(&cryptoEnc, sizeof(phCryptoSym_Sw_DataParams_t), &keyStore);
    CHECK_SUCCESS(status);
    status = phCryptoSym_Sw_Init(&cryptoSymRnd, sizeof(phCryptoSym_Sw_DataParams_t), &keyStore);
    CHECK_SUCCESS(status);
    status = phCryptoRng_Sw_Init(&cryptoRng, sizeof(phCryptoRng_Sw_DataParams_t), &cryptoSymRnd);
    CHECK_SUCCESS(status);
    memset(pSeed, 0x00, 0x08);
    status = phCryptoRng_Seed(&cryptoRng, pSeed, 8);
    CHECK_SUCCESS(status);

    /* Initialise the Reader BAL component */
    status = phbalReg_Rd70xUsbWin_Init(&balReader, sizeof(phbalReg_Rd70xUsbWin_DataParams_t));
    CHECK_SUCCESS(status);

    /* Get list of connected pegoda readers */
    status = phbalReg_GetPortList(&balReader, sizeof(bReaderListPegoda), bReaderListPegoda, &wCount);
    CHECK_SUCCESS(status);

    /* Connect to the first reader */
    status = phbalReg_SetPort(&balReader, bReaderListPegoda);
    CHECK_SUCCESS(status);

    /* open the reader port */
    status = phbalReg_OpenPort(&balReader);
    CHECK_SUCCESS(status);

    /* init the Reader HAL component */
    status = phhalHw_Rd70x_Init(
        &halReader,
        sizeof(phhalHw_Rd70x_DataParams_t),
        &balReader,
        gaHalRxReader,
        sizeof(gaHalRxReader),
        gaHalTxReader,
        sizeof(gaHalTxReader));

    CHECK_SUCCESS(status);

    /* The default reader is the Pegoda */
    pHal = &halReader;

    /* init the 14443-3A component */
    status = phpalI14443p3a_Sw_Init(&I14443p3a, sizeof(phpalI14443p3a_Sw_DataParams_t), pHal);
    CHECK_SUCCESS(status);

    /* init the 14443-4A component */
    status = phpalI14443p4a_Sw_Init(&I14443p4a, sizeof(phpalI14443p4a_Sw_DataParams_t), pHal);
    CHECK_SUCCESS(status);

    /* init the 14443-4 component */
    status = phpalI14443p4_Sw_Init(&I14443p4, sizeof(phpalI14443p4_Sw_DataParams_t), pHal);
    CHECK_SUCCESS(status);

    /* init. mifare pal */
    status = phpalMifare_Sw_Init(&palMifare, sizeof(phpalMifare_Sw_DataParams_t), pHal, &I14443p4);
    CHECK_SUCCESS(status);

    /* Initialize logging */
#ifdef NXPBUILD__PH_LOG
    /* Register the component for logging */
    status = phLog_Init(phalMfdf_log, logRegisterEntries, (uint16_t)(sizeof(logRegisterEntries) / sizeof(phLog_RegisterEntry_t)));
	CHECK_SUCCESS(status);
    status = phLog_Register(&cryptoEnc, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(status);
    status = phLog_Register(&cryptoRng, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(status);
    status = phLog_Register(&alMfdf, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(status);
#endif /* NXPBUILD__PH_LOG */

    /* ####################Card Initialization########################### */

    /* Switch on the field */
    status = phhalHw_FieldOn(pHal);
    CHECK_SUCCESS(status);

    /* Configure HAL for Type-A cards */
    status = phhalHw_ApplyProtocolSettings(pHal, PHHAL_HW_CARDTYPE_ISO14443A);
    CHECK_SUCCESS(status);

    /* Activate Layer 3 card. In loop till a card is detected. */
    do
    {
        status = phpalI14443p3a_ActivateCard(&I14443p3a, NULL, 0x00, bUid, &bLength, bSak, &bMoreCardsAvaliable);
    } while (status != PH_ERR_SUCCESS);
    CHECK_SUCCESS(status);

    /* switch on time mesaurement */
    status = phhalHw_SetConfig(&halReader, PHHAL_HW_CONFIG_TIMING_MODE, PHHAL_HW_TIMING_MODE_COMM);
    CHECK_SUCCESS(status);

    /* Send RATS */
    status = phpalI14443p4a_Rats(&I14443p4a, 0x08, 0x01, pAts);
    CHECK_SUCCESS(status);

    /* Obtain the protocol parameters from layer 4a and then store them in layer 4 */
    /*
    bCidEnable indicates if the card supports Cid. if 0, then no cid is supported.
    bNadSupported indicates if card supports Node address. if 0, NAD is not supported.
    bFwi => Framw waiting integer. can be zero to maximum value can be 0 TO 15.
    bFsdi -> Frame size indicator of pcd 0 to 8
    bFsci -> Frame size indicator of Card 0 - 8
    */
    status = phpalI14443p4a_GetProtocolParams(&I14443p4a, &bCidEnable, &bCid, &bNadSupported, &bFwi, &bFsdi, &bFsci);
    CHECK_SUCCESS(status);

    status = phpalI14443p4_SetProtocol(&I14443p4, bCidEnable, bCid, bNadSupported, 0, bFwi, bFsdi, bFsci);
    CHECK_SUCCESS(status);
    /* ####################End Card Initialization########################### */

    /* Initialize the MF DesFire EV1 component */
    status = phalMfdf_Sw_Init(&alMfdf,      /* [In] Pointer to this layer's parameter structure. */
        sizeof(phalMfdf_Sw_DataParams_t),   /* [In] Specifies the size of the data parameter structure */
        &palMifare,                         /* [In] Pointer to a palMifare component context. */
        &keyStore,                          /* [In] Pointer to Key Store data parameters. */
        &cryptoEnc,                         /* [In] Pointer to a Crypto component context for encryption. */
        &cryptoRng,                         /* [In] Pointer to a CryptoRng component context. */
        &halReader);                        /* [In] Pointer to the HAL parameters structure. */
    CHECK_SUCCESS(status);

    /* Set all the keys in the S/W key store */
    status = StoreKeysInKeyStore(&keyStore);
    CHECK_SUCCESS(status);

    printf("A Ticket is discovered \n");
    printf("Press 1 for Personalization and Accreditation of AFC App \n");
    printf("Press 2 for Checkout - AFC App \n");
    printf("Press 3 for Crediting - AFC App\n");
    printf("Press 4 for Personalization - NextPACs App\n");
    printf("Press 5 to print the card object and PACs object of NextPacs \n");
    printf("Press x to quit \n");
    iChoice = _getch();

    switch(iChoice)
    {
    case('1'):
        /* Personalization and Accreditation of AFC App */

        /* Pre personalization */
        status = PrePersonalization_AFCApp(&alMfdf, bCardUid);
        CHECK_SUCCESS(status);

        /* Personalization */
        status = Personalization_AFCApp(&keyStore, &alMfdf, bCardUid);
        CHECK_SUCCESS(status);

        /* Accreditation */
        status = Accreditation(&alMfdf);
        CHECK_SUCCESS(status);

#if 0
        /* #################### Finalization ############################### */
        /* CURRENTLY DISABLED. CAN BE UNCOMMENTED IN THE FINAL CODE. THIS
        * CHANGES THE PICC MASTER KEY FROM 2K3DES TO AES128
        * IF THIS IS ENABLED, PLEASE ENSURE THAT YOU USE PICCAESKEY for
        * PICC KEY authentication instead of PICCKEY.
        */
        /* Select Master application */
        memset(pAid, 0, 3);
        status = phalMfdf_SelectApplication(&alMfdf, pAid);
        CHECK_SUCCESS(status);

        status = phalMfdf_AuthenticateISO(&alMfdf,
            PHAL_MFDF_NO_DIVERSIFICATION,
            PICCKEY,
            0x00,
            0x00,
            NULL,
            0);
        CHECK_SUCCESS(status);

        /* Change PICC Master key from 2K3DES to AES128 */
        status = phalMfdf_ChangeKey(&alMfdf,
            PHAL_MFDF_NO_DIVERSIFICATION,
            PICCKEY,
            0x00,
            PICCAESKEY,
            0x00,
            0x80,
            NULL,
            0x00);
        CHECK_SUCCESS(status);

        /* Re authenticate */
        status = phalMfdf_SelectApplication(&alMfdf, pAid);
        CHECK_SUCCESS(status);

        status = phalMfdf_AuthenticateAES(&alMfdf,
            PHAL_MFDF_NO_DIVERSIFICATION,
            PICCAESKEY,
            0x00,
            0x00,
            NULL,
            0);
        CHECK_SUCCESS(status);

        status = phalMfdf_ChangeKey(&alMfdf,
            PHAL_MFDF_NO_DIVERSIFICATION,
            PICCAESKEY,
            0x00,
            PICCKEY,
            0x00,
            0x00,
            NULL,
            0x00);
        CHECK_SUCCESS(status);
        /* #################### End Finalization ############################### */
#endif
        /* Deselect */
        phhalHw_FieldReset(&halReader);
        break;

    case '2':
        status = CheckOutAfcApp(&alMfdf);
        CHECK_SUCCESS(status);
        printf("DOOR OPENED !!! \n");

        /* Deselect */
        phhalHw_FieldReset(&halReader);
        break;

    case '3':
        /* Accreditation */
        memset(bValue, 0, 4);
        bValue[0] = 50;
        printf(" Crediting 50 units to your account \n");
        status =  Accreditation_WithValue(&alMfdf,
            bValue);
        CHECK_SUCCESS(status);

        /* Deselect */
        phhalHw_FieldReset(&halReader);
        break;

    case '4':
        /* Personalization of NextPACS App */
        status = Perso_NextPACSApp(&alMfdf);
        CHECK_SUCCESS(status);
        /* Deselect */
        phhalHw_FieldReset(&halReader);
        break;

    case '5':
        /* Print the Card Object and PACS object of NP app */
        status = PrintNextPACSApp(&alMfdf);
        CHECK_SUCCESS(status);
        /* Deselect */
        phhalHw_FieldReset(&halReader);
        break;

    default:
        printf(" Quitting ... \n");
        break;
    }
    printf("-------------------------\n");
    printf("Press any key to continue...\n\n");
    iChoice = _getch();

    /* Returns zero if the procedure was successful. */
    return 0;
}
