/*
 * Copyright 2013, 2016, 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for Rd710 MF0EV1.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/**
* Header for this file
*/
#include "Example-Rd710_MF0EV1.h"
#pragma warning(disable:4996)

#define PAGE_ADDRESS_2              0x02U   /**< Page 2 used for BCC1, internal, Lock Byte 0 and Lock Byte 1 */
#define OTP_ADDRESS                 0x03U   /**< One time programmable (OTP) page address */
#define OTP_EXAMPLE_DATA_ADDRESS    0x08U   /**< OTP example data address */
#define NAME_PAGE_ADDRESS           0x08U   /**< MF0EV1 example: 'name' page address */
#define ADDRESS_PAGE_ADDRESS        0x09U   /**< MF0EV1 example: 'address' page address */
#define LOCK_BYTE_1_LOCK_PAGE_8     0x01U   /**< MF0EV1 example: Lock byte 1 should lock page 8 */
#define LOCK_BYTE_1_LOCK_PAGE_9     0x02U   /**< MF0EV1 example: Lock byte 1 should lock page 9 */
#define BLOCK_LOCK0_PAGE            0x02U   /**< Block lock bit page address*/
#define NFC_COUNTER_ADDRESS         0x84U   /**< NFC counter enable page address*/
#define ASCII_MIRROR_ADDRESS        0x83U   /**< ASCII mirror configuration page address*/
#define NFC_ASCII_MIRROR_PAGE       0x20U   /**< NFC ASCII mirror page address*/
#define UID_ASCII_MIRROR_PAGE       0x30U   /**< UID ASCII mirror page address*/

#define PCSC_READER_P2_NO_SAM_NAME  "NXP Pegoda N CL 0 0"   /**< Pegoda 2: no SAM reader */
#define READ_RESPONSE_SIZE          0x10U                   /**< MF0EV1 read command response size */
#define GLOBAL_BUFFER_SIZE          0x012CU                 /**< global size of transmit and recieve buffer */
#define PHAL_MFUL_UNUSED_VARIABLE(x)  {for( ( x ) = ( x ) ; ( x ) != ( x ) ; );}


#define CHECK_SUCCESS(x)                                                \
    if ((x) != PH_ERR_SUCCESS)                                          \
{                                                                       \
    printf("An error occured (0x%04X), press any key to exit...", (x)); \
    _getch();                                                           \
    return 0;                                                           \
}

/* Utility function to print a buffer */
void PRINT_BUFFER(uint8_t * pBuffer, uint16_t wLength)
{
    uint16_t wIndex;

    for (wIndex = 0; wIndex < wLength; ++wIndex)
    {
        printf("%02X ", pBuffer[wIndex]);
    }
    printf("\n");
}

/* Logging function. Turned OFF currently */
#ifdef NXPBUILD__PH_LOG
void phalMful_log(void * pDataParams, uint8_t bOption, phLog_LogEntry_t * pEntries, uint16_t wEntryCount)
{
    uint16_t i;
    PHAL_MFUL_UNUSED_VARIABLE(bOption);
    PHAL_MFUL_UNUSED_VARIABLE(pDataParams);

    for (i = 0; i < wEntryCount; i++ )
    {
        if ( i != 0 )
        {
            printf("%s : ", pEntries[i].pString);
        }
        else
        {
            printf("-----------------------------------------\n");
            printf("%s \n", pEntries[0].pString);
            printf("-----------------------------------------\n");
        }
        PRINT_BUFFER((uint8_t *)(pEntries[i].pData), pEntries[i].wDataLen);
    }
    printf("\n\n");
}
#endif /* NXPBUILD__PH_LOG */


static uint8_t aUid[10];  /**< complete Uid; uint8_t[4/7/10]. */
static uint8_t aPwd[PHAL_MFUL_WRITE_BLOCK_LENGTH] = {0xFF, 0xFF, 0xFF, 0xFF};   /* Password for pwdAuth */

#ifdef NXPBUILD__PH_LOG
phLog_RegisterEntry_t logRegisterEntries[16];
phLog_LogEntry_t logEntries[48];
#endif /* NXPBUILD__PH_LOG */

/**
* Reset card : Function to reset and activate the PICC
*
*/
uint8_t resetCard(phalMful_Sw_DataParams_t * palMful,             /**< Pointer to MIFARE Ultralight parameter structure. */
                  phpalMifare_Sw_DataParams_t * palMifare,        /**< Pointer to MIFARE parameter structure. */
                  phhalHw_Rd70x_DataParams_t * pHal,              /**< Pointer to HAL parameter structure */
                  phpalI14443p3a_Sw_DataParams_t * palI14443p3a)  /**< PAL-ISO14443P3A parameter structure */
{
    phStatus_t status;
    uint8_t aSak[1];                /**< Select Acknowledge; uint8_t. */
    uint8_t bMoreCardsAvaliable;    /**< more card available */
    uint8_t bLength;                /**< Length */

    /**
    * init. Ultralight component
    */
    status = phalMful_Sw_Init(palMful,    /**< [In] Pointer to this layer's parameter structure. */
        sizeof(phalMful_Sw_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
        palMifare,                        /**< [In] Pointer to palMifare parameter structure. */
        NULL,                             /**< [In] Pointer to phKeystore parameter structure. */
        NULL,                             /**< [In] Pointer to phCrypto data parameters structure. */
        NULL);                            /**< [In] Pointer to the parameter structure of the CryptoRng layer. */
    CHECK_SUCCESS(status);

    /**
    * Field Reset
    */
    status = phhalHw_FieldReset(pHal);  /**< [In] Pointer to this layer's parameter structure. */
    CHECK_SUCCESS(status);

    /**
    * Activate Layer 3 card
    */
    status = phpalI14443p3a_ActivateCard(palI14443p3a,  /**< [In] PAL-ISO14443P3A parameter structure */
        NULL,                                           /**< [In] unknown UID */
        0x00,                                           /**< [In] UID length 0 */
        aUid,                                           /**< [Out] complete Uid; uint8_t[4/7/10]. */
        &bLength,                                       /**< [Out] Length of Uid; 4/7/10. */
        aSak,                                           /**< [Out] Select Acknowledge; uint8_t. */
        &bMoreCardsAvaliable);                          /**< [Out] Whether there are more cards in the field or not; uint8_t. */
    CHECK_SUCCESS(status);


    return PH_ERR_SUCCESS;
}

/**
* Password Auth : Function to do password auth
*
*/
uint8_t Password_Auth(phalMful_Sw_DataParams_t * palMful)
{
    phStatus_t status = 0;
    uint8_t aPack[PHAL_MFUL_PACK_LENGTH] = {0};

    status = phalMful_PwdAuth(palMful,  /**< [In] Pointer to this layer's parameter structure. */
		0x00,
		0x00,							/**< [In] Key number to be used in Sam hardware. Only valid for Sam AV3 NonX and X mode. */
		0x00,							/**< [In] Key version to be used in Sam hardware. Only valid for Sam AV3 NonX and X mode. */
		NULL,							/**< [In] Diversification input. Only valid for Sam AV3 NonX and X mode. */
		0,								/**< [In] Diversification input length. Only valid for Sam AV3 NonX and X mode. */
        aPwd,                           /**< [In] Four byte array of type uint8_t containing four password bytes */
        aPack);                         /**< [Out] pCntValue[3] counter value. LSB first. Three bytes counter value returned from PICC. */
    CHECK_SUCCESS(status);

    return PH_ERR_SUCCESS;
}

/**
* ReadWriteDemo : read, write, lock, OTP usecase
*
*/
uint8_t ReadWriteDemo(phalMful_Sw_DataParams_t * palMful)   /**< Pointer to MIFARE Ultralight parameter structure. */
{
    phStatus_t status;
    uint8_t aData[READ_RESPONSE_SIZE];           /**< Response data */
    uint8_t aVersion[PHAL_MFUL_VERSION_LENGTH] = {0};

    /**
    * a.) Read, write two parts: One is 'name' and the other 'address'
    * Perform Ultralight Read at NAME_PAGE_ADDRESS
    */
    status = phalMful_Read(palMful, /**< [In] Pointer to this layer's parameter structure. */
        NAME_PAGE_ADDRESS,          /**< [In] Address on Picc to read from. */
        aData);                     /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing data returned from the Picc. */
    CHECK_SUCCESS(status);

    /**
    * Perform Ultralight Read at ADDRESS_PAGE_ADDRESS
    */
    status = phalMful_Read(palMful, /**< [In] Pointer to this layer's parameter structure. */
        ADDRESS_PAGE_ADDRESS,       /**< [In] Address on Picc to read from. */
        aData);                     /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing data returned from the Picc. */
    CHECK_SUCCESS(status);

    /**
    * Perform Ultralight Write at NAME_PAGE_ADDRESS
    */
    status = phalMful_Write(palMful,    /**< [In] Pointer to this layer's parameter structure. */
        NAME_PAGE_ADDRESS,              /**< [In] Address on Picc to write to. */
        aData);                         /**< [In] pData[#PHAL_MFUL_WRITE_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);

    /**
    * Perform Ultralight Write at ADDRESS_PAGE_ADDRESS
    */
    status = phalMful_Write(palMful,    /**< [In] Pointer to this layer's parameter structure. */
        ADDRESS_PAGE_ADDRESS,           /**< [In] Address on Picc to write to. */
        aData);                         /**< [In] pData[#PHAL_MFUL_WRITE_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);

    /**
    * Get Version : Get the version details of the PICC
    */
    status = phalMful_GetVersion(palMful,   /**< [In] Pointer to this layer's parameter structure. */
        aVersion);                          /**< [Out] An eight byte array containing version information bytes */
    CHECK_SUCCESS(status);

    return PH_ERR_SUCCESS;
}

/**
* Count command Demo App
*
*/
uint8_t CounterDemo(phalMful_Sw_DataParams_t * palMful) /**< Pointer to MIFARE Ultralight parameter structure. */
{
    phStatus_t status;
    uint8_t aCnt[PHAL_MFUL_COUNTER_WR_VALUE_LENGTH] = {0, 1, 2, 4};
    uint8_t bCntNum = 0x0;
    uint8_t bValidFlag;

    /**
    * Increment Count Demo : Increment the counter with specified value
    */
    status = phalMful_IncrCnt(palMful,  /**< [In] Pointer to this layer's parameter structure. */
        bCntNum,                        /**< [In] One byte counter number 00 to 02 */
        aCnt);                          /**< [In] pCnt[4] counter value LSB first. Only first three data bytes are used by the PICC. The 4th byte is ignored */
    CHECK_SUCCESS(status);

    memset(aCnt, 0, PHAL_MFUL_COUNTER_WR_VALUE_LENGTH);
    /**
    * Read Count Demo : Read the counter value specified
    */
    status = phalMful_ReadCnt(palMful,  /**< [In] Pointer to this layer's parameter structure. */
        bCntNum,                        /**< [In] One byte counter number 00 to 02 */
        aCnt);                          /**< [Out] pCntValue[3] counter value. LSB first. Three bytes counter value returned from PICC. */
    CHECK_SUCCESS(status);

    /**
    * Check tearing event Demo : Check for the tearing event with the associated counter
    */
    status = phalMful_ChkTearingEvent(palMful,  /**< [In] Pointer to this layer's parameter structure. */
        bCntNum,                                /**< [In] Value specifying the counter number 00 to 02 */
        &bValidFlag);                           /**< [Out] Address of one byte containing the valid flag byte */
    CHECK_SUCCESS(status);

    return PH_ERR_SUCCESS;
}

/**
* Block Lock Demo App
*
*/
uint8_t BlockLockDemo(phalMful_Sw_DataParams_t * palMful,         /**< Pointer to MIFARE Ultralight parameter structure. */
                      phpalMifare_Sw_DataParams_t * palMifare,        /**< Pointer to MIFARE parameter structure. */
                      phhalHw_Rd70x_DataParams_t * pHal,              /**< Pointer to HAL parameter structure */
                      phpalI14443p3a_Sw_DataParams_t * palI14443p3a)
{
    phStatus_t status = 0;
    uint8_t aData[PHAL_MFUL_READ_BLOCK_LENGTH] = {0};
    uint8_t bAddr = 0;
    /**
    * Perform Ultralight Write: lock page 8
    * aData byte 0: BCC1 = 0x00
    * aData byte 1: internal  = 0x00
    * aData byte 2: Lock Byte 0 = 0x00
    * aData byte 3: Lock Byte 1 = LOCK_BYTE_1_LOCK_PAGE_8
    */
    aData[0] = aData[1] = aData[2] = 0x00;
    aData[3] = 0;/*LOCK_BYTE_1_LOCK_PAGE_8;*/
    bAddr = PAGE_ADDRESS_2;
    status = phalMful_Write(palMful,    /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                          /**< [In] Address on Picc to write to. */
        aData);                         /**< [In] pData[#PHAL_MFUL_WRITE_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);

    /**
    * read
    */
    bAddr = NAME_PAGE_ADDRESS;
    status = phalMful_Read(palMful, /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                      /**< [In] Address on Picc to read from. */
        aData);                     /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing data returned from the Picc. */
    CHECK_SUCCESS(status);

    /**
    * write impossible
    */
    bAddr = NAME_PAGE_ADDRESS;
    status = phalMful_Write(palMful,    /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                          /**< [In] Address on Picc to write to. */
        aData);                         /**< [In] pData[#PHAL_MFUL_WRITE_BLOCK_LENGTH] containing block to be written to the Picc. */

    if ((status) != PH_ERR_SUCCESS)
    {
        /**
        * write was impossible
        */
        status = PH_ERR_SUCCESS;
    }
    else
    {
        printf("An error occurred: write should be impossible, press any key to exit...");
        _getch();
        return 0;
    }

    /**
    * reset card
    */
    status = resetCard(palMful, /**< Pointer to MIFARE Ultralight parameter structure. */
        palMifare,              /**< Pointer to MIFARE parameter structure. */
        pHal,                   /**< Pointer to HAL parameter structure */
        palI14443p3a);          /**< PAL-ISO14443P3A parameter structure */
    CHECK_SUCCESS(status);

    aData[0] = 0x0;
    aData[1] = 0x0;
    aData[2] = 0x02;
    aData[3] = 0x0;
    bAddr = BLOCK_LOCK0_PAGE;
        status = phalMful_Write(palMful,    /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                          /**< [In] Address on Picc to write to. */
        aData);                         /**< [In] pData[#PHAL_MFUL_WRITE_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);
    printf("Lock bit for Page 4 to 9 is blocked \n");

    aData[0] = aData[1] = aData[2] = 0x00;
    aData[3] = 0;/*LOCK_BYTE_1_LOCK_PAGE_9;*/
    bAddr = PAGE_ADDRESS_2;
    status = phalMful_Write(palMful,    /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                          /**< [In] Address on Picc to write to. */
        aData);                         /**< [In] pData[#PHAL_MFUL_WRITE_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);

    printf("Not able to change the lock bit of Page 9 \n");

    bAddr = PAGE_ADDRESS_2;
    status = phalMful_Read(palMful, /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                      /**< [In] Address on Picc to read from. */
        aData);                     /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing data returned from the Picc. */
    CHECK_SUCCESS(status);

    bAddr = PAGE_ADDRESS_2;
    status = phalMful_Read(palMful, /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                      /**< [In] Address on Picc to read from. */
        aData);                     /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing data returned from the Picc. */
    CHECK_SUCCESS(status);

    /**
    * reset card
    */
    status = resetCard(palMful, /**< Pointer to MIFARE Ultralight parameter structure. */
        palMifare,              /**< Pointer to MIFARE parameter structure. */
        pHal,                   /**< Pointer to HAL parameter structure */
        palI14443p3a);          /**< PAL-ISO14443P3A parameter structure */
    CHECK_SUCCESS(status);

    return PH_ERR_SUCCESS;
}
/**
* OTP Demo App
*
*/
uint8_t OtpDemo(phalMful_Sw_DataParams_t * palMful)
{
    phStatus_t status = 0;
    uint8_t bAddr = OTP_ADDRESS;
    uint8_t aData[PHAL_MFUL_READ_BLOCK_LENGTH] = {0};

    status = phalMful_Read(palMful, /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                      /**< [In] Address on Picc to read from. */
        aData);                     /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing data returned from the Picc. */
    CHECK_SUCCESS(status);

    /* Set the OTP bit */
    aData[0] = 0x00;
    status = phalMful_Write(palMful,    /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                          /**< [In] Address on Picc to write to. */
        aData);                         /**< [In] pData[#PHAL_MFUL_WRITE_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);

    /* Read the Page at address */
    bAddr = OTP_EXAMPLE_DATA_ADDRESS;
    status = phalMful_Read(palMful, /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                      /**< [In] Address on Picc to read from. */
        aData);                     /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing data returned from the Picc. */
    CHECK_SUCCESS(status);

    printf("The card is active and no Reset is needed after writing to OTP \n");

    /* Try setting the OTP bit again */
    aData[0] = 0x01;
    bAddr = OTP_ADDRESS;
    /*Check out whether the overwrite works out or not*/
    status = phalMful_Write(palMful,    /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                          /**< [In] Address on Picc to write to. */
        aData);                         /**< [In] pData[#PHAL_MFUL_WRITE_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);

    return PH_ERR_SUCCESS;
}

/**
* Read Signature Demo App
*
*/
uint8_t ReadSignatureDemo(phalMful_Sw_DataParams_t * palMful)

{
    phStatus_t status;
    uint8_t bAddr = 0x0;
    uint8_t *pSignature;

    /* Read the signature from the PICC */
    status = phalMful_ReadSign(palMful, /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                          /**< [In] Value is always 00. Present for forward compatibility reasons */
        &pSignature);                   /**< [Out] Pointer to a 32 byte signature read from the card */
    CHECK_SUCCESS(status);

    return PH_ERR_SUCCESS;

}

/**
* Fast read command Check Demo App
*
*/
uint8_t FastReadDemo(phalMful_Sw_DataParams_t * palMful)
{
    phStatus_t status = 0;
    uint8_t bStartAddr = NAME_PAGE_ADDRESS;
    uint8_t bEndAddr = 0x19;
    uint8_t *pData;
    uint16_t wNumBytes = 0;

    status = phalMful_FastRead(palMful, /**< [In] Pointer to this layer's parameter structure. */
        bStartAddr,                     /**< [In] One byte start address */
        bEndAddr,                       /**< [In] One byte end address */
        &pData,                         /**< [Out] Pointer to the data read from the card. The bytes are stored on the HAL's Rx buffer. */
        &wNumBytes);                    /**< [Out] Contains number of bytes read from the card */
    CHECK_SUCCESS(status);

    return PH_ERR_SUCCESS;
}

/**
* Compatibility write Demo App
*
*/
uint8_t CompWriteDemo(phalMful_Sw_DataParams_t * palMful)
{
    phStatus_t status = 0;
    uint8_t bAddr = NAME_PAGE_ADDRESS;
    uint8_t aData[PHAL_MFUL_READ_BLOCK_LENGTH] =
    {
        0xFF, 0xF5, 0xFF, 0xAA,
        0xFF, 0xF5, 0xFF, 0xAA,
        0xFF, 0xF5, 0xFF, 0xAA,
        0xFF, 0xF5, 0xFF, 0xAA,
    };
    status = phalMful_CompatibilityWrite(palMful,   /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                                      /**< [In] Address on Picc to write to. */
        aData);                                     /**< [In] pData[#PHAL_MFUL_COMPWRITE_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);

    memset(aData, 0, PHAL_MFUL_READ_BLOCK_LENGTH);
    status = phalMful_Read(palMful, /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                      /**< [In] Address on Picc to read from. */
        aData);                     /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing data returned from the Picc. */
    CHECK_SUCCESS(status);

    return PH_ERR_SUCCESS;
}
/**
* Short read demo (ACL2 skip) App
*
*/
uint8_t ShortReadDemo(phalMful_Sw_DataParams_t * palMful,            /**< Pointer to MIFARE Ultralight parameter structure. */
                     phpalMifare_Sw_DataParams_t * palMifare,        /**< Pointer to MIFARE parameter structure. */
                     phhalHw_Rd70x_DataParams_t * pHal,              /**< Pointer to HAL parameter structure */
                     phpalI14443p3a_Sw_DataParams_t * palI14443p3a)  /**< PAL-ISO14443P3A parameter structure */
{
    phStatus_t status;
    uint8_t aAtqa[4];
    uint8_t aNvbUid[10];
    uint8_t bCascadeLevel = 0x93;
    uint8_t bAddr = 0x00;
    uint8_t aData[PHAL_MFUL_READ_BLOCK_LENGTH];

    /**
    * init. Ultralight component
    */
    status = phalMful_Sw_Init(palMful,    /**< [In] Pointer to this layer's parameter structure. */
        sizeof(phalMful_Sw_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
        palMifare,                        /**< [In] Pointer to palMifare parameter structure. */
        NULL,                             /**< [In] Pointer to phKeystore parameter structure. */
        NULL,                             /**< [In] Pointer to phCrypto data parameters structure. */
        NULL);                            /**< [In] Pointer to the parameter structure of the CryptoRng layer. */
    CHECK_SUCCESS(status);

    /**
    * Field Reset
    */
    status = phhalHw_FieldReset(pHal);  /**< [In] Pointer to this layer's parameter structure. */
    CHECK_SUCCESS(status);

    status = phpalI14443p3a_RequestA(palI14443p3a,  /**< [In] Pointer to this layer's parameter structure. */
        aAtqa);                                     /**< [Out] AtqA; uint8_t[2]. */
    CHECK_SUCCESS(status);

    status = phpalI14443p3a_Anticollision(palI14443p3a,     /**< [In] Pointer to this layer's parameter structure. */
        bCascadeLevel,                                      /**< [In] cascade level code.  */
        NULL,                                               /**< [In] known Uid, can be NULL; uint8_t[0-4]. */
        NULL,                                               /**< [In] number of valid bits of UidIn. MSB codes the valid bytes, LSB codes the valid bits.  */
        aUid,                                               /**< [Out] complete Uid; uint8_t[4]. */
        aNvbUid);                                           /**< [Out] number of valid bits of UidOut. MSB codes the valid bytes, LSB codes the valid bits.  */
    CHECK_SUCCESS(status);

    status = phalMful_Read(palMful, /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                      /**< [In] Address on Picc to read from. */
        aData);                     /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing data returned from the Picc. */
    CHECK_SUCCESS(status);

    printf("Demo to show that Read can skip ACL2\n");

    return PH_ERR_SUCCESS;
}

/**
* NFC counter and UID ASCII Mirror page Demo App
*
*/
uint8_t NfcCounterAndAsciiMirrorDemo(
                      phalMful_Sw_DataParams_t * palMful,             /**< Pointer to MIFARE Ultralight parameter structure. */
                      phpalMifare_Sw_DataParams_t * palMifare,        /**< Pointer to MIFARE parameter structure. */
                      phhalHw_Rd70x_DataParams_t * pHal,              /**< Pointer to HAL parameter structure */
                      phpalI14443p3a_Sw_DataParams_t * palI14443p3a)
{
    phStatus_t status;
    uint8_t bAddr = 0;
    uint8_t bCntNum = 2;
    uint8_t aData[PHAL_MFUL_READ_BLOCK_LENGTH] = {0};
    uint8_t aCnt[PHAL_MFUL_READ_BLOCK_LENGTH] = {0};

    printf("Use UC5-MS2 Type card \n\n");
    printf("Press Enter after placing the card... \n");
    _getch();


    Password_Auth(palMful);

    aData[0] = 0x10; /* Enable NFC counter */
    aData[1] = 0x00;
    aData[2] = 0x00;
    aData[3] = 0x00;
    bAddr = NFC_COUNTER_ADDRESS;
    status = phalMful_Write(palMful,    /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                          /**< [In] Address on Picc to write to. */
        aData);                         /**< [In] pData[#PHAL_MFUL_WRITE_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);

    aData[0] = 0x00; /* Set the UID byte address(UIDASCIIByte) and NFC Counter ASCII byte address(NFCCounterASCIIByte) to b0*/
    aData[1] = NFC_ASCII_MIRROR_PAGE; /* Set the NFC mirror page address(NFCCounterASCIIPage) to P20 */
    aData[2] = UID_ASCII_MIRROR_PAGE; /* Set the UID mirror page address(UIDASCIIPage) to P30 */
    aData[3] = 0x40; /* Set the first Auth data page (FirstProtectedPage) to P40 */
    bAddr = ASCII_MIRROR_ADDRESS;
    status = phalMful_Write(palMful,    /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                          /**< [In] Address on Picc to write to. */
        aData);                         /**< [In] pData[#PHAL_MFUL_WRITE_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);

    /* Perform Read NFC counter 02 */
    status = phalMful_ReadCnt(palMful,  /**< [In] Pointer to this layer's parameter structure. */
        bCntNum,                        /**< [In] One byte counter number 00 to 02 */
        aCnt);                          /**< [Out] pCntValue[3] counter value. LSB first. Three bytes counter value returned from PICC. */
    CHECK_SUCCESS(status);

    /* Read the NFC Counter ASCII mirror */
    bAddr = NFC_ASCII_MIRROR_PAGE;
    status = phalMful_Read(palMful,     /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                          /**< [In] Address on Picc to write to. */
        aData);                         /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);

    printf("The NFC counter ASCII mirror value is : \n");
    PRINT_BUFFER(aData, PHAL_MFUL_READ_BLOCK_LENGTH);

    /* Read the UID ASCII mirror */
    bAddr = UID_ASCII_MIRROR_PAGE;
    status = phalMful_Read(palMful,     /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                          /**< [In] Address on Picc to write to. */
        aData);                         /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);

    printf("The UID ASCII mirror value is : \n");
    PRINT_BUFFER(aData, PHAL_MFUL_READ_BLOCK_LENGTH);
    /**
    * reset card
    */
    status = resetCard(palMful, /**< Pointer to MIFARE Ultralight parameter structure. */
        palMifare,              /**< Pointer to MIFARE parameter structure. */
        pHal,                   /**< Pointer to HAL parameter structure */
        palI14443p3a);          /**< PAL-ISO14443P3A parameter structure */
    CHECK_SUCCESS(status);

    Password_Auth(palMful);

    /* Dummy read */
    bAddr = 0x0;
    status = phalMful_Read(palMful,     /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                          /**< [In] Address on Picc to write to. */
        aData);                         /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);

    /* Read the NFC Counter ASCII mirror */
    bAddr = NFC_ASCII_MIRROR_PAGE;
    status = phalMful_Read(palMful,     /**< [In] Pointer to this layer's parameter structure. */
        bAddr,                          /**< [In] Address on Picc to write to. */
        aData);                         /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);

    printf("The NFC counter ASCII mirror value is : \n");
    PRINT_BUFFER(aData, PHAL_MFUL_READ_BLOCK_LENGTH);

    return PH_ERR_SUCCESS;
}

int __cdecl main()
{
    phStatus_t status;
    uint8_t aHalBufferReaderTx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader transmit buffer */
    uint8_t aHalBufferReaderRx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader receive buffer */
    uint8_t aSak[1];                  /**< Select Acknowledge; uint8_t. */
    uint8_t aAtr[256];                /**< Atr */
    uint8_t bMoreCardsAvailable;      /**< more card available */
    uint8_t bLength;                  /**< Length */
    int iChoice;                      /**< choice */
    /**
    * Bfl data parameter storage
    */
    phbalReg_PcscWin_DataParams_t balPcsc;          /**< PCSC (Windows) BAL parameter structure */
    phhalHw_Rd710_DataParams_t halRd710;            /**< Rd710 HAL parameter structure */
    phpalI14443p3a_Sw_DataParams_t palI14443p3a;    /**< PAL-ISO14443P3A parameter structure */
    phpalI14443p4a_Sw_DataParams_t palI14443p4a;    /**< PAL-ISO14443P4A parameter structure */
    phpalI14443p4_Sw_DataParams_t palI14443p4;      /**< ISO14443-4 parameter structure */
    phpalMifare_Sw_DataParams_t palMifare;          /**< PAL-MIFARE parameter structure */
    phalMful_Sw_DataParams_t alMful;                /**< Pointer to MIFARE Ultralight parameter structure. */
    void * pHal = NULL;                             /**< Pointer to the parameter structure of the HAL layer. */
    uint8_t flag = 1;

    printf("\nNxpRdLib ANSI-C Example Program V1.0a\n\n");
    printf("Please ensure that a Pegoda reader is connected and in working condition.\n\n");
    printf("Performing startup...\n\n");

    /**
    * Init. PCSC BAL
    */
    status = phbalReg_PcscWin_Init(&balPcsc,    /**< [In] Pointer to this layer's parameter structure. */
        sizeof(phbalReg_PcscWin_DataParams_t),  /**< [In] Specifies the size of the data parameter structure. */
        aAtr,                                   /**< [In] Pointer to buffer used for storing the ATR after card activation. */
        256);                                   /**< [In] Size of the ATR buffer. */
    CHECK_SUCCESS(status);

    /**
    * set the BAL Communication Configuration values
    */
    status = phbalReg_SetConfig(&balPcsc,           /**< [In] Pointer to this layer's parameter structure. */
        PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL,          /**< BAL Communication Configs */
        PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_UNDEFINED);/**< BAL Communication Configuration values */
    CHECK_SUCCESS(status);
    /**
    * set the BAL Communication Configs
    */
    status = phbalReg_SetConfig(&balPcsc,       /**< [In] Pointer to this layer's parameter structure. */
        PHBAL_REG_PCSCWIN_CONFIG_SHARE,         /**< BAL Communication Configs */
        PHBAL_REG_PCSCWIN_VALUE_SHARE_DIRECT);  /**< BAL Communication Configuration values */
    CHECK_SUCCESS(status);

    /**
    * Select Port to be used (no SAM reader)
    */
    status = phbalReg_SetPort(&balPcsc,         /**< [In] Pointer to this layer's parameter structure. */
        (uint8_t*)PCSC_READER_P2_NO_SAM_NAME);  /**< [In] Port Name as String. */
    CHECK_SUCCESS(status);

    /**
    * Initialise the Rd710 HAL component:
    *   RD710 with GLOBAL_BUFFER_SIZE bytes send/receive buffer
    */
    status = phhalHw_Rd710_Init(&halRd710,  /**< [In] Pointer to this layer's parameter structure. */
        sizeof(phhalHw_Rd710_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
        &balPcsc,                           /**< [In] Pointer to the lower layers parameter structure */
        0,                                  /**< [In] Slot number */
        aHalBufferReaderTx,                 /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        sizeof(aHalBufferReaderTx),         /**< [In] Size of the global transmit buffer. */
        aHalBufferReaderRx,                 /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        sizeof(aHalBufferReaderRx));        /**< [In] Size of the global receive buffer. */
    CHECK_SUCCESS(status);

    pHal = &halRd710;  /**< Set HAL pointer */

    /**
    * Open the PCSC port
    */
    status = phbalReg_OpenPort(&balPcsc);  /**< [In] Pointer to this layer's parameter structure. */
    CHECK_SUCCESS(status);

    /**
    * Standard Rd710 Cmd Init.
    */
    status = (phStatus_t)phhalHw_Rd710_Cmd_InitReader(&halRd710);  /**< [In] Pointer to this layer's parameter structure. */
    CHECK_SUCCESS(status);

    /**
    * init the 14443-3A component
    */
    status = phpalI14443p3a_Sw_Init(&palI14443p3a,  /**< [In] PAL-ISO14443P3A parameter structure */
        sizeof(palI14443p3a),                       /**< [In] Specifies the size of the data parameter structure */
        pHal);                                      /**< [In] Pointer to the parameter structure of the underlying layer.*/
    CHECK_SUCCESS(status);

    /**
    * init the 14443-4A component
    */
    status = phpalI14443p4a_Sw_Init(&palI14443p4a,  /**< [In]  PAL-ISO14443P4A parameter structure */
        sizeof(palI14443p4a),                       /**< [In] Specifies the size of the data parameter structure */
        pHal);                                      /**< [In] Pointer to the parameter structure of the underlying layer.*/
    CHECK_SUCCESS(status);

    /**
    * init the 14443-4 component
    */
    status = phpalI14443p4_Sw_Init(&palI14443p4,    /**< [In] ISO14443-4 parameter structure */
        sizeof(palI14443p4),                        /**< [In] Specifies the size of the data parameter structure */
        pHal);                                      /**< [In] Pointer to the parameter structure of the underlying HAL layer.*/
    CHECK_SUCCESS(status);

    /**
    * init. MIFARE
    */
    status = phpalMifare_Sw_Init(&palMifare,    /**< [In] PAL-MIFARE parameter structure */
        sizeof(palMifare),                      /**< [In] Specifies the size of the data parameter structure */
        pHal,                                   /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
        &palI14443p4);                          /**< [In] Pointer to the parameter structure of the underlying ISO14443-4 layer. */
    CHECK_SUCCESS(status);

    /**
    * init. Ultralight component
    */
    status = phalMful_Sw_Init(&alMful,      /**< [In] Pointer to MIFARE Ultralight parameter structure. */
        sizeof(phalMful_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
        &palMifare,                         /**< [In] PAL-MIFARE parameter structure */
        NULL,                               /**< [In] Pointer to a key store structure */
        NULL,                               /**< [In] CryptoSym parameter structure for ENC */
        NULL);                              /**< [In] CryptoRng parameter structure for Rng */
    CHECK_SUCCESS(status);

    /* Initialize logging */
#ifdef NXPBUILD__PH_LOG
    /* Register the component for logging */
    status = phLog_Init(phalMful_log, logRegisterEntries, (uint16_t)(sizeof(logRegisterEntries) / sizeof(phLog_RegisterEntry_t)));
	CHECK_SUCCESS(status);
    status = phLog_Register(&alMful, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(status);
#endif /* NXPBUILD__PH_LOG */

    /**
    * Field Reset
    */
    status = phhalHw_FieldReset(pHal);  /**< [In] Pointer to the parameter structure of the HAL layer. */
    CHECK_SUCCESS(status);

    /**
    * Configure HAL for Type-A cards
    */
    status = phhalHw_ApplyProtocolSettings(pHal,    /**< [In] Pointer to this layer's parameter structure. */
        PHHAL_HW_CARDTYPE_ISO14443A);               /**< [In] Type of card for which the hal should be configured for. */
    CHECK_SUCCESS(status);

    printf("Please insert a UL_EV1 card for the examples!\n");

    /**
    * Activate Layer 3 card
    */
    status = phpalI14443p3a_ActivateCard(&palI14443p3a, /**< [In] PAL-ISO14443P3A parameter structure */
        NULL,                                           /**< [In] unknown UID */
        0x00,                                           /**< [In] UID length 0 */
        aUid,                                           /**< [Out] complete Uid; uint8_t[4/7/10]. */
        &bLength,                                       /**< [Out] Length of Uid; 4/7/10. */
        aSak,                                           /**< [Out] Select Acknowledge; uint8_t. */
        &bMoreCardsAvailable);                          /**< [Out] Whether there are more cards in the field or not; uint8_t. */
    CHECK_SUCCESS(status);

    printf("Perform the examples: \n");

    printf("---------------------------\n");
    while(flag)
    {
        printf("MIFARE Ultralight_EV1 Example\n");
        printf("[0] Reset Card \n");
        printf("[1] Card Authenticate \n");
        printf("[2] Read/Write Demo \n");
        printf("[3] OTP Demo \n");
        printf("[4] Block Lock Demo \n");
        printf("[5] Read Signature Demo \n");
        printf("[6] ACL2Skip Demo \n");
        printf("[7] Fast Read Demo \n");
        printf("[8] Comp write Demo \n");
        printf("[9] NFC Counter ASCII mirror Demo \n");
        printf("[x] Exit\n");
        printf("----------------\n");
        printf("Choice:");
        iChoice = _getch();
        printf(" %c", iChoice);
        printf("\n\n");
        printf("---------------------------\n");
        printf("\n\n");

        switch (iChoice)
        {
        case '0':
            printf("Resetting the card\n");
            resetCard(&alMful,   /**< MIFARE Ultralight parameter structure. */
                &palMifare,      /**< PAL-MIFARE parameter structure */
                pHal,            /**< Pointer to the parameter structure of the HAL layer. */
                &palI14443p3a);  /**< PAL-ISO14443P3A parameter structure */
            break;

        case '1':
            printf("Authenticating the card\n");
            Password_Auth(&alMful); /**< MIFARE Ultralight parameter structure. */
            break;

        case '2':
            printf("Read/Write Demo \n");
            ReadWriteDemo(&alMful); /**< MIFARE Ultralight parameter structure. */
            break;

        case '3':
            printf("OTP Demo \n");
            OtpDemo(&alMful);   /**< MIFARE Ultralight parameter structure. */
            break;

        case '4':
            printf("Block Lock Demo \n");
            BlockLockDemo(&alMful,  /**< MIFARE Ultralight parameter structure. */
                &palMifare,         /**< PAL-MIFARE parameter structure */
                pHal,               /**< Pointer to the parameter structure of the HAL layer. */
                &palI14443p3a);     /**< PAL-ISO14443P3A parameter structure */
            break;

        case '5':
            printf("Read Signature Demo \n");
            ReadSignatureDemo(&alMful);  /**< MIFARE Ultralight parameter structure. */
            break;

        case '6':
            printf("Short cut to Read Demo (Skipping ACL2)\n\n");
            ShortReadDemo(
                &alMful,         /**< MIFARE Ultralight parameter structure. */
                &palMifare,      /**< PAL-MIFARE parameter structure */
                pHal,            /**< Pointer to the parameter structure of the HAL layer. */
                &palI14443p3a);  /**< PAL-ISO14443P3A parameter structure */
            break;

        case '7':
            printf("Fast Read Demo \n\n");
            FastReadDemo(&alMful);  /**< MIFARE Ultralight parameter structure. */
            break;

        case '8':
            printf("Comp Read Demo \n\n");
            CompWriteDemo(&alMful); /**< MIFARE Ultralight parameter structure. */
            break;

        case '9':
            printf("NFC Counter and UID ASCII Mirror Page Demo \n\n");
            NfcCounterAndAsciiMirrorDemo(
                &alMful,         /**< MIFARE Ultralight parameter structure. */
                &palMifare,      /**< PAL-MIFARE parameter structure */
                pHal,            /**< Pointer to the parameter structure of the HAL layer. */
                &palI14443p3a);  /**< PAL-ISO14443P3A parameter structure */
            break;

        default:
            flag = 0;
            break;
        }
    }
    return 0;
}