/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for Rd710/Rd852 in X-Mode with Mifare Plus.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/**
 * Header for this file
 */
#include "Example-Rd710_Rd852_SamX_MfPlus.h"

/**
 * Reader Library Headers
 */
#include <phbalReg.h>
#include <phhalHw.h>
#include <phpalI14443p3a.h>
#include <phpalI14443p4a.h>
#include <phpalI14443p4.h>
#include <phCidManager.h>
#include <phpalMifare.h>
#include <phalMful.h>
#include <phKeyStore.h>
#include <phCryptoRng.h>
#include <phCryptoSym.h>
#include <phalMfp.h>
#include <phalMfc.h>
#include <phalVca.h>
#include <phhalHw_Rd710_Cmd.h>

/**
 * Block Addresses
 */
#define CARD_MASTER_KEY               0x9000U /**< Card Master Key Address */
#define CARD_CONFIGURATIOM_KEY        0x9001U /**< Card Configuration Key Address */
#define LEVEL_2_SWITCH_KEY            0x9002U /**< Level 2 Switch Key Address */
#define LEVEL_3_SWITCH_KEY            0x9003U /**< Level 3 Switch Key Address */
#define SL_1_CARD_AUTHENTICATION_KEY  0x9004U /**< SL1 Card Authentication Key Address */
#define SELECT_VC_KEY                 0xA000U /**< Select VC Key Address */
#define PROXIMITY_CHECK_KEY           0xA001U /**< Proximity Check Key Address */
#define VC_POLLING_ENC_KEY            0xA080U /**< VC Polling ENC Key Address */
#define VC_POLLING_MAC_KEY            0xA081U /**< VC Polling MAC Key Address */
#define MFP_CONFIGURATION_BLOCK       0xB000U /**< MIFARE Plus Configuration block Address */
#define INSTALLATION_IDENTIFIER       0xB001U /**< Installation Identifier Address */
#define FIELD_CONFIGURATION_BLOCK     0xB003U /**< Field Configuration block Address */
#define PHYSICAL_AES_START_ADDRESS    0x4000U /**< physical start address of AES key location Address */

/**
 * sectors
 */
#define TWO_AES_KEYS              0x02U  /**< Two AES keys */
#define AES_FIRST_SECTOR          0x00U  /**< first AES sector */
#define AES_EXAMPLE_SL2_SECTOR    0x00U  /**< example AES sector in SL2 */
#define AES_EXAMPLE_SL3_SECTOR    0x01U  /**< example AES sector in SL3 */
#define LOWER_SECTOR_SIZE         0x04U  /**< EEProm memory: the lower 2kB contain 32 sectors with 4 blocks*/

/**
 * Key store
 */
#define KEYCOUNT                 0xFFU /**< number of keys */
#define KEYVERSIONS              0x01U /**< number of key versions */
#define SL_1_KEY_ADDRESS         0x01U /**< SL1 key address in keystore */
#define SL_1_KEY_VERSION         0x00U /**< SL1 key version in keystore */
#define MFC_KEY_ADDRESS          0x02U /**< MIFARE classic key address in keystore */
#define MFC_KEY_VERSION          0x00U /**< MIFARE classic key version in keystore */
#define SL_2_KEY_ADDRESS         0x03U /**< SL2 key address in keystore */
#define SL_2_KEY_VERSION         0x00U /**< SL2 key version in keystore */
#define SL_2_AUTH_KEY_ADDRESS    0x04U /**< SL2 AUTH key address in keystore */
#define SL_2_AUTH_KEY_VERSION    0x00U /**< SL2 AUTH key version in keystore */
#define SL_2_MFP_KEY_ADDRESS     0x05U /**< SL2 MFP key address in keystore */
#define SL_2_MFP_KEY_VERSION     0x00U /**< SL2 MFP key version in keystore */
#define SL_3_KEY_ADDRESS         0x06U /**< SL3 key address in keystore */
#define SL_3_KEY_VERSION         0x00U /**< SL3 key version in keystore */
#define SL_3_AUTH_KEY_ADDRESS    0x07U /**< SL3 AUTH key address in keystore */
#define SL_3_AUTH_KEY_VERSION    0x00U /**< SL3 AUTH key version in keystore */
#define MASTER_KEY_ADDRESS       0x08U /**< Master key address in keystore */
#define MASTER_KEY_VERSION       0x00U /**< Master key version in keystore */
#define SL_3_AES_KEY_ADDRESS     0x09U /**< SL3 AES key address in keystore */
#define SL_3_AES_KEY_VERSION     0x00U /**< SL3 AES key version in keystore */
#define SL_3_VC_CONFIG_KEY_ADDRESS  0x0AU /**< SL3 VC Config key address in keystore */
#define SL_3_VC_CONFIG_KEY_VERSION  0x00U /**< SL3 VC Config key version in keystore */
#define VC_POLLING_ENC_KEY_ADDRESS  0x0BU /**< VC Polling ENC key address in keystore */
#define VC_POLLING_ENC_KEY_VERSION  0x00U /**< VC Polling ENC key version in keystore */
#define VC_POLLING_MAC_KEY_ADDRESS  0x0CU /**< VC Polling MAC key address in keystore */
#define VC_POLLING_MAC_KEY_VERSION  0x00U /**< VC Polling MAC key version in keystore */
#define VC_SELECT_KEY_ADDRESS    0x0DU /**< VC Select key address in keystore */
#define VC_SELECT_KEY_VERSION    0x00U /**< VC Select key version in keystore */
#define KEY_POS                  0x00U /**< Key position */
#define MIFARE_KEY_SIZE          0x06U /**< MIFARE Key size */
#define MIFARE_KEY_A_B_SIZE      0x0CU /**< MIFARE size of Key A and Key B */

#define ONE_VALUE_BLOCK_SECTOR_TRAILER  0x13U   /**< one block sector trailer */
#define ONE_VALUE_BLOCK         0x10U  /**< one block as a value block */
#define BLOCK_SIZE              0x10U  /**< block size */
#define SL_1_EXAMPLE_BLOCK      0x0CU  /**< SL1 example block */
#define SL_1_EXAMPLE_SECTOR_TRAILER  0x0FU /**< SL1 example sector trailer */

#define IID_KEY_COUNT      0x13U /**< number of IID keys */

#define SAM_KEY                  0x00U /**< SAM Key address */

#define SAM_AV1                  0x01U  /**< SAM AV1 */
#define SAM_AV2                  0x02U  /**< SAM AV2 */

#define SAM_LC0         0x00U /**< SAM logical channel 0 */
#define SAM_LC1         0x01U /**< SAM logical channel 1 */

#define PLAIN       0x00U    /**< plain mode */
#define MAC         0x01U    /**< CMAC mode */
#define FULL        0x02U    /**< full mode */

/**
 * ISO
 */
#define ISO14443_3    0x00U  /**< ISO14443-3 protocol */
#define ISO14443_4    0x01U  /**< ISO14443-4 protocol */

/**
 * flags
 */
#define ENCRYPTED_OFF     0x00U    /**< Plain */
#define ENCRYPTED_ON      0x01U    /**< Encrypted */
#define MAC_RESPONSE_OFF  0x00U    /**< No MAC on response */
#define MAC_RESPONSE_ON   0x01U    /**< MAC on response */
#define MAC_COMMAND_OFF   0x00U    /**< No MAC on command */
#define MAC_COMMAND_ON    0x01U    /**< MAC on command */
#define FOLLOWING_AUTH    0x00U    /**< Following Authentication */
#define FIRST_AUTH        0x01U    /**< First Authentication */

/**
  * EEProm memory size
  */
#define EEPROM2KB      0x00U    /**< 2kB EEProm memory size */
#define EEPROM4KB      0x01U    /**< 4kB EEProm memory size */

/**
 * MIFARE Plus SL3
 */
#define PHYSICAL_AES_START_ADDRESS    0x4000U /**< physical start address of AES key location Address */

/**
 * Pegoda 2 configurations
 */
#define PCSC_READER_P2_X_NAME       "NXP Pegoda X 0 0"        /**< Pegoda 2: SAM in X-Mode */

/**
 * EEProm memory  size set to 4kB
 */
static const uint8_t gbEEPROMSize = EEPROM4KB;

static  uint8_t gaVCMasterKey[] =       /**< Virtual Card Master Key */
  { 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x01};
static uint8_t gaVCConfigKey[] =        /**< Virtual Card Configuration Key */
  { 0x02, 0x00, 0x02, 0x00, 0x02, 0x00, 0x02, 0x00, 0x02, 0x00, 0x02, 0x00, 0x02, 0x00, 0x02, 0x02};
static uint8_t gaLevel2SwitchKey[] =    /**< Key to switch in Security Level 2 */
  { 0x03, 0x00, 0x03, 0x00, 0x03, 0x00, 0x03, 0x00, 0x03, 0x00, 0x03, 0x00, 0x03, 0x00, 0x03, 0x03};
static uint8_t gaLevel3SwitchKey[] =    /**< Key to switch in Security Level 3 */
  { 0x04, 0x00, 0x04, 0x00, 0x04, 0x00, 0x04, 0x00, 0x04, 0x00, 0x04, 0x00, 0x04, 0x00, 0x04, 0x04};
static  uint8_t gaSelectVCKey[] =       /**< Select Virtual Card Key */
  { 0x05, 0x00, 0x05, 0x00, 0x05, 0x00, 0x05, 0x00, 0x05, 0x00, 0x05, 0x00, 0x05, 0x00, 0x05, 0x05};
static uint8_t gaVCPollingENCKey[] =    /**< Virtual Card Polling ENC Key */
  { 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x00, 0x06, 0x06};
static uint8_t gaVCPollingMACKey[] =    /**< Virtual Card Polling MAC Key */
  { 0x07, 0x00, 0x07, 0x00, 0x07, 0x00, 0x07, 0x00, 0x07, 0x00, 0x07, 0x00, 0x07, 0x00, 0x07, 0x07};
static uint8_t gaVCProximityKey[] =     /**< Virtual Card Proximity Key */
  { 0x08, 0x00, 0x08, 0x00, 0x08, 0x00, 0x08, 0x00, 0x08, 0x00, 0x08, 0x00, 0x08, 0x00, 0x08, 0x08};
static uint8_t gaSecurityLevel1AESKey[] =/**< Key to switch in Security Level 1 */
  { 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09};
/**
 * MFC Used Sector Trailer in SL1 and SL2
 * Byte 0-5: Key A: {0x01, 0x00, 0x00, 0x00, 0xFE}
 * Byte 6-9: Access Bits:
 * read Key A|B, write key A|B, increment key A|B, decrement/transfer/restore key A|B
 * Access Bytes for value block 0:  (C1_0,C2_0,C3_0) = (0,0,0)
 * Access Bytes for value block 1:  (C1_1,C2_1,C3_1) = (0,0,0)
 * Access Bytes for value block 2:  (C1_2,C2_2,C3_2) = (0,0,0)
 * Access Bytes for sector trailer: transport configuration (C1_3,C2_3,C3_3) = (0,0,1)
 * Byte 6: [(!C2_3), (!C2_2), (!C2_1), (!C2_0), (!C1_3), (!C1_2), (!C1_1), (!C1_0)]
 *         [      1,       1,       1,       1,       1,       1,       1,      1 ] = 0xFFU
 * Byte 7: [(C1_3), (C1_2), (C1_1), (C1_0), (!C3_3), (!C3_2), (!C3_1), (!C3_0)]
 *         [     0,      0,      0,      0,       0,       1,       1,      1 ] = 0x07U
 * Byte 8: [(C3_3), (C3_2), (C3_1), (C3_0), (C2_3), (C2_2), (C2_1), (C2_0)]
 *         [     1,      0,      0,      0,      0,      0,      0,      0] = 0x80U
 * Byte 10-15: Key B: {0x00, 0x00, 0x00, 0x00, 0x00, 0x00}
 */
static uint8_t gaMFCUsedSectorTrailer[] =/**< MFC Sector Trailer with a special Key A: 0x01, 0x00, 0x00, 0x00, 0xFE */
  { 0x01, 0x00, 0x00, 0x00, 0xFE, 0xFF, 0xFF, 0x07, 0x80, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
static uint8_t gaEmptyBlock[] =          /**< empty block */
  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
/**
 * MFC Sector Trailer
 * Byte 0-5: Key A: {0x00, 0xFF, 0xFF, 0xFF, 0xF0}
 * In SL3 Byte 5 is used for plain communication
 * Byte 5: [1,1,1,1,0,0,0,0] = 0xF0U: reading in plain is impossible in the Blocks 0-3
 * Byte 6-9: Access Bits:
 * read Key A|B, write key A|B, increment key A|B, decrement/transfer/restore key A|B
 * Access Bytes for value block 0:  (C1_0,C2_0,C3_0) = (0,0,0)
 * Access Bytes for value block 1:  (C1_1,C2_1,C3_1) = (0,0,0)
 * Access Bytes for value block 2:  (C1_2,C2_2,C3_2) = (0,0,0)
 * Access Bytes for sector trailer: transport configuration (C1_3,C2_3,C3_3) = (0,0,1)
 * Byte 6: [(!C2_3), (!C2_2), (!C2_1), (!C2_0), (!C1_3), (!C1_2), (!C1_1), (!C1_0)]
 *         [      1,       1,       1,       1,       1,       1,       1,      1 ] = 0xFFU
 * Byte 7: [(C1_3), (C1_2), (C1_1), (C1_0), (!C3_3), (!C3_2), (!C3_1), (!C3_0)]
 *         [     0,      0,      0,      0,       0,       1,       1,      1 ] = 0x07U
 * Byte 8: [(C3_3), (C3_2), (C3_1), (C3_0), (C2_3), (C2_2), (C2_1), (C2_0)]
 *         [     1,      0,      0,      0,      0,      0,      0,      0] = 0x80U
 * Byte 10-15: Key B: {0x01, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF}
 */
static uint8_t gaMFCSectorTrailer[] =  /**< standard MFC Sector Trailer */
    { 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xF0, 0xFF, 0x07, 0x80, 0x69, 0x01, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};
static uint8_t gaVCIID[] =             /**< Virtual Card Installation Identifier */
  { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F};
static uint8_t gaKey[] =               /**< standard key block */
  { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};
static uint8_t gaReceivedBlock[] =     /**< receive block */
  {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

phLog_RegisterEntry_t logRegisterEntries[5];
phLog_LogEntry_t logEntries[32];

/**
 * calculate the coresponding AES key Address
 * Key A even and Key B odd at the Physical AES start address 0x4000U
 */
#define AES_SECTOR_KEYS_A(sectorNr) (((sectorNr)*2) + PHYSICAL_AES_START_ADDRESS)
#define AES_SECTOR_KEYS_B(sectorNr) ((((sectorNr)*2) + 1) + PHYSICAL_AES_START_ADDRESS)

#define CHECK_SUCCESS(x)                                                \
if ((x) != PH_ERR_SUCCESS)                                              \
{                                                                       \
    printf("An error occurred (0x%04X), press any key to exit...", (x)); \
    _getch();                                              \
    return 0;                                                           \
}

/**
 * Mfp Value File Operations in Security Level 3
 */
phStatus_t Mfp_VFOperations(phalMfp_SamAV2_X_DataParams_t * palMfp,  /**< [In] MIFARE Plus Software parameter structure */
              uint8_t bEnc,            /**< [In] \c 0: Plain communication; \c 1: Encrypted communication; */
              uint8_t bResponseMACed,  /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
              uint8_t bMACOnCmd,       /**< [In] \c 0: No MAC on command; \c 1: MAC on command; */
              uint8_t bBlockNumber)    /**< [In] MIFARE block number. */
{
  phStatus_t status;
  uint8_t aValue[] = { 0x00, 0x00, 0x00, 0x01};  /**< Value 0x00000001 */
  uint8_t bAddrData = 0;  /**< address of the data */

  /**
   * Mfp write
   */
  status = phalMfp_Write(palMfp,          /**< [In] MIFARE Plus Software parameter structure */
                         bEnc,            /**< [In] \c 0: Plain communication; \c 1: Encrypted communication; */
                         bResponseMACed,  /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                         bBlockNumber,    /**< [In] MIFARE block number. */
                         0x01,            /**< [In] Number of blocks to write (must not be more than 3). */
                         gaEmptyBlock);   /**< [In] Block(s) (16*bNumBlocks bytes).  */
  CHECK_SUCCESS(status);

  /**
   * Mfp write value
   */
  status = phalMfp_WriteValue(palMfp,          /**< [In] MIFARE Plus Software parameter structure */
                              bEnc,            /**< [In] \c 0: Plain communication; \c 1: Encrypted communication; */
                              bResponseMACed,  /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                              bBlockNumber,    /**< [In] MIFARE block number. */
                              aValue,          /**< [In] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                              bBlockNumber);   /**< [In] bAddrData containing address written to the MIFARE(R) card value block. */
  CHECK_SUCCESS(status);

  /**
   * Mfp read
   */
  status = phalMfp_Read(palMfp,           /**< [In] MIFARE Plus Software parameter structure */
                        bEnc,             /**< [In] \c 0: Plain communication; \c 1: Encrypted communication; */
                        bResponseMACed,   /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                        bMACOnCmd,        /**< [In] \c 0: No MAC on command; \c 1: MAC on command; */
                        bBlockNumber,     /**< [In] MIFARE block number. */
                        0x01,             /**< [In] Number of blocks to read. */
                        gaReceivedBlock); /**< [Out] Block(s) (16*bNumBlocks bytes).  */
  CHECK_SUCCESS(status);

  /**
   * Mfp increment
   */
  status = phalMfp_Increment(palMfp,          /**< [In] MIFARE Plus Software parameter structure */
                             bResponseMACed,  /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                             bBlockNumber,    /**< [In] MIFARE block number. */
                             aValue);         /**< [In] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
  CHECK_SUCCESS(status);

  /**
   * Mfp transfer
   */
  status = phalMfp_Transfer(palMfp,          /**< [In] MIFARE Plus Software parameter structure */
                            bResponseMACed,  /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                            bBlockNumber);   /**< [In] MIFARE block number. */
  CHECK_SUCCESS(status);

  /**
   * Mfp increment transfer
   */
  status = phalMfp_IncrementTransfer(palMfp,          /**< [In] MIFARE Plus Software parameter structure */
                                     bResponseMACed,  /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                                     bBlockNumber,    /**< [In] MIFARE Source block number. */
                                     bBlockNumber,    /**< [In] MIFARE Destination block number. */
                                     aValue);         /**< [In] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
  CHECK_SUCCESS(status);

  /**
   * Mfp read value
   */
  status = phalMfp_ReadValue(palMfp,          /**< [In] MIFARE Plus Software parameter structure */
                             bEnc,            /**< [In] \c 0: Plain communication; \c 1: Encrypted communication; */
                             bResponseMACed,  /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                             bMACOnCmd,       /**< [In] \c 0: No MAC on command; \c 1: MAC on command; */
                             bBlockNumber,    /**< [In] MIFARE block number. */
                             aValue,          /**< [Out] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                             &bAddrData);     /**< [Out] bAddrData containing address read from the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * change value for decrement
   */
  aValue[3] = 0x01;

  /**
   * Mfp decrement transfer
   */
  status = phalMfp_DecrementTransfer(palMfp,         /**< [In] MIFARE Plus Software parameter structure */
                                     bResponseMACed, /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                                     bBlockNumber,   /**< [In] Source block number. */
                                     bBlockNumber,   /**< [In] Destination block number. */
                                     aValue);        /**< [In] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
  CHECK_SUCCESS(status);

  /**
   * Mfp decrement
   */
  status = phalMfp_Decrement(palMfp,             /**< [In] MIFARE Plus Software parameter structure */
                             bResponseMACed,     /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                             bBlockNumber,       /**< [In] MIFARE Source block number. */
                             aValue);            /**< [In] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
  CHECK_SUCCESS(status);

  /**
   * Mfp restore
   */
  status = phalMfp_Restore(palMfp,            /**< [In] MIFARE Plus Software parameter structure */
                           bResponseMACed,    /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                           bBlockNumber);     /**< [In] MIFARE Source block number. */
  CHECK_SUCCESS(status);

  /**
   * Mfp transfer
   */
  status = phalMfp_Transfer(palMfp,               /**< [In] MIFARE Plus Software parameter structure */
                            bResponseMACed,       /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                            bBlockNumber);        /**< [In] MIFARE Destination block number. */
  CHECK_SUCCESS(status);

  /**
   * Mfp Read
   */
  status = phalMfp_Read(palMfp,               /**< [In] MIFARE Plus Software parameter structure */
                        bEnc,                 /**< [In] \c 0: Plain communication; \c 1: Encrypted communication; */
                        bResponseMACed,       /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                        bMACOnCmd,            /**< [In] \c 0: No MAC on command; \c 1: MAC on command; */
                        bBlockNumber,         /**< [In] MIFARE block number. */
                        0x01,                 /**< [In] Number of blocks to read. */
                        gaEmptyBlock);        /**< [Out] Block(s) (16*bNumBlocks bytes).  */
  CHECK_SUCCESS(status);

  return PH_ERR_SUCCESS;
}

/**
 * Activate Layer 3 card, Activate card I14443p4a, switch to ISO4, get and Set the ISO14443-4 protocol parameters
 */
uint8_t ActivateCard(phpalI14443p3a_SamAV2_X_DataParams_t * palI14443p3a, /**< [In] PAL-ISO14443P3A parameter structure */
                     phpalI14443p4a_SamAV2_X_DataParams_t * palI14443p4a, /**< [In] PAL-ISO14443P4A parameter structure */
                     uint8_t bDri,  /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3 */
                     uint8_t bDsi)  /**< [In] Divisor Send (PICC to PCD) Integer; 0-3 */
{
  phStatus_t status;
  uint8_t aSak[1];      /**< Select Acknowledge; uint8_t. */
  uint8_t aUid[10];     /**< complete Uid; uint8_t[4/7/10]. */
  uint8_t bMoreCardsAvailable; /**< more card available */
  uint8_t bLength;      /**< Length */
  uint8_t aAts[256];    /**< Answer to Select; uint8_t[1-254] */
  uint8_t bCidEnabled;  /**< Unequal '0' if Card Identifier is enabled. */
  uint8_t bCid = 0;     /**< Card Identifier. */
  uint8_t bNadSupported;/**< Node Address Support; Unequal '0' if supported. */
  uint8_t bFwi;         /**< Frame Waiting Integer. */
  uint8_t bFsdi = 5;    /**< PCD Frame Size Integer; 0-8 */
  uint8_t bFsci;        /**< PICC Frame Size Integer; 0-8. */

  /**
   * Activate Layer 3 card
   */
  status = phpalI14443p3a_ActivateCard(palI14443p3a,  /**< [In] PAL-ISO14443P3A parameter structure */
                                       NULL,          /**< [In] unknown UID */
                                       0x00,          /**< [In] UID length 0 */
                                       aUid,          /**< [Out] complete Uid; uint8_t[7]. */
                                       &bLength,      /**< [Out] Length of Uid; 4/7/10. */
                                       aSak,          /**< [Out] Select Acknowledge; uint8_t. */
                                       &bMoreCardsAvailable);  /**< [Out] Whether there are more cards in the field or not; uint8_t. */
  CHECK_SUCCESS(status);

  /**
   * Activate card I14443p4a, switch to ISO4
   */
  status = phpalI14443p4a_ActivateCard(palI14443p4a,  /**< [In] PAL-ISO14443P4A parameter structure */
                                       bFsdi,         /**< [In] Frame Size Integer; 0-8 */
                                       bCid,          /**< [In] Card Identifier; 0-14 */
                                       bDri,          /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3 */
                                       bDsi,          /**< [In] Divisor Send (PICC to PCD) Integer; 0-3 */
                                       aAts);         /**< [Out] Answer to Select; uint8_t[1-254] */
  CHECK_SUCCESS(status);

  /**
   * Retrieve the ISO14443-4A protocol parameters
   */
  status = phpalI14443p4a_GetProtocolParams(palI14443p4a,    /**< [In]  PAL-ISO14443P4A parameter structure */
                                            &bCidEnabled,    /**< [Out] Unequal '0' if Card Identifier is enabled. */
                                            &bCid,           /**< [Out] Card Identifier. */
                                            &bNadSupported,  /**< [Out] Node Address Support; Unequal '0' if supported. */
                                            &bFwi,           /**< [Out] Frame Waiting Integer. */
                                            &bFsdi,          /**< [Out] PCD Frame Size Integer; 0-8. */
                                            &bFsci);         /**< [Out] PICC Frame Size Integer; 0-8. */
  CHECK_SUCCESS(status);

  return PH_ERR_SUCCESS;
  }

/**
 * 1. Personalization Security Level 0
 */
uint8_t Personalization(phalMfp_SamAV2_X_DataParams_t * palMfp)  /**< [In] Pointer to this layer's parameter structure. */
{
  phStatus_t status;
  uint16_t wBlockNr = 0;  /**< Block number */
  uint16_t wSectorNr = 0; /**< Sector number */

  /**
   * standard key block initialized with 47
   * byte 0-3  = Value:  { 0x2F, 0x00, 0x00, 0x00}
   * byte 4-7  = !Value: { 0xD0, 0xFF, 0xFF, 0xFF}
   * byte 8-11 = Value:  { 0x2F, 0x00, 0x00, 0x00}
   * byte 12: Address 0x00; byte 13: !Address 0xFF; byte 14: Address 0x00; byte 15: !Address 0xFF
   */
  uint8_t aOneValueBlock[] =
    { 0x2F, 0x00, 0x00, 0x00, 0xD0, 0xFF, 0xFF, 0xFF, 0x2F, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00, 0xFF};

  /**
   * MFP Configuration Block: Default Mac on read and impossible read plain
   * byte 0: 0x00 MAC on Read Mandatory, this is default value
   * byte 1: Read Plain impossible for all blocks: 0xF0 = 11110000
   * bit 7: !Block3; bit 6: !Block2; bit 5: !Block1; bit 4: !Block0
   * bit 3: Block3; bit 2: Block2; bit 1: Block1; bit 0: Block0
   * byte 2-15: RFU
   */
  uint8_t aMfpConfigBlockMACOn[] =
    { 0x00, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

  uint8_t aSectorTrailerOneValueBlock[BLOCK_SIZE];  /**< Sector Trailer */

  /**
   * a. configure all blocks with different Crypto1 KeyA and KeyB
   * 32 sectors with 4 blocks = 128 blocks
   * each sector has a sector trailer in the last block
   */
  for (wBlockNr = 3; wBlockNr < 128; wBlockNr+=4)
  {
    /**
     * sector trailer
     */
    status = phalMfp_WritePerso(palMfp,     /**< [In] Pointer to this layer's parameter structure. */
                                ISO14443_4, /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                                wBlockNr,   /**< [In] MIFARE Block number. */
                                gaMFCSectorTrailer);  /**< [In] Value (16 bytes). */

    CHECK_SUCCESS(status);
  }

  /**
   * if EEProm memory size is 4kB, than the upper 2kB contain 8 sectors with 16 blocks = 128
   */
  for ( ; (gbEEPROMSize == EEPROM4KB) && ((wBlockNr < 256) && (wBlockNr >= 128)); wBlockNr+=16)
  {
    /**
     * sector trailer
     */
    status = phalMfp_WritePerso(palMfp,     /**< [In] Pointer to this layer's parameter structure. */
                                ISO14443_4, /**< [In]  \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                                wBlockNr,   /**< [In] MIFARE Block number. */
                                gaMFCSectorTrailer); /**< [In] Value (16 bytes). */
    CHECK_SUCCESS(status);
  }

  /**
   * b. configure one block as a value block
   * copy standard Sector Trailer
   */
  memcpy(aSectorTrailerOneValueBlock, gaMFCSectorTrailer, sizeof(gaMFCSectorTrailer));
  /**
   * Set access bytes in the Sector Trailer for a value block: Example blocks 0,1,2 = ONE_VALUE_BLOCK
   * Access Bytes for value block 0:  (C1_0,C2_0,C3_0) = (1,1,0)
   * Access Bytes for value block 1:  (C1_1,C2_1,C3_1) = (1,1,0)
   * Access Bytes for value block 2:  (C1_2,C2_2,C3_2) = (1,1,0)
   * Access Bytes for sector trailer: (C1_3,C2_3,C3_3) = (1,0,0)
   * read Key A|B, write key B, increment key B, decrement/transfer/restore key A|B
   * Byte 6: [(!C2_3), (!C2_2), (!C2_1), (!C2_0), (!C1_3), (!C1_2), (!C1_1), (!C1_0)]
   *         [      1,       0,       0,       0,      0,       0,       0,      0 ] = 0x80U
   */
  aSectorTrailerOneValueBlock[0x06] = 0x80;
  /**
   * Byte 7: [(C1_3), (C1_2), (C1_1), (C1_0), (!C3_3), (!C3_2), (!C3_1), (!C3_0)]
   *         [     1,      1,     1,      1,       1,       1,       1,      1 ] = 0xFFU
   */
  aSectorTrailerOneValueBlock[0x07] = 0xFF;
  /**
   * Byte 8: [(C3_3), (C3_2), (C3_1), (C3_0), (C2_3), (C2_2), (C2_1), (C2_0)]
   *         [     0,      0,      0,      0,      0,      1,      1,      1] = 0x07U
   */
  aSectorTrailerOneValueBlock[0x08] = 0x07;
  /**
   * write perso
   */
  status = phalMfp_WritePerso(palMfp,             /**< [In] Pointer to this layer's parameter structure. */
                              ISO14443_4,         /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                              ONE_VALUE_BLOCK_SECTOR_TRAILER,    /**< [In] MIFARE Block number. */
                              aSectorTrailerOneValueBlock);/**< [In] Value (16 bytes). */
  CHECK_SUCCESS(status);

  /*
   * Write AES keys
   * a. configure 2 blocks with identical AES KeyA and KeyB and identical access rights
   */
  for (wSectorNr = 0x00; wSectorNr < TWO_AES_KEYS; wSectorNr++)
  {
    /**
     * Write Key A
     */
    status = phalMfp_WritePerso(palMfp,     /**< [In] Pointer to this layer's parameter structure. */
                                ISO14443_4, /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                                AES_SECTOR_KEYS_A(wSectorNr),   /**< [In] MIFARE Block number. */
                                gaKey);     /**< [In] Value (16 bytes). */
    CHECK_SUCCESS(status);

    /**
     * Write Key B
     */
    status = phalMfp_WritePerso(palMfp,     /**< [In] Pointer to this layer's parameter structure. */
                                ISO14443_4, /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                                AES_SECTOR_KEYS_B(wSectorNr),   /**< [In] MIFARE Block number. */
                                gaKey);     /**< [In] Value (16 bytes). */
    CHECK_SUCCESS(status);
  }

  /**
   * Write Card Master
   */
  status = phalMfp_WritePerso(palMfp,         /**< [In] Pointer to this layer's parameter structure. */
                              ISO14443_4,     /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                              CARD_MASTER_KEY,/**< [In] MIFARE Block number. */
                              gaVCMasterKey); /**< [In] Value (16 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Write Card Configuration
   */
  status = phalMfp_WritePerso(palMfp,           /**< [In] Pointer to this layer's parameter structure. */
                              ISO14443_4,       /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                              CARD_CONFIGURATIOM_KEY,    /**< [In] MIFARE Block number. */
                              gaVCConfigKey);   /**< [In] Value (16 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Write Level 2 Switch
   */
  status = phalMfp_WritePerso(palMfp,            /**< [In] Pointer to this layer's parameter structure. */
                              ISO14443_4,        /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                              LEVEL_2_SWITCH_KEY,/**< [In] MIFARE Block number. */
                              gaLevel2SwitchKey);/**< [In] Value (16 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Write Level 3 Switch
   */
  status = phalMfp_WritePerso(palMfp,             /**< [In] Pointer to this layer's parameter structure. */
                              ISO14443_4,         /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                              LEVEL_3_SWITCH_KEY, /**< [In] MIFARE Block number. */
                              gaLevel3SwitchKey); /**< [In] Value (16 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Write SL1 Card Authentication
   */
  status = phalMfp_WritePerso(palMfp,             /**< [In] Pointer to this layer's parameter structure. */
                              ISO14443_4,         /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                              SL_1_CARD_AUTHENTICATION_KEY,  /**< [In] MIFARE Block number. */
                              gaSecurityLevel1AESKey);  /**< [In] Value (16 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Write Select VC
   */
  status = phalMfp_WritePerso(palMfp,          /**< [In] Pointer to this layer's parameter structure. */
                              ISO14443_4,      /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                              SELECT_VC_KEY,   /**< [In] MIFARE Block number. */
                              gaSelectVCKey);  /**< [In] Value (16 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Write VC Polling ENC
   */
  status = phalMfp_WritePerso(palMfp,              /**< [In] Pointer to this layer's parameter structure. */
                              ISO14443_4,          /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                              VC_POLLING_ENC_KEY,  /**< [In] MIFARE Block number. */
                              gaVCPollingENCKey);  /**< [In] Value (16 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Write VC Polling MAC
   */
  status = phalMfp_WritePerso(palMfp,              /**< [In] Pointer to this layer's parameter structure. */
                              ISO14443_4,          /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                              VC_POLLING_MAC_KEY,  /**< [In] MIFARE Block number. */
                              gaVCPollingMACKey);  /**< [In] Value (16 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Write Proximity Check Keys
   */
  status = phalMfp_WritePerso(palMfp,               /**< [In] Pointer to this layer's parameter structure. */
                              ISO14443_4,           /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                              PROXIMITY_CHECK_KEY,  /**< [In] MIFARE Block number. */
                              gaVCProximityKey);    /**< [In] Value (16 bytes). */
  CHECK_SUCCESS(status);

  /**
   * write MFP configuration block (used in SL3: no unMACed commands, plain not allowed)
   */
  status = phalMfp_WritePerso(palMfp,                 /**< [In] Pointer to this layer's parameter structure. */
                              ISO14443_4,             /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                              MFP_CONFIGURATION_BLOCK,/**< [In] MIFARE Block number. */
                              aMfpConfigBlockMACOn);  /**< [In] Value (16 bytes). */
  CHECK_SUCCESS(status);

  /**
   * write MFP Installation Identifier: used in the Virtual Card example
   */
  status = phalMfp_WritePerso(palMfp,         /**< [In] Pointer to this layer's parameter structure. */
                              ISO14443_4,     /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                              INSTALLATION_IDENTIFIER,  /**< [In] MIFARE Block number. */
                              gaVCIID);       /**< [In] Value (16 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Configure one value sector:
   * write example block 12: value 47
   */
  status = phalMfp_WritePerso(palMfp,           /**< [In] Pointer to this layer's parameter structure. */
                              ISO14443_4,       /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                              SL_1_EXAMPLE_BLOCK,  /**< [In] MIFARE Block number. */
                              aOneValueBlock);  /**< [In] Value (16 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Configure used data sector trailer:
   * write SL_1_EXAMPLE_SECTOR_TRAILER: key: 01 00 00 00 FE FF
   */
  status = phalMfp_WritePerso(palMfp,         /**< [In] Pointer to this layer's parameter structure. */
                              ISO14443_4,     /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol. */
                              SL_1_EXAMPLE_SECTOR_TRAILER,  /**< [In] MIFARE Block number. */
                              gaMFCUsedSectorTrailer);  /**< [In] Value (16 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Returns SUCCESS
   */
  printf("(Success)\n");
  return PH_ERR_SUCCESS;
}



/**
 * 2. Security Level 1 Example (MIFARE Classic compatible)
 */
uint8_t SecurityLevel1(phalMfp_SamAV2_X_DataParams_t * palMfp,        /**< [In] MIFARE Plus parameter structure. */
                       phalMfc_SamAV2_X_DataParams_t * palMfc,        /**< [In] MIFARE Classic parameter structure. */
                       phKeyStore_SamAV2_DataParams_t * pKeyStore,  /**< [In] keystore parameter structure. */
                       uint8_t * pUid,                          /**< [In] Complete serial number; uint8_t[4/7/10]. */
                       uint8_t bUidLength)                      /**< [In] Length of provided serial number (4/7/10). */
{
  phStatus_t status;
  uint8_t aPcdCap2Out[6];   /**< Pointer to PCDCaps sent from the card (6 bytes). */
  uint8_t aPdCap2[6];       /**< Pointer to PDCaps. (6 bytes) */
  uint8_t aSectorTrailerKeyA[MIFARE_KEY_A_B_SIZE]; /**< Sector Trailer Key A */
  uint8_t aSectorTrailerKeyB[MIFARE_KEY_A_B_SIZE]; /**< Sector Trailer Key B */
  uint8_t aValue[] = {0x01, 0x02, 0x03, 0x04}; /**< Value 0x01020304U */
  uint8_t bValueAddress = 0; /**< Value address */

  /**
   * Load key(SecurityLevel1AESKey) in KeyStore
   * First format the key entry
   */
  status = phKeyStore_FormatKeyEntry(pKeyStore,           /**< [In] Pointer to this layer's parameter structure. */
                                     SL_1_KEY_ADDRESS,    /**< [In] Key number of the key to be loaded. */
                                     PH_KEYSTORE_KEY_TYPE_AES128); /**< [In] Key type of the key to be loaded.*/
  CHECK_SUCCESS(status);
  /**
   * Second set the key at position
   */
  status = phKeyStore_SetKeyAtPos(pKeyStore,              /**< [In] Pointer to this layer's parameter structure. */
                                  SL_1_KEY_ADDRESS,       /**< [In] Key number of the key to be loaded. */
                                  KEY_POS,                /**< [In] Key position to be updated. */
                                  PH_KEYSTORE_KEY_TYPE_AES128, /**< [In] Key type of the key to be loaded.*/
                                  gaSecurityLevel1AESKey, /**< [In] Pointer to the key itself.*/
                                  SL_1_KEY_VERSION);      /**< [In] New Key version of the key to be updated. */
  CHECK_SUCCESS(status);

  /**
   * SL1 Card Authentication
   */
  status = phalMfp_AuthenticateSL1(palMfp,            /**< [In] Pointer to this layer's parameter structure. */
                                   ISO14443_3,        /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol; */
                                   FOLLOWING_AUTH,    /**< [In] \c 0: Following Authentication; \c 1: First Authentication; */
                                   SL_1_CARD_AUTHENTICATION_KEY,    /**< [In] Key Block number. */
                                   SL_1_KEY_ADDRESS,  /**< [In] Key Storage number. */
                                   SL_1_KEY_VERSION,  /**< [In] Key Storage version. */
                                   0,                 /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
                                   NULL,              /**< [In] Diversification Input used to diversify the key. */
                                   0,                 /**< [In] Length of the supplied PCDCaps. */
                                   NULL,              /**< [In] Pointer to PCDCaps (bLenPcdCap2 bytes), ignored if bLenPcdCap2 == 0. */
                                   aPcdCap2Out,       /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                   aPdCap2);          /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
  CHECK_SUCCESS(status);

  /**
   * get secret key A
   * keystore writes MIFARE Key A and B at once
   * only Key A is needed so the Key B is set to 0x00
   */
  memset(aSectorTrailerKeyA, 0x00, sizeof(aSectorTrailerKeyA));        /**< clear array */
  memcpy(aSectorTrailerKeyA, gaMFCUsedSectorTrailer, MIFARE_KEY_SIZE); /**< set Key A*/

  /**
   * Load MFC key(in KeyStore)
   * First format the key entry
   */
  status = phKeyStore_FormatKeyEntry(pKeyStore,
                                     MFC_KEY_ADDRESS,
                                     PH_KEYSTORE_KEY_TYPE_MIFARE);
  CHECK_SUCCESS(status);
  /**
   * Second set the key at position
   */
  status = phKeyStore_SetKeyAtPos(pKeyStore,
                                  MFC_KEY_ADDRESS,
                                  KEY_POS,
                                  PH_KEYSTORE_KEY_TYPE_MIFARE,
                                  aSectorTrailerKeyA,
                                  MFC_KEY_VERSION);
  CHECK_SUCCESS(status);

  /**
   * Authenticate
   */
  status = phalMfc_Authenticate(palMfc,              /**< [In] Pointer to this layer's parameter structure. */
                                SL_1_EXAMPLE_BLOCK,  /**< [In] Blocknumber on card to authenticate to. */
                                PHHAL_HW_MFC_KEYA,   /**< [In] Either #PHAL_MFC_KEYA or #PHAL_MFC_KEYB. */
                                MFC_KEY_ADDRESS,     /**< [In] Key number to be used in authentication. */
                                MFC_KEY_VERSION,     /**< [In] Key version to be used in authentication. */
                                pUid,                /**< [In] Complete serial number; uint8_t[4/7/10]. */
                                bUidLength);         /**< [In] Length of provided serial number (4/7/10). */
  CHECK_SUCCESS(status);

  /**
   * Read
   */
  status = phalMfc_Read(palMfc,              /**< [In] Pointer to this layer's parameter structure. */
                        SL_1_EXAMPLE_BLOCK,  /**< [In] Blocknumber on card to authenticate to. */
                        gaReceivedBlock);    /**< [Out] pBlockData[16] containing block read from the MIFARE(R) card */
  CHECK_SUCCESS(status);

  /**
   * Write
   */
  status = phalMfc_Write(palMfc,              /**< [In] Pointer to this layer's parameter structure. */
                         SL_1_EXAMPLE_BLOCK,  /**< [In] Blocknumber on card to authenticate to. */
                         gaEmptyBlock);       /**< [In] pBlockData[16] containing block to be written to the MIFARE(R) card */
  CHECK_SUCCESS(status);

  /**
   * get secret key B from gaMFCSectorTrailer
   * keystore writes MIFARE Key A and B at once
   * only Key B is needed so the first bytes for Key A are set to 0x00
   */
  memset(aSectorTrailerKeyB, 0x00, sizeof(aSectorTrailerKeyB));
  memcpy(&aSectorTrailerKeyB[6], &gaMFCSectorTrailer[10], MIFARE_KEY_SIZE);

  /**
   * Load MFC key(in KeyStore)
   * First format the key entry
   */
  status = phKeyStore_FormatKeyEntry(pKeyStore,
                                     MFC_KEY_ADDRESS,
                                     PH_KEYSTORE_KEY_TYPE_MIFARE);
  CHECK_SUCCESS(status);
  /**
   * Second set the key at position
   */
  status = phKeyStore_SetKeyAtPos(pKeyStore,
                                  MFC_KEY_ADDRESS,
                                  KEY_POS,
                                  PH_KEYSTORE_KEY_TYPE_MIFARE,
                                  aSectorTrailerKeyB,
                                  MFC_KEY_VERSION);
  CHECK_SUCCESS(status);

  /**
   * Authenticate to Value sector
   */
  status = phalMfc_Authenticate(palMfc,              /**< [In] Pointer to this layer's parameter structure. */
                                ONE_VALUE_BLOCK,     /**< [In] Blocknumber on card to authenticate to. */
                                PHHAL_HW_MFC_KEYB,   /**< [In] Either #PHAL_MFC_KEYA or #PHAL_MFC_KEYB. */
                                MFC_KEY_ADDRESS,     /**< [In] Key number to be used in authentication. */
                                MFC_KEY_VERSION,     /**< [In] Key version to be used in authentication. */
                                pUid,                /**< [In] Complete serial number; uint8_t[4/7/10]. */
                                bUidLength);         /**< [In] Length of provided serial number (4/7/10). */
  CHECK_SUCCESS(status);

  /**
   * Write value
   */
  status = phalMfc_WriteValue(palMfc,                /**< [In] Pointer to this layer's parameter structure. */
                              ONE_VALUE_BLOCK,       /**< [In] Blocknumber on card to authenticate to. */
                              aValue,                /**< [In] pValue[4] containing value (LSB first) to be written to the MIFARE(R) card */
                              ONE_VALUE_BLOCK);      /**< [In] bAddrData containing address written to the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Clear pValue
   */
  aValue[3] = aValue[2] = aValue[1] = aValue[0] = 0x00;

  /**
   * Read value
   */
  status = phalMfc_ReadValue(palMfc,           /**< [In] Pointer to this layer's parameter structure. */
                             ONE_VALUE_BLOCK,  /**< [In] Blocknumber on card to authenticate to. */
                             aValue,           /**< [Out] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                             &bValueAddress);  /**< [Out] bAddrData containing address read from the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Increment value
   */
  status = phalMfc_Increment(palMfc,          /**< [In] Pointer to this layer's parameter structure. */
                             ONE_VALUE_BLOCK, /**< [In] Blocknumber on card to authenticate to. */
                             aValue);         /**< [In] pValue[4] containing value (LSB first) to be incremented on the MIFARE(R) card */
  CHECK_SUCCESS(status);

  /**
   * Read value before transferred
   */
  status = phalMfc_ReadValue(palMfc,          /**< [In] Pointer to this layer's parameter structure. */
                             ONE_VALUE_BLOCK, /**< [In] block number to be read. */
                             aValue,          /**< [Out] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                             &bValueAddress); /**< [Out] bAddrData containing address read from the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Transfer value to EEProm
   */
  status = phalMfc_Transfer(palMfc,            /**< [In] Pointer to this layer's parameter structure. */
                            ONE_VALUE_BLOCK);  /**< [In] block number the transfer buffer shall be transferred to. */
  CHECK_SUCCESS(status);

  /**
   * Read the incremented value
   */
  status = phalMfc_ReadValue(palMfc,           /**< [In] Pointer to this layer's parameter structure. */
                             ONE_VALUE_BLOCK,  /**< [In] block number to be read. */
                             aValue,           /**< [Out] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                             &bValueAddress);  /**< [Out] bAddrData containing address read from the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Increment and transfer value
   */
  status = phalMfc_IncrementTransfer(palMfc,          /**< [In] Pointer to this layer's parameter structure. */
                                     ONE_VALUE_BLOCK, /**< [In] block number to be incremented. */
                                     ONE_VALUE_BLOCK, /**< [In] block number to be transferred to. */
                                     aValue);         /**< [In] pValue[4] containing value (LSB first) to be incremented on the MIFARE(R) card */
  CHECK_SUCCESS(status);

  /**
   * Read the incremented value
   */
  status = phalMfc_ReadValue(palMfc,           /**< [In] Pointer to this layer's parameter structure. */
                             ONE_VALUE_BLOCK,  /**< [In] block number to be read. */
                             aValue,           /**< [Out] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                             &bValueAddress);  /**< [Out] bAddrData containing address read from the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Set pValue (0x00000001)
   */
  aValue[3] = aValue[2] = aValue[1] = 0x00;
  aValue[0] = 1;

  /**
   * Decrement and transfer value
   */
  status = phalMfc_DecrementTransfer(palMfc,          /**< [In] Pointer to this layer's parameter structure. */
                                     ONE_VALUE_BLOCK, /**< [In] block number to be decremented. */
                                     ONE_VALUE_BLOCK, /**< [In] block number to be transferred to. */
                                     aValue);         /**< [In] pValue[4] containing value (LSB first) to be decremented on the MIFARE(R) card */
  CHECK_SUCCESS(status);

  /**
   * Read value
   */
  status = phalMfc_ReadValue(palMfc,           /**< [In] Pointer to this layer's parameter structure. */
                             ONE_VALUE_BLOCK,  /**< [In] block number to be read. */
                             aValue,           /**< [Out] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                             &bValueAddress);  /**< [Out] bAddrData containing address read from the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Set pValue (0x00000001)
   */
  aValue[3] = aValue[2] = aValue[1] = 0x00;
  aValue[0] = 1;

  /**
   * Write value
   */
  status = phalMfc_WriteValue(palMfc,           /**< [In] Pointer to this layer's parameter structure. */
                              ONE_VALUE_BLOCK,  /**< [In] block number to be written. */
                              aValue,           /**< [In] pValue[4] containing value (LSB first) to be written to the MIFARE(R) card */
                              ONE_VALUE_BLOCK); /**< [In] bAddrData containing address written to the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Read value
   */
  status = phalMfc_ReadValue(palMfc,           /**< [In] Pointer to this layer's parameter structure. */
                             ONE_VALUE_BLOCK,  /**< [In] block number to be read. */
                             aValue,           /**< [Out] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                             &bValueAddress);  /**< [Out] bAddrData containing address read from the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Restore
   */
  status = phalMfc_Restore(palMfc,            /**< [In] Pointer to this layer's parameter structure. */
                           ONE_VALUE_BLOCK);  /**< [In] block number the transfer buffer shall be restored from. */
  CHECK_SUCCESS(status);

  /**
   * Transfer value to EEProm
   */
  status = phalMfc_Transfer(palMfc,           /**< [In] Pointer to this layer's parameter structure. */
                            ONE_VALUE_BLOCK); /**< [In] block number the transfer buffer shall be transferred to. */
  CHECK_SUCCESS(status);

  /**
   * Read value
   */
  status = phalMfc_ReadValue(palMfc,           /**< [In] Pointer to this layer's parameter structure. */
                             ONE_VALUE_BLOCK,  /**< [In] block number to be read. */
                             aValue,           /**< [Out] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                             &bValueAddress);  /**< [Out] bAddrData containing address read from the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Returns zero if successful.
   */
  printf("(Success)\n");
  return PH_ERR_SUCCESS;

}


/**
 * 3. Security Level 2 Example (MIFARE Classic migration)
 */
uint8_t SecurityLevel2(phalMfp_SamAV2_X_DataParams_t * palMfp,   /**< [In] MIFARE Plus parameter structure. */
             phalMfc_SamAV2_X_DataParams_t * palMfc,             /**< [In] MIFARE Classic parameter structure. */
             phKeyStore_SamAV2_DataParams_t * pKeyStore,       /**< [In] keystore parameter structure. */
             uint8_t * pUid,                               /**< [In] Complete serial number; uint8_t[4/7/10]. */
             uint8_t bUidLength)                           /**< [In] Length of provided serial number (4/7/10). */
{
  phStatus_t status;
  uint8_t aKmf[6];          /**< MIFARE Sector Key Modifier (6 bytes). */
  uint8_t aPcdCap2Out[6];   /**< Pointer to PCDCaps sent from the card (6 bytes). */
  uint8_t aPdCap2[6];       /**< Pointer to PCDCaps. */
  const uint8_t bNrOfMultiBlocks = 0x02; /**<Number of multi blocks (must not be more than 3)*/
  const uint8_t bBlockNumber = 0x01;   /**< used block number */
  uint8_t aUsedDataBlock[MIFARE_KEY_A_B_SIZE]; /**< used data block */
  uint8_t aValue[] = { 0x05, 0x06, 0x07, 0x08}; /**< value: 0x05060708U */
  uint8_t bValueAddress = 0; /**< value address */
  uint8_t aDoubleBlock[] =    /**< two blocks in one array */
    { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
      0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};

  /**
   * AES Authenticate
   * Load key(in KeyStore)
   * First format the key entry
   */
  status = phKeyStore_FormatKeyEntry(pKeyStore, SL_2_AUTH_KEY_ADDRESS, PH_KEYSTORE_KEY_TYPE_AES128);
  CHECK_SUCCESS(status);
  /**
   * Second set the key at position
   */
  status = phKeyStore_SetKeyAtPos(pKeyStore, SL_2_AUTH_KEY_ADDRESS, KEY_POS, PH_KEYSTORE_KEY_TYPE_AES128, gaKey, SL_2_AUTH_KEY_VERSION);
  CHECK_SUCCESS(status);

  /**
   * AES Authentication
   */
  status = phalMfp_AuthenticateSL2(palMfp,        /**< [In] Pointer to this layer's parameter structure. */
                                   ISO14443_3,    /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol; */
                                   FOLLOWING_AUTH,  /**< [In] \c 0: Following Authentication; \c 1: First Authentication; */
                                   AES_SECTOR_KEYS_A(AES_EXAMPLE_SL2_SECTOR),    /**< [In] Key Block number. */
                                   SL_2_AUTH_KEY_ADDRESS,    /**< [In] Key Storage number. */
                                   SL_2_AUTH_KEY_VERSION,      /**< [In] Key Storage version. */
                                   0,             /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
                                   NULL,          /**< [In] Diversification Input used to diversify the key. */
                                   0,             /**< [In] Length of the supplied PCDCaps. */
                                   NULL,          /**< [In] Pointer to PCDCaps (bLenPcdCap2 bytes), ignored if bLenPcdCap2 == 0. */
                                   aPcdCap2Out,   /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                   aPdCap2,       /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                   aKmf);         /**< [Out] MIFARE Sector Key Modifier (6 bytes). */
    CHECK_SUCCESS(status);


  /**
   * MF Classic Authenticate
   * get secret key A
   * keystore writes MIFARE Key A and B at once
   * only Key A is needed so the Key B is set to 0x00
   */
  memset(aUsedDataBlock, 0x00, sizeof(aUsedDataBlock));
  memcpy(aUsedDataBlock, gaMFCSectorTrailer, MIFARE_KEY_SIZE);

  /**
   * Load key(in KeyStore)
   * First format the key entry
   */
  status = phKeyStore_FormatKeyEntry(pKeyStore,
                                     SL_2_MFP_KEY_ADDRESS,
                                     PH_KEYSTORE_KEY_TYPE_MIFARE);
  CHECK_SUCCESS(status);
  /**
   * Second set the key at position
   */
  status = phKeyStore_SetKeyAtPos(pKeyStore,
                                  SL_2_MFP_KEY_ADDRESS,
                                  KEY_POS,
                                  PH_KEYSTORE_KEY_TYPE_MIFARE,
                                  aUsedDataBlock,
                                  SL_2_MFP_KEY_VERSION);
  CHECK_SUCCESS(status);

  /**
   * Authenticate classic
   */
  status = phalMfp_AuthenticateClassicSL2(palMfp,                /**< [In] Pointer to this layer's parameter structure. */
                                          ONE_VALUE_BLOCK,       /**< [In] Blocknumber on card to authenticate to. */
                                          PHHAL_HW_MFC_KEYA,     /**< [In] Either #PHAL_MFC_KEYA or #PHAL_MFC_KEYB. */
                                          SL_2_MFP_KEY_ADDRESS,  /**< [In] Key number to be used in authentication. */
                                          SL_2_MFP_KEY_VERSION,  /**< [In] Key version to be used in authentication. */
                                          pUid,                  /**< [In] Complete serial number; uint8_t[4/7/10]. */
                                          bUidLength);           /**< [In] UID length provided. */
  CHECK_SUCCESS(status);

  /**
   * Read
   */
  status = phalMfc_Read(palMfc,         /**< [In] MIFARE Classic parameter structure. */
                        bBlockNumber,   /**< [In] Blocknumber on card to authenticate to. */
                        gaKey);         /**< [Out] pBlockData[16] containing block read from the MIFARE(R) card */
  CHECK_SUCCESS(status);

  /**
   * Write
   */
  status = phalMfc_Write(palMfc,        /**< [In] MIFARE Classic parameter structure. */
                         bBlockNumber,  /**< [In] Blocknumber on card to authenticate to. */
                         gaEmptyBlock); /**< [In] pBlockData[16] containing block to be written to the MIFARE(R) card */
  CHECK_SUCCESS(status);

  /**
   * SL2 AES Card Authentication Key B
   */
  status = phalMfp_AuthenticateSL2(palMfp,        /**< [In] Pointer to this layer's parameter structure. */
                                   ISO14443_3,    /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol; */
                                   FOLLOWING_AUTH,  /**< [In] \c 0: Following Authentication; \c 1: First Authentication; */
                                   AES_SECTOR_KEYS_B(AES_EXAMPLE_SL2_SECTOR),    /**< [In] Key Block number. */
                                   SL_2_AUTH_KEY_ADDRESS,    /**< [In] Key Storage number. */
                                   SL_2_AUTH_KEY_VERSION,      /**< [In] Key Storage version. */
                                   0,             /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
                                   NULL,          /**< [In] Diversification Input used to diversify the key. */
                                   0,             /**< [In] Length of the supplied PCDCaps. */
                                   NULL,          /**< [In] Pointer to PCDCaps (bLenPcdCap2 bytes), ignored if bLenPcdCap2 == 0. */
                                   aPcdCap2Out,   /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                   aPdCap2,       /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                   aKmf);         /**< [Out] MIFARE Sector Key Modifier (6 bytes). */
    CHECK_SUCCESS(status);

  /**
   * get secret key B from gaMFCSectorTrailer
   * keystore writes MIFARE Key A and B at once
   * only Key B is needed so the first bytes for Key A are set to 0x00
   */
  memset(aUsedDataBlock, 0x00, sizeof(aUsedDataBlock));
  memcpy(&aUsedDataBlock[6], &gaMFCSectorTrailer[10], MIFARE_KEY_SIZE);

  /**
   * Load key(in KeyStore)
   * First format the key entry
   */
  status = phKeyStore_FormatKeyEntry(pKeyStore,
                                     SL_2_MFP_KEY_ADDRESS,
                                     PH_KEYSTORE_KEY_TYPE_MIFARE);
  CHECK_SUCCESS(status);
  /**
   * Second set the key at position
   */
  status = phKeyStore_SetKeyAtPos(pKeyStore,
                                  SL_2_MFP_KEY_ADDRESS,
                                  KEY_POS,
                                  PH_KEYSTORE_KEY_TYPE_MIFARE,
                                  aUsedDataBlock,
                                  SL_2_MFP_KEY_VERSION);
  CHECK_SUCCESS(status);

  /**
   * MF Classic Authenticate to Value Sector
   */
  status = phalMfp_AuthenticateClassicSL2(palMfp,                /**< [In] Pointer to this layer's parameter structure. */
                                          ONE_VALUE_BLOCK,       /**< [In] Blocknumber on card to authenticate to. */
                                          PHHAL_HW_MFC_KEYB,     /**< [In] Either #PHAL_MFC_KEYA or #PHAL_MFC_KEYB. */
                                          SL_2_MFP_KEY_ADDRESS,  /**< [In] Key number to be used in authentication. */
                                          SL_2_MFP_KEY_VERSION,  /**< [In] Key version to be used in authentication. */
                                          pUid,                  /**< [In] Complete serial number; uint8_t[4/7/10]. */
                                          bUidLength);           /**< [In] UID length provided. */
  CHECK_SUCCESS(status);

  /**
   * Write value
   */
  status = phalMfc_WriteValue(palMfc,           /**< [In] MIFARE Classic parameter structure. */
                              ONE_VALUE_BLOCK,  /**< [In] Blocknumber on card to authenticate to. */
                              aValue,           /**< [In] pValue[4] containing value (LSB first) to be written to the MIFARE(R) card */
                              ONE_VALUE_BLOCK); /**< [In] bAddrData containing address written to the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Clear pValue
   */
  aValue[3] = aValue[2] = aValue[1] = aValue[0] = 0x00;

  /**
   * Read value
   */
  status = phalMfc_ReadValue(palMfc,           /**< [In] MIFARE Classic parameter structure. */
                             ONE_VALUE_BLOCK,  /**< [In] Blocknumber on card to authenticate to. */
                             aValue,           /**< [Out] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                             &bValueAddress);  /**< [Out] bAddrData containing address read from the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Increment value
   */
  status = phalMfc_Increment(palMfc,           /**< [In] MIFARE Classic parameter structure. */
                             ONE_VALUE_BLOCK,  /**< [In] block number to be incremented. */
                             aValue);          /**< [In] pValue[4] containing value (LSB first) to be incremented on the MIFARE(R) card */
  CHECK_SUCCESS(status);

  /**
   * Read value before transferred
   */
  status = phalMfc_ReadValue(palMfc,           /**< [In] MIFARE Classic parameter structure. */
                             ONE_VALUE_BLOCK,  /**< [In] Blocknumber on card to authenticate to. */
                             aValue,           /**< [Out] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                             &bValueAddress);  /**< [Out] bAddrData containing address read from the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Transfer value to EEProm
   */
  status = phalMfc_Transfer(palMfc,            /**< [In] MIFARE Classic parameter structure. */
                            ONE_VALUE_BLOCK);  /**< [In] block number the transfer buffer shall be transferred to. */
  CHECK_SUCCESS(status);

  /**
   * Read the incremented value
   */
  status = phalMfc_ReadValue(palMfc,           /**< [In] MIFARE Classic parameter structure. */
                             ONE_VALUE_BLOCK,  /**< [In] Blocknumber on card to authenticate to. */
                             aValue,           /**< [Out] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                             &bValueAddress);  /**< [Out] bAddrData containing address read from the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Increment and transfer value
   */
  status = phalMfc_IncrementTransfer(palMfc,          /**< [In] MIFARE Classic parameter structure. */
                                     ONE_VALUE_BLOCK, /**< [In] block number to be incremented. */
                                     ONE_VALUE_BLOCK, /**< [In] block number to be transferred to. */
                                     aValue);         /**< [In] pValue[4] containing value (LSB first) to be incremented on the MIFARE(R) card */
  CHECK_SUCCESS(status);

  /**
   * Read the incremented value
   */
  status = phalMfc_ReadValue(palMfc,           /**< [In] MIFARE Classic parameter structure. */
                             ONE_VALUE_BLOCK,  /**< [In] Blocknumber on card to authenticate to. */
                             aValue,           /**< [Out] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                             &bValueAddress);  /**< [Out] bAddrData containing address read from the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
  * Set pValue (0x00000001)
  */
  aValue[3] = aValue[2] = aValue[1] = 0x00;
  aValue[0] = 1;

  /**
   * Decrement and transfer value
   */
  status = phalMfc_DecrementTransfer(palMfc,           /**< [In] Pointer to this layer's parameter structure. */
                                     ONE_VALUE_BLOCK,  /**< [In] block number to be decremented. */
                                     ONE_VALUE_BLOCK,  /**< [In] block number to be transferred to. */
                                     aValue);          /**< [In] pValue[4] containing value (LSB first) to be decremented on the MIFARE(R) card */
  CHECK_SUCCESS(status);

  /**
   * Read value
   */
  status = phalMfc_ReadValue(palMfc,           /**< [In] MIFARE Classic parameter structure. */
                             ONE_VALUE_BLOCK,  /**< [In] Blocknumber on card to authenticate to. */
                             aValue,           /**< [Out] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                             &bValueAddress);  /**< [Out] bAddrData containing address read from the MIFARE(R) card value block*/
    CHECK_SUCCESS(status);

  /**
   * Set pValue (0x00000001)
   */
  aValue[3] = aValue[2] = aValue[1] = 0x00;
  aValue[0] = 1;

  /**
   * Write value
   */
  status = phalMfc_WriteValue(palMfc,           /**< [In] MIFARE Classic parameter structure. */
                              ONE_VALUE_BLOCK,  /**< [In] block number to be written. */
                              aValue,           /**< [In] pValue[4] containing value (LSB first) to be written to the MIFARE(R) card */
                              bValueAddress);            /**< [In] bAddrData containing address written to the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Read value
   */
  status = phalMfc_ReadValue(palMfc,           /**< [In] MIFARE Classic parameter structure. */
                             ONE_VALUE_BLOCK,  /**< [In] Blocknumber on card to authenticate to. */
                             aValue,           /**< [Out] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                             &bValueAddress);  /**< [Out] bAddrData containing address read from the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Restore
   */
  status = phalMfc_Restore(palMfc,            /**< [In] MIFARE Classic parameter structure. */
                           ONE_VALUE_BLOCK);  /**< [In] block number the transfer buffer shall be restored from. */
  CHECK_SUCCESS(status);

  /**
   * Transfer value to EEProm
   */
  status = phalMfc_Transfer(palMfc,           /**< [In] MIFARE Classic parameter structure. */
                            ONE_VALUE_BLOCK); /**< [In] block number the transfer buffer shall be transferred to. */
  CHECK_SUCCESS(status);

  /**
   * Read value
   */
  status = phalMfc_ReadValue(palMfc,           /**< [In] MIFARE Classic parameter structure. */
                             ONE_VALUE_BLOCK,  /**< [In] Blocknumber on card to authenticate to. */
                             aValue,           /**< [Out] pValue[4] containing value (LSB first) read from the MIFARE(R) card */
                             &bValueAddress);  /**< [Out] bAddrData containing address read from the MIFARE(R) card value block*/
  CHECK_SUCCESS(status);

  /**
   * Multi Block Write (over 2 blocks)
   */
  status = phalMfp_MultiBlockWrite(palMfp,             /**< [In] MIFARE Plus Software parameter structure */
                                   ONE_VALUE_BLOCK,    /**< [In] block number to be decremented. */
                                   bNrOfMultiBlocks,   /**< [In] Number of blocks to write (must not be more than 3). */
                                   aDoubleBlock);      /**< [In] Block(s) (16*bNumBlocks bytes). */
  CHECK_SUCCESS(status);

  /**
   * clear array
   */
  memset(aDoubleBlock, 0x00, sizeof(aDoubleBlock));

  /**
   * Multi Block Read (over 2 blocks)
   */
  status = phalMfp_MultiBlockRead(palMfp,             /**< [In] MIFARE Plus Software parameter structure */
                                  ONE_VALUE_BLOCK,    /**< [In] block number to be decremented. */
                                  bNrOfMultiBlocks,   /**< [In] Number of blocks to read (must not be more than 3). */
                                  aDoubleBlock);      /**< [Out] Block(s) (16*bNumBlocks bytes). */
  CHECK_SUCCESS(status);

  /**
   * Returns zero if successful.
   */
  printf("(Success)\n");
  return PH_ERR_SUCCESS;
}


/**
 * 4. Security Level 3 Example
 */
uint8_t SecurityLevel3(phalMfp_SamAV2_X_DataParams_t * palMfp,
                       phhalHw_SamAV2_DataParams_t * pHal,
                       phKeyStore_SamAV2_DataParams_t * pKeyStore,
                       phpalI14443p3a_SamAV2_X_DataParams_t * palI14443p3a,
                       phpalI14443p4a_SamAV2_X_DataParams_t * palI14443p4a)
{
  phStatus_t status;
  uint8_t aPcdCap2Out[6];             /**< Pointer to PCDCaps sent from the card (6 bytes). */
  uint8_t aPdCap2[6];                 /**< Pointer to PDCaps sent from the card (6 bytes). */
  uint8_t aMultiBlocks[16*0x10];      /**< Number of Blocks * 16 bytes(block size) */
  const uint8_t bBlockNumber = 0x04;  /**< first block of a lower sector */
  uint8_t nextSector = bBlockNumber + LOWER_SECTOR_SIZE; /**< first block of the next sector after the bBlockNumber sector */
  const uint8_t bSectorNumber = 0x07; /**< used sector trailer */
  uint8_t aMfpConfigBlockPlain[] =    /**< MFP Configuration Block: MAC on command send is optional and plain is possible */
    { 0xFF, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
  uint8_t aMFCSectorTrailerPlain[] =  /**< MFC Plain Sector Trailer */
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F, 0xFF, 0x07, 0x80, 0x69, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};
  uint8_t aFieldConfigBlock[] =       /**< Field Configuration Block */
    { 0x00, 0xAA, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

  /**
   * AES Authenticate
   * reset gaKey
   */
  memset(gaKey, 0xFF, sizeof(gaKey));

  /**
   * Load key(in KeyStore)
   * First format the key entry
   */
  status = phKeyStore_FormatKeyEntry(pKeyStore,
                                     SL_3_AUTH_KEY_ADDRESS,
                                     PH_KEYSTORE_KEY_TYPE_AES128);
  CHECK_SUCCESS(status);
  /**
   * Second set the key at position
   */
  status = phKeyStore_SetKeyAtPos(pKeyStore,
                                  SL_3_AUTH_KEY_ADDRESS,
                                  KEY_POS,
                                  PH_KEYSTORE_KEY_TYPE_AES128,
                                  gaKey,
                                  SL_3_AUTH_KEY_VERSION);
  CHECK_SUCCESS(status);

  /**
   * SL3 AES Card Authentication
   */
  status = phalMfp_AuthenticateSL3(palMfp,       /**< [In] MIFARE Plus Software parameter structure */
                                   FIRST_AUTH,   /**< [In] \c 0: Following Authentication; \c 1: First Authentication; */
                                   AES_SECTOR_KEYS_A(AES_FIRST_SECTOR),    /**< [In] Key Block number. */
                                   SL_3_AUTH_KEY_ADDRESS, /**< [In] Key Storage number. */
                                   SL_3_AUTH_KEY_VERSION,  /**< [In] Key Storage version. */
                                   0,             /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
                                   NULL,          /**< [In] Diversification Input used to diversify the key. */
                                   0,             /**< [In] Length of the supplied PCDCaps. */
                                   NULL,          /**< [In] Pointer to PCDCaps (bLenPcdCap2 bytes), ignored if bLenPcdCap2 == 0. */
                                   aPcdCap2Out,   /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                   aPdCap2);      /**< [Out] Pointer to PDCaps sent from the card (6 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Multiblock(5 blocks) read
   */
  status = phalMfp_Read(palMfp,          /**< [In] MIFARE Plus Software parameter structure */
                        ENCRYPTED_ON,    /**< [In] \c 0: Plain communication; \c 1: Encrypted communication; */
                        MAC_RESPONSE_ON, /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                        MAC_COMMAND_ON,  /**< [In] \c 0: No MAC on command; \c 1: MAC on command; */
                        bBlockNumber,    /**< [In] MIFARE block number. */
                        0x05,            /**< [In] Number of blocks to read. */
                        aMultiBlocks);   /**< [Out] Block(s) (16*bNumBlocks bytes).  */
  CHECK_SUCCESS(status);

  /**
   * clear blocks
   */
  memset(aMultiBlocks, 0x00, sizeof(aMultiBlocks));

  /**
   * Multiblock(3 blocks) write
   */
  status = phalMfp_Write(palMfp,        /**< [In] MIFARE Plus Software parameter structure */
                         ENCRYPTED_ON,  /**< [In] \c 0: Plain communication; \c 1: Encrypted communication; */
                         MAC_RESPONSE_ON,  /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                         bBlockNumber,  /**< [In] MIFARE block number. */
                         0x03,          /**< [In] Number of blocks to write (must not be more than 3). */
                         aMultiBlocks); /**< [In] Block(s) (16*bNumBlocks bytes).  */
  CHECK_SUCCESS(status);

  /**
   * Another multiblock(3 blocks) write at the next lower sector
   */
  status = phalMfp_Write(palMfp,
                         ENCRYPTED_ON,
                         MAC_RESPONSE_ON,
                         nextSector,
                         0x03,
                         aMultiBlocks);
  CHECK_SUCCESS(status);

  /**
   * Field Reset
   */
  status = phhalHw_FieldReset(pHal);
  CHECK_SUCCESS(status);

  /**
   * Activate Layer 3 card, Activate card I14443p4a, switch to ISO4, get and Set the ISO14443-4 protocol parameters
   */
  status = ActivateCard(palI14443p3a,    /**< [In] PAL-ISO14443P3A parameter structure */
                        palI14443p4a,    /**< [In] PAL-ISO14443P4A parameter structure */
                        PHPAL_I14443P4A_DATARATE_106,  /**< DRI/DSI value for 106 kBit/s */
                        PHPAL_I14443P4A_DATARATE_106); /**< DRI/DSI value for 106 kBit/s */
  CHECK_SUCCESS(status);

  /**
   * AES Authentication to Value sector
   */
  status = phalMfp_AuthenticateSL3(palMfp,
                                   FIRST_AUTH,    /**< [In] \c 0: Following Authentication; \c 1: First Authentication; */
                                   AES_SECTOR_KEYS_A(AES_FIRST_SECTOR),    /**< [In] Key Block number. */
                                   SL_3_AUTH_KEY_ADDRESS,  /**< [In] Key Storage number. */
                                   SL_3_AUTH_KEY_VERSION,  /**< [In] Key Storage version. */
                                   0,             /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
                                   NULL,          /**< [In] Diversification Input used to diversify the key. */
                                   0,             /**< [In] Length of the supplied PCDCaps. */
                                   NULL,          /**< [In] Pointer to PCDCaps (bLenPcdCap2 bytes), ignored if bLenPcdCap2 == 0. */
                                   aPcdCap2Out,   /**< [Out] Pointer to PCD Caps sent from the card (6 bytes). */
                                   aPdCap2);      /**< [Out] Pointer to PDCaps sent from the card (6 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Value File operation encrypted, MAC on response, MAC on command
   */
  status  = Mfp_VFOperations(palMfp,           /**< [In] MIFARE Plus Software parameter structure */
                             ENCRYPTED_ON,     /**< [In] \c 0: Plain communication; \c 1: Encrypted communication; */
                             MAC_RESPONSE_ON,  /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                             MAC_COMMAND_ON,   /**< [In] \c 0: No MAC on command; \c 1: MAC on command; */
                             bBlockNumber);    /**< [In] MIFARE block number. */
  CHECK_SUCCESS(status);

  /**
   * Value File operation encrypted, no MAC on response, MAC on command
   */
  status  = Mfp_VFOperations(palMfp,           /**< [In] MIFARE Plus Software parameter structure */
                             ENCRYPTED_ON,     /**< [In] \c 0: Plain communication; \c 1: Encrypted communication; */
                             MAC_RESPONSE_OFF, /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                             MAC_COMMAND_ON,   /**< [In] \c 0: No MAC on command; \c 1: MAC on command; */
                             bBlockNumber);    /**< [In] MIFARE block number. */
  CHECK_SUCCESS(status);

  /**
   * enable Plain access in MFP Configuration block
   * configure one sector to allow plain configuration
   */
  status = phalMfp_Write(palMfp,            /**< [In] MIFARE Plus Software parameter structure */
                         ENCRYPTED_ON,      /**< [In] \c 0: Plain communication; \c 1: Encrypted communication; */
                         MAC_RESPONSE_ON,   /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                         bSectorNumber,     /**< [In] MIFARE block number. */
                         0x01,              /**< [In] Number of blocks to write (must not be more than 3). */
                         aMFCSectorTrailerPlain); /**< [In] Block(s) (16*bNumBlocks bytes).  */
  CHECK_SUCCESS(status);

  /**
   * Value File operation MAC on response, MAC on command
   */
  status  = Mfp_VFOperations(palMfp,           /**< [In] MIFARE Plus Software parameter structure */
                             ENCRYPTED_OFF,    /**< [In] \c 0: Plain communication; \c 1: Encrypted communication; */
                             MAC_RESPONSE_ON,  /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                             MAC_COMMAND_ON,   /**< [In] \c 0: No MAC on command; \c 1: MAC on command; */
                             bBlockNumber);    /**< [In] MIFARE block number. */
  CHECK_SUCCESS(status);

  /**
   * Value File operation no MAC on response, MAC on command
   */
  status  = Mfp_VFOperations(palMfp,           /**< [In] MIFARE Plus Software parameter structure */
                             ENCRYPTED_OFF,    /**< [In] \c 0: Plain communication; \c 1: Encrypted communication; */
                             MAC_RESPONSE_OFF, /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                             MAC_COMMAND_ON,   /**< [In] \c 0: No MAC on command; \c 1: MAC on command; */
                             bBlockNumber);    /**< [In] MIFARE block number. */
  CHECK_SUCCESS(status);

  /**
   * Load key(VC Master Key) in KeyStore
   * First format the key entry
   */
  status = phKeyStore_FormatKeyEntry(pKeyStore,
                                     MASTER_KEY_ADDRESS,
                                     PH_KEYSTORE_KEY_TYPE_AES128);
  CHECK_SUCCESS(status);
  /**
   * Second set the key at position
   */
  status = phKeyStore_SetKeyAtPos(pKeyStore,
                                  MASTER_KEY_ADDRESS,
                                  KEY_POS,
                                  PH_KEYSTORE_KEY_TYPE_AES128,
                                  gaVCMasterKey,
                                  MASTER_KEY_VERSION);
  CHECK_SUCCESS(status);

  /**
   * Card Authentication Master Key
   */
  status = phalMfp_AuthenticateSL3(palMfp,             /**< [In] MIFARE Plus Software parameter structure */
                                   FIRST_AUTH,         /**< [In] \c 0: Following Authentication; \c 1: First Authentication; */
                                   CARD_MASTER_KEY,    /**< [In] Key Block number. */
                                   MASTER_KEY_ADDRESS, /**< [In] Key Storage number. */
                                   MASTER_KEY_VERSION, /**< [In] Key Storage version. */
                                   0,                  /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
                                   NULL,               /**< [In] Diversification Input used to diversify the key. */
                                   0,                  /**< [In] Length of the supplied PCDCaps. */
                                   NULL,               /**< [In] Pointer to PCDCaps (bLenPcdCap2 bytes), ignored if bLenPcdCap2 == 0. */
                                   aPcdCap2Out,        /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                   aPdCap2);           /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
    CHECK_SUCCESS(status);

  /**
   * Mfp Write one block(enc and mac used): Set MFP Configuration Block to maximum unMaced commands
   */
  status = phalMfp_Write(palMfp,
                         ENCRYPTED_ON,
                         MAC_RESPONSE_ON,
                         MFP_CONFIGURATION_BLOCK,
                         0x01,
                         aMfpConfigBlockPlain);
  CHECK_SUCCESS(status);

  /**
   * Reset Authentication
   */
  status = phalMfp_ResetAuth(palMfp);
  CHECK_SUCCESS(status);

  /**
   * SL3 Card Authentication
   */
  status = phalMfp_AuthenticateSL3(palMfp,       /**< [In] MIFARE Plus Software parameter structure */
                                   FIRST_AUTH,   /**< [In] \c 0: Following Authentication; \c 1: First Authentication; */
                                   AES_SECTOR_KEYS_A(AES_FIRST_SECTOR),    /**< [In] Key Block number. */
                                   SL_3_AUTH_KEY_ADDRESS,  /**< [In] Key Storage number. */
                                   SL_3_AUTH_KEY_VERSION,  /**< [In] Key Storage version. */
                                   0,            /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
                                   NULL,         /**< [In] Diversification Input used to diversify the key. */
                                   0,            /**< [In] Length of the supplied PCDCaps. */
                                   NULL,         /**< [In] Pointer to PCDCaps (bLenPcdCap2 bytes), ignored if bLenPcdCap2 == 0. */
                                   aPcdCap2Out,  /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                   aPdCap2);     /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Value File operation encrypted, MAC on response, no MAC on command
   */
  status  = Mfp_VFOperations(palMfp,
                             ENCRYPTED_ON,
                             MAC_RESPONSE_ON,
                             MAC_COMMAND_OFF,
                             bBlockNumber);
  CHECK_SUCCESS(status);

  /**
   * Value File operation encrypted, no MAC on response, no MAC on command
   */
  status  = Mfp_VFOperations(palMfp,
                             ENCRYPTED_ON,
                             MAC_RESPONSE_OFF,
                             MAC_COMMAND_OFF,
                             bBlockNumber);
  CHECK_SUCCESS(status);

  /**
   * Value File operation MAC on response, no MAC on command
   */
  status  = Mfp_VFOperations(palMfp,
                             ENCRYPTED_OFF,
                             MAC_RESPONSE_ON,
                             MAC_COMMAND_OFF,
                             bBlockNumber);
  CHECK_SUCCESS(status);

  /**
   * Value File operation no MAC on response, no MAC on command
   */
  status  = Mfp_VFOperations(palMfp,
                             ENCRYPTED_OFF,
                             MAC_RESPONSE_OFF,
                             MAC_COMMAND_OFF,
                             bBlockNumber);
  CHECK_SUCCESS(status);

  /**
   * Key update and authentication with new key
   * Load key(in KeyStore)
   * First format the key entry
   */
  gaKey[0] = 0x00;
  status = phKeyStore_FormatKeyEntry(pKeyStore,
                                     SL_3_AES_KEY_ADDRESS,
                                     PH_KEYSTORE_KEY_TYPE_AES128);
  CHECK_SUCCESS(status);
  /**
   * Second set the key at position
   */
  status = phKeyStore_SetKeyAtPos(pKeyStore,
                                  SL_3_AES_KEY_ADDRESS,
                                  KEY_POS,
                                  PH_KEYSTORE_KEY_TYPE_AES128,
                                  gaKey,
                                  SL_3_AES_KEY_VERSION);
  CHECK_SUCCESS(status);

  /**
   * Change MFP key A, AES_EXAMPLE_SL3_SECTOR
   */
  status = phalMfp_ChangeKey(palMfp,               /**< [In] MIFARE Plus Software parameter structure */
                             MAC_RESPONSE_ON,      /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                             AES_SECTOR_KEYS_A(AES_EXAMPLE_SL3_SECTOR),/**< [In] MIFARE block number. */
                             SL_3_AES_KEY_ADDRESS, /**< [In] Key number of key to be written to the card. */
                             SL_3_AES_KEY_VERSION, /**< [In] Key version of key to be written to the card. */
                             0,                    /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
                             NULL);                /**< [In] Diversification Input used to diversify the key. */
  CHECK_SUCCESS(status);

  /**
   * Reset Authentication
   */
  status = phalMfp_ResetAuth(palMfp);
  CHECK_SUCCESS(status);

  /**
   * SL3 Card Authentication AES_EXAMPLE_SL3_SECTOR key A
   */
  status = phalMfp_AuthenticateSL3(palMfp,      /**< [In] MIFARE Plus Software parameter structure */
                                   FIRST_AUTH,  /**< [In] \c 0: Following Authentication; \c 1: First Authentication; */
                                   AES_SECTOR_KEYS_A(AES_EXAMPLE_SL3_SECTOR),    /**< [In] Key Block number. */
                                   SL_3_AES_KEY_ADDRESS,  /**< [In] Key Storage number. */
                                   SL_3_AES_KEY_VERSION,  /**< [In] Key Storage version. */
                                   0,           /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
                                   NULL,        /**< [In] Diversification Input used to diversify the key. */
                                   0,           /**< [In] Length of the supplied PCDCaps. */
                                   NULL,        /**< [In] Pointer to PCDCaps (bLenPcdCap2 bytes), ignored if bLenPcdCap2 == 0. */
                                   aPcdCap2Out, /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                   aPdCap2);    /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
  CHECK_SUCCESS(status);

  /**
   * High data rate (424 kBit/s) example with multi-block read
   * Field Reset
   */
  status = phhalHw_FieldReset(pHal);
  CHECK_SUCCESS(status);

  /**
   * Activate Layer 3 card, Activate card I14443p4a, switch to ISO4, get and Set the ISO14443-4 protocol parameters
   */
  status = ActivateCard(palI14443p3a,    /**< [In] PAL-ISO14443P3A parameter structure */
                        palI14443p4a,    /**< [In] PAL-ISO14443P4A parameter structure */
                        PHPAL_I14443P4A_DATARATE_424,  /**< DRI/DSI value for 424 kBit/s */
                        PHPAL_I14443P4A_DATARATE_424); /**< DRI/DSI value for 424 kBit/s */
  CHECK_SUCCESS(status);

  /**
   * SL3 Card Authentication
   */
  status = phalMfp_AuthenticateSL3(palMfp,      /**< [In] MIFARE Plus Software parameter structure */
                                   FIRST_AUTH,  /**< [In] \c 0: Following Authentication; \c 1: First Authentication; */
                                   AES_SECTOR_KEYS_A(AES_EXAMPLE_SL3_SECTOR),    /**< [In] Key Block number. */
                                   SL_3_AES_KEY_ADDRESS,  /**< [In] Key Storage number. */
                                   SL_3_AES_KEY_VERSION,  /**< [In] Key Storage version. */
                                   0,           /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
                                   NULL,        /**< [In] Diversification Input used to diversify the key. */
                                   0,           /**< [In] Length of the supplied PCDCaps. */
                                   NULL,        /**< [In] Pointer to PCDCaps (bLenPcdCap2 bytes), ignored if bLenPcdCap2 == 0. */
                                   aPcdCap2Out, /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                   aPdCap2);    /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Multiblock(3 blocks) read
   */
  status = phalMfp_Read(palMfp,
                        ENCRYPTED_ON,
                        MAC_RESPONSE_ON,
                        MAC_COMMAND_ON,
                        bBlockNumber,
                        0x03,
                        aMultiBlocks);
  CHECK_SUCCESS(status);

  /**
   * clear blocks
   */
  memset(aMultiBlocks, 0x00, sizeof(aMultiBlocks));

  /**
   * Multiblock(2 blocks) write
   */
  status = phalMfp_Write(palMfp,
                         ENCRYPTED_ON,
                         MAC_RESPONSE_ON,
                         bBlockNumber,
                         0x02,
                         aMultiBlocks);
  CHECK_SUCCESS(status);

  /**
   * Configure Random ID
   * Reset Authentication
   */
  status = phalMfp_ResetAuth(palMfp);
  CHECK_SUCCESS(status);

  /**
   * Load VCConfigKey into the keystore
   * First format the key entry
   */
  status = phKeyStore_FormatKeyEntry(pKeyStore,
                                     SL_3_VC_CONFIG_KEY_ADDRESS,
                                     PH_KEYSTORE_KEY_TYPE_AES128);
  CHECK_SUCCESS(status);
  /**
   * Second set the key at position
   */
  status = phKeyStore_SetKeyAtPos(pKeyStore,
                                  SL_3_VC_CONFIG_KEY_ADDRESS,
                                  KEY_POS,
                                  PH_KEYSTORE_KEY_TYPE_AES128,
                                  gaVCConfigKey,
                                  SL_3_VC_CONFIG_KEY_VERSION);
  CHECK_SUCCESS(status);

  /**
   * Auth VCConfigKey
   */
  status = phalMfp_AuthenticateSL3(palMfp,      /**< [In] MIFARE Plus Software parameter structure */
                                   FIRST_AUTH,  /**< [In] \c 0: Following Authentication; \c 1: First Authentication; */
                                   CARD_CONFIGURATIOM_KEY,    /**< [In] Key Block number. */
                                   SL_3_VC_CONFIG_KEY_ADDRESS,    /**< [In] Key Storage number. */
                                   SL_3_VC_CONFIG_KEY_VERSION,  /**< [In] Key Storage version. */
                                   0,           /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
                                   NULL,        /**< [In] Diversification Input used to diversify the key. */
                                   0,           /**< [In] Length of the supplied PCDCaps. */
                                   NULL,        /**< [In] Pointer to PCDCaps (bLenPcdCap2 bytes), ignored if bLenPcdCap2 == 0. */
                                   aPcdCap2Out, /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                   aPdCap2);    /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Write Random ID is used
   */
  status = phalMfp_Write(palMfp,
                         ENCRYPTED_ON,
                         MAC_RESPONSE_ON,
                         FIELD_CONFIGURATION_BLOCK,
                         0x01,
                         aFieldConfigBlock);
  CHECK_SUCCESS(status);

  /**
   * Returns zero if successful.
   */
  printf("(Success)\n");
  return PH_ERR_SUCCESS;
}

/**
 * Virtual Card Example
 */
uint8_t VirtualCard(phalMfp_SamAV2_X_DataParams_t * palMfp,
                    phalVca_SamAV2_X_DataParams_t * phalVca,
                    phKeyStore_SamAV2_DataParams_t * pKeyStore)
{
  phStatus_t status;
  uint16_t wNumValidIids;     /**< number of valid IIDS */
  uint8_t aPcdCap2Out[6];     /**< Pointer to PCDCaps sent from the card (6 bytes). */
  uint8_t aPdCap2[6];         /**< Pointer to PCDCaps. */
  uint16_t wIidIndex;         /**< corresponding IID to the key pair where the MAC was matching in VCSL command call. */
  uint8_t bVcUidSize;         /**< Size of the VC UID (4 or 7 bytes). */
  uint8_t aVcUid[7];          /**< VC UID (aVcUidSize bytes). */
  uint8_t aInfo[1];           /**< Info byte. */
  uint8_t aPdCapabilities[2];

  /**
   * Start Card Selection
   */
  status = phalVca_StartCardSelection(phalVca);  /**< [In] VCA Software parameter structure */
  CHECK_SUCCESS(status);

  /**
   * Load VC polling ENC Key into the keystore
   * First format the key entry
   */
  status = phKeyStore_FormatKeyEntry(pKeyStore,
                                     VC_POLLING_ENC_KEY_ADDRESS,
                                     PH_KEYSTORE_KEY_TYPE_AES128);
  CHECK_SUCCESS(status);
  /**
   * Second set the key at position
   */
  status = phKeyStore_SetKeyAtPos(pKeyStore,
                                  VC_POLLING_ENC_KEY_ADDRESS,
                                  KEY_POS,
                                  PH_KEYSTORE_KEY_TYPE_AES128,
                                  gaVCPollingENCKey,
                                  VC_POLLING_ENC_KEY_VERSION);
  CHECK_SUCCESS(status);

  /**
   * Load VC polling MAC Key into the keystore
   * First format the key entry
   */
  status = phKeyStore_FormatKeyEntry(pKeyStore,
                                     VC_POLLING_MAC_KEY_ADDRESS,
                                     PH_KEYSTORE_KEY_TYPE_AES128);
  CHECK_SUCCESS(status);
  /**
   * Second set the key at position
   */
  status = phKeyStore_SetKeyAtPos(pKeyStore,
                                 VC_POLLING_MAC_KEY_ADDRESS,
                                 KEY_POS,
                                 PH_KEYSTORE_KEY_TYPE_AES128,
                                 gaVCPollingMACKey,
                                 VC_POLLING_MAC_KEY_VERSION);
  CHECK_SUCCESS(status);

  /**
   * VC Support with wrong VCIID
   */
  gaVCIID[0] = 0x01;
  status = phalVca_VcSupport(phalVca,            /**< [In] VCA Software parameter structure */
                             gaVCIID,            /**< [In] Installation Identifier (16 bytes). */
                             VC_POLLING_ENC_KEY_ADDRESS,  /**< [In] Key number of the ENC key associated to the Iid. */
                             VC_POLLING_ENC_KEY_VERSION,  /**< [In] Key version of the ENC key associated to the Iid. */
                             VC_POLLING_MAC_KEY_ADDRESS,  /**< [In] Key number of the MAC key associated to the Iid. */
                             VC_POLLING_MAC_KEY_VERSION); /**< [In] Key version of the MAC key associated to the Iid. */
  CHECK_SUCCESS(status);

  /**
   * Set correct VCIID
   */
  gaVCIID[0] = 0x00;

  /**
   * VC Support Last with correct VCIID
   */
  status = phalVca_VcSupportLast(phalVca,            /**< [In] VCA Software parameter structure */
                                 gaVCIID,            /**< [In] Installation Identifier (16 bytes). */
                                 0,                  /**< [In] Length of the PCDcaps (0-6). */
                                 NULL,               /**< [In] PCD Capabilities (bLenCap bytes), ignored if bLenCap == 0. */
                                 VC_POLLING_ENC_KEY_ADDRESS,  /**< [In] Key number of the ENC key used in VCSL command. */
                                 VC_POLLING_ENC_KEY_VERSION,  /**< [In] Key version of the ENC key used in VCSL command. */
                                 VC_POLLING_MAC_KEY_ADDRESS,  /**< [In] Key number of the MAC key used in VCSL command. */
                                 VC_POLLING_MAC_KEY_VERSION); /**< [In] Key version of the MAC key used in VCSL command. */
  CHECK_SUCCESS(status);

  /**
   * Finalize Card Selection, returns the number of valid Iids
   */
  status = phalVca_FinalizeCardSelection(phalVca,         /**< [In] VCA Software parameter structure */
                                         &wNumValidIids); /**< [Out] Number of valid IIDs detected during VCS/VCSL sequence execution. */
  CHECK_SUCCESS(status);

  /**
   * Returns the ID of the valid Iid
   */
  status = phalVca_GetIidInfo(phalVca,         /**< [In] VCA Software parameter structure */
                              wNumValidIids-1, /**< [In] index of valid IID, needs to be < pNumValidIids in FinalizeCardSelection call. */
                              &wIidIndex,      /**< [Out] corresponding IID to the key pair where the MAC was matching in VCSL command call. */
                              &bVcUidSize,     /**< [Out] Size of the VC UID (4 or 7 bytes). */
                              aVcUid,          /**< [Out] VC UID (aVcUidSize bytes). */
                              aInfo,           /**< [Out] Info byte. */
                              aPdCapabilities);/**< [Out] PD Capabilities (2 bytes). */
  CHECK_SUCCESS(status);

  /**
   * Load VCConfigKey into the keystore
   * First format the key entry
   */
  status = phKeyStore_FormatKeyEntry(pKeyStore,
                                     VC_SELECT_KEY_ADDRESS,
                                     PH_KEYSTORE_KEY_TYPE_AES128);
  CHECK_SUCCESS(status);
  /**
   * Second set the key at position
   */
  status = phKeyStore_SetKeyAtPos(pKeyStore,
                                  VC_SELECT_KEY_ADDRESS,
                                  KEY_POS,
                                  PH_KEYSTORE_KEY_TYPE_AES128,
                                  gaSelectVCKey,
                                  VC_SELECT_KEY_VERSION);
  CHECK_SUCCESS(status);

  /**
   * Select VC
   */
  status = phalVca_SelectVc(phalVca,                /**< [In] VCA Software parameter structure */
                            (wNumValidIids-1),      /**< [In] index of valid IID, needs to be < pNumValidIids in FinalizeCardSelection call. */
                            VC_SELECT_KEY_ADDRESS,  /**< [In] Key number of the MAC key used in SVC command. */
                            VC_SELECT_KEY_VERSION); /**< [In] Key version of the MAC key used in SVC command. */
  CHECK_SUCCESS(status);

  /**
   * SL3 Card Authentication
   */
  status = phalMfp_AuthenticateSL3(palMfp,
                                   FIRST_AUTH,   /**< [In] \c 0: Following Authentication; \c 1: First Authentication; */
                                   AES_SECTOR_KEYS_A(AES_FIRST_SECTOR),    /**< [In] Key Block number. */
                                   SL_3_AUTH_KEY_ADDRESS, /**< [In] Key Storage number. */
                                   SL_3_AUTH_KEY_VERSION,  /**< [In] Key Storage version. */
                                   0,            /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
                                   NULL,         /**< [In] Diversification Input used to diversify the key. */
                                   0,            /**< [In] Length of the supplied PCDCaps. */
                                   NULL,         /**< [In] Pointer to PCDCaps (bLenPcdCap2 bytes), ignored if bLenPcdCap2 == 0. */
                                   aPcdCap2Out,  /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                   aPdCap2);     /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
    CHECK_SUCCESS(status);

  /**
   * Returns zero if successful.
   */
  printf("(Success)\n");
  return PH_ERR_SUCCESS;
}

void Log_Callback(void * pDataParams, uint8_t bOption, phLog_LogEntry_t * pEntries, uint16_t wEntryCount)
{
    phStatus_t status;
    uint8_t bIsFirstString;
    uint8_t bIsFirstParam;
    uint16_t wIndex;
    uint16_t wByteIndex;
    uint16_t wOption;
    char sOutput[512];
    char sTmp[7];

    /* satisfy compiler */
    if (pDataParams);

    /* Ignore some calls */
    if ((strncmp((char*)pEntries[0].pString, "phhalHw_SetConfig", 17) == 0) ||
        (strncmp((char*)pEntries[0].pString, "phhalHw_GetConfig", 17) == 0) ||
        (strncmp((char*)pEntries[0].pString, "phhalHw_ReadReg", 15) == 0) ||
        (strncmp((char*)pEntries[0].pString, "phhalHw_WriteReg", 16) == 0))
    {
        return;
    }

    /* Clear output string */
    sprintf(sOutput, "");

    /* Init. states */
    bIsFirstString = 1;
    bIsFirstParam = 1;

    if (bOption != PH_LOG_OPTION_CATEGORY_GEN)
    {
        strcat_s(sOutput, sizeof(sOutput), "\n");
    }

    for (wIndex = 0; wIndex < wEntryCount; ++wIndex)
    {
        if ((pEntries[wIndex].bLogType == PH_LOG_LOGTYPE_INFO) && pEntries[wIndex].wDataLen == 0)
        {
            strcat_s(sOutput, sizeof(sOutput), (char*)pEntries[wIndex].pString);
            if (bIsFirstString)
            {
                switch (bOption)
                {
                case PH_LOG_OPTION_CATEGORY_ENTER:
                    strcat_s(sOutput, sizeof(sOutput), " [ENTRY] (");
                    break;
                case PH_LOG_OPTION_CATEGORY_GEN:
                    strcat_s(sOutput, sizeof(sOutput), " [GEN] (");
                    break;
                case PH_LOG_OPTION_CATEGORY_LEAVE:
                    strcat_s(sOutput, sizeof(sOutput), " [LEAVE] (");
                    break;
                }
                bIsFirstString = 0;
            }
        }
        else
        {
            if (!bIsFirstParam)
            {
                if ((strcmp((char*)pEntries[wIndex].pString, "status") != 0) && (pEntries[wIndex].wDataLen > 0))
                {
                    strcat_s(sOutput, sizeof(sOutput), ", ");
                }
            }
            else
            {
                bIsFirstParam = 0;
            }
        }

        if ((strcmp((char*)pEntries[wIndex].pString, "wOption") == 0) && (pEntries[wIndex].wDataLen == 2))
        {
            wOption = (((uint8_t*)(pEntries[wIndex].pData))[0] << 8) | ((uint8_t*)(pEntries[wIndex].pData))[1];
            strcat_s(sOutput, sizeof(sOutput), "wOption = ");
            switch (wOption & PH_EXCHANGE_MODE_MASK)
            {
            case PH_EXCHANGE_DEFAULT:
                strcat_s(sOutput, sizeof(sOutput), "DEFAULT");
                break;
            case PH_EXCHANGE_TXCHAINING:
                strcat_s(sOutput, sizeof(sOutput), "TXCHAINING");
                break;
            case PH_EXCHANGE_RXCHAINING:
                strcat_s(sOutput, sizeof(sOutput), "RXCHAINING");
                break;
            case PH_EXCHANGE_RXCHAINING_BUFSIZE:
                strcat_s(sOutput, sizeof(sOutput), "RXCHAINING_BUFSIZE");
                break;
            }
            if ((wOption & PH_EXCHANGE_BUFFERED_BIT) != 0)
            {
                strcat_s(sOutput, sizeof(sOutput), " | BUFFERED_BIT");
            }
            if ((wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT) != 0)
            {
                strcat_s(sOutput, sizeof(sOutput), " | LEAVE_BUFFER_BIT");
            }
        }
        else if ((strcmp((char*)pEntries[wIndex].pString, "status") == 0) && (pEntries[wIndex].wDataLen == 2))
        {
            strcat_s(sOutput, sizeof(sOutput), ") = ");
            status = (((uint8_t*)(pEntries[wIndex].pData))[0] << 8) | ((uint8_t*)(pEntries[wIndex].pData))[1];
            sprintf_s(sTmp, sizeof(sTmp), "0x%04X", status);
            strcat_s(sOutput, sizeof(sOutput), sTmp);
        }
        else if (pEntries[wIndex].wDataLen > 0)
        {
            strcat_s(sOutput, sizeof(sOutput), (char*)pEntries[wIndex].pString);
            strcat_s(sOutput, sizeof(sOutput), "(");
            sprintf_s(sTmp, sizeof(sTmp), "%d", pEntries[wIndex].wDataLen);
            strcat_s(sOutput, sizeof(sOutput), sTmp);
            strcat_s(sOutput, sizeof(sOutput), ") = ");

            for (wByteIndex = 0; wByteIndex < pEntries[wIndex].wDataLen; ++wByteIndex)
            {
                sprintf_s(sTmp, sizeof(sTmp), "%02X", ((uint8_t*)pEntries[wIndex].pData)[wByteIndex]);
                strcat_s(sOutput, sizeof(sOutput), sTmp);
                if ((wByteIndex + 1) < pEntries[wIndex].wDataLen)
                {
                    strcat_s(sOutput, sizeof(sOutput), " ");
                }
            }
        }
    }

    if (bOption == PH_LOG_OPTION_CATEGORY_LEAVE)
    {
        strcat_s(sOutput, sizeof(sOutput), "\n");
    }
    else
    {
        strcat_s(sOutput, sizeof(sOutput), ")");
    }
    printf(sOutput);
}

int __cdecl main()
{
    phStatus_t status;
    uint8_t aSak[1];                  /**< Select Acknowledge; uint8_t. */
    uint8_t aUid[10];                 /**< complete Uid; uint8_t[4/7/10]. */
    uint8_t aAtr[256];                /**< Atr */
    uint8_t aSeed[8];                 /**< Seed */
    uint8_t bMoreCardsAvailable;      /**< more card available */
    uint8_t bLength;                  /**< Length */
	uint16_t aVersion[0x1F];          /**< SAM Version */
    uint8_t aHalBufferSam[261];       /**< Hal SAM Buffer array */
	uint8_t bSamHostMode = 0;         /**< SAM host mode */
	uint8_t div = 0;                  /**< diversivication */
    uint8_t authtype = PLAIN;         /**< authentication type: PLAIN, MAC, FULL */
    uint8_t aPcdCap2Out[6];           /**< Pointer to PCDCaps sent from the card (6 bytes). */
    uint8_t aPdCap2[6];               /**< Pointer to PCDCaps. */
    uint8_t aKmf[6];                  /**< MIFARE Sector Key Modifier (6 bytes). */

    /**
     * keystore
     */
    phKeyStore_Sw_KeyEntry_t aKeyEntry[KEYCOUNT];
    phKeyStore_Sw_KeyVersionPair_t aKeyVersion[KEYCOUNT * KEYVERSIONS];
    phKeyStore_Sw_KUCEntry_t aKeyUsage[KEYCOUNT];

    /**
	 * Bfl data parameter storage
	 */
    phbalReg_PcscWin_DataParams_t balPcsc;            /**< PCSC (Windows) BAL parameter structure */
    phpalI14443p3a_SamAV2_X_DataParams_t I14443p3a_X; /**< SAM PAL-ISO14443P3A parameter structure */
	phpalI14443p4a_SamAV2_X_DataParams_t I14443p4a_X; /**< SAM PAL-ISO14443P4A parameter structure */
    phpalI14443p4_SamAV2_X_DataParams_t I14443p4_X;   /**< SAM PAL-ISO14443P4 parameter structure */
	phpalMifare_SamAV2_X_DataParams_t palMifare_X;    /**< SAM PAL-MIFARE parameter structure */
    phCryptoSym_Sw_DataParams_t cryptoEnc;            /**< CryptoSym parameter structure for ENC */
    phCryptoSym_Sw_DataParams_t cryptoMac;            /**< CryptoSym parameter structure for MAC */
    phCryptoSym_Sw_DataParams_t cryptoSymRnd;         /**< CryptoSym parameter structure for SymRng */
    phCryptoRng_Sw_DataParams_t cryptoRng;            /**< CryptoRng parameter structure for Rng */
    phKeyStore_Sw_DataParams_t keyStoreSw;            /**< Software KeyStore parameter structure */
    phalMfc_SamAV2_X_DataParams_t alMfc_X;            /**< SamAV2 AL-MFC parameter structure */
    phhalHw_SamAV2_DataParams_t halSamLC0;            /**< SamAV2 HAL parameter structure LC0 */
    phhalHw_SamAV2_DataParams_t halSamLC1;            /**< SamAV2 HAL parameter structure LC1 */
    phKeyStore_SamAV2_DataParams_t keyStoreSAM;       /**< SAM KeyStore parameter structure */
    phalMfp_SamAV2_X_DataParams_t alMfp_X;            /**< MIFARE Plus Software parameter structure */
	void * pKeyStore;                                 /**< Pointer to a KeyStore parameter structure */

    /**
     * Virtual Card
     */
    phalVca_SamAV2_X_DataParams_t alVca;                               /**< VCA Software parameter structure */
    phalVca_SamAV2_X_IidTable_t aIidTableStorage[IID_KEY_COUNT];  /**< Pointer to the Iid Table storage for the layer. */
    uint16_t wNumIidTableStorageEntries = IID_KEY_COUNT;         /**< Number of possible Iid table entries in the storage. */
    phalVca_SamAV2_X_CardTableEntry_t aCardTableStorage[IID_KEY_COUNT];/**< Pointer to the Card Table storage for the layer. */
    uint16_t wNumCardTableStorageEntries = IID_KEY_COUNT;        /**< Number of possible Card table entries in the storage. */
    phalVca_SamAV2_X_KeysTable_t pKeysTable[IID_KEY_COUNT];

    printf("\nNxpRdLib ANSI-C Example Program V1.0a\n\n");
    printf("Please ensure that a Pegoda reader is connected and in working condition.\n\n");
    printf("Performing startup...\n\n");

    /* Initialize logging */
	status = phLog_Init(Log_Callback, logRegisterEntries, (uint16_t)(sizeof(logRegisterEntries) / sizeof(phLog_RegisterEntry_t)));
	CHECK_SUCCESS(status);
    status = phLog_Register(&halSamLC0, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(status);
    status = phLog_Register(&halSamLC1, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(status);
    status = phLog_Register(&keyStoreSAM, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(status);
    status = phLog_Register(&alMfc_X, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(status);
    status = phLog_Register(&alMfp_X, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(status);

    /**
	 * init. Sw keystore
	 */
    status = phKeyStore_Sw_Init(&keyStoreSw,   /**< [In] Pointer to this layer's parameter structure. */
                                sizeof(phKeyStore_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
                                aKeyEntry,     /**< [In] Pointer to a storage containing the key entries. */
                                KEYCOUNT,      /**< [In] size of pKeyEntries. */
                                aKeyVersion,   /**< [In] Pointer to a storage containing the key version pairs. */
                                KEYVERSIONS,   /**< [In] amount of key versions available in each key entry. */
                                aKeyUsage,     /**< [In] Key usage counter entry storage, size = sizeof(phKeyStore_Sw_KUCEntry_t) * wNumKUCEntries */
                                KEYCOUNT);     /**< [In] Number of Key usage counter entries. */
    CHECK_SUCCESS(status);
	pKeyStore = &keyStoreSw;  /**< set the software keystore */

    /**
	 * init. crypto
	 */
    status = phCryptoSym_Sw_Init(&cryptoEnc,    /**< [In] Pointer to this layer's parameter structure. */
		                         sizeof(phCryptoSym_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
								 pKeyStore);  /**< [In] Pointer to a key store structure (can be null).*/
    CHECK_SUCCESS(status);
    status = phCryptoSym_Sw_Init(&cryptoMac,    /**< [In] Pointer to this layer's parameter structure. */
		                         sizeof(phCryptoSym_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
								 pKeyStore);    /**< [In] Pointer to a key store structure (can be null).*/
    CHECK_SUCCESS(status);
    status = phCryptoSym_Sw_Init(&cryptoSymRnd, /**< [In] Pointer to this layer's parameter structure. */
		                         sizeof(phCryptoSym_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
								 pKeyStore);    /**< [In] Pointer to a key store structure (can be null).*/
    CHECK_SUCCESS(status);
    status = phCryptoRng_Sw_Init(&cryptoRng,    /**< [In] Pointer to this layer's parameter structure. */
		                         sizeof(phCryptoRng_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
								 &cryptoSymRnd);  /**< [In] Pointer to the parameter structure of the symmetric crypto layer. */
    CHECK_SUCCESS(status);

    /**
     * clear seed
     */
    memset(aSeed, 0x00, 0x08);
    /**
     * seed the random number generator with the given seed
     */
    status = phCryptoRng_Seed(&cryptoRng,      /**< [In] CryptoSym parameter structure for Rng */
                              aSeed,           /**< [In] Seed */
                              sizeof(aSeed));  /**< [In] Size of the seed. */
    CHECK_SUCCESS(status);

    printf("----------------\n");
    printf("Connected RD710/RD852 Sam in X-Mode\n");
    printf("----------------\n");
	printf("\n\n");

 	/**
	 * Init. PCSC BAL
	 */
	status = phbalReg_PcscWin_Init(&balPcsc,                                /**< [In] Pointer to this layer's parameter structure. */
		                           sizeof(phbalReg_PcscWin_DataParams_t),   /**< [In] Specifies the size of the data parameter structure. */
                                   aAtr,                                    /**< [In] ATR buffer. */
                                   sizeof(aAtr));                           /**< [In] ATR buffer size. */
	CHECK_SUCCESS(status);

    /**
	 * Select Port to be used (SAM in X-Mode)
	 */
    status = phbalReg_SetPort(&balPcsc, /**< [In] Pointer to this layer's parameter structure. */
		                      (uint8_t*)PCSC_READER_P2_X_NAME);  /**< [In] Port Name as String. */
	CHECK_SUCCESS(status);

	/**
	 * init the SAM HAL LC0 component (SAM in X-Mode)
	 */
	status = phhalHw_SamAV2_Init(&halSamLC0,                    /**< SamAV2 HAL parameter structure LC0 */
	 					         sizeof(phhalHw_SamAV2_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
								 &balPcsc,                      /**< PCSC (Windows) BAL parameter structure */
								 NULL,                          /**< Pointer to the hal data params of a reader. NULL in case of X-Mode */
                                 pKeyStore,                     /**< Pointer to KeyStore parameter structure */
                                 &cryptoEnc,                    /**< CryptoSym parameter structure for ENC */
                                 &cryptoMac,                    /**< CryptoSym parameter structure for MAC */
                                 &cryptoRng,                    /**< Pointer to the parameter structure of the CryptoRng layer. */
                                 PHHAL_HW_SAMAV2_OPMODE_X_RC523,   /**< Operation mode */
                                 SAM_LC0,                       /**< Logical channel */
                                 aHalBufferSam,                 /**< Hal SAM Buffer array TX */
                                 sizeof(aHalBufferSam),         /**< Hal SAM Buffer array length TX */
                                 aHalBufferSam,                 /**< Hal SAM Buffer array RX */
                                 sizeof(aHalBufferSam));        /**< Hal SAM Buffer array length RX */
   CHECK_SUCCESS(status)

    /**
	 * Init hal of sam with HComm AV1 object
     * init the SAM HAL LC1 component
	 */
	status = phhalHw_SamAV2_Init(&halSamLC1,                    /**< SamAV2 HAL parameter structure LC1 */
							     sizeof(phhalHw_SamAV2_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
							     &balPcsc,                      /**< PCSC (Windows) BAL parameter structure */
							     NULL,                          /**< Pointer to the hal data params of a reader. NULL in case of X-Mode */
                                 pKeyStore,                     /**< Pointer to KeyStore parameter structure */
                                 &cryptoEnc,                    /**< CryptoSym parameter structure for ENC */
                                 &cryptoMac,                    /**< CryptoSym parameter structure for MAC */
                                 &cryptoRng,                    /**< Pointer to the parameter structure of the CryptoRng layer. */
                                 PHHAL_HW_SAMAV2_OPMODE_X_RC523,   /**< Operation mode */
                                 SAM_LC1,                       /**< Logical channel */
                                 aHalBufferSam,                 /**< Hal SAM Buffer array TX */
                                 sizeof(aHalBufferSam),         /**< Hal SAM Buffer array length TX */
                                 aHalBufferSam,                 /**< Hal SAM Buffer array RX */
                                 sizeof(aHalBufferSam));        /**< Hal SAM Buffer array length RX */
	CHECK_SUCCESS(status);

    /**
	 * Init SAM as keyStore object
	 */
    status = phKeyStore_SamAV2_Init(&keyStoreSAM,  /**< [In] Pointer to this layer's parameter structure. */
								    sizeof(phKeyStore_SamAV2_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
									&halSamLC1);  /**< [In] Pointer to the parameter structure of the underlying layer.*/
	CHECK_SUCCESS(status);

	/**
	 * Open the PCSC port
	 */
	status = phbalReg_OpenPort(&balPcsc);  /**< [In] Pointer to this layer's parameter structure. */
	CHECK_SUCCESS(status);

	/**
	 * Detect SAM settings LC0
	 */
	status = phhalHw_SamAV2_DetectMode(&halSamLC0);  /**< [In] Pointer to this layer's parameter structure. */
	CHECK_SUCCESS(status);

	/**
	 * Detect SAM settings LC1
	 */
	status = phhalHw_SamAV2_DetectMode(&halSamLC1);  /**< [In] Pointer to this layer's parameter structure. */
	CHECK_SUCCESS(status);

    /**
	 * Get SAM hostmode
	 */
    status = phhalHw_GetConfig(&halSamLC0, PHHAL_HW_SAMAV2_CONFIG_HOSTMODE, aVersion);
	CHECK_SUCCESS(status);

    if (aVersion[0] == SAM_AV1)
    {
        /**
		 * AV1:
		 * Load SAM DES key at position 0 in KeyStore
		 * set the host mode
		 */
        bSamHostMode = SAM_AV1;
        /**
		 * format the key entry
		 */
        status = phKeyStore_FormatKeyEntry(pKeyStore,  /**< [In] Pointer to the parameter structure of the KeyStore component. */
			                               SAM_KEY,       /**< [In] KeyEntry number to be Formatted. */
										   PH_KEYSTORE_KEY_TYPE_DES);  /**< [In] New Key type of the KeyEntry (predefined type of KeyType).*/
        CHECK_SUCCESS(status);
	}
    else
    {
        /**
		 * AV2:
		 * Load SAM AES128 key at position 0 in KeyStore
		 * set the host mode
		 */
        bSamHostMode = SAM_AV2;
        /**
		 * format the key entry
		 */
        status = phKeyStore_FormatKeyEntry(pKeyStore,  /**< [In] Pointer to the parameter structure of the KeyStore component. */
			                               SAM_KEY,    /**< [In] KeyEntry number to be Formatted. */
										   PH_KEYSTORE_KEY_TYPE_AES128);  /**< [In] New Key type of the KeyEntry (predefined type of KeyType).*/
        CHECK_SUCCESS(status);
	}

	pKeyStore = &keyStoreSAM;  /**< set the sam keystore */

    /**
	 * init the 14443-3A component (SAM in X-Mode)
	 */
    status = phpalI14443p3a_SamAV2_X_Init(&I14443p3a_X,  /**< [In] Pointer to this layer's parameter structure. */
		                                  sizeof(phpalI14443p3a_SamAV2_X_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
										  &halSamLC0);  /**< [In] Pointer to the parameter structure of the underlying layer.*/
	CHECK_SUCCESS(status);

    /**
	 * init the 14443-4A component (SAM in X-Mode)
	 */
	status = phpalI14443p4a_SamAV2_X_Init(&I14443p4a_X,  /**< [In] Pointer to this layer's parameter structure. */
		                                  sizeof(phpalI14443p4a_SamAV2_X_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
										  &halSamLC0);  /**< [In] Pointer to the parameter structure of the underlying layer.*/
	CHECK_SUCCESS(status);

	/**
	 * init the 14443-4 component (SAM in X-Mode)
	 */
    status = phpalI14443p4_SamAV2_X_Init(&I14443p4_X,  /**< [In] Pointer to this layer's parameter structure. */
		                                 sizeof(phpalI14443p4_SamAV2_X_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
										 &halSamLC0);  /**< [In] Pointer to the parameter structure of the underlying layer.*/
	CHECK_SUCCESS(status);

    /**
	 * init. mifare pal (SAM in X-Mode)
	 */
    status = phpalMifare_SamAV2_X_Init(&palMifare_X,  /**< [In] Pointer to this layer's parameter structure. */
		                               sizeof(phpalMifare_SamAV2_X_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
									   &halSamLC0,    /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
									   &I14443p4_X);  /**< [In] Pointer to the parameter structure of the underlying ISO14443-4 layer. */
	CHECK_SUCCESS(status);

	/**
	 * init. MIFARE classic (SAM in X-Mode)
	 */
	status = phalMfc_SamAV2_X_Init(&alMfc_X,        /**< [In] Pointer to this layer's parameter structure. */
								   sizeof(phalMfp_SamAV2_X_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
								   &halSamLC0,      /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
								   &palMifare_X);   /**< [In] Pointer to the parameter structure of the palMifare component. */
	CHECK_SUCCESS(status);

	status = phalVca_SamAV2_X_Init(&alVca,                  /**< [In] VCA Software parameter structure */
                                   sizeof(alVca),           /**< [In] Specifies the size of the data parameter structure */
                                   &halSamLC0,      /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
						           &palMifare_X,              /**< [In] Pointer to a palMifare component context. */
						           aIidTableStorage,        /**< [In] Pointer to the IidTable of the layer */
                                   pKeysTable,               /**< [In] Pointer to the KeysTable of the layer  */
                                   wNumIidTableStorageEntries,    /**< [In] Number of possible Iid table entries in the storage. */
                                   aCardTableStorage,        /**< [In] Pointer to the Card Table storage for the layer. */
                                   wNumCardTableStorageEntries);  /**< [In] Number of possible Card table entries in the storage. */
    CHECK_SUCCESS(status);

    /**
	 * AV1 or AV2?
	 */
    if (bSamHostMode == SAM_AV1)
    {
        /**
		 * Plain Mode
		 * Perform HOST AUTHENTICATION
		 */
		status = phhalHw_SamAV2_Cmd_SAM_AuthenticateHost(&halSamLC1,  /**< Pointer to this layer's parameter structure. */
					  					    0x00,       /**< AuthType: Default */
										    0,          /**< Host Key Number */
										    0x00,       /**< Host Key Version */
										    0x00,       /**< Sam Key Number */
										    0x00,       /**< Sam Key Version */
										    &div,       /**< No Diversification input */
											0);         /**< Diversification length */
		CHECK_SUCCESS(status);
	}
    else
    {
	    /**
		 * Perform HOST AUTHENTICATION
		 */
		status = phhalHw_SamAV2_Cmd_SAM_AuthenticateHost(&halSamLC1,  /**< Pointer to this layer's parameter structure. */
											authtype,   /**< AuthType: Default */
											0,          /**< Host Key Number */
											0x00,       /**< Host Key Version */
											0x00,       /**< Sam Key Number */
											0x00,       /**< Sam Key Version */
											&div,       /**< No Diversification input */
											0);         /**< Diversification length */
		CHECK_SUCCESS(status);
	}

	printf("----------------\n");
	printf("Card Example: MF Plus (crypto in SW)\n");
    printf("----------------\n");
	printf("\n\n");

	/**
	 * MIFARE Plus SL3 (AES)
     * Activate Idle + RATS + Authenticate Key A (= all bytes FF), Block 0 + Read Block 0
	 */

    /**
     * init. MIFARE Plus X-mode
     */
    status = phalMfp_SamAV2_X_Init(&alMfp_X,        /**< [In] MIFARE Plus Software parameter structure */
                                   sizeof(phalMfp_SamAV2_X_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
                                   &halSamLC0,
	  							   &palMifare_X); /**< [In] Pointer to a palMifare component context. */
    CHECK_SUCCESS(status);

	/**
	 * Perform RF-Reset
	 */
	status = phhalHw_FieldReset(&halSamLC0);    /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
	CHECK_SUCCESS(status);

	/**
	 * Configure Hardware for Type-A cards
	 */
	status = phhalHw_ApplyProtocolSettings(&halSamLC0,      /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
										   PHHAL_HW_CARDTYPE_ISO14443A);  /**< [In] Type of card for which the hal should be configured for. */
	CHECK_SUCCESS(status);

  /**
   * Activate Layer 3 card, Activate card I14443p4a, switch to ISO4, get and Set the ISO14443-4 protocol parameters
   */
  status = ActivateCard(&I14443p3a_X,    /**< [In] PAL-ISO14443P3A parameter structure */
                        &I14443p4a_X,    /**< [In] PAL-ISO14443P4A parameter structure */
                        PHPAL_I14443P4A_DATARATE_106,  /**< DRI/DSI value for 106 kBit/s */
                        PHPAL_I14443P4A_DATARATE_106); /**< DRI/DSI value for 106 kBit/s */
  CHECK_SUCCESS(status);

  printf("Preform the examples: \n");

  /**
   * 1. Personalization
   */
  printf("1. Personalisation... ");
  status = Personalization(&alMfp_X);      /**< [In] MIFARE Plus Software parameter structure */
  CHECK_SUCCESS(status);

  /**
   * Switch to Level 1
   */
  printf("   Switch to security level 1... ");
  status = phalMfp_CommitPerso(&alMfp_X,       /**< [In] MIFARE Plus Software parameter structure */
                               ISO14443_4);  /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol; */
  CHECK_SUCCESS(status);

  /**
   * Field Reset
   */
  status = phhalHw_FieldReset(&halSamLC0);
  CHECK_SUCCESS(status);

  /**
   * Activate Layer 3 card
   */
  status = phpalI14443p3a_ActivateCard(&I14443p3a_X,  /**< [In] PAL-ISO14443P3A parameter structure */
                                       NULL,           /**< [In] unknown UID */
                                       0x00,           /**< [In] UID length 0 */
                                       aUid,           /**< [Out] complete Uid; uint8_t[7]. */
                                       &bLength,       /**< [Out] Length of Uid; 4/7/10. */
                                       aSak,           /**< [Out] Select Acknowledge; uint8_t. */
                                       &bMoreCardsAvailable);  /**< [Out] Whether there are more cards in the field or not; uint8_t. */
  CHECK_SUCCESS(status);
  printf("(Success)\n");

  /**
   * 2. Security Level 1 Example (MIFARE Classic compatible)
   */
  printf("2. Security Level 1 Example... ");
  status = SecurityLevel1(&alMfp_X,      /**< [In] MIFARE Plus Software parameter structure */
                          &alMfc_X,      /**< [In] MIFARE Classic parameter structure. */
                          pKeyStore,   /**< [In] Pointer to the parameter structure of the KeyStore component. */
                          aUid,        /**< [In] complete Uid; uint8_t[7]. */
                          bLength);    /**< [In] Uid Length. */
  CHECK_SUCCESS(status);

  /**
   * Field Reset
   */
  status = phhalHw_FieldReset(&halSamLC0);
  CHECK_SUCCESS(status);

  /**
   * Activate Layer 3 card, Activate card I14443p4a, switch to ISO4, get and Set the ISO14443-4 protocol parameters
   */
  status = ActivateCard(&I14443p3a_X,    /**< [In] PAL-ISO14443P3A parameter structure */
                        &I14443p4a_X,    /**< [In] PAL-ISO14443P4A parameter structure */
                        PHPAL_I14443P4A_DATARATE_106,  /**< DRI/DSI value for 106 kBit/s */
                        PHPAL_I14443P4A_DATARATE_106); /**< DRI/DSI value for 106 kBit/s */
  CHECK_SUCCESS(status);

  /**
   * Load key
   * First format the key entry
   */
  status = phKeyStore_FormatKeyEntry(pKeyStore,         /**< [In] Pointer to the parameter structure of the KeyStore component. */
                                     SL_2_KEY_ADDRESS,  /**< [In] KeyEntry number to be Formatted. */
                                     PH_KEYSTORE_KEY_TYPE_AES128);  /**< [In] New Key type of the KeyEntry (predefined type of KeyType).*/
  CHECK_SUCCESS(status);
  /**
   * Second set the key at position
   */
  status = phKeyStore_SetKeyAtPos(pKeyStore,          /**< [In] Pointer to the parameter structure of the KeyStore component. */
                                  SL_2_KEY_ADDRESS,   /**< [In] KeyEntry number to be Formatted. */
                                  KEY_POS,            /**< [In] Key position to be updated. */
                                  PH_KEYSTORE_KEY_TYPE_AES128,  /**< [In] New Key type of the KeyEntry (predefined type of KeyType).*/
                                  gaLevel2SwitchKey,  /**< [In] Pointer to the key itself.*/
                                  SL_2_KEY_VERSION);  /**< [In] New Key version of the key to be updated. */
  CHECK_SUCCESS(status);

  /**
   * Switch to Level 2
   */
  printf("   Switch to security level 2... ");
  status = phalMfp_AuthenticateSL1(&alMfp_X,
                                   ISO14443_4,         /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol; */
                                   FOLLOWING_AUTH,     /**< [In] \c 0: Following Authentication; \c 1: First Authentication; */
                                   LEVEL_2_SWITCH_KEY, /**< [In] Key Block number. */
                                   SL_2_KEY_ADDRESS,   /**< [In] Key Storage number. */
                                   SL_2_KEY_VERSION,   /**< [In] Key Storage version. */
                                   0,                  /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
                                   NULL,               /**< [In] Diversification Input used to diversify the key. */
                                   0,                  /**< [In] Length of the supplied PCDCaps. */
                                   NULL,               /**< [In] Pointer to PCDCaps (bLenPcdCap2 bytes), ignored if bLenPcdCap2 == 0. */
                                   aPcdCap2Out,        /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                   aPdCap2);           /**< [Out] Pointer to PDCaps sent from the card (6 bytes). */
  CHECK_SUCCESS(status);
  printf("(Success)\n");

  /**
   * Field Reset
   */
  status = phhalHw_FieldReset(&halSamLC0);
  CHECK_SUCCESS(status);

  /**
   * Activate Layer 3 card
   */
  status = phpalI14443p3a_ActivateCard(&I14443p3a_X,  /**< [In] PAL-ISO14443P3A parameter structure */
                                       NULL,           /**< [In] unknown UID */
                                       0x00,           /**< [In] UID length 0 */
                                       aUid,           /**< [Out] complete Uid; uint8_t[7]. */
                                       &bLength,       /**< [Out] Length of Uid; 4/7/10. */
                                       aSak,           /**< [Out] Select Acknowledge; uint8_t. */
                                       &bMoreCardsAvailable);  /**< [Out] Whether there are more cards in the field or not; uint8_t. */
  CHECK_SUCCESS(status);


  /**
   * 3. Security Level 2 Example (MIFARE Classic migration)
   */
  printf("3. Security Level 2 Example... ");
  status = SecurityLevel2(&alMfp_X,    /**< [In] MIFARE Plus Software parameter structure */
                          &alMfc_X,    /**< [In] MIFARE Classic parameter structure. */
                          pKeyStore, /**< [In] Pointer to the parameter structure of the KeyStore component. */
                          aUid,      /**< [In] complete Uid; uint8_t[7]. */
                          bLength);  /**< [In] Uid Length. */
  CHECK_SUCCESS(status);

  /**
   * Field Reset
   */
  status = phhalHw_FieldReset(&halSamLC0);
  CHECK_SUCCESS(status);

  /**
   * Activate Layer 3 card, Activate card I14443p4a, switch to ISO4, get and Set the ISO14443-4 protocol parameters
   */
  status = ActivateCard(&I14443p3a_X,    /**< [In] PAL-ISO14443P3A parameter structure */
                        &I14443p4a_X,    /**< [In] PAL-ISO14443P4A parameter structure */
                        PHPAL_I14443P4A_DATARATE_106,  /**< DRI/DSI value for 106 kBit/s */
                        PHPAL_I14443P4A_DATARATE_106); /**< DRI/DSI value for 106 kBit/s */
  CHECK_SUCCESS(status);

  /**
   * Load key(SecurityLevel3Key)
   * First format the key entry
   */
  status = phKeyStore_FormatKeyEntry(pKeyStore,         /**< [In] Pointer to the parameter structure of the KeyStore component. */
                                     SL_3_KEY_ADDRESS,  /**< [In] KeyEntry number to be Formatted. */
                                     PH_KEYSTORE_KEY_TYPE_AES128);  /**< [In] New Key type of the KeyEntry (predefined type of KeyType).*/
  CHECK_SUCCESS(status);
  /**
   * Second set the key at position
   */
  status = phKeyStore_SetKeyAtPos(pKeyStore,          /**< [In] Pointer to the parameter structure of the KeyStore component. */
                                  SL_3_KEY_ADDRESS,   /**< [In] KeyEntry number to be Formatted. */
                                  KEY_POS,            /**< [In] Key position to be updated. */
                                  PH_KEYSTORE_KEY_TYPE_AES128,  /**< [In] New Key type of the KeyEntry (predefined type of KeyType).*/
                                  gaLevel3SwitchKey,  /**< [In] Pointer to the key itself.*/
                                  SL_3_KEY_VERSION);  /**< [In] New Key version of the key to be updated. */
  CHECK_SUCCESS(status);

  /**
   * Switch to Level 3
   */
  printf("   Switch to security level 3... ");
  status = phalMfp_AuthenticateSL2(&alMfp_X,            /**< [In] MIFARE Plus Software parameter structure */
                                   ISO14443_4,        /**< [In] \c 0: use ISO14443-3 protocol; \c 1: use ISO14443-4 protocol; */
                                   FOLLOWING_AUTH,    /**< [In] \c 0: Following Authentication; \c 1: First Authentication; */
                                   LEVEL_3_SWITCH_KEY,/**< [In] Key Block number. */
                                   SL_3_KEY_ADDRESS,  /**< [In] Key Storage number. */
                                   SL_3_KEY_VERSION,  /**< [In] Key Storage version. */
                                   0,                 /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
                                   NULL,              /**< [In] Diversification Input used to diversify the key. */
                                   0,                 /**< [In] Length of the supplied PCDCaps. */
                                   NULL,              /**< [In] Pointer to PCDCaps (bLenPcdCap2 bytes), ignored if bLenPcdCap2 == 0. */
                                   aPcdCap2Out,       /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                   aPdCap2,           /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                   aKmf);             /**< [Out] MIFARE Sector Key Modifier (6 bytes). */
  CHECK_SUCCESS(status);
  printf("(Success)\n");

  /**
   * Field Reset
   */
  status = phhalHw_FieldReset(&halSamLC0);
  CHECK_SUCCESS(status);

  /**
   * Activate Layer 3 card, Activate card I14443p4a, switch to ISO4, get and Set the ISO14443-4 protocol parameters
   */
  status = ActivateCard(&I14443p3a_X,    /**< [In] PAL-ISO14443P3A parameter structure */
                        &I14443p4a_X,    /**< [In] PAL-ISO14443P4A parameter structure */
                        PHPAL_I14443P4A_DATARATE_106,  /**< DRI/DSI value for 106 kBit/s */
                        PHPAL_I14443P4A_DATARATE_106); /**< DRI/DSI value for 106 kBit/s */
  CHECK_SUCCESS(status);

  /**
   * 4. Security Level 3 Example
   */
  printf("4. Security Level 3 Example... ");
  status = SecurityLevel3(&alMfp_X,         /**< [In] MIFARE Plus Software parameter structure */
                          &halSamLC0,           /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
                          pKeyStore,      /**< [In] Pointer to the parameter structure of the KeyStore component. */
                          &I14443p3a_X,  /**< [In] PAL-ISO14443P3A parameter structure */
                          &I14443p4a_X);  /**< [In] PAL-ISO14443P4A parameter structure */
  CHECK_SUCCESS(status);

  /**
   * Field Reset
   */
  status = phhalHw_FieldReset(&halSamLC0);
  CHECK_SUCCESS(status);

  /**
   * Activate Layer 3 card, Activate card I14443p4a, switch to ISO4, get and Set the ISO14443-4 protocol parameters
   */
  status = ActivateCard(&I14443p3a_X,    /**< [In] PAL-ISO14443P3A parameter structure */
                        &I14443p4a_X,    /**< [In] PAL-ISO14443P4A parameter structure */
                        PHPAL_I14443P4A_DATARATE_106,  /**< DRI/DSI value for 106 kBit/s */
                        PHPAL_I14443P4A_DATARATE_106); /**< DRI/DSI value for 106 kBit/s */
  CHECK_SUCCESS(status);

  /**
   * Virtual Card Example
   */
  printf("5. Virtual Card Example... ");
  status = VirtualCard(&alMfp_X,      /**< [In] MIFARE Plus Software parameter structure */
                       &alVca,      /**< [In] VCA Software parameter structure */
                       pKeyStore);  /**< [In] Pointer to the parameter structure of the KeyStore component. */
  CHECK_SUCCESS(status);

    /**
	 * Switch off the field
	 */
    status = phhalHw_FieldOff(&halSamLC0);
    CHECK_SUCCESS(status);

    /**
	 * we are finished, release bal handle
	 */
    status = phbalReg_ClosePort(&balPcsc);
    CHECK_SUCCESS(status);

    printf("-------------------------\n");
    printf("No more tests available.\n");
    printf("Press any key to continue...\n\n");
    _getch();

    /**
	 * Returns zero if the procedure was successful.
	 */
    return 0;
}

