/*
 * Copyright 2013, 2015, 2017, 2020, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic KeyStore Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHKEYSTORE_H
#define PHKEYSTORE_H

#include <ph_Status.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PH_KEYSTORE_SW

#define PH_KEYSTORE_SW_ID                                       0x01U   /**< ID for Software KeyStore component. */

/**
 * \defgroup phKeyStore_Sw Component : Software
 * \brief KeyStore implementation in Software.
 * @{
 */

#ifdef NXPBUILD__PH_KEYSTORE_ASYM
#define PH_KEYSTORE_SW_MAX_KEY_SIZE                             48U     /**< Maximum size of a Symmetric and ASymetric Key. */
#endif /* NXPBUILD__PH_KEYSTORE_ASYM */

#ifndef NXPBUILD__PH_KEYSTORE_ASYM
#define PH_KEYSTORE_SW_MAX_KEY_SIZE                             32U     /**< Maximum size of a Symmetric Key. */
#endif /* NXPBUILD__PH_KEYSTORE_ASYM */

#define PH_KEYSTORE_MAX_KEY_SIZE        PH_KEYSTORE_SW_MAX_KEY_SIZE     /**< Maximum size of a Symmetric and ASymetric Key. */

/** \brief Software KeyVersionPair structure for Symmetric and ASymmetric keys. */
typedef struct
{
    uint8_t pKey[PH_KEYSTORE_MAX_KEY_SIZE];                             /**< Array containing a Symmetric Key or ASymmetric Private Key. */
    uint16_t wVersion;                                                  /**< Versions related to the Symmetric Key. Not applicable for ASymmetric keys storage. */

#ifdef NXPBUILD__PH_KEYSTORE_ASYM
    uint8_t pPubKey[(PH_KEYSTORE_MAX_KEY_SIZE * 2) + 1];                /**< Array containing a ASymmetric Public Key including the Un-compressed notation. */
    uint16_t wKeyPairType;                                              /**< KeyPair being used for key to be updated. */
    uint8_t bCurveID;                                                   /**< Curve Type of Private / Public Key. One of the values mentioned in
                                                                         *   \ref phKeyStore_ASym_Defines_CurveID "ECC CurveID's"
                                                                         */
#endif /* NXPBUILD__PH_KEYSTORE_ASYM */
} phKeyStore_Sw_KeyVersionPair_t;

/** \brief Software KeyEntry structure. */
typedef struct
{
    uint16_t wKeyType;                                                  /**< Type of the keys in \ref phKeyStore_Sw_KeyVersionPair_t
                                                                         *   "Key-Version Pair".
                                                                         */
    uint16_t wRefNoKUC;                                                 /**< Key usage counter number of the keys in pKeys. */
} phKeyStore_Sw_KeyEntry_t;

/** \brief Software KeyUsageCounter structure. */
typedef struct
{
    uint32_t dwLimit;                                                   /**< Limit of the Key Usage Counter. */
    uint32_t dwCurVal;                                                  /**< Current Value of the KUC. */
} phKeyStore_Sw_KUCEntry_t;

/** \brief Software parameter structure. */
typedef struct
{
    uint16_t  wId;                                                      /**< Layer ID for this component, NEVER MODIFY! */
    phKeyStore_Sw_KeyEntry_t * pKeyEntries;                             /**< Key entry storage, size = sizeof(\ref phKeyStore_Sw_KeyEntry_t "Key Entry") *
                                                                         *   \b wNoOfKeyEntries.
                                                                         */
    phKeyStore_Sw_KeyVersionPair_t * pKeyVersionPairs;                  /**< Key version pairs, size = sizeof(\ref phKeyStore_Sw_KeyVersionPair_t "Key Entry") *
                                                                         *   \b wNoOfKeyEntries * \b wNoOfVersions.
                                                                         */
    uint16_t wNoOfKeyEntries;                                           /**< Number of key entries in that storage. */
    uint16_t wNoOfVersions;                                             /**< Number of versions in each key entry. */
    phKeyStore_Sw_KUCEntry_t * pKUCEntries;                             /**< Key usage counter entry storage, size = sizeof(\ref phKeyStore_Sw_KUCEntry_t "KUC Entry") *
                                                                         *   \b wNoOfKUCEntries.
                                                                         */
    uint16_t wNoOfKUCEntries;                                           /**< Number of Key usage counter entries. */
} phKeyStore_Sw_DataParams_t;

/**
 * \brief Initializes the KeyStore component as software component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phKeyStore_Sw_Init(
        phKeyStore_Sw_DataParams_t * pDataParams,                       /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                     /**< [In] Specifies the size of the data parameter structure. */
        phKeyStore_Sw_KeyEntry_t * pKeyEntries,                         /**< [In] Pointer to a storage containing the key entries. */
        uint16_t wNoOfKeyEntries,                                       /**< [In] Size of \b pKeyEntries. */
        phKeyStore_Sw_KeyVersionPair_t * pKeyVersionPairs,              /**< [In] Pointer to a storage containing the key version pairs. */
        uint16_t wNoOfVersionPairs,                                     /**< [In] Amount of key versions available in each key entry. */
        phKeyStore_Sw_KUCEntry_t * pKUCEntries,                         /**< Key usage counter entry storage, size = sizeof(\ref phKeyStore_Sw_KUCEntry_t "KUC Entry") *
                                                                         *   \b wNoOfKUCEntries.
                                                                         */
        uint16_t wNoOfKUCEntries                                        /**< [In] Number of Key usage counter entries. */
    );
/**
 * end of group phKeyStore_Sw
 * @}
 */
#endif /* NXPBUILD__PH_KEYSTORE_SW */

#ifdef NXPBUILD__PH_KEYSTORE_RC663

#include <phhalHw.h>

#define PH_KEYSTORE_RC663_ID                                    0x02U   /**< ID for Rc663 KeyStore component. */

/**
 * \defgroup phKeyStore_Rc663 Component : Rc663
 * \brief KeyStore implementation to use RC663 Hardware KeyStore.
 * @{
 */
#define PH_KEYSTORE_RC663_NUM_KEYS                              0x80U   /**< Maximum number of keys storable in Rc663. */
#define PH_KEYSTORE_RC663_NUM_VERSIONS                          0x01U   /**< Amount of versions for each key entry in the key store. */

/** \brief Rc663 parameter structure. */
typedef struct
{
    uint16_t  wId;                                                      /**< Layer ID for this component, NEVER MODIFY! */
    void * pHalDataParams;                                              /**< Pointer to the parameter structure of the underlying layer. */
} phKeyStore_Rc663_DataParams_t;

/**
 * \brief Initializes the KeyStore component as RC663 component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phKeyStore_Rc663_Init(
        phKeyStore_Rc663_DataParams_t * pDataParams,                    /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                     /**< [In] Specifies the size of the data parameter structure. */
        void * pHalDataParams                                           /**< [In] Pointer to a HAL based on NXP RC663 IC. */
    );
/**
 * end of group phKeyStore_Rc663
 * @}
 */
#endif /* NXPBUILD__PH_KEYSTORE_RC663 */

#ifdef NXPBUILD__PH_KEYSTORE_RC632

#include <phhalHw.h>

#define PH_KEYSTORE_RC632_ID                                    0x03U   /**< ID for Rc632 KeyStore component. */
/**
 * \defgroup phKeyStore_Rc632 Component : Rc632
 * \brief KeyStore implementation to use RC632 Hardware KeyStore.
 * @{
 */
#define PH_KEYSTORE_RC632_NUM_KEYS                              16U     /**< Maximum number of keys storable in Rc632. */
#define PH_KEYSTORE_RC632_NUM_VERSIONS                          0x01U   /**< Amount of versions for each key entry in the key store. */

/** \brief Rc632 parameter structure */
typedef struct
{
    uint16_t  wId;                                                      /**< Layer ID for this component, NEVER MODIFY! */
    void * pHalDataParams;                                              /**< Pointer to the parameter structure of the underlying layer. */
} phKeyStore_Rc632_DataParams_t;

/**
 * \brief Initializes the KeyStore component as RC632 component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phKeyStore_Rc632_Init(
        phKeyStore_Rc632_DataParams_t * pDataParams,                    /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                     /**< [In] Specifies the size of the data parameter structure. */
        void * pHalDataParams                                           /**< [In] Pointer to a HAL based on NXP RC632 IC. */
    );
/**
 * end of group phKeyStore_Rc632
 * @}
 */
#endif /* NXPBUILD__PH_KEYSTORE_RC632 */

#ifdef NXPBUILD__PH_KEYSTORE_RD710
#include <phhalHw.h>

#define PH_KEYSTORE_RD710_ID                                    0x04U   /**< ID for Rd710 KeyStore component. */

/**
 * \defgroup phKeyStore_Rd710 Component : Rd710
 * \brief KeyStore implementation to use RD710 Hardware KeyStore.
 * @{
 */

/** \brief Rd710 parameter structure */
typedef struct
{
    uint16_t  wId;                                                      /**< Layer ID for this component, NEVER MODIFY! */
    phhalHw_Rd710_DataParams_t * pHalDataParams;                        /**< Pointer to the parameter structure of the underlying layer.*/
} phKeyStore_Rd710_DataParams_t;

/**
 * \brief Initializes the KeyStore component as RD710 component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phKeyStore_Rd710_Init(
        phKeyStore_Rd710_DataParams_t * pDataParams,                    /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                     /**< [In] Specifies the size of the data parameter structure. */
        phhalHw_Rd710_DataParams_t * pHalDataParams                     /**< [In] Pointer to a HAL based on NXP RD710 IC. */
    );
/**
 * end of group phKeyStore_Rd710
 * @}
 */
#endif /* NXPBUILD__PH_KEYSTORE_RD710 */

#ifdef NXPBUILD__PH_KEYSTORE_SAMAV2

#include <phhalHw.h>

#define PH_KEYSTORE_SAMAV2_ID                                   0x05U   /**< ID for SamAV2 KeyStore component. */

/**
 * \defgroup phKeyStore_SamAV2 Component : SamAV2
 * \brief SamAV2 implementation of the phSam interface
 * @{
 */

#define PH_KEYSTORE_SAMAV2_AV1_MODE                             0x01U   /**< Define the AV1 mode of the KeyStore */
#define PH_KEYSTORE_SAMAV2_AV2_MODE                             0x02U   /**< Define the AV2 mode of the KeyStore */

/** \name Key Classes */
/** @{ */
#define PH_KEYSTORE_SAMAV2_KEY_CLASS_HOST                       0x00U   /**< Host key: has to be either AES128 or AES192. */
#define PH_KEYSTORE_SAMAV2_KEY_CLASS_PICC                       0x01U   /**< PICC key: has to be either AES128, single DES, 2 Key Triple DES, 3 Key Triple DES or MIFARE. */
#define PH_KEYSTORE_SAMAV2_KEY_CLASS_OFFLINE_CHANGE             0x02U   /**< Offline change key: has to be either AES128 or AES192. */
#define PH_KEYSTORE_SAMAV2_KEY_CLASS_OFFLINE_CRYPTO             0x04U   /**< Offline crypto key: has to be either AES128, AES192, single DES with CRC32 protection, 2 Key Triple DES with CRC32 protection or 3 Key Triple DES. */
/** @} */

/** \name DES Key Options */
/** @{ */
#define PH_KEYSTORE_SAMAV2_DES_OPTION_DESFIRE4                  0x00U    /**< DESFire 4 compatibility mode. */
#define PH_KEYSTORE_SAMAV2_DES_OPTION_ISO_CRC16                 0x01U    /**< ISO 10116 mode with CRC16 protection and 4 bytes MAC. */
#define PH_KEYSTORE_SAMAV2_DES_OPTION_ISO_CRC32                 0x02U    /**< ISO 10116 mode with CRC32 protection and 8 bytes MAC. */
/** @} */

/**
* \name KeyStore Configs
*/
/** @{ */
#define PH_KEYSTORE_SAMAV2_CONFIG_ALLOW_DUMP_SESSION_KEY        0x0000U /**< Enable or Disable SAM_ChangeKeyMIFARE and SAM_DumpSessionKey command. */
#define PH_KEYSTORE_SAMAV2_CONFIG_ALLOW_CRYPTO_SECRET_KEY       0x0001U /**< Enable or Disable crypto based on the secret key for AV1 mode. */
#define PH_KEYSTORE_SAMAV2_CONFIG_ALLOW_DUMP_SECRET_KEY         0x0002U /**< Enable or Disable SAM_DumpSecretKey command for AV2 mode. */
#define PH_KEYSTORE_SAMAV2_CONFIG_KEEP_IV                       0x0003U /**< Enable or Disable the reset of init vector after a crypto command. */
#define PH_KEYSTORE_SAMAV2_CONFIG_ENABLE_HIGH_SEC_LEVEL         0x0004U /**< Enable or Disable Higher Command Security Level for AV1 Master Key. */
#define PH_KEYSTORE_SAMAV2_CONFIG_ENABLE_HOST_AUTH              0x0005U /**< Enable or Disable Host Authentication for AV2 mode. */
#define PH_KEYSTORE_SAMAV2_CONFIG_ENABLE_HOST_AUTH_AFTER_RESET  0x0006U /**< Enable or Disable Host Authentication after reset for AV1 Master Key. */
#define PH_KEYSTORE_SAMAV2_CONFIG_ENABLE_LOCK_UNLOCK            0x0007U /**< Enable or Disable LockUnlock for AV2 mode. */
#define PH_KEYSTORE_SAMAV2_CONFIG_DISABLE_CHANGE_KEY_PICC       0x0008U /**< Enable or Disable SAM_ChangeKeyPICC command. */
#define PH_KEYSTORE_SAMAV2_CONFIG_DISABLE_ENCRYPTION            0x0009U /**< Enable or Disable SAM_DecipherData command. */
#define PH_KEYSTORE_SAMAV2_CONFIG_DISABLE_DECRYPTION            0x000AU /**< Enable or Disable SAM_EncipherData command. */
#define PH_KEYSTORE_SAMAV2_CONFIG_DISABLE_VERIFY_MAC            0x000BU /**< Enable or Disable SAM_VerifyMAC command. */
#define PH_KEYSTORE_SAMAV2_CONFIG_DISABLE_GENERATE_MAC          0x000CU /**< Enable or Disable SAM_GenerateMAC command. */
#define PH_KEYSTORE_SAMAV2_CONFIG_MANDATE_KEY_DIVERSIFICATION   0x000DU /**< Mandate or not key diversification. */
#define PH_KEYSTORE_SAMAV2_CONFIG_CHANGE_KEY_PLAIN              0x000EU /**< Enable plain exchange of ChangeKey functions for AV1 mode. */
#define PH_KEYSTORE_SAMAV2_CONFIG_KEYCLASS                      0x000FU /**< Key Class used in key store. */
#define PH_KEYSTORE_SAMAV2_CONFIG_KEYNO_CEK                     0x0010U /**< Key Number of Change Entry key. */
#define PH_KEYSTORE_SAMAV2_CONFIG_KEYV_CEK                      0x0011U /**< Key Version of Change Entry key. */
#define PH_KEYSTORE_SAMAV2_CONFIG_KEYNO_CKUC                    0x0012U /**< Key Number of Change  KUC. */
#define PH_KEYSTORE_SAMAV2_CONFIG_KEYV_CKUC                     0x0013U /**< Key Version of Change KUC. */
#define PH_KEYSTORE_SAMAV2_CONFIG_DISABLE_KEY_ENTRY             0x0014U /**< Enable or Disable Key Entry. */
#define PH_KEYSTORE_SAMAV2_CONFIG_DF_AID                        0x0015U /**< MIFARE DESFire application ID. */
#define PH_KEYSTORE_SAMAV2_CONFIG_DF_KEY_NO                     0x0016U /**< MIFARE DESFire key number. */
#define PH_KEYSTORE_SAMAV2_CONFIG_REF_NO_KUC                    0x0017U /**< Reference number of key usage counter. */
#define PH_KEYSTORE_SAMAV2_CONFIG_DES_KEY_OPTION                0x0018U /**< Option for single DES and 2 Key Triple DES keys. */
#define PH_KEYSTORE_SAMAV2_CONFIG_KEYNO_MF_DIV_A                0x0019U /**< Key Number used for MIFARE key A diversification. */
#define PH_KEYSTORE_SAMAV2_CONFIG_KEYV_MF_DIV_A                 0x001AU /**< Key Version used for MIFARE key A diversification. */
#define PH_KEYSTORE_SAMAV2_CONFIG_KEYNO_MF_DIV_B                0x001BU /**< Key Number used for MIFARE key B diversification. */
#define PH_KEYSTORE_SAMAV2_CONFIG_KEYV_MF_DIV_B                 0x001CU /**< Key Version used for MIFARE key B diversification. */
/** @} */

/** \brief Sam parameter structure */
typedef struct
{
    uint16_t  wId;                                                      /**< Layer ID for this component, NEVER MODIFY! */
    phhalHw_SamAV2_DataParams_t * pHalDataParams;                       /**< Pointer to the parameter structure of the underlying layer.*/
    uint8_t pSet[2];                                                    /**< Configuration settings. */
    uint8_t bExtSet;                                                    /**< Extended configuration settings. */
    uint8_t bChangeKeyPlain;                                            /**< Use plain transmission for ChangeKey operations in AV1 mode. */
    uint8_t pDFAid[3];                                                  /**< MIFARE DESFire application ID. */
    uint8_t bDFKeyNo;                                                   /**< MIFARE DESFire key number. */
    uint8_t bKeyNoCEK;                                                  /**< Key Number of Change Entry Key. */
    uint8_t bKeyVCEK;                                                   /**< Key Version of Change Entry Key. */
    uint8_t bRefNoKUC;                                                  /**< Reference number of key usage counter. */
    uint8_t bKeyNoCKUC;                                                 /**< Key Number of Change KUC. */
    uint8_t bKeyVCKUC;                                                  /**< Key Version of Change KUC. */
    uint8_t bKeyNoMfDivA;                                               /**< Key Number used for MIFARE key A diversification (has to point to a DES key). */
    uint8_t bKeyVMfDivA;                                                /**< Key Version used for MIFARE key A diversification (has to point to a DES key). */
    uint8_t bKeyNoMfDivB;                                               /**< Key Number used for MIFARE key B diversification (has to point to a DES key). */
    uint8_t bKeyVMfDivB;                                                /**< Key Version used for MIFARE key B diversification (has to point to a DES key). */
    uint8_t b2K3DESOption;                                              /**< Option for single DES and 2 Key Triple DES keys. Can be set either to
                                                                         *   - #PH_KEYSTORE_SAMAV2_DES_OPTION_DESFIRE4
                                                                         *   - #PH_KEYSTORE_SAMAV2_DES_OPTION_ISO_CRC16
                                                                         *   - #PH_KEYSTORE_SAMAV2_DES_OPTION_ISO_CRC32
                                                                         */
} phKeyStore_SamAV2_DataParams_t;

/**
 * \brief Initializes the KeyStore component as SAM AV2 component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phKeyStore_SamAV2_Init(
        phKeyStore_SamAV2_DataParams_t * pDataParams,                   /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                     /**< [In] Specifies the size of the data parameter structure. */
        phhalHw_SamAV2_DataParams_t * pHalDataParams                    /**< [In] Pointer to the parameter structure of the underlying layer. */
    );
/**
 * end of group phKeyStore_SamAV2
 * @}
 */
#endif /* NXPBUILD__PH_KEYSTORE_SAMAV2 */

#ifdef NXPBUILD__PH_KEYSTORE_SAMAV3

#include <phhalHw.h>

#define PH_KEYSTORE_SAMAV3_ID                                       0x06U       /**< ID for SamAV3 KeyStore component. */

/**
 * \defgroup phKeyStore_SamAV3 Component : SamAV3
 * \brief SamAV3 implementation of the phSam interface
 * @{
 */

#define PH_KEYSTORE_SAMAV3_AV2_MODE                                 0x02U       /**< Define the AV2 mode of the KeyStore */
#define PH_KEYSTORE_SAMAV3_AV3_MODE                                 0x03U       /**< Define the AV3 mode of the KeyStore */

/** \name Key Classes. Should be used to set the ExtSET information. */
/** @{ */
#define PH_KEYSTORE_SAMAV3_KEY_CLASS_HOST                           0x00U       /**< Configuring key entry as Host. */
#define PH_KEYSTORE_SAMAV3_KEY_CLASS_PICC                           0x01U       /**< Configuring key entry as PICC. */
#define PH_KEYSTORE_SAMAV3_KEY_CLASS_OFFLINE_CHANGE                 0x02U       /**< Configuring key entry as Offline Change. */
#define PH_KEYSTORE_SAMAV3_KEY_CLASS_OFFLINE_CRYPTO                 0x04U       /**< Configuring key entry as Offline Crypto. */
#define PH_KEYSTORE_SAMAV3_KEY_CLASS_OFFLINE_UPLOAD                 0x05U       /**< Configuring key entry as Offline Upload. */
#define PH_KEYSTORE_SAMAV3_KEY_CLASS_OFFLINE_PERSO                  0x06U       /**< Configuring key entry as Offline Perso. */
/** @} */

/** \name DES Key Options */
/** @{ */
#define PH_KEYSTORE_SAMAV3_DES_OPTION_DESFIRE4                      0x00U       /**< DESFire 4 compatibility mode. */
#define PH_KEYSTORE_SAMAV3_DES_OPTION_ISO_CRC16                     0x01U       /**< ISO 10116 mode with CRC16 protection and 4 bytes MAC. */
#define PH_KEYSTORE_SAMAV3_DES_OPTION_ISO_CRC32                     0x02U       /**< ISO 10116 mode with CRC32 protection and 8 bytes MAC. */
/** @} */

/** \name KeyStore Configs for SET configurations. */
/** @{ */
#define PH_KEYSTORE_SAMAV3_CONFIG_ALLOW_DUMP_SESSION_KEY            0x0000U     /**< Enable or Disable SAM_ChangeKeyMIFARE and SAM_DumpSessionKey command. */
#define PH_KEYSTORE_SAMAV3_CONFIG_KEEP_IV                           0x0001U     /**< Enable or Disable the reset of init vector after a crypto command. */
#define PH_KEYSTORE_SAMAV3_CONFIG_PL_KEY                            0x0002U     /**< Enable or Disable the Host key type to provide permissions for Cmd.PLExec execution. */
#define PH_KEYSTORE_SAMAV3_CONFIG_AUTH_KEY                          0x0003U     /**< Enable or Disable Host Authentication with key other that MasterKey. */
#define PH_KEYSTORE_SAMAV3_CONFIG_DISABLE_KEY_ENTRY                 0x0004U     /**< Enable or Disable Key Entry. */
#define PH_KEYSTORE_SAMAV3_CONFIG_LOCK_KEY                          0x0005U     /**< Enable or Disable LockUnlock. */
#define PH_KEYSTORE_SAMAV3_CONFIG_DISABLE_CHANGE_KEY_PICC           0x0006U     /**< Enable or Disable writing the key to a PICC. */
#define PH_KEYSTORE_SAMAV3_CONFIG_DISABLE_DECRYPTION                0x0007U     /**< Enable or Disable SAM_DecipherData command. */
#define PH_KEYSTORE_SAMAV3_CONFIG_DISABLE_ENCRYPTION                0x0008U     /**< Enable or Disable SAM_EncipherData command. */
#define PH_KEYSTORE_SAMAV3_CONFIG_DISABLE_VERIFY_MAC                0x0009U     /**< Enable or Disable SAM_VerifyMAC command. */
#define PH_KEYSTORE_SAMAV3_CONFIG_DISABLE_GENERATE_MAC              0x000AU     /**< Enable or Disable SAM_GenerateMAC command. */
/** @} */

/** \name KeyStore Configs for ExtSET configurations. */
/** @{ */
#define PH_KEYSTORE_SAMAV3_CONFIG_KEYCLASS                          0x000BU     /**< Key Class used in key store. */
#define PH_KEYSTORE_SAMAV3_CONFIG_ALLOW_DUMP_SECRET_KEY             0x000CU     /**< Enable or Disable SAM_DumpSecretKey command. */
#define PH_KEYSTORE_SAMAV3_CONFIG_MANDATE_KEY_DIVERSIFICATION       0x000DU     /**< Mandate or not key diversification. */
#define PH_KEYSTORE_SAMAV3_CONFIG_RESERVED_SAM_PRESONALIZATION      0x000EU     /**< Enable or disable the Key Entry for Sam Personalization. */
#define PH_KEYSTORE_SAMAV3_CONFIG_KEY_USAGE_INT_HOST                0x000FU     /**< Enable or disable the Key Entry usage by Internal Host. */
#define PH_KEYSTORE_SAMAV3_CONFIG_KEY_CHANGE_INT_HOST               0x0010U     /**< Enable or disable the Key Entry change by Internal Host. */
#define PH_KEYSTORE_SAMAV3_CONFIG_SESSION_KEY_USAGE_INT_HOST        0x0011U     /**< Enable or disable the Session Key usage by Internal Host. */
#define PH_KEYSTORE_SAMAV3_CONFIG_ALLOW_DUMP_SECRET_KEY_INT_HOST    0x0012U     /**< Enable or disable the dumping of Secret Key by Internal Host. */
#define PH_KEYSTORE_SAMAV3_CONFIG_ALLOW_DUMP_SESSION_KEY_INT_HOST   0x0013U     /**< Enable or disable the dumping of Session Key by Internal Host. */
/** @} */

/** \name KeyStore Configs for SAM Key Storage Table's Key Entry. */
/** @{ */
#define PH_KEYSTORE_SAMAV3_CONFIG_DF_AID                            0x0014U     /**< DESFire application ID. */
#define PH_KEYSTORE_SAMAV3_CONFIG_DF_KEY_NO                         0x0015U     /**< DESFire key number. */
#define PH_KEYSTORE_SAMAV3_CONFIG_KEYNO_CEK                         0x0016U     /**< Key Number of Change Entry key. */
#define PH_KEYSTORE_SAMAV3_CONFIG_KEYV_CEK                          0x0017U     /**< Key Version of Change Entry key. */
#define PH_KEYSTORE_SAMAV3_CONFIG_REF_NO_KUC                        0x0018U     /**< Reference number of key usage counter. */
#define PH_KEYSTORE_SAMAV3_CONFIG_KEYNO_AEK                         0x0019U     /**< Key Number of Access Entry key. */
#define PH_KEYSTORE_SAMAV3_CONFIG_KEYV_AEK                          0x001AU     /**< Key Version of Access Entry key. */
#define PH_KEYSTORE_SAMAV3_CONFIG_KEYNO_CKUC                        0x001BU     /**< Key Number of Change KUC. */
#define PH_KEYSTORE_SAMAV3_CONFIG_KEYV_CKUC                         0x001CU     /**< Key Version of Change KUC. */
#define PH_KEYSTORE_SAMAV3_CONFIG_DES_KEY_OPTION                    0x001DU     /**< Option for single DES and 2 Key Triple DES keys. */
#define PH_KEYSTORE_SAMAV3_CONFIG_KEYNO_MF_DIV_A                    0x001EU     /**< Key Number used for MIFARE key A diversification. */
#define PH_KEYSTORE_SAMAV3_CONFIG_KEYV_MF_DIV_A                     0x001FU     /**< Key Version used for MIFARE key A diversification. */
#define PH_KEYSTORE_SAMAV3_CONFIG_KEYNO_MF_DIV_B                    0x0020U     /**< Key Number used for MIFARE key B diversification. */
#define PH_KEYSTORE_SAMAV3_CONFIG_KEYV_MF_DIV_B                     0x0021U     /**< Key Version used for MIFARE key B diversification. */
#define PH_KEYSTORE_SAMAV3_CONFIG_ENABLE_LRP                        0x0022U     /**< The AES key to be used is for LRP algorithm. */
/** @} */

/** \brief Sam parameter structure */
typedef struct
{
    uint16_t  wId;                                                              /**< Layer ID for this component, NEVER MODIFY! */
    phhalHw_SamAV3_DataParams_t * pHalDataParams;                               /**< Pointer to the parameter structure of the underlying layer.*/
    uint8_t aSet[2];                                                            /**< Configuration settings. */
    uint8_t aExtSet[2];                                                         /**< Extended configuration settings. */
    uint8_t aDFAid[3];                                                          /**< DESFire application ID. */
    uint8_t bDFKeyNo;                                                           /**< DESFire key number. */
    uint8_t bKeyNoCEK;                                                          /**< Key Number of Change Entry Key. */
    uint8_t bKeyVCEK;                                                           /**< Key Version of Change Entry Key. */
    uint8_t bRefNoKUC;                                                          /**< Reference number of key usage counter. */
    uint8_t bKeyNoAEK;                                                          /**< Key Number of Access Entry Key. */
    uint8_t bKeyVAEK;                                                           /**< Key Version of Access Entry Key. */
    uint8_t bKeyNoCKUC;                                                         /**< Key Number of Change KUC. */
    uint8_t bKeyVCKUC;                                                          /**< Key Version of Change KUC. */
    uint8_t bKeyNoMfDivA;                                                       /**< Key Number used for MIFARE key A diversification (has to point to a DES key). */
    uint8_t bKeyVMfDivA;                                                        /**< Key Version used for MIFARE key A diversification (has to point to a DES key). */
    uint8_t bKeyNoMfDivB;                                                       /**< Key Number used for MIFARE key B diversification (has to point to a DES key). */
    uint8_t bKeyVMfDivB;                                                        /**< Key Version used for MIFARE key B diversification (has to point to a DES key). */
    uint8_t b2K3DESOption;                                                      /**< Option for single DES and 2 Key Triple DES keys. Can be set either to
                                                                                 *   #PH_KEYSTORE_SAMAV3_DES_OPTION_DESFIRE4, #PH_KEYSTORE_SAMAV3_DES_OPTION_ISO_CRC16
                                                                                 *   or #PH_KEYSTORE_SAMAV3_DES_OPTION_ISO_CRC32
                                                                                 */
    uint8_t bIsLRPKey;                                                          /**< Option for LRP key type. If set indicated that the AES key is of LRP type. */
} phKeyStore_SamAV3_DataParams_t;

/**
 * \brief Initializes the KeyStore component as SAM AV3 component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phKeyStore_SamAV3_Init(
        phKeyStore_SamAV3_DataParams_t * pDataParams,                           /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                             /**< [In] Specifies the size of the data parameter structure. */
        phhalHw_SamAV3_DataParams_t * pHalDataParams                            /**< [In] Pointer to the parameter structure of the underlying layer.*/
    );
/**
 * end of group phKeyStore_SamAV3
 * @}
 */
#endif /* NXPBUILD__PH_KEYSTORE_SAMAV3 */



#ifdef NXPBUILD__PH_KEYSTORE

/** \addtogroup phKeyStore KeyStore
 *
 * \brief This is only a wrapper layer to abstract the different KeyStore implementations.
 * With this wrapper it is possible to support more than one CryptoSym implementation
 * in parallel, by adapting this wrapper.
 *
 * Important hints for users of this component:
 * - Before use of any function, the dedicated KeyStore implementation has to be initialized using either of the interfaces mentioned below.
 *      - \ref phKeyStore_Sw "Software Initialization"
 *      - \ref phKeyStore_Rc663 "RC663 Hardware KeyStore Initialization"
 *      - \ref phKeyStore_Rc632 "RC632 Hardware KeyStore Initialization"
 *      - \ref phKeyStore_Rd710 "RD710 Hardware KeyStore Initialization"
 *      - \ref phKeyStore_SamAV2 "SAM AV2 Hardware KeyStore Initialization"
 *      - \ref phKeyStore_SamAV3 "SAM AV3 Hardware KeyStore Initialization"
 * @{
 */

/**
 * \defgroup phKeyStore_Defines Defines
 * \brief Definitions for KeyStore layer.
 * @{
 */

/**
 * \defgroup phKeyStore_Defines_Config Configuration
 * \brief Definitions for KeyStore layer configuration.
 * @{
 */
#define PH_KEYSTORE_CONFIG_SET_DEFAULT                                  0xFFFFU /**< Reset all bit of SET param. */
#define PH_KEYSTORE_INVALID_ID                                          0xFFFFU /**< ID used for various parameters as a invalid default */
#define PH_KEYSTORE_DEFAULT_ID                                          0x0000U /**< ID used for various parameters as a default */
/**
 * end of group phKeyStore_Defines_Config
 * @}
 */

/**
 * end of group phKeyStore_Defines
 * @}
 */

/**
 * \brief Format a key entry to a new KeyType.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_FormatKeyEntry(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wKeyNo,                                                        /**< [In] KeyEntry number to be Formatted. */
        uint16_t wNewKeyType                                                    /**< [In] New Key type of the KeyEntry (predefined type of KeyType).
                                                                                 *           - \ref phKeyStore_Sym_Defines_KeyType "Symmetric KeyTypes"
                                                                                 *           - \ref phKeyStore_ASym_Defines_KeyType "ASymmetric KeyTypes"
                                                                                 */
    );

/**
 * \brief Change the KUC of a key entry.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_SetKUC(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wKeyNo,                                                        /**< [In] KeyEntry number. */
        uint16_t wRefNoKUC                                                      /**< [In] Reference Number of the key usage counter used together with that key.*/
    );

/**
 * \brief Obtain a key usage counter entry.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_GetKUC(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wRefNoKUC,                                                     /**< [In] Number of the key usage counter to be looked at (00h to 0Fh) */
        uint32_t * pdwLimit,                                                    /**< [Out] Currently Set Limit in the KUC */
        uint32_t * pdwCurVal                                                    /**< [Out] Currently set value in the KUC */
    );

/**
 * \brief Change a key usage counter entry.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_ChangeKUC(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wRefNoKUC,                                                     /**< [In] Number of key entry. */
        uint32_t dwLimit                                                        /**< [In] Limit of the Key Usage Counter. */
    );

/**
 * \brief Set configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_SetConfig(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                                                       /**< [In] Configuration Identifier */
        uint16_t wValue                                                         /**< [In] Configuration Value */
    );

/**
 * \brief Set configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_SetConfigStr(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                                                       /**< [In] Configuration Identifier. */
        uint8_t *pBuffer,                                                       /**< [In] Buffer containing the configuration string. */
        uint16_t wBufferLength                                                  /**< [In] Length of configuration string. */
    );

/**
 * \brief Get configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_GetConfig(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                                                       /**< [In] Configuration Identifier */
        uint16_t * pValue                                                       /**< [Out] Configuration Value */
    );

/**
 * \brief Get configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_GetConfigStr(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                                                       /**< [In] Configuration Identifier */
        uint8_t ** ppBuffer,                                                    /**< [Out] Pointer to the buffer containing the configuration string. */
        uint16_t * pBufferLength                                                /**< [Out] Amount of valid bytes in the configuration string buffer. */
    );

/**
 * \defgroup phKeyStore_Sym Symmetric
 * \brief Interfaces for Symmetric keys of KeyStore component.
 * @{
 */

/**
 * \defgroup phKeyStore_Sym_Defines Common Definitions
 * \brief Definitions for Symmetric keys of KeyStore component.
 * @{
 */

/**
 * \defgroup phKeyStore_Sym_Defines_KeyType KeyType
 * \brief Definitions for Symmetric Key types.
 * @{
 */
#define PH_KEYSTORE_KEY_TYPE_AES128                                     0x00U   /**< AES 128 Key [16]. */
#define PH_KEYSTORE_KEY_TYPE_AES192                                     0x01U   /**< AES 192 Key [24]. */
#define PH_KEYSTORE_KEY_TYPE_AES256                                     0x02U   /**< AES 256 Key [32]. */
#define PH_KEYSTORE_KEY_TYPE_DES                                        0x03U   /**< DES Single Key [8 Bytes]. This is basically the 56-Bit DES key. */
#define PH_KEYSTORE_KEY_TYPE_2K3DES                                     0x04U   /**< 2 Key Triple Des [16 Bytes]. This is basically the 112-Bit DES key. */
#define PH_KEYSTORE_KEY_TYPE_3K3DES                                     0x05U   /**< 3 Key Triple Des [24 Bytes]. This is basically the 168-Bit DES key. */
#define PH_KEYSTORE_KEY_TYPE_MIFARE                                     0x06U   /**< MIFARE (R) Key. */
/**
 * end of group phKeyStore_Sym_Defines_KeyType
 * @}
 */

/**
 * \defgroup phKeyStore_Sym_Defines_Size Key Size
 * \brief Definitions for Symmetric Key sizes.
 * @{
 */
#define PH_KEYSTORE_KEY_TYPE_MIFARE_SIZE                                0x0CU   /**< Size of an MIFARE Key. */
#define PH_KEYSTORE_KEY_TYPE_AES128_SIZE                                0x10U   /**< Size of an AES128 Key. */
#define PH_KEYSTORE_KEY_TYPE_AES192_SIZE                                0x18U   /**< Size of an AES192 Key. */
#define PH_KEYSTORE_KEY_TYPE_AES256_SIZE                                0x20U   /**< Size of an AES256 Key. */
 /**
 * end of group phKeyStore_Sym_Defines_Size
 * @}
 */

/**
 * end of group phKeyStore_Sym_Defines
 * @}
 */

/**
 * \brief Change a symmetric key entry at a given version.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_SetKey(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wKeyNo,                                                        /**< [In] Key number of the key to be loaded. Should be the Key index to
                                                                                 *        set the key to DataParams \b pKeyEntries buffer.
                                                                                 */
        uint16_t wKeyVersion,                                                   /**< [In] Key version of the key to be loaded. */
        uint16_t wKeyType,                                                      /**< [In] New Key type of the KeyEntry (predefined type of KeyType).
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_AES128 "AES 128Bit Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_AES192 "AES 192Bit Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_AES256 "AES 256Bit Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_DES "DES Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_2K3DES "TripleDES - 2Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_3K3DES "TripleDES - 3Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_MIFARE "CRYPTO-1 Key"
                                                                                 */
        uint8_t * pNewKey,                                                      /**< [In] Pointer to the key itself. */
        uint16_t wNewKeyVersion                                                 /**< [In] New Key version of the key to be updated. */
    );

/**
 * \brief Change a symmetric key entry at the specified position.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_SetKeyAtPos(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wKeyNo,                                                        /**< [In] Key number of the key to be loaded. Should be the Key index to
                                                                                 *        set the key to DataParams \b pKeyEntries member.
                                                                                 */
        uint16_t wPos,                                                          /**< [In] Key position to be updated.
                                                                                 *          - Should be the Key position to set the key to DataParams \b pKeyEntries buffer.
                                                                                 */
        uint16_t wKeyType,                                                      /**< [In] New Key type of the KeyEntry (predefined type of KeyType).
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_AES128 "AES 128Bit Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_AES192 "AES 192Bit Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_AES256 "AES 256Bit Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_DES "DES Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_2K3DES "TripleDES - 2Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_3K3DES "TripleDES - 3Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_MIFARE "CRYPTO-1 Key"
                                                                                 */
        uint8_t * pNewKey,                                                      /**< [In] Pointer to the key itself. */
        uint16_t wNewKeyVersion                                                 /**< [In] New Key version of the key to be updated. */
    );

/**
 * \brief Change a full symmetric key entry.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_SetFullKeyEntry(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wNoOfKeys,                                                     /**< [In] Number of keys in pNewKeys. */
        uint16_t wKeyNo,                                                        /**< [In] Key number of the key to be loaded. */
        uint16_t wNewRefNoKUC,                                                  /**< [In] Number of the key usage counter used together with that key. */
        uint16_t wNewKeyType,                                                   /**< [In] Key type of the key (if the current keyType of KeyEntry is different, error). */
        uint8_t * pNewKeys,                                                     /**< [In] Array of Keys to load. */
        uint16_t * pNewKeyVersionList                                           /**< [In] KeyVersionList of the key to be loaded. */
    );

/**
 * \brief Get a symmetric key entry information block.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_GetKeyEntry(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wKeyNo,                                                        /**< [In] Key number of the key entry of interest. */
        uint16_t wKeyVersionBufSize,                                            /**< [In] Buffer Size of wKeyVersion in Bytes. */
        uint16_t * wKeyVersion,                                                 /**< [Out] Array for version information. */
        uint16_t * wKeyVersionLength,                                           /**< [Out] Length of valid data in wKeyVersion. */
        uint16_t * pKeyType                                                     /**< [Out] Type of the key. */
    );

/**
 * \brief Get a symmetric key.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_GetKey(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wKeyNo,                                                        /**< [In] Key number of the key to be retrieved. */
        uint16_t wKeyVersion,                                                   /**< [In] Key version of the key to be retrieved. */
        uint8_t bKeyBufSize,                                                    /**< [In] Size of the key buffer. */
        uint8_t * pKey,                                                         /**< [Out] Pointer to the key itself. */
        uint16_t * pKeyType                                                     /**< [Out] Type of the key. */
    );

/**
 * end of group phKeyStore_Sym
 * @}
 */

#ifdef  NXPBUILD__PH_KEYSTORE_ASYM
/**
 * \defgroup phKeyStore_ASym ASymmetric
 * \brief Interfaces for ASymmetric keys of KeyStore component.
 * @{
 */

/**
 * \defgroup phKeyStore_ASym_Defines Common Definitions
 * \brief Definitions for ASymetric keys of KeyStore component.
 * @{
 */

/**
 * \defgroup phKeyStore_ASym_Defines_KeyType KeyType
 * \brief Definitions for ASymmetric Key types.
 * @{
 */
#define PH_KEYSTORE_KEY_TYPE_ECC                                        0x0200U /**< ASymmetric ECC (Elliptical Curve Cryptography) key type. */
/**
 * end of group phKeyStore_ASym_Defines_KeyType
 * @}
 */

/**
 * \defgroup phKeyStore_ASym_Defines_KeyPair KeyPair
 * \brief Definitions for ASymmetric Key pair.
 * @{
 */
#define PH_KEYSTORE_KEY_PAIR_MASK                                       0xF000U /**< Masking of Key-pair types. */
#define PH_KEYSTORE_KEY_PAIR_INVALID                                    0xC000U /**< ASymmetric key pair as Invalid. */
#define PH_KEYSTORE_KEY_PAIR_PRIVATE                                    0x1000U /**< ASymmetric key pair as Private Key. */
#define PH_KEYSTORE_KEY_PAIR_PUBLIC                                     0x2000U /**< ASymmetric key pair as Public Key. */
/**
 * end of group phKeyStore_ASym_Defines_KeyPair
 * @}
 */

/** \defgroup phKeyStore_ASym_Defines_CurveID CurveID
 * @{
 */
#define PH_KEYSTORE_CURVE_ID_MASK                                       0x000FU /**< Masking of CurveID's. */
#define PH_KEYSTORE_CURVE_ID_NONE                                       0x0000U /**< ECC Curve ID as none. */
#define PH_KEYSTORE_CURVE_ID_SECP256R1                                  0x0003U /**< Domain parameters for the 256-bit curve defined by FIPS 186-4 and SEC1. */
#define PH_KEYSTORE_CURVE_ID_SECP384R1                                  0x0004U /**< Domain parameters for the 384-bit curve defined by FIPS 186-4 and SEC1. */
#define PH_KEYSTORE_CURVE_ID_BRAINPOOL256R1                             0x0006U /**< Domain parameters for 256-bit BrainPool curve. */
#define PH_KEYSTORE_CURVE_ID_BRAINPOOL384R1                             0x0007U /**< Domain parameters for 384-bit BrainPool curve. */
/**
 * end of group phKeyStore_ASym_Defines_CurveID
 * @}
 */

/**
 * \defgroup phKeyStore_ASym_Defines_Size Key Size
 * \brief Definitions for ASymmetric Key sizes.
 * @{
 */
#define PH_KEYSTORE_KEY_TYPE_ECC_256_SIZE                                 32U   /**< Size of an ASymmetric key with SECP-256 or BrainPool-256 Curve Type.
                                                                                 *      - This is the actual size for Private Key.
                                                                                 *      - For Public Key it will be twice of Private Key + 1 byte fixed to 0x04
                                                                                 *        at beginning of the key indicating as uncompressed binary format.
                                                                                 */
#define PH_KEYSTORE_KEY_TYPE_ECC_384_SIZE                                 48U   /**< Size of an ASymmetric key with SECP-384 or BrainPool-384 Curve Type.
                                                                                 *      - This is the actual size for Private Key.
                                                                                 *      - For Public Key it will be twice of Private Key + 1 byte fixed to 0x04
                                                                                 *        at beginning of the key indicating as uncompressed binary format.
                                                                                 */
/**
 * end of group phKeyStore_ASym_Defines_Size
 * @}
 */

/**
 * end of group phKeyStore_ASym_Defines
 * @}
 */

/**
 * \brief Change a Asymmetric Private or Public key entry in Un-compressed Binary format.
 *
 * \note
 *      - This interface should be called more than ones to load Private or Public Key.
 *      - One Key number at a specific position can hold both Private and Public Key.
 *      - Public Key alone can be stored also.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_SetKeyASym(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wKeyNo,                                                        /**< [In] Key number of the KeyStore to be loaded. */
        uint16_t wPos,                                                          /**< [In] Position at which the Key should be updated.
                                                                                 *          - Should be the Key position to set the key to DataParams \b pKeyEntries member.
                                                                                 */
        uint16_t wKeyType,                                                      /**< [In] New Key type of the KeyEntry (predefined type of KeyType).
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_ECC "ECC Key Type"
                                                                                 */
        uint16_t wKeyInfo,                                                      /**< [In] Type of Key pair and Curve to save.
                                                                                 *          - Refer \ref phKeyStore_ASym_Defines_KeyPair "Key Pair" to save.
                                                                                 *          - Refer \ref phKeyStore_ASym_Defines_CurveID "CurveID" for type of Curve
                                                                                 *            available in \b pKey buffer.
                                                                                 */
        uint8_t * pKey,                                                         /**< [In] Pointer to the key itself.
                                                                                 *          - Private Key should be equal to the provided CurveID in \b wKeyInfo parameter.
                                                                                 *          - Public Key should be twice the length of Private Key including the Uncompressed
                                                                                 *            notation at beginning of the buffer.
                                                                                 */
        uint16_t wKeyLen                                                        /**< [In] Length of bytes available in \b pKey buffer. */
    );

/**
 * \brief Get a asymmetric key.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_GetKeyASym(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wKeyNo,                                                        /**< [In] Key number of the KeyStore to be retrieved. */
        uint16_t wPos,                                                          /**< [In] Position at which the Key should be retrieved.
                                                                                 *          - Should be the Key position to get the key from DataParams
                                                                                 *            \b pKeyEntries member.
                                                                                 */
        uint16_t wKeyPairType,                                                  /**< [In] Type of Key-pair to be retrieved.
                                                                                 *          - \ref PH_KEYSTORE_KEY_PAIR_PRIVATE "Private Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_PAIR_PUBLIC "Public Key"
                                                                                 */
        uint16_t * pKeyType,                                                    /**< [In] Type of key being saved earlier.
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_ECC "ECC Key Type"
                                                                                 */
        uint8_t * pCurveID,                                                     /**< [In] One of the \ref phKeyStore_ASym_Defines_CurveID "Curve ID's". */
        uint8_t * pKey,                                                         /**< [Out] Pointer to the key itself. The buffer size should be equal or greater
                                                                                 *         than the Size of the stored key.
                                                                                 */
        uint16_t * pKeyLen                                                      /**< [Out] Length of bytes available in \b pKey buffer. */
    );

/**
 * \brief Get a asymmetric key ECC Curve ID.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phKeyStore_GetCurveID(
        void * pDataParams,                                                     /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wKeyNo,                                                        /**< [In] Key number of the KeyStore to be retrieved. */
        uint16_t wPos,                                                          /**< [In] Position at which the Key should be retrieved.
                                                                                 *          - Should be the Key position to get the key from DataParams
                                                                                 *            \b pKeyEntries member.
                                                                                 */
        uint8_t * pCurveID                                                      /**< [Out] Provided the CurveID of the provided Key informations.
                                                                                 *         Will return one of the following mentioned in
                                                                                 *         \ref phKeyStore_ASym_Defines_CurveID "Curve ID's"
                                                                                 */
    );

/**
 * end of group phKeyStore_ASym
 * @}
 */
#endif /* NXPBUILD__PH_KEYSTORE_ASYM */

/**
 * \defgroup phKeyStore_Utility Utility
 * \brief Interfaces for utility interfaces for KeyStore component.
 * @{
 */
/**
 * \brief Gets the size of Symmetric key.
 *
 * \retval Symmetric Key Size.
 */
uint8_t phKeyStore_GetKeySize(
        uint16_t wKeyType                                                       /**< [In] Key type of the KeyEntry (predefined type of KeyType).
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_AES128 "AES 128Bit Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_AES192 "AES 192Bit Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_AES256 "AES 256Bit Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_DES "DES Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_2K3DES "TripleDES - 2Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_3K3DES "TripleDES - 3Key"
                                                                                 *          - \ref PH_KEYSTORE_KEY_TYPE_MIFARE "CRYPTO-1 Key"
                                                                                 */
    );

/**
 * end of group phKeyStore_Utility
 * @}
 */

/**
 * end of group phKeyStore
 * @}
 */
#endif /* NXPBUILD__PH_KEYSTORE */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHKEYSTORE_H */
