/*
 * Copyright 2013 - 2020, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic BAL Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHBALREG_H
#define PHBALREG_H

#include <ph_Status.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHBAL_REG_SERIALWIN

/**
 * \defgroup phbalReg_SerialWin Component : RS232 (Windows)
 * \brief RS232 BAL (Windows)
 *
 * Serial-Line (COM-Port) interface for Windows.
 * @{
 */

#define PHBAL_REG_SERIALWIN_ID                                                              0x01U   /**< ID for Serial BAL component (Win32) */

/** \brief Serial (Windows) BAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                                   /**< Layer ID for this BAL component, NEVER MODIFY! */
    int8_t * pComPortString;                                                                        /**< Com-Port string. */
    void * pComHandle;                                                                              /**< Handle to communication port. */
    uint32_t dwBitRate;                                                                             /**< Storage for bit rate setting. */
    uint16_t wTimeout;                                                                              /**< Storage for timeout setting. */
    uint16_t wAdditionalRxDelayBytes;                                                               /**< Storage for additional rx delay setting. */
    uint16_t wTimeoutMultiplier;                                                                    /**< Storage for timeout multiplier setting. */
    uint16_t wEnumBufSize;                                                                          /**< Size of the enumeration buffer. Maximum possible size is 255 bytes. */
    uint8_t * pEnumBuffer;                                                                          /**< Pointer to the enumeration buffer for the GetPortList() function. */
    uint8_t bStopBits;                                                                              /**< Number of stopbits. */
    uint8_t bDtr;                                                                                   /**< DTR line */
    uint8_t bRts;                                                                                   /**< RTS line */
    uint8_t bParity;                                                                                /**< Parity */
    uint8_t bFI;                                                                                    /**< FI */
    uint8_t bDI;                                                                                    /**< DI */
} phbalReg_SerialWin_DataParams_t;

/** \name BAL Communication Configs */
/** @{ */
#define PHBAL_REG_SERIALWIN_CONFIG_BITRATE                          (PH_CONFIG_CUSTOM_BEGIN + 0)    /**< Configure communication speed. */
#define PHBAL_REG_SERIALWIN_CONFIG_STOPBITS                         (PH_CONFIG_CUSTOM_BEGIN + 1)    /**< Configure number of stop bits. */
#define PHBAL_REG_SERIALWIN_CONFIG_DTR                              (PH_CONFIG_CUSTOM_BEGIN + 2)    /**< Configure dtr line. */
#define PHBAL_REG_SERIALWIN_CONFIG_RTS                              (PH_CONFIG_CUSTOM_BEGIN + 3)    /**< Configure rts line. */
#define PHBAL_REG_SERIALWIN_CONFIG_PARITY                           (PH_CONFIG_CUSTOM_BEGIN + 4)    /**< Configure parity. */
#define PHBAL_REG_SERIALWIN_CONFIG_FI                               (PH_CONFIG_CUSTOM_BEGIN + 7)    /**< Configure FI. */
#define PHBAL_REG_SERIALWIN_CONFIG_DI                               (PH_CONFIG_CUSTOM_BEGIN + 8)    /**< Configure DI. */
#define PHBAL_REG_SERIALWIN_CONFIG_TIMEOUT_MULTIPLIER_MS            (PH_CONFIG_CUSTOM_BEGIN + 9)    /**< Configure Timeout Multiplier. */
#define PHBAL_REG_SERIALWIN_CONFIG_ADDITIONAL_RXDELAY_BYTES         (PH_CONFIG_CUSTOM_BEGIN + 10)   /**< Needed because WriteFile() sometimes returns too early. */
/** @} */

/** \name Option to supress checks after ReadFile is called. */
/** @{ */
#define PHBAL_REG_SERIALWIN_SUPRESS_CHECKS                                                  0x0080  /**< Option to supress checks after ReadFile is called.  */
/** @} */

/** \name BAL Configuration Values */
/** @{ */
#define PHBAL_REG_SERIALWIN_VALUE_BITRATE_9600                                              0x0000U /**< Serial bitrate as 9600 kBit/s. */
#define PHBAL_REG_SERIALWIN_VALUE_BITRATE_19200                                             0x0001U /**< Serial bitrate as 19200 kBit/s. */
#define PHBAL_REG_SERIALWIN_VALUE_BITRATE_38400                                             0x0002U /**< Serial bitrate as 38400 kBit/s. */
#define PHBAL_REG_SERIALWIN_VALUE_BITRATE_57600                                             0x0003U /**< Serial bitrate as 57600 kBit/s. */
#define PHBAL_REG_SERIALWIN_VALUE_BITRATE_115200                                            0x0004U /**< Serial bitrate as 115200 kBit/s. */
#define PHBAL_REG_SERIALWIN_VALUE_BITRATE_230400                                            0x0005U /**< Serial bitrate as 230400 kBit/s. */
#define PHBAL_REG_SERIALWIN_VALUE_BITRATE_460800                                            0x0006U /**< Serial bitrate as 460800 kBit/s. */
#define PHBAL_REG_SERIALWIN_VALUE_STOPBITS_ONE                                              0x0000U /**< One stopbit. */
#define PHBAL_REG_SERIALWIN_VALUE_STOPBITS_ONE5                                             0x0001U /**< One and a half stopbit. */
#define PHBAL_REG_SERIALWIN_VALUE_STOPBITS_TWO                                              0x0002U /**< Two stopbit. */
#define PHBAL_REG_SERIALWIN_VALUE_DTR_LOW                                                   0x0000U /**< DTR Signal low. */
#define PHBAL_REG_SERIALWIN_VALUE_DTR_HIGH                                                  0x0001U /**< DTR Signal high. */
#define PHBAL_REG_SERIALWIN_VALUE_RTS_LOW                                                   0x0000U /**< RTS Signal low. */
#define PHBAL_REG_SERIALWIN_VALUE_RTS_HIGH                                                  0x0001U /**< RTS Signal high. */
#define PHBAL_REG_SERIALWIN_VALUE_PARITY_NONE                                               0x0000U /**< Parity as None. */
#define PHBAL_REG_SERIALWIN_VALUE_PARITY_ODD                                                0x0001U /**< Parity as ODD. */
#define PHBAL_REG_SERIALWIN_VALUE_PARITY_EVEN                                               0x0002U /**< Parity as EVEN. */
/** @} */

/**
 * \brief Initialise the RS232 BAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS Parameter structure size is invalid.
 * \retval #PH_ERR_INVALID_PARAMETER \b wEnumBufSize is invalid.
 */
phStatus_t phbalReg_SerialWin_Init(
        phbalReg_SerialWin_DataParams_t * pDataParams,                                              /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                                 /**< [In] Specifies the size of the data parameter structure. */
        uint16_t wEnumBufSize,                                                                      /**< [In] Size of the enumeration buffer. Maximum possible size is 255 bytes. */
        uint8_t * pEnumBuffer                                                                       /**< [In] Pointer to the enumeration buffer for the GetPortList() function. */
    );

/**
 * end of group phbalReg_SerialWin
 * @}
 */
#endif /* NXPBUILD__PHBAL_REG_SERIALWIN */


#ifdef NXPBUILD__PHBAL_REG_RDFPGAV6

/**
 * \cond INCLUDE_INTERNALS
 * \defgroup phbalReg_RdFpgaV6 Component : RdFpgaV6
 * \brief BAL-Component for the FPGA-Box Reader
 * @{
 */

#define PHBAL_REG_RDFPGAV6_ID                                                               0x03U   /**< ID for RdFpga BAL component */

/** \name Custom Configs */
/** @{ */
#define PHBAL_REG_RDFPGAV6_ADDITIONAL_INFO                          (PH_CONFIG_CUSTOM_BEGIN + 0)    /**< Set / Get additional information . */
/** @} */

/** \name Function pointers */
/** @{ */
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_OPEN) (uint16_t wReaderType, uint8_t * paForceSoftwareStack);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_CLOSE)();
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_INIT) (uint16_t wProtocolType);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_SETPARAM) (uint32_t dwIdentifier, uint32_t dwValue, uint8_t bParamSetMode);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_GETPARAM) (uint32_t dwIdentifier, uint32_t * pdwValue);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_EXCHANGEBITS) (uint8_t * pTxBuffer, uint16_t wTxBytes, uint8_t bNumValidTxBits, uint8_t * pRxBuffer, uint16_t dwRxBufferSize, uint16_t * pdwRxBytes, uint8_t * pbNumValidRxBits);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_SETTARGETJOINER) (uint16_t joiner);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_SETJOINERREG) (uint8_t address, uint8_t value);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_GETJOINERREG) (uint8_t address, uint8_t * pvalue);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_LOADJOINERINIFILE) (uint8_t * filename);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_ACTFIELD) ();
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_DEACTIVATEFIELD) ();
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_HFPAUSE) (uint32_t pause_time);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_SET_POTIPOS) (uint16_t pos);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_SET_DEFAULTPOTIPOS) (uint16_t pos);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_GET_POTIPOS) (uint16_t * pos);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_INC_POTIPOS)();
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_DEC_POTIPOS)();
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_SET_TIMEOUTVALUE) (uint32_t timeout_value);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_GET_TIMEOUTVALUE) (uint32_t * timeout_value);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_LOAD_PATTERN) (uint8_t * filename, uint8_t id);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_PLAY_PATTERN) (uint8_t id);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_PLAYSEQA) ();
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_PLAYSEQB) ();
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_PLAYSINUS) ();
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_SELECT_TRIGGER) (uint16_t source);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_LBLRST) (uint16_t rising_edge_cnt, uint32_t sinus, uint32_t hf_pause);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_LBLRST_COUNTERRESET) (uint16_t rising_edge_cnt, uint32_t sinus, uint32_t hf_pause);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_LBLRST_SSCOUNTERRESET) (uint16_t rising_edge_cnt, uint32_t sinus, uint32_t hf_pause);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_LBLRST_SINGLESHOT) (uint16_t rising_edge_cnt, uint32_t sinus, uint32_t hf_pause);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_LBLRST_WITHCOMMAND) (uint32_t sinus);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_ACT_FDT_MEASUREMENT) ();
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_GET_FDT) (uint32_t * fdt);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_GET_FDT_FAST) (uint32_t * fdt);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_ACT_ANTITEARINGMODE) (uint32_t sinus, uint32_t hf_pause);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_RESONANCE_FREQ_TEST) (uint32_t hf_pause, uint16_t repeat_freq_id);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_ACT_FDT_PICC_TO_PCD_MEASUREMENT) (uint32_t time, uint8_t check_resp, uint16_t rx_byte_cnt, uint8_t rx_bit_cnt, uint8_t * rx_expected);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_ACT_LMA_MEASUREMENT) ();
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_ACT_SEQUENCERMODE)();
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_DEACTIVATE_MODES)();
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_SET_JOINER_RXGAIN) (uint8_t value);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_GET_JOINER_RXGAIN) (uint8_t * pvalue);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_ACTIVEDS_SET_FDT) (uint32_t value);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_DS_SET_FDT) (uint32_t value, uint8_t id);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_GET_VERSION) (uint8_t * pvalue);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_GET_REVISION) (uint8_t * pvalue);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_GET_ERROR_CODES) (uint8_t * perrors, uint8_t max_error_cnt, uint8_t * perror_cnt);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_SET_CLOCK_SOURCE) (uint16_t clock_source);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_GET_FIRST_RESPONSE) (uint8_t * pRxBuffer, uint16_t wRxBufferSize, uint16_t * pwRxBytes, uint8_t * pNumValidRxBits, int16_t * pwError);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_GET_RESPONSE) (uint8_t * pRxBuffer, uint16_t wRxBufferSize, uint16_t * pwRxBytes, uint8_t * pNumValidRxBits, int16_t * pwError);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_GET_IQ_TIMEOUTVALUE) (uint32_t * pdwTimeoutValue);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_SET_IQ_TIMEOUTVALUE) (uint32_t dwTimeoutValue);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_GET_TERMINALID) (uint8_t * pTerminalId);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_RESET_JOINER) ( );
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_LOADFILTERFILE) (uint8_t * filename);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_EXCHANGEFPGACOMMAND) (uint8_t * txData, uint32_t txLen, uint8_t * rxData, uint32_t maxRxLen, uint32_t * rxLen);
typedef int16_t (__stdcall*PHBAL_REG_RDFPGAV6_ACT_TYPEBMODE)();
/** @} */

/** \brief RDFPGA BAL parameter structure */
typedef struct
{
    uint16_t                                            wId;                                        /**< Layer ID for this BAL component, NEVER MODIFY! */
    uint8_t *                                           pDeviceName;                                /**< Name of FPGA Box Device device. */
    uint8_t                                             bPortOpen;                                  /**< Internal State if port is opened. */
    uint16_t                                            wAdditionalInfo;                            /**< Storage for additional error information. */
    uint16_t                                            wTimeout;                                   /**< Shadow for the timeout. */
    void *                                              pHandle;                                    /**< Handle of the phCDllReaderLib.dll. */
    PHBAL_REG_RDFPGAV6_OPEN                             pfRdFpga_Open;                              /**< function pointer for pfRdFpga_Open. */
    PHBAL_REG_RDFPGAV6_CLOSE                            pfRdFpga_Close;                             /**< function pointer for pfRdFpga_Close. */
    PHBAL_REG_RDFPGAV6_INIT                             pfRdFpga_Init;                              /**< function pointer for pfRdFpga_Init. */
    PHBAL_REG_RDFPGAV6_SETPARAM                         pfRdFpga_SetParam;                          /**< function pointer for pfRdFpga_SetParam. */
    PHBAL_REG_RDFPGAV6_GETPARAM                         pfRdFpga_GetParam;                          /**< function pointer for pfRdFpga_GetParam. */
    PHBAL_REG_RDFPGAV6_EXCHANGEBITS                     pfRdFpga_ExchangeBits;                      /**< function pointer for pfRdFpga_ExchangeBits. */
    PHBAL_REG_RDFPGAV6_SETTARGETJOINER                  pfRdFpga_SetTargetJoiner;                   /**< function pointer for pfRdFpga_SetTargetJoiner. */
    PHBAL_REG_RDFPGAV6_SETJOINERREG                     pfRdFpga_SetJoinerReg;                      /**< function pointer for pfRdFpga_SetJoinerReg. */
    PHBAL_REG_RDFPGAV6_GETJOINERREG                     pfRdFpga_GetJoinerReg;                      /**< function pointer for pfRdFpga_GetJoinerReg. */
    PHBAL_REG_RDFPGAV6_LOADJOINERINIFILE                pfRdFpga_LoadJoinerIniFile;                 /**< function pointer for pfRdFpga_LoadJoinerIniFile. */
    PHBAL_REG_RDFPGAV6_ACTFIELD                         pfRdFpga_ActivateField;                     /**< function pointer for pfRdFpga_ActivateField. */
    PHBAL_REG_RDFPGAV6_DEACTIVATEFIELD                  pfRdFpga_DeactivateField;                   /**< function pointer for pfRdFpga_DeactivateField. */
    PHBAL_REG_RDFPGAV6_HFPAUSE                          pfRdFpga_HFPause;                           /**< function pointer for pfRdFpga_HFPause. */
    PHBAL_REG_RDFPGAV6_SET_POTIPOS                      pfRdFpga_SetPotiPos;                        /**< function pointer for pfRdFpga_SetPotiPos. */
    PHBAL_REG_RDFPGAV6_SET_DEFAULTPOTIPOS               pfRdFpga_SetDefaultPotiPos;                 /**< function pointer for pfRdFpga_SetDefaultPotiPos. */
    PHBAL_REG_RDFPGAV6_GET_POTIPOS                      pfRdFpga_GetPotiPos;                        /**< function pointer for pfRdFpga_GetPotiPos. */
    PHBAL_REG_RDFPGAV6_INC_POTIPOS                      pfRdFpga_IncPotiPos;                        /**< function pointer for pfRdFpga_IncPotiPos. */
    PHBAL_REG_RDFPGAV6_DEC_POTIPOS                      pfRdFpga_DecPotiPos;                        /**< function pointer for pfRdFpga_DecPotiPos. */
    PHBAL_REG_RDFPGAV6_SET_TIMEOUTVALUE                 pfRdFpga_SetTimeoutValue;                   /**< function pointer for pfRdFpga_SetTimeoutValue. */
    PHBAL_REG_RDFPGAV6_GET_TIMEOUTVALUE                 pfRdFpga_GetTimeoutValue;                   /**< function pointer for pfRdFpga_GetTimeoutValue. */
    PHBAL_REG_RDFPGAV6_LOAD_PATTERN                     pfRdFpga_LoadPattern;                       /**< function pointer for pfRdFpga_LoadPattern. */
    PHBAL_REG_RDFPGAV6_PLAY_PATTERN                     pfRdFpga_PlayPattern;                       /**< function pointer for pfRdFpga_PlayPattern. */
    PHBAL_REG_RDFPGAV6_PLAYSEQA                         pfRdFpga_PlaySeqA;                          /**< function pointer for pfRdFpga_PlaySeqA. */
    PHBAL_REG_RDFPGAV6_PLAYSEQB                         pfRdFpga_PlaySeqB;                          /**< function pointer for pfRdFpga_PlaySeqB. */
    PHBAL_REG_RDFPGAV6_PLAYSINUS                        pfRdFpga_PlaySinus;                         /**< function pointer for pfRdFpga_PlaySinus. */
    PHBAL_REG_RDFPGAV6_SELECT_TRIGGER                   pfRdFpga_SelectTrigger;                     /**< function pointer for pfRdFpga_SelectTrigger. */
    PHBAL_REG_RDFPGAV6_LBLRST                           pfRdFpga_LabelReset;                        /**< function pointer for pfRdFpga_LabelReset. */
    PHBAL_REG_RDFPGAV6_LBLRST_COUNTERRESET              pfRdFpga_LabelResetCounterReset;            /**< function pointer for pfRdFpga_LabelResetCounterReset. */
    PHBAL_REG_RDFPGAV6_LBLRST_SSCOUNTERRESET            pfRdFpga_LabelResetCounterResetSingleShot;  /**< function pointer for pfRdFpga_LabelResetCounterResetSingleShot. */
    PHBAL_REG_RDFPGAV6_LBLRST_SINGLESHOT                pfRdFpga_LabelResetSingleShot;              /**< function pointer for pfRdFpga_LabelResetSingleShot. */
    PHBAL_REG_RDFPGAV6_LBLRST_WITHCOMMAND               pfRdFpga_LabelResetWithCommand;             /**< function pointer for pfRdFpga_LabelResetWithCommand. */
    PHBAL_REG_RDFPGAV6_ACT_FDT_MEASUREMENT              pfRdFpga_ActivateFDTMeasurement;            /**< function pointer for pfRdFpga_ActivateFDTMeasurement. */
    PHBAL_REG_RDFPGAV6_GET_FDT                          pfRdFpga_GetFDT;                            /**< function pointer for pfRdFpga_GetFDT. */
    PHBAL_REG_RDFPGAV6_GET_FDT_FAST                     pfRdFpga_GetFDTFast;                        /**< function pointer for pfRdFpga_GetFDTFast. */
    PHBAL_REG_RDFPGAV6_ACT_ANTITEARINGMODE              pfRdFpga_ActivateAntiTearingMode;           /**< function pointer for pfRdFpga_ActivateAntiTearingMode. */
    PHBAL_REG_RDFPGAV6_RESONANCE_FREQ_TEST              pfRdFpga_ResonanceFrequenceTest;            /**< function pointer for pfRdFpga_ResonanceFrequenceTest. */
    PHBAL_REG_RDFPGAV6_ACT_FDT_PICC_TO_PCD_MEASUREMENT  pfRdFpga_ActivateFDTMeasurementPICCtoPCD;   /**< function pointer for pfRdFpga_ActivateFDTMeasurementPICCtoPCD. */
    PHBAL_REG_RDFPGAV6_ACT_LMA_MEASUREMENT              pfRdFpga_ActivateLMAMeasurement;            /**< function pointer for pfRdFpga_ActivateLMAMeasurement. */
    PHBAL_REG_RDFPGAV6_ACT_SEQUENCERMODE                pfRdFpga_ActivateSequencerMode;             /**< function pointer for pfRdFpga_ActivateSequencerMode. */
    PHBAL_REG_RDFPGAV6_DEACTIVATE_MODES                 pfRdFpga_DeactivateModes;                   /**< function pointer for pfRdFpga_DeactivateModes. */
    PHBAL_REG_RDFPGAV6_SET_JOINER_RXGAIN                pfRdFpga_SetJoinerRxGain;                   /**< function pointer for pfRdFpga_SetJoinerRxGain. */
    PHBAL_REG_RDFPGAV6_GET_JOINER_RXGAIN                pfRdFpga_GetJoinerRxGain;                   /**< function pointer for pfRdFpga_GetJoinerRxGain. */
    PHBAL_REG_RDFPGAV6_ACTIVEDS_SET_FDT                 pfRdFpga_SetFDTPCDforActiveDS;              /**< function pointer for pfRdFpga_SetFDTPCDforActiveDS. */
    PHBAL_REG_RDFPGAV6_DS_SET_FDT                       pfRdFpga_SetFDTPCDforDS;                    /**< function pointer for pfRdFpga_SetFDTPCDforDS. */
    PHBAL_REG_RDFPGAV6_GET_VERSION                      pfRdFpga_GetVersion;                        /**< function pointer for pfRdFpga_GetVersion. */
    PHBAL_REG_RDFPGAV6_GET_REVISION                     pfRdFpga_GetRevision;                       /**< function pointer for pfRdFpga_GetRevision. */
    PHBAL_REG_RDFPGAV6_GET_ERROR_CODES                  pfRdFpga_GetErrorCodes;                     /**< function pointer for pfRdFpga_GetErrorCodes. */
    PHBAL_REG_RDFPGAV6_SET_CLOCK_SOURCE                 pfRdFpga_SetClockSource;                    /**< function pointer for pfRdFpga_SetClockSource. */
    PHBAL_REG_RDFPGAV6_GET_FIRST_RESPONSE               pfRdFpga_GetFirstResponse;                  /**< function pointer for pfRdFpga_GetFirstResponse. */
    PHBAL_REG_RDFPGAV6_GET_RESPONSE                     pfRdFpga_GetResponse;                       /**< function pointer for pfRdFpga_GetResponse. */
    PHBAL_REG_RDFPGAV6_GET_IQ_TIMEOUTVALUE              pfRdFpga_GetIQTimeoutValue;                 /**< function pointer for pfRdFpga_GetIQTimeoutValue. */
    PHBAL_REG_RDFPGAV6_SET_IQ_TIMEOUTVALUE              pfRdFpga_SetIQTimeoutValue;                 /**< function pointer for pfRdFpga_SetIQTimeoutValue. */
    PHBAL_REG_RDFPGAV6_GET_TERMINALID                   pfRdFpga_GetTerminalId;                     /**< function pointer for pfRdFpga_GetTerminalId. */
    PHBAL_REG_RDFPGAV6_RESET_JOINER                     pfRdFpga_ResetJoiner;                       /**< function pointer for pfRdFpga_ResetJoiner. */
    PHBAL_REG_RDFPGAV6_LOADFILTERFILE                   pfRdFpga_LoadFilterFile;                    /**< function pointer for pfRdFpga_LoadFilterFile. */
    PHBAL_REG_RDFPGAV6_EXCHANGEFPGACOMMAND              pfRdFpga_ExchangeFPGACommand;               /**< function pointer for pfRdFpga_ExchangeFPGACommand. */
    PHBAL_REG_RDFPGAV6_ACT_TYPEBMODE                    pfRdFpga_ActivateTypeBMode;                 /**< function pointer for pfRdFpga_ActivateTypeBMode. */
} phbalReg_RdFpgaV6_DataParams_t;

/**
 * \brief Initialise the RDFPGA BAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS Parameter structure size is invalid.
 * \retval #PH_ERR_USE_CONDITION Required dependencies not available.
 */
phStatus_t phbalReg_RdFpgaV6_Init(
        phbalReg_RdFpgaV6_DataParams_t * pDataParams,                                               /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams                                                                  /**< [In] Specifies the size of the data parameter structure. */
    );

/**
 * end of group phbalReg_RdFpgaV6
 * @}
 * \endcond
 */
#endif /* NXPBUILD__PHBAL_REG_RDFPGAV6 */

#ifdef NXPBUILD__PHBAL_REG_PS6000USB

/**
 * \defgroup phbalReg_Ps6000Usb Component : Ps6000Usb
 * \brief BAL-Component for the PicoScope PS6000 USB Oscilloscope
 * @{
 */

#define PHBAL_REG_PS6000USB_ID                                                              0x04U   /**< ID for Ps6000Usb BAL component */

#define PHBAL_REG_PS6000_GET_UNIT_INFO_BUFFER_SIZE 16

/** \name Custom Configs */
/** @{ */
#define PHBAL_REG_PS6000USB_ADDITIONAL_INFO                         (PH_CONFIG_CUSTOM_BEGIN + 0)    /**< Set / Get additional information . */
/** @} */

/** \name Function pointers */
/** @{ */
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_OPEN_UNIT) (int16_t * pwHandle, int8_t * pSerial);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_OPEN_UNIT_ASYNC) (int16_t * pwStatus, int8_t * pSerial);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_OPEN_UNIT_PROGRESS) (int16_t * pwHandle, int16_t * pwProgressPercent, int16_t * pwComplete);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_UNIT_INFO) (int16_t wHandle, int8_t * pString, int16_t wStringLength, int16_t * pwRequiredSize, uint32_t dwInfo);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_FLASH_LED) (int16_t wHandle, int16_t wStart);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_CLOSE_UNIT) (int16_t wHandle);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_MEMORY_SEGMENTS) (int16_t wHandle, uint32_t dwNSegments, uint32_t * pdwNMaxSamples);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_CHANNEL) (int16_t wHandle, uint32_t dwChannel, int16_t wEnabled, uint32_t dwType, uint32_t dwRange, int32_t dwAnalogueOffset, uint32_t dwBandwidth);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_TIMEBASE) (int16_t wHandle, uint32_t dwTimebase, uint32_t dwNoSamples, int32_t * pdwTimeIntervalNanoseconds, int16_t wOversample, uint32_t * pdwMaxSamples, uint32_t dwSegmentIndex);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_TIMEBASE2) (int16_t wHandle, uint32_t dwTimebase, uint32_t dwNoSamples, int32_t * pdwTimeIntervalNanoseconds, int16_t wOversample, uint32_t * pdwMaxSamples, uint32_t dwSegmentIndex);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_SIG_GEN_ARBITRARY) (int16_t wHandle, int32_t dwOffsetVoltage, uint32_t dwPkToPk, uint32_t dwStartDeltaPhase, uint32_t dwStopDeltaPhase, uint32_t dwDeltaPhaseIncrement, uint32_t dwDwellCount, int16_t * pwArbitraryWaveform, int32_t dwArbitraryWaveformSize, uint32_t dwSweepType, uint32_t dwOperation, uint32_t dwIndexMode, uint32_t dwShots, uint32_t dwSweeps, uint32_t dwTriggerType, uint32_t dwTriggerSource, int16_t wExtInThreshold);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_SIG_GEN_BUILT_IN) (int16_t wHandle, int32_t dwOffsetVoltage, uint32_t dwPkToPk, int16_t wWavetype, int32_t dwStartFrequency, int32_t dwStopFrequency, int32_t dwIncrement, int32_t dwDwellTime, uint32_t dwSweepType, uint32_t dwOperation, uint32_t dwShots, uint32_t dwSweeps, uint32_t dwTriggerType, uint32_t dwTriggerSource, int16_t wExtInThreshold);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SIG_GEN_SOFTWARE_CONTROL) (int16_t wHandle, int16_t wState);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_SIMPLE_TRIGGER) (int16_t wHandle, int16_t wEnable, uint32_t dwSource, int16_t wThreshold, uint32_t dwDirection, uint32_t dwDelay, int16_t wAutoTrigger_ms);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_ETS) (int16_t wHandle, uint32_t dwMode, int16_t wEtsCycles, int16_t wEtsInterleave, int32_t * pdwSampleTimePicoseconds);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_TRIGGER_CHANNEL_PROPERTIES) (int16_t wHandle, uint32_t * pdwChannelProperties, int16_t wNChannelProperties, int16_t wAuxOutputEnable, int32_t dwAutoTriggerMilliseconds);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_TRIGGER_CHANNEL_CONDITIONS) (int16_t wHandle, uint32_t * pdwConditions, uint16_t wNConditions);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_TRIGGER_CHANNEL_DIRECTIONS) (int16_t wHandle, uint32_t dwChannelA, uint32_t dwChannelB, uint32_t dwChannelC, uint32_t dwChannelD, uint32_t dwExt, uint32_t dwAux);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_TRIGGER_DELAY) (int16_t wHandle, uint32_t dwDelay);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_PULSE_WIDTH_QUALIFIER) (int16_t wHandle, uint32_t * pdwConditions, int16_t wNConditions, uint32_t dwDirection, uint32_t dwLower, uint32_t dwUpper, uint32_t dwType);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_IS_TRIGGER_OR_PULSE_WIDTH_QUALIFIER_ENABLED) (int16_t wHandle, int16_t * pwTriggerEnabled, int16_t * pwPulseWidthQualifierEnabled);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_TRIGGER_TIME_OFFSET) (int16_t wHandle, uint32_t * pdwTimeUpper, uint32_t * pdwTimeLower, uint32_t * pdwTimeUnits, uint32_t dwSegmentIndex);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_TRIGGER_TIME_OFFSET64) (int16_t wHandle, uint64_t * pqwTime, uint32_t * pdwTimeUnits, uint32_t dwSegmentIndex);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_VALUES_TRIGGER_TIME_OFFSET_BULK) (int16_t wHandle, uint32_t * pdwTimesUpper, uint32_t * pdwTimesLower, uint32_t * pdwTimeUnits, uint32_t dwFromSegmentIndex, uint32_t dwToSegmentIndex);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_VALUES_TRIGGER_TIME_OFFSET_BULK64) (int16_t wHandle, uint64_t * pqwTimes, uint32_t * pdwTimeUnits, uint32_t * pdwFromSegmentIndex, uint32_t * pdwToSegmentIndex);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_DATA_BUFFERS) (int16_t wHandle, uint32_t dwChannel, int16_t * pwBufferMax, int16_t * pwBufferMin, uint32_t dwBufferLth, uint32_t dwDownSampleRatioMode);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_DATA_BUFFER) (int16_t wHandle, uint32_t dwChannel, int16_t * pwBuffer, uint32_t dwBufferLth, uint32_t dwDownSampleRatioMode);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_DATA_BUFFER_BULK) (int16_t wHandle, uint32_t dwChannel, int16_t * pwBuffer, uint32_t dwBufferLth, uint32_t dwWaveform, uint32_t dwDownSampleRatioMode);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_DATA_BUFFERS_BULK) (int16_t wHandle, uint32_t dwChannel, int16_t * pwBufferMax, int16_t * pwBufferMin, uint32_t dwBufferLth, uint32_t dwWaveform, uint32_t dwDownSampleRatioMode);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_ETS_TIME_BUFFER) (int16_t wHandle, uint64_t * pqwBuffer, uint32_t dwBufferLth);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_ETS_TIME_BUFFERS) (int16_t wHandle, uint32_t * pdwTimeUpper, uint32_t * pdwTimeLower, uint32_t dwBufferLth);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_RUN_BLOCK) (int16_t wHandle, uint32_t dwNoOfPreTriggerSamples, uint32_t dwNoOfPostTriggerSamples, uint32_t dwTimebase, int16_t wOversample, int32_t * pdwTimeIndisposedMs, uint32_t dwSegmentIndex, uint32_t dwLpReady, void * pParameter);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_IS_READY) (int16_t wHandle, int16_t * pwReady);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_RUN_STREAMING) (int16_t wHandle, uint32_t * pdwSampleInterval, uint32_t dwSampleIntervalTimeUnits, uint32_t dwMaxPreTriggerSamples, uint32_t dwMaxPostTriggerSamples, int16_t wAutoStop, uint32_t dwDownSampleRatio, uint32_t dwDownSampleRatioMode, uint32_t dwOverviewBufferSize);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_STREAMING_LATEST_VALUES) (int16_t wHandle, void * pLpPs6000Ready, void * pParameter);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_NO_OF_STREAMING_VALUES) (int16_t wHandle, uint32_t * pdwNoOfValues);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_MAX_DOWN_SAMPLE_RATIO) (int16_t wHandle, uint32_t dwNoOfUnaggregatedSamples, uint32_t * pdwMaxDownSampleRatio, uint32_t dwDownSampleRatioMode, uint32_t dwSegmentIndex);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_VALUES) (int16_t wHandle, uint32_t dwStartIndex, uint32_t * pdwNoOfSamples, uint32_t dwDownSampleRatio, uint32_t dwDownSampleRatioMode, uint32_t dwSegmentIndex, int16_t * pwOverflow);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_VALUES_BULK) (int16_t wHandle, uint32_t * pdwNoOfSamples, uint32_t dwFromSegmentIndex, uint32_t dwToSegmentIndex, uint32_t dwDownSampleRatio, uint32_t dwDownSampleRatioMode, int16_t * pwOverflow);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_VALUES_ASYNC) (int16_t wHandle, uint32_t dwStartIndex, uint32_t dwNoOfSamples, uint32_t dwDownSampleRatio, uint32_t dwDownSampleRatioMode, uint32_t dwSegmentIndex, void * pLpDataReady, void * pParameter);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_VALUES_OVERLAPPED) (int16_t wHandle, uint32_t dwStartIndex, uint32_t * pdwNoOfSamples, uint32_t dwDownSampleRatio, uint32_t dwDownSampleRatioMode, uint32_t dwSegmentIndex, int16_t * pwOverflow);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_VALUES_OVERLAPPED_BULK) (int16_t wHandle, uint32_t dwStartIndex, uint32_t * pdwNoOfSamples, uint32_t dwDownSampleRatio, uint32_t dwDownSampleRatioMode, uint32_t dwFromSegmentIndex, uint32_t dwToSegmentIndex, int16_t * pwOverflow);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_VALUES_BULK_ASYNC) (int16_t wHandle, uint32_t dwStartIndex, uint32_t * pdwNoOfSamples, uint32_t dwDownSampleRatio, uint32_t dwDownSampleRatioMode, uint32_t dwFromSegmentIndex, uint32_t dwToSegmentIndex, int16_t * pwOverflow);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_GET_NO_OF_CAPTURES) (int16_t wHandle, uint32_t * pdwNCaptures);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_STOP) (int16_t wHandle);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_NO_OF_CAPTURES) (int16_t wHandle, uint32_t dwNCaptures);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_WAVEFORM_LIMITER) (int16_t wHandle, uint32_t dwNWaveformsPerSecond);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_ENUMERATE_UNITS) (int16_t * pwCount, int8_t * pSerials, int16_t * pwSerialLth);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_SET_EXTERNAL_CLOCK) (int16_t wHandle, uint32_t dwFrequency, int16_t wThreshold);
typedef uint32_t (__stdcall *PHBAL_REG_PS6000USB_PING_UNIT) (int16_t wHandle);
/** @} */

/** \brief PS6000USB BAL parameter structure */
typedef struct
{
    uint16_t                                                        wId;                                                                    /**< Layer ID for this BAL component, NEVER MODIFY! */
    uint8_t *                                                       pDeviceName;                                                            /**< Name of PicoScope Osci device. */
    int16_t                                                         pDeviceHandle;                                                          /**< Handle of PicoScope Osci device. */
    uint8_t                                                         bPortOpen;                                                              /**< Internal State if port is opened. */
    uint32_t                                                        dwAdditionalInfo;                                                       /**< Storage for additional error information. */
    uint16_t                                                        wTimeout;                                                               /**< Shadow for the timeout. */
    void *                                                          pLibraryHandle;                                                         /**< Handle of the PS6000.dll. */
    uint8_t                                                         pDriverVersion[PHBAL_REG_PS6000_GET_UNIT_INFO_BUFFER_SIZE];             /**< This value stores the Version number of PicoScope 6000 DLL */
    uint8_t                                                         pUSBVersion[PHBAL_REG_PS6000_GET_UNIT_INFO_BUFFER_SIZE];                /**< This value stores the Type of USB connection to device: 1.1, 2.0 or 3.0 */
    uint8_t                                                         pHardwareVersion[PHBAL_REG_PS6000_GET_UNIT_INFO_BUFFER_SIZE];           /**< This value stores the Hardware version of device */
    uint8_t                                                         pVariantInfo[PHBAL_REG_PS6000_GET_UNIT_INFO_BUFFER_SIZE];               /**< This value stores the Model number of device */
    uint8_t                                                         pBatchAndSerial[PHBAL_REG_PS6000_GET_UNIT_INFO_BUFFER_SIZE];            /**< This value stores the Batch and serial number of device */
    uint8_t                                                         pCalibrationDate[PHBAL_REG_PS6000_GET_UNIT_INFO_BUFFER_SIZE];           /**< This value stores the Calibration date of device */
    uint8_t                                                         pKernelVersion[PHBAL_REG_PS6000_GET_UNIT_INFO_BUFFER_SIZE];             /**< This value stores the Version of kernel driver */
    uint8_t                                                         pDigitalHardwareVersion[PHBAL_REG_PS6000_GET_UNIT_INFO_BUFFER_SIZE];    /**< This value stores the Hardware version of the digital section */
    uint8_t                                                         pAnalogueHardwareVersion[PHBAL_REG_PS6000_GET_UNIT_INFO_BUFFER_SIZE];   /**< This value stores the Hardware version of the analogue section */
    uint8_t                                                         pFirmwareVersion1[PHBAL_REG_PS6000_GET_UNIT_INFO_BUFFER_SIZE];          /**< This value stores the Version information of Firmware 1 */
    uint8_t                                                         pFirmwareVersion2[PHBAL_REG_PS6000_GET_UNIT_INFO_BUFFER_SIZE];          /**< This value stores the Version information of Firmware 2 */
    PHBAL_REG_PS6000USB_OPEN_UNIT                                   pfPs6000Usb_OpenUnit;                                                   /**< function pointer for pfPs6000Usb_OpenUnit. */
    PHBAL_REG_PS6000USB_OPEN_UNIT_ASYNC                             pfPs6000Usb_OpenUnitAsync;                                              /**< function pointer for pfPs6000Usb_OpenUnitAsync. */
    PHBAL_REG_PS6000USB_OPEN_UNIT_PROGRESS                          pfPs6000Usb_OpenUnitProgress;                                           /**< function pointer for pfPs6000Usb_OpenUnitProgress. */
    PHBAL_REG_PS6000USB_GET_UNIT_INFO                               pfPs6000Usb_GetUnitInfo;                                                /**< function pointer for pfPs6000Usb_GetUnitInfo. */
    PHBAL_REG_PS6000USB_FLASH_LED                                   pfPs6000Usb_FlashLed;                                                   /**< function pointer for pfPs6000Usb_FlashLed. */
    PHBAL_REG_PS6000USB_CLOSE_UNIT                                  pfPs6000Usb_CloseUnit;                                                  /**< function pointer for pfPs6000Usb_CloseUnit. */
    PHBAL_REG_PS6000USB_MEMORY_SEGMENTS                             pfPs6000Usb_MemorySegments;                                             /**< function pointer for pfPs6000Usb_MemorySegments. */
    PHBAL_REG_PS6000USB_SET_CHANNEL                                 pfPs6000Usb_SetChannel;                                                 /**< function pointer for pfPs6000Usb_SetChannel. */
    PHBAL_REG_PS6000USB_GET_TIMEBASE                                pfPs6000Usb_GetTimebase;                                                /**< function pointer for pfPs6000Usb_GetTimebase. */
    PHBAL_REG_PS6000USB_GET_TIMEBASE2                               pfPs6000Usb_GetTimebase2;                                               /**< function pointer for pfPs6000Usb_GetTimebase2. */
    PHBAL_REG_PS6000USB_SET_SIG_GEN_ARBITRARY                       pfPs6000Usb_SetSigGenArbitrary;                                         /**< function pointer for pfPs6000Usb_SetSigGenArbitrary. */
    PHBAL_REG_PS6000USB_SET_SIG_GEN_BUILT_IN                        pfPs6000Usb_SetSigGenBuiltIn;                                           /**< function pointer for pfPs6000Usb_SetSigGenBuiltIn. */
    PHBAL_REG_PS6000USB_SIG_GEN_SOFTWARE_CONTROL                    pfPs6000Usb_SigGenSoftwareControl;                                      /**< function pointer for pfPs6000Usb_SigGenSoftwareControl. */
    PHBAL_REG_PS6000USB_SET_SIMPLE_TRIGGER                          pfPs6000Usb_SetSimpleTrigger;                                           /**< function pointer for pfPs6000Usb_SetSimpleTrigger. */
    PHBAL_REG_PS6000USB_SET_ETS                                     pfPs6000Usb_SetEts;                                                     /**< function pointer for pfPs6000Usb_SetEts. */
    PHBAL_REG_PS6000USB_SET_TRIGGER_CHANNEL_PROPERTIES              pfPs6000Usb_SetTriggerChannelProperties;                                /**< function pointer for pfPs6000Usb_SetTriggerChannelProperties. */
    PHBAL_REG_PS6000USB_SET_TRIGGER_CHANNEL_CONDITIONS              pfPs6000Usb_SetTriggerChannelConditions;                                /**< function pointer for pfPs6000Usb_SetTriggerChannelConditions. */
    PHBAL_REG_PS6000USB_SET_TRIGGER_CHANNEL_DIRECTIONS              pfPs6000Usb_SetTriggerChannelDirections;                                /**< function pointer for pfPs6000Usb_SetTriggerChannelDirections. */
    PHBAL_REG_PS6000USB_SET_TRIGGER_DELAY                           pfPs6000Usb_SetTriggerDelay;                                            /**< function pointer for pfPs6000Usb_SetTriggerDelay. */
    PHBAL_REG_PS6000USB_SET_PULSE_WIDTH_QUALIFIER                   pfPs6000Usb_SetPulseWidthQualifier;                                     /**< function pointer for pfPs6000Usb_SetPulseWidthQualifier. */
    PHBAL_REG_PS6000USB_IS_TRIGGER_OR_PULSE_WIDTH_QUALIFIER_ENABLED pfPs6000Usb_IsTriggerOrPulseWidthQualifierEnabled;                      /**< function pointer for pfPs6000Usb_IsTriggerOrPulseWidthQualifierEnabled. */
    PHBAL_REG_PS6000USB_GET_TRIGGER_TIME_OFFSET                     pfPs6000Usb_GetTriggerTimeOffset;                                       /**< function pointer for pfPs6000Usb_GetTriggerTimeOffset. */
    PHBAL_REG_PS6000USB_GET_TRIGGER_TIME_OFFSET64                   pfPs6000Usb_GetTriggerTimeOffset64;                                     /**< function pointer for pfPs6000Usb_GetTriggerTimeOffset64. */
    PHBAL_REG_PS6000USB_GET_VALUES_TRIGGER_TIME_OFFSET_BULK         pfPs6000Usb_GetValuesTriggerTimeOffsetBulk;                             /**< function pointer for pfPs6000Usb_GetValuesTriggerTimeOffsetBulk. */
    PHBAL_REG_PS6000USB_GET_VALUES_TRIGGER_TIME_OFFSET_BULK64       pfPs6000Usb_GetValuesTriggerTimeOffsetBulk64;                           /**< function pointer for pfPs6000Usb_GetValuesTriggerTimeOffsetBulk64. */
    PHBAL_REG_PS6000USB_SET_DATA_BUFFERS                            pfPs6000Usb_SetDataBuffers;                                             /**< function pointer for pfPs6000Usb_SetDataBuffers. */
    PHBAL_REG_PS6000USB_SET_DATA_BUFFER                             pfPs6000Usb_SetDataBuffer;                                              /**< function pointer for pfPs6000Usb_SetDataBuffer. */
    PHBAL_REG_PS6000USB_SET_DATA_BUFFER_BULK                        pfPs6000Usb_SetDataBufferBulk;                                          /**< function pointer for pfPs6000Usb_SetDataBufferBulk. */
    PHBAL_REG_PS6000USB_SET_DATA_BUFFERS_BULK                       pfPs6000Usb_SetDataBuffersBulk;                                         /**< function pointer for pfPs6000Usb_SetDataBuffersBulk. */
    PHBAL_REG_PS6000USB_SET_ETS_TIME_BUFFER                         pfPs6000Usb_SetEtsTimeBuffer;                                           /**< function pointer for pfPs6000Usb_SetEtsTimeBuffer. */
    PHBAL_REG_PS6000USB_SET_ETS_TIME_BUFFERS                        pfPs6000Usb_SetEtsTimeBuffers;                                          /**< function pointer for pfPs6000Usb_SetEtsTimeBuffers. */
    PHBAL_REG_PS6000USB_RUN_BLOCK                                   pfPs6000Usb_RunBlock;                                                   /**< function pointer for pfPs6000Usb_RunBlock. */
    PHBAL_REG_PS6000USB_IS_READY                                    pfPs6000Usb_IsReady;                                                    /**< function pointer for pfPs6000Usb_IsReady. */
    PHBAL_REG_PS6000USB_RUN_STREAMING                               pfPs6000Usb_RunStreaming;                                               /**< function pointer for pfPs6000Usb_RunStreaming. */
    PHBAL_REG_PS6000USB_GET_STREAMING_LATEST_VALUES                 pfPs6000Usb_GetStreamingLatestValues;                                   /**< function pointer for pfPs6000Usb_GetStreamingLatestValues. */
    PHBAL_REG_PS6000USB_NO_OF_STREAMING_VALUES                      pfPs6000Usb_NoOfStreamingValues;                                        /**< function pointer for pfPs6000Usb_NoOfStreamingValues. */
    PHBAL_REG_PS6000USB_GET_MAX_DOWN_SAMPLE_RATIO                   pfPs6000Usb_GetMaxDownSampleRatio;                                      /**< function pointer for pfPs6000Usb_GetMaxDownSampleRatio. */
    PHBAL_REG_PS6000USB_GET_VALUES                                  pfPs6000Usb_GetValues;                                                  /**< function pointer for pfPs6000Usb_GetValues. */
    PHBAL_REG_PS6000USB_GET_VALUES_BULK                             pfPs6000Usb_GetValuesBulk;                                              /**< function pointer for pfPs6000Usb_GetValuesBulk. */
    PHBAL_REG_PS6000USB_GET_VALUES_ASYNC                            pfPs6000Usb_GetValuesAsync;                                             /**< function pointer for pfPs6000Usb_GetValuesAsync. */
    PHBAL_REG_PS6000USB_GET_VALUES_OVERLAPPED                       pfPs6000Usb_GetValuesOverlapped;                                        /**< function pointer for pfPs6000Usb_GetValuesOverlapped. */
    PHBAL_REG_PS6000USB_GET_VALUES_OVERLAPPED_BULK                  pfPs6000Usb_GetValuesOverlappedBulk;                                    /**< function pointer for pfPs6000Usb_GetValuesOverlappedBulk. */
    PHBAL_REG_PS6000USB_GET_VALUES_BULK_ASYNC                       pfPs6000Usb_GetValuesBulkAsync;                                         /**< function pointer for pfPs6000Usb_GetValuesBulkAsync. */
    PHBAL_REG_PS6000USB_GET_NO_OF_CAPTURES                          pfPs6000Usb_GetNoOfCaptures;                                            /**< function pointer for pfPs6000Usb_GetNoOfCaptures. */
    PHBAL_REG_PS6000USB_STOP                                        pfPs6000Usb_Stop;                                                       /**< function pointer for pfPs6000Usb_Stop. */
    PHBAL_REG_PS6000USB_SET_NO_OF_CAPTURES                          pfPs6000Usb_SetNoOfCaptures;                                            /**< function pointer for pfPs6000Usb_SetNoOfCaptures. */
    PHBAL_REG_PS6000USB_SET_WAVEFORM_LIMITER                        pfPs6000Usb_SetWaveformLimiter;                                         /**< function pointer for pfPs6000Usb_SetWaveformLimiter. */
    PHBAL_REG_PS6000USB_ENUMERATE_UNITS                             pfPs6000Usb_EnumerateUnits;                                             /**< function pointer for pfPs6000Usb_EnumerateUnits. */
    PHBAL_REG_PS6000USB_SET_EXTERNAL_CLOCK                          pfPs6000Usb_SetExternalClock;                                           /**< function pointer for pfPs6000Usb_SetExternalClock. */
    PHBAL_REG_PS6000USB_PING_UNIT                                   pfPs6000Usb_PingUnit;                                                   /**< function pointer for pfPs6000Usb_PingUnit. */
} phbalReg_Ps6000Usb_DataParams_t;

/**
 * \brief Initialise the PS6000USB BAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS Parameter structure size is invalid.
 * \retval #PH_ERR_USE_CONDITION Required dependencies not available.
 */
phStatus_t phbalReg_Ps6000Usb_Init(
        phbalReg_Ps6000Usb_DataParams_t * pDataParams,                                                                                      /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams                                                                                                          /**< [In] Specifies the size of the data parameter structure. */
    );

/**
 * end of group phbalReg_Ps6000Usb
 * @}
 */
#endif /* NXPBUILD__PHBAL_REG_PS6000USB */

#ifdef NXPBUILD__PHBAL_REG_RD70XUSBWIN

/**
 * \defgroup phbalReg_Rd70xUsbWin Component : Rd70x USB (Windows)
 * \brief Rd70x BAL (Windows)
 * @{
 */

#define PHBAL_REG_RD70X_USBWIN_ID                                                           0x05U   /**< ID for PEGODA USB component */

/** \brief Rd70x USB (Windows) BAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                                   /**< Layer ID for this BAL component, NEVER MODIFY! */
    uint8_t ucTxSeq;                                                                                /**< Sequence counter for packets. */
    uint8_t * pDeviceName;                                                                          /**< Name of currently open USB device. */
    void * pDeviceHandle;                                                                           /**< Handle to the USB device. */
    void * pPipeOut;                                                                                /**< Handle to Usb Out-pipe */
    void * pPipeIn;                                                                                 /**< Handle to Usb In-pipe. */
    uint16_t wTimeoutWrMs;                                                                          /**< TO value for Usb Write pipe transfer. */
    uint16_t wTimeoutRdMs;                                                                          /**< TO value for Usb Read pipe transfer. */
} phbalReg_Rd70xUsbWin_DataParams_t;

/**
 * \brief Initialise the PEGODA USB BAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS Parameter structure size is invalid.
 */
phStatus_t phbalReg_Rd70xUsbWin_Init(
        phbalReg_Rd70xUsbWin_DataParams_t * pDataParams,                                            /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams                                                                  /**< [In] Specifies the size of the data parameter structure. */
    );

/**
 * end of group phbalReg_Rd70xUsbWin
 * @}
 */
#endif /* NXPBUILD__PHBAL_REG_RD70XUSBWIN */

#ifdef NXPBUILD__PHBAL_REG_PCSCWIN

/**
 * \defgroup phbalReg_PcscWin Component : PCSC (Windows)
 * \brief PC/SC BAL Windows
 *
 * Smartcard Interface.
 * @{
 */

#define PHBAL_REG_PCSCWIN_ID                                                                0x06U   /**< ID for PCSC BAL component (Win32) */

/** \name Custom Option Bits: Can be used for \ref phhalHw_Exchange. */
/** @{ */

/**
 * \brief Calls \b SCardControl with given parameters.
 *
 * Make sure \b pTxBuffer contents are as follows:
 * \code
 *        pTxBuffer[0]                pTxBuffer[1]               pTxBuffer[2]               pTxBuffer[3]           [pTxBuffer[4-n]]
 * +--------------------------+--------------------------+--------------------------+--------------------------+----------------------+
 * | dwControlCode(bit 31-24) | dwControlCode(bit 23-16) |  dwControlCode(bit 15-8) |  dwControlCode(bit 7-0)  |       [Data]         |
 * +--------------------------+--------------------------+--------------------------+--------------------------+----------------------+
 * \endcode
 */
#define PHBAL_REG_PCSCWIN_OPTION_SCARDCONTROL                                           0x0010U
/** @} */

/** \brief PCSC (Windows) BAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                                   /**< Layer ID for this BAL component, NEVER MODIFY! */
    uint8_t bProtocolType;                                                                          /**< Define protocol type of contact interface that is required during open. */
    uint8_t bActiveProtocolType;                                                                    /**< Define protocol type that is active after open. */
    uint8_t bShare;                                                                                 /**< Define share type of PCSC interface. */
    uint8_t bPegoda2Mode;                                                                           /**< This is mainly for Pegoda 2 (RD710) reader usage. To be set only if firmware higher that 2.2.7 is used. */
    uint8_t * pAtrBuffer;                                                                           /**< Pointer to buffer used for storing the ATR after card activation. */
    uint8_t bUseWarmATR;                                                                            /**< Indicates that the warm activation should be used */
    uint16_t wMaxAtrBufSize;                                                                        /**< The maximum size of the ATR buffer. */
    uint16_t wAtrBufSize;                                                                           /**< Size of the ATR buffer. */
    uint8_t * pVendorName;                                                                          /**< Pointer to buffer used for storing the vendor name. */
    uint16_t wVendorNameSize;                                                                       /**< Size of the VendorName buffer. */
    uint8_t * pVendorIFDType;                                                                       /**< Pointer to buffer used for storing the vendor ifd type. */
    uint16_t wVendorIFDTypeSize;                                                                    /**< Size of the VendorIFDType buffer. */
    uint8_t * pVendorIFDVersion;                                                                    /**< Pointer to buffer used for storing the vendor ifd version. */
    uint16_t wVendorIFDVersionSize;                                                                 /**< Size of the VendorIFDVersion buffer. */
    uint8_t * pVendorIFDSerial;                                                                     /**< Pointer to buffer used for storing the vendor ifd serial. */
    uint16_t wVendorIFDSerialSize;                                                                  /**< Size of the VendorIFDSerial buffer. */
    void * hContext;                                                                                /**< Context for SmartCard class. */
    void * hCard;                                                                                   /**< Handle of card. */
    uint8_t * pReaderName;                                                                          /**< Name of the reader. */
    uint16_t wTimeoutWrMs;                                                                          /**< TO value for Write transfer. */
    uint16_t wTimeoutRdMs;                                                                          /**< TO value for Read transfer. */
} phbalReg_PcscWin_DataParams_t;

/** \name BAL Communication Configs */
/** @{ */
#define PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL                           (PH_CONFIG_CUSTOM_BEGIN + 0)    /**< Used Protocol; e.g. #PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_T0. */
#define PHBAL_REG_PCSCWIN_CONFIG_SHARE                              (PH_CONFIG_CUSTOM_BEGIN + 1)    /**< Used Access mode; e.g. #PHBAL_REG_PCSCWIN_VALUE_SHARE_SHARED. */
#define PHBAL_REG_PCSCWIN_CONFIG_USE_WARM_ATR                       (PH_CONFIG_CUSTOM_BEGIN + 2)    /**< Define if the warm atr should used. Values: PH_ON, PH_OFF. */
#define PHBAL_REG_PCSCWIN_CONFIG_CARD_STATE                         (PH_CONFIG_CUSTOM_BEGIN + 3)    /**< Current state of the smart card in the reader. See SCardStatus */
#define PHBAL_REG_PCSCWIN_CONFIG_ACTIVE_PROTOCOL                    (PH_CONFIG_CUSTOM_BEGIN + 4)    /**< Get the used Protocol; e.g. #PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_T0. */
#define PHBAL_REG_PCSCWIN_CONFIG_PEGODA2_MODE                       (PH_CONFIG_CUSTOM_BEGIN + 5)    /**< Define if the Pegoda 2 Mode should used. Values: PH_ON, PH_OFF. */
/** @} */

/** \name BAL Communication Configuration values */
/** @{ */
#define PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_T0                                                 0x0000U /**< T=0 protocol. */
#define PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_T1                                                 0x0001U /**< T=1 protocol. */
#define PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_RAW                                                0x0002U /**< RAW protocol. */
#define PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_AUTO                                               0x0003U /**< Use T=0|T=1|RAW protocol. */
#define PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_UNDEFINED                                          0x0004U /**< UNDEFINED protocol. */
#define PHBAL_REG_PCSCWIN_VALUE_SHARE_SHARED                                                0x0000U /**< SHARED access. */
#define PHBAL_REG_PCSCWIN_VALUE_SHARE_EXCLUSIVE                                             0x0001U /**< EXCLUSIVE access. */
#define PHBAL_REG_PCSCWIN_VALUE_SHARE_DIRECT                                                0x0002U /**< DIRECT access. */
/** @} */

/**
 * \brief Initialise the PCSC BAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS Parameter structure size is invalid.
 * \retval #PH_ERR_INVALID_PARAMETER \b wAtrBufSize is invalid.
 */
phStatus_t phbalReg_PcscWin_Init(
        phbalReg_PcscWin_DataParams_t * pDataParams,                                                /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                                 /**< [In] Specifies the size of the data parameter structure. */
        uint8_t * pAtrBuffer,                                                                       /**< [In] Pointer to buffer used for storing the ATR after card activation. */
        uint16_t wAtrBufSize                                                                        /**< [In] Size of the ATR buffer. */
    );

/**
 * end of group phbalReg_PcscWin
 * @}
 */
#endif /* NXPBUILD__PHBAL_REG_PCSCWIN */

#ifdef NXPBUILD__PHBAL_REG_PIPELIN

/**
 * \defgroup phbalReg_PipeLin Component : Pipe (Linux)
 * \brief Pipe BAL (Unix/Linux)
 *
 * Interfaces NCSim Simulation over Pipes.
 * @{
 */

#define PHBAL_REG_PIPELIN_ID            0x07U                                                       /**< ID for Pipe Linux */

#define PHBAL_REG_PIPELIN_MODE_RAW      0x00U                                                       /**< RAW exchange mode. Exchanges null-terminated strings. */
#define PHBAL_REG_PIPELIN_MODE_RS232    0x01U                                                       /**< RS232 exchange mode according to Maat DigiSpec; Exchanges binary data. */
#define PHBAL_REG_PIPELIN_MODE_I2C      0x02U                                                       /**< I2C exchange mode according to Maat DigiSpec; Exchanges binary data. */
#define PHBAL_REG_PIPELIN_MODE_SPI      0x03U                                                       /**< SPI exchange mode according to Maat DigiSpec; Exchanges binary data. */

/** \name PipeLin specific configs */
/** @{ */
#define PHBAL_REG_PIPELIN_CONFIG_MODE                               (PH_CONFIG_CUSTOM_BEGIN + 0)    /**< Set operating mode. (e.g. #PHBAL_REG_PIPELIN_MODE_RS232). */
#define PHBAL_REG_PIPELIN_CONFIG_BITRATE                            (PH_CONFIG_CUSTOM_BEGIN + 1)    /**< Configure communication speed. */
#define PHBAL_REG_PIPELIN_CONFIG_I2C_SLAVE_ADDR                     (PH_CONFIG_CUSTOM_BEGIN + 2)    /**< Configure slave address for I2C communication mode. */
#define PHBAL_REG_PIPELIN_CONFIG_SETCARD                            (PH_CONFIG_CUSTOM_BEGIN + 3)    /**< Set card configuration for simulation. */
#define PHBAL_REG_PIPELIN_CONFIG_RESET                              (PH_CONFIG_CUSTOM_BEGIN + 4)    /**< Reset the simulation. */
/** @} */

/** \name PipeLin Bitrate Values */
/** @{ */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_RS232_9600                                          0x0000U /**< 9600 kBit/s RS232. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_RS232_19200                                         0x0001U /**< 19200 kBit/s RS232. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_RS232_38400                                         0x0002U /**< 38400 kBit/s RS232. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_RS232_57600                                         0x0003U /**< 57600 kBit/s RS232. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_RS232_115200                                        0x0004U /**< 115200 kBit/s RS232. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_RS232_128000                                        0x0005U /**< 128000 kBit/s RS232. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_RS232_230400                                        0x0006U /**< 230400 kBit/s RS232. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_RS232_460800                                        0x0007U /**< 460800 kBit/s RS232. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_RS232_921600                                        0x0008U /**< 921600 kBit/s RS232. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_RS232_1228800                                       0x0009U /**< 1228800 kBit/s RS232. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_SPI_10K                                             0x0010U /**< 10 kBit/s SPI. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_SPI_100K                                            0x0011U /**< 100 kBit/s SPI. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_SPI_1M                                              0x0012U /**< 1 MBit/s SPI. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_SPI_3M                                              0x0013U /**< 3 MBit/s SPI. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_SPI_5M                                              0x0014U /**< 5 MBit/s SPI. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_SPI_8M                                              0x0015U /**< 8 MBit/s SPI. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_I2C_10K                                             0x0020U /**< 10 KBit/s I2C. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_I2C_100K                                            0x0021U /**< 100 KBit/s I2C. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_I2C_200K                                            0x0022U /**< 200 KBit/s I2C. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_I2C_400K                                            0x0023U /**< 400 KBit/s I2C. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_I2CL_1M                                             0x0044U /**< 1 MBit/s I2CL. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_I2CL_3M                                             0x0045U /**< 3 MBit/s I2CL. */
#define PHBAL_REG_PIPELIN_VALUE_BITRATE_I2CL_5M                                             0x0046U /**< 5 MBit/s I2CL. */
/** @} */

/**
 * \name PipeLin Bitrate Values
 * Note: Set in MSB(yte), they set the TX-Mode, in the LSB the RX-Mode.
 */
/** @{ */
#define PHBAL_REG_PIPELIN_TX_TWO_CARDS                                                      0x80U   /**< Set this in the MSB(yte) to have two cards instead of one. */
#define PHBAL_REG_PIPELIN_RXTX_MIFARE_106                                                   0x00U   /**< MIFARE(R) Operating mode at 106kbit/s. */
#define PHBAL_REG_PIPELIN_RXTX_MIFARE_212                                                   0x01U   /**< MIFARE(R) Operating mode at 212kbit/s. */
#define PHBAL_REG_PIPELIN_RXTX_MIFARE_424                                                   0x02U   /**< MIFARE(R) Operating mode at 414kbit/s. */
#define PHBAL_REG_PIPELIN_RXTX_MIFARE_848                                                   0x03U   /**< MIFARE(R) Operating mode at 848kbit/s. */

#define PHBAL_REG_PIPELIN_RXTX_TYPEB_106                                                    0x04U   /**< ISO14443B Operating mode at 106kbit/s. */
#define PHBAL_REG_PIPELIN_RXTX_TYPEB_212                                                    0x05U   /**< ISO14443B Operating mode at 212kbit/s. */
#define PHBAL_REG_PIPELIN_RXTX_TYPEB_424                                                    0x06U   /**< ISO14443B Operating mode at 414kbit/s. */
#define PHBAL_REG_PIPELIN_RXTX_TYPEB_848                                                    0x07U   /**< ISO14443B Operating mode at 848kbit/s. */

#define PHBAL_REG_PIPELIN_RXTX_FELICA_212                                                   0x08U   /**< FeliCa Operating mode at 212kbit/s. */
#define PHBAL_REG_PIPELIN_RXTX_FELICA_424                                                   0x09U   /**< FeliCa Operating mode at 424kbit/s. */

#define PHBAL_REG_PIPELIN_RXTX_I15693_1OO4                                                  0x0AU   /**< ISO15693 One-Out-Of-Four Operating mode. */
#define PHBAL_REG_PIPELIN_RXTX_I15693_1OO256                                                0x0BU   /**< ISO15693 One-Out-Of-256 Operating mode. */

#define PHBAL_REG_PIPELIN_RXTX_EPCV2_424_DUAL                                               0x0FU   /**< ICode EPC V2 424 kBit/s (2 Subcarriers). */
#define PHBAL_REG_PIPELIN_RXTX_EPCV2_424_QUAD                                               0x10U   /**< ICode EPC V2 424 kBit/s (4 Subcarriers). */
#define PHBAL_REG_PIPELIN_RXTX_EPCV2_848_DUAL                                               0x11U   /**< ICode EPC V2 848 kBit/s (2 Subcarriers). */
#define PHBAL_REG_PIPELIN_RXTX_EPCV2_848_QUAD                                               0x12U   /**< ICode EPC V2 848 kBit/s (4 Subcarriers). */
/** @} */

/** \brief BAL PipeLin parameter structure */
typedef struct
{
    uint16_t wId;                                                                                   /**< Layer ID for this BAL component, NEVER MODIFY! */
    uint8_t bMode;                                                                                  /**< Operating mode. */
    uint8_t * pPipeNameRx;                                                                          /**< Identifier of the named Rx-Pipe. */
    uint8_t * pPipeNameTx;                                                                          /**< Identifier of the named Tx-Pipe. */
    void * pPipeRx;                                                                                 /**< Pointer to RX communication pipe. */
    void * pPipeTx;                                                                                 /**< Pointer to TX communication pipe. */
    uint8_t * pTxBuffer;                                                                            /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                            /**< Size of the global transmit buffer. */
    uint8_t bI2cSlaveAddr;                                                                          /**< I2C slave address. */
    uint8_t bConfigValue;                                                                           /**< current configuration value. */
} phbalReg_PipeLin_DataParams_t;

/**
 * \brief Initialize the BAL PipeLinux layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS Parameter structure size is invalid.
 * \retval #PH_ERR_INVALID_PARAMETER \b bMode or \b wTxBufSize is invalid.
 */
phStatus_t phbalReg_PipeLin_Init(
        phbalReg_PipeLin_DataParams_t * pDataParams,                                                /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                                 /**< [In] Specifies the size of the data parameter structure. */
        uint8_t bMode,                                                                              /**< [In] Initial operating mode; \see #PHBAL_REG_PIPELIN_CONFIG_MODE. */
        uint16_t wTxBufSize,                                                                        /**< [In] Size of the global transmit buffer. */
        uint8_t * pTxBuffer                                                                         /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
    );

/**
 * end of group phbalReg_PipeLin
 * @}
 */
#endif /* NXPBUILD__PHBAL_REG_PIPELIN */

#ifdef NXPBUILD__PHBAL_REG_STUB

/**
 * \defgroup phbalReg_Stub Component : Stub
 * \brief Component without functionality to ease implementation of additional busses.
 * @{
 */

#define PHBAL_REG_STUB_ID                                                                   0x08U   /**< ID for Stub BAL component */

/** \brief BAL Stub parameter structure */
typedef struct
{
    uint16_t       wId;                                                                             /**< Layer ID for this BAL component, NEVER MODIFY! */
    /* TODO: Place private parameters which are needed across different functions here. */
} phbalReg_Stub_DataParams_t;

/**
 * \brief Initialise the BAL Stub layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS Parameter structure size is invalid.
 */
phStatus_t phbalReg_Stub_Init(
        phbalReg_Stub_DataParams_t * pDataParams,                                                   /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams                                                                  /**< [In] Specifies the size of the data parameter structure. */
    );

/**
 * end of group phbalReg_Stub
 * @}
 */
#endif /* NXPBUILD__PHBAL_REG_STUB */

#ifdef NXPBUILD__PHBAL_REG_SOCKETWIN

/**
 * \defgroup phbalReg_SocketWin Component : Socket (Windows)
 * \brief Socket BAL (Windows)
 *
 * Interfaces NCSim Simulation over Network Sockets.
 * @{
 */

#define PHBAL_REG_SOCKETWIN_ID                                                              0x09U   /**< ID for SOCKETWIN component */

/** \name SocketWin specific configs */
/** @{ */
#define PHBAL_REG_SOCKETWIN_CONFIG_MODE                             (PH_CONFIG_CUSTOM_BEGIN + 0)    /**< Set operating mode. (e.g. #PHBAL_REG_SOCKETWIN_MODE_RS232). */
#define PHBAL_REG_SOCKETWIN_CONFIG_BITRATE                          (PH_CONFIG_CUSTOM_BEGIN + 1)    /**< Configure communication speed. */
#define PHBAL_REG_SOCKETWIN_CONFIG_I2C_SLAVE_ADDR                   (PH_CONFIG_CUSTOM_BEGIN + 2)    /**< Configure slave address for I2C communication mod. */
#define PHBAL_REG_SOCKETWIN_CONFIG_SETCARD                          (PH_CONFIG_CUSTOM_BEGIN + 3)    /**< Set card configuration for simulation. */
#define PHBAL_REG_SOCKETWIN_CONFIG_RESET                            (PH_CONFIG_CUSTOM_BEGIN + 4)    /**< Reset the simulation. */
/** @} */

/** \name SocketWin operation mode values */
/** @{ */
#define PHBAL_REG_SOCKETWIN_MODE_RAW                                                        0x00U   /**< RAW exchange mode. Exchanges null-terminated strings. */
#define PHBAL_REG_SOCKETWIN_MODE_RS232                                                      0x01U   /**< RS232 exchange mode according to Maat DigiSpec; Exchanges binary data. */
#define PHBAL_REG_SOCKETWIN_MODE_I2C                                                        0x02U   /**< I2C exchange mode according to Maat DigiSpec; Exchanges binary data. */
#define PHBAL_REG_SOCKETWIN_MODE_SPI                                                        0x03U   /**< SPI exchange mode according to Maat DigiSpec; Exchanges binary data. */
#define PHBAL_REG_SOCKETWIN_MODE_HEX                                                        0x04U   /**< RAW exchange mode. Exchanges of hex data. */
#define PHBAL_REG_SOCKETWIN_MODE_TELNET                                                     0x05U   /**< Telnet exchange mode. */
/** @} */

/** \name SocketWin Bitrate Values */
/** @{ */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_RS232_9600                                        0x0000U /**< 9600 kBit/s RS232. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_RS232_19200                                       0x0001U /**< 19200 kBit/s RS232. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_RS232_38400                                       0x0002U /**< 38400 kBit/s RS232. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_RS232_57600                                       0x0003U /**< 57600 kBit/s RS232. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_RS232_115200                                      0x0004U /**< 115200 kBit/s RS232. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_RS232_128000                                      0x0005U /**< 128000 kBit/s RS232. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_RS232_230400                                      0x0006U /**< 230400 kBit/s RS232. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_RS232_460800                                      0x0007U /**< 460800 kBit/s RS232. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_RS232_921600                                      0x0008U /**< 921600 kBit/s RS232. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_RS232_1228800                                     0x0009U /**< 1228800 kBit/s RS232. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_SPI_10K                                           0x0010U /**< 10 kBit/s SPI. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_SPI_100K                                          0x0011U /**< 100 kBit/s SPI. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_SPI_1M                                            0x0012U /**< 1 MBit/s SPI. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_SPI_3M                                            0x0013U /**< 3 MBit/s SPI. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_SPI_5M                                            0x0014U /**< 5 MBit/s SPI. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_SPI_8M                                            0x0015U /**< 8 MBit/s SPI. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_I2C_10K                                           0x0020U /**< 10 KBit/s I2C. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_I2C_100K                                          0x0021U /**< 100 KBit/s I2C. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_I2C_200K                                          0x0022U /**< 200 KBit/s I2C. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_I2C_400K                                          0x0023U /**< 400 KBit/s I2C. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_I2CL_1M                                           0x0044U /**< 1 MBit/s I2CL. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_I2CL_3M                                           0x0045U /**< 3 MBit/s I2CL. */
#define PHBAL_REG_SOCKETWIN_VALUE_BITRATE_I2CL_5M                                           0x0046U /**< 5 MBit/s I2CL. */
/** @} */

/**
 * \name SocketWin Card Values
 *
 * Note: Set in MSB(yte), they set the TX-Mode, in the LSB the RX-Mode.
 */
/** @{ */
#define PHBAL_REG_SOCKETWIN_TX_TWO_CARDS                                                    0x80U   /**< Set this in the MSB(yte) to have two cards instead of one. */
#define PHBAL_REG_SOCKETWIN_RXTX_MIFARE_106                                                 0x00U   /**< MIFARE(R) Operating mode at 106kbit/s. */
#define PHBAL_REG_SOCKETWIN_RXTX_MIFARE_212                                                 0x01U   /**< MIFARE(R) Operating mode at 212kbit/s. */
#define PHBAL_REG_SOCKETWIN_RXTX_MIFARE_424                                                 0x02U   /**< MIFARE(R) Operating mode at 424kbit/s. */
#define PHBAL_REG_SOCKETWIN_RXTX_MIFARE_848                                                 0x03U   /**< MIFARE(R) Operating mode at 848kbit/s. */

#define PHBAL_REG_SOCKETWIN_RXTX_TYPEB_106                                                  0x04U   /**< ISO14443B Operating mode at 106kbit/s. */
#define PHBAL_REG_SOCKETWIN_RXTX_TYPEB_212                                                  0x05U   /**< ISO14443B Operating mode at 212kbit/s. */
#define PHBAL_REG_SOCKETWIN_RXTX_TYPEB_424                                                  0x06U   /**< ISO14443B Operating mode at 424kbit/s. */
#define PHBAL_REG_SOCKETWIN_RXTX_TYPEB_848                                                  0x07U   /**< ISO14443B Operating mode at 848kbit/s. */

#define PHBAL_REG_SOCKETWIN_RXTX_FELICA_212                                                 0x08U   /**< FeliCa Operating mode at 212kbit/s. */
#define PHBAL_REG_SOCKETWIN_RXTX_FELICA_424                                                 0x09U   /**< FeliCa Operating mode at 424kbit/s. */

#define PHBAL_REG_SOCKETWIN_RXTX_I15693_1OO4                                                0x0AU   /**< ISO15693 One-Out-Of-Four Operating mode. */
#define PHBAL_REG_SOCKETWIN_RXTX_I15693_1OO256                                              0x0BU   /**< ISO15693 One-Out-Of-256 Operating mode. */

#define PHBAL_REG_SOCKETWIN_RXTX_EPCV2_424_DUAL                                             0x0FU   /**< ICode EPC V2 424 kBit/s (2 Subcarriers). */
#define PHBAL_REG_SOCKETWIN_RXTX_EPCV2_424_QUAD                                             0x10U   /**< ICode EPC V2 424 kBit/s (4 Subcarriers). */
#define PHBAL_REG_SOCKETWIN_RXTX_EPCV2_848_DUAL                                             0x11U   /**< ICode EPC V2 848 kBit/s (2 Subcarriers). */
#define PHBAL_REG_SOCKETWIN_RXTX_EPCV2_848_QUAD                                             0x12U   /**< ICode EPC V2 848 kBit/s (4 Subcarriers). */
/** @} */

/** \ Socket Win Callback Transfer Type. */
typedef enum
{
    SEND_DATA = 1,
    RECEIVE_DATA = 2
} TyTransferType;

/** \brief function pointer to register socketWin Callback. */
typedef void (*TySocketWinCallback)(uint8_t* data, uint32_t length, TyTransferType transferType);

/** \brief SocketWin(Windows) BAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                                   /**< Layer ID for this BAL component, NEVER MODIFY! */
    void * pWsaData;                                                                                /**< Pointer to WsaData buffer. */
    uint8_t * pSocketHostName;                                                                      /**< Socket Host Name String e.g. "localhost". */
    uint8_t * pSocketPortName;                                                                      /**< Socket Port Name String e.g. "1234". */
    uint8_t * pUserName;                                                                            /**< User Name String e.g. "nxp12607". */
    SOCKET_T dwServerSocket;                                                                        /**< Server Socket ID. */
    uint8_t * pTxBuffer;                                                                            /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                            /**< Size of the global transmit buffer. */
    uint8_t bMode;                                                                                  /**< Operating mode. */
    uint8_t bConfigValue;                                                                           /**< Current configuration value. */
    uint8_t bI2cSlaveAddr;                                                                          /**< I2C slave address. */
} phbalReg_SocketWin_DataParams_t;

/**
 * \brief Initialise the Socket Win BAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS Parameter structure size is invalid.
 * \retval #PH_ERR_INVALID_PARAMETER \b bMode or \b wTxBufSize is invalid.
 * \retval #PH_ERR_BUFFER_OVERFLOW \b pWsaBuffer size is too small.
 */
phStatus_t phbalReg_SocketWin_Init(
        phbalReg_SocketWin_DataParams_t * pDataParams,                                              /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                                 /**< [In] Specifies the size of the data parameter structure. */
        uint8_t bMode,                                                                              /**< [In] Initial operating mode; \see #PHBAL_REG_SOCKETWIN_CONFIG_MODE. */
        uint16_t wTxBufSize,                                                                        /**< [In] Size of the global transmit buffer. */
        uint8_t * pTxBuffer,                                                                        /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wWsaBufferSize,                                                                    /**< [In] Size of the Windsocket WSADATA structure buffer; Refer to WinSock documentation (should be about 400 bytes). */
        uint8_t * pWsaBuffer                                                                        /**< [In] Pointer to the Winsocket WSADATA structure buffer. */
    );

/**
 * end of group phbalReg_SocketWin
 * @}
 */
#endif /* NXPBUILD__PHBAL_REG_SOCKETWIN */

#ifdef NXPBUILD__PHBAL_REG_SAM

 /**< Macro replacement to call Bal exchange based on different platforms  */
#ifdef _WIN32
#define PHBALREG_EXCHANGE                                       phbalReg_Exchange
#else
#define PHBALREG_EXCHANGE                                       phbalReg_Sam_Exchange
#endif /* _WIN32 */

/**
 * \defgroup phbalReg_Sam Component : SAM
 * \brief To communicate to SAM using Pegoda Reader or DUT readers.
 *
 * \note
 *      - Implements SAM interface for Direct-Mode Communication.\n
 *        Needs to be linked between SAM HAL and the actual communication BAL that interfaces with the Pegoda or DUT readers.
 *      - By default TDA communication is configured.
 *        To enable I2C communication mode, use \ref phbalReg_SetConfig "Set Config" interface with \ref PHBAL_REG_SAM_CONFIG_COMMUNICATION_TYPE
 *        "Communication Type" as configuration identifier and \ref PHBAL_REG_SAM_COMMUNICATION_TYPE_I2C "I2C" as value.
 *      - Default 0x2B will be used as Slave Address and 200ms timeout for I2C Communication.
 * @{
 */

#define PHBAL_REG_SAM_ID                                                                    0x0AU   /**< ID for SAM component */

/**
 * \defgroup phbalReg_Defines Configuration
 * @{
 */

/**
 * \defgroup phbalReg_Defines_General Common
 * @{
 */
#define PHBAL_REG_SAM_CONFIG_COMMUNICATION_TYPE                     (PH_CONFIG_CUSTOM_BEGIN + 0U)   /**< Configure communication type.
                                                                                                     *      - \ref PHBAL_REG_SAM_COMMUNICATION_TYPE_TDA "TDA"
                                                                                                     *      - \ref PHBAL_REG_SAM_COMMUNICATION_TYPE_I2C "I2C"
                                                                                                     */
#define PHBAL_REG_SAM_COMMUNICATION_TYPE_TDA                        0x01U                           /**< SAM Communication type as TDA. SAM APDU's will be exchanged
                                                                                                     *   via TDA interface.
                                                                                                     */
#define PHBAL_REG_SAM_COMMUNICATION_TYPE_I2C                        0x02U                           /**< SAM Communication type as I2C. SAM APDU's will be exchanged
                                                                                                     *   via I2C interface.
                                                                                                     */
/**
 * end of group phbalReg_Defines_General
 * @}
 */

/**
 * \defgroup phbalReg_Defines_I2C I2C
 * @{
 */
#define PHBAL_REG_SAM_CONFIG_I2C_SLAVE_ADDRESS                      (PH_CONFIG_CUSTOM_BEGIN + 1U)   /**< Configure I2C Slave Address. Default configured is 2Bh */
#define PHBAL_REG_SAM_CONFIG_I2C_TIMEOUT_MS                         (PH_CONFIG_CUSTOM_BEGIN + 2U)   /**< Configure I2C BitRate. By Default 200ms will be used. */
#define PHBAL_REG_SAM_CONFIG_I2C_BITRATE                            (PH_CONFIG_CUSTOM_BEGIN + 3U)   /**< Configure I2C BitRate. By Default 400 KBits/s will be used. */
#define PHBAL_REG_SAM_CONFIG_BITRATE_STANDARD_MODE                                          0x00U   /**< I2C bit rate configuration value for 100 KBits/s. */
#define PHBAL_REG_SAM_CONFIG_BITRATE_FAST_MODE                                              0x01U   /**< I2C bit rate configuration value for 400 KBits/s. */
#define PHBAL_REG_SAM_CONFIG_BITRATE_FAST_PLUS_MODE                                         0x02U   /**< I2C bit rate configuration value for 1 MBits/s. */
#define PHBAL_REG_SAM_CONFIG_BITRATE_HIGH_SPEED_MODE                                        0x03U   /**< I2C bit rate configuration value for 3.4 MBits/s. */
/**
 * end of group phbalReg_Defines_I2C
 * @}
 */

/**
 * end of group phbalReg_Defines
 * @}
 */

/** \brief Sam BAL parameter structure */
typedef struct
{
    uint16_t wId;                                                                                   /**< Layer ID for this BAL component, NEVER MODIFY! */
    void * pLowerBalDataParams;                                                                     /**< Pointer to the BAL which actually connects to reader in direct mode. */
    uint8_t * pAtrBuffer;                                                                           /**< Pointer to buffer used for storing the ATR after card activation. */
    uint16_t wMaxAtrBufSize;                                                                        /**< The maximum size of the ATR buffer. */
    uint16_t wAtrBufSize;                                                                           /**< Length of bytes available in ATR buffer. This will be filled once ATR is
                                                                                                     *   received from SAM.
                                                                                                     */
    uint8_t * pTxBuffer;                                                                            /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                            /**< Size of the global transmit buffer. */
    uint8_t * pRxBuffer;                                                                            /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                            /**< Size of the global receive buffer. */
    uint8_t bCommType;                                                                              /**< Communication type to use  for exchanging with SAM. Will be one of
                                                                                                     *   the following,
                                                                                                     *      - \ref PHBAL_REG_SAM_COMMUNICATION_TYPE_TDA "ISO7816 Interface"
                                                                                                     *      - \ref PHBAL_REG_SAM_COMMUNICATION_TYPE_I2C "I2C T = 1 Interface"
                                                                                                     *
                                                                                                     */
    uint8_t bIsPortOpened;                                                                          /**< Indicator if connection to SAM was established already. */
    uint16_t wI2C_SlaveAddr;                                                                        /**< Address of I2C Slave. */
    uint16_t wI2C_TimeoutMs;                                                                        /**< For saving / configuring I2C communication timeout. */
    uint8_t  bI2C_BitRate;                                                                              /**< I2C BitRate. One of the following
                                                                                                     *      - \ref PHBAL_REG_SAM_CONFIG_BITRATE_STANDARD_MODE "100 KBits/s"
                                                                                                     *      - \ref PHBAL_REG_SAM_CONFIG_BITRATE_FAST_MODE "400 KBits/s"
                                                                                                     *      - \ref PHBAL_REG_SAM_CONFIG_BITRATE_FAST_PLUS_MODE "1 MBits/s"
                                                                                                     *      - \ref PHBAL_REG_SAM_CONFIG_BITRATE_HIGH_SPEED_MODE "3.4 MBits/s"
                                                                                                     */
    uint8_t bI2C_PcbBlockNum;                                                                       /**< Protocol Control Byte. */
    uint8_t bI2C_NextFrame;                                                                         /**< Next Frame Type. Whether its I Frame or S Frame */
} phbalReg_Sam_DataParams_t;

/**
 * \brief Initialize the SAM BAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS Parameter structure size is invalid.
 * \retval #PH_ERR_INVALID_PARAMETER \b wAtrBufSize, \b wTxBufSize or \b wRxBufSize is invalid.
 */
phStatus_t phbalReg_Sam_Init(
        phbalReg_Sam_DataParams_t * pDataParams,                                                    /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                                 /**< [In] Specifies the size of the data parameter structure. */
        void * pLowerBalDataParams,                                                                 /**< [In] Pointer to the BAL which actually connects to Pegoda or DUT
                                                                                                     *        reader in direct mode.
                                                                                                     */
        uint8_t * pAtrBuffer,                                                                       /**< [In] Pointer to buffer used for storing the ATR after card activation. */
        uint16_t wAtrBufSize,                                                                       /**< [In] Size of the ATR buffer. */
        uint8_t * pTxBuffer,                                                                        /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wTxBufSize,                                                                        /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                                                        /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t wRxBufSize                                                                         /**< [In] Size of the global receive buffer. */
    );

/**
 * end of group phbalReg_Sam
 * @}
 */
#endif /* NXPBUILD__PHBAL_REG_SAM */

#ifdef NXPBUILD__PHBAL_REG_RD710SAM

/**
 * \deprecated
 * RD710SAM component is obsolete and will be removed.
 * Refer \ref phbalReg_Sam "Bal_SAM" component for accessing SAM using Pegoda or DUT reader(s).
 *
 * \defgroup phbalReg_Rd710Sam Component : Rd710 SAM
 * \brief Rd710 SAM BAL
 *
 * Implements SAM interface for Direct-Mode Communication.\n
 * Needs to be linked between SAM HAL and the BAL that interfaces with the Rd710.
 * @{
 */

#define PHBAL_REG_RD710_SAM_ID                                                              0x14U   /**< ID for RD710_SAM component */

/**
* \brief RD710Sam (Windows) BAL parameter structure
*/
typedef struct
{
    uint16_t wId;                                                                                   /**< Layer ID for this BAL component, NEVER MODIFY! */
    void * pLowerBalDataParams;                                                                     /**< Pointer to the BAL which actually connects to Rd710 in direct mode. */
    uint8_t * pAtrBuffer;                                                                           /**< Pointer to buffer used for storing the ATR after card activation. */
    uint16_t wMaxAtrBufSize;                                                                        /**< The maximum size of the ATR buffer. */
    uint16_t wAtrBufSize;                                                                           /**< Size of the ATR buffer. */
    uint8_t * pTxBuffer;                                                                            /**< Pointer to global transmit buffer used by the Exchange() function. */
    uint16_t wTxBufSize;                                                                            /**< Size of the global transmit buffer. */
    uint8_t * pRxBuffer;                                                                            /**< Pointer to global receive buffer used by the Exchange() function. */
    uint16_t wRxBufSize;                                                                            /**< Size of the global receive buffer. */
    uint8_t bCardOpened;                                                                            /**< Indicator if card was already opened. */
} phbalReg_Rd710Sam_DataParams_t;

/**
 * \brief Initialise the RD710SAM BAL component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS Parameter structure size is invalid.
 * \retval #PH_ERR_INVALID_PARAMETER \b wAtrBufSize, \b wTxBufSize or \b wRxBufSize is invalid.
 */
phStatus_t phbalReg_Rd710Sam_Init(
        phbalReg_Rd710Sam_DataParams_t * pDataParams,                                               /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                                                 /**< [In] Specifies the size of the data parameter structure. */
        void * pLowerBalDataParams,                                                                 /**< [In] Pointer to the BAL which actually connects to Rd710 in direct mode. */
        uint8_t * pAtrBuffer,                                                                       /**< [In] Pointer to buffer used for storing the ATR after card activation. */
        uint16_t wAtrBufSize,                                                                       /**< [In] Size of the ATR buffer. */
        uint8_t * pTxBuffer,                                                                        /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        uint16_t wTxBufSize,                                                                        /**< [In] Size of the global transmit buffer. */
        uint8_t * pRxBuffer,                                                                        /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        uint16_t wRxBufSize                                                                         /**< [In] Size of the global receive buffer. */
    );

/**
 * end of group phbalReg_Rd710Sam
 * @}
 */
#endif /* NXPBUILD__PHBAL_REG_RD710SAM */








#ifdef NXPBUILD__PHBAL_REG

/**
 * \defgroup phbalReg Bus Abstraction Layer
 * \brief These Components implement the interfaces between physical Host-Device and physical Reader-Device.
 * @{
 */

/** \name Generic BAL Configs */
/** @{ */
#define PHBAL_REG_CONFIG_WRITE_TIMEOUT_MS                                                   0x0000U /**< Configure transmission timeout. */
#define PHBAL_REG_CONFIG_READ_TIMEOUT_MS                                                    0x0001U /**< Configure reception timeout. */
/** @} */

/** \name Custom Error Codes */
/** @{ */
#define PHBAL_REG_ERR_DLL                                               (PH_ERR_CUSTOM_BEGIN + 0)   /**< used dll threw an non standard error read wAddionalInfo to get error code */
/** @} */


/**
 * \brief List all available ports.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_BUFFER_OVERFLOW Given enumeration buffer is too small.
 * \retval #PH_ERR_INTERFACE_ERROR Error while enumerating devices.
 */
phStatus_t phbalReg_GetPortList(
        void * pDataParams,                                                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wPortBufSize,                                                                      /**< [In] Buffer Size of Port Name String. */
        uint8_t * pPortNames,                                                                       /**< [Out] Port Name as Multi-String. */
        uint16_t * pNumOfPorts                                                                      /**< [Out] Number of found port strings. */
    );

/**
 * \brief Select Port to be used.
 *
 * \note
 * The caller has to ensure that \c pPortName is valid throughout the whole lifetime of \c pDataParams.\n
 * Furthermore, the caller is responsible for prepending <var>\\\\\.\\</var> escape format if COM ports above COM9 need to be accessed.
 *
 * \b Example:
 * \code
 * strcpy(pPortName, "COM9");         <-- correct
 * strcpy(pPortName, "\\\\.\\COM9");  <-- correct
 * strcpy(pPortName, "\\\\.\\COM10"); <-- correct
 * strcpy(pPortName, "COM10");        <-- wrong
 * \endcode
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phbalReg_SetPort(
        void * pDataParams,                                                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pPortName                                                                         /**< [In] Port Name as String. */
    );

/**
* \brief Open communication port.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_USE_CONDITION Communication port is already open.
* \retval #PH_ERR_INTERFACE_ERROR Error while opening port.
*/
phStatus_t phbalReg_OpenPort(
        void * pDataParams                                                                          /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief Close communication port.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_USE_CONDITION Communication port is not open.
 * \retval #PH_ERR_INTERFACE_ERROR Error while closing port.
 */
phStatus_t phbalReg_ClosePort(
        void * pDataParams                                                                          /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief Perform Data Exchange on the bus.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_PARAMETER \b wOption is invalid.
 * \retval #PH_ERR_IO_TIMEOUT No response received within given time frame.
 * \retval #PH_ERR_BUFFER_OVERFLOW Response is too big for either given receive buffer or internal buffer.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 */
phStatus_t phbalReg_Exchange(
        void * pDataParams,                                                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                                           /**< [In] Option parameter. */
        uint8_t * pTxBuffer,                                                                        /**< [In] Data to transmit. */
        uint16_t wTxLength,                                                                         /**< [In] Number of bytes to transmit. */
        uint16_t wRxBufSize,                                                                        /**< [In] Size of receive buffer / Number of bytes to receive (depending on implementation). */
        uint8_t * pRxBuffer,                                                                        /**< [Out] Received data. */
        uint16_t * pRxLength                                                                        /**< [Out] Number of received data bytes. */
    );

/**
 * \brief Set configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
 * \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 */
phStatus_t phbalReg_SetConfig(
        void * pDataParams,                                                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                                                                           /**< [In] Configuration Identifier. */
        uint16_t wValue                                                                             /**< [In] Configuration Value. */
    );
/**
 * \brief Get configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 */
phStatus_t phbalReg_GetConfig(
        void * pDataParams,                                                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                                                                           /**< [In] Configuration Identifier. */
        uint16_t * pValue                                                                           /**< [Out] Configuration Value. */
    );

/**
 * end of group phbalReg
 * @}
 */
#endif /* NXPBUILD__PHBAL_REG */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHBALREG_H */
