/*
 * Copyright 2013, 2016 - 2020, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic ISO14443-4A Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHPALI14443P4A_H
#define PHPALI14443P4A_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

#ifdef NXPBUILD__PHPAL_I14443P4A_SW

/** \defgroup phpalI14443p4a_Sw Component : Software
* @{
*/

#define PHPAL_I14443P4A_SW_ID   0x01    /**< ID for Software ISO14443-4A layer     */

/**
* \brief Software PAL-ISO14443P4A parameter structure
*/
typedef struct
{
    uint16_t wId;           /**< Layer ID for this component, NEVER MODIFY! */
    void  * pHalDataParams; /**< Pointer to the parameter structure of the underlying layer. */
    uint8_t bCidSupported;  /**< Cid Support indicator; Unequal '0' if supported. */
    uint8_t bNadSupported;  /**< Nad Support indicator; Unequal '0' if supported. */
    uint8_t bCid;           /**< Card Identifier; Ignored if bCidSupported is equal '0'. */
    uint8_t bBitRateCaps;   /**< TA(1) byte of ATS; ISO/IEC 14443-4:2008, Section 5.2.4. */
    uint8_t bFwi;           /**< Frame Waiting Integer. */
    uint8_t bFsci;          /**< PICC Frame Size Integer; 0-0xC; */
    uint8_t bFsdi;          /**< (Current) PCD Frame Size Integer; 0-0xC; */
    uint8_t bDri;           /**< (Current) Divisor Receive (PCD to PICC) Integer; 0-3; */
    uint8_t bDsi;           /**< (Current) Divisor Send (PICC to PCD) Integer; 0-3; */
} phpalI14443p4a_Sw_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4a_Sw_Init(
                                  phpalI14443p4a_Sw_DataParams_t * pDataParams, /**< [In] Pointer to this layer's parameter structure. */
                                  uint16_t wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure. */
                                  void * pHalDataParams                         /**< [In] Pointer to the parameter structure of the underlying layer.*/
                                  );

/** @} */
#endif /* NXPBUILD__PHPAL_I14443P4A_SW */

#ifdef NXPBUILD__PHPAL_I14443P4A_SAMAV2_X

/** \defgroup phpalI14443p4a_SamAV2_X Component : SamAV2_X
* @{
*/

#define PHPAL_I14443P4A_SAMAV2_X_ID   0x02    /**< ID for SamAV2 ISO14443-3A layer       */

/**
* \brief ISO14443-4a parameter structure
*/
typedef struct
{
    uint16_t wId;           /**< Layer ID for this component, NEVER MODIFY!             */
    phhalHw_SamAV2_DataParams_t * pHalDataParams; /**< pointer to the parameter structure of the underlying layer */
    uint8_t bCidSupported;  /**< Cid Support indicator; Unequal '0' if supported.           */
    uint8_t bNadSupported;  /**< Nad Support indicator; Unequal '0' if supported.           */
    uint8_t bCid;           /**< Card Identifier; Ignored if bCidSupported is equal '0'.    */
    uint8_t bBitRateCaps;   /**< TA(1) byte of ATS; ISO/IEC 14443-4:2008, Section 5.2.4     */
    uint8_t bFwi;           /**< Frame Waiting Integer.                                     */
    uint8_t bFsci;          /**< PICC Frame Size Integer; 0-8                               */
    uint8_t bDri;           /**< (Current) Divisor Receive (PCD to PICC) Integer; 0-3       */
    uint8_t bDsi;           /**< (Current) Divisor Send (PICC to PCD) Integer; 0-3          */
} phpalI14443p4a_SamAV2_X_DataParams_t;


/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4a_SamAV2_X_Init(
                                        phpalI14443p4a_SamAV2_X_DataParams_t * pDataParams,     /**< [In] Pointer to this layer's parameter structure */
                                        uint16_t wSizeOfDataParams,                             /**< [In] Specifies the size of the data parameter structure. */
                                        phhalHw_SamAV2_DataParams_t * pHalDataParams            /**< [In] Pointer to the parameter structure of the underlying layer */
                                        );

/** @} */
#endif /* NXPBUILD__PHPAL_I14443P4A_SAMAV2_X */

#ifdef NXPBUILD__PHPAL_I14443P4A_SAMAV3_X

/** \defgroup phpalI14443p4a_SamAV3_X Component : SamAV3_X
* @{
*/

#define PHPAL_I14443P4A_SAMAV3_X_ID   0x04    /**< ID for SamAV3 ISO14443-3A layer       */

/**
* \brief ISO14443-4a parameter structure
*/
typedef struct
{
    uint16_t wId;           /**< Layer ID for this component, NEVER MODIFY!             */
    phhalHw_SamAV3_DataParams_t * pHalDataParams; /**< pointer to the parameter structure of the underlying layer */
    uint8_t bCidSupported;  /**< Cid Support indicator; Unequal '0' if supported.           */
    uint8_t bNadSupported;  /**< Nad Support indicator; Unequal '0' if supported.           */
    uint8_t bCid;           /**< Card Identifier; Ignored if bCidSupported is equal '0'.    */
    uint8_t bBitRateCaps;   /**< TA(1) byte of ATS; ISO/IEC 14443-4:2008, Section 5.2.4     */
    uint8_t bFwi;           /**< Frame Waiting Integer.                                     */
    uint8_t bFsci;          /**< PICC Frame Size Integer; 0-8                               */
    uint8_t bDri;           /**< (Current) Divisor Receive (PCD to PICC) Integer; 0-3       */
    uint8_t bDsi;           /**< (Current) Divisor Send (PICC to PCD) Integer; 0-3          */
} phpalI14443p4a_SamAV3_X_DataParams_t;


/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4a_SamAV3_X_Init(
                                        phpalI14443p4a_SamAV3_X_DataParams_t * pDataParams,     /**< [In] Pointer to this layer's parameter structure */
                                        uint16_t wSizeOfDataParams,                             /**< [In] Specifies the size of the data parameter structure. */
                                        phhalHw_SamAV3_DataParams_t * pHalDataParams            /**< [In] Pointer to the parameter structure of the underlying layer */
                                        );

/** @} */
#endif /* NXPBUILD__PHPAL_I14443P4A_SAMAV3_X */


#ifdef NXPBUILD__PHPAL_I14443P4A_RD710

/** \defgroup phpalI14443p4a_Rd710 Component : Rd710
* @{
*/

#define PHPAL_I14443P4A_RD710_ID   0x03    /**< ID for RD710 ISO14443-4A layer     */

/**
* \brief ISO14443-4a parameter structure
*/
typedef struct
{
    uint16_t wId;           /**< Layer ID for this component, NEVER MODIFY!             */
    phhalHw_Rd710_DataParams_t * pHalDataParams; /**< pointer to the parameter structure of the underlying layer */
} phpalI14443p4a_Rd710_DataParams_t;


/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4a_Rd710_Init(
                                        phpalI14443p4a_Rd710_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure */
                                        uint16_t wSizeOfDataParams,                         /**< [In] Specifies the size of the data parameter structure. */
                                        phhalHw_Rd710_DataParams_t * pHalDataParams         /**< [In] Pointer to the parameter structure of the underlying layer */
                                        );

/** @} */
#endif /* NXPBUILD__PHPAL_I14443P4A_RD710 */

#ifdef NXPBUILD__PHPAL_I14443P4A_PCSC

/** \defgroup phpalI14443p4a_Pcsc Component : PCSC
* @{
*/

#define PHPAL_I14443P4A_PCSC_ID     0x05    /**< ID for PCSC ISO14443-4A layer     */

/**
* \brief PCSC ISO14443-4A parameter structure
*/
typedef struct
{
    uint16_t wId;                           /**< Layer ID for this component, NEVER MODIFY!. */
    void * pHalDataParams;                  /**< Pointer to the parameter structure of HAL layer. */
    phbalReg_PcscWin_DataParams_t * pBalDataParams; /**< pointer to the parameter structure of the underlying pcsc bal. Is used to access the pcsc parameter */
    phhalHw_Pcsc_DataParams_t * pHalPcscDataParams; /**< pointer to the parameter structure of the underlying pcsc hal. Is used to access the pcsc parameter */
    uint8_t bCidSupported;                  /**< Cid Support indicator; Unequal '0' if supported. */
    uint8_t bNadSupported;                  /**< Nad Support indicator; Unequal '0' if supported. */
    uint8_t bCid;                           /**< Card Identifier; Ignored if bCidSupported is equal '0'. */
    uint8_t bBitRateCaps;                   /**< TA(1) byte of ATS; ISO/IEC 14443-4:2008, Section 5.2.4. */
    uint8_t bFwi;                           /**< Frame Waiting Integer. */
    uint8_t bFsci;                          /**< PICC Frame Size Integer; 0-0xC; */
    uint8_t bFsdi;                          /**< (Current) PCD Frame Size Integer; 0-0xC; */
    uint8_t bDri;                           /**< (Current) Divisor Receive (PCD to PICC) Integer; 0-3; */
    uint8_t bDsi;                           /**< (Current) Divisor Send (PICC to PCD) Integer; 0-3; */
    uint8_t bActStatus;                     /**< Get the Activated Status; */
} phpalI14443p4a_Pcsc_DataParams_t;

/**
* \brief Initialise PCSC ISO14443-4A layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4a_Pcsc_Init(
                                    phpalI14443p4a_Pcsc_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                    uint16_t wSizeOfDataParams,                        /**< [In] Specifies the size of the data parameter structure. */
                                    void * pHalDataParams                              /**< [In] Pointer to the parameter structure of HAL layer. */
                                    );

/** @} */
#endif /* NXPBUILD__PHPAL_I14443P4A_PCSC */


#ifdef NXPBUILD__PHPAL_I14443P4A

/** \defgroup phpalI14443p4a ISO/IEC14443-4A
* \brief These Components implement the additional ISO/IEC 14443-4 activation steps
* needed for ISO14443 Type-A cards.
* @{
*/

/**
* \name DRI/DSI Values
*/
/** @{ */
#define PHPAL_I14443P4A_DATARATE_106    0x00U   /**< DRI/DSI value for 106 kBit/s   */
#define PHPAL_I14443P4A_DATARATE_212    0x01U   /**< DRI/DSI value for 212 kBit/s   */
#define PHPAL_I14443P4A_DATARATE_424    0x02U   /**< DRI/DSI value for 424 kBit/s   */
#define PHPAL_I14443P4A_DATARATE_848    0x03U   /**< DRI/DSI value for 848 kBit/s   */
/** @} */

/**
* \brief Perform a ISO14443-4A "Request Answer to Select" command.
*
\verbatim
[FSDI to FSD conversion]
----------------------------------------------------------------------------------------
FSDI         | 00 | 01 | 02 | 03 | 04 | 05 | 06 | 07  | 08  | 09  | 0A   | 0B   | 0C   |
----------------------------------------------------------------------------------------
FSD (bytes)  | 16 | 24 | 32 | 40 | 48 | 64 | 96 | 128 | 256 | 512 | 1024 | 2018 | 4096 |
----------------------------------------------------------------------------------------
\endverbatim
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_PROTOCOL_ERROR Invalid response received.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4a_Rats(
                               void * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                               uint8_t bFsdi,       /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
                               uint8_t bCid,        /**< [In] Card Identifier; 0-14 */
                               uint8_t * pAts       /**< [Out] Answer to Select; uint8_t[255] */
                               );

/**
* \brief Perform a ISO14443-4A "Protocol and Parameter Selection" command.
*
* The following values for \c bDri and \c bDsi are possible:\n
* \li #PHPAL_I14443P4A_DATARATE_106
* \li #PHPAL_I14443P4A_DATARATE_212
* \li #PHPAL_I14443P4A_DATARATE_424
* \li #PHPAL_I14443P4A_DATARATE_848
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_PROTOCOL_ERROR Invalid response received.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4a_Pps(
                              void * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
                              uint8_t bDri,         /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3 */
                              uint8_t bDsi          /**< [In] Divisor Send (PICC to PCD) Integer; 0-3 */
                              );

/**
* \brief Perform ISO14443-4A Rats and Pps commands.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_PROTOCOL_ERROR Invalid response received.
* \retval #PH_ERR_FRAMING_ERROR Bcc invalid.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phpalI14443p4a_ActivateCard(
                                       void * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                                       uint8_t bFsdi,       /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
                                       uint8_t bCid,        /**< [In] Card Identifier; 0-14 */
                                       uint8_t bDri,        /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3 */
                                       uint8_t bDsi,        /**< [In] Divisor Send (PICC to PCD) Integer; 0-3 */
                                       uint8_t * pAts       /**< [Out] Answer to Select; uint8_t[255] */
                                       );

/**
* \brief Retrieve the ISO14443-4A protocol parameters.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phpalI14443p4a_GetProtocolParams(
    void * pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
    uint8_t * pCidEnabled,      /**< [Out] Unequal '0' if Card Identifier is enabled. */
    uint8_t * pCid,             /**< [Out] Card Identifier. */
    uint8_t * pNadSupported,    /**< [Out] Node Address Support; Unequal '0' if supported. */
    uint8_t * pFwi,             /**< [Out] Frame Waiting Integer. */
    uint8_t * pFsdi,            /**< [Out] PCD Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames). */
    uint8_t * pFsci             /**< [Out] PICC Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames). */
    );

/** @} */
#endif /* NXPBUILD__PHPAL_I14443P4A */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHPALI14443P4A_H */
