/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software Cid Manager Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phCidManager.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PH_CIDMANAGER_SW

#include "phCidManager_Sw.h"

phStatus_t phCidManager_Sw_Init(
                             phCidManager_Sw_DataParams_t * pDataParams,
                             uint16_t wSizeOfDataParams
                             )
{
    if (sizeof(phCidManager_Sw_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_CIDMANAGER);
    }
	PH_ASSERT_NULL (pDataParams);

    /* Set identifier */
    pDataParams->wId = PH_COMP_CIDMANAGER | PH_CIDMANAGER_SW_ID;

    /* Initialise Cid list */
    memset(pDataParams->bCidList, 0x00, sizeof(pDataParams->bCidList));  /* PRQA S 3200 */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_CIDMANAGER);
}

phStatus_t phCidManager_Sw_GetFreeCid(
                                   phCidManager_Sw_DataParams_t * pDataParams,
                                   uint8_t * pCid
                                   )
{
    uint8_t PH_MEMLOC_COUNT bIndex;

    /* Search for first free Cid */
    for (bIndex = 0; bIndex < PH_CIDMANAGER_LAST_CID; ++bIndex)
    {
        if (pDataParams->bCidList[bIndex] == 0)
        {
            break;
        }
    }

    /* If a free slot has been found */
    if (bIndex < PH_CIDMANAGER_LAST_CID)
    {
        /* Set slot to occupied status */
        pDataParams->bCidList[bIndex] = 1;

        /* return the free Cid */
        *pCid = bIndex + 1;
    }
    /* return PH_ERR_USE_CONDITION if no free slot has been found */
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_CIDMANAGER);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_CIDMANAGER);
}

phStatus_t phCidManager_Sw_FreeCid(
                                phCidManager_Sw_DataParams_t * pDataParams,
                                uint8_t bCid
                                )
{
    /* Bail out if given Cid is out of range */
    if (bCid == 0 || bCid > PH_CIDMANAGER_LAST_CID)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_CIDMANAGER);
    }

    /* Bail out if given Cid is not occupied */
    if (pDataParams->bCidList[bCid-1] == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_CIDMANAGER);
    }

    /* Free Cid slot */
    pDataParams->bCidList[bCid-1] = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_CIDMANAGER);
}

#endif /* NXPBUILD__PH_CIDMANAGER_SW */
