/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) KeyStore component.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <phKeyStore.h>
#include <phhalHw.h>

#ifdef NXPBUILD__PH_KEYSTORE_SAM

#include "phKeyStore_Sam.h"
#include "phKeyStore_Sam_Int.h"

#include "../../comps/phhalHw/src/Sam/Commands/03_KeyManagement/phhalHw_Sam_Cmd_KM.h"

phStatus_t phKeyStore_Sam_Init(phKeyStore_Sam_DataParams_t * pDataParams, uint16_t wSizeOfDataParams,
    phhalHw_Sam_DataParams_t * pHalDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    if(sizeof(phKeyStore_Sam_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_KEYSTORE);
    }

    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_KEYSTORE);
    PH_ASSERT_NULL_PARAM(pHalDataParams, PH_COMP_KEYSTORE);

    /* Init private data */
    pDataParams->wId = PH_COMP_KEYSTORE | PH_KEYSTORE_SAM_ID;
    pDataParams->pHalDataParams = pHalDataParams;
    pDataParams->bIsLRPKey = PH_OFF;

    /* Set defaults */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_SetConfig(
        pDataParams,
        PH_KEYSTORE_CONFIG_SET_DEFAULT,
        PH_ON));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

phStatus_t phKeyStore_Sam_FormatKeyEntry(phKeyStore_Sam_DataParams_t * pDataParams, uint16_t wKeyNo,
    uint16_t wKeyType)
{
    phStatus_t                PH_MEMLOC_REM wStatus = 0;
    uint8_t                   PH_MEMLOC_REM bCurrKeyNoCEK = 0;
    uint8_t                   PH_MEMLOC_REM bProMas = 0;  /* For updating the P2 information byte of SAM ChangeKeyEntry command frame. */
    uint8_t                   PH_MEMLOC_REM bIsRamKey = PH_OFF;

    uint8_t                   PH_MEMLOC_REM aNullKey[PH_KEYSTORE_KEY_ENTRY_SIZE];

    phKeyStore_Sam_KeyEntry_t PH_MEMLOC_REM stKeyEntry;
    uint8_t                   PH_MEMLOC_REM bKeyEntryLen = 0;

    uint8_t                   PH_MEMLOC_REM aKeyEntryBuff[PH_KEYSTORE_KEY_ENTRY_SIZE_MAX];
    uint8_t                   PH_MEMLOC_REM bKeyEntryBuffLen = 0;

    /* Check if its RAM key. */
    bIsRamKey = PH_KEYSTORE_IS_RAM_KEY(wKeyNo);

    /* Reset the buffer. */
    memset(aNullKey, 0x00, sizeof(aNullKey));

    /* Get the current KeyEntry information from SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_GetKeyEntry(
        pDataParams,
        (uint8_t) wKeyNo,
        bIsRamKey,
        &stKeyEntry));

    /* Save Current Change Entry key value. */
    bCurrKeyNoCEK = stKeyEntry.bKeyNoCEK;

    /* Set the key buffers with NullKey buffer. */
    memcpy(stKeyEntry.aKeyData, aNullKey, sizeof(aNullKey));

    /* Update the retrieved Set and ExtSET values to internal KeyEntry structure. */
    stKeyEntry.aSet[0U] = pDataParams->aSet[0U];
    stKeyEntry.aSet[1U] = pDataParams->aSet[1U];
    stKeyEntry.aExtSet[0U] = pDataParams->aExtSet[0U];
    stKeyEntry.aExtSet[1U] = pDataParams->aExtSet[1U];

    /* RESET old key entry setting */
    stKeyEntry.aSet[0] &= (uint8_t) ~(uint8_t) PH_KEYSTORE_SAM_KEYTYPE_MASK;

    /* Define new Key type and Key B, Key C validity. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_SetKeyType(
        &stKeyEntry,
        wKeyType,
        pDataParams->b2K3DESOption,
        pDataParams->bIsLRPKey));

    /* Reset Key version. */
    stKeyEntry.bVersionKeyA = 0x00;
    stKeyEntry.bVersionKeyB = 0x00;
    stKeyEntry.bVersionKeyC = 0x00;

    /* Copy the DFAid and DF_KeyNo to internal KeyEntry structure. */
    memcpy(stKeyEntry.aDFAid, pDataParams->aDFAid, sizeof(pDataParams->aDFAid));
    stKeyEntry.bDFKeyNo = pDataParams->bDFKeyNo;

    /* Update the internal KeyEntry structure with reference KUC number, CEK, AEK key number and version. */
    stKeyEntry.bKeyNoCEK = pDataParams->bKeyNoCEK;
    stKeyEntry.bKeyVCEK = pDataParams->bKeyVCEK;
    stKeyEntry.bRefNoKUC = pDataParams->bRefNoKUC;
    stKeyEntry.bKeyNoAEK = pDataParams->bKeyNoAEK;
    stKeyEntry.bKeyVAEK = pDataParams->bKeyVAEK;

    /* Update the Program mask which will be used for P2 information of ChangeKeyEntry command. */
    if(bIsRamKey == (uint8_t) PH_ON)
    {
        /* Update ProMask for Key A only. */
        bProMas = PH_KEYSTORE_SAM_PRO_BIT_MASK_RAM_INCLUDE_VER_A;
    }
    else
    {
        bProMas = PH_KEYSTORE_SAM_PRO_BIT_MASK_NVM_INCLUDE_VER_A;

        /* Update the Program Mask to include Key B information. */
        if(stKeyEntry.bVersionKeyBValid == PH_ON)
            bProMas |= PH_KEYSTORE_SAM_PRO_BIT_MASK_NVM_INCLUDE_VER_AB;

        /* Update the Program Mask to include Key C information. */
        if(stKeyEntry.bVersionKeyCValid == PH_ON)
            bProMas |= PH_KEYSTORE_SAM_PRO_BIT_MASK_NVM_INCLUDE_VER_ABC;
    }

    /* Update the key length. */
    bKeyEntryLen = (uint8_t) ((stKeyEntry.bVersionKeyCValid || stKeyEntry.bVersionKeyBValid) ? 48U : 32U);

    /* Convert the KeyEntry information to bytes. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_ConvertKeyEntryToBuffer(
        &stKeyEntry,
        stKeyEntry.aKeyData,
        bKeyEntryLen,
        aKeyEntryBuff,
        &bKeyEntryBuffLen));

    /* Update the current key information to newly configured key information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_SAM_ChangeKeyEntry(
        pDataParams->pHalDataParams,
        (uint8_t) wKeyNo,
        bProMas,
        aKeyEntryBuff,
        bKeyEntryBuffLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

phStatus_t phKeyStore_Sam_SetKUC(phKeyStore_Sam_DataParams_t * pDataParams, uint16_t wKeyNo,
    uint16_t wRefNoKUC)
{
    phStatus_t                PH_MEMLOC_REM wStatus = 0;
    uint8_t                   PH_MEMLOC_REM bCurKeyNoCEK = 0;
    uint8_t                   PH_MEMLOC_REM bProMas = 0;
    uint8_t                   PH_MEMLOC_REM bKeyLen = 0;
    phKeyStore_Sam_KeyEntry_t PH_MEMLOC_REM stKeyEntry;

    uint8_t                   PH_MEMLOC_REM aKeyEntryBuff[PH_KEYSTORE_KEY_ENTRY_SIZE_MAX];
    uint8_t                   PH_MEMLOC_REM bKeyEntryBuffLen = 0;

    /* Get the current KeyEntry */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_GetKeyEntry(
        pDataParams,
        (uint8_t) wKeyNo,
        PH_OFF,
        &stKeyEntry));

    /* Save the current KeyNoCEK. */
    bCurKeyNoCEK = stKeyEntry.bKeyNoCEK;

    /* Update the KUC number. */
    stKeyEntry.bRefNoKUC = (uint8_t) wRefNoKUC;

    /* Update the programming mask information. */
    bProMas = 0x05U;

    /* Convert the KeyEntry information to bytes. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_ConvertKeyEntryToBuffer(
        &stKeyEntry,
        stKeyEntry.aKeyData,
        bKeyLen,
        aKeyEntryBuff,
        &bKeyEntryBuffLen));

    /* Update the current reference KUC information to newly configured KUC information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_SAM_ChangeKeyEntry(
        pDataParams->pHalDataParams,
        (uint8_t) wKeyNo,
        bProMas,
        aKeyEntryBuff,
        bKeyEntryBuffLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

phStatus_t phKeyStore_Sam_GetKUC(phKeyStore_Sam_DataParams_t * pDataParams, uint16_t wRefNoKUC,
    uint32_t * pdwLimit, uint32_t * pdwCurVal)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t *   PH_MEMLOC_REM pKucData = NULL;
    uint16_t    PH_MEMLOC_REM wKucDataLen = 0;

    /* Get the KUCEntry information from Sam. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_SAM_GetKUCEntry(
        pDataParams->pHalDataParams,
        (uint8_t) wRefNoKUC,
        &pKucData,
        &wKucDataLen));

    /* Update the Limit parameter. */
    *pdwLimit = (uint32_t) ((uint32_t) (pKucData[3U]) << 24U);
    *pdwLimit |= (uint32_t) ((uint32_t) (pKucData[2U]) << 16U);
    *pdwLimit |= (uint32_t) ((uint32_t) (pKucData[1U]) << 8U);
    *pdwLimit |= (uint32_t) ((uint32_t) (pKucData[0U]) << 0);

    /* Update the KeyNo and Version to DataParams. */
    pDataParams->bKeyNoCKUC = pKucData[4U];
    pDataParams->bKeyVCKUC = pKucData[5U];

    /* Update the Current value of Key Usage Counter parameter. */
    *pdwCurVal = (uint32_t) ((uint32_t) (pKucData[9U]) << 24U);
    *pdwCurVal |= (uint32_t) ((uint32_t) (pKucData[8U]) << 16U);
    *pdwCurVal |= (uint32_t) ((uint32_t) (pKucData[7U]) << 8U);
    *pdwCurVal |= (uint32_t) ((uint32_t) (pKucData[6U]) << 0);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

phStatus_t phKeyStore_Sam_ChangeKUC(phKeyStore_Sam_DataParams_t * pDataParams, uint16_t wRefNoKUC,
    uint32_t dwLimit)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bProMas = 0x00;
    uint8_t    PH_MEMLOC_REM aKucData[0x06U /* KUC Limit (4 byte), KeyNoCKUC (1 byte), KeyVCKUC (1 byte) */];

    /* Reset the KucData buffer. */
    memset(aKucData, 0x00, sizeof(aKucData));

    bProMas |= PH_KEYSTORE_SAM_UPDATE_LIMIT_MASK;
    aKucData[0U] = (uint8_t) (dwLimit >> 0);
    aKucData[1U] = (uint8_t) (dwLimit >> 8U);
    aKucData[2U] = (uint8_t) (dwLimit >> 16U);
    aKucData[3U] = (uint8_t) (dwLimit >> 24U);

    if(pDataParams->bKeyNoCKUC != (PH_KEYSTORE_INVALID_ID & 0xFFU))
    {
        bProMas |= PH_KEYSTORE_SAM_UPDATE_KEYNO_CKUC_MASK;
        aKucData[4U] = pDataParams->bKeyNoCKUC;
    }

    if(pDataParams->bKeyVCKUC != (PH_KEYSTORE_INVALID_ID & 0xFFU))
    {
        bProMas |= PH_KEYSTORE_SAM_UPDATE_KEY_VCKUC_MASK;
        aKucData[5U] = pDataParams->bKeyVCKUC;
    }

    /* Update the KUCEntry information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_SAM_ChangeKUCEntry(
        pDataParams->pHalDataParams,
        (uint8_t) wRefNoKUC,
        bProMas,
        aKucData,
        (uint16_t) sizeof(aKucData)));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

phStatus_t phKeyStore_Sam_SetConfig(phKeyStore_Sam_DataParams_t * pDataParams, uint16_t wConfig,
    uint16_t wValue)
{
    switch(wConfig)
    {
        case PH_KEYSTORE_SAM_CONFIG_ALLOW_DUMP_SESSION_KEY:
            if(wValue)
                pDataParams->aSet[0] |= PH_KEYSTORE_SAM_SET0_ALLOW_DUMP_SESSION_KEY;
            else
                pDataParams->aSet[0] = (uint8_t) (pDataParams->aSet[0] & (~PH_KEYSTORE_SAM_SET0_ALLOW_DUMP_SESSION_KEY));
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEEP_IV:
            if(wValue)
                pDataParams->aSet[0] |= PH_KEYSTORE_SAM_SET0_KEEP_IV;
            else
                pDataParams->aSet[0] = (uint8_t) (pDataParams->aSet[0] & (~PH_KEYSTORE_SAM_SET0_KEEP_IV));
            break;

        case PH_KEYSTORE_SAM_CONFIG_PL_KEY:
            if(wValue)
                pDataParams->aSet[0] |= PH_KEYSTORE_SAM_SET0_PL_KEY;
            else
                pDataParams->aSet[0] = (uint8_t) (pDataParams->aSet[0] & (~PH_KEYSTORE_SAM_SET0_PL_KEY));
            break;

        case PH_KEYSTORE_SAM_CONFIG_AUTH_KEY:
            if(wValue)
                pDataParams->aSet[1U] |= PH_KEYSTORE_SAM_SET1_AUTH_KEY;
            else
                pDataParams->aSet[1U] = (uint8_t) (pDataParams->aSet[1U] & (~PH_KEYSTORE_SAM_SET1_AUTH_KEY));
            break;

        case PH_KEYSTORE_SAM_CONFIG_DISABLE_KEY_ENTRY:
            if(wValue)
                pDataParams->aSet[1U] |= PH_KEYSTORE_SAM_SET1_DISABLE_KEY_ENTRY;
            else
                pDataParams->aSet[1U] = (uint8_t) (pDataParams->aSet[1U] & (~PH_KEYSTORE_SAM_SET1_DISABLE_KEY_ENTRY));
            break;

        case PH_KEYSTORE_SAM_CONFIG_LOCK_KEY:
            if(wValue)
                pDataParams->aSet[1U] |= PH_KEYSTORE_SAM_SET1_LOCK_KEY;
            else
                pDataParams->aSet[1U] = (uint8_t) (pDataParams->aSet[1U] & (~PH_KEYSTORE_SAM_SET1_LOCK_KEY));
            break;

        case PH_KEYSTORE_SAM_CONFIG_DISABLE_CHANGE_KEY_PICC:
            if(wValue)
                pDataParams->aSet[1U] |= PH_KEYSTORE_SAM_SET1_DISABLE_CHANGE_KEY_PICC;
            else
                pDataParams->aSet[1U] = (uint8_t) (pDataParams->aSet[1U] & (~PH_KEYSTORE_SAM_SET1_DISABLE_CHANGE_KEY_PICC));
            break;

        case PH_KEYSTORE_SAM_CONFIG_DISABLE_DECRYPTION:
            if(wValue)
                pDataParams->aSet[1U] |= PH_KEYSTORE_SAM_SET1_DISABLE_DECRYPTION;
            else
                pDataParams->aSet[1U] = (uint8_t) (pDataParams->aSet[1U] & (~PH_KEYSTORE_SAM_SET1_DISABLE_DECRYPTION));
            break;

        case PH_KEYSTORE_SAM_CONFIG_DISABLE_ENCRYPTION:
            if(wValue)
                pDataParams->aSet[1U] |= PH_KEYSTORE_SAM_SET1_DISABLE_ENCRYPTION;
            else
                pDataParams->aSet[1U] = (uint8_t) (pDataParams->aSet[1U] & (~PH_KEYSTORE_SAM_SET1_DISABLE_ENCRYPTION));
            break;

        case PH_KEYSTORE_SAM_CONFIG_DISABLE_VERIFY_MAC:
            if(wValue)
                pDataParams->aSet[1U] |= PH_KEYSTORE_SAM_SET1_DISABLE_VERIFY_MAC;
            else
                pDataParams->aSet[1U] = (uint8_t) (pDataParams->aSet[1U] & (~PH_KEYSTORE_SAM_SET1_DISABLE_VERIFY_MAC));
            break;

        case PH_KEYSTORE_SAM_CONFIG_DISABLE_GENERATE_MAC:
            if(wValue)
                pDataParams->aSet[1U] |= PH_KEYSTORE_SAM_SET1_DISABLE_GENERATE_MAC;
            else
                pDataParams->aSet[1U] = (uint8_t) (pDataParams->aSet[1U] & (~PH_KEYSTORE_SAM_SET1_DISABLE_GENERATE_MAC));
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEYCLASS:
            pDataParams->aExtSet[0] = (uint8_t) (pDataParams->aExtSet[0] & (~PH_KEYSTORE_SAM_KEYCLASS_MASK));
            pDataParams->aExtSet[0] |= (uint8_t) wValue;
            break;

        case PH_KEYSTORE_SAM_CONFIG_ALLOW_DUMP_SECRET_KEY:
            if(wValue)
                pDataParams->aExtSet[0] |= PH_KEYSTORE_SAM_EXTSET0_ALLOW_DUMP_SECRET_KEY;
            else
                pDataParams->aExtSet[0] = (uint8_t) (pDataParams->aExtSet[0] & (~PH_KEYSTORE_SAM_EXTSET0_ALLOW_DUMP_SECRET_KEY));
            break;

        case PH_KEYSTORE_SAM_CONFIG_MANDATE_KEY_DIVERSIFICATION:
            if(wValue)
                pDataParams->aExtSet[0] |= PH_KEYSTORE_SAM_EXTSET0_MANDATE_KEY_DIVERSIFICATION;
            else
                pDataParams->aExtSet[0] = (uint8_t) (pDataParams->aExtSet[0] & (~PH_KEYSTORE_SAM_EXTSET0_MANDATE_KEY_DIVERSIFICATION));
            break;

        case PH_KEYSTORE_SAM_CONFIG_RESERVED_SAM_PRESONALIZATION:
            if(wValue)
                pDataParams->aExtSet[0] |= PH_KEYSTORE_SAM_EXTSET0_PERSONALIZATION_SAM;
            else
                pDataParams->aExtSet[0] = (uint8_t) (pDataParams->aExtSet[0] & (~PH_KEYSTORE_SAM_EXTSET0_PERSONALIZATION_SAM));
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEY_USAGE_INT_HOST:
            if(wValue)
                pDataParams->aExtSet[1U] |= PH_KEYSTORE_SAM_EXTSET1_KEY_USAGE_INT_HOST;
            else
                pDataParams->aExtSet[1U] = (uint8_t) (pDataParams->aExtSet[1U] & (~PH_KEYSTORE_SAM_EXTSET1_KEY_USAGE_INT_HOST));
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEY_CHANGE_INT_HOST:
            if(wValue)
                pDataParams->aExtSet[1U] |= PH_KEYSTORE_SAM_EXTSET1_KEY_CHANGE_INT_HOST;
            else
                pDataParams->aExtSet[1U] = (uint8_t) (pDataParams->aExtSet[1U] & (~PH_KEYSTORE_SAM_EXTSET1_KEY_CHANGE_INT_HOST));
            break;

        case PH_KEYSTORE_SAM_CONFIG_SESSION_KEY_USAGE_INT_HOST:
            if(wValue)
                pDataParams->aExtSet[1U] |= PH_KEYSTORE_SAM_EXTSET1_SESSION_KEY_USAGE_INT_HOST;
            else
                pDataParams->aExtSet[1U] = (uint8_t) (pDataParams->aExtSet[1] & (~PH_KEYSTORE_SAM_EXTSET1_SESSION_KEY_USAGE_INT_HOST));
            break;

        case PH_KEYSTORE_SAM_CONFIG_ALLOW_DUMP_SECRET_KEY_INT_HOST:
            if(wValue)
                pDataParams->aExtSet[1U] |= PH_KEYSTORE_SAM_EXTSET1_DUMP_SECRET_KEY_INT_HOST;
            else
                pDataParams->aExtSet[1U] = (uint8_t) (pDataParams->aExtSet[1U] & (~PH_KEYSTORE_SAM_EXTSET1_DUMP_SECRET_KEY_INT_HOST));
            break;

        case PH_KEYSTORE_SAM_CONFIG_ALLOW_DUMP_SESSION_KEY_INT_HOST:
            if(wValue)
                pDataParams->aExtSet[1U] |= PH_KEYSTORE_SAM_EXTSET1_DUMP_SESSION_KEY_INT_HOST;
            else
                pDataParams->aExtSet[1U] = (uint8_t) (pDataParams->aExtSet[1U] & (~PH_KEYSTORE_SAM_EXTSET1_DUMP_SESSION_KEY_INT_HOST));
            break;

        case PH_KEYSTORE_SAM_CONFIG_DF_KEY_NO:
            pDataParams->bDFKeyNo = (uint8_t) wValue;
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEYNO_CEK:
            pDataParams->bKeyNoCEK = (uint8_t) wValue;
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEYV_CEK:
            pDataParams->bKeyVCEK = (uint8_t) wValue;
            break;

        case PH_KEYSTORE_SAM_CONFIG_REF_NO_KUC:
            pDataParams->bRefNoKUC = (uint8_t) wValue;
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEYNO_AEK:
            pDataParams->bKeyNoAEK = (uint8_t) wValue;
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEYV_AEK:
            pDataParams->bKeyVAEK = (uint8_t) wValue;
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEYNO_CKUC:
            pDataParams->bKeyNoCKUC = (uint8_t) wValue;
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEYV_CKUC:
            pDataParams->bKeyVCKUC = (uint8_t) wValue;
            break;

        case PH_KEYSTORE_SAM_CONFIG_DES_KEY_OPTION:
            pDataParams->b2K3DESOption = (uint8_t) wValue;
            break;

        case PH_KEYSTORE_SAM_CONFIG_ENABLE_LRP:
            pDataParams->bIsLRPKey = (uint8_t) wValue;
            break;

        case PH_KEYSTORE_CONFIG_SET_DEFAULT:
            pDataParams->aSet[0] = 0x00;
            pDataParams->aSet[1U] = 0x00;
            pDataParams->aExtSet[0] = 0x00;
            pDataParams->aExtSet[1U] = 0x00;
            memset(pDataParams->aDFAid, 0, 3U);
            pDataParams->bDFKeyNo = 0x00;
            pDataParams->b2K3DESOption = 0x00;
            pDataParams->bKeyNoCEK = 0x00;
            pDataParams->bKeyVCEK = 0x00;
            pDataParams->bRefNoKUC = 0xFFU;
            pDataParams->bKeyNoCKUC = 0x00;
            pDataParams->bKeyVCKUC = 0x00;
            pDataParams->bKeyNoAEK = 0x00;
            pDataParams->bKeyVAEK = 0x00;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_KEYSTORE);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

phStatus_t phKeyStore_Sam_GetConfig(phKeyStore_Sam_DataParams_t * pDataParams, uint16_t wConfig,
    uint16_t * pValue)
{
    switch(wConfig)
    {
        case PH_KEYSTORE_SAM_CONFIG_ALLOW_DUMP_SESSION_KEY:
            *pValue = (uint16_t) ((pDataParams->aSet[0] & PH_KEYSTORE_SAM_SET0_ALLOW_DUMP_SESSION_KEY) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEEP_IV:
            *pValue = (uint16_t) ((pDataParams->aSet[0] & PH_KEYSTORE_SAM_SET0_KEEP_IV) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_PL_KEY:
            *pValue = (uint16_t) ((pDataParams->aSet[0] & PH_KEYSTORE_SAM_SET0_PL_KEY) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_AUTH_KEY:
            *pValue = (uint16_t) ((pDataParams->aSet[1U] & PH_KEYSTORE_SAM_SET1_AUTH_KEY) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_DISABLE_KEY_ENTRY:
            *pValue = (uint16_t) ((pDataParams->aSet[1U] & PH_KEYSTORE_SAM_SET1_DISABLE_KEY_ENTRY) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_LOCK_KEY:
            *pValue = (uint16_t) ((pDataParams->aSet[1U] & PH_KEYSTORE_SAM_SET1_LOCK_KEY) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_DISABLE_CHANGE_KEY_PICC:
            *pValue = (uint16_t) ((pDataParams->aSet[1U] & PH_KEYSTORE_SAM_SET1_DISABLE_CHANGE_KEY_PICC) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_DISABLE_DECRYPTION:
            *pValue = (uint16_t) ((pDataParams->aSet[1U] & PH_KEYSTORE_SAM_SET1_DISABLE_DECRYPTION) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_DISABLE_ENCRYPTION:
            *pValue = (uint16_t) ((pDataParams->aSet[1U] & PH_KEYSTORE_SAM_SET1_DISABLE_ENCRYPTION) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_DISABLE_VERIFY_MAC:
            *pValue = (uint16_t) ((pDataParams->aSet[1U] & PH_KEYSTORE_SAM_SET1_DISABLE_VERIFY_MAC) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_DISABLE_GENERATE_MAC:
            *pValue = (uint16_t) ((pDataParams->aSet[1U] & PH_KEYSTORE_SAM_SET1_DISABLE_GENERATE_MAC) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEYCLASS:
            *pValue = (uint16_t) (pDataParams->aExtSet[0] & PH_KEYSTORE_SAM_KEYCLASS_MASK);
            break;

        case PH_KEYSTORE_SAM_CONFIG_ALLOW_DUMP_SECRET_KEY:
            *pValue = (uint16_t) ((pDataParams->aExtSet[0] & PH_KEYSTORE_SAM_EXTSET0_ALLOW_DUMP_SECRET_KEY) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_MANDATE_KEY_DIVERSIFICATION:
            *pValue = (uint16_t) ((pDataParams->aExtSet[0] & PH_KEYSTORE_SAM_EXTSET0_MANDATE_KEY_DIVERSIFICATION) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_RESERVED_SAM_PRESONALIZATION:
            *pValue = (uint16_t) ((pDataParams->aExtSet[0] & PH_KEYSTORE_SAM_EXTSET0_PERSONALIZATION_SAM) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEY_USAGE_INT_HOST:
            *pValue = (uint16_t) ((pDataParams->aExtSet[1] & PH_KEYSTORE_SAM_EXTSET1_KEY_USAGE_INT_HOST) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEY_CHANGE_INT_HOST:
            *pValue = (uint16_t) ((pDataParams->aExtSet[1U] & PH_KEYSTORE_SAM_EXTSET1_KEY_CHANGE_INT_HOST) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_SESSION_KEY_USAGE_INT_HOST:
            *pValue = (uint16_t) ((pDataParams->aExtSet[1U] & PH_KEYSTORE_SAM_EXTSET1_SESSION_KEY_USAGE_INT_HOST) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_ALLOW_DUMP_SECRET_KEY_INT_HOST:
            *pValue = (uint16_t) ((pDataParams->aExtSet[1U] & PH_KEYSTORE_SAM_EXTSET1_DUMP_SECRET_KEY_INT_HOST) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_ALLOW_DUMP_SESSION_KEY_INT_HOST:
            *pValue = (uint16_t) ((pDataParams->aExtSet[1U] & PH_KEYSTORE_SAM_EXTSET1_DUMP_SESSION_KEY_INT_HOST) ? 1U : 0);
            break;

        case PH_KEYSTORE_SAM_CONFIG_DF_KEY_NO:
            *pValue = (uint16_t) pDataParams->bDFKeyNo;
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEYNO_CEK:
            *pValue = (uint16_t) pDataParams->bKeyNoCEK;
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEYV_CEK:
            *pValue = (uint16_t) pDataParams->bKeyVCEK;
            break;

        case PH_KEYSTORE_SAM_CONFIG_REF_NO_KUC:
            *pValue = (uint16_t) pDataParams->bRefNoKUC;
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEYNO_AEK:
            *pValue = (uint16_t) pDataParams->bKeyNoAEK;
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEYV_AEK:
            *pValue = (uint16_t) pDataParams->bKeyVAEK;
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEYNO_CKUC:
            *pValue = (uint16_t) pDataParams->bKeyNoCKUC;
            break;

        case PH_KEYSTORE_SAM_CONFIG_KEYV_CKUC:
            *pValue = (uint16_t) pDataParams->bKeyVCKUC;
            break;

        case PH_KEYSTORE_SAM_CONFIG_DES_KEY_OPTION:
            *pValue = (uint16_t) pDataParams->b2K3DESOption;
            break;

        case PH_KEYSTORE_SAM_CONFIG_ENABLE_LRP:
            *pValue = pDataParams->bIsLRPKey;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_KEYSTORE);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

phStatus_t phKeyStore_Sam_SetConfigStr(phKeyStore_Sam_DataParams_t * pDataParams, uint16_t wConfig,
    uint8_t * pBuffer, uint16_t wBufferLen)
{
    switch(wConfig)
    {
        case PH_KEYSTORE_SAM_CONFIG_DF_AID:
            if(wBufferLen != 3U)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_KEYSTORE);
            }

            memcpy(pDataParams->aDFAid, pBuffer, 3U);
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_KEYSTORE);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

phStatus_t phKeyStore_Sam_GetConfigStr(phKeyStore_Sam_DataParams_t * pDataParams, uint16_t wConfig,
    uint8_t ** ppBuffer, uint16_t * pBufferLen)
{
    switch(wConfig)
    {
        case PH_KEYSTORE_SAM_CONFIG_DF_AID:
            *ppBuffer = pDataParams->aDFAid;
            *pBufferLen = 3U;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_KEYSTORE);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}





phStatus_t phKeyStore_Sam_SetKey(phKeyStore_Sam_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer,
    uint16_t wKeyType, uint8_t * pNewKey, uint16_t wNewKeyVer)
{
    phStatus_t                PH_MEMLOC_REM wStatus = 0;
    uint16_t                  PH_MEMLOC_REM wCurKeyType = 0;
    uint8_t                   PH_MEMLOC_REM bProMas = 0x01U;
    uint8_t                   PH_MEMLOC_REM bKeySize = 0;
    uint8_t                   PH_MEMLOC_REM bIsLRPKey = 0;
    uint8_t                   PH_MEMLOC_REM bIsRamKey = PH_OFF;
    phKeyStore_Sam_KeyEntry_t PH_MEMLOC_REM stKeyEntry;

    uint8_t                   PH_MEMLOC_REM aKeyEntryBuff[PH_KEYSTORE_KEY_ENTRY_SIZE_MAX];
    uint8_t                   PH_MEMLOC_REM bKeyEntryBuffLen = 0;

    uint8_t                   PH_MEMLOC_REM aKeyBuff[PH_KEYSTORE_KEY_ENTRY_SIZE];
    uint8_t                   PH_MEMLOC_REM bKeyBuffLen = 0;

    /* Check if its RAM key. */
    bIsRamKey = PH_KEYSTORE_IS_RAM_KEY(wKeyNo);

    /* Get the current KeyEntry */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_GetKeyEntry(
        pDataParams,
        (uint8_t) wKeyNo,
        bIsRamKey,
        &stKeyEntry));

    /* Get the KeyType of the KeyEntry. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_GetKeyType(
        &stKeyEntry,
        &wCurKeyType,
        &bIsLRPKey));

    /* The Key Type to be loaded must match with the current keytype format */
    if(wKeyType == PH_KEYSTORE_KEY_TYPE_DES)
    {
        if((wCurKeyType - 1U) != wKeyType)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_KEYSTORE);
        }
    }
    /* Reset of th eKeyTypes */
    else
    {
        if(wCurKeyType != wKeyType)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_KEYSTORE);
        }
    }

    /* Get the Key Size. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_GetKeySize(
        wKeyType,
        &bKeySize));

    /* Reset the internal Key buffer. */
    memset(aKeyBuff, 0, bKeySize);

    /* Update the internal buffer with the Key information. */
    memcpy(aKeyBuff, pNewKey, PH_KEYSTORE_KEY_ENTRY_SIZE);

    /* Update Key Version A */
    if(stKeyEntry.bVersionKeyA == (uint8_t) wKeyVer)
    {
        memcpy(&stKeyEntry.aKeyData[0], aKeyBuff, bKeySize);
        bKeyBuffLen = bKeySize;

        stKeyEntry.bVersionKeyA = (uint8_t) wNewKeyVer;
        bProMas |= 0x80U;
    }

    /* Update Key Version B */
    else if((stKeyEntry.bVersionKeyB == (uint8_t) wKeyVer) && (stKeyEntry.bVersionKeyBValid == PH_ON))
    {
        memcpy(&stKeyEntry.aKeyData[bKeySize], aKeyBuff, bKeySize);
        bKeyBuffLen = (uint8_t) (bKeySize * 2U);

        stKeyEntry.bVersionKeyB = (uint8_t) wNewKeyVer;
        bProMas |= 0x40U;
    }

    /* Update Key Version C */
    else if((stKeyEntry.bVersionKeyC == (uint8_t) wKeyVer) && (stKeyEntry.bVersionKeyCValid == PH_ON))
    {
        memcpy(&stKeyEntry.aKeyData[bKeySize * 2U], aKeyBuff, bKeySize);
        bKeyBuffLen = (uint8_t) (bKeySize * 3U);

        stKeyEntry.bVersionKeyC = (uint8_t) wNewKeyVer;
        bProMas |= 0x20U;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_KEYSTORE);
    }

    /* Update DFAid and DFKeyNo in KeyEntry. */
    if(bIsRamKey == (uint8_t) PH_OFF)
    {
        bProMas |= 0x10U;
        memcpy(stKeyEntry.aDFAid, pDataParams->aDFAid, sizeof(stKeyEntry.aDFAid));
        stKeyEntry.bDFKeyNo = pDataParams->bDFKeyNo;
    }

    /* Update KeyNo / Version CEK in KeyEntry. */
    bProMas |= 0x08U;
    stKeyEntry.bKeyNoCEK = pDataParams->bKeyNoCEK;
    stKeyEntry.bKeyVCEK = pDataParams->bKeyVCEK;

    /* Update KUC reference number in KeyEntry. */
    bProMas |= 0x04U;
    stKeyEntry.bRefNoKUC = pDataParams->bRefNoKUC;

    /* Update SET configuration in KeyEntry. */
    bProMas |= 0x02U;
    stKeyEntry.aSet[0U] &= PH_KEYSTORE_SAM_KEYTYPE_MASK;
    stKeyEntry.aSet[0U] |= pDataParams->aSet[0U];
    stKeyEntry.aSet[1U] = pDataParams->aSet[1U];
    stKeyEntry.aExtSet[0] = pDataParams->aExtSet[0U];
    stKeyEntry.aExtSet[1U] = pDataParams->aExtSet[1U];

    /* Convert the KeyEntry information to bytes. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_ConvertKeyEntryToBuffer(
        &stKeyEntry,
        stKeyEntry.aKeyData,
        bKeyBuffLen,
        aKeyEntryBuff,
        &bKeyEntryBuffLen));

    /* Update the current key information to newly configured key information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_SAM_ChangeKeyEntry(
        pDataParams->pHalDataParams,
        (uint8_t) wKeyNo,
        bProMas,
        aKeyEntryBuff,
        bKeyEntryBuffLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

phStatus_t phKeyStore_Sam_SetKeyAtPos(phKeyStore_Sam_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wPos,
    uint16_t wKeyType, uint8_t * pKey, uint16_t wKeyVer)
{
    phStatus_t                PH_MEMLOC_REM wStatus = 0;
    uint16_t                  PH_MEMLOC_REM wCurKeyType = 0;
    uint8_t                   PH_MEMLOC_REM bProMas = 0x01U;
    uint8_t                   PH_MEMLOC_REM bKeySize = 0;
    uint8_t                   PH_MEMLOC_REM bIsLRPKey = 0;
    uint8_t                   PH_MEMLOC_REM bIsRamKey = PH_OFF;
    phKeyStore_Sam_KeyEntry_t PH_MEMLOC_REM stKeyEntry;

    uint8_t                   PH_MEMLOC_REM aKeyBuff[PH_KEYSTORE_KEY_ENTRY_SIZE];
    uint8_t                   PH_MEMLOC_REM bKeyBuffLen = 0;

    uint8_t                   PH_MEMLOC_REM aKeyEntryBuff[PH_KEYSTORE_KEY_ENTRY_SIZE_MAX];
    uint8_t                   PH_MEMLOC_REM bKeyEntryBuffLen = 0;

    /* Check if its RAM key. */
    bIsRamKey = PH_KEYSTORE_IS_RAM_KEY(wKeyNo);

    /* Get the current KeyEntry */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_GetKeyEntry(
        pDataParams,
        (uint8_t) wKeyNo,
        bIsRamKey,
        &stKeyEntry));

    /* Get the KeyType of the KeyEntry. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_GetKeyType(
        &stKeyEntry,
        &wCurKeyType,
        &bIsLRPKey));

    /* The Key Type to be loaded must match with the current keytype format */
    if(wKeyType == PH_KEYSTORE_KEY_TYPE_DES)
    {
        if((wCurKeyType - 1U) != wKeyType)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_KEYSTORE);
        }
    }
    /* Reset of th eKeyTypes */
    else
    {
        if(wCurKeyType != wKeyType)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_KEYSTORE);
        }
    }

    /* Get the Key Size. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_GetKeySize(
        wKeyType,
        &bKeySize));

    /* Reset the internal Key buffer. */
    memset(aKeyBuff, 0, PH_KEYSTORE_KEY_ENTRY_SIZE);

    /* Update the internal buffer with the Key information. */
    memcpy(aKeyBuff, pKey, bKeySize);

    /* Check if keys should be updated */
    switch(wPos)
    {
        case PH_KEYSTORE_SAM_VERSION_POSITION_A:
            memcpy(&stKeyEntry.aKeyData[0], aKeyBuff, bKeySize);
            bKeyBuffLen = bKeySize;

            stKeyEntry.bVersionKeyA = (uint8_t) wKeyVer;
            bProMas |= 0x80U;
            break;

        case PH_KEYSTORE_SAM_VERSION_POSITION_B:
            memcpy(&stKeyEntry.aKeyData[bKeySize], aKeyBuff, bKeySize);
            bKeyBuffLen = (uint8_t) (bKeySize * 2U);

            stKeyEntry.bVersionKeyB = (uint8_t) wKeyVer;
            bProMas |= 0x40U;
            break;

        case PH_KEYSTORE_SAM_VERSION_POSITION_C:
            memcpy(&stKeyEntry.aKeyData[bKeySize * 2U], aKeyBuff, bKeySize);
            bKeyBuffLen = (uint8_t) (bKeySize * 3U);

            stKeyEntry.bVersionKeyC = (uint8_t) wKeyVer;
            bProMas |= 0x20U;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_KEYSTORE);
    }

    /* Update DFAid and DFKeyNo in KeyEntry. */
    if(bIsRamKey == (uint8_t) PH_OFF)
    {
        bProMas |= 0x10U;
        memcpy(stKeyEntry.aDFAid, pDataParams->aDFAid, 3U);
        stKeyEntry.bDFKeyNo = pDataParams->bDFKeyNo;
    }

    /* Update KeyNo / Version CEK in KeyEntry. */
    bProMas |= 0x08U;
    stKeyEntry.bKeyNoCEK = pDataParams->bKeyNoCEK;
    stKeyEntry.bKeyVCEK = pDataParams->bKeyVCEK;

    /* Update KUC reference number in KeyEntry. */
    bProMas |= 0x04U;
    stKeyEntry.bRefNoKUC = pDataParams->bRefNoKUC;

    /* Update SET configuration in KeyEntry. */
    bProMas |= 0x02U;
    stKeyEntry.aSet[0] &= PH_KEYSTORE_SAM_KEYTYPE_MASK;
    stKeyEntry.aSet[0] |= pDataParams->aSet[0];
    stKeyEntry.aSet[1U] = pDataParams->aSet[1U];
    stKeyEntry.aExtSet[0] = pDataParams->aExtSet[0];
    stKeyEntry.aExtSet[1U] = pDataParams->aExtSet[1U];

    /* Convert the KeyEntry information to bytes. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_ConvertKeyEntryToBuffer(
        &stKeyEntry,
        stKeyEntry.aKeyData,
        bKeyBuffLen,
        aKeyEntryBuff,
        &bKeyEntryBuffLen));

    /* Update the current key information to newly configured key information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_SAM_ChangeKeyEntry(
        pDataParams->pHalDataParams,
        (uint8_t) wKeyNo,
        bProMas,
        aKeyEntryBuff,
        bKeyEntryBuffLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

phStatus_t phKeyStore_Sam_GetKey(phKeyStore_Sam_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer,
    uint8_t bKeyBufSize, uint8_t * pKey, uint16_t * pKeyType)
{
    phStatus_t                PH_MEMLOC_REM wStatus = 0;
    uint8_t                   PH_MEMLOC_REM bIsLRPKey = 0;
    uint8_t                   PH_MEMLOC_REM bIsRamKey = PH_OFF;

    phKeyStore_Sam_KeyEntry_t PH_MEMLOC_REM stKeyEntry;
    uint16_t                  PH_MEMLOC_REM wKeyLen = 0;

    uint8_t *                 PH_MEMLOC_REM pKeyTmp = NULL;

    /* Check if its RAM key. */
    bIsRamKey = PH_KEYSTORE_IS_RAM_KEY(wKeyNo);

    /* Get the current KeyEntry. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_GetKeyEntry(
        pDataParams,
        (uint8_t) wKeyNo,
        bIsRamKey,
        &stKeyEntry));

    /* Get the KeyType of the KeyEntry. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_GetKeyType(
        &stKeyEntry,
        pKeyType,
        &bIsLRPKey));

    /* Check if Secret Key Dump is allowed. */
    if(!(stKeyEntry.aExtSet[0] & PH_KEYSTORE_SAM_EXTSET0_ALLOW_DUMP_SECRET_KEY))
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_KEYSTORE);
    }

    /* Get the Key Entry information from SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_SAM_DumpSecretKey(
        pDataParams->pHalDataParams,
        0,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        NULL,
        0,
        &pKeyTmp,
        &wKeyLen));

    /* Check if size is equal to the required length. */
    if(bKeyBufSize < (uint8_t) wKeyLen)
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_KEYSTORE);
    }

    /* Copy Key information to parameter. */
    if(pKeyTmp != NULL)
        memcpy(pKey, pKeyTmp, wKeyLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

phStatus_t phKeyStore_Sam_GetKeyEntry(phKeyStore_Sam_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t * pKeyVer,
    uint16_t * pKeyVerLen, uint16_t * pKeyType)
{
    phStatus_t                PH_MEMLOC_REM wStatus = 0;
    uint8_t                   PH_MEMLOC_REM bIsRamKey = PH_OFF;
    phKeyStore_Sam_KeyEntry_t PH_MEMLOC_REM stKeyEntry;

    /* Check if its RAM key. */
    bIsRamKey = PH_KEYSTORE_IS_RAM_KEY(wKeyNo);

    /* Update the parameter. */
    *pKeyVerLen = 0x00;

    /* First try to find the correct key position. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_GetKeyEntry(
        pDataParams,
        (uint8_t) wKeyNo,
        bIsRamKey,
        &stKeyEntry));

    /* Update the version information. */
    pKeyVer[0U] = stKeyEntry.bVersionKeyA;
    pKeyVer[1U] = stKeyEntry.bVersionKeyB;
    pKeyVer[2U] = stKeyEntry.bVersionKeyC;

    /* Update the version length. */
    *pKeyVerLen = (uint8_t) ((stKeyEntry.bVersionKeyCValid) ? 3U : 2U);

    /* Get the key type. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_Sam_Int_GetKeyType(
        &stKeyEntry,
        pKeyType,
        &pDataParams->bIsLRPKey));

    /* Set the DESFire additional option settings. */
    switch(((stKeyEntry.aSet[0] & PH_KEYSTORE_SAM_KEYTYPE_MASK) >> 3U))
    {
        case PH_KEYSTORE_SAM_KEYTYPE_2K3DES_MASK:
            pDataParams->b2K3DESOption = PH_KEYSTORE_SAM_DES_OPTION_ISO_CRC16;
            break;

        case PH_KEYSTORE_SAM_KEYTYPE_3DESDF4_MASK:
            pDataParams->b2K3DESOption = PH_KEYSTORE_SAM_DES_OPTION_DESFIRE4;
            break;

        case PH_KEYSTORE_SAM_KEYTYPE_2K3DESDF8_MASK:
            pDataParams->b2K3DESOption = PH_KEYSTORE_SAM_DES_OPTION_ISO_CRC32;
            break;
    }

    /* Update SET and ExtSET configuration. */
    pDataParams->aSet[0] = (uint8_t) (stKeyEntry.aSet[0] & ~PH_KEYSTORE_SAM_KEYTYPE_MASK);
    pDataParams->aSet[1U] = stKeyEntry.aSet[1U];
    pDataParams->aExtSet[0] = stKeyEntry.aExtSet[0];
    pDataParams->aExtSet[1U] = stKeyEntry.aExtSet[1U];

    /* Update KeyNo and Version of Change Entry Key. */
    pDataParams->bKeyNoCEK = stKeyEntry.bKeyNoCEK;
    pDataParams->bKeyVCEK = stKeyEntry.bKeyVCEK;

    /* Update DF_AID and KeyNo. */
    if(bIsRamKey == (uint8_t) PH_OFF)
    {
        memcpy(pDataParams->aDFAid, stKeyEntry.aDFAid, sizeof(stKeyEntry.aDFAid));
        pDataParams->bDFKeyNo = stKeyEntry.bDFKeyNo;
    }

    /* Update KUC Reference number. */
    pDataParams->bRefNoKUC = stKeyEntry.bRefNoKUC;

    /* Update KeyNo and Version of Access KeyEntry. */
    pDataParams->bKeyNoAEK = stKeyEntry.bKeyNoAEK;
    pDataParams->bKeyVAEK = stKeyEntry.bKeyVAEK;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

#endif /* NXPBUILD__PH_KEYSTORE_SAM */
