/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal logic file of SAM (AV4 and future SAM's) KeyStore component.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHKEYSTORE_SAM_INT_H
#define PHKEYSTORE_SAM_INT_H

#include <ph_Status.h>
#include <phKeyStore.h>

#define PH_KEYSTORE_SAM_NVM_KEY_MAX                                             0x7FU   /* Definition for Maximum Key entries in SAM NVM. */
#define PH_KEYSTORE_KEY_ENTRY_SIZE              PH_KEYSTORE_KEY_TYPE_AES128_SIZE * 3U   /* Definition for Maximum size to allocate 3 AES128 keys. */
#define PH_KEYSTORE_KEY_ENTRY_SIZE_MAX                                          0x40U   /* Definition for Maximum possible size of a key entry.*/

#define PH_KEYSTORE_SAM_KEYTYPE_MASK                                            0x78U   /* Definition for masking out the Key Type. */
#define PH_KEYSTORE_SAM_KEYCLASS_MASK                                           0x07U   /* Definition for masking out the Key Class. */

/* Key types for SAM (AV4 and future SAM's). Macros to update the SET configuration's first byte information. */
#define PH_KEYSTORE_SAM_KEYTYPE_3DESDF4_MASK                                    0x00U   /* Definition for Key Type as DESFire4 (TDEA DESFire4 (DES or 2TDEA key)) */
#define PH_KEYSTORE_SAM_KEYTYPE_2K3DES_MASK                                     0x01U   /* Definition for Key Type as ISO_TDEA16 (TDEA ISO 10116 (16-bits CRC,
                                                                                         * 4-bytes MAC) (DES or 2TDEA key))
                                                                                         */
#define PH_KEYSTORE_SAM_KETYPE_3K3DES_MASK                                      0x03U   /* Definition for Key Type as 3TDEA (3TDEA ISO 10116 (3TDEA key)) */
#define PH_KEYSTORE_SAM_KEYTYPE_AES128_MASK                                     0x04U   /* Definition for Key Type as AES_128 (AES 128-bits key (AES_128 key)) */
#define PH_KEYSTORE_SAM_KEYTYPE_AES192_MASK                                     0x05U   /* Definition for Key Type as AES_192 (AES 192-bits key (AES_1925 key)) */
#define PH_KEYSTORE_SAM_KEYTYPE_2K3DESDF8_MASK                                  0x06U   /* Definition for Key Type as ISO_TDEA32 (TDEA ISO 10116 (32-bits CRC,
                                                                                         * 8-bytes MAC) (DES or 2TDEA key))
                                                                                         */
#define PH_KEYSTORE_SAM_KEYTYPE_AES256_MASK                                     0x07U   /* Definition for Key Type as AES_256 (AES 256-bits key (AES_256 key)) */
#define PH_KEYSTORE_SAM_KEYTYPE_LRP_AES128_MASK                                 0x08U   /* Definition for Key Type as AES_128LRP (AES 128-bits key (AES_128)
                                                                                         * restricted for LRP usage.)
                                                                                         */

/* Macros to update the SET configuration's first byte information. */
#define PH_KEYSTORE_SAM_SET0_ALLOW_DUMP_SESSION_KEY                             0x01U   /* Definition for allow dump session key / MIFARE key (from standard host) */
#define PH_KEYSTORE_SAM_SET0_KEEP_IV                                            0x04U   /* Definition for not resetting Initialization Vector. After finishing a
                                                                                         * cryptographic command the init vector (IV) will be kept and not reset.
                                                                                         */
#define PH_KEYSTORE_SAM_SET0_PL_KEY                                             0x80U   /* Definition for Programmable Logic Key. When this bit is set in SAM Master
                                                                                         * Key, it indicates for HOST Keys whether this key entry can be used to get
                                                                                         * access to SAM_PLExec command
                                                                                         */

/* Macros to update the SET configuration's second byte information. */
#define PH_KEYSTORE_SAM_SET1_AUTH_KEY                                           0x01U   /* Definition for Authentication Key. Indicates for Host Keys whether this key
                                                                                         * entry can be used to execute the SAM_Authenticate Host command
                                                                                         */
#define PH_KEYSTORE_SAM_SET1_DISABLE_KEY_ENTRY                                  0x02U   /* Definition for Disable key entry. The key entry is disabled and can be
                                                                                         * reactivated by a SAM_ChangeKeyEntry command (default configuration).
                                                                                         */
#define PH_KEYSTORE_SAM_SET1_LOCK_KEY                                           0x04U   /* Definition for Lock Key. Indicates for Host Keys whether this key entry can
                                                                                         * be used to execute the SAM_LockUnlock command
                                                                                         */
#define PH_KEYSTORE_SAM_SET1_DISABLE_CHANGE_KEY_PICC                            0x08U   /* Definition for Disable writing the key to a PICC. Indicates if the key stored
                                                                                         * in the key entry can be downloaded into a PICC
                                                                                         */
#define PH_KEYSTORE_SAM_SET1_DISABLE_DECRYPTION                                 0x10U   /* Definition for Disable decryption. Indicates if the key entry can be used to
                                                                                         * decrypt data with SAM_Decipher_Data and SAM_Decipher Offline_Data commands
                                                                                         */
#define PH_KEYSTORE_SAM_SET1_DISABLE_ENCRYPTION                                 0x20U   /* Definition for Disable encryption. Indicates if the key entry can be used to
                                                                                         * encrypt data with SAM_Encipher_Data and SAM_Encipher Offline_Data commands.
                                                                                         */
#define PH_KEYSTORE_SAM_SET1_DISABLE_VERIFY_MAC                                 0x40U   /* Definition for Disable verify MAC. Indicates if the key entry can be used to
                                                                                         * verify a MAC with SAM_Verify_MAC and SAM_Remove_SM commands.
                                                                                         */
#define PH_KEYSTORE_SAM_SET1_DISABLE_GENERATE_MAC                               0x80U   /* Definition for Disable generate MAC. Indicates if the key entry can be used
                                                                                         * to generate a MAC with SAM_Generate_MAC and SAM_Apply_SM commands.
                                                                                         */

/* Macros to update the ExtSET configuration's first byte information. */
#define PH_KEYSTORE_SAM_EXTSET0_ALLOW_DUMP_SECRET_KEY                           0x08U   /* Definition for Allow dump secret key (from standard host). The command
                                                                                         * SAM_DumpSecretKey is allowed for this key entry.
                                                                                         */
#define PH_KEYSTORE_SAM_EXTSET0_MANDATE_KEY_DIVERSIFICATION                     0x10U   /* Definition for Diversification restriction. Diversification when using
                                                                                         * this key entry is mandatory. The stored value of this key will only be used
                                                                                         * directly to compute a diversified key. Use for other cryptographic operations
                                                                                         * is impossible.
                                                                                         */
#define PH_KEYSTORE_SAM_EXTSET0_PERSONALIZATION_SAM                             0x20U   /* Definition for Reserved for personalization SAM. The key entry is reserved for
                                                                                         * personalization and is disabled for any other operations. The key can only be
                                                                                         * used for inclusion by SAM_EncipherKeyEntry and PKI_EncipherKeyEntries. The values
                                                                                         * of the change entry key (KeyNoCEK and KeyVerCEK) are ignored
                                                                                         */

/* Macros to update the ExtSET configuration's second byte information. */
#define PH_KEYSTORE_SAM_EXTSET1_KEY_USAGE_INT_HOST                              0x01U   /* Definition for Force key usage by Internal Host. Indicates whether the usage of
                                                                                         * this key entry are restricted to the Internal Host. When enabled, the key entry
                                                                                         * are considered disabled when accessed from the standard host.
                                                                                         */
#define PH_KEYSTORE_SAM_EXTSET1_KEY_CHANGE_INT_HOST                             0x02U   /* Definition for Force key change by Internal Host. Indicates whether the change
                                                                                         * of this key entry are restricted to the Internal Host. When enabled, the key
                                                                                         * entry cannot be changed from the standard host.
                                                                                         */
#define PH_KEYSTORE_SAM_EXTSET1_SESSION_KEY_USAGE_INT_HOST                      0x04U   /* Definition for Force session usage by Internal Host. Indicates whether the PICC
                                                                                         * session key derived from the usage of this key entry are restricted to the Internal
                                                                                         * Host. When enabled, the PICC session key entry is not usable by the standard host.
                                                                                         * In other words, the PICC secure messaging is controlled by the Programmable Logic.
                                                                                         */
#define PH_KEYSTORE_SAM_EXTSET1_DUMP_SECRET_KEY_INT_HOST                        0x08U   /* Definition for Allow dump secret key by Internal Host. When executed by the Internal
                                                                                         * Host, SAM_DumpSecretKey command is allowed for this key entry.
                                                                                         */
#define PH_KEYSTORE_SAM_EXTSET1_DUMP_SESSION_KEY_INT_HOST                       0x10U   /* Definition for Allow dump session key by Internal Host. When executed by the Internal
                                                                                         * Host, SAM_DumpSessionKey command is allowed for this key entry.
                                                                                         */

/* Macros to represent the key entry length returned by Sam for Cmd.GetKeyEntry command. */
#define PH_KEYSTORE_SAM_KEY_ENTRY_LEN_SAMAV2_FORMAT_VER_ABC                     13U     /* Definition for Key Entry length with Version ABC in SAM AV2 format. */
#define PH_KEYSTORE_SAM_KEY_ENTRY_LEN_SAMAV2_FORMAT_VER_AB                      12U     /* Definition for Key Entry length with Version AC in SAM AV2 format. */
#define PH_KEYSTORE_SAM_KEY_ENTRY_LEN_SAM_FORMAT_VER_ABC                        16U     /* Definition for Key Entry length with Version ABC in SAM (AV4 and future SAM's) format. */
#define PH_KEYSTORE_SAM_KEY_ENTRY_LEN_SAM_FORMAT_VER_AB                         15U     /* Definition for Key Entry length with Version AB in SAM (AV4 and future SAM's) format. */
#define PH_KEYSTORE_SAM_KEY_ENTRY_LEN_SAM_FORMAT_VER_A                          14U     /* Definition for Key Entry length with Version A in SAM (AV4 and future SAM's) format. */
#define PH_KEYSTORE_SAM_KEY_ENTRY_LEN_SAM_RAM_KEY                               9U      /* Definition for Key Entry length for SAM RAM Keys. */

/* Macros to represent the version positions. */
#define PH_KEYSTORE_SAM_VERSION_POSITION_A                                      0U      /* Definition for SAM Version Position A. */
#define PH_KEYSTORE_SAM_VERSION_POSITION_B                                      1U      /* Definition for SAM Version Position B. */
#define PH_KEYSTORE_SAM_VERSION_POSITION_C                                      2U      /* Definition for SAM Version Position C. */

/* Macros to represent the Programming Bit Mask for SAM_ChangeKeyEntry command. */
#define PH_KEYSTORE_SAM_PRO_BIT_MASK_RAM_INCLUDE_VER_A                          0x8FU   /* Definition for SAM Non-volatile programming mask of SAM_ChangeKeyEntry command.
                                                                                         * Here VersionA alone and KeyNoCEK, KeyVCEK, RefNoKUC, SET, ExtSET and, if present,
                                                                                         * KeyNoAEK, KeyVAEK, Key versions specified separately will be included for Key
                                                                                         * Entry update
                                                                                         */
#define PH_KEYSTORE_SAM_PRO_BIT_MASK_NVM_INCLUDE_VER_A                          0x9FU   /* Definition for SAM Non-volatile programming mask of SAM_ChangeKeyEntry command.
                                                                                         * Here VersionA, DF_AID, KeyNoCEK, KeyVCEK, RefNoKUC, SET, ExtSET and, if present,
                                                                                         * KeyNoAEK, KeyVAEK, Key versions specified separately will be included for Key
                                                                                         * Entry update.
                                                                                         */
#define PH_KEYSTORE_SAM_PRO_BIT_MASK_NVM_INCLUDE_VER_AB                         0xDFU   /* Definition for SAM Non-volatile programming mask of SAM_ChangeKeyEntry command.
                                                                                         * Here VersionAB, DF_AID, KeyNoCEK, KeyVCEK, RefNoKUC, SET, ExtSET and, if present,
                                                                                         * KeyNoAEK, KeyVAEK, Key versions specified separately will be included for Key
                                                                                         * Entry update.
                                                                                         */
#define PH_KEYSTORE_SAM_PRO_BIT_MASK_NVM_INCLUDE_VER_ABC                        0xFFU   /* Definition for SAM Non-volatile programming mask of SAM_ChangeKeyEntry command.
                                                                                         * Here VersionABC, DF_AID, KeyNoCEK, KeyVCEK, RefNoKUC, SET, ExtSET and, if present,
                                                                                         * KeyNoAEK, KeyVAEK, Key versions specified separately will be included for Key Entry
                                                                                         * update.
                                                                                         */

/* Macros to represent the Programming Bit Mask for SAM_ChangeKUCEntry command. */
#define PH_KEYSTORE_SAM_UPDATE_LIMIT_MASK                                       0x80U   /* Definition for SAM Non-volatile programming mask of SAM_ChangeKUCEntry command.
                                                                                         * Update limit of the KUC entry.
                                                                                         */
#define PH_KEYSTORE_SAM_UPDATE_KEYNO_CKUC_MASK                                  0x40U   /* Definition for SAM Non-volatile programming mask of SAM_ChangeKUCEntry command.
                                                                                         * Update KeyNoCKUC
                                                                                         */
#define PH_KEYSTORE_SAM_UPDATE_KEY_VCKUC_MASK                                   0x20U   /* Definition for SAM Non-volatile programming mask of SAM_ChangeKUCEntry command.
                                                                                         * Update KeyVCKUC
                                                                                         */

#define PH_KEYSTORE_IS_RAM_KEY(KeyNo)   \
    (uint8_t) (((KeyNo) >= 0xE0U) && ((KeyNo) <= 0xE3U))

/**
 * Gets the information available in the key entry structure.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS          Operation successful.
 * \retval #PH_ERR_LENGTH_ERROR     If the received length is not proper.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phKeyStore_Sam_Int_GetKeyEntry(
        phKeyStore_Sam_DataParams_t * pDataParams,                                      /* Pointer to this layer's parameter structure. */
        uint8_t bKeyNo,                                                                 /* The key number to use for retrieving the key entry
                                                                                         * information from SAM KeyStore. One of the following,
                                                                                         *  - NVM Keys: 0x00 - 0x7F
                                                                                         *  - RAM Keys: 0xE0 - 0xE3
                                                                                         */
        uint8_t bIsRamKey,                                                              /* Indicates if the key number provided in \b bKeyNo is
                                                                                         * RAM Key.
                                                                                         *  - \ref PH_OFF "Standard NVM Key"
                                                                                         *  - \ref PH_ON "RAM Key"
                                                                                         */
        phKeyStore_Sam_KeyEntry_t * pKeyEntry                                           /* The key entry information for the mentioned key number. */
    );

/**
 * Convert the information available in the key entry structure to bytes.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS          Operation successful.
 * \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phKeyStore_Sam_Int_ConvertKeyEntryToBuffer(
        phKeyStore_Sam_KeyEntry_t * pKeyEntry,                                          /* The key entry information for the mentioned key number. */
        uint8_t * pKey,                                                                 /* The buffer containing the Key information. Here the KeyA,
                                                                                         * KeyB and KeyC will be combined together.
                                                                                         */
        uint8_t bKeyLen,                                                                /* Length of bytes available in \b pKey buffer. */
        uint8_t * pKeyEntryBuff,                                                        /* The buffer containing the KeyEntry information. */
        uint8_t * pKeyEntryBuffLen                                                      /* Length of bytes available in \b pKeyEntryBuff buffer. */
    );

/**
 * Sets the KeyType of the key entry.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_PARAMETER
 *                                      - For unsupported \b b2K3DESOption values.
                                        - For unsupported \b wKeyType values.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phKeyStore_Sam_Int_SetKeyType(
        phKeyStore_Sam_KeyEntry_t * pKeyEntry,                                          /* The key entry information for the mentioned key number. */
        uint16_t wKeyType,                                                              /* The key type of the KeyEntry. */
        uint8_t b2K3DESOption,                                                          /* Option to represent the different DES key types.
                                                                                         *  - \ref PH_KEYSTORE_SAM_DES_OPTION_DESFIRE4 "DESFire 4 compatibility mode"
                                                                                         *  - \ref PH_KEYSTORE_SAM_DES_OPTION_ISO_CRC16 "ISO 10116 mode with CRC16"
                                                                                         *  - \ref PH_KEYSTORE_SAM_DES_OPTION_ISO_CRC32 "ISO 10116 mode with CRC32"
                                                                                         */
        uint8_t bIsLRPKey                                                               /* Option to represent that the AES key to be used is for LRP.
                                                                                         *  - \ref PH_OFF "Non LRP Key"
                                                                                         *  - \ref PH_ON "LEP Key"
                                                                                         */
    );

/**
 * Gets the KeyType of the key entry.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_PARAMETER    For unsupported \b wKeyType values.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phKeyStore_Sam_Int_GetKeyType(
        phKeyStore_Sam_KeyEntry_t * pKeyEntry,                                          /* The key entry information for the mentioned key number. */
        uint16_t * pKeyType,                                                            /* The key type of the KeyEntry. */
        uint8_t * pIsLRPKey                                                             /* Option to represent that the AES key to be used is for LRP.
                                                                                         *  - \ref PH_OFF "Non LRP Key"
                                                                                         *  - \ref PH_ON "LEP Key"
                                                                                         */
    );

/**
 * Gets the size of key.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_PARAMETER    For unsupported \b wKeyType values.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phKeyStore_Sam_Int_GetKeySize(
        uint16_t wKeyType,                                                              /* The key type of the KeyEntry. */
        uint8_t * pKeySize                                                              /* The size of key for the provided key type. */
    );

#endif /* PHKEYSTORE_SAM_INT_H */
