/*
 * Copyright 2013 - 2014, 2017, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Log Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phLog.h>

#ifdef NXPBUILD__PH_LOG

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include "phLog_Int.h"

static const uint8_t *      PH_MEMLOC_CONST_ROM gpkphLog_String_BufferOverflow = (uint8_t*)"OVERFLOW";
static const uint8_t *      PH_MEMLOC_CONST_ROM gpkphLog_String_Null = (uint8_t*)"NULL";
static phLog_DataParams_t   PH_MEMLOC_REM gphLog_LogDataParams = {NULL, NULL, 0, 0};

phStatus_t phLog_Init(
                      pphLog_Callback_t pLogCallback,
                      phLog_RegisterEntry_t * pRegisterEntries,
                      uint16_t wMaxRegisterEntries
                      )
{
    /* Check parameters */
    if (pLogCallback != NULL)
    {
        if (wMaxRegisterEntries == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_LOG);
        }
        PH_ASSERT_NULL (pRegisterEntries);
    }

    gphLog_LogDataParams.pLogCallback           = pLogCallback;
    gphLog_LogDataParams.pRegisterEntries       = pRegisterEntries;
    gphLog_LogDataParams.wNumRegisterEntries    = 0;
    gphLog_LogDataParams.wMaxRegisterEntries    = wMaxRegisterEntries;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_LOG);
}

phStatus_t phLog_Register(
                          void * pDataParams,
                          phLog_LogEntry_t * pLogEntries,
                          uint16_t wMaxLogEntries
                          )
{
    /* Parameter check */
    if ((pDataParams == NULL) || (pLogEntries == NULL) || (wMaxLogEntries == 0))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_LOG);
    }

    /* Logging has to be initialized */
    if (gphLog_LogDataParams.pLogCallback == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_LOG);
    }

    /* Check for buffer overflow */
    if (gphLog_LogDataParams.wMaxRegisterEntries < (gphLog_LogDataParams.wNumRegisterEntries + 1))
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_LOG);
    }

    /* Add Register Entry */
    gphLog_LogDataParams.pRegisterEntries[gphLog_LogDataParams.wNumRegisterEntries].pDataParams = pDataParams;
    gphLog_LogDataParams.pRegisterEntries[gphLog_LogDataParams.wNumRegisterEntries].pLogEntries = pLogEntries;
    gphLog_LogDataParams.pRegisterEntries[gphLog_LogDataParams.wNumRegisterEntries].wNumLogEntries = 0;
    gphLog_LogDataParams.pRegisterEntries[gphLog_LogDataParams.wNumRegisterEntries].wMaxLogEntries = wMaxLogEntries;

    /* Increment number of registered entry pairs */
    ++gphLog_LogDataParams.wNumRegisterEntries;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_LOG);
}

void phLog_AddString(
                     void * pDataParams,
                     uint8_t bLogType,
                     char const * pFunctionName
                     )
{
    phLog_AddParam_Raw(pDataParams, bLogType, pFunctionName, NULL, 0, PH_LOG_DATATYPE_BUFFER);
}

void phLog_AddParam_Uint8(
                          void * pDataParams,
                          uint8_t bLogType,
                          char const * pName,
                          uint8_t * pParam
                          )
{
    phLog_AddParam_Raw(pDataParams, bLogType, pName, pParam, sizeof(uint8_t), PH_LOG_DATATYPE_VALUE);
}

void phLog_AddParam_Uint16(
                           void * pDataParams,
                           uint8_t bLogType,
                           char const * pName,
                           uint16_t * pParam
                           )
{
    phLog_AddParam_Raw(pDataParams, bLogType, pName, pParam, sizeof(uint16_t), PH_LOG_DATATYPE_VALUE);
}

void phLog_AddParam_Uint32(
                           void * pDataParams,
                           uint8_t bLogType,
                           char const * pName,
                           uint32_t * pParam
                           )
{
    phLog_AddParam_Raw(pDataParams, bLogType, pName, pParam, sizeof(uint32_t), PH_LOG_DATATYPE_VALUE);
}

void phLog_AddParam_Uint64(
                           void * pDataParams,
                           uint8_t bLogType,
                           char const * pName,
                           uint64_t * pParam
                           )
{
    phLog_AddParam_Raw(pDataParams, bLogType, pName, pParam, sizeof(uint64_t), PH_LOG_DATATYPE_VALUE);
}

void phLog_AddParam_Int16(
	void * pDataParams,
	uint8_t bLogType,
	char const * pName,
	int16_t * pParam
)
{
	phLog_AddParam_Raw(pDataParams, bLogType, pName, pParam, sizeof(int16_t), PH_LOG_DATATYPE_VALUE);
}

void phLog_AddParam_Int32(
	void * pDataParams,
	uint8_t bLogType,
	char const * pName,
	int32_t * pParam
)
{
	phLog_AddParam_Raw(pDataParams, bLogType, pName, pParam, sizeof(int32_t), PH_LOG_DATATYPE_VALUE);
}

void phLog_AddParam_Int64(
                          void * pDataParams,
                          uint8_t bLogType,
                          char const * pName,
                          int64_t * pParam
                         )
{
    phLog_AddParam_Raw(pDataParams, bLogType, pName, pParam, sizeof(int64_t), PH_LOG_DATATYPE_VALUE);
}

void phLog_AddParam_Float64(
                            void * pDataParams,
                            uint8_t bLogType,
                            char const * pName,
                            float64_t * pParam
                           )
{
    phLog_AddParam_Raw(pDataParams, bLogType, pName, pParam, sizeof(float64_t), PH_LOG_DATATYPE_VALUE);
}

void phLog_AddParam_Buffer(
                           void * pDataParams,
                           uint8_t bLogType,
                           char const * pName,
                           void const * pParam,
                           uint16_t wLength
                           )
{
    phLog_AddParam_Raw(pDataParams, bLogType, pName, pParam, wLength, PH_LOG_DATATYPE_BUFFER);
}

void phLog_Execute(
                   void * pDataParams,
                   uint8_t bOption
                   )
{
    phLog_RegisterEntry_t * PH_MEMLOC_REM pRegisterEntry;
    uint16_t wOldLogEntriesStart;

    /* Don't do anything if no callback has been registered */
    if (gphLog_LogDataParams.pLogCallback == NULL)
    {
        return;
    }

    /* Retrieve corresponding Register Entry */
    pRegisterEntry = phLog_GetRegisteredEntry(pDataParams);
    if (pRegisterEntry == NULL)
    {
        return;
    }

    /* Store old number of entries */
    wOldLogEntriesStart = pRegisterEntry->wLogEntriesStart;
    /* Change to actual used number so a new logging starts at pRegisterEntry->wNumLogEntries */
    pRegisterEntry->wLogEntriesStart = pRegisterEntry->wNumLogEntries;

    /* Invoke the external callback */
    (*(gphLog_LogDataParams.pLogCallback))(pDataParams, bOption, &pRegisterEntry->pLogEntries[wOldLogEntriesStart], pRegisterEntry->wNumLogEntries - wOldLogEntriesStart);

    /* Restore wLogEntriesStart */
    pRegisterEntry->wLogEntriesStart = wOldLogEntriesStart;

    /* Reset/Restore entries */
    pRegisterEntry->wNumLogEntries = wOldLogEntriesStart;
}

phLog_RegisterEntry_t * phLog_GetRegisteredEntry(
    void * pDataParams
    )
{
    phLog_RegisterEntry_t * PH_MEMLOC_REM pRegisterEntry;
    uint16_t                PH_MEMLOC_COUNT wIndex;

    /* Init. return value to NULL */
    pRegisterEntry = NULL;

    /* Iterate through all Register Entrys to find given one */
    for (wIndex = 0; wIndex < gphLog_LogDataParams.wNumRegisterEntries; ++wIndex)
    {
        if (pDataParams == gphLog_LogDataParams.pRegisterEntries[wIndex].pDataParams)
        {
            pRegisterEntry = &gphLog_LogDataParams.pRegisterEntries[wIndex];
        }
    }

    return pRegisterEntry;
}

void phLog_AddParam_Raw(
                        void * pDataParams,
                        uint8_t bLogType,
                        char const * pName,
                        void const * pParam,
                        uint16_t wLength,
                        uint8_t bDataType
                        )
{
    phLog_RegisterEntry_t * PH_MEMLOC_REM pRegisterEntry;

    /* Don't do anything if no callback has been registered */
    if (gphLog_LogDataParams.pLogCallback == NULL)
    {
        return;
    }

    /* Retrieve corresponding Register Entry */
    pRegisterEntry = phLog_GetRegisteredEntry(pDataParams);
    if (pRegisterEntry == NULL)
    {
        return;
    }

    /* check if entry is available */
    if (pRegisterEntry->wMaxLogEntries <= pRegisterEntry->wNumLogEntries)
    {
        return;
    }

    /* Build up the paramter */
    if (pRegisterEntry->wMaxLogEntries > (pRegisterEntry->wNumLogEntries + 1))
    {
        pRegisterEntry->pLogEntries[pRegisterEntry->wNumLogEntries].bLogType = bLogType;
        pRegisterEntry->pLogEntries[pRegisterEntry->wNumLogEntries].pString = (uint8_t const*)pName;
        pRegisterEntry->pLogEntries[pRegisterEntry->wNumLogEntries].pData = (pParam == NULL) ? (uint8_t const*) gpkphLog_String_Null : pParam;
        pRegisterEntry->pLogEntries[pRegisterEntry->wNumLogEntries].wDataLen = (pParam == NULL) ? 0 : wLength;
    }
    /* Else return buffer overflow error */
    else
    {
        pRegisterEntry->pLogEntries[pRegisterEntry->wNumLogEntries].bLogType = PH_LOG_LOGTYPE_ERROR;
        pRegisterEntry->pLogEntries[pRegisterEntry->wNumLogEntries].pString = (uint8_t const*)gpkphLog_String_BufferOverflow;
        pRegisterEntry->pLogEntries[pRegisterEntry->wNumLogEntries].pData = NULL;
        pRegisterEntry->pLogEntries[pRegisterEntry->wNumLogEntries].wDataLen = 0;
    }

    /* Also store the endianness of the data */
    pRegisterEntry->pLogEntries[pRegisterEntry->wNumLogEntries].bDataType = bDataType;

    /* Increment buffer entries */
    ++pRegisterEntry->wNumLogEntries;
}

#ifdef _DEBUG
phStatus_t phLog_PrintCheckError(uint8_t *pFilename, uint32_t dwLine, phStatus_t status)
{
    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "PH_CHECK_SUCCESS");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(file);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(line);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    /* pVar is a warning free pointer that points to address '1' */
    uint8_t * pVar = 0;
    pVar++;

    phLog_AddString((void *)pVar, PH_LOG_LOGTYPE_INFO, bFunctionName);
    phLog_AddParam_Buffer((void *)pVar, PH_LOG_LOGTYPE_DEBUG, file_log, pFilename, (uint16_t)strlen((const char *)pFilename));
    phLog_AddParam_Uint32((void *)pVar, PH_LOG_LOGTYPE_DEBUG, line_log, &dwLine);
    phLog_AddParam_Uint16((void *)pVar, PH_LOG_LOGTYPE_DEBUG, status_log, &status);
    phLog_Execute((void *)pVar, PH_LOG_OPTION_CATEGORY_ENTER);

    phLog_AddString((void *)pVar, PH_LOG_LOGTYPE_INFO, bFunctionName);
    phLog_Execute((void *)pVar, PH_LOG_OPTION_CATEGORY_LEAVE);
    return status;
}

phStatus_t phLog_PrintSetError(uint8_t *pFilename, uint32_t dwLine, uint16_t stat, uint16_t code)
{
    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "PH_ADD_COMPCODE");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(file);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(line);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    phStatus_t result = ((phStatus_t)(((stat) == PH_ERR_SUCCESS) || ((stat) & PH_COMP_MASK) ? (stat) : (((stat) & PH_ERR_MASK) | ((code) & PH_COMP_MASK))));
    /* pVar is a warning free pointer that points to address '1' */
    uint8_t * pVar = 0;
    pVar++;

    if (result != PH_ERR_SUCCESS)
    {
		phLog_AddString((void *)pVar, PH_LOG_LOGTYPE_INFO, bFunctionName);
        phLog_AddParam_Buffer((void *)pVar, PH_LOG_LOGTYPE_DEBUG, file_log, pFilename, (uint16_t)strlen((const char *)pFilename));
        phLog_AddParam_Uint32((void *)pVar, PH_LOG_LOGTYPE_DEBUG, line_log, &dwLine);
        phLog_AddParam_Uint16((void *)pVar, PH_LOG_LOGTYPE_DEBUG, status_log, &result);
        phLog_Execute((void *)pVar, PH_LOG_OPTION_CATEGORY_ENTER);

        phLog_AddString((void *)pVar, PH_LOG_LOGTYPE_INFO, bFunctionName);
        phLog_Execute((void *)pVar, PH_LOG_OPTION_CATEGORY_LEAVE);
    }
    return result;
}
#else
phStatus_t phLog_PrintCheckError(uint8_t *pFilename, uint32_t dwLine, phStatus_t status)
{
    /* Fill Dummy */
	pFilename =  NULL;
	dwLine = 0;
	status = PH_ERR_SUCCESS;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_LOG);
}

phStatus_t phLog_PrintSetError(uint8_t *pFilename, uint32_t dwLine, uint16_t stat, uint16_t code)
{
    /* Fill Dummy */
	pFilename =  NULL;
	dwLine = 0;
	stat = 0;
	code = 0;
   return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_LOG);
}
#endif
#endif /* NXPBUILD__PH_LOG */
