/*
 * Copyright 2013, 2017, 2019, 2020, 2022, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Tools Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phTools.h>
#include <ph_RefDefs.h>

#pragma warning(push)                   /* PRQA S 3116 */
#pragma warning(disable:4001)           /* PRQA S 3116 */
#pragma warning(disable:4200)           /* PRQA S 3116 */
#pragma warning(disable:4201)           /* PRQA S 3116 */
#include <windows.h>
#include <stdio.h>                      /* PRQA S 5124 */
#pragma warning(pop)                    /* PRQA S 3116 */

static uint8_t phTools_CalcParity(uint8_t bDataByte, uint8_t bMode);

phStatus_t phTools_EncodeParity(
                                uint8_t bOption,
                                uint8_t * pInBuffer,
                                uint16_t wInBufferLength,
                                uint8_t bInBufferBits,
                                uint16_t wOutBufferSize,
                                uint8_t * pOutBuffer,
                                uint16_t * pOutBufferLength,
                                uint8_t * pOutBufferBits
                                )
{
    uint16_t    PH_MEMLOC_REM wByteIndexIn;
    uint16_t    PH_MEMLOC_REM wByteIndexOut;
    uint8_t     PH_MEMLOC_REM bBitPosition;
    uint16_t    PH_MEMLOC_REM wInByteCount;

    /* Parameter check */
    if (((bOption != PH_TOOLS_PARITY_OPTION_EVEN) && (bOption != PH_TOOLS_PARITY_OPTION_ODD)) || (bInBufferBits > 7))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);
    }

    /* Retrieve full input byte count */
    if (bInBufferBits == 0)
    {
        wInByteCount = wInBufferLength;
    }
    else
    {
        wInByteCount = wInBufferLength - 1;
    }

    /* Retrieve number of (additional) full bytes */
    *pOutBufferLength = (uint16_t)((uint16_t)(wInByteCount + bInBufferBits) >> 3);

    /* Retrieve output bits */
    *pOutBufferBits = (uint8_t)((uint16_t)(wInByteCount + bInBufferBits) % 8);

    /* Increment output length in case of incomplete byte */
    if (*pOutBufferBits > 0)
    {
        ++(*pOutBufferLength);
    }

    /* Overflow check */
    if (*pOutBufferLength > (0xFFFF - wInByteCount))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);
    }

    /* Calculate number of output bytes */
    *pOutBufferLength = wInByteCount + *pOutBufferLength;

    /* Buffer overflow check*/
    if (wOutBufferSize < *pOutBufferLength)
    {
        *pOutBufferLength = 0;
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_TOOLS);
    }

    /* Prepare output buffer */
    pOutBuffer[0] = 0x00;

    /* Prepare loop vars */
    wByteIndexIn = 0;
    wByteIndexOut = 0;
    bBitPosition = 7;

    /* Do for each byte */
    for (; wByteIndexIn < wInBufferLength; ++wByteIndexIn, ++wByteIndexOut, --bBitPosition)
    {
        /* Append source bits to output */
        pOutBuffer[wByteIndexOut] |= (uint8_t)(pInBuffer[wByteIndexIn] << (7 - bBitPosition));

        /* If there is more data bits in the sourcebyte append it to next data byte */
        if ((wByteIndexOut + 1) < *pOutBufferLength)
        {
            pOutBuffer[wByteIndexOut + 1]  = (uint8_t)(pInBuffer[wByteIndexIn] >> (1 + bBitPosition));

            /* Perform parity appending if this isn't an incomplete byte */
            if ((bInBufferBits == 0) || ((wByteIndexIn + 1) < wInBufferLength))
            {
                pOutBuffer[wByteIndexOut + 1] |= (uint8_t)(phTools_CalcParity(pInBuffer[wByteIndexIn], bOption) << (7 - bBitPosition));
            }
        }

        /* We have reached the 8th parity bit, the output buffer index is now one ahead */
        if (bBitPosition == 0)
        {
            if ((wByteIndexOut + 2) < *pOutBufferLength)
            {
                bBitPosition = 8;
                pOutBuffer[++wByteIndexOut + 1] = 0x00;
            }
        }
    }

    /* Mask out invalid bits of last byte */
    if (*pOutBufferBits > 0)
    {
        pOutBuffer[*pOutBufferLength - 1] &= (uint8_t)(0xFF >> (8 - *pOutBufferBits));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phTools_DecodeParity(
                                uint8_t bOption,
                                uint8_t * pInBuffer,
                                uint16_t wInBufferLength,
                                uint8_t bInBufferBits,
                                uint16_t wOutBufferSize,
                                uint8_t * pOutBuffer,
                                uint16_t * pOutBufferLength,
                                uint8_t * pOutBufferBits
                                )
{
    uint16_t    PH_MEMLOC_REM wByteIndexIn;
    uint16_t    PH_MEMLOC_REM wByteIndexOut;
    uint8_t     PH_MEMLOC_REM bBitPosition;
    uint16_t    PH_MEMLOC_REM wDiv;
    uint8_t     PH_MEMLOC_REM bMod;
    uint8_t     PH_MEMLOC_REM bParity;

    /* Parameter check */
    if (((bOption != PH_TOOLS_PARITY_OPTION_EVEN) && (bOption != PH_TOOLS_PARITY_OPTION_ODD) && (bOption != PH_TOOLS_PARITY_OPTION_DONT_CARE)) || (bInBufferBits > 7))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);
    }

    /* Parameter check */
    if (wInBufferLength == 0)
    {
        /* Zero input length is simply passed through */
        if (bInBufferBits == 0)
        {
            *pOutBufferLength = 0;
            *pOutBufferBits = 0;
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
        }
        /* Invalid parameter */
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);
        }
    }

    /* Retrieve DIV and MOD */
    if (bInBufferBits == 0)
    {
        wDiv = (uint16_t)(wInBufferLength / 9);
        bMod = (uint8_t)(wInBufferLength % 9);
    }
    else
    {
        wDiv = (uint16_t)((wInBufferLength - 1) / 9);
        bMod = (uint8_t)((wInBufferLength - 1) % 9);
    }

    /* Calculate number of output bytes */
    *pOutBufferLength = (uint16_t)((wDiv << 3) + bMod);
    if (bMod > bInBufferBits)
    {
        --(*pOutBufferLength);
    }

    /* Calculate number of rest-bits of output */
    *pOutBufferBits = (uint8_t)((8 - (((8 + (*pOutBufferLength % 8)) - bInBufferBits) % 8)) % 8);

    /* Increment output length in case of incomplete byte */
    if (*pOutBufferBits > 0)
    {
        ++(*pOutBufferLength);
    }

    /* Buffer overflow check*/
    if (wOutBufferSize < *pOutBufferLength)
    {
        *pOutBufferLength = 0;
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_TOOLS);
    }

    /* Prepare loop vars */
    wByteIndexIn = 0;
    wByteIndexOut = 0;
    bBitPosition = 7;

    /* Do for each byte */
    for (; wByteIndexOut < *pOutBufferLength; ++wByteIndexOut, ++wByteIndexIn, --bBitPosition)
    {
        /* Append source bits to output */
        pOutBuffer[wByteIndexOut] = (uint8_t)(pInBuffer[wByteIndexIn] >> (7 - bBitPosition));

        /* If there is more data bits in the sourcebyte append it to next data byte */
        if ((wByteIndexIn + 1) < wInBufferLength)
        {
            /* Append remaining bits to output */
            pOutBuffer[wByteIndexOut] |= (uint8_t)(pInBuffer[wByteIndexIn + 1] << (1 + bBitPosition));

            /* Perform parity checking if this isn't an incomplete byte */
            if ((*pOutBufferBits == 0) || ((wByteIndexOut + 1) < *pOutBufferLength))
            {
                if (bOption != PH_TOOLS_PARITY_OPTION_DONT_CARE)
                {
                    bParity = phTools_CalcParity(pOutBuffer[wByteIndexOut], bOption);
                    if ((pInBuffer[wByteIndexIn + 1] & (uint8_t)(1 << (7 - bBitPosition))) != (bParity << (7 - bBitPosition)))
                    {
                        return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_TOOLS);
                    }
                }
            }
        }

        /* We have reached the 8th parity bit, the input buffer index is now one ahead */
        if (bBitPosition == 0)
        {
            bBitPosition = 8;
            ++wByteIndexIn;
        }
    }

    /* Mask out invalid bits of last byte */
    if (*pOutBufferBits > 0)
    {
        pOutBuffer[*pOutBufferLength - 1] &= (uint8_t)(0xFF >> (8 - *pOutBufferBits));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phTools_CalculateCrc5(
                                 uint8_t bOption,
                                 uint8_t bPreset,
                                 uint8_t bPolynom,
                                 uint8_t * pData,
                                 uint16_t wDataLength,
                                 uint8_t * pCrc
                                 )
{
    uint16_t PH_MEMLOC_REM wDataIndex = 0;
    uint8_t  PH_MEMLOC_REM bBitIndex;
    uint8_t  PH_MEMLOC_REM bBitMax;

    if (bOption & (uint8_t)~(uint8_t)PH_TOOLS_CRC_OPTION_MASK)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);
    }

    *pCrc = bPreset;

    if (bOption & PH_TOOLS_CRC_OPTION_MSB_FIRST)
    {
        /* Shift 5bit preset to 8bit (data) alignment */
        *pCrc <<= 3;

        /* Shift 5bit polinom to 8bit (data) alignment */
        bPolynom <<= 3;
    }

    /* Loop through all data bytes */
    while (wDataLength)
    {
        /* XOR input data */
        if (bOption & PH_TOOLS_CRC_OPTION_BITWISE)
        {
            if (wDataLength < 8)
            {
                bBitMax = (uint8_t)wDataLength;
                wDataLength = 0;
            }
            else
            {
                bBitMax = 8;
                wDataLength -= 8;
            }
        }
        else
        {
            bBitMax = 8;
            /* Decrement DataLen */
            --wDataLength;
        }

        /* CRC polynom (MSB first) */
        if (bOption & PH_TOOLS_CRC_OPTION_MSB_FIRST)
        {
            *pCrc ^= pData[wDataIndex++] & (0xFFU << (8 - bBitMax));

            for (bBitIndex = 0; bBitIndex < bBitMax; ++bBitIndex)
            {
                if ((*pCrc) & 0x80)
                {
                    *pCrc = (uint8_t)(((*pCrc) << 1) ^ bPolynom);
                }
                else
                {
                    *pCrc = (*pCrc) << 1;
                }
            }
        }
        /* CRC polynom (LSB first) */
        else
        {
            *pCrc ^= pData[wDataIndex++] & (0xFFU >> (8 - bBitMax));

            for (bBitIndex = 0; bBitIndex < bBitMax; ++bBitIndex)
            {
                if ((*pCrc) & 0x01)
                {
                    *pCrc = (uint8_t)(((*pCrc) >> 1) ^ bPolynom);
                }
                else
                {
                    *pCrc = ((*pCrc) >> 1);
                }
            }
        }
    }

    if (bOption & PH_TOOLS_CRC_OPTION_MSB_FIRST)
    {
        /* Shift back for 5bit alignment */
        *pCrc >>= 3;
    }

    /* Invert CRC if requested */
    if (bOption & PH_TOOLS_CRC_OPTION_OUPUT_INVERTED)
    {
        *pCrc ^= 0x1FU;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phTools_CalculateCrc8(
                                 uint8_t bOption,
                                 uint8_t bPreset,
                                 uint8_t bPolynom,
                                 uint8_t * pData,
                                 uint16_t wDataLength,
                                 uint8_t * pCrc
                                 )
{
    uint16_t PH_MEMLOC_REM wDataIndex = 0;
    uint8_t  PH_MEMLOC_REM bBitIndex;
    uint8_t  PH_MEMLOC_REM bBitMax;

    if (bOption & (uint8_t)~(uint8_t)PH_TOOLS_CRC_OPTION_MASK)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);
    }

    *pCrc = bPreset;

    /* Loop through all data bytes */
    while (wDataLength)
    {
        /* XOR input data */
        if (bOption & PH_TOOLS_CRC_OPTION_BITWISE)
        {
            if (wDataLength < 8)
            {
                bBitMax = (uint8_t)wDataLength;
                wDataLength = 0;
            }
            else
            {
                bBitMax = 8;
                wDataLength -= 8;
            }
        }
        else
        {
            bBitMax = 8;
            /* Decrement DataLen */
            --wDataLength;
        }

        /* CRC polynom (MSB first) */
        if (bOption & PH_TOOLS_CRC_OPTION_MSB_FIRST)
        {
            *pCrc ^= pData[wDataIndex++] & (0xFFU << (8 - bBitMax));

            for (bBitIndex = 0; bBitIndex < bBitMax; ++bBitIndex)
            {
                if ((*pCrc) & 0x80)
                {
                    *pCrc = (uint8_t)(((*pCrc) << 1) ^ bPolynom);
                }
                else
                {
                    *pCrc = ((*pCrc) << 1);
                }
            }
        }
        /* CRC polynom (LSB first) */
        else
        {
            *pCrc ^= pData[wDataIndex++] & (0xFFU >> (8 - bBitMax));

            for (bBitIndex = 0; bBitIndex < bBitMax; ++bBitIndex)
            {
                if ((*pCrc) & 0x01)
                {
                    *pCrc = (uint8_t)(((*pCrc) >> 1) ^ bPolynom);
                }
                else
                {
                    *pCrc = ((*pCrc) >> 1);
                }
            }
        }
    }

    /* Invert CRC if requested */
    if (bOption & PH_TOOLS_CRC_OPTION_OUPUT_INVERTED)
    {
        *pCrc ^= 0xFFU;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phTools_CalculateCrc16(
                                  uint8_t bOption,
                                  uint16_t wPreset,
                                  uint16_t wPolynom,
                                  uint8_t * pData,
                                  uint16_t wDataLength,
                                  uint16_t * pCrc
                                  )
{
    uint16_t PH_MEMLOC_REM wDataIndex = 0;
    uint8_t  PH_MEMLOC_REM bBitIndex;
    uint8_t  PH_MEMLOC_REM bBitMax;

    if (bOption & (uint8_t)~(uint8_t)PH_TOOLS_CRC_OPTION_MASK)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);
    }

    *pCrc = wPreset;

    /* Loop through all data bytes */
    while (wDataLength)
    {
        /* XOR input data */
        if (bOption & PH_TOOLS_CRC_OPTION_BITWISE)
        {
            if (wDataLength < 8)
            {
                bBitMax = (uint8_t)wDataLength;
                wDataLength = 0;
            }
            else
            {
                bBitMax = 8;
                wDataLength -= 8;
            }
        }
        else
        {
            bBitMax = 8;
            /* Decrement DataLen */
            --wDataLength;
        }

        /* CRC polynom (MSB first) */
        if (bOption & PH_TOOLS_CRC_OPTION_MSB_FIRST)
        {
            *pCrc ^= (uint16_t)pData[wDataIndex++] << 8;

            for (bBitIndex = 0; bBitIndex < bBitMax; ++bBitIndex)
            {
                if ((*pCrc) & 0x8000)
                {
                    *pCrc = (uint16_t)(((*pCrc) << 1) ^ wPolynom);
                }
                else
                {
                    *pCrc = ((*pCrc) << 1);
                }
            }
        }
        /* CRC polynom (LSB first) */
        else
        {
            *pCrc ^= pData[wDataIndex++];

            for (bBitIndex = 0; bBitIndex < bBitMax; ++bBitIndex)
            {
                if ((*pCrc) & 0x0001)
                {
                    *pCrc = (uint16_t)(((*pCrc) >> 1) ^ wPolynom);
                }
                else
                {
                    *pCrc = ((*pCrc) >> 1);
                }
            }
        }
    }

    /* Invert CRC if requested */
    if (bOption & PH_TOOLS_CRC_OPTION_OUPUT_INVERTED)
    {
        *pCrc ^= 0xFFFFU;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phTools_CalculateCrc32(
                                  uint8_t bOption,
                                  uint32_t dwPreset,
                                  uint32_t dwPolynom,
                                  uint8_t * pData,
                                  uint32_t dwDataLength,
                                  uint32_t * pCrc
                                  )
{
    uint32_t PH_MEMLOC_REM dwDataIndex = 0;
    uint8_t  PH_MEMLOC_REM bBitIndex;
    uint8_t  PH_MEMLOC_REM bBitMax;
    uint8_t  PH_MEMLOC_REM bData;

    if (bOption & (uint8_t)~(uint8_t)PH_TOOLS_CRC32_OPTION_MASK)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);
    }

    *pCrc = dwPreset;

    /* Loop through all data bytes */
    while (dwDataLength)
    {
        /* XOR input data */
        if (bOption & PH_TOOLS_CRC_OPTION_BITWISE)
        {
            if (dwDataLength < 8)
            {
                bBitMax = (uint8_t)dwDataLength;
                dwDataLength = 0;
            }
            else
            {
                bBitMax = 8;
                dwDataLength -= 8;
            }
        }
        else
        {
            bBitMax = 8;
            /* Decrement DataLen */
            --dwDataLength;
        }

        bData = pData[dwDataIndex++];
        if (bOption & PH_TOOLS_CRC32_OPTION_REVERSE_DATA_BYTE)
        {
            bData = ((bData & 0x55) <<  1) | ((bData >>  1) & 0x55);
            bData = ((bData & 0x33) <<  2) | ((bData >>  2) & 0x33);
            bData = ((bData & 0x0F) <<  4) | ((bData >>  4) & 0x0F);
        }

        /* CRC polynom (MSB first) */
        if (bOption & PH_TOOLS_CRC_OPTION_MSB_FIRST)
        {
            *pCrc ^= (uint32_t)bData << 24;

            for (bBitIndex = 0; bBitIndex < bBitMax; ++bBitIndex)
            {
                if ((*pCrc) & 0x80000000)
                {
                    *pCrc = (uint32_t)(((*pCrc) << 1) ^ dwPolynom);
                }
                else
                {
                    *pCrc = ((*pCrc) << 1);
                }
            }
        }
        /* CRC polynom (LSB first) */
        else
        {
            *pCrc ^= bData;

            for (bBitIndex = 0; bBitIndex < bBitMax; ++bBitIndex)
            {
                if ((*pCrc) & 0x00000001)
                {
                    *pCrc = (uint32_t)(((*pCrc) >> 1) ^ dwPolynom);
                }
                else
                {
                    *pCrc = ((*pCrc) >> 1);
                }
            }
        }
    }

    if (bOption & PH_TOOLS_CRC32_OPTION_REVERSE_CRC)
    {
        *pCrc = ((*pCrc & 0x55555555) <<  1) | ((*pCrc >>  1) & 0x55555555);
        *pCrc = ((*pCrc & 0x33333333) <<  2) | ((*pCrc >>  2) & 0x33333333);
        *pCrc = ((*pCrc & 0x0F0F0F0F) <<  4) | ((*pCrc >>  4) & 0x0F0F0F0F);
        *pCrc = (*pCrc << 24) | ((*pCrc & 0xFF00) << 8) |
            ((*pCrc >> 8) & 0xFF00) | (*pCrc >> 24);
    }

    /* Invert CRC if requested */
    if (bOption & PH_TOOLS_CRC_OPTION_OUPUT_INVERTED)
    {
        *pCrc ^= 0xFFFFFFFFU;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phTools_CalculateLrc(
                                uint8_t * pData,
                                uint16_t wDataLength,
                                uint8_t * pLrc
                                )
{
    uint16_t PH_MEMLOC_REM i;
    *pLrc = 0;

    for (i = 0; i < wDataLength; i++)
    {
        *pLrc ^= pData[i];
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

static uint8_t phTools_CalcParity(
                                  uint8_t bDataByte,
                                  uint8_t bOption
                                  )
{
    uint8_t PH_MEMLOC_REM bBit;
    uint8_t PH_MEMLOC_REM bParity;

    if (bOption == PH_TOOLS_PARITY_OPTION_EVEN)
    {
        bParity = 0x00;
    }
    else
    {
        bParity = 0x01;
    }

    for (bBit = 0; bBit < 8; ++bBit)
    {
        if (bDataByte & (uint8_t)(1 << bBit))
        {
            ++bParity;
        }
    }
    return bParity & 0x01;
}

phStatus_t phTools_HammingEncode(
                                 uint8_t* pData,
                                 uint8_t* pEnhancedSubBlock,
                                 uint16_t wEnhancedSubBlockSize,
                                 uint16_t wEnhancedSubBlockOffset
                                 )
{
    /* Hamming control byte */
    uint8_t c = 0x00;

    uint8_t currByte = 0;
    uint8_t currBit = 0;

    /* Hamming control generation matrix A */
    uint8_t A[] = {
                0x03, 0x05, 0x06, 0x07, 0x09, 0x0A, 0x0B, 0x0C,
                0x0D, 0x0E, 0x0F, 0x11, 0x12, 0x13, 0x14, 0x15,
                0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D,
                0x1E, 0x1F, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
                0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E,
                0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36,
                0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E,
                };

    if ((uint32_t)wEnhancedSubBlockOffset + 8 > (uint32_t)wEnhancedSubBlockSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_TOOLS);
    }

    /* copy data */
    memcpy(pEnhancedSubBlock + wEnhancedSubBlockOffset, pData, 7);

    /* iterate through all bits of the 7-byte sub-block from enhanced block */
    for (currByte = 0; currByte < 7; currByte++) /* 7 bytes */
    {
        for (currBit = 0; currBit < 8; currBit++) /* 8 bits each */
        {
            /* if bit is equal 1 XOR with corresponding column of matrix A */
            if ((pEnhancedSubBlock[wEnhancedSubBlockOffset + currByte] >> currBit) & 0x01)
            {
                c ^= A[currByte * 8 + currBit];
            }
        }
    }

    /* add padding bits */
    c = (c << 1) | 0x81;

    /* add HammingByte to SubBlock */
    pEnhancedSubBlock[wEnhancedSubBlockOffset + 7] = c;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phTools_HammingEncodeMoreBlocks(
                                           uint8_t*  pData,
                                           uint16_t  wDataLen,
                                           uint8_t*  pEnhancedBlocks,
                                           uint16_t  wEnhancedBlocksSize,
                                           uint16_t* pwEnhancedBlocksLen
                                           )
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint16_t wCurrentBlockIndex;
    uint16_t wNumOfBlocks;

    /* Check if the input has an valid length (multiple of block size) */
    if (wDataLen % 7 != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
    }

    /* Calc number of blocks used (always round up) */
    wNumOfBlocks = wDataLen / 7;
    *pwEnhancedBlocksLen = wNumOfBlocks * 8;

    /* Check result buffer size */
    if (*pwEnhancedBlocksLen > wEnhancedBlocksSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_ISO14443P4);
    }

    /* Encode all blocks */
    for (wCurrentBlockIndex = 0; wCurrentBlockIndex < wNumOfBlocks; wCurrentBlockIndex++)
    {
        PH_CHECK_SUCCESS_FCT(status, phTools_HammingEncode(pData + (wCurrentBlockIndex*7),
            pEnhancedBlocks, wEnhancedBlocksSize, wCurrentBlockIndex * 8));
    }
    return status;
}

phStatus_t phTools_HammingDecode(
                                        uint8_t* pEnhancedSubBlock,
                                        uint8_t* pDecodedData,
                                        uint16_t wDecodedDataSize,
                                        uint16_t wDecodedDataOffset
                                        )
{
    uint16_t bit_pos = 0;
    uint16_t parity = 1;
    uint16_t subtract = 0;
    uint8_t pCmpEnhancedSubBlock[8] = {0};

    phStatus_t status = PH_ERR_SUCCESS;

    if ((uint32_t)wDecodedDataOffset + 7 > (uint32_t)wDecodedDataSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_TOOLS);
    }

    PH_CHECK_SUCCESS_FCT(status, phTools_HammingEncode(pEnhancedSubBlock,
        pCmpEnhancedSubBlock, sizeof(pCmpEnhancedSubBlock), 0));

    /* check if Hamming Byte is equal */
    if(pCmpEnhancedSubBlock[7] == pEnhancedSubBlock[7])
    {
        memcpy(pDecodedData + wDecodedDataOffset, pCmpEnhancedSubBlock, 7);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
    }

    /* cut padding bits and XOR hamming bytes */
    bit_pos = ((pCmpEnhancedSubBlock[7] & 0x7F) >> 1)  ^
        ((pEnhancedSubBlock[7] & 0x7F) >> 1);

    /* if bit_pos = 0, 1, 2, 4, 8, 16, 32 or 63 no change in received bits */
    if(bit_pos == 0 || bit_pos == 1 || bit_pos == 2 || bit_pos == 4 || bit_pos == 8 || bit_pos == 16 || bit_pos == 32 || bit_pos == 63)
    {
        memcpy(pDecodedData + wDecodedDataOffset, pCmpEnhancedSubBlock, 7);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
    }

    /* subtract number of parity bits */
    while(parity < bit_pos)
    {
        parity *= 2;
        subtract++;
    }
    bit_pos -= subtract;

    /* error detected! correct error */
    pCmpEnhancedSubBlock[(bit_pos) / 8] ^= (0x01 << ((bit_pos - 1) % 8));

    /* do the check again */
    PH_CHECK_SUCCESS_FCT(status, phTools_HammingEncode(pCmpEnhancedSubBlock,
        pCmpEnhancedSubBlock, sizeof(pCmpEnhancedSubBlock), 0));

    /* check if Hamming Byte is equal */
    if(pCmpEnhancedSubBlock[7] == pEnhancedSubBlock[7])
    {
        memcpy(pDecodedData + wDecodedDataOffset, pCmpEnhancedSubBlock, 7);
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_TOOLS);
    }
}

phStatus_t phTools_HammingDecodeMoreBlocks(
                                           uint8_t* pEnhancedBlocks,
                                           uint16_t wEnhancedBlocksLen,
                                           uint8_t*  pDecodedBlocks,
                                           uint16_t  wDecodedBlocksSize,
                                           uint16_t* wDecodedBlocksLen
                                           )
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint16_t wCurrentBlockIndex;
    uint16_t wNumOfBlocks;

    /* Check if the input has an valid length (multiple of block size) */
    if (wEnhancedBlocksLen % 8 != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4);
    }

    /* Calc number of blocks used (always round up) */
    wNumOfBlocks = wEnhancedBlocksLen / 8;
    *wDecodedBlocksLen = wNumOfBlocks * 7;

    /* Check result buffer size */
    if (*wDecodedBlocksLen > wDecodedBlocksSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_PAL_ISO14443P4);
    }

    /* Encode all blocks */
    for (wCurrentBlockIndex = 0; wCurrentBlockIndex < wNumOfBlocks; wCurrentBlockIndex++)
    {
        PH_CHECK_SUCCESS_FCT(status, phTools_HammingDecode(pEnhancedBlocks + (wCurrentBlockIndex*8),
            pDecodedBlocks, wDecodedBlocksSize, wCurrentBlockIndex * 7));
    }
    return status;
}

phStatus_t phTools_Sleep(
                        uint32_t dwTimeMilliSeconds
                        )
{
    Sleep(dwTimeMilliSeconds);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}
