/*
 * Copyright 2019 - 2020, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Tools Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phToolsMicropross.h>
#include <phTools.h>
#include <ph_RefDefs.h>
#include "..\..\phHalHwContact\src\ContactMicropross\phhalHwContact_Micropross_Int.h"
#include <stdio.h>              /* PRQA S 5124 */
#include <stdarg.h>

#define SWAP(x)         ((((x) & 0xFF) << 8) | (((x) >> 8) & 0xFF))
#define SWAPDWORD(x)    SWAP(((x >> 16) & 0xffff)) | (SWAP(x & 0xffff)<<16)
#define SWAPDDWORD(x)    ((x & 0xff00ff00ff00ff00ull)>>8ull) | ((x & 0x00ff00ff00ff00ffull)<<8ull)

phStatus_t phToolsMicropross_EvaluateStatus(
                                            phhalHwContact_Micropross_DataParams_t * pDataParams,
                                            uint16_t wStatus
                                            )
{
    phStatus_t PH_MEMLOC_REM status;

    switch (wStatus)
    {
    case PHTOOLS_MICROPROSS_ERR_OK:
        status = PH_ERR_SUCCESS;
        break;

    case PHTOOLS_MICROPROSS_ERR_INVALID_PARAMETER:
        status = PH_ERR_INVALID_PARAMETER;
        break;

    case PHTOOLS_MICROPROSS_ERR_TIMEOUTRX:
        status = PH_ERR_IO_TIMEOUT;
        break;

    case PHTOOLS_MICROPROSS_ERR_PARITY:
    case PHTOOLS_MICROPROSS_ERR_CRC:
        status = PH_ERR_INTEGRITY_ERROR;
        break;

    case PHTOOLS_MICROPROSS_ERR_INCONSISTENT_RECEIVED_FRAME:
    case PHTOOLS_MICROPROSS_ERR_BITCOUNTERR:
        status = PH_ERR_FRAMING_ERROR;
        break;

    case PHTOOLS_MICROPROSS_ERR_ATS_INVALID_LENGTH:
    case PHTOOLS_MICROPROSS_ERR_INVALID_ATS_CRC:
    case PHTOOLS_MICROPROSS_ERR_INVALID_ATS_T0:
    case PHTOOLS_MICROPROSS_ERR_INVALID_ATS_FSCI:
    case PHTOOLS_MICROPROSS_ERR_INVALID_ATS_FWI:
    case PHTOOLS_MICROPROSS_ERR_INVALID_ATS_TC1:
    case PHTOOLS_MICROPROSS_ERR_INVALID_ATS_TA1:
    case PHTOOLS_MICROPROSS_ERR_SELECT_FAILED:
    case PHTOOLS_MICROPROSS_ERR_ANTICOL_FAILED:
    case PHTOOLS_MICROPROSS_ERR_ANSWER_TO_HALT_DETECTED:
        status = PH_ERR_PROTOCOL_ERROR;
        break;

    case PHTOOLS_MICROPROSS_ERR_COLLISION:
        status = PH_ERR_COLLISION_ERROR;
        break;

    case PHTOOLS_MICROPROSS_ERR_DATA_RATE_NOT_SUPPORTED:
        status = PH_ERR_UNSUPPORTED_PARAMETER;
        break;

    case PHTOOLS_MICROPROSS_ERR_INVALID_ANSWER_TO_HALT:
    case PHTOOLS_MICROPROSS_ERR_WRONG_ATTRIB:
    case PHTOOLS_MICROPROSS_ERR_DESELECT_SENT:
    case PHTOOLS_MICROPROSS_ERR_INVALID_ANSWER_TO_DESELECT:
    case PHTOOLS_MICROPROSS_ERR_TIMEOUTRX_CMD1:
    case PHTOOLS_MICROPROSS_ERR_TIMEOUTTX:
    case PHTOOLS_MICROPROSS_ERR_UNEXPECTED_RX:
    case PHTOOLS_MICROPROSS_ERR_FELICA_INVALID_LENGTH:
    case PHTOOLS_MICROPROSS_ERR_FELICA_INVALID_SYNC_CODE:
    case PHTOOLS_MICROPROSS_ERR_PPS_INVALID_PARAMETER:
    case PHTOOLS_MICROPROSS_ERR_PPS_INVALID_ANSWER:
    case PHTOOLS_MICROPROSS_ERR_CALIBRATION_1:
    case PHTOOLS_MICROPROSS_ERR_CALIBRATION_2:
    case PHTOOLS_MICROPROSS_ERR_CALIBRATION_3:
    case PHTOOLS_MICROPROSS_ERR_CALIBRATION_4:
    case PHTOOLS_MICROPROSS_ERR_INVALID_IMPEDANCE_CALIBRATION_TABLE:
    case PHTOOLS_MICROPROSS_ERR_NO_CABLE_INFORMATION:
    case PHTOOLS_MICROPROSS_ERR_NO_HEAD_INFORMATION:
    case PHTOOLS_MICROPROSS_ERR_RESONANCE_FREQUENCY_MEASUREMENT:
    case PHTOOLS_MICROPROSS_ERR_ADJUSTRX_CHANNEL_2:
    case PHTOOLS_MICROPROSS_ERR_NO_CABLE_PLUS_HEAD_INFORMATION:
    case PHTOOLS_MICROPROSS_ERR_NO_SHAPE_SET:
        pDataParams->wAdditionalInfo = wStatus;
        status = PHTOOLS_MICROPROSS_ERR_MICROPROSS;
        break;

    case PHTOOLS_MICROPROSS_ERR_CARD_MUTE:
    case PHTOOLS_MICROPROSS_ERR_CARD_TIMEOUT:
        pDataParams->wAdditionalInfo = wStatus;
        status = PH_ERR_IO_TIMEOUT;
        break;

    /* Sequencer */
    case PHTOOLS_MICROPROSS_ERR_SEQUENCER_INVALID_PIN_STATE:
    case PHTOOLS_MICROPROSS_ERR_SEQUENCER_FULL:
    case PHTOOLS_MICROPROSS_ERR_SEQUENCER_NO_SUCH_SEQUENCEID:
    case PHTOOLS_MICROPROSS_ERR_SEQUENCER_NO_MORE_BUFFER_SPACE:
    case PHTOOLS_MICROPROSS_ERR_SEQUENCER_RUNNING:
    case PHTOOLS_MICROPROSS_ERR_SEQUENCER_MAX_SEQUENCE_REACHED:
        pDataParams->wAdditionalInfo = wStatus;
        status = PHTOOLS_MICROPROSS_ERR_MICROPROSS;
        break;
    case PHTOOLS_MICROPROSS_ERR_SEQUENCER_TIMEOUT:
        pDataParams->wAdditionalInfo = wStatus;
        status = PH_ERR_IO_TIMEOUT;
        break;

    case PHTOOLS_MICROPROSS_ERR_I2C_NOT_OPEN:
        status = PH_ERR_USE_CONDITION;
        break;

    case PHTOOLS_MICROPROSS_ERR_I2C_RX_ERROR:
    case PHTOOLS_MICROPROSS_ERR_I2C_TRANSMISSION_ERROR:
        status = PH_ERR_INTERFACE_ERROR;
        break;

    case PHTOOLS_MICROPROSS_ERR_I2C_RX_OVERRUN:
        status = PH_ERR_BUFFER_OVERFLOW;
        break;

    case PHTOOLS_MICROPROSS_ERR_I2C_TIMEOUT:
        status = PH_ERR_IO_TIMEOUT;
        break;

    default:
        pDataParams->wAdditionalInfo = wStatus;
        status = PH_ERR_INTERNAL_ERROR;
        break;
    }

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phToolsMicropross_GetStatus(
                                       phhalHwContact_Micropross_DataParams_t * pDataParams,
                                       uint8_t * pCmd
                                       )
{
    phStatus_t PH_MEMLOC_REM status;
    uint16_t PH_MEMLOC_REM wStatus;

    /* Reset the scratch buffer read position */
    pDataParams->wIntBufferReadPos = 0;

    /* Buffer overflow check */
    if (strlen((char*)pCmd) + 1 > pDataParams->wIntBufferLen)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* check if command in the response match the expected */
    if (strncmp((char*)pCmd,(char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos], strlen((char*)pCmd)))
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    pDataParams->wIntBufferReadPos = pDataParams->wIntBufferReadPos + (uint16_t)strlen((char*)pCmd);

    if (pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos] != ' ')
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    pDataParams->wIntBufferReadPos++;

    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_GetParam_Uint16Hex(pDataParams, &wStatus));

    /* Evaluate status */
    return phToolsMicropross_EvaluateStatus(pDataParams, wStatus);
}

phStatus_t phToolsMicropross_SendCommand(
                                         phhalHwContact_Micropross_DataParams_t * pDataParams
                                         )
{
    phStatus_t PH_MEMLOC_REM status;
    uint16_t   PH_MEMLOC_REM wNumCharPrint;
    uint8_t *pTransmitBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wTransmitBufferSize = 0;
    uint8_t *pReceiveBuffer = NULL;
    uint16_t   PH_MEMLOC_REM wReceiveBufferSize = 0;
    uint16_t   PH_MEMLOC_REM wReceiveBufferLength = 0;
    uint8_t PH_MEMLOC_REM bRetryCnt = 0;

#ifdef _DEBUG
/* FIXXME */
    FILE * fh = NULL;
#endif

    /* 3 byte for CR LF NULL */
    if (3 > pDataParams->wIntBufferSize - pDataParams->wIntBufferLen)
    {
        status = PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /* Add CR LF */
    wNumCharPrint = (uint16_t)sprintf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen], "\r\n");
    pDataParams->wIntBufferLen = pDataParams->wIntBufferLen + wNumCharPrint;

#ifdef _DEBUG
/* FIXXME */
    fh = fopen(".\\mp300CB.log","a");
    fprintf(fh,"%s",pDataParams->pIntBuffer);
#endif

    /* Use temporary buffer */
    pTransmitBuffer = pDataParams->pIntBuffer;
    wTransmitBufferSize = pDataParams->wIntBufferLen;
    pReceiveBuffer = pDataParams->pIntBuffer;
    wReceiveBufferSize = pDataParams->wIntBufferSize - 1; /* To have place for the NULL sign */
    pDataParams->wIntBufferLen = 0;

    do
    {
        /* Perform actual exchange */
        status = phbalReg_Exchange(
            pDataParams->pBalDataParams,
            PH_EXCHANGE_DEFAULT,
            pTransmitBuffer,
            wTransmitBufferSize,
            wReceiveBufferSize,
            pReceiveBuffer,
            &wReceiveBufferLength);

        /* If it is a BAL IO Timeout give it a retry in case we are socket win */
        if ((status & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT && ((phbalReg_SocketWin_DataParams_t *)pDataParams->pBalDataParams)->wId == (PH_COMP_BAL | PHBAL_REG_SOCKETWIN_ID))
        {
            bRetryCnt++;
            /* Perform retry only 2 times */
            if (bRetryCnt <= 2)
            {
#ifdef _DEBUG /* FIXXME */
                fprintf(fh, "BAL Contact timeout -> retry\r\n");
#endif
                phTools_Sleep(5000);

                /* Clear transmit buffer and retry */
                pTransmitBuffer = NULL;
                wTransmitBufferSize = 0;
                continue;
            }

            if (pDataParams->pBalTimeoutHandling != NULL)
            {
                pDataParams->pBalTimeoutHandling(pDataParams->pBalTimeoutHandlingContext, pDataParams->bCouplerNumber);

                /* Flush buffer */
                phbalReg_Exchange(
                    pDataParams->pBalDataParams,
                    PH_EXCHANGE_DEFAULT,
                    NULL,
                    0,
                    wReceiveBufferSize,
                    pReceiveBuffer,
                    &wReceiveBufferLength);
            }
        }

        /* Since we provide the buffer, it's a HAL buffer overflow */
        if ((status & PH_ERR_MASK) == PH_ERR_BUFFER_OVERFLOW)
        {
#ifdef _DEBUG
/* FIXXME */
            fclose(fh);
#endif
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Add NULL Sign */
        pReceiveBuffer[wReceiveBufferLength] = 0;

#ifdef _DEBUG /* FIXXME */
        fprintf(fh,"%s",pReceiveBuffer);
#endif

        /* Check that there is a valid exchange on BAL layer */
        if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS)
        {
#ifdef _DEBUG
/* FIXXME */
            fclose(fh);
#endif
            return status;
        }

        /* Reception of one block done -> clear send buffer and adjused next receive buffer */
        pTransmitBuffer = NULL;
        wTransmitBufferSize = 0;
        pReceiveBuffer += wReceiveBufferLength;
        wReceiveBufferSize -= wReceiveBufferLength;
        pDataParams->wIntBufferLen += wReceiveBufferLength;

        if (pDataParams->wIntBufferLen < 2)
        {
#ifdef _DEBUG
/* FIXXME */
            fclose(fh);
#endif
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }
    }
    while(pDataParams->pIntBuffer[pDataParams->wIntBufferLen-2] != '\r' ||
        (pDataParams->pIntBuffer[pDataParams->wIntBufferLen-1] != '\n' &&
         pDataParams->pIntBuffer[pDataParams->wIntBufferLen-1] != 0)
        );

    pDataParams->wIntBufferLen-=2;
#ifdef _DEBUG
/* FIXXME */
    fclose(fh);
#endif
    return status;
}

phStatus_t phToolsMicropross_GetParam_HexString(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    uint8_t * pData,
    uint16_t wDataSize,
    uint16_t * pwDataLen
    )
{
    uint32_t PH_MEMLOC_REM dwValue;

    *pwDataLen = 0;
    while (pDataParams->wIntBufferReadPos < pDataParams->wIntBufferLen)
    {
        /* Overflow detection */
        if ((*pwDataLen) >= wDataSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        if (sscanf((char *)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos],"%02lX",&dwValue) != 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }

        if (dwValue > 0xFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        pData[(*pwDataLen)++] = (uint8_t)dwValue;
        pDataParams->wIntBufferReadPos += 2;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phToolsMicropross_GetParam_Uint8(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    uint8_t * pData
    )
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t   PH_MEMLOC_REM dwValue;

    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_GetParam_Uint32(pDataParams, &dwValue));

    if (dwValue > 0xFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    *pData = (uint8_t)dwValue;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phToolsMicropross_GetParam_Uint16(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    uint16_t * pData
    )
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t   PH_MEMLOC_REM dwValue;

    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_GetParam_Uint32(pDataParams, &dwValue));

    if (dwValue > 0xFFFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    *pData = (uint16_t)dwValue;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phToolsMicropross_GetParam_Uint32(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    uint32_t * pData
    )
{
    uint16_t PH_MEMLOC_REM wIndex;

    /* find the next space and terminate the string at this position */
    for ( wIndex = pDataParams->wIntBufferReadPos; wIndex < pDataParams->wIntBufferLen; wIndex++ )
    {
        if (pDataParams->pIntBuffer[wIndex] == ' ')
        {
            pDataParams->pIntBuffer[wIndex] = 0;
            break;
        }
    }

    /* scan the return code */
    if (sscanf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos],"%lu",pData ) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* move the read position after the status */
    pDataParams->wIntBufferReadPos = wIndex+1;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phToolsMicropross_GetParam_Int32(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    int32_t * pData
)
{
    uint16_t PH_MEMLOC_REM wIndex;

    /* find the next space and terminate the string at this position */
    for (wIndex = pDataParams->wIntBufferReadPos; wIndex < pDataParams->wIntBufferLen; wIndex++)
    {
        if (pDataParams->pIntBuffer[wIndex] == ' ')
        {
            pDataParams->pIntBuffer[wIndex] = 0;
            break;
        }
    }

    /* scan the return code */
    if (sscanf((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos], "%ld", pData) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* move the read position after the status */
    pDataParams->wIntBufferReadPos = wIndex + 1;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phToolsMicropross_GetParam_Uint8Hex(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    uint8_t * pData
    )
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t   PH_MEMLOC_REM dwValue;

    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_GetParam_Uint32Hex(pDataParams, &dwValue));

    if (dwValue > 0xFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    *pData = (uint8_t)dwValue;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phToolsMicropross_GetParam_Uint16Hex(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    uint16_t * pData
    )
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t   PH_MEMLOC_REM dwValue;

    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_GetParam_Uint32Hex(pDataParams, &dwValue));

    if (dwValue > 0xFFFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    *pData = (uint16_t)dwValue;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phToolsMicropross_GetParam_Uint32Hex(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    uint32_t * pData
    )
{
    uint16_t PH_MEMLOC_REM wIndex;

    /* find the next space and terminate the string at this position */
    for ( wIndex = pDataParams->wIntBufferReadPos; wIndex < pDataParams->wIntBufferLen; wIndex++ )
    {
        if (pDataParams->pIntBuffer[wIndex] == ' ')
        {
            pDataParams->pIntBuffer[wIndex] = 0;
            break;
        }
    }

    /* scan the return code */
    if (sscanf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos],"%lX",pData ) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* move the read position after the status */
    pDataParams->wIntBufferReadPos = wIndex+1;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phToolsMicropross_OpenResource(phhalHwContact_Micropross_DataParams_t * pDataParams)
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t   PH_MEMLOC_REM dwResourceId;
    uint8_t    PH_MEMLOC_REM bOpenResourceId;

    pDataParams->wCouplerResourceId = NULL;

    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u", CONTACT_MICROPROSS_CMD_GET_COUPLER_INFO, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_GetParam_Uint32Hex(pDataParams, &dwResourceId));

    switch ( dwResourceId )
    {
    case 0x81:
        bOpenResourceId = PHHAL_HW_CONTACT_MICROPROSS_RESSOURCE_TC1;
        break;
    case 0x82:
        bOpenResourceId = PHHAL_HW_CONTACT_MICROPROSS_RESSOURCE_TC2;
        break;
    case 0x83:
        bOpenResourceId = PHHAL_HW_CONTACT_MICROPROSS_RESSOURCE_TC3;
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* Try to open the resource not blocking */
    /* If the reader is allready used by the current task it return RET_SEMAPHORE_ALREADY_LOCK F3D */
    /* If the reader is free it allocate it */
    /* If it is allocated by another thread just force */

    status = phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u %u", CONTACT_MICROPROSS_CMD_OPEN_RESSOURCE, bOpenResourceId, pDataParams->bCouplerNumber, CONTACT_MICROPROSS_RESSOURCE_NOT_BLOCKING);

    if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        if (bOpenResourceId != PHHAL_HW_CONTACT_MICROPROSS_RESSOURCE_TC1)
        {
            /* Close the Spy resource again so it can be used from another thread */
            PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SpyClose(pDataParams));
        }
    }
    else if ((status & PH_ERR_MASK) == PH_ERR_INTERNAL_ERROR && pDataParams->wAdditionalInfo == 0x0F3D)
    {
        /* Ok we have a already opened so we just say this open is OK */
        status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }
    else if ((status & PH_ERR_MASK) == PH_ERR_INTERNAL_ERROR && pDataParams->wAdditionalInfo != 0x0F3D)
    {
        /* If the resourse is not open: Open it with force */
        PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u %u", CONTACT_MICROPROSS_CMD_OPEN_RESSOURCE, bOpenResourceId, pDataParams->bCouplerNumber, CONTACT_MICROPROSS_RESSOURCE_OVERRIDE));

        if (bOpenResourceId != PHHAL_HW_CONTACT_MICROPROSS_RESSOURCE_TC1)
        {
            /* Close the Spy resource again so it can be used from another thread */
            PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SpyClose(pDataParams));
        }
    }
    /* Status must be OK now */
    PH_CHECK_SUCCESS(status);

    pDataParams->wCouplerResourceId = bOpenResourceId;
    return phToolsMicropross_GetReaderVersions(pDataParams);
}

phStatus_t phToolsMicropross_SpyClose(phhalHwContact_Micropross_DataParams_t * pDataParams)
{
    return phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u ", CONTACT_MICROPROSS_CMD_SPY_CLOSE, pDataParams->bCouplerNumber);
}

phStatus_t phToolsMicropross_GetResourceID(phhalHwContact_Micropross_DataParams_t * pDataParams, uint8_t bCouplerNumber, uint32_t * pdwResourceID)
{
    phStatus_t PH_MEMLOC_REM status;
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u ", CONTACT_MICROPROSS_CMD_GET_RESOURCE_ID, bCouplerNumber));
    return phToolsMicropross_GetParam_Uint32(pDataParams, pdwResourceID);
}

phStatus_t phToolsMicropross_GetRxBuffer(
                                         phhalHwContact_Micropross_DataParams_t * pDataParams,
                                         uint8_t ** pRxBuffer,
                                         uint16_t * pRxBufferLen,
                                         uint16_t * pRxBufferSize
                                         )
{
    *pRxBuffer = &pDataParams->pRxBuffer[pDataParams->wRxBufStartPos];
    *pRxBufferSize = pDataParams->wRxBufSize - pDataParams->wRxBufStartPos - 2;
    *pRxBufferLen = pDataParams->wRxBufLen;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phToolsMicropross_AddToIntBufferHexString(
                                                     phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                     uint8_t * pData,
                                                     uint16_t wLen
                                                     )
{
    uint16_t PH_MEMLOC_REM wIndex;
    uint16_t PH_MEMLOC_REM wNumCharPrint;

    /* Buffer overflow check */
    if ((wLen*2)+1 > pDataParams->wIntBufferSize-pDataParams->wIntBufferLen)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    if(wLen > 0)
    {
        pDataParams->pIntBuffer[pDataParams->wIntBufferLen] = ' ';
        pDataParams->wIntBufferLen++;
    }

    for ( wIndex = 0; wIndex < wLen; wIndex++ )
    {
        wNumCharPrint = (uint16_t)sprintf( (char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen], "%02X", pData[wIndex]);
        if (wNumCharPrint != 2)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }
        pDataParams->wIntBufferLen += 2;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phToolsMicropross_GetReaderVersions(phhalHwContact_Micropross_DataParams_t * pDataParams)
{
    phStatus_t PH_MEMLOC_REM status;
    uint32_t PH_MEMLOC_REM dwResourceId;
    uint8_t PH_MEMLOC_BUF bDeviceFamily[50];
    uint8_t PH_MEMLOC_BUF bDeviceReferenceWithinFamily[50];
    uint8_t PH_MEMLOC_BUF bMotherboardBootVersion[50];
    uint8_t PH_MEMLOC_BUF bMotherboardSystemVersion[50];
    uint32_t PH_MEMLOC_REM dwHWConfigCoupler;
    uint8_t PH_MEMLOC_BUF bCouplerDriverVersion[50];
    uint8_t PH_MEMLOC_BUF bCouplerSerialNumber[50];
    uint8_t PH_MEMLOC_BUF bCouplerRevision[50];
    uint16_t PH_MEMLOC_REM wDummy;

    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_GetEepromSys(
        pDataParams,
        CONTACT_MICROPROSS_PARAMETER_GET_EEPROM_SYS_EEP_SERIAL_NUMBER,
        pDataParams->bSerialNumber,
        PHHAL_HW_CONTACT_MICROPROSS_SERIAL_NUMBER_SIZE));

    /* Get Version */
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s", CONTACT_MICROPROSS_CMD_GET_VERSION));

    /* Trim leading spaces */
    for ( ; pDataParams->wIntBufferReadPos < pDataParams->wIntBufferLen; pDataParams->wIntBufferReadPos++ )
    {
        if (pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos] != ' ')
        {
            break;
        }
    }
    /* Fifty char are the max size for this string */
    if (strlen((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos]) > 50)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    /* scan the values code */
    if (sscanf((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos], "%s %s %s %s",
        bDeviceFamily, bDeviceReferenceWithinFamily, bMotherboardBootVersion, bMotherboardSystemVersion) != 4)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    if (strlen((char*)bDeviceFamily) >= PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE ||
        strlen((char*)bDeviceReferenceWithinFamily) >= PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE ||
        strlen((char*)bMotherboardBootVersion) >= PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE ||
        strlen((char*)bMotherboardSystemVersion) >= PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }
    strncpy((char*)pDataParams->bDeviceFamily, (char*)bDeviceFamily, PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE); /* MP */
    strncpy((char*)pDataParams->bDeviceReferenceWithinFamily, (char*)bDeviceReferenceWithinFamily, PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE); /* 500 */
    strncpy((char*)pDataParams->bMotherboardBootVersion, (char*)bMotherboardBootVersion, PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE); /* 1.07 */
    strncpy((char*)pDataParams->bMotherboardSystemVersion, (char*)bMotherboardSystemVersion, PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE); /* 5.08 */

    /* Coupler Info & CouplerFPGAVersion */
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u", CONTACT_MICROPROSS_CMD_GET_COUPLER_INFO, pDataParams->bCouplerNumber));
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_GetParam_Uint32Hex(pDataParams, &dwResourceId));

    switch ( dwResourceId )
    {
        case 0x09: strncpy((char*)pDataParams->bCouplerName, "TCL3", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0x0A: strncpy((char*)pDataParams->bCouplerName, "TCL3", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0x41: strncpy((char*)pDataParams->bCouplerName, "C1", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0x42: strncpy((char*)pDataParams->bCouplerName, "C2", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0x43: strncpy((char*)pDataParams->bCouplerName, "C3", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0xE0: strncpy((char*)pDataParams->bCouplerName, "CSWP", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0xE4: strncpy((char*)pDataParams->bCouplerName, "MC1", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0xD3: strncpy((char*)pDataParams->bCouplerName, "MX1", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0x81: strncpy((char*)pDataParams->bCouplerName, "TC1", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0x82: strncpy((char*)pDataParams->bCouplerName, "TC2", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0x83: strncpy((char*)pDataParams->bCouplerName, "TC3", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0x61:
        case 0x62: strncpy((char*)pDataParams->bCouplerName, "CL1", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0x63: strncpy((char*)pDataParams->bCouplerName, "CL2", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0x78:
        case 0x79: strncpy((char*)pDataParams->bCouplerName, "TCL1", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0x7A: strncpy((char*)pDataParams->bCouplerName, "TCL2", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0xC0: strncpy((char*)pDataParams->bCouplerName, "MHC6", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0xE8: strncpy((char*)pDataParams->bCouplerName, "SWPSPY", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        /*case 0xE8: This is also defined in the documentation but then it is used twice */
        case 0xE9:
        case 0xEA: strncpy((char*)pDataParams->bCouplerName, "SC1", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0x70: strncpy((char*)pDataParams->bCouplerName, "MT1", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0xC8: strncpy((char*)pDataParams->bCouplerName, "SCL1", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        case 0xD0: strncpy((char*)pDataParams->bCouplerName, "TPC1", PHHAL_HW_CONTACT_MICROPROSS_COUPLER_NAME_SIZE); break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* Trim leading spaces */
    for ( ; pDataParams->wIntBufferReadPos < pDataParams->wIntBufferLen; pDataParams->wIntBufferReadPos++ )
    {
        if (pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos] != ' ')
        {
            break;
        }
    }
    /* Fifty char are the max size for this string */
    if (strlen((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos]) > 50)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    /* scan the values code */
    if (sscanf((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos], "%lX %s",
        &dwHWConfigCoupler, bCouplerDriverVersion) != 2)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    if (strlen((char*)bCouplerDriverVersion) >= PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE || dwHWConfigCoupler > 0xFFFF)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }
    strncpy((char*)pDataParams->bCouplerDriverVersion, (char*)bCouplerDriverVersion, PHHAL_HW_CONTACT_MICROPROSS_VERSION_STR_SIZE); /* 1.11 */
    pDataParams->wCouplerFPGAVersion = (uint16_t)dwHWConfigCoupler;

    /* bCouplerSerialNumber */
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u",
        CONTACT_MICROPROSS_CMD_COUPLER_EEPROM_SYS, pDataParams->bCouplerNumber, CONTACT_MICROPROSS_PARAMETER_COUPLER_EEPROM_SYS_EEP_COUPLER_IDENT));

    /* Trim leading spaces */
    for ( ; pDataParams->wIntBufferReadPos < pDataParams->wIntBufferLen; pDataParams->wIntBufferReadPos++ )
    {
        if (pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos] != ' ')
        {
            break;
        }
    }
    /* Fifty char are the max size for this string */
    if (strlen((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos]) > 50)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    /* scan the values code */
    if (sscanf((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos], "%s", bCouplerSerialNumber) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    if (strlen((char*)bCouplerSerialNumber) >= PHHAL_HW_CONTACT_MICROPROSS_COUPLER_SERIAL_NUMBER_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }
    strncpy((char*)pDataParams->bCouplerSerialNumber, (char*)bCouplerSerialNumber, PHHAL_HW_CONTACT_MICROPROSS_COUPLER_SERIAL_NUMBER_SIZE); /* 0x21270215 : Module ID=2127, Unique number : 0215 */

    /* bCouplerRevision */
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u %u",
        CONTACT_MICROPROSS_CMD_COUPLER_EEPROM_SYS, pDataParams->bCouplerNumber, CONTACT_MICROPROSS_PARAMETER_COUPLER_EEPROM_SYS_EEP_COUPLER_REVISION));

    /* Trim leading spaces */
    for ( ; pDataParams->wIntBufferReadPos < pDataParams->wIntBufferLen; pDataParams->wIntBufferReadPos++ )
    {
        if (pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos] != ' ')
        {
            break;
        }
    }
    /* Fifty char are the max size for this string */
    if (strlen((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos]) > 50)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    /* scan the values code */
    if (sscanf((char*)&pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos], "%s", bCouplerRevision) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    if (strlen((char*)bCouplerRevision) >= PHHAL_HW_CONTACT_MICROPROSS_COUPLER_REVISION_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }
    strncpy((char*)pDataParams->bCouplerRevision, (char*)bCouplerRevision, PHHAL_HW_CONTACT_MICROPROSS_COUPLER_REVISION_SIZE); /* 0x3045 : Original version E (no upgrade) */

    /* Calibration Date */
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u 10 2", CONTACT_MICROPROSS_CMD_EEPROM_C_READ, pDataParams->bCouplerNumber));

    /* Trim leading spaces */
    for ( ; pDataParams->wIntBufferReadPos < pDataParams->wIntBufferLen; pDataParams->wIntBufferReadPos++ )
    {
        if (pDataParams->pIntBuffer[pDataParams->wIntBufferReadPos] != ' ')
        {
            break;
        }
    }
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_GetParam_HexString(pDataParams, pDataParams->bCouplerCalibrationDate, PHHAL_HW_CONTACT_MICROPROSS_COUPLER_CALC_DATE_SIZE, &wDummy)); /* EECR 0000 0A060B1B  10.06.11 Checksum */
    if (wDummy != PHHAL_HW_CONTACT_MICROPROSS_COUPLER_CALC_DATE_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* There is also an additional information that can be obtained: */
    /* --->Remote command HEHW
       <---HEHW 0000 23 3 6 8000000 */

    return status;
}


phStatus_t phToolsMicropross_GetEepromSys(phhalHwContact_Micropross_DataParams_t * pDataParams, uint8_t bParamId, uint8_t * pbParamValue, uint16_t wParamSize)
{
    phStatus_t PH_MEMLOC_REM status;
    uint16_t PH_MEMLOC_REM dummy = 0;

    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u", CONTACT_MICROPROSS_CMD_GET_EEPROM_SYS, bParamId));
    return phToolsMicropross_GetParam_HexString(pDataParams, pbParamValue, wParamSize, &dummy);
}

phStatus_t phToolsMicropross_SendAndCheckCmd(phhalHwContact_Micropross_DataParams_t * pDataParams, const char *format, ...)
{
    phStatus_t      PH_MEMLOC_REM status;
    uint16_t        PH_MEMLOC_REM nbytes;

    va_list args;
    va_start(args, format);

    /* Reset the scratch buffer len and add string to buffer */
    pDataParams->wIntBufferLen = 0;

    /* Buffer overflow check +1 for the '\0' */
    nbytes = (uint16_t)vsnprintf(NULL, 0, format, args) + 1;
    if(nbytes > pDataParams->wIntBufferSize - pDataParams->wIntBufferLen)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    pDataParams->wIntBufferLen += (uint16_t)vsnprintf((int8_t *)&pDataParams->pIntBuffer[pDataParams->wIntBufferLen],
        pDataParams->wIntBufferSize - pDataParams->wIntBufferLen, format, args);

    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_SendCommand(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phToolsMicropross_GetStatus(pDataParams, va_arg(args, unsigned char *)));

    va_end(args);
    return status;
}

phStatus_t phToolsMicropross_GetLocalIp(uint32_t NetworkAddress, uint32_t SubnetMask, uint8_t *ip)
{
    uint8_t name[255];
    uint32_t dwip;

    DWORD dwRetval;
    struct addrinfo *result = NULL;
    struct addrinfo *ptr = NULL;
    struct sockaddr_in  *sockaddr_ipv4;

    if (gethostname((char *)name, sizeof(name)) == 0)
    {
        dwRetval = getaddrinfo((char *)name, /*(char *)port*/ NULL, NULL/*&hints*/, &result);
        for (ptr = result; ptr != NULL;ptr = ptr->ai_next)
        {
            if (ptr->ai_family == AF_INET)
            {
                sockaddr_ipv4 = (struct sockaddr_in *) ptr->ai_addr;

                dwip = ((((sockaddr_ipv4->sin_addr.S_un.S_addr) & 0xff000000) >> 24) |
                    (((sockaddr_ipv4->sin_addr.S_un.S_addr) & 0x00ff0000) >>  8) |
                    (((sockaddr_ipv4->sin_addr.S_un.S_addr) & 0x0000ff00) <<  8) |
                    (((sockaddr_ipv4->sin_addr.S_un.S_addr) & 0x000000ff) << 24));

                if((dwip & SubnetMask) == NetworkAddress)
                {
                    sprintf_s((char *)ip, 20, "%d.%d.%d.%d",
                        (((sockaddr_ipv4->sin_addr.S_un.S_addr) & 0x000000ff) >> 0),
                        (((sockaddr_ipv4->sin_addr.S_un.S_addr) & 0x0000ff00) >> 8),
                        (((sockaddr_ipv4->sin_addr.S_un.S_addr) & 0x00ff0000) >> 16),
                        (((sockaddr_ipv4->sin_addr.S_un.S_addr) & 0xff000000) >> 24));

                    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
                }
            }
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_HAL);
}

phStatus_t phToolsMicropross_CreateServerSocket(void *GEvtReady, uint8_t *pExpectedClientSocketHostName, SOCKET *ClientSocket, SOCKET *ServerSocket)
{
    struct sockaddr_in server, client;
    int32_t sizeOfSockAddr;
    uint8_t bClientAddressBuffer[20];

    /* Create Socket */
    if((*ServerSocket = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) == INVALID_SOCKET)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    memset(&server, 0, sizeof(struct sockaddr));
    server.sin_family = AF_INET;
    server.sin_addr.s_addr = INADDR_ANY;
    server.sin_port = htons(61641);

    if(bind(*ServerSocket ,(struct sockaddr *)&server , sizeof(server)) == SOCKET_ERROR)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    listen(*ServerSocket, 3);

    if(GEvtReady != NULL)
    {
        /* Thread ready */
        SetEvent (GEvtReady);
    }

    sizeOfSockAddr = sizeof(struct sockaddr_in);
    *ClientSocket = accept(*ServerSocket , (struct sockaddr *)&client, (int *)&sizeOfSockAddr);
    if (*ClientSocket == INVALID_SOCKET)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    sprintf_s((char *)bClientAddressBuffer, sizeof(bClientAddressBuffer), "%d.%d.%d.%d",
        (((client.sin_addr.S_un.S_addr) & 0x000000ff) >> 0),
        (((client.sin_addr.S_un.S_addr) & 0x0000ff00) >> 8),
        (((client.sin_addr.S_un.S_addr) & 0x00ff0000) >> 16),
        (((client.sin_addr.S_un.S_addr) & 0xff000000) >> 24));

    /* Check if client hast correct ip and port as set in LGBD command */
    if(strcmp((const char*)bClientAddressBuffer, (const char*)pExpectedClientSocketHostName))
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phToolsMicropross_CheckSuccessCloseSpy(phhalHwContact_Micropross_DataParams_t * pDataParams, phStatus_t status)
{
    if(status != PH_ERR_SUCCESS)
    {
        if(pDataParams->SpyThreadHandle != NULL)
        {
            TerminateThread(pDataParams->SpyThreadHandle, 0);
        }

        phToolsMicropross_SendAndCheckCmd(pDataParams, "%s %u", CONTACT_MICROPROSS_CMD_SPY_CLOSE, pDataParams->bCouplerNumber);
        return status;
    }
    else
    {
        return status;
    }
}

void phToolsMicropross_SwapHeaderContact(SpyHeaderContact *gHeader)
{
    gHeader->Version = SWAP(gHeader->Version);
    gHeader->TimeBase = SWAP(gHeader->TimeBase);
    gHeader->ClockDivider = SWAPDWORD(gHeader->ClockDivider);
    gHeader->NumberOfEvents = SWAPDWORD(gHeader->NumberOfEvents);
    gHeader->ConfigFlags = SWAP(gHeader->ConfigFlags);
    gHeader->EventsMask1 = SWAP(gHeader->EventsMask1);
    gHeader->EventsMask2 = SWAP(gHeader->EventsMask2);
    gHeader->EventsMask3 = SWAP(gHeader->EventsMask3);
    gHeader->VccThresholdLow = SWAP(gHeader->VccThresholdLow);
    gHeader->VccThresholdHigh = SWAP(gHeader->VccThresholdHigh);
    gHeader->VIL = SWAP(gHeader->VIL);
    gHeader->VIH = SWAP(gHeader->VIH);
    gHeader->FICC = SWAPDWORD(gHeader->FICC);

    if(gHeader->Version >= 3)
    {
        gHeader->OpenLogEventsMask = SWAPDWORD(gHeader->OpenLogEventsMask);
        gHeader->SWPS2Threshold = SWAP(gHeader->SWPS2Threshold);
    }
    if(gHeader->Version >= 5)
    {
        gHeader->BurstDelay = SWAPDWORD(gHeader->BurstDelay);
    }
}

void phToolsMicropross_SwapHeaderContactless(SpyHeaderContactless *gHeader)
{
    gHeader->Version = SWAP(gHeader->Version);
    gHeader->TimeBase = SWAP(gHeader->TimeBase);
    gHeader->NumberOfEvents = SWAPDWORD(gHeader->NumberOfEvents);
    gHeader->ConfigFlags = SWAP(gHeader->ConfigFlags);
    gHeader->EventsMasks = SWAPDWORD(gHeader->EventsMasks);

    if(gHeader->Version >= 3)
    {
        gHeader->FRfMeasure = SWAPDWORD(gHeader->FRfMeasure);
    }
    if(gHeader->Version >= 4)
    {
        gHeader->BurstDelay = SWAPDWORD(gHeader->BurstDelay);
    }
}

void phToolsMicropross_SwapEvents16(uint16_t *buf, uint32_t n)
{
    uint32_t i;
    for (i = 0; i < n; i++)
    {
        buf[i] = SWAP(buf[i]);
    }
}

void phToolsMicropross_SwapEvents32(uint32_t *buf, uint32_t n)
{
    uint32_t i;
    for (i = 0; i < n; i++)
    {
        buf[i] = SWAPDWORD(buf[i]);
    }
}

void phToolsMicropross_SwapEvents64(uint64_t *buf, uint32_t n)
{
    uint32_t i;
    for (i = 0; i < n; i++)
    {
        buf[i] = SWAPDDWORD(buf[i]);
    }
}
