﻿/*
 * Copyright 2017 - 2020, 2022 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal functions of both Software and SamNonX implementation of ICode (R) application layer.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 *
 * History:
 *  CHu: Generated 02. January 2017
 *
 */

#include <ph_RefDefs.h>
#include <phpalSli15693.h>
#include <phalICode.h>

#ifdef NXPBUILD__PH_CRYPTOSYM
#include <phKeyStore.h>
#include <phCryptoSym.h>
#include <phCryptoRng.h>
#endif /* NXPBUILD__PH_CRYPTOSYM */

#include "phalICode_Int.h"

#ifdef NXPBUILD__PHAL_ICODE

phStatus_t phalICode_Int_SetOptionBit(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bUpdateTiming)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wCurFlag = 0;
    uint16_t    PH_MEMLOC_REM wNewFlag = 0;

    /* Check if the Option Byte has a valid value. */
    if(bOption > PHAL_ICODE_OPTION_ON)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Retrieve the flags byte */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_FLAGS,
        &wCurFlag));

    /* Update Option bit with provided information. */
    if(bOption != PHAL_ICODE_OPTION_OFF)
    {
        wNewFlag = wCurFlag | PHPAL_SLI15693_FLAG_OPTION;
    }
    /* Clear option bit */
    else
    {
        wNewFlag = wCurFlag &(uint8_t) ~(uint8_t) PHPAL_SLI15693_FLAG_OPTION;
    }

    /* Update the Option bit in the Flag's byte. */
    if(wNewFlag != wCurFlag)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
            pPalSli15693DataParams,
            PHPAL_SLI15693_CONFIG_FLAGS,
            wNewFlag));
    }

    if(bUpdateTiming == PH_ON)
    {
        /* Set special frame EOF timeout. */
        if(bOption != PHAL_ICODE_OPTION_OFF)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
                pPalSli15693DataParams,
                PHPAL_SLI15693_CONFIG_TIMEOUT_US,
                PHPAL_SLI15693_TIMEOUT_EOF_US));
        }

        /* Set long timeout. */
        else
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
                pPalSli15693DataParams,
                PHPAL_SLI15693_CONFIG_TIMEOUT_US,
                PHPAL_SLI15693_TIMEOUT_LONG_US));
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_WriteAlikeHandling(void * pPalSli15693DataParams, phStatus_t wExchangeStatus)
{
    uint8_t     PH_MEMLOC_REM bDsfid = 0;
    uint8_t     PH_MEMLOC_REM bUid[PHPAL_SLI15693_UID_LENGTH];
    uint8_t     PH_MEMLOC_REM bUidLen = 0;
    uint8_t     PH_MEMLOC_REM aData[1];
    uint16_t    PH_MEMLOC_REM wDataLen = 0;

    switch(wExchangeStatus & PH_ERR_MASK)
    {
        /* Check for protocol error.  */
        case PH_ERR_SUCCESS:
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);

        /* Timeout is correct behavior, send EOF. */
        case PH_ERR_IO_TIMEOUT:
            /* card answers after next EOF -> correct status is timeout */
            return phpalSli15693_SendEof(
                pPalSli15693DataParams,
                PHPAL_SLI15693_EOF_WRITE_ALIKE,
                &bDsfid,
                bUid,
                &bUidLen,
                aData,
                &wDataLen);

        /* Framing errors etc. are ignored and the waiting until EOF sending is continued. */
        case PH_ERR_INTEGRITY_ERROR:
        case PH_ERR_COLLISION_ERROR:
        case PH_ERR_FRAMING_ERROR:
            return phpalSli15693_SendEof(
                pPalSli15693DataParams,
                PHPAL_SLI15693_EOF_WRITE_ALIKE_WITH_WAIT,
                &bDsfid,
                bUid,
                &bUidLen,
                aData,
                &wDataLen);

        /* Directly return all other errors */
        default:
            return wExchangeStatus;
    }
}

phStatus_t phalICode_Int_ComputeErrorCode(void * pPalSli15693DataParams, phStatus_t wStatus)
{
    phStatus_t  PH_MEMLOC_REM wStatusTmp = 0;
    uint16_t    PH_MEMLOC_REM wValue = 0;

    /* Update the temporary status variable. */
    wStatusTmp = wStatus;

    /*  Compute the custom error codes in case PAL returns an error. */
    if((wStatus & PH_ERR_MASK) == PHPAL_SLI15693_ERR_ISO15693)
    {
        /* Get the error code from additional info. */
        PH_CHECK_SUCCESS_FCT(wStatusTmp, phpalSli15693_GetConfig(
            pPalSli15693DataParams,
            PHPAL_SLI15693_CONFIG_ADD_INFO,
            &wValue));

        /* Compute the custom code. */
        if((wValue >= 0xA0U) && (wValue <= 0xDFU))
        {
            /* Error mapping for error codes returned by Custom commands. */
            wStatusTmp = PHAL_ICODE_ERR_CUSTOM_COMMANDS_ERROR;
        }
        else
        {
            switch(wValue)
            {
                case PHAL_ICODE_RESP_ERR_COMMAND_NOT_SUPPORTED: wStatusTmp = PHAL_ICODE_ERR_COMMAND_NOT_SUPPORTED;        break;
                case PHAL_ICODE_RESP_ERR_COMMAND_NOT_RECOGNIZED: wStatusTmp = PHAL_ICODE_ERR_COMMAND_NOT_RECOGNIZED;       break;
                case PHAL_ICODE_RESP_ERR_COMMAND_OPTION_NOT_SUPPORTED: wStatusTmp = PHAL_ICODE_ERR_COMMAND_OPTION_NOT_SUPPORTED; break;
                case PHAL_ICODE_RESP_ERR_NO_INFORMATION: wStatusTmp = PHAL_ICODE_ERR_NO_INFORMATION;               break;
                case PHAL_ICODE_RESP_ERR_BLOCK_NOT_AVAILABLE: wStatusTmp = PHAL_ICODE_ERR_BLOCK_NOT_AVAILABLE;          break;
                case PHAL_ICODE_RESP_ERR_BLOCK_LOCKED: wStatusTmp = PHAL_ICODE_ERR_BLOCK_LOCKED;                 break;
                case PHAL_ICODE_RESP_ERR_CONTENT_CHANGE_FAILURE: wStatusTmp = PHAL_ICODE_ERR_CONTENT_CHANGE_FAILURE;       break;
                case PHAL_ICODE_RESP_ERR_BLOCK_PROGRAMMING_FAILURE: wStatusTmp = PHAL_ICODE_ERR_BLOCK_PROGRAMMING_FAILURE;    break;
                case PHAL_ICODE_RESP_ERR_BLOCK_NOT_LOCKED: wStatusTmp = PHAL_ICODE_ERR_BLOCK_NOT_LOCKED;             break;
                case PHAL_ICODE_RESP_ERR_BLOCK_PROTECTED: wStatusTmp = PHAL_ICODE_ERR_BLOCK_PROTECTED;              break;
                case PHAL_ICODE_RESP_ERR_GENERIC_CRYPTO_ERROR: wStatusTmp = PHAL_ICODE_ERR_GENERIC_CRYPTO_ERROR;         break;

                default: wStatusTmp = PH_ERR_PROTOCOL_ERROR; break;
            }
        }

        /* Merge the status code with component code. */
        wStatusTmp = PH_ADD_COMPCODE(wStatusTmp, PH_COMP_AL_ICODE);
    }

    return wStatusTmp;
}

phStatus_t phalICode_Int_GetFlags(void * pPalSli15693DataParams, uint16_t * pFlags)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    switch(PH_GET_COMPID(pPalSli15693DataParams))
    {
#ifdef NXPBUILD__PHPAL_SLI15693_SW
        case PHPAL_SLI15693_SW_ID:
            *pFlags = ((phpalSli15693_Sw_DataParams_t *) pPalSli15693DataParams)->bFlags;
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_SW */

#ifdef NXPBUILD__PHPAL_SLI15693_RD710
        case PHPAL_SLI15693_RD710_ID:
            *pFlags = ((phpalSli15693_Rd710_DataParams_t *) pPalSli15693DataParams)->bFlags;
            break;
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_ICODE);
            break;
    }

    return wStatus;
}

phStatus_t phalICode_Int_SetChallengeType(void * pDataParams, uint8_t bChallengeType)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_ICODE_SW
        case PHAL_ICODE_SW_ID:
            ((phalICode_Sw_DataParams_t *) pDataParams)->bChallengeType = bChallengeType;
            break;
#endif /* PHAL_ICODE_SW_ID */

#ifdef NXPBUILD__PHAL_ICODE_SAM_NONX
        case PHAL_ICODE_SAMAV3_NONX_ID:
            ((phalICode_SamAV3_NonX_DataParams_t *) pDataParams)->bChallengeType = bChallengeType;
            break;
#endif /* NXPBUILD__PHAL_ICODE_SAM_NONX */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_ICODE);
            break;
    }

    return wStatus;
}

phStatus_t phalICode_Int_GetChallengeType(void * pDataParams, uint8_t * pChallengeType)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_ICODE_SW
        case PHAL_ICODE_SW_ID:
            *pChallengeType = ((phalICode_Sw_DataParams_t *) pDataParams)->bChallengeType;
            break;
#endif /* PHAL_ICODE_SW_ID */

#ifdef NXPBUILD__PHAL_ICODE_SAM_NONX
        case PHAL_ICODE_SAMAV3_NONX_ID:
            *pChallengeType = ((phalICode_SamAV3_NonX_DataParams_t *) pDataParams)->bChallengeType;
            break;
#endif /* NXPBUILD__PHAL_ICODE_SAM_NONX */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_ICODE);
            break;
    }

    return wStatus;
}

phStatus_t phalICode_Int_SetFast_DataRate(void * pPalSli15693DataParams, uint16_t *pDataRate)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wFlags = 0;
    uint16_t    PH_MEMLOC_REM wRxDataRate_New = 0;

    /* Get the existing Rx DataRate */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_RXDATARATE,
        pDataRate));

    /* Get the Flag information based on component */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_GetFlags(pPalSli15693DataParams, &wFlags));

    /* Configuring the baud rate based on b2 and b8 value */
    if((wFlags & PHAL_ICODE_FLAG_DATA_RATE) > 0U)
    {
        wRxDataRate_New = (uint16_t) ((wFlags & PHAL_ICODE_FLAG_FAST_DATA_RATE) ? PHPAL_SLI15693_106KBPS_DATARATE :
            PHPAL_SLI15693_26KBPS_DATARATE);
    }
    else
    {
        wRxDataRate_New = (uint16_t) ((wFlags & PHAL_ICODE_FLAG_FAST_DATA_RATE) ? PHPAL_SLI15693_53KBPS_DATARATE :
            PHPAL_SLI15693_212KBPS_DATARATE);
    }

    /* Update the Rx BaudRate to new value. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_RXDATARATE,
        wRxDataRate_New));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ReadBlocks(void * pPalSli15693DataParams, uint8_t bCmdCode, uint8_t bEnableBuffering, uint8_t bUpdateTiming,
    uint8_t bOption, uint16_t wBlockNo, uint16_t wNumBlocks, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint16_t    PH_MEMLOC_REM wRxDataRate = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[5];
    uint8_t     PH_MEMLOC_REM bcmdLen = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;

    /* This flag enables buffering of response data received from ICode tags. This is purely for applications that run on desktop.
    * This flag by default be placed in preprocessor sections. Additionally the application has to enable a flag bEnableBuffering
    * to make this chaining work.
    *
    * To disable this flag remove this macro from ProjectProperties-> C/C++ -> Preprocessor -> Preprocessor Definitions for both
    * DEBUG and RELEASE configurations.
    */
#ifdef PHAL_ICODE_ENABLE_CHAINING
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint16_t    PH_MEMLOC_REM wCurrBlocksToRead = 0;
    uint16_t    PH_MEMLOC_REM wCurrBlockNo = 0;
    uint16_t    PH_MEMLOC_REM wMaxNoBlocks = 0;
    uint8_t     PH_MEMLOC_REM bFinish = PH_OFF;
#endif /* PHAL_ICODE_ENABLE_CHAINING */

    /* Set or clear the flags option bit indicated by bOption. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Configuring the Rx BaudRate based on b2 and b8 value */
    if(bUpdateTiming == PH_ON)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetFast_DataRate(pPalSli15693DataParams, &wRxDataRate));
    }

    /* Reset command buffer and its length variable. */
    bcmdLen = 0U;
    (void) memset(aCmdBuff, 0x00U, (size_t) sizeof(aCmdBuff));

    /* Frame the initial command. */
    aCmdBuff[bcmdLen++] = bCmdCode;

#ifdef PHAL_ICODE_ENABLE_CHAINING
    /* Buffer the response data if Buffering flag is set. */
    if(0U != bEnableBuffering)
    {
        /* Update the maximum number of blocks with respect to Option flag setting. The value for the blocks is fixed to 60 and 40 to avoid multiple
         * handling of different data in response. RD70x can respond with more amount of data but RD710 cannot. So fixing the blocks count to a lower
         * value.
         */
        wMaxNoBlocks = (uint8_t) ((bOption != 0U) ? PHAL_ICODE_MAX_BLOCKS_CM1_OPTION_FLAG_SET : PHAL_ICODE_MAX_BLOCKS_CM1_OPTION_FLAG_NOT_SET);

        /* Blocks to read. */
        wCurrBlocksToRead = wMaxNoBlocks;

        /* Update the number of blocks to read if its less than the internal required one. */
        if(wNumBlocks < wMaxNoBlocks)
        {
            wCurrBlocksToRead = wNumBlocks;
            bFinish = PH_ON;
        }

        /* Read the blocks. */
        do
        {
            /* Frame command information. */
            aCmdBuff[bcmdLen++] = (uint8_t) ((wCurrBlockNo + wBlockNo) & 0x00FFU);

            /* Add the next byte of block number for extended commands. */
            if((bCmdCode == PHAL_ICODE_CMD_EXTENDED_READ_MULTIPLE_BLOCKS) || (bCmdCode == PHAL_ICODE_CMD_EXTENDED_GET_MULTIPLE_BLOCK_SECURITY_STATUS) ||
                (bCmdCode == PHAL_ICODE_CMD_EXTENDED_FAST_READ_MULTIPLE_BLOCKS))
            {
                aCmdBuff[bcmdLen++] = (uint8_t) (((wCurrBlockNo + wBlockNo) & 0xFF00U) >> 8U);
            }

            /* Adjust number of blocks. Adjustment is made because the User or the application will pass
            * the number of blocks starting from 1 to N. But as per Iso15693 specification the number
            * of blocks ranges from 0 - (N - 1).
            */
            --wCurrBlocksToRead;

            /* Add number of blocks. */
            aCmdBuff[bcmdLen++] = (uint8_t) (wCurrBlocksToRead & 0x00FFU);

            /* Add the next byte of number of blocks for extended commands. */
            if((bCmdCode == PHAL_ICODE_CMD_EXTENDED_READ_MULTIPLE_BLOCKS) || (bCmdCode == PHAL_ICODE_CMD_EXTENDED_GET_MULTIPLE_BLOCK_SECURITY_STATUS) ||
                (bCmdCode == PHAL_ICODE_CMD_EXTENDED_FAST_READ_MULTIPLE_BLOCKS))
            {
                aCmdBuff[bcmdLen++] = (uint8_t) ((wCurrBlocksToRead & 0xFF00U) >> 8U);
            }

            /* Exchange the command information to PAL layer. */
            wStatus = phpalSli15693_Exchange(
                pPalSli15693DataParams,
                PH_EXCHANGE_DEFAULT,
                aCmdBuff,
                bcmdLen,
                &pResponse,
                &wRespLen);

            /* Compute the status code. */
            wStatus = phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus);
            if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            {
                /* Copy the received data to internal buffer. */
                if(wRespLen != 0U)
                {
                    (void) memcpy(&pData[*pDataLen], pResponse, wRespLen);
                }
                *pDataLen += wRespLen;

                /* Update the variables to read the remaining data. */
                wCurrBlockNo += wMaxNoBlocks;

                /* Update the Current blocks to read. */
                wCurrBlocksToRead = wMaxNoBlocks;

                /* Reset the command buffer length. */
                bcmdLen = 1;

                /* Set the remaining blocks to read. */
                if((wNumBlocks - wCurrBlockNo) < wMaxNoBlocks)
                {
                    wCurrBlocksToRead = (uint16_t) (wNumBlocks - wCurrBlockNo);
                }

                /* Set the flag to finish the loop. */
                switch(bCmdCode)
                {
                    case PHAL_ICODE_CMD_READ_MULTIPLE_BLOCKS:
                    case PHAL_ICODE_CMD_FAST_READ_MULTIPLE_BLOCKS:
                    case PHAL_ICODE_CMD_EXTENDED_READ_MULTIPLE_BLOCKS:
                    case PHAL_ICODE_CMD_EXTENDED_FAST_READ_MULTIPLE_BLOCKS:
                    case PHAL_ICODE_CMD_READ_SRAM:
                        if((!(bOption > 0U) && ((wNumBlocks * 4U) == *pDataLen)) || ((bOption > 0U) && (((wNumBlocks * 4U) + wNumBlocks) == *pDataLen)))
                        {
                            bFinish = PH_ON;
                        }
                        break;

                    case PHAL_ICODE_CMD_GET_MULTIPLE_BLOCK_SECURITY_STATUS:
                    case PHAL_ICODE_CMD_EXTENDED_GET_MULTIPLE_BLOCK_SECURITY_STATUS:
                    case PHAL_ICODE_CMD_GET_MULTIPLE_BLOCK_PROTECTION_STATUS:
                        if(wNumBlocks == *pDataLen)
                        {
                            bFinish = PH_ON;
                        }
                        break;
                }
            }
            else
            {
                bFinish = PH_ON;
            }
        } while(0U == bFinish);
    }
    else
#endif /* PHAL_ICODE_ENABLE_CHAINING */
    {
        /* To avoid build failures. */
        bEnableBuffering = 0U;

        /* Adjust number of blocks. Adjustment is made because the User or the application will pass
        * the number of blocks starting from 1 to N. But as per Iso15693 specification the number
        * of blocks ranges from 0 - (N - 1).
        */
        --wNumBlocks;

        /* Frame ReadMultipleBlock command information. */
        aCmdBuff[bcmdLen++] = (uint8_t) (wBlockNo & 0x00FFU);

        /* Add the next byte of block number for extended commands. */
        if((bCmdCode == PHAL_ICODE_CMD_EXTENDED_READ_MULTIPLE_BLOCKS) || (bCmdCode == PHAL_ICODE_CMD_EXTENDED_GET_MULTIPLE_BLOCK_SECURITY_STATUS) ||
            (bCmdCode == PHAL_ICODE_CMD_EXTENDED_FAST_READ_MULTIPLE_BLOCKS))
        {
            aCmdBuff[bcmdLen++] = (uint8_t) ((wBlockNo & 0xFF00U) >> 8U);
        }

        /* Add number of blocks. */
        aCmdBuff[bcmdLen++] = (uint8_t) (wNumBlocks & 0x00FFU);

        /* Add the next byte of number of blocks for extended commands. */
        if((bCmdCode == PHAL_ICODE_CMD_EXTENDED_READ_MULTIPLE_BLOCKS) || (bCmdCode == PHAL_ICODE_CMD_EXTENDED_GET_MULTIPLE_BLOCK_SECURITY_STATUS) ||
            (bCmdCode == PHAL_ICODE_CMD_EXTENDED_FAST_READ_MULTIPLE_BLOCKS))
        {
            aCmdBuff[bcmdLen++] = (uint8_t) ((wNumBlocks & 0xFF00U) >> 8U);
        }

        /* Exchange the command information to PAL layer. */
        wStatus = phpalSli15693_Exchange(
            pPalSli15693DataParams,
            PH_EXCHANGE_DEFAULT,
            aCmdBuff,
            bcmdLen,
            &pResponse,
            pDataLen);

        /* Copy the received data to internal buffer. */
        (void) memcpy(pData, pResponse, *pDataLen);

        /* Compute the status code. */
        wStatus = phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus);

        /* Revert back to Old BaudRate. */
        if(bUpdateTiming == PH_ON)
        {
            PH_CHECK_SUCCESS_FCT(wStatus1, phpalSli15693_SetConfig(
                pPalSli15693DataParams,
                PHPAL_SLI15693_CONFIG_RXDATARATE,
                wRxDataRate));
        }
    }

    return PH_ADD_COMPCODE(wStatus, PH_COMP_AL_ICODE);
}

void phalICode_Int_Reverse(uint8_t * pData, uint16_t wLength)
{
    uint8_t PH_MEMLOC_REM bTemp = 0;

    uint8_t bLSB = 0U;
    uint8_t bMSB = (uint8_t) (wLength - 1U);

    while(bLSB <= bMSB)
    {
        bTemp = pData[bLSB];
        pData[bLSB] = pData[bMSB];
        pData[bMSB] = bTemp;

        bLSB++;
        bMSB--;
    }
}





phStatus_t phalICode_Int_ReadSingleBlock(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bBlockNo, uint8_t ** ppData,
    uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wRetries = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];

    /* Retrieve max retry count */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT,
        &wRetries));

    wRetries += 1U;

    /* Frame ReadSingleBlock command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_READ_SINGLE_BLOCK;
    aCmdBuff[1U] = bBlockNo;

    do
    {
        /* Set or clear the flags option bit indicated by bOption. */
        wStatus = phalICode_Int_SetOptionBit(
            pPalSli15693DataParams,
            bOption,
            PH_OFF);

        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Set short + Tolerance(Delta) Timeout(50ms). */
            wStatus = phpalSli15693_SetConfig(
                pPalSli15693DataParams,
                PHPAL_SLI15693_CONFIG_TIMEOUT_US,
                PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_TIMEOUT_DELTA_US);

            if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            {
                /* Exchange the command information to PAL layer. */
                wStatus = phpalSli15693_Exchange(
                    pPalSli15693DataParams,
                    PH_EXCHANGE_DEFAULT,
                    aCmdBuff,
                    2U,
                    ppData,
                    pDataLen);

                /* Compute the status code. */
                wStatus = phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus);
            }
        }

        wRetries--;

    } while((wStatus != PH_ERR_SUCCESS) && (wRetries != 0U));

    return wStatus;
}

phStatus_t phalICode_Int_WriteSingleBlock(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bBlockNo, uint8_t * pData,
    uint8_t bDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint16_t    PH_MEMLOC_REM wRetries = 0;

    /* Retrieve max retry count */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT,
        &wRetries));

    wRetries += 1U;

    /* Frame WriteSingleBlock command. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_WRITE_SINGLE_BLOCK;
    aCmdBuff[1U] = bBlockNo;

    do
    {
        /* Set or clear the flags option bit indicated by bOption. */
        wStatus = phalICode_Int_SetOptionBit(
            pPalSli15693DataParams,
            bOption,
            PH_ON);

        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Buffer the command information to exchange buffer. */
            PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Exchange(
                pPalSli15693DataParams,
                PH_EXCHANGE_BUFFER_FIRST,
                aCmdBuff,
                2U,
                NULL,
                NULL));

            /* Buffer the data to exchange buffer and exchange the buffered information to PAL layer. */
            wStatus = phpalSli15693_Exchange(
                pPalSli15693DataParams,
                PH_EXCHANGE_BUFFER_LAST,
                pData,
                bDataLen,
                &pResponse,
                &wRespLen);

            /* Write-alike handling */
            if(0U != bOption)
            {
                wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
            }

            /* Compute the status code. */
            wStatus = phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus);
        }

        wRetries--;

    } while((wStatus != PH_ERR_SUCCESS) && (wRetries != 0U));

    return wStatus;
}

phStatus_t phalICode_Int_LockBlock(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bBlockNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint16_t    PH_MEMLOC_REM wRetries = 0;

    /* Retrieve max retry count */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT,
        &wRetries));

    wRetries += 1U;

    /* Frame LockBlock command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_LOCK_BLOCK;
    aCmdBuff[1U] = bBlockNo;

    do
    {
        /* Set or clear the flags option bit indicated by bOption. */
        wStatus = phalICode_Int_SetOptionBit(
            pPalSli15693DataParams,
            bOption,
            PH_ON);

        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Exchange the command information to PAL layer. */
            wStatus = phpalSli15693_Exchange(
                pPalSli15693DataParams,
                PH_EXCHANGE_DEFAULT,
                aCmdBuff,
                2U,
                &pResponse,
                &wRespLen);

            /* Write-alike handling */
            if(0U != bOption)
            {
                wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
            }

            /* Compute the status code. */
            wStatus = phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus);
        }

        wRetries--;

    } while((wStatus != PH_ERR_SUCCESS) && (wRetries != 0U));

    return wStatus;
}

phStatus_t phalICode_Int_ReadMultipleBlocks(void * pPalSli15693DataParams, uint8_t bEnableBuffering, uint8_t bOption, uint8_t bBlockNo,
    uint8_t bNumBlocks, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wRetries = 0;

    /* Number of bNumBlocks can't be zero */
    if(bNumBlocks == 0U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Check number of blocks doesn't exceed 256. */
    if(((uint16_t) bBlockNo + bNumBlocks) > PHAL_ICODE_MAX_BLOCKS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Retrieve max retry count */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT,
        &wRetries));

    wRetries += 1U;

    do
    {
        /* Set short + Tolerance(Delta) Timeout(50ms). */
        wStatus = phpalSli15693_SetConfig(
            pPalSli15693DataParams,
            PHPAL_SLI15693_CONFIG_TIMEOUT_US,
            PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_TIMEOUT_DELTA_US);

        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Read the blocks. */
            wStatus = phalICode_Int_ReadBlocks(
                pPalSli15693DataParams,
                PHAL_ICODE_CMD_READ_MULTIPLE_BLOCKS,
                bEnableBuffering,
                PH_OFF,
                bOption,
                bBlockNo,
                bNumBlocks,
                pData,
                pDataLen);
        }

        wRetries--;

    } while((wStatus != PH_ERR_SUCCESS) && (wRetries != 0U));

    return wStatus;
}

phStatus_t phalICode_Int_WriteAFI(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bAfi)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuf[2];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame WriteAFI command information. */
    aCmdBuf[0U] = PHAL_ICODE_CMD_WRITE_AFI;
    aCmdBuf[1U] = bAfi;

    /* Set or clear the flags option bit indicated by bOption. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuf,
        2U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_LockAFI(void * pPalSli15693DataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame LockAFI command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_LOCK_AFI;

    /* Set or clear the flags option bit indicated by bOption. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_WriteDSFID(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bDsfid)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuf[2];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame WriteDSFID command information. */
    aCmdBuf[0U] = PHAL_ICODE_CMD_WRITE_DSFID;
    aCmdBuf[1U] = bDsfid;

    /* Set or clear the flags option bit indicated by bOption. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuf,
        2U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_LockDSFID(void * pPalSli15693DataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame LockDSFID command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_LOCK_DSFID;

    /* Set or clear the flags option bit indicated by bOption. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_GetSystemInformation(void * pPalSli15693DataParams, uint8_t ** ppSystemInfo, uint16_t * pSystemInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];

    /* Frame GetSystemInformation command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_GET_SYSTEM_INFORMATION;

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Set short timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        ppSystemInfo,
        pSystemInfoLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_GetMultipleBlockSecurityStatus(void * pPalSli15693DataParams, uint8_t bEnableBuffering, uint8_t bBlockNo,
    uint8_t bNoOfBlocks, uint8_t * pStatus, uint16_t * pStatusLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Number of bNoOfBlocks can't be zero */
    if(bNoOfBlocks == 0U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Check number of blocks doesn't exceed 256 */
    if(((uint16_t) bBlockNo + bNoOfBlocks) > PHAL_ICODE_MAX_BLOCKS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Set short timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Read the blocks. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadBlocks(
        pPalSli15693DataParams,
        PHAL_ICODE_CMD_GET_MULTIPLE_BLOCK_SECURITY_STATUS,
        bEnableBuffering,
        PH_OFF,
        PHAL_ICODE_OPTION_OFF,
        bBlockNo,
        bNoOfBlocks,
        pStatus,
        pStatusLen));

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_FastReadMultipleBlocks(void * pPalSli15693DataParams, uint8_t bEnableBuffering, uint8_t bOption,
    uint8_t bBlockNo, uint8_t bNumBlocks, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Number of bNumBlocks can't be zero */
    if(bNumBlocks == 0U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Check number of blocks doesn't exceed 255. */
    if(((uint16_t) bBlockNo + bNumBlocks) > PHAL_ICODE_MAX_BLOCKS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Set short timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Read the blocks. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadBlocks(
        pPalSli15693DataParams,
        PHAL_ICODE_CMD_FAST_READ_MULTIPLE_BLOCKS,
        bEnableBuffering,
        PH_ON,
        bOption,
        bBlockNo,
        bNumBlocks,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ExtendedReadSingleBlock(void * pPalSli15693DataParams, uint8_t bOption, uint16_t wBlockNo, uint8_t ** ppData,
    uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[3];
    uint16_t    PH_MEMLOC_REM wRetries = 0;

    /* Frame ExtendedReadSingleBlock command information. */
    aCmdBuff[0] = PHAL_ICODE_CMD_EXTENDED_READ_SINGLE_BLOCK;
    aCmdBuff[1] = (uint8_t) (wBlockNo & 0x00FFU);
    aCmdBuff[2] = (uint8_t) ((wBlockNo & 0xFF00U) >> 8U);

    /* Retrieve max retry count */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT,
        &wRetries));

    wRetries += 1U;

    do
    {
        /* Set or clear the flags option bit indicated by bOption. */
        wStatus = phalICode_Int_SetOptionBit(
            pPalSli15693DataParams,
            bOption,
            PH_OFF);

        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Set short + Tolerance(Delta) Timeout(50ms). */
            wStatus = phpalSli15693_SetConfig(
                pPalSli15693DataParams,
                PHPAL_SLI15693_CONFIG_TIMEOUT_US,
                PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_TIMEOUT_DELTA_US);

            if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            {
                /* Exchange the command information to PAL layer. */
                wStatus = phpalSli15693_Exchange(
                    pPalSli15693DataParams,
                    PH_EXCHANGE_DEFAULT,
                    aCmdBuff,
                    3U,
                    ppData,
                    pDataLen);

                /* Compute the status code. */
                wStatus = phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus);
            }
        }

        wRetries--;

    } while((wStatus != PH_ERR_SUCCESS) && (wRetries != 0U));

    return wStatus;
}

phStatus_t phalICode_Int_ExtendedWriteSingleBlock(void * pPalSli15693DataParams, uint8_t bOption, uint16_t wBlockNo, uint8_t * pData,
    uint8_t bDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[3];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint16_t    PH_MEMLOC_REM wRetries = 0;

    /* Frame ExtendedWriteSingleBlock command. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_EXTENDED_WRITE_SINGLE_BLOCK;
    aCmdBuff[1U] = (uint8_t) (wBlockNo & 0x00FFU);
    aCmdBuff[2U] = (uint8_t) ((wBlockNo & 0xFF00U) >> 8U);

    /* Retrieve max retry count */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT,
        &wRetries));

    wRetries += 1U;

    do
    {
        /* Set or clear the flags option bit indicated by bOption. */
        wStatus = phalICode_Int_SetOptionBit(
            pPalSli15693DataParams,
            bOption,
            PH_ON);

        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Buffer the command information to exchange buffer. */
            PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Exchange(
                pPalSli15693DataParams,
                PH_EXCHANGE_BUFFER_FIRST,
                aCmdBuff,
                3U,
                NULL,
                NULL));

            /* Buffer the data to exchange buffer and exchange the buffered information to PAL layer. */
            wStatus = phpalSli15693_Exchange(
                pPalSli15693DataParams,
                PH_EXCHANGE_BUFFER_LAST,
                pData,
                bDataLen,
                &pResponse,
                &wRespLen);

            /* Write-alike handling */
            if(0U != bOption)
            {
                wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
            }

            /* Compute the status code. */
            wStatus = phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus);
        }

        wRetries--;

    } while((wStatus != PH_ERR_SUCCESS) && (wRetries != 0U));

    return wStatus;
}

phStatus_t phalICode_Int_ExtendedLockBlock(void * pPalSli15693DataParams, uint8_t bOption, uint16_t wBlockNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[3];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint16_t    PH_MEMLOC_REM wRetries = 0;

    /* Frame ExtendedLockBlock command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_EXTENDED_LOCK_BLOCK;
    aCmdBuff[1U] = (uint8_t) (wBlockNo & 0x00FFU);
    aCmdBuff[2U] = (uint8_t) ((wBlockNo & 0xFF00U) >> 8U);

    /* Retrieve max retry count */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT,
        &wRetries));

    wRetries += 1U;

    do
    {
        /* Set or clear the flags option bit indicated by bOption. */
        wStatus = phalICode_Int_SetOptionBit(
            pPalSli15693DataParams,
            bOption,
            PH_ON);

        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Exchange the command information to PAL layer. */
            wStatus = phpalSli15693_Exchange(
                pPalSli15693DataParams,
                PH_EXCHANGE_DEFAULT,
                aCmdBuff,
                3U,
                &pResponse,
                &wRespLen);

            /* Write-alike handling */
            if(0U != bOption)
            {
                wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
            }

            /* Compute the status code. */
            wStatus = phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus);
        }

        wRetries--;

    } while((wStatus != PH_ERR_SUCCESS) && (wRetries != 0U));

    return wStatus;
}

phStatus_t phalICode_Int_ExtendedReadMultipleBlocks(void * pPalSli15693DataParams, uint8_t bEnableBuffering, uint8_t bOption,
    uint16_t wBlockNo, uint16_t wNumBlocks, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wRetries = 0;

    /* Number of bNumBlocks can't be zero */
    if(wNumBlocks == 0U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Check number of blocks doesn't exceed 512. */
    if(((uint16_t) wBlockNo + wNumBlocks) > PHAL_ICODE_MAX_BLOCKS_EXTENDED)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Retrieve max retry count */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT,
        &wRetries));

    wRetries += 1U;

    do
    {
        /* Set short + Tolerance(Delta) Timeout(50ms). */
        wStatus = phpalSli15693_SetConfig(
            pPalSli15693DataParams,
            PHPAL_SLI15693_CONFIG_TIMEOUT_US,
            PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_TIMEOUT_DELTA_US);

        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Read the blocks. */
            wStatus = phalICode_Int_ReadBlocks(
                pPalSli15693DataParams,
                PHAL_ICODE_CMD_EXTENDED_READ_MULTIPLE_BLOCKS,
                bEnableBuffering,
                PH_OFF,
                bOption,
                wBlockNo,
                wNumBlocks,
                pData,
                pDataLen);
        }
        wRetries--;

    } while((wStatus != PH_ERR_SUCCESS) && (wRetries != 0U));

    return wStatus;
}

#ifdef NXPBUILD__PHAL_ICODE_INTERNAL
phStatus_t phalICode_Int_WriteCertificate(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bCertNo, uint16_t wWordPtr,
    uint8_t * pCertificate, uint8_t bCertLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wRetries = 0;
    uint16_t    PH_MEMLOC_REM wRetries_Bkp = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint16_t    PH_MEMLOC_REM wStartAddr = 0;
    uint16_t    PH_MEMLOC_REM wEndAddr = 0;
    uint16_t    PH_MEMLOC_REM wIndex = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint8_t     PH_MEMLOC_REM bOffset = 0;

    uint8_t     PH_MEMLOC_REM aCmdBuff[23];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;

    /* Check if Certificate length is multiple of 4 bytes. */
    if((bCertLen % PHAL_ICODE_BLOCK_SIZE) != 0U)
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_SIZE, PH_COMP_AL_ICODE);

    /* Retrieve max retry count */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT,
        &wRetries_Bkp));

    wRetries_Bkp += 1U;

    /* Get Start and Stop address. */
    wStartAddr = wWordPtr;
    wEndAddr = (uint16_t) (bCertLen / PHAL_ICODE_BLOCK_SIZE);

    /* Iterate through the WordPtrs */
    for(wIndex = wStartAddr; wIndex < wEndAddr; wIndex++)
    {
        /* Reset Retries */
        wRetries = wRetries_Bkp;

        /* Frame WriteSingleBlock command. */
        aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_WRITE_CERTIFICATE;
        aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_DH;
        aCmdBuff[bCmdLen++] = PHAL_ICODE_AUTH_PARAM_TAM1_0;
        aCmdBuff[bCmdLen++] = bCertNo;

        (void) memcpy(&aCmdBuff[bCmdLen], (uint8_t *) &wIndex, 2U);
        bCmdLen += 2;

        do
        {
            /* Set or clear the flags option bit indicated by bOption. */
            wStatus = phalICode_Int_SetOptionBit(
                pPalSli15693DataParams,
                bOption,
                PH_ON);

            if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            {
                /* Buffer the command information to exchange buffer. */
                PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Exchange(
                    pPalSli15693DataParams,
                    PH_EXCHANGE_BUFFER_FIRST,
                    aCmdBuff,
                    bCmdLen,
                    NULL,
                    NULL));

                /* Buffer the data to exchange buffer and exchange the buffered information to PAL layer. */
                wStatus = phpalSli15693_Exchange(
                    pPalSli15693DataParams,
                    PH_EXCHANGE_BUFFER_LAST,
                    &pCertificate[bOffset],
                    PHAL_ICODE_BLOCK_SIZE,
                    &pResponse,
                    &wRespLen);

                /* Write-alike handling */
                if(0U != bOption)
                {
                    wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
                }

                /* Compute the status code. */
                wStatus = phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus);
            }

            wRetries--;

        } while((wStatus != PH_ERR_SUCCESS) && (wRetries != 0U));

        /* Reset command length */
        bCmdLen = 0U;

        /* Compute the offset */
        bOffset += PHAL_ICODE_BLOCK_SIZE;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ReadCertificate(void * pPalSli15693DataParams, uint8_t bCertNo, uint16_t wWordPtr, uint16_t wWordCount,
    uint8_t ** ppCertificate, uint16_t *pCertLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wRetries = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[8];

    /* Retrieve max retry count */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT,
        &wRetries));

    wRetries += 1U;

    /* Frame ReadSingleBlock command information. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_READ_CERTIFICATE;
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_DH;
    aCmdBuff[bCmdLen++] = PHAL_ICODE_AUTH_PARAM_TAM1_1;
    aCmdBuff[bCmdLen++] = bCertNo;

    (void) memcpy(&aCmdBuff[bCmdLen], (uint8_t *) &wWordPtr, 2U);
    bCmdLen += 2;

    (void) memcpy(&aCmdBuff[bCmdLen], (uint8_t *) &wWordCount, 2U);
    bCmdLen += 2;

    do
    {
        /* Set or clear the flags option bit indicated by bOption. */
        wStatus = phalICode_Int_SetOptionBit(
            pPalSli15693DataParams,
            PHAL_ICODE_OPTION_OFF,
            PH_OFF);

        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Set short + Tolerance(Delta) Timeout(50ms). */
            wStatus = phpalSli15693_SetConfig(
                pPalSli15693DataParams,
                PHPAL_SLI15693_CONFIG_TIMEOUT_US,
                PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_TIMEOUT_DELTA_US);

            if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            {
                /* Exchange the command information to PAL layer. */
                wStatus = phpalSli15693_Exchange(
                    pPalSli15693DataParams,
                    PH_EXCHANGE_DEFAULT,
                    aCmdBuff,
                    bCmdLen,
                    ppCertificate,
                    pCertLen);

                /* Compute the status code. */
                wStatus = phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus);
            }
        }

        wRetries--;

    } while((wStatus != PH_ERR_SUCCESS) && (wRetries != 0U));

    return wStatus;
}
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalICode_Int_Challenge(void * pPalSli15693DataParams, void * pCryptoRngDataParams, uint16_t wOption, uint8_t bChallengeType,
    uint16_t wTimeout, uint16_t wTimeout_Unit, uint8_t bKeyID_CertNum, uint8_t * pChallenge, uint16_t wChallengeLen, uint8_t * pRnd)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wGenRnd = 0;
    uint8_t     PH_MEMLOC_REM bRndSize = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    uint8_t     PH_MEMLOC_REM aCmdBuff[70];
    uint8_t     PH_MEMLOC_REM aIChallenge[PHAL_ICODE_RANDOM_NUMBER_MAX_SIZE];

    /* Extract Parameters */
    wGenRnd = (uint16_t) (wOption & 0x0F00U);

    /* Validate Parameter */
    PHAL_ICODE_VALIDATE_CHALLENGETYPE(bChallengeType);

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        wTimeout_Unit,
        wTimeout));

    /* Clear all the local variables. */
    (void) memset(aCmdBuff, 0x00U, (size_t) sizeof(aCmdBuff));
    (void) memset(aIChallenge, 0x00U, (size_t) sizeof(aIChallenge));

    /* Frame and exchange the Challenge message based on Challenge type. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_CHALLENGE;

    /* Prepare Message based on Challenge Type */
#ifdef NXPBUILD__PHAL_ICODE_INTERNAL
    switch(bChallengeType)
    {
        case PHAL_ICODE_CHALLENGE_TAM_1_2:
            aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_DH;
            aCmdBuff[bCmdLen++] = PHAL_ICODE_AUTH_PARAM_TAM1_2;
            break;

        case PHAL_ICODE_CHALLENGE_TAM_1_3:
            aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_DH;
            aCmdBuff[bCmdLen++] = PHAL_ICODE_AUTH_PARAM_TAM1_3;
            aCmdBuff[bCmdLen++] = bKeyID_CertNum;
            break;

        default:
            aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_AES;
            aCmdBuff[bCmdLen++] = PHAL_ICODE_AUTHPROC_TAM | PHAL_ICODE_TAM_CUSTOMDATA_CLEAR;
            aCmdBuff[bCmdLen++] = bKeyID_CertNum;
            break;
    }
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */
#ifndef NXPBUILD__PHAL_ICODE_INTERNAL
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_AES;
    aCmdBuff[bCmdLen++] = PHAL_ICODE_AUTHPROC_TAM | PHAL_ICODE_TAM_CUSTOMDATA_CLEAR;
    aCmdBuff[bCmdLen++] = bKeyID_CertNum;
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */

    /* Get the random number. */
    if(wGenRnd == PHAL_ICODE_GENERATE_CHALLENGE_INTERNAL)
    {
        /* Get the Random number size to use. */
        bRndSize = PHAL_ICODE_RANDOM_NUMBER_MIN_SIZE;
#ifdef NXPBUILD__PHAL_ICODE_INTERNAL
        switch(bChallengeType)
        {
            case PHAL_ICODE_CHALLENGE_TAM_1_2:
            case PHAL_ICODE_CHALLENGE_TAM_1_3:
                bRndSize = PHAL_ICODE_RANDOM_NUMBER_MAX_SIZE;
                break;

            default:
                bRndSize = PHAL_ICODE_RANDOM_NUMBER_MIN_SIZE;
                break;
        }
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */

        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Rnd(
            pCryptoRngDataParams,
            bRndSize,
            aIChallenge));
    }
    else
    {
        (void) memcpy(aIChallenge, pChallenge, wChallengeLen);
        bRndSize = (uint8_t) wChallengeLen;
    }

    /* Add the random number. */
    (void) memcpy(&aCmdBuff[bCmdLen], aIChallenge, bRndSize);
    bCmdLen += bRndSize;

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        NULL,
        NULL);

    if((wStatus & PH_ERR_MASK) != PH_ERR_IO_TIMEOUT)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
    }

    /* Copy generated random number to parameter. */
    (void) memcpy(pRnd, aIChallenge, bRndSize);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ReadBuffer(uint16_t wCompID, void * pPalSli15693DataParams, void * pCryptoDataParams, void * pKeyStoreDataParams,
    uint8_t * pRndNo, uint8_t bChallengeType, uint8_t bVerify, uint8_t bKeyNo, uint8_t bKeyVer, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwTRnd_TAM1 = 0;
    uint16_t    PH_MEMLOC_REM wC_TAM1 = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;
    uint16_t    PH_MEMLOC_REM wTimeout = 0;
    uint16_t    PH_MEMLOC_REM wTimeout_Unit = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t     PH_MEMLOC_REM aKey[PH_KEYSTORE_KEY_TYPE_AES128_SIZE];
    uint8_t     PH_MEMLOC_REM aRespPlain[16];
    uint8_t     PH_MEMLOC_REM aIChallenge_TAM1[PHAL_ICODE_RANDOM_NUMBER_MAX_SIZE];

#ifndef NXPBUILD__PHAL_ICODE_INTERNAL
    PH_UNUSED_VARIABLE(bChallengeType);
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */

    /* Check if bVerify parameter is greater than 1 .*/
    if(bVerify > 0x01U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Update the Timing according to Challenge Type. */
    wTimeout = PHPAL_SLI15693_TIMEOUT_LONG_US;
    wTimeout_Unit = PHPAL_SLI15693_CONFIG_TIMEOUT_US;
#ifdef NXPBUILD__PHAL_ICODE_INTERNAL
    switch(bChallengeType)
    {
        case PHAL_ICODE_CHALLENGE_TAM_1_2:
        case PHAL_ICODE_CHALLENGE_TAM_1_3:
            wTimeout = 825U + 1U /* As per specification, extended by the additional 1ms to be able to compute and store the results */;
            wTimeout_Unit = PHPAL_SLI15693_CONFIG_TIMEOUT_MS;
            break;

        default:
            wTimeout = PHPAL_SLI15693_TIMEOUT_LONG_US;
            wTimeout_Unit = PHPAL_SLI15693_CONFIG_TIMEOUT_US;
            break;
    }
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        wTimeout_Unit,
        wTimeout));

    /* Clear all the local variables. */
    (void) memset(aCmdBuff, 0x00U, (size_t)sizeof(aCmdBuff));
    (void) memset(aKey, 0x00U, PH_KEYSTORE_KEY_TYPE_AES128_SIZE);
    (void) memset(aRespPlain, 0x00U, (size_t)sizeof(aRespPlain));
    (void) memset(aIChallenge_TAM1, 0x00U, (size_t)sizeof(aIChallenge_TAM1));

    /* Frame the command. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_READ_BUFFER;

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        ppResponse,
        pRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    /* Check if verification is requested. */
    if(0U != bVerify)
    {
        /* Reverse the buffer. */
        phalICode_Int_Reverse(*ppResponse, *pRespLen);

        /* Get the key from key store. */
        PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
            pKeyStoreDataParams,
            bKeyNo,
            bKeyVer,
            (uint8_t) sizeof(aKey),
            aKey,
            &wKeyType));

        /* Check if key type is of type AES. */
        if(wKeyType != PH_KEYSTORE_KEY_TYPE_AES128)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_ICODE);
        }

#ifdef NXPBUILD__PHAL_ICODE_SW
        /* Reverse the Key before loading to crypto params. */
        if((wCompID & PH_COMPID_MASK) == PHAL_ICODE_SW_ID)
        {
            phalICode_Int_Reverse(aKey, (uint16_t) sizeof(aKey));
        }
#endif /* NXPBUILD__PHAL_ICODE_SW */

        /* Load the key to crypto params. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pCryptoDataParams,
            aKey,
            wKeyType));

        /* Decrypt the response to extract the random numbers. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
            pCryptoDataParams,
            (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_ECB),
            *ppResponse,
            PH_CRYPTOSYM_AES_BLOCK_SIZE,
            *ppResponse));

        /* Reverse the buffer. */
        phalICode_Int_Reverse(*ppResponse, PH_CRYPTOSYM_AES_BLOCK_SIZE);

        /* Extract constant and random numbers. */
        (void) memcpy(aIChallenge_TAM1, *ppResponse, PHAL_ICODE_RANDOM_NUMBER_MIN_SIZE);
        (void) memcpy(&dwTRnd_TAM1, (*ppResponse + PHAL_ICODE_RANDOM_NUMBER_MIN_SIZE), 4U);
        (void) memcpy(&wC_TAM1, (*ppResponse + (PHAL_ICODE_RANDOM_NUMBER_MIN_SIZE + 4U /* TChallenge */)), 2U);

        /* Verify the received constant Tag authentication value. */
        if(wC_TAM1 != PHAL_ICODE_CONST_TAM1)
        {
            return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_ICODE);
        }

        /* Verify the IChallenge. */
        if(memcmp(pRndNo, aIChallenge_TAM1, 10U) != 0x00U)
        {
            return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_ICODE);
        }

        /* Reverse the buffer. */
        phalICode_Int_Reverse(*ppResponse, PH_CRYPTOSYM_AES_BLOCK_SIZE);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalICode_Int_ExtendedGetSystemInformation(void * pPalSli15693DataParams, uint8_t bInfoParams, uint8_t ** ppSystemInfo,
    uint16_t * pSystemInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];

    /* Frame ExtendedGetSystemInformation command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_EXTENDED_GET_SYSTEM_INFORMATION;
    aCmdBuff[1U] = bInfoParams;

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Set short timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        2U,
        ppSystemInfo,
        pSystemInfoLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ExtendedGetMultipleBlockSecurityStatus(void * pPalSli15693DataParams, uint8_t bEnableBuffering, uint16_t wBlockNo,
    uint16_t wNoOfBlocks, uint8_t * pStatus, uint16_t * pStatusLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Number of bNoOfBlocks can't be zero */
    if(wNoOfBlocks == 0U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Check number of blocks doesn't exceed 512 */
    if(((uint16_t) wBlockNo + wNoOfBlocks) > PHAL_ICODE_MAX_BLOCKS_EXTENDED)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Set short timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Get the block status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadBlocks(
        pPalSli15693DataParams,
        PHAL_ICODE_CMD_EXTENDED_GET_MULTIPLE_BLOCK_SECURITY_STATUS,
        bEnableBuffering,
        PH_OFF,
        PHAL_ICODE_OPTION_OFF,
        wBlockNo,
        wNoOfBlocks,
        pStatus,
        pStatusLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ExtendedFastReadMultipleBlocks(void * pPalSli15693DataParams, uint8_t bEnableBuffering, uint8_t bOption,
    uint16_t wBlockNo, uint16_t wNumBlocks, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Number of bNumBlocks can't be zero */
    if(wNumBlocks == 0U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Check number of blocks doesn't exceed 512. */
    if(((uint16_t) wBlockNo + wNumBlocks) > PHAL_ICODE_MAX_BLOCKS_EXTENDED)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Set short timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Read the blocks. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadBlocks(
        pPalSli15693DataParams,
        PHAL_ICODE_CMD_EXTENDED_FAST_READ_MULTIPLE_BLOCKS,
        bEnableBuffering,
        PH_ON,
        bOption,
        wBlockNo,
        wNumBlocks,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}





phStatus_t phalICode_Int_SetEAS(void * pPalSli15693DataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    aCmdBuff[0U] = PHAL_ICODE_CMD_SET_EAS;

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ResetEAS(void * pPalSli15693DataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    aCmdBuff[0U] = PHAL_ICODE_CMD_RESET_EAS;

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_LockEAS(void * pPalSli15693DataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    aCmdBuff[0U] = PHAL_ICODE_CMD_LOCK_EAS;

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_EASAlarm(void * pPalSli15693DataParams, uint8_t bOption, uint8_t * pEasIdValue, uint8_t bEasIdMaskLen,
    uint8_t ** ppEas, uint16_t * pEasLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[4];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;

    aCmdBuff[wCmdLen++] = PHAL_ICODE_CMD_EAS_ALARM;

    /* Set or clear the flags option bit indicated by bOption. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    if(bOption != PHAL_ICODE_OPTION_OFF)
    {
        wCmdLen = 2U + (uint16_t) (((uint16_t) bEasIdMaskLen) >> 3U);
        aCmdBuff[1U] = bEasIdMaskLen;
        (void) memcpy(&aCmdBuff[2U], pEasIdValue, (size_t) wCmdLen - 2U);
    }

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        wCmdLen,
        ppEas,
        pEasLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_PasswordProtectEAS(void * pPalSli15693DataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    aCmdBuff[0U] = PHAL_ICODE_CMD_PASSWORD_PROTECT_EAS_AFI;

    /* Clear Option bit to protect the EAS password. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_PasswordProtectAFI(void * pPalSli15693DataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    aCmdBuff[0U] = PHAL_ICODE_CMD_PASSWORD_PROTECT_EAS_AFI;

    /* Set Option bit to protect the AFI password. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_ON,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_WriteEAS_ID(void * pPalSli15693DataParams, uint8_t bOption, uint8_t * pEasIdValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[3];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    aCmdBuff[0U] = PHAL_ICODE_CMD_WRITE_EAS_ID;
    aCmdBuff[1U] = pEasIdValue[0U];
    aCmdBuff[2U] = pEasIdValue[1U];

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        3U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ReadEPC(void * pPalSli15693DataParams, uint8_t ** ppEpc, uint16_t * pEpcLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];

    aCmdBuff[0U] = PHAL_ICODE_CMD_READ_EPC;

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Proceed with the command in lower layers */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        ppEpc,
        pEpcLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_GetNXPSystemInformation(void * pPalSli15693DataParams, uint8_t ** ppSystemInfo, uint16_t * pSystemInfoLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];

    /* Frame GetNXPSystemInformation command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_GET_NXP_SYSTEM_INFORMATION;

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Set short timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        ppSystemInfo,
        pSystemInfoLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_GetRandomNumber(void * pPalSli15693DataParams, uint8_t ** ppRnd, uint16_t * pRndLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];

    aCmdBuff[0U] = PHAL_ICODE_CMD_GET_RANDOM_NUMBER;

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        ppRnd,
        pRndLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_SetPassword(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bPwdIdentifier, uint8_t * pXorPwd)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[6];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    aCmdBuff[0U] = PHAL_ICODE_CMD_SET_PASSWORD;
    aCmdBuff[1U] = bPwdIdentifier;
    (void) memcpy(&aCmdBuff[2], pXorPwd, 4);

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        6U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling
     * Only processed for Privacy and Destroy passwords
     */
    if((0U != bOption) && (bPwdIdentifier == PHAL_ICODE_SET_PASSWORD_PRIVACY))
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_WritePassword(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bPwdIdentifier, uint8_t * pPwd)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[6];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    aCmdBuff[0U] = PHAL_ICODE_CMD_WRITE_PASSWORD;
    aCmdBuff[1U] = bPwdIdentifier;
    (void) memcpy(&aCmdBuff[2], pPwd, 4U);

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        6U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_LockPassword(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bPwdIdentifier)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    aCmdBuff[0U] = PHAL_ICODE_CMD_LOCK_PASSWORD;
    aCmdBuff[1U] = bPwdIdentifier;

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Proceed with the command in lower layers */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        2U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ProtectPage(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bPPAdd_PageNo, uint8_t bProtectionStatus)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[3];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    aCmdBuff[0U] = PHAL_ICODE_CMD_PROTECT_PAGE;
    aCmdBuff[1U] = bPPAdd_PageNo;
    aCmdBuff[2U] = bProtectionStatus;

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Proceed with the command in lower layers */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        3U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_LockPageProtectionCondition(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bPageNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    aCmdBuff[0U] = PHAL_ICODE_CMD_LOCK_PAGE_PROTECTION_CONDITION;
    aCmdBuff[1U] = bPageNo;

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        2U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_GetMultipleBlockProtectionStatus(void * pPalSli15693DataParams, uint8_t bEnableBuffering, uint8_t bBlockNo,
    uint8_t bNoOfBlocks, uint8_t * pProtectionStates, uint16_t * pNumReceivedStates)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Number of bNoOfBlocks can't be zero */
    if(bNoOfBlocks == 0U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Check number of blocks doesn't exceed 256 */
    if((uint16_t) bBlockNo + bNoOfBlocks > PHAL_ICODE_MAX_BLOCKS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Get the blocks status. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadBlocks(
        pPalSli15693DataParams,
        PHAL_ICODE_CMD_GET_MULTIPLE_BLOCK_PROTECTION_STATUS,
        bEnableBuffering,
        PH_OFF,
        PHAL_ICODE_OPTION_OFF,
        bBlockNo,
        bNoOfBlocks,
        pProtectionStates,
        pNumReceivedStates));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_Destroy(void * pPalSli15693DataParams, uint8_t bOption, uint8_t * pXorPwd)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[5];
    uint8_t     PH_MEMLOC_REM bCmdBuffLen = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    aCmdBuff[bCmdBuffLen++] = PHAL_ICODE_CMD_DESTROY;

    /* For SLI-S and SLI-L, the XOR password is not required. */
    if(pXorPwd != NULL)
    {
        (void) memcpy(&aCmdBuff[1], pXorPwd, 4U);
        bCmdBuffLen += (uint8_t) 4U;
    }

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdBuffLen,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    wStatus = phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_EnablePrivacy(void * pPalSli15693DataParams, uint8_t bOption, uint8_t * pXorPwd)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[5];
    uint8_t     PH_MEMLOC_REM bCmdBuffLen = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    aCmdBuff[bCmdBuffLen++] = PHAL_ICODE_CMD_ENABLE_PRIVACY;

    /* For SLI-S and SLI-L, the XOR password is not required. */
    if(pXorPwd != NULL)
    {
        (void) memcpy(&aCmdBuff[1], pXorPwd, 4U);
        bCmdBuffLen += (uint8_t) 4U;
    }

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdBuffLen,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_64BitPasswordProtection(void * pPalSli15693DataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    aCmdBuff[0U] = PHAL_ICODE_CMD_64_BIT_PASSWORD_PROTECTION;

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ReadSignature(void * pPalSli15693DataParams, uint8_t ** ppSign, uint16_t * pSignLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];

    aCmdBuff[0U] = PHAL_ICODE_CMD_READ_SIGNATURE;

    /* Clear Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Exchange the command information to PAL layer */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        ppSign,
        pSignLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ReadConfig(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bBlockAddr, uint8_t bNoOfBlocks,
    uint8_t ** ppData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[4];

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Clear the command buffer. */
    (void) memset(aCmdBuff, 0x00U, (size_t)sizeof(aCmdBuff));

    /* Frame the command. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_READ_CONFIG;

    /* Append the block address. */
    aCmdBuff[bCmdLen++] = bBlockAddr;

    /* Append the no of blocks to read. */
    aCmdBuff[bCmdLen++] = bNoOfBlocks - 1U;

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        ppData,
        pDataLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_WriteConfig(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bBlockAddr, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    uint8_t     PH_MEMLOC_REM aCmdBuff[7];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Clear the command buffer. */
    (void) memset(aCmdBuff, 0x00U, (size_t)sizeof(aCmdBuff));

    /* Frame the command. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_WRITE_CONFIG;

    /* Append the block address. */
    aCmdBuff[bCmdLen++] = bBlockAddr;

    /* Append the block data. */
    (void) memcpy(&aCmdBuff[bCmdLen], pData, 4U);
    bCmdLen = bCmdLen + 4U;

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_PickRandomID(void * pPalSli15693DataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Clear the command buffer. */
    (void) memset(aCmdBuff, 0x00U, (size_t)sizeof(aCmdBuff));

    /* Frame the command. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_PICK_RANDOM_ID;

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ReadTT(void * pPalSli15693DataParams, uint8_t bOption, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;

    uint8_t     PH_MEMLOC_REM aCmdBuff[1];

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_LONG_US));

    /* Clear the command buffer. */
    (void) memset(aCmdBuff, 0x00U, sizeof(aCmdBuff));

    /* Frame the command. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_READ_TT;

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        ppResponse,
        pRspLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ParameterRequest(void * pPalSli15693DataParams, uint8_t * pBitRate, uint8_t * pTiming)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Clear the command buffer. */
    (void) memset(aCmdBuff, 0x00U, (size_t)sizeof(aCmdBuff));

    /* Frame ParameterRequest command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_PARAMETER_REQUEST;

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    /* Update the BitRate and timing parameters with received response. */
    *pBitRate = pResponse[0U];
    *pTiming = pResponse[1U];

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ParameterSelect(void * pPalSli15693DataParams, uint8_t bBitRate, uint8_t bTiming)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[3];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint16_t    PH_MEMLOC_REM wTxDataRate = 0;
    uint16_t    PH_MEMLOC_REM wRxDataRate = 0;
    uint16_t    PH_MEMLOC_REM wTiming = 0;

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Clear the command buffer. */
    (void) memset(aCmdBuff, 0x00, sizeof(aCmdBuff));

    /* Frame ParameterSelect command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_PARAMETER_SELECT;
    aCmdBuff[1U] = bBitRate;
    aCmdBuff[2U] = bTiming;

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        3U,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    /* Calculating the wTxDataRate out of the BitRate */
    if(bBitRate & PHAL_ICODE_PARAMETERS_BITRATE_53KBPS_VCD_VICC)
    {
        wTxDataRate = PHPAL_SLI15693_53KBPS_DATARATE;
    }
    else if(bBitRate & PHAL_ICODE_PARAMETERS_BITRATE_106KBPS_VCD_VICC)
    {
        wTxDataRate = PHPAL_SLI15693_106KBPS_DATARATE;
    }
    else if(bBitRate & PHAL_ICODE_PARAMETERS_BITRATE_212KBPS_VCD_VICC)
    {
        wTxDataRate = PHPAL_SLI15693_212KBPS_DATARATE;
    }
    else
    {
        wTxDataRate = PHPAL_SLI15693_26KBPS_DATARATE;
    }

    /* Calculating the wRxDataRate out of the BitRate */
    if(bBitRate & PHAL_ICODE_PARAMETERS_BITRATE_53KBPS_VICC_VCD)
    {
        wRxDataRate = PHPAL_SLI15693_53KBPS_DATARATE;
    }
    else if(bBitRate & PHAL_ICODE_PARAMETERS_BITRATE_106KBPS_VICC_VCD)
    {
        wRxDataRate = PHPAL_SLI15693_106KBPS_DATARATE;
    }
    else if(bBitRate & PHAL_ICODE_PARAMETERS_BITRATE_212KBPS_VICC_VCD)
    {
        wRxDataRate = PHPAL_SLI15693_212KBPS_DATARATE;
    }
    else
    {
        wRxDataRate = PHPAL_SLI15693_26KBPS_DATARATE;
    }

    /* Calculating the Timings out of the bTiming */
    switch(bTiming)
    {
        case PHAL_ICODE_PARAMETERS_TIMING_320_9_US:
            wTiming = PHPAL_SLI15693_TIMEOUT_SHORT_US;
            break;

        case PHAL_ICODE_PARAMETERS_TIMING_160_5_US:
            wTiming = PHPAL_SLI15693_TIMEOUT_NTAG5_I2C_US;
            break;

        case PHAL_ICODE_PARAMETERS_TIMING_80_2_US:
            wTiming = PHPAL_SLI15693_TIMEOUT_NTAG5_I2C_81_US;
            break;
        default:
            break;
    }

    /* Set TXDATARATE. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TXDATARATE,
        wTxDataRate));

    /* Set RXDATARATE. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_RXDATARATE,
        wRxDataRate));

    /* Set TIMINGS. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_T1_PARAMETER,
        wTiming));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ReadSRAM(void * pPalSli15693DataParams, uint8_t bEnableBuffering, uint8_t bOption, uint8_t bBlockNo,
    uint8_t bNumBlocks, uint8_t * pData, uint16_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Number of bNumBlocks can't be zero */
    if(bNumBlocks == 0U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Check number of blocks doesn't exceed 0x3F. */
    if(((uint16_t) bBlockNo + bNumBlocks) > 0x40U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Set short timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Read the blocks. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ReadBlocks(
        pPalSli15693DataParams,
        PHAL_ICODE_CMD_READ_SRAM,
        bEnableBuffering,
        PH_OFF,
        bOption,
        bBlockNo,
        bNumBlocks,
        pData,
        pDataLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_WriteSRAM(void * pPalSli15693DataParams, uint8_t bEnableBuffering, uint8_t bOption, uint8_t bBlockNo,
    uint8_t bNumBlocks, uint8_t * pData, uint16_t wDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[3];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

/* This flag enables buffering of response data received from ICode tags. This is purely for applications that run on desktop.
 * This flag by default be placed in preprocessor sections. Additionally the application has to enable a flag bEnableBuffering
 * to make this chaining work.
 *
 * To disable this flag remove this macro from ProjectProperties-> C/C++ -> Preprocessor -> Preprocessor Definitions for both
 * DEBUG and RELEASE configurations.
 */
#ifdef PHAL_ICODE_ENABLE_CHAINING
    uint8_t     PH_MEMLOC_REM bCurrBlocksToWrite = 0;
    uint8_t     PH_MEMLOC_REM bCurrBlockNo = 0;
    uint8_t     PH_MEMLOC_REM bMaxNoBlocks = 0;
    uint8_t     PH_MEMLOC_REM bAllBlocksWritten = 0;
#endif /* PHAL_ICODE_ENABLE_CHAINING */

    /* Number of bNumBlocks can't be zero */
    if(bNumBlocks == 0U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Check number of blocks doesn't exceed 0x3F. */
    if(((uint16_t) bBlockNo + bNumBlocks) > 0x40U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_ICODE);
    }

    /* Set or clear the flags option bit indicated by bOption. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_OFF));

    /* Set short timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Reset command buffer and its length variable. */
    bCmdLen = 0U;
    (void) memset(aCmdBuff, 0x00U, (size_t) sizeof(aCmdBuff));

    /* Frame the initial command. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_WRITE_SRAM;

#ifdef PHAL_ICODE_ENABLE_CHAINING
    /* Buffer the response data if Buffering flag is set. */
    if(bEnableBuffering)
    {
        /* Update the maximum number of blocks with respect to Option flag setting. The value for the blocks is fixed to 60 and 40 to avoid multiple
         * handling of different data in response. RD70x can respond with more amount of data but CM1 cannot. So fixing the blocks count to a lower
         * value.
         */
        bMaxNoBlocks = (uint8_t) (bOption ? PHAL_ICODE_MAX_BLOCKS_CM1_OPTION_FLAG_SET : PHAL_ICODE_MAX_BLOCKS_CM1_OPTION_FLAG_NOT_SET);

        /* Blocks to Write. */
        bCurrBlocksToWrite = bMaxNoBlocks;

        /* Update the number of blocks to write if its less than the internal required one. */
        if(bNumBlocks < bMaxNoBlocks)
        {
            bCurrBlocksToWrite = bNumBlocks;
            bAllBlocksWritten = 1U;
        }

        /* Write the blocks. */
        do
        {
            /* If blocks to write and current block sum is more than total no of blocks to write, exit after the transaction */
            if((bCurrBlocksToWrite +  bCurrBlockNo) >= bNumBlocks)
            {
                bAllBlocksWritten = 1U;
            }
            /* Frame command information. */
            aCmdBuff[bCmdLen++] = (uint8_t) (bCurrBlockNo + bBlockNo);

            /* Adjust number of blocks. Adjustment is made because the User or the application will pass
             * the number of blocks starting from 1 to N. But as per Iso15693 specification the number
             * of blocks ranges from 0 - (N - 1).
             */
            /*--bCurrBlocksToWrite;*/

            /* Add number of blocks. */
            aCmdBuff[bCmdLen++] = (uint8_t) (bCurrBlocksToWrite - 1U);

            /* Exchange the command information to PAL layer. */
            PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Exchange(
                pPalSli15693DataParams,
                PH_EXCHANGE_BUFFER_FIRST,
                aCmdBuff,
                bCmdLen,
                NULL,
                NULL));

            /* Exchange the information to PAL layer. */
            wStatus = phpalSli15693_Exchange(
                pPalSli15693DataParams,
                PH_EXCHANGE_BUFFER_LAST,
                &pData[bCurrBlockNo * PHAL_ICODE_BLOCK_SIZE],
                (bCurrBlocksToWrite * PHAL_ICODE_BLOCK_SIZE),
                &pResponse,
                &wRespLen);

            /* Write-alike handling */
            if(bOption)
            {
                wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
            }

            /* Compute the status code. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

            /* Update the block number to a new one. */
            bCurrBlockNo += bMaxNoBlocks;

            /* Update the Current blocks to write. */
            bCurrBlocksToWrite = bMaxNoBlocks;

            /* Reset the command buffer length. */
            bCmdLen = 1U;

            /* Set the remaining blocks to read. */
            if((bNumBlocks - bCurrBlockNo) < bMaxNoBlocks)
                bCurrBlocksToWrite = (uint8_t) (bNumBlocks - bCurrBlockNo);

        } while(!bAllBlocksWritten);
    }
    else
#endif /* PHAL_ICODE_ENABLE_CHAINING */
    {
        /* To avoid build failures. */
        bEnableBuffering = 0U;

        /* Adjust number of blocks. Adjustment is made because the User or the application will pass
         * the number of blocks starting from 1 to N. But as per Iso15693 specification the number
         * of blocks ranges from 0 - (N - 1).
         */
        --bNumBlocks;

        /* Frame ReadMultipleBlock command information. */
        aCmdBuff[bCmdLen++] = bBlockNo;

        /* Add number of blocks. */
        aCmdBuff[bCmdLen++] = bNumBlocks;

        /* Buffer the command information to PAL layer. */
        PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Exchange(
            pPalSli15693DataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            aCmdBuff,
            bCmdLen,
            NULL,
            NULL));

        /* Exchange the information to PAL layer. */
        wStatus = phpalSli15693_Exchange(
            pPalSli15693DataParams,
            PH_EXCHANGE_BUFFER_LAST,
            pData,
            wDataLen,
            &pResponse,
            &wRespLen);

        /* Write-alike handling */
        if(bOption)
        {
            wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
        }

        /* Compute the status code. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_I2CMRead(void * pPalSli15693DataParams, uint8_t bI2CParam, uint16_t wDataLen, uint8_t * pData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[3];
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Validate Data Length. Should not be more than 256 bytes. */
    if(wDataLen > PHAL_ICODE_MAX_I2C_DATA_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_SIZE, PH_COMP_AL_ICODE);
    }

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Set short timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Clear the command buffer. */
    (void) memset(aCmdBuff, 0x00U, (size_t) sizeof(aCmdBuff));

    /* Frame I2CMRead command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_I2CM_READ;
    aCmdBuff[1U] = bI2CParam;
    aCmdBuff[2U] = (uint8_t) (wDataLen - 1U);

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        3U,
        &pData,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_I2CMWrite(void * pPalSli15693DataParams, uint8_t bI2CParam, uint8_t * pData, uint16_t wDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[3];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Validate Data Length. Should not be more than 256 bytes. */
    if(wDataLen > PHAL_ICODE_MAX_I2C_DATA_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_SIZE, PH_COMP_AL_ICODE);
    }

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Set long timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Clear the command buffer. */
    (void) memset(aCmdBuff, 0x00U, (size_t) sizeof(aCmdBuff));

    /* Frame I2CMWrite command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_I2CM_WRITE;
    aCmdBuff[1U] = bI2CParam;
    aCmdBuff[2U] = (uint8_t) (wDataLen - 1U);

    /* Buffer the command information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmdBuff,
        3U,
        NULL,
        NULL));

    /* Buffer data and exchange the buffered information. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pData,
        wDataLen,
        &pResponse,
        &wRespLen);

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

#ifdef NXPBUILD__PHAL_ICODE_INTERNAL
phStatus_t phalICode_Int_IncrCnt(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bCounter, uint16_t wValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_ON));

    /* Frame IncrCnt command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_INCR_CNT;
    aCmdBuff[1U] = bCounter;

    /* Buffer the command information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmdBuff,
        2U,
        NULL,
        NULL));

    /* Buffer data and exchange the buffered information. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_BUFFER_LAST,
        (uint8_t *) &wValue,
        2U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_ReadCnt(void * pPalSli15693DataParams, uint8_t bCounter, uint16_t * pValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Set short timeout */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Frame ReadCnt command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_READ_CNT;
    aCmdBuff[1U] = bCounter;

    /* Exchange the command information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        2U,
        &pResponse,
        &wRespLen));

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    /* Copy response to parameter */
    if((pResponse != NULL) && (wRespLen == 2U))
    {
        *pValue = (uint16_t) ((pResponse[1] << 8) | pResponse[0]);
    }
    else
    {
        *pValue = 0;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_WriteCnt(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bCounter, uint16_t wValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_ON));

    /* Frame WriteCnt command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_WRITE_CNT;
    aCmdBuff[1U] = bCounter;

    /* Buffer the command information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmdBuff,
        2U,
        NULL,
        NULL));

    /* Buffer data and exchange the buffered information. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_BUFFER_LAST,
        (uint8_t *) &wValue,
        2U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_LockCertificate(void * pPalSli15693DataParams, uint8_t bOption, uint8_t bCertNo)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        bOption,
        PH_ON));

    /* Frame I2CMWrite command information. */
    aCmdBuff[0U] = PHAL_ICODE_CMD_LOCK_CERTIFICATE;
    aCmdBuff[1U] = bCertNo;

    /* Buffer the command information. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        2U,
        &pResponse,
        &wRespLen);

    /* Write-alike handling */
    if(0U != bOption)
    {
        wStatus = phalICode_Int_WriteAlikeHandling(pPalSli15693DataParams, wStatus);
    }

    /* Compute the status code. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}

phStatus_t phalICode_Int_FastReadCertificate(void * pPalSli15693DataParams, uint8_t bCertNo, uint16_t wWordPtr, uint16_t wWordCount,
    uint8_t ** ppCertificate, uint16_t *pCertLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint16_t    PH_MEMLOC_REM wRetries = 0;
    uint16_t    PH_MEMLOC_REM wRxDataRate = 0;
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[8];

    /* Retrieve max retry count */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT,
        &wRetries));

    wRetries += 1U;

    /* Frame ReadSingleBlock command information. */
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CMD_FAST_READ_CERTIFICATE;
    aCmdBuff[bCmdLen++] = PHAL_ICODE_CSI_DH;
    aCmdBuff[bCmdLen++] = PHAL_ICODE_AUTH_PARAM_TAM1_1;
    aCmdBuff[bCmdLen++] = bCertNo;

    (void) memcpy(&aCmdBuff[bCmdLen], (uint8_t *) &wWordPtr, 2U);
    bCmdLen += 2;

    (void) memcpy(&aCmdBuff[bCmdLen], (uint8_t *) &wWordCount, 2U);
    bCmdLen += 2;

    do
    {
        /* Set or clear the flags option bit indicated by bOption. */
        wStatus = phalICode_Int_SetOptionBit(
            pPalSli15693DataParams,
            PHAL_ICODE_OPTION_OFF,
            PH_OFF);

        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            /* Set short + Tolerance(Delta) Timeout(50ms). */
            wStatus = phpalSli15693_SetConfig(
                pPalSli15693DataParams,
                PHPAL_SLI15693_CONFIG_TIMEOUT_US,
                PHPAL_SLI15693_TIMEOUT_SHORT_US + PHPAL_SLI15693_TIMEOUT_DELTA_US);

            if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            {
                /* Configure Fast DataRate. */
                wStatus = phalICode_Int_SetFast_DataRate(pPalSli15693DataParams, &wRxDataRate);

                if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
                {
                    /* Exchange the command information to PAL layer. */
                    wStatus = phpalSli15693_Exchange(
                        pPalSli15693DataParams,
                        PH_EXCHANGE_DEFAULT,
                        aCmdBuff,
                        bCmdLen,
                        ppCertificate,
                        pCertLen);

                    /* Compute the status code. */
                    wStatus = phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus);
                }

                /* Revert back DataRate  */
                PH_CHECK_SUCCESS_FCT(wStatus1, phpalSli15693_SetConfig(
                    pPalSli15693DataParams,
                    PHPAL_SLI15693_CONFIG_RXDATARATE,
                    wRxDataRate));
            }
        }

        wRetries--;

    } while((wStatus != PH_ERR_SUCCESS) && (wRetries != 0U));

    return wStatus;
}

phStatus_t phalICode_Int_FastReadBuffer(void * pPalSli15693DataParams, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint16_t    PH_MEMLOC_REM wRxDataRate = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1];

    /* Clear all the local variables. */
    (void) memset(aCmdBuff, 0x00U, (size_t)sizeof(aCmdBuff));

    /* Update Option bit */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetOptionBit(
        pPalSli15693DataParams,
        PHAL_ICODE_OPTION_OFF,
        PH_OFF));

    /* Set short timeout. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_TIMEOUT_US,
        PHPAL_SLI15693_TIMEOUT_SHORT_US));

    /* Configure Fast DataRate. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalICode_Int_SetFast_DataRate(
        pPalSli15693DataParams,
        &wRxDataRate));

    /* Frame the command. */
    aCmdBuff[0] = PHAL_ICODE_CMD_FAST_READ_BUFFER;

    /* Exchange the command. */
    wStatus = phpalSli15693_Exchange(
        pPalSli15693DataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        ppResponse,
        pRespLen);

    /* Compute the status code. */
    wStatus = phalICode_Int_ComputeErrorCode(pPalSli15693DataParams, wStatus);

    /* Revert back DataRate  */
    PH_CHECK_SUCCESS_FCT(wStatus1, phpalSli15693_SetConfig(
        pPalSli15693DataParams,
        PHPAL_SLI15693_CONFIG_RXDATARATE,
        wRxDataRate));

    return wStatus;
}
#endif /* NXPBUILD__PHAL_ICODE_INTERNAL */





phStatus_t phalICode_Int_GetTagType(void * pPalSli15693DataParams, uint16_t * pTagType)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aUID[PHPAL_SLI15693_UID_LENGTH];
    uint8_t     PH_MEMLOC_REM bUidLen = 0;
    uint16_t    PH_MEMLOC_REM wTagType;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalSli15693_GetSerialNo(
        pPalSli15693DataParams,
        aUID,
        &bUidLen));

    /* Reverse the buffer. */
    phalICode_Int_Reverse(aUID, PHPAL_SLI15693_UID_LENGTH);

    /* Extract the tag type from the UID. */
    wTagType = (uint16_t) ((aUID[3U] << 8U) | (aUID[2U]));

    /* Apply the tag type. */
    if(wTagType == PHAL_ICODE_TAG_TYPE_ICODE_SLI)
        *pTagType = PHAL_ICODE_TAG_TYPE_ICODE_SLI;
    else if(wTagType == PHAL_ICODE_TAG_TYPE_ICODE_SLI_S)
        *pTagType = PHAL_ICODE_TAG_TYPE_ICODE_SLI_S;
    else if(wTagType == PHAL_ICODE_TAG_TYPE_ICODE_SLI_L)
        *pTagType = PHAL_ICODE_TAG_TYPE_ICODE_SLI_L;
    else if(wTagType == PHAL_ICODE_TAG_TYPE_ICODE_SLIX)
        *pTagType = PHAL_ICODE_TAG_TYPE_ICODE_SLIX;
    else if(wTagType == PHAL_ICODE_TAG_TYPE_ICODE_SLIX_S)
        *pTagType = PHAL_ICODE_TAG_TYPE_ICODE_SLIX_S;
    else if(wTagType == PHAL_ICODE_TAG_TYPE_ICODE_SLIX_L)
        *pTagType = PHAL_ICODE_TAG_TYPE_ICODE_SLIX_L;
    else if(wTagType == PHAL_ICODE_TAG_TYPE_ICODE_DNA)
        *pTagType = PHAL_ICODE_TAG_TYPE_ICODE_DNA;
    else if(wTagType == PHAL_ICODE_TAG_TYPE_ICODE_NTAG_5)
        *pTagType = PHAL_ICODE_TAG_TYPE_ICODE_NTAG_5;
    else if(wTagType == PHAL_ICODE_TAG_TYPE_ICODE_SLI_X2)
        *pTagType = PHAL_ICODE_TAG_TYPE_ICODE_SLI_X2;
    else if(wTagType == PHAL_ICODE_TAG_TYPE_ICODE_SLI_X3)
        *pTagType = PHAL_ICODE_TAG_TYPE_ICODE_SLI_X3;
    else
        *pTagType = PHAL_ICODE_TAG_TYPE_UNKNOWN;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_ICODE);
}
#endif /* NXPBUILD__PHAL_ICODE */
